// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "loops.h"
#include "after_opts.h"
#include "angles.h"
#include "cenlist.h"
#include "files_paths.h"
#include "fragments_ns.h"
#include "fullatom.h"
#include "input_pdb.h"
#include "jumping_loops.h"
#include "knots.h"
#include "loop_class.h"
#include "loops_ns.h"
#include "make_pdb.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "native.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "pdb.h"
#include "pose_loops.h"
#include "random_numbers.h"
#include "recover.h"
#include "refold.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "status.h"
#include "termini.h"
#include "util_vector.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <sstream>


////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_start_loops
///
/// @brief
///
/// @detailed define loop regions; setup for segmental folding
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_start_loops()
{
	using namespace files_paths;
	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace pdb;

	if ( !get_loop_flag() ) return;

	static bool init = { false };
	static std::string extension;
	static std::string one_loop_file;
	static bool start_from_extend = { false };

	bool exist = false;
	FArray1D_bool allow_insert( MAX_RES()() );

	if ( !init ) {
		init = true;
		set_pose_loop_flag( truefalseoption("pose") );
		fix_natsc = truefalseoption("fix_natsc");
		start_from_extend = truefalseoption("start_from_extend");
		if ( fix_natsc ) input_fa = true;
		stringafteroption( "one_loop_file", "none", one_loop_file );
		if ( one_loop_file == "none" )
			stringafteroption( "loop_library", "loops", extension );
	}

	if ( get_pose_loop_flag() ) {
		pose_ns::Loops loops;
		exist = pose_loops_init_from_file( loops );
	}
//car define loop regions, read in library conformations
	if ( !exist ) {
		std::string filename;
		if ( one_loop_file != "none" ) {
			filename = one_loop_file;
			strip_path( one_loop_file, libraryname );
		} else {
			libraryname = start_file + '.' + extension;
			filename = start_path + libraryname;
		}
		read_loop_library( filename, start_x, exist );
	}

	if ( !exist ) {
		std::cout << "ERROR :: library file required!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	save_START_loops();

//car template modifications; search for undefined coordiates
	for ( int j = 1; j <= total_residue; ++j ) {
		name(j) = "TMPL";
	}
	if ( get_pose_loop_flag() ) {
		for ( int kk = 1; kk <= num_loop; ++kk ) {
			for ( int j = loop_begin(kk), je = loop_end(kk); j <= je; ++j ) {
				name(j) = "INPT";
			}
		}
	} else {
		for ( int kk = 1; kk <= num_loop; ++kk ) {
			for ( int j = loop_begin(kk) + 1, je = loop_end(kk) - 1; j <= je; ++j ) {
				if ( occupancy(1,j) < 0.0 || occupancy(2,j) < 0.0 || occupancy(3,j) < 0.0 ) {
					//N,CA or C missing
					set_extended_angles(j);
				}	else if ( start_from_extend ) { //chu make it command-line flag dependent
					set_extended_angles(j);
				} else {
					name(j) = "INPT";
				}
			}
			if ( loop_begin(kk) < total_residue ) {
				if ( name(loop_begin(kk)+1) == "EXTD" ) set_extended_angles(loop_begin(kk));
			}
			if ( loop_end(kk) > 1 ) {
				if ( name(loop_end(kk)-1) == "EXTD" ) set_extended_angles(loop_end(kk));
			}
		}
	}
	crossing_loops();

//car prepare for scoring
//car--------------------
	score_set_loop_weight(0.0);
	overlap_coord = 0.0;

//car prepare for folding
//car--------------------
//car define splice direction
//car non-extended, non-terminal, measure gaps
	select_splice_dir(randomize_dir);
	setup_loop_splicing(randomize_dir);

//car generate coordinates for extended regions if necessary
	set_allow_insert_by_extended(true,exist);
	if ( exist ) {
		monte_carlo_reset();
		refold(1,total_residue);    // must refold to generate extended coords;
		if ( input_fa ) {
//car if fullatom input and regions of undefined coords exist, then
//car sidechain coords will have been improperly built in input_pdb.
			retrieve_allow_insert(allow_insert,total_residue);
			set_allow_repack(allow_insert,total_residue);
		//replaces fullatom coords with those from position over refolded regions only (N,CA,C,O,CB)
		//fixed regions also copied, but these will be identical in coord and fullcoord
			fullatom_pack_position(false);   //discard existing rotamers on sites to be repacked.
		}
	}

//car set the maps according to loop definitions
	reset_loops();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin select_splice_dir
///
/// @brief select direction for folding
///
/// @detailed
///
/// @param[out] random   out - was random splice direction selected?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car if template appears to have loops folded in a
///car particular direction, and we aren't folding from
///car scratch, use this direction
///car otherwise random direction will be used (random for each decoy)
///car template direction checked by comparing input
///car torsion angles with measured torsion angles
///
/// should add command line options
///            checking of a screened loop library so that same direction used
///
/// @references
///
/// @authors car
///
/// @last_modified 9/5/2003
/////////////////////////////////////////////////////////////////////////////////

void
select_splice_dir( bool & random /* was direction set randomly? */ )
{
	using namespace loops_ns;
	using namespace misc;

//car constants
	float const threshold = { 1.0 };

	splice_dir = 0;
	random = true;

//car could look for command line options here

	if ( fold ) return;
	float ndiff = 0.0;
	float cdiff = 0.0;
	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_is_extended(i) ) goto L100;
		{ // Scope
		int const first = loop_begin(i);
		int const last = loop_end(i);
		if ( first == 1 ) goto L100;
		if ( last == total_residue ) goto L100;
		if ( last == 1 ) goto L100;
		if ( first == total_residue ) goto L100;
		ndiff += std::abs(psi(first) -
		 dihedral(Eposition(1,1,first),Eposition(1,2,first),Eposition(1,4,first),
		 Eposition(1,1,first+1))); // psi
		cdiff += std::abs(phi(last) -
		 dihedral(Eposition(1,4,last-1),Eposition(1,1,last),Eposition(1,2,last),
		 Eposition(1,4,last+1))); // phi
		}
L100:;
	}
//car if both ndiff and cdiff are non-zero, then gaps in boths directions
//car no obvious direction to choose, so randomize
	if ( ndiff > threshold && cdiff > threshold ) return;

	if ( ndiff > threshold ) splice_dir = c2n;
	if ( cdiff > threshold ) splice_dir = n2c;
	if ( splice_dir != 0 ) random = false;
	std::cout << "FOLDING DIRECTION " << splice_dir << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_decoy_loops
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_decoy_loops()
{
	using namespace loops_ns;

	if ( ! get_loop_flag() ) return;
	for ( int i = 1; i <= num_loop; ++i ) {
		loop_begin(i) = start_loop_begin(i);
		loop_end(i) = start_loop_end(i);
		loop_depth(i) = start_loop_depth(i);
	}
	num_loop = start_num_loop;
	reset_loops();
	score_set_loop_weight(0.0);
	if ( randomize_dir ) setup_loop_splicing(true);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin save_START_loops
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
save_START_loops()
{
	using namespace loops_ns;

	for ( int i = 1; i <= num_loop; ++i ) {
		start_loop_begin(i) = loop_begin(i);
		start_loop_end(i) = loop_end(i);
		start_loop_depth(i) = loop_depth(i);
	}
	start_num_loop = num_loop;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_extended_angles
///
/// @brief
///
/// @detailed
///
/// @param[in] j   in - residue for which angles should be set
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///
////////////////////////////////////////////////////////////////////////////////
void
set_extended_angles( int j )
{
	using namespace misc;
	using namespace param;

	phi(j) = init_phi;
	psi(j) = init_psi;
	omega(j) = init_omega;
	secstruct(j) = 'L';
	name(j) = "EXTD";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_loop_trial
///
/// @brief
///
/// @detailed
///
/// @param  score_fxn - [in/out]? -
/// @param  cycle - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_loop_trial(
	Scoring_Function score_fxn,
	int cycle
)
{
	using namespace loops_ns;

	std::string const move_type( "loop" );

	int begin = 0, end = 0, loop_num = 0, depth;
	bool err;

	choose_random_loop(begin,end,loop_num);
	depth = static_cast< int >( ran3() * loop_depth(loop_num) + 1 );
	loop_insert(loop_num,depth,err);

	if ( err ) {  //cems this should never happen
		std::cout << "POSITION IMPOSSIBLE: return from loop insert error" << std::endl;
		std::cout << "loop_num " << loop_num << " depth " << depth << ' ' <<
		 loop_depth(loop_num) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	refold(begin,end);
	mc_global_track::mc_score::score = score_fxn();

	save_status_info(move_type,begin,end-begin+1);
	monte_carlo(cycle);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_loop_min_trial
///
/// @brief
///
/// @detailed
///
/// @param  score_fxn - [in/out]? -
/// @param  cycle - [in/out]? -
/// @param  type - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_loop_min_trial(
	Scoring_Function score_fxn,
	int cycle,
	std::string const & type
)
{
	using namespace loops_ns;

	std::string const move_type( "loopmin" );

	int begin = 0, end = 0, loop_num = 0, depth;
	bool err,gfrag;

	choose_random_loop(begin,end,loop_num);
	depth = static_cast< int >( ran3() * loop_depth(loop_num) + 1 );
	loop_insert(loop_num,depth,err);

	if ( err ) {  //cems this should never happen
		std::cout << "POSITION IMPOSSIBLE: return from loop insert error" << std::endl;
		std::cout << "loop_num " << loop_num << " depth " << depth << ' ' <<
		 loop_depth(loop_num) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	refold(begin,end);
	minimize( type, move_type, score_fxn, begin, end, gfrag );

	save_status_info( move_type, begin, end-begin+1 );
	monte_carlo(cycle);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_loop_insert
///
/// @brief
///
/// @detailed
///
/// @param  scoring_function - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
main_loop_insert( Scoring_Function scoring_function )
{
	using namespace loops_ns;

	int begin = 0, end = 0, loop_num = 0, depth;
	bool err;

	choose_random_loop(begin,end,loop_num);
	depth = static_cast< int >( ran3() * loop_depth(loop_num) + 1 );
	loop_insert(loop_num,depth,err);

	if ( err ) {  //cems this should never happen
		std::cout << "POSITION IMPOSSIBLE: return from loop insert error" << std::endl;
		std::cout << "loop_num " << loop_num << " depth " << depth << ' ' <<
		 loop_depth(loop_num) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	refold(begin,end);
	mc_global_track::mc_score::score = scoring_function();

// can't use mc reset cause don't want to accept low
	save_status_info("loop-forced",loop_num,end-begin+1);
	output_status_file(2,-1.0);
	monte_carlo_accept_best();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_loop_deriv
///
/// @brief
///compute the derivative of f = (V-V')**2 with respect to each
///of the torsion angles in the loop
///
/// @detailed
///V is overlap_coord (ie loop) and V' is position (ie template)
///at the overlap site  (assume overlap of 1 residue)
///this is the function used in segment_overlap/splice_rms
///Using the chain rule, we have
///
///   df/d phi = df/dri dri/dphi = 2ri dri/dphi.
///
///   dri/dphi = Eab x (V-Vb) . (V' - V)/|V-V'|
///
///   (the first cross product is the displacement of V upon a rotation dphi
///   around the unit vector Eab, Vb is the coordinates of the second atom in
///   the bond)
///
///   since | V-V'| = ri,
///
///   df/ dphi = 2 Eab x (V-Vb) . (V' - V)
///
/// @param  itorsion - [in/out]? - residue whose torsion the deriv is w/ respect to
/// @param  phi_or_psi - [in/out]? -
/// @param  Eab - [in/out]? - unit vector along bond
/// @param  loop_deriv - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
eval_loop_deriv(
	int & itorsion, // residue whose torsion the deriv is w/ respect to
	int & phi_or_psi,
	FArray1Da_float Eab, // unit vector along bond
	float & loop_deriv
)
{
	using namespace loops_ns::loop_flag_bool;
	using namespace loops_ns;
	using namespace misc;
	using namespace scorefxns;
	using numeric::conversions::radians;

	Eab.dimension( 3 );

//car parameters
	int const natom = { 5 };

//car local
	int iatom = 0, ires; // atom and residue of 2nd atom in torsion angle
	int iloop,key;
	int jatom,jres,jstart; // j refers to overlap position
	FArray1D_float vec( 3 );
	FArray1D_float Eabxvec( 3 );

	loop_deriv = 0.;
	if ( !loop_flag ) return; // no loops

	if ( minimize_res_is_excluded(itorsion) ) return; // not a variable residue

	ires = itorsion;
	iloop = loop_map(itorsion);
	if ( loop_begin(iloop) == 1 || loop_end(iloop) == total_residue ) return;

	jstart = 1;
	if ( phi_or_psi == 1 ) {
		iatom = 2; // CA
	} else if ( phi_or_psi == 2 ) {
		iatom = 4; // C
	} else if ( phi_or_psi == 3 ) {
		iatom = 1; // N
		ires = itorsion+1; // of i+1
//car special case: omega of last residue in loop - no N in deriv
		if ( loop_get_refold_dir(iloop) == n2c && itorsion == loop_end(iloop) ) jstart = 2;
	} else {
		std::cout << "unknown phipsi type in eval_loop_deriv " << phi_or_psi <<
		 std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	int loop_dir = loop_get_refold_dir(iloop);
	if ( loop_dir == n2c ) {
		jres = loop_end(iloop)+1;
		key = 2;
	} else if ( loop_dir == c2n ) {
		jres = loop_begin(iloop)-1;
		key = 1;
	} else {
		std::cout << "loop closing deriv not supported for fold direction " <<
		 loop_dir << ' ' << iloop << ' ' << ires << ' ' << phi_or_psi <<
		 std::endl;
		return;
	}

	for ( jatom = jstart; jatom <= natom; ++jatom ) {
//car vec = V-Vb
		vec(1) = overlap_coord(1,jatom,iloop,key) - Eposition(1,iatom,ires);
		vec(2) = overlap_coord(2,jatom,iloop,key) - Eposition(2,iatom,ires);
		vec(3) = overlap_coord(3,jatom,iloop,key) - Eposition(3,iatom,ires);

//car Eabxvec = Eab X vec
//car note have to reverse sign, cause Eab is from Cterm atom to Nterm
		cros(vec,Eab,Eabxvec); // ( vec X Eab ) == -( Eab X vec )

//car vec = V'-V
		vec(1) = Eposition(1,jatom,jres) - overlap_coord(1,jatom,iloop,key);
		vec(2) = Eposition(2,jatom,jres) - overlap_coord(2,jatom,iloop,key);
		vec(3) = Eposition(3,jatom,jres) - overlap_coord(3,jatom,iloop,key);

//car dE/dphi = 2 * Eabxvec . vec (converted to radians)
//car why not a factor of 2?!?
		loop_deriv += radians( dotprod(Eabxvec,vec) );
	}
	loop_deriv *= 2 * splicemsd_weight / ( natom - jstart + 1 );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_loop_flag
///
/// @brief
///
/// @detailed
///
/// @param  yes_no - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_loop_flag( bool yes_no )
{
	using namespace loops_ns;
	using namespace loops_ns::loop_flag_bool;

	loop_flag = yes_no;

//car if turning on, set num_loop, loop_begin,loop_end,
//car then reset_loops()

	if ( !loop_flag ) {
		num_loop = 0;
		reset_loops();
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_loop_flag
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_loop_flag()
{
	return loops_ns::loop_flag_bool::loop_flag;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin set_pose_loop_flag
///
/// @brief set flag for pose_loop modeling
///
/// @detailed
///
/// @param  yes_no - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_pose_loop_flag( bool yes_no )
{
	loops_ns::loop_flag_bool::pose_loop_flag = yes_no;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_pose_loop_flag
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_pose_loop_flag()
{
	return loops_ns::loop_flag_bool::pose_loop_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin choose_random_loop
///
/// @brief
/// choose a random loop, weighted by the number of variable residues
/// in each loop
///
/// @detailed
///
/// @param  begin - [in/out]? -
/// @param  end - [in/out]? -
/// @param  num - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
choose_random_loop(
	int & begin,
	int & end,
	int & num
)
{

	using namespace protein_maps;

	int residue = insert_map( static_cast< int >( ran3() * total_insert ) + 1 );
	identify_loop(residue,residue,begin,end,num);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_save_overlap_coord
///
/// @brief
///
/// @detailed
///
/// @param  loop_num - [in/out]? -
/// @param  key - [in/out]? - 1 = Nterminal, 2 = Cterminal
/// @param  position - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_save_overlap_coord(
	int loop_num,
	int key, // 1 = Nterminal, 2 = Cterminal
	FArray2Da_float position
)
{
	using namespace loops_ns;
	using namespace param;

	position.dimension( 3, MAX_POS );

	for ( int j = 1; j <= 5; ++j ) {
		for ( int i = 1; i <= 3; ++i ) {
			overlap_coord(i,j,loop_num,key) = position(i,j);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_get_overlap
///
/// @brief
///
/// @detailed
///
/// @param  loop_num - [in/out]? -
/// @param  key - [in/out]? - 1 = Nterminal, 2 = Cterminal
/// @param  position - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_get_overlap(
	int loop_num,
	int key, // 1 = Nterminal, 2 = Cterminal
	FArray2Da_float position
)
{
	using namespace loops_ns;
	using namespace param;

	position.dimension( 3, MAX_POS );

	for ( int j = 1; j <= 5; ++j ) {
		for ( int i = 1; i <= 3; ++i ) {
			position(i,j) = overlap_coord(i,j,loop_num,key);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_get_best_overlap
///
/// @brief
///
/// @detailed
///
/// @param  loop_num - [in/out]? -
/// @param  key - [in/out]? - 1 = Nterminal, 2 = Cterminal
/// @param  position - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_get_best_overlap(
	int loop_num,
	int key, // 1 = Nterminal, 2 = Cterminal
	FArray2Da_float position
)
{
	using namespace loops_ns;
	using namespace param;

	position.dimension( 3, MAX_POS );

	for ( int j = 1; j <= 5; ++j ) {
		for ( int i = 1; i <= 3; ++i ) {
			position(i,j) = best_overlap_coord(i,j,loop_num,key);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_score_start
///
/// @brief
/// score the template region only
///
/// @detailed
///
/// @param[in]  filename
/// @param[in]  fullatom
/// @param[in]  rotamers_exist
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// right now this code assumes the loops are in sequential order!!
/// if fullatom is true, scores fullatom terms even if the scoring
/// function isn't a fullatom scoring function.  If false, score fullatom
/// only if scoring function is fullatom
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_score_start(
	std::string const & filename,
	bool const & fullatom,
	bool const & rotamers_exist
)
{
	using namespace loops_ns;
	using namespace misc;
	using namespace pdb;
	using namespace protein_maps;

	if ( !get_loop_flag() ) return;
	if ( !truefalseoption("score_all_loops") ) return;

	std::string label;
	for ( int i = 0; i <= num_loop; ++i ) {
		if ( i > 0 ) {
			if ( occupancy(1,loop_begin(i)+1) < 0.0 ) continue; // skip extended loops
			label = "loop_" + lead_zero_string_of( i, 3 );
		} else {
			label = "template";
		}
		move_loop_into_position( i, total_residue );
		if ( fullatom ) {
			fullatom_score_position( true, rotamers_exist );
		} else {
			mc_global_track::mc_score::score = scorefxn();
		}
		scorefile_output( filename, label, fullatom );
		if ( i == 0 ) {
			if ( !knot_filter() ) std::cout <<
			 "WARNING:: starting template is knotted!" << std::endl;
		}
		loop_restore_position();
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin move_loop_into_position
///
/// @brief
///
/// @detailed
/// saves the values in global current arrays and
/// moves a loop plus some surrounding residues into the global current arrays
/// so that all operations (except frag insertion and loop insertion)
/// can be performed on this smaller section
///
/// @param   loop_num_in - [in/out]? -
/// @param   context - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// note that this is terminated with a monte_carlo_reset, so only
/// the saved structure will be recoverd by loop_restore_position (ie
/// not best or LOW).  START can be recovered.
///
/// the refold hijack flag is not set because we're still in loop mode,
/// the position of the loop has just changed.
///
/// @references
///
/// @authors car
///
/// @authors
////////////////////////////////////////////////////////////////////////////////
void
move_loop_into_position(
	int & loop_num_in,
	int & context // number of surrounding residues to include
)
{
	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace loops_ns::pose_hijack;

	int const max_atom = MAX_ATOM();

	FArray2D_float overlapN( 3, 5 );
	FArray2D_float overlapC( 3, 5 );

//car misc.h variables to be saved + what we moved in

//car loops.h variables to be saved (only the first loop which we will overwrite)

	loop_num = loop_num_in;
//car save variables  save best, copy position, mc reset
//car                 will restore to position then mc reset
	store_residue = total_residue;
	for ( int i = 1; i <= total_residue; ++i ) {
		store_secstruct(i) = secstruct(i);
		store_res(i) = res(i);
		store_resvariant(i) = res_variant(i);
		store_phi(i) = phi(i);
		store_psi(i) = psi(i);
		store_omega(i) = omega(i);
		store_name(i) = name(i);
		for ( int k = 1; k <= 3; ++k ) {
			for ( int j = 1; j <= 5; ++j ) {
				Estore_position(k,j,i) = Eposition(k,j,i);
			}
			for ( int j = 1; j <= max_atom; ++j ) {
				store_fullcoord(k,j,i) = full_coord(k,j,i);
			}
			store_centroid(k,i) = centroid(k,i);
		}
	}

	store_num_loop = num_loop;
	int begin, end;
	if ( loop_num != 0 ) {
		store_loop_begin = loop_begin(1);
		store_loop_end = loop_end(1);
		loop_get_overlap(1,1,store_overlapN);
		loop_get_overlap(1,2,store_overlapC);

//car decide which residues to move to front of arrays
		begin = loop_begin(loop_num);
		end = loop_end(loop_num);
	} else {
		begin = total_residue+1;
		end = 0;
	}
	begin = std::max(begin-context,1);
	end = std::min(end+context,total_residue);

//car replace variables
	int ires = 0;
	for ( int i = begin; i <= end; ++i ) {
		if ( loop_map(i) != 0 && loop_map(i) != loop_num ) goto L200;
		++ires;
		res(ires) = store_res(i);
		res_variant(ires) = store_resvariant(i);
		secstruct(ires) = store_secstruct(i);
		phi(ires) = store_phi(i);
		psi(ires) = store_psi(i);
		omega(ires) = store_omega(i);
		for ( int k = 1; k <= 3; ++k ) {
			for ( int j = 1; j <= 5; ++j ) {
				Eposition(k,j,ires) = Estore_position(k,j,i);
			}
			for ( int j = 1; j <= max_atom; ++j ) {
				full_coord(k,j,ires) = store_fullcoord(k,j,i);
			}
			centroid(k,ires) = store_centroid(k,i);
		}

		if ( loop_num != 0 ) {
			if ( loop_num == 1 ) {
				if ( i == store_loop_begin ) loop_begin(1) = ires;
				if ( i == store_loop_end ) loop_end(1) = ires;
			} else {
				if ( i == loop_begin(loop_num) ) loop_begin(1) = ires;
				if ( i == loop_end(loop_num) ) loop_end(1) = ires;
			}
		}

L200:;
	}
	total_residue = ires;
	update_sequence();

	if ( loop_num != 0 ) {
		num_loop = 1;
		loop_get_overlap(loop_num,1,overlapN);
		loop_get_overlap(loop_num,2,overlapC);
		loop_save_overlap_coord(1,1,overlapN);
		loop_save_overlap_coord(1,2,overlapC);

		setup_loop_splicing(false); // reset directions,segments
	} else {
		num_loop = 0;
	}

	reset_loops();
	score_set_new_pose();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_restore_position
///
/// @brief
/// restore position, don't keep any of the modifications that may have been
/// made.
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_restore_position()
{

	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace loops_ns::pose_hijack;

	int const max_atom = MAX_ATOM();

	FArray2D_float overlapN( 3, 5 );
	FArray2D_float overlapC( 3, 5 );

//car misc.h variables to be saved + what we moved in

//car loops.h variables to be saved (only the first loop which is overwriten)

//car put modifed region into proper place:
	if ( loop_num != 0 ) {
		int ii = 0;
		if ( loop_num == 1 ) {
			ii = store_loop_begin;
		} else {
			ii = loop_begin(loop_num);
		}
		for ( int i = loop_begin(1), ie = loop_end(1); i <= ie; ++i ) {
			store_phi(ii) = phi(i);
			store_psi(ii) = psi(i);
			store_omega(ii) = omega(i);
			store_res(ii) = res(i);
			store_resvariant(ii) = res_variant(i);
			store_secstruct(ii) = secstruct(i);
			store_name(ii) = name(i);

			for ( int k = 1; k <= 3; ++k ) {
				for ( int j = 1; j <= 5; ++j ) {
					Estore_position(k,j,ii) = Eposition(k,j,i);
				}
				for ( int j = 1; j <= max_atom; ++j ) {
					store_fullcoord(k,j,ii) = full_coord(k,j,i);
				}
				store_centroid(k,ii) = centroid(k,i);
			}
			++ii;
		}

//car retrieve new loop overlap
		loop_get_overlap(1,1,overlapN);
		loop_get_overlap(1,2,overlapC);
	}

//car put saved arrays back in current
	for ( int i = 1; i <= store_residue; ++i ) {
		secstruct(i) = store_secstruct(i);
		res(i) = store_res(i);
		res_variant(i) = store_resvariant(i);
		phi(i) = store_phi(i);
		psi(i) = store_psi(i);
		omega(i) = store_omega(i);
		name(i) = store_name(i);
		for ( int k = 1; k <= 3; ++k ) {
			for ( int j = 1; j <= 5; ++j ) {
				Eposition(k,j,i) = Estore_position(k,j,i);
			}
			for ( int j = 1; j <= max_atom; ++j ) {
				full_coord(k,j,i) = store_fullcoord(k,j,i);
			}
			centroid(k,i) = store_centroid(k,i);
		}
	}
	total_residue = store_residue;
	update_sequence();

	num_loop = store_num_loop;
//car restore information about loop 1
	if ( loop_num != 0 ) {
		loop_begin(1) = store_loop_begin;
		loop_end(1) = store_loop_end;
		loop_save_overlap_coord(1,1,store_overlapN);
		loop_save_overlap_coord(1,2,store_overlapC);

//car save new information about transported loop
		loop_save_overlap_coord(loop_num,1,overlapN);
		loop_save_overlap_coord(loop_num,2,overlapC);
	}
	setup_loop_splicing(false); // reset directions,segments
	reset_loops();
	score_set_new_pose();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin trim_template
///
/// @brief
///
/// @detailed
/// saves the values in global current arrays and moves variable regions
/// (via allow_insert) plus some surrounding residues (by distance) into
/// the global current arrays so that all operations (including frag
/// insertion and loop insertion) can be performed on this smaller section
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// note that this is terminated with a monte_carlo_reset, so only
/// the saved structure will be recoverd by trim_template_reset (ie
/// not best or LOW).  START can be recovered.
///
/// the refold hijack flag is not set because we're still in loop mode,
/// the position of the loop has just changed.
///
///chu trim_template only changes the current pose,  but not the starting pose.
///chu If the fullatom_coord is built after trimming,(set_fullatom_flag(true))
///chu starting side-chains will not be added to the current backbone correctly
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
trim_template()
{
	using namespace cenlist_ns;
	using namespace files_paths;
	using namespace fragments;
	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace protein_maps;
	using namespace loops_ns::trim_hijack;

	int const max_atom = MAX_ATOM();

	float const STANDARD_CUTOFF = { cen_dist_cutoff2 + 25.0 }; //Angstroms**2
	float const EXTENDED_CUTOFF = { STANDARD_CUTOFF + 56.0 }; //Angstroms**2
	FArray1D_bool trim_res( MAX_RES()() );
	FArray1D_bool trim_loop( max_loop );
	bool check_res;
	FArray1D_int reverse_map( MAX_RES()() );

	assert( ! trim_on );

//car save variables  save best, copy position, mc reset
//car                 will restore to position then mc reset
	store_residue = total_residue;
	for ( int i = 1; i <= total_residue; ++i ) {
		store_secstruct(i) = secstruct(i);
		store_res(i) = res(i);
		store_resvariant(i) = res_variant(i);
		store_phi(i) = phi(i);
		store_psi(i) = psi(i);
		store_omega(i) = omega(i);
		store_name(i) = name(i);
		for ( int j = 1; j <= 5; ++j ) {
			for ( int k = 1; k <= 3; ++k ) Estore_position(k,j,i) = Eposition(k,j,i);
			for ( int j = 1; j <= max_atom; ++j ) {
				for ( int k = 1; k <= 3; ++k ) {
					store_fullcoord(k,j,i) = full_coord(k,j,i);
				}
			}
			for ( int k = 1; k <= 3; ++k ) store_centroid(k,i) = centroid(k,i);
		}
	}

	{ //scope: store total_domains and domain_end
		store_total_domains = total_domains;
		for ( int i = 1; i <= total_domains; ++i ) {
			store_domain_end(i) = domain_end(i);
		}
	} // scope end

//car loops.h variables
	store_num_loop = num_loop;
	for ( int i = 1; i <= store_num_loop; ++i ) {
		store_loop_begin(i) = loop_begin(i);
		store_loop_end(i) = loop_end(i);
		store_loop_depth(i) = loop_depth(i);
		int const loop_length = store_loop_end(i) - store_loop_begin(i) +1;
		for ( int k = 1, ke = store_loop_depth(i); k <= ke; ++k ) {
			for ( int j = 1; j <= loop_length; ++j ) {
				store_loop_phi(j,k,i) = loop_phi(j,k,i);
				store_loop_psi(j,k,i) = loop_psi(j,k,i);
				store_loop_omega(j,k,i) = loop_omega(j,k,i);
				store_loop_secstruct(j,k,i) = loop_secstruct(j,k,i);
			}
			store_loop_name(k,i) = loop_name(k,i);
		}
	}

//car fragments_ns.h
	for ( int k = 1, ke = n_frag_sizes(); k <= ke; ++k ) {
		for ( int j = 1, je = MAX_NEIGH(); j <= je; ++j ) {   // could use align_depth(k,i)
			for ( int i = 1; i <= total_residue; ++i ) {
				store_align_name(i,j,k) = align_name(i,j,k);
				store_align_chain(i,j,k) = align_chain(i,j,k);
				store_align_resseq(i,j,k) = align_resseq(i,j,k);
			}
		}
	}
	for ( int k = 1, ke = n_frag_sizes(); k <= ke; ++k ) {
		int const length = frag_sizes(k) - 1;
		for ( int kk = 0; kk <= length; ++kk ) {
			for ( int j = 1, je = MAX_NEIGH(); j <= je; ++j ) { // could use align_depth(k,i)
				for ( int i = 1; i <= total_residue; ++i ) {
					store_align_res_id(i,j,kk,k) = align_res_id(i,j,kk,k);
					store_align_phi(i,j,kk,k) = align_phi(i,j,kk,k);
					store_align_psi(i,j,kk,k) = align_psi(i,j,kk,k);
					store_align_omega(i,j,kk,k) = align_omega(i,j,kk,k);
					store_ss_type(i,j,kk,k) = ss_type(i,j,kk,k);
				}
			}
		}
	}
	store_align_depth = align_depth;

//car decide which residues to retain
	trim_res = false;

	update_cendist(total_residue,centroid);

	for ( int i = 1; i <= total_residue; ++i ) {
		check_res = false;
		float trim_cutoff = STANDARD_CUTOFF;

// include residues in active loops plus at least one adjacent residue and all residues they contact
		if ( allow_insert(i) ) check_res = true;
		if ( i > 1 ) {
			if ( allow_insert(i-1) ) check_res = true; // residue follows active loop (or within it)
		}
		if ( i < total_residue ) {
			if ( allow_insert(i+1) ) check_res = true; // residue precedes active loop (or within it)
		}

// handle extended loops differently
// include extended loop region in map, but not its contacts
		if ( allow_insert(i) && loop_is_extended(loop_map(i)) ) {
			check_res = false;
			trim_res(i) = true;
		}

// instead, if residue precedes an active extended loop or follows it
// include it and residues within the extended loop cutoff
	if ( ( i < total_residue  && loop_map(i+1) != 0 &&
	       i == loop_begin(loop_map(i+1)) - 1 && allow_insert(i+1) &&
               loop_is_extended(loop_map(i+1)) ) ||
             ( i > 1 && loop_map(i-1) != 0 && i == loop_end(loop_map(i-1)) + 1
	       && allow_insert(i-1) && loop_is_extended(loop_map(i-1)) ) ) {
           check_res = true;
           trim_cutoff = EXTENDED_CUTOFF;
        }

// find contacts for those residues with check_res true
// contacts to extended regions are ignored
		if ( check_res ) {
			trim_res(i) = true;
			for ( int j = 1; j <= total_residue; ++j ) {
				if (!loop_is_extended(loop_map(j)) && cendist(i,j) < trim_cutoff )
					trim_res(j) = true;
			}
		}

		store_allow_insert(i) = allow_insert(i);
	}

//car replace variables
	int ires = 0;
	for ( int i = 1; i <= store_residue; ++i ) {
		reverse_map(i) = 0;
		if ( trim_res(i) ) {
			++ires;
			trim_map(ires) = i;
			reverse_map(i) = ires;

			res(ires) = store_res(i);
			res_variant(ires) = store_resvariant(i);
			secstruct(ires) = store_secstruct(i);
			phi(ires) = store_phi(i);
			psi(ires) = store_psi(i);
			omega(ires) = store_omega(i);
			name(ires) = store_name(i);
			for ( int j = 1; j <= 5; ++j ) {
				for ( int k = 1; k <= 3; ++k ) Eposition(k,j,ires) = Estore_position(k,j,i);
			}
			for ( int j = 1; j <= max_atom; ++j ) {
				for ( int k = 1; k <= 3; ++k ) full_coord(k,j,ires) = store_fullcoord(k,j,i);
			}
			for ( int k = 1; k <= 3; ++k ) centroid(k,ires) = store_centroid(k,i);


			for ( int k = 1, ke = n_frag_sizes(); k <= ke; ++k ) {
				int length = frag_sizes(k) - 1;
				for ( int j = 1, je = MAX_NEIGH(); j <= je; ++j ) {//could use align_depth(k,ires)
					align_name(ires,j,k) = store_align_name(i,j,k);
					align_chain(ires,j,k) = store_align_chain(i,j,k);
					align_resseq(ires,j,k) = store_align_resseq(i,j,k);
				}
				for ( int kk = 0; kk <= length; ++kk ) {
					for ( int j = 1, je = MAX_NEIGH(); j <= je; ++j ) {//could use align_depth(k,ires)
						align_res_id(ires,j,kk,k) = store_align_res_id(i,j,kk,k);
						align_phi(ires,j,kk,k) = store_align_phi(i,j,kk,k);
						align_psi(ires,j,kk,k) = store_align_psi(i,j,kk,k);
						align_omega(ires,j,kk,k) = store_align_omega(i,j,kk,k);
						ss_type(ires,j,kk,k) = store_ss_type(i,j,kk,k);
					}
				}
				align_depth(ires,k) = store_align_depth(i,k);
			}
		}
	}
	total_residue = ires;
	update_sequence();

	{ //scope: update total_domain and domain_end after trim
		FArray1D_int old_domain_map( store_residue, 0 );
		int idomain = 1;
		for ( int i = 1; i <= store_residue; ++i ) {
			old_domain_map(i) = idomain;
			if ( i == domain_end(idomain) ) idomain++;
		}
		assert( store_total_domains == idomain-1 );

		FArray1D_int new_domain_map( total_residue, 0 );
		idomain = 1;
		for ( int i = 1; i <= total_residue; ++i ) {
			new_domain_map(i) = old_domain_map( trim_map(i) );
		}
		for ( int i = 1; i < total_residue; ++i ) {
			if ( new_domain_map(i) != new_domain_map(i+1) ) {
				domain_end(idomain) = i;
				idomain++;
			}
		}
		total_domains = idomain;
		domain_end(total_domains) = total_residue;
	} //scope end

//car update loops  -renumbering as appropriate
	num_loop = 0;
	for ( int i = 1; i <= store_num_loop; ++i ) {
		trim_loop(i) = false;
		int const jstart = store_loop_begin(i);
		int const jend = store_loop_end(i);
		for ( int j = jstart; j <= jend; ++j ) {
			if ( !allow_insert(j) ) goto L100;
			trim_loop(i) = true;
		}
L100:
		if ( trim_loop(i) ) {
			++num_loop;
			loop_begin(num_loop) = reverse_map(store_loop_begin(i));
			loop_end(num_loop) = reverse_map(store_loop_end(i));
			loop_depth(num_loop) = store_loop_depth(i);
			int const loop_length = loop_end(num_loop) - loop_begin(num_loop) + 1;
			for ( int k = 1, ke = loop_depth(num_loop); k <= ke; ++k ) {
				for ( int j = 1; j <= loop_length; ++j ) {
					loop_phi(j,k,num_loop) = store_loop_phi(j,k,i);
					loop_psi(j,k,num_loop) = store_loop_psi(j,k,i);
					loop_omega(j,k,num_loop) = store_loop_omega(j,k,i);
					loop_secstruct(j,k,num_loop) = store_loop_secstruct(j,k,i);
					loop_name(k,num_loop) = store_loop_name(k,i);
				}
			}
		}
	}

	setup_loop_splicing(false); // reset directions,segments
	reset_loops(); //set allow_insert[] for renumbering, and other stuff

	score_set_new_pose();
	score_enable_rotamer_trials(false); //chu make sure rotamer_trials is off
	scorefxn();                         //because best_energies not update yet
	score_enable_rotamer_trials(true);
	monte_carlo_reset();

	trim_on = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin trim_template_reset
///
/// @brief
/// car restore position, keeping modifications
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
trim_template_reset()
{

	using namespace files_paths;
	using namespace fragments;
	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace protein_maps;
	using namespace loops_ns::trim_hijack;


	assert( trim_on );

	int const max_atom = MAX_ATOM();

//car put modifed region into proper place:
	for ( int i = 1; i <= total_residue; ++i ) {
		int const ii = trim_map(i);
		store_phi(ii) = phi(i);
		store_psi(ii) = psi(i);
		store_omega(ii) = omega(i);
		store_res(ii) = res(i);
		store_resvariant(ii) = res_variant(i);
		store_secstruct(ii) = secstruct(i);
		store_name(ii) = name(i);
		for ( int j = 1; j <= 5; ++j ) {
			for ( int k = 1; k <= 3; ++k ) Estore_position(k,j,ii) = Eposition(k,j,i);
		}
		for ( int j = 1; j <= max_atom; ++j ) {
			for ( int k = 1; k <= 3; ++k ) store_fullcoord(k,j,ii) = full_coord(k,j,i);
		}
		for ( int k = 1; k <= 3; ++k ) store_centroid(k,ii) = centroid(k,i);
	}

//car put saved arrays back in current
	total_residue = store_residue;
	for ( int i = 1; i <= total_residue; ++i ) {
		secstruct(i) = store_secstruct(i);
		res(i) = store_res(i);
		res_variant(i) = store_resvariant(i);
		phi(i) = store_phi(i);
		psi(i) = store_psi(i);
		omega(i) = store_omega(i);
		name(i) = store_name(i);
		for ( int j = 1; j <= 5; ++j ) {
			for ( int k = 1; k <= 3; ++k ) Eposition(k,j,i) = Estore_position(k,j,i);
		}
		for ( int j = 1; j <= max_atom; ++j ) {
			for ( int k = 1; k <= 3; ++k ) full_coord(k,j,i) = store_fullcoord(k,j,i);
		}
		for ( int k = 1; k <= 3; ++k ) centroid(k,i) = store_centroid(k,i);
	}
	// chu: total_domains and domain_end
	total_domains = store_total_domains;
	for ( int i = 1; i <= total_domains; i++ ) {
		domain_end(i) = store_domain_end(i);
	}

//car fragments_ns.h
	for ( int k = 1, ke = n_frag_sizes(); k <= ke; ++k ) {
		for ( int j = 1, je = MAX_NEIGH(); j <= je; ++j ) {  // could use align_depth(k,i)
			for ( int i = 1; i <= total_residue; ++i ) {
				align_name(i,j,k) = store_align_name(i,j,k);
				align_chain(i,j,k) = store_align_chain(i,j,k);
				align_resseq(i,j,k) = store_align_resseq(i,j,k);
			}
		}
	}
	for ( int k = 1, ke = n_frag_sizes(); k <= ke; ++k ) {
		int const length = frag_sizes(k) - 1;
		for ( int kk = 0; kk <= length; ++kk ) {
			for ( int j = 1, je = MAX_NEIGH(); j <= je; ++j ) {  // could use align_depth(k,i)
				for ( int i = 1; i <= total_residue; ++i ) {
					align_res_id(i,j,kk,k) = store_align_res_id(i,j,kk,k);
					align_phi(i,j,kk,k) = store_align_phi(i,j,kk,k);
					align_psi(i,j,kk,k) = store_align_psi(i,j,kk,k);
					align_omega(i,j,kk,k) = store_align_omega(i,j,kk,k);
					ss_type(i,j,kk,k) = store_ss_type(i,j,kk,k);
				}
			}
		}
	}
	align_depth = store_align_depth;

//car loops.h variables
	num_loop = store_num_loop;
	for ( int i = 1; i <= num_loop; ++i ) {
		loop_begin(i) = store_loop_begin(i);
		loop_end(i) = store_loop_end(i);
		loop_depth(i) = store_loop_depth(i);
		int const loop_length = loop_end(i) - loop_begin(i) + 1;
		for ( int k = 1, ke = loop_depth(i); k <= ke; ++k ) {
			for ( int j = 1; j <= loop_length; ++j ) {
				loop_phi(j,k,i) = store_loop_phi(j,k,i);
				loop_psi(j,k,i) = store_loop_psi(j,k,i);
				loop_omega(j,k,i) = store_loop_omega(j,k,i);
				loop_secstruct(j,k,i) = store_loop_secstruct(j,k,i);
				loop_name(k,i) = store_loop_name(k,i);
			}
		}
	}

	update_sequence();
	setup_loop_splicing(false); // reset directions,segments
	reset_loops();

//car restore allow_insert and maps to original settings
	for ( int i = 1; i <= total_residue; ++i ) {
		allow_insert(i) = store_allow_insert(i);
	}
	reset_insert_map();

	score_set_new_pose();
	score_enable_rotamer_trials(false); //chu make sure rotamer_trials is off
	scorefxn();                         //because best_energies not updated yet.
	score_enable_rotamer_trials(true);
	monte_carlo_reset();

  trim_on = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_loops
///
/// @brief
/// reset arrays for loops
///
/// @detailed
/// sets allow_insert,loop_map
/// and insert_map
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_loops()
{

	using namespace loops_ns;
	using namespace misc;
	using namespace protein_maps;

	if ( num_loop < 1 ) return;
	for ( int j = 1; j <= total_residue; ++j ) {
		allow_insert(j) = false;
		allow_repack(j) = !fix_natsc;
		allow_rottrial(j) = !fix_natsc;
		loop_map(j) = 0; // no residues in loops
		}

	for ( int kk = 1; kk <= num_loop; ++kk ) {
		for ( int j = loop_begin(kk), je = loop_end(kk); j <= je; ++j ) {
			allow_insert(j) = true;
			allow_repack(j) = true;
			allow_rottrial(j) = true;
			loop_map(j) = kk; // loop number that residue j is in
		}
	}
	reset_insert_map();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin identify_loop
///
/// @brief
/// returns the endpoints of the loop containing res1 and res2
/// and the loop number.
///
/// @detailed
/// the endpoints are the first variable residue of the loop
///
/// @param  res1 - [in/out]? -
/// @param  res2 - [in/out]? -
/// @param  begin - [in/out]? -
/// @param  end - [in/out]? -
/// @param  loop_num - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
identify_loop(
	int & res1,
	int & res2,
	int & begin,
	int & end,
	int & loop_num
)
{

	using namespace loops_ns;
	using namespace misc;

	if ( !get_loop_flag() ) {
		begin = 1;
		end = total_residue;
		return;
	}

	if ( loop_map(res1) != loop_map(res2) ) {
		std::cout << "Unable to identify loop:" << std::endl;
		std::cout << res1 << " in loop " << loop_map(res1) << std::endl;
		std::cout << res2 << " in loop " << loop_map(res2) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( loop_map(res1) == 0 ) {
		std::cout << "WARNING! nonloop residues in identify loop " <<
		 res1 << ' ' << res2 << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	loop_num = loop_map(res1);
	begin = loop_begin(loop_num);
	end = loop_end(loop_num);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_loop_ends
///
/// @brief
/// returns the endpoints of loop number loop_num.
/// returns entire chain endpoints for loop_num < 1 or > total_residue
///
/// @detailed
///
/// @param  loop_num - [in/out]? -
/// @param  begin - [in/out]? -
/// @param  end - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_loop_ends(
	int loop_num,
	int & begin,
	int & end
)
{

	using namespace loops_ns;
	using namespace misc;

	if ( loop_num < 1 || loop_num > num_loop ) {
		begin = 1;
		end = total_residue;
		return;
	}

	begin = loop_begin(loop_num);
	end = loop_end(loop_num);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_loop_splicemsd
///
/// @brief calculate and return the splice msd, summed over all loops
///
/// @detailed see notes for loop_get_one_splicerms
///
/// @return
///
/// @global_read Eposition, overlap_coord, loop ends
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/28/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_loop_splicemsd()
{
	using namespace loops_ns;
	using namespace misc;

	int key,lap_pos;

	if ( num_loop == 0 ) return 0.0f;

	double msd = 0.0;
	for ( int iloop = 1; iloop <= num_loop; ++iloop ) {
		if ( loop_get_refold_dir(iloop) == n2c ) {
			key = 2;
			lap_pos = loop_end(iloop)+1;
		} else {
			key = 1;
			lap_pos = loop_begin(iloop) - 1;
		}

		if ( lap_pos > 1 && lap_pos < total_residue ) {
			for ( int i = 1; i <= 5; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					double const dif = overlap_coord(j,i,iloop,key) - Eposition(j,i,lap_pos);
					msd += dif * dif;
				}
			}
		}
	}
	return msd / 5.0; // mean
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_get_one_splicerms
///
/// @brief
///    calculate and return the splice rms for one loop
///
/// @detailed
///      splice rms is the distance between open terminus of the loop
///      and the takeoff residue it is supposed to superpose upon.
///      distance is calculted as an rms average over the five
///      centroid-mode atoms
///
/// @param[in]   iloop - in -
/// @param[out]   rms - out -
///
/// @global_read Eposition,
///             loop variables: overlap_coord, loop_begin, loop_end
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/27/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_get_one_splicerms(
	int iloop,
	float & rms
)
{
	using namespace loops_ns;
	using namespace misc;

	int key,lap_pos;

	rms = 0.0;
	if ( num_loop == 0 ) return;
	if ( loop_get_refold_dir(iloop) == n2c ) {
		key = 2;
		lap_pos = loop_end(iloop) + 1;
	} else {
		key = 1;
		lap_pos = loop_begin(iloop) - 1;
	}

	if ( lap_pos > 1 && lap_pos < total_residue ) {
		double rms_sum = 0.0;
		for ( int i = 1; i <= 5; ++i ) {
			for ( int j = 1; j <= 3; ++j ) {
				double const dif = overlap_coord(j,i,iloop,key) - Eposition(j,i,lap_pos);
				rms_sum += dif * dif;
			}
		}
		rms_sum /= 5.0; // mean
		rms_sum = std::sqrt(rms_sum); // root
		rms = rms_sum;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_get_best_splicemsd
///
/// @brief splice rms of the best_position
///
/// @detailed see notes for loop_get_one_splicerms
///
/// @param  iloop - [in/out]? -
/// @param  msd - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/27/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_get_best_splicemsd(
	int iloop,
	float & msd
)
{
	using namespace loops_ns;
	using namespace misc;

	int key,lap_pos;

	msd = 0.0;
	if ( num_loop == 0 ) return;
	if ( loop_get_refold_dir(iloop) == n2c ) {
		key = 2;
		lap_pos = loop_end(iloop)+1;
	} else {
		key = 1;
		lap_pos = loop_begin(iloop)-1;
	}

	if ( lap_pos > 1 && lap_pos < total_residue ) {
		double msd_sum = 0.0;
		for ( int i = 1; i <= 5; ++i ) {
			for ( int j = 1; j <= 3; ++j ) {
				double const dif = best_overlap_coord(j,i,iloop,key) - Ebest_position(j,i,lap_pos);
				msd_sum += dif * dif;
			}
		}
		msd_sum /= 5.0; // mean
		msd = msd_sum;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_get_all_splicerms
///
/// @brief splice rms for each loops
///
/// @detailed see notes for loop_get_one_splicerms
///
/// @param[out]   rms - out - array of splice rms values
/// @param[in]   num - in - number of loops
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/27/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_get_all_splicerms(
	FArray1Da_float rms,
	int & num
)
{
	using namespace loops_ns;
	using namespace misc;

	rms.dimension( star );

	int key,lap_pos;

	num = num_loop;
	for ( int iloop = 1; iloop <= num_loop; ++iloop ) {
		if ( loop_get_refold_dir(iloop) == n2c ) {
			key = 2;
			lap_pos = loop_end(iloop) + 1;
		} else {
			key = 1;
			lap_pos = loop_begin(iloop) - 1;
		}

		rms(iloop) = 0.0;
		if ( lap_pos > 1 && lap_pos < total_residue ) {
			double rms_sum = 0.0;
			for ( int i = 1; i <= 5; ++i ) {
				for ( int j = 1; j <= 3; ++j ) {
					double const dif = overlap_coord(j,i,iloop,key) - Eposition(j,i,lap_pos);
					rms_sum += dif * dif;
				}
			}
			rms_sum /= 5.0; // mean
			rms_sum = std::sqrt(rms_sum); // root
			rms(iloop) = rms_sum;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_retrieve_best_pose
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_retrieve_best_pose()
{
	using namespace loops_ns;

	if ( !get_loop_flag() ) return;
	for ( int i = 1; i <= num_loop; ++i ) {
		for ( int k = 1; k <= 5; ++k ) {
			for ( int j = 1; j <= 3; ++j ) {
				overlap_coord(j,k,i,1) = best_overlap_coord(j,k,i,1);
				overlap_coord(j,k,i,2) = best_overlap_coord(j,k,i,2);
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_retrieve_low_pose
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_retrieve_low_pose()
{
	using namespace loops_ns;

	if ( !get_loop_flag() ) return;
	for ( int i = 1; i <= num_loop; ++i ) {
		for ( int k = 1; k <= 5; ++k ) {
			for ( int j = 1; j <= 3; ++j ) {
				overlap_coord(j,k,i,1) = low_overlap_coord(j,k,i,1);
				overlap_coord(j,k,i,2) = low_overlap_coord(j,k,i,2);
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_accept_best_pose
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_accept_best_pose()
{
	using namespace loops_ns;

	if ( !get_loop_flag() ) return;
	for ( int i = 1; i <= num_loop; ++i ) {
		for ( int k = 1; k <= 5; ++k ) {
			for ( int j = 1; j <= 3; ++j ) {
				best_overlap_coord(j,k,i,1) = overlap_coord(j,k,i,1);
				best_overlap_coord(j,k,i,2) = overlap_coord(j,k,i,2);
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_accept_low_pose
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_accept_low_pose()
{
	using namespace loops_ns;

	if ( !get_loop_flag() ) return;
	for ( int i = 1; i <= num_loop; ++i ) {
		for ( int k = 1; k <= 5; ++k ) {
			for ( int j = 1; j <= 3; ++j ) {
				low_overlap_coord(j,k,i,1) = overlap_coord(j,k,i,1);
				low_overlap_coord(j,k,i,2) = overlap_coord(j,k,i,2);
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_segment_overlap
///
/// @brief
///
/// @detailed
///
/// @param  seg_num - [in/out]? -
/// @param  fold_begin - [in/out]? -
/// @param  fold_end - [in/out]? -
/// @param  seg_begin - [in/out]? -
/// @param  seg_end - [in/out]? -
/// @param  frag_begin - [in/out]? -
/// @param  frag_end - [in/out]? -
/// @param  dir - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_segment_overlap(
	int const seg_num,
	int const fold_begin,
	int const fold_end,
	int const seg_begin,
	int const seg_end,
	int const dir
)
{
	using namespace loops_ns;
	using namespace misc;
	using namespace param;
	using namespace termini_ns;

	if ( seg_begin == 1 && seg_end == total_residue ) return; // no breaks
	if ( seg_num < 1 || seg_num > num_loop ) return; // fixed segment
	if ( fold_begin > fold_end ) return; // no change in this segment
	if ( loop_begin(seg_num) <= 1 ) return; // terminal loops
	if ( loop_end(seg_num) >= total_residue ) return;

//car local
	int key;
	int lap_pos;

	static FArray3D_float overlap( 3, MAX_POS, MAX_RES() );
	static FArray2D_float overlap_cen( 3, MAX_RES() );
	static FArray3D_float overlap_full( 3, MAX_ATOM(), MAX_RES() );

	if ( dir == n2c ) {
		key = 2;
		lap_pos = seg_end + 1;
//car fold lap_pos using position as source!!
//car this could be done with lower level function?

//car fold res 2 & 3 of a 3 residue segment; must start from seg_end -1
//car because of the way the refold stub is defined: c(seg_end-1),
//car n(seg_end), and c(seg_end).
//car  inside refold_update_coords
//car   lap_pos = res3, seg_end = res2, seg_end -1 = res1

		int slice = seg_end - 1;
		int fold_start = 2;
		int fold_end = 3;
		if (seg_end == 1) {
			//car fold 2 residues of a 2 residue N-terminal segment:
			slice = seg_end;
			fold_start = 1;
			fold_end = 2;
		}
		refold_update_coords(dir,fold_start,fold_end,fold_end,slice,phi(slice),
		 psi(slice),omega(slice),res(slice),res_variant(slice),
		 is_N_terminus(slice),is_C_terminus(slice),
		 Eposition(1,1,slice),overlap(1,1,slice),centroid(1,slice),
		 overlap_cen(1,slice),false,full_coord(1,1,slice),overlap_full(1,1,slice));
	} else {

		key = 1;
		lap_pos = seg_begin - 1;
//car fold lap_pos using position as source!!

//car fold res1-2 of 3 res segment starting at lap_pos position as source
//car segment must be 3 residues because of definition of refold stub:
//car  c(seg_begin),n(seg_begin-1),ca(seg_begin-1)
//car  inside refold_update_coords
//car   lap_pos = res1, seg_begin = res2, seg_begin+1 = res3

		int fold_start = 1;
		int fold_end = 2;
		int seg_length = 3;
		if (seg_begin == total_residue) {
			seg_length = 2;  //fold c-term 2 residues of 2 residue segment.
		}
		refold_update_coords(dir,fold_start,fold_end,seg_length,lap_pos,phi(lap_pos),
		 psi(lap_pos),omega(lap_pos),res(lap_pos),res_variant(lap_pos),
		 is_N_terminus(lap_pos),is_C_terminus(lap_pos),
     Eposition(1,1,lap_pos),overlap(1,1,lap_pos),centroid(1,lap_pos),
     overlap_cen(1,lap_pos),false,full_coord(1,1,lap_pos),
     overlap_full(1,1,lap_pos));

	}
	loop_save_overlap_coord(seg_num,key,overlap(1,1,lap_pos));

}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_halfsplice_reverse
///
/// @brief
/// calculates the rotation and offset between pos1 and pos2, applies it
/// to nres residues beginning at position3
///
/// @detailed
/// rotation and offset to move pos1 onto pos2, applies same rot/off to
/// pos3
/// intended to switch a loop with a Cterminal discontinuity to one
/// with an N-terminal discontinuity and vice versa.
///
/// @param  pos1 - [in/out]? -
/// @param  pos2 - [in/out]? -
/// @param  pos3 - [in/out]? -
/// @param  nres - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_halfsplice_reverse(
	FArray2Da_float pos1,
	FArray2Da_float pos2,
	FArray3Da_float pos3,
	int nres
)
{
	int const natoms = { 5 };
	pos1.dimension( 3, natoms );
	pos2.dimension( 3, natoms );
	pos3.dimension( 3, natoms, star );

	FArray2D_double xx( 3, 5 );
	FArray2D_double yy( 3, 5 );
	FArray1D_double WW( 30 );
	double sigma3;
	FArray1D_double xx_offset( 3 );
	FArray1D_double yy_offset( 3 );
	FArray2D_double UU( 3, 3 );

	for ( int k = 1; k <= 3; ++k ) {
		for ( int j = 1; j <= natoms; ++j ) {
			xx(k,j) = pos1(k,j);
			yy(k,j) = pos2(k,j);
		}
	}

	for ( int i = 1; i <= natoms; ++i ) {
		WW(i) = 1;
	}
	findUU_trans(xx,yy,WW,natoms,UU,sigma3,xx_offset,yy_offset);

	UU_rotate(pos3,nres*natoms,xx_offset,yy_offset,UU);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_loop_library
///
/// @brief
///
/// @detailed
///
/// @param[in]   filename - in  - loop file name
/// @param[in]   funit    - in  - input file stream
/// @param[out]   exist    - out - notification that library was read successfully
///
/// @global_read namespace param
///
/// @global_write namespace loops
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 3/12/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_loop_library(
	std::string const & filename,
	utility::io::izstream & funit,
	bool & exist
)
{
	using namespace loops_ns;
	using namespace param;


	funit.clear();
	funit.open( filename );
	std::cout << "Looking for loop file: " << funit.filename() << std::endl;
	if ( !funit ) {
		std::cout << "Loop file not found!" << std::endl;
		exist = false;
		funit.clear();
		return;
	}

	int const max_line = { 103 + ( 27 * max_loop_size ) };
	num_loop = 0;
	exist = true;
	int isize;
	int begin,end; // frag size and location
	int last_begin = 0;
	int depth;
	std::string line;
	while ( ! funit.eof() ) {
		funit.clear();
		funit >> bite( 3, isize ) >> skip( 1 ) >> bite( 4, begin ) >> skip( 1 ) >>
		 bite( 4, end ) >> skip( 1 ) >> bite( line ) >> skip;
		if ( ! funit.eof() ) {
			isize = end - begin + 1;
			if ( isize > max_loop_size ) {
				std::cout << "WARNING!! max_loop_size exceeded" << std::endl;
				std::cout << "Increase max_loop_size in loops_ns.cc to " << isize <<
				 std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			if ( begin != last_begin ) { // new loop
				if ( num_loop == max_loop ) {
					std::cout << "WARNING: max_loop exceeded." << std::endl;
					std::cout << "increase max_loop in loops_ns.cc to " << num_loop + 1 <<
					 std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				++num_loop; // new loop
				loop_begin(num_loop) = begin;
				loop_end(num_loop) = end;
				loop_depth(num_loop) = 0;
				last_begin = begin;
			}
			++loop_depth(num_loop);
			depth = loop_depth(num_loop); // local copy

			if ( depth > max_loop_depth ) {
				std::cout << "increase max_loop_depth in loops_ns.cc" << std::endl;
				std::cout << SS( max_loop_depth ) << SS( depth ) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			line.resize( max_line, ' ' ); // Pad line with blanks
			for ( int i = 1; i <= isize; ++i ) {
				loop_secstruct(i,depth,num_loop) = line[i-1];
			}
			line.erase( 0, isize + 98 );
			std::istringstream line_stream( line );
			line_stream >> bite( 4, loop_name(depth,num_loop) );
			line.erase( 0, 5 );
			line_stream.clear();
			line_stream.str( line );
			line_stream.seekg( std::ios_base::beg );
			for ( int i = 1; i <= isize; ++i ) {
				line_stream >> bite( 7, loop_phi( i, depth,num_loop ) ) >> skip( 1 ) >>
				 bite( 7, loop_psi( i, depth, num_loop ) ) >> skip( 1 ) >>
				 bite( 7, loop_omega( i, depth, num_loop ) ) >> skip( 3 );
			}
			if ( loop_phi(1,depth,num_loop) == 0.0 &&
			 loop_psi(1,depth,num_loop) == 0.0 &&
			 loop_omega(1,depth,num_loop) == 0.0 ) { // not a real conformation
				for ( int i = 1; i <= isize; ++i ) { // include extended conformation
					loop_phi(i,1,num_loop) = init_phi; // extended conformation
					loop_psi(i,1,num_loop) = init_psi;
					loop_omega(i,1,num_loop) = init_omega;
					loop_secstruct(i,1,num_loop) = 'L';
				}
			}
		}
	}

//car check that loops don't overlap

	int i, j;
	for ( i = 1; i <= num_loop; ++i ) {
		std::cout << "loop number " << I( 2, i ) << ": " <<
		 I( 3, loop_begin(i) ) << " -" << I( 3, loop_end(i) ) <<
		 " length: " << I( 4, loop_end(i)-loop_begin(i)+1 ) <<
		 " depth: " << I( 4, loop_depth(i) ) << std::endl;
		for ( j = i+1; j <= num_loop; ++j ) {
			if ( loop_begin(j) == loop_begin(i) ) goto L200;
			if ( loop_end(j) == loop_begin(i) ) goto L200;
			if ( loop_begin(j) == loop_end(i) ) goto L200;
			if ( loop_end(j) == loop_end(i) ) goto L200;
			if ( loop_begin(j) > loop_begin(i) && loop_begin(j) < loop_end(i) ) goto L200;
			if ( loop_end(j) > loop_begin(i) && loop_end(j) < loop_end(i) ) goto L200;
			if ( loop_begin(j) < loop_begin(i) ) std::cout <<
			 "WARNING: loops not in sequential order. " <<
			 "Template only score will be incorrect!" << std::endl;
		}
	}

	funit.close();
	funit.clear();

	return;

L200:
	std::cout << "There is a problem in the loop library." << std::endl;
	std::cout << "Overlapping loops are defined:" << std::endl;
	std::cout << "loop " << i << " : " << loop_begin(i) << " - " << loop_end(i) << std::endl;
	std::cout << "loop " << j << " : " << loop_begin(j) << " - " << loop_end(j) << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin write_loop_library
///
/// @brief write phi/psi angles defining a single loop to a loop library file
///
/// @detailed also includes SS, various scores
///
/// @param[in]   funit - in
/// @param[in]   num - in
/// @param[in]   depth - in
/// @param[in]   score1 - in
/// @param[in]   score2 - in
///
/// @global_read loop torsion angles
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/28/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
write_loop_library(
	std::ostream & funit,
	int num,
	int depth,
	float score1,
	float score2,
	bool const knots,
	float score3
)
{
//car write a single loop to an open library file

	using namespace loops_ns;

	std::string line;

	int length = loop_end(num) - loop_begin(num) + 1;
	std::ostringstream line_stream;
	line_stream << I( 3, length ) << ' '
					<<	I( 4, loop_begin(num) ) << ' '
					<< I( 4, loop_end(num) ) << ' ';
	line = line_stream.str();

	for ( int ii = 1; ii <= length; ++ii ) {
		line += loop_secstruct( ii, depth, num );
	}

	line_stream.str( "" ); // Clear the stream
	int loop_dir = loop_get_refold_dir(num);
	line_stream << ' ' << F( 9, 4, score1 )
					<< ' ' << F( 9, 4, score2 )
					<< ' ' << L( 1, knots )
					<< ' ' << I( 2, loop_dir )
					<< ' ' << I( 9, num )
					<< ' ' << I( 9, depth )
					<< ' ' << I( 9, num*1000 + depth )
					<< ' ' << F( 9, 4, score3 )
					<< space( 33 ) << loop_name(depth,num) << ' ';
	line += line_stream.str();

	line_stream.str( "" ); // Clear the stream
	for ( int ii = 1; ii <= length; ++ii ) {
		line_stream << F( 7, 2, loop_phi( ii, depth, num ) ) << ' '
						<< F( 7, 2, loop_psi( ii, depth, num ) ) << ' '
						<< F( 7, 2, loop_omega( ii, depth, num ) ) << "   ";
		line += line_stream.str();
		line_stream.str( "" ); // Clear the stream
	}

	funit << line << '\n'; //std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_insert
///
/// @brief
/// inserts the jth loop of loop number loop_no into supplied phi-psi arrays
/// returns an error if j is greater than the number of available loops
///
/// @detailed
///
/// @param  loop_no - [in/out]? -
/// @param  j - [in/out]? -
/// @param  err - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_insert(
	int const loop_no,
	int const j,
	bool & err
)
{

	using namespace loops_ns;
	using namespace misc;

	err = true;
	int i = loop_no;
	if ( loop_depth(i) < j ) return;
	int kk = 1;
	for ( int k = loop_begin(i), ke = loop_end(i); k <= ke; ++k ) {
		phi(k) = loop_phi(kk,j,i);
		psi(k) = loop_psi(kk,j,i);
		omega(k) = loop_omega(kk,j,i);
		secstruct(k) = loop_secstruct(kk,j,i);
		name(k) = loop_name(j,i);
		++kk;
	}
	err = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin select_starting_loops
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
select_starting_loops()
{
	using namespace loops_ns;
	using namespace param;

//car local
	int last_res;
	bool err;
	FArray1D_bool allow_insert( MAX_RES()() );

	last_res = loop_end(num_loop);
	retrieve_allow_insert(allow_insert,last_res);
	for ( int i = 1; i <= num_loop; ++i ) {
		for ( int j = loop_begin(i), je = loop_end(i); j <= je; ++j ) {
		 // if any residue fixed, leave alone
			if ( !allow_insert(j) ) goto L100;
		}
		{ // Scope
			int const j = static_cast< int >( ran3() * loop_depth(i) ) + 1;
			 // random loop conf.
			loop_insert(i,j,err);
			std::cout << "loop number: " << SS( i ) <<
			 " starting conformation: " << SS( j ) << std::endl;
		}

L100:;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_secstruct
///
/// @brief
/// this is a messy fix to secondary structure rules for fragment insertion
///
/// @detailed
/// sometimes loop insertions cause disallowed ss patterns, which causes
/// problems for choose_fragment. The fix here is just to modify the ss
/// so that the violation isn't in the ss.
///
/// @param  ss - [in/out]? -
/// @param  nres - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// note that the ss may not be accurate
/// also a mc-reset occurs.
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
check_secstruct(
	FArray1Da_char ss,
	int & nres
)
{
	ss.dimension( nres );

	char current = ss(1);
	int length = 1;
	for ( int i = 2; i <= nres; ++i ) {
		if ( ss(i) == current ) {
			++length;
		} else {
			if ( current == 'H' ) {
				if ( length < 3 ) {
					for ( int j = i-1, e = i - length; j >= e; --j ) {
						ss(j) = 'L';
					}
				}
			} else if ( current == 'E' ) {
				if ( length < 2 ) {
					for ( int j = i-1, e = i - length; j >= e; --j ) {
						ss(j) = 'L';
					}
				}
			}
			current = ss(i);
			length = 1;
		}
	}
	std::cout << "Secondary structure after check:" << std::endl;
	for ( int j = 1; j <= nres; ++j ) {
		std::cout << ss(j);
	} std::cout << std::endl;

	save_status_info("ss_check",0,0);
	monte_carlo_reset();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fix_terminal_loops
///
/// @brief
/// terminal loops set to yes_no, others not affected
///
/// @detailed
///
/// @param  yes_no - [in/out]? -
/// @param  exist - [in/out]? - exist true if any variable residues exist
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fix_terminal_loops(
	bool yes_no,
	bool & exist
)
{

	using namespace loops_ns;
	using namespace misc;
	using namespace protein_maps;

	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_begin(i) == 1 || loop_end(i) == total_residue ) {
			for ( int k = loop_begin(i), ke = loop_end(i); k <= ke; ++k ) {
				allow_insert(k) = !yes_no;
			}
		}
	}

	exist = reset_insert_map();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_allow_insert_by_looplength
///
/// @brief
/// loop <= length allow_insert set to yes_no
/// loop > length allow_insert set to ! yes_no
///
/// @detailed
/// includes terminal residues
/// exist returns true if anything is set true
///
/// @param  length - [in/out]? -
/// @param  yes_no - [in/out]? -
/// @param  exist - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_allow_insert_by_looplength(
	int length,
	bool yes_no,
	bool & exist
)
{

	using namespace loops_ns;
	using namespace protein_maps;

	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_end(i) - loop_begin(i) + 1 <= length ) {
			for ( int k = loop_begin(i), ke = loop_end(i); k <= ke; ++k ) {
				allow_insert(k) = yes_no;
			}
		} else {
			for ( int k = loop_begin(i), ke = loop_end(i); k <= ke; ++k ) {
				allow_insert(k) = !yes_no;
			}
		}
	}
	exist = reset_insert_map();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_allow_insert_by_looprms
///
/// @brief
///splicerms <= rms allow_insert set to yes_no
///splicerms > length allow_insertset to ! yes_no
///
/// @detailed
/// includes terminal residues
/// exist returns true if anything is set true
///
/// @param  rms - [in/out]? -
/// @param  yes_no - [in/out]? -
/// @param  exist - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_allow_insert_by_looprms(
	float rms,
	bool yes_no,
	bool & exist
)
{

	using namespace loops_ns;
	using namespace protein_maps;

	FArray1D_float splicerms( max_loop );
	int num;

	loop_get_all_splicerms(splicerms,num);

	for ( int i = 1; i <= num_loop; ++i ) {
		if ( splicerms(i) <= rms ) {
			for ( int k = loop_begin(i), ke = loop_end(i); k <= ke; ++k ) {
				allow_insert(k) = yes_no;
			}
		} else {
			for ( int k = loop_begin(i), ke = loop_end(i); k <= ke; ++k ) {
				allow_insert(k) = !yes_no;
			}
		}
	}
	exist = reset_insert_map();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loopnum_is_valid
///
/// @brief returns true if loopnum is a valid loop number
///
/// @detailed
///
/// @param[in]   loop_num - in - loop number to check
///
/// @return    true if loop_num between 1 and num_loop
///
/// @global_read
///  num_loops loops.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 9/4/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
loopnum_is_valid( int loop_num )
{
	using namespace loops_ns;

	bool loopnum_is_valid = true; // Return value
	if ( loop_num < 1 ) loopnum_is_valid = false;
	if ( loop_num > num_loop ) loopnum_is_valid = false;
	return loopnum_is_valid;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_is_extended
///
/// @brief returns true if loop is in an extended conformation
///
/// @detailed
///
/// @param[in]   loop_num - in - loop number to check
///
/// @return    true if phi/psi/omega set to init values (ie extended)
///
/// @global_read
///  phi,psi,omega        misc.h
///  loop_begin,loop_end  loops.h
///  init_phi,init_psi,init_omega  param.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/29/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
loop_is_extended( int loop_num )
{
	using namespace loops_ns;
	using namespace misc;
	using namespace param;

	if ( !loopnum_is_valid(loop_num) ) return false;
	for ( int i = loop_begin(loop_num), ie = loop_end(loop_num); i <= ie; ++i ) {
		if ( phi(i) != init_phi ) return false;
		if ( psi(i) != init_psi ) return false;
		if ( omega(i) != init_omega ) return false;
	}
	return true;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin loop_is_terminal
///
/// @brief returns true if loop is a terminal extension
///
/// @detailed
///
/// @param[in]   loop_num - in - loop number to check
///
/// @return    true if loop_begin==1 or loop_end==total_residue
///
/// @global_read
///  total_residue         misc.h
///  loop_begin,loop_end  loops.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 8/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
loop_is_terminal( int loop_num )
{
	using namespace loops_ns;
	using namespace misc;

	if ( !loopnum_is_valid(loop_num) ) return false;
	if ( loop_begin(loop_num) == 1 ) return true;
	if ( loop_end(loop_num) == total_residue ) return true;
	return false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_allow_insert_by_extended
///
/// @brief
/// if loop in extended conformation, sets to  yes_no, other loops set to !yes_no
///
/// @detailed
///
/// @param  yes_no - [in/out]? -
/// @param  exist - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_allow_insert_by_extended(
	bool yes_no,
	bool & exist
)
{

	using namespace loops_ns;
	using namespace protein_maps;

	bool setting;

	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_is_extended(i) ) {
			setting = yes_no;
		} else {
			setting = !yes_no;
		}
		for ( int k = loop_begin(i), ke = loop_end(i); k <= ke; ++k ) {
			allow_insert(k) = setting;
		}
	}
	exist = reset_insert_map();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_allow_insert_by_loopnum
///
/// @brief
/// loop loop_num set to yes_no, others set to ! yes_no
///
/// @detailed
/// exist true if any variable residues exist
///
/// @param  loop_num - [in/out]? -
/// @param  yes_no - [in/out]? -
/// @param  exist - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_allow_insert_by_loopnum(
	int loop_num,
	bool yes_no,
	bool & exist
)
{

	using namespace loops_ns;
	using namespace protein_maps;

	bool setting;

	for ( int j = 1; j <= num_loop; ++j ) {
		if ( j == loop_num ) {
			setting = yes_no;
		} else {
			setting = !yes_no;
		}

		for ( int i = loop_begin(j), ie = loop_end(j); i <= ie; ++i ) {
			allow_insert(i) = setting;
		}
	}

	exist = reset_insert_map();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fix_lowrms_loops
///
/// @brief
/// sets allow_insert to !yes_no or loops with splice rms <= rms.
/// other loops not affected!
///
/// @detailed
/// exist true if any variable residues exist
///
/// @param  rms - [in/out]? -
/// @param  yes_no - [in/out]? -
/// @param  exist - [in/out]? - if loops >rms exist
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fix_lowrms_loops(
	float rms,
	bool yes_no,
	bool & exist // if loops >rms exist
)
{

	using namespace loops_ns;
	using namespace protein_maps;
	using namespace runlevel_ns;

	FArray1D_float splicerms( max_loop );
	int num;

	loop_get_all_splicerms(splicerms,num);

	for ( int i = 1; i <= num_loop; ++i ) {
		if ( splicerms(i) <= rms ) {
			for ( int j = loop_begin(i), je = loop_end(i); j <= je; ++j ) {
				allow_insert(j) = !yes_no;
			}
			if ( runlevel >= gush )  std::cout << "loop num " << i << " set to " <<
			 L( yes_no ) << std::endl;
		}
	}

	exist = reset_insert_map();
}


void
loops_reorder_initializer( FArray1D_int & reorder )
{
	reorder( 1 ) = 1;
	reorder( 2 ) = 2;
	reorder( 3 ) = 4;
	reorder( 4 ) = 5;
	reorder( 5 ) = 3;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_global_rms
///
/// @brief calculate rms over loops while aligned to template (ie default alignment)
///
/// @detailed
///
/// @param[in]   Eposition - in -
/// @param[out]   rms - out -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/28/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_global_rms(
	int const loop_num,
	FArray3Da_float Eposition,
	float & rms
)
{
	using namespace loops_ns;
	using namespace native;
	using namespace param;

	rms = 0.0;
	if (! native_exists) return;

	Eposition.dimension( 3, MAX_POS, MAX_RES() );

//car local
	static FArray1D_int const reorder( 5, loops_reorder_initializer );

	int npoints = 0;
	int jstart = 1;
	int jend = num_loop;
	if ( loopnum_is_valid(loop_num) ) {
		jstart = loop_num;
		jend = loop_num;
	}
	for ( int j = jstart; j <= jend; ++j ) {
		for ( int i = loop_begin(j), ie = loop_end(j); i <= ie; ++i ) {
			for ( int m = 1; m <= 4; ++m ) { // N,CA,C,O
				++npoints;
				for ( int k = 1; k <= 3; ++k ) {
					double const dif = Eposition(k,reorder(m),i) - native_coord(k,m,i);
					rms += dif * dif;
				}
			}
		}
	}
	if ( npoints == 0 ) return;
	rms = std::sqrt(rms/npoints);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin loop_global_CArms
///
/// @brief
/// calculate rms over loops while aligned to template (ie default alignment)
///
/// @detailed
///
/// @param  Eposition - [in/out]? -
/// @param  rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks currently not used (4/28/4)
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_global_CArms(
	int const loop_num,
	FArray3Da_float Eposition,
	float & rms
)
{
	using namespace loops_ns;
	using namespace native;
	using namespace param;

	rms = 0.0;
	if (! native_exists) return;

	Eposition.dimension( 3, MAX_POS, MAX_RES() );

	int npoints = 0;
	int jstart = 1;
	int jend = num_loop;
	if ( loopnum_is_valid(loop_num) ) {
	  jstart = loop_num;
		jend = loop_num;
	}
	for ( int j = jstart; j <= jend; ++j ) {
		for ( int i = loop_begin(j), ie = loop_end(j); i <= ie; ++i ) {
			++npoints;
			for ( int k = 1; k <= 3; ++k ) {
				double const dif = Eposition(k,2,i) - native_ca(k,i);
				rms += dif * dif;
			}
		}
	}
	if ( npoints == 0 ) return;
	rms = std::sqrt(rms/npoints);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_local_rms
///
/// @brief align over loops regions only (one at a time) and calculate rms
///      (average per loop)
///
/// @detailed
///
/// @param[in]   p2 - in - coordinate array,position format
/// @param[out]   avg_rms - out -
///
/// @global_read loop bookkeeping
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/28/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_local_rms(
	int const loop_num,
	FArray3Da_float p2, // coordinate array,position format
	float & avg_rms
)
{
	using namespace loops_ns;
	using namespace native;
	using namespace param;


	avg_rms = 0.0;
	if (! native_exists) return;

	p2.dimension( 3, MAX_POS, star );

//car align over loops regions only and calculate rms

//car local
	FArray2D_double p1a( 3, MAX_RES()() * MAX_POS );
	FArray2D_double p2a( 3, MAX_RES()() * MAX_POS);
	FArray1D_double ww( MAX_RES()() * MAX_POS);
	FArray2D_double UU( 3, 3 );
	float fast_rms;
	double ctx;
	static FArray1D_int const reorder( 5, loops_reorder_initializer );

	int num = 0;
	if ( !get_loop_flag() ) return;
	int kkstart = 1;
	int kkend = num_loop;
	if ( loopnum_is_valid(loop_num) ) {
	  kkstart = loop_num;
		kkend = loop_num;
	}
	for ( int kk = kkstart; kk <= kkend; ++kk ) {
		int nPoints = 0;
		for ( int k = loop_begin(kk), ke = loop_end(kk); k <= ke; ++k ) {
			for ( int j = 1; j <= 4; ++j ) { // N,CA,C,O
				++nPoints;
				ww(nPoints) = 1.0;
				for ( int i = 1; i <= 3; ++i ) {
					p1a(i,nPoints) = native_coord(i,j,k); // single to double
					p2a(i,nPoints) = p2(i,reorder(j),k); // reorder atoms
				}
			}
		}
		if ( nPoints != 0 ) {
			findUU(p1a,p2a,ww,nPoints,UU,ctx);
			calc_rms_fast(fast_rms,p1a,p2a,ww,nPoints,ctx);
			avg_rms += fast_rms;
			++num;
		}

	}
	if ( num == 0 ) return;
	avg_rms /= num;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_local_CArms
///
/// @brief
/// align over loops regions only and calculate rms
///
/// @detailed
///
/// @param  p2 - [in/out]? - coordinate array,position format
/// @param  avg_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks currently not used (4/28/4)
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_local_CArms(
	int const loop_num,
	FArray3Da_float p2, // coordinate array,position format
	float & avg_rms
)
{
	using namespace loops_ns;
	using namespace native;
	using namespace param;

	avg_rms = 0.0;
	if (! native_exists) return;

	p2.dimension( 3, MAX_POS, star );

//car local
	FArray2D_double p1a( 3, MAX_RES()() );
	FArray2D_double p2a( 3, MAX_RES()() );
	FArray1D_double ww( MAX_RES()() );
	FArray2D_double UU( 3, 3 );
	float fast_rms;
	double ctx;

	int num = 0;
	if ( !get_loop_flag() ) return;
	int kkstart = 1;
	int kkend = num_loop;
	if ( loopnum_is_valid(loop_num) ) {
	  kkstart = loop_num;
		kkend = loop_num;
	}
	for ( int kk = kkstart; kk <= kkend; ++kk ) {
		int nPoints = 0;
		for ( int k = loop_begin(kk), ke = loop_end(kk); k <= ke; ++k ) {
			++nPoints;
			ww(nPoints) = 1.0;
			for ( int i = 1; i <= 3; ++i ) {
				p1a(i,nPoints) = native_ca(i,k); // single to double
				p2a(i,nPoints) = p2(i,2,k); // pick out c_alphas
			}
		}
		if ( nPoints != 0 ) {
			findUU(p1a,p2a,ww,nPoints,UU,ctx);
			calc_rms_fast(fast_rms,p1a,p2a,ww,nPoints,ctx);
			avg_rms += fast_rms;
			++num;
		}

	}
	if ( num == 0 ) return;
	avg_rms /= num;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_gaps
///
/// @brief calculate average distance in angstroms between the end of the loop
///       and the adjacent takeoff residue.
///
/// @detailed tracked separately for c2n and n2c loops.  done using atom 2 (CA?)
///
/// @param[out]   ngap - out - gap for c2n loops
/// @param[out]   cgap - out - gap for n2c loops
///
/// @global_read Eposition
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray 4/28/4
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_gaps(
	float & ngap,
	float & cgap
)
{
	using namespace loops_ns;
	using namespace misc;

	float dis;
	ngap = 0.0;
	cgap = 0.0;
	int numn = 0;
	int numc = 0;

	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_get_refold_dir(i) == c2n ) {
			if ( loop_begin(i) == 1 ) goto L100;
			distance_bk(Eposition(1,2,loop_begin(i)-1),
			 Eposition(1,2,loop_begin(i)),dis);
			ngap += dis;
			++numn;
		}
		if ( loop_get_refold_dir(i) == n2c ) {
			if ( loop_end(i) == total_residue ) goto L100;
			distance_bk(Eposition(1,2,loop_end(i)+1),Eposition(1,2,loop_end(i)),
			 dis);
			cgap += dis;
			++numc;
		}
L100:;
	}

	if ( numn > 0 ) ngap /= numn;
	if ( numc > 0 ) cgap /= numc;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_get_refold_dir
///
/// @brief
///
/// @detailed
///
/// @param  loop_num - [in]
/// @return dir
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
loop_get_refold_dir(
	int const loop_num)
{
	using namespace loops_ns;

	if ( loop_num < 1 ) {
		return c2n;
	}
	if ( loop_num > num_loop ) {
		return n2c;
	}
	int seg_num = identify_segment(loop_begin(loop_num));
	int dir;
	refold_select_dir(seg_num,loop_begin(loop_num),loop_end(loop_num),dir);
	return dir;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin loop_set_bfactor
///
/// @brief
///
/// @detailed
///
/// @param  bfactor - [in/out]? -
/// @param  residue - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
loop_set_bfactor(
	float & bfactor,
	int residue
)
{
	using namespace loops_ns;

	if ( !get_loop_flag() ) return;
	bfactor = 0.0;
	if ( loop_map(residue) != 0 ) bfactor = 10.0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_loop_splicing
///
/// @brief
///
/// @detailed
///
/// @param  randomize - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_loop_splicing( bool randomize )
{
	using namespace loops_ns;
	using namespace misc;
	using namespace runlevel_ns;


	static int splice_mode = { 1 };
//car local
	FArray1D_int seg_begin( DRange( 0, max_loop+1 ) );
	FArray1D_int seg_end( DRange( 0, max_loop+1 ) );
	FArray1D_int seg_dir( DRange( 0, max_loop+1 ) );

	if ( num_loop == 0 ) return;
	if ( randomize ) {
		if ( ran3() > 0.5 ) {
			splice_mode = -1;
		} else {
			splice_mode = 1;
		}
		splice_mode = 1; // DEBUG!!
	}
	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_begin(i) == 2 ) splice_mode = 1;
	}

	if ( runlevel > inform ) std::cout << "SPLICEMODE: " << splice_mode << std::endl;
	if ( splice_mode != 1 && splice_mode !=-1 ) {
		std::cout << "unsupported splicemode for loops: " << splice_mode << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	seg_dir(0) = 0;
	for ( int i = 1; i <= num_loop; ++i ) {
		seg_dir(i) = splice_mode;
		if ( loop_begin(i) == 1 ) seg_dir(i) = c2n;
		if ( loop_end(i) == total_residue ) seg_dir(i) = n2c;
	}
	seg_dir(num_loop+1) = 0;

	int direction = splice_mode;
	if ( num_loop == 1 ) direction = seg_dir(1);

	int first_seg = 1;
	int last_seg = num_loop;
	seg_begin(0) = 1; // initialize; may not be set
	seg_end(0) = 0;


	if ( direction == n2c ) {
		if ( loop_end(num_loop) != total_residue ) last_seg = num_loop+1;
		for ( int seg_num = first_seg; seg_num <= last_seg; ++seg_num ) {
			if ( seg_num == 1 ) {      // note first_seg = 1
				seg_begin(seg_num) = 1;
				seg_end(seg_num) = loop_end(1);
			} else if ( seg_num == num_loop+1 ) {
				seg_begin(seg_num) = loop_end(num_loop)+1;
				seg_end(seg_num) = total_residue;
			} else {
				seg_begin(seg_num) = loop_end(seg_num-1)+1;
				seg_end(seg_num) = loop_end(seg_num);
			}
		}
	} else {
		if ( loop_begin(1) != 1 ) first_seg = 0;
		for ( int seg_num = first_seg; seg_num <= last_seg; ++seg_num ) {
			if ( seg_num == 0 ) {
				seg_begin(seg_num) = 1;
				seg_end(seg_num) = loop_begin(1) - 1;
			} else if ( seg_num == num_loop ) { // note last_seg = num_loop
				seg_begin(seg_num) = loop_begin(seg_num);
				seg_end(seg_num) = total_residue;
			} else {
				seg_begin(seg_num) = loop_begin(seg_num);
				seg_end(seg_num) = loop_begin(seg_num+1) - 1;
			}
		}
	}

	refold_set_segments(0,last_seg,seg_begin,seg_end,seg_dir);

//car special cases-- do not allow loop to be entire segment
//car---------------------------------------

	for ( int i = 1; i <= num_loop; ++i ) {
		if ( loop_begin(i) > seg_begin(i) || loop_end(i) < seg_end(i) ) goto L100;

		if ( seg_dir(i) == n2c && i > 1 ) {
			if ( loop_end(i-1) < seg_begin(i)-1 ) {
				--seg_begin(i);
				goto L100;
			}
		}
		if ( seg_dir(i) == c2n && i < num_loop ) {
			if ( loop_begin(i+1) > seg_end(i)+1 ) {
				++seg_end(i);
				goto L100;
			}
		}

//car try opposite direction
		seg_dir(i) = -seg_dir(i);
		if ( seg_dir(i) == n2c && i > 1 ) {
			if ( loop_end(i-1) < seg_begin(i)-1 ) {
				--seg_begin(i);
				goto L100;
			}
		}
		if ( seg_dir(i) == c2n && i < num_loop ) {
			if ( loop_begin(i+1) > seg_end(i)+1 ) {
				++seg_end(i);
				goto L100;
			}
		}

		std::cout << "ERROR:: Unable to find anchor residue for loop " << i <<
		 std::endl;
		std::cout << "begin,end,total_res,seg_dir,seg_begin,seg_end, " <<
		 loop_begin(i) << ' ' << loop_end(i) << ' ' << total_residue << ' ' <<
		 seg_dir(i) << ' ' << seg_begin(i) << ' ' << seg_end(i) << std::endl;
		std::cout << "Stopping..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

L100:;
	}
	refold_adjust_segment_ends(0,last_seg,seg_begin,seg_end,seg_dir);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_compressed_loops
///
/// @brief
///
/// @detailed
///
/// @param  total_residue - [in/out]? -
/// @param  output_res   [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_compressed_loops(
	int & total_residue,
	FArray1Da_bool output_res
)
{
	using namespace loops_ns;

	output_res.dimension( total_residue );

// write loop-specific header info to silent format output files

	if ( !get_loop_flag() ) return;
	output_res = false;

	for ( int i = 1; i <= num_loop; ++i ) {
		for ( int j = loop_begin(i), je = loop_end(i); j <= je; ++j ) {
			output_res(j) = true; // write out loop residues only
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_compressed_loops_header
///
/// @brief
/// write loop specific header info to silent format output files
///
/// @detailed
///
/// @param  iunit - [in/out]? -
/// @param  total_residue - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_compressed_loops_header(
	utility::io::orstream & iunit
)
{

	using namespace files_paths;
	using namespace loops_ns;

	if ( !get_loop_flag() ) return;

	for ( int i = 1; i <= num_loop; ++i ) {
		iunit << "LOOP: " << I( 6, loop_begin(i) ) <<
		 I( 6, loop_end(i) ) << std::endl;
	}
//car write out template coordinates
//car this is a bit of a hack:  no tag, but means pdb reader will work
//car loop coordinates are completely arbitrary
	if ( output_fa ) {
		output_fullcoord_pdb(iunit);
	} else {
		output_position_pdb(iunit);
	}

}

//////////////////////////////////////////////////////////////////////////////
bool
get_loop_fold_with_dunbrack()
{
	return loops_ns::loop_bool::fold_with_dunbrack;
}
/////////////////////////////////////////////////////////////////////////////////
/// @begin close_loop_after_moves
///
/// @brief
/// call dunbrack loop closure after a certain torsion_bb_moves
///
/// @detailed
/// close each loop by dunbrack loop closure method sequentially
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// best must be hijacked since refold relys on it to build current
///
/// @references
///
/// @authors Chu Wang
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
close_loop_after_moves()
{
	using namespace loops_ns;
	using namespace scorefxns;
	using namespace runlevel_ns;

	float forward_deviation, backward_deviation;

	if ( runlevel > inform ) std::cout << "close_loop_after_moves" << std::endl;

	hijack_best_pose();
	for ( int i = 1; i <= num_loop; ++i ) {

		scored_ccd_loop_closure(loop_begin(i), loop_end(i), loop_end(i), 100,
														score12, forward_deviation, backward_deviation,
														false, false);

		if ( runlevel > inform ) {
			std::cout << "loop_num: " << SS(i) << std::endl;
			std::cout << "forward_deviation: " << SS(forward_deviation) << std::endl;
			std::cout << "backward_deviation: " << SS(backward_deviation)
								<< std::endl;
		}
		accept_best_coord();
	}
	hijack_best_pose_release();
}
