// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "minimize_chi.h"
#include "aaproperties_pack.h"
#include "angles.h"
#include "fullatom.h"
#include "input_pdb.h"
#include "maps.h"
#include "minimize_ns.h"
#include "misc.h"
#include "param.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>

// Numeric Headers
#include <numeric/conversions.hh>


////////////////////////////////////////////////////////////////////////////////
/// @begin refold_new_chi
///
/// @brief update full_coord sidechain coordinates to new chi angles
///
/// @detailed
///
///  rebuild full_coord sidechain coordinates so that the chi angles
///  specified in the update_chi array have the chi values specified
///  in the new_chi array
///
/// @global_read
///
/// misc.h: full_coord,res,res_variant
///
/// namespace minimize_ns::unpack_refoldchi_share: update_chi,new_chi
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_new_chi(int & first_res, int & last_res)
{
//ctsa
//ctsa update_chi - specify which chi angles should be refolded
//ctsa new_chi - specifies what these chi angles should be refolded to
//ctsa

	using namespace misc;
	using namespace minimize_ns::unpack_refoldchi_share;

//------------------------------------------------------------------------------
	if ( !get_fullatom_flag() ) return;

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( refold_new_chi_residue(full_coord(1,1,seqpos),res(seqpos),
																res_variant(seqpos),new_chi(1,seqpos),
																update_chi(1,seqpos)) ) {
			maps_set_new_rotamer(seqpos);
			first_res = ( seqpos < first_res ? seqpos : first_res ); //min
			last_res = ( seqpos > last_res ? seqpos : last_res ); //max
		} // if
	} // for
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_new_chi_residue
///
/// @brief rebuild residue coordinates with specified chi angles
///
/// @detailed
///
///     given the coordinates of a sidechain, rebuild coordinates such
///     that the chi angles of the sidechain which are specified by the
///     change_chi array are changed to match the chi angles specified by
///     the chi array.
///
/// @param  rescoord - [in/out]? - coordinates of existing sidechain
/// @param[in]   aa - in - sidechain amino acid
/// @param[in]   aav - in - sidechain amino acid variant
/// @param[in]   chi - in - array of desired chi angles
/// @param[in]   change_chi - in - array specifying which chi angles to change
///
/// @return    true if sidechain coords changed
///
/// @global_read
///
/// param.h: max_[many]
/// aaproperties_pack.h: chi_required,nchi
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
refold_new_chi_residue(
	FArray2Da_float rescoord,
	int const aa,
	int const aav,
	FArray1Da_float chi,
	FArray1Da_bool change_chi
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using numeric::conversions::radians;

	rescoord.dimension( 3, MAX_ATOM() );
	chi.dimension( MAX_CHI );
	change_chi.dimension( MAX_CHI );

//ctsa ( after bk )


// local
	float ichi,rot;

	static FArray2D_float mat( 3, 3 );
	static FArray1D_float vec( 3 );

//------------------------------------------------------------------------------
	bool rotamer_changed = false;
	for ( int this_chi = 1; this_chi <= MAX_CHI; ++this_chi ) {
		if ( nchi(aa,aav) < this_chi ) return rotamer_changed;
		if ( !change_chi(this_chi) ) goto L785;
		rotamer_changed = true;

		//// get initial chi angle
		ichi = dihedral(rescoord(1,chi_atoms(1,this_chi,aa,aav)),
		 rescoord(1,chi_atoms(2,this_chi,aa,aav)),
		 rescoord(1,chi_atoms(3,this_chi,aa,aav)),
		 rescoord(1,chi_atoms(4,this_chi,aa,aav)));

		//// find rotation angle
		rot = radians( chi(this_chi) - ichi );

		//// generate rotation vector and matrix
		Sgetrot(rescoord(1,chi_atoms(2,this_chi,aa,aav)),
		 rescoord(1,chi_atoms(3,this_chi,aa,aav)),rot,mat,vec);

		//// move atoms
		for ( int i = 1, e = natoms(aa,aav); i <= e; ++i ) {
			if ( chi_required(this_chi,i,aa,aav) ) {
				Smove_in_place(rescoord(1,i),mat,vec);
			}
		}

L785:;
	}
	return rotamer_changed;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_closest_chi
///
/// @brief create and return the closest chi index
///
/// @detailed
///
///     Calculate and return the closest_chi index. For any atom, the
///     closest_chi value is the highest numbered chi angle on the
///     sidechain which is capable of rotating this atom (when the
///     backbone is considered a fixed reference point). The closest_chi
///     index stores this value for all atoms in all amino acid types.
///
/// @param[out]   closest_chi - out - closest chi index structure
///
/// @global_read
/// param.h: max_[many]
/// aaproperties_pack.h: n_atoms,chi_required
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_closest_chi( FArray4DB_bool & closest_chi )
{
	using namespace aaproperties_pack;
	using namespace param;

//------------------------------------------------------------------------------
	for ( int aa = 1, aae = MAX_AA(); aa <= aae; ++aa ) {
		for ( int aav = 1, aave = MAX_AA_VARIANTS(); aav <= aave; ++aav ) {
			int const nchi_aa = nchi(aa,aav);
			for ( int atm = 1, lcc = closest_chi.index(1,atm,aa,aav),
			 atme = natoms(aa,aav); atm <= atme; ++atm ) {

				//// ctsa - init
				////
				for ( int chino = 1; chino <= MAX_CHI; ++chino, ++lcc ) {
					closest_chi[ lcc ] = false; // closest_chi(chino,atm,aa,aav)
				}

				//// ctsa - an atom's closest chi is the highest numbered chi angle
				//// required to build it
				////
				for ( int chino = nchi_aa, lccn = closest_chi.index(chino,atm,aa,aav);
				 chino >= 1; --chino, --lccn ) {
				 // Assumes closest_chi and chi_required have same dimensions
					if ( ( closest_chi[ lccn ] = chi_required[ lccn ] ) ) goto L145;
//Objexx: Replaced by above for speed
//					closest_chi(chino,atm,aa,aav) = chi_required(chino,atm,aa,aav);
//					if ( closest_chi(chino,atm,aa,aav) ) goto L145;
				}
L145:;
			}
		}
	}

}
