// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 16636 $
//  $Date: 2007-08-21 15:47:16 -0700 (Tue, 21 Aug 2007) $
//  $Author: ashworth $


// Rosetta Headers
#include "after_opts.h"
#include "multistate_design.h"
#include "design.h"
#include "dna.h"
#include "DnaPose.h"
#include "dna_classes.h"
#include "dna_ns.h"
#include "dock_structure.h"
#include "FArray_xyz_functions.h"
#include "files_paths.h"
#include "fullatom_energy.h"
#include "ga_seq_opt.h"
#include "input_pdb.h"
#include "jumping_refold.h"
#include "ligand_ns.h"
#include "ligand.h"
#include "misc.h"
#include "namespace_fullatom_flag.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "pose_io.h"
#include "pose_dna.h"
#include "random_numbers.h"
#include "score.h"
#include "DesignMap.h"
#include "PackerTask.h"
#include "RotamerSet.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <fstream>
#include <functional>
#include <iostream>
#include <list>
#include <sstream>
#include <vector>

////////////////////////////////////////////////////////////////////////////////
// Helper class that defines a functor/closure over a vector of
// single states.  Invoking the functor reads in a pdb file, then
// packs up data from misc.h into a 'state' in the form of a pose
// plus an array with heteroatom coordinates, then adds it to the
// vector 'closed over'.

class StatePDBFileReader {
	private:
		std::vector< SingleState > & my_db;
	public:
		StatePDBFileReader( std::vector< SingleState > & _my_db ):
			my_db( _my_db ) {};

		void operator() ( std::string this_filename ) {
			using namespace files_paths;
			using namespace misc;

			start_x.clear();
			start_x.open( start_path + this_filename );
			if ( !start_x ) {
				std::cout << "Can't find starting structure: " <<
											start_x.filename() << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			std::cout << "reading starting structure: " <<
										this_filename << std::endl;
			bool coord_fail;
			input_pdb(start_x,use_fasta,input_fa,coord_fail);
			start_x.close();
			start_x.clear();
			my_db.push_back( single_state_from_misc() );
		}
};

//////////////////////////////////////////////////////////////////////////////
// Functor to randomize a full sequence - constructed with the info
// needed to know which positions and how many options to use for
// randomizing.

class PopMemberRandomizer {
	private:
		FArray1D_int const & res_mapping; // 1-indexed
		DesignMap const & mdm; // 1-indexed
	public:
		PopMemberRandomizer( FArray1D_int & rm, DesignMap & dm ):
			res_mapping( rm ), mdm( dm ) {}

		void operator() ( ga_seq_opt::PopMember & pm ) {

//			std::cout << "Starting member:";
			for ( int seq(1), end( pm._sequence.size() ); seq <= end; ++seq ) {
				// Be careful about 0/1 offsets for sequence positions:
				// pm.sequence (a std::vector) is 0-indexed
				pm._sequence[seq-1] = mdm.random_allowed_aa( res_mapping(seq) );
//				std::cout << " " << pm._sequence[seq-1];
			}
//			std::cout << std::endl;
		}
};

//////////////////////////////////////////////////////////////////////////////
// Functor to calculate the fitness for a single sequence
// The state energies are assumed to have already been determined.

class FitnessCalculator {

	private:

		float ref_energy, Boltz_temp;
		std::vector< SingleState > & state_db;
		std::vector< float > floor_energy, adjusted_energy;

	public:

		FitnessCalculator(
			std::vector< SingleState > & _state_db,
			int const _num_states,
			float const ref,
			float const temp
		) :
			ref_energy( ref ),
			Boltz_temp( temp ),
			state_db( _state_db )
		{
			floor_energy.clear();
			// Add one for reference
			floor_energy.assign( _num_states + 1, 0.0 );
			adjusted_energy.assign( _num_states + 1, 0.0 );
		};

		void operator() ( ga_seq_opt::PopMember & pm ) {
			using namespace ga_seq_opt;

			int nstates( floor_energy.size() - 1 );

			if ( !pm.cached() ) {

				// First, get an ensemble energy for the competitors
				for ( int tloop = 0; tloop < nstates; ++tloop ) {
					// This gets energy - best for this state
					adjusted_energy[tloop] = pm.state_energies(tloop) -
																		state_db[tloop].best_score;
				}

				for ( int tloop = 1; tloop < nstates; ++tloop ) {
					// This gets energy - target energy
					floor_energy[tloop] = adjusted_energy[tloop] - adjusted_energy[0];
				}
				floor_energy[ nstates ] = ref_energy - adjusted_energy[0];

				// Need the lowest energy to do robust exp math stuff
				float lowest_energy = *min_element( floor_energy.begin()+1,
																						floor_energy.end() );

				// Now the actual partition sum
				float partition = 0.0;
				float const beta( 1.0 / Boltz_temp );
				for ( int tloop = 1; tloop < (nstates + 1 ); ++tloop ) {
					float argument = (-1.0)*(floor_energy[tloop]-lowest_energy)*beta;
					// Exp math can easily under/overflow
					if ( argument < -150 ) {
						partition += 7.1751e-32;
					} else if ( argument > 150 ) {
						partition += 1.0e32;
					} else {
						partition += std::exp(argument);
					}
				}
				pm.set_fitness( Boltz_temp * std::log(partition) - lowest_energy );

//				std::cout << "FITNESS IS " << SS( pm.fitness() ) << std::endl;

// Cache this sequence's information

//				PopMember new_member( pm.fitness(), pm._sequence, pm.state_energies() );
//				_ms_hash_map[ pm._sequence ] =  new_member;
//				_ms_hash_map.insert( std::make_pair( pm._sequence, new_member ) );
				_ms_hash_map.insert( std::make_pair( pm._sequence, pm ) );

			}
		}
};

////////////////////////////////////////////////////////////////////////////////
// alternative fitness calculator, closer to Boltzmann probability
// ashworth
class JA_FitnessCalculator {

	private:
		std::vector< SingleState > & state_db;
		float ref_offset, Boltz_temp;

	public:
		JA_FitnessCalculator(
			std::vector< SingleState > & _state_db,
			float const ref,
			float const temp
		) :
			state_db( _state_db ),
			ref_offset( ref ),
			Boltz_temp( temp )
		{}

		void
		operator() ( ga_seq_opt::PopMember & pm )
		{
			using namespace ga_seq_opt;

			if ( !pm.cached() ) {
				// the 'affinity anchor'
				float const target_energy( state_db[0].best_score + ref_offset ),
				            inv_temp( 1.0 / Boltz_temp );
				// find lowest energy, to use as a zero point
				float const low(
					std::min(
						*min_element( pm.state_energies().begin(), pm.state_energies().end() ),
						target_energy
					)
				);
				// numerator initialized to the energy for the target state
				float const numer(
					std::exp( ( low - pm.state_energies(0) ) * inv_temp )
				);
				// denominator initialized to the anchor value
				float denom( std::exp( ( low - target_energy ) * inv_temp ) );

				for ( std::vector< float >::const_iterator
				      E( pm.state_energies().begin() );
				      E != pm.state_energies().end(); ++E ) {
					if ( E == pm.state_energies().begin() ) denom += numer;
					else denom += std::exp( ( low - *E ) * inv_temp );
				}
				// flip sign on the Boltzmann probability for proper comparison behavior in other functions
				pm.set_fitness( -1*(numer/denom) );

				// insert into hash
				_ms_hash_map.insert( std::make_pair( pm._sequence, pm ) );
			}
		}
};

////////////////////////////////////////////////////////////////////////////////
/// @begin StatePreparer
///
/// @brief
/// Class for getting/storing rotamer sets and interaction graphs for each of the physical states in a multistate run.  (I think it needs to close over the precomputed 'localtask.')
///
/// @authors
/// havranek
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
class StatePreparer {
	private:
		PackerTask const & pt;
		std::string outfile_root;
		int counter;

		void store_state_rotamers_and_ig( SingleState & sstate )
		{
			// Reinstate coords, res, etc. to misc
			copy_ms_set_to_current( sstate );

			// In pack_rotamers(), this dim is MAX_RES(), but I don't want it to be updated dynamically, so using the pose total_residue here
			int total_residue_inout( sstate.pose.total_residue() );
			sstate.current_rot_index.dimension( total_residue_inout );
			sstate.current_rot_index = 0;

			// Get rotamer_set and ig, as advertized
			pack_rotamers_setup( pt, sstate.p_rotamer_set_, sstate.ig, sstate.pose,
				sstate.current_rot_index, sstate.ligenergy1b_old );
		}

		void design_best_state(
			SingleState & sstate,
			bool ms_share_data = false
		)
		{
			using namespace files_paths;

			// Reinstate coords, res, etc. to misc (sometimes unnecessary)
			copy_ms_set_to_current( sstate );

			std::vector< int > rot_to_pack;
			if ( ms_share_data ) {
				// restrict DNA rotamer selection from master RotamerSet
				restrict_dna_rotamers( sstate.p_rotamer_set_, sstate.pose.res(),
				                       rot_to_pack );
			}

			pack_rotamers_run( pt, sstate.p_rotamer_set_, sstate.ig, sstate.pose,
				sstate.current_rot_index, sstate.ligenergy1b_old, rot_to_pack );

			pose_ns::Score_weight_map wm;
			setup_score_weight_map( wm, score12 );
			sstate.best_score = sstate.pose.score(wm);

			std::cout << "Got best energy of " << sstate.best_score <<
			 " for state " << counter << std::endl;

			sstate.pose.dump_pdb(
				pdb_out_path + outfile_root + "_SSD_" +
				lead_zero_string_of(counter, 2) + ".pdb"
			);
			counter++;
		}

	public:
		StatePreparer(
			PackerTask & _pt,
			std::string const filename
		): pt( _pt ), outfile_root( filename ), counter( 0 ) {};

		// used as functor
		void operator() ( SingleState & sstate )
		{
			store_state_rotamers_and_ig( sstate );
			design_best_state( sstate );
		}

	//ja If the only variability between states can be described at the rotamer level, then they can all share the same master rotamer_set and interaction graph.  (This is the case for comparing rotameric DNA sequence variation, for example.)
		void shared_rotamers_and_ig(
			std::vector< SingleState > & state_db
		)
		{
			// prepare for the first (reference/target) state
			SingleState & ref_state( state_db.front() );
			store_state_rotamers_and_ig( ref_state );

			// assign the rotamer_set and interaction graph to the other states
			for ( std::vector< SingleState >::iterator
			      state( state_db.begin() ); state != state_db.end(); ++state ) {
				if ( state != state_db.begin() ) {
					state->copy_rotamers_and_ig_from( ref_state );
				}
				// all
				design_best_state( *state, true );
			}
		}

};

////////////////////////////////////////////////////////////////////////////////
/// @begin multistate_design
///
/// @brief
///
/// @authors
/// havranek, ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
multistate_design( std::string const & packmode )
{
	using namespace design;
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace ga_seq_opt;
	using namespace misc;
	using namespace param;
	using namespace param_aa;


	std::string tmpc, filename;
	FArray2D_int extra_rot( MAX_CHI, MAX_RES() );
	FArray2D_float extra_chi( MAX_CHI, MAX_RES() );

	// may be useful to output a pedantic statistical analysis of the appropriate population size given the number of positions (and choices per position)
	int ms_pop_size(1000), ms_max_generations(50), num_packs(1);
	intafteroption( "ms_pop_size", ms_pop_size, ms_pop_size );
	intafteroption( "ms_generations", ms_max_generations, ms_max_generations );
	intafteroption( "ms_num_packs", num_packs, num_packs );
	float ref_offset(0.), Boltz_temp(0.6);
	realafteroption( "ms_ref_offset", ref_offset, ref_offset );
	realafteroption( "Boltz_temp", Boltz_temp, Boltz_temp );
	bool const ms_share_data( truefalseoption("ms_share_data") ),
	           ms_checkpoint( truefalseoption("ms_checkpoint") );

	std::cout << std::endl << " *** Starting multistate design: *** " << std::endl
	          << ms_max_generations << " generations"
						<< " with a population of " << ms_pop_size
						<< ", a ref_offset of " << F(4,1,ref_offset)
						<< ", and a Boltzmann temperature of " << F(4,1,Boltz_temp)
						<< std::endl << std::endl;

	std::string outfile_root;
	if ( pdbout == "des" ) outfile_root = output_file;
	else outfile_root = pdbout;

	// this will hold the states: Pose, RotamerSet, I.G., etc.
	std::vector< SingleState > state_db;
	// copy the starting structure over from misc as state 0
	state_db.push_back( single_state_from_misc() );

	// set up a PackerTask
	PackerTask localtask( state_db.front().pose );
	FArray1D_bool allow_repack( param::MAX_RES(), true );
	bool const make_output_file(false), include_current(true);
	localtask.set_task( packmode, make_output_file, allow_repack,
	                    include_current );
	localtask.set_nloop(1); // ensure single runs

	// analyze the resfile to guide sequence changes
	localtask.setup_residues_to_vary();

	if ( dna_interface && dna_variables::design_by_base ) {
		// temporarily make a DnaPose to get DNA interface information
		DnaPose dnapose;
		// restrict residue behavior based on automated interface detection
		// configures the DesignMap in localtask, saves interface info in DnaPose
		dnapose.find_dna_interface( localtask, 0., 0. );
		// build target and competitor DNA states internally
		build_DNA_competitors( dnapose, state_db );

	} else {
		// read paths to competitors from file
		std::vector< std::string > ms_filenames;
		read_ms_filenames( ms_filenames, ms_comp_list );
		// read in the competitors and store their structures
		StatePDBFileReader competitor_reader( state_db );
		for_each( ms_filenames.begin(), ms_filenames.end(), competitor_reader );
	}

	int num_states = static_cast<int>( state_db.size() );

	if ( num_states <= 1 ) {
		std::cout << "COMPETITOR LIST IS EMPTY" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else {
		std::cout << "Total number of states is " << SS( num_states ) << std::endl;
		std::cout << "MULTISTATE DESIGN: USING " << SS( num_states - 1 ) <<
								" COMPETITORS" << std::endl;
	}

	// Master Design Map
	DesignMap mdm( localtask.get_designmap() );

	// This memory allocation is more than required, but
	// I don't want to do a second run through or use a
	// std::vector here
	FArray1D_int res_mapping( MAX_RES(), -1 ), // 1-indexed
	             inv_res_mapping( MAX_RES(), -1 ); // 0-indexed
	FArray1D_bool is_res_designed( MAX_RES(), false ); // 1-indexed

	// determine and set up positions to design
	std::cout << "Design positions are";
	int num_des_pos = 0;
	for ( int i = 1, ie = MAX_RES(); i <= ie; ++i ) {
		// skip non-protein residues (e.g. rotameric DNA)
		if ( !param_aa::is_protein( state_db.front().pose.res(i) ) ) continue;
		if( mdm.num_allowed_aa( i ) > 1 ) {
			inv_res_mapping( i ) = num_des_pos; // 0-indexed
			res_mapping( ++num_des_pos ) = i; // 1-indexed
			is_res_designed( i ) = true; // 1-indexed
			std::cout << " " << i << "(pdb " << pdb::pdb_res_num(i) << ")";
		}
	}
	if ( num_des_pos == 0 ) {
		std::cerr << "ERROR: No design positions" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	std::cout << std::endl;

//	for ( int i = 1, ie = MAX_RES(); i <= ie; ++i ) {
//		std::cout << "inv mapping pos " << i << " is " << inv_res_mapping(i) << std::endl;
//	}

//	for ( int i = 0, ie = num_des_pos; i < ie; ++i ) {
//		std::cout << "res mapping option " << i << " is " << res_mapping(i+1) << std::endl;
//	}

	// Calculate and store rotamer sets, ig, single-state design for all states
	StatePreparer full_design_info( localtask, outfile_root );
	if ( ms_share_data ) {
		// do this only if state variability can be described at the rotamer level
		full_design_info.shared_rotamers_and_ig( state_db );
	} else {
		// functor prepares separate rotamer sets and ig's for each state
		for_each( state_db.begin(), state_db.end(), full_design_info );
	}

// Make the population
	PopMember default_member( num_des_pos, num_states );
	std::vector< PopMember > ms_population( ms_pop_size, default_member );

// Initialize the first generation
	PopMemberRandomizer sized_randomize( res_mapping, mdm );
	std::cout << "Starting population:" << std::endl;
	for_each( ms_population.begin(), ms_population.end(), sized_randomize );

	// clear the hash table for caching fitness values
	_ms_hash_map.clear(); // GLOBAL

//	JA_FitnessCalculator ms_fitness_calculator(
//		state_db, ref_offset, Boltz_temp
//	);

	FitnessCalculator ms_fitness_calculator(
		state_db, num_states, ref_offset, Boltz_temp
	);

// Enter loop over generations
	for ( int ms_iter = 1; ms_iter <= ms_max_generations; ++ms_iter ) {
		// if the previous job was interrupted prior to finishing, start where it left off
		if ( ms_checkpoint && ( ms_iter == 1 ) ) {
			load_checkpoint( ms_population, ms_iter, outfile_root );
		}

		int state_id(0);
		for ( std::vector< SingleState >::iterator state_itr = state_db.begin(),
					estate_itr = state_db.end();
					state_itr != estate_itr ;
					++state_itr, ++state_id ) {

//			std::ostringstream pdbname;
//			pdbname << state_id << ".pdb";
//			state_itr->pose.dump_pdb( pdbname.str() );

			std::cout << "Starting calculations for next state " << std::endl;

//    Now get energies for all sequences in the current population
//    in this structural context
			for ( std::vector< PopMember >::iterator p_itr = ms_population.begin(),
						e_itr = ms_population.end() ; p_itr != e_itr ; ++p_itr ) {

				if ( !p_itr->cached() ) {

//        Filter down to the rotamers needed for this single sequence
					std::vector< int > limited_rotamer_set;
					ms_set_limit_rotamer_set( limited_rotamer_set, is_res_designed,
					 inv_res_mapping, state_itr, p_itr );

//        Restore structural info for this template/state - must be done
//        everytime, since packer overwrites
					copy_ms_set_to_current( *state_itr );

					float E, bestE(9999.);
					pose_ns::Score_weight_map wm;
					setup_score_weight_map( wm, score12 );

//					std::cout << "Sequence " << seqstr( p_itr->_sequence )
//					          << "vs. state " << state_id << ": ";

					// optionally pack multiple times to find best energy
					for ( int i(0); i < num_packs; ++i ) {
						pack_rotamers_run( localtask, state_itr->p_rotamer_set_,
						 state_itr->ig, state_itr->pose, state_itr->current_rot_index,
						 state_itr->ligenergy1b_old, limited_rotamer_set );

						E = state_itr->pose.score(wm);
						if ( E < bestE ) bestE = E;
					}
					p_itr->state_energies(state_id) = bestE;

//					std::cout << F( 7, 2, p_itr->state_energies(state_id) )
//					          << std::endl;

				} // end of energy calculation for uncached (new) sequences
			} // end loop over all sequences
		} // end loop over all states

		// Convert template scores to fitness
		for_each( ms_population.begin(), ms_population.end(),
							ms_fitness_calculator );

		// Create next generation based on fitness
		ms_evolve_next_gen( ms_iter, num_states, mdm, res_mapping, ms_population );
		if ( ms_checkpoint ) {
			output_population( ms_population, outfile_root, ms_iter );
		}
	} // end loop over generations

// OUTPUT TOP SEQUENCE

	SingleState & tgt( state_db.front() ); // shorthand reference to target state
	copy_ms_set_to_current( tgt );

// Filter down to the rotamers needed for this single sequence
	std::vector< int > limited_rotamer_set;
	ms_set_limit_rotamer_set( limited_rotamer_set, is_res_designed,
	 inv_res_mapping, state_db.begin(), ms_population.begin() );

// Pack it
	pack_rotamers_run( localtask, tgt.p_rotamer_set_, tgt.ig, tgt.pose,
	 tgt.current_rot_index, tgt.ligenergy1b_old, limited_rotamer_set );

// Output
	if ( dna_interface && dna_variables::design_by_base ) {
		// use DnaPose output methods
		// (would have been cleaner if the state_db stored pointers to Pose
		//  capable of casting to DnaPose objects, but I ran into difficulties...)
		DnaPose dnapose;
		dnapose.init( tgt.pose );
		dnapose.find_dna_interface( localtask, 0., 0. );
		dnapose.outfile_root_ = outfile_root + "_ms_best_fit";
		dnapose.output_pdb();
	} else { // default
		tgt.pose.dump_pdb( pdb_out_path + outfile_root + "_ms_best_fit.pdb" );
	}

// DONE

}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_generation
///
/// @brief
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
output_population(
	std::vector< ga_seq_opt::PopMember > const & population,
	std::string const outfile_root,
	int const iteration
)
{
	using namespace ga_seq_opt;

	utility::io::ozstream outfile( outfile_root + ".pop_info" );
	outfile << "Generation " << iteration << std::endl;

	// write out current population
	for ( std::vector< PopMember >::const_iterator
		    member( population.begin() ); member != population.end(); ++member ) {
		outfile << "Current " << *member;
	}

	// write out all cached members
	for ( MultistateHashMap::const_iterator entry( _ms_hash_map.begin() );
	      entry != _ms_hash_map.end(); ++entry ) {
		outfile << "Cached " << entry->second;
	}

	outfile.close();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin load_checkpoint
///
/// @brief
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
load_checkpoint(
	std::vector< ga_seq_opt::PopMember > & population,
	int & ms_iter,
	std::string const outfile_root
)
{
	using namespace ga_seq_opt;

	std::ifstream checkpoint;
	std::string name( outfile_root + ".pop_info" );
	checkpoint.open( name.c_str() );
	if ( checkpoint ) {
		std::cout << "Reading checkpointed fitnesses from file" << std::endl;

		std::string line, word;
		// get generation index
		int last_iter;
		checkpoint >> word >> last_iter >> skip; // first line
		if ( ( word != "Generation" ) ) return;
		ms_iter = last_iter;

		// for checking compatibility
		size_t const seq_length( population.front()._sequence.size() ),
		             num_states( population.front().state_energies().size() );
		// store new PopMembers and hash entries in temp containers for validation
		std::vector< PopMember > in_pop;

		_ms_hash_map.clear();
		while ( std::getline( checkpoint, line ) ) {

			if ( line.substr(0,7) == "Current" ) {
				in_pop.push_back( PopMember( line ) );
				if ( !in_pop.back().validate( seq_length, num_states ) ) {
					in_pop.pop_back();
				}

			} else if ( line.substr(0,6) == "Cached" ) {
				PopMember entry( line );
				if ( entry.validate( seq_length, num_states ) ) {
					_ms_hash_map.insert( std::make_pair( entry._sequence, entry ) );
				}
			}
		} // end while

		// check for correct population size
		if ( in_pop.size() != population.size() ) {
			std::cerr << "Failed to read population from file" << std::endl;
			return;
		}
		// copy over
		population.clear();
		population.assign( in_pop.begin(), in_pop.end() );

		std::cerr << "Read population for generation " << ms_iter
		          << " from file:" << std::endl;
		for ( std::vector< PopMember >::const_iterator member( population.begin() );
		      member != population.end(); ++member ) {
			std::cout << *member;
		}
	}

	checkpoint.close();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin copy_ms_set_to_current
///
/// @brief
///
/// @authors
///
////////////////////////////////////////////////////////////////////////////////
void
copy_ms_set_to_current(
	SingleState const & state
)
{
	using namespace ga_seq_opt;
	using namespace ligand;
	using namespace misc;
	using namespace param;

	state.pose.copy_to_misc();

// Can't make head nor tail of these ligand arrays
	for ( size_t j = 0; j < ligand::ligand_one->atom_vector.size(); ++j ) {
		ligand::ligand_one->atom_vector[j]->set_coordinates(
									state.hetero_atom_coord.at(j));
	}
}

/////////////////////////////////////////////////////////////////////////////
/// PopMember comparison function
bool
fitness_compare(
	ga_seq_opt::PopMember const & pm1,
	ga_seq_opt::PopMember const & pm2
)
{
	return ( pm1.fitness() <= pm2.fitness() );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin ms_evolve_next_gen
///
/// @brief
///
/// @detailed
///
/// @param  mdm - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
ms_evolve_next_gen(
	int ms_iter,
	int num_states,
	DesignMap const & mdm, // 1-indexed
	FArray1DB_int const & res_mapping, // 1-indexed
	std::vector< ga_seq_opt::PopMember > & curr_pop )
{
	using namespace ga_seq_opt;
	using namespace param;
	using namespace param_aa;

	float const GA_RECOMBINE = { 0.95 };

//  The key tasks are:
// 1. copy over the best sequence
// 2. create a certain percentage of the next gen by recombination
// 3. do the rest by mutation
// 4. copy over to the working array
// 5. check the new generation to see if they've already been calculated

	int seq_len( (curr_pop.begin())->_sequence.size() );

	int ms_pop_size( curr_pop.size() );

	std::vector< PopMember > next_pop;
	next_pop.clear();
	next_pop.reserve( curr_pop.size() );

	next_pop.push_back( *min_element( curr_pop.begin(), curr_pop.end(),
																		fitness_compare ) );

	// The best of the current population is first, so comparison of firsts
	// tells if this one is new
	if( fitness_compare( *next_pop.begin(), *curr_pop.begin() ) ) {
		std::cout << "Found better fitness " << next_pop.begin()->fitness() <<
					std::endl;
	}

	std::cout << "Generation " << SS( ms_iter ) << std::endl;
	std::cout << "Best " << next_pop.front() << std::endl;

	int num_by_recombine = static_cast< int >(GA_RECOMBINE*ms_pop_size);
	if ( num_by_recombine > ms_pop_size - 1 ) {
		num_by_recombine = ms_pop_size - 1;
	}

// Get more by recombination
	for ( int ploop = 1; ploop <= num_by_recombine; ++ploop ) {
		int parent1 = ms_tournament_select( curr_pop );
		int parent2 = ms_tournament_select( curr_pop );

		PopMember new_member( seq_len, num_states );
		recombine_int_vectors( curr_pop[ parent1 ]._sequence,
														curr_pop[ parent2 ]._sequence,
														new_member._sequence );

		next_pop.push_back( new_member );
	}

// Get the rest by mutation
	int num_by_mutate = ms_pop_size - ( num_by_recombine + 1 );
	if ( num_by_mutate > 0 ) {
		for ( int ploop = 1; ploop <= num_by_mutate; ++ploop ) {
			int parent1 = ms_tournament_select( curr_pop );

			PopMember new_member( seq_len, num_states );

			float mutate_rate = 2.0 / seq_len;

			for ( int sloop = 1; sloop <= seq_len ; ++sloop ) {
				if ( mutate_rate > ran3() ) {
					new_member._sequence[ sloop - 1 ] =
						mdm.random_allowed_aa( res_mapping(sloop) );
				} else {
					new_member._sequence[ sloop - 1 ] =
						curr_pop[ parent1 ]._sequence[ sloop - 1 ];
				}
			}

		next_pop.push_back( new_member );
		}
	}

//	std::cout << "Current pop size is " << curr_pop.size() << " next is " << next_pop.size() << std::endl;

//	for( int ploop = 0, eloop = curr_pop.size() ; ploop < eloop ; ++ploop ) {
//		std::cout << "old pop_member " << ploop << " seq 0 " << curr_pop[ploop]._sequence[0] << std::endl;
//		std::cout << "old pop_member " << ploop << " seq 1 " << curr_pop[ploop]._sequence[1] << std::endl;
//		std::cout << "old pop_member " << ploop << " seq 2 " << curr_pop[ploop]._sequence[2] << std::endl;
//		std::cout << "old pop_member " << ploop << " seq 3 " << curr_pop[ploop]._sequence[3] << std::endl;
//		std::cout << "old pop_member " << ploop << " seq 4 " << curr_pop[ploop]._sequence[4] << std::endl;
//	}

// Swap the next generation in for the current one
	swap( curr_pop, next_pop );

//	for( int ploop = 0, eloop = curr_pop.size() ; ploop < eloop ; ++ploop ) {
//		std::cout << "new pop_member " << ploop << " seq 0 " << curr_pop[ploop]._sequence[0] << std::endl;
//		std::cout << "new pop_member " << ploop << " seq 1 " << curr_pop[ploop]._sequence[1] << std::endl;
//		std::cout << "new pop_member " << ploop << " seq 2 " << curr_pop[ploop]._sequence[2] << std::endl;
//		std::cout << "new pop_member " << ploop << " seq 3 " << curr_pop[ploop]._sequence[3] << std::endl;
//		std::cout << "new pop_member " << ploop << " seq 4 " << curr_pop[ploop]._sequence[4] << std::endl;
//	}

// Check all the new population members to see if we've cached their sequences
	for_each( curr_pop.begin(), curr_pop.end(), check_for_cached );

// Count those that are not cached (i.e., new sequences )

	std::cout << "There are " <<
		SS( count_if( curr_pop.begin(), curr_pop.end(), std::mem_fun_ref(&PopMember::not_cached) ) ) <<
	 " new sequences this generation" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin ms_tournament_select
///
/// @brief
///
/// @authors
/// havranek
///
/////////////////////////////////////////////////////////////////////////////////
int
ms_tournament_select( std::vector< ga_seq_opt::PopMember > & pop)
{
	int const pick1 = static_cast< int >(ran3()*pop.size());
	int const pick2 = static_cast< int >(ran3()*pop.size());

	return ( pop[ pick1 ].fitness() < pop[ pick2 ].fitness() ?
						pick1 :
						pick2 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @authors
/// havranek
///
/////////////////////////////////////////////////////////////////////////////
void
read_ms_filenames(
	std::vector< std::string > & filenames,
	std::string & list_file
)
{
	using namespace files_paths;

	std::string tmpc;

	filenames.clear();

// First, make sure we have a list of competitors.
	if ( list_file == "default" ) {
		std::cout << "FATAL ERROR:  MS DESIGN REQUIRES ms_comp_list FILE" <<
				std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else {
		std::cout << "Reading competitors from " << ms_comp_list << std::endl;
	}

	start_x.clear();
	start_x.open( list_file );
	if ( !start_x ) {
		std::cout << "FATAL ERROR: file " << start_x.filename() <<
		 " doesn't exist" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Now read them in and push into the return vector
	while ( start_x ) {
		start_x >> bite( tmpc ) >> skip;
		ObjexxFCL::strip_whitespace( tmpc ); // Trim surrounding whitespace
		if ( start_x.eof() ) {
			break;
		} else if ( start_x.fail() ) {
			start_x.clear();
			start_x >> skip;
			continue;
		}

		filenames.push_back( tmpc );
		std::cout << "Competitor " << SS( filenames.size() ) << " is " << tmpc << std::endl;
	}

	start_x.close();
	start_x.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @authors
/// havranek
///
////////////////////////////////////////////////////////////////////////////////
SingleState
single_state_from_misc(){

	SingleState new_one;

	pose_from_misc( new_one.pose, true, false, true );
	new_one.hetero_atom_coord = get_ligand_cartesian_conformation( *ligand::ligand_one );

	return new_one;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @authors
/// havranek
///
////////////////////////////////////////////////////////////////////////////////
void
ms_set_limit_rotamer_set(
  std::vector< int > & limited_rotamer_set,
  FArray1DB_bool const & is_res_designed, // 1-indexed
  FArray1DB_int const & inv_res_mapping, // 0-indexed
  std::vector< SingleState >::iterator state_itr,
  std::vector< ga_seq_opt::PopMember >::iterator p_itr
)
{
	using namespace misc;

	limited_rotamer_set.clear();
	// Allocate enough to accomodate full rotamer set
	limited_rotamer_set.reserve( state_itr->p_rotamer_set_.nrotamers() );

	// Top loop is over rotamers
	for ( int irot(1), erot( state_itr->p_rotamer_set_.nrotamers() );
	      irot <= erot; ++irot ) {

		int rot_pos( state_itr->p_rotamer_set_.report_seqpos( irot ) ),
		    rot_aa( state_itr->p_rotamer_set_.report_aa( irot ) );

		if( is_res_designed( rot_pos ) == false ) {
			//ja careful not to allow rotameric DNA's to mutate
			if ( param_aa::is_NA(rot_aa) ) {
				if ( rot_aa != state_itr->pose.res(rot_pos) ) continue;
//				std::cout << "Allowing DNA rotamer " << param_aa::aa_name3(rot_aa)
//				          << " at pos " << rot_pos << std::endl;
			}
			limited_rotamer_set.push_back( irot );
		} else {
			// Pick the aa from the sequence in order
			int const member_aa(
				p_itr->_sequence[ inv_res_mapping( rot_pos ) ]
			);
//			std::cout << "pos " << rot_pos << " have " << rot_aa << " want " << pop_aa << " indices " << inv_res_mapping(rot_pos)+1 << " and " << p_itr->_sequence[ inv_res_mapping( rot_pos )  ]  << " sub " << inv_res_mapping( rot_pos ) << std::endl;
			if ( rot_aa == member_aa ) {
				limited_rotamer_set.push_back( irot );
//				std::cout << "Adding rotamer pos " << rot_pos << " aa " << rot_aa << std::endl;
			}
		}
	}

//	std::cout << "Using " << limited_rotamer_set.size() << " rotamers " << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @authors
/// havranek
///
////////////////////////////////////////////////////////////////////////////////
void
recombine_int_vectors(
	std::vector< int > & parent1,
	std::vector< int > & parent2,
	std::vector< int > & child
)
{

	int choose_total = parent1.size();
	int choose_0 = static_cast< int >( ran3() * (choose_total-1) )+1;

// convention for mask - 0 = from 1, 1 = from 2
// need to set choose_0 elements of mask array to 0, the
// rest to 1
//

//	std::cout << "choose_total is " << choose_total << std::endl;
//	std::cout << "choose_0 is " << choose_0 << std::endl;
	for ( int sloop = 0, eloop = choose_total ; sloop < eloop ; ++sloop ) {
		float chance = float(choose_0)/float(choose_total);
		float random_num = ran3();
		if ( random_num <= chance ) {
			child[ sloop ] = parent1[ sloop ];
//			std::cout << "Taking " << parent1[ sloop ] << " from 1 " << std::endl;
			--choose_0;
		} else {
			child[ sloop ] = parent2[ sloop ];
//			std::cout << "Taking " << parent1[ sloop ] << " from 2 " << std::endl;
		}
		--choose_total;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @authors
/// havranek
///
////////////////////////////////////////////////////////////////////////////////
void
check_for_cached(
	ga_seq_opt::PopMember & pm
)
{
	using namespace ga_seq_opt;

	// default value
	pm.set_cached( false );

	hash_map< std::vector< int >, PopMember, ga_seq_hash >::iterator hash_itr;
	hash_itr = _ms_hash_map.find( pm._sequence );
	if( hash_itr != _ms_hash_map.end() ) {
			pm.set_from_cached( hash_itr->second );
//			std::cout << "Found cached value fitness " << pm.fitness() << " state energy 0 " << pm.state_energies(0) << " 1 " << pm.state_energies(1) << std::endl;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin build_DNA_competitors
///
/// @brief
/// Builds internally-generated competitor DNA states for ga-multistate design.  (No external structures required.)
///
/// DnaPose is a derived class of Pose.  SingleState stores a pointer of type Pose.  This pointer can point to a DnaPose as a Pose; static_cast back to DnaPose is necessary to get back proper access to DnaPose members.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
build_DNA_competitors(
	DnaPose const & dnapose,
	std::vector< SingleState > & states
)
{
	std::cout << "Adding internally-generated DNA competitors" << std::endl;

	// get the user-specified target sequence
//	std::string const target( states.front().pose.opt_.userseq );
	std::string const target( dnapose.opt_.userseq );

	// set up list of single-basepair mutant competitor sequences
	std::list< DnaSeqInfo > seqs(
		( dnapose.dna_info_.target_set( target ) ).single_mutant_list()
	);

	int counter(0);
	for ( std::list< DnaSeqInfo >::const_iterator seq( seqs.begin() );
	      seq != seqs.end(); ++seq ) {
		// the first sequence in the list is the target, apply to existing state 0
		// otherwise, add a new copy of state 0 to make a competitor
		if ( seq != seqs.begin() ) states.push_back( states.front() );
		std::cout << "State " << counter << std::endl;
		++counter;
		mutate_bases( states.back().pose, *seq );
	}
}

