// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20540 $
//  $Date: 2008-02-22 11:53:44 -0800 (Fri, 22 Feb 2008) $
//  $Author: sid $


// Rosetta Headers
#include "orient_rms.h"
#include "current_pose.h"
#include "design.h"
#include "dock_loops.h"
#include "dock_structure.h"
#include "docking.h"
#include "docking_ns.h"
#include "docking_movement.h"
#include "files_paths.h"
#include "initialize.h"
#include "loops.h"
#include "maxsub_threshold.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "native.h"
#include "param.h"
#include "pose.h"
#include "pose_rms.h"
#include "runlevel.h"
#include "score.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyz.functions.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <list>
#include <vector>
#include <map>


//car  functions for aligning coordinates:
//car calc_rms             rms to native; wrapper, special conditions for
//car                      docking, loopmode, etc.
//car calc_dme             dme to native
//car
//car fast_rms_to_native   rms to native, called by calc_rms
//car
//car orient_rms_to_native aligns coordinate array onto the native ca array
//car                      returns rms, input array is modified
//car orient_rms           align one coord array onto a second by CA rms
//car                      second array is modified
//car orient_regions       takes two coordinate arrays and aligns the with
//car                      respect to a specified region, returns rms over
//car                      specified region, one input array is modified
//car fast_rms_x           calc CA rms between two arrays when they are
//car                      aligned over the total length, neither array is moved
//car fast_region_rms_x    calc rms between two arrays over specified region
//car                      when they are aligned over a specified region
//car                      neither array is modified
//car
//car --------------------------------------------------------------------------
//car  underlying math:
//car   find_UU - return the rotation matrix between two sets of coordinates
//car             input coordinates are translated to the origin
//car   find_UUtrans  -identical to find_UU but also returns the offset
//car                  vectors used to translate the coordinate sets
//car   UU_rotate - applies the rotation/offset returned by find_UUtrans
//car
//car   calc_rms_fast - fast rms calculation using parameters returned by
//car                   find_UU
//car
//car  accessory functions:
//car     fixEigenvector
//car     MatrixMult(4),BlankMatrixMult(4)  single and double precision
//car                                       matrix multiplication

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_rms
///
/// @brief
/// calculates rms of the position array to native CA  if native_exists not, returns 0
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_rms()
{
	using namespace misc;
	using namespace mc_global_track::diagnose; // yab: misc removal

	if ( !get_native_exists() ) {
		rms_err = 0.0;
		return;
	}

	if ( score_check_current_pose() ) {
		pose_ns::Pose & pose( score_get_current_pose() );
		calc_rms( pose );
		rms_err = pose.get_0D_score( pose_ns::RMSD );
	}

	if ( pose_flag() ) {
		// not sure what situations get me here
		rms_err = 0.0;
		return;
	}

	if ( files_paths::multi_chain && !design::dna_interface && !files_paths::antibody_modeler ) {
		if ( get_loop_flag() ) {
			loop_global_rms(0,position,rms_err);
			set_dock_loop_rmsd(rms_err);
		}
		if ( get_pose_symmetry_docking_flag() ) {

			 FArray2D_float shuffled_position;
			float rms_err_shuffle;
			// we need to change the order of the chains. rms is calculate for all
			// monomers not as in regular docking
			shuffle_chains_for_symmetry(shuffled_position);
			fast_rms_to_native(shuffled_position,rms_err_shuffle);
			fast_rms_to_native(position,rms_err);
			std::cout<< " rms_err : rms_err_shuffled " << rms_err
							 << " : " << rms_err_shuffle << std::endl;
			if ( rms_err_shuffle < rms_err )	rms_err = rms_err_shuffle;
			return;
		}
		calc_docking_rmsd();
		rms_err = get_docking_rmsd();
		return;
	}

	if ( get_loop_flag() ) {
		loop_global_rms(0,position,rms_err);
		return;
	}

	fast_rms_to_native(position,rms_err);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_dme
///
/// @brief
/// CA dme between native and current structure
/// call mk_position2 first to update calpha from position
///
/// @detailed
///
/// @param  dme_err - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_dme( float & dme_err )
{

	using namespace misc;
	using namespace native;

//  COMPUTE DISTANCES, INITIALIZE VARIABLES
	int tot = 0;
	dme_err = 0.0;
	for ( int i = 1; i <= total_residue; ++i ) {
		float const E1i = Eposition(1,2,i);
		float const E2i = Eposition(2,2,i);
		float const E3i = Eposition(3,2,i);
		float const n1i = native_ca(1,i);
		float const n2i = native_ca(2,i);
		float const n3i = native_ca(3,i);
		for ( int j = i+1; j <= total_residue; ++j ) {
			float const Edif1 = E1i - Eposition(1,2,j);
			float const Edif2 = E2i - Eposition(2,2,j);
			float const Edif3 = E3i - Eposition(3,2,j);
			float const dist_sim =
			 std::sqrt( ( Edif1 * Edif1 ) + ( Edif2 * Edif2 ) + ( Edif3 * Edif3 ) );

			float const ndif1 = n1i - native_ca(1,j);
			float const ndif2 = n2i - native_ca(2,j);
			float const ndif3 = n3i - native_ca(3,j);
			float const dist_nat =
			 std::sqrt( ( ndif1 * ndif1 ) + ( ndif2 * ndif2 ) + ( ndif3 * ndif3 ) );

			dme_err += ( ( dist_sim - dist_nat ) * ( dist_sim - dist_nat ) );
			++tot;
		}
	}
	if ( tot > 0 ) dme_err = std::sqrt(dme_err/tot);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin fast_rms_to_native
///
/// @brief
/// this a minor modification of orient_rms and fast_rms_x, to allow
/// for a function that computes the fast rms to native without changing
/// any position arrays
///
/// @detailed
/// compute rms of calpha between two structures
/// using function findUU() and the companion function calc_rms_fast
/// it reads in two arrays of x,y,z points (called p1 and p2)
/// it then finds the 3x3 matrix UU which rotates P2 onto P1 (or vica versa)
/// and returns the rms
///
/// @param  p2 - [in/out]? -
/// @param  fast_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// neither input array is modified
///
/// @references
/// Mathethematical Basis from paper:
/// april 20 1998 : found more recent paper which corrected errors
///
/// @authors
///  Charlie Strauss 1999
///  working revision feb 3 1999
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fast_rms_to_native(
	FArray2DB_float const & p2,
	float & fast_rms
)
{
	using namespace misc;
	using namespace native;
	using namespace param;

	FArray2D_double p1a( 3, total_residue );
	FArray2D_double p2a( 3, total_residue );
	FArray1D_double const ww( total_residue, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;

	int j = -3;
	int n_points = 0;
	for ( int k = 1; k <= total_residue; ++k ) {
		j += 5;
		if ( native_occupancy( 2, k ) <= 0 ) continue; // only if not missing
		n_points++;
		for ( int i = 1; i <= 3; ++i ) {
			p1a(i,n_points) = native_ca(i,k); // single to double
			p2a(i,n_points) = p2(i,j); // pick out c_alphas
		}
	}

	findUU( p1a, p2a, ww, n_points, uu, ctx );
	calc_rms_fast( fast_rms, p1a, p2a, ww, n_points, ctx );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin orient_rms_to_native
///
/// @brief
/// align c_alpha backbone of position matrix p2 to native ca backbone
/// r using function findUU() and the companion function calc_RMS
/// { it ends.
///
/// @detailed
/// it reads in two arrays of x,y,z points (called P1.dat and P2.dat
/// it then finds the 3x3 matrix UU which rotates P2 onto P1 (or vica versa)
/// it also  applies UU to P2 and writes the results to the console
///
/// @param  p2 - [in/out]? -
/// @param  fast_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
/// Mathethematical Basis from paper:
/// april 20 1998 : found more recent paper which corrected errors
///
///
/// @authors
/// Charlie Strauss 1999
/// working revision feb 3 1999
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
orient_rms_to_native(
	FArray2Da_float p2,
	float & fast_rms
)
{
	using namespace misc;
	using namespace native;
	using namespace param;

	p2.dimension( 3, total_residue*5 );


	FArray2D_double p1a( 3, total_residue );
	FArray2D_double p2a( 3, total_residue );
	FArray1D_double const ww( total_residue, 1.0 );
	FArray2D_double uu( 3, 3 );
	FArray1D_double rtemp( 3 );
	double ctx;

	if ( !get_native_exists() ) {
		fast_rms = 0.0;
		return;
	}

	int nPoints = total_residue*5;
	int j = 2;
	for ( int k = 1; k <= total_residue; ++k ) {
		for ( int i = 1; i <= 3; ++i ) {
			p1a(i,k) = native_ca(i,k); // single to double
			p2a(i,k) = p2(i,j); // pick out c_alphas
		}
		j += 5;
	}

	findUU( p1a, p2a, ww, total_residue, uu, ctx );

// now have 3x3 rotation array UU
	calc_rms_fast( fast_rms, p1a, p2a, ww, total_residue, ctx );

// in-place rotation of p2 using UU
	for ( int k = 1; k <= nPoints; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			rtemp(j) = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				rtemp(j) += uu(j,i)*p2(i,k);
			}
		}
		for ( int j = 1; j <= 3; ++j ) {
			p2(j,k) = rtemp(j);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin orient_rms
///
/// @brief
/// align c_alpha backbone of position matrix p2 to p1 backbone
/// r using function findUU() and the companion function calc_RMS
///
/// @detailed
/// it reads in two arrays of x,y,z points (called P1.dat and P2.dat
/// it then finds the 3x3 matrix UU which rotates P2 onto P1 (or vica versa)
/// it also  applies UU to P2 and writes the results to the console
/// { it ends.
///
/// @param  p1 - [in/out]? -
/// @param  p2 - [in/out]? -
/// @param  nres - [in/out]? -
/// @param  fast_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
/// Mathethematical Basis from paper:
/// april 20 1998 : found more recent paper which corrected errors
///
/// @authors
/// Charlie Strauss 1999
/// working revision feb 3 1999
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
orient_rms(
	FArray3Da_float p1,
	FArray3Da_float p2,
	int nres,
	float & fast_rms
)
{
	using namespace param;

	p1.dimension( 3, MAX_POS, nres );
	p2.dimension( 3, MAX_POS, nres );

	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	FArray1D_double const ww( nres, 1.0 );
	FArray2D_double uu( 3, 3 );
	double rtemp;
	double ctx;

	for ( int j = 1; j <= nres; ++j ) {
		for ( int i = 1; i <= 3; ++i ) {
			p1a(i,j) = p1(i,2,j); // single to double
			p2a(i,j) = p2(i,2,j); // pick out c_alphas
		}
	}

	findUU( p1a, p2a, ww, nres, uu, ctx );

// now have 3x3 rotation array UU
	calc_rms_fast( fast_rms, p1a, p2a, ww, nres, ctx );

// in-place rotation of p2 using UU
	for ( int k = 1; k <= nres; ++k ) {
		for ( int kk = 1; kk <= 5; ++kk ) {
			for ( int j = 1; j <= 3; ++j ) {
				rtemp = 0.0;
				for ( int i = 1; i <= 3; ++i ) {
					rtemp += uu(j,i)*p2(i,kk,k);
				}
				p2(j,kk,k) = rtemp;
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin orient_region
///
/// @brief - orient a specified region (both backbone and full_coord)
///         to xyz2 backbone
///
/// @detailed
/// orient xyz1 on top of xyz2 using regions specified by 'align' variables
/// return rms specified by rms variables, xyz1 is modified by this function
///
/// @param[in]   nres - in - logical size of  coordinate
/// @param  xyz1 - [in/out]? - coordinate array
/// @param  xyz1_full - [in/out]? - full_coord to be moved
/// @param[in]   xyz2 - in - coordinate array
/// @param[in]   alignregions - in - number of contig regions to align over
/// @param[in]   alignstart - in -
/// @param[in]   alignend - in -
/// @param[in]   rmsregions - in - number of contig regions to calc rms over
/// @param[in]   rmsstart - in -
/// @param[in]   rmsend - in -
/// @param[out]   rms - out -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   xyz1 and xyz1_full are modified by this function
///
/// @references
///
/// @authors GLB
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
orient_region(
	int nres, // logical size of  coordinate
	FArray3Da_float xyz1, // coordinate array
	FArray3Da_float xyz1_full, // full_coord to be moved
	FArray3Da_float xyz2, // coordinate array
	int alignregions, // number of contig regions to align over
	FArray1Da_int alignstart,
	FArray1Da_int alignend,
	int rmsregions, // number of contig regions to calc rms over
	FArray1Da_int rmsstart,
	FArray1Da_int rmsend,
	float & rms
)
{
	using namespace param;

	xyz1.dimension( 3, MAX_POS, nres );
	xyz1_full.dimension( 3, MAX_ATOM(), nres );
	xyz2.dimension( 3, MAX_POS, nres );
	alignstart.dimension( alignregions );
	alignend.dimension( alignregions );
	rmsstart.dimension( rmsregions );
	rmsend.dimension( rmsregions );


	int const MAXREGIONS = { 50 }; //changed by Sid for aligning variable interfaces

//car local
	FArray2D_double p1( 3, MAX_RES()() );
	FArray2D_double p2( 3, MAX_RES()() );
	FArray1D_double ww( MAX_RES()() );
	FArray1D_double p1_offset( 3 );
	FArray1D_double p2_offset( 3 );
	FArray2D_double uu( 3, 3 );
	double sigma3;

	alignend(1) = std::min(nres,alignend(1));
	rmsend(1) = std::min(nres,rmsend(1));
	if ( alignregions > MAXREGIONS ) std::cout <<
	 "increase MAXREGIONS in orient_region to : " << alignregions << std::endl;

//car pack arrays for find_UU,UU_rotate
	int aligncount = 0;
	for ( int i = 1; i <= alignregions; ++i ) {
		for ( int j = alignstart(i), je = alignend(i); j <= je; ++j ) {
			++aligncount;
			for ( int k = 1; k <= 3; ++k ) {
				p1(k,aligncount) = xyz1(k,2,j);
				p2(k,aligncount) = xyz2(k,2,j);
			}
			ww(aligncount) = 1.0; // weight all residues equally
		}
	}

	findUU_trans( p1, p2, ww, aligncount, uu, sigma3, p1_offset, p2_offset );
	UU_rotate(xyz1,nres*MAX_POS,p1_offset,p2_offset,uu);
	UU_rotate(xyz1_full,nres*MAX_ATOM()(),p1_offset,p2_offset,uu);

	int rmscount = 0;
	rms = 0.0;

	for ( int i = 1; i <= rmsregions; ++i ) {
		for ( int j = rmsstart(i), je = rmsend(i); j <= je; ++j ) {
			++rmscount;
			for ( int k = 1; k <= 3; ++k ) {
				float const xyz12 = xyz1(k,2,j) - xyz2(k,2,j);
				rms += xyz12 * xyz12;
			}
		}
	}

	rms = std::sqrt(rms/rmscount);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fast_rms_x
///
/// @brief
///
/// @detailed
/// compute rms of calpha between two structures
/// using function findUU() and the companion function calc_rms_fast
/// it reads in two arrays of x,y,z points (called p1 and p2)
/// it then finds the 3x3 matrix UU which rotates P2 onto P1 (or vica versa)
/// and returns the rms
///
/// @param  p1 - [in/out]? -
/// @param  p2 - [in/out]? -
/// @param  fast_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car  neither input array is modified
///
/// @references
/// Mathethematical Basis from paper:
/// april 20 1998 : found more recent paper which corrected errors
///
/// @authors
/// Charlie Strauss 1999
///  working revision feb 3 1999
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fast_rms_x(
	FArray2Da_float p1,
	FArray2Da_float p2,
	float & fast_rms
)
{
	using namespace misc;
	using namespace param;

	p1.dimension( 3, total_residue*5 );
	p2.dimension( 3, total_residue*5 );

	FArray2D_double p1a( 3, total_residue );
	FArray2D_double p2a( 3, total_residue );
	FArray1D_double const ww( total_residue, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;

//	int nPoints = total_residue*5;
	int j = 2;
	for ( int k = 1; k <= total_residue; ++k ) {
		for ( int i = 1; i <= 3; ++i ) {
			p1a(i,k) = p1(i,j); // single to double
			p2a(i,k) = p2(i,j); // pick out c_alphas
		}
		j += 5;
	}

	findUU( p1a, p2a, ww, total_residue, uu, ctx );
	calc_rms_fast( fast_rms, p1a, p2a, ww, total_residue, ctx );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin fast_region_rms_x
///
/// @brief
///
/// @detailed
/// return rms between xyz1 and xyz2 over regions specified by rms variables
/// when arrays are oriented with respect to regions specified by align
/// variables, neither xyz1 or xyz2 is modified
///
/// @param  nres - [in/out]? - logical size of  coordinate
/// @param  xyz1 - [in/out]? - coordinate arrays
/// @param  xyz2 - [in/out]? -
/// @param  alignregions - [in/out]? - number of contig regions to align over
/// @param  alignstart - [in/out]? -
/// @param  alignend - [in/out]? -
/// @param  rmsregions - [in/out]? - number of contig regions to calc rms over
/// @param  rmsstart - [in/out]? -
/// @param  rmsend - [in/out]? -
/// @param  rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fast_region_rms_x(
	int nres, // logical size of  coordinate
	FArray3Da_float xyz1, // coordinate arrays
	FArray3Da_float xyz2,
	int alignregions, // number of contig regions to align over
	FArray1Da_int alignstart,
	FArray1Da_int alignend,
	int rmsregions, // number of contig regions to calc rms over
	FArray1Da_int rmsstart,
	FArray1Da_int rmsend,
	float & rms
)
{
	using namespace param;

	xyz1.dimension( 3, MAX_POS, nres );
	xyz2.dimension( 3, MAX_POS, nres );
	alignstart.dimension( alignregions );
	alignend.dimension( alignregions );
	rmsstart.dimension( rmsregions );
	rmsend.dimension( rmsregions );

	int const MAXREGIONS = { 10 };

//car local
	FArray2D_double p1( 3, MAX_RES()() );
	FArray2D_double p2( 3, MAX_RES()() );
	FArray1D_double ww( MAX_RES()() );
	FArray1D_double p1_offset( 3 );
	FArray1D_double p2_offset( 3 );
	FArray2D_double uu( 3, 3 );
	double sigma3;
	FArray2D_float p3( 3, MAX_RES()() );

	alignend(1) = std::min(nres,alignend(1));
	rmsend(1) = std::min(nres,rmsend(1));
	if ( alignregions > MAXREGIONS ) std::cout <<
	 "increase MAXREGIONS in fast_region_rms_x to : " << alignregions << std::endl;

//car pack arrays for find_UU,UU_rotate
	int aligncount = 0;
	for ( int i = 1; i <= alignregions; ++i ) {
		for ( int j = alignstart(i), je = alignend(i); j <= je; ++j ) {
			++aligncount;
			for ( int k = 1; k <= 3; ++k ) {
				p1(k,aligncount) = xyz1(k,2,j);
				p2(k,aligncount) = xyz2(k,2,j);
			}
			ww(aligncount) = 1.0; // weight all residues equally
		}
	}

	int rmscount = 0;
	for ( int i = 1; i <= rmsregions; ++i ) {
		for ( int j = rmsstart(i), je = rmsend(i); j <= je; ++j ) {
			++rmscount;
			for ( int k = 1; k <= 3; ++k ) {
				p3(k,rmscount) = xyz1(k,2,j);
			}
		}
	}

	findUU_trans(p1,p2,ww,aligncount,uu,sigma3,p1_offset,p2_offset);
	UU_rotate(p3,nres,p1_offset,p2_offset,uu);

	rms = 0.0;
	for ( int i = 1; i <= rmscount; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			float const p3_xyz2 = p3(k,i)-xyz2(k,2,i);
			rms += p3_xyz2 * p3_xyz2;
		}
	}
	rms = std::sqrt(rms/rmscount);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fast_rms_ca
///
/// @brief
/// modified from fast_rms_x so that input is just CA positions and they can
/// be the subset of the whole sequence. for docking_interf_rmsd.
///
/// @detailed
/// compute rms of calpha between two structures
/// using function findUU() and the companion function calc_rms_fast
/// it reads in two arrays of x,y,z points (called p1 and p2)
/// it then finds the 3x3 matrix UU which rotates P2 onto P1 (or vica versa)
/// and returns the rms
///
/// @param  p1- [in/out]? -
/// @param  p2 - [in/out]? -
/// @param  nres - [in/out]? -
/// @param  fast_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// one of the inputs will be modified.
///
/// @references
/// Mathethematical Basis from paper:
/// april 20 1998 : found more recent paper which corrected errors
/// working revision feb 3 1999
///
/// @authors Charlie Strauss 1999
///
/////////////////////////////////////////////////////////////////////////////////
void
fast_rms_ca(
	FArray2Da_double p1,
	FArray2Da_double p2,
	int nres,
	float & fast_rms
)
{
	using namespace param;

	p1.dimension( 3, MAX_RES() );
	p2.dimension( 3, MAX_RES() );

	FArray1D_double const ww( nres, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;

	findUU( p1, p2, ww, nres, uu, ctx );
	calc_rms_fast( fast_rms, p1, p2, ww, nres, ctx );

}

void
fast_rms_atoms(
	FArray2Da_double p1,
	FArray2Da_double p2,
	int nres,
	float & fast_rms
)
{
	using namespace param;

	p1.dimension( 3, MAX_ATOM()());
	p2.dimension( 3, MAX_ATOM()());

	FArray1D_double const ww( nres, 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;

	findUU( p1, p2, ww, nres, uu, ctx );
	calc_rms_fast( fast_rms, p1, p2, ww, nres, ctx );

}
////////////////////////////////////////////////////////////////////////////////
/// @begin calc_rms_maximal_minimal_rms(
///
/// @detailed
///
/// @param[in] points1 list of coordinates
/// @param[in] points2 list of coordinates
/// @param [out] resulting_vector
/// @param [in] max_size_of_points1
/// @param [in] minimal_rms_limit
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  Kristian Kaufmann
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////


float
calc_rms_maximal_minimal_rms(
	std::list< std::vector < numeric::xyzVector< double > > > & points1,
	std::list< std::vector < numeric::xyzVector< double > > > & points2,
	std::list< std::vector < numeric::xyzVector< double > > > & resulting_vector,
	size_t max_size_of_points1,
	float minimal_rms_limit
){
	//The purpose of this function is to load into points1 elements from points2
	// that maximized the coverage of points1 with the number of elements up to
	// max_size_of_points1 with distance between all element greater than
	// minimal_rms_limits.

	//iwd  To restate: resulting_vector is set to all the conformations from points1,
	//iwd  plus some of the conformations from points2,
	//iwd  up to a total number of conformations <= max_size_of_points1.
	//iwd  The goal is to iteratively add the conformation from points2 that is most different
	//iwd  (measured by rms) from all the conformations already in resulting_vector.
	//iwd  The actual implementation tries to be more efficient than the naive; see below.
	resulting_vector.clear();
	resulting_vector=points1;

	//Calculate the minimal rms
	//iwd  That is, the rms between each element of point2
	//iwd  and the element of resulting_vector it's closest to.
	//iwd  If it's within minimal_rms of an element in resulting_vector,
	//iwd  mark it for removal so we don't consider it again.
	//iwd  Otherwise, store it in minimal_rms (sorted by minimal rms, ascending).
	typedef std::list< std::vector< numeric::xyzVector< double > > >::iterator list_conf_iter;
	std::list< list_conf_iter > added_to_resulting_vector;
	std::multimap< float, list_conf_iter > minimal_rms;
	std::list< list_conf_iter > points_to_remove;
	float rms=10000;
	float max_rms_under_minimum=0;
	float best_rms=0;
	for(list_conf_iter p2itr=points2.begin(); p2itr!=points2.end(); p2itr++){

		bool found_rms=false;
		float min_rms=1000;
		for( list_conf_iter p1itr=resulting_vector.begin();
			p1itr!=resulting_vector.end(); p1itr++){

			rms=calc_rms_wrapper((*p2itr),(*p1itr));
			if( rms >0 && rms < minimal_rms_limit){
				points_to_remove.push_back(p2itr);
				if( rms > max_rms_under_minimum){
					max_rms_under_minimum=rms;
				}
				break;
			}else if( rms < min_rms && rms > 0 ){
				min_rms=rms;
				found_rms=true;
			}
		}//points1
		if( found_rms ){
			if( min_rms > best_rms){
				best_rms=min_rms;
			}
			std::pair< float, list_conf_iter > rms_ptr_pair( min_rms, p2itr );
			minimal_rms.insert(rms_ptr_pair);
		}else if ( max_rms_under_minimum > best_rms ){
			best_rms=max_rms_under_minimum;
		}
	}//points2
	//removing all elements within minimal_rms_limit of and element of points1
	int removed_points=0;
	while( !points_to_remove.empty()){
		points2.remove((*points_to_remove.back()));
		points_to_remove.pop_back();
		removed_points++;
	}
	if( points2.empty() || resulting_vector.size() >= max_size_of_points1 ){
		//all points2 were within minimal_rms_limit
		return best_rms;
	}

	//iwd  Iteratively add the candidate with the maximal minimal rms,
	//iwd  updating all other candidates minimal rms on each pass,
	//iwd  until we've accumulated the desired number of conformations.
	//iwd  This code must be careful about holding onto iterators that point
	//iwd  to erase()'d elements (e.g. minimal_rms.second holds iterators
	//iwd  from points2, but we may remove things from points2 also).
	do {
		assert( !minimal_rms.empty()); //iwd  Really only useful on first pass thru loop

		//iwd  Pop the last element from minimal_rms (the conf with the maximal minimal rms)
		//iwd  Below, a goofy way of getting a forward iterator to the last element, so we can erase() it.
		//iwd  The ++ is necessary because otherwise it gives the one-past-the-end element.
		std::multimap< float, list_conf_iter >::iterator const
			best = (++(minimal_rms.rbegin())).base();
		assert(best->second == minimal_rms.rbegin()->second);
		best_rms = best->first;
		resulting_vector.push_back(*best->second);
		points2.erase(best->second);
		minimal_rms.erase(best); // from here out, best is invalid/unusable

		//iwd  Update the minimal rms for remaining candidates.
		//iwd  This is the min of their previous min rms and their rms to the new addition.
		//iwd  Also, eliminate from further consideration any candidates
		//iwd  that fall within minimal_rms_limit of the new addition.
		float min_rms=1000;
		typedef std::multimap< float, list_conf_iter >::iterator map_conf_iter;
		for(map_conf_iter i = minimal_rms.begin(); i != minimal_rms.end(); ) {
			map_conf_iter j = i;
			j++; // have to make a copy and increment before erasing i
			rms = calc_rms_wrapper(*i->second, (resulting_vector.back()));
			if( rms > 0 && rms < minimal_rms_limit){
				points2.erase( i->second );
				minimal_rms.erase(i);
			}else if( rms >0 && rms < min_rms ){
				min_rms=rms;
				std::pair< float, list_conf_iter > rms_ptr_pair( min_rms, i->second );
				// need to remove the old one so this is a replace rather than an insert
				minimal_rms.insert(rms_ptr_pair);
				minimal_rms.erase(i);
			}
			i = j; // now i is valid again after having been erased
		}

	} while( !minimal_rms.empty() && resulting_vector.size() < max_size_of_points1 );

	return best_rms;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin calc_rms_wrapper
///
/// @detailed
///
/// @param[in] points1 cartesian coordinates
/// @param[in] points2 cartesian coordinates
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  Kristian Kaufmann
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////


float
calc_rms_wrapper(
	std::vector< numeric::xyzVector< double > > & points1,
	std::vector< numeric::xyzVector< double > > & points2
){


	if( points1.size()==points2.size()){
	//kwk variables needed
	FArray2D_double XX(3, points1.size());
	FArray2D_double YY(3, points1.size());
	FArray1D_double WW(points1.size(), 1.0 );
	FArray2D_double UU(3, 3);
	float rms=-9999.999;
	double sigma3=99.99;
	size_t point_index=1;
	std::vector< numeric::xyzVector< double > >::iterator p1itr=points1.begin();
	for(std::vector< numeric::xyzVector< double > >::iterator p2itr=points2.begin();
		p2itr!=points2.end(); p2itr++){

		XX(1,point_index)=p1itr->x();
		XX(2,point_index)=p1itr->y();
		XX(3,point_index)=p1itr->z();
		YY(1,point_index)=p2itr->x();
		YY(2,point_index)=p2itr->y();
		YY(3,point_index)=p2itr->z();
		point_index++;
		p1itr++;
	}

	findUU(XX,YY,WW,int(points1.size()),UU,sigma3);
	calc_rms_fast(rms,XX,YY,WW,int(points1.size()),sigma3);
	return rms;

	}
	return -9999.999;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin findUU
///
/// @brief
/// intended to rotate one protein xyz array onto another one such that
/// the point-by-point rms is minimized.
///
/// @detailed
///   1) ORIGINAL PAPER HAD ERROR IN HANDEDNESS OF VECTORS, LEADING
///      TO INVERSION MATRICIES ON OCCASION. OOPS. NOW FIXED.
///       SEE ACTA CRYST(1978) A34 PAGE 827 FOR REVISED MATH
///   2) TRAP DIVIDE BY ZERO ERRORS WHEN NO ROTATIONS REQUIRED.
///   3) ADDED WEIGHTS (WEIGHTS NOW WORK)
///   4) ADDED FAST RMS CALC AUXILIRARY ROUTINE.
///   5) CHANGED TO double TO DEAL WITH HIGHLY DISSIMILAR BUT LARGE PROTEINS.
///
/// switched order of array subscripts so that can use logical array sizes
/// XX and YY are lists of Npoints  XYZ vectors (3xNpoint matrix) to be co-aligned
/// these matrices are returned slightly modified: they are translated so their origins
/// are at the center of mass (see Weights below ).
/// WW is the weight or importance of each point (weighted RMS) vector of size Npoints
/// The center of mass is figured including this variable.
/// UU is a 3x3 symmetric orthonornal rotation matrix that will rotate YY onto XX such that
/// the weighted RMS distance is minimized.
///
/// @param[in]   XX - in - XX,YY  are 2D arrays of x,y,z position of each atom
/// @param[in]   YY - in -
/// @param[in]   WW - in -  a weight matrix for the points
/// @param[in]   Npoints - in - the number of XYZ points (need not be physical array size)
/// @param[out]   UU - out - 3x3 rotation matrix.
/// @param[out]   sigma3 - out - TO BE PASSED TO OPTIONAL FAST_RMS CALC ROUTINE.
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// SIDEEFECTS: the Matrices XX, YY are Modified so that their weighted center
///
/// CAVEATS:
///      1) it is CRITICAL that the first physical dimension of XX and YY is 3
///
///      2) an iterative approx algorithm computes the diagonalization of
///         a 3x3 matrix.  if this program needs a speed up this could be
///         made into an analytic but uggggly diagonalization function.
///         of mass is  moved to (0,0,0).
///
/// @references
/// Mathethematical Basis from paper:
/// (Wolfgang Kabsch) acta Cryst (1976) A32 page 922
///
/// @authors
///  Charlie Strauss 1999
///  Revised april 22
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
findUU(
	FArray2DB_double & XX,
	FArray2DB_double & YY,
	FArray1DB_double const & WW,
	int Npoints,
	FArray2DB_double & UU,
	double & sigma3
)
{
	using namespace runlevel_ns;
	using numeric::xyzMatrix_double;
	using numeric::xyzVector_double;

	FArray1D_int sort( 3 );
	FArray2D_double eVec( 3, 3 );
	FArray2D_double bb( 3, 3 );
	FArray1D_double w_w( 3 );
	FArray2D_double m_moment( 3, 3 );
	FArray2D_double rr_moment( 3, 3 );
	double temp1;
	double temp2;
	double temp3;
	FArray1D_double Ra( 3 );

	if ( Npoints < 1 ) {

// return identity rotation matrix to moron
		for ( int i = 1; i <= 3; ++i ) {
			for ( int k = 1; k <= 3; ++k ) {
				UU(i,k) = 0.0;
				if ( i == k ) UU(i,i) = 1.0;
			}
		}
		sigma3 = 0.0;
		return;
	}

// align center of mass to origin
	for ( int k = 1; k <= 3; ++k ) {
		temp1 = 0.0;
		temp2 = 0.0;
		temp3 = 0.0;
		for ( int j = 1; j <= Npoints; ++j ) {
			temp1 += XX(k,j) * WW(j);
			temp2 += YY(k,j) * WW(j);
			temp3 += WW(j);
		}
		if (temp3 > 0.001) temp1 /= temp3;
		if (temp3 > 0.001) temp2 /= temp3;

		for ( int j = 1; j <= Npoints; ++j ) {
			XX(k,j) -= temp1;
			YY(k,j) -= temp2;
		}
	}

// Make cross moments matrix   INCLUDE THE WEIGHTS HERE
	for ( int k = 1; k <= 3; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			temp1 = 0.0;
			for ( int i = 1; i <= Npoints; ++i ) {
				temp1 += WW(i) * YY(k,i) * XX(j,i);
			}
			m_moment(k,j) = temp1;
		}
	}

// Multiply CROSS MOMENTS by transpose
	BlankMatrixMult(m_moment,3,3,1,m_moment,3,0,rr_moment);

	// Copy to/from xyzMatrix/xyzVector since rest of function uses FArrays
	xyzMatrix_double xyz_rr_moment( xyzMatrix_double::cols( &rr_moment( 1,1 ) ) );
	xyzVector_double xyz_w_w;
	xyzMatrix_double xyz_eVec;

	// Find eigenvalues, eigenvectors of symmetric matrix rr_moment
   xyz_w_w = eigenvector_jacobi( xyz_rr_moment, 1E-9, xyz_eVec );

	// Copy eigenvalues/vectors back to FArray
	for ( int i = 1; i <= 3; ++i ) {
		w_w( i ) = xyz_w_w( i );
		for ( int j = 1; j <= 3; ++j ) {
			eVec( i, j ) = xyz_eVec( i, j );
		}
	}

//     std::cout << "avec" << std::endl;
//     debug(eVec,3)


// explicitly coded 3 level index sort using eigenvalues
	for ( int i = 1; i <= 3; ++i ) {
		sort(i) = i;
	}

	if ( w_w(1) < w_w(2) ) {
		sort(2) = 1;
		sort(1) = 2;
	}

	if ( w_w(sort(2)) < w_w(3) ) {
		sort(3) = sort(2);
		sort(2) = 3;

		if ( w_w(sort(1)) < w_w(3) ) {
			sort(2) = sort(1);
			sort(1) = 3;
		}
	}

// sort is now an index to order of eigen values

	if ( w_w(sort(2)) == 0.0 ) { // holy smokes,  two eigen values are zeros
		if ( runlevel >= inform ) std::cout << "**** findUU:degenerate rotation! ***" <<
		 SS( w_w(1) ) << SS( w_w(2) ) << SS( w_w(3) ) << std::endl;
// return identity rotation matrix to moron
		for ( int i = 1; i <= 3; ++i ) {
			for ( int k = 1; k <= 3; ++k ) {
				UU(i,k) = 0.0;
			}
			UU(i,i) = 1.0;
		}
		if ( w_w(sort(1)) < 0.0 && runlevel >= inform ) {
			std::cout << "findUU: negative eigen value" << SS( w_w(sort(1)) ) << std::endl;
			w_w(sort(1)) = std::abs(w_w(sort(1)));
		}
		sigma3 = std::sqrt(w_w(sort(1)));

		return; // make like a prom dress and slip off
	}

// sort eigen values
	temp1 = w_w(sort(1));
	temp2 = w_w(sort(2));
	w_w(3) = w_w(sort(3));
	w_w(2) = temp2;
	w_w(1) = temp1;
// sort first two eigen vectors (dont care about third)
	for ( int i = 1; i <= 3; ++i ) {
		temp1 = eVec(i,sort(1));
		temp2 = eVec(i,sort(2));
		eVec(i,1) = temp1;
		eVec(i,2) = temp2;
	}


// april 20: the fix not only fixes bad eigen vectors but solves
// a problem of forcing a right-handed coordinate system

	fixEigenvector(eVec);
//     std::cout << "favec" << std::endl;
//     debug(eVec,3);
// at this point we now have three good eigenvectors in a right hand coordinate system.

// make bb basis vectors   = moments*eVec

	BlankMatrixMult(m_moment,3,3,0,eVec,3,0,bb);
//     std::cout << "m_moment" << std::endl;
//     debug(m_moment,3);
// squirrel away a free copy of the third eigenvector before normalization/fix
	for ( int j = 1; j <= 3; ++j ) {
		Ra(j) = bb(j,3);
	}

// normalize first two bb-basis vectors
// dont care about third since were going to replace it with b1xb2
// this also avoids problem of possible zero third eigen value
	for ( int j = 1; j <= 2; ++j ) {
		temp1 = 1.0/std::sqrt(w_w(j)); // zero checked for above
		for ( int k = 1; k <= 3; ++k ) { // x,y,z
			bb(k,j) *= temp1;
		}
	}

//  fix things so that bb eigenvecs are right handed

	fixEigenvector(bb); // need to fix this one too
//     std::cout << "Bvec" << std::endl;
//     debug(eVec,3);
// find  product of eVec and bb matrices

	BlankMatrixMult(eVec,3,3,0,bb,3,1,UU);
// result is returned in UU.


// and lastly determine a value used in another function to compute the
// rms
	sigma3 = 0.0;
	for ( int j = 1; j <= 3; ++j ) {
		sigma3 += bb(j,3)*Ra(j);
	}
//cems the abs() fixes some round off error situations where the w_w values are
//cems very small and accidentally negative.  (theoretically they are positive,
//cems but in practice round off error makes them negative)
	if ( sigma3 < 0.0 ) {
		sigma3 = std::sqrt(std::abs(w_w(1))) + std::sqrt(std::abs(w_w(2))) -
		 std::sqrt(std::abs(w_w(3)));
	} else {
		sigma3 = std::sqrt(std::abs(w_w(1))) + std::sqrt(std::abs(w_w(2))) +
		 std::sqrt(std::abs(w_w(3)));
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin findUU_trans
///
/// @brief
/// this is just like findUU except that the offset vectors for XX and YY
/// are returned.
///
/// @detailed
/// purpose: intended to rotate one protein xyz array onto
/// another one such that the point-by-point rms is minimized.
///
/// REVISED APRIL 21
///   1) ORIGINAL PAPER HAD ERROR IN HANDEDNESS OF VECTORS, LEADING
///      TO INVERSION MATRICIES ON OCCASION. OOPS. NOW FIXED.
///       SEE ACTA CRYST(1978) A34 PAGE 827 FOR REVISED MATH
///   2) TRAP DIVIDE BY ZERO ERRORS WHEN NO ROTATIONS REQUIRED.
///   3) ADDED WEIGHTS (WEIGHTS NOW WORK)
///   4) ADDED FAST RMS CALC AUXILIRARY ROUTINE.
///   5) CHANGED TO double TO DEAL WITH HIGHLY DISSIMILAR BUT LARGE PROTEINS.
///
///  Revised april 22
///   switched order of array subscripts so that can use logical array sizes
/// XX and YY are lists of Npoints  XYZ vectors (3xNpoint matrix) to be co-aligned
/// these matrices are returned slightly modified: they are translated so their origins
/// are at the center of mass (see Weights below ).
/// WW is the weight or importance of each point (weighted RMS) vector of size Npoints
/// The center of mass is figured including this variable.
/// UU is a 3x3 symmetric orthonornal rotation matrix that will rotate YY onto XX such that
/// the weighted RMS distance is minimized.
///
/// @param[in]   XX - in - XX,YY  are 2D arrays of x,y,z position of each atom
/// @param[in]   YY - in -
/// @param[in]   WW - in - weight matrix for the points
/// @param[in]   Npoints - in - number of XYZ points (need not be physical array size)
/// @param[out]   UU - out - 3x3 rotation matrix.
/// @param[out]   sigma3 - out - SIGMA3 IS TO BE PASSED TO OPTIONAL FAST_RMS CALC ROUTINE.
/// @param  xx_offset - [out] -
/// @param  yy_offset - [out] -
///
/// @global_read
///
/// @global_write
///
/// @references
/// Mathethematical Basis from paper:
/// (Wolfgang Kabsch) acta Cryst (1976) A32 page 922
///
/// @remarks
/// SIDEEFECTS: the Matrices XX, YY are Modified so that their weighted center
///            of mass is  moved to (0,0,0).
/// CAVEATS:
///      1) it is CRITICAL that the first physical dimension of XX and YY is 3
///      2) an iterative approx algorithm computes the diagonalization of
///              a 3x3 matrix.  if this program needs a speed up this could be
///               made into an analytic but uggggly diagonalization function.
///
/// @references
///
/// @authors
///  Charlie Strauss 1999
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
findUU_trans(
	FArray2Da_double XX,
	FArray2Da_double YY,
	FArray1Da_double WW,
	int Npoints,
	FArray2Da_double UU,
	double & sigma3,
	FArray1Da_double xx_offset,
	FArray1Da_double yy_offset
)
{
	using namespace runlevel_ns;
	using numeric::xyzMatrix_double;
	using numeric::xyzVector_double;

	XX.dimension( 3, Npoints );
	YY.dimension( 3, Npoints );
	WW.dimension( Npoints );
	UU.dimension( 3, 3 );
	xx_offset.dimension( 3 );
	yy_offset.dimension( 3 );
	FArray1D_int sort( 3 );

	FArray2D_double eVec( 3, 3 );
	FArray2D_double bb( 3, 3 );
	FArray1D_double w_w( 3 );
	FArray2D_double m_moment( 3, 3 );
	FArray2D_double rr_moment( 3, 3 );
	double temp1;
	double temp2;
	double temp3;
	FArray1D_double Ra( 3 );

	if ( Npoints < 1 ) {
// return identity rotation matrix to moron
		for ( int i = 1; i <= 3; ++i ) {
			for ( int k = 1; k <= 3; ++k ) {
				UU(i,k) = 0.0;
				if ( i == k ) UU(i,i) = 1.0;
			}
			xx_offset(i) = 0.0;
			yy_offset(i) = 0.0;
		}
		sigma3 = 0.0;
		return;
	}

// align center of mass to origin
	for ( int k = 1; k <= 3; ++k ) {
		temp1 = 0.0;
		temp2 = 0.0;
		temp3 = 0.0;
		for ( int j = 1; j <= Npoints; ++j ) {
			temp1 += XX(k,j)*WW(j);
			temp2 += YY(k,j)*WW(j);
			temp3 += WW(j);
		}
		if (temp3 > 0.001) temp1 /= temp3;
		if (temp3 > 0.001) temp2 /= temp3;
//db
		xx_offset(k) = temp1;
		yy_offset(k) = temp2;
//db
		for ( int j = 1; j <= Npoints; ++j ) {
			XX(k,j) -= temp1;
			YY(k,j) -= temp2;
		}
	}

// Make cross moments matrix   INCLUDE THE WEIGHTS HERE
	for ( int k = 1; k <= 3; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			temp1 = 0.0;
			for ( int i = 1; i <= Npoints; ++i ) {
				temp1 += WW(i)*YY(k,i)*XX(j,i);
			}
			m_moment(k,j) = temp1;
		}
	}

// Multiply CROSS MOMENTS by transpose
	BlankMatrixMult(m_moment,3,3,1,m_moment,3,0,rr_moment);

	// Copy to/from xyzMatrix/xyzVector since rest of function uses FArrays
	xyzMatrix_double xyz_rr_moment( xyzMatrix_double::cols( &rr_moment( 1,1 ) ) );
	xyzVector_double xyz_w_w;
	xyzMatrix_double xyz_eVec;

	// Find eigenvalues, eigenvectors of symmetric matrix rr_moment
   xyz_w_w = eigenvector_jacobi( xyz_rr_moment, 1E-9, xyz_eVec );

	// Copy eigenvalues/vectors back to FArray
	for ( int i = 1; i <= 3; ++i ) {
		w_w( i ) = xyz_w_w( i );
		for ( int j = 1; j <= 3; ++j ) {
			eVec( i, j ) = xyz_eVec( i, j );
		}
	}

//     std::cout << "avec" << std::endl;
//     debug(eVec,3)


// explicitly coded 3 level index sort using eigenvalues
	for ( int i = 1; i <= 3; ++i ) {
		sort(i) = i;
	}

	if ( w_w(1) < w_w(2) ) {
		sort(2) = 1;
		sort(1) = 2;
	}

	if ( w_w(sort(2)) < w_w(3) ) {
		sort(3) = sort(2);
		sort(2) = 3;

		if ( w_w(sort(1)) < w_w(3) ) {
			sort(2) = sort(1);
			sort(1) = 3;
		}
	}

// sort is now an index to order of eigen values

	if ( w_w(sort(2)) == 0.0 ) { // holy smokes,  two eigen values are zeros
		if ( runlevel >= inform ) std::cout <<
		 "*** findUU_trans:degenerate rotation! ***" <<
		 SS( w_w(1) ) << SS( w_w(2) ) << SS( w_w(3) ) << std::endl;
// return identity rotation matrix to moron
		for ( int i = 1; i <= 3; ++i ) {
			for ( int k = 1; k <= 3; ++k ) {
				UU(i,k) = 0.0;
				if ( i == k ) UU(i,i) = 1.0;
			}
		}
		if ( w_w(sort(1)) <= 0.0 && runlevel >= inform ) {
			std::cout << "findUU_trans: negative eigen value" << SS( w_w(sort(1)) ) <<
			 std::endl;
			w_w(sort(1)) = std::abs(w_w(sort(1)));
		}
		sigma3 = std::sqrt(w_w(sort(1)));

		return; // make like a prom dress and slip off
		// comment courtesy of P.M. Bowers
	}

// sort eigen values
	temp1 = w_w(sort(1));
	temp2 = w_w(sort(2));
	w_w(3) = w_w(sort(3));
	w_w(2) = temp2;
	w_w(1) = temp1;
// sort first two eigen vectors (dont care about third)
	for ( int i = 1; i <= 3; ++i ) {
		temp1 = eVec(i,sort(1));
		temp2 = eVec(i,sort(2));
		eVec(i,1) = temp1;
		eVec(i,2) = temp2;
	}


// april 20: the fix not only fixes bad eigen vectors but solves
// a problem of forcing a right-handed coordinate system

	fixEigenvector(eVec);
//     std::cout << "favec" << std::endl;
//     debug(eVec,3)
// at this point we now have three good eigenvectors in a right hand coordinate system.

// make bb basis vectors   = moments*eVec

	BlankMatrixMult(m_moment,3,3,0,eVec,3,0,bb);
//     std::cout << "m_moment" << std::endl;
//     debug(m_moment,3)
// squirrel away a free copy of the third eigenvector before normalization/fix
	for ( int j = 1; j <= 3; ++j ) {
		Ra(j) = bb(j,3);
	}

// normalize first two bb-basis vectors
// dont care about third since were going to replace it with b1xb2
// this also avoids problem of possible zero third eigen value
	for ( int j = 1; j <= 2; ++j ) {
		temp1 = 1/std::sqrt(w_w(j)); // zero checked for above
		for ( int k = 1; k <= 3; ++k ) { // x,y,z
			bb(k,j) *= temp1;
		}
	}


//  fix things so that bb eigenvecs are right handed

	fixEigenvector(bb); // need to fix this one too
//     std::cout << "Bvec" << std::endl;
//     debug(eVec,3)
// find  product of eVec and bb matrices

	BlankMatrixMult(eVec,3,3,0,bb,3,1,UU);
// result is returned in UU.


// and lastly determine a value used in another function to compute the rms
	sigma3 = 0.0;
	for ( int j = 1; j <= 3; ++j ) {
		sigma3 += bb(j,3)*Ra(j);
	}
	if ( sigma3 < 0.0 ) {
		sigma3 = std::sqrt(w_w(1))+std::sqrt(w_w(2))-std::sqrt(w_w(3));
	} else {
		sigma3 = std::sqrt(w_w(1))+std::sqrt(w_w(2))+std::sqrt(w_w(3));
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin fixEigenvector
///
/// @brief
///
/// @detailed
///       m_v is a 3x3  matrix of 3 eigen vectors
///       replaces the third  eigenvector by taking cross product of
///       of the first two eigenvectors
///
/// @param  m_v - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fixEigenvector( FArray2Da_double m_v )
{
	m_v.dimension( 3, 3 );

	double const m_v_13 = m_v(2,1)*m_v(3,2) - m_v(3,1)*m_v(2,2);
	double const m_v_23 = m_v(3,1)*m_v(1,2) - m_v(1,1)*m_v(3,2);
	double const m_v_33 = m_v(1,1)*m_v(2,2) - m_v(2,1)*m_v(1,2);
//     normalize it to 1 (should already be one but lets be safe)
	double const norm = std::sqrt( 1 /
	 ( ( m_v_13 * m_v_13 ) + ( m_v_23 * m_v_23 ) + ( m_v_33 * m_v_33 ) ) );

	m_v(1,3) = m_v_13 * norm;
	m_v(2,3) = m_v_23 * norm;
	m_v(3,3) = m_v_33 * norm;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin UU_rotate
///
/// @brief
/// subtracts xx_off,rotates by uu, then adds yy_off  for num coordinates
/// in the coordinate array
///
/// @param coord - [in/out]? -
/// @param num - [in/out]? -
/// @param xx_off - [in/out]? -
/// @param yy_off - [in/out]? -
/// @param uu - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
UU_rotate(
	FArray2Da_float coord,
	int num,
	FArray1DB_double const & xx_off,
	FArray1DB_double const & yy_off,
	FArray2DB_double const & uu
)
{
	coord.dimension( 3, num );

	double x,y,z;

	for ( int i = 1; i <= num; ++i ) {
		coord(1,i) -= xx_off(1);
		coord(2,i) -= xx_off(2);
		coord(3,i) -= xx_off(3);
		x = uu(1,1)*coord(1,i)+uu(2,1)*coord(2,i)+uu(3,1)*coord(3,i)+yy_off(1);
		y = uu(1,2)*coord(1,i)+uu(2,2)*coord(2,i)+uu(3,2)*coord(3,i)+yy_off(2);
		z = uu(1,3)*coord(1,i)+uu(2,3)*coord(2,i)+uu(3,3)*coord(3,i)+yy_off(3);
		coord(1,i) = x;
		coord(2,i) = y;
		coord(3,i) = z;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_rms_fast
///
/// @brief
/// companion function for findUU ( it is optional ) computes the minimum
/// RMS deviation beteen XX and YY as though it rotated the arrays, without
/// actually rotating them.
///
/// @detailed
///
/// @param  rms_out   [in/out]? - the  float  output value of the rms deviation
/// @param  XX - [in/out]? -
/// @param  YY - [in/out]? -
/// @param  WW - [in/out]? -
/// @param  npoints - [in/out]? -
/// @param  ctx - [in/out]? - magic number computed during find UU that is needed
///               for this calculation
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// the XX, YY, WW must be the same as call to findUU (remember that findUU
/// offsets the XX and YY weighted  COM to the origin!)
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_rms_fast(
	float & rms_out,
	FArray2Da_double xx,
	FArray2Da_double yy,
	FArray1Da_double ww,
	int npoints,
	double ctx
)
{
	xx.dimension( 3, npoints );
	yy.dimension( 3, npoints );
	ww.dimension( npoints );


	double rms = 0.0;

	for ( int k = 1; k <= npoints; ++k ) {
		double const ww_k = ww(k);
		for ( int j = 1; j <= 3; ++j ) {
			double const xx_jk = xx(j,k);
			double const yy_jk = yy(j,k);
			rms += ww_k * ( ( xx_jk * xx_jk ) + ( yy_jk * yy_jk ) );
		}
	}

	rms -= 2 * ctx;
	// abs on next line catches small difference of large numbers that turn out
	// to be accidentall small but negative
	rms_out = std::sqrt(std::abs(rms/npoints)); // retrurn a float
}
////////////////////////////////////////////////////////////////////////////////
/// @begin BlankMatrixMult
///
/// @brief
///
/// @detailed
///
/// @param  A - [in/out]? -
/// @param  n - [in/out]? -
/// @param  np - [in/out]? -
/// @param  transposeA - [in/out]? -
/// @param  B - [in/out]? -
/// @param  m - [in/out]? -
/// @param  transposeB - [in/out]? -
/// @param  AxB_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
BlankMatrixMult(
	FArray2Da_double A,
	int n,
	int np,
	int transposeA,
	FArray2Da_double B,
	int m,
	int transposeB,
	FArray2Da_double AxB_out
)
{
	A.dimension( np, n );
	B.dimension( np, m );
	AxB_out.dimension( m, n );

// fills output matrix with zeros before calling matrix multiply
	AxB_out = 0.0;

	MatrixMult(A,n,np,transposeA,B,m,transposeB,AxB_out);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin BlankMatrixMult4
///
/// @brief
///
/// @detailed
///
/// @param  A - [in/out]? -
/// @param  n - [in/out]? -
/// @param  np - [in/out]? -
/// @param  transposeA - [in/out]? -
/// @param  B - [in/out]? -
/// @param  m - [in/out]? -
/// @param  transposeB - [in/out]? -
/// @param  AxB_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
BlankMatrixMult4(
	FArray2Da_float A,
	int n,
	int np,
	int transposeA,
	FArray2Da_float B,
	int m,
	int transposeB,
	FArray2Da_float AxB_out
)
{
	A.dimension( np, n );
	B.dimension( np, m );
	AxB_out.dimension( m, n );

//jg single precision version

// fills output matrix with zeros before calling matrix multiply
	AxB_out = 0.0;

	MatrixMult4(A,n,np,transposeA,B,m,transposeB,AxB_out);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MatrixMult
///
/// @brief
///
/// @detailed
/// multiplys matrices A (npXn) and B (npXn). results in AxB.out
/// IF THE MATRICES are SQUARE.  you can also multiply the transposes of these matrices
/// to do so set the transposeA or transposeB flags to 1, otherwise they should be zero.
///
/// @param  A - [in/out]? -
/// @param  n - [in/out]? -
/// @param  np - [in/out]? -
/// @param  transposeA - [in/out]? -
/// @param  B - [in/out]? -
/// @param  m - [in/out]? -
/// @param  transposeB - [in/out]? -
/// @param  AxB_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// transpose only works correctly for square matricies!
///
/// this function does SUMS to the old value of AxB_out...
/// you might want to call BlankMatrixMult instead (see above) 4/30/01 jjg
///
///this function works on double values
///float version below. jjg
///
/// @references
///
/// @authors
/// charlie strauss 1999
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
MatrixMult(
	FArray2Da_double A,
	int n,
	int np,
	int transposeA,
	FArray2Da_double B,
	int m,
	int transposeB,
	FArray2Da_double AxB_out
)
{
	A.dimension( np, n );
	B.dimension( np, m );
	AxB_out.dimension( m, n );



	if ( transposeA == 0 ) {
		if ( transposeB == 0 ) {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(k,i)*B(i,j);
					}
				}
			}
		} else {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(k,i)*B(j,i);
					}
				}
			}
		}
	} else {
		if ( transposeB == 0 ) {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(i,k)*B(i,j);
					}
				}
			}
		} else {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(i,k)*B(j,i);
					}
				}
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin MatrixMult4
///
/// @brief
///
/// @detailed
///
/// @param  A - [in/out]? -
/// @param  n - [in/out]? -
/// @param  np - [in/out]? -
/// @param  transposeA - [in/out]? -
/// @param  B - [in/out]? -
/// @param  m - [in/out]? -
/// @param  transposeB - [in/out]? -
/// @param  AxB_out - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
MatrixMult4(
	FArray2Da_float A,
	int n,
	int np,
	int transposeA,
	FArray2Da_float B,
	int m,
	int transposeB,
	FArray2Da_float AxB_out
)
{
	A.dimension( np, n );
	B.dimension( np, m );
	AxB_out.dimension( m, n );

//jjg single precision version


	if ( transposeA == 0 ) {
		if ( transposeB == 0 ) {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(k,i)*B(i,j);
					}
				}
			}
		} else {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(k,i)*B(j,i);
					}
				}
			}
		}
	} else {
		if ( transposeB == 0 ) {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(i,k)*B(i,j);
					}
				}
			}
		} else {
			for ( int k = 1; k <= m; ++k ) {
				for ( int j = 1; j <= n; ++j ) {
					for ( int i = 1; i <= np; ++i ) {
						AxB_out(k,j) += A(i,k)*B(j,i);
					}
				}
			}
		}
	}

}

/// @begin calc_residue_dist
///
/// @brief finds the individual ca-ca distances between the position
/// array and the native array, after oriented to minimize the rmsd

void
calc_residue_dist(
	FArray1D_float & residue_dist
)
{
	using namespace misc;
	using namespace param;
	using namespace native;

	for ( int i = 1; i <= total_residue; ++i ) {
		residue_dist(i) = 0;
	}
	if ( !get_native_exists() ) {
		return;
	} else {
		//		FArray1D_float residue_dist(total_residue);

		int orient_start = { 1 };
		int orient_end = { total_residue };

		float local_rms;
		//chu BUGFIX 2005-03-22
		// use a tmp copy of current pose to do the following calculations
		// because orient_region will transform the input coordinate.
		// the current pose should not be changed here.

		FArray3D_float tmp_Eposition( Eposition );
		FArray3D_float tmp_full_coord( full_coord );
		orient_region(total_residue,tmp_Eposition,tmp_full_coord,native_Eposition,1,
		 orient_start,orient_end,1,1,total_residue,local_rms);
		for ( int i = 1; i <= total_residue; ++i ) {
			residue_dist(i) = 0.0;
			for ( int j = 1; j <= 3; ++j ) {
				float const d = tmp_Eposition(j,2,i) - native_Eposition(j,2,i);
				residue_dist(i) += d*d;
			}
			residue_dist(i) = std::sqrt(residue_dist(i));
		}
	}
}

/////////////////////////////////////////////////////////////////////////////////
void
get_gdtmm_scores(
	float & mm_1_1,
	float & mm_2_2,
	float & mm_3_3,
	float & mm_4_3,
	float & mm_7_4,
	float & gdtmm
)
{
	using namespace misc;
	using namespace native;
	using namespace param;

	if ( !get_native_exists() ) {
		mm_1_1 = 0.0;
		mm_2_2 = 0.0;
		mm_3_3 = 0.0;
		mm_4_3 = 0.0;
		mm_7_4 = 0.0;
		gdtmm = 0.0;
		return;
	}

	FArray1D_bool maxsub_alignment( total_residue );

	// copy coords into double arrays
	FArray2D_double p1a( 3, total_residue );
	FArray2D_double p2a( 3, total_residue );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Eposition( k, atom_index, i );
			p2a(k,i) = native_ca(k,i);
		}
	}

	// now find out the maxsub alignment
	double mxrms,mxpsi,mxzscore,mxscore,mxeval,rmstol,distance_tolerance;
	int nali;
	rmstol = 1.0;
	distance_tolerance = 1.0;
	maxsub_threshold(total_residue,p1a,p2a,mxrms,mxpsi,nali,mxzscore,mxeval,
	 mxscore,rmstol,maxsub_alignment,distance_tolerance);
	mm_1_1 = float(nali)/float(total_residue);

	rmstol = 2.0;
	distance_tolerance = 2.0;
	maxsub_threshold(total_residue,p1a,p2a,mxrms,mxpsi,nali,mxzscore,mxeval,
	 mxscore,rmstol,maxsub_alignment,distance_tolerance);
	mm_2_2 = float(nali)/float(total_residue);

	rmstol = 3.0;
	distance_tolerance = 3.0;
	maxsub_threshold(total_residue,p1a,p2a,mxrms,mxpsi,nali,mxzscore,mxeval,
	 mxscore,rmstol,maxsub_alignment,distance_tolerance);
	mm_3_3 = float(nali)/float(total_residue);

	rmstol = 3.0;
	distance_tolerance = 4.0;
	maxsub_threshold(total_residue,p1a,p2a,mxrms,mxpsi,nali,mxzscore,mxeval,
	 mxscore,rmstol,maxsub_alignment,distance_tolerance);
	mm_4_3 = float(nali)/float(total_residue);

	rmstol = 4.0;
	distance_tolerance = 7.0;
	maxsub_threshold(total_residue,p1a,p2a,mxrms,mxpsi,nali,mxzscore,mxeval,
	 mxscore,rmstol,maxsub_alignment,distance_tolerance);
	mm_7_4 = float(nali)/float(total_residue);

	gdtmm = (mm_1_1 + mm_2_2 + mm_3_3 + mm_4_3 + mm_7_4 )/5.0;

}

/// @begin shuffle_chains_for_symmetry
///
/// @brief changes the order of chains of symmetrical proteins. Used for rms
// calculation

void
shuffle_chains_for_symmetry(
	 FArray2D_float & shuffled_position
){

		using namespace docking;
		using namespace misc;

		shuffled_position.dimension( 3, total_residue*5 );

		int const N_monomer ( docking::part_end(1) );
		int const f ( pose_symm_n_monomers == 2 ? 1 : 2 );

    for ( int i = 0; i < total_residue*5*3; ++i ) {
			if ( i < N_monomer*5*3 )	{
				shuffled_position[i] = position[i+f*N_monomer*5*3];
			} else if (i >= N_monomer*5*3 && i < f*N_monomer*5*3 ) {
				shuffled_position[i] = position[i];
			} else
				shuffled_position[i] =  position[i -f*N_monomer*5*3];
			}
}

void
shuffle_chains_for_symmetry(
	FArray2D_double & start,
	FArray2D_double & shuffled,
	int const N,
	int const nres,
	int const nres_monomer
){

/// for D2

	if ( docking::symm_type == "dn" && N == 4 ) {

    for ( int i = 0; i < nres*3; ++i ) {
      if ( i < 2*nres_monomer*3 ) {
        shuffled[i] = start[i];
      } else if (i >= 2*nres_monomer*3 && i < 3*nres_monomer*3 ) {
        shuffled[i] = start[i+nres_monomer*3];
      } else
        shuffled[i] =  start[i - nres_monomer*3];
      }

	} else {

		int const f ( N == 2 ? 1 : 2 );

    for ( int i = 0; i < nres*3; ++i ) {
			if ( i < nres_monomer*3 )	{
				shuffled[i] = start[i+f*nres_monomer*3];
			} else if (i >= nres_monomer*3 && i < f*nres_monomer*3 ) {
				shuffled[i] = start[i];
			} else
				shuffled[i] =  start[i -f*nres_monomer*3];
			}
	}
}


