// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

// Rosetta Headers
#include "pH_main.h"
#include "aaproperties_pack.h"
#include "design.h"
#include "files_paths.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pH_ns.h"
#include "read_aaproperties.h"
#include "runlevel.h"
#include "after_opts.h"
#include "initialize.h"

// ObjexxFLC Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>

// C++ Headers
#include <iostream>

// forward reference.
// JSS the following do not belong in the interface (.h) file.
// They should not be called outside this file because there are assumptions
// about which variants are created in which order.
void create_pH_asp1(); //1HD1
void create_pH_asp2(); //1HD2
void create_pH_asp_duplicate(); //Make degeneracy of the protonated and deprotonated states equal
void create_pH_glu1(); //1HE1
void create_pH_glu2(); //1HE2
void create_pH_glu_duplicate(); //Make degeneracy of the protonated and deprotonated states equal
void create_pH_his(); // 1ND, 1NE + 0 state
void create_pH_tyr();
void create_pH_lys();
void create_pH_arg();


////////////////////////////////////////////////////////////////////////////////
/// @begin get_pH_packing_options()
///
/// @brief Options for pH-sensitive packing
///
/// @detailed
///
/// @return
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/31/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_pH_packing_options()
{
	set_pH_packing_flag(true);
	realafteroption("WpH",1.00,pH_ns::WpH);
	realafteroption("Wgb",1.00,pH_ns::Wgb);
	realafteroption("Wscale",1.00,pH_ns::Wscale);
	design::design_commands::try_both_his_tautomers = true;
	std::cerr << "WARNING: Use pH-sensitive packing AT YOUR OWN RISK." << std::endl;
	std::cerr << "using pH-sensitive rotamers" << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_pH_packing_flag()
///
/// @brief Detect pH-sensitive side chain packing
///
/// @detailed
///
/// @return
/// pH
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/31/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_pH_packing_flag()
{
	return param_pack::packer_logical::pH_rot;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_pH_packing_flag()
///
/// @brief pH-sensitive side chain packing on/off
///
/// @detailed
///
/// @return
///
/// @global_read
/// pH
///
/// @global_write
/// pH
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/31/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_pH_packing_flag( bool true_false )
{
	param_pack::packer_logical::pH_rot = true_false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_current_pH()
///
/// @brief Return the current pH value
///
/// @detailed
///
/// @return
///
/// @global_read
/// pH
///
/// @global_write
///
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/31/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
get_current_pH()
{
//	if ( param_pack::packer_logical::pH_rot ); // JSS this did nothing
	return pH_ns::pH;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_dGprotonation_table()
///
/// @brief Calculates the chemical protonation potential of an amino acid, then
/// prints it to a table.
///
/// @detailed
///
/// @return
///
/// @global_read
/// dGprotonation_table()
///
/// @global_write
/// dGprotonation_table()
///
/// @remarks
/// dG(protonation) = RT(pH - pKa) --> dG(deprotonation) = -RT(pH - pKa)
/// 1.36 = 2.3RT = 2.3*0.001987*298 (kcal/mol)
/// 1.42 = 2.3RT = 2.3*0.001987*310 (kcal/mol)
///
/// JSS This should not be a separate function because it
/// depends on the variant numbers coming from create_pH_variants()
///
/// @references
///
/// @authors Ryan Harrison 05/31/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_dGprotonation_table()
{
	using namespace LookUp_aav_pH;
	using namespace param_aa;
	using namespace param_pack;
	using namespace pH_ns;

	if (! get_pH_packing_flag() ) return;

		get_aa_pKa();

		int aavbase  = 1;
		int his_taut = 2;

		
		dGprotonation_table( aa_asp, aavbase )              = -1.36*( pH - aa_pKa( aa_asp ) ); // Original Variant - Need two originals to fix degeneracy
		dGprotonation_table( aa_asp, asp_duplicate1     )   = -1.36*( pH - aa_pKa( aa_asp ) ); // Original Variant
		dGprotonation_table( aa_asp, asp_duplicate2     )   = -1.36*( pH - aa_pKa( aa_asp ) ); // Original Variant
		dGprotonation_table( aa_asp, asp_duplicate3     )   = -1.36*( pH - aa_pKa( aa_asp ) ); // Original Variant
		dGprotonation_table( aa_asp, asp_duplicate4     )   = -1.36*( pH - aa_pKa( aa_asp ) ); // Original Variant
		dGprotonation_table( aa_asp, asp_duplicate5     )   = -1.36*( pH - aa_pKa( aa_asp ) ); // Original Variant
		dGprotonation_table( aa_asp, asp_OD1protonated  )   =  1.36*( pH - aa_pKa( aa_asp ) ); // Added 1HD1
		dGprotonation_table( aa_asp, asp_OD2protonated  )   =  1.36*( pH - aa_pKa( aa_asp ) ); // Added 1HD2

		dGprotonation_table( aa_glu, aavbase           )    = -1.36*( pH - aa_pKa( aa_glu ) ); // Original variant - Need two originals to fix degeneracy
		dGprotonation_table( aa_glu, glu_duplicate1    )    = -1.36*( pH - aa_pKa( aa_glu ) ); // Original variant
		dGprotonation_table( aa_glu, glu_duplicate2    )    = -1.36*( pH - aa_pKa( aa_glu ) ); // Original variant
		dGprotonation_table( aa_glu, glu_duplicate3    )    = -1.36*( pH - aa_pKa( aa_glu ) ); // Original variant
		dGprotonation_table( aa_glu, glu_duplicate4    )    = -1.36*( pH - aa_pKa( aa_glu ) ); // Original variant
		dGprotonation_table( aa_glu, glu_duplicate5    )    = -1.36*( pH - aa_pKa( aa_glu ) ); // Original variant
		dGprotonation_table( aa_glu, glu_OE1protonated )    =  1.36*( pH - aa_pKa( aa_glu ) ); // Added 1HE1
		dGprotonation_table( aa_glu, glu_OE2protonated )    =  1.36*( pH - aa_pKa( aa_glu ) ); // Added 1HE2

		dGprotonation_table( aa_his, aavbase )              = -1.36*( pH - aa_pKa( aa_his ) ); // Original variant - flip H position in +1 state
		dGprotonation_table( aa_his, his_taut )             = -1.36*( pH - aa_pKa( aa_his ) ); // alternate histidine tautomer - flip H in +1 positions
		dGprotonation_table( aa_his, his_protonated1 )      =  1.36*( pH - aa_pKa( aa_his ) );  // 1ND, 1NE + 0 state
		dGprotonation_table( aa_his, his_protonated2 )      =  1.36*( pH - aa_pKa( aa_his ) );  // 1ND, 1NE + 0 state

		dGprotonation_table( aa_lys, aavbase )              =  1.36*( pH - aa_pKa( aa_lys ) );  // Original variant
		dGprotonation_table( aa_lys, lys_3HZdeprotonated )  = -1.36*( pH - aa_pKa( aa_lys ) ); // Deleted 3HZ

		dGprotonation_table( aa_arg, aavbase )              =  1.36*( pH - aa_pKa( aa_arg ) );  // Original variant
		dGprotonation_table( aa_arg, arg_1HH2deprotonated ) = -1.36*( pH - aa_pKa( aa_arg ) ); // Deleted 2HH2

		dGprotonation_table( aa_tyr, aavbase )              =  1.36*( pH - aa_pKa( aa_tyr ) );  // Original variant
		dGprotonation_table( aa_tyr, tyr_HHdeprotonated )   = -1.36*( pH - aa_pKa( aa_tyr ) ); // Deleted HH

		//}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin create_pH_variants()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  This routine creates duplicates of the standard variants so
///  they can balance the pKa of the ionized variants.  This is a hack...
/// jss If you fix this, also change calc_dGprotonation_table above.
///
/// Proton Degeneracy for ASP/GLU
///
/// create_pH_asp1() - creates the OD1 variant
/// create_pH_asp2() - creates the OD2 variant
///
/// With this set up there are 2x as many rotamers for the protonated
/// state as the deprotonated state. To correct this over sampling I
/// doubled the number of deprotonated states (create_pH_asp_degeneracy).
///
/// @references
///
/// @authors Ryan Harrison 01/05/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_variants( int aa ) {

	using namespace param_aa;

	if ( !get_pH_packing_flag() ) return;

	if ( aa == aa_asp ) {
		create_pH_asp_duplicate();
		create_pH_asp_duplicate();
		create_pH_asp_duplicate();
		create_pH_asp_duplicate();
		create_pH_asp_duplicate();
		create_pH_asp1();
		create_pH_asp2();
	}
	if ( aa == aa_glu ) {
		create_pH_glu_duplicate();
		create_pH_glu_duplicate();
		create_pH_glu_duplicate();
		create_pH_glu_duplicate();
		create_pH_glu_duplicate();
		create_pH_glu1();
		create_pH_glu2();
	}
	if ( aa == aa_his ) {
		create_pH_his();
		create_pH_his();
	}
	design::design_commands::try_both_his_tautomers = true;
	if ( aa == aa_tyr ) create_pH_tyr();
	if ( aa == aa_lys ) create_pH_lys();
	if ( aa == aa_arg ) create_pH_arg();

	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_asp1()
///
/// @brief Creates a protonated variant of asp (pKa =~ 4.0)
///
/// @detailed
/// Explicitly add H (1HD1)
/// Partial Charges modified to reflect additional proton on sidechain
/// carboxyl. (+0 charge)
/// Partial charges are best guess estimates based on electronegativity
/// Solvation Parameters modified
/// Best guess solvation paramaters based on Lazardis & Karplus
/// See FArray1D_float & fa_lk_dgfree in etable.h for detailed summary of
/// solvation parameters.
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 2 created for asp
///
/// @remarks
/// Add HD1 to OD1
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_asp1()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

	int const var = create_new_variant(aa_asp, 1, aav_pH);
	LookUp_aav_pH::asp_OD1protonated = var;

	int HD1pos = HNpos(aa_asp, var) + 1;

	make_room_for_atom(aa_asp, var, HD1pos);
	atom_name(HD1pos, aa_asp, var) = "1HD1";

//bk   variant2 ---------- ASP

	natoms( aa_asp,var) = 13; // number of total atoms
	nheavyatoms( aa_asp,var) =  8; // number of heavy atoms
	nchi( aa_asp,var) =  3; // number of chi angles

//bk   atom names ASP

	int const Npos    = LookupByName(aa_asp, var, " N  ");
	int const CApos   = LookupByName(aa_asp, var, " CA ");
	int const Cpos    = LookupByName(aa_asp, var, " C  ");
	int const Opos    = LookupByName(aa_asp, var, " O  ");
	int const CBpos   = LookupByName(aa_asp, var, " CB ");
	int const CGpos   = LookupByName(aa_asp, var, " CG ");
	int const OD1pos  = LookupByName(aa_asp, var, " OD1");
	int const OD2pos  = LookupByName(aa_asp, var, " OD2");
	int const Hpos    = LookupByName(aa_asp, var, " H  ");
	//int const HD1pos  = LookupByName(aa_asp, var, "1HD1");
	int const HHApos  = LookupByName(aa_asp, var, " HA ");
	int const HB2pos  = LookupByName(aa_asp, var, "2HB ");
	int const HB3pos  = LookupByName(aa_asp, var, "3HB ");

//jjh intra residue bonding
	nbonded_neighbors( OD1pos, aa_asp, var) = 2; // OD1
		bonded_neighbor(1, OD1pos, aa_asp, var) = CGpos; // OD1--CG
		bonded_neighbor(2, OD1pos, aa_asp, var) = HD1pos;//OD1--1HD1
	nbonded_neighbors( HD1pos, aa_asp, var) = 1; // 1HD1
		bonded_neighbor(1, HD1pos, aa_asp, var) = OD1pos; //1HD1--OD1

//jjh get the angles automatically
	get_angles_from_bonds(aa_asp, var);

//chu   base atom numbers ASP
	atom_base(HD1pos, aa_asp,var) = OD1pos; //  1HD1-> OD1

//bk   atom type numbers ASP
	fullatom_type( OD1pos, aa_asp,var) = 13; // OH     OD1
	fullatom_type( OD2pos, aa_asp,var) = 15; // OOC    OD2
	fullatom_type( HD1pos, aa_asp,var) = 22; // Hpol  1HD1 22

//cmd  atomic charge ASP
//rh From CHARMM22 Residue Topology File
//rh MacKrell Lab (UM Baltimore, July 2004)
	atomic_charge( Npos,   aa_asp,var) = -0.47; //    N
	atomic_charge( CApos,  aa_asp,var) =  0.07; //    CA
	atomic_charge( Cpos,   aa_asp,var) =  0.51; //    C
	atomic_charge( Opos,   aa_asp,var) = -0.51; //    O
	atomic_charge( CBpos,  aa_asp,var) = -0.21; //    CB
	atomic_charge( CGpos,  aa_asp,var) =  0.75; //    CG
	atomic_charge( OD1pos, aa_asp,var) = -0.61; //    OD1
	atomic_charge( OD2pos, aa_asp,var) = -0.55; //    OD2
	atomic_charge( Hpos,   aa_asp,var) =  0.31; //    H
	atomic_charge( HD1pos, aa_asp,var) =  0.44; //    1HD1
	atomic_charge( HHApos, aa_asp,var) =  0.09; //    HA
	atomic_charge( HB2pos, aa_asp,var) =  0.09; //   2HB
	atomic_charge( HB3pos, aa_asp,var) =  0.09; //   3HB

//rh Ryan's best guess partial charges based on electronegativity.
//	atomic_charge( 1, aa_asp,var) = -0.47; //    N
//	atomic_charge( 2, aa_asp,var) =  0.07; //    CA
//	atomic_charge( 3, aa_asp,var) =  0.51; //    C
//	atomic_charge( 4, aa_asp,var) = -0.51; //    O
//	atomic_charge( 5, aa_asp,var) = -0.28; //    CB
//	atomic_charge( 6, aa_asp,var) =  0.70; //    CG
//	atomic_charge( 7, aa_asp,var) = -0.38; //    OD1
//	atomic_charge( 8, aa_asp,var) = -0.53; //    OD2
//	atomic_charge( 9, aa_asp,var) =  0.31; //    H
//	atomic_charge(10, aa_asp,var) =  0.31; //    1HD1
//	atomic_charge(11, aa_asp,var) =  0.09; //    HA
//	atomic_charge(12, aa_asp,var) =  0.09; //   2HB
//	atomic_charge(13, aa_asp,var) =  0.09; //   3HB

	assert_total_charge( aa_asp, var, 0.0);

//bk   template for building  1HD1
	ta(1,HD1pos, aa_asp,var) =  OD1pos; //  OD1
	ta(2,HD1pos, aa_asp,var) =  CGpos; //   CG
	ta(3,HD1pos, aa_asp,var) =  CBpos; //   CB

//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3, aa_asp, var) =  CBpos; //   CB
	chi_atoms( 2, 3, aa_asp, var) =  CGpos; //   CG
	chi_atoms( 3, 3, aa_asp, var) =  OD1pos; //   OD1
	chi_atoms( 4, 3, aa_asp, var) =  HD1pos; //  1HD1

//bk   chi angles required to build atoms ASP
//bk   chi angles needed for building  1HD1
	chi_required(1, HD1pos, aa_asp,var) =  true;
	chi_required(2, HD1pos, aa_asp,var) =  true;
	chi_required(3, HD1pos, aa_asp,var) =  true;
	chi_required(4, HD1pos, aa_asp,var) =  false;

	HNpos( aa_asp,var) = Hpos; // atom numbers forbackbone HN
	HApos( aa_asp,var) = HHApos; // atom numbers forbackbone HA
	nH_polar( aa_asp,var) =  2; // number of polar hydrogens
	nH_aromatic( aa_asp,var) =  0; // number of aromatic hydrogens
	nH_apolar( aa_asp,var) =  3; // number of apolar hydrogens
	nacceptors( aa_asp,var) =  3; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_asp,var) =  Hpos; //  H
	Hpos_polar( 2, aa_asp,var) =  HD1pos; //  1HD1

//bk   acceptor positions
	accpt_pos( 1, aa_asp, var) =  Opos; //  O
	accpt_pos( 2, aa_asp, var) =  OD1pos; //  OD1
	accpt_pos( 3, aa_asp, var) =  OD2pos; //  OD2

//bk   acceptor base2 positions
	abase2( 7, aa_asp, var) =  HD1pos; // HD1 base2 for  OD1 is the only change

//bk   hydrogens on each atom

	nhydrogens_on_atm( OD1pos, aa_asp, var) =  1; //  OD1
  hydrogens_on_atm( 1, OD1pos, aa_asp, var) = HD1pos; //  1HD1

//bk   template coordinates for the amino acid
	//rh COO-H Bond Legnth: 0.96
	icoor( 1,HD1pos, aa_asp, var) =   -2.8450; //  1HD1
	icoor( 2,HD1pos, aa_asp, var) =    9.1410; //  1HD1
	icoor( 3,HD1pos, aa_asp, var) =    9.7720; //  1HD1

	//rh COO-H Bond Legnth: 1.09
	//icoor( 1,HD1pos, aa_asp, var) =   -1.9170; //  1HD1
	//icoor( 2,HD1pos, aa_asp, var) =   10.4180; //  1HD1
	//icoor( 3,HD1pos, aa_asp, var) =   10.6070; //  1HD1
}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_asp2()
///
/// @brief Creates a protonated variant of asp (pKa =~ 4.0)
///
/// @detailed
/// Explicitly add H (1HD2)
/// Partial Charges modified to reflect additional proton on sidechain
/// carboxyl. (+0 charge)
/// Partial charges are best guess estimates based on electronegativity
/// Solvation Parameters modified
/// Best guess solvation paramaters based on Lazardis & Karplus
/// See FArray1D_float & fa_lk_dgfree in etable.h for detailed summary of
/// solvation parameters.
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 2 created for asp
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_asp2()
{
	using namespace aaproperties_pack;
	using namespace param_aa;

	int const var = create_new_variant(aa_asp, 1, aav_pH);
	LookUp_aav_pH::asp_OD2protonated = var;

	int HD2pos = HNpos(aa_asp, var) + 1;

	make_room_for_atom(aa_asp, var, HD2pos);
	atom_name(HD2pos, aa_asp, var) = "1HD2";

//bk   variant2 ---------- ASP

	natoms( aa_asp,var) = 13; // number of total atoms
	nheavyatoms( aa_asp,var) =  8; // number of heavy atoms
	nchi( aa_asp,var) =  3; // number of chi angles

//bk   atom names ASP

	int const Npos    = LookupByName(aa_asp, var, " N  ");
	int const CApos   = LookupByName(aa_asp, var, " CA ");
	int const Cpos    = LookupByName(aa_asp, var, " C  ");
	int const Opos    = LookupByName(aa_asp, var, " O  ");
	int const CBpos   = LookupByName(aa_asp, var, " CB ");
	int const CGpos   = LookupByName(aa_asp, var, " CG ");
	int const OD1pos  = LookupByName(aa_asp, var, " OD1");
	int const OD2pos  = LookupByName(aa_asp, var, " OD2");
	int const Hpos    = LookupByName(aa_asp, var, " H  ");
	//int const HD2pos  = LookupByName(aa_asp, var, "1HD2");
	int const HHApos  = LookupByName(aa_asp, var, " HA ");
	int const HB2pos  = LookupByName(aa_asp, var, "2HB ");
	int const HB3pos  = LookupByName(aa_asp, var, "3HB ");

//jjh intra residue bonding
	nbonded_neighbors( OD2pos, aa_asp, var) = 2; // OD2
		bonded_neighbor(1, OD2pos, aa_asp, var) = CGpos; // OD2--CG
		bonded_neighbor(2, OD2pos, aa_asp, var) = HD2pos;//OD2--1HD2
	nbonded_neighbors( HD2pos, aa_asp, var) = 1; // 1HD2
		bonded_neighbor(1, HD2pos, aa_asp, var) = OD2pos; //1HD2--OD2

//jjh get the angles automatically
	get_angles_from_bonds(aa_asp, var);

//chu   base atom numbers ASP
	atom_base(HD2pos, aa_asp,var) = OD2pos; //  1HD2-> OD2

//bk   atom type numbers ASP
	fullatom_type( OD1pos, aa_asp,var) = 15; // OOC    OD1
	fullatom_type( OD2pos, aa_asp,var) = 13; // OH     OD2
	fullatom_type( HD2pos, aa_asp,var) = 22; // Hpol  1HD2 22

//cmd  atomic charge ASP
//rh From CHARMM22 Residue Topology File
//rh MacKrell Lab (UM Baltimore, July 2004)
	atomic_charge( Npos, aa_asp,var) = -0.47; //    N
	atomic_charge( CApos, aa_asp,var) =  0.07; //    CA
	atomic_charge( Cpos, aa_asp,var) =  0.51; //    C
	atomic_charge( Opos, aa_asp,var) = -0.51; //    O
	atomic_charge( CBpos, aa_asp,var) = -0.21; //    CB
	atomic_charge( CGpos, aa_asp,var) =  0.75; //    CG
	atomic_charge( OD1pos, aa_asp,var) = -0.55; //    OD1
	atomic_charge( OD2pos, aa_asp,var) = -0.61; //    OD2
	atomic_charge( Hpos, aa_asp,var) =  0.31; //    H
	atomic_charge( HD2pos, aa_asp,var) =  0.44; //    1HD2
	atomic_charge( HHApos, aa_asp,var) =  0.09; //    HA
	atomic_charge( HB2pos, aa_asp,var) =  0.09; //   2HB
	atomic_charge( HB3pos, aa_asp,var) =  0.09; //   3HB

//rh Ryan's best guess partial charges based on electronegativity.
//	atomic_charge( 1, aa_asp,var) = -0.47; //    N
//	atomic_charge( 2, aa_asp,var) =  0.07; //    CA
//	atomic_charge( 3, aa_asp,var) =  0.51; //    C
//	atomic_charge( 4, aa_asp,var) = -0.51; //    O
//	atomic_charge( 5, aa_asp,var) = -0.28; //    CB
//	atomic_charge( 6, aa_asp,var) =  0.70; //    CG
//	atomic_charge( 7, aa_asp,var) = -0.38; //    OD1
//	atomic_charge( 8, aa_asp,var) = -0.53; //    OD2
//	atomic_charge( 9, aa_asp,var) =  0.31; //    H
//	atomic_charge(10, aa_asp,var) =  0.31; //    1HD2
//	atomic_charge(11, aa_asp,var) =  0.09; //    HA
//	atomic_charge(12, aa_asp,var) =  0.09; //   2HB
//	atomic_charge(13, aa_asp,var) =  0.09; //   3HB

	assert_total_charge( aa_asp, var, 0.0);

//bk   template for building  1HD2
	ta(1,HD2pos, aa_asp,var) =  OD2pos; //  OD2
	ta(2,HD2pos, aa_asp,var) =  CGpos;  //  CG
	ta(3,HD2pos, aa_asp,var) =  CBpos;  //  CB

//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3, aa_asp, var) =  CBpos; //   CB
	chi_atoms( 2, 3, aa_asp, var) =  CGpos; //   CG
	chi_atoms( 3, 3, aa_asp, var) =  OD2pos; //   OD2
	chi_atoms( 4, 3, aa_asp, var) =  HD2pos; //  1HD2

//bk   chi angles required to build atoms ASP
//bk   chi angles needed for building  1HD2
	chi_required(1, HD2pos, aa_asp,var) =  true;
	chi_required(2, HD2pos, aa_asp,var) =  true;
	chi_required(3, HD2pos, aa_asp,var) =  true;
	chi_required(4, HD2pos, aa_asp,var) =  false;

	HNpos( aa_asp,var) = Hpos; // atom numbers forbackbone HN
	HApos( aa_asp,var) = HHApos; // atom numbers forbackbone HA
	nH_polar( aa_asp,var) =  2; // number of polar hydrogens
	nH_aromatic( aa_asp,var) =  0; // number of aromatic hydrogens
	nH_apolar( aa_asp,var) =  3; // number of apolar hydrogens
	nacceptors( aa_asp,var) =  3; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_asp,var) =  Hpos; //  H
	Hpos_polar( 2, aa_asp,var) =  HD2pos; //  1HD2

//bk   acceptor positions
	accpt_pos( 1, aa_asp, var) =  Opos; //  O
	accpt_pos( 2, aa_asp, var) =  OD1pos; //  OD1
	accpt_pos( 3, aa_asp, var) =  OD2pos; //  OD2

//bk   acceptor base2 positions
	abase2( 8, aa_asp, var) =  HD2pos; //  HD2  base2 for  OD2 is only change

//bk   hydrogens on each atom
	nhydrogens_on_atm( OD2pos, aa_asp, var) =  1; //  OD2
  hydrogens_on_atm( 1, OD2pos, aa_asp, var) = HD2pos; //  1HD2

//bk   template coordinates for the amino acid
	//rh COO-H Bond Legnth: 0.96
	icoor( 1,HD2pos, aa_asp, var) =   -0.2570; //  1HD2
	icoor( 2,HD2pos, aa_asp, var) =    8.0290; //  1HD2
	icoor( 3,HD2pos, aa_asp, var) =   12.0200; //  1HD2

	//rh COO-H Bond Legnth: 0.9
	//icoor( 1,HD2pos, aa_asp, var) =   -0.6040; //  1HD2
	//icoor( 2,HD2pos, aa_asp, var) =   9.6720; //  1HD2
	//icoor( 3,HD2pos, aa_asp, var) =   12.1390; //  1HD2
}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_asp_duplicate()
///
/// @brief Creates a de-protonated variant of asp (pKa =~ 4.0)
///        This additional variant equalizes the degeneracy between protonated
///        and deprotonated states (Added 09-28-06)
///
/// @detailed
///
/// @return
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 09/28/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
create_pH_asp_duplicate(){
	using namespace aaproperties_pack;
	using namespace LookUp_aav_pH;
	using namespace param_aa;

	int const var = create_new_variant(aa_asp, 1, aav_base);

	//rh Create 5 duplicate ASP
	if ( asp_duplicate1 == 0 ) {
		asp_duplicate1 = var;
	} else if ( asp_duplicate2 == 0 ) {
		asp_duplicate2 = var;
	} else if ( asp_duplicate3 == 0 ) {
		asp_duplicate3 = var;
	} else if ( asp_duplicate4 == 0 ) {
		asp_duplicate4 = var;
	} else if ( asp_duplicate5 == 0 ) {
		asp_duplicate5 = var;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_glu1()
///
/// @brief Creates a protonated variant of glu (pKa =~ 4.4)
///
/// @detailed
/// Explicitly add H (1HE1)
/// Partial Charges modified to reflect additional proton on sidechain
/// carboxyl. (+0 charge)
/// Solvation Parameters modified
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 2 created for glu
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_glu1()
{

	using namespace aaproperties_pack;
	using namespace param_aa;

	int var =	create_new_variant(aa_glu, 1, aav_pH);
	LookUp_aav_pH::glu_OE1protonated = var;

	make_room_for_atom(aa_glu, var, 11);

//bk   variant2 ---------- GLU

	natoms( aa_glu,var) = 16; // number of total atoms
	nheavyatoms( aa_glu,var) =  9; // number of heavy atoms
	nchi( aa_glu,var) =  4; // number of chi angles

//bk   atom names GLU
	atom_name(11, aa_glu,var) = "1HE1";

//jjh intra residue bonding
	nbonded_neighbors( 8, aa_glu, var) = 2; // OE1
		bonded_neighbor(1, 8, aa_glu, var) = 7; // OE1--CD
		bonded_neighbor(2, 8, aa_glu, var) = 11; // OE1--1HE1
	nbonded_neighbors( 11, aa_glu, var) = 1; // 1HE1
		bonded_neighbor(1, 11, aa_glu, var) = 8; //1HE1--OE1

//jjh get the angles automatically
	get_angles_from_bonds(aa_glu, var);

//chu   base atom numbers GLU
	atom_base(11, aa_glu,var) = 8; //  1HE1 -> OE1

//bk   atom type numbers GLU
	fullatom_type( 8, aa_glu,var) = 13; // OH     OE1
	fullatom_type( 9, aa_glu,var) = 15; // OOC    OE2
	fullatom_type(11, aa_glu,var) = 22; // Hpol   1HE1

//cmd  atomic charge GLU
//rh From CHARMM22 Residue Topology File
//rh MacKrell Lab (UM Baltimore, July 2004)
	atomic_charge( 1, aa_glu,var) = -0.47; //    N
	atomic_charge( 2, aa_glu,var) =  0.07; //    CA
	atomic_charge( 3, aa_glu,var) =  0.51; //    C
	atomic_charge( 4, aa_glu,var) = -0.51; //    O
	atomic_charge( 5, aa_glu,var) = -0.18; //    CB
	atomic_charge( 6, aa_glu,var) = -0.21; //    CG
	atomic_charge( 7, aa_glu,var) =  0.75; //    CD
	atomic_charge( 8, aa_glu,var) = -0.61; //    OG1
	atomic_charge( 9, aa_glu,var) = -0.55; //    OG2
	atomic_charge(10, aa_glu,var) =  0.31; //    H
	atomic_charge(11, aa_glu,var) =  0.44; //   1HE1
	atomic_charge(12, aa_glu,var) =  0.09; //    HA
	atomic_charge(13, aa_glu,var) =  0.09; //   2HB
	atomic_charge(14, aa_glu,var) =  0.09; //   3HB
	atomic_charge(15, aa_glu,var) =  0.09; //   2HG
	atomic_charge(16, aa_glu,var) =  0.09; //   3HG

//rh Ryan's best guess partial charges based on electronegativity.
//	atomic_charge( 1, aa_glu,var) = -0.47; //    N
//	atomic_charge( 2, aa_glu,var) =  0.07; //    CA
//	atomic_charge( 3, aa_glu,var) =  0.51; //    C
//	atomic_charge( 4, aa_glu,var) = -0.51; //    O
//	atomic_charge( 5, aa_glu,var) = -0.18; //    CB
//	atomic_charge( 6, aa_glu,var) = -0.28; //    CG
//	atomic_charge( 7, aa_glu,var) =  0.70; //    CD
//	atomic_charge( 8, aa_glu,var) = -0.38; //    OG1
//	atomic_charge( 9, aa_glu,var) = -0.53; //    OG2
//	atomic_charge(10, aa_glu,var) =  0.31; //    H
//	atomic_charge(11, aa_glu,var) =  0.31; //   1HE1
//	atomic_charge(12, aa_glu,var) =  0.09; //    HA
//	atomic_charge(13, aa_glu,var) =  0.09; //   2HB
//	atomic_charge(14, aa_glu,var) =  0.09; //   3HB
//	atomic_charge(15, aa_glu,var) =  0.09; //   2HG
//	atomic_charge(16, aa_glu,var) =  0.09; //   3HG

  assert_total_charge( aa_glu, var, 0.0);

//bk   template atoms used for placing atoms GLU

//bk   template for building 1HE1
	ta(1,11, aa_glu,var) =  8; //   OG1
	ta(2,11, aa_glu,var) =  7; //   CD
	ta(3,11, aa_glu,var) =  6; //   CG

//bk   chi angles required to build atoms GLU
//bk   chi angles needsed for building  1HE1
	chi_required(1, 11, aa_glu,var) =  true;
	chi_required(2, 11, aa_glu,var) =  true;
	chi_required(3, 11, aa_glu,var) =  true;
	chi_required(4, 11, aa_glu,var) =  true;
	// FIX RMH: Should there be 4 chi angles to represent polar hydrogen degrees of freedom.
	// FIX RMH: Does optimize hydrogen, or rotate_proton take care of this??

//bk   four atoms that define chi angle  4
	chi_atoms( 1, 4, aa_glu, var) =  6; //   CG
	chi_atoms( 2, 4, aa_glu, var) =  7; //   CD
	chi_atoms( 3, 4, aa_glu, var) =  8; //   OE1
	chi_atoms( 4, 4, aa_glu, var) =  11; //  1HE1

	HNpos( aa_glu,var) = 10; // atom numbers forbackbone HN
	HApos( aa_glu,var) = 12; // atom numbers forbackbone HA
	nH_polar( aa_glu,var) =  2; // number of polar hydrogens
	nH_aromatic( aa_glu,var) =  0; // number of aromatic hydrogens
	nH_apolar( aa_glu,var) =  5; // number of apolar hydrogens
	nacceptors( aa_glu,var) =  3; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_glu,var) = 10; //  H
	Hpos_polar( 2, aa_glu,var) = 11; //  1HG1

//bk   atom number for aromatic hydrogens

//bk   acceptor positions
// FIX RMH: Should this still be an acceptor - a WEAK acceptor?
// FIX RMH: Only accepting at one of two positions instead of 3 positions.
// FIX RMH: Look at how hydrogen is handled for proton OH on Serine
// FIX RMH: Look in dunbrack and hydrogen functions
// FIX RMH: Protonation for terminii??? (Not currently in)
	accpt_pos( 1, aa_glu, var) =  4; //  O
	accpt_pos( 2, aa_glu, var) =  8; //  OE1
	accpt_pos( 3, aa_glu, var) =  9; //  OE2

//bk   acceptor base2 positions
	abase2( 8, aa_glu, var) = 11; // 1HE1 base2 for  OE1 is only change

//bk   hydrogens on each atom
	nhydrogens_on_atm( 8, aa_glu, var) =  1; //  OG1
  hydrogens_on_atm( 1, 8, aa_glu, var) = 11; // 1HE1

//bk   template coordinates for the amino acid
	//rh COO-H Bond Legnth: 0.96
	icoor( 1,11, aa_glu, var) =    2.1140; //  1HE1
	icoor( 2,11, aa_glu, var) =    5.6400; //  1HE1
	icoor( 3,11, aa_glu, var) =   18.3500; //  1HE1

	//rh COO-H Bond Legnth: 0.9
	//icoor( 1,11, aa_glu, var) =    2.9940; //  1HE1
	//icoor( 2,11, aa_glu, var) =    4.3550; //  1HE1
	//icoor( 3,11, aa_glu, var) =   17.8190; //  1HE1
}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_glu2()
///
/// @brief Creates a protonated variant of glu (pKa =~ 4.4)
///
/// @detailed
/// Explicitly add H (1HE2)
/// Partial Charges modified to reflect additional proton on sidechain
/// carboxyl. (+0 charge)
/// Solvation Parameters modified
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 2 created for glu
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_glu2()
{

	using namespace aaproperties_pack;
	using namespace param_aa;

	int var =	create_new_variant(aa_glu, 1, aav_pH);
	LookUp_aav_pH::glu_OE2protonated = var;

	int HE2pos = HNpos(aa_glu, var) + 1;

	make_room_for_atom(aa_glu, var, HE2pos);
	atom_name(HE2pos, aa_glu, var) = "1HE2";

//bk   variant2 ---------- GLU

	natoms( aa_glu,var) = 16; // number of total atoms
	nheavyatoms( aa_glu,var) =  9; // number of heavy atoms
	nchi( aa_glu,var) =  4; // number of chi angles

//bk   atom names GLU

	int const Npos    = LookupByName(aa_glu, var, " N  ");
	int const CApos   = LookupByName(aa_glu, var, " CA ");
	int const Cpos    = LookupByName(aa_glu, var, " C  ");
	int const Opos    = LookupByName(aa_glu, var, " O  ");
	int const CBpos   = LookupByName(aa_glu, var, " CB ");
	int const CGpos   = LookupByName(aa_glu, var, " CG ");
	int const CDpos   = LookupByName(aa_glu, var, " CD ");
	int const OE1pos  = LookupByName(aa_glu, var, " OE1");
	int const OE2pos  = LookupByName(aa_glu, var, " OE2");
	int const Hpos    = LookupByName(aa_glu, var, " H  ");
	//int const HE2pos  = LookupByName(aa_glu, var, "1HE2");
	int const HHApos  = LookupByName(aa_glu, var, " HA ");
	int const HB2pos  = LookupByName(aa_glu, var, "2HB ");
	int const HB3pos  = LookupByName(aa_glu, var, "3HB ");
	int const HG2pos  = LookupByName(aa_glu, var, "2HG ");
	int const HG3pos  = LookupByName(aa_glu, var, "3HG ");

//jjh intra residue bonding
	nbonded_neighbors( OE2pos, aa_glu, var) = 2; // OE2
		bonded_neighbor(1, OE2pos, aa_glu, var) = CDpos; // OE2--CD
		bonded_neighbor(2, OE2pos, aa_glu, var) = HE2pos;//OE2--1HE2
	nbonded_neighbors( HE2pos, aa_glu, var) = 1; // 1HE2
		bonded_neighbor(1, HE2pos, aa_glu, var) = OE2pos; //1HE2--OE2

//jjh get the angles automatically
	get_angles_from_bonds(aa_glu, var);

//chu   base atom numbers GLU
	atom_base(HE2pos, aa_glu,var) = OE2pos; //  1HE2-> OE2

//bk   atom type numbers GLU
	fullatom_type( OE1pos, aa_glu,var) = 15; // OOC    OE1
	fullatom_type( OE2pos, aa_glu,var) = 13; // OH     OE2
	fullatom_type( HE2pos, aa_glu,var) = 22; // Hpol  1HE2

//cmd  atomic charge GLU
//rh From CHARMM22 Residue Topology File
//rh MacKrell Lab (UM Baltimore, July 2004)
	atomic_charge(Npos, aa_glu,var)   = -0.47; //    N
	atomic_charge(CApos, aa_glu,var)  =  0.07; //    CA
	atomic_charge(Cpos, aa_glu,var)   =  0.51; //    C
	atomic_charge(Opos, aa_glu,var)   = -0.51; //    O
	atomic_charge(CBpos, aa_glu,var)  = -0.18; //    CB
	atomic_charge(CGpos, aa_glu,var)  = -0.21; //    CG
	atomic_charge(CDpos, aa_glu,var)  =  0.75; //    CD
	atomic_charge(OE1pos, aa_glu,var) = -0.55; //    OE1
	atomic_charge(OE2pos, aa_glu,var) = -0.61; //    OE2
	atomic_charge(Hpos, aa_glu,var)   =  0.31; //    H
	atomic_charge(HE2pos, aa_glu,var) =  0.44; //   1HE2
	atomic_charge(HHApos, aa_glu,var) =  0.09; //    HA
	atomic_charge(HB2pos, aa_glu,var) =  0.09; //   2HB
	atomic_charge(HB3pos, aa_glu,var) =  0.09; //   3HB
	atomic_charge(HG2pos, aa_glu,var) =  0.09; //   2HG
	atomic_charge(HG3pos, aa_glu,var) =  0.09; //   3HG

//rh Ryan's best guess partial charges based on electronegativity.
//	atomic_charge( 1, aa_glu,var) = -0.47; //    N
//	atomic_charge( 2, aa_glu,var) =  0.07; //    CA
//	atomic_charge( 3, aa_glu,var) =  0.51; //    C
//	atomic_charge( 4, aa_glu,var) = -0.51; //    O
//	atomic_charge( 5, aa_glu,var) = -0.18; //    CB
//	atomic_charge( 6, aa_glu,var) = -0.28; //    CG
//	atomic_charge( 7, aa_glu,var) =  0.70; //    CD
//	atomic_charge( 8, aa_glu,var) = -0.38; //    OG1
//	atomic_charge( 9, aa_glu,var) = -0.53; //    OG2
//	atomic_charge(10, aa_glu,var) =  0.31; //    H
//	atomic_charge(11, aa_glu,var) =  0.31; //   1HE2
//	atomic_charge(12, aa_glu,var) =  0.09; //    HA
//	atomic_charge(13, aa_glu,var) =  0.09; //   2HB
//	atomic_charge(14, aa_glu,var) =  0.09; //   3HB
//	atomic_charge(15, aa_glu,var) =  0.09; //   2HG
//	atomic_charge(16, aa_glu,var) =  0.09; //   3HG

  assert_total_charge( aa_glu, var, 0.0);

//bk   template for building  1HE2
	ta(1,HE2pos, aa_glu,var) =  OE2pos; //  OE2
	ta(2,HE2pos, aa_glu,var) =  CDpos;  //  CG
	ta(3,HE2pos, aa_glu,var) =  CGpos;  //  CB

//bk   four atoms that define chi angle  4
	chi_atoms( 1, 4, aa_glu, var) =  CGpos; //   CG
	chi_atoms( 2, 4, aa_glu, var) =  CDpos; //   CD
	chi_atoms( 3, 4, aa_glu, var) =  OE2pos; //   OE2
	chi_atoms( 4, 4, aa_glu, var) =  HE2pos; //  1HE2

//bk   chi angles required to build atoms GLU
//bk   chi angles needed for building  1HE2
	chi_required(1, HE2pos, aa_glu,var) =  true;
	chi_required(2, HE2pos, aa_glu,var) =  true;
	chi_required(3, HE2pos, aa_glu,var) =  true;
	chi_required(4, HE2pos, aa_glu,var) =  true;
	// FIX RMH: Should there be 4 chi angles to represent polar hydrogen degrees of freedom.
	// FIX RMH: Does optimize hydrogen, or rotate_proton take care of this??

	HNpos( aa_glu,var) = Hpos;   // atom numbers forbackbone HN
	HApos( aa_glu,var) = HHApos; // atom numbers forbackbone HA

	nH_polar( aa_glu,var) =  2; // number of polar hydrogens
	nH_aromatic( aa_glu,var) =  0; // number of aromatic hydrogens
	nH_apolar( aa_glu,var) =  5; // number of apolar hydrogens
	nacceptors( aa_glu,var) =  3; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_glu,var) =  Hpos;   //  H
	Hpos_polar( 2, aa_glu,var) =  HE2pos; //  1HE2

//bk   acceptor positions
	accpt_pos( 1, aa_glu, var) =  Opos; //  O
	accpt_pos( 2, aa_glu, var) =  OE1pos; //  OE1
	accpt_pos( 3, aa_glu, var) =  OE2pos; //  OE2

//bk   acceptor base2 positions
	abase2( OE2pos, aa_glu, var) =  HE2pos; //  1HE2  base2 for  OE2 is only change

//bk   hydrogens on each atom
	nhydrogens_on_atm( OE2pos, aa_glu, var) =  1; //  OE2
	 hydrogens_on_atm( 1, OE2pos, aa_glu, var) = HE2pos; //  1HE2

//bk   template coordinates for the amino acid
	//rh COO-H Bond Legnth: 0.96
	icoor( 1,11, aa_glu, var) =    3.2790; //  1HE2
	icoor( 2,11, aa_glu, var) =    5.5680; //  1HE2
	icoor( 3,11, aa_glu, var) =   14.9240; //  1HE2

//rh COO-H Bond Legnth: 0.96
	//icoor( 1,11, aa_glu, var) =    2.7320; //  1HE2
	//icoor( 2,11, aa_glu, var) =    4.1550; //  1HE2
	//icoor( 3,11, aa_glu, var) =   15.3280; //  1HE2

}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_glu_duplicate()
///
/// @brief Creates a de-protonated variant of glu (pKa =~ 4.4)
///        This additional variant equalizes the degeneracy between protonated
///        and deprotonated states (Added 09-28-06)
///
/// @detailed
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 4 created for glu
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 09/28/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
create_pH_glu_duplicate(){
	using namespace aaproperties_pack;
	using namespace LookUp_aav_pH;
	using namespace param_aa;

	int const var = create_new_variant(aa_glu, 1, aav_base);
	
	//rh Create 5 duplicate GLU
	if ( glu_duplicate1 == 0 ) {
		glu_duplicate1 = var;
	} else if ( glu_duplicate2 == 0 ) {
		glu_duplicate2 = var;
	} else if ( glu_duplicate3 == 0 ) {
		glu_duplicate3 = var;
	} else if ( glu_duplicate4 == 0 ) {
		glu_duplicate4 = var;
	} else if ( glu_duplicate5 == 0 ) {
		glu_duplicate5 = var;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_his()
///
/// @brief Creates a protonated variant of HIS (pKa =~ 6.3)
///
/// @detailed
/// Explicitly add H (HD1) to the original HIS, which already has HE2.
/// Partial Charges modified to reflect additional proton on ring
/// nitrogen. (+1 charge)
/// Solvation Parameters modified
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant created for his
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_his()
{

	using namespace aaproperties_pack;
	using namespace param_aa;

	int const var = create_new_variant(aa_his, 1, aav_pH);

	//rh First function call is for his_protonated1
	if ( LookUp_aav_pH::his_protonated1 == 0 ) {
		LookUp_aav_pH::his_protonated1 = var;
	}

	//rh If there is a duplicate protonated variant for HIS store in his_protonated2
	if ( var == LookUp_aav_pH::his_protonated1 + 1 && LookUp_aav_pH::his_protonated2 == 0 ) {
		LookUp_aav_pH::his_protonated2 = var;
	}

	make_room_for_atom(aa_his, var, 12);

//bk   variant var---------- HIS

	natoms( aa_his,var) = 18; // number of total atoms
	nheavyatoms( aa_his,var) = 10; // number of heavy atoms
	nchi( aa_his,var) =  2; // number of chi angles

//bk   atom names HIS
	atom_name(12, aa_his,var) = " HD1";

//jjh intra residue bonding
	nbonded_neighbors( 7, aa_his, var) = 3; // ND1
		bonded_neighbor(1, 7, aa_his, var) = 6; // ND1--CG
		bonded_neighbor(2, 7, aa_his, var) = 9; // ND1--CE1
		bonded_neighbor(3, 7, aa_his, var) =12; // ND1--HD1
	nbonded_neighbors( 12, aa_his, var) = 1; // 1HD1
		bonded_neighbor(1, 12, aa_his, var) = 7; //HD1--ND1

//jjh get the angles automatically
	get_angles_from_bonds(aa_his, var);

//chu   base atom numbers HIS
	atom_base(12, aa_his,var) = 7; //  HD1 -> ND1

//bk   atom type numbers HIS
	fullatom_type( 7, aa_his,var) =  8; // Nhis   ND1
	fullatom_type(12, aa_his,var) = 22; // Hpol   HD1

//cmd  atomic charge HIS
//rh From CHARMM22 Residue Topology File
//rh MacKrell Lab (UM Baltimore, July 2004)
	atomic_charge( 1, aa_his,var) = -0.47; //    N
	atomic_charge( 2, aa_his,var) =  0.07; //    CA
	atomic_charge( 3, aa_his,var) =  0.51; //    C
	atomic_charge( 4, aa_his,var) = -0.51; //    O
	atomic_charge( 5, aa_his,var) = -0.05; //    CB
	atomic_charge( 6, aa_his,var) =  0.19; //    CG
	atomic_charge( 7, aa_his,var) = -0.51; //    ND1
	atomic_charge( 8, aa_his,var) =  0.19; //    CD2
	atomic_charge( 9, aa_his,var) =  0.32; //    CE1
	atomic_charge(10, aa_his,var) = -0.51; //    NE2
	atomic_charge(11, aa_his,var) =  0.31; //    H
	atomic_charge(12, aa_his,var) =  0.44; //    HD1
	atomic_charge(13, aa_his,var) =  0.44; //    HE2
	atomic_charge(14, aa_his,var) =  0.09; //    HA
	atomic_charge(15, aa_his,var) =  0.09; //   2HB
	atomic_charge(16, aa_his,var) =  0.09; //   3HB
	atomic_charge(17, aa_his,var) =  0.18; //    HE1
	atomic_charge(18, aa_his,var) =  0.13; //    HD2

//rh Ryan's best guess partial charges based on electronegativity.
//	atomic_charge( 1, aa_his,var) = -0.47; //    N
//	atomic_charge( 2, aa_his,var) =  0.07; //    CA
//	atomic_charge( 3, aa_his,var) =  0.51; //    C
//	atomic_charge( 4, aa_his,var) = -0.51; //    O
//	atomic_charge( 5, aa_his,var) = -0.09; //    CB
//	atomic_charge( 6, aa_his,var) =  0.35; //    CG
//	atomic_charge( 7, aa_his,var) = -0.36; //    ND1
//	atomic_charge( 8, aa_his,var) =  0.05; //    CD2
//	atomic_charge( 9, aa_his,var) =  0.35; //    CE1
//	atomic_charge(10, aa_his,var) = -0.36; //    NE2
//	atomic_charge(11, aa_his,var) =  0.31; //    H
//	atomic_charge(12, aa_his,var) =  0.33; //    HD1
//	atomic_charge(13, aa_his,var) =  0.33; //    HE2
//	atomic_charge(14, aa_his,var) =  0.09; //    HA
//	atomic_charge(15, aa_his,var) =  0.09; //   2HB
//	atomic_charge(16, aa_his,var) =  0.09; //   3HB
//	atomic_charge(17, aa_his,var) =  0.13; //    HE2
//	atomic_charge(18, aa_his,var) =  0.09; //    HD2

	assert_total_charge( aa_his, var, 1.0);

//bk   template atoms used for placing atoms HIS

//bk   template for building  HD1
	ta(1,12, aa_his,var) =  7; //   ND1
	ta(2,12, aa_his,var) =  6; //   CG
	ta(3,12, aa_his,var) =  9; //   CE1

//bk   chi angles required to build atoms HIS
//bk   chi angles needed for building  HD1
	chi_required(1,12, aa_his,var) =  true;
	chi_required(2,12, aa_his,var) =  true;

	HNpos( aa_his,var) = 11; // atom number for backbone HN
	HApos( aa_his,var) = 14; // atom number for backbone HA
	nH_polar( aa_his,var) =  3; // number of polar hydrogens
	nH_aromatic( aa_his,var) =  0; // number of aromatic hydrogens
	nH_apolar( aa_his,var) =  5; // number of apolar hydrogens
	nacceptors( aa_his,var) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_his,var) = 11; //  H
	Hpos_polar( 2, aa_his,var) = 12; //  HD1
	Hpos_polar( 3, aa_his,var) = 13; //  HE2

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1, aa_his,var) = 14; //  HA
	Hpos_apolar( 2, aa_his,var) = 15; // 2HB
	Hpos_apolar( 3, aa_his,var) = 16; // 3HB
	Hpos_apolar( 4, aa_his,var) = 17; //  HE2
	Hpos_apolar( 5, aa_his,var) = 18; //  HD2

//bk   acceptor positions
	accpt_pos( 1, aa_his, var) =  4; //  O

//bk   acceptor base2 positions

//bk   hydrogens on each atom
	nhydrogens_on_atm(7, aa_his, var) =  1; //  ND1
	hydrogens_on_atm( 1,7, aa_his, var) = 12; //  HD1


//bk   template coordinates for the amino acid
	icoor( 1,12, aa_his, var) =   -3.0550; //   HD1
	icoor( 2,12, aa_his, var) =   16.2160; //   HD1
	icoor( 3,12, aa_his, var) =   22.9340; //   HD1
}


////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_tyr()
///
/// @brief Creates a de-protonated variant of tyr (pKa =~ 9.8)
///
/// @detailed
/// Explicitly remove H (HH)
/// Partial Charges modified to reflect removal of proton. (+0 charge)
/// Solvation Parameters modified
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 2 created for arg
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_tyr()
{

	using namespace aaproperties_pack;
	using namespace param_aa;

	int const var = create_new_variant(aa_tyr, 1, aav_pH);
	LookUp_aav_pH::tyr_HHdeprotonated = var;

	remove_atom(aa_tyr, var, 14);

//bk   variant2 ---------- TYR

	natoms(aa_tyr,var) = 20; // number of total atoms
	nheavyatoms(aa_tyr,var) = 12; // number of heavy atoms
	nchi(aa_tyr,var) =  2; // number of chi angles

//bk   atom type numbers TYR
	fullatom_type(12,aa_tyr,var) = 14; // OH     OH  // Changed from 13  (14 ONH2; 20 OCbb)

	nbonded_neighbors( 12, aa_tyr, var) = 1; // OH
		bonded_neighbor(2,12, aa_tyr, var) =0; // OH--HH
	nbonded_neighbors( 21, aa_tyr, var) = 0;
		bonded_neighbor(1,21, aa_tyr, var) = 0;

//jjh get the angles automatically
	get_angles_from_bonds(aa_tyr, var);

//chu   base atom numbers TYR
	atom_base(21,aa_tyr,var) = 0; // 3HB  on  CB
//bk   atom type numbers TYR
	fullatom_type(21,aa_tyr,var) = 0; // Hapo  3HB


//cmd  atomic charge TYR
	atomic_charge( 1,aa_tyr,var) = -0.47;  //    N
	atomic_charge( 2,aa_tyr,var) =  0.07;  //    CA
	atomic_charge( 3,aa_tyr,var) =  0.51;  //    C
	atomic_charge( 4,aa_tyr,var) = -0.51;  //    O
	atomic_charge( 5,aa_tyr,var) = -0.18;  //    CB
	atomic_charge( 6,aa_tyr,var) =  0.00;  //    CG
	atomic_charge( 7,aa_tyr,var) = -0.115; //    CD1
	atomic_charge( 8,aa_tyr,var) = -0.115; //    CD2
	atomic_charge( 9,aa_tyr,var) = -0.115; //    CE1
	atomic_charge(10,aa_tyr,var) = -0.115; //    CE2
	atomic_charge(11,aa_tyr,var) = -0.150;  //    CZ
	atomic_charge(12,aa_tyr,var) = -0.85;  //    OH
	atomic_charge(13,aa_tyr,var) =  0.31;  //    H
	atomic_charge(14,aa_tyr,var) =  0.115; //    HD1
	atomic_charge(15,aa_tyr,var) =  0.115; //    HE1
	atomic_charge(16,aa_tyr,var) =  0.115; //    HE2
	atomic_charge(17,aa_tyr,var) =  0.115; //    HD2
	atomic_charge(18,aa_tyr,var) =  0.09;  //    HA
	atomic_charge(19,aa_tyr,var) =  0.09;  //   2HB
	atomic_charge(20,aa_tyr,var) =  0.09;  //   3HB

	assert_total_charge( aa_tyr, var, -1.0);

//bk   template for building 3HB
	ta(1,21,aa_tyr,var) =  0; //   CB
	ta(2,21,aa_tyr,var) =  0; //   CA
	ta(3,21,aa_tyr,var) =  0; //   CG

//bk   chi angles needed for building 3HB
	chi_required(1,21,aa_tyr,var) = false;
	chi_required(2,21,aa_tyr,var) = false;
	chi_required(3,21,aa_tyr,var) = false;
	chi_required(4,21,aa_tyr,var) = false;

//bk   four atoms that define chi angle  3
	chi_atoms( 1, 3,aa_tyr, var) = 0; //   CE2
	chi_atoms( 2, 3,aa_tyr, var) = 0; //   CZ
	chi_atoms( 3, 3,aa_tyr, var) = 0; //   OH
	chi_atoms( 4, 3,aa_tyr, var) = 0; //   HH

	HNpos(aa_tyr,var) = 13; // atom number for backbone HN
	HApos(aa_tyr,var) = 18; // atom number for backbone HA
	nH_polar(aa_tyr,var) =  1; // number of polar hydrogens
	nH_aromatic(aa_tyr,var) =  4; // number of aromatic hydrogens
	nH_apolar(aa_tyr,var) =  3; // number of apolar hydrogens
	nacceptors(aa_tyr,var) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_tyr,var) = 13; //  H
	Hpos_polar( 2,aa_tyr,var) = 0; //  HH

//bk   atom number for aromatic hydrogens
	Hpos_aromatic( 1,aa_tyr,var) = 14; //  HD1
	Hpos_aromatic( 2,aa_tyr,var) = 15; //  HE1
	Hpos_aromatic( 3,aa_tyr,var) = 16; //  HE2
	Hpos_aromatic( 4,aa_tyr,var) = 17; //  HD2

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_tyr,var) = 18; //  HA
	Hpos_apolar( 2,aa_tyr,var) = 19; // 2HB
	Hpos_apolar( 3,aa_tyr,var) = 20; // 3HB

//bk   acceptor positions
	accpt_pos( 1,aa_tyr, var) =  4; //  O
	accpt_pos( 2,aa_tyr, var) = 12; //  OH

//bk   acceptor base2 positions
	abase2(12,aa_tyr, var) = atom_base(11, aa_tyr, var); //  get base2 for  OH from CZ
	nhydrogens_on_atm(12,aa_tyr, var) =  0; //  OH
}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_lys()
///
/// @brief Creates a de-protonated variant of lys (pKa =~ 10.4)
///
/// @detailed
// Explicitly remove H (3HZ)
// Partial Charges modified to reflect removal of proton on amine. (+0 charge)
// Solvation Parameters modified
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 2 created for lys
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_lys()
{

	using namespace aaproperties_pack;
	using namespace param_aa;

	int const var = create_new_variant(aa_lys, 1, aav_pH);
	LookUp_aav_pH::lys_3HZdeprotonated = var;

	remove_atom(aa_lys, var, 13);

//bk   variant2 ---------- LYS

	natoms( aa_lys,var) = 21; // number of total atoms
	nheavyatoms( aa_lys,var) =  9; // number of heavy atoms
	nchi( aa_lys,var) =  4; // number of chi angles


//bk   atom type numbers LYS
	fullatom_type( 9, aa_lys, var) =  9; // Nlys   NZ  - changed from 10

	nbonded_neighbors( 9, aa_lys, var) = 3; // NZ
	  bonded_neighbor(4, 9, aa_lys, var) = 0; // NZ--3HZ
	nhydrogens_on_atm( 9, aa_lys, var) =  2; //  NZ
	  hydrogens_on_atm( 3, 9, aa_lys, var) = 0; // 3HZ

//jjh get the angles automatically
	get_angles_from_bonds(aa_lys, var);


//cmd  atomic charge LYS
//rh From CHARMM22 Residue Topology File
//rh MacKrell Lab (UM Baltimore, July 2004)
	atomic_charge( 1, aa_lys,var) = -0.47; //    N
	atomic_charge( 2, aa_lys,var) =  0.07; //    CA
	atomic_charge( 3, aa_lys,var) =  0.51; //    C
	atomic_charge( 4, aa_lys,var) = -0.51; //    O
	atomic_charge( 5, aa_lys,var) = -0.18; //    CB
	atomic_charge( 6, aa_lys,var) = -0.18; //    CG
	atomic_charge( 7, aa_lys,var) = -0.18; //    CD
	atomic_charge( 8, aa_lys,var) =  0.13; //    CE
	atomic_charge( 9, aa_lys,var) = -0.96; //    NZ
	atomic_charge(10, aa_lys,var) =  0.31; //    H
	atomic_charge(11, aa_lys,var) =  0.34; //   1HZ
	atomic_charge(12, aa_lys,var) =  0.34; //   2HZ
	atomic_charge(13, aa_lys,var) =  0.09; //    HA
	atomic_charge(14, aa_lys,var) =  0.09; //   2HB
	atomic_charge(15, aa_lys,var) =  0.09; //   3HB
	atomic_charge(16, aa_lys,var) =  0.09; //   2HG
	atomic_charge(17, aa_lys,var) =  0.09; //   3HG
	atomic_charge(18, aa_lys,var) =  0.09; //   2HD
	atomic_charge(19, aa_lys,var) =  0.09; //   3HD
	atomic_charge(20, aa_lys,var) =  0.075; //   2HE
	atomic_charge(21, aa_lys,var) =  0.075; //   3HE

//rh Ryan's best guess partial charges based on electronegativity.
//	atomic_charge( 1, aa_lys,var) = -0.47; //    N
//	atomic_charge( 2, aa_lys,var) =  0.07; //    CA
//	atomic_charge( 3, aa_lys,var) =  0.51; //    C
//	atomic_charge( 4, aa_lys,var) = -0.51; //    O
//	atomic_charge( 5, aa_lys,var) = -0.18; //    CB
//	atomic_charge( 6, aa_lys,var) = -0.18; //    CG
//	atomic_charge( 7, aa_lys,var) = -0.18; //    CD
//	atomic_charge( 8, aa_lys,var) =  0.10; //    CE
//	atomic_charge( 9, aa_lys,var) = -0.68; //    NZ
//	atomic_charge(10, aa_lys,var) =  0.31; //    H
//	atomic_charge(11, aa_lys,var) =  0.25; //   1HZ
//	atomic_charge(12, aa_lys,var) =  0.25; //   2HZ
//	atomic_charge(13, aa_lys,var) =  0.09; //    HA
//	atomic_charge(14, aa_lys,var) =  0.09; //   2HB
//	atomic_charge(15, aa_lys,var) =  0.09; //   3HB
//	atomic_charge(16, aa_lys,var) =  0.09; //   2HG
//	atomic_charge(17, aa_lys,var) =  0.09; //   3HG
//	atomic_charge(18, aa_lys,var) =  0.09; //   2HD
//	atomic_charge(19, aa_lys,var) =  0.09; //   3HD
//	atomic_charge(20, aa_lys,var) =  0.04; //   2HE
//	atomic_charge(21, aa_lys,var) =  0.04; //   3HE

  assert_total_charge( aa_lys, var, 0.0);

	HNpos( aa_lys,var) = 10; // atom number for backbone HN
	HApos( aa_lys,var) = 13; // atom number for backbone HA
	nH_polar( aa_lys,var) =  3; // number of polar hydrogens
	nH_aromatic( aa_lys,var) =  0; // number of aromatic hydrogens
	nH_apolar( aa_lys,var) =  9; // number of apolar hydrogens
	nacceptors( aa_lys,var) =  2; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1, aa_lys,var) = 10; //  H
	Hpos_polar( 2, aa_lys,var) = 11; // 1HZ
	Hpos_polar( 3, aa_lys,var) = 12; // 2HZ

//bk   acceptor positions
	accpt_pos( 1, aa_lys, var) =  4; //  O
	accpt_pos( 2, aa_lys, var) =  9; //  NZ

//bk   acceptor base2 positions
	abase2( 9, aa_lys, var) =  11; //  CE  base for  NZ

	//icoor( 1,12, aa_lys, var) =   -4.4710; //  2HZ   //rh 2HZ coords for sp2 hybidized NZ..
	//icoor( 2,12, aa_lys, var) =   19.7800; //  2HZ   //rh Almost certain of sp3 hyridization
	//icoor( 3,12, aa_lys, var) =   31.0740; //  2HZ

}

////////////////////////////////////////////////////////////////////////////////
/// @begin  create_pH_arg()
///
/// @brief Creates a de-protonated variant of lys (pKa =~ 12.5)
///
/// @detailed
/// Explicitly remove H (1HH2) from position 15 and slides rest down.
/// Partial Charges modified to reflect removal of proton. (+0 charge)
/// Solvation Parameters modified
///
/// @return
///
///
/// @global_read
///
/// @global_write
/// Variant 2 created for arg
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 05/27/05
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
create_pH_arg()
{

	using namespace aaproperties_pack;
	using namespace param_aa;

	int const var = create_new_variant(aa_arg, 1, aav_pH);
	LookUp_aav_pH::arg_1HH2deprotonated = var;

	remove_atom(aa_arg, var, 15);

//bk   variant2 ---------- ARG

	nvar(aa_arg) = var;
	variant_type( aav_pH, aa_arg, var ) = true; // type of aa variant

	natoms(aa_arg,var) = 23; // number of total atoms
	nheavyatoms(aa_arg,var) = 11; // number of heavy atoms
	nchi(aa_arg,var) =  4; // number of chi angles

//bk   atom type numbers ARG
	fullatom_type( 8,aa_arg,var) = 11; // Narg   NE
	fullatom_type(10,aa_arg,var) = 11; // Narg   NH1
	fullatom_type(11,aa_arg,var) =  9; // Narg   NH2 //Changed from 11

	nbonded_neighbors( 11, aa_arg, var) = 2; // NH2
		bonded_neighbor(2,11, aa_arg, var) =15; // NH2--2HH2 // was 16, but 15 is deleted
		bonded_neighbor(3,11, aa_arg, var) =0; // NH2--1HH2
	nhydrogens_on_atm(11,aa_arg, var) =  1; //  NH2
	  hydrogens_on_atm( 2,11,aa_arg, var) = 0; // 1HH2

	nbonded_neighbors( 24, aa_arg, var) = 0;
		bonded_neighbor(1,24, aa_arg, var) = 0;

//jjh get the angles automatically
	get_angles_from_bonds(aa_arg, var);

	atom_base(24,aa_arg,var) =  0;
	atom_base(14,aa_arg,var) = 10; // 2HH1 on  NH1
	fullatom_type(24,aa_arg,var) = 0; // Hapo  3HD

	ta(1,24,aa_arg,var) =  0; //   CD
	ta(2,24,aa_arg,var) =  0; //   CG
	ta(3,24,aa_arg,var) =  0; //   NE

	chi_required(1,24,aa_arg,var) = false;
	chi_required(2,24,aa_arg,var) = false;
	chi_required(3,24,aa_arg,var) = false;
	chi_required(4,24,aa_arg,var) = false;

//cmd  atomic charge ARG
	atomic_charge( 1,aa_arg,var) = -0.47; //    N
	atomic_charge( 2,aa_arg,var) =  0.07; //    CA
	atomic_charge( 3,aa_arg,var) =  0.51; //    C
	atomic_charge( 4,aa_arg,var) = -0.51; //    O
	atomic_charge( 5,aa_arg,var) = -0.18; //    CB
	atomic_charge( 6,aa_arg,var) = -0.18; //    CG
	atomic_charge( 7,aa_arg,var) =  0.20; //    CD
	atomic_charge( 8,aa_arg,var) = -0.80; //    NE
	atomic_charge( 9,aa_arg,var) =  0.54; //    CZ
	atomic_charge(10,aa_arg,var) = -1.07; //    NH1
	atomic_charge(11,aa_arg,var) = -0.87; //    NH2
	atomic_charge(12,aa_arg,var) =  0.31; //    H
	atomic_charge(13,aa_arg,var) =  0.46; //   1HH1
	atomic_charge(14,aa_arg,var) =  0.46; //   2HH1
	atomic_charge(15,aa_arg,var) =  0.46; //   2HH2
	atomic_charge(16,aa_arg,var) =  0.44; //    HE
	atomic_charge(17,aa_arg,var) =  0.09; //    HA
	atomic_charge(18,aa_arg,var) =  0.09; //   2HB
	atomic_charge(19,aa_arg,var) =  0.09; //   3HB
	atomic_charge(20,aa_arg,var) =  0.09; //   2HG
	atomic_charge(21,aa_arg,var) =  0.09; //   3HG
	atomic_charge(22,aa_arg,var) =  0.09; //   2HD
	atomic_charge(23,aa_arg,var) =  0.09; //   3HD

	assert_total_charge( aa_arg, var, 0.0);

	HNpos(aa_arg,var) = 12; // atom number for backbone HN
	HApos(aa_arg,var) = 17; // atom number for backbone HA
	nH_polar(aa_arg,var) =  5; // number of polar hydrogens
	nH_aromatic(aa_arg,var) =  0; // number of aromatic hydrogens
	nH_apolar(aa_arg,var) =  7; // number of apolar hydrogens
	nacceptors(aa_arg,var) =  1; // number of acceptors

//bk   atom numbers for polar H
	Hpos_polar( 1,aa_arg,var) = 12; //  H
	Hpos_polar( 2,aa_arg,var) = 13; // 1HH1
	Hpos_polar( 3,aa_arg,var) = 14; // 2HH1
	Hpos_polar( 4,aa_arg,var) = 15; // 2HH2
	Hpos_polar( 5,aa_arg,var) = 16; //  HE
	Hpos_polar( 6,aa_arg,var) = 0;

//bk   atom number for aromatic hydrogens

//bk   atom number for apolar hydrogens
	Hpos_apolar( 1,aa_arg,var) = 17; //  HA
	Hpos_apolar( 2,aa_arg,var) = 18; // 2HB
	Hpos_apolar( 3,aa_arg,var) = 19; // 3HB
	Hpos_apolar( 4,aa_arg,var) = 20; // 2HG
	Hpos_apolar( 5,aa_arg,var) = 21; // 3HG
	Hpos_apolar( 6,aa_arg,var) = 22; // 2HD
	Hpos_apolar( 7,aa_arg,var) = 23; // 3HD

//bk   acceptor positions
	accpt_pos( 1,aa_arg, var) =  4; //  O
}
