// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

// Rosetta Headers
#include "aaproperties_pack.h"
#include "are_they_neighbors.h"
#include "files_paths.h"
#include "fullatom.h"
#include "gb_elec.h"
#include "initialize.h"
#include "input_pdb.h" //tmp
#include "maps_ns.h"
#include "misc.h"
#include "nblist.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdb.h"
#include "pH_ns.h"
#include "pH_main.h"
#include "pKa_mode.h"
#include "pose_ligand.h" //tmp
#include "current_pose.h" //tmp
#include "pose_io.h" //tmp
#include "PackerTask.h" //tmp
#include "runlevel.h"
#include "RotamerSet.h"
#include "rotamer_trials.h"
#include "score.h"
#include "after_opts.h"
#include "monte_carlo.h"
#include "status.h"

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// ObjexxFLC Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray.io.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>
#include <cmath>


namespace pKa_mode {

	// Mode Control Options
	bool pKa_flag = false;
	namespace pH_titration {
		bool pH_titration_flag = false;
		float start_pH;
		float end_pH;
		float delta_pH;
		FArray1D_float titration_array(files_paths::default_nstruct);
		using namespace pH_titration;
	}

	namespace pKa_control_flags {
		// General Options
		bool use_monte_carlo_flag = false;

		// Repack Options
		bool read_repack_flag = false;
		bool full_repack_flag = false;
		bool neighbor_repack_flag = false;
		bool site_repack_flag = false;
		bool optH_repack_flag = false;

		// Rotamer Trials Options
		bool full_rt_flag = false;
		bool neighbor_rt_flag = false;
		bool site_rt_flag = false;

		// File Output Options
		bool output_ensemble_stats_flag = false;
		bool condensed = false;

		// Global Options
		std::string repack_file = "repackfile"; // In - Repack Filename
		std::string ensemble_file = "ensemblefile"; // Out - Ensemble file. Used only if
		                                            // output_ensemble_stats_flag is true
		std::string ensemble_outfile = "ensemblefile";

		FArray1D_int repack_site_index; // In - Repack File Indexes of sites to repack.
		int nrepack_sites; // Number of sites to repack. Input from repack_file
		int current_repack_number;

		using namespace pKa_control_flags;
	}

	namespace pKa_packer_flags {
		using namespace param;
		// Packer Options
		std::string packer_mode;
		bool calc_rot_freq = false; // Used in Sim Annealing
		bool track_pack = false;
		int pKa_outputnum = 100;
		bool packer_set_temp = false;
		float packer_temp = 1;
		FArray1D_float store_rot_delta_energy;
		FArray1D_float store_rotstep_E;
		FArray1D_float theo_sum_rot_poll( MAX_RES(), 0.0);
		using namespace pKa_packer_flags;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_commandline_options()
///
/// @brief Sets Options for pKa mode
///
/// @detailed Sets variables for pKa mode in files_paths,pH_titration and
///           pKa_control_flags namespaces.
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	get_commandline_options() {
		using namespace files_paths;
		using namespace pH_titration;
		using namespace pKa_control_flags;

		std::cout << "WARNING! : pKa is an experimental mode. Use at your own risk!!!" << std::endl;

		// Mode Control Options
		set_pKa_flag(true);
		set_pH_packing_flag( true ); // pH is always on in pKa_mode
		if ( truefalseoption("pH_titration") ) {
			pH_titration_flag = true;
			real3afteroption("pH_titration",1.0,start_pH,14.0,end_pH,1.0,delta_pH);
			overwrite_pdbs = false;
			pH_ns::pH = start_pH;
		}

		// General Options
		if ( truefalseoption("use_monte_carlo") ) {
			use_monte_carlo_flag = true;
		}

		// Using Repack File
		if ( truefalseoption("read_repack_file") ) {
			set_read_repack_flag(true);
			stringafteroption("read_repack_file",protein_name_prefix + protein_name + ".repack",repack_file);
				if ( truefalseoption("neighbor_repack") ) {
					set_neighbor_repack_flag(true);
				}
			if ( truefalseoption("site_repack") ) {
				set_site_repack_flag(true);
			}
		} else {
			// Using Commandline
			if ( truefalseoption("full_repack") ) {
				set_full_repack_flag(true);
			}
			if ( truefalseoption("neighbor_repack") ) {
				set_neighbor_repack_flag(true);
				repack_site_index.dimension( 1 );
				nrepack_sites = 1;
				intafteroption("neighbor_repack",1,repack_site_index(1));
				std::cerr << "Number of Sites: " << nrepack_sites << std::endl;
				std::cerr << "Repack Site Index: " << repack_site_index(1) << std::endl;
			}
			if ( truefalseoption("site_repack") ) {
				set_site_repack_flag(true);
				repack_site_index.dimension( 1 );
				nrepack_sites = 1;
				intafteroption("site_repack",1,repack_site_index(1));
				std::cerr << "Number of Sites: " << nrepack_sites << std::endl;
				std::cerr << "Repack Site Index: " << repack_site_index(1) << std::endl;
			}
		}

		// Rotamer Trials Options
		if ( truefalseoption("full_rt") ) {
			set_full_rt_flag(true);
		}
		if ( truefalseoption("neighbor_rt") ) {
			set_neighbor_rt_flag(true);
			stringafteroption("neighbor_rt",protein_name_prefix + protein_name + ".repack",repack_file);
		}
		if ( truefalseoption("site_rt") ) {
			set_site_rt_flag(true);
			stringafteroption("site_rt",protein_name_prefix + protein_name + ".repack",repack_file);
		}

		// File Output Options
		if ( truefalseoption("output_ensemble_stats") ) {
			output_ensemble_stats_flag = true;
			stringafteroption("output_ensemble_stats",(protein_name_prefix + protein_name + ".ensemble"),ensemble_file);
			ensemble_outfile = score_path + code + ensemble_file;
			std::cerr << "Ensemble File: " << seq_path + ensemble_file << std::endl;
		}

		get_pH_packing_options();

		// If there is no option set, the optimizeH structure is output

		// Global Rosetta Options
		// Stuff from files paths
		gb_set_use_no_intra_res(true); // Intraresidue gb electrostatics disabled
		idealized_structure = false;
		require_frags       = false;
		require_start       = true;
		use_fasta           = false;
		default_nstruct     = 1 + static_cast<int>((pH_titration::end_pH - pH_titration::start_pH)/pH_titration::delta_pH);
		use_constraints     = false;
		input_fa            = true;
		output_fa           = true;
		repack_input        = true; // pKa may or may not repack input
		disable_filters     = true;
		use_pdb_numbering   = true;
		read_all_chains     = true;
		refold_input        = false;

	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_start_list()
///
/// @brief Use start list in main.cc to store copies of pdb to be
///        titrated
///
/// @detailed The modified start_list is used in initialize_pKa_start() to
//            loop through the pdbs at user defined pH steps.
///
/// @return
///
/// @global_read startnm, startch, nstartnm, pH_ns::pH
///
/// @global_write startnm, startch, nstartnm
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_titration_list()
	{
		using namespace files_paths;
		using namespace param;
		using namespace pH_titration;

		if ( !get_pKa_flag() || pH_titration_flag==false) return;
		if ( get_read_repack_flag() == true ) {
			read_repack_file();
		}

		float tmp_pH = start_pH;
		pH_ns::pH = start_pH;

		for ( int i = 1; i <= default_nstruct; i++ ) {
			titration_array(i) = tmp_pH;
			tmp_pH = tmp_pH + delta_pH;
			std::cerr << "pH: " << titration_array(i) << std::endl;
		}

		std::cerr << "Titration Size: " << default_nstruct << std::endl;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_pKa_start()
///
/// @brief Options for pKa mode
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	initialize_pKa_start(int const curr_outnum) {
		using namespace param;
		using namespace pH_titration;
		using namespace pKa_packer_flags;

		if (!get_pKa_flag() && !pH_titration_flag ) return;
		pKa_outputnum = curr_outnum;
		pH_ns::pH = titration_array(pKa_outputnum);

		std::cout << "pH Titration: " << titration_array(pKa_outputnum) << "\t current output number: " << pKa_outputnum << std::endl;
		calc_dGprotonation_table();
		std::cerr << "Titration Index Number: " << pKa_outputnum << std::endl;
		std::cerr << "Setting pH: " << pH_ns::pH << std::endl;

		theo_sum_rot_poll = 0.0;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_pdb_name()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read output_pH
///
/// @global_write filename
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	output_pdb_name( std::string & filename, int curr_outnum ) {
		using namespace files_paths;
		
		if ( !get_pKa_flag() ) {
			std::cout << "WARNING: pKa_flag is false. Improper Call to fuction pKa_mode::output_pdb_name." << std::endl;
			return;
		}

		filename = filename + "_pH" + string_of( pH_titration::titration_array(curr_outnum), 4 );

	}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_repack_file()
///
/// @brief Input repack sites from repack file
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	read_repack_file() {
		using namespace files_paths;
		using namespace param;
		using namespace pKa_control_flags;

		if ( !get_pKa_flag() || get_full_repack_flag() || get_full_rt_flag() ) { return; }
		if ( get_read_repack_flag() == false ) { return; }

		//seq_path
		std::string const filename( start_path + repack_file );
		utility::io::izstream instream;
		instream.clear();
		instream.open( filename );
		std::cerr << "Looking for repack file: " << start_path << instream.filename() << std::endl;

		if ( !instream ) {
			std::cout << "WARNING!! Repack file not found" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
			std::cerr << "Reading repack file: " << start_path << instream.filename() << std::endl;
		}

		instream >> nrepack_sites >> skip;
		std::cerr << "Number of Sites: " << nrepack_sites << std::endl;

		instream >> skip; // Skip: ----------------
		instream >> skip; // Skip: start

		repack_site_index.dimension(nrepack_sites);
		for (int i = 1; i <= nrepack_sites; i++) {
			instream >> repack_site_index(i) >> skip;
			std::cerr << "Repack Site Index: " << repack_site_index(i) << std::endl;
		}

		instream.close();

	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allow_repack()
///
/// @brief Setup the allow_repack and allow_rottrial arrays according to pKa
///        repacking or rotamer trials sub-mode.
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_allow_repack() {
		using namespace protein_maps;
		using namespace misc;
		using namespace param;

		allow_repack     = false;
		allow_rottrial   = false;

		// Setip allow_repack and allow_rottrial arrays
		setup_allow_full_repack();
		setup_allow_neighbor_repack();
		setup_allow_site_repack();
		setup_allow_full_rt();
		setup_allow_neighbor_rt();
		setup_allow_site_rt();

	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allow_full_repack()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @global_write protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_allow_full_repack() {
		if (! get_full_repack_flag() ) return;
		protein_maps::allow_repack     = true;
		protein_maps::allow_rottrial   = false;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allow_neighbor_repack()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @global_write protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_allow_neighbor_repack() {
		if (! get_neighbor_repack_flag() ) return;

		using namespace misc;
		using namespace param;
		using namespace pKa_control_flags;

		float dis2;
		bool neighbors;

		//rh Allow sites in file to be repacked
		for ( int i = 1; i <= nrepack_sites; i++ ) {
			protein_maps::allow_repack( repack_site_index(i) ) = true;

			//rh Repack residues that are neighbors of site i
			for ( int j = 1; j <= total_residue; ++j ) {
				are_they_neighbors(res(repack_site_index(i)),res(j),full_coord(1,1,repack_site_index(i)),
													 misc::full_coord(1,1,j),dis2,neighbors);
				if ( neighbors ) {
					protein_maps::allow_repack(j) = true;
				}
			}
		}
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allow_site_repack()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @global_write protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_allow_site_repack() {
		using namespace pKa_control_flags;

		if (! get_site_repack_flag() ) return;

		//rh Allow sites in file to be repacked
		for ( int i = 1; i <= nrepack_sites; i++ ) {
			protein_maps::allow_repack( repack_site_index(i) ) = true;
		}
		//if ( residue3(i) == "ASP" || residue3(i) == "GLU" || residue3(i) == "HIS" || residue3(i) == "TYR" || residue3(i) == "LYS" || residue3(i) == "ARG" ) {
		//allow_repack(i) = true;
		//}
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allow_full_rt()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @global_write protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_allow_full_rt() {
		if (! get_full_rt_flag() ) return;
		protein_maps::allow_rottrial   = true;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allow_neighbor_rt()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @global_write protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_allow_neighbor_rt() {
		if (! get_neighbor_rt_flag() ) return;

		using namespace misc;
		using namespace param;
		using namespace pKa_control_flags;

		float dis2;
		bool neighbors;

		//rh Allow sites in file to undergo rotamer trials
		for ( int i = 1; i <= nrepack_sites; i++ ) {
			protein_maps::allow_rottrial( repack_site_index(i) ) = true;

			//rh Rotamer trials on residues that are neighbors of site i
			for ( int j = 1; j <= total_residue; ++j ) {
				are_they_neighbors(res(repack_site_index(i)),res(j),full_coord(1,1,repack_site_index(i)),
													 misc::full_coord(1,1,j),dis2,neighbors);
				if ( neighbors ) {
					protein_maps::allow_rottrial(j) = true;
				}
			}
		}
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_allow_site_rt()
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @global_write protein_maps::allow_repack, protein_maps::allow_rottrial
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	setup_allow_site_rt() {
		using namespace pKa_control_flags;

		if (! get_site_rt_flag() ) return;

		//rh Allow sites in file to undergo rotamer trials
		for ( int i = 1; i <= nrepack_sites; i++ ) {
			protein_maps::allow_rottrial( repack_site_index(i) ) = true;
		}
		//if ( residue3(i) == "ASP" || residue3(i) == "GLU" || residue3(i) == "HIS" || residue3(i) == "TYR" || residue3(i) == "LYS" || residue3(i) == "ARG" ) {
		//allow_repack(i) = true;
		//}
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin main_protocol()
///
/// @brief Main protocol for pKa mode. Called from main.cc
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	main_protocol() {
		using namespace aaproperties_pack;
		using namespace misc;
		using namespace param;
		using namespace param_aa;
		using namespace pKa_control_flags;
		using namespace pKa_packer_flags;
		using namespace runlevel_ns;

		std::cerr << "mode: pKa" << std::endl;

		if (! get_pKa_flag() ) {
			std::cerr << "WARNING: pKa mode is not set properly. Must set from pKa class." << std::endl;
			return;
		}

		if ( !use_monte_carlo_flag && get_optH_repack_flag() ) {
		}

		// Full, Neighbor and Site Repack without monte_carlo outer loop
		if ( !use_monte_carlo_flag && ( get_full_repack_flag() || get_neighbor_repack_flag() || get_site_repack_flag() )) {
			monte_carlo_reset();
			select_rotamer_set( "large" );
			score_set_try_rotamers(false); // don't let scoring function change rotamers
			set_fullatom_flag(true);
			// score = score12();
			setup_allow_repack();
			
			// Use simple gen_born electrostatics by default
			//param_pack::gen_born = true;
			//gb_set_use_simple_electrostatics( true );
			
			//car local
			//FArray1D_bool missingres( MAX_RES()() );
			//std::string pack_mode;
			//FArray2D_int extra_rot( MAX_CHI, MAX_RES()() );
			//FArray2D_float extra_chi( MAX_CHI, MAX_RES()() );

			//bk set missingres to true if any heavy atoms were missing from a residue
			
			//include_extra = false;

			//rh Removed optimizeH from cycle [03-22-06]
			//   No longer needed because included ASP and GLU
			//   hydrogen rotamers in calc_exrotset_from_exflags()
			//pack_mode = "optimizeH";
			//std::cout << "optimizing hydrogen positions" << std::endl;
			//pack_rotamers( res, full_coord, res_variant, total_residue, phi, psi,
			//							 include_current, include_extra, extra_rot, extra_chi, missingres, pack_mode,
			//							 make_output_file );

			//for ( int i = 1; i <= 1; i++ ) {
			//int i = 1;
			//mc_global_track::mc_score::score = score13();
			
			std::cout << "Temperature Quelch (T: 100:exp(-outeriterations):0.3)" << std::endl;
			std::cout << "Collect Rotamer Statistics (T = 1)" << std::endl;

			calc_rot_freq = true;
			packer_set_temp = false;
			packer_temp = 0.3;
			//if ( get_neighbor_repack_flag() || get_site_repack_flag() ) {
				current_repack_number = 0;

				FArray1D_bool missingres( MAX_RES()() );
				std::string pack_mode;
				bool const fullatom( true );
				bool const ideal_pose( false ); // non-ideal backbone geometry
				bool const coords_init( true );
				bool include_current = true;

				pose_ns::Pose pose;

				pack_mode = "optimizeH";
				std::cout << "optimizing hydrogen positions (pKa mode)" << std::endl;
				pose_from_misc(pose,fullatom, ideal_pose, coords_init);
				
				PackerTask Task2( pose );
				Task2.set_task("optimizeH", false, missingres, include_current);
				Task2.setup_residues_to_vary();
				
				pack_rotamers( pose, Task2 );
				pose.copy_to_misc();
				
				current_repack_number = 2;
				main_repack_trial(score13,current_repack_number,true);
				//} else {
				//current_repack_number = 2;
				//main_repack_trial(score13,current_repack_number,true);
				//}

			//std::cout << "Equilibrate System: T = 0.3" << std::endl;
			//calc_rot_freq = false;
			//packer_set_temp = true;
			//packer_temp = 0.3;
			//current_repack_number = i+1;
			//main_repack_trial(score13,current_repack_number,false);

			//std::cout << "Collect Rotamer Statistics: T = 0.3" << std::endl;
			//calc_rot_freq = true;
			//packer_set_temp = true;
			//packer_temp = 0.3;
			//current_repack_number = i+1;
			//main_repack_trial(score13,current_repack_number,false);

			//monte_carlo_reset();
			//}
		}

		// Full, Neighbor and Site Repack with monte_carlo outer loop
		if ( use_monte_carlo_flag && ( get_full_repack_flag() || get_neighbor_repack_flag() || get_site_repack_flag() ) ) {
			//rh From docking_repack in docking_movement.cc
			select_rotamer_set( "large" );
			score_set_try_rotamers(false); // don't let scoring function change rotamers
			set_fullatom_flag(true);
			//output_status_file(0,5,-1.0);
			monte_carlo_reset();
			setup_allow_repack();
			mc_global_track::mc_score::score = score12();
			current_repack_number = 0;
			calc_rot_freq = true;
			main_repack(true);

			for ( int i = 1; i <= 5; i++ ) {
				calc_rot_freq = false;
				setup_allow_repack();
				current_repack_number = i;
				main_repack_trial(score12,current_repack_number);
			}

			calc_rot_freq = true;
			setup_allow_repack();
			current_repack_number = 6;
			main_repack_trial(score12,current_repack_number);

		}

		// Full, Neighbor and Site Rot Trials without monte_carlo outer loop
		if ( !use_monte_carlo_flag && ( get_full_rt_flag() || get_neighbor_rt_flag() || get_site_rt_flag() )) {
			monte_carlo_reset();
			select_rotamer_set( "large" );
			score_set_try_rotamers(true); // don't let scoring function change rotamers
			set_fullatom_flag(true);
			// score = score12();
			setup_allow_repack();

			for ( int i = 1; i <= 1; i++ ) {
				calc_rot_freq = true;
				current_repack_number = i;
				//rotamer_trials();
				mc_global_track::mc_score::score = score12();
				main_repack_trial(score12,current_repack_number);
				monte_carlo_reset();
			}
		}

		//for ( int i = 1; i <= total_residue; i++) {
		//	std::cerr << residue3(i) << "\t" << "allow_repack: " << protein_maps::allow_repack(i) << "\t" << "allow_rottrial: " << protein_maps::allow_rottrial(i) << std::endl;
		//}
		calc_rot_freq = false;

	}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_ensemble_stats()
///
/// @brief Output titration ensemble statistics to a file
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ryan Harrison 01/10/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
	void
	output_ensemble_stats( FArray1DB_float & rot_freq, RotamerSet const & rotamer_set ) {
		using namespace files_paths;
		using namespace misc;
		using namespace param;
		using namespace pKa_control_flags;
		using namespace pKa_packer_flags;

		if (! get_pKa_flag() && get_output_ensemble_stats_flag() ) {
			return;
		}

		std::fstream ensemble_teststream( ensemble_outfile.c_str(), std::ios_base::in|std::ios_base::out );
		utility::io::ozstream ensemble_stream;
	
		if ( !ensemble_teststream ) { // file is new so write out headers
			ensemble_teststream.close();
			ensemble_teststream.clear();
		
			ensemble_stream.open( ensemble_outfile );
			if ( !ensemble_stream ) {
				std::cout << "ABORT: cannot create or find scorefile" << std::endl;
				std::cout << "scorefile: " << ensemble_outfile << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			ensemble_stream << "EN" << "\t packmode" << "\t CycleN" << "\t pH" << "\t AA" << "\t AAN" << "\t RotNum" << "\t AAV" << "\t RotFreq" << "\t RotFreqTheo" << "\t RotFreqError" << "\t Energy" << std::endl;
		}
		
		ensemble_stream.open_append( ensemble_outfile );

		ensemble_stream << "# Output: " << pKa_outputnum << "\t packmode: " << packer_mode << "\t pH: " << pH_ns::pH << std::endl;
		if ( packer_mode=="optimizeH") current_repack_number=0;

		//rh Calculate theoretical rot freq from store_rot_delta_energy
		//rh e^-B*E(j)      : theo_rot_poll
		//rh sum( e^-B*E(j) : theo_sum_rot_poll
		FArray1D_float theo_rot_poll( rotamer_set.nrotamers(), 0.0);
		FArray1D_float theo_rot_freq( rotamer_set.nrotamers(), 0.0);
		FArray1D_float error_rot_freq( rotamer_set.nrotamers(), 0.0);

		theo_sum_rot_poll = 0;

		for ( int aan = 1; aan <= total_residue; ++aan ) {
			for ( int rot = 1; rot <= rotamer_set.nrotamers(); ++rot ) {
				if ( aan == rotamer_set.report_seqpos(rot) && (store_rot_delta_energy(rot) <= 40 ))   {
					theo_rot_poll(rot) = (std::exp(-store_rot_delta_energy(rot)));
					theo_sum_rot_poll(aan) += theo_rot_poll(rot);
				}
			}
		}

			for ( int rot = 1; rot <= rotamer_set.nrotamers(); ++rot ) {
				theo_rot_freq(rot) = ((theo_rot_poll(rot)) / (theo_sum_rot_poll(rotamer_set.report_seqpos(rot))));
					error_rot_freq(rot) = theo_rot_freq(rot) - rot_freq(rot);

					ensemble_stream << pKa_outputnum << "\t " << packer_mode << "\t " << current_repack_number << "\t" << F( 6, 5, pH_ns::pH) << "\t " << residue3( rotamer_set.report_seqpos(rot) ) << "\t " << rotamer_set.report_seqpos(rot) << "\t " << rotamer_set.report_rotnum(rot) << "\t " << rotamer_set.report_aav(rot) << "\t" << F( 9, 10, rot_freq(rot)) << "\t" <<  F( 9, 10, theo_rot_freq(rot)) << "\t" <<  F( 9, 10, error_rot_freq(rot)) << "\t" << F( 9, 10, store_rot_delta_energy(rot)) << std::endl;
			}
			ensemble_stream << std::endl;

		ensemble_stream.close();
		ensemble_stream.clear();

	}

} // Namespace


