// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $

#ifndef INCLUDED_pack_geom_inline
#define INCLUDED_pack_geom_inline


// Rosetta Headers
//#include "pack.h" -- NOOO!! This #include defeats the whole purpose of this .h file

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/Fmath.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>



// geometry Function Declarations

inline
void
getrot_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	float const chi,
	FArray2DB_float & mat,
	FArray1DB_float & vec
);

inline
void
getmat_bk(
	float & phi,
	float & psi,
	float const kappa,
	FArray2DB_float & aa
);

inline
void
dihedral_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float a3,
	FArray1Da_float a4,
	float & ang
);

inline
void
lineup_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float b1,
	FArray1Da_float b2,
	FArray2DB_float & mat,
	FArray1DB_float & vec
);


inline
void
distance_bk_dp(
	FArray1Da_double atom1,
	FArray1Da_double atom2,
	double & dis
);


inline
void
angle_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float b1,
	FArray1Da_float b2,
	float & ang
);


inline
void
distance_bk(
	FArray1Da_float atom1, // dimension( 3 )
	FArray1Da_float atom2, // dimension( 3 )
	float & dis
)
{
	float const a_1 = atom1[ 0 ] - atom2[ 0 ]; // atom1(1) - atom2(1)
	float const a_2 = atom1[ 1 ] - atom2[ 1 ]; // atom1(2) - atom2(2)
	float const a_3 = atom1[ 2 ] - atom2[ 2 ]; // atom1(3) - atom2(3)
	dis = std::sqrt( ( a_1 * a_1 ) + ( a_2 * a_2 ) + ( a_3 * a_3 ) );
}


inline
void
distance2_bk(
	FArray1Da_float atom1, // dimension( 3 )
	FArray1Da_float atom2, // dimension( 3 )
	float & dis2
)
{
	float const a_1 = atom1[ 0 ] - atom2[ 0 ]; // atom1(1) - atom2(1)
	float const a_2 = atom1[ 1 ] - atom2[ 1 ]; // atom1(2) - atom2(2)
	float const a_3 = atom1[ 2 ] - atom2[ 2 ]; // atom1(3) - atom2(3)
	dis2 = ( ( a_1 * a_1 ) + ( a_2 * a_2 ) + ( a_3 * a_3 ) );
}


template< typename T >
inline
void
subvec_bk(
	FArray1DB< T > const & v1,
	FArray1DB< T > const & v2,
	FArray1DB< T > & v3
)
{
//* v3 = v1 - v2
	v3(1) = v1(1) - v2(1);
	v3(2) = v1(2) - v2(2);
	v3(3) = v1(3) - v2(3);
}


template< typename T >
inline
void
rotate_bk(
	FArray2DB< T > const & mat,
	FArray1DB< T > const & v1,
	FArray1DB< T > & v2
)
{
	v2(1) = mat[0]*v1(1) + mat[1]*v1(2) + mat[2]*v1(3);
	v2(2) = mat[3]*v1(1) + mat[4]*v1(2) + mat[5]*v1(3);
	v2(3) = mat[6]*v1(1) + mat[7]*v1(2) + mat[8]*v1(3);
}


template< typename T >
inline
void
center_angle( T & ang )
{
	ang -= static_cast< long int >( ( ang + T( 180.0 ) ) / T( 360.0 ) ) * T( 360.0 );
}


template< typename T >
inline
void
angle_between(
	T ang1,
	T ang2,
	T & diff
)
{
	T const d12 = ang1 - ang2;
	diff = std::abs( nlint( d12 / T( 360.0 ) ) * T( 360.0 ) - d12 );
}


inline
void
move_bk(
	FArray1Da_float c, // dimension( 3 )
	FArray2DB_float const & mat,
	FArray1DB_float const & vec
)
{
	static FArray1D_float t( 3, 0.0f );

	rotate_bk(mat,c,t);
	c[0] = t(1) + vec(1);
	c[1] = t(2) + vec(2);
	c[2] = t(3) + vec(3);
}


template< typename T >
inline
void
cross_bk(
	FArray1DB< T > const & a,
	FArray1DB< T > const & b,
	FArray1DB< T > & c // A X B
)
{
// Cross product A X B
	c(1) = a(2) * b(3) - a(3) * b(2);
	c(2) = a(3) * b(1) - a(1) * b(3);
	c(3) = a(1) * b(2) - a(2) * b(1);
}


template< typename T >
inline
void
cross_bk(
	FArray1DB< T > const & a,
	FArray1DB< T > const & b,
	FArray1DB< T > & c, // A X B
	T & d // Length of A X B
)
{
// Cross product A X B
	c(1) = a(2) * b(3) - a(3) * b(2);
	c(2) = a(3) * b(1) - a(1) * b(3);
	c(3) = a(1) * b(2) - a(2) * b(1);
	d = std::sqrt( c(1)*c(1) + c(2)*c(2) + c(3)*c(3) );
}


inline
void
align_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float a3,
	FArray1Da_float b1,
	FArray2Da_float mat,
	FArray1Da_float vec
)
{
	a1.dimension( 3 );
	a2.dimension( 3 );
	a3.dimension( 3 );
	b1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

	static FArray1D_float a( 3, 0.0f );
	static FArray1D_float b( 3, 0.0f );
	static FArray1D_float c( 3, 0.0f );
	static FArray1D_float ab( 3, 0.0f );
	static FArray1D_float ac( 3, 0.0f );
	static FArray1D_float aba( 3, 0.0f );
	float dab,daba,ab_ac,aba_ac,chi;

//* get difference vectors
	for ( int i = 1; i <= 3; ++i ) {
		a(i) = a1(i) - a2(i);
		b(i) = a3(i) - a2(i);
		c(i) = b1(i) - a2(i);
	}
	cross_bk(b,a,ab,dab);
	cross_bk(c,a,ac);
	cross_bk(ab,a,aba,daba);
//* get vector lengths
//* get ab.dot.ac
	ab_ac = ab(1)*ac(1) + ab(2)*ac(2) + ab(3)*ac(3);
//*    aba.dot.ac
	aba_ac = aba(1)*ac(1) + aba(2)*ac(2) + aba(3)*ac(3);
//* cosine ab_ac
	ab_ac /= dab;
//* cosine aba_ac = sine ab_ac
	aba_ac /= daba;
//* get chi
	chi = std::atan2(aba_ac,ab_ac);
//* get matrix for chi rotation
	getrot_bk(a1,a2,chi,mat,vec);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin align_atom
///
/// @brief
/// Given two bonds that originate from the same point, generate a rotation
/// matrix for moving the mobile bond and its children to superimpose with the
/// reference bond.  Simpler than lineup_bk().
///
/// @detailed
/// Written to allow the preservation of c-beta conformation from the input
/// structure by rotational adjustment of rotamer sidechains. ( -use_input_cb )
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

inline
void
align_atom(
	FArray1Da_float mob,   // mobile atom
	FArray1Da_float ref,   // reference atom
	FArray1Da_float cen,   // equivalent atom (center of rotation)
	FArray2Da_float mat,
	FArray1Da_float vec
)
{
	mob.dimension( 3 );
	ref.dimension( 3 );
	cen.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

	// vectors to center of rotation
	static FArray1D_float a( 3, 0.0f );
	static FArray1D_float b( 3, 0.0f );
	static FArray1D_float ab( 3, 0.0f );
	for ( int i = 1; i <= 3; ++i ) {
		a(i) = mob(i) - cen(i);
		b(i) = ref(i) - cen(i);
	}
	cross_bk( a, b, ab ); // rotation axis through center atom

	float dist, radius;
	distance_bk(mob,ref,dist); // displacement
	distance_bk(ref,cen,radius); // bond length

	float rot_angle = 2*std::asin(dist/(2*radius)); // angle to rotate

	static FArray1D_float dummy_atom( 3, 0.0f );
	for ( int i = 1; i <= 3; ++i ) dummy_atom(i) = cen(i) + ab(i);

	getrot_bk( cen, dummy_atom, rot_angle, mat, vec ); // get rotation matrix

}

////////////////////////////////////////////////////////////////////////////////
/// @begin getrot_bk
///
/// @brief
/// find the matrix (mat) and vector (vec) that rotate chi deg about an axis
/// defined by a1-->a2 rotation is right-handed. That is, it is a clockwise
/// rotation when looking from a1 to a2 chi in radians
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  chi - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
getrot_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	float const chi,
	FArray2DB_float & mat,
	FArray1DB_float & vec
)
{
	using numeric::sin_cos_range;

	if ( std::abs(chi) >= 1.0E-4 ) {
		a1.dimension( 3 );
		a2.dimension( 3 );
		FArray1D_float a( 3 );
		FArray1D_float c( 3 );
//* get difference vector: a
		subvec_bk(a2,a1,a);
//* get length of vector A
		float const da = std::sqrt( a(1)*a(1) + a(2)*a(2) + a(3)*a(3) );
//* get phi, psi
		float psi = std::acos(sin_cos_range(a(3)/da));
		float phi = 0.0;
		if ( a(1) != 0.0 || a(2) != 0.0 ) phi = std::atan2(-a(1),a(2));
//* get matrix
		getmat_bk(phi,psi,chi,mat);
//* get vector, = -Mat*a1 + a1
		rotate_bk(mat,a1,c);
		subvec_bk(a1,c,vec);
		return;
	}
	mat = 0.0;
	for ( int i = 1; i <= 3; ++i ) {
		mat(i,i) = 1.0;
	}
	vec = 0.0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin getmat_bk
///
/// @brief
/// get matrix for Tanaka convention polar angles
///
/// @detailed
///* CB 14-JAN-1991, function 3-JUN-1991
///
/// @param  phi - [in/out]? - in radians
/// @param  psi - [in/out]? - in radians
/// @param  kappa - [in/out]? - in radians
/// @param  aa - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
getmat_bk(
	float & phi,
	float & psi,
	float const kappa,
	FArray2DB_float & aa
)
{

	// Check aa matrix is 3x3
	assert( ( aa.I1() == SRange( 3 ) ) );
	assert( ( aa.I2() == SRange( 3 ) ) );

//***
//***cb Try to avoid underflows...
	if ( std::abs(phi) < 1.0E-4 ) phi = 0.0;
	if ( std::abs(psi) < 1.0E-4 ) psi = 0.0;
	if ( std::abs(kappa) < 1.0E-4 ) { // Return identity
		aa[1] = aa[2] = aa[3] = aa[5] = aa[6] = aa[7] = 0.0;
		aa[0] = aa[4] = aa[8] = 1.0; // aa(1,1) = aa(2,2) = aa(3,3) = 1.0;
		return;
	}
	float const sf = std::sin(phi);
	float const cf = std::cos(phi);
	float const ss = std::sin(psi);
	float const cs = std::cos(psi);
	float const sk = std::sin(-kappa);
	float const ck = std::cos(-kappa);
//* now calculate the product matrix of the five rotation matrices
	aa[ 0 ] = (cf*cf+sf*sf*cs*cs)*ck+sf*sf*ss*ss; // aa(1,1)
	aa[ 3 ] = ss*ss*sf*cf*(ck-1.0f)-cs*sk;        // aa(1,2)
	aa[ 6 ] = sf*cs*ss*(ck-1.0f)+cf*sk*ss;        // aa(1,3)
	aa[ 1 ] = ss*ss*cf*sf*(ck-1.0f)+cs*sk;        // aa(2,1)
	aa[ 4 ] = (sf*sf+cf*cf*cs*cs)*ck+cf*cf*ss*ss; // aa(2,2)
	aa[ 7 ] = cf*ss*cs*(1.0f-ck)+sf*ss*sk;        // aa(2,3)
	aa[ 2 ] = cs*ss*sf*(ck-1.0f)-cf*ss*sk;        // aa(3,1)
	aa[ 5 ] = cf*ss*cs*(1.0f-ck)-sf*ss*sk;        // aa(3,2)
	aa[ 8 ] = ss*ss*ck+cs*cs;                    // aa(3,3)
//* done.

//cb  std::cout << std::endl;
//cb  for ( int j = 1; j <= 3; ++j ) {
//cb    for ( int i = 1; i <= 3; ++i ) {
//cb      std::cout << F( 12, 8, aa(i,j) );
//cb    } std::cout << std::endl;
//cb  }
}

/////////////////////////////////////////////////////////////////////////////////
/// @begin dihedral_bk
///
/// @brief calculates dihedral angle
///
/// @detailed
/// Given four atoms connected in a chain (a1, a2, a3, and a4),
/// dihedral calculates the torsion angle(deg) between the vectors
/// a2->a1 and a3->a4 while sighting along the axis defined
/// by the vector a2->a3 (positive indicates right handed twist).
///
/// @param[in] a1  input - atom coordinates (same for a2-a4)
/// @param[out] ang  output - dihedral angle in degrees
///
/// @return
///
///\GLOBAL_PREREQ
///
///\GLOBAL_CHANGED
///
/// @remarks
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
dihedral_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float a3,
	FArray1Da_float a4,
	float & ang
)
{
	using namespace numeric::constants::f;
	using numeric::conversions::degrees;
	using numeric::sin_cos_range;

	a1.dimension( 3 );
	a2.dimension( 3 );
	a3.dimension( 3 );
	a4.dimension( 3 );

	float psi,phi;
	//Objexx: static for speed
	static FArray1D_float a1t( 3 );
	static FArray1D_float a2t( 3 );
	static FArray1D_float a3t( 3 );
	static FArray1D_float a4t( 3 );
	static FArray1D_float a1r( 3 );
	static FArray1D_float a2r( 3 );
	static FArray1D_float a3r( 3 );
	static FArray1D_float a4r( 3 );
	static FArray1D_float a( 3 );
	static FArray1D_float b1( 3 );
	static FArray1D_float b2( 3 );
	float da;
	static FArray2D_float mat( 3, 3 ); // rotation matrix
	float sf, cf, ss, cs; // trig operations
	float dis1,dis2,dis3;

//**     make sure sequential atoms are not on top of each other
	distance2_bk(a1,a2,dis1);
	distance2_bk(a2,a3,dis2);
	distance2_bk(a3,a4,dis3);

	if ( dis1 < 0.001 || dis2 < 0.001 || dis3 < 0.001 ) {
		ang = 0.0; // not 4 separate points so give dummy value of 0
		return;
	}

//**     make the point a2 the center of the coordinate system

	for ( int i = 1; i <= 3; ++i ) {
		a1t(i) = a1(i) - a2(i);
		a3t(i) = a3(i) - a2(i);
		a4t(i) = a4(i) - a2(i);
		a2t(i) = 0.0;
	}

//**     determine tanaka convention polar angles between the
//**        vector a2->a3 and the z axis, steps(1-3)

//**     (1) get difference vector: a
	subvec_bk(a3t,a2t,a);

//**     (2) get length of vector a
	da = a(1)*a(1) + a(2)*a(2) + a(3)*a(3);
	da = std::sqrt(da);

//**     (3) get phi, psi (tanaka convention polar coordinates)
	psi = std::acos(sin_cos_range(a(3)/da));
	if ( (a(1) != 0.00) && (a(2) != 0.00) ) {
		phi = std::atan2(a(2),a(1));
	} else if ( a(1) > 0.0 ) {
		phi = 0.0;
	} else if ( a(1) < 0.0 ) {
		phi = pi;
	} else if ( a(2) > 0.0 ) {
		phi = pi_over_2;
	} else if ( a(2) < 0.0 ) {
		phi = -pi_over_2;
	} else {
		phi = 0.0;
	}

//**     determine rotation matrix that will bring a3 on to the z axis,
//**     rotate coordinate system by phi degrees around the z axis and
//**     by -psi degrees around the y axis

	sf = std::sin(phi);
	cf = std::cos(phi);
	ss = std::sin(-psi);
	cs = std::cos(-psi);

	mat[0] = cs*cf; //  mat(1,1)
	mat[1] = cs*sf; //  mat(2,1)
	mat[2] = ss;     // mat(3,1)
	mat[3] = -sf;   //  mat(1,2)
	mat[4] = cf;     // mat(2,2)
	mat[5] = 0.0;   //  mat(3,2)
	mat[6] = -ss*cf; // mat(1,3)
	mat[7] = -ss*sf; // mat(2,3)
	mat[8] = cs;     // mat(3,3)

//**     multiply a1t, a2t, a3t, a4t by the rotation matrix
//**     store new coordinates in a1r, a2r, a3r, a4r

	for ( int i = 1; i <= 3; ++i ) {
		int const l = mat.index(1,i);
		float const m1 = mat[ l ], m2 = mat[ l+1 ], m3 = mat[ l+2 ];
		a1r(i) = m1*a1t(1) + m2*a1t(2) + m3*a1t(3);
		a2r(i) = m1*a2t(1) + m2*a2t(2) + m3*a2t(3);
		a3r(i) = m1*a3t(1) + m2*a3t(2) + m3*a3t(3);
		a4r(i) = m1*a4t(1) + m2*a4t(2) + m3*a4t(3);
//Objexx: Replaced by above for speed
//    a1r(i) = mat(1,i)*a1t(1)+mat(2,i)*a1t(2)+mat(3,i)*a1t(3);
//    a2r(i) = mat(1,i)*a2t(1)+mat(2,i)*a2t(2)+mat(3,i)*a2t(3);
//    a3r(i) = mat(1,i)*a3t(1)+mat(2,i)*a3t(2)+mat(3,i)*a3t(3);
//    a4r(i) = mat(1,i)*a4t(1)+mat(2,i)*a4t(2)+mat(3,i)*a4t(3);
	}

//  std::cout << "a3r" << SS( a3r(1) ) << SS( a3r(2) ) << SS( a3r(3) ) << std::endl;
//**  compute the difference vectors for a3r->a4r and a2r->a1r
//**  b2 and b1 are the output storage vectors
	subvec_bk(a4r,a3r,b2);
	subvec_bk(a1r,a2r,b1);

//**  rotate b1 and b2 around the z axis so that b2 runs along the
//**  x axis
	float const theta = std::atan2(b1(2),b1(1));
	float const sin_theta = std::sin(theta);
	float const cos_theta = std::cos(theta);
	float const rb2_1 = cos_theta*b2(1) + sin_theta*b2(2);
	float const rb2_2 = -sin_theta*b2(1) + cos_theta*b2(2);
//Objexx: Replaced by above for speed
//  FArray1D_float rb1( 3 );
//  FArray1D_float rb2( 3 );
//  rb1(1) = std::cos(theta)*b1(1) + std::sin(theta)*b1(2);
//  rb1(2) = -std::sin(theta)*b1(1) + std::cos(theta)*b1(2);
//  rb2(1) = std::cos(theta)*b2(1) + std::sin(theta)*b2(2);
//  rb2(2) = -std::sin(theta)*b2(1) + std::cos(theta)*b2(2);

//**  compute the dihedral angle
	ang = degrees( std::atan2( rb2_2, rb2_1 ) );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin lineup_bk
///
/// @brief
///* find the matrix (mat) and vector (vec) that line up a1->a2 with b1->b2,
///* with a1 superimposed on b1.
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  b1 - [in/out]? -
/// @param  b2 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
lineup_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float b1,
	FArray1Da_float b2,
	FArray2DB_float & mat,
	FArray1DB_float & vec
)
{
	using numeric::sin_cos_range;

//Objexx: Skip dimensioning for speed
//  a1.dimension( 3 );
//  a2.dimension( 3 );
//  b1.dimension( 3 );
//  b2.dimension( 3 );

	static FArray1D_float a( 3 );
	static FArray1D_float b( 3 );
	static FArray1D_float c( 3 );

//* get difference vectors
	a[0] = a2[0] - a1[0];
	a[1] = a2[1] - a1[1];
	a[2] = a2[2] - a1[2];
	b[0] = b2[0] - b1[0];
	b[1] = b2[1] - b1[1];
	b[2] = b2[2] - b1[2];

//* get vector lengths
//* get a.dot.b
	float const da_db = std::sqrt(
	 ( a(1)*a(1) + a(2)*a(2) + a(3)*a(3) ) * ( b(1)*b(1) + b(2)*b(2) + b(3)*b(3) ) );
	float dab = a(1)*b(1) + a(2)*b(2) + a(3)*b(3);

//* get chi
//cb The following is to protect against underflows on the Mac:
	float const chi =
	 ( ( std::abs(dab-da_db) >= 0.000001 ) ? std::acos(sin_cos_range(dab/da_db)) : 0.0 );

	float psi;
	float phi;
	if ( chi >= 1.0E-4 ) {
//* cross product A X B = C
		cross_bk(a,b,c,dab);
//* get phi, psi
		psi = std::acos(sin_cos_range(c(3)/dab));
		phi = ( ( c(1) != 0.0 || c(2) != 0.0 ) ? std::atan2(-c(1),c(2)) : 0.0 );
	} else {
		psi = 0.0;
		phi = 0.0;
	}

//* get matrix
	getmat_bk(phi,psi,chi,mat);

//* get vector, = -Mat*a1 + b1
	rotate_bk(mat,a1,c);
	vec[0] = -c[0] + b1[0];
	vec[1] = -c[1] + b1[1];
	vec[2] = -c[2] + b1[2];
}



//******************************************************************************

////////////////////////////////////////////////////////////////////////////////
/// @begin angle_bk
///
/// @brief
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  b1 - [in/out]? -
/// @param  b2 - [in/out]? -
/// @param  ang - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
angle_bk(
	FArray1Da_float a1,
	FArray1Da_float a2,
	FArray1Da_float b1,
	FArray1Da_float b2,
	float & ang
)
{
	using numeric::sin_cos_range;

	a1.dimension( 3 );
	a2.dimension( 3 );
	b1.dimension( 3 );
	b2.dimension( 3 );

//cb determine the angle between the two vectors a1->a2,b1->b2
	static FArray1D_float a( 3 );
	static FArray1D_float b( 3 );

	ang = 0.0;
//* get difference vectors
	for ( int i = 1; i <= 3; ++i ) {
		a(i) = a2(i) - a1(i);
		b(i) = b2(i) - b1(i);
	}
//* get vector lengths
//* get a.dot.b
	float const da = a(1)*a(1) + a(2)*a(2) + a(3)*a(3);
	float const db = b(1)*b(1) + b(2)*b(2) + b(3)*b(3);
	float const dab = a(1)*b(1) + a(2)*b(2) + a(3)*b(3);
	float const da_db = std::sqrt( da * db );
//* get angle
//cb The following is to protect against underflows on the Mac:
	if ( std::abs(dab-da_db) < 0.000001 ) {
		ang = 0.0;
	} else {
		ang = std::acos(sin_cos_range(dab/da_db));
	}

}


#endif
