// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 18214 $
//  $Date: 2007-11-05 08:32:40 -0800 (Mon, 05 Nov 2007) $
//  $Author: ora $


// Rosetta Headers
#include "pdb.h"
#include "enzyme.h"
#include "docking_ns.h"
#include "misc.h"
#include "files_paths.h"
#include "param.h"
#include "pose.h"

// ObjexxFCL Headers
// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <iostream>

// Using
using namespace param;


namespace pdb {
//jg  data from the input pdb file
//car atom numbers in these arrays use the fullcoord numbering scheme: n,ca,c,o,cb...
	FArray1D_int pdb_res_num( MAX_RES(), 0 );
	FArray1D_char res_chain( MAX_RES(), ' ' );
	FArray1D_char pdb_insert_let( MAX_RES(), ' ' );
	FArray2D_float bvalue( MAX_ATOM(), MAX_RES(), 0.0 );
	FArray2D_float occupancy( MAX_ATOM(), MAX_RES(), 1.0 );
	FArray2D_bool missing_input_atom( MAX_ATOM(), MAX_RES(), 0.0 );
}

 Pdb_info::Pdb_info(){
	use_pdb_numbering_ = false;
	multi_chain_ = false;
}

// returns the private information
int
Pdb_info::pdb_res_num(
		int const pos
		) const
{
	return pdb_res_num_(pos);
}

char
Pdb_info::res_chain(
		int const pos
		) const
{
	return res_chain_(pos);
}

char
Pdb_info::pdb_insert_let(
		int const pos
		) const
{
	return pdb_insert_let_(pos);
}

//lin funtion for set pdb_res, pdb_chain, pdb_insert_let
	/////////////////////////////////////////////////////////////////////////////
	void
	Pdb_info::set_pdb_res(
		int const pos,
		int const value
	)
	{
		pdb_res_num_(pos) = value;
	}

	void
	Pdb_info::set_pdb_occ(
		int const pos,
		float const value
	)
	{
		for ( int j=1; j<=param::MAX_ATOM()(); j++ ) {
			occupancy_(j,pos) = value;
		}
	}

	void
	Pdb_info::set_pdb_bvalue(
												int const atom_index,
												int const pos,
												float const value
	)
	{
		bvalue_(atom_index,pos) = value;
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pdb_info::set_pdb_chain(
		int const pos,
		char const value
	)
	{
		res_chain_(pos) = value;
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Pdb_info::set_pdb_insert_let(
		int const pos,
		char const value
	)
	{
		pdb_insert_let_(pos) = value;
	}


 void Pdb_info::set_pdb_info(
  bool const use_pdb_numbering_in,
  FArray1D_int & res_num_in,
  FArray1D_char & res_chain_in,
  FArray1D_char & pdb_insert_let_in,
  FArray2D_float & bvalue_in,
  FArray2D_float & occupancy_in,
  FArray2D_bool & missing_input_atom_in
 )
 {
	 dimension( misc::ints::total_residue );
	 use_pdb_numbering_ = use_pdb_numbering_in;
	 set_pdb_info( 1, misc::ints::total_residue, res_num_in, res_chain_in, pdb_insert_let_in, bvalue_in, occupancy_in, missing_input_atom_in );
 }

 void Pdb_info::set_pdb_info(
	int const begin, int const end,
	FArray1D_int const & res_num_in,
  FArray1D_char const & res_chain_in,
  FArray1D_char const & pdb_insert_let_in,
  FArray2D_float const & bvalue_in,
  FArray2D_float const & occupancy_in,
  FArray2D_bool const & missing_input_atom_in
 )
 {
  for ( int i=begin; i<= end; ++i ) {
	 pdb_res_num_(i) = res_num_in(i);
	 res_chain_(i) = res_chain_in(i);
	 pdb_insert_let_(i) = pdb_insert_let_in(i);
	 for ( int j=1; j<=param::MAX_ATOM()(); j++ ) {
	   bvalue_(j,i) = bvalue_in(j,i);
	   occupancy_(j,i) = occupancy_in(j,i);
	   missing_input_atom_(j,i) = missing_input_atom_in(j,i);
	 }
	}
 }

//lin stort the multi chain info :here is only docking_part, can be more
void Pdb_info::set_multi_chain_info(
		 bool const use_multi_chain,
		 int const size_in,
     FArray1D_int const & docking_part_begin,
     FArray1D_int const & docking_part_end
		 ) {
	multi_chain_ = use_multi_chain;

	if( ! multi_chain_ && !get_enable_ligaa_flag() ) return;
	docking_part_begin_		 .dimension(size_in);
	docking_part_end_			 .dimension(size_in);
	for ( int i=1; i<= size_in; ++i ) {
		docking_part_begin_(i) = docking_part_begin(i) ;
		docking_part_end_(i) = docking_part_end(i) ;
	}
}

void Pdb_info::get_multi_chain_info( int const begin, int const end ) const {
	// copy the local to the global
	if( ! multi_chain_ ) return;
	for ( int i=begin; i<=end; ++i ) {
		docking::docking_query::part_begin(i) =	docking_part_begin_(i) ;
		docking::docking_query::part_end(i) =	docking_part_end_(i) ;
	}
}

void Pdb_info::pdb_info_from_global()
{
	set_pdb_info( files_paths::use_pdb_numbering, pdb::pdb_res_num, pdb::res_chain, pdb::pdb_insert_let, pdb::bvalue, pdb::occupancy, pdb::missing_input_atom);
	set_multi_chain_info( files_paths::io_options_bool::multi_chain,
												docking::dock_multi::max_monomers,
												docking::docking_query::part_begin,
												docking::docking_query::part_end );
}

void Pdb_info::pdb_info_to_global( ) const
{
	pdb_info_to_global( 1, misc::ints::total_residue );
	// comment this out as copying multichain back to global namespace is
	// dangerous if there is no pdb info initialized with pose --chu.
	// get_multi_chain_info( 1, docking::dock_sym::max_monomers );
}

void Pdb_info::pdb_info_to_global( int begin, int end ) const
{
 for ( int i=begin; i<= end; ++i ) {
	pdb::pdb_res_num(i) = pdb_res_num_(i);
	pdb::res_chain(i) = res_chain_(i);
	pdb::pdb_insert_let(i) = pdb_insert_let_(i);
  for ( int j=1; j<=param::MAX_ATOM()(); j++ ) {
	  pdb::bvalue(j,i) = bvalue_(j,i);
	  pdb::occupancy(j,i) = occupancy_(j,i);
	  pdb::missing_input_atom(j,i) = missing_input_atom_(j,i);
	}
 }
}

//////////////////////////////////////////////////////////////////////////////
/// @begin print_pdb_information()
///
/// @brief print the information of the pdb that is in pose
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 07 2006
////////////////////////////////////////////////////////////////////////////////
void
Pdb_info::print_pdb_information(int const size) const
{
	print_pdb_information( 1, size );
}

void
Pdb_info::print_pdb_information(int const begin, int const end) const
{
	std::cout << "\nPdb information\n" << std::endl;

	for ( int i = begin; i<= end; ++i ) {
		std::cout << I( 4, pdb_res_num(i) ) << " (Rosetta" << I( 4, i ) << ")";
	}
	std::cout << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin copy_segment()
///
/// @brief copy a segment of information from the pdb_info of a pose into the global pdb_info
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified June 28 2006
////////////////////////////////////////////////////////////////////////////////
void
Pdb_info::copy_segment(
	int const size,
	Pdb_info const & p,
	int const begin,
	int const src_begin
	)
{
//	p.print_pdb_information(src_begin+size-1);
//	print_pdb_information(begin+size-1);
	for ( int i=0; i<size; ++i ) {
		int const pos ( begin+i );
		int const src_pos ( src_begin+i );

	  pdb_res_num_(pos) = p.pdb_res_num(src_pos);
	  res_chain_(pos) = p.res_chain(src_pos);
	  pdb_insert_let_(pos) = p.pdb_insert_let(src_pos);

		for ( int j=1; j<=param::MAX_ATOM()(); j++ ) {
			bvalue_(j, pos) = p.bvalue()(j, src_pos);
			occupancy_(j,pos) = p.occupancy()(j, src_pos);
			missing_input_atom_(j,pos) = p.missing_input_atom()(j,src_pos);
		}
	}
}

void
Pdb_info::dimension( const int size_in ) {
	pdb_res_num_			 .dimension(size_in);
	res_chain_				 .dimension(size_in);
	pdb_insert_let_		 .dimension(size_in);
	bvalue_						 .dimension(MAX_ATOM(), size_in);
	occupancy_				 .dimension(MAX_ATOM(), size_in);
	missing_input_atom_.dimension(MAX_ATOM(), size_in);
}

Pdb_info &
Pdb_info::operator =(
		Pdb_info const & src
		)
{
	pdb_res_num_ = src.pdb_res_num_;
	res_chain_ = src.res_chain_;
	pdb_insert_let_ = src.pdb_insert_let_;
	bvalue_ = src.bvalue_;
	occupancy_ = src.occupancy_;
	missing_input_atom_ = src.missing_input_atom_;
	use_pdb_numbering_ = src.use_pdb_numbering_;
	multi_chain_ = src.multi_chain_;
	docking_part_begin_ = src.docking_part_begin_ ;
	docking_part_end_ = src.docking_part_end_ ;
	return *this;
}


//////////////////////////////////////////////////////////////
// Just some default values.
void
Pdb_info::set_pdb_info( const int size_in ) {
	dimension( size_in );
	//Default values?
	for (int i = 1; i <= size_in; i++){
		pdb_res_num_( i ) = i;
		res_chain_( i )    = ' ';
		pdb_insert_let_( i )    = ' ';
		for (int j = 1; j <= MAX_ATOM(); j++){
			bvalue_( j, i )    = 0.0;
			occupancy_( j, i )    = 1.0;
			missing_input_atom_( j, i ) = false;
		}
	}
	use_pdb_numbering_ = false;
	multi_chain_ = false;
}


//////////////////////////////////////////////////////////////
// Default values, plus look inside pose for
// pseudo residues, to determine occupancy.
void
Pdb_info::figure_out_pdb_info( pose_ns::Pose & pose ) {
	set_pdb_info( pose.total_residue() );

	//Set occupancy of pseudoresidues to zero.
	for (int i=1; i <= pose.total_residue(); i++) {
		float const occ = pose.pseudo(i) ? 0.0: 1.0;
		set_pdb_occ( i, occ );
	}

}
