// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 35419 $
//  $Date: 2010-02-23 08:52:19 -0800 (Tue, 23 Feb 2010) $
//  $Author: barak $

#ifndef INCLUDED_pose
#define INCLUDED_pose


// Rosetta Headers
#include "fold_tree.h"
#include "bonds_class.h"
#include "cst_packer.h"
#include "cst_set.h"
#include "jump_classes.h"
#include "kin_atom_tree.h"
#include "packer_weights.h"
#include "param.h"
#include "param_aa.h"
#include "param_torsion.h" // phi_torsion, etc
#include "pdb.h"
#include "pose_param.h"
#include "pose_rms.h"
#include "score_data.h"
#include "symmetry_info.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <numeric/all.fwd.hh>

// C++ Headers
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <map>
#include <string>

//Utility Headers
#include <utility/basic_sys_util.hh>

// Forward declarations
#include "PackerTask_fwd.h"
#include "RotamerSetFWD.h"

/////////////////////////////////////////////////////////////////////////////80
// first try at making a "pose" object
// I wrote these comments a while ago -- some of them may have become a little
// out of date. I am trying to update them right now. 03/04/05
//
// interface: .set_phi(...) .score() .refold()
//
// book-keeping:
// torsions are accessible through set() functions
// these calls automatically set both the corresponding variable and
// internal book-keeping variables that look like new_<tag>_refold
// where <tag> is "bb","chi","jump"
//
// .refold() checks these variables to see what torsions have changed and folds
// accordingly. So the job of these vars is to record mismatches between the
// torsions and the coordinates
//
// at the end of .refold() the new_*_refold vars all set to false, and
// corresponding new_*_score variables are set to true. These variables
// record changes in the coordinates so that stored score data can be
// updated properly.
//
// outside routines can get (res_moved, pair_moved) and/or the domain_map
// to figure out which parts of the structure have changed.
//
// score book-keeping: this is probably the trickiest part. The idea is to
// include in the pose all the common variables like cendist
// and fullatom energies that are used to make scoring fast, but that
// make book-keeping such a headache. All this stuff is bundled inside
// the "Score_data" obect, defined in score_data.h
// currently it just holds 0,1,2 dimensional FArrays (0==>just a float)
//
// each piece of score data is assigned a state: BAD,OK,GOOD
// GOOD means fully up-to-date; OK means up to date except for
// changes to the structure as recorded by res_moved, pair_moved and
// domain_map; and BAD is all bad. After refold all GOOD scores are
// set to OK; during a score evaluation, when a score is updated it's
// state goes back to GOOD; at the end all scores still in state OK are reset
// to BAD. This is necessary since after a score evaluation the
// book-keeping data is reset.
//
// During the course of a score
// evaluation routines can access and/or modify this data, by calls
// to score_data.get_?D_score() or score_data.set_?D_score()
// get_* return const references, set_* return non-const references
// The key assumption is that calls to set_* accessing a given type
// of stored score data will update this data to accurately measure
// the current coordinates. This allows us to know for any type of
// score data what its status is, and this status is returned along
// with the references in calls to get_ or set_
// score data that isn't touched in the course of a score-evaluation
//

////////////////////////////////////////////////////////////
// thinking more about how to efficiently mesh with Rosetta:
// maybe use the misc arrays to communicate with scoring functions
//
// the main things to take from pose:
//
// ** stored score data like cendist, bumps, atr_pair
// ** allow arrays like allow_bb_move ( --> allow_insert )
// ** book-keeping arrays like domain_map, pair_moved
//
// ** also monte_carlo will take stored data from get_current_MC()
//    which returns the current Monte Carlo book-keeping object
//
// torsion modifications have to occur through calls to pose.set_*
//
// changing the torsions in misc will have no effect on the next refold,
// just might screw up rama,paa,etc evaluations ( checking for this in after_scoring)
//
// so the torsion-modifying routines like small/shear-moves, choose_fragment,
// and unpack_phipsi have to work with the pose.
//
// refold_pose ( Pose & pose ) will be in charge of filling misc arrays



namespace pose_ns
{
	////////////////////////////////////////////////////////////////
	// dimensioning of FArrays is done by update_dimensions(), which
	// gets total_residue, num_jump from the fold_tree

	class Pose {
	public:
		// constructor!
		Pose():
			fullatom_(false),
			num_jump_(0),      // Ensure these are 0 to signal that arrays
			total_residue_(0), // need to be dimensioned by update_dimensions()
			pose_bonds_(0), // ptr
			new_torsions(true),
			pose_constraints_(0), // ptr
			ptr_references_(1), // ptr
			native_pose_(0), // ptr
			start_pose_(0), // ptr
			refold_from_current(true),
			refolding_pose(0), // ptr
			symm_info(0), // ptr
			atom_tree_(0), // ptr
			use_actual_centroids_(false),
			vary_bond_geometry_(false)
		{}

		// destructor -- > kill data on the heap
		~Pose();

		////////////////
		// tree builders
		void
		set_fold_tree(
			Fold_tree const & fold_tree
		);

		void
		simple_fold_tree(
			int const nres
		);

		void
		fold_in_rb_deltas();

		void
		one_jump_tree(
			int const nres,
			int const jump_pos1,
			int const jump_pos2,
			int const cutpoint,
			int const root = 1
		);

		bool
		check_dimensions() const;


		/////////////////////////////
		// access to bond geometry data:

		inline
		bool
		ideal_backbone() const;

		inline
		bonds_class::Bonds const &
		bonds() const;

		FArray1D_bool
		flag_nonideal();

		// this routine will allow folding through chainbreaks
		// useful if you have a chainbreak that is currently
		// located at a cutpoint in the fold_tree, but you want
		// to switch to a tree without a cutpoint at the chainbreak
		// location
		void
		update_backbone_bonds_and_torsions();


		// puts ideal bond-lenths into segment from start to stop
		void
		insert_ideal_bonds(
			int const start,
			int const stop
		);

		void
		set_segment_extended(
			int const start,
			int const stop
		);

		void
		insert_ideal_bonds_at_fold_tree_cutpoints();

		void
		insert_ideal_bonds_at_backbone_chainbreaks(
			float const tolerance = 0.5
		);

		void
		identify_backbone_chainbreaks(
			FArray1D_bool & chainbreaks,
			float const tolerance = 1.0
		);

		void
		insert_ideal_bonds_at_cutpoint(
			int const cutpoint
		);

		void
		reset_bonds();

		void
		set_bonds(
			bonds_class::Bonds const & bonds
		);

		// access constraints
		bool
		constraints_exist() const;

		cst_set_ns::Cst_set const &
		constraints() const;

		void
		set_constraints(
			cst_set_ns::Cst_set const & constraints_in
		);

		void
		reset_constraints();

 		//lin method for check fullatom_id with pose
		bool match_aa_aav(  kin::Fullatom_id const & atom ) const ;

		kin::Fullatom_id
		get_fullatom_id( kin::Atom_id const & atom ) const;

		// doesnt copy everything; see pose_private.cc
		void copy_coords(
			Pose const & src,
			bool const trust_src_score_data
		);

		Pose & operator =( Pose const & src );

		// forget all of the internal score info
		void new_score_pose();

		// this will recalculate the torsion angles
		// and bond data (if !ideal_pos)
		void
		set_coords(
			bool const ideal_pos,
			FArray3DB_float const & Epos,
			FArray3DB_float const & fcoord,
			bool const check_missing = true
			);

		void
		set_segment_full_coord(
			int const seg_size,
			int const seg_begin,
			FArray3Da_float seg_coord
		);

		// torsion modifiers:
		// return false if the corresponding move is not allowed
		bool
		set_phi(
			int const pos,
			float const value
		);

		bool
		set_psi(
			int const pos,
			float const value
		);

		bool
		set_omega(
			int const pos,
			float const value
		);

		bool
		set_secstruct(
			int const pos,
			char const value
		);

		bool
		set_name(
			int const pos,
			std::string const value
		);

		void
		set_sequence( std::string const & seq );

		bool
		set_res(
			const FArray1D_int & value
		);

		bool
		set_res_variant(
			const FArray1D_int & value
		);

		bool set_res(
			int const pos,
			int const value
		);

		bool
		set_res_variant(
			int const pos,
			int const value
		);

		bool set_res_all(
			int const value
		);

		bool
		set_res_variant_all(
			int const value
		);

		bool
		set_chi(
			int const chino,
			int const pos,
			float const value
		);

		bool set_pdb_res(
			int const pos,
			int const value
		);

		bool set_pdb_chain(
			int const pos,
			char const value
		);

		bool set_pdb_insert_let(
			int const pos,
			char const value
		);

		bool
		set_torsion_by_number(
			int const pos,
			int const torsion,
			float const value
		);

		void
		set_pdb_information(
			Pdb_info const & p
		);

		// jumps
		bool
		set_jump(
			int const jump_number,
			const Jump & new_jump
		);

		bool
		set_jump_rb_delta(
			int const jump_number,
			int const rb_no,
			int const dir,
			double value
		);

		void
		set_jump_rb_center(
			int const jump_number,
			numeric::xyzVector_double & center
		);

		// get
		inline double
		get_jump_rb_delta(
			int const jump_number,
			int const rb_no,
			int const dir
		) const;

		inline int
		total_residue() const;

		inline int
		num_jump() const;

		inline bool
		fullatom() const;

		void
		center( int const seqpos );

		void
		transform_GL(
			FArray2D_float const & Mgl
		);

		void
		transform_Ax_plus_b(
			numeric::xyzMatrix_float const & A,
			numeric::xyzVector_float const & b
		);


		// debugging
		bool
		debug_torsions() const;

		void
		dump_pdb(
			std::string const & filename
		) const;

		void
		dump_scored_pdb(
		  std::string const & filename,
		  Score_weight_map const & w
		) ;

		static void
		open_scored_pdb_outfile(
      std::string const & filename,
			utility::io::ozstream & out
    ) ;

		void
		dump_scored_pdb( utility::io::ozstream & out, Score_weight_map const & w );

		void
		pose_to_design_output( int const run );

		// when setting fullatom flag true, if repack_rotamers is false
		// this will just set the flag and dimension the fullatom arrays
		// you still have to fill full_coord. This might be useful if you
		// are about to fill the full_coord array by copying from another
		// pose
		// if repack_rotamers is true, repack_pose(false) will be called
		// this will initialize the full_coord array from Eposition
		// and call pack_rotamers(...) to generate side chains
		// in EITHER case, the rosetta fullatom_flag will be set to true
		// by calling set_fullatom_flag_simple which does NO other setup
		void
		set_fullatom_flag(
			bool const setting,
			bool const repack_rotamers = true
		);

		void
		recover_sidechain( pose_ns::Pose & src_pose );

		void
		full_repack(
			bool const rotamers_exist
		);

		void repack(
			FArray1D_bool const & allow_repack,
			bool const include_current = true
		);

		void
		pack(
			PackerTask const & task
		);

		void pack_hack(
			bool const include_current,
			std::string const & mode
		);

		void rottrial(
			Score_weight_map const & weight_map,
			const int cycles
		);

		// called by rotamer-trials at the end, copies new rotamers to full_coord
		void after_rotamer_trials(
			int const nres,
			FArray1DB_int const & resv,
			FArray3DB_float const & xyz
		);

		//In pose_rms.
		void
		CA_orient( pose_ns::Pose const & pose );

		void
		CA_orient_subset( pose_ns::Pose const & pose, FArray1D_bool const & subset );

		void
		allatom_orient( pose_ns::Pose const & pose );

		void
		allatom_orient_subset( pose_ns::Pose const & pose, FArray1D_bool const & subset );


		inline FArray1D_char   const & secstruct  () const { return secstruct_; }
		inline FArray1D_string const & name       () const { return name_; }
		inline FArray1D_int    const & res        () const { return res_; }
		inline FArray1D_int    const & res_variant() const { return res_variant_; }
		inline FArray2D_float  const & chi        () const {
			assert (!atom_tree_); return chi_; }

		inline float phi         ( int const pos ) const;
		inline float psi         ( int const pos ) const;
		inline float omega       ( int const pos ) const;
		inline char  secstruct   ( int const pos ) const;
		inline std::string name  ( int const pos ) const;
		inline int   res         ( int const pos ) const;
		inline int   res_variant ( int const pos ) const;

		inline bool is_protein( int const pos ) const;
		inline bool is_NA     ( int const pos ) const;
		inline bool is_DNA    ( int const pos ) const;
		inline bool is_RNA    ( int const pos ) const;

		inline numeric::xyzVector_float xyz( kin::Atom_id const & id ) const;

		inline int   basepair_partner ( int const pos ) const;
		inline void set_basepair( int const pos, int const val )
				{ basepair_partner_( pos ) = val; };

		// return std::string of the sequence
		std::string
		sequence() const;

		inline bool symmetric() const;
		inline int total_residue_for_scoring() const;
		inline int total_pseudo_residue() const;
		inline bool pseudo( int const seqpos ) const;
		void
		setup_symm_info(
			int const nres_monomer,
			int const njump_monomer,
			int const N,
			std::string const & type
		);
		void
		setup_symm_info(
      int const nres_monomer,
      int const njump_monomer,
      int const N,
			int score_monomer,
      std::string const & type
    );

		void
		setup_symm_info( Symmetry_info const & s );

		inline Symmetry_info const & symmetry_info() const;

		inline float
		chi(
			int const chino,
			int const pos
		) const;

		inline float get_torsion_by_number(
			int const pos,
			int const torsion
		) const;

		// copy just sidechain coords for a residue
		// unsafe: must be followed eventually by a call to update_chi
		// find better solution.. update_chi_rsd( seqpos ) method?
		void copy_sidechain( const int seqpos, int const aa, int const aav,
												 FArray2Da_float fcoord, bool keep_bb = true,
												 bool align_bb = true );


		void
		delete_segment(
			int const seg_begin,
			int const seg_end
		);

		// copy a stretch of coordinates/torsions from another pose
		void
		copy_segment(
			int const size,
			Pose const & src,
			int const begin,
			int const src_begin,
			bool const update_jumps = true // should we update any affected jumps?
		);

		//// get/set allow move maps //
		void set_allow_move( kin::Kin_torsion_type const type, bool const setting);

		void show_allow_move();

		// bb
		//FArray1D_bool const & get_allow_bb_move () const;
		void
		retrieve_allow_move(
			FArray1D_bool & bb_move,
			FArray1D_bool & chi_move,
			FArray1D_bool & jump_move
		) const;
		bool get_allow_bb_move ( int const seqpos ) const;
		void set_allow_bb_move( int const pos, bool const setting );
		void set_allow_bb_move ( bool const setting );
		void set_allow_bb_move ( const FArray1D_bool & setting );
		// jump
		//inline FArray1D_bool const & get_allow_jump_move () const;
		bool get_allow_jump_move ( int const seqpos ) const;
		void set_allow_jump_move ( bool const setting );
		void set_allow_jump_move ( int const jump_number, bool const setting );
		void set_allow_jump_move ( const FArray1D_bool & setting );
		void set_allow_rb_move( int const rb_no, int const jump_number,
														bool const setting );
		bool get_allow_rb_move( int const rb_no, int const jump_number ) const;
		// chi
		//inline FArray1D_bool const & get_allow_chi_move () const;
		bool get_allow_chi_move ( int const seqpos ) const;
		void set_allow_chi_move ( bool const setting );
		void set_allow_chi_move ( int const pos, bool const setting );
		void set_allow_chi_move ( const FArray1D_bool & setting );
		// bb+chi
		void set_allow_bbchi_move(
			const FArray1D_bool & bb_move,
			const FArray1D_bool & chi_move
		);

		// get allow-move derived data
		int get_total_insert () const;
		FArray1D_int const & get_insert_size () const;
		FArray1D_int const & get_insert_map ( int & total_insert_out ) const;

		void
		copy_to_misc() const;

		// prevents refolding through non-ideal stuff
		// believes that the torsions and coordinates are in sync
		//void accept_coordinates();

		// generate coordinates from torsions
		// uses the new_XXX_refold arrays; sets to false at the end unless
		// refold_from_current == false
		// sets the new_XXX_score arrays true based on new_XXX_refold arrays
		// calls score_data->after_refolding() at the end

		// this silly routine just calls refold with the current chi
		// angles. The effect is to switch symmetric sidechains
		// so that chi angles calculated from the coordinates are in
		// dunbrack standard ranges. This is not useful for decoys, just
		// for native structures read in from the pdb.
		// it will change the score slightly if the rotamers are not ideal
		// eg 1b72 has PHE that is a little warped.
		void refold_sidechains_from_chi();

		void fold_from_here(); // save current coords for refolding from
		void fold_from_current(); // revert to default behavior
		//Pose const & get_refolding_pose() const;

		// fills internal score data, re-uses as appropriate
		// sets new_XXX_score arrays to false at the end
		// calls score_data->after_scoring() at the end
		float score( pose_param::Scoring_Function score_fxn );
		float score( const Score_weight_map & wt_map );

		void
		main_minimize(
			const Score_weight_map & wt_map,
			std::string const & min_type
		);

		void
		redesign( FArray2D_bool const & design_matrix_in,
							bool const include_current = true
							);

		void
		redesign(	FArray2D_bool const & design_matrix_in,
							PackerWeights design_weight );


		// modify the pdb information
		inline Pdb_info & pdb_info() {return pdb_info_;}
		inline Pdb_info const & pdb_info() const {return pdb_info_;}

		// view / modify the fold_tree
		inline Fold_tree const & fold_tree() const {return fold_tree_;}

		bool
		is_cutpoint( int const seqpos ) const
		{ return fold_tree_.get_is_cutpoint()(seqpos); }

		bool
		is_jump_point( int const seqpos ) const
		{ return fold_tree_.get_is_jump_point()(seqpos); }

		// access score book-keeper
		// should check that it has been updated
		// uses new_XXX_score arrays
		FArray1D_int const & get_domain_map() const;
		FArray1D_bool const & get_res_moved() const;
		FArray2D_bool const & get_pair_moved() const;

		// this is needed by jmp_refold_tree:
		// note that if refold_from_current == false, these will return
		// the new_XXX_refold arrays from refolding_pose
		FArray1D_bool const & get_new_bb_refold() const;
		FArray1D_bool const & get_new_jump_refold() const;
		FArray1D_bool const & get_new_chi_refold() const;

		inline
		Jump const &
		get_jump( int const jump_number ) const;

		///////////////////
		// access to coords
		inline
		const FArray3D_float &
		Eposition() const;

		inline
		const FArray2D_float &
		centroid() const;

		inline
		numeric::xyzVector_float
		centroid( int const seqpos ) const;

		inline
		const FArray3D_float &
		full_coord() const;

		// just reutrn full_coord_ which uses in packer to avoid refold() in packer
		inline
		const FArray3D_float &
		full_coord_simple_return_packer() const;


		inline
		numeric::xyzVector_float
		full_coord( int const atomno, int const seqpos ) const;

		// coordinate setters
		// these will trigger recalculation of the torsions
		void set_Eposition (
			FArray3Da_float Epos,
			bool const ideal_pos
		);
		// backbone here should agree with Eposition
		// call set_Eposition first
		void set_full_coord(
			FArray3Da_float fcoord
		);

		// set sidechian coordinates from rotamer set
		void set_rotcoord(const int & total_residue,
											const RotamerSet & rotamer_set,
											const int & rot_index
											);

		inline
    void set_coords_pos(const int& ires,
                        const int& iatom,
                        const int& ixyz,
                        const float& fxyz){
      full_coord_(ixyz, iatom, ires) = fxyz;
      update_chi_residue(ires);
    }



		const FArray4D_float &
		get_overlap_Eposition(
			int const overlap
		) const;

		// for setting extra scores by hand:
		// this is useful if you want some extra quantity to be output
		// on the scoreline / in the final decoy
		void set_extra_score( std::string const & name, float const value );
		void reset_extra_scores();

		bool has_extra_score( std::string const & name );
		void unset_extra_score( std::string const & name );
		float get_extra_score( std::string const & name );

		// for clearing score_data by hand: --chu
		void reset_score_data();

		////////////////////////////////
		// access native and start poses
		void
		set_native_pose(
			Pose const & pose
		);

		void
		set_native_pose(
		);

		void
		reset_native_pose();

		bool
		native_pose_exists() const;

		Pose const &
		native_pose() const;

		bool
		start_pose_exists() const;

		void
		set_start_pose(
			Pose const & pose
		);

		void
		reset_start_pose();

		Pose const &
		start_pose() const;

		// score_data interface: ////////////////////////////
		// have 2 over-loaded versions of each get/set
		// if the score state is not part of the call, then
		// we assert that the state is GOOD
		//
		//0D
		inline void set_0D_score( const Score_name & name , float const value );
		inline float & set_0D_score( const Score_name & name , Score_state & state );
		inline float & set_0D_score( const Score_name & name );
		inline float   get_0D_score( const Score_name & name , Score_state & state ) const;
		inline float   get_0D_score( const Score_name & name ) const;
		//1D
		inline FArray1D_float & set_1D_score( const Score_name & name , Score_state & state );
		inline FArray1D_float & set_1D_score( const Score_name & name );
		inline FArray1D_float const & get_1D_score( const Score_name & name , Score_state & state ) const;
		inline FArray1D_float const & get_1D_score( const Score_name & name ) const;
		//2D
		inline FArray2D_float & set_2D_score( const Score_name & name , Score_state & state );
		inline FArray2D_float & set_2D_score( const Score_name & name );
		inline FArray2D_float const & get_2D_score( const Score_name & name , Score_state & state ) const;
		inline FArray2D_float const & get_2D_score( const Score_name & name ) const;
		//3D
		inline FArray3D_float & set_3D_score( const Score_name & name , Score_state & state, int const size1, int const size2, int const size3 );
		inline FArray3D_float & set_3D_score( const Score_name & name, int const size1, int const size2, int const size3 );
		inline FArray3D_float const & get_3D_score( const Score_name & name , Score_state & state ) const;
		inline FArray3D_float const & get_3D_score( const Score_name & name ) const;

		///////////////// other

		std::string show_scores() const;
		void show_scores      ( std::ostream & os ) const;
		void show_score_values( std::ostream & os ) const;
		void show_score_tags  ( std::ostream & os ) const;


		inline
		std::map< std::string, float >
		get_extra_scores() const	{	return extra_scores; }

		inline
		void
		set_extra_scores( std::map< std::string, float > const extra_scores_input )
		{	extra_scores = extra_scores_input; }

		//////////////// chu add some for docking
		int residue_center_of_mass( int const start, int const stop ) const;
		/// overloaded version of the same as above, Monica Berrondo
		int residue_center_of_mass( std::vector < numeric::xyzVector<double> > const partner_positions ) const;

		// function called to return the residue nearest to a location in space
		int return_nearest_residue( int const begin, int const positions_size, numeric::xyzVector_float center ) const;

		//////////////// Atom_tree methods
		void
		reattach_atom(
			kin::Atom_id const & root_id,
			kin::Atom_id const & anchor_id
		);

		void
		attach_ligand(
			int const lig_aa,
			int const lig_aav,
			int const anchor_atomno,
			int const anchor_rsd,
			int const lig_atomno,
			FArray2Da_float lig_coord,
			bool const attach_by_jump
		);

		inline
		kin::Atom_tree const *
		atom_tree() const { return atom_tree_; }

		inline
		void
		set_allow_move( int const atomno, int const rsd,
										kin::Kin_torsion_type const type, bool const setting ){
			atom_tree_->atom( atomno, rsd )->set_allow_move( type,setting );
		}

		inline
		void
		set_allow_move( kin::Atom_id const & atom_id,
										kin::Kin_torsion_type const type, bool const setting ){
			//std::cout<<"set_allow_move: "<< atom_id << " " << type << " " << setting << std::endl;
			atom_tree_->atom( atom_id )->set_allow_move( type,setting );
		}

		inline
		void
		set_allow_move( kin::Torsion_id const & id, bool const setting ) {
			atom_tree_->atom( id.atom_id() )->set_allow_move( id.type(), setting );
		}

		void
		setup_atom_tree();

		void
		setup_atom_tree_preserving_allow_move();

		void
		clear_atom_tree();

		inline
		kin::Atom const *
		get_atom_tree_atom( int const atomno, int const rsd ) const {
			return atom_tree_->atom( atomno, rsd );
		}

		inline
		kin::Atom const *
		get_atom_tree_atom( kin::Atom_id const & id ) const {
			return atom_tree_->atom( id.atomno(), id.rsd() );
		}

		kin::Stub
		upstream_jump_stub( int const jump_number ) {
			assert( atom_tree_ && fullatom_ );
			if ( new_torsions ) refold();
			kin::Coords_FArray_const coords( full_coord_ );
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			return atom_tree_->get_jump_atom( seqpos )->get_input_stub(coords);
		}

		kin::Stub
		downstream_jump_stub( int const jump_number ) {
			assert( atom_tree_ && fullatom_ );
			if ( new_torsions ) refold();
			kin::Coords_FArray_const coords( full_coord_ );
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			return atom_tree_->get_jump_atom( seqpos )->get_stub(coords);
		}

		float
		get_distance(
       kin::Atom_id const & a1,
			 kin::Atom_id const & a2 ) const;

		float
		get_angle(
       kin::Atom_id const & a1,
			 kin::Atom_id const & a2,
			 kin::Atom_id const & a3 ) const;

		float
		get_torsion_angle(
       kin::Atom_id const & a1,
			 kin::Atom_id const & a2,
			 kin::Atom_id const & a3,
			 kin::Atom_id const & a4 ) const;

		inline
		float
		get_atom_tree_torsion( int const atomno, int const rsd,
													 kin::Kin_torsion_type const type ) const {
			return atom_tree_->atom( atomno, rsd )->get_torsion( type );
		}

		inline
		float
		get_atom_tree_torsion( kin::Torsion_id const & id ) const {
			return atom_tree_->torsion( id );
		}

		inline
		void
		set_atom_tree_torsion( int const atomno, int const rsd,
			kin::Kin_torsion_type const type, float const value ) {
			new_torsions = true;
			atom_tree_->atom( atomno, rsd )->set_torsion( type, value );
		}

		inline
		void
		set_atom_tree_torsion( kin::Atom_id const & atom_id,
			kin::Kin_torsion_type const type, float const value ) {
			new_torsions = true;
			if( cst_set_ns::debug_output ) {
				std::cout<<"set_atom_tree_torsion "<< atom_id << " "
								 << type << " " << value << std::endl;
			}
			atom_tree_->atom( atom_id )->set_torsion( type, value );
		}

		inline
		void
		set_atom_tree_torsion( kin::Torsion_id const & id, float const value ) {
			new_torsions = true;
			atom_tree_->atom( id.atom_id() )->set_torsion( id.type(), value );
		}

		inline
		void
		set_atom_tree_moved()
		{
			domain_map_updated = false;
			atom_tree_->root()->set_torsion_moved( true, false, true );
		}

		inline
		void
		set_use_actual_centroids( bool const value){
			use_actual_centroids_ = value;
		}

		inline
		void
		set_vary_bond_geometry_flag( bool const value){
			vary_bond_geometry_ = value;
		}

		inline bool
		get_vary_bond_geometry_flag() const {
			return vary_bond_geometry_;
		}

	private:
		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////
		///////////////////////// private /////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////

		Fold_tree fold_tree_;

		// fullatom?
		bool fullatom_;

		// these two vars are derived from fold_tree. They are
		// kept synced by update_dimensions(), which is also responsible for
		// dimensioning all the member FArrays of the Pose object.
		int num_jump_;
		int total_residue_;

		// torsions:
		FArray1D< Jump > jump_transform; // now includes rb deltas

		FArray1D_float phi_;
		FArray1D_float psi_;
		FArray1D_float omega_;
		FArray2D_float chi_;

		FArray1D_char secstruct_;
		FArray1D_string name_;

		// backbone bond geometry (initialized to 0 ==> ideal bond geometry)
		bonds_class::Bonds * pose_bonds_;

		// sequence
		FArray1D_int res_;
		FArray1D_int res_variant_;

		// pdb information
		Pdb_info pdb_info_;

		// position arrays
		// why are these mutable??
		// well, to make refolding transparent I had to adopt an
		// on-the-fly updating scheme: const methods like get_Eposition
		// have to be able to call refold() if the torsions have
		// been changed
		//
		// but also, now that I think about it, this is a consequence
		// of the fact that the coordinates and torsion angles are
		// redundant sources of information. In this case, we are
		// choosing the torsions as the "primary" description.
		//

		mutable FArray3D_float Eposition_;
		mutable FArray3D_float full_coord_;
		mutable FArray2D_float centroid_;
		mutable FArray4D_float overlap_Eposition_;

		// allow-move arrays
		FArray1D_bool allow_bb_move;
		FArray1D_bool allow_chi_move;
		FArray1D_bool allow_jump_move;
		FArray2D_bool allow_rb_move;

		// for fragment insertions, mutable to allow updating on the fly
		mutable int total_insert;
		mutable FArray1D_int insert_map;
		mutable FArray1D_int insert_size;
		mutable bool insert_maps_updated;

		//// book-keeping
		// torsion changes:
		mutable bool new_torsions;
		mutable FArray1D_bool new_bb_refold;
		mutable FArray1D_bool new_bb_score;
		mutable FArray1D_bool new_chi_refold;
		mutable FArray1D_bool new_chi_score;
		mutable FArray1D_bool new_jump_refold;
		mutable FArray1D_bool new_jump_score;

		// differences between position arrays and stored score data
		// for exporting to other routines:
		// mutable for updating on the fly by const member functions get_*
		mutable FArray1D_int domain_map;
		mutable FArray1D_bool res_moved;
		mutable FArray2D_bool pair_moved;
		mutable bool domain_map_updated;
		mutable bool pair_moved_updated;

		// stored data on the structure for scoring
		mutable Score_data score_data; // cendist, bumps, stored scores

		// jjh base-pairing information
		FArray1D_int basepair_partner_;

		// silly extra scores that are filled just before output'ing a decoy
		// for things like decoy-time, etc, not set by scorefxn(), but by calls
		// to set_extra_score. NOT Reset every time we call refold() anymore!
		std::map< std::string, float > extra_scores;

		// pointers
		cst_set_ns::Cst_set const * pose_constraints_; // CONST

// 		// lin pointers linking to the packer cst
// 		packer_cst_ns::Packer_cst_set const * packer_constraints_; // CONST

		// The number of pointers referring to this pose. This is 1 if the pose has an
		// actual variable name, and is incremented as this pose is used in references.
		// This is needed for garbage collection of dynamically allocated "anonymous" poses
		mutable int ptr_references_;

		// pointer to native structure, for calculating rmsd, etc
		Pose const * native_pose_;
		Pose const * start_pose_;

		// handle refolding during minimization so that coords dont get
		// warped by too many translations and rotations
		bool refold_from_current;
		Pose * refolding_pose;

		Symmetry_info * symm_info;
		kin::Atom_tree* atom_tree_;

		// For fullatom poses, place centroid at actual
		// sidechain center of mass.
		bool use_actual_centroids_;

		// flag that bond distances or angles have been minimized and are neither
		// the starting values nor ideal values.
		// (Note: if one of the latter holds, information will be
		//  efficiently stored in things like pose_bonds_).
		bool vary_bond_geometry_;

		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////
		///////////////////// internal methods ////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////
		///////////////////////////////////////////////////////////////////////////

		// for dimensioning the private data arrays
		void update_dimensions(); // re-derives total_residue, num_jump from fold_tree
		void redimension_fullatom_arrays();
		void new_Eposition() const; // signal change to Eposition array

		void update_insert_maps() const;
		void update_domain_map() const;
		void update_pair_moved() const;

		void reset_refold_moved_arrays() const;
		void update_score_moved_arrays() const;
		void reset_score_moved_arrays();

		// these guys are called before and after entering score_fxn()
		// respectively:
		void before_scoring();
		void after_scoring();

		bool
		check_new_torsions() const;

		bool
		torsions_init() const;

		bool
		backbone_coords_init(
			int const seqpos
		) const;

		// private copy-constructor guarantees no pass-by-value, but we will have operator=
		Pose( Pose & ) {}

		void
		refold() const;

		void
		atom_tree_refold() const;

		void
		rebuild_centroids() const;

		void
		rebuild_centroid( int const i ) const;

		// private methods for deriving torsions from coords
		void
		update_chi();

		void
		update_jump_from_coords(
			int const seqpos
		);

		void
		update_chi_residue(
			int const seqpos
		);

		void
		jumps_from_position();

		void
		backbone_torsions_from_position();

		void
		calc_bonds();

		void
		update_full_coord() const;

		// initialize full_coord from Eposition. build HA HN:
		//void
		//setup_full_coord();

		bool
		check_chi() const;

	}; // class Pose

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	////////////////////// inline definitions ///////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////


	// helper function:
	inline
	bool
	check_xyz(
		FArray1Da_float xyz
	)
	{
		xyz.dimension(3);
		return ( std::abs( xyz(1) - pose_param::BAD_POS ) +
						 std::abs( xyz(2) - pose_param::BAD_POS ) +
						 std::abs( xyz(3) - pose_param::BAD_POS ) >= 0.1 );
	}


	inline int
	Pose::total_residue() const
	{
		if ( symm_info ) {
			return total_residue_ - symm_info->exclude_from_total_residue();
		}
		return total_residue_;
	}

	inline int
	Pose::num_jump() const
	{
		return num_jump_;
	}

	inline
	bool
	Pose::fullatom() const
	{
		return fullatom_;
	}

	inline
	bool
	Pose::ideal_backbone() const
	{
		return ( pose_bonds_ == 0 );
	}

	inline
	const bonds_class::Bonds &
	Pose::bonds() const
	{
		assert( pose_bonds_ != 0 );
		return *pose_bonds_;
	}

	inline
	FArray3D_float const &
	Pose::Eposition() const
	{
		if ( new_torsions ) refold();
		return Eposition_;
	}

	inline FArray2D_float const &
	Pose::centroid() const
	{
		if ( new_torsions ) refold();
		return centroid_;
	}

	inline
	numeric::xyzVector_float
	Pose::centroid( int const seqpos ) const
	{
		if ( new_torsions ) refold();
		return numeric::xyzVector_float( &(centroid_(1,seqpos)));
	}

	inline const FArray3D_float &
	Pose::full_coord() const
	{
		if ( new_torsions ) refold();
		if ( !fullatom_ ) update_full_coord();
		assert( check_xyz( full_coord_ ) );
		return full_coord_;
	}


	inline const FArray3D_float &
	Pose::full_coord_simple_return_packer() const
	{
		return full_coord_;
	}

	inline
	numeric::xyzVector_float
	Pose::full_coord( int const atomno, int const seqpos ) const {
		if ( new_torsions ) refold();
		return numeric::xyzVector_float( &(full_coord_(1,atomno,seqpos) ) );
	}

	inline float
	Pose::phi(
		int const pos
	) const
	{
		if ( atom_tree_ ) {
			return atom_tree_->get_rosetta_torsion( param_torsion::phi_torsion, pos);
		}
		return phi_(pos);
	}

	inline float
	Pose::psi(
		int const pos
	) const
	{
		if ( atom_tree_ ) {
			return atom_tree_->get_rosetta_torsion( param_torsion::psi_torsion, pos);
		}
		return psi_(pos);
	}

	inline float
	Pose::omega(
		int const pos
	) const
	{
		if ( atom_tree_ ) {
			if ( fold_tree_.get_is_cutpoint()(pos) ) return 180.0;
			return atom_tree_->get_rosetta_torsion(param_torsion::omega_torsion,pos);
		}
		return omega_(pos);
	}

	inline
	numeric::xyzVector_float
	Pose::xyz( kin::Atom_id const & id ) const
	{
		if ( new_torsions ) refold();
		return numeric::xyzVector_float
			( &(full_coord_( 1, id.atomno(), id.rsd() ) ) );
	}

	inline int
	Pose::res(
		int const pos
	) const
	{
		return res_(pos);
	}

	inline int
	Pose::res_variant(
		int const pos
	) const
	{
		return res_variant_(pos);
	}

	inline char
	Pose::secstruct(
		int const pos
	) const
	{
		return secstruct_(pos);
	}

	inline std::string
	Pose::name(
		int const pos
	) const
	{
		return name_(pos);
	}

	inline bool
	Pose::is_protein(
		int const pos
	) const
	{
		return param_aa::is_protein( res_(pos) );
	}

	inline bool
	Pose::is_NA(
		int const pos
	) const
	{
		return param_aa::is_NA( res_(pos) );
	}

	inline bool
	Pose::is_DNA(
		int const pos
	) const
	{
		return param_aa::is_DNA( res_(pos) );
	}

	inline bool
	Pose::is_RNA(
		int const pos
	) const
	{
		return param_aa::is_RNA( res_(pos) );
	}

	inline float
	Pose::chi(
		int const chino,
		int const pos
	) const
	{
		if ( atom_tree_ ) {
			int const torsion_number( 3 + chino ); // offset by phi,psi,omega
			return atom_tree_->get_rosetta_torsion( torsion_number, pos );
		}
		assert( pos<= total_residue_ && chino <= 4 );
		return chi_( chino, pos );
	}

	inline
	const Jump &
	Pose::get_jump(
		int const jump_number
	) const
	{
		if ( atom_tree_ ) {
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			kin::Atom* atom( atom_tree_->get_jump_atom( seqpos ) );
			if ( atom ) {
				return atom->jump();
			} else {
				std::cout << "WARNING: Pose::get_jump: cant find jump atom: " <<
					jump_number << std::endl;
				std::exit( EXIT_FAILURE );
			}
		}
		return jump_transform ( jump_number );
	}

	inline
	double
	Pose::get_jump_rb_delta(
		int const jump_number,
		int const rb_no,
		int const dir
	) const
	{
		if ( atom_tree_ ) {
			int const seqpos( fold_tree_.downstream_jump_residue( jump_number ) );
			kin::Atom* atom( atom_tree_->get_jump_atom( seqpos ) );
			if ( atom ) {
				return atom->jump().get_rb_delta( rb_no, dir );
			} else {
				std::cout << "WARNING: Pose::get_jump: cant find jump atom: " <<
					jump_number << std::endl;
				std::exit( EXIT_FAILURE );
			}
		}
		return jump_transform( jump_number ).get_rb_delta( rb_no, dir );
	}


// 	inline
// 	bool
// 	dna_exists() const
// 	{
// 		return ( dna != 0 );
// 	}

	inline bool
	Pose::symmetric() const
	{
		return ( symm_info != 0 );
	}

	inline
	bool
	Pose::pseudo( int const seqpos ) const
	{
		return symmetric() && ( seqpos + symm_info->npseudo() > total_residue_ );
	}

	inline
	int
	Pose::total_pseudo_residue() const
	{
		return ( symmetric() ? symm_info->npseudo() : 0 );
	}

	inline
	int
	Pose::total_residue_for_scoring() const
	{
		return ( symmetric() ?
						 total_residue_ - symm_info->npseudo() : total_residue_ );
	}

	inline
	Symmetry_info const &
	Pose::symmetry_info() const
	{
		assert( symm_info );
		return *symm_info;
	}

	inline int
	Pose::basepair_partner(
		int const pos
	) const
	{
	return basepair_partner_(pos);
	}

	inline
	float
	Pose::get_torsion_by_number(
		int const pos,
		int const torsion
	) const
	{
		// check for non-protein, currently only alternative is dna
		if ( atom_tree() ) {
			// handle atom_tree stuff
			return atom_tree_->get_rosetta_torsion( torsion, pos );
		} else if ( torsion == param_torsion::phi_torsion ) {
			return phi( pos );
		} else if ( torsion == param_torsion::psi_torsion ) {
			return psi( pos );
		} else if ( torsion == param_torsion::omega_torsion ) {
			return omega( pos );
		} else if ( torsion >= param_torsion::chi1_torsion &&
								torsion <= param_torsion::chi4_torsion ) {
			return chi( torsion - param_torsion::chi1_torsion + 1, pos );
		} else {
			std::cout << "Pose::get_torsion_by_number: cant handle rb-torsions: "
								<< torsion << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return 0.0; // shouldnt hit here
	}

	///////////////////////////////////////////////////////////////
	//// score_data interface /////////////////////////////////////
	///////////////////////////////////////////////////////////////
	// this very long section contains over-loaded get and set
	// functions for score_data
	//
	// NOTE: the get/set_?D( name ) versions will DIE if either
	// the score doesnt exist or the state is not already GOOD
	// so you have to pass in a state variable if you want to catch this
	//
	// So that the get_* methods can be CONST, we dont create any new
	// score entries if they are called with non-existent scores
	// the (name) version will just DIE, while the (name,state) version
	// will return a reference to a dummy FArray and state = BAD
	//
	// ALSO NOTE that the set_* methods leave the score in state GOOD
	// after the call. So if you call a set_* method you are responsible
	// for completely updating it. The state value returned in
	// set_*( name, state ) is the state of score BEFORE the call.
	//
	//////////////////////////// 0D ////////////////////////////////
	// set 0D, pass in desired value
	// currently this overloaded version only exists for 0D scores
	inline void Pose::set_0D_score( const Score_name & name, float const value ) {
		int const dimension (0);
		if ( new_torsions ) refold();
		score_data.set_score ( name, dimension, total_residue_ ).set0() = value;
	}
	// set 0D
	inline float & Pose::set_0D_score( const Score_name & name,
																		 Score_state & state ) {
		int const dimension (0);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, total_residue_ ) );
		state = score.get_state();
		return score.set0(); // automatically sets the state to GOOD
	}
	// set 0D, assert already GOOD
	inline float & Pose::set_0D_score( const Score_name & name ) {
		int const dimension (0);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, total_residue_ ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "set_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.set0(); // automatically sets the state to GOOD
	}
	// get 0D:
	inline float Pose::get_0D_score( const Score_name & name,
																	 Score_state & state ) const
	{
		int const dimension (0);
		if ( new_torsions ) refold();
		static FArray1D_float const dummy;
		if ( !score_data.has_score( name ) ) {
			state = BAD;
			return 0.0;
		} else {
			Stored_score const & score
				( score_data.get_score ( name, dimension, total_residue_ ) );
			state = score.get_state();
			return score.get0();
		}
	}
	// get 0D, assert already exists and GOOD
	inline float Pose::get_0D_score( const Score_name & name ) const
	{
		int const dimension (0);
		if ( new_torsions ) refold();
		Stored_score const & score // will DIE if score doesnt already exist:
			( score_data.get_score ( name, dimension, total_residue_ ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "get_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.get0();
	}
	//////////////////////////// 1D ////////////////////////////////
	// set 1D
	inline FArray1D_float & Pose::set_1D_score( const Score_name & name,
																							Score_state & state ) {
		int const dimension (1);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, total_residue_ ) );
		state = score.get_state();
		return score.set1(); // automatically sets the state to GOOD
	}
	// set 1D, assert already GOOD
	inline FArray1D_float & Pose::set_1D_score( const Score_name & name ) {
		int const dimension (1);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, total_residue_ ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "set_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.set1(); // automatically sets the state to GOOD
	}
	// get 1D:
	inline FArray1D_float const & Pose::get_1D_score( const Score_name & name,
																										Score_state & state ) const
	{
		int const dimension (1);
		if ( new_torsions ) refold();
		static FArray1D_float const dummy;
		if ( !score_data.has_score( name ) ) {
			state = BAD;
			return dummy;
		} else {
			Stored_score const & score
				( score_data.get_score ( name, dimension, total_residue_ ) );
			state = score.get_state();
			return score.get1();
		}
	}
	// get 1D, assert already exists and GOOD
	inline FArray1D_float const & Pose::get_1D_score( const Score_name & name ) const
	{
		int const dimension (1);
		if ( new_torsions ) refold();
		Stored_score const & score // will DIE if score doesnt already exist:
			( score_data.get_score ( name, dimension, total_residue_ ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "get_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.get1();
	}
	//////////////////////////// 2D ////////////////////////////////
	// set 2D
	inline FArray2D_float & Pose::set_2D_score( const Score_name & name,
																							Score_state & state ) {
		int const dimension (2);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, total_residue_ ) );
		state = score.get_state();
		return score.set2(); // automatically sets the state to GOOD
	}
	// set 2D, assert already GOOD
	inline FArray2D_float & Pose::set_2D_score( const Score_name & name ) {
		int const dimension (2);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, total_residue_ ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "set_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.set2(); // automatically sets the state to GOOD
	}
	// get 2D:
	inline FArray2D_float const & Pose::get_2D_score( const Score_name & name,
																										Score_state & state ) const
	{
		int const dimension (2);
		if ( new_torsions ) refold();
		static FArray2D_float const dummy;
		if ( !score_data.has_score( name ) ) {
			state = BAD;
			return dummy;
		} else {
			Stored_score const & score
				( score_data.get_score ( name, dimension, total_residue_ ) );
			state = score.get_state();
			return score.get2();
		}
	}
	// get 2D, assert already exists and GOOD
	inline FArray2D_float const & Pose::get_2D_score( const Score_name & name ) const
	{
		int const dimension (2);
		if ( new_torsions ) refold();
		Stored_score const & score // will DIE if score doesnt already exist:
			( score_data.get_score ( name, dimension, total_residue_ ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "get_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.get2();
	}

	//////////////////////////// 3D ////////////////////////////////
	// set 3D
	// This is kind of special, currently used only for RNA.
	// Need a 3D array with sizes: total_residue by total_residue by 3.
	// So go ahead and specify this explicitly as size1, size2, size3, rather than
	// filling a massive array with total_residue^3 entries!
	//
	inline FArray3D_float & Pose::set_3D_score( const Score_name & name,
																							Score_state & state, int const size1, int const size2, int const size3 ) {
		int const dimension (3);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, size1, size2, size3 ) );
		state = score.get_state();
		return score.set3(); // automatically sets the state to GOOD
	}
	// set 3D, assert already GOOD
	inline FArray3D_float & Pose::set_3D_score( const Score_name & name, int const size1, int const size2, int const size3 ) {
		int const dimension (3);
		if ( new_torsions ) refold();
		Stored_score & score
			( score_data.set_score ( name, dimension, size1, size2, size3 ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "set_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.set3(); // automatically sets the state to GOOD
	}
	// get 3D:
	inline FArray3D_float const & Pose::get_3D_score( const Score_name & name,
																										Score_state & state ) const
	{
		int const dimension (3);
		if ( new_torsions ) refold();
		static FArray3D_float const dummy;
		if ( !score_data.has_score( name ) ) {
			state = BAD;
			return dummy;
		} else {
			Stored_score const & score
				( score_data.get_score ( name, dimension, total_residue_ ) );
			state = score.get_state();
			return score.get3();
		}
	}
	// get 3D, assert already exists and GOOD
	inline FArray3D_float const & Pose::get_3D_score( const Score_name & name ) const
	{
		int const dimension (3);
		if ( new_torsions ) refold();
		Stored_score const & score // will DIE if score doesnt already exist:
			( score_data.get_score ( name, dimension, total_residue_ ) );
		if ( score.get_state() != GOOD ) {
			std::cout << "get_" << dimension << "D_score( " << name <<
				" ) called for non-up-to-date score!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		return score.get3();
	}

	// helper functions for checking if coords are up-to-date
	inline void reset_xyz( FArray1Da_float xyz ) {
		xyz.dimension(3);
		xyz(1) = pose_param::BAD_POS;
		xyz(2) = pose_param::BAD_POS;
		xyz(3) = pose_param::BAD_POS;
	}


	// more inline definitions


	/** some thoughts:
			get_*() returns const references, set_*() returns non-const references
			  for fold_tree and score_data, res_moved, pair_moved, etc

			score():
				At the end -- notify score data
				that we are done. Any data that hasn't been touched is
				reset to the empty state by score data function.
				sets new_XXX_score false at the end of the call

			Score_data:
			  holds single scores, 1d and 2d scores
	**/

	/////////////////////////////////////////////////////////////////////////////
	class Monte_carlo {
	private:
		Pose best_pose_;
		Pose low_pose_;
		float temperature_;
		Score_weight_map weight_map_;
		bool autotemp_;
		float quench_temp_;
		int last_accept_;
		float last_score_;
		bool save_mc_accepted_;
		int mc_accepted_;
		std::map< std::string, int > trial_counter;
		std::map< std::string, int > accept_counter;
		std::map< std::string, float > energy_drop_counter;

		// for checkpointing
		bool do_checkpointing;
		std::string checkpointfiletag;
		std::string currposeoutfile;
		std::string lowposeoutfile;
		std::string bestposeoutfile;
		std::string checkpointfile;
		int mc_checkpoint_isfullatom;
		int mc_checkpoint_ntrials;

	public:
		Monte_carlo(
			Pose const & init_pose,
			Score_weight_map const & weight_map,
			float const temperature
		);
		void save_mc_accepted();
		void set_temperature( float const temp );
		void set_autotemp( bool const setting, float const quench_temp );
		bool boltzmann( Pose & pose, std::string const & move_type = "unk", bool const & movie = "true" );
		void reset( Pose const & pose );
		Pose const & best_pose() const;
		Pose const & low_pose() const;
		void recover_low( Pose & pose );
		// re-scores best,low
		void set_weight_map( Score_weight_map const & weight_map );
		Score_weight_map const & weight_map() const;
		void show_scores() const;
		float last_score() const;
		void reset_counters();
		void store_counters_in_pose( Pose & pose );
		void show_counters() const;

		inline float best_score() const { return best_pose_.get_0D_score( SCORE );}
		inline float low_score() const { return low_pose_.get_0D_score( SCORE );}
		inline float temperature() const { return temperature_;}

		// for checkpointing
		void restore_checkpoint_counts();
		bool time_to_checkpoint();
		void restore_pose_from_checkpoint( Pose & pose, std::string const & outfile, bool const & isfullatom );
		bool checkpoint( Pose & pose );

	private:
		// for managing the temperature, if we need to do so
		void
		autotemp_reject();

		void
		autotemp_accept();

		void
		update_monte_carlo_globals( int const accept_type );

	};

	/////////////////////////////////////////////////////////////////////////////
	inline
	void
	pose_update_MAX_RES(
		Pose const & pose
	)
	{
		if ( !param::MAX_RES().initialized() ||
				 param::MAX_RES()() < pose.total_residue() ) {
			std::cout << "WARNING:: pose.total_residue() > " <<
				"param::MAX_RES()() -- updating MAX_RES\nThis will erase " <<
				"any data in MAX_RES sized arrays, eg fragments." << std::endl;
			param::MAX_RES_assign_res( pose.total_residue() ); // Set MAX_RES
		}
	}
}

void
pose_to_native(pose_ns::Pose const & native_pose);

void
pose_to_low(pose_ns::Pose const & low_pose);

void
pose_to_best(pose_ns::Pose const & best_pose);

void
fill_misc_mc_score( float const & score, float const & best_score, float const & low_score, float const & rms_err, float const & low_rms, float const & best_rms);

#endif
