// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20531 $
//  $Date: 2008-02-21 17:00:06 -0800 (Thu, 21 Feb 2008) $
//  $Author: chu $


// Rosetta Headers
#include "pose_benchmark.h"
#include "after_opts.h"
//#include "param.h"
#include "barcode_stats.h" // testing
#include "files_paths.h"
#include "fragment_class.h"
#include "fragments_pose.h"
#include "jumping_util.h"
#include "jumping_minimize.h"
#include "map_sequence.h"
#include "minimize.h"
#include "nblist.h"
#include "param_pack.h"
#include "pose.h"
#include "pose_abinitio.h"
#include "pose_constraints.h"
#include "pose_idealize.h"
#include "pose_design.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "prof.h"
#include "runlevel.h"
#include "score.h"
#include "silent_input.h"
#include "util_basic.h"
//#include "vanita.h"

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <fstream>

//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>


///////////////////////////////////////////////////////////////////////////////
//
// useful debugging routine
//
// checks all the components of the weight_map one at a time
//

void
pose_deriv_check(
	pose_ns::Pose const & start_pose,
	pose_ns::Score_weight_map const & start_w,
	std::string const & tag
)
{
	using namespace pose_ns;

	/////////////////////////
	// minimizer diagnostics:
	bool const save_deriv_check( minimize_get_deriv_check() );
	minimize_set_deriv_check( true );

	{ // first check the full weight map
		Pose pose;
		pose = start_pose;

		pose.main_minimize( start_w, "linmin" );

		std::cout << "start_w: "; start_w.show();
		minimize_show_deriv_check_stats( std::cout );
		std::cout <<  " full_weight_map - " << tag << std::endl;
	}


	Score_weight_map w2( start_w );

	// now loop over all the non-zero weights and check derivs
	for ( Score_weight_map::Weight_map::const_iterator it = start_w.begin();
				it != start_w.end(); ++it ) {
		Score_name const & name( it->first );

		float const setting( it->second );
		if ( std::abs( setting ) > 1e-3 ) {

			{ // minimize this guy by himself
				Score_weight_map w;
				w.set_weight( name, setting );

				Pose pose;
				pose = start_pose;

				pose.main_minimize( w, "linmin" );

				minimize_show_deriv_check_stats( std::cout );
				std::cout <<  ' ' << Score_data::name2string(name) << ' ' <<
					name << ' ' << tag << std::endl;

			} // non-zero weight


			{
				// now set this guy to zero
				w2.set_weight( name, 0.0 );
				Pose pose;
				pose = start_pose;

				pose.main_minimize( w2, "linmin" );

				minimize_show_deriv_check_stats( std::cout );
				std::cout <<  ' ' << Score_data::name2string(name) << "-zeroed " <<
					name << ' ' << tag << std::endl;
			}
		}

	} // loop over the individual scores

	minimize_set_deriv_check( save_deriv_check );
}

///////////////////////////////////////////////////////////////////////////////
void
pose_benchmark()
{
	using namespace pose_ns;

	utility::io::ozstream log( stringafteroption("log_file").c_str() );

	// silent I/O benchmark
	pose_test_silent_io( false, log ); // centroid
	pose_test_silent_io( true, log ); // fullatom

	// minimization benchmark
	pose_test_fa_min( log );

	// loop bmark1
	pose_simple_loop_test( log );

	// test pose_idealize
	pose_test_fa_idealize( log );

	// close the log
	log.close();

	// silly message for autobench
	std::cout << "DONE :: pose_test1" << std::endl;
	exit(0);

}

///////////////////////////////////////////////////////////////////////////////
void
pose_test_silent_io(
	bool const fullatom,
	std::ostream & log
)
{
	using namespace pose_ns;
	using namespace silent_io;
	log << "pose_test_silent_io:: fullatom= " << fullatom << '\n';

	// setup scorefxn
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, fullatom ? score12 : score4 );

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// first, read and score a set of decoys:
	{
		Pose pose;

		std::string filename( benchmark_filename( fullatom ? "aai001_fa.out" :
																							"aai001_cen.out" ) );
		Silent_file_data data( filename, fullatom );

		// loop over the decoys in the silent file
		for ( Silent_file_data::const_iterator it = data.begin(),
						it_end = data.end(); it != it_end; ++it) {
			std::string const tag( it->first );
			Silent_structure const & structure( *(it->second) );

			structure.fill_pose( pose, true ); // check ca coord deviation

			pose.score( weight_map );
			log << "reconstruct and score: " << tag << ' ' <<
				pose.show_scores() << '\n';
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// read a pdb, write as silent file, read back in -- compare rmsd, score
	{
		Pose pose;

		// read from pdb file
		std::string pdb_filename( benchmark_filename("1b72.pdb") );
		bool const ideal_pose( false ); // non-ideal backbone geometry
		pose_from_pdb( pose, pdb_filename, fullatom, ideal_pose );
		// flip symmetric sidechains to match rosetta's chi-angle ranges
		if ( fullatom ) pose.refold_sidechains_from_chi();
		pose.score( weight_map );
		log << "score: " << pdb_filename << ' ' << pose.show_scores() << '\n';
		float const start_score( pose.get_0D_score( SCORE ));

		// write to silent file
		std::string silent_filename( fullatom ? "pt1_fa.out" : "pt1_cen.out" );
		Silent_out out( silent_filename );
		if ( fullatom ) out.store_rotamers( pose ); // non-ideal sidechains
		out.write( "test1", pose );

		// update fold_tree, write to silent file again
		pose.one_jump_tree( pose.total_residue(), 5, 20, 10 );
		pose.score( weight_map );
		log << "rescore_w_new_tree: " << pdb_filename <<
			" score_delta= " << pose.get_0D_score( SCORE ) - start_score << '\n';
		out.write( "test2", pose );

		// read back in
		Pose new_pose;
		Silent_file_data data( silent_filename, fullatom );
		for ( int repeat=1; repeat<=2; ++repeat ) {
			std::string tag( "test" + string_of( repeat ) );
			data.get_structure( tag ).fill_pose( new_pose, true );
			new_pose.score( weight_map );
			float const final_score( new_pose.get_0D_score( SCORE ));

			// calculate rmsd:
			float rms;
			if ( fullatom ) {
				rms = allatom_rmsd( pose, new_pose );
			} else {
				rms = CA_rmsd( pose, new_pose );
			}

			log << "read back in: " << pdb_filename << ' ' << tag <<
				" rms_to_start= " << rms <<
				" score_delta= " <<	final_score - start_score << ' ' <<
				new_pose.show_scores() << '\n';
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
void
pose_test_fa_min(
	std::ostream & log
)
{
	using namespace pose_ns;
	using pose_param::PEPTIDE;

	// read fullatom pdb
	Pose pose, start_pose;
	bool const fullatom( true );
	std::string pdb_filename( benchmark_filename( "1b72.pdb" ) );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	pose_from_pdb( pose, pdb_filename, fullatom, ideal_pose );
	int const nres( pose.total_residue() );

	// flip symmetric sidechains to match rosetta's chi-angle ranges
	// otherwise happens the first time we call func w/ chi angles in
	// the minimizer
	pose.refold_sidechains_from_chi();
	start_pose = pose;
	start_pose.set_allow_bb_move( false );
	start_pose.set_allow_chi_move( false );
	start_pose.set_allow_jump_move( false );

	// minimizer default setup: everything can move by default
	// local_min = FALSE
	// exclude_sstype = FALSE,FALSE
	float const minimize_tolerance( 1.0 ); // very large for benchmarking!
	pose_setup_minimizer( minimize_tolerance );

	/////////////////////////
	// minimizer diagnostics:
	minimize_set_deriv_check( true );

	// setup the functions to be minimized
	std::vector< std::vector< Score_name > > chi_min_list;
	std::vector< std::vector< Score_name > > bb_min_list;

	{
		std::vector< Score_name > v;
		// atr,rep,&sol
		v.push_back( FA_ATR );
		v.push_back( FA_REP );
		v.push_back( FA_SOL );
    v.push_back( FA_ELEC );
		chi_min_list.push_back(v);
		bb_min_list.push_back(v);

		// hbonds
		v.clear();
		v.push_back( HB_SC );
		chi_min_list.push_back( v ); // sc
		v.clear();
		v.push_back( HB_SRBB);
		v.push_back( HB_LRBB);
		bb_min_list.push_back( v ); // bb
		v.push_back( HB_SC );
		bb_min_list.push_back( v ); // all

		// rama
		v.clear();
		v.push_back( RAMACHANDRAN );
		bb_min_list.push_back( v );

		// prob
		v.clear();
		v.push_back( FA_PROB );
		bb_min_list.push_back( v );

		// pair
		v.clear();
		v.push_back( FA_PAIR );
		bb_min_list.push_back( v );

		// dun -- bb + chi
		v.clear();
		v.push_back( FA_DUN );
		bb_min_list.push_back( v );
		chi_min_list.push_back( v );
	}

	// skip
	//bb_min_list.clear();
	//chi_min_list.clear();


	// set what's allowed to vary
	FArray1D_bool allow_move( nres, false );
	for ( int i=1; i<= nres; ++i ) if ( i%3 == 0 ) allow_move(i) = true;

	///////////////////////////////////////////////////
	// loop over the different weight sets to be tested
	//
	for ( int repeat=1; repeat<= 2; ++repeat ) {
		bool const bb( repeat == 1 );
		std::vector< std::vector< Score_name > > const & weights_list
			( bb ? bb_min_list : chi_min_list );

		for ( int ii=0; ii< int(weights_list.size()); ++ii ) {
			std::vector< Score_name > const & weights( weights_list[ii] );

			// fill in the weights
			Score_weight_map weight_map;
			std::string weight_tag( bb ? "BBMIN" : "CHIMIN" );
			for( std::vector< Score_name >::const_iterator it=weights.begin(),
						 it_end = weights.end(); it != it_end; ++it ) {
				weight_map.set_weight( *it, 1.0 );
				weight_tag += ( "," + Score_data::name2string( *it ) );
			}


			// setup the pose
			pose = start_pose;
			if ( bb ) {
				pose.set_allow_bb_move( allow_move );
			} else {
				pose.set_allow_chi_move( allow_move );
			}

			pose.main_minimize( weight_map, "linmin" );
			minimize_show_deriv_check_stats( log );
			log <<  ' ' << weight_tag << '\n';

		} // ii
	} // repeat

	/////////////////////////////////////////////////////////////////////////////
	// derivatives across jumps
	// including chainbreak derivative test
	int const loop1_begin( 18 ); // 1b72 turn 1
	int const loop1_end( 22 );
	int const loop2_begin( 34 ); // 1b72 turn 2
	int const loop2_end( 36 );
	assert( nres > loop2_end );


	{ ///////////////////////////////////////////////////////////////////////////
		// chainbreak
		int const cutpoint( loop1_begin+2 );
		int const jump_pos1( loop1_begin-1);
		int const jump_pos2( loop1_end+1);
		Fold_tree f;
		f.add_edge( 1, jump_pos1, PEPTIDE );
		f.add_edge( jump_pos1, cutpoint, PEPTIDE );
		f.add_edge( cutpoint+1, jump_pos2, PEPTIDE );
		f.add_edge( jump_pos2, nres, PEPTIDE );
		f.add_edge( jump_pos1, jump_pos2, 1); // jump
		f.reorder(1);

		Score_weight_map weight_map;
		weight_map.set_weight( CHAINBREAK, 1.0 );
		weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );

		// set fold_tree
		pose = start_pose;
		//pose.set_fullatom_flag( false );
		pose.set_fold_tree( f );
		pose.set_allow_jump_move( false );

		// perturb loop
		for ( int i=loop1_begin; i<= loop1_end; ++i ) {
			// pseudo-random??
 			float const phi_delta( periodic_range( i * 10000.0, 30.12345 ) );
 			float const psi_delta( periodic_range( i * 20000.0, 30.12345 ) );
			std::cout << "tor-deltas: " << phi_delta << ' ' << psi_delta <<
				std::endl;
			pose.set_allow_bb_move( i, true );
			pose.set_phi( i, periodic_range( pose.phi(i) + phi_delta , 360.0 ));
			pose.set_psi( i, periodic_range( pose.psi(i) + psi_delta , 360.0 ));
		}

		pose.main_minimize( weight_map, "linmin" );
		minimize_show_deriv_check_stats( log );
		log << " BBMIN,CHAINBREAK\n";
	}


	{ ///////////////////////////////////////////////////////////////////////////
		// jump min: delete both loops, minimize
		// the tricky thing in rigid body minimization is keeping the
		// protein from blowing up, or at least from moving far enough
		// during minimization that the nblist becomes unreliable
		//
		// this will happen for this example if FA_REP weight is left at 1.0
		//
		// TO DO: control the size of steps during linmin; have to pack
		// translational degrees of freedom into phipsi so that they are
		// comparable with angle steps taken in degrees.

		pose = start_pose;
		pose.set_allow_bb_move( true );
		pose.set_allow_chi_move( true );
		pose.set_allow_jump_move( true );

		// setup a 2 jump tree
		int const num_jump(2);
		FArray1D_int cuts( num_jump );
		FArray2D_int jump_point( 2, num_jump );
		cuts(1) = loop1_begin;
		cuts(2) = loop2_begin;
		jump_point(1,1) = loop1_begin-1;
		jump_point(2,1) = loop1_end+1;
		jump_point(1,2) = loop1_begin-1;
		jump_point(2,2) = loop2_end+1;
		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jump_point, cuts );

		// now we can delete the loops
		// note that we do it loop2 and then loop1: because the sequence
		// numbers of loop2 would change if we deleted loop1 first...
		pose.set_fold_tree(f);
		pose.delete_segment( loop2_begin, loop2_end );
		pose.delete_segment( loop1_begin, loop1_end );

		// set the allow-move arrays for minimization
		pose.set_allow_bb_move( false );
		pose.set_allow_chi_move( false );
		pose.set_allow_jump_move( true );

		// damp the repulsive
		Score_weight_map weight_map( score12 );
		weight_map.set_weight( FA_REP, 0.1 );

		// save starting point:
		Pose tmp_pose;
		tmp_pose = pose;

		// minimize: with and w/o the nblist
		pose.main_minimize( weight_map, "linmin" );
		minimize_show_deriv_check_stats( log );
		log << " JUMPMIN,SCORE12\n";

		pose_set_use_nblist( false );
		pose = tmp_pose;
		pose.main_minimize( weight_map, "linmin" );
		minimize_show_deriv_check_stats( log );
		log << " JUMPMIN,SCORE12-no-nblist\n";
		pose_set_use_nblist( true );
	}

	{ ///////////////////////////////////////////////////////////////////////////
		// test some of the pose constraint derivs
		pose = start_pose;
		pose.set_allow_bb_move( true );
		//pose.set_allow_chi_move( true );
		//pose.set_allow_jump_move( true );

		// define a constraint-set object
		cst_set_ns::Cst_set cst_set;

		// calculate atompair constraints from a pose
		// this fills the atompair constraints in cst_set
		// actually it also fills the torsion and coordinate tethers
		// but the weights on those guys are 0 below; at first, then
		// we test the phipsi torsion tether
		//
		bool const just_use_bb_heavyatoms( false );
		fill_cst_set( cst_set, pose, just_use_bb_heavyatoms );

		// perturb structure
		pose.insert_ideal_bonds( 1,nres );

		// scorefxn
		// a little tricky because the constraint score is not internally
		// normalized by the number of constraints
		//
		// in this case it doesnt matter since we have no other score
		// components. In general it's necessary to set the weight as follows
		// to get the correct balance between constraints and other scores.
		//
		Score_weight_map weight_map;
		weight_map.set_weight( ATOMPAIR_CST,
			500.0 / cst_set.count_atompair_constraints() );

		// put constraints into pose
		pose.set_constraints( cst_set );

		// minimize
		pose.set_allow_bb_move( allow_move );
		pose.main_minimize( weight_map, "linmin" );
		minimize_show_deriv_check_stats( log );
		log <<  " BBMIN,ATOMPAIR_CST\n";

		// now minimize with just torsion angle constraints
		weight_map.clear();
		weight_map.set_weight( PHIPSI_CST, 0.2 / (2*nres) );
		pose.main_minimize( weight_map, "linmin" );
		minimize_show_deriv_check_stats( log );
		log <<  " BBMIN,PHIPSI_CST\n";
	}



	minimize_set_deriv_check( false );
}

///////////////////////////////////////////////////////////////////////////////
void
pose_simple_loop_test(
	std::ostream & log
)
{
	using namespace pose_ns;

	{ // test simple loop rebuilding routines used in map_sequence.cc
		int const frag_size(3);

		Pose pdb_pose, pose;
		bool const fullatom( false );
		bool const ideal_pose( false );
		pose_from_pdb( pdb_pose, benchmark_filename("1b72.pdb"), fullatom,
									 ideal_pose );
		int const nres( pdb_pose.total_residue() );

		// define the loop
		int const loop_begin( 18 ); // 1b72 turn 1
		int const loop_end( 22 );
		int const loop_size( loop_end - loop_begin + 1 );
		assert( loop_end+1 <= nres );
		int const cutpoint( loop_begin+1 );

		// setup a fold_tree with jump from loop_begin-1 to loop_end+1
		pose = pdb_pose;
		pose.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint);

		// reset torsions and bonds in the loop region
		insert_init_frag( pose, loop_begin, loop_size );
		pose.insert_ideal_bonds( loop_begin, loop_end );

		// for debugging: setup a mapping that excludes the loop residues
		// use this to confirm that the structure is unchanged outside the
		// loop
		FArray1D_int mapping( nres );
		for ( int i=1; i<= nres; ++i ) {
			if ( loop_begin <= i && i <= loop_end ) {
				mapping(i) = -1;
			} else {
				mapping(i) = i;
			}
		}

		{ // debug rmsd over non-loop region after this change
			float const rmsd
				( CA_rmsd_by_mapping( pdb_pose, pose, mapping ) );
			log << "non-loop-rmsd_to_start= " << rmsd << '\n';
			assert( rmsd < 0.01 );
		}

		// scorefxn
		Score_weight_map weight_map( score3 );
		weight_map.set_weight( CHAINBREAK, 1.0 );
		weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );

		for ( int repeat =1; repeat <= 2; ++repeat ) {
			if ( repeat == 1 ) {
				// read standard rosetta fragment file
				read_fragments_simple( "1b72_", nres );
			} else {
				files_paths::n_frag_files().assign_if_nic( 0 );
				param::n_frag_sizes().assign_if_nic( 1 );
				std::string ss;
				for ( int i=1; i<= nres; ++i ) ss += pose.secstruct(i);
				get_frags_by_ss(ss, loop_begin, loop_end, frag_size, 200,
												true, true, true, 1 );
			}

			// list of fragments that close the loop
			std::vector< Fragment > loop_list;

			{
				// params for scored_frag_close
				int const cycles1(100);
				int const cycles2(20 * loop_size );
				int const frag_size(3);
				int const frag_offset(0); // frag file and pose numbering agree
				bool const do_ccd_moves(false );

				scored_frag_close( weight_map, pose, loop_begin, loop_end, frag_size,
					frag_offset, cycles1, cycles2, do_ccd_moves, loop_list );

				log << "scored_frag_close: found: " << loop_list.size() << " frags\n";
			}

			{
				// params for frag_close:
				int const cycles1(400), cycles2(10*loop_size),
					big_num_fragments(100), little_num_fragments(50),
					frag_close_ccd_cycles(50), frag_offset( 0 );

				frag_close( pose, loop_begin, loop_size, cutpoint,
					frag_offset, cycles1, cycles2, big_num_fragments,
					little_num_fragments, frag_close_ccd_cycles, loop_list );

				log << "frag_close: now found: " << loop_list.size() << " frags\n";
			}

			// confirm that frags really do close the loop perfectly:
			Pose noloop_pose;
			noloop_pose = pose;
			noloop_pose.simple_fold_tree( nres );
			for ( std::vector< Fragment >::const_iterator it=loop_list.begin(),
							it_end = loop_list.end(); it != it_end; ++it ) {
				it->insert( noloop_pose, loop_begin );
				noloop_pose.score( weight_map );
				float const rmsd( CA_rmsd_by_mapping( noloop_pose, pose,
																							mapping) );
				float const start_rmsd( CA_rmsd( noloop_pose, pdb_pose ));
				assert( rmsd < 0.1 );
				log << "noloop_refold: " << F(9,3,rmsd) << F(9,3,start_rmsd) <<
					F(9,3,noloop_pose.get_0D_score( SCORE )) << '\n';
			}
		} // repeat
	}
}


///////////////////////////////////////////////////////////////////////////////
void
pose_test_fa_idealize(
	std::ostream & log
)
{
	using namespace pose_ns;

	// silly example -- just idealize 3 residues -- dont want it to be
	// too slow
	int const idl_begin( 18 );
	int const idl_end( 20 );

	Pose pose, pdb_pose;
	bool const fullatom( true );
	bool const ideal_pose( false );

	pose_from_pdb( pdb_pose, benchmark_filename("1b72.pdb"), fullatom,
								 ideal_pose );

	// flip symmetric sidechains
	pdb_pose.refold_sidechains_from_chi();

	pose = pdb_pose;

	pose.set_allow_bb_move( false );
	pose.set_allow_chi_move( false );
	pose.set_allow_jump_move( false );

	// tell pose_idealize which residues to idealize:
	for ( int i=idl_begin; i<= idl_end; ++i ) {
		pose.set_allow_bb_move( i, true );
		pose.set_allow_chi_move( i, true );
	}

	idealize_pose( pose, false );

	log << "idealize_pose: " << CA_rmsd( pose, pdb_pose ) <<
		' ' << allatom_rmsd( pose, pdb_pose ) << std::endl;
}

///////////////////////////////////////////////////////////////////////////////
void
pose_test_ai(
	std::ostream & log
)
{

	// make single AI decoy with sequence of 1b72.pdb
	pose_ns::Pose pose;


	// this is a little silly: just setting nres, sequence
	bool const fullatom( false );
	bool const ideal_pose( false );
	pose_from_pdb( pose, benchmark_filename("1b72.pdb"), fullatom, ideal_pose );
	int const nres( pose.total_residue() );
	read_fragments_simple( "1b72_", nres );

	// reset backbone:
	pose.insert_ideal_bonds( 1, nres );
	insert_init_frag( pose, 1, nres );

	fold_abinitio( pose );

	log << "fold_abinitio( pose ): final_score= " <<
		pose.get_0D_score( pose_ns::SCORE ) << '\n';

}

///////////////////////////////////////////////////////////////////////////////
//
//

void
bk_min_test()
{
	using namespace pose_ns;
	using numeric::xyzVector_float;

	////////////////////
	// params: CHANGE ME
	int const lj_ramp_cycles( 8 );
	float const initial_rep_weight( 0.02 ); // make lower?
	float const final_rep_weight( 1.0 );
	float const ramp_min_tolerance( 0.001 );
	float const final_min_tolerance( 0.000001 );
	bool const use_nblist( truefalseoption( "nblist" ) );
	float const ca_dis2_threshold( 12.0 * 12.0 ); // for interface defn
	bool const use_cst( truefalseoption( "cst" ) );
	float const atompair_weight( 500.0 * realafteroption("cst",0.0) );
	float const MAX_CN_DISTANCE( 2.0 );
	////////////////////

	// setup minimizer
	pose_setup_minimizer( ramp_min_tolerance );
	//minimize_set_vary_rb_angle( false );
	pose_set_use_nblist( use_nblist );

	//minimize_ns::rb_translation_scale_factor = intafteroption("scale");

	// scorefxn
	Score_weight_map weight_map( score12 );

	// i/o
	Pose pose,pdb_pose;
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const read_all_chains( true );
	pose_from_pdb( pdb_pose, stringafteroption("s"), fullatom, ideal_pose,
								 read_all_chains );

	// flip symmetric sidechains
	{
		float const start_score( pdb_pose.score( weight_map ) );
		pdb_pose.refold_sidechains_from_chi();
		std::cout << "score-delta from sidechain flip: " <<
			pdb_pose.score( weight_map ) - start_score << std::endl;
	}

	// atompair constraints
	cst_set_ns::Cst_set cst_set;

	if ( use_cst ) {
		fill_cst_set( cst_set, pdb_pose, true );
		weight_map.set_weight( ATOMPAIR_CST,
			atompair_weight / cst_set.count_atompair_constraints() );
	}

	int const nres( pdb_pose.total_residue() );
	int nres1(0);
	// set length of partner1:
	if ( truefalseoption( "partner2_begin" ) ) {
		// get from commandline
		nres1 = intafteroption("partner2_begin") - 1;
	} else {
		// try to determine from chainbreaks
		std::vector< int > cuts;
		for ( int i=1; i<nres; ++i ) {
			float const cn_distance = distance
				( xyzVector_float( &pdb_pose.Eposition()(1,4,i) ),
					xyzVector_float( &pdb_pose.Eposition()(1,1,i+1) ) );
			if ( cn_distance > MAX_CN_DISTANCE ) {
				std::cout << "chainbreak detected: " << i << ' ' <<
					cn_distance << std::endl;
				cuts.push_back( i );
			}
		}
		if ( cuts.size() != 1 ) {
			std::cout << "Couldnt determine partner2_begin: must set on command " <<
				"line." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		nres1 = cuts[0];
	}

	// setup the fold_tree, interface rsd set
	pose = pdb_pose;
	if ( use_cst ) pose.set_constraints( cst_set );
	FArray1D_bool interface( nres, false );
	if ( nres1 >= 1 || nres1 < nres )  // chu this is a complex
	{
		int jump_pos1(0), jump_pos2(0), interface_count(0);

		// find the closest c-alphas between the two partners, anchor the
		// jump at these residues

		float min_dis2(10000.0);
		for ( int i=1; i<= nres; ++i ) {
			xyzVector_float i_ca( &pose.Eposition()(1,2,i) );
			float i_min_dis2(10000.0);
			for ( int j=1; j<= nres; ++j ) {
				if ( i<= nres1 && j<=nres1 || i>nres1 && j>nres1 ) continue;
				xyzVector_float j_ca( &pose.Eposition()(1,2,j) );
				float const dis2( distance_squared( i_ca, j_ca ) );
				if ( dis2 < min_dis2 ) {
					min_dis2 = dis2;
					jump_pos1 = std::min(i,j);
					jump_pos2 = std::max(i,j);
				}
				i_min_dis2 = std::min( dis2, i_min_dis2 );
			}
			if ( i_min_dis2 <= ca_dis2_threshold ) {
				++interface_count;
				interface(i) = true;
			}
		}


		if ( truefalseoption("jump_pos1" ) ) {
			jump_pos1 = intafteroption("jump_pos1");
			jump_pos2 = intafteroption("jump_pos2");
			assert( jump_pos1 <= nres1 && jump_pos2 > nres1 );
			min_dis2 = distance_squared
				( xyzVector_float( &pose.Eposition()(1,2,jump_pos1) ),
					xyzVector_float( &pose.Eposition()(1,2,jump_pos2) ) );
		}


		std::cout << "Anchoring jump at residues: " << jump_pos1 << " and " <<
			jump_pos2 << " CA_distance= " << std::sqrt( min_dis2 ) << std::endl;

		std::cout << "Found " << interface_count << " interface rsds" <<
			std::endl;

		// set the fold_tree
		pose.one_jump_tree( nres, jump_pos1, jump_pos2, nres1 );
	}
	else { // this is a monomeric protein
		pose.simple_fold_tree(nres);
		interface = true;
	}
	// what can move:
	bool const bb( truefalseoption("bb") );
	bool const chi( truefalseoption("chi") );
	bool const int_chi( truefalseoption("int_chi") );
	bool const jump( truefalseoption("jump") );
	pose.set_allow_bb_move( bb );
	if ( int_chi ) {
		pose.set_allow_chi_move( interface ); // pass FArray
	} else {
		pose.set_allow_chi_move( chi ); // pass bool
	}
	pose.set_allow_jump_move( jump );

	// for profiling
	prof::reset();
	PROF_START( prof::TEST0 );

	// starting score
	pose.score( weight_map );
	std::cout << "start_score= " << pose.get_0D_score( SCORE ) <<
		" start_fa_rep= " << pose.get_0D_score( FA_REP ) <<
		" start_int_rep= " << sum_interface_fa_rep( pose ) << std::endl;


	// minimize: ramp up the repulsive geometrically
	// rep_delta^(lj_ramp_cycles-1) = end_rep_weight / start_rep_weight
	float const rep_delta
		( std::exp( std::log( final_rep_weight / initial_rep_weight ) /
								(lj_ramp_cycles-1) ) );
	float rep_weight = initial_rep_weight / rep_delta;

	for ( int k = 1; k <= lj_ramp_cycles; ++k ) {
		rep_weight *= rep_delta;
		weight_map.set_weight( FA_REP, rep_weight );
		pose.main_minimize( weight_map, "dfpmin" );

		// diagnostics
		float const score( pose.get_0D_score( SCORE ));
		float const fa_rep( pose.get_0D_score( FA_REP ));
		float const interface_rep( sum_interface_fa_rep( pose ) );
		float const ca_rms( CA_rmsd( pose, pdb_pose ));
		float const allatom_rms( allatom_rmsd( pose, pdb_pose ));
		float const cst_score( use_cst ? pose.get_0D_score( ATOMPAIR_CST ) : 0.0 );
		std::cout << "rep_weight: " << F(9,3,rep_weight ) <<
			" score= " << F(9,3,score) <<
			" fa_rep= " << F(9,3,fa_rep) <<
			" int_rep= " << F(9,3,interface_rep) <<
			" cst_score= " << F(9,3,cst_score) << // NOTE: includes weight
			" ca_rms= " << F(9,3,ca_rms ) <<
			" allatom_rms= " << F(9,3,allatom_rms) << std::endl;
		prof::show();
	}

	// final minimize:
	minimize_set_tolerance( final_min_tolerance );
	pose.main_minimize( weight_map, "dfpmin" );

	// diagnostics
	float const score( pose.get_0D_score( SCORE ));
	float const fa_rep( pose.get_0D_score( FA_REP ));
	float const interface_rep( sum_interface_fa_rep( pose ) );
	float const ca_rms( CA_rmsd( pose, pdb_pose ));
	float const allatom_rms( allatom_rmsd( pose, pdb_pose ));
	float const cst_score( use_cst ? pose.get_0D_score( ATOMPAIR_CST ) : 0.0 );
	std::cout << "rep_weight: " << F(9,3,rep_weight ) <<
		" score= " << F(9,3,score) <<
		" fa_rep= " << F(9,3,fa_rep) <<
		" int_rep= " << F(9,3,interface_rep) <<
		" cst_score= " << F(9,3,cst_score) << // NOTE: includes weight
		" ca_rms= " << F(9,3,ca_rms ) <<
		" allatom_rms= " << F(9,3,allatom_rms) << std::endl;

	pose.dump_pdb("bk_min_test.pdb");

	// profiling
	PROF_STOP( prof::TEST0 );
	prof::show();

}

///////////////////////////////////////////////////////////////////////////////
float
sum_interface_fa_rep(
	pose_ns::Pose const & pose
)
{
	// assumes that scores are up-to-date, or will die
	if ( pose.num_jump() != 1 ) { return 0.0; };
	std::vector< int > cuts( pose.fold_tree().cutpoints() );
	assert( cuts.size() == 1 );
	int const nres1( cuts[0] );
	int const nres( pose.total_residue() );
	FArray2D_float rep_pair( pose.get_2D_score( pose_ns::REP_PAIR ) );
	float fa_rep(0.0);
	for ( int i=1; i<= nres1; ++i ) {
		for ( int j=nres1+1; j<= nres; ++j ) {
			fa_rep += rep_pair(i,j);
		}
	}
	return fa_rep;
}

///////////////////////////////////////////////////////////////////////////////
std::string
benchmark_filename(
	std::string const & file
)
{
	static bool init( false );
	static std::string bmark_dir;

	if (!init) {
		init = true;
		bmark_dir = stringafteroption("bmark_dir");
	}

	return bmark_dir+"/"+file;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
pose_deriv_check( pose_ns::Pose const & start_pose )
{
	using namespace pose_ns;

	pose_setup_minimizer( 0.001 );

	/////////////////////////
	// minimizer diagnostics:
	minimize_set_deriv_check( true );

	// setup the functions to be minimized
	std::vector< std::vector< Score_name > > chi_min_list;
	std::vector< std::vector< Score_name > > bb_min_list;

	{
		std::vector< Score_name > v;
		// atr,rep,&sol
		v.push_back( FA_ATR );
		v.push_back( FA_REP );
		v.push_back( FA_SOL );
		chi_min_list.push_back(v);
		bb_min_list.push_back(v);

		// gb
		if ( param_pack::gen_born ) {
			v.clear();
			v.push_back( GB );
			chi_min_list.push_back(v);
			bb_min_list.push_back(v);
		}

		// hbonds
		v.clear();
		v.push_back( HB_SC );
		chi_min_list.push_back( v ); // sc
		v.clear();
		v.push_back( HB_SRBB);
		v.push_back( HB_LRBB);
		bb_min_list.push_back( v ); // bb
		v.push_back( HB_SC );
		bb_min_list.push_back( v ); // all

		// rama
		v.clear();
		v.push_back( RAMACHANDRAN );
		bb_min_list.push_back( v );

		// prob
		v.clear();
		v.push_back( FA_PROB );
		bb_min_list.push_back( v );

		// pair
		v.clear();
		v.push_back( FA_PAIR );
		bb_min_list.push_back( v );

		// dun -- bb + chi
		v.clear();
		v.push_back( FA_DUN );
		bb_min_list.push_back( v );
		chi_min_list.push_back( v );
	}

	//
	FArray1D_bool start_allow_bb_move, start_allow_chi_move,
		start_allow_jump_move;

	start_pose.retrieve_allow_move( start_allow_bb_move, start_allow_chi_move,
																	start_allow_jump_move );


	///////////////////////////////////////////////////
	// loop over the different weight sets to be tested
	//
	for ( int repeat=0; repeat< 4; ++repeat ) {
		bool const bb( repeat%2 == 0 );
		bool const use_nblist( repeat/2 == 0 );
		pose_set_use_nblist( use_nblist );

		std::vector< std::vector< Score_name > > const & weights_list
			( bb ? bb_min_list : chi_min_list );

		for ( int ii=0; ii< int(weights_list.size()); ++ii ) {
			std::vector< Score_name > const & weights( weights_list[ii] );

			// fill in the weights
			Score_weight_map weight_map;
			std::string weight_tag
				( std::string( bb ? "BBMIN" : "CHIMIN" ) +
					std::string( use_nblist ? "-nblist" : "-no_nblist" ) );
			for( std::vector< Score_name >::const_iterator it=weights.begin(),
						 it_end = weights.end(); it != it_end; ++it ) {
				weight_map.set_weight( *it, 1.0 );
				weight_tag += ( "," + Score_data::name2string( *it ) );
			}


			// setup the pose
			Pose pose;
			pose = start_pose;
			pose.set_allow_bb_move( false );
			pose.set_allow_chi_move( false );
			pose.set_allow_jump_move( false );

			if ( bb ) {
				pose.set_allow_bb_move( start_allow_bb_move );
			} else {
				pose.set_allow_chi_move( start_allow_chi_move );
			}

			pose.main_minimize( weight_map, "linmin" );
			minimize_show_deriv_check_stats( std::cout );
			std::cout <<  ' ' << weight_tag << '\n';

		} // ii
	} // repeat
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
