// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_dna.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "DesignMap.h"
#include "dna.h"
#include "dna_classes.h"
#include "dna_ns.h"
#include "fragments_pose.h"
#include "fullatom_energies.h"
#include "fullatom_setup.h"
#include "gl_graphics.h"
#include "hbonds_ns.h"
#include "jumping_util.h" // debug
#include "jumping_loops.h"
#include "kin_stub.h"
#include "minimize.h"
#include "nblist.h"
#include "PackerTask.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "pose_benchmark.h"
#include "pose_constraints.h"
#include "pose_design.h"
#include "pose_rotamer_trials.h"
#include "pose_vdw.h"
#include "prof.h"
#include "read_aaproperties.h"
#include "random_numbers.h"
#include "score.h"
#include "score_ns.h"
#include "silent_input.h"
#include "smallmove.h"
#include "torsion_bbmove_trials.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>


typedef numeric::xyzVector_float Vec;
typedef numeric::xyzVector_double DVec;
typedef numeric::xyzMatrix_float Mat;
typedef numeric::xyzMatrix_double DMat;


// PBHACK
int debug_bp_deriv( 0 );
int debug_bs_deriv( 0 );
bool verbose( false );

inline
float
arccos(
	float const x
)
{
	return std::acos( numeric::sin_cos_range( x ) );
}


///////////////////////////////////////////////////////////////////////////////
// Much of this is just re-hashing Alex Morozov's (very nice)
// DNA code in dna.cc. I do a few things differently.
//
// duplicate some code for two reasons:
// 1 -- make sure I know how things work
// 2 -- need to change interface slightly, avoid some dna global vars
//
// Adding (I hope) derivatives (probably approximate) for the
// knowledge-based potential
//
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// analogous to count_pair
//
// refers to the base_step i -> i+1
//
// LOGIC: rsd i and i+1 are DNA and in basepairs but not paired to each other
//        the rsd numbers of their basepair partners are compatible w/
//        antiparallel strands
//
inline
bool
count_pair_bs(
	int const i,
	pose_ns::Pose const & pose
)
{
	return ( i>=1 && i < pose.total_residue() &&
					 param_aa::is_DNA( pose.res(i) ) &&
					 param_aa::is_DNA( pose.res(i+1) ) &&
					 pose.basepair_partner(i  ) >= 1   &&
					 pose.basepair_partner(i+1) >= 1   &&
					 pose.basepair_partner(i  ) != i+1 &&
					 pose.basepair_partner(i+1) == pose.basepair_partner(i)-1 );
}


///////////////////////////////////////////////////////////////////////////////
//
// LOGIC: rsd i is DNA and is in a basepair
//

inline
bool
count_pair_bp(
	int const i,
	pose_ns::Pose const & pose
)
{
	return ( i>=1 && i <= pose.total_residue() &&
					 param_aa::is_DNA( pose.res(i) ) &&
					 pose.basepair_partner(i) >= 1 );
}


///////////////////////////////////////////////////////////////////////////////
void
calc_rms_zf(
	pose_ns::Pose & pose
)
{
	using namespace aaproperties_pack;
	using numeric::xyzVector_float;
	using param_aa::aa_name3;

	float const CONTACT_THRESHOLD(3.5);
	std::vector< int > protein_pos_list; //, dna_pos_list;
	protein_pos_list.push_back( 16 );
	protein_pos_list.push_back( 18 );
	protein_pos_list.push_back( 19 );
	protein_pos_list.push_back( 22 );


	pose_ns::Pose const & native_pose( pose.native_pose() );

	int const nres( pose.total_residue() );
	{	// debug: is DNA in same place??
		float dev(0.0);
		for ( int i=1; i<=nres; ++i ) {
			assert( pose.res(i) == native_pose.res(i) );
			if ( pose.is_DNA( i ) ) {
				dev += distance( pose.full_coord(5,i), native_pose.full_coord(5,i));
			}
		}
		std::cout << "dna_dev: " << dev << std::endl;
		assert( dev < 0.01 );
	}


	{ // calc rms over protein_pos_list sidechains
		// no fitting, just use dna reference frame
		int count(0),count2(0);
		float dev(0.0),dev2(0.0);
		for ( int ii=0; ii< int(protein_pos_list.size()); ++ii ) {
			int const i( protein_pos_list[ii] );
			int const aa ( pose.res        (i));
			int const aav( pose.res_variant(i));
			for ( int j=5; j<= nheavyatoms(aa,aav); ++j ) {
				++count;
				dev += distance_squared( pose.full_coord(j,i),
																 native_pose.full_coord(j,i) );

				if ( chi_required( nchi(aa,aav),j,aa,aav) ) {
					// including only the terminal atoms
					++count2;
					dev2 += distance_squared( pose.full_coord(j,i),
																		native_pose.full_coord(j,i) );
				}
			}
		}
		dev  = std::sqrt( dev  / count  );
		dev2 = std::sqrt( dev2 / count2 );

		pose.set_extra_score( "dna_sc_rms", dev );
		pose.set_extra_score( "dna_sc_rms_term", dev2 );
	}

	{ // compare contacts with dna
		int
			dna_bb_contacts_native(0),
			dna_sc_contacts_native(0),
			dna_bb_contacts_native_correct(0),
			dna_sc_contacts_native_correct(0);

		for ( int i=1; i<= nres; ++i ) {
			if ( !pose.is_DNA(i)) continue;
			int const na ( pose.res        (i));
			int const nav( pose.res_variant(i));
			for ( int ii=1; ii<= nheavyatoms(na,nav); ++ii ) {
				if ( i>1 && pose.is_cutpoint(i-1) && ii<=3 ) continue;

				xyzVector_float const xyz_i( native_pose.full_coord(ii,i) );
				bool const is_bb( is_NA_backbone_atom( ii, na, nav ) );
				for ( int j=1; j<= nres; ++j ) {
					if ( !pose.is_protein(j) ) continue;
					int const aa ( pose.res        (j));
					int const aav( pose.res_variant(j));
					float min_native_d(999.9);
					int min_jj(0);
					for ( int jj=1; jj<= nheavyatoms(aa,aav); ++jj ) {
						min_native_d = std::min( min_native_d,
														distance( xyz_i, native_pose.full_coord(jj,j)));
						min_jj = jj;
					}

					if ( min_native_d < CONTACT_THRESHOLD ) {
						if ( is_bb ) ++dna_bb_contacts_native;
						else ++dna_sc_contacts_native;

						// check for contact in pose
						float min_d(999.9);
						for ( int j2=1; j2<= nheavyatoms(aa,aav); ++j2 ) {
							float const d
								( distance( pose.full_coord(ii,i), pose.full_coord(j2,j)));
							if ( d < min_d ) {
								min_d = d;
							}
						}

						if ( min_d < CONTACT_THRESHOLD || min_d < min_native_d + 0.3 ) {
							if ( is_bb ) ++dna_bb_contacts_native_correct;
							else ++dna_sc_contacts_native_correct;
						} else {
							std::cout << "Missed contact: " << is_bb << ' ' << i << ' ' <<
								aa_name3(na) <<
								' ' << atom_name(ii,na,nav) << ' ' << aa_name3(aa) <<
								' ' << atom_name(min_jj,aa,aav) <<
								' ' << j << ' ' << min_native_d << ' ' << min_d << std::endl;
						}
					}
				}
			}
		}

		pose.set_extra_score( "bbdna-nat-total", dna_bb_contacts_native );
		pose.set_extra_score( "scdna-nat-total", dna_sc_contacts_native );
		pose.set_extra_score( "bbdna-nat-found", dna_bb_contacts_native_correct );
		pose.set_extra_score( "scdna-nat-found", dna_sc_contacts_native_correct );
	}


}



///////////////////////////////////////////////////////////////////////////////
//
void
show_clashes(
						 pose_ns::Pose const & pose,
						 float const threshold
						 )
{
	using namespace param_aa;

	FArray2D_float const & rep_pair( pose.get_2D_score( pose_ns::REP_PAIR ) );
	int const nres( pose.total_residue() );
	for ( int i=1; i<= nres; ++i ) {
		for ( int j=i; j<= nres; ++j ) {
			float const repE( rep_pair(j,i));
			if ( repE >= threshold ) {
				std::cout << "RepE: " << F(9,3,repE) <<
					I(4,i) << ' ' << aa_name3(pose.res(i)) <<
					I(4,j) << ' ' << aa_name3(pose.res(j)) << std::endl;
			}
		}
	}



}


///////////////////////////////////////////////////////////////////////////////
//

void
mutate_test( pose_ns::Pose & pose )
{
	int const nres( pose.total_residue() );
	for ( int i=1; i<= nres; ++i ) {
		int const aa ( pose.res(i));
		int const aav( pose.res_variant(i));
		if ( param_aa::is_DNA( aa ) ) {
			if ( pose.is_jump_point( i ) ) continue;

			pose.copy_sidechain( i, pose.res(i), pose.res_variant(i),
													 aaproperties_pack::icoor(1,1,aa,aav) );
		}
	}
	pose.dump_pdb("idl.pdb");



// 			for ( int j=1; j<= nres; ++j ) {
// 				if ( param_aa::is_DNA( pose.res(j) ) ) {

// 					FArray3D_float fcoord( pose.full_coord() ); // make a copy


// 					pose.copy_sidechain( i, pose.res(j), pose.res_variant(j),
// 															 fcoord(1,1,j) );
// 					pose.dump_pdb("copy_from_"+string_of(j)+"_to_"+string_of(i)+".pdb");

// 				}
// 			}
// 		}
// 	}

	exit(0);
}



///////////////////////////////////////////////////////////////////////////////

// //
// void
// mutate_base(
// 						pose_ns::Pose & pose,
// 						int const seqpos,
// 						int const new_na
// 						)
// {

// 	FArray2D_float coord(3,MAX_ATOM()());
// 	{ // get current coords
// 		FArray3D_float const & fcoord( pose.full_coord() );
// 		for ( int j=1; j<= MAX_ATOM()(); ++j ) {
// 			for ( int k=1; k<= 3; ++k ) {
// 				coord(k,j) = fcoord(k,j,seqpos);
// 			}
// 		}
// 	}


// 	FArray1D_int aan_tmp(1,pose.res(seqpos) );

// 	mutate_base( 1, aan_tmp, coord, new_na );

// 	copy_sidechain( seqpos, new_na, new_nav, coord, false );


// }




///////////////////////////////////////////////////////////////////////////////
//
void
fix_phosphates(
	pose_ns::Pose & pose,
	float const threshold
)
{
	using namespace pose_ns;
	using namespace param_aa;

	Score_weight_map w( score12 );
	//w.set_weight( FA_REP, 1.0 );

	pose.score( w );

	FArray2D_float const & rep_pair( pose.get_2D_score( REP_PAIR ) );

	int const nres( pose.total_residue() );

	for ( int i=2; i<= nres; ++i ) {
		std::cout << i << ' ' << is_DNA( pose.res(i)) << ' ' <<
			pose.fold_tree().get_is_cutpoint()(i-1) << std::endl;

		if ( is_DNA( pose.res(i)) && pose.fold_tree().get_is_cutpoint()(i-1)) {

			// accumulate score
			float repE(0.0);
			for ( int j=1; j<= nres; ++j ) {
				repE += rep_pair( j,i);
			}

			std::cout << "fix? " << repE << ' ' << i <<std::endl;

			if ( repE > threshold ) {

				if ( pose.fold_tree().get_is_jump_point()(i) ) {
					std::cout << "big repE but cant fix phosphate at jump pos: " <<
						i << ' ' << repE << std::endl;
					continue;
				}

				std::cout << "Try fixing phosphate: " << i << ' ' << repE << std::endl;
				pose.set_allow_bb_move( i, true );
				Monte_carlo mc( pose, w, 0.0 );

				for ( int j=2; j<= nres; ++j ) {
					if ( is_DNA( pose.res(j)) ) {
						if ( pose.fold_tree().get_is_cutpoint()(j-1) ) continue;

						// try copying torsions into this position:
						for ( int k=2; k<=3; ++k ) {
							float const val( pose.get_torsion_by_number( j,k ));
							pose.set_torsion_by_number( i, k, val );
							std::cout << ' ' << k << ' ' << F(9,3,val);
						}
						std::cout << std::endl;

						if ( mc.boltzmann( pose ) ) {
							FArray2D_float const & new_rep_pair
								( pose.get_2D_score( REP_PAIR ) );

							// accumulate score
							float new_repE(0.0);
							for ( int jj=1; jj<= nres; ++jj ) {
								new_repE += new_rep_pair( jj,i);
							}

							//pose.dump_pdb("fix_phospho."+string_of(i)+"."+string_of(j)+
							//							"."+string_of(new_repE)+".pdb");
						}
					}
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


namespace pose_ns {
class Smap {
public:
	std::map< std::string, std::vector< float > > M;
	typedef std::map< std::string, std::vector< float > >::const_iterator
	const_iterator;

	void
	add_score(
		std::string const & tag,
		float const score
		)
	{
		if ( !M.count( tag ) ) {
			M.insert( std::make_pair( tag, std::vector< float >() ) );
		}
		std::vector< float > & scores( M.find( tag )->second );
		scores.push_back( score );
	}


	float
	mean(
		std::string const & tag
	)
	{
		assert( M.count(tag) );
		float m(0.0);
		std::vector< float > & scores( M.find( tag )->second );
		for ( int i=0; i<int(scores.size()); ++i ) m+= scores[i];
		m/= scores.size();
		return m;
	}
};
}


///////////////////////////////////////////////////////////////////////////////
//

void
store_scores(
	pose_ns::Pose & pose,
	pose_ns::Score_weight_map const & w,
	pose_ns::Smap & smap
)
{
	using namespace fullatom_energies;
	using namespace param_pack;
	using namespace scorefxns;
	using namespace scores;


	pose.score( w );

	// total scores
	smap.add_score("score", pose.get_0D_score(pose_ns::SCORE) );
	smap.add_score("atr_tot", fa_atr_score );
	smap.add_score("rep_tot", fa_rep_score );
	smap.add_score("sol_tot", fa_solv_score );
	smap.add_score("dun_tot", fa_dun_score );
	smap.add_score("hb_tot",
								 pack_wts.Whb_srbb() * hb_srbb_score +
								 pack_wts.Whb_lrbb() * hb_lrbb_score +
								 pack_wts.Whb_sc() * hb_sc_score ); //JSS hbscore
	smap.add_score("pair_tot", fa_pair_score );
	smap.add_score("prob_tot", fa_prob1b_score );
	smap.add_score("gb_tot", fa_gb_elec_score );

	// per-rsd scores
	for ( int i=1; i<= pose.total_residue(); ++i ) {

		smap.add_score( "atr_" +string_of(i),     atrenergy(i) );
		smap.add_score( "rep_" +string_of(i),     repenergy(i) );
		smap.add_score( "sol_" +string_of(i),     solenergy(i) );
		smap.add_score( "dun_" +string_of(i),     dunenergy(i) );
		smap.add_score( "hb_"  +string_of(i),      hbenergy(i) );
		smap.add_score( "pair_"+string_of(i),    probenergy(i) );
		smap.add_score( "prob_"+string_of(i),    probenergy(i) );
		smap.add_score( "gb_"  +string_of(i), gb_elecenergy(i) );
		smap.add_score( "res_" +string_of(i),     resenergy(i) );
	}



}



///////////////////////////////////////////////////////////////////////////////
//
void
compare_decoys()
{
	using namespace pose_ns;
	Score_weight_map w( score12 );

	Smap smap1,smap2;
	for ( int n=1; n<= 2; ++n ) {
		std::ifstream data( stringafteroption("l"+string_of(n)).c_str());
		std::string line;
		while ( getline( data,line ) ) {
			Pose pose;
			bool ok = pose_from_pdb( pose, line, true, false, true );
			if ( !ok ) {
				std::cout << "input failed: " << line << std::endl;
				continue;
			}

			FArray1D_bool exclude_none( pose.total_residue(), false );
			setup_simple_dna_fold_tree( pose, exclude_none );

			store_scores( pose, w, n==1 ? smap1 : smap2 );

		}
		data.close();

	}


	for ( Smap::const_iterator it = smap1.M.begin(); it != smap1.M.end(); ++it ){
		std::string const & tag( it->first );
		float const m1( smap1.mean(tag));
		float const m2( smap2.mean(tag));
		std::cout << "means: " << F(9,3,m1) << F(9,3,m2) << F(9,3,m1-m2) <<' '<<
			tag << std::endl;
	}

	std::exit(0);
}

////////////////////////////////////////////////////////////////////////////////

void
zf_linker_relax_mutate(
	pose_ns::Pose & pdb_pose
)
{
	using namespace pose_ns;

	////////////////////
	// angle move params
	float small_angle( 2.0 );
// 	float big_angle( 4.0 ), small_angle( 2.0 );


	std::vector< std::string > proteins, dnas;
	proteins.push_back( "RDER" ); dnas.push_back( "GCGT" );
	proteins.push_back( "DSNR" ); dnas.push_back( "GACC" );
	proteins.push_back( "QGSR" ); dnas.push_back( "GCAC" );
	proteins.push_back( "RADR" ); dnas.push_back( "GCAC" );

	std::vector< int > protein_pos_list, dna_pos_list;
	protein_pos_list.push_back( 16 );
	protein_pos_list.push_back( 18 );
	protein_pos_list.push_back( 19 );
	protein_pos_list.push_back( 22 );

	dna_pos_list.push_back(  8 );
	dna_pos_list.push_back(  9 );
	dna_pos_list.push_back( 10 );
	dna_pos_list.push_back( 11 );


	Pose native_pose;
	native_pose = pdb_pose;
	pdb_pose.set_native_pose( native_pose );

	prof::reset( true );

	//pdb_pose.dump_pdb("start.pdb");


	// setup fold_tree
	int const nres( pdb_pose.total_residue() );
	int nres_protein(0);
	//int const dna_jump(1);
	{
		for ( int i=1; i<= nres; ++i ) {
			if ( pdb_pose.is_protein(i) ) nres_protein = i;
		}
		assert( nres_protein + 11 + 11 == nres );
		int const num_jump(2);
		FArray1D_int cuts(num_jump);
		FArray2D_int jumps(2,num_jump);
		jumps(1,1) = 47;
		jumps(2,1) = 90;
		jumps(1,2) = 90; // might not be paired!
		jumps(2,2) = 103;
		cuts(1) = nres_protein;
		cuts(2) = nres_protein+11;
		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );
		f.reorder( 90 );
		pdb_pose.set_fold_tree(f);
	}

	fix_phosphates( pdb_pose, 1.0 );
	//pdb_pose.dump_pdb("fixed.pdb");
	native_pose = pdb_pose;


	// get interface residues
	int const linker_begin(28);
	int const linker_end  (31);

	FArray1D_bool interface( nres, false ), linker( nres, false );
	{ // identify interface residues
		using aaproperties_pack::natoms;
		for ( int i=1; i<= nres; ++i ) {
			if ( i>= linker_begin && i<= linker_end ) {
				linker(i) = true;
				interface(i) = true; // include in interface
			}
		}
		for ( int i=1; i<= linker_end /** only up to linker **/; ++i ) {
			if ( pdb_pose.is_protein( i ) ) {
				float min_d(1000.0);
				int const aa ( pdb_pose.res(i));
				int const aav( pdb_pose.res_variant(i));
				for ( int j=1; j<= nres; ++j ) {
					int const bb ( pdb_pose.res(j) );
					int const bbv( pdb_pose.res_variant(j) );
					if ( param_aa::is_DNA( bb ) || linker( j ) ) {
						for ( int ii=5 /*cb*/; ii<= natoms(aa,aav); ++ii ) {
							for ( int jj=1; jj<= natoms(bb,bbv); ++jj ) {
								min_d = std::min( distance( pdb_pose.full_coord(ii,i),
																						pdb_pose.full_coord(jj,j)), min_d);
							}
						}
					}
				}
				if ( min_d < 6.0 ) {
					std::cout << "interface rsd: " << i << ' ' << min_d << std::endl;
					interface(i) = true;
				}
			}
		}

		// exclude zn-coordinating residues
		interface(5 ) = false;
		interface(10) = false;
		interface(23) = false;
		interface(27) = false;

	} // find interface residues

	std::map< int, int > partner;
	{ // get base partners
		DnaSeqInfo base_pairs;
		find_basepairs( pdb_pose, base_pairs );
	}


	// scorefxn
	Score_weight_map w( score12 );

	minimize_set_tolerance( 0.001 );
	pose_setup_packer();
	pose_set_use_nblist( false ); // safety


	// minimize interface sc
	pdb_pose.set_allow_chi_move( interface );
	pdb_pose.set_allow_bb_move( false );
	pdb_pose.set_allow_jump_move( false ); /////////////////////PERMANENT NOW

	pdb_pose.main_minimize( w, "dfpmin" );
	calc_rms( pdb_pose );

	// minimize linker + interface sc
	pdb_pose.set_allow_chi_move( interface );
	pdb_pose.set_allow_bb_move( linker );

	pdb_pose.main_minimize( w, "dfpmin" );
	calc_rms( pdb_pose );

	// minimize linker + interface sc + dna_jump
// 	pdb_pose.set_allow_chi_move( interface );
// 	pdb_pose.set_allow_bb_move( linker );
// 	pdb_pose.set_allow_jump_move( dna_jump, true );

// 	pdb_pose.main_minimize( w, "dfpmin" );
// 	calc_rms( pdb_pose );

	// decoy output
	silent_io::Decoy_out out( false /* silent_output */ );


	for ( int nn=1; nn<= 400; ++nn ) {

		// choose one of the DNA sites
		for ( int d=0; d< int(dnas.size()); ++d ) {
			std::string const & dna_word( dnas[d] );

			// choose one of the PROTEIN sites
			for ( int p=0; p<int(proteins.size()); ++p ) {
				std::string const & protein_word( proteins[p] );


				// mut2 -- do rottrial before minimize
				std::string const tag( "_zf_mut2.dna="+string_of(d)+".protein="+
															 string_of(p)+"."+string_of( nn ) );
// 				std::string const tag( "_zf_mut.dna="+string_of(d)+".protein="+
// 															 string_of(p)+"."+string_of( nn ) );

				if ( out.decoy_exists(tag)) continue;


				// recover starting conformation
				Pose pose;
				pose = pdb_pose;


				///////////////////////////
				// mutate the dna positions
				for ( int i=0; i<int(dna_word.size()); ++i ) {
					using aaproperties_pack::icoor;
					int na, pos, nav(1);
					num_from_name("  "+dna_word.substr(i,1), na );
					assert(na);

					// mutate primary strand
					pos = dna_pos_list[i] + nres_protein;
					pose.copy_sidechain( pos, na, nav, icoor(1,1,na,nav) );

					// mutate complementary strand
					na = na_partner(na);
					pos = partner[pos];
					pose.copy_sidechain( pos, na, nav, icoor(1,1,na,nav) );
				}


				///////////////////////////////
				// mutate the protein positions
				bool same_protein_sequence( true );
				for ( int i=0; i<int(protein_word.size()); ++i ) {
					using aaproperties_pack::icoor;
					int aa, pos, aav(1);
					num_from_res1(protein_word[i], aa );
					assert(aa);

					// mutate primary strand
					pos = protein_pos_list[i];
					if ( aa != pose.res(pos) ) {
						same_protein_sequence = false;
						pose.copy_sidechain( pos, aa, aav, icoor(1,1,aa,aav) );
					}
				}


				// randomly perturb the first finger
// 				if ( nn > 1 ) {

// 					for ( int i=linker_begin; i<= linker_end; ++i ) {
// 						pose.set_phi( i, pose.phi(i) + gaussian()*big_angle );
// 					}

// 				}


				// repack interface
				pose.repack( interface, same_protein_sequence );


				// minimize linker
				pose.set_allow_jump_move( false );
				pose.set_allow_bb_move( linker );
				pose.set_allow_chi_move( false );

				pose.main_minimize( w, "dfpmin" );

				// minimize everything
				//pose.set_allow_jump_move( dna_jump, true );NO MORE!!!!!!
				pose.set_allow_bb_move( linker );
				pose.set_allow_chi_move( interface );

				pose.main_minimize( w, "dfpmin" );

				Pose start_pose;
				start_pose = pose;
				pose.set_start_pose( start_pose );

				calc_rms( pose );

				Monte_carlo mc( pose, w, 0.8 );

				// rottrial:
				{ // rottrial trial -- all interface positions
					pose.rottrial( w, 1 );
					pose.new_score_pose();
					pose.score(w);
					mc.boltzmann( pose, "Rottrial" );
				}


				for ( int n=1; n<= 3; ++n ) {
					for ( int m=1; m<= 10; ++m ) {
						// small moves

						for ( int i=linker_begin; i<= linker_end; ++i ) {
							pose.set_phi( i, pose.phi(i) + gaussian()*small_angle );
							pose.set_psi( i, pose.psi(i) + gaussian()*small_angle );
						}

						pose.rottrial(w,1);
						pose.new_score_pose();

						pose.main_minimize(w, "dfpmin");


						if ( false ) { // try rotamer trialing
							Pose save_pose;
							save_pose = pose;
							float const old_score( save_pose.get_0D_score( SCORE ) );
							pose.rottrial(w,1);
							float const rottrial_score( pose.get_0D_score( SCORE ) );
							pose.new_score_pose();
							pose.score(w);
							float const new_score( pose.get_0D_score( SCORE ) );
							if ( new_score > old_score ) {
								std::cout << "rottrial BAD: " << old_score << ' ' <<
									rottrial_score << ' ' << new_score << ' ' <<
									new_score - rottrial_score << std::endl;
								pose = save_pose;
							}
						}

						// accept?
						mc.boltzmann( pose, "linker-bb-min" );

						calc_rms( pose );
						std::cout << pose.show_scores() << std::endl;
					}
					pose = mc.low_pose();
					prof::show();

				}

				mc.show_counters();

				calc_rms(pose);
				out.write( tag, pose );
				prof::show();
			} // p
		} // d
	} // nn -- decoy loop

	std::exit(0);

}






///////////////////////////////////////////////////////////////////////////////
//
// try rebuilding a loop using fragments and ccd-closure
//
//
//

void
jim_test()
{
	using namespace pose_ns;
	using namespace kin;

	int const loop_begin( 24 );
	int const loop_end( 40 );
	int const loop_size( loop_end-loop_begin+1 );
	int const moving_residue( 30 );


	Pose pose;
	pose_from_pdb( pose, "/work/pbradley/dna/endo/cter_scid.pdb",
								 true, false, true );

	FArray1D_bool exclude_none( pose.total_residue(), false );
	setup_simple_dna_fold_tree( pose, exclude_none );
	Fold_tree simple_dna_tree( pose.fold_tree() );

	pose.dump_pdb("test1.pdb");

	{

		int moving_jump_number;
		{ // setup loop fold_tree
			Fold_tree f( pose.fold_tree() );

			// stupid sanity check
			for ( int i=loop_begin; i<= loop_end; ++i ) {
				assert( !f.get_is_jump_point()(i) );
			}

			moving_jump_number =
				f.new_jump( loop_begin-1, moving_residue, moving_residue -1 );
			f.new_jump( loop_begin-1, loop_end+1, moving_residue  );

			pose.set_fold_tree( f );
			pose.dump_pdb("test2.pdb");
		}


		// set loop residues to ALA
		int ala_pos(0);
		for ( int i=1; i< loop_begin; ++i ) {
			if ( pose.res(i) == param_aa::aa_ala ) {
				ala_pos = i;
				break;
			}
		}
		assert( ala_pos );

		for ( int i=loop_begin; i<= loop_end; ++i ) {
			if ( i != moving_residue ) {
				pose.copy_sidechain( i, param_aa::aa_ala, 1,
														 pose.full_coord()(1,1,ala_pos) );
			}
		}
		pose.dump_pdb("ala_loop.pdb");

		// now perturb the jump
		int const n2c(1);
		pose.set_jump_rb_delta( moving_jump_number, 1, n2c, 1.5 );
		pose.set_jump_rb_delta( moving_jump_number, 2, n2c, 1.5 );
		pose.set_jump_rb_delta( moving_jump_number, 3, n2c, 1.5 );

		pose.dump_pdb("move_jump.pdb");


		// try fragment insertion protocol
		// see if copying of the pose is a problem, timewise

		// let loop bb move
		pose.set_allow_move( kin::ALL, false );
		FArray1D_bool loop_rsd( pose.total_residue(), false );
 		for ( int i=loop_begin; i<= loop_end; ++i ) {
 			pose.set_allow_bb_move( i, true );
			loop_rsd(i) = true;
 		}


		///////////
		// scorefxn
		Score_weight_map weight_map( score12 );
		weight_map.set_weight( CHAINBREAK_CST, 1.0 );

		// setup chainbreak constraints
		cst_set_ns::Cst_set cst_set;
		pose.set_constraints( cst_set );
		cst_set.add_chainbreak( moving_residue-1, 1.0 );
		cst_set.add_chainbreak( moving_residue  , 1.0 );


		{ // experiment with dna vdw and fast hbonds...
			Score_weight_map w;
			w.set_weight( VDW, 1.0 );
			w.set_weight( HB_SRBB, 1.0 );
			w.set_weight( HB_LRBB, 1.0 );
			w.set_weight( RAMACHANDRAN, 0.2 );
			w.set_weight( CHAINBREAK_CST, 0.5 );

			Monte_carlo mc( pose, w, 0.8 );

			prof::reset( true );
			pose_setup_minimizer( 0.001 );

			float chainbreak_weight(0.5);
			for ( int c1=1; c1<= 5; ++c1 ) {
				chainbreak_weight *= 2.0;
				w.set_weight( CHAINBREAK_CST, chainbreak_weight );
				mc.set_weight_map( w );

				//pose.main_minimize( w, "dfpmin" );
				//mc.boltzmann( pose, "min" );

				for ( int c2=1; c2<= 50; ++c2 ) {
					ccd_moves( 5, pose, loop_begin, moving_residue-1, moving_residue-1 );
					mc.boltzmann( pose, "nterm" );

					ccd_moves( 5, pose, moving_residue+1, loop_end, moving_residue );
					mc.boltzmann( pose, "cterm" );

					if ( false ) {
						ccd_moves( 5, pose, moving_residue+1, loop_end, moving_residue );
						pose.main_minimize( w, "linmin" );
						mc.boltzmann( pose, "cterm-min" );

						ccd_moves( 5, pose, loop_begin, moving_residue-1,moving_residue-1);
						pose.main_minimize( w, "linmin" );
						mc.boltzmann( pose, "nterm-min" );
					}
				}
				pose = mc.low_pose();
				std::cout << "round " << c1 << ' ' << pose.show_scores() << std::endl;
				mc.show_counters();
			}
			pose = mc.low_pose();
			pose.dump_pdb( "final.pdb" );
			prof::show();

//			exit(0);
		}



		{ // test

			// first try to mostly close the loop
			for ( int n=1; n<=1; ++n ) {
				float fdev,bdev,tor_delta,rama_delta;
				int const ccd_cycles(0);
				fast_ccd_loop_closure( pose,loop_begin,moving_residue-1,
					moving_residue-1, ccd_cycles, 0.01, true, 2.0, 10, 30, 90, fdev, bdev,
					tor_delta, rama_delta );
				std::cout << "fdev: " << F(9,3,fdev) << F(9,3,bdev) <<
					F(9,3,tor_delta) << F(9,3,rama_delta ) << std::endl;

				pose.score( weight_map );
				std::cout << pose.show_scores() << std::endl;
				pose.dump_pdb("nterm_ccd.pdb"+string_of(n));

				fast_ccd_loop_closure( pose, moving_residue+1,loop_end,moving_residue,
															 ccd_cycles, 0.01, true, 2.0, 10, 30, 90, fdev, bdev,
															 tor_delta, rama_delta );
				std::cout << "fdev: " << F(9,3,fdev) << F(9,3,bdev) <<
					F(9,3,tor_delta) << F(9,3,rama_delta ) << std::endl;
				pose.score( weight_map );
				std::cout << pose.show_scores() << std::endl;
				pose.dump_pdb("cterm_ccd.pdb"+string_of(n));
			}


			// now connect moving residue to previous position

			// residue 30 is an ASN
			// add a coordinate tether holding OD1 and ND2 to their current locations
			{
				using namespace param_aa;
				assert( pose.res( moving_residue ) == aa_asn );
				Atom_id const OD1( 7,moving_residue ), ND2( 8, moving_residue );
				cst_set.add_coordinate_constraint( OD1, pose.xyz( OD1 ) );
				cst_set.add_coordinate_constraint( ND2, pose.xyz( ND2 ) );
			}

			weight_map.set_weight( COORD_CST, 1.0 );

			pose.score( weight_map );
			pose.dump_pdb("before.pdb");
			std::cout << "before gluing: " << pose.show_scores() <<std::endl;

			// connect to previous position
			Fold_tree f( simple_dna_tree );
			//f.new_jump( loop_begin-1, loop_end+1, moving_residue-1 );
			f.new_jump( loop_begin-1, loop_end+1, moving_residue );
			pose.set_fold_tree( f );
			pose.set_allow_bb_move( loop_rsd );

			//
			pose.score( weight_map );
			pose.dump_pdb("after1.pdb");
			std::cout << "after gluing1: " << pose.show_scores() <<std::endl;


			// insert ideal bonds at junction of moving_residue-1 and moving_residue
			// insert ideal bonds at junction of moving_residue and moving_residue+1
			pose.insert_ideal_bonds_at_cutpoint( moving_residue-1 );
			pose.score( weight_map );
			pose.dump_pdb("after2.pdb");
			std::cout << "after gluing2: " << pose.show_scores() <<std::endl;


			{ //test min
				Score_weight_map w;
				w.set_weight( COORD_CST, 1.0 );
				pose.main_minimize( w, "dfpmin" );
				std::cout << "after w-min: " << pose.show_scores() <<std::endl;
				exit(0);
			}



			pose.main_minimize(weight_map,"dfpmin");
			pose.score( weight_map );
			pose.dump_pdb("min1.pdb");
			std::cout << "after gluing2: " << pose.show_scores() <<std::endl;

			weight_map.set_weight( COORD_CST, 10000.0 );
			pose.main_minimize(weight_map,"dfpmin");
			pose.score( weight_map );
			pose.dump_pdb("min2.pdb");
			std::cout << "after min2: " << pose.show_scores() <<std::endl;


			exit(0);

		}

// 		prof::show();
// 		exit(0);
		if ( false ) {
			prof::reset();


			pose_setup_minimizer( 0.001 );
			pose.main_minimize( weight_map, "dfpmin" );


			prof::show();
			exit(0);
		}

		{ // slower protocol for closing using smallish steps and energy
			// monte carlo
			Monte_carlo mc( pose, weight_map, 0.8 );

			prof::reset();
			pose_setup_minimizer( 0.001 );

			float chainbreak_weight(0.5);
			for ( int c1=1; c1<= 5; ++c1 ) {
				chainbreak_weight *= 2.0;
				weight_map.set_weight( CHAINBREAK_CST, chainbreak_weight );
				mc.set_weight_map( weight_map );

				pose.main_minimize( weight_map, "dfpmin" );
				mc.boltzmann( pose );

				for ( int c2=1; c2<= 30; ++c2 ) {
					ccd_moves( 5, pose, loop_begin, moving_residue-1, moving_residue-1 );
					pose.main_minimize( weight_map, "dfpmin" );
					mc.boltzmann( pose );

					ccd_moves( 5, pose, moving_residue+1, loop_end, moving_residue );
					pose.main_minimize( weight_map, "dfpmin" );
					mc.boltzmann( pose );
				}
				pose = mc.low_pose();
				std::cout << "round " << c1 << ' ' << pose.show_scores() << std::endl;
			}
			pose = mc.low_pose();
			pose.dump_pdb( "final.pdb" );
			prof::show();

			exit(0);
		}
	}


// 	{ // this code is no longer valid!
// 		int const cutpoint(0);
// 		std::cout << pose.fold_tree() << std::endl;
// 		pose.atom_tree()->root()->show();

// 		prof::reset();
// 		//Score_weight_map weight_map( score12 );
// 		Score_weight_map weight_map;
// 		weight_map.set_weight( ATOMPAIR_CST, 1.0 );
// 		weight_map.set_weight( KIN_3D_CST, 1.0 );

// 		pose.set_allow_move( kin::ALL, false );
// // 		for ( int i=loop_begin; i<= loop_end; ++i ) {
// // 			pose.set_allow_bb_move( i, true );
// // 		}


// 		// setup chainbreak constraints
// 		cst_set_ns::Cst_set cst_set;
// 		add_chainbreak_constraint( cutpoint, pose.res(cutpoint+1),
// 															 pose.res_variant( cutpoint+1), cst_set );


// 		pose.set_constraints( cst_set );

// 		pose.score( weight_map );
// 		std::cout << pose.show_scores() << std::endl;

// 		// perturb
// 		int const ppos( loop_begin );
// 		pose.set_allow_bb_move( ppos, true );
// 		pose.set_allow_bb_move( ppos+1, true );
// 		pose.set_phi( ppos  , pose.phi( ppos   ) + 15 );
// 		pose.set_psi( ppos  , pose.psi( ppos   ) - 15 );
// 		pose.set_phi( ppos+1, pose.phi( ppos+1 ) + 15 );
// 		pose.set_psi( ppos+1, pose.psi( ppos+1 ) - 15 );

// 		pose.score( weight_map );
// 		pose.dump_pdb("test_perturb.pdb" );
// 		std::cout << pose.show_scores() << std::endl;

// 		pose_setup_minimizer( 0.001 );
// 		pose.main_minimize( weight_map, "dfpmin" );

// 		pose.dump_pdb("test_perturb_min.pdb" );
// 		std::cout << pose.show_scores() << std::endl;

// 		prof::show();
// 		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
// 	}

	{
		// try making some fragment insertions
		prof::reset();

		for ( int i=loop_begin; i<= loop_end; ++i ) {
			pose.set_allow_bb_move( i, true );
		}


		Score_weight_map weight_map;
		pose.score( weight_map );

		insert_init_frag( pose, loop_begin, loop_size );

		pose.score( weight_map );
		pose.dump_pdb("test3.pdb");

		read_fragments_simple("ani1Z",127);

		choose_fragment_pose( pose, 3 );

		pose.score( weight_map );
		pose.dump_pdb("test4.pdb");

		prof::show();
		exit(0);
	}
}

///////////////////////////////////////////////////////////////////////////////
// identify helix in the major groove

void
helix_in_groove(
	pose_ns::Pose & pose
)
{
	using namespace param_aa;

	float const d1_cutoff( 12.0 );
	int const ca_index(2);
	int const p_index(2);

	// identify basepairs
	std::map< int, int > partner;
	DnaSeqInfo base_pairs;
	int const nres( pose.total_residue() );
	find_basepairs( pose, base_pairs );

	// identify triplets of helical residues
	for ( int i=2; i<= nres-1; ++i ) {
		if ( pose.is_protein(i-1) &&
				 pose.is_protein( i ) &&
				 pose.is_protein(i+1) ) {

			if ( pose.secstruct(i-1) == 'H' &&
					 pose.secstruct( i ) == 'H' &&
					 pose.secstruct(i+1) == 'H' ) {

				Vec
					ca1( pose.full_coord(ca_index,i-1) ),
					ca2( pose.full_coord(ca_index,i  ) ),
					ca3( pose.full_coord(ca_index,i+1) );

				// construct a position along the helical axis:
				Vec c( 0.5f * ca1 + 0.5f * ca3 );
				c += 0.1f * ( ca2 - c );

				// try to find a major groove spot
				float min_d(1000.0);
				int jj1(0), jj2(0);

				for ( int j=1; j<= nres; ++j ) {
					if ( !is_DNA(pose.res(j)) ) continue;
					if ( !partner[j] || partner[j] < j ) continue;

					Vec p( pose.full_coord(p_index, j ) );
					float const d1( distance( p, c ) );
					if ( d1 > d1_cutoff ) continue;
					for ( int k=0; k <= 9; ++k ) {
						int const j2( partner[j] - k );
						Vec p2( pose.full_coord( p_index, j2 ) );
						float const dot2( dot( ( p-c  ).normalized(),
																	 ( c-p2 ).normalized() ) );
						if ( dot2 < 0.5 ) continue;
						float const d2( distance( p2, c ) );
						if ( d1+d2 < min_d ) {
							min_d = d1+d2;
							jj1 = j;
							jj2 = j2;
							std::cout << "min_d: " << min_d << ' ' << k << ' ' << i << ' ' <<
								jj1 << ' ' << jj2 << ' ' << d1 << ' ' << d2 << ' ' << dot2 <<
								std::endl;
						}
						if ( pose.fold_tree().get_is_cutpoint()(j2-1)) break;
					}
				}
				if ( min_d <= 20 ) {
					std::cout << "groove: " << i << ' ' << jj1 << ' ' << jj2 << ' ' <<
						min_d << std::endl;
				}
			}
		}
	}


}


///////////////////////////////////////////////////////////////////////////////
// loop-dock test
//
// test redocking of the 1st zn finger from 1aay.pdb.orig
// onto the DNA
//
// 1. sample by modifying loop phipsi angles
//

// OLD


///////////////////////////////////////////////////////////////////////////////
void
perturb_dna_jump(
	pose_ns::Pose & pose,
	int const dna_jump,
	float const trans_delta,
	float const rot_delta
)
{
	pose.fold_in_rb_deltas();
	for( int i=1; i<=3; ++i ) { // first 3 are translation
		pose.set_jump_rb_delta(dna_jump,i,1,trans_delta*gaussian());
	}

	for( int i=4; i<=6; ++i ) { // next 3 are rotations
		pose.set_jump_rb_delta(dna_jump,i,1,rot_delta*gaussian());
	}
	pose.fold_in_rb_deltas();
}


///////////////////////////////////////////////////////////////////////////////
// look at docking winged helices from the case paper
//

void
winged_dock_test()
{
	using namespace pose_ns;
	using param_aa::is_protein;
	using param_aa::is_DNA;

	// try mutating dna bases in 3-bp stretch:
	//int const mutate_begin(8);
	//int const mutate_begin_partner(26);

	prof::reset( true );

	Pose pdb_pose;
 	pose_from_pdb( pdb_pose, "1pue.ABE.pdb", true,
 								 false, true );
//  	pose_from_pdb( pdb_pose, "/work/pbradley/dna/winged/1pue.ABE.pdb", true,
//  								 false, true );

	// Rosetta setup:
	minimize_set_tolerance( 1e-3 );
	pose_setup_packer();
	pose_set_use_nblist( false );


	// build fold_tree:
	int const dna_jump(2); // should match tree below
	int const nres( pdb_pose.total_residue() );
	int const num_jump( 2 );
	{
		FArray1D_int cuts(num_jump);
		FArray2D_int jump_point(2,num_jump);

		cuts(1) = 16;
		cuts(2) = 32;
		jump_point(1,1) = 10;
		jump_point(2,1) = 26;
		jump_point(1,2) = 10;
		jump_point(2,2) = 96;
		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jump_point, cuts );
		f.reorder(10);
		pdb_pose.set_fold_tree(f);
	}
	//	pdb_pose.dump_pdb("pdb_pose.pdb");


	FArray1D_bool protein( nres, false ), interface( nres, false );
	{

		// identify interface residues
		using aaproperties_pack::natoms;
		for ( int i=1; i<= nres; ++i ) {
			protein(i) = pdb_pose.is_protein( i );
			if ( protein(i) ) {
				float min_d(1000.0);
				int const aa ( pdb_pose.res(i));
				int const aav( pdb_pose.res_variant(i));
				for ( int j=1; j<= nres; ++j ) {
					int const bb ( pdb_pose.res(j) );
					int const bbv( pdb_pose.res_variant(j) );
					if ( param_aa::is_DNA( bb ) ) {
						for ( int ii=1; ii<= natoms(aa,aav); ++ii ) {
							for ( int jj=1; jj<= natoms(bb,bbv); ++jj ) {
								min_d = std::min( distance( pdb_pose.full_coord(ii,i),
																						pdb_pose.full_coord(jj,j) ), min_d );
							}
						}
					}
				}
				if ( min_d < 6.0 ) {
					std::cout << "interface rsd: " << i << ' ' << min_d << std::endl;
					interface(i) = true;
				}
			}
		}
	} // find interface residues


	Score_weight_map w( score12 );

	// FIRST: minimize just the interface sidechains
	pdb_pose.set_allow_bb_move( false );
	pdb_pose.set_allow_chi_move( interface ); // b/c there's no intra-dna scoring
	pdb_pose.set_allow_jump_move( false );

	pdb_pose.main_minimize( w, "dfpmin" );
	//pdb_pose.dump_pdb("after_chi_min.pdb");

	// SECOND: optimize jump and interface chi
	pdb_pose.set_allow_jump_move( dna_jump, true );
	pdb_pose.set_allow_chi_move( interface );

	pdb_pose.main_minimize( w, "dfpmin" );

	//pdb_pose.dump_pdb("after_jump_min.pdb");

	// setup MOnte carlo
	Monte_carlo mc( pdb_pose, w, 0.8 /*temp*/);

	{ // rottrial trial -- all interface positions
		pdb_pose.rottrial( w, 1 );
		pdb_pose.new_score_pose();
		pdb_pose.score(w);
		mc.boltzmann( pdb_pose, "Rottrial" );
	}

	float const trans_delta( 0.2 );
	float const rot_delta( 5 );
	float const trans_delta_large( 1.0 );
	float const rot_delta_large( 20 );

	silent_io::Decoy_out out( false /* silent_output */ );

	for ( int nn=1; nn<= 20; ++nn ) {

		Pose pose, start_pose;
		pose = pdb_pose;
		pose.set_native_pose( pdb_pose );

		for ( int b1=1; b1<=4; ++b1 ) {
			int const na1( b1+21 );
			for ( int b2=1; b2<=4; ++b2 ) {
				int const na2( b2+21 );
				for ( int b3=1; b3<=4; ++b3 ) {
					int const na3( b3+21 );
					assert( param_aa::is_DNA(na1) && param_aa::is_DNA(na2) &&
									param_aa::is_DNA(na3) );

//					mutate_three_bases( pose, mutate_begin, na1, na2, na3 );
				}
			}
		}



//		std::string const tag
//			( "1pue_perturb."+param_aa::aa_name1(na1)+param_aa::aa_name1(na2)+
//				param_aa::aa_name1(na3)+"."+ string_of(nn) );

		std::string const tag( "PHIL_THE_ABOVE_DOESNT_COMPILE" );

		if ( out.decoy_exists( tag ) ) continue;


		if ( nn > 1 ) {
			// perturb the starting conformation
			perturb_dna_jump( pose, dna_jump, trans_delta_large, rot_delta_large );

			pose.repack( interface, false /*dont include current*/ );
		}

		start_pose = pose;

		pose.set_start_pose( start_pose );

		pose.main_minimize( w, "dfpmin" );

		mc.reset( pose );
		calc_rms(pose);
		std::cout << "starting score: " << nn << ' ' << pose.show_scores() <<
			std::endl;
		//pose.dump_pdb("start."+string_of(nn)+".pdb");

		for ( int n=1; n<= 3; ++n ) {

			mc.reset_counters();
			for ( int m=1; m<= 10; ++m ) {

				{ // perturb jump trial
					perturb_dna_jump( pose, dna_jump, trans_delta, rot_delta );

					// minimize
					pose.main_minimize( w, "dfpmin" );

					Pose save_pose;
					save_pose = pose;

					{ // try rotamer trialing
						float const old_score( save_pose.get_0D_score( SCORE ) );
						pose.rottrial(w,1);
						float const rottrial_score( pose.get_0D_score( SCORE ) );
						pose.new_score_pose();
						pose.score(w);
						float const new_score( pose.get_0D_score( SCORE ) );
						if ( new_score > old_score ) {
							std::cout << "rottrial BAD: " << old_score << ' ' <<
								rottrial_score << ' ' << new_score << ' ' <<
								new_score - rottrial_score << std::endl;
							pose = save_pose;
						}
					}

					// accept?
					mc.boltzmann( pose, "rbmin" );
				}

				if ( false ) { // alternate linmin, rottrials
					perturb_dna_jump( pose, dna_jump, trans_delta, rot_delta );
					pose.score(w);
					pose.rottrial( w, 1 );
					pose.new_score_pose();
					pose.score(w);
					pose.main_minimize( w, "linmin");
					pose.rottrial( w, 1 );
					pose.new_score_pose();
					pose.score(w);
					pose.main_minimize( w, "dfpmin");
					pose.rottrial( w, 1 );
					pose.new_score_pose();
					pose.score(w);
					mc.boltzmann( pose, "Rottrial-linmin" );
				}
			} // inner loop

			pose = mc.low_pose();
			//pose.dump_pdb("round"+string_of(n)+".pdb");
			mc.show_counters();
			prof::show();
		} // outer loop

		mc.show_counters();

		calc_rms(pose);
		out.write( tag, pose );
		//pose.dump_pdb("final."+string_of(nn)+".pdb");

		prof::show();
	} // decoy loop

}


///////////////////////////////////////////////////////////////////////////////
inline
int
bin_cen_d(
	float const cend
)
{
	if ( cend < 6 ) {
		return 1;
	} else if ( cend < 8 ) {
		return 2;
	} else if ( cend < 10 ) {
		return 3;
	} else if ( cend < 12 ) {
		return 4;
	} else {
		return 1000;
	}
}

///////////////////////////////////////////////////////////////////////////////
// Plan: implement three types of scores:
// 1. aa propensity given base and distance
// 2. distance frequency given base+distance / overall distance frequency
//    ( w/pseudo counts as in paper )
// 3. protein-dna vdw score

// then setup centroid mode atom_tree stuff.

// add weights for new scoring routines

// try simple docking simulations, create decoys, set weights on terms!



///////////////////////////////////////////////////////////////////////////////
namespace dna_scoring
{
	int const MAX_CEN_D( 4 );

	// interaction propensities
	//                          d           AA                 NA
	FArray3D_float aa_score( MAX_CEN_D, param::MAX_AA(), param::MAX_AA() );
	FArray3D_float r_score ( MAX_CEN_D, param::MAX_AA(), param::MAX_AA() );

	// for vdw
	//                         bb-atoms     dna-atoms          NA
	FArray3D_float closest_approach( 5, param::MAX_ATOM(), param::MAX_AA() );
	FArray3D_float atom_vdw_dna    ( 5, param::MAX_ATOM(), param::MAX_AA() );


	int aa_begin, aa_end, na_begin, na_end;
}

///////////////////////////////////////////////////////////////////////////////
void
setup_aa_bounds()
{
	using namespace dna_scoring;
	aa_begin = 1;
	aa_end   = 20;
	na_begin = 0;
	for ( int i=1; i<= param::MAX_AA()(); ++i ) {
		if ( !param_aa::is_DNA(i) ) continue;
		na_end = i;
		if ( !na_begin ) na_begin = i;
	}
}


///////////////////////////////////////////////////////////////////////////////
void
dna_pair_score(
							 pose_ns::Pose & pose,
							 float & aa_score_total,
							 float & r_score_total
							 )
{
	using namespace dna_scoring;


	aa_score_total = 0.0;
	r_score_total = 0.0;

	if ( !setup_dna_scoring() ) return;

	pose_update_cendist(pose);
	FArray2D_float const & cendist( pose.get_2D_score( pose_ns::CENDIST ) );

	int const nres( pose.total_residue() );
	for ( int i=1; i<= nres; ++i ) {
		int const aa( pose.res(i) );
		if ( !param_aa::is_protein(aa) ) continue;
		for ( int j=1; j<= nres; ++j ) {
			int const na(pose.res(j));
			if ( !param_aa::is_DNA(na)) continue;
			float const dist( std::sqrt( cendist(j,i)));
			int const d( bin_cen_d( dist ));
			if ( d <= MAX_CEN_D ) {
				aa_score_total += dna_scoring::aa_score(d,aa,na);
				r_score_total  += dna_scoring::r_score (d,aa,na);
			}
		}
	}




}


///////////////////////////////////////////////////////////////////////////////
float
dna_vdw(
	pose_ns::Pose & pose,
	int const i, // amino acid position
	int const j // dna position
)
{
	using namespace param_aa;
	using namespace dna_scoring;

	static bool init( false ), setup_success( true );
	if ( !init ) {
		setup_success = setup_dna_scoring();
		init = true;
	}
	if ( !setup_success ) return 0.0;

	int const aa( pose.res(i) );
	assert( is_protein(aa));

	int const na( pose.res(j) );
	assert( is_DNA(na));

	int const natoms( aaproperties_pack::natoms(na,pose.res_variant(j) ) );

	FArray3D_float const & full_coord( pose.full_coord() );
	int const fc_index_i( full_coord.index(1,1,i));
	int const fc_index_j( full_coord.index(1,1,j));

	float score(0.0);
	for ( int jj=1, jj_index=fc_index_j; jj <= natoms; ++jj, jj_index+=3 ) {
		numeric::xyzVector_float xyz_j( &full_coord[ jj_index ] );
		assert( xyz_j(1) == full_coord(1,jj,j) &&
						xyz_j(2) == full_coord(2,jj,j) &&
						xyz_j(3) == full_coord(3,jj,j) );

		int vdw_index = atom_vdw_dna.index(1,jj,na);
		for ( int ii=1, ii_index=fc_index_i; ii <= 5; ++ii, ii_index+=3,
						++vdw_index ) {
			if ( ii==5 && aa == aa_gly ) continue;
			numeric::xyzVector_float xyz_i( &full_coord[ ii_index ] );
			assert( xyz_i(1) == full_coord(1,ii,i) &&
							xyz_i(2) == full_coord(2,ii,i) &&
							xyz_i(3) == full_coord(3,ii,i) );

			float const atomvdw = atom_vdw_dna[ vdw_index ];
			assert( atomvdw == atom_vdw_dna( ii,jj,na ) );
			float const dist2 = atomvdw - distance_squared( xyz_i, xyz_j );
			if ( dist2 > 0.0 ) score += ( dist2 * dist2 ) / atomvdw;
		} // jj
	} // ii
	return score;
}



///////////////////////////////////////////////////////////////////////////////
// returns TRUE on success

bool
setup_dna_scoring()
{
	using namespace dna_scoring;
	static bool init( false ), ok( true );
	if ( init ) return ok;
	init = true;


	// read from file
	std::ifstream data( "/work/havranek/dna_scores.53.txt" ); //ja not there?
	if ( !data.good() ) {
		data.clear();
		data.close();
		data.clear();
		data.open( "/users/pbradley/dna/dna_scores.53.txt" );
	}

	if ( !data.good() ) {
		std::cout << "WARNING!! WARNING!! WARNING!! WARNING!! WARNING!!\n";
		std::cout << "couldnt open data file for dna centroid scoring!" <<
			std::endl;
		ok = false;
		return ok;
	}

	//std::ifstream data( "/work/havranek/dna_scores.53.txt" ); //ja not there?
	std::string line, tag, tag2;

	// initialize
	float const BOGUS_SCORE( 999.0 );
	aa_score = BOGUS_SCORE;
	r_score = BOGUS_SCORE;
	closest_approach = BOGUS_SCORE;

	while ( getline( data,line ) ) {
		std::istringstream l(line);
		l >> tag;
		int d,aa,na,bb,atomno;
		float f1,f2;
		if ( tag == "aa_score:" ) {
			l >> d >> aa >> na >> f1 >> tag2 >> f2;
			if ( l.fail() ) {
				std::cout << "PARSE ERROR! " << line << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			aa_score(d,aa,na) = f1;
			r_score(d,aa,na) = f2;
		} else if ( tag == "closest_approach:" ) {
			l >> bb >> atomno >> na >> f1;
			if ( l.fail() ) {
				std::cout << "PARSE ERROR1 " << line << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			closest_approach(bb,atomno,na) = f1;
			atom_vdw_dna(bb,atomno,na) = f1*f1;
		}
	} // getline

	setup_aa_bounds();

	for ( int d=1; d<= MAX_CEN_D; ++d ) {
		for ( int aa=aa_begin; aa<= aa_end; ++aa ) {
			for ( int na=na_begin; na<= na_end; ++na ) {
				if ( aa_score(d,aa,na) == BOGUS_SCORE ||
						 r_score(d,aa,na) == BOGUS_SCORE ) {
					std::cout << "PARSE ERROR2 " << d << ' ' << aa << ' ' << na <<
						std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

	for ( int i=1; i<=5; ++i ) {
		for ( int na=na_begin; na<= na_end; ++na ) {
			int const nav(1);
			for ( int jj=1; jj<= aaproperties_pack::natoms(na,nav); ++jj ) {
				if ( closest_approach(i,jj,na) == BOGUS_SCORE ) {
					std::cout << "PARSE ERROR3 " << i << ' ' << jj << ' ' << na <<
						std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

	return true;
}


///////////////////////////////////////////////////////////////////////////////
void
dna_centroid_docking()
{
	using namespace pose_ns;
	using namespace param_aa;

	Pose pdb_pose;
 	pose_from_pdb( pdb_pose, "/work/pbradley/dna/winged/1pue.ABE.pdb", true,
 								 false, true );


	// HACKING
	helix_in_groove( pdb_pose );
	std::exit(0);


	int const dna_jump(2); // should match tree below
	int const nres( pdb_pose.total_residue() );
	int const num_jump( 2 );
	{
		FArray1D_int cuts(num_jump);
		FArray2D_int jump_point(2,num_jump);

		cuts(1) = 16;
		cuts(2) = 32;
		jump_point(1,1) = 10;
		jump_point(2,1) = 26;
		jump_point(1,2) = 10;
		jump_point(2,2) = 96;
		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jump_point, cuts );
		f.reorder(10);
		pdb_pose.set_fold_tree(f);
	}


	pdb_pose.set_allow_chi_move( false );
	pdb_pose.set_allow_bb_move ( false );
	pdb_pose.set_allow_jump_move( dna_jump, true );

	// start profiling
	prof::reset( true );

	// scorefxn
	Score_weight_map w;
	w.set_weight( DNA_AA, 1.0 );
	w.set_weight( DNA_R, 1.0 );
	w.set_weight( VDW, 1.0 );


	// Monte carlo object
	Monte_carlo mc( pdb_pose, w, 2.0 /*temp*/);
	//mc.set_autotemp

	float const trans_delta( 1.0 );
	float const rot_delta( 5 );

	pdb_pose.dump_pdb("pdb_pose.pdb");

	for ( int nn=1; nn<= 500; ++nn ) {

		Pose pose;
		pose = pdb_pose;
		pose.set_native_pose( pdb_pose );

		pose.score(w);

		float const pdb_vdw( pose.get_0D_score( VDW ) );

		int const nres( pose.total_residue() );

		// randomly re-orient the protein
		{

			Vec protein_center(0.0);
			int count(0);
			for ( int i=1; i<= nres; ++i ) {
				if ( !pose.is_protein(i)) continue;
				protein_center += pose.centroid(i);
				++count;
			}
			protein_center /= count;

			Jump jump( pose.get_jump( dna_jump ) );

			Mat M( random_reorientation_matrix() );

			jump.rotation_by_matrix( pose.upstream_jump_stub(dna_jump),
															 protein_center,
															 M );

			pose.set_jump( dna_jump, jump );


			{ // recalc protein center
				Vec new_protein_center(0.0);
				// 			protein_center = 0.0;
				// 			assert( std::abs( protein_center(1) + protein_center(2) +
				// 												protein_center(3) ) < 1e-3 );
				count = 0;
				for ( int i=1; i<= nres; ++i ) {
					if ( !pose.is_protein(i)) continue;
					new_protein_center += pose.centroid(i);
					++count;
				}
				new_protein_center /= count;

				std::cout << "center dev: " <<
					distance( protein_center, new_protein_center ) << std::endl;
			}

			// now slide out of contact
			// find dna centroid nearest the protein center
			Vec axis(999.9);
			for ( int i=1; i<= nres; ++i ) {
				if (!is_DNA(pose.res(i))) continue;
				if ( distance( pose.centroid(i), protein_center ) < axis.length() ) {
					axis = protein_center - pose.centroid(i);
// 					std::cout << "closest dna: " << i << ' ' << axis.length() <<
// 						std::endl;
				}
			}

			pose.score( w );
			//pose.dump_pdb("random_rot.pdb");


			while ( pose.get_0D_score( VDW ) > pdb_vdw + 0.5 ) {
				// the protein is moving (downstream)
				// axis points from dna to protein
				// so want positive value (1.0) here
				jump.translation_along_axis( pose.upstream_jump_stub(dna_jump),
																		 axis, 1.0 );
				pose.set_jump( dna_jump, jump );
				pose.score( w );
				//static int counter(0);
				//pose.dump_pdb("slide_out."+string_of(++counter)+".pdb");
			}

			while ( pose.get_0D_score( VDW ) < pdb_vdw + 0.25 ) {
				jump.translation_along_axis( pose.upstream_jump_stub(dna_jump),
																		 axis, -0.5 );
				pose.set_jump( dna_jump, jump );
				pose.score( w );
				//static int counter(0);
				//pose.dump_pdb("slide_in."+string_of(++counter)+".pdb");
			}

		}

		pose.score(w);
		mc.reset(pose);
		mc.reset_counters();

		for ( int n=1; n<= 10; ++n ) {

			for ( int m=1; m<= 100; ++m ) {

				perturb_dna_jump( pose, dna_jump, trans_delta, rot_delta );
				mc.boltzmann( pose, "rbmove" );
			}
			pose = mc.low_pose();
			//pose.dump_pdb("round"+string_of(n)+".pdb");
			//mc.show_counters();
			//prof::show();

		} // outer loop

		mc.show_counters();

		calc_rms( pose );
		std::cout << pose.show_scores() << std::endl;

		if ( !score_filter( pose.get_0D_score( SCORE ), "dock", 0.1 ) ) {
			--nn;
			continue;
		}

		pose.dump_pdb("final."+string_of(nn)+".pdb");
		prof::show();
	} // decoy loop



}




///////////////////////////////////////////////////////////////////////////////
void
test_dna_scores()
{
	using namespace pose_ns;

	setup_dna_scoring(); // auto-initialize ? not done yet for vdw

	// setup scorefxn
	Score_weight_map w;
	w.set_weight( DNA_AA, 1.0 );
	w.set_weight( DNA_R, 1.0 );
	w.set_weight( VDW, 1.0 );

	std::ifstream data( stringafteroption("l").c_str() );

	prof::reset( true );
	std::string line;
	while ( getline( data, line ) ) {
		pose_ns::Pose pose;
		bool const ok( pose_from_pdb( pose, line, true, false, true ) );
		if ( !ok ) {
			std::cout << "WARNING:: failure to read pdb! " << line << std::endl;
			continue;
		}

		pose.score(w);

		helix_in_groove( pose );

		std::cout << line << ' ' << pose.show_scores() << std::endl;
	}
	prof::show();

	exit(0);
}



///////////////////////////////////////////////////////////////////////////////
//
// collect stats for centroid mode dna stuff

void
collect_dna_stats()
{
	using namespace param;
	using namespace param_aa;
	using namespace aaproperties_pack;
	using namespace dna_scoring;

	float const weight( 0.2 );


	setup_aa_bounds();

	assert( truefalseoption("enable_dna" ) );
// 	assert( truefalseoption("enable_dna" ) &&
// 					truefalseoption("dna_weights" ) &&
// 					truefalseoption("gen_born") );

	int const CA_index( 2 );

	// histograms of centroid-centroid distances
	FArray3D_int cend( MAX_CEN_D, MAX_AA()(), MAX_AA()() );

	FArray1D_int aa_count( MAX_AA()(), 0 );

	// initialize
	cend = 0;
	closest_approach = 1000.0;

	int file_count(0);

	std::ifstream data( stringafteroption("l").c_str() );

	std::string line;
	while ( getline( data, line ) ) {
		pose_ns::Pose pose;
		bool const ok( pose_from_pdb( pose, line, true, false, true ) );
		if ( !ok ) {
			std::cout << "WARNING:: failure to read pdb! " << line << std::endl;
			continue;
		}

		++file_count;
		int const nres( pose.total_residue() );

		// identify chain breaks
		int ncuts;
		FArray1D_int cuts(nres);
		find_chainbreaks( nres, pose.res(), pose.full_coord(), ncuts, cuts );
		FArray1D_bool is_cutpoint( nres, false );
		for ( int i=1; i<= ncuts; ++i ) {
			is_cutpoint( cuts(i) ) = true;
		}

		// get closest distances
		for ( int i=1; i<= nres; ++i ) {
			int const aa ( pose.res(i));
			int const aav( pose.res_variant(i));
			if ( is_protein( aa ) ) {
				++aa_count(aa);
				for ( int j=1; j<= nres; ++j ) {
					int const na ( pose.res(j));
					int const nav( pose.res_variant(j));
					if ( is_DNA( na ) ) {
						if ( j==1 || is_cutpoint(j-1) ) continue; // might have added P
						for ( int ii=1; ii<=5; ++ii ) {
							if ( ii == 5 && aa == aa_gly ) continue;
							for ( int jj=1; jj<= natoms( na,nav); ++jj ) {
								float const d
									( distance( pose.full_coord(ii,i), pose.full_coord(jj,j) ) );
								if ( d < closest_approach(ii,jj,na) ) {
									closest_approach(ii,jj,na) = d;
								}

								if ( d<2.0f ) {
									std::cout << "CLOSE_D: " << line << ' ' <<
										i << ' ' << ii << ' ' << atom_name(ii,aa,aav) << ' ' <<
										aa_name3(aa) << ' ' <<
										j << ' ' << jj << ' ' << atom_name(jj,na,nav) << ' ' <<
										aa_name3(na) << ' ' <<
										d << std::endl;
								}
							}
						}

						float const dist(distance(pose.centroid(i),pose.centroid(j)));
						float const ca_dist(distance(pose.full_coord(CA_index,i),
																				 pose.centroid(j)));
						int const d( bin_cen_d(dist));
						assert( d>=1 );
						if ( d <= MAX_CEN_D ) {
							std::cout << "CEN_D: " << aa_name3(na) << ' ' << dist << ' ' <<
								ca_dist << ' ' << aa << ' ' << aa_name3(aa) << std::endl;
							++cend(d,aa,na);
						}
					}
				}
			}
		}

	}


	// get overall amino acid frequencies
	FArray1D_float F_aa( MAX_AA()() );
	{
		int total(0);
		for ( int aa=aa_begin; aa<= aa_end; ++aa ) {
			total += aa_count(aa);
		}
		for ( int aa=aa_begin; aa<= aa_end; ++aa ) {
			assert( is_protein(aa) );
			F_aa(aa) = float( aa_count(aa)) / total;
			std::cout << "F_aa: " << aa_name3(aa) << F(9,3,F_aa(aa)) <<
				std::endl;
		}
	}

	// get overall distance distribution
	FArray1D_float F_d( MAX_CEN_D );
	{
		int total(0);
		for ( int d=1; d<= MAX_CEN_D; ++d ) {
			for ( int aa = aa_begin; aa<= aa_end; ++aa ) {
				for ( int na = na_begin; na<= na_end; ++na ) {
					F_d(d) += cend(d,aa,na);
					total += cend(d,aa,na);
				}
			}
		}

		for ( int d=1; d<= MAX_CEN_D; ++d ) {
			F_d(d) /= total;
		}
	}

	// calculate r_score
	for ( int aa = aa_begin; aa<= aa_end; ++aa ) {
		for ( int na = na_begin; na<= na_end; ++na ) {
			int total(0);
			for ( int d=1; d<= MAX_CEN_D; ++d ) {
				total += cend(d,aa,na);
			}

			for ( int d=1; d<= MAX_CEN_D; ++d ) {
				float freq( cend(d,aa,na) );

				if ( total ) freq /= total; // frequency of d given aa,na

				float const wfreq = // add pseudo counts from overall frequency of d
					F_d(d) / ( 1 + total * weight ) +
					total * weight * freq / ( 1 + total * weight );

				r_score(d,aa,na) = -std::log( wfreq / F_d(d) );
			}
		}
	}


	// calc aa_score
	for ( int na = na_begin; na<= na_end; ++na ) {
		for ( int d=1; d<= MAX_CEN_D; ++d ) {


			// total for aa_score
			int total(0);
			for ( int aa=aa_begin; aa<= aa_end; ++aa ) {
				total += cend(d,aa,na);
			}

			for ( int aa=aa_begin; aa<= aa_end; ++aa ) {
				if (!is_protein(aa)) continue;
				float freq( cend(d,aa,na) );
				if ( total ) freq /= total;
				float const propensity( freq/F_aa(aa));
				std::cout << "aa_prop: " << I(4,d) << ' ' <<
					aa_name3(na) << ' ' << aa_name3(aa) <<
					I(6,total) << I(6,cend(d,aa,na)) <<
					F(9,3,freq) << F(9,3,F_aa(aa)) << ' ' << propensity << std::endl;

				// no fewer than half the expected -- floors the score
				freq = std::max( freq, 0.3f * F_aa(aa) );

				aa_score(d,aa,na) = -std::log( freq / F_aa(aa) );
			}
		} // d
	} // na


	utility::io::ozstream out
		( "/users/pbradley/dna/dna_scores."+string_of(file_count)+".txt" );
	for ( int na=na_begin; na<= na_end; ++na ) {
		for ( int aa=aa_begin; aa<= aa_end; ++aa ) {
			for ( int d=1; d<= MAX_CEN_D; ++d ) {
				out << "aa_score: " << d << ' ' << aa << ' ' << na << ' ' <<
					aa_score( d,aa,na) << " r_score: " << r_score( d,aa,na) << ' ' <<
					aa_name3( aa) << ' ' << aa_name3(na) << '\n';
			}
		}
	}

	for ( int na=na_begin; na<= na_end; ++na ) {
		int const nav(1);
		for ( int j=1; j<= natoms(na,nav); ++j ) {
			for ( int bb=1; bb<= 5; ++bb ) {
				out << "closest_approach: " << bb << ' ' << j << ' ' << na << ' ' <<
					closest_approach(bb,j,na) << '\n';
			}
		}
	}
	out.close();

	exit(0);


}



///////////////////////////////////////////////////////////////////////////////




///////////////////////////////////////////////////////////////////////////////
void
setup_simple_dna_fold_tree(
	pose_ns::Pose & pose,
	FArray1Da_bool exclude
)
{
	using namespace pose_ns;
	using namespace kin;
	using namespace param_aa;


	///////////////////////
	int const nres( pose.total_residue() );
	FArray1D_int const & res( pose.res() );

	exclude.dimension( nres );

	/////////////////////////////////////////
	// now try to setup the correct fold_tree
	Fold_tree f;
	std::map< int, int > base_partner;
	int dna_jump(0);
	FArray1D_bool protein_interface( nres, false );
	{ // scope

		// identify basepairs
		DnaSeqInfo base_pairs;
		find_basepairs( pose, base_pairs );

		// identify chain breaks
		int ncuts;
		FArray1D_int cuts(nres);
		find_chainbreaks( nres, res, pose.full_coord(), ncuts, cuts );

		// first determine closest approach between protein and dna backbones
		int protein_anchor(0);
		int dna_anchor(0);
		{
			int const CA(2), P(dna_variables::p);
			float min_d(1000.0);
			for ( int i=1; i<= nres; ++i ) {
				if ( !is_protein(res(i) ) ) continue;
				for ( int j=1; j<= nres; ++j ) {
					if ( !is_DNA( res(j) ) || !base_partner[j] ) continue;
					float const d( distance( Vec( &pose.full_coord()(1,CA,i)),
																	 Vec( &pose.full_coord()(1,P,j)) ) );
					///jjh Exclude user-specified positions from anchor
					///jjh points
					if ( d<min_d && !exclude( i ) ) {
						min_d = d;
						protein_anchor = i;
						dna_anchor = j;
					}

					if ( d<12.0 ) {
						protein_interface( i ) = true;
					}

				}
			}
		}

		int const dna_anchor_partner( base_partner[ dna_anchor ] );
		std::cout << "protein_anchor: " << protein_anchor <<
			" dna_anchor: " << dna_anchor <<
			" dna_anchor_partner: " << dna_anchor_partner << std::endl;

		FArray2D_int jump_point( 2, ncuts );
		int njump(0);
		for ( int i=1; i<= ncuts+1; ++i ) {
			int const seg_begin( i==1       ? 1    : cuts(i-1)+1 );
			int const seg_end  ( i==ncuts+1 ? nres : cuts(i) );
			std::cout << "segs: " << i << ' ' << seg_begin << ' ' << seg_end <<
				std::endl;
			// couple of cases
			// 1 -- DNA or protein segment not containing anchor residue
			// 2 -- DNA or protein segment containing anchor residue
			int const aa( res(seg_begin ) );
			if ( is_protein( aa ) ) {
				if ( seg_begin > protein_anchor || seg_end < protein_anchor ) {
					// add a jump to protein_anchor
					// could pick a better jump_pos but we aren't doing protein
					// internal minimization yet
					jump_point(1,++njump) = std::min( seg_begin, protein_anchor );
					jump_point(2,  njump) = std::max( seg_begin, protein_anchor );
				}
			} else if ( is_DNA( aa ) ) {
				if ( seg_begin <= dna_anchor && seg_end >= dna_anchor ) {
					// add a jump to protein_anchor
					jump_point(1,++njump) = std::min( dna_anchor, protein_anchor );
					jump_point(2,  njump) = std::max( dna_anchor, protein_anchor );
					dna_jump = njump;
				} else if ( seg_begin <= dna_anchor_partner &&
										seg_end >= dna_anchor_partner ) {
					// add a jump to dna_anchor
					jump_point(1,++njump) = std::min( dna_anchor_partner, dna_anchor );
					jump_point(2,  njump) = std::max( dna_anchor_partner, dna_anchor );
				} else {
					std::cout << "TOO MUCH DNA! " << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			} else {
				std::cout << "BAD AA: " << aa << ' ' << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
		assert( njump == ncuts );


		if( jump_point(1, 2) == 14 ) jump_point(1, 2) = 21;


		f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );
		f.reorder( protein_anchor ); // guarantee some c2n folding
	} // scope
	pose.set_fold_tree( f );
}

///////////////////////////////////////////////////////////////////////////////
void
dna_test_old()
{
	assert( truefalseoption("enable_dna" ) &&
					truefalseoption("dna_weights" ) &&
					truefalseoption("gen_born") );

	std::ifstream data( stringafteroption("l").c_str() );

	std::string line;
	while ( getline( data, line ) ) {
		pose_ns::Pose pose;
		bool const ok( pose_from_pdb( pose, line, true, false, true ) );
		if ( ok ) dna_info( pose, line );
	}
	exit(0);
}


///////////////////////////////////////////////////////////////////////////////
// float
// knowledge_based_score(
// 											pose_ns::Pose const & pose
// 											)
// {
// 	// assumes basepairs have already been setup
// 	// probably dont want to recalculate on the fly b/c we might
// 	// get in a situation where a really strained base pair should
// 	// be penalized but simply stops being registered as a base pair...









// 	return 0.0;




// }



///////////////////////////////////////////////////////////////////////////////
void
find_chainbreaks(
	int const nres,
	FArray1D_int const & res,
	FArray3D_float const & coords,
	int & ncuts,
	FArray1D_int & cuts
)
{
	using namespace param_aa;

	int const N(1), C(3), O3STAR( dna_variables::o3star ), P( dna_variables::p );
	float const MAX_N2C_DIST( 2.0 ), MAX_O2P_DIST( 2.0 );

	ncuts = 0;
	for ( int i=1; i<= nres-1; ++i ) {
		int const aa( res(i) ), bb( res(i+1) );
		if ( is_protein(aa) && is_protein(bb) ) {
			if ( distance( Vec( &coords(1,C,i)),
										 Vec( &coords(1,N,i+1) ) ) > MAX_N2C_DIST ){
				std::cout << "Protein chainbreak: " << i << std::endl;
				cuts( ++ncuts ) = i;
			}
		} else if ( is_DNA(aa) && is_DNA(bb) ) {
			if ( distance( Vec( &coords(1,O3STAR,i)),
										 Vec( &coords(1,P,i+1) ) ) > MAX_O2P_DIST ){
				std::cout << "DNA chainbreak: " << i << std::endl;
				cuts( ++ncuts ) = i;
			}
		} else {
			cuts( ++ncuts ) = i;
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
// no convention for the sign of lsf_normal??

numeric::xyzVector_float
lsf_normal(
	std::vector< numeric::xyzVector_float > const & atoms_in
)
{
	using numeric::xyzVector_float;

	// translate the atoms so center of mass is at the origin
	xyzVector_float cm(0.0);
	int const natoms( atoms_in.size() );
	for ( int i=0; i<natoms; ++i ) {
		cm += atoms_in[i];
	}

	cm /= natoms;

	std::vector< xyzVector_float > atoms; atoms.reserve( natoms );
	for ( int i=0; i<natoms; ++i ) {
		atoms.push_back( atoms_in[i] - cm );
	}


	/////////////////////////////////////////////////////////////////////////////
	// this is all stolen from Alex Morozov's code:
	/////////////////////////////////////////////////////////////////////////////

	// Create a matrix which provides coeffs for the cubic equation.
	// Note that A(i,j) = A(j,i).
	FArray2D_float A( 3, 3, 0.0f );
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = i; j <= 3; ++j ) {
			for ( int a = 0; a < natoms; ++a ) {
				A(i,j) += atoms[a](i) * atoms[a](j);//coords2fit(i,a)*coords2fit(j,a);
			}
		}
	}

	// Compute cubic eqn. coeffs:
	float a = A(1,1) + A(2,2) + A(3,3); // alpha
	float b = A(1,3)*A(1,3) + A(1,2)*A(1,2) + A(2,3)*A(2,3)
		- A(2,2)*A(3,3) - A(1,1)*A(3,3) - A(1,1)*A(2,2); // beta
	float g = A(1,1)*A(2,2)*A(3,3) + 2*A(1,2)*A(2,3)*A(1,3)
		- A(1,1)*A(2,3)*A(2,3) - A(2,2)*A(1,3)*A(1,3)
		- A(3,3)*A(1,2)*A(1,2); // gamma

	// The solution to the cubic eqn. is obtained through neglecting the
	// lambda**3 term (lambda is small for nearly planar atom sets):
	// WARNING: bound to fail for REALLY nonplanar atom sets!
	float lambda = (-b - std::sqrt(b*b - 4*a*g) )/(2*a);

	// debug
	//std::cout << "lambda = " << lambda << std::endl;
	//std::cout << "D = " << (b*b - 4*a*g) << std::endl;
	// Finally, compute the eigenvector corresponding to the least squares plane:

	xyzVector_float normal_f;
	normal_f(1) = (A(2,2)-lambda)*A(1,3) - A(1,2)*A(2,3);
	normal_f(2) = (A(1,1)-lambda)*A(2,3) - A(1,2)*A(1,3);
	normal_f(3) = A(1,2)*A(1,2) - (A(2,2)-lambda)*(A(1,1)-lambda);

	normal_f.normalize();

	if ( false ) { // debug
		for ( int i=0; i<= 20; ++i ) {
			// hack b/c random_rotation... returns double
			numeric::xyzVector_double Dtmp_normal
				( normal_f(1), normal_f(2), normal_f(3) );
			if ( i ) {
				Dtmp_normal = random_axis_rotation_matrix( 30.0f ) * Dtmp_normal;
			}
			xyzVector_float tmp_normal
				( Dtmp_normal(1),Dtmp_normal(2),Dtmp_normal(3));
			assert( tmp_normal.is_normalized( 0.001 ) );
			float avg(0.0);
			int count(0);
			for ( int j=0; j< natoms; ++j ) {
				for ( int k=j+1; k< natoms; ++k ) {
					++count;
					avg += std::abs( dot( tmp_normal, atoms[j] - atoms[k] ) ) /
						( atoms[j] - atoms[k] ).length();
				}
			}
			avg/=count;
			std::cout << "avg= " << i << ' ' <<
				90.0f - numeric::conversions::degrees( arccos( avg ) ) << std::endl;
		}
	}

	return normal_f;
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
setup_base_atoms(
								 int const aa,
								 FArray2DB_float const & coords,
								 std::vector< numeric::xyzVector_float > & atoms
								 )
{
	static bool init(false );
	if ( !init ) {
		init = true;
		init_triad_coords();
	}
	using namespace param_aa;
	int const icoor_index( aa == na_ade ? 1 : ( aa == na_cyt ? 2 :
											 ( aa == na_gua ? 3 : 4 ) ) );
	for ( int i=1; i<= param::MAX_ATOM()(); ++i ) {
		if ( dna_variables::triad_icoor(1,i,icoor_index) !=
				 static_cast< float >( dna_variables::empty_value ) ) {
			atoms.push_back( numeric::xyzVector_float( &coords(1,i) ) );
		}
	}
	assert( int( atoms.size() ) ==
					dna_variables::triad_icoor_natoms( icoor_index ) );
}

///////////////////////////////////////////////////////////////////////////////
bool
is_orthonormal(
							 numeric::xyzMatrix_float const & M,
							 float const tol
							 )
{
	numeric::xyzMatrix_float X( M*M.transposed() );
	float dev( 0.0 );
	for( int i=1; i<=3; ++i ) {
		for( int j=1; j<=3; ++j ) {
			dev += std::abs( X(i,j) - ( i == j ? 1.0 : 0.0 ) );
		}
	}
	//std::cout << "ortho dev: " << dev << std::endl;
	return dev < tol;
}

///////////////////////////////////////////////////////////////////////////////
// stolen/borrowed from Alex Morozov!

kin::Stub
get_base_stub(
							int const aa,
							int const strand,
							FArray2Da_float coords
							)
{
	using namespace param_aa;
	using namespace numeric;
	using numeric::conversions::degrees;
	using dna_variables::c4star;
	using dna_variables::c3star;

	int a1(0),a2(0);
	get_y_axis_atoms( aa, strand, a1, a2 );

	xyzVector_float x_axis, y_axis, z_axis, origin,
		v1( &coords(1,a1)), v2( &coords(1,a2) );

	y_axis = v2 - v1;
	y_axis.normalize();

	origin = 0.5f * (v1 + v2 );


	std::vector< xyzVector_float > base_atoms;
	setup_base_atoms( aa, coords, base_atoms );

	z_axis = lsf_normal( base_atoms );

	// flip z-axis if necessary
	// not handling base flips correctly
	xyzVector_float orient
		( xyzVector_float(&coords(1,c3star)) - xyzVector_float(&coords(1,c4star)));
	if ( strand == 2 ) orient *= -1.0f;
	orient.normalize();
	if ( dot( z_axis, orient ) < 0.0 )  z_axis *= -1.0f;
	//std::cout << "dot_orient: " << degrees( arccos( dot( z_axis, orient )))<<
	//std::endl;


	// subtract off component in y-axis direction
	z_axis -= dot( z_axis, y_axis ) * y_axis;
	z_axis.normalize();
	assert( std::abs( dot(y_axis, z_axis) ) < 1e-3 );

	x_axis = cross( y_axis, z_axis );
	x_axis.normalize();

	return kin::Stub( xyzMatrix_double::cols( x_axis, y_axis, z_axis ), origin );
}

///////////////////////////////////////////////////////////////////////////////
// stolen directly from Alex

void
calc_dna_dihedrals(
	int const nres,
	FArray1D_int const & aan,
	FArray3D_float const & xyz,
	FArray2D_float & dna_dihedrals
	)
{
	using dna_variables::p;
	using dna_variables::o5star;
	using dna_variables::c5star;
	using dna_variables::c4star;
	using dna_variables::c3star;
	using dna_variables::o3star;
	using dna_variables::c2;
	using dna_variables::c4A;
	using dna_variables::c4G;
	using dna_variables::o4star;
	using dna_variables::c1star;
	using dna_variables::n9A;
	using dna_variables::n9G;
	using dna_variables::n1C;
	using dna_variables::n1T;
	using param_aa::aa_name3;
	using param_aa::is_DNA;

	for ( int i=1; i<= nres; ++i ) {
		int const aa(aan(i));
		if ( !is_DNA(aa) ) continue;
		if ( i == 1 || !is_DNA( aan(i-1) ) ) {
			dna_dihedrals(1,i) = 0.0; //alpha: O3*_{i-1} P O5* C5* [9 1 4 5]
		} else {
			//alpha: O3*_{i-1} P O5* C5* [9 1 4 5]
			dihedral_bk(xyz(1,o3star,i-1),xyz(1,p,i),xyz(1,o5star,i),xyz(1,c5star,i),
									dna_dihedrals(1,i) );
		}
		//beta: P O5* C5* C4* [1 4 5 6]
		dihedral_bk(xyz(1,p,i),xyz(1,o5star,i),xyz(1,c5star,i),xyz(1,c4star,i),
								dna_dihedrals(2,i) );
		//gamma: O5* C5* C4* C3* [4 5 6 8]
		dihedral_bk(xyz(1,o5star,i),xyz(1,c5star,i),xyz(1,c4star,i),
								xyz(1,c3star,i),dna_dihedrals(3,i) );
		//delta: C5* C4* C3* O3* [5 6 8 9]
		dihedral_bk(xyz(1,c5star,i),xyz(1,c4star,i),xyz(1,c3star,i),xyz(1,o3star,i),
								dna_dihedrals(4,i) );
		if ( i == nres || !is_DNA( aan(i+1) ) ) {
			dna_dihedrals(5,i) = 0.0; //epsilon: C4* C3* O3* P_{i+1} [6 8 9 1]
			dna_dihedrals(6,i) = 0.0; //xi: C3* O3* P_{i+1} O5*_{i+1} [8 9 1 4]
		} else {
			//epsilon: C4* C3* O3* P_{i+1} [6 8 9 1]
			dihedral_bk(xyz(1,c4star,i),xyz(1,c3star,i),xyz(1,o3star,i),xyz(1,p,i+1),
									dna_dihedrals(5,i) );
			//xi: C3* O3* P_{i+1} O5*_{i+1} [8 9 1 4]
			dihedral_bk(xyz(1,c3star,i),xyz(1,o3star,i),xyz(1,p,i+1),
									xyz(1,o5star,i+1),dna_dihedrals(6,i) );
		}
		if (aa_name3(aa)=="  A") {
			//chi: O4* C1* N9 C4 for A
			dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n9A,i),xyz(1,c4A,i),
									dna_dihedrals(7,i) );
		} else if (aa_name3(aa)=="  G") {
			//chi: O4* C1* N9 C4 for G
			dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n9G,i),xyz(1,c4G,i),
									dna_dihedrals(7,i) );
		} else if (aa_name3(aa)=="  C") {
			//chi: O4* C1* N1 C2 for C
			dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n1C,i),xyz(1,c2,i),
									dna_dihedrals(7,i) );
		} else if (aa_name3(aa)=="  T") {
			//chi: O4* C1* N1 C2 for T
			dihedral_bk(xyz(1,o4star,i),xyz(1,c1star,i),xyz(1,n1T,i),xyz(1,c2,i),
									dna_dihedrals(7,i) );
		} else {
			std::cerr << "Unknown basepair name: ==" << aa_name3(aa) << "==!"
								<< std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
}

// these are indexed using the indices returned by
// Alex's function get_basestep_index
//
namespace new_bs_info
{
	FArray3D_float new_bs_Fij   ( 6, 6, 16 );
	FArray2D_float new_bs_mean  ( 6, 16 );
	FArray2D_float new_bs_stddev( 6, 16 );
}



//
void
load_dna_score_tables()
{
	using namespace new_bs_info;

	static bool init( false );
	if ( init ) return;
	init = true;
	LoadDNAParams(); // in dna.cc

	// load my personal base-step parameters

	{
		std::ifstream data( "/work/pbradley/dna/jim_set/58.params" );
		std::string line;
		bool fail( false );
		while ( getline( data,line ) ) {
			std::istringstream l( line );
			std::string tag, step;

			l >> tag;
			if ( l.fail() ) continue;
			if ( tag == "SD:" ) {
				int prm;
				float mean, stddev;
				l >> step >> prm >> mean >> stddev;
				if ( l.fail() || prm < 1 || prm > 6 ) {
					std::cout << "failline:" << line << std::endl;
					fail = true;
					break;
				}
				int const index
					( get_basestep_index( step.substr(0,1), step.substr(1,1) ) );
				new_bs_mean  ( prm, index ) = mean;
				new_bs_stddev( prm, index ) = stddev;


			} else if ( tag == "stiffness:" ) {
				int i;
				l >> step >> i;
				int const index
					( get_basestep_index( step.substr(0,1), step.substr(1,1) ) );
				for ( int j=1; j<= 6; ++j ) {
					l >> new_bs_Fij( i, j, index );
				}
				if ( l.fail() ) {
					std::cout << "failline:" << line << std::endl;
					fail = true;
					break;
				}
			}
		}

		assert(!fail);

		// check params:
		for ( int i=1; i<= 6; ++i ) {
			for ( int j=1; j<= 6; ++j ) {
				for ( int k=1; k<= 16; ++k ) {
					assert( std::abs( new_bs_Fij(i,j,k) - new_bs_Fij(j,i,k) )<1e-3);
				}
			}
		}
	}
}



// STOLEN from Alex Morozov
float
base_pair_score(
	int const aa,
	FArray1Da_float params
)
{
	using namespace dna_variables;
	using namespace param;
	using namespace param_aa;

	load_dna_score_tables();

	params.dimension(6);

	// max allowed deviation from average beyond which the score is capped
	// PB -- this seems very large! it's in standard deviations, right?
	float const MAX_SCORE_DEV = 100.0;

	// accumulate score values
	float score(0.0);

	//Make a label for the basestep type lookup:
	int const ind = get_basepair_index( aa_name3(aa)[2] );

	for ( int i = 1; i <= 6; ++i ) {
		float const max_delta_i = MAX_SCORE_DEV*stddevs_bp(ind,i);

		float delta_i = params(i) - means_bp(ind,i);
		// reset outliers to max allowed values
		// (beyond which the potential is unreliable)
		if ( delta_i > max_delta_i ) delta_i = max_delta_i;
		if ( delta_i < -max_delta_i ) delta_i = -max_delta_i;

		for ( int j = 1; j <= 6; ++j ) {
			float const max_delta_j = MAX_SCORE_DEV*stddevs_bp(ind,j);

			float delta_j = params(j) - means_bp(ind,j);
			// reset outliers to max allowed values
			// (beyond which the potential is unreliable)
			if ( delta_j > max_delta_j ) delta_j = max_delta_j;
			if ( delta_j < -max_delta_j ) delta_j = -max_delta_j;

			score += Gij(ind,i,j) * delta_i * delta_j;
		}
	}

	if ( debug_bp_deriv ) {
		score = 0;
		for ( int i=1; i<= 6; ++i ) {
			if ( i == debug_bp_deriv || debug_bp_deriv > 6 ) {
				score +=
					( params( i ) - means_bp( ind, i ) ) *
					( params( i ) - means_bp( ind, i ) );
			}
		}
	}
	return score;
}



///////////////////////////////////////////////////////////////////////////////
// STOLEN from Alex Morozov
float
new_base_step_score(
	int const aa1,
	int const aa2,
	FArray1Da_float params
)
{
	//using namespace dna_variables;
	using namespace param;
	using namespace param_aa;
	using namespace new_bs_info;

	load_dna_score_tables();

	params.dimension(6);

	// max allowed deviation from average beyond which the score is capped
	// PB -- this seems very large! it's in standard deviations, right?
	float const MAX_SCORE_DEV = 100.0;

	// accumulate score values
	float score(0.0);

	//Make a label for the basestep type lookup:
	int const ind = get_basestep_index( aa_name3(aa1).substr(2),
																			aa_name3(aa2).substr(2) );

	for ( int i = 1; i <= 6; ++i ) {
		float const max_delta_i = MAX_SCORE_DEV * new_bs_stddev(i,ind);

		float delta_i = params(i) - new_bs_mean(i,ind);
		// reset outliers to max allowed values
		// (beyond which the potential is unreliable)
		if ( delta_i > max_delta_i ) delta_i = max_delta_i;
		if ( delta_i < -max_delta_i ) delta_i = -max_delta_i;

		for ( int j = 1; j <= 6; ++j ) {
			float const max_delta_j = MAX_SCORE_DEV * new_bs_stddev(j,ind);

			float delta_j = params(j) - new_bs_mean(j,ind);
			// reset outliers to max allowed values
			// (beyond which the potential is unreliable)
			if ( delta_j > max_delta_j ) delta_j = max_delta_j;
			if ( delta_j < -max_delta_j ) delta_j = -max_delta_j;

			score += new_bs_Fij(i,j,ind) * delta_i * delta_j;
		}
	}

	if ( debug_bs_deriv ) {
		score = 0;
		for ( int i=1; i<= 6; ++i ) {
			if ( i == debug_bs_deriv || debug_bs_deriv > 6 ) {
				score +=
					( params( i ) - new_bs_mean( i, ind ) ) *
					( params( i ) - new_bs_mean( i, ind ) );
			}
		}
	}
	return score;
}



///////////////////////////////////////////////////////////////////////////////
//
// aa1 is in strand1, aa2 is in strand2
//
void
base_pair_params(
	int const aa1,
	int const aa2,
	FArray2Da_float base1_coords,
	FArray2Da_float base2_coords,
	FArray1D_float & params // output
)
{
	using namespace param_aa;
	using namespace numeric;
	using numeric::conversions::degrees;

	base1_coords.dimension( 3, param::MAX_ATOM()() );
	base2_coords.dimension( 3, param::MAX_ATOM()() );


	kin::Stub const
		stub1( get_base_stub( aa1, 1, base1_coords ) ), // strand = 1
		stub2( get_base_stub( aa2, 2, base2_coords ) ); // strand = 2

	// copy matrices
	xyzMatrix_float M1( stub1.M ), M2( stub2.M );

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );

	bool base_flipped = false;
	if ( dot( M1.col_z(), M2.col_z() ) < 0.0 ) {
		base_flipped = true;
		std::cout << "base_pair_params: base flip!!!" << std::endl;
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// get angle between the y-axes
	float const gamma( arccos( dot( M1.col_y(), M2.col_y() ) ) );

	xyzVector_float const bo( ( cross( M2.col_y(), M1.col_y() ) ).normalized() );

	xyzMatrix_float R_gamma_2( rotation_matrix( bo, gamma/2.0f ) );

	M2 = R_gamma_2 * M2;
	M1 = R_gamma_2.transposed() * M1;

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );

	// build mid-base-pair triad
	assert( distance( M1.col_y(), M2.col_y() ) < 1e-3 );
	assert( std::abs( dot( bo, M1.col_y() ) ) < 1e-3 );

	xyzMatrix_float MBT;
	MBT.col_y( M1.col_y() );

	assert( std::abs( dot( M1.col_z(), MBT.col_y() ) ) < 1e-3 );
	assert( std::abs( dot( M2.col_z(), MBT.col_y() ) ) < 1e-3 );
	assert( std::abs( dot( M1.col_x(), MBT.col_y() ) ) < 1e-3 );
	assert( std::abs( dot( M2.col_x(), MBT.col_y() ) ) < 1e-3 );

	// get
	MBT.col_x( ( 0.5f * ( M1.col_x() + M2.col_x() ) ).normalized() );
	MBT.col_z( ( 0.5f * ( M1.col_z() + M2.col_z() ) ).normalized() );

	assert( is_orthonormal( MBT, 1e-3 ) );

	// angular params

	// propellor
	// z,x,y make rh coord system
	params(1) = std::atan2( dot( M1.col_z(), M2.col_x() ),
													dot( M1.col_z(), M2.col_z() ) );
	{
		float const tmp1( std::abs( params(1) ) );
		float const tmp2( arccos( dot( M1.col_z(), M2.col_z() ) ) );
		assert( std::abs( tmp1 - tmp2 ) <1e-2 );
	}
	assert( std::abs( std::abs( params(1) ) -
										arccos( dot( M1.col_z(), M2.col_z() ) ) )<1e-2 );



	// buckle:
	params(2) = gamma * dot( bo, MBT.col_x() );

	// opening
	params(3) = gamma * dot( bo, MBT.col_z() );

	// translational params
	xyzVector_float const displacement( stub1.v - stub2.v );

	params(4) = dot( displacement, MBT.col_x() );
	params(5) = dot( displacement, MBT.col_y() );
	params(6) = dot( displacement, MBT.col_z() );

	/////////////
	// debugging:
	{
		{ // sin gamma version of params(2) is a simple dot product:
			float const tmp1 = std::sin( gamma ) * params(2) / gamma;
			float const tmp2 = dot( xyzVector_float( cross( stub2.M.col_y(),
																											stub1.M.col_y() ) ),
															MBT.col_x() );
			assert( std::abs( tmp1-tmp2)<1e-2 );
		}

		{ // sin gamma version of params(3) is a simple dot product:
			float const tmp1( std::sin( gamma ) * params(3) / gamma );
			float const tmp2( dot( xyzVector_float( cross( stub2.M.col_y(),
																										 stub1.M.col_y() )),
														 MBT.col_z() ) );
			assert( std::abs( tmp1-tmp2)<1e-2 );
		}

		// check sign conventions
		float phi_prime( arccos( dot( bo, MBT.col_x() ) ) );
		if ( dot( cross( bo, MBT.col_x() ), MBT.col_y() ) < 0.0f ) {
			phi_prime *= -1.0f;
		}

		xyzVector_float tmp( cross( M2.col_z(), M1.col_z() ) );
		assert( cross( tmp, MBT.col_y() ).length() <1e-2 );

		float const p1x =
			std::asin( dot( MBT.col_y(), cross( M2.col_x(), M1.col_x() ) ) );
		float const p1z =
			std::asin( dot( MBT.col_y(), cross( M2.col_z(), M1.col_z() ) ) );
		assert( base_flipped ||
						( std::abs( params(1) - p1x ) + std::abs( params(1) - p1z )<1e-2));
		//std::cout << "equal? p1: " << params(1) << ' ' << p1x << ' ' << p1z <<
 		//	std::endl;

		float const p2 = gamma * std::cos( phi_prime );
		float const p3 = gamma * std::sin( phi_prime );
		float const dev( std::abs( p2 - params(2) ) + std::abs( p3 - params(3) ) );
		//std::cout << "dev: " << dev << std::endl;
		assert( dev < 1e-2 );

	}
	// check sign conventions
	assert( params(1) * dot( MBT.col_y(), cross( M2.col_x(), M1.col_x() ) ) > 0);

	// convert to degrees
	params(1) = degrees( params(1) );
	params(2) = degrees( params(2) );
	params(3) = degrees( params(3) );

	if ( verbose ) {
		std::cout << "params: " << aa_name3(aa1) << ' ' << aa_name3(aa2) <<
			" Prop: " << F(9,3,params(1)) <<
			" Buck: " << F(9,3,params(2)) <<
			" Open: " << F(9,3,params(3)) <<
			" Sher: " << F(9,3,params(4)) <<
			" Strc: " << F(9,3,params(5)) <<
			" Stag: " << F(9,3,params(6)) <<
			" gamm: " << F(9,3,degrees(gamma)) << std::endl;
	}

}

///////////////////////////////////////////////////////////////////////////////
//
// aa1 is rsd i, aa2 is rsd i+1
//
void
new_base_step_params(
	int const aa1,
	int const aa2,
	FArray2Da_float base1_coords,
	FArray2Da_float base2_coords,
	FArray1D_float & params // output
)
{
	using namespace param_aa;
	using namespace numeric;
	using numeric::conversions::degrees;

	base1_coords.dimension( 3, param::MAX_ATOM()() );
	base2_coords.dimension( 3, param::MAX_ATOM()() );


	kin::Stub const
		stub1( get_base_stub( aa1, 1, base1_coords ) ), // strand = 1
		stub2( get_base_stub( aa2, 1, base2_coords ) ); // strand = 2

	// copy matrices
	xyzMatrix_float M1( stub1.M ), M2( stub2.M );

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );

	if ( dot( M1.col_z(), M2.col_z() ) < 0.0 ) {
		std::cout << "new_base_step_params: base flip!" << std::endl;
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// get angle between the z-axes
	float const gamma( arccos( dot( M1.col_z(), M2.col_z() ) ) );

	xyzVector_float const rt( ( cross( M2.col_z(), M1.col_z() ) ).normalized() );

	xyzMatrix_float R_gamma_2( rotation_matrix( rt, gamma/2.0f ) );

	M2 = R_gamma_2 * M2;
	M1 = R_gamma_2.transposed() * M1;

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );

	// build mid-base-pair triad
	assert( distance( M1.col_z(), M2.col_z() ) < 1e-3 );
	assert( std::abs( dot( rt, M1.col_z() ) ) < 1e-3 );

	xyzMatrix_float MBT;
	MBT.col_z( M1.col_z() );

	assert( std::abs( dot( M1.col_x(), MBT.col_z() ) ) < 1e-3 );
	assert( std::abs( dot( M2.col_x(), MBT.col_z() ) ) < 1e-3 );
	assert( std::abs( dot( M1.col_y(), MBT.col_z() ) ) < 1e-3 );
	assert( std::abs( dot( M2.col_y(), MBT.col_z() ) ) < 1e-3 );

	// get
	MBT.col_y( ( 0.5f * ( M1.col_y() + M2.col_y() ) ).normalized() );
	MBT.col_x( ( 0.5f * ( M1.col_x() + M2.col_x() ) ).normalized() );

	assert( is_orthonormal( MBT, 1e-3 ) );

	// angular params

	// TWIST
	// x,y,z make rh coord system
	params(1) = std::atan2( dot( M1.col_x(), M2.col_y() ),
													dot( M1.col_x(), M2.col_x() ) );
	{
		float const tmp1( std::abs( params(1) ) );
		float const tmp2( arccos( dot( M1.col_x(), M2.col_x() ) ) );
		assert( std::abs( tmp1 - tmp2 ) <1e-2 );
	}
	assert( std::abs( std::abs( params(1) ) -
										arccos( dot( M1.col_x(), M2.col_x() ) ) )<1e-2 );



	// ROLL:
	params(2) = gamma * dot( rt, MBT.col_y() );

	// TILT
	params(3) = gamma * dot( rt, MBT.col_x() );

	// translational params
	xyzVector_float const displacement( stub1.v - stub2.v );

	params(4) = dot( displacement, MBT.col_x() ); // SHIFT
	params(5) = dot( displacement, MBT.col_y() ); // SLIDE
	params(6) = dot( displacement, MBT.col_z() ); // RISE

	/////////////
	// debugging:
	{
		{ // sin gamma version of params(2) (roll) is a simple dot product:
			float const tmp1 = std::sin( gamma ) * params(2) / gamma;
			float const tmp2 = dot( xyzVector_float( cross( stub2.M.col_z(),
																											stub1.M.col_z() ) ),
															MBT.col_y() );
			assert( std::abs( tmp1-tmp2)<1e-2 );
		}

		{ // sin gamma version of params(3) (tilt) is a simple dot product:
			float const tmp1( std::sin( gamma ) * params(3) / gamma );
			float const tmp2( dot( xyzVector_float( cross( stub2.M.col_z(),
																										 stub1.M.col_z() )),
														 MBT.col_x() ) );
			assert( std::abs( tmp1-tmp2)<1e-2 );
		}

		// check sign conventions
		float phi_prime( arccos( dot( rt, MBT.col_y() ) ) );
		if ( dot( cross( rt, MBT.col_y() ), MBT.col_z() ) < 0.0f ) {
			phi_prime *= -1.0f;
		}

		xyzVector_float tmp( cross( M2.col_x(), M1.col_x() ) );
		assert( cross( tmp, MBT.col_z() ).length() <1e-2 );

		float const p1x =
			std::asin( dot( MBT.col_z(), cross( M2.col_y(), M1.col_y() ) ) );
		float const p1z =
			std::asin( dot( MBT.col_z(), cross( M2.col_x(), M1.col_x() ) ) );
		//std::cout << "equal? p1: " << params(1) << ' ' << p1x << ' ' << p1z <<
 		//	std::endl;
		assert( std::abs( params(1) - p1x ) + std::abs( params(1) - p1z )<1e-2);

		float const p2 = gamma * std::cos( phi_prime );
		float const p3 = gamma * std::sin( phi_prime );
		float const dev( std::abs( p2 - params(2) ) + std::abs( p3 - params(3) ) );
		//std::cout << "dev: " << dev << std::endl;
		assert( dev < 1e-2 );

	}
	// check sign conventions
	assert( params(1) * dot( MBT.col_z(), cross( M2.col_y(), M1.col_y() ) ) > 0);

	// convert to degrees
	params(1) = degrees( params(1) );
	params(2) = degrees( params(2) );
	params(3) = degrees( params(3) );

	if ( verbose ) {
		std::cout << "bs-params: " << aa_name3(aa1) << ' ' << aa_name3(aa2) <<
			" Twst: " << F(9,3,params(1)) <<
			" Roll: " << F(9,3,params(2)) <<
			" Tilt: " << F(9,3,params(3)) <<
			" Shft: " << F(9,3,params(4)) <<
			" Slid: " << F(9,3,params(5)) <<
			" Rise: " << F(9,3,params(6)) <<
			" gamm: " << F(9,3,degrees(gamma)) << std::endl;
	}

}


///////////////////////////////////////////////////////////////////////////////
float
eval_dna_bp_score(
									pose_ns::Pose const & pose
									)
{
	int const nres( pose.total_residue() );

	float score(0.0);
	FArray1D_float params(6);

	for ( int i=1; i<= nres; ++i ) {
		if ( count_pair_bp( i, pose ) ) {
			int const j( pose.basepair_partner( i ) );
			if ( j > i ) {
				// calculate params
				base_pair_params( pose.res(i), pose.res(j), pose.full_coord()(1,1,i),
													pose.full_coord()(1,1,j), params );
				score += base_pair_score( pose.res(i), params );
			}
		}
	}
	return score;



}



///////////////////////////////////////////////////////////////////////////////
float
eval_new_dna_bs_score(
	pose_ns::Pose const & pose
)
{
	int const nres( pose.total_residue() );

	float score(0.0);
	FArray1D_float params(6);

	for ( int i=1; i<= nres-1; ++i ) {
		if ( count_pair_bs( i, pose ) ) {

			// calculate params
			new_base_step_params( pose.res(i), pose.res(i+1),
														pose.full_coord()(1,1,i),
														pose.full_coord()(1,1,i+1), params );
			float const tmp_score
				( new_base_step_score( pose.res(i), pose.res(i+1), params ) );
			score += tmp_score;
			if ( verbose ) std::cout << "new_bs_score: " << i << ' ' <<
											 param_aa::aa_name3( pose.res(i) ) << tmp_score <<
											 std::endl;
		}
	}
	return score;
}



///////////////////////////////////////////////////////////////////////////////
void
bqian_dna_test()
{
	// read, repack, and minimize bin's decoys

	assert( truefalseoption("enable_dna" ) &&
					truefalseoption("dna_weights" ) &&
					truefalseoption("gen_born") );


	// setup
	pose_setup_packer();
	pose_setup_minimizer(1e-4);

	std::ifstream data( stringafteroption("l").c_str() );

	std::string line;
	while ( getline( data, line ) ) {
		pose_ns::Pose pose;

		bool const ok( pose_from_pdb( pose, line, true, false, true ) );
		if (!ok ) {
			std::cout << "input failed: " << line << std::endl;
			continue;
		}

		dna_info( pose, line );



	}
	exit(0);




}

///////////////////////////////////////////////////////////////////////////////
void
specificity_check_zf(
	pose_ns::Pose & start_pose
)
{
	using namespace pose_ns;

	///////////////////
	// start_pose info:
	int const nres( start_pose.total_residue() );
	int nres_protein(0);
	for ( int i=1; i<= nres; ++i ) {
		if ( start_pose.is_protein(i) ) nres_protein = i;
	}
	assert( nres_protein + 11 + 11 == nres );
	std::map< int, int > partner;
	{ // get base partners
		DnaSeqInfo base_pairs;
		find_basepairs( start_pose, base_pairs );
	}


	////////
	// setup
	std::vector< int > dna_pos_list;

	dna_pos_list.push_back(  8 );
	dna_pos_list.push_back(  9 );
	dna_pos_list.push_back( 10 );
	dna_pos_list.push_back( 11 );

	std::vector< std::string > dnas;

	// "wt"
	dnas.push_back( "GCGT" );

	// 1 rsd changes from "wt" -- should be bad for wt protein
	dnas.push_back( "GAGT" );
	dnas.push_back( "GCTT" );

	// three from rebar
	dnas.push_back( "GACC" );
	dnas.push_back( "GCAC" );
	dnas.push_back( "GCGC" );

	// scorefxn
	Score_weight_map w( score12 );

	for ( int d=0; d< int(dnas.size()); ++d ) {

		Pose pose;
		pose = start_pose;

		std::string const & dna_word( dnas[d] );

		///////////////////////////
		// mutate the dna positions
		for ( int i=0; i<int(dna_word.size()); ++i ) {
			using aaproperties_pack::icoor;
			int na, pos, nav(1);
			num_from_name("  "+dna_word.substr(i,1), na );
			assert(na);

			// mutate primary strand
			pos = dna_pos_list[i] + nres_protein;
			pose.copy_sidechain( pos, na, nav, icoor(1,1,na,nav) );

			// mutate complementary strand
			na = na_partner(na);
			pos = partner[pos];
			pose.copy_sidechain( pos, na, nav, icoor(1,1,na,nav) );
		}


		pose.rottrial(w,1);

		pose.main_minimize(w,"dfpmin");
		start_pose.set_extra_score( "spec:"+dna_word, pose.get_0D_score( SCORE ));
	}


}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//

void
zf_linker_relax(
	pose_ns::Pose & pdb_pose,
	std::string const & run_tag
)
{
	using namespace pose_ns;

	float big_angle( 4.0 ), small_angle( 2.0 );

	int const linker_begin(28);
	int const linker_end  (31);

	Pose native_pose;
	native_pose = pdb_pose;
	pdb_pose.set_native_pose( native_pose );

	prof::reset( true );

	//////////////////
	// setup fold_tree
	int const nres( pdb_pose.total_residue() );
	int nres_protein(0);
	//int const dna_jump(1);
	{
		for ( int i=1; i<= nres; ++i ) {
			if ( pdb_pose.is_protein(i) ) nres_protein = i;
		}
		assert( nres_protein + 11 + 11 == nres );
		int const num_jump(2);
		FArray1D_int cuts(num_jump);
		FArray2D_int jumps(2,num_jump);
		jumps(1,1) = linker_end+1;
		jumps(2,1) = nres_protein+2; // dont want this to overlap spec region
		jumps(1,2) = nres_protein+2; // ditto
		jumps(2,2) = nres; // ditto
		cuts(1) = nres_protein;
		cuts(2) = nres_protein+11;
		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );
		f.reorder( nres_protein+2 );
		pdb_pose.set_fold_tree(f);
	}

	fix_phosphates( pdb_pose, 1.0 );

	native_pose = pdb_pose;

	FArray1D_bool interface( nres, false ), linker( nres, false );
	{ // identify interface residues
		using aaproperties_pack::natoms;
		for ( int i=1; i<= nres; ++i ) {
			if ( i>= linker_begin && i<= linker_end ) {
				linker(i) = true;
				interface(i) = true; // include in interface
			}
		}
		for ( int i=1; i<= linker_end /** only up to linker **/; ++i ) {
			if ( pdb_pose.is_protein( i ) ) {
				float min_d(1000.0);
				int const aa ( pdb_pose.res(i));
				int const aav( pdb_pose.res_variant(i));
				for ( int j=1; j<= nres; ++j ) {
					int const bb ( pdb_pose.res(j) );
					int const bbv( pdb_pose.res_variant(j) );
					if ( param_aa::is_DNA( bb ) || linker( j ) ) {
						for ( int ii=5 /*cb*/; ii<= natoms(aa,aav); ++ii ) {
							for ( int jj=1; jj<= natoms(bb,bbv); ++jj ) {
								min_d = std::min( distance( pdb_pose.full_coord(ii,i),
																						pdb_pose.full_coord(jj,j)), min_d);
							}
						}
					}
				}
				if ( min_d < 6.0 ) {
					std::cout << "interface rsd: " << i << ' ' << min_d << std::endl;
					interface(i) = true;
				}
			}
		}

		// exclude zn-coordinating residues
		interface(5 ) = false;
		interface(10) = false;
		interface(23) = false;
		interface(27) = false;

	} // find interface residues



	// scorefxn
	Score_weight_map w( score12 );

	minimize_set_tolerance( 0.001 );
	pose_setup_packer();
	pose_set_use_nblist( false ); // safety


	////////////////////////
	// minimize interface sc
	pdb_pose.set_allow_chi_move( interface );
	pdb_pose.set_allow_bb_move( false );
	pdb_pose.set_allow_jump_move( false );

	pdb_pose.main_minimize( w, "dfpmin" );
	calc_rms( pdb_pose );

	/////////////////////////////////
	// minimize linker + interface sc
	pdb_pose.set_allow_chi_move( interface );
	pdb_pose.set_allow_bb_move( linker );

	pdb_pose.main_minimize( w, "dfpmin" );
	calc_rms( pdb_pose );

	// minimize linker + interface sc + dna_jump
// 	pdb_pose.set_allow_chi_move( interface );
// 	pdb_pose.set_allow_bb_move( linker );
// 	pdb_pose.set_allow_jump_move( dna_jump, true );

// 	pdb_pose.main_minimize( w, "dfpmin" );
// 	calc_rms( pdb_pose );

	// decoy output
	silent_io::Decoy_out out( false /* silent_output */ );


	for ( int nn=1; nn<= 4000; ++nn ) {

		bool const perturb( nn%10 != 0 );
		bool const extra_relax(  perturb && nn%11 == 0 ||
														!perturb && nn%30 == 0 );

		std::string tag( "-zf-linker-relax_s="+run_tag+"_nn="+string_of(nn));
		if ( perturb ) tag += "_pert";
		if ( extra_relax ) tag += "_extra-rlx";

		if ( out.decoy_exists(tag) ) continue;

		// recover starting conformation
		Pose pose;
		pose = pdb_pose;


		// randomly perturb the first finger
		if ( perturb ) {

			for ( int i=linker_begin; i<= linker_end; ++i ) {
				pose.set_phi( i, pose.phi(i) + gaussian()*big_angle );
				pose.set_psi( i, pose.psi(i) + gaussian()*big_angle );
			}

		}

		// repack interface
		pose.repack( interface, !perturb );

		// minimize linker
		pose.set_allow_jump_move( false ); // stays fixed now
		pose.set_allow_bb_move( linker );
		pose.set_allow_chi_move( false );

		pose.main_minimize( w, "dfpmin" );

		// minimize everything
		pose.set_allow_bb_move( linker );
		pose.set_allow_chi_move( interface );

		pose.main_minimize( w, "dfpmin" );

		Pose start_pose;
		start_pose = pose;
		pose.set_start_pose( start_pose );

		calc_rms( pose );

		Monte_carlo mc( pose, w, 0.8 );

		// rottrial:
		{ // rottrial trial -- all interface positions
			pose.rottrial( w, 1 );
			pose.new_score_pose();
			pose.score(w);
			mc.boltzmann( pose, "Rottrial" );
		}

		int const inner_cycles( extra_relax ? 20 : 10 );
		int const outer_cycles( extra_relax ? 5  : 3 );


		for ( int n=1; n<= outer_cycles; ++n ) {
			for ( int m=1; m<= inner_cycles; ++m ) {
				// small moves
				for ( int i=linker_begin; i<= linker_end; ++i ) {
					pose.set_phi( i, pose.phi(i) + gaussian()*small_angle );
					pose.set_psi( i, pose.psi(i) + gaussian()*small_angle );
				}

				{ // try rotamer trialing
					pose.rottrial(w,1);
					pose.new_score_pose();
					pose.score(w);
				}

				pose.main_minimize(w, "dfpmin");
				mc.boltzmann( pose, "linker-bb-min" );

			}
			pose = mc.low_pose();
			prof::show();

		}

		mc.show_counters();


		// spec check shouldnt modify the pose structure
		// just stores some extra data in the pose extra_scores...
		// confirm settings for spec minimize
		pose.set_allow_chi_move( interface );
		pose.set_allow_bb_move ( linker );
		pose.set_allow_jump_move( false );
		specificity_check_zf( pose );

		// rms calcs
		calc_rms(pose);
		calc_rms_zf(pose);

		out.write( tag, pose );
		prof::show();

	} // nn -- decoy loop

	std::exit(0);

}

///////////////////////////////////////////////////////////////////////////////
//
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
//

void
zf_linker_rb_relax(
	pose_ns::Pose & pdb_pose
)
{
	using namespace pose_ns;

	// get interface residues
	int const linker_begin(28);
	int const linker_end  (31);

	int const jump1_pos1( 24 );

	// params
	float const trans_delta( 0.2 );
	float const rot_delta( 5 );
	float const trans_delta_large( 1.0 );
	float const rot_delta_large( 10 );

	Pose native_pose;
	native_pose = pdb_pose;
	pdb_pose.set_native_pose( native_pose );

	prof::reset( true );

	//pdb_pose.dump_pdb("start.pdb");


	// setup fold_tree
	int const nres( pdb_pose.total_residue() );
	int nres_protein(0);
	int const dna_jump(1);
	{
		for ( int i=1; i<= nres; ++i ) {
			if ( pdb_pose.is_protein(i) ) nres_protein = i;
		}
		assert( nres_protein + 11 + 11 == nres );
		int const num_jump(3);
		FArray1D_int cuts(num_jump);
		FArray2D_int jumps(2,num_jump);
		jumps(1,1) = jump1_pos1;
		jumps(2,1) = nres_protein+2;
		jumps(1,2) = linker_end+1;
		jumps(2,2) = nres_protein+2;
		jumps(1,3) = nres_protein+2;
		jumps(2,3) = nres_protein+13;
		cuts(1) = linker_end;
		cuts(2) = nres_protein;
		cuts(3) = nres_protein+11;
		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );
		f.reorder( nres_protein+2 );
		pdb_pose.set_fold_tree(f);
	}

	fix_phosphates( pdb_pose, 1.0 );
	//pdb_pose.dump_pdb("fixed.pdb");
	native_pose = pdb_pose;


	FArray1D_bool interface( nres, false ), linker( nres, false );
	{ // identify interface residues
		using aaproperties_pack::natoms;
		for ( int i=1; i<= nres; ++i ) {
			if ( i>= linker_begin && i<= linker_end ) {
				linker(i) = true;
				interface(i) = true; // include in interface
			}
		}
		for ( int i=1; i<= linker_end /** only up to linker **/; ++i ) {
			if ( pdb_pose.is_protein( i ) ) {
				float min_d(1000.0);
				int const aa ( pdb_pose.res(i));
				int const aav( pdb_pose.res_variant(i));
				for ( int j=1; j<= nres; ++j ) {
					int const bb ( pdb_pose.res(j) );
					int const bbv( pdb_pose.res_variant(j) );
					if ( param_aa::is_DNA( bb ) || linker( j ) ) {
						for ( int ii=5 /*cb*/; ii<= natoms(aa,aav); ++ii ) {
							for ( int jj=1; jj<= natoms(bb,bbv); ++jj ) {
								min_d = std::min( distance( pdb_pose.full_coord(ii,i),
																						pdb_pose.full_coord(jj,j)), min_d);
							}
						}
					}
				}
				if ( min_d < 6.0 ) {
					std::cout << "interface rsd: " << i << ' ' << min_d << std::endl;
					interface(i) = true;
				}
			}
		}

		// exclude zn-coordinating residues
		interface(5 ) = false;
		interface(10) = false;
		interface(23) = false;
		interface(27) = false;

	} // find interface residues



	// scorefxn
	Score_weight_map w( score12 );
	w.set_weight( CHAINBREAK_CST, 0.2 );
	cst_set_ns::Cst_set cst_set;
	pdb_pose.set_constraints( cst_set );
	cst_set.add_chainbreak( linker_end, 1.0 );


	minimize_set_tolerance( 0.001 );
	pose_setup_packer();
	pose_set_use_nblist( false ); // safety


	// minimize interface sc
	pdb_pose.set_allow_chi_move( interface );
	pdb_pose.set_allow_bb_move( false );
	pdb_pose.set_allow_jump_move( false );

	pdb_pose.main_minimize( w, "dfpmin" );
	calc_rms( pdb_pose );

	// minimize linker + interface sc + dna_jump
	pdb_pose.set_allow_chi_move( interface );
	pdb_pose.set_allow_bb_move( linker );
	pdb_pose.set_allow_jump_move( dna_jump, true );

	pdb_pose.main_minimize( w, "dfpmin" );
	calc_rms( pdb_pose );

	// decoy output
	silent_io::Decoy_out out( false /* silent_output */ );


	for ( int nn=1; nn<= 4000; ++nn ) {

		std::string const tag( "_zf_linker_rb_"+string_of( nn ) );

		if ( out.decoy_exists(tag)) continue;

		// recover starting conformation
		Pose pose;
		pose = pdb_pose;

		bool const perturb( nn%10 != 1 );

		// randomly perturb the first finger
		float fdev,bdev,tor_delta,rama_delta;
		if ( perturb ) {
			perturb_dna_jump( pose, dna_jump, trans_delta_large, rot_delta_large );
			fast_ccd_loop_closure( pose, linker_begin, linker_end, linker_end,
														 100 /*ccd_cycles*/, 0.01, true, 2.0, 10, 30, 90,
														 fdev, bdev,tor_delta, rama_delta );
		}

		// repack interface
		pose.repack( interface, !perturb /* only include for 1st decoy */ );


		// minimize linker
		pose.set_allow_jump_move( false );
		pose.set_allow_bb_move( linker );
		pose.set_allow_chi_move( false );

		pose.main_minimize( w, "dfpmin" );

		// minimize everything
		pose.set_allow_jump_move( dna_jump, true );
		pose.set_allow_bb_move( linker );
		pose.set_allow_chi_move( interface );

		pose.main_minimize( w, "dfpmin" );

		Pose start_pose;
		start_pose = pose;
		pose.set_start_pose( start_pose );

		calc_rms( pose );
		//pose.dump_pdb( "start."+string_of(nn)+".pdb");


		Monte_carlo mc( pose, w, 0.8 );

		// rottrial:
		{ // rottrial trial -- all interface positions
			pose.rottrial( w, 1 );
			pose.new_score_pose();
			pose.score(w);
			mc.boltzmann( pose, "Rottrial" );
		}



		for ( int n=1; n<= 3; ++n ) {
			for ( int m=1; m<= 10; ++m ) {
				// small moves
				perturb_dna_jump( pose, dna_jump, trans_delta, rot_delta );
				fast_ccd_loop_closure( pose, linker_begin, linker_end, linker_end,
															 50 /*ccd_cycles*/, 0.01, true, 2.0, 10, 30, 90,
															 fdev, bdev,tor_delta, rama_delta );

				pose.main_minimize(w, "dfpmin");

				Pose save_pose;
				save_pose = pose;

				{ // try rotamer trialing
					float const old_score( save_pose.get_0D_score( SCORE ) );
					pose.rottrial(w,1);
					float const rottrial_score( pose.get_0D_score( SCORE ) );
					pose.new_score_pose();
					pose.score(w);
					float const new_score( pose.get_0D_score( SCORE ) );
					if ( new_score > old_score ) {
						std::cout << "rottrial BAD: " << old_score << ' ' <<
							rottrial_score << ' ' << new_score << ' ' <<
							new_score - rottrial_score << std::endl;
						pose = save_pose;
					}
				}

				// accept?
				mc.boltzmann( pose, "linker-bb-min" );

				calc_rms( pose );
				std::cout << pose.show_scores() << std::endl;
			}
			pose = mc.low_pose();
			prof::show();

		}

		mc.show_counters();

		calc_rms(pose);
		calc_rms_zf(pose);
		out.write( tag, pose );
		prof::show();

	} // nn -- decoy loop

	std::exit(0);

}

///////////////////////////////////////////////////////////////////////////////
//

void
zf_test()
{
	pose_ns::Pose pdb_pose;
	std::string pdb( stringafteroption("s"));
	pose_from_pdb( pdb_pose, pdb, true, false, true );

	zf_linker_relax( pdb_pose, pdb.substr(0,pdb.size()-4));
// 	zf_linker_relax_mutate( pdb_pose );
}


// void
// zf_test()
// {
// 	// read and process each of the zf structures

// 	std::ifstream data( stringafteroption("l").c_str() );

// 	std::string filename;
// 	while ( getline( data, filename ) ) {
// 		pose_ns::Pose pdb_pose;
// 		bool const ok( pose_from_pdb( pdb_pose, filename, true, false, true ) );
// 		if ( !ok ) {
// 			std::cout << "WARNING:: failure to read pdb! " << filename << std::endl;
// 			continue;
// 		}

// 		//zf_linker_rb_relax( pdb_pose );
// // 		zf_linker_relax( pdb_pose );
//  		zf_linker_relax_mutate( pdb_pose );

// 	}
// }

///////////////////////////////////////////////////////////////////////////////
void
set_basepairs_from_pair_vector(
	pose_ns::Pose & pose,
	DnaSeqInfo & pairs
)
{
	for ( std::vector< DnaPosInfo >::const_iterator bp( pairs.begin() );
	      bp != pairs.end(); ++bp ) {
		pose.set_basepair( bp->fwdpos(), bp->rvspos() );
		pose.set_basepair( bp->rvspos(), bp->fwdpos() );
	}
}

///////////////////////////////////////////////////////////////////////////////
void
set_basepairs(
	pose_ns::Pose & pose
)
{
	std::map< int, int > base_partner;
	DnaSeqInfo base_pairs;
	find_basepairs( pose, base_pairs );
	set_basepairs_from_pair_vector( pose, base_pairs );
}

///////////////////////////////////////////////////////////////////////////////
//
void
analyze_dna()
{
	using namespace pose_ns;

	std::vector< std::string > files;
	if ( truefalseoption("s") ) {
		files.push_back( stringafteroption("s"));
	} else {
		std::ifstream data( stringafteroption("l").c_str());
		std::string line;
		while ( getline( data,line ) ) {
			files.push_back(line);
		}
		data.close();
	}

	Pose native_pose;
	if ( truefalseoption("n") ) {
		bool ok = pose_from_pdb( native_pose, stringafteroption("n"), true, false,
														 true );
		if ( !ok ) {
			std::cout << "native input failed!!!" << std::endl;
			std::exit( EXIT_FAILURE );
		}
	}

	Smap smap;

	for ( int i=0; i< int(files.size()); ++i ) {
		Pose pose;
		bool ok = pose_from_pdb( pose, files[i], true, false, true );
		if ( !ok ) {
			std::cout << "input failed: " << files[i] << std::endl;
			continue;
		}

		FArray1D_bool exclude_none( pose.total_residue(), false );
		setup_simple_dna_fold_tree( pose, exclude_none );

		Score_weight_map w( score12 );

		{ // spec check
			specificity_check_zf( pose );
			std::cout << files[i] << ' ' << pose.show_scores() << std::endl;
			continue;
		}


		{ // get scores
			store_scores( pose, w, smap );
			continue;

		}




		{ // look at hbonds
			pose.score(w);
			pose.dump_pdb( files[i]+".new_pdb");
			continue;
		}


		{ // look at gb

			pose.score(w);

			pose_io::bval_src = pose_io::BORN_RADIUS;
			pose.dump_pdb( files[i]+".born_rad.pdb" );

			pose_io::bval_src = pose_io::CHARGE;
			pose.dump_pdb( files[i]+".charge.pdb" );

			continue;
		}




		if ( native_pose.total_residue() ) {
			pose.set_native_pose( native_pose );
			calc_rms_zf( pose );
			calc_rms( pose );

			std::ifstream data(files[i].c_str() );
			std::string info,line;
			while ( getline(data,line) ) {
				if ( line.substr(0,12) == "REMARK SCORE" ) {
					info = line.substr(7);
					break;
				}
			}
			data.close();

			std::cout << files[i] << ' ' << info << ' ' << pose.show_scores() <<
				std::endl;
			continue;
		}

		pose.score(w);

		mutate_test( pose );

		show_clashes( pose, 1.0 );


		fix_phosphates( pose, 1.0 );
	}
	exit(0);

}

///////////////////////////////////////////////////////////////////////////////
#ifdef DOH
void
justin_test()
{
	using namespace pose_ns;

	Pose big_pose;

	// get protein and dna into big pose


	Pose protein_pose;
	protein_pose.simple_fold_tree( nres_protein );
	protein_pose.copy_segment( nres_protein, big_pose, 1, 1 );

	Pose dna_pose;
	dna_pose.simple_fold_tree( nres_dna );
	dna_pose.copy_segment( nres_dna, big_pose, 1, nres_protein+1 );
	//dna_pose.setup_atom_tree(); // really should use atom_trees for DNA

}
#endif


///////////////////////////////////////////////////////////////////////////////
void
dna_info(
	pose_ns::Pose & pose,
	std::string const & tag
)
{
	using namespace pose_ns;
	using namespace kin;
	using namespace param_aa;


	///////////////////////
	// read in starting pdb
	int const nres( pose.total_residue() );
	FArray1D_int const & res( pose.res() );

	/////////////////////////////////////////
	// now try to setup the correct fold_tree
	Fold_tree f;
	std::map< int, int > base_partner;
	int dna_jump(0);
	FArray1D_bool protein_interface( nres, false );
	{ // scope

		// identify basepairs
		DnaSeqInfo base_pairs;
		find_basepairs( pose, base_pairs );

		// identify chain breaks
		int ncuts;
		FArray1D_int cuts(nres);
		find_chainbreaks( nres, res, pose.full_coord(), ncuts, cuts );

		// first determine closest approach between protein and dna backbones
		int protein_anchor(0);
		int dna_anchor(0);
		{
			int const CA(2), P(dna_variables::p);
			float min_d(1000.0);
			for ( int i=1; i<= nres; ++i ) {
				if ( !is_protein(res(i) ) ) continue;
				for ( int j=1; j<= nres; ++j ) {
					if ( !is_DNA( res(j) ) || !base_partner[j] ) continue;
					float const d( distance( Vec( &pose.full_coord()(1,CA,i)),
																	 Vec( &pose.full_coord()(1,P,j)) ) );
					if ( d<min_d ) {
						min_d = d;
						protein_anchor = i;
						dna_anchor = j;
					}

					if ( d<12.0 ) {
						protein_interface( i ) = true;
					}

				}
			}
		}

		int const dna_anchor_partner( base_partner[ dna_anchor ] );
		std::cout << "dna_anchor: " << tag << ' ' << protein_anchor << ' ' <<
			dna_anchor << ' ' << dna_anchor_partner << std::endl;

		FArray2D_int jump_point( 2, ncuts );
		int njump(0);
		for ( int i=1; i<= ncuts+1; ++i ) {
			int const seg_begin( i==1       ? 1    : cuts(i-1)+1 );
			int const seg_end  ( i==ncuts+1 ? nres : cuts(i) );
			std::cout << "segs: " << i << ' ' << res(seg_begin) << ' ' <<
				seg_begin << ' ' << seg_end << std::endl;
			// couple of cases
			// 1 -- DNA or protein segment not containing anchor residue
			// 2 -- DNA or protein segment containing anchor residue
			int const aa( res(seg_begin ) );
			if ( is_protein( aa ) ) {
				if ( seg_begin > protein_anchor || seg_end < protein_anchor ) {
					// add a jump to protein_anchor
					// could pick a better jump_pos but we aren't doing protein
					// internal minimization yet
					jump_point(1,++njump) = std::min( seg_begin, protein_anchor );
					jump_point(2,  njump) = std::max( seg_begin, protein_anchor );
				}
			} else if ( is_DNA( aa ) ) {
				if ( seg_begin <= dna_anchor && seg_end >= dna_anchor ) {
					// add a jump to protein_anchor
					jump_point(1,++njump) = std::min( dna_anchor, protein_anchor );
					jump_point(2,  njump) = std::max( dna_anchor, protein_anchor );
					dna_jump = njump;
				} else if ( seg_begin <= dna_anchor_partner &&
										seg_end >= dna_anchor_partner ) {
					// add a jump to dna_anchor
					jump_point(1,++njump) = std::min( dna_anchor_partner, dna_anchor );
					jump_point(2,  njump) = std::max( dna_anchor_partner, dna_anchor );
				} else {
					std::cout << "TOO MUCH DNA? " << tag << std::endl;
					int const seg_anchor( ( seg_begin + seg_end ) / 2 );
					jump_point(1,++njump) = std::min( seg_anchor, dna_anchor );
					jump_point(2,  njump) = std::max( seg_anchor, dna_anchor );
					//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			} else {
				std::cout << "BAD AA: " << aa << ' ' << tag << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
		assert( njump == ncuts );


		//if( jump_point(1, 2) == 14 ) jump_point(1, 2) = 21;


		f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );
		f.reorder( protein_anchor ); // guarantee some c2n folding
	} // scope
	pose.set_fold_tree( f );


	//pose.atom_tree()->root()->show();

	pose.dump_pdb(tag+"_test2.pdb");


	// look at basepair params
	{
		FArray1D_float params(6);
		FArray3D_float const & xyz( pose.full_coord() );
		for ( int i=1; i<= nres; ++i ) {
			int const j( base_partner[i] );
			if ( j > i ) {
				base_pair_params( pose.res(i), pose.res(j), xyz(1,1,i), xyz(1,1,j),
													params );
				float const score( base_pair_score( pose.res(i), params ) );
				std::cout << "base_pair_score: " << I(4,i) << F(9,3,score) <<std::endl;
			}
		}
		for ( int i=1; i<= nres-1; ++i ) {
			if ( base_partner[i] && base_partner[i+1] &&
					 base_partner[i] != i+1 &&
					 base_partner[i+1] == base_partner[i]-1 ) {

				new_base_step_params( pose.res(i), pose.res(i+1), xyz(1,1,i),
															xyz(1,1,i+1), params );

				float const score( new_base_step_score( pose.res(i), pose.res(i+1),
																								params ) );
				std::cout << "new_base_step_score: " << I(4,i) << F(9,3,score) <<
					std::endl;
			}
		}
		return;
	}


	// try copying poses
	{
		Pose new_pose;
		new_pose = pose;

		new_pose.dump_pdb(tag+"_copy.pdb");
	}

	{
		// try copying sidechains
		int const src_pos( 34 ), dest_pos( 36 );

		pose.copy_sidechain( dest_pos, pose.res( src_pos ),
			pose.res_variant( src_pos ), pose.full_coord()(1,1,src_pos));
		pose.dump_pdb(tag+"_new_sc_36_from_34.pdb");
	}


	exit(0);

	if ( true ) { // look at torsion angles
		Pose protein_pose;
		pose_from_pdb( protein_pose, tag, true, false, false, 'A' );

		FArray2D_float dna_dihedrals( 7, nres );
		calc_dna_dihedrals( nres, pose.res(), pose.full_coord(), dna_dihedrals);

		for ( int i=1; i<= nres; ++i ) {
			int const aa ( pose.res        (i) );
			int const aav( pose.res_variant(i) );
			int max_tor(0);
			if ( is_protein( aa ) ) {
				max_tor = 3 + aaproperties_pack::nchi( aa,aav );
			} else {
				assert( is_DNA( aa ) );
				max_tor = 7;
			}
			std::cout << "pose_torsions: " << i << ' ' << aa_name3(aa);
			for ( int tor=1; tor<= max_tor; ++tor ) {
				std::cout << F(9,3, pose.get_torsion_by_number( i,tor ) );
			}
			std::cout << std::endl;
			if ( is_protein(aa) ) {
				assert( i <= protein_pose.total_residue() && aa ==protein_pose.res(i));
				for ( int tor=1; tor<= max_tor; ++tor ) {
					std::cout << "protein-torsion-dev: " <<
						I(5,i) << ' ' << aa_name3(aa) << ' ' << tor << ' ' <<
						F(9,3, pose.get_torsion_by_number( i,tor ) ) <<
						F(9,3, protein_pose.get_torsion_by_number( i,tor ) ) << std::endl;
				}
			}
			if ( is_DNA(aa) ) {
				for ( int tor=1; tor<= max_tor; ++tor ) {
					std::cout << "dna-torsion-dev: " <<
						I(5,i) << ' ' << aa_name3(aa) << ' ' << tor << ' ' <<
						F(9,3, pose.get_torsion_by_number( i,tor ) ) <<
						F(9,3, dna_dihedrals( tor, i ) ) << std::endl;
				}
			}
		}
	}

	//exit(0);

	if ( false ) { // look at torsion angles, restrict to internal paired rsds
		for ( int i=2; i<= nres-1; ++i ) {
			if ( !is_DNA( pose.res( i-1 ) ) || !base_partner[ i-1 ] ||
					 !is_DNA( pose.res( i   ) ) || !base_partner[  i  ] ||
					 !is_DNA( pose.res( i+1 ) ) || !base_partner[ i+1 ] ) continue;

			std::cout << "dna_torsions: " << tag << ' ' << i << ' ' <<
				aa_name3( pose.res(i) ) << ' ';
			for ( int j=1; j<= 7; ++j ) {
				std::cout << F(9,3,pose.get_torsion_by_number( i,j ));
			}
			std::cout << std::endl;
		}
	}


	{

		// try scoring the complex, minimizing
		Score_weight_map weight_map( score12 );
		//Score_weight_map weight_map;
// 		weight_map.set_weight( FA_ATR, 1.0 );
// 		weight_map.set_weight( FA_REP, 1.0 );
		//weight_map.set_weight( FA_DUN, 1.0 );


		pose.score( weight_map );

		// minimize protein sidechains
		pose_set_use_nblist( true );
		minimize_set_tolerance( 1e-6 );
		pose_setup_packer();



		pose.set_allow_move( ALL, false );


		{
			pose.set_allow_jump_move( true );
			pose.main_minimize( weight_map, "dfpmin" );

			exit(0);
		}



		//pose.set_allow_chi_move( true );
 		pose.set_allow_chi_move( protein_interface );



		if ( false ) {
			pose.repack( protein_interface, true /*include_current*/ );
			std::cout << pose.show_scores() << std::endl;
			pose.dump_pdb(tag+"repack2_inc_current.pdb");
			exit(0);
		}

		if ( false ) {
			// minimize just gb!
			Score_weight_map gb_weight_map;
			gb_weight_map.set_weight( GB, 1.0 );
			pose.score( gb_weight_map );
			std::cout << pose.show_scores() << std::endl;


			pose.set_allow_move( ALL, false );
			pose.set_allow_jump_move( true );
			pose.main_minimize( weight_map, "dfpmin" );
			std::cout << pose.show_scores() << std::endl;

			pose.main_minimize( gb_weight_map, "dfpmin" );
			std::cout << pose.show_scores() << std::endl;
			exit(0);
		}

		pose.main_minimize( weight_map, "dfpmin" );
		pose.dump_pdb(tag+"chi_min.pdb");
		exit(0);

		// repack interface sidechains
// 		FArray1D_bool allow_repack( nres, false );
// 		for ( int i=1; i<= nres; ++i ) {
// 			if ( pose.is_protein(i) ) allow_repack(i) = true;
// 		}
		pose.repack( protein_interface, true /*include_current*/ );
// 		pose.repack( allow_repack, true /*include_current*/ );
		std::cout << pose.show_scores() << std::endl;
		pose.dump_pdb(tag+"repack.pdb");


		// minimize docking arrangement
		pose.set_allow_chi_move( false );
		pose.set_allow_jump_move( dna_jump, true );
		pose.main_minimize( weight_map, "dfpmin" );
		pose.dump_pdb(tag+"jump_min.pdb");

		// repack no input sc
		if ( true ) {
			pose.repack( protein_interface, false /*include_current*/ );
			std::cout << pose.show_scores() << std::endl;
			pose.dump_pdb(tag+"repack3.pdb");
		}
	}


}

///////////////////////////////////////////////////////////////////////////////
int
get_dna_bp_atom(
								int const aa,
								int const aav
								)
{
	int const atomno( LookupByName( aa, aav, " C5 " ) );
	if ( atomno <1 || atomno > param::MAX_ATOM()() ) {
		std::cout << "Unable to find C5 atom!" << std::endl;
		std::exit( EXIT_FAILURE );
	}
	return atomno;
}



void
calc_single_dna_bs_deriv(
	int const rsd1,
	pose_ns::Pose const & pose,
	numeric::xyzVector_float & F1,
	numeric::xyzVector_float & F2,
	float const external_sign_factor // should probably be +1 or -1
	)
{
	using namespace param_aa;
	using namespace numeric;
	using numeric::conversions::degrees;
	using numeric::conversions::radians;
	using namespace new_bs_info;

	int const rsd2( rsd1+1 );
	assert( is_DNA( pose.res(rsd2) ) );

	int const aa1( pose.res(rsd1) );
	int const aa2( pose.res(rsd2) );

	// get coord slices from the pose
	FArray2Dp_float base1_coords
		( pose.full_coord()(1,1,rsd1), 3, param::MAX_ATOM()() );
	FArray2Dp_float base2_coords
		( pose.full_coord()(1,1,rsd2), 3, param::MAX_ATOM()() );

	kin::Stub const
		stub1( get_base_stub( aa1, 1, base1_coords ) ), // strand = 1
		stub2( get_base_stub( aa2, 1, base2_coords ) ); // strand = 1

	// copy matrices
	xyzMatrix_float M1( stub1.M ), M2( stub2.M );

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );

	if ( dot( M1.col_z(), M2.col_z() ) < 0.0 ) {
		std::cout << "dna_bs_deriv: base flip!" << std::endl;
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// get angle between the z-axes
	float const gamma( arccos( dot( M1.col_z(), M2.col_z() ) ) );

	xyzVector_float const rt( ( cross( M2.col_z(), M1.col_z() ) ).normalized() );

	xyzMatrix_float R_gamma_2( rotation_matrix( rt, gamma/2.0f ) );

	M2 = R_gamma_2 * M2;
	M1 = R_gamma_2.transposed() * M1;

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );

	// build mid-base-pair triad
	assert( distance( M1.col_z(), M2.col_z() ) < 1e-3 );
	assert( std::abs( dot( rt, M1.col_z() ) ) < 1e-3 );

	xyzMatrix_float MBT;
	MBT.col_z( M1.col_z() );

	assert( std::abs( dot( M1.col_x(), MBT.col_z() ) ) < 1e-3 );
	assert( std::abs( dot( M2.col_x(), MBT.col_z() ) ) < 1e-3 );
	assert( std::abs( dot( M1.col_y(), MBT.col_z() ) ) < 1e-3 );
	assert( std::abs( dot( M2.col_y(), MBT.col_z() ) ) < 1e-3 );

	// get
	MBT.col_y( ( 0.5f * ( M1.col_y() + M2.col_y() ) ).normalized() );
	MBT.col_x( ( 0.5f * ( M1.col_x() + M2.col_x() ) ).normalized() );

	assert( is_orthonormal( MBT, 1e-3 ) );

	// angular params

	// TWIST
	// x,y,z make rh coord system
	FArray1D_float params(6);

	params(1) = std::atan2( dot( M1.col_x(), M2.col_y() ),
													dot( M1.col_x(), M2.col_x() ) );
	// ROLL:
	params(2) = gamma * dot( rt, MBT.col_y() );

	// TILT
	params(3) = gamma * dot( rt, MBT.col_x() );

	// translational params
	xyzVector_float const displacement( stub1.v - stub2.v );

	params(4) = dot( displacement, MBT.col_x() ); // SHIFT
	params(5) = dot( displacement, MBT.col_y() ); // SLIDE
	params(6) = dot( displacement, MBT.col_z() ); // RISE

	// check sign conventions
	assert( params(1) * dot( MBT.col_z(), cross( M2.col_y(), M1.col_y() ) ) > 0);

	// convert to degrees
	float const twist( params(1) );
	float const roll ( params(2) );
	float const tilt ( params(3) );

	params(1) = degrees( params(1) );
	params(2) = degrees( params(2) );
	params(3) = degrees( params(3) );


	load_dna_score_tables();

	params.dimension(6);

	// max allowed deviation from average beyond which the score is capped
	// PB -- this seems very large! it's in standard deviations, right?
	float const MAX_SCORE_DEV = 100.0;

	//Make a label for the basestep type lookup:
	int const ind = get_basestep_index( aa_name3(aa1).substr(2),
																			aa_name3(aa2).substr(2) );

	FArray1D_float dE_dp(6,0.0);

	bool out_of_bounds( false );

	for ( int i = 1; i <= 6; ++i ) {
		float const max_delta_i = MAX_SCORE_DEV * new_bs_stddev(i,ind);

		float delta_i = params(i) - new_bs_mean(i,ind);
		// reset outliers to max allowed values
		// (beyond which the potential is unreliable)
		if ( delta_i > max_delta_i ) {
			out_of_bounds = true;
			delta_i = max_delta_i;
		}
		if ( delta_i < -max_delta_i ) {
			out_of_bounds = true;
			delta_i = -max_delta_i;
		}

		for ( int j = 1; j <= 6; ++j ) {
			float const max_delta_j = MAX_SCORE_DEV * new_bs_stddev(j,ind);

			float delta_j = params(j) - new_bs_mean(j,ind);
			// reset outliers to max allowed values
			// (beyond which the potential is unreliable)
			if ( delta_j > max_delta_j ) {
				out_of_bounds = true;
				delta_j = max_delta_j;
			}
			if ( delta_j < -max_delta_j ) {
				delta_j = -max_delta_j;
				out_of_bounds = true;
			}

			dE_dp(i) += new_bs_Fij(i,j,ind) * delta_j;
			dE_dp(j) += new_bs_Fij(i,j,ind) * delta_i;
		}
	}

	if ( debug_bs_deriv ) {
		dE_dp = 0.0;
		for ( int i=1; i<= 6; ++i ) {
			if ( i == debug_bs_deriv || debug_bs_deriv > 6 ) {
				dE_dp(i) = 2 * ( params( i ) - new_bs_mean( i, ind ) );
			}
		}
	}

	if ( out_of_bounds ) {
		// this is not really a big deal
		std::cout << "WARNING:: out_of_bounds in dna base-pair derivative!" <<
			std::endl;
	}


	// adjust for the external sign factor:
	for ( int i=1; i<= 6; ++i ) dE_dp(i) *= external_sign_factor;


	//////////////////////////
	// twist deriv
	//////////////////////////
	//
	// let xi be the original x-axis vector in triad i
	// xi' the vector transformed so that the z-axes coincide
	//
	// cos( twist ) = dot( x1' , x2' )
	//              = dot( x1, R_gamma( x2 ) ) // apply R_gamma/2 to both sides
	//
	// ASSUMPTION: R_gamma doesn't depend on the torsion angle, which it
	//  does! but I think the primary contribution is still captured here.
	//
	// d/dphi( dot( x1, R_gamma(x2) )) = dot( cross(u_phi,x1), R_gamma(x2) )
	// = dot( -u_phi , cross( R_gamma(x2), x1 )

	float const rad2deg( degrees(1.0));

	{
		float const dE_dtwist( dE_dp(1) );

		// these are for preventing d_arccos from blowing up
		static float const small_angle( radians( 0.1f ) );
		static float const big_angle( radians( 179.9f ) );
		static float const max_c( std::cos( small_angle ));
		static float const min_c( std::cos( big_angle ));

		Vec x1( stub1.M.col_x() );
		Vec R_gamma_x2( rotation_matrix( rt, gamma) * Vec(stub2.M.col_x()) );

		assert( distance( Vec(stub1.M.col_z()),
											rotation_matrix(rt,gamma) * Vec(stub2.M.col_z())) <1e-2);

		float c( std::cos( twist ) );
		assert( std::abs( c - dot( x1,R_gamma_x2 ) ) < 1e-2 );
		c = std::min( std::max( min_c, c ), max_c );

		float const dtwist_dc = -1 / std::sqrt( 1 - c * c );

		// since the derivatives factor through cosine we are getting
		// the derivative of an unsigned angle

		int const sign_factor( twist < 0 ? -1 : 1 );

		if ( verbose ) std::cout << "dE_dtwist: " <<
										 F(9,3, dE_dtwist ) <<
										 F(9,3, dtwist_dc ) <<
										 F(9,3, cross(R_gamma_x2,x1).length() ) << std::endl;

		F1 += sign_factor * dE_dtwist * rad2deg * dtwist_dc*
			cross( R_gamma_x2, x1 );
	}

	///////////////////
	// roll + tilt
	///////////////////
	//
	// as confirmed in the base_pair_params routine,
	// if we let tilt' = sin(gamma) * tilt / gamma,
	// then tilt' = dot( cross( z2, z1 ), x_MBT )
	//
	// likewise roll' = dot( cross( z2, z1 ), y_MBT )
	//
	// but how do x_MBT and y_MBT vary as we rotate about the phi axis??
	//
	// since these are approximately averages between a moving vector and
	// a fixed vector, we can assume:
	//
	// d/dphi( x_MBT ) = cross( u_phi, x_MBT ) / 2.0
	//
	// have to see if this actually works
	//
	// in addition we can assume that sin(gamma)/gamma is constant
	// since gamma is small
	//

	{
		float const dE_droll( dE_dp(2) );
		float const dE_dtilt( dE_dp(3) );

		Vec const z1( stub1.M.col_z() ), z2( stub2.M.col_z() );

		float const gamma_sin_gamma( gamma / std::sin( gamma ) );

		assert( std::abs( roll - gamma_sin_gamma *
											dot( cross( z2, z1 ), MBT.col_y() ) ) < 1e-2 );

		assert( std::abs( tilt - gamma_sin_gamma *
											dot( cross( z2, z1 ), MBT.col_x() ) ) < 1e-2 );

		F1 += dE_dtilt * gamma_sin_gamma * rad2deg *
			( cross( z1, cross( z2, MBT.col_x() ) ) +        // 1st term: dz1/dphi
				0.5f * cross( MBT.col_x(), cross( z1, z2 ) ) );// 2nd term: dx_MBT/dphi

		F1 += dE_droll * gamma_sin_gamma * rad2deg *
			( cross( z1, cross( z2, MBT.col_y() ) ) +
				0.5f * cross( MBT.col_y(), cross( z1, z2 ) ) );

	}

	///////////////////////
	// translational derivs
	///////////////////////
	//
	// need d/dphi( dot( M-F, x_MBT ) ) where M is rotated, F is fixed,
	// and x_MBT is varying in some complicated way
	//
	// from cst_set.cc, comments to Angle_cst::helper(...)
	// the term w x_MBT constant contributes x_MBT to F2
	// and cross( x_MBT, M) to F1
	//
	// for the other term we use d/dphi( x_MBT ) = 0.5 * cross( u_phi, X_MBT)
	// and the standard vector rearrangement
	{
		Vec M( stub1.v ), F( stub2.v );

		F1 += dE_dp(4) * ( cross( MBT.col_x(), M - 0.5f * ( M - F ) ) );
		F2 += dE_dp(4) * ( MBT.col_x() );

		F1 += dE_dp(5) * ( cross( MBT.col_y(), M - 0.5f * ( M - F ) ) );
		F2 += dE_dp(5) * ( MBT.col_y() );

		F1 += dE_dp(6) * ( cross( MBT.col_z(), M - 0.5f * ( M - F ) ) );
		F2 += dE_dp(6) * ( MBT.col_z() );

	}


}


///////////////////////////////////////////////////////////////////////////////
//
// this is a little tricky:
//
// we may have
//
void
calc_new_dna_bs_deriv(
	int const rsd,
	pose_ns::Pose const & pose,
	numeric::xyzVector_float & F1,
	numeric::xyzVector_float & F2
)
{
	using namespace param_aa;
	using namespace numeric;
	using numeric::conversions::degrees;

	if ( count_pair_bs( rsd, pose ) ) {
		// the rsd->rsd+1 base_step

		calc_single_dna_bs_deriv( rsd, pose, F1, F2,  1.0 /* sign-factor */ );

	}
	if ( count_pair_bs( rsd-1, pose ) ) {
		// the rsd-1 -> rsd base_step

		calc_single_dna_bs_deriv( rsd-1, pose, F1, F2, -1.0 /* sign-factor */ );

	}

}

///////////////////////////////////////////////////////////////////////////////
void
calc_dna_bp_deriv(
	int const seqpos,
	pose_ns::Pose const & pose,
	numeric::xyzVector_float & F1,
	numeric::xyzVector_float & F2
)
{
	using namespace param_aa;
	using namespace numeric;
	using numeric::conversions::degrees;
	using numeric::conversions::radians;
	using namespace dna_variables;


	// need for derivs
	load_dna_score_tables();

	if ( !count_pair_bp( seqpos, pose ) ) return;

	int const partner( pose.basepair_partner( seqpos ) );

	int rsd1, rsd2, sign_factor;
	if ( partner > seqpos ) {
		rsd1 = seqpos;
		rsd2 = partner;
		sign_factor = 1;
	} else {
		rsd1 = partner;
		rsd2 = seqpos;
		sign_factor = -1;
	}

	int const aa1( pose.res( rsd1 ) );
	int const aa2( pose.res( rsd2 ) );

	FArray1D_float params(6,0.0);
	float const MAX_SCORE_DEV = 100.0; // should be same as in dna_bp_score

	// get coord slices from the pose
	FArray2Dp_float base1_coords
		( pose.full_coord()(1,1,rsd1), 3, param::MAX_ATOM()() );
	FArray2Dp_float base2_coords
		( pose.full_coord()(1,1,rsd2), 3, param::MAX_ATOM()() );


	kin::Stub const
		stub1( get_base_stub( aa1, 1, base1_coords ) ), // strand = 1
		stub2( get_base_stub( aa2, 2, base2_coords ) ); // strand = 2

	// copy matrices
	xyzMatrix_float M1( stub1.M ), M2( stub2.M );

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );

	if ( dot( M1.col_z(), M2.col_z() ) < 0.0 ) {
		std::cout << "dna_bp_deriv: base flip!" << std::endl;
		//utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// get angle between the y-axes
	float const gamma( arccos( dot( M1.col_y(), M2.col_y() ) ) );

	xyzVector_float const bo( ( cross( M2.col_y(), M1.col_y() ) ).normalized() );

	xyzMatrix_float R_gamma_2( rotation_matrix( bo, gamma/2.0f ) );

	M2 = R_gamma_2 * M2;
	M1 = R_gamma_2.transposed() * M1;

	assert( is_orthonormal( M1, 1e-3 ) );
	assert( is_orthonormal( M2, 1e-3 ) );
	assert( distance( M1.col_y(), M2.col_y() ) < 1e-3 );
	assert( std::abs( dot( bo, M1.col_y() ) ) < 1e-3 );

	// build mid-base-pair triad
	xyzMatrix_float MBT;
	MBT.col_y( M1.col_y() );
	MBT.col_x( ( 0.5f * ( M1.col_x() + M2.col_x() ) ).normalized() );
	MBT.col_z( ( 0.5f * ( M1.col_z() + M2.col_z() ) ).normalized() );

	/////////////////
	// angular params

	// propeller
	// z,x,y make rh coord system
	float const omega = std::atan2( dot( M1.col_z(), M2.col_x() ),
																	dot( M1.col_z(), M2.col_z() ) );

	assert( ( std::abs( std::abs( omega ) -
											arccos( dot( M1.col_z(), M2.col_z() ) ) )<1e-2 ) &&
					( std::abs( std::abs( omega ) -
											arccos( dot( M1.col_x(), M2.col_x() ) ) )<1e-2 ) );

	// buckle:
	float const kappa = gamma * dot( bo, MBT.col_x() );

	// opening
	float const sigma = gamma * dot( bo, MBT.col_z() );

	///////////////////////
	// translational params
	xyzVector_float const displacement( stub1.v - stub2.v );

	params(4) = dot( displacement, MBT.col_x() );
	params(5) = dot( displacement, MBT.col_y() );
	params(6) = dot( displacement, MBT.col_z() );

	/////////////
	// check sign conventions
	assert( omega * dot( MBT.col_y(), cross( M2.col_x(), M1.col_x() ) ) > 0);

	// convert to degrees
	params(1) = degrees( omega );
	params(2) = degrees( kappa );
	params(3) = degrees( sigma );

	if ( verbose ) {
		std::cout << "params: " << aa_name3(aa1) << ' ' << aa_name3(aa2) <<
			" Prop: " << F(9,3,params(1)) <<
			" Buck: " << F(9,3,params(2)) <<
			" Open: " << F(9,3,params(3)) <<
			" Sher: " << F(9,3,params(4)) <<
			" Strc: " << F(9,3,params(5)) <<
			" Stag: " << F(9,3,params(6)) <<
			" gamm: " << F(9,3,degrees(gamma)) << std::endl;
	}

	// now do deriv stuff:
	FArray1D_float dE_dp( 6, 0.0 );

	//Make a label for the basestep type lookup:
	int const ind = get_basepair_index( param_aa::aa_name3(aa1)[2] );

	bool out_of_bounds( false );

	for ( int i = 1; i <= 6; ++i ) {
		float delta_i = params(i) - means_bp(ind,i);

		// reset outliers to max allowed values
		// (beyond which the potential is unreliable)
		float const max_delta_i( MAX_SCORE_DEV*stddevs_bp(ind,i) );
		if ( delta_i >  max_delta_i ) {
			delta_i =  max_delta_i;
			out_of_bounds = true;
		}
		if ( delta_i < -max_delta_i ) {
			delta_i = -max_delta_i;
			out_of_bounds = true;
		}

		for ( int j = 1; j <= 6; ++j ) {
			float delta_j = params(j) - means_bp(ind,j);

			// reset outliers to max allowed values
			// (beyond which the potential is unreliable)
			float const max_delta_j( MAX_SCORE_DEV*stddevs_bp(ind,j) );
			if ( delta_j >  max_delta_j ) {
				delta_j =  max_delta_j;
				out_of_bounds = true;
			}
			if ( delta_j < -max_delta_j ) {
				delta_j = -max_delta_j;
				out_of_bounds = true;
			}

			// score += Gij(ind,i,j)*delta_i*delta_j;

			dE_dp(i) += Gij(ind,i,j)*delta_j;
			dE_dp(j) += Gij(ind,i,j)*delta_i;
		}
	}

	if ( out_of_bounds ) {
		// this is not really a big deal
		std::cout << "WARNING:: out_of_bounds in dna base-pair derivative!" <<
			std::endl;
	}

	// HACK
	if ( debug_bp_deriv ) {
		dE_dp = 0.0;
		for ( int i=1; i<= 6; ++i ) {
			if ( i == debug_bp_deriv || debug_bp_deriv > 6 ) {
				dE_dp(i) = 2 * ( params( i ) - means_bp( ind, i ) );
			}
		}
	}

	// apply sign_factor
	for ( int i=1; i<= 6; ++i ) {
		dE_dp(i) *= sign_factor;
	}


	//////////////////////////
	// propeller (omega) deriv
	//////////////////////////
	//
	// let xi be the original x-axis vector in triad i
	// xi' the vector transformed so that the y-axes coincide
	//
	// cos(omega) = dot( x1' , x2' )
	//            = dot( x1, R_gamma( x2 ) ) // apply R_gamma/2 to both sides
	//
	// ASSUMPTION: R_gamma doesn't depend on the torsion angle, which it
	//  does! but I think the primary contribution is still captured here.
	//
	// d/dphi( dot( x1, R_gamma(x2) )) = dot( cross(u_phi,x1), R_gamma(x2) )
	// = dot( -u_phi , cross( R_gamma(x2), x1 )

	float const rad2deg( degrees(1.0));

	{
		float const dE_domega( dE_dp(1) );

		// these are for preventing d_arccos from blowing up
		static float const small_angle( radians( 0.1f ) );
		static float const big_angle( radians( 179.9f ) );
		static float const max_c( std::cos( small_angle ));
		static float const min_c( std::cos( big_angle ));

		Vec x1( stub1.M.col_x() );
		Vec R_gamma_x2( rotation_matrix( bo, gamma) * Vec(stub2.M.col_x()) );

		assert( distance( Vec(stub1.M.col_y()),
											rotation_matrix(bo,gamma) * Vec(stub2.M.col_y())) <1e-2);

		float c( std::cos( omega ) );
		assert( std::abs( c - dot( x1,R_gamma_x2 ) ) < 1e-2 );
		c = std::min( std::max( min_c, c ), max_c );

		float const domega_dc = -1 / std::sqrt( 1 - c * c );

		// since the derivatives factor through cosine we are getting
		// the derivative of an unsigned angle

		int const sign_factor2( omega < 0 ? -1 : 1 );

		F1 += sign_factor2 * dE_domega * rad2deg * domega_dc *
			cross( R_gamma_x2, x1 );
	}

	///////////////////
	// buckle + opening
	///////////////////
	//
	// as confirmed in the base_pair_params routine,
	// if we let kappa' = sin(gamma) * kappa / gamma,
	// then kappa' = dot( cross( y2, y1 ), x_MBT )
	//
	// likewise sigma' = dot( cross( y2, y1 ), z_MBT )
	//
	// but how do x_MBT and z_MBT vary as we rotate about the phi axis??
	//
	// since these are approximately averages between a moving vector and
	// a fixed vector, we can assume:
	//
	// d/dphi( x_MBT ) = cross( u_phi, x_MBT ) / 2.0
	//
	// have to see if this actually works
	//
	// in addition we can assume that sin(gamma)/gamma is constant
	// since gamma is small
	//

	{
		float const dE_dkappa( dE_dp(2) );
		float const dE_dsigma( dE_dp(3) );

		Vec const y1( stub1.M.col_y() ), y2( stub2.M.col_y() );

		float const gamma_sin_gamma( gamma / std::sin( gamma ) );

		F1 += dE_dkappa * gamma_sin_gamma * rad2deg *
			( cross( y1, cross( y2, MBT.col_x() ) ) +        // 1st term: dy1/dphi
				0.5f * cross( MBT.col_x(), cross( y1, y2 ) ) );// 2nd term: dx_MBT/dphi

		F1 += dE_dsigma * gamma_sin_gamma * rad2deg *
			( cross( y1, cross( y2, MBT.col_z() ) ) +
				0.5f * cross( MBT.col_z(), cross( y1, y2 ) ) );

	}

	///////////////////////
	// translational derivs
	///////////////////////
	//
	// need d/dphi( dot( M-F, x_MBT ) ) where M is rotated, F is fixed,
	// and x_MBT is varying in some complicated way
	//
	// from cst_set.cc, comments to Angle_cst::helper(...)
	// the term w x_MBT constant contributes x_MBT to F2
	// and cross( x_MBT, M) to F1
	//
	// for the other term we use d/dphi( x_MBT ) = 0.5 * cross( u_phi, X_MBT)
	// and the standard vector rearrangement
	{
		Vec M( stub1.v ), F( stub2.v );

		F1 += dE_dp(4) * ( cross( MBT.col_x(), M - 0.5f * ( M - F ) ) );
		F2 += dE_dp(4) * ( MBT.col_x() );

		F1 += dE_dp(5) * ( cross( MBT.col_y(), M - 0.5f * ( M - F ) ) );
		F2 += dE_dp(5) * ( MBT.col_y() );

		F1 += dE_dp(6) * ( cross( MBT.col_z(), M - 0.5f * ( M - F ) ) );
		F2 += dE_dp(6) * ( MBT.col_z() );

	}

}

///////////////////////////////////////////////////////////////////////////////
void
dna_bs_deriv_test()
{
	using namespace pose_ns;
	using namespace param_aa;

	stringafteroption("enable_dna");

	// read pose
	Pose pdb_pose, pose;
	pose_from_pdb( pdb_pose, "/work/pbradley/dna/jim_set/1qn4_trimmed.pdb",
								 true, false, true );
	pdb_pose.dump_pdb("test.pdb");


	pose = pdb_pose;

	int const nres( pose.total_residue() );

	assert( is_protein( pose.res(1) ) );

	setup_simple_dna_fold_tree( pose, FArray1D_bool(nres,false)/*exclude*/);

	set_basepairs( pose );

	assert( pose.num_jump() == 2 );

	Score_weight_map w;
	w.set_weight( DNA_BS, realafteroption("bs") );
	w.set_weight( DNA_BP, realafteroption("bp") );
	w.set_weight( BBONLY_SCOREFXN, 1.0 );


	pose.set_allow_bb_move( false );
	pose.set_allow_chi_move( false );
	pose.set_allow_jump_move( false );
	for ( int i=1; i<= nres; ++i ) {
		if ( pose.is_DNA(i) ) {
			pose.set_allow_bb_move( i, true );
			pose.set_allow_chi_move( i, true );
		}
	}

	pose.score(w);
	pose.dump_pdb("start.pdb");

	prof::reset( true );
	pose.main_minimize( w, "dfpmin" );
	prof::show();

	pose.dump_pdb("final.pdb");

	exit(0);
}


///////////////////////////////////////////////////////////////////////////////
void
dna_bs_deriv_test_jumps()
{
	using namespace pose_ns;
	using namespace param_aa;

	stringafteroption("enable_dna");

	// read pose
	Pose pdb_pose, pose;
	pose_from_pdb( pdb_pose, "/work/pbradley/dna/jim_set/1qn4_trimmed.pdb",
								 true, false, true );
	pdb_pose.dump_pdb("test.pdb");


	pose = pdb_pose;

	int const nres( pose.total_residue() );

	assert( is_protein( pose.res(1) ) );

	setup_simple_dna_fold_tree( pose, FArray1D_bool(nres,false)/*exclude*/);

	set_basepairs( pose );

	assert( pose.num_jump() == 2 );

	int const nres_protein( pose.fold_tree().cutpoints()[0] );
	int const dna_cutpoint( pose.fold_tree().cutpoints()[1] );


	int dna_anchor( 0 );
	for ( int i=nres_protein+1; i<= nres; ++i ) {
		if ( pose.basepair_partner(i) >=1 ) {
			dna_anchor = i;
			break;
		}
	}

	std::cout << "dna_anchor: " << dna_anchor << std::endl;

	FArray1D_bool no_jump(nres, false );
	for ( int i=nres_protein+1; i<= dna_cutpoint; ++i ) {

		int const bp1( pose.basepair_partner(i  ) );
		int const bp2( pose.basepair_partner(i+1) );
		if ( bp1 >= 1 && bp2 >= 1 ) {
			assert( bp1 == bp2 + 1 );

			if ( ( i-dna_anchor )%2 == 0 ) {
				no_jump(i) = true;
			} else {
				no_jump(bp2) = true;
			}
		}
	}

	{ //
		Fold_tree f;
		FArray2D_int jumps(2,1000);
		FArray1D_int cuts(1000);
		int njump(0), ncut(0);

		jumps(1,++njump) = nres_protein;
		jumps(2,  njump) = nres_protein+1;
		cuts (++ncut) = nres_protein;

		for ( int i=nres_protein+1; i<= nres; ++i ) {
			int const j( pose.basepair_partner(i) );
			if ( j >= i ) {
				jumps(1, ++njump) = i;
				jumps(2,   njump) = j;
			}
			if ( i<nres && !no_jump(i) ) {
				jumps(1, ++njump) = i;
				jumps(2,   njump) = i+1;
			}

			if ( i<nres ) {
				cuts(++ncut) = i;
			}
		}

		assert( ncut == njump );
		f.tree_from_jumps_and_cuts( nres, njump, jumps,cuts);

		// setup jump atoms!
		for ( Fold_tree::const_iterator edge = f.begin(); edge != f.end(); ++edge){
			if ( edge->is_jump() ) {
				int const jump( edge->label );
				int const aa1( pose.res( edge->start ) );
				int const aa2( pose.res( edge->stop  ) );
				if ( is_DNA(aa1) && is_DNA(aa2) ) {
					f.set_jump_atoms( jump,
														LookupByName( aa1, 1, " C2 " ),
														LookupByName( aa2, 1, " C2 " ), true );
				}
			}
		}
		pose.set_fold_tree( f );
	}


	Score_weight_map w;
	w.set_weight( DNA_BS, realafteroption("bs") );
	w.set_weight( DNA_BP, realafteroption("bp") );
	w.set_weight( BBONLY_SCOREFXN, 1.0 );


	pose.set_allow_bb_move( false );
	pose.set_allow_jump_move( true );
	pose.set_allow_chi_move( false );

	pose.score(w);
	pose.dump_pdb("start.pdb");

	prof::reset( true );
	pose.main_minimize( w, "dfpmin" );
	prof::show();

	pose.dump_pdb("final.pdb");

	exit(0);
}


///////////////////////////////////////////////////////////////////////////////
void
dna_bs_deriv_test_old()
{
	using namespace pose_ns;
	using namespace param_aa;

	stringafteroption("enable_dna");

	// read pose
	Pose pdb_pose;
	pose_from_pdb( pdb_pose, "/work/pbradley/dna/jim_set/1aay.pdb", true, false,
								 true );


	pdb_pose.dump_pdb("test.pdb");

	// create mini pose -- two paired residues with a jump between them
	int const Apos( 77 ), Tpos( 63 );
	assert( pdb_pose.res(Apos) == na_ade && pdb_pose.res(Tpos) == na_thy );

	Pose pose, start_pose;

	if ( false ) {

		// one jump
		Fold_tree f;
		int const num_jump( 1 );
		int const nres(2);
		FArray2D_int jumps(2,num_jump);
		FArray1D_int cuts(num_jump);

		jumps(1,1) = 1;
		jumps(2,1) = 2;
		cuts(1) = 1;

		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );

		f.reorder(1);

		f.set_jump_atoms( 1,
											LookupByName( pdb_pose.res(Apos  ), 1, " C2 " ),
											LookupByName( pdb_pose.res(Apos+1), 1, " C2 " ), true );
		pose.set_fold_tree( f );

		pose.set_fullatom_flag( true, false );

		pose.copy_segment( 2, pdb_pose, 1, Apos );

	} else { // setup fold_tree
		int const nres(4);
		pose.simple_fold_tree(nres);

		pose.copy_segment( 2, pdb_pose, 1, Apos   );
		pose.copy_segment( 2, pdb_pose, 3, Tpos-1 );

		Fold_tree f;
		int const num_jump( 3 );
		FArray2D_int jumps(2,num_jump);
		FArray1D_int cuts(num_jump);

		jumps(1,1) = 1;
		jumps(2,1) = 4;
		jumps(1,2) = 3;
		jumps(2,2) = 4;
		jumps(1,3) = 2;
		jumps(2,3) = 3;

		cuts(1) = 1;
		cuts(2) = 2;
		cuts(3) = 3;

		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );

		f.reorder(1);

		f.set_jump_atoms( 1,
											LookupByName( pose.res(1), 1, " C2 " ),
											LookupByName( pose.res(4), 1, " C2 " ), true );

		f.set_jump_atoms( 2,
											LookupByName( pose.res(4), 1, " C2 " ),
											LookupByName( pose.res(3), 1, " C2 " ), true );

		f.set_jump_atoms( 1,
											LookupByName( pose.res(3), 1, " C2 " ),
											LookupByName( pose.res(2), 1, " C2 " ), true );

		pose.set_fold_tree(f);

		pose.set_fullatom_flag( true, false );

		pose.copy_segment( 2, pdb_pose, 1, Apos   );
		pose.copy_segment( 2, pdb_pose, 3, Tpos-1 );

	}

	pose.setup_atom_tree();

	pose.dump_pdb("test2.pdb");

	Score_weight_map w;
	w.set_weight( DNA_BS, 1.0 );
	w.set_weight( DNA_BP, 1.0 );


	pose.set_allow_bb_move( false );
	pose.set_allow_jump_move( true );
// 	pose.set_allow_jump_move( false );
	pose.set_allow_jump_move( 2, true );
	pose.set_allow_chi_move( false );

	set_basepairs( pose );

	debug_bs_deriv = intafteroption("d");

	for ( int i=1; i<= 3; ++i ) {
		pose.set_jump_rb_delta(i,1,1,0.5);
		pose.set_jump_rb_delta(i,2,1,0.5);
		pose.set_jump_rb_delta(i,3,1,0.5);
		pose.set_jump_rb_delta(i,4,1,10.0);
		pose.set_jump_rb_delta(i,5,1,10.0);
		pose.set_jump_rb_delta(i,6,1,10.0);
	}

	pose.fold_in_rb_deltas();

	pose.dump_pdb("start.pdb");

	pose.main_minimize( w, "dfpmin" );

	pose.dump_pdb("final."+string_of(debug_bs_deriv)+".pdb");

	exit(0);
}


///////////////////////////////////////////////////////////////////////////////
void
dna_bp_deriv_test()
{
	using namespace pose_ns;
	using namespace param_aa;

	stringafteroption("enable_dna");

	// read pose
	Pose pdb_pose;
	pose_from_pdb( pdb_pose, "/work/pbradley/dna/jim_set/1aay.pdb", true, false,
								 true );


	pdb_pose.dump_pdb("test.pdb");

	// create mini pose -- two paired residues with a jump between them
	int const Apos( 77 ), Tpos( 63 );
	assert( pdb_pose.res(Apos) == na_ade && pdb_pose.res(Tpos) == na_thy );

	Pose pose, start_pose;
	pose.one_jump_tree( 2, 1, 2, 1 );

	pose.set_res( 1, na_ade );
	pose.set_res( 2, na_thy );
	pose.set_res_variant( 1, 1 );
	pose.set_res_variant( 2, 1 );

	pose.set_fullatom_flag( true, false );

	pose.copy_segment( 1, pdb_pose, 1, Apos );
	pose.copy_segment( 1, pdb_pose, 2, Tpos );

	pose.setup_atom_tree();

	pose.dump_pdb("test2.pdb");

	pose.atom_tree()->root()->show();

	{
		Fold_tree f( pose.fold_tree() );
		f.reorder(1); // prob unnec
		f.set_jump_atoms( 1, LookupByName( na_ade, 1, " N1 " ),
											LookupByName( na_thy, 1, " N3 " ), true );
		pose.set_fold_tree( f );

	}
	std::cout << "setup new atom-tree:" << std::endl;

	pose.atom_tree()->root()->show();

	pose.dump_pdb("test2b.pdb");

	Score_weight_map w;
	w.set_weight( DNA_BP, 1.0 );

	pose.set_allow_bb_move( false );
	pose.set_allow_jump_move( true );
	pose.set_allow_chi_move( false );

	set_basepairs( pose );

	debug_bp_deriv = intafteroption("d");

	pose.set_jump_rb_delta(1,1,1,0.5);
	pose.set_jump_rb_delta(1,2,1,0.5);
	pose.set_jump_rb_delta(1,3,1,0.5);
	pose.set_jump_rb_delta(1,4,1,10.0);
	pose.set_jump_rb_delta(1,5,1,10.0);
	pose.set_jump_rb_delta(1,6,1,10.0);

	pose.fold_in_rb_deltas();

	pose.dump_pdb("start.pdb");

	pose.main_minimize( w, "dfpmin" );

	pose.dump_pdb("final."+string_of(debug_bp_deriv)+".pdb");

	exit(0);
}


///////////////////////////////////////////////////////////////////////////////
void
bzip_test_old()
{
	using namespace pose_ns;

	stringafteroption  ("hydrate_dna") +
		stringafteroption("enable_dna") +
		stringafteroption("dna_interface") +
		stringafteroption("ex_dna_waters");


	bool const repack_interface( truefalseoption("repack") );

	// read in 2dgc == gcn4
	Pose gcn4_pose, pose;

	std::string const filename
		( "/work/pbradley/dna/bzip/gcn4/2dgc_biomolecule.pdb" );

	pose_from_pdb( pose, filename, true, false, true );
	pose.copy_to_misc(); // STUPID -- for designmap? -- apl, probably.  We're working on it

	int const nres( pose.total_residue() );

	// identify interface protein residues

	// these are the bases of interest:
	FArray1D_bool hydrate( nres, false );
	for ( int i=1; i<= nres; ++i ) {
		if ( i >= 109 && i <= 113 ||
				 i >= 123 && i <= 127 ) hydrate(i) = true;
	}

	int const interface_helix_end( 20 );

	std::cout << "select ";
	FArray1D_bool interface( nres, false );
	for ( int i=1; i<= interface_helix_end; ++i ){
		using aaproperties_pack::natoms;
		bool contact( false );
		if ( !pose.is_protein( i ) ) continue;
		for ( int j=1; j<= nres; ++j ) {
			if ( !hydrate(j) ) continue;
			for ( int ii=1; ii<= natoms( pose.res(i), pose.res_variant(i) ); ++ii ) {
				for ( int jj=1; jj<= natoms( pose.res(j),pose.res_variant(j)); ++jj ) {
					if ( distance( pose.full_coord(ii,i), pose.full_coord(jj,j) )<4.5 ){
						contact = true;
						break;
					}
				}
				if ( contact ) break;
			}
			if ( contact ) break;
		}
		if ( contact ) std::cout << i << ", ";
		interface(i) = contact;
	}
	std::cout << std::endl;

	// setup the freakin design_map
	DesignMap design_map( nres, pose.res() );
	{
		using namespace aaproperties_pack;

		FArray1D_bool default_variant_setting( aaproperties_pack::number_aav_type,
																					 false );

		default_variant_setting( aaproperties_pack::aav_base ) = true;
		//default_variant_setting( aaproperties_pack::aav_his_taut ) = true;
		default_variant_setting( aaproperties_pack::aav_water ) = true;

		design_map.set_variant_defaults( default_variant_setting );

		// this applies the defaults
		for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
			design_map.fix_sidechain_allowing_variants( seqpos );

			if ( repack_interface ) {
				// packing
				if ( interface( seqpos ) ) {
					design_map.set( seqpos, pose.res(seqpos) );
					design_map.require_base_rotamer( seqpos );
				} else if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}
				std::cout << "design_map: repack: " << seqpos << ' ' <<
					design_map.repack_residue( seqpos ) << std::endl;

			} else {
				// just hydrate
				if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}
			}

		}



	}


	Score_weight_map w12( score12 );
	if ( truefalseoption("premin") ) {

		pose.set_allow_bb_move( false );
		pose.set_allow_chi_move( interface );
		pose.set_allow_jump_move( false );

		pose.dump_scored_pdb("premin.pdb",w12);
		pose.main_minimize( w12, "dfpmin" );
		pose.dump_scored_pdb("postmin.pdb",w12);

	}


	water::use_hbond_potential = !truefalseoption("std_water");
	bool const inc_current( truefalseoption("inc_current") );

	std::string run_tag;
	if ( repack_interface ) run_tag += ".repack_interface";
	else run_tag += ".repack_waters";
	if ( inc_current ) run_tag += ".inc_current";
	else run_tag += ".no_inc_current";
	if ( water::use_hbond_potential ) run_tag += ".hbond_waters";
	else run_tag += ".std_waters";

	// rotamer defaults
	pose_setup_packer();

	PackerTask task( pose );
	FArray1D_bool allow_repack( nres, true );

	task.set_task( "packrot", false/*make-outputfile*/, allow_repack,
								 inc_current );
	task.set_designmap( design_map);
	//pose.dump_scored_pdb( "gcn4_before.pdb", w12 );

	task.set_nloop( 100 );
	prof::reset( true );
	pose.pack( task );
	prof::show();

	pose.dump_scored_pdb( "gcn4"+run_tag+".pdb", w12 );


	if ( truefalseoption("postmin" ) ) {

		Score_weight_map w;
		w.set_weight( FA_H2O, 1.0 );

		pose.set_allow_bb_move( false );
		pose.set_allow_chi_move( interface );
		pose.set_allow_jump_move( false );

		pose.dump_scored_pdb("pre_postmin.pdb",w);
		pose.main_minimize( w, "dfpmin" );
		pose.dump_scored_pdb("post_postmin.pdb",w);

	}

	exit(0);


	// read in 1gd2 == pap1
	Pose pap1_pose;
	pose_from_pdb( pose, "/work/pbradley/dna/bzip/pap1/1gd2.pdb",
								 true, false, true );

	pose.dump_pdb( "pap1.pdb" );


}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
bzip_repack_test()
{
	using namespace pose_ns;

	stringafteroption  ("hydrate_dna"    ) +
		stringafteroption("enable_dna"     ) +
		stringafteroption("dna_interface"  ) +
		stringafteroption("gen_born"       ) +
		stringafteroption("extrachi_cutoff") +
		stringafteroption("fa_input"       ) +
		stringafteroption("no_optH"        ) +
		stringafteroption("ex_dna_waters"  );


	for ( int r=1; r<=3; ++r ) {
		std::string const mode( r == 1 ? "gcn4" :
														( r == 2 ? "pap1" : "cebp" ) );

		for ( int rr=-1; rr <= 4; ++rr ) {

			// -1 - repack only waters
			// 0 - repack sidechains w/ water
			// 1 - repack sidechains w/ water, inc_current
			// 2 - repack sidechains w/o water
			// 3 - repack sidechains w/o water, inc_current

			bool repack_interface( true ), inc_current( false ), use_h2o( true );
			if ( rr < 0 ) {
				repack_interface = false;
			} else {
				if ( rr%2 == 1 ) inc_current = true;
				if ( rr/2 == 1 ) use_h2o = false;
			}


	// read one of the three subset pdbs
	Pose pose;

	std::string const filename
		( "/work/pbradley/dna/bzip/three_test/"+mode+"_subset.pdb");

	pose_from_pdb( pose, filename, true, false, true );
	pose.copy_to_misc(); // STUPID -- for designmap?


	int const nres( pose.total_residue() );

	FArray1D_bool hydrate( nres, false ), interface( nres, false );
	for ( int i=1; i<= nres; ++i ) {
		if ( pose.is_DNA(i) ) hydrate(i)= true;
		else interface(i) = true;
	}

	// setup the freakin design_map
	DesignMap design_map( nres, pose.res() );
	{
		using namespace aaproperties_pack;

		FArray1D_bool default_variant_setting( aaproperties_pack::number_aav_type,
																					 false );

		// setup the variants
		default_variant_setting( aaproperties_pack::aav_base ) = true;
		//default_variant_setting( aaproperties_pack::aav_his_taut ) = true;
		if ( use_h2o ) {
			default_variant_setting( aaproperties_pack::aav_water ) = true;
		}
		design_map.set_variant_defaults( default_variant_setting );

		// fix_sidechain_allowing_variants allows the default variants!
		for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
			design_map.fix_sidechain_allowing_variants( seqpos );

			if ( repack_interface ) {
				// packing
				if ( interface( seqpos ) ) {
					design_map.set( seqpos, pose.res(seqpos) );
					design_map.require_base_rotamer( seqpos );
				} else if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}

			} else {
				// just hydrate
				if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}
			}

		}
	}


	water::use_hbond_potential = !truefalseoption("std_water");
	//bool const inc_current( truefalseoption("inc_current") );

	std::string run_tag;
	if ( repack_interface ) run_tag += ".repack_interface";
	else run_tag += ".repack_waters";
	if ( inc_current ) run_tag += ".inc_current";
	else run_tag += ".no_inc_current";
	if ( water::use_hbond_potential ) run_tag += ".hbond_waters";
	else run_tag += ".std_waters";
	if ( use_h2o ) run_tag += ".waters";
	else run_tag += ".no_waters";

	// rotamer defaults
	pose_setup_packer();

	PackerTask task( pose );
	FArray1D_bool allow_repack( nres, true );

	task.set_task( "packrot", false/*make-outputfile*/, allow_repack,
								 inc_current );
	task.set_designmap( design_map);

	task.set_nloop( 100 );
	prof::reset( true );
	pose.pack( task );
	prof::show();

	pose.dump_scored_pdb( mode+run_tag+".pdb", Score_weight_map( score12 ) );

	if ( truefalseoption("postmin" ) ) {

		Score_weight_map w;
		w.set_weight( FA_H2O, 100.0 );

		pose_set_use_nblist( false );
		minimize_set_tolerance( 1e-6 );

		pose.set_allow_bb_move( false );
		pose.set_allow_chi_move( interface );
		pose.set_allow_jump_move( false );

		pose.dump_scored_pdb(mode+run_tag+".pre_postmin.pdb",w);
		pose.main_minimize( w, "dfpmin" );
		pose.dump_scored_pdb(mode+run_tag+".post_postmin.pdb",w);

	}
		} // rr
	} // r
}


///////////////////////////////////////////////////////////////////////////////
int
setup_dna_jump( pose_ns::Pose & pose )
{
	set_basepairs( pose );

	//
	int const num_jump( pose.num_jump() );
	int dna_jump(0), pos1(0), pos2(0);

	pose_ns::Fold_tree const & f( pose.fold_tree() );
	for ( int i=1; i<= num_jump; ++i ) {
		if ( pose.is_DNA    ( f.  upstream_jump_residue(i) ) &&
				 pose.is_protein( f.downstream_jump_residue(i) ) ) {
			pos1 = f.  upstream_jump_residue(i);
			pos2 = f.downstream_jump_residue(i);
			dna_jump = i;
			break;
		}
	}

	//
	// set the rotation centers for this jump
	//

	// upstream center should be at the midpoint of the basepair
	//
	pose_ns::Jump jump( pose.get_jump( dna_jump ) );
	{
		int const partner1( pos1 );
		int const partner2( pose.basepair_partner( pos1 ) );
		int const aa1 ( pose.res        ( partner1 ) );
		int const aa2 ( pose.res        ( partner2 ) );
		int const aav1( pose.res_variant( partner1 ) );
		int const aav2( pose.res_variant( partner2 ) );
		int const atom1( LookupByName( aa1, aav1, " C1*" ) );
		int const atom2( LookupByName( aa2, aav2, " C1*" ) );
		Vec center( 0.5f * ( pose.full_coord(atom1,partner1) +
												 pose.full_coord(atom2,partner2) ) );

		jump.set_rb_center( -1 /*c2n*/, pose.upstream_jump_stub( dna_jump ),
												center );
	}


	// downstream center should be at the c-alpha of the jump residue
	// which it probably already is by default
	//
	{
		jump.set_rb_center( 1 /*n2c*/, pose.downstream_jump_stub( dna_jump ),
												pose.full_coord( 2 /*CA*/, pos2 ) );
	}

	pose.set_jump( dna_jump, jump );

	return dna_jump;
}


///////////////////////////////////////////////////////////////////////////////
// assumes that the pose is anchored in the DNA
//
void
fast_dna_relax(
							 pose_ns::Pose & pose,
							 int const main_cycles
)
{
	using namespace pose_ns;
	// PARAMS
	set_smallmove_size(2.0,2.0,3.0); // helix,strand,other
	float const temperature( 0.8 );
	int const max_lj_ramp_cycles( 6 );
	float const min_lj_rep_weight( 0.02 );
	int const lj_ramp_inner_cycles( 10 );
	float const max_fa_rep( 10.0 );
	float const trans_mag( 0.25 );
	float const rot_mag( 2.0 );
	minimize_set_tolerance( 1e-4 );
	set_use_nblist( truefalseoption("nblist") );
	float const energycut( 0.1 );

	assert( pose.fullatom() );

	prof::reset( true );

	// setup the rb centers
	// also fills basepairing information
	int const dna_jump( setup_dna_jump( pose ) );

	// are we moving proteinbb?
	bool do_small_moves( false );
	for ( int i=1; i<= pose.total_residue(); ++i ) {
		if ( pose.get_allow_bb_move(i) ){
			do_small_moves = true;
			break;
		}
	}

	// setup basepairswapping
	std::vector< int > bases, bp_swap_list;
	{
		using namespace param_aa;
		bases.push_back( na_ade );
		bases.push_back( na_thy );
		bases.push_back( na_gua );
		bases.push_back( na_cyt );
		bp_swap_list.push_back( 21 );
		bp_swap_list.push_back( 22 );
		bp_swap_list.push_back( 23 );
	}


	//////////////////////////////
	// ramp repulsive if necessary
	float lj_rep_weight(1.0);
	{
		pose.score( score12 );
		float const fa_rep( pose.get_0D_score( FA_REP ) );
		if ( fa_rep > max_fa_rep ) {
			lj_rep_weight =
				( std::max( min_lj_rep_weight, max_fa_rep / fa_rep ));
		}
	}

	Score_weight_map w( score12 );
	w.set_weight( FA_REP, lj_rep_weight );
	Monte_carlo mc( pose, w, temperature );

	if ( truefalseoption("pose_deriv_check") ) {
		pose_deriv_check( pose );
	}


	/////////////////////////////////////////////////////////////////////////////
	//// LJ-RAMPING CYCLES
	int lj_ramp_cycle(0);
// 	bool lj_ramped( false );
	while ( lj_rep_weight < 1.0 - 1e-3 ) {
// 		lj_ramped = true;
		++lj_ramp_cycle;
		if ( lj_ramp_cycle > max_lj_ramp_cycles ) break;

		std::cout << "lj_ramp: " << lj_ramp_cycle << ' ' << lj_rep_weight <<
			std::endl;

		w.set_weight( FA_REP, lj_rep_weight );
		mc.set_weight_map( w );

		pose.full_repack( true );
		mc.boltzmann( pose, "ljramp_repack" );

		pose.main_minimize( w, "dfpmin" );
		mc.boltzmann( pose, "ljramp_main_minimize" );

		for ( int i=1; i<= lj_ramp_inner_cycles; ++i ) {
			int const dir( ran3() < 0.5 ? 1 : -1 );
			pose_rb_min_trial( pose, mc, "linmin" /* !!!!!! */, dna_jump, dir,
												 trans_mag, rot_mag, true, 0.1 );
		}

		mc.recover_low( pose );

		// update the weight
		float const fa_rep( pose.get_0D_score( FA_REP ) );
		lj_rep_weight =
			std::min( 1.0f , std::max( 2*lj_rep_weight, max_fa_rep / fa_rep ) );
	}

	// ensure full weight
	w.set_weight( FA_REP, 1.0 );
	mc.set_weight_map( w );

	// minimize
	pose.main_minimize( w, "dfpmin" );
	mc.boltzmann( pose, "main_minimize" );

	// repack
	pose.full_repack( true );
	mc.boltzmann( pose, "repack" );

	if ( truefalseoption("pose_deriv_check") ) {
		pose_deriv_check( pose );
	}


	/////////////////////////////////////////////////////////////////////////////
	// MAIN CYCLES

	mc.recover_low( pose );
	for ( int i=1; i<= main_cycles; ++i ) {
		if ( i%5 == 0 && do_small_moves ) {
			pose_small_min_trial( pose, mc, w, "dfpmin", 3, true, energycut );
		}

		int const dir( ran3() > 0.5 ? 1 : -1 );
		pose_rb_min_trial( pose, mc, "dfpmin", dna_jump, dir, trans_mag, rot_mag,
											 true, energycut );


		{ // try dna basepair change
			using aaproperties_pack::icoor;
			int const seqpos1
				( bp_swap_list[ static_cast< int >( ran3() * bp_swap_list.size() ) ] );
			int new_aa(0);
			while (!new_aa){
				new_aa = bases[ static_cast< int >( ran3() * bases.size() ) ];
				if ( new_aa == pose.res(seqpos1) ) new_aa = 0;
			}
			int const seqpos2( pose.basepair_partner( seqpos1 ) );
			int const aav(1);
			int const new_aa2( na_partner(new_aa));
			int const old_aa( pose.res(seqpos1) );
			pose.copy_sidechain( seqpos1, new_aa , aav, icoor(1,1,new_aa ,aav) );
			pose.copy_sidechain( seqpos2, new_aa2, aav, icoor(1,1,new_aa2,aav) );


			std::cout << "baseswap: " << seqpos1 << ' ' << seqpos2 << ' ' <<
				param_aa::aa_name3( old_aa ) << " to " <<
				param_aa::aa_name3( new_aa ) << std::endl;

			score_set_try_rotamers( true );
			set_rotamer_trials_by_deltaE(
				 pose_ns::RESENERGY, energycut, pose.total_residue(),
				 mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 3 /*cycles*/ );

			// calls score --> rottrials inside here
			pose.main_minimize( mc.weight_map(), "dfpmin" );
			score_set_try_rotamers( false );
			mc.boltzmann( pose, "base-swap-min" );
		}


		if ( i%10 == 0 ) {
			pose.full_repack( true );
			mc.boltzmann( pose, "repack" );
		}
	}

	mc.recover_low( pose );

	// minimize
	pose.main_minimize( w, "dfpmin" );
	mc.boltzmann( pose, "main_minimize" );
	mc.recover_low( pose );

	mc.show_counters();
	prof::show();

}



///////////////////////////////////////////////////////////////////////////////
// specificity test

void
bzip_test()
{
	using namespace pose_ns;

	bool const use_h2o( truefalseoption("use_h2o") );
	bool const inc_current( false );

	stringafteroption  ("hydrate_dna"    ) +
		stringafteroption("enable_dna"     ) +
		stringafteroption("dna_interface"  ) +
		stringafteroption("gen_born"       ) +
		stringafteroption("extrachi_cutoff") +
		stringafteroption("fa_input"       ) +
		stringafteroption("no_optH"        ) +
		stringafteroption("ex_dna_waters"  );

	std::vector< std::string > bzips;
	bzips.push_back( "gcn4" );
	bzips.push_back( "cebp" );
// 	bzips.push_back( "pap1" );
	int const nzip( bzips.size() );

	// first read the three poses
	std::vector< Pose * > bzip_poses;

	for ( int r=0; r<nzip; ++r ) {
		Pose * const pose( new Pose );
		bzip_poses.push_back( pose );

		std::string const filename
			( "/work/pbradley/dna/bzip/three_test/"+bzips[r]+"_subset.pdb");

		pose_from_pdb( *pose, filename, true, false, true );

		// setup fold_tree
		{
			int const num_jump(2);
			FArray1D_int cuts(num_jump);
			FArray2D_int jumps(2,num_jump);
			jumps(1,1) = 11;
			jumps(2,1) = 22;
			jumps(1,2) = 22;
			jumps(2,2) = 29;


			cuts(1) = 18;
			cuts(2) = 25;

			Fold_tree f;
			f.tree_from_jumps_and_cuts( pose->total_residue(), num_jump, jumps,
																	cuts );
			f.reorder( 22 );
			pose->set_fold_tree(f);
		}
	}


	// now try specificity calcs:
	for ( int r1=0; r1< nzip; ++r1 ) {
		std::string const & bzip1( bzips[r1] );

		//get dna sequence from another pose
		for ( int r2=0; r2< nzip; ++r2 ) {
			std::string const & bzip2( bzips[r2] );

		//get dna sequence from another pose
		for ( int r3=0; r3< nzip; ++r3 ) {
			std::string const & bzip3( bzips[r3] );

			Pose pose, dna_src_pose, protein_src_pose;

			pose             = *bzip_poses[r1];
			dna_src_pose     = *bzip_poses[r2];
			protein_src_pose = *bzip_poses[r3];

			int const nres( pose.total_residue() );

			// update the dna sequence
			for ( int i=1; i<= nres; ++i ) {
				if ( pose.is_DNA(i) ) {
					int const aa ( dna_src_pose.res(i) );
					int const aav( dna_src_pose.res_variant(i) );
					if ( aa != pose.res(i) ) {
						pose.copy_sidechain( i, aa, aav,
																 aaproperties_pack::icoor(1,1,aa,aav) );
					}
				}
				if ( i != 1 && pose.is_protein(i) ) {   // !!!!!!!!!!!!!!!!!!!!!!!!!
					int const aa ( protein_src_pose.res(i) );
					int const aav( protein_src_pose.res_variant(i) );
					if ( aa != pose.res(i) ) {
						pose.copy_sidechain( i, aa, aav,
																 aaproperties_pack::icoor(1,1,aa,aav) );
					}
				}
			}



			// now try repacking

			FArray1D_bool hydrate( nres, false ), interface( nres, false );
			for ( int i=1; i<= nres; ++i ) {
				if ( pose.is_DNA(i) ) hydrate(i)= true;
				else interface(i) = true;
			}

			// setup the freakin design_map
			DesignMap design_map( nres, pose.res() );
			{
				using namespace aaproperties_pack;

				FArray1D_bool default_variant_setting
					( aaproperties_pack::number_aav_type, false );

				// setup the variants
				default_variant_setting( aaproperties_pack::aav_base ) = true;
				//default_variant_setting( aaproperties_pack::aav_his_taut ) = true;
				if ( use_h2o ) {
					default_variant_setting( aaproperties_pack::aav_water ) = true;
				}
				design_map.set_variant_defaults( default_variant_setting );

				// fix_sidechain_allowing_variants allows the default variants!
				for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
					design_map.fix_sidechain_allowing_variants( seqpos );

					// packing
					if ( interface( seqpos ) ) {
						design_map.set( seqpos, pose.res(seqpos) );
						design_map.require_base_rotamer( seqpos );
					} else if ( !hydrate(seqpos) ) {
						design_map.require_base_rotamer( seqpos );
					}
				}
			}


			water::use_hbond_potential = !truefalseoption("std_water");


			// rotamer defaults
			pose_setup_packer();

			PackerTask task( pose );
			FArray1D_bool allow_repack( nres, true );

			task.set_task( "packrot", false/*make-outputfile*/, allow_repack,
										 inc_current );
			task.set_designmap( design_map);

			task.set_nloop( 100 );
			prof::reset( true );
			pose.pack( task );
			prof::show();

			std::string tag( "scaffold-"+bzip1+".dna-"+bzip2+".protein-"+bzip3 );
			if ( use_h2o ) tag += ".h2o";
			pose.dump_scored_pdb( tag+".pdb",
														Score_weight_map( score12 ) );


			if ( truefalseoption("postmin" ) ) {

				Score_weight_map w( score12 );

				pose_set_use_nblist( false );
				minimize_set_tolerance( 1e-6 );

				pose.set_allow_bb_move( false );
				pose.set_allow_chi_move( interface );
				pose.set_allow_jump_move( false );

				pose.main_minimize( w, "dfpmin" );
				pose.dump_scored_pdb( tag+".postmin.pdb", w );
			} else if ( truefalseoption("post_relax") ) {

				for ( int i=1; i<= nres; ++i ) {
					if ( pose.is_DNA(i) ) {
						pose.set_allow_bb_move ( i, false );
						pose.set_allow_chi_move( i, false );
					} else {
						pose.set_allow_bb_move ( i, true );
						pose.set_allow_chi_move( i, true );
					}
				}
				pose.set_allow_jump_move( false );
				pose.set_allow_jump_move( 1, true );
				fast_dna_relax( pose, 20 );
				pose.dump_scored_pdb( tag+".relaxed.pdb", Score_weight_map( score12) );

			}
		} // r3
		} // r2
	} // r1
}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
bzip_relax_test()
{
	using namespace pose_ns;
	using namespace param_pack; // for pack_wts

	stringafteroption  ("enable_dna"     ) +
		stringafteroption("dna_interface"  ) +
		stringafteroption("gen_born"       ) +
		stringafteroption("extrachi_cutoff") +
		stringafteroption("fa_input"       ) +
		//stringafteroption("hydrate_dna"    ) +
		//stringafteroption("ex_dna_waters"  ) +
		stringafteroption("no_optH"        );

	std::string const mode( stringafteroption("mode") );
	int const nstruct( intafteroption("nstruct"));

	bool const repack_interface( false ), inc_current( true ),
		use_h2o( truefalseoption("use_h2o") );

	// NOTE: ex flags are only on by command line!

	// prefix:
	std::string prefix
		( files_paths_pdb_out_prefix()+"bzip_spec_relax_"+mode+"_" );

	// adjust param pack weights:
	initialize_fullatom(); // ensure that this has been done

	if ( truefalseoption("Wdun_factor") ) {
		pack_wts.set_Wdun( pack_wts.Wdun() * realafteroption("Wdun_factor") );
		prefix += "Wdun"+string_of( pack_wts.Wdun() )+"_";
	}
	if ( truefalseoption("Wsol_factor") ) {
		pack_wts.set_Wsol( pack_wts.Wsol() *realafteroption("Wsol_factor") );
		prefix += "Wsol"+string_of( pack_wts.Wsol() )+"_";
	}
	if ( truefalseoption("Whbond_factor") ) {
		pack_wts.set_Whbond_bb( pack_wts.Whbond_bb() * realafteroption("Whbond_factor") );
		pack_wts.set_Whbond_sc( pack_wts.Whbond_sc() * realafteroption("Whbond_factor") );
		pack_wts.set_Whbond_bb_sc( pack_wts.Whbond_bb_sc() * realafteroption("Whbond_factor") );
		prefix += "Whbond_sc"+string_of( pack_wts.Whbond_sc() )+"_";
	}
	if ( truefalseoption("Wh2o_intro_factor") ) {
		pack_wts.set_Wh2o_hb( pack_wts.Wh2o_intra() * realafteroption("Wh2o_intra_factor") );
		prefix += "Wh2o_intra"+string_of( pack_wts.Wh2o_intra() )+"_";
	}
	if ( truefalseoption("Wh2o_hb_factor") ) {
		pack_wts.set_Wh2o_hb( pack_wts.Wh2o_hb() * realafteroption("Wh2o_hb_factor") );
		prefix += "Wh2o_hb"+string_of( pack_wts.Wh2o_hb() )+"_";
	}

	silent_io::Silent_out out( prefix+".out");

	for ( int n=1; n<= nstruct; ++n ) {

		if ( !out.start_decoy( string_of(n) ) ) continue;

#ifdef GL_GRAPHICS
		gl_graphics_set_title( mode + "relax_test" + string_of( n ) );
#endif



	// read one of the three subset pdbs
	Pose pose;

	std::string const filename( mode+"_subset.pdb");

	pose_from_pdb( pose, filename, true, false, true );
	set_basepairs( pose );

	pose.copy_to_misc(); // STUPID -- for designmap?

	int const nres( pose.total_residue() );

	// setup fold_tree
	{
		int const num_jump(2);
		FArray1D_int cuts(num_jump);
		FArray2D_int jumps(2,num_jump);
		jumps(1,1) = 11;
		jumps(2,1) = 22;
		jumps(1,2) = 22;
		jumps(2,2) = 29;


		cuts(1) = 18;
		cuts(2) = 25;

		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );
		f.reorder( 22 );
		pose.set_fold_tree(f);

		setup_dna_jump(pose); // testing only
	}



	FArray1D_bool hydrate( nres, false ), interface( nres, false );
	for ( int i=1; i<= nres; ++i ) {
		if ( pose.is_DNA(i) ) hydrate(i)= true;
		else interface(i) = true;
	}

	// setup the freakin design_map
	DesignMap design_map( nres, pose.res() );
	{
		using namespace aaproperties_pack;

		FArray1D_bool default_variant_setting( aaproperties_pack::number_aav_type,
																					 false );

		// setup the variants
		default_variant_setting( aaproperties_pack::aav_base ) = true;
		//default_variant_setting( aaproperties_pack::aav_his_taut ) = true;
		if ( use_h2o ) {
			default_variant_setting( aaproperties_pack::aav_water ) = true;
		}
		design_map.set_variant_defaults( default_variant_setting );

		// this design_map.reset allows the default variants!
		for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
			design_map.fix_sidechain_allowing_variants( seqpos );

			if ( repack_interface ) {
				// packing
				if ( interface( seqpos ) ) {
					design_map.set( seqpos, pose.res(seqpos) );
					design_map.require_base_rotamer( seqpos );
				} else if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}

			} else {
				// just hydrate
				if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}
			}

		}
	}


	water::use_hbond_potential = true;


	// rotamer defaults
	pose_setup_packer();

	design::active_rotamer_options.set_ex12( truefalseoption("ex1"), truefalseoption("ex2"),
						truefalseoption("ex1aro") );

	PackerTask task( pose );
	FArray1D_bool allow_repack( nres, true );

	task.set_task( "packrot", false/*make-outputfile*/, allow_repack,
								 inc_current );
	task.set_designmap( design_map);

	task.set_nloop( 100 );
	prof::reset( true );
	pose.pack( task );
	prof::show();


	// only the rigid-body and the protein sidechains
	pose.set_allow_jump_move( false );
	pose.set_allow_jump_move( 1, true );
	for ( int i=1; i<= nres; ++i ) {
		if ( pose.is_protein(i) ) {
			pose.set_allow_bb_move ( i, false );
			pose.set_allow_chi_move( i, true );
		} else {
			pose.set_allow_bb_move ( i, false );
			pose.set_allow_chi_move( i, false );
		}
	}

	fast_dna_relax( pose, 25 );

	pose.dump_scored_pdb( prefix+string_of(n)+".pdb",
												Score_weight_map( score12) );
	} // n=1,nstruct
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
bzip_swap_test()
{
	using namespace pose_ns;

	stringafteroption  ("enable_dna"     ) +
		stringafteroption("dna_interface"  ) +
		stringafteroption("gen_born"       ) +
		stringafteroption("extrachi_cutoff") +
		stringafteroption("fa_input"       ) +
		stringafteroption("no_optH"        );


	int const nstruct( intafteroption("nstruct") );
	std::string const target  ( stringafteroption("target"));
	std::string const scaffold( stringafteroption("scaffold"));

	bool const repack_interface( true ), inc_current( false ),
		use_h2o( truefalseoption("use_h2o") );

	if ( use_h2o )
		stringafteroption("hydrate_dna")+stringafteroption("ex_dna_waters");

	// read the scaffold
	Pose start_pose;
	{
		std::string const filename( scaffold+"_subset.pdb");

		pose_from_pdb( start_pose, filename, true, false, true );
	}
	int const nres( start_pose.total_residue() );

	// read the target sequence
	{
		Pose target_pose;

		std::string const filename( target+"_subset.pdb");

		pose_from_pdb( target_pose, filename, true, false, true );

		assert( target_pose.total_residue() == nres );
		for ( int i=1; i<= nres; ++i ) {
			if ( start_pose.is_protein(i) ) {
				assert( target_pose.is_protein(i) );
				start_pose.copy_sidechain( i, target_pose.res(i), target_pose.res_variant(i),
														 target_pose.full_coord()(1,1,i) );
			}
		}
	}

	set_basepairs( start_pose );

	start_pose.copy_to_misc(); // STUPID -- for designmap?

	// setup fold_tree
	{
		int const num_jump(2);
		FArray1D_int cuts(num_jump);
		FArray2D_int jumps(2,num_jump);
		jumps(1,1) = 11;
		jumps(2,1) = 22;
		jumps(1,2) = 22;
		jumps(2,2) = 29;


		cuts(1) = 18;
		cuts(2) = 25;

		Fold_tree f;
		f.tree_from_jumps_and_cuts( nres, num_jump, jumps, cuts );
		f.reorder( 22 );
		start_pose.set_fold_tree(f);
	}



	FArray1D_bool hydrate( nres, false ), interface( nres, false );
	for ( int i=1; i<= nres; ++i ) {
		if ( start_pose.is_DNA(i) ) hydrate(i)= true;
		else interface(i) = true;
	}

	// setup the freakin design_map
	DesignMap design_map( nres, start_pose.res() );
	{
		using namespace aaproperties_pack;

		FArray1D_bool default_variant_setting( aaproperties_pack::number_aav_type,
																					 false );

		// setup the variants
		default_variant_setting( aaproperties_pack::aav_base ) = true;
		//default_variant_setting( aaproperties_pack::aav_his_taut ) = true;
		if ( use_h2o ) {
			default_variant_setting( aaproperties_pack::aav_water ) = true;
		}
		design_map.set_variant_defaults( default_variant_setting );

		// this design_map.reset allows the default variants!
		for ( int seqpos = 1; seqpos <= nres; ++seqpos ) {
			design_map.fix_sidechain_allowing_variants( seqpos );

			if ( repack_interface ) {
				// packing
				if ( interface( seqpos ) ) {
					design_map.set( seqpos, start_pose.res(seqpos) );
					design_map.require_base_rotamer( seqpos );
				} else if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}

			} else {
				// just hydrate
				if ( !hydrate(seqpos) ) {
					design_map.require_base_rotamer( seqpos );
				}
			}

		}
	}


	water::use_hbond_potential = true;


	// rotamer defaults
	pose_setup_packer();

	design::active_rotamer_options.set_ex12( truefalseoption("ex1"), truefalseoption("ex2"),
						truefalseoption("ex1aro") );

	PackerTask task( start_pose );
	FArray1D_bool allow_repack( nres, true );

	task.set_task( "packrot", false/*make-outputfile*/, allow_repack,
								 inc_current );
	task.set_designmap( design_map);

	task.set_nloop( 100 );
	prof::reset( true );
	start_pose.pack( task );
	prof::show();


	for ( int n=1; n<= nstruct; ++n ) {
		Pose pose;
		pose = start_pose;

		std::string const tag( "bzip_swap_"+scaffold+"_to_"+target+string_of(n) );

#ifdef GL_GRAPHICS
		gl_graphics_set_title( tag );
#endif


		// only the rigid-body and the protein sidechains
		pose.set_allow_jump_move( false );
		pose.set_allow_jump_move( 1, true );
		for ( int i=1; i<= nres; ++i ) {
			if ( pose.is_protein(i) ) {
				pose.set_allow_bb_move ( i, false );
				pose.set_allow_chi_move( i, true );
			} else {
				pose.set_allow_bb_move ( i, false );
				pose.set_allow_chi_move( i, false );
			}
		}

		fast_dna_relax( pose, 25 );

		std::string dna_seq;
		{
			for ( int i=21; i<= 23; ++i ) {
				dna_seq += param_aa::aa_name3( pose.res(i) )[2];
			}
		}

		pose.set_extra_score( dna_seq, 1.0 );

		pose.dump_scored_pdb( files_paths_pdb_out_prefix()+tag+".pdb",
													Score_weight_map( score12) );

		// undo the seq hack:
		pose.unset_extra_score( dna_seq );

	} // n=1,nstruct
}
///////////////////////////////////////////////////////////////////////////////

void
dna_test()
{

	if ( truefalseoption("bzip_relax") ) {
		bzip_relax_test();
		exit(0);
	} else if ( truefalseoption("bzip_swap") ) {
		bzip_swap_test();
		exit(0);
	}

	bzip_test();
	exit(0);

	bzip_relax_test();
	exit(0);

	bzip_test();
	exit(0);

//	zf_test();

//	compare_decoys();

	dna_bs_deriv_test();
	exit(0);

	dna_bp_deriv_test();
	exit(0);

	dna_test_old();
	exit(0);

	dna_test_old();
	exit(0);

	jim_test();

	winged_dock_test();

	test_dna_scores();

	dna_centroid_docking();

	collect_dna_stats();

	std::exit( 0 );

}

