// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revislion: 13847 $
//  $Date: 2008-03-20 14:27:59 -0700 (Thu, 20 Mar 2008) $
//  $Author: jiangl $


// Rosetta Headers
#include "pose_io.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "enzyme.h"
#include "enzyme_ns.h"
#include "files_paths.h"
#include "fullatom.h"
#include "fullatom_energies.h"
//#include "gb_elec_ns.h"
#include "gl_graphics.h"
#include "input_pdb.h"
#include "jumping_refold.h"
#include "ligand.h"
#include "make_pdb.h"
#include "misc.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdb.h"
#include "pose.h"
#include "pose_ligand.h"
#include "output_decoy.h"
#include "ramachandran.h"
#include "read_aa_ss.h"
#include "read_aaproperties.h"
#include "structure.h"
#include "template_pack.h"
#include "score_ns.h"
#include "util_basic.h"

#include "misc_removal.h"

#include "prof.h" //diagnostics REMOVE!

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
//#include <algorithm>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <string>



///////////////////////////////////////////////////////////////////////////////

namespace pose_io {
	//enum BVAL_SRC { BORN_RADIUS, CHARGE, PDB_BVAL };
	BVAL_SRC bval_src( CHARGE );
}
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
//
void
pose_from_misc(
	pose_ns::Pose & pose,
	bool const fullatom,
	bool const ideal_pos,
	bool const coords_init
	)
{
	//std::cout << "pose_from_misc:: ideal_pos: " << ideal_pos <<" coords_init: "
	//					<< coords_init << " fullatom: " << fullatom << std::endl;

	if ( pose.total_residue() != misc::total_residue ) {
		// DANGER: set a simple fold tree
		//std::cout << "pose_from_misc:: setting a simple fold_tree" << std::endl;
		pose.simple_fold_tree( misc::total_residue );
	}

	// take the fullatom setting from ROSETTA:
	pose.set_fullatom_flag ( fullatom, false ); // no repack

	// fill in backbone torsions, sequence
	int const total_residue ( pose.total_residue() );
	for ( int i=1; i<= total_residue; ++i ) {
		pose.set_phi         ( i, misc::phi(i) );
		pose.set_psi         ( i, misc::psi(i) );
		pose.set_omega       ( i, misc::omega(i) );
		pose.set_secstruct   ( i, misc::secstruct(i) );
		pose.set_name        ( i, misc::name(i) );
		pose.set_res         ( i, misc::res(i) );
		pose.set_res_variant ( i, misc::res_variant(i) );
		// now pose.new_torsions() will be true...
	}

	//lin fill the pdb info
	Pdb_info pdb_info;
	pdb_info.pdb_info_from_global();
	pose.set_pdb_information( pdb_info );

  if( pose.total_residue()==0 && get_read_ligaa_flag() ) return;//skip if only ligand

	// fill in the coordinates
	if ( coords_init ) {
		// this will recalculate the torsion angles:
		pose.set_coords( ideal_pos, misc::Eposition, misc::full_coord, true);

		//copy_misc_position_to_pose( pose );
		// set new_XXX_refold = false to prevent initial refold
		// alse sets new_bb_score = true to guarantee full score calculation
		// also calls update_chi if fullatom=true
		//
		// PHIL: think about this: when do we want update_chi to be
		// called???
		//pose.accept_coordinates();
	} else {
		if ( !ideal_pos ) {
			std::cout << "STOP:: pose_from_misc: cant have ideal_pos=F and " <<
				"coords_init=F" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

#ifdef GL_GRAPHICS
	gl_graphics_new_pose(misc::total_residue,true,1);//update the pose frame to show
#endif
}

void
fullatom_nonideal_initialized_pose_from_misc( pose_ns::Pose & pose )
{
	bool const fullatom( true );
	bool const ideal_pose( false ); // non-ideal backbone geometry
	bool const coords_init( true );

	pose_from_misc(pose, fullatom, ideal_pose, coords_init);
}


///////////////////////////////////////////////////////////////////////////////
void
read_ligand(
						std::string const & filename
						)
{
	assert( get_ligand_flag() || get_enable_ligaa_flag() );
	utility::io::izstream data( filename );
	if( truefalseoption("mdlfile") ) {
		utility::io::izstream lig_iunit;
		lig_iunit.clear();
		lig_iunit.open( stringafteroption("mdlfile").c_str() );
	  delete ligand::ligand_one;
	  ligand::ligand_one = new Ligand();
		read_mdl( lig_iunit,(ligand::ligand_one));
	} else {
		read_pdb_hetero( data,(*ligand::ligand_one) );
	}
	data.close();

}

///////////////////////////////////////////////////////////////////////////////
bool
pose_from_pdb(
	pose_ns::Pose & pose,
	std::string const & filename,
	bool const fullatom,
	bool const ideal_pose,
	bool const read_all_chains, // = false
	char const chain, // = '-'
	bool const skip_missing, // = false
	bool const allow_missing // = false
)
{
	utility::io::izstream data( filename );

	if ( !data().good() ) {
		std::cout << "unable to open pdb file: " << filename << std::endl;
		return false; // failed
	}

	// set some i/o params used by input_pdb
	bool const save_pose_flag( pose_flag() );
	set_pose_flag( true ); // currently, this is to catch update_sequence calls

	files_paths::protein_chain = chain; // read first chain
	files_paths::read_all_chains = read_all_chains; //
	//files_paths::multi_chain = false; //lin enable multi_chain by flag
	files_paths::skip_missing = skip_missing; //
	files_paths::allow_missing = allow_missing; //
	bool fail;
	input_pdb( data, false, fullatom, fail ); // seq_defined=F

	if ( fail ) {
		std::cout << "input_pdb failed: " << filename << std::endl;
		set_pose_flag( save_pose_flag );
		return false;
	}

	// try reading ss
	read_pdb_phipsi( data, fail );
	data.close();
	data.clear();

	if ( fail && misc::total_residue > 0 ) {
		bool defined( false );
		misc::secstruct = ' '; //signal to dssp that we need to read from file

		std::string dssp_filename( filename );
		std::string::size_type const pos( dssp_filename.find( ".pdb" ) );
		if ( pos != std::string::npos ) {
			dssp_filename.erase( pos );
			dssp_filename = dssp_filename + ".dssp";

			// this is such a hack:
			read_dssp( dssp_filename, misc::total_residue,
			 misc::residue1, files_paths::protein_chain, misc::secstruct, defined );
		}
		if ( !defined ) {
			std::cout << "calculating secondary structure from torsion angles" << std::endl;
			ss_from_phipsi( misc::secstruct, misc::phi, misc::psi, misc::total_residue);
		}
	}

	//lin if multi_chain, set up the part_begin and part_end
	setup_part_from_domains();

	//// yab: misc removal
	// see function definition above, ensure initialization of certain new classes
	// Please DO NOT add anything here without consulting yab, as this
	// functions in this code block are scheduled for removal.
	misc_removal::init_modules_TEMPORARY();
	//// yab: misc removal

	pose_from_misc( pose, fullatom, ideal_pose, true ); // coords_init

	set_pose_flag( save_pose_flag );
	return true; // success

}


///////////////////////////////////////////////////////////////////////////////
// this is a total ugly hack, to minimize interference with current CVS
// code. In jumping_refactor I dont do this, b/c fullatom_energy() is different
// soon rosetta will be more objectified and we probably won't have to
// do this...
//
// called before scoring

void
copy_pose_fullatom_energies_to_best(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	if ( !pose.fullatom() ) return;

	int const nres( pose.total_residue() );

	Score_state atr_state, rep_state, sol_state, gb_state, unf_state,
		dun_state, prob_state, intrares_state, cst_state, bondangle_state;
  Score_state elec_state;

	// 2D:
	FArray2D_float const & atr_pair ( pose.get_2D_score( ATR_PAIR, atr_state ) );
	FArray2D_float const & rep_pair ( pose.get_2D_score( REP_PAIR, rep_state ) );
	FArray2D_float const & sol_pair ( pose.get_2D_score( SOL_PAIR, sol_state ) );
	FArray2D_float const & cst_pair ( pose.get_2D_score( CST_PAIR, cst_state ) );
  FArray2D_float const & elec_pair ( pose.get_2D_score( ELEC_PAIR, elec_state ) );

	// 1D:
	FArray1D_float const & resenergy
		( pose.get_1D_score( RESENERGY, dun_state ) ); //dont care about the state
	FArray1D_float const & dunenergy
		( pose.get_1D_score( DUNENERGY, dun_state ) );
	FArray1D_float const & probenergy
		( pose.get_1D_score( PROBENERGY, prob_state ) );
	FArray1D_float const & unfenergy
		( pose.get_1D_score( UNFENERGY, unf_state ) );
	FArray1D_float const & intraresenergy
		( pose.get_1D_score( INTRARESENERGY, intrares_state ) );
	FArray1D_float const & bondangleenergy
		( pose.get_1D_score( BONDANGLEENERGY, bondangle_state ) );

	fullatom_energies::best_protonation_energy = 0.0;

	if ( ! ( ( dun_state      == GOOD || dun_state      == OK ) &&
					 ( prob_state     == GOOD || prob_state     == OK ) &&
					 ( unf_state      == GOOD || unf_state      == OK ) &&
					 ( intrares_state == GOOD || intrares_state == OK ) &&
					 ( bondangle_state == GOOD || bondangle_state == OK ) &&
					 ( atr_state      == GOOD || atr_state      == OK ) &&
					 ( cst_state      == GOOD || cst_state      == OK ) &&
					 ( rep_state      == GOOD || rep_state      == OK ) &&
					 ( sol_state      == GOOD || sol_state      == OK ) )) { //&&
 //          ( elec_state     == GOOD || elec_state     == OK ) )) {
		// this will set res_moved, pair_moved to TRUE everywhere,
		// preventing copying from best
		pose.new_score_pose();
		return;
	}

	FArray2D_float & ap( fullatom_energies::best_atr_pair );
	FArray2D_float & rp( fullatom_energies::best_rep_pair );
	FArray2D_float & sp( fullatom_energies::best_sol_pair );
	FArray2D_float & cp( fullatom_energies::best_cst_pair );
  FArray2D_float & ep( fullatom_energies::best_elec_pair );

	for ( int i=1; i<= nres; ++i ) {
		fullatom_energies::best_dunenergy(i) = dunenergy(i);
		fullatom_energies::best_probenergy(i) = probenergy(i);
		fullatom_energies::best_unfenergy(i) = unfenergy(i);
		fullatom_energies::best_intraresenergy(i) = intraresenergy(i);
		fullatom_energies::best_bondangleenergy(i) = bondangleenergy(i);
		fullatom_energies::best_resenergy(i) = resenergy(i);

		for ( int j = 1, l = atr_pair.index(1,i), m = ap.index(1,i); j <= nres;
		 ++j, ++l, ++m ) {
			ap[m] = atr_pair[l];
			rp[m] = rep_pair[l];
			sp[m] = sol_pair[l];
			cp[m] = cst_pair[l];
      ep[m] = elec_pair[l];
		}
	}

	// generalized born
	if ( param_pack::gen_born ) {
		FArray2D_float const & gb_pair
			( pose.get_2D_score( GB_PAIR, gb_state ) );

		if ( !( gb_state == GOOD || gb_state == OK ) ) {
			pose.new_score_pose();
			return;
		}

		FArray2D_float & gb( fullatom_energies::best_gb_elec_pair );

		for ( int i=1; i<= nres; ++i ) {
			for ( int j = 1, l = gb_pair.index(1,i), m = gb.index(1,i); j <= nres;
						++j, ++l, ++m ) {
				gb[m] = gb_pair[l];
			}
		}
	} else {
		// no gen_born
		for ( int i=1; i<= nres; ++i ) {
			for ( int j=1; j<= nres; ++j ) {
				fullatom_energies::best_gb_elec_pair(i,j) = 0.0;
			}
		}
	}

	// currently non-cached energies -- these scores dont work w/ pose
	// easy to add cacheing; I think Bin has done this in his version
	// for the CST energy. On "to do" list
	for ( int i=1; i<= nres; ++i ) {
		for ( int j=1; j<= nres; ++j ) {
			fullatom_energies::best_h2o_pair(i,j) = 0.0;
			fullatom_energies::best_h2o_hb_pair(i,j) = 0.0;
			//	fullatom_energies::best_cst_pair = 0.0;
		}
		//Isn't this a repeat of above?
		fullatom_energies::best_protonation_energy(i) = 0.0;
	}
}

///////////////////////////////////////////////////////////////////////////////
// called after scoring
//
void
copy_fullatom_energies_to_pose(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	if ( !pose.fullatom() ) return;

	int const nres( pose.total_residue() );

	Score_state dont_care;

	// 2D:
	FArray2D_float & atr_pair ( pose.set_2D_score( ATR_PAIR, dont_care ) );
	FArray2D_float & rep_pair ( pose.set_2D_score( REP_PAIR, dont_care ) );
	FArray2D_float & sol_pair ( pose.set_2D_score( SOL_PAIR, dont_care ) );
	FArray2D_float & cst_pair ( pose.set_2D_score( CST_PAIR, dont_care ) );
  FArray2D_float & elec_pair (pose.set_2D_score( ELEC_PAIR, dont_care ));

	// 1D:
	FArray1D_float & resenergy      ( pose.set_1D_score( RESENERGY     , dont_care ) );
	FArray1D_float & dunenergy      ( pose.set_1D_score( DUNENERGY     , dont_care ) );
	FArray1D_float & unfenergy      ( pose.set_1D_score( UNFENERGY     , dont_care ) );
	FArray1D_float & probenergy     ( pose.set_1D_score( PROBENERGY    , dont_care ) );
	FArray1D_float & intraresenergy ( pose.set_1D_score( INTRARESENERGY, dont_care ) );
	FArray1D_float & bondangleenergy ( pose.set_1D_score( BONDANGLEENERGY, dont_care ) );

	FArray2D_float & ap( fullatom_energies::atr_pair );
	FArray2D_float & rp( fullatom_energies::rep_pair );
	FArray2D_float & sp( fullatom_energies::sol_pair );
	FArray2D_float & cp( fullatom_energies::cst_pair );
  FArray2D_float & ep( fullatom_energies::elec_pair);

	for ( int i = 1; i <= nres; ++i ) {
		dunenergy     (i) = fullatom_energies::dunenergy      (i);
		unfenergy     (i) = fullatom_energies::unfenergy      (i);
		probenergy    (i) = fullatom_energies::probenergy     (i);
		intraresenergy(i) = fullatom_energies::intraresenergy (i);
		bondangleenergy(i) = fullatom_energies::bondangleenergy (i);
		resenergy     (i) = fullatom_energies::resenergy      (i);

		for ( int j = 1, l = atr_pair.index(1,i), m = ap.index(1,i); j <= nres;
		 ++j, ++l, ++m ) {
			atr_pair[l] = ap[m];
			rep_pair[l] = rp[m];
			sol_pair[l] = sp[m];
			cst_pair[l] = cp[m];
      elec_pair[l] = ep[m];
		}
	}

	if ( param_pack::gen_born ) {
		FArray2D_float & gb_pair ( pose.set_2D_score( GB_PAIR, dont_care ) );

		FArray2D_float & gb( fullatom_energies::gb_elec_pair );
		for ( int i = 1; i <= nres; ++i ) {
			for ( int j = 1, l = gb_pair.index(1,i), m = gb.index(1,i); j <= nres;
						++j, ++l, ++m ) {
				gb_pair[l] = gb[m];
			}
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
void
dump_Eposition(
	pose_ns::Pose const & pose,
	const std::string tag,
	bool const overwrite // = false
)
{
	static std::map< std::string, int > count;

	std::string filename( files_paths::query_defined ?
												files_paths::code + files_paths::protein_name : "" );
	filename = filename+tag;
	const int n( ++count[filename] );
	if ( n > 1 && !overwrite ) {
		filename = filename + '_' + lead_zero_string_of( n, 5);
	}

	utility::io::ozstream out( filename.c_str() );
	if (!out.good() ) {
		std::cout << "STOP: cant open file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	dump_Eposition( pose, out );
	out.close();
}

///////////////////////////////////////////////////////////////////////////////
void
dump_Eposition(
	pose_ns::Pose const & pose,
	std::ostream & out
)
{
	FArray1D_string atom_name(5);
	atom_name( 1 ) = " N  ";
	atom_name( 2 ) = " CA ";
	atom_name( 3 ) = " CB ";
	atom_name( 4 ) = " C  ";
	atom_name( 5 ) = " O  ";

	// array to rearrange backbone atoms
	FArray1D_int reorder(5);
	reorder( 1 ) = 1;
	reorder( 2 ) = 2;
	reorder( 3 ) = 4;
	reorder( 4 ) = 5;
	reorder( 5 ) = 3;

	float occ(1.0),bfactor(0.0);
	int atom_number(0);
	char chain_out = ' ';
	char insert_let_out = ' ';

	const int nres( pose.total_residue_for_scoring() );
	const FArray3D_float Eposition( pose.Eposition() );
	const FArray2D_float centroid( pose.centroid() );

	for ( int seqpos = 1; seqpos<= nres; ++seqpos ) {
		const int aa( pose.res(seqpos) );
		for ( int i = 1; i <= 5; ++i ) { // loop over atoms in each residue
			const int ii( reorder(i) );
			++atom_number; // want predictable atom numbers
			if ( aa != param_aa::aa_gly || ii != 3 ) {
				out << "ATOM  " << I( 5, atom_number ) << ' ' <<
					A( 4, atom_name(ii) ) << ' ' << param_aa::aa_name3(aa) << ' ' <<
					chain_out << I( 4, seqpos ) << insert_let_out << "   " <<
					F( 8, 3, Eposition(1,ii,seqpos) ) <<
					F( 8, 3, Eposition(2,ii,seqpos) ) <<
					F( 8, 3, Eposition(3,ii,seqpos) ) <<
					F( 6, 2, occ ) << F( 6, 2, bfactor ) << '\n';
			}                  // atoms
		}

		if (files_paths::output_centroids) {
			++atom_number;
			out << "ATOM  " << I( 5, atom_number ) << ' ' <<
				A( 4, " CEN" ) << ' ' <<  param_aa::aa_name3(aa)  << ' ' <<
				chain_out << I( 4, seqpos ) << insert_let_out << "   " <<
				F( 8, 3, centroid(1,seqpos) ) <<
				F( 8, 3, centroid(2,seqpos) ) <<
				F( 8, 3, centroid(3,seqpos) ) <<
				F( 6, 2, occ ) << F( 6, 2, bfactor ) << std::endl;
		}

	} // residues

	out << "TER\n";;
}


///////////////////////////////////////////////////////////////////////////////
void
output_CA_pdb(
	int const nres,
	FArray1D_int const & res,
	FArray1D_int const &, //res_variant,
	FArray3D_float const & xyz, // Epos or full_coord
	utility::io::orstream & out
)
{

	for ( int seqpos = 1; seqpos<= nres; ++seqpos ) {
		const int aa( res(seqpos) );
		out << "ATOM  " << I( 5, seqpos ) << ' ' <<
			A( 4, " CA " ) << ' ' << param_aa::aa_name3(aa) << ' ' <<
			' ' /*chain_ou*/ << I( 4, seqpos ) << ' ' << "   " <<
			F( 8, 3, xyz(1,2,seqpos) ) <<
			F( 8, 3, xyz(2,2,seqpos) ) <<
			F( 8, 3, xyz(3,2,seqpos) ) <<
			F( 6, 2, 1.0 /*occ*/ ) << F( 6, 2, 1.0 /*bfactor*/ ) << '\n';
	}





}
///////////////////////////////////////////////////////////////////////////////
void
open_full_coord_outfile(
	utility::io::ozstream & out,
	std::string const & tag,
	bool const overwrite // = false
)
{
	static std::map< std::string, int > count;

	// construct filename from tag
	std::string filename;
	if ( files_paths::query_defined ) {
		filename = filename + files_paths::code + files_paths::protein_name;
	}
	filename = filename+tag;
	const int n( ++count[filename] );
	if ( !overwrite ) {
		filename = filename + '_' +
		           lead_zero_string_of( n, files_paths::NUMBER_MAX_LENGTH );
	}
	filename += ".pdb";
	// open file
	out.clear();
	out.open( filename.c_str() );
	if (!out.good() ) {
		std::cout << "STOP: cant open file: " << filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}
///////////////////////////////////////////////////////////////////////////////
void
dump_full_coord(
	pose_ns::Pose const & pose,
	std::string const & tag,
	bool const overwrite // = false
)
{
	utility::io::ozstream out;

	open_full_coord_outfile(out, tag, overwrite);

	// output coords
	dump_full_coord( pose, out );

	out << pose.show_scores() << "\n";

	// all done
	out.close();
	out.clear();
}

///////////////////////////////////////////////////////////////////////////////
void
dump_full_coord(
	pose_ns::Pose const & pose,
	std::ostream & out,
	std::string const extra_tag // = ""
)
{
	int atom_number(0);
	const int nres( pose.total_residue_for_scoring() );
	for ( int seqpos = 1; seqpos<= nres; ++seqpos ) {
		dump_rsd( seqpos, pose, atom_number, out, extra_tag );
		if ( ( pose.pdb_info().multi_chain() || get_enable_ligaa_flag() )
				 && seqpos == pose.pdb_info().docking_part_end()(1) ) {
			out << "TER\n" ;
		}
	}

	if ( extra_tag.length() < 1 ) 	out << "TER\n" ;
}


///////////////////////////////////////////////////////////////////////////////
void
dump_rsd(
	int const seqpos,
	pose_ns::Pose const & pose,
	int & atom_number,
	std::ostream & out,
	std::string const extra_tag // = ""
)
{
	const int aa( pose.res(seqpos));
	const int aav( pose.res_variant(seqpos));
	const int natoms( aaproperties_pack::natoms(aa,aav));
	const FArray3D_float & xyz( pose.full_coord() );
	char chain_out = ' ';
	char insert_let_out = ' ';
	std::string atomname;
	std::string resname;
	std::string tag = " ";
	int res_number_out;

	if( !param_aa::is_ligand(pose.res(seqpos)) ) {
		tag = "ATOM  ";
	}else{//lin tag for ligand
		tag = "HETATM";
	}

  //lin enable use_pdb_numbering
  if ( pose.pdb_info().use_pdb_numbering() &&
			 !param_aa::is_ligand(pose.res(seqpos)) ) {
    res_number_out = pose.pdb_info().pdb_res_num(seqpos);
    insert_let_out = pose.pdb_info().pdb_insert_let(seqpos);
    chain_out = pose.pdb_info().res_chain(seqpos);
  } else {
    res_number_out = seqpos;
  }

	FArray2D_float const & pdb_bvalue ( pose.pdb_info().bvalue() );

	name_from_num(aa,resname);
	for ( int j=1; j<= natoms; ++j ) {
		atom_name_from_atom_num(j,aa,aav,atomname);
		++atom_number;
		float occ(1.0);
		float bvalue(1.0);
		if ( pose_io::bval_src == pose_io::BORN_RADIUS ) {
			bvalue = template_pack::born_radius(j,seqpos);
		} else if ( pose_io::bval_src == pose_io::CHARGE ) {
			bvalue = 50.0 + aaproperties_pack::atomic_charge( j,aa,aav);
		} else if ( pose_io::bval_src == pose_io::PDB_BVAL ) {
			bvalue = pdb_bvalue( j, seqpos );
		}
		if( param_aa::is_ligand(pose.res(seqpos)) ) {
			atom_number=j;
			if( enzyme::read_occ_weights ){
				occ = enable_ligaa_ns::get_ligaa_iocc_weight(aa)(j);
			}
			if( enzyme::read_bvalue_charge ){
				bvalue = enable_ligaa_ns::get_ligaa_icharge(aa)(j);
			}
		}

		out << tag << I( 5, atom_number ) << ' ' <<
			atomname << ' ' << resname << ' ' << chain_out << I( 4, res_number_out ) <<
			insert_let_out << "   " << F( 8, 3, xyz(1,j,seqpos) ) <<
			F( 8, 3, xyz(2,j,seqpos) ) << F( 8, 3, xyz(3,j,seqpos) ) <<
			F( 6, 2, occ ) << ' ' << F( 5, 2, bvalue ) << extra_tag << '\n';
	}

}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////
// note: this doesnt do any score calculation
// and there's no filtering here

void
output_decoy_pose(
	pose_ns::Pose const & pose
)
{
	// doesnt take a const pointer!
	pose.copy_to_misc();

	std::string filename;
	retrieve_decoy_name(filename);
	std::string fullname( files_paths::pdb_out_path + filename );

	std::cout<<"output the file: "<<fullname<<std::endl;

	utility::io::ozstream pdb_out_stream( fullname );

	if ( !pdb_out_stream ) {
	  std::cout << "Open failed for file: " << fullname  << std::endl;
	  utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	make_pdb( pdb_out_stream, pose.fullatom() );

	pdb_out_stream.close();
	pdb_out_stream.clear();
	remove( (fullname+".in_progress").c_str() );

}

///////////////////////////////////////////////////////////////////////////////
void
output_torsions(
	pose_ns::Pose const & pose,
	std::ostream & out
)
{
	bool const start (  pose.start_pose_exists() );
	bool const native( pose.native_pose_exists() );

	pose_ns::Pose const & start_pose ( start  ? pose.start_pose()  : pose );
	pose_ns::Pose const & native_pose( native ? pose.native_pose() : pose );

	int const total_residue( pose.total_residue() );

	// retrieve cutpoint info
	int n_cut;
	FArray1D_int const & fold_tree_cutpoint(
		pose.fold_tree().get_fold_tree_cutpoint( n_cut ) );
	FArray1D_bool const & is_cutpoint(
		pose.fold_tree().get_is_cutpoint() );
	FArray1D_float cut_weight( n_cut,
		scorefxns::jmp_chainbreak_weight == 0.0 ? 0.0 : 1.0 );
	if( cut_weight.size1() == scorefxns::cut_weight.size1() )
		cut_weight = scorefxns::cut_weight;

	FArray1D_bool eval_rama(total_residue, true );
	eval_rama(1) = false;
	for ( int i = 2; i < total_residue; i++ ) {
		if ( is_cutpoint(i) ) {
			eval_rama(i) = false;
			eval_rama(i+1) = false;
		}
	}
	eval_rama(total_residue) = false;
	for ( int i = 1; i <= n_cut; ++i ) {
		if ( cut_weight(i) != 0.0 ) {
			int j = fold_tree_cutpoint(i);
			eval_rama(j) = true;
			eval_rama(j+1) = true;
		}
	}
//car 'complete' is the tag searched for to read in torsion angles
//car from rosetta files and should be the last line before the angles begin
	out << "complete: res ss phi psi omega frag rama seq ";
	if ( start ) {
		out << " dstart_tor dstart_rama";
	}
	if ( native ) {
		out << " dnat_tor dnat_rama";
	}
  out << '\n';
	for ( int i = 1; i <= total_residue; ++i ) {
		if( param_aa::is_ligand(pose.res(i)) ) continue;
		float rama(0.0), tmp1, tmp2;
		if ( eval_rama(i) ) eval_rama_score_residue( pose.res(i), pose.phi(i),
			pose.psi(i), pose.secstruct(i), rama, tmp1, tmp2 );

		float start_dev(0.0), start_rama(0.0), native_dev(0.0), native_rama(0.0);

		if ( start ) {
			start_dev =
				( std::abs(subtract_degree_angles(pose.phi  (i),start_pose.phi  (i))) +
					std::abs(subtract_degree_angles(pose.psi  (i),start_pose.psi  (i))) +
					std::abs(subtract_degree_angles(pose.omega(i),start_pose.omega(i))));

			if( eval_rama(i) ) eval_rama_score_residue( start_pose.res(i),
				start_pose.phi(i), start_pose.psi(i), start_pose.secstruct(i),
				start_rama, tmp1, tmp2 );
		}

		if ( native ) {
			native_dev =
				(std::abs(subtract_degree_angles(pose.phi  (i),native_pose.phi  (i))) +
				 std::abs(subtract_degree_angles(pose.psi  (i),native_pose.psi  (i))) +
				 std::abs(subtract_degree_angles(pose.omega(i),native_pose.omega(i))));

			if ( eval_rama(i) ) eval_rama_score_residue( native_pose.res(i),
				native_pose.phi(i), native_pose.psi(i), native_pose.secstruct(i),
				native_rama, tmp1, tmp2 );
		}

		std::string name( pose.name(i) );
		if ( name.size() != 4 ) name = "----";

		out << I( 4, i ) << ' ' << pose.secstruct(i) << ' ' <<
			F( 9, 3, pose.phi(i) ) << F( 9, 3, pose.psi(i) ) <<
			F( 9, 3, pose.omega(i) ) << ' ' <<
			name << ' ' << F( 9, 2, rama ) << ' ' <<
			param_aa::aa_name1( pose.res(i) );

		if ( start ) out << F( 9, 3, start_dev ) << F( 9, 3, rama - start_rama );
		if ( native) out << F( 9, 3, native_dev) << F( 9, 3, rama - native_rama);
		out << '\n';
	}
}

///////////////////////////////////////////////////////////////////////////////


