// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $


#ifndef INCLUDED_pose_jjh_loops
#define INCLUDED_pose_jjh_loops

#include <vector>

#include "dna_motifs_ns.h"
#include "dna_motifs.h"
#include "pose_fwd.h"
#include "fold_tree.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray3D.hh>

void dna_jjh_loops();

void dna_jjh_refine();

///jjh Create an abstract loop design template class, in the spirit
///jjh of the Template Design Pattern rather than the type
///jjh parameterization template facility of C++.  The idea is to
///jjh sketch out the guts of how a loop design would work,
///jjh compartmentalizing problem-specific parts as class methods which
///jjh are intended to be defined/overwritten by subclasses.  The
///jjh parent class defines the invariants of loop design protocols:
///jjh cycling through a library of loops, perhaps permuting them; selecting
///jjh promising loops using problem specific criteria which may require
///jjh specific data (e.g. screening loop backbones against a library of
///jjh inverse rotamers);  minimizing loop backbone positions if desired;
///jjh identifying positions to redesign/repack/hold fixed, again by
///jjh problem specific rules; performing a design using the loop backbone
///jjh and DesignMap generated previously;  filtering to see if it is
///jjh worthy of outputing, and perhaps doing so.

namespace LoopDesign {

	///jjh Simple class to store the coordinates, aa, aav for
	///jjh a loop region as well as a mask array that tells whether
	///jjh a given position in the loop has been 'set' by the motif
	///jjh matching process
	class LoopState {

	public:
		int start;
		int end;
		FArray3D_float coords;
		FArray1D_int   aa    ;
		FArray1D_int   aav   ;

		///jjh Lots of copy constructors and such, since this
		///jjh will be passed as an argument by _value_
		LoopState() {};
		LoopState(int start, int end);
		LoopState( const LoopState &ls );
		LoopState& operator=( const LoopState &ls );
		int size() const { return (end - start + 1) ; };

	};

	class LoopDesignProtocol {

	public:
		///jjh This is the invariant loop design protocol
		void do_design( pose_ns::Pose & pose );

		///jjh Set up a virtual destructor
		virtual ~LoopDesignProtocol( ) {};

	private:
		///jjh All the variable sub-methods that are called by the
		///jjh invariant protocol are declared here.  Some must
		///jjh be provided by subclasses, and so are declared
		///jjh pure virtual.  Some may be left as is - these generally
		///jjh provide a default behavior of 'do nothing', such as
		///jjh an optional DoMinimization()

			virtual void do_initialize( pose_ns::Pose & pose ) = 0;

		///jjh This need to be written, since they contain the problem-specific
		///jjh criteria for whether or not you bother to write a structure out

			virtual void do_loop_shooting( pose_ns::Pose & pose ) = 0;


	};

	class DNAMotifLoopDesign : public LoopDesignProtocol {

	public:
			DNAMotifLoopDesign() : initialized( false ) {};

	private:
		///jjh All the variable sub-methods that are called by the
		///jjh invariant protocol are declared here.  Some must
		///jjh be provided by subclasses, and so are declared
		///jjh pure virtual.  Some may be left as is - these generally
		///jjh provide a default behavior of 'do nothing', such as
		///jjh an optional DoMinimization()

		void do_initialize( pose_ns::Pose & pose );
		void do_loop_shooting( pose_ns::Pose & pose );
		void do_find_second_shell( pose_ns::Pose & pose );

		void try_for_more( pose_ns::Pose & pose, FArray1D_int setpos,
					LoopState ls, int depth );
		void copy_loop_state_to_pose( pose_ns::Pose & pose, LoopState &ls );
		LoopState get_loop_state_from_pose( pose_ns::Pose & pose );
		std::string make_loop_name( int nres, FArray1Da_int setpos );
		DNA_Motifs::RegionPair check_free_length( int nres, int mp, FArray1Da_int setpos );
		bool is_clash_free( pose_ns::Pose & pose, int seqpos,
					FArray1Da_int setpos, int aa, int aav, FArray2Da_float coords);
		void get_closest_pos_and_rmsd( pose_ns::Pose & pose, FArray1Da_int pos,
						FArray1Da_float rmsd, FArray1Da_float ca_dist );
		bool successful_loop_closure( pose_ns::Pose & pose, FArray1Da_int setpos,
							int try_mot, int seqpos, DNA_Motifs::RegionPair & free );

		///jjh Here are the DNA motif specific helper functions

		float check_bump_with_template( );

		///jjh Below follows the problem-specific data for this protocol

		int max_recursion_depth;
		bool initialized;
		int number_of_hits;
		float saveWgb_elec;

		std::vector<int> second_shell_positions;

		DNA_Motifs::DNAMotifRotamerLibrary inv_rot_lib;

		DNA_Motifs::RegionPair loop_region;
		DNA_Motifs::RegionPairLibrary omit_regions;
		DNA_Motifs::RegionPairLibrary target_regions;

		pose_ns::Fold_tree saved_fold_tree;

	};

}

void enable_gb( bool onoff );

void dna_loop_min();

#endif
