// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.1 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_ligand.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "docking_score.h"
#include "etable.h"
#include "enzyme_ns.h"
#include "cst_set.h"
#include "files_paths.h"
#include "hbonds_ns.h"
#include "ligand.h"
#include "kin_atom.h"
#include "nblist.h"
#include "minimize.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack_geom_inline.h"
#include "pack_fwd.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pose.h"
#include "pose_docking.h"
#include "pose_io.h"
#include "random_numbers.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/Time_Date.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>
#include <utility/io/ozstream.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>

// Types

namespace enable_ligaa_ns {

	int const MAX_LIGAND_ROTAMERS = { 600 };
	int const MAX_LIGAND_CHI = { 20 };
	//ligand_cartesian_rotamers(xyz, atom_index, rotnum, aa, aav)
	FArray5D_float ligand_cartesian_rotamers( 3, param::MAX_ATOM(), MAX_LIGAND_ROTAMERS, MAX_LIGAA(), 1, 0.0);
	FArray4D_float ligand_rotamer_chi_angles( MAX_LIGAND_CHI, MAX_LIGAND_ROTAMERS, MAX_LIGAA(), 1, 0.0);
	//kwk added duplicate chi arrays to circumvent the MAX_CHI cutoff which could create difficulties in
	//portion of the code such as in the DNA rotamers
	FArray2D< Ligand * > ligand_ptr_array( MAX_LIGAA(), param::MAX_AA_VARIANTS() );
	FArray3D< size_t > ligaa_ligand_atom_map( param::MAX_ATOM(), MAX_LIGAA(), param::MAX_AA_VARIANTS() );
	FArray2D< utility::vector0< int > > ligand_ligaa_atom_map( MAX_LIGAA(), param::MAX_AA_VARIANTS() );
	FArray2D_int ligand_nchi( MAX_LIGAA(), param::MAX_AA_VARIANTS() );
	FArray4D_int ligand_chi_atoms( 4, MAX_LIGAND_CHI, MAX_LIGAA(), param::MAX_AA_VARIANTS() ); // chi atoms; four for each nchi
	FArray4D_bool ligand_chi_required( MAX_LIGAND_CHI, param::MAX_ATOM(), MAX_LIGAA(), param::MAX_AA_VARIANTS() ); // chi needed to bld atm?
	FArray3D_int ligand_length_chi_rigid_atom_list( MAX_LIGAND_CHI, MAX_LIGAA(), param::MAX_AA_VARIANTS() );
	FArray4D_int ligand_chi_rigid_atom_list( param::MAX_ATOM(), MAX_LIGAND_CHI, MAX_LIGAA(), param::MAX_AA_VARIANTS() );
	FArray2D_int nligandrotamers( param::MAX_AA(), 1, 0);
	FArray3D_float lig_icoord( 3, param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
	FArray2D_float lig_iocc_weight( param::MAX_ATOM(), MAX_LIGAA(), 1.0 );
	FArray2D_float lig_icharge( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );

	namespace ligaa_atomEarrays {
		FArray2D_float ligaa_atrE( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
		FArray2D_float ligaa_repE( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
		FArray2D_float ligaa_solE( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
		FArray2D_float ligaa_hbE( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
		FArray2D_float ligaa_sasa5( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
		FArray2D_float ligaa_sasa14( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
		FArray2D_float ligaa_sasa_pack( param::MAX_ATOM(), MAX_LIGAA(), 0.0 );
// 		FArray1D_float ligaa_pairE( param::MAX_ATOM(), 0.0 );
// 		FArray1D_float ligaa_intraE( param::MAX_ATOM(), 0.0 );
// 		FArray1D_float ligaa_gbE( param::MAX_ATOM(), 0.0 );
// 		FArray1D_float ligaa_cstE( param::MAX_ATOM(), 0.0 );
// 		FArray1D_float ligaa_h2oE( param::MAX_ATOM(), 0.0 );
	}

  FArray2Da_float
  get_ligaa_icoord( int const LG ){
		if( !param_aa::is_ligand( LG ) ) {
			std::cout<<"get_ligaa_icoord LG is out of range"<<std::endl;
			assert(false);
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
			return lig_icoord( 1, 1, param_aa::has_ligand_no( LG ) );
		}
		return lig_icoord( 1, 1, 1 );
  }

  FArray1Da_float
  get_ligaa_iocc_weight( int const LG ){
		if( !param_aa::is_ligand( LG ) ) {
			std::cout<<"get_ligaa_iocc_weight LG is out of range"<<std::endl;
			assert(false);
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
			return lig_iocc_weight( 1, param_aa::has_ligand_no( LG ) );
		}
			return lig_iocc_weight( 1, 1 );
	}

  FArray1Da_float
  get_ligaa_icharge( int const LG ){
		if( !param_aa::is_ligand( LG ) ) {
			std::cout<<"get_ligaa_icharge LG is out of range"<<std::endl;
			assert(false);
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
			return lig_icharge( 1, param_aa::has_ligand_no( LG ) );
		}
			return lig_icharge( 1, 1 );
	}

	void
	read_ligand( std::string const & filename) {
		utility::io::izstream lig_iunit;
		lig_iunit.clear();
		lig_iunit.open( files_paths::start_path + filename.c_str() );
		read_ligand( lig_iunit );
		lig_iunit.close();
	}

  void
  read_ligand( utility::io::irstream & iunit )
  {
		delete ligand::ligand_one;
		ligand::ligand_one = new Ligand();
		delete ligand::ligand_two;
		ligand::ligand_two = new Ligand();
		utility::io::izstream lig_iunit;
		lig_iunit.clear();

		//lin read ligand from the file
    if( enzyme::read_ligaa_mdl ) {
      lig_iunit.open( files_paths::start_path + stringafteroption("mdlfile").c_str() );
      if( read_mdl( lig_iunit,(ligand::ligand_one)) == -1 ) {
        std::cout<<"enable_ligand_aa failed in reading from the mdlfile "
                 <<files_paths::start_path + stringafteroption("mdlfile").c_str()<<std::endl;
        assert(false);
        utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
      }
      // amw215 - if using mdl connect, then read hetero atom info from the pdbfile
      if ( enzyme::use_mdl_connect ) {
        read_pdb_hetero( iunit, (*ligand::ligand_two) );
        int nligatoms = ligand::ligand_one->atom_vector.size();
        for (int i=0; i < nligatoms; i++ ) {
          Atom *pAtom1 = ligand::ligand_one->atom_vector[i];
          Atom *pAtom2 = ligand::ligand_two->atom_vector[i];

					if ( ( pAtom1->get_element_type() != atom_chem::VIRTUAL || pAtom1->get_element_type() != atom_chem::VIRTUAL )
							 && pAtom1->get_atom_name() != pAtom2->get_atom_name()) {
						std::cout << "ligand ordering in mdl and pdb files differ" << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					}

          pAtom1->set_coordinates( pAtom2->get_coordinates() );
					pAtom1->set_occupancy( pAtom2->get_occupancy() );
        }
      }
      lig_iunit.close();
    } else {
			if( enzyme::read_lig2 ) {
				lig_iunit.open( files_paths::start_path + stringafteroption("lig2_file").c_str() );
				if( !lig_iunit ) {
					read_pdb_hetero( iunit,(*ligand::ligand_one), "LG1" );
				} else {
					read_pdb_hetero( iunit,(*ligand::ligand_one) );
				}
			} else {
				read_pdb_hetero( iunit,(*ligand::ligand_one) );
			}
    }

    //lin setup ligand aa type from get_ligand_one
    FArray2D_float lig_coord( 3, param::MAX_ATOM()() );
    setup_ligand_aa( param_aa::ligand_aa_vector[1], 1, (*ligand::ligand_one), lig_coord ); // aav=1

		//lin the number of the ligand as aa
		enzyme::ligaaN=1;

		if( enzyme::read_lig2 ) {
			if( !lig_iunit ) {
				read_pdb_hetero( iunit,(*ligand::ligand_two), "LG2" );
			} else {
				read_pdb_hetero( lig_iunit, (*ligand::ligand_two) );
			}
			setup_ligand_aa( param_aa::ligand_aa_vector[2], 1, (*ligand::ligand_two), lig_coord ); // aav=1
			enzyme::ligaaN++;
		}

		lig_iunit.close();

		////lin add the ligand into misc
		//enable_ligaa_ns::add_rsd_icoord_to_misc( param_aa::lig1 );
		//enable_ligaa_ns::add_rsd_icoord_to_misc( param_aa::lig2 );
  }

	void fill_ligaa_repEarray(
					 int const atom,
					 int const aa,
					 float const repE
					 )
	{
		using namespace param_pack;
		using namespace param_aa;

		if( !param_aa::is_ligand(aa) ) return;

		float temp = 0.5f * pack_wts.Wrep() * repE;
		ligaa_repE(atom, has_ligand_no(aa)) += temp;
	}

	void fill_ligaa_hbEarray(
					 int const atom1,
					 int const aa1,
					 bool const atom1_is_sc,
					 int const atom2,
					 int const aa2,
					 bool const atom2_is_sc,
					 float const hbE
					 )
	{
		using namespace param_pack;
		using namespace param_aa;

		if( !param_aa::is_ligand(aa1) && !param_aa::is_ligand(aa2) ) return;

		float weight = 1;
		if( atom1_is_sc && atom2_is_sc ) {
			weight = pack_wts.Whbond_sc();
		} else if( !atom1_is_sc && atom2_is_sc ) {
			weight = pack_wts.Whbond_bb();
		} else {
			weight = pack_wts.Whbond_bb_sc();
		}
		float temp = 0.5f * weight * hbE;
		if( param_aa::is_ligand(aa1) ) {
			ligaa_hbE(atom1, has_ligand_no(aa1)) += temp;
		}
		if( param_aa::is_ligand(aa2) ) {
			ligaa_hbE(atom2, has_ligand_no(aa2)) += temp;
		}
	}

	void fill_ligaa_pairEarrays(
					int const atom,
					int const aa,
					float const solE,
					float const atrE,
					float const repE
					)
	{
		using namespace param_pack;
		using namespace param_aa;

		if( !param_aa::is_ligand(aa) ) return;

		float temp = 0.5f * pack_wts.Watr() * atrE;
		ligaa_atrE(atom, has_ligand_no(aa)) += temp;

		temp = 0.5f * pack_wts.Wrep() * repE;
		ligaa_repE(atom, has_ligand_no(aa)) += temp;

		temp = 0.5f * pack_wts.Wsol() * solE;
		ligaa_solE(atom, has_ligand_no(aa)) += temp;
	}

	void fill_ligaa_sasa_arrays(
		int const atom,
		int const aa,
		float const sasa14,
		float const sasa5,
		float const sasa_pack
	)
	{
		using namespace param_aa;
		if( !param_aa::is_ligand(aa) ) return;

		ligaa_sasa5(atom, has_ligand_no(aa)) = sasa5;
		ligaa_sasa14(atom, has_ligand_no(aa)) = sasa14;
		ligaa_sasa_pack(atom, has_ligand_no(aa)) = sasa_pack;
	}
}

///////////////////////////////////////////////////////////////////////////////
///@begin ligand_docking_function
///@brief this function serves as the main driving function for a
///flexible ligand docking protocol whereby the ligand is perturbed by a given
///amount and then grandient minimized after a run through the packer with ligand
///rotamers.
///
///@author Kristian Kaufmann
///////////////////////////////////////////////////////////////////////////////
void
ligand_docking_function(
)
{

	std::vector< std::string > input_filenames;
	files_paths::require_start=true; //this states that input pdbs are required
	if( truefalseoption("l")){
		std::ifstream data( stringafteroption("l").c_str() );
		std::string line;
		while ( getline( data, line ) ) {

			unsigned int n = line.rfind(".pdb");
			line.replace(n,4,"");
			input_filenames.push_back(line);
		}
	}else if(truefalseoption("s")){
		std::string line;
		stringafteroption("s", "input_pdb", line);
		unsigned int n = line.rfind(".pdb");
		line.replace(n,4,"");
		input_filenames.push_back(line);
	}else{
		std::cout << "==========================================================" << std::endl;
		std::cout << "Did not find either -s or -l in commandline. Cannot continue without" << std::endl;
		std::cout << "start structure filename. Exiting now." << std::endl;
		std::cout << "==========================================================" << std::endl;

		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( truefalseoption("series") ) {
		stringafteroption( "series", "--", files_paths::code );
		if ( files_paths::code[0] == '-' ) {
			std::cout << "must specify 2 letter series code after -series" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
	for( std::vector< std::string >::iterator c_file=input_filenames.begin(); c_file!=input_filenames.end(); c_file++){

		std::string line=(*c_file);
		files_paths::output_file=(*c_file);
		// read in the starting structure
		// will trigger read_pdb_hetero
		pose_ns::Pose pose;
		bool const fullatom( true ), ideal_pose( false ), read_all_chains( true );
		//ligand flag and ligand_flexible flag and ligand_mdlfile flag are all assumed to be true in this instance
		ligand::ligand_flag=true;
		ligand::ligand_flexible=true;
		stringafteroption( "ligand_mdlfile", files_paths::start_path + line + ".mol" , files_paths::ligand_input_mdlfile);

		std::string input_filename = files_paths::start_path + line + ".pdb";
		std::cout << "input_file: " << input_filename << std::endl;
		pose_from_pdb( pose, input_filename, fullatom, ideal_pose, read_all_chains );


		// setup ligand aa type
		FArray3D_float lig_coord( 3, param::MAX_ATOM()(), enable_ligaa_ns::MAX_LIGAA() );
		//load the topological information in the mdlfile into the aaproperties file
		assert( param_aa::ligand_aa_vector.size() >= ligand::ligand_ptr_vector.size() );
		size_t ligaa=0;
		for(std::vector< Ligand * >::iterator cligand=ligand::ligand_ptr_vector.begin(); cligand!=ligand::ligand_ptr_vector.end();
			cligand++){

			ligaa++;
			(*cligand)->setup_ligand_res( param_aa::ligand_aa_vector[ligaa], 1, false, false, lig_coord );
			// aa,aav=1, read_weights from occ, read_weights from temperature, lig_coord
		}


		// no longer using ligand mode to manage the ligand
		// it has been setup as an amino acid now
		set_ligand_flag( false );
		ligand::ligand_flexible=false;
		ligand::ligand_mdlfile=false;


		//attach each ligand as directed in the mdlfile.
		// this is intended to allow multiple copies of same ligand to be represented in a protein ligand complex.
		// this requires the user to supply a transformation/rotation matrix to each ligand position.
		// this is done in the mdlfile by providing the coordinates of four atoms from the transformed position.
		// the four atoms assure a unique placement of the ligand. for further information see the read_mdl function
		// in ligand.cc under the MPROT anchor_residue option.
		// the default behavior is the assign one copy of the ligand anchored to the nearest CA atom.

		std::cout << "Now setting jumps from ligands to protein" << std::endl;

		ligaa=0;
		int ligands_added=0;
		for(std::vector< Ligand * >::iterator cligand=ligand::ligand_ptr_vector.begin(); cligand!=ligand::ligand_ptr_vector.end();
			cligand++){

			ligaa++;
			//first find the nearest CA for the input conformation
			numeric::xyzVector_float lig_root_xyz( &( lig_coord( 1, 1, ligaa)));

			int anchor_rsd=0;
			anchor_rsd=find_nearest_res( pose, lig_root_xyz);
			std::cout << "Anchor residue for ligand:" << anchor_rsd << " " << lig_root_xyz << std::endl;
			pose.attach_ligand( param_aa::ligand_aa_vector[ligaa],1,2,anchor_rsd,1,lig_coord(1,1,ligaa),true);
			pose.set_allow_jump_move( pose.num_jump(), true);
			ligands_added++;

			for ( size_t ligand_copy=0; ligand_copy < (*cligand)->ligand_copy_trans_rot.size(); ligand_copy++){
				//first job is to apply the translation rotation to the lig_coord array;
				//the translation rotation will be accomplished using the findUU_trans
				//and UU_rotate functions in orient_rms
				double sigma3=0.0;
				FArray1D_double weights(4,1.0);
				FArray1D_double lig_orig_trans(3,0.0);
				FArray1D_double ligand_trans_after(3,0.0);
				FArray2D_double trans_rot_matrix(3,3,0.0);
				FArray2D_double ligand_new(3,4,0.0);
				FArray2D_double ligand_original(3,4,0.0);
				FArray2D_float lig_coordinates_new(3,aaproperties_pack::natoms(param_aa::ligand_aa_vector[ligaa],1),0.0);
				for(size_t atom_ind=0;atom_ind<4;atom_ind++){
					ligand_original(1,int(atom_ind+1))=
							lig_coord(1,enable_ligaa_ns::ligand_ligaa_atom_map(ligaa,1)[(*cligand)->ligand_copy_trans_rot[ligand_copy][atom_ind].first], ligaa);
					ligand_new(1,int(atom_ind+1))=(*cligand)->ligand_copy_trans_rot[ligand_copy][atom_ind].second.x();
					ligand_original(2,int(atom_ind+1))=
							lig_coord(2,enable_ligaa_ns::ligand_ligaa_atom_map(ligaa,1)[(*cligand)->ligand_copy_trans_rot[ligand_copy][atom_ind].first], ligaa);
					ligand_new(2, int(atom_ind+1))=(*cligand)->ligand_copy_trans_rot[ligand_copy][atom_ind].second.y();
					ligand_original(3,int(atom_ind+1))=
							lig_coord(3,enable_ligaa_ns::ligand_ligaa_atom_map(ligaa,1)[(*cligand)->ligand_copy_trans_rot[ligand_copy][atom_ind].first], ligaa);
					ligand_new(3,int(atom_ind+1))=(*cligand)->ligand_copy_trans_rot[ligand_copy][atom_ind].second.z();
				}
				for( int atom_ind=1; atom_ind<=aaproperties_pack::natoms(param_aa::ligand_aa_vector[ligaa],1); atom_ind++){
					lig_coordinates_new(1,atom_ind)=lig_coord(1,atom_ind,ligaa);
					lig_coordinates_new(2,atom_ind)=lig_coord(2,atom_ind,ligaa);
					lig_coordinates_new(3,atom_ind)=lig_coord(3,atom_ind,ligaa);
				}
				findUU_trans(ligand_original, ligand_new, weights, 4, trans_rot_matrix, sigma3, lig_orig_trans, ligand_trans_after);

				UU_rotate(lig_coordinates_new, aaproperties_pack::natoms(param_aa::ligand_aa_vector[ligaa],1),lig_orig_trans, ligand_trans_after, trans_rot_matrix);
				numeric::xyzVector_float lig_root_xyz( &( lig_coordinates_new( 1, 1)));
				int anchor_rsd=0;
				anchor_rsd=find_nearest_res( pose, lig_root_xyz);
				std::cout << "anchor residue:" << anchor_rsd << " " << lig_root_xyz << std::endl;
				pose.attach_ligand( param_aa::ligand_aa_vector[ligaa],1,2,anchor_rsd,1,lig_coordinates_new(1,1),true);
				pose.set_allow_jump_move( pose.num_jump(), true);
				ligands_added++;
			}//ligand_copy_translation_rotation

		}//for ligand
		//pose.dump_pdb( "test1.pdb" );

		int num_output_structures=1;
		if(truefalseoption("nstruct")){
			intafteroption("nstruct",1,num_output_structures);
		}

		for ( int current_structure=1; current_structure <= num_output_structures; current_structure++ ){
			pose_ns::Pose start_pose;
			start_pose=pose;
			bool structure_exists=true;
			std::string outfilename;
			check_decoy_exists("dock", 1, current_structure, structure_exists, outfilename);
			std::cout << "Next Structure "<< outfilename << std::endl;
			if( !structure_exists){
				ligand_docking_protocol1( start_pose, outfilename);
			}
		}
	}
	exit(0);

}

////////////////////////////////////////////////////////////////////////////////
///@begin ligand_docking_protocol1
///@brief first ligand docking protocol perturbs the ligands by the parameters
///passed on commandline to after the flag ligand_dock_pert following that the
///perturbation is refined by 0.2 Angstrom translation 5 degree rotations in
///a metropolis monte carlo search with the centroid scoring function. Following
///the low resolution search a fullatom sidechain repack of the protein with
///ligand rotamers is performed.
///@param [in] Pose & pose
///@param [in] int structure
////////////////////////////////////////////////////////////////////////////////
int
ligand_docking_protocol1(
	pose_ns::Pose & pose,
	std::string const filename
){

	//Now need to perturb docking positions of ligands
	float trans_delta, rot_delta, perturb_ligand_trans, perturb_ligand_rot;
	int pert_cycles;
	real2afteroption("ligand_dock_pert", 1.0, trans_delta, 10, rot_delta);
	intafteroption("ligand_dock_pert_cycles", 5, pert_cycles);
	real2afteroption("perturb_ligand_rotamers", 0.0, perturb_ligand_trans, 0.0, perturb_ligand_rot);
	pose_ns::Score_weight_map wt_map( score10d );
	pose_ligand_docking_rigid_body_trial( pose, pert_cycles, wt_map, trans_delta, rot_delta, perturb_ligand_trans, perturb_ligand_rot );

// now try minimizing
// first set up what is allowed to vary

	pose.set_allow_move( kin::ALL, false );

	bool const minimize_ligand_bonds   ( truefalseoption("bonds") );
	bool const minimize_ligand_angles  ( truefalseoption("angles") );
	bool const minimize_ligand_torsions( truefalseoption("torsions") );

	// protein backbone
	pose.set_allow_bb_move( truefalseoption("bb") );

	// protein chi angles
	pose.set_allow_chi_move( truefalseoption("chi") );

	// only one jump: ligand rigid-body degrees of freedom
	pose.set_allow_jump_move( truefalseoption("jump") );


	// params
		// note that bond angles and torsions are stored in the atom_tree
		// as radians, hence the degrees(1.0) term in the tether weights
		// below -- this converts the tether score to degrees, which
		// is easier to think about
		//
		//int const lig_root_atomno( 1 ); // 1st hetero heavyatom in pdb file
		float const bond_length_tether( 10000.0 );
		float const bond_angle_tether( numeric::constants::f::rad2deg*1.0 * numeric::constants::f::rad2deg*1.0 * 10.0 );
		float const bond_torsion_tether( numeric::constants::f::rad2deg * 1.0 * numeric::constants::f::rad2deg*1.0 );
		//float const bond_length_tether( 0.01 );
		//float const bond_angle_tether( 0.01 );
		//float const bond_torsion_tether( 0.01 );

	// ligand internal dof
	// setup a constraint set to tether these to current values
	for(int lig_rsd = 1; lig_rsd <= pose.total_residue();lig_rsd++){

		if(!param_aa::is_ligand(pose.res(lig_rsd))){continue;}
		cst_set_ns::Cst_set cst_set;
		pose.set_constraints( cst_set );
		int const natoms( aaproperties_pack::natoms( pose.res(lig_rsd), 1 ) );
		for ( int i=2; i<= natoms; ++i ) {

			kin::Atom_id atom( i, lig_rsd );

			if ( minimize_ligand_bonds ) { // bond length
				kin::Torsion_id this_bond_length( atom, kin::D );
				float const current_val( pose.get_atom_tree_torsion(this_bond_length));
				//float const eps( current_val*0.1 );
				cst_set.add_kin_torsion_constraint( this_bond_length, current_val, bond_length_tether );
				pose.set_allow_move( this_bond_length, true );
			}

			if ( minimize_ligand_angles ) { // bond angle
				kin::Torsion_id this_bond_angle( atom, kin::THETA );
				float const current_val( pose.get_atom_tree_torsion( this_bond_angle));
				//float const eps( current_val*0.1 );
				cst_set.add_kin_torsion_constraint( this_bond_angle, current_val,bond_angle_tether );
				pose.set_allow_move( this_bond_angle, true );
			}

			if ( minimize_ligand_torsions ) { // torsion angle or torsion-offset
				kin::Torsion_id this_bond_torsion( atom, kin::PHI );
				float const current_val(pose.get_atom_tree_torsion(this_bond_torsion));
				//float const eps( current_val*0.1 );
				cst_set.add_kin_torsion_constraint( this_bond_torsion, current_val, bond_torsion_tether );
				pose.set_allow_move( this_bond_torsion, true );
			}
		}
	}


	pose_ns::Score_weight_map weight_map( score12 );
	weight_map.set_weight( pose_ns::KIN_1D_CST, 1.0 ); // the torsion/bond/angle tether
	set_use_nblist( true );
	minimize_set_tolerance( 1e-6 );


	pose.main_minimize( weight_map, "dfpmin" );

	std::string outfile= files_paths::pdb_out_path + filename;
	pose.dump_scored_pdb( outfile, weight_map );
	remove((outfile +".in_progress").c_str());
	return 1;
}

////////////////////////////////////////////////////////////////////////////////
///@begin pose_ligand_docking_rigid_body_trial
///@brief This function mirrors the mmc search in the pose_docking.cc function
///pose_docking_rigid_body_trial. Some difference exists hence the duplication.
///@param[in/out] pose_ns::Pose & pose
///@param[in] int const cycles number of monte carlo trials
///@param[in] const pose_ns::Score_weight_map & wt_map scoring function
///@param[in] bool const repack controls fullatom repack after perturbation
///@param[in] float const trans_mag maximum translation magnitude in angstrom
///@param[in] float const rot_mag maximum rotational magnitude in degrees
////////////////////////////////////////////////////////////////////////////////
float
pose_ligand_docking_rigid_body_trial(
	pose_ns::Pose & pose,
	int const cycles,
	const pose_ns::Score_weight_map & wt_map,
	float const trans_mag,
	float const rot_mag,
	float const perturb_ligand_trans,
	float const perturb_ligand_rot
){

	//for benchmarking case we may want to force the randomization of the start point
	//these lines shoudl be commented out unless you want to force a random start point
	//for ( int current_jump=1; current_jump <= pose.num_jump(); current_jump++){
	//		pose_docking_gaussian_rigid_body_move( pose, current_jump, trans_mag, rot_mag );
	//}

//	float const temperature( 0.8 );
//	pose_ns::Monte_carlo mc( pose, wt_map, temperature );
	int n_accepted = 0;

//	for ( int i = 1; i <= cycles; ++i ) {
//		std::cout << "Cycle:" << i << std::endl;
		for ( int current_jump=1; current_jump <= pose.num_jump(); current_jump++){
			pose_docking_gaussian_rigid_body_move( pose, current_jump, trans_mag, rot_mag );
		}


		repack_for_ligand( pose, perturb_ligand_trans , perturb_ligand_rot);

//		std::cout << "After repack" << std::endl;
		pose.score( wt_map );
		//std::string outfile = "MC_" + lead_zero_string_of(i,3) + ".pdb";
		//pose.dump_scored_pdb( outfile, wt_map );
//		if ( mc.boltzmann( pose ) ) {
//		++n_accepted;
//		}
//		std::cout << "After boltzmann" << std::endl;
//	}

//	pose = mc.low_pose();
	//std::string outfile = "MC_" + lead_zero_string_of(cycles+1,3) + ".pdb";
	//pose.dump_scored_pdb( outfile, wt_map );
//	std::cout << "pose_docking_rigid_body_trial -- "
//			<< (repack ? "fullatom ":"centroid ")
//			<< n_accepted << " out of " << cycles << " accepted\n";

	return n_accepted/cycles;

}


////////////////////////////////////////////////////////////////////////////////
///@begin find_nearest_res
///@brief This function returns the pose residue number of the residue nearest
///given point.
///
///@param[in] const Pose & pose the input pose
///@param[in] const numeric::xyzVector_float point
////////////////////////////////////////////////////////////////////////////////
int
find_nearest_res(
	const pose_ns::Pose & pose,
	const numeric::xyzVector_float point
){
	int anchor_rsd(0);
	{//restricting scope of using numeric::xyzVector statement
				using numeric::xyzVector_float;
				float min_d( 1000.0f );
				for ( int i=1; i<= pose.total_residue(); ++i ) {
					float const d = distance( xyzVector_float( &(pose.full_coord()(1,2,i))),point );
					if ( d<min_d && !param_aa::is_ligand(pose.res(i))){
						min_d = d;
						anchor_rsd = i;
					}
				}
	}//restricting scope of using numeric::xyzVector statement
	if ( anchor_rsd == 0 ) {
		anchor_rsd =1;
	}
	return anchor_rsd;
}


///////////////////////////////////////////////////////////////////////////////
///@begin repack_for_ligand
///@brief This function setups and calls pack_rotamers. It allows repacking of
///all residues with CA 12 A of the first three heavy atoms of any ligand.
///
///@param[in] Pose & pose
///@param[in] float const translation amount perturbation to allow for the ligand in the packer
///@param[in] float const rotation perturbation to allow for the ligand in the packer
///
///////////////////////////////////////////////////////////////////////////////
int
repack_for_ligand(
	pose_ns::Pose & pose,
	float const translation,
	float const rotation
){


	FArray1D_bool repack_res(pose.total_residue(),false);

	//first find ligand residues
	std::vector< int > ligand_residues;
	for( int i =1; i <= pose.total_residue(); ++i ){
		if( param_aa::is_ligand(pose.res(i))){
			ligand_residues.push_back(i);
			repack_res(i)=true;
		}
	}

	//next_find the residues within 12 A of the ligand
	std::vector< int >::iterator lig_res_first_ptr=ligand_residues.begin();
	std::vector< int >::iterator lig_res_last_ptr=ligand_residues.end();
	int num_interface_res=0;
	for( std::vector< int >::iterator c_lig_res_ptr=lig_res_first_ptr; c_lig_res_ptr!=lig_res_last_ptr; c_lig_res_ptr++){
		for( int i=1; i <= pose.total_residue(); ++i ){
			if( !param_aa::is_ligand(pose.res(i))){
				if( distance( numeric::xyzVector_float( &(pose.full_coord()(1,2,i))), numeric::xyzVector_float( &(pose.full_coord()(1,1,(*c_lig_res_ptr))))) < 12 ||
						distance( numeric::xyzVector_float( &(pose.full_coord()(1,2,i))), numeric::xyzVector_float( &(pose.full_coord()(1,2,(*c_lig_res_ptr))))) < 12 ||
						distance( numeric::xyzVector_float( &(pose.full_coord()(1,2,i))), numeric::xyzVector_float( &(pose.full_coord()(1,3,(*c_lig_res_ptr))))) < 12
				){
					num_interface_res++;
					repack_res(i)=true;
				}
			}
		}
	}
	//Setup a packer task to repack residues

	std::cout << "Number of interface residues: " << num_interface_res << std::endl;
	PackerTask repack_task(pose, "packrot", false, repack_res);
	repack_task.setup_residues_to_vary();
	repack_task.set_perturb_ligand_rotamers(translation, rotation);
	pack_rotamers(pose,repack_task);

	return 1;
}
///////////////////////////////////////////////////////////////////////////////
///@begin small_dock_perturbation_pose_ligand
///@brief This function generates a ligand ensemble of 1000
///placements and conformations from the ligand_rotamers stored
///in enable_ligaa_ns
///@param[in] int aa
///@param[in] int aav
///@param[in] float max_translation maximum transtion in angstroms about a random axis
///@param[in] float max_rotation maximum angle of rotation about a random axis
///@param[in] FArray2D_float & input_conformation
///
///@author Kristian Kaufmann
///////////////////////////////////////////////////////////////////////////////

	FArray3D_float
	small_dock_perturbation_pose_ligand(
		int const aa,
		int const aav,
		float const max_translation,
		float const max_rotation,
		FArray2D_float & input_conformation,
		int & nconformations
	){


		FArray3D_float return_conformations(4, param::MAX_ATOM()(), 1000, 0.0);
//		utility::io::ozstream iunit( "coordinates.txt");
		if ( max_translation >= 0.00000001 || max_rotation <= -0.0000001 || max_rotation >= 0.00000001 ){
			//if(move ligand)
			nconformations=1000;
//			for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//				iunit << F(10,3, input_conformation(1,atom)) << " ";
//				iunit << F(10,3, input_conformation(2,atom)) << " ";
//				iunit << F(10,3, input_conformation(3,atom)) << std::endl;
//			}
//			iunit << "TER" << std::endl;
			for( int conf=1;conf<=1000;conf++){
				FArray2D_float new_conformation( 3, aaproperties_pack::natoms(aa,aav), 0.0);
				//choose a conformation
				int c_ligand=random_range(1,enable_ligaa_ns::nligandrotamers(aa,aav));
				FArray2D_float rotamer=change_conformation_to_rotamer( c_ligand, input_conformation, aa, aav);
//				for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//					iunit << F(10,3, rotamer(1,atom)) << " ";
//					iunit << F(10,3, rotamer(2,atom)) << " ";
//					iunit << F(10,3, rotamer(3,atom)) << std::endl;
//				}
//				iunit << "TER" << std::endl;
				//translate and rotate the conformation
				new_conformation=rotate_translate_conformation(rotamer, aa, aav, max_rotation, max_translation);
				load_conformation_into_FArray3D_float( new_conformation, conf, c_ligand, aa, aav, return_conformations);
			}
		}else{
			nconformations=enable_ligaa_ns::nligandrotamers(aa,aav);
//			for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//				iunit << F(10,3, input_conformation(1,atom)) << " ";
//				iunit << F(10,3, input_conformation(2,atom)) << " ";
//				iunit << F(10,3, input_conformation(3,atom)) << std::endl;
//			}
//			iunit << "TER" << std::endl;
			for( int conf=1; conf<=enable_ligaa_ns::nligandrotamers(aa,aav); conf++){
				FArray2D_float rotamer=change_conformation_to_rotamer(conf, input_conformation, aa, aav);
//				for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//					iunit << F(10,3, enable_ligaa_ns::ligand_cartesian_rotamers(1,atom, conf, param_aa::has_ligand_no(aa),1)) << " ";
//					iunit << F(10,3, enable_ligaa_ns::ligand_cartesian_rotamers(2,atom, conf, param_aa::has_ligand_no(aa),1)) << " ";
//					iunit << F(10,3, enable_ligaa_ns::ligand_cartesian_rotamers(3,atom, conf, param_aa::has_ligand_no(aa),1)) << " ";
//				}
//				iunit << "TER" << std::endl;
//				for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//					iunit << F(10,3, rotamer(1,atom)) << " ";
//					iunit << F(10,3, rotamer(2,atom)) << " ";
//					iunit << F(10,3, rotamer(3,atom)) << std::endl;
//				}
//				iunit << "TER" << std::endl;
				load_conformation_into_FArray3D_float( rotamer, conf, conf, aa, aav, return_conformations);
			}
		}
//		utility::io::ozstream iunit( "coordinates.txt");
//		for( int conf=1; conf<=nconformations; conf++){
//			for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//				iunit << F(10,3, return_conformations(1,atom,conf)) << " ";
//				iunit << F(10,3, return_conformations(2,atom,conf)) << " ";
//				iunit << F(10,3, return_conformations(3,atom,conf)) << std::endl;
//			}
//			iunit << "TER" << std::endl;
//		}
//		iunit.close();
		return return_conformations;
	}
////////////////////////////////////////////////////////////////////////////////
///@begin change_conformation_to_rotamer
///@brief creates a copy of the passed in coordinates and changes the chi angles
///of the copy to match those of the selected conformation of the ligand aa
///@param int const conf the integer of the ligand rotamer to change to
///@param FArray2D_float & coord input coordinates
///@param int const aa the integer identifier of the ligand aa
///@param int const aav the integger identifier of the ligand aa variant
///
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
	FArray2D_float
	change_conformation_to_rotamer(
		int const conf,
		FArray2D_float & coord,
		int const aa,
		int const aav
	){
//		utility::io::ozstream iunit( "coordinates3.txt");
//		utility::io::ozstream iunit2( "chi_required.txt");
//		for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//			iunit << F(10,3, coord(1,atom)) << " ";
//			iunit << F(10,3, coord(2,atom)) << " ";
//			iunit << F(10,3, coord(3,atom)) << std::endl;
//		}
//		iunit << "TER" << std::endl;

		FArray2D_float return_conformation=coord;
		FArray1D_float current_chi(enable_ligaa_ns::ligand_nchi(param_aa::has_ligand_no(aa),aav));
		//current_chi=get_current_chi(return_conformation, aa,aav);
		for( int chi=1; chi<=enable_ligaa_ns::ligand_nchi(param_aa::has_ligand_no(aa),aav); chi++){

			// rotate angle by new-initial degrees
			current_chi=get_current_chi(return_conformation, aa,aav);
			float rot = numeric::conversions::radians(  enable_ligaa_ns::ligand_rotamer_chi_angles(chi,conf,param_aa::has_ligand_no(aa),aav) - current_chi(chi) );
			// generate rotation vector and matrix
			FArray2D_float mat(3,3,0.0);
			FArray1D_float vec(3,0.0);
			int c2=enable_ligaa_ns::ligand_chi_atoms(2,chi,param_aa::has_ligand_no(aa),aav);
			int c3=enable_ligaa_ns::ligand_chi_atoms(3,chi,param_aa::has_ligand_no(aa),aav);
			getrot_bk(return_conformation(1,c2), return_conformation(1,c3), rot, mat, vec);
			// move atoms
			for ( int i = 1; i <= aaproperties_pack::natoms(aa,aav); ++i ) {
//				iunit2 << "atom: " << i << " chi: " << chi << " c2 " << c2 << " c3 " << c3;
				if ( enable_ligaa_ns::ligand_chi_required(chi,i,param_aa::has_ligand_no(aa),aav) ) { // chi_required(ch,i,aa,aav)
					move_bk(return_conformation(1,i),mat,vec);
//					iunit2 << " true";
				}
//				iunit2 << std::endl;
			}//for natoms
//			for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//				iunit << F(10,3, return_conformation(1,atom)) << " ";
//				iunit << F(10,3, return_conformation(2,atom)) << " ";
//				iunit << F(10,3, return_conformation(3,atom)) << std::endl;
//			}
//			iunit << "TER" << std::endl;
		}//for nchi
//		iunit.close();
//		iunit2.close();
		return return_conformation;
	}

////////////////////////////////////////////////////////////////////////////////
///@begin create_rotid
///@brief create a mapping of a int to a MAX_CHI digit number traceable back to
///the ligand rotamer index
///@param int const rot_integer
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
	FArray1D_int
	create_rotid(
		int const rot_integer
	){
		FArray1D_int return_rotid(param::MAX_CHI,0);
		for( int i=1;i<=param::MAX_CHI; i++){
			return_rotid(i)=rot_integer%(10^i);
			for( int j=1;j<i;j++){
				return_rotid(i)=return_rotid(i)-return_rotid(j);
			}
		}
		return return_rotid;
	}

////////////////////////////////////////////////////////////////////////////////
///@begin create_int_from_rotid
///@brief combine digits for rotid FArray1D_int into an integer
///@param FArray1D_int
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
	int
	create_int_from_rotid(
		FArray1D_int rotid
	){
		int return_value=0;
		for( int i=1; i<=param::MAX_CHI; i++){
			return_value+=rotid(i)*(10^(i-1));
		}
		return return_value;
	}

////////////////////////////////////////////////////////////////////////////////
///@begin load_conformation_into_FArray3D_float
///@brief transfer coordinates from passed 2D array into the 3D array at the
///index conf
///@param FArray2D_float & coord input_conformation of a ligand aa
///@param int const conf the index of the 3D array
///@param int const rotid the rotamer number of the conformation being loaded
///@param int const aa the integer identifier of the ligand aa
///@param int const aav the integer identifier of the ligand aa variant
///
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
	void
	load_conformation_into_FArray3D_float(
		FArray2D_float & coord,
		int const conf,
		int const rotid,
		int const aa,
		int const aav,
		FArray3D_float & conf_lib
	){
		for( int atom=1; atom <= aaproperties_pack::natoms(aa,aav); atom++){
			conf_lib(1,atom,conf)=coord(1,atom);
			conf_lib(2,atom,conf)=coord(2,atom);
			conf_lib(3,atom,conf)=coord(3,atom);
			conf_lib(4,atom,conf)=rotid;
		}
		return;
	}

////////////////////////////////////////////////////////////////////////////////
///@begin load_conformation_from_FArray3d_float
///@brief extract conformation from FArray3D_float into FArray2D_float
///
///@param int const conf conformation to extract
///@param int const aa ligand aa integer identifier
///@param int const aav ligand aa variant integer identifier
///
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
	FArray2D_float
	load_conformation_from_FArray3D_float(
		FArray3D_float & conf_lib,
		int const conf,
		int const aa,
		int const aav
	){
		FArray2D_float return_conformation(4,param::MAX_ATOM()(),0.0);

		for( int atom = 1 ;atom <= aaproperties_pack::natoms(aa,aav); atom++){
			return_conformation(1,atom)=conf_lib(1,atom,conf);
			return_conformation(2,atom)=conf_lib(2,atom,conf);
			return_conformation(3,atom)=conf_lib(3,atom,conf);
			return_conformation(4,atom)=conf_lib(4,atom,conf);
		}

		return return_conformation;
	}

////////////////////////////////////////////////////////////////////////////////
///@begin rotate_translate_conformation
///@brief performs a random translation and rotation on the input conformations
///bounded by the passed values
///@param FArray2D_float & input_conformation
///@param int const aa the ligand aa integer identifier
///@param int const aav the ligand aav integer identifier
///@param float max_rotation the maximum value in degrees for the random rotation
///@param float max_translation the maximum posible translation value
///
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////

	FArray2D_float
	rotate_translate_conformation(
		FArray2D_float & input_conformation,
		int const aa,
		int const aav,
		float const max_rotation,
		float const max_translation
	){
		//choose the random translation vector
		numeric::xyzVector< float > trans_vector;
		if( max_translation >= 0.000000001){
			trans_vector.assign( ran3()-0.5, ran3()-0.5, ran3()-0.5);
			trans_vector.normalize();
			trans_vector=trans_vector*max_translation;
		}else{
			trans_vector.assign( 0.0,0.0,0.0);
		}
		//choose the rotation matrix
		FArray2D_float rotation_matrix(3,3,0.0);
		if( max_rotation >=0.00000001 || max_rotation <= -0.00000001){
			float rotation_angle=0.0;
			numeric::xyzVector< float > rotation_axis( ran3()-0.5, ran3()-0.5, ran3()-0.5);
			rotation_axis.normalize();
			if( max_rotation > 180.0 || max_rotation < -180 ){
				rotation_angle=180.0;
			}else{
				rotation_angle=std::abs(max_rotation);
			}
			rotation_angle=2*ran3()*rotation_angle-rotation_angle;
			FArray1D_float quaternion(4);
			float sin_theta_2=sin(rotation_angle/2);
			quaternion(1)=cos(rotation_angle/2.0);
			quaternion(2)=sin_theta_2*rotation_axis.x();
			quaternion(3)=sin_theta_2*rotation_axis.y();
			quaternion(4)=sin_theta_2*rotation_axis.z();
			rotation_matrix(1,1)=1-2*quaternion(3)*quaternion(3)-2*quaternion(4)*quaternion(4);//xx
			rotation_matrix(1,2)=2*quaternion(2)*quaternion(3)-2*quaternion(1)*quaternion(4);//xy
			rotation_matrix(1,3)=2*quaternion(2)*quaternion(4)+2*quaternion(3)*quaternion(1);//xz
			rotation_matrix(2,1)=2*quaternion(2)*quaternion(3)+2*quaternion(1)*quaternion(4);//yx
			rotation_matrix(2,2)=1-2*quaternion(2)*quaternion(2)-2*quaternion(4)*quaternion(4);//yy
			rotation_matrix(2,3)=2*quaternion(3)*quaternion(4)-2*quaternion(1)*quaternion(2);//yz
			rotation_matrix(3,1)=2*quaternion(2)*quaternion(4)-2*quaternion(1)*quaternion(3);//zx
			rotation_matrix(3,2)=2*quaternion(3)*quaternion(4)+2*quaternion(1)*quaternion(2);//zy
			rotation_matrix(3,3)=1-2*quaternion(2)*quaternion(2)-2*quaternion(3)*quaternion(3);//zz
		}else{
			//assign an identity matrix
			rotation_matrix(1,1)=1.0;
			rotation_matrix(1,2)=0.0;
			rotation_matrix(1,3)=0.0;
			rotation_matrix(2,1)=0.0;
			rotation_matrix(2,2)=1.0;
			rotation_matrix(2,3)=0.0;
			rotation_matrix(3,1)=0.0;
			rotation_matrix(3,2)=0.0;
			rotation_matrix(3,3)=1.0;
		}
		//using coords of the first atom as the origin
		// new_coord= rotation_matrix*(coord-origin)+origin+trans_vector
		FArray1D_float origin(3,0.0);
		FArray1D_float trans_plus_origin(3,0.0);
		FArray2D_float return_conformation(3,aaproperties_pack::natoms(aa,aav));
		origin(1)=input_conformation(1,1);
		origin(2)=input_conformation(2,1);
		origin(3)=input_conformation(3,1);
		trans_plus_origin(1)=trans_vector.x()+origin(1);
		trans_plus_origin(2)=trans_vector.y()+origin(2);
		trans_plus_origin(3)=trans_vector.z()+origin(3);
		return_conformation=input_conformation;
		for( int atom =1; atom<=aaproperties_pack::natoms(aa,aav);atom++){
			FArray1D_float point(3,0.0);
			point(1)=return_conformation(1,atom)-origin(1);
			point(2)=return_conformation(2,atom)-origin(2);
			point(3)=return_conformation(3,atom)-origin(3);
			move_bk(point(1), rotation_matrix, trans_plus_origin);
			return_conformation(1,atom)=point(1);
			return_conformation(2,atom)=point(2);
			return_conformation(3,atom)=point(3);
		}
		return return_conformation;
	}

////////////////////////////////////////////////////////////////////////////////
///@begin get_current_chi
///@brief given an FArray2D_float containing the coordinates of a ligand aa
///this function returns an array with the values of the "chi" angles of the
///ligand aa
///@param FArray2D_float input_conformation
///@param int const aa
///@param int const aav
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
	FArray1D_float
	get_current_chi(
		FArray2D_float input_conformation,
		int const aa,
		int const aav
	){
		FArray1D_float return_chi(enable_ligaa_ns::ligand_nchi(param_aa::has_ligand_no(aa),aav));
		for( int chi=1; chi<=enable_ligaa_ns::ligand_nchi(param_aa::has_ligand_no(aa),aav); chi++){
			float ichi=0.0;
			int l = enable_ligaa_ns::ligand_chi_atoms.index(1,chi,param_aa::has_ligand_no(aa),aav);
			int c1 = enable_ligaa_ns::ligand_chi_atoms[  l ]; // chi_atoms(1,ch,aa,aav);
			int c2 = enable_ligaa_ns::ligand_chi_atoms[ ++l ]; // chi_atoms(2,ch,aa,aav);
			int c3 = enable_ligaa_ns::ligand_chi_atoms[ ++l ]; // chi_atoms(3,ch,aa,aav);
			int c4 = enable_ligaa_ns::ligand_chi_atoms[ ++l ]; // chi_atoms(4,ch,aa,aav);
			// determine initial chi angle
			dihedral_bk(input_conformation(1,c1),input_conformation(1,c2),input_conformation(1,c3),input_conformation(1,c4),ichi);
			return_chi(chi)=ichi;
		}
		return return_chi;
	}




inline
bool
atom_type_is_hydrogen(
	int const type
)
{
	return ( type >= 22 && type <= 25 );
}

inline
bool
atom_type_is_polar_hydrogen(
											int const type
											)
{
	return ( type == 22 || type == 25 );
}

inline
bool
atom_type_is_apolar_hydrogen(
											int const type
											)
{
	return ( type == 23 );
}

inline
bool
atom_type_is_aromatic_hydrogen(
											int const type
											)
{
	return ( type == 24 );
}


///iwd  Helper function for setup_ligand_aa().  Connects atom b2 to b1.
///iwd  b1 should already be bonded, unless this is the first bond.
void make_ligaa_bond(
	int const b1,
	int const b2,
	int const aa,
	int const aav,
	FArray1D_bool & bonded,
	bool isFirstBond = false
	)
{
	using namespace aaproperties_pack;
	if ( isFirstBond ) {
		//iwd  Special setup particular to the first bond:
		assert(!bonded(b1));
		assert(nbonded_neighbors(b1,aa,aav) == 0);
		bonded(b1) = true;
		atom_base(b1, aa, aav) = b2;
	}
	assert(bonded(b1) && !bonded(b2));
	assert(nbonded_neighbors(b2,aa,aav) == 0);
	bonded(b2) = true;
	bonded_neighbor( ++nbonded_neighbors(b1,aa,aav), b1, aa, aav ) = b2;
	bonded_neighbor( ++nbonded_neighbors(b2,aa,aav), b2, aa, aav ) = b1;
	atom_base(b2, aa, aav) = b1;
}


void
setup_ligand_aa(
	int const aa,
	int const aav,
	Ligand & ligand,
	FArray2D_float & coords // make a copy of coords
)
{
	using namespace aaproperties_pack;
  using namespace hbonds;
	using namespace param;
	using namespace enzyme;
	using numeric::xyzVector_float;

	assert( MAX_AA()() >= aa );

	typedef std::vector< Atom* > Atom_list;
	typedef Atom_list::iterator Atom_iter;
	Atom_list & atoms( ligand.atom_vector );
  std::vector<int> mdl_indices;
  mdl_indices.reserve(atoms.size());

	// default
	nchi( aa, aav ) = 0;
	first_scatom( aa, aav ) = 1;
	HNpos( aa, aav ) = -1;
	HApos( aa, aav ) = -1;
	nh2o( aa, aav ) = 0;

	// get heavy atoms
	int atom_count = 0;
	FArray1D< Atom * > atom_array( 10000 );
  int counter = 0;
	for( Atom_iter atom_iter=atoms.begin(), atom_end = atoms.end();
			 atom_iter != atom_end; ++atom_iter ) {
		Atom & atom( **atom_iter );
		int const type( atom.get_ros_atom_type() );
		if ( !atom_type_is_hydrogen( type ) && type <= MAX_REALTYPES ) {
			atom_array( ++atom_count ) = &atom;
      mdl_indices[counter] = atom_count;
		}
    counter++;
	}
	// store the number of heavy atoms
	nheavyatoms(aa,aav) = atom_count;
	// now the hydrogens
  counter = 0;
	for( Atom_iter atom_iter=atoms.begin(), atom_end = atoms.end();
			 atom_iter != atom_end; ++atom_iter ) {
		Atom & atom( **atom_iter );
		int const type( atom.get_ros_atom_type() );
		if ( atom_type_is_hydrogen( type ) || type > MAX_REALTYPES ) {
			atom_array( ++atom_count ) = &atom;
      mdl_indices[counter] = atom_count;
		}
    counter++;
	}

	std::cout << "Found " << atom_count << " total atoms!" << std::endl;
	if ( atom_count > MAX_ATOM()() ) {
		std::cout << "DOH! ligand atom count > MAX_ATOM()() " <<
			atom_count << ' ' << MAX_ATOM()() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// fill in the easy stuff
	natoms( aa, aav ) = atom_count;
	nH_polar( aa, aav ) = 0;
	nH_apolar( aa, aav ) = 0;
	nH_aromatic( aa, aav ) = 0;

	//if( natoms( aa, aav ) ) == 0 ) return;

	for ( int i=1; i<= atom_count; ++i ) {
		Atom & atom( *atom_array(i) );
		atom_name(i,aa,aav)=atom.get_atom_name();
		fullatom_type( i, aa, aav ) = atom.get_ros_atom_type();
		atomic_charge( i, aa, aav ) = atom.get_partial_charge();
		int const type( fullatom_type( i, aa, aav ) );
		if ( type == 22 || type == 25 ) {
			Hpos_polar( ++nH_polar( aa, aav ), aa, aav ) = i;
		} else if ( type == 23 ) {
			Hpos_apolar( ++nH_apolar( aa, aav ), aa, aav ) = i;
		} else if ( type == 24 ) {
			Hpos_aromatic( ++nH_aromatic( aa, aav ), aa, aav ) = i;
		}

		for ( int k=1; k<= 3; ++k ) {
			enable_ligaa_ns::lig_icoord( k, i, param_aa::has_ligand_no(aa) ) = coords( k, i )
										= atom_array(i)->get_coordinates()(k);
			icoor( k, i, aa, aav ) = atom_array(i)->get_coordinates()(k);
		}
		if( read_occ_weights ) {
			enable_ligaa_ns::lig_iocc_weight( i, param_aa::has_ligand_no(aa) ) = atom_array(i)->get_occupancy();
		} else {
			enable_ligaa_ns::lig_iocc_weight( i, param_aa::has_ligand_no(aa) ) = 1.0;
		}
		if( read_bvalue_charge ) {
		  enable_ligaa_ns::lig_icharge( i, param_aa::has_ligand_no(aa)  ) =  atom_array(i)->get_bvalue();
		} else {
			enable_ligaa_ns::lig_icharge( i, param_aa::has_ligand_no(aa) ) =  0.0;
		}
	} // i


	// calculate approx bonding information
	FArray2D_float dist( atom_count, atom_count );
	for ( int i=1; i<= atom_count; ++i ) {
		dist( i,i) = 0.0;
		for ( int j=i+1; j<= atom_count; ++j ) {
			float const d = distance( xyzVector_float( &coords(1,i) ),
																xyzVector_float( &coords(1,j) ) );
			dist( i,j ) = dist( j,i ) = d;
		}
	}

	// start with atom number 1
	FArray1D_bool bonded( atom_count, false );
	for ( int i=1; i<= atom_count; ++i ) {
		nbonded_neighbors(i,aa,aav) = 0;
	}

  // amw215 - setup bonding information from the ligand
  std::list<numeric::xyzVector < size_t > > bond_map;
  std::list<numeric::xyzVector < size_t > >::iterator bnd;
  ligand.get_bond_map(bond_map);
  if (bond_map.size() > 0) {  // read from an mdl file
		std::cout << "using mdl bond information" << std::endl;
		//iwd  The old code assumed that bonds are listed as (existing atom, new or existing atom)
		//iwd  except, of course, for the first, where both atoms are new (and assumed heavy).
		//iwd  (existing, existing) bonds were ignored and (existing, new) bonds necessarily
		//iwd  formed a spanning tree (and one in which Hs are leaves.)
		//iwd  The order is used to set up the coordinate frames for acceptors -- atom_base and abase2
		//iwd  set properly to be able to have hydrogen bonds from ligand atoms go in the right directions.
		//iwd  (Description courtesy JSS.)
		//iwd
		//iwd  It also assumes that the first atom in the first bond (the new,new bond)
		//iwd  is the first atom in atom_array (i.e. atom_array(1)).
		//iwd  It also appears that bond_map isn't in the same order as in the original molfile anyway.
		//iwd  So I'm not sure why the original algorithm had these expectations...
		//iwd  The warnings about out-of-order bonds don't catch the case where atom 1 of bond 1
		//iwd  doesn't also have index 1.  In this case, we succeed where it fails...
		//iwd  There doesn't seem to be any reason some other atom can't be part of the first bond.
		//iwd
		//iwd  The new code should produce the same result if the input conforms to that standard.
		//iwd  If it doesn't, however, we still manage to connect everything up.
		//iwd  Bonds are *ERASED* from the bond_map as we "consume" them.
		//iwd
		//iwd  (1) Find the first bond between two heavy atoms and connect them to each other.
		//iwd  (2) Cycle through the other bonds, skipping things we don't know how to deal with.
		//iwd  If we ever go through a cycle without adding any bonds, die.
		//iwd  Probably not the most efficient algorithm, but if the bonds are ordered as
		//iwd  the old code expected, it should produce the same results as the old code!
		bool warnBondsOutOfOrder = false;
		bool foundFirstBond = false;
		while ( !bond_map.empty() ) {
			bool foundSomeBond = false;
			for (bnd = bond_map.begin(); bnd != bond_map.end(); bnd++) {
				//iwd  mdl_b[12] are (0-based) indexes of atoms in mdl file
				int mdl_b1 = (*bnd)[0];
				int mdl_b2 = (*bnd)[1];
				int b1 = mdl_indices[mdl_b1];
				int b2 = mdl_indices[mdl_b2];
				//iwd  Insist that the first bond be between heavy atoms
				if ( !foundFirstBond ) {
					if ( !atom_type_is_hydrogen(fullatom_type( b1, aa, aav )) && !atom_type_is_hydrogen(fullatom_type( b2, aa, aav )) ) {
						make_ligaa_bond(b1, b2, aa, aav, bonded, true /* first bond */);
						foundFirstBond = foundSomeBond = true;
					}
				} else { // first bond already established
					//iwd  If the bond is "backwards", just swap b1 and b2.
					if (!bonded(b1) && bonded(b2)) {
						int b_tmp = b1; b1 = b2; b2 = b_tmp;
						warnBondsOutOfOrder = true;
					}
					if (bonded(b1)) {
						if (!bonded(b2)) {
							make_ligaa_bond(b1, b2, aa, aav, bonded);
						} else {
							//iwd  Both are already bonded (ring closure)
							// JSS note that this code will silently ignore ring-closing bonds
							assert(bonded(b1) && bonded(b2));
						}
						foundSomeBond = true;
					} else {
						//iwd  Neither is bonded yet -- skip until later.
						assert(!bonded(b1) && !bonded(b2));
					}
				}
				if ( foundSomeBond ) {
					std::cout << "Found new bond: " << mdl_b1+1 << ' ' << atom_name(b1,aa,aav) << ' ' <<
						mdl_b2+1 << ' ' << atom_name(b2,aa,aav) << ' ' << dist(b1,b2) << std::endl;
					bond_map.erase(bnd);
					break; // escape from for-each-bond loop
				}
				//iwd  If we make it to the end of the loop even once, we skipped a bond!
				warnBondsOutOfOrder = true;
			}//end for-each-bond
			if ( !foundSomeBond ) {
				if ( !foundFirstBond ) {
					std::cout << "No heavyatom - heavyatom bond could be found to base the ligand on" << std::endl;
				} else {
					std::cout << "Ackk!  Ligand bonds do not form a connected graph.  Dying..." << std::endl;
					std::cout << "  Disconnected atoms:" << std::endl;
					for(int i = 1; i <= atom_count; ++i) {
						if(!bonded(i)) std::cout << "    " << i << " " << atom_name(i,aa,aav) << std::endl;
					}
				}
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}//end while-bonds-remaining
		if ( warnBondsOutOfOrder ) {
			std::cout << "Warning:  Ligand bonds were used out-of-order and/or reversed." << std::endl;
			std::cout << "  Ligand atom_base() values may not be what you intended." << std::endl;
		}
	} else { // We don't have MDL bonds records
		bonded(1) = true; //start with bond 1 and build bonds to nearest neighbors
  	while ( true ) {  //should replace by a good minimum spanning tree algorithm (boost?)
  		int closest_i(0);
  		int closest_j(0);
  		float closest_d(1000.0);
  		for ( int i=1; i<= atom_count; ++i ) {
  			if (!bonded(i) ) continue;
  			for ( int j=1; j<= atom_count; ++j ) {
  				if ( bonded(j) ) continue;
  				if ( dist(i,j) < closest_d ) {
  					closest_d = dist(i,j);  // record closest pair between bonded and unbonded
  					closest_i = i;
  					closest_j = j;
  				}
  			}
  		}
  		if ( closest_i == 0 ) break; // done if no new pair

  		// add new bond between i,j
  		{
  			int const i( closest_i );
  			int const j( closest_j );
  			std::cout << "Found new bond: " << atom_name(i,aa,aav) << ' ' << I(2,i) << ' ' <<
  				atom_name(j,aa,aav) << ' ' << I(2,j) << ' ' << dist(i,j) << std::endl;
  			assert( bonded(i) && !bonded(j) );
  			assert( nbonded_neighbors(j,aa,aav) == 0 );
  			bonded(j) = true;
				atom_base(j,aa,aav) = i; //JSS need to define an atom_base for j in any case.
				if( nbonded_neighbors(i,aa,aav) < max_bonded_neighbors ) { // we know nbd_nhbs(j,..)==0.
          // every atom needs a base, so handle atom 1 specially for its first bond.
					if (i==1 && nbonded_neighbors(i,aa,aav)==0) atom_base(i,aa,aav) = j;
					bonded_neighbor( ++nbonded_neighbors(i,aa,aav), i, aa, aav ) = j;
					bonded_neighbor( ++nbonded_neighbors(j,aa,aav), j, aa, aav ) = i;
				} else {
					std::cout << "WARNING: too many bonded_neighbors "<< " between "<<i<<" "<<j
										<<" , please increase the max_bonded_neighbors in aapproperties_pack.cc"<<std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
  		}
  	}
  }

	// or maybe do this with build_aa??
	//kin::Atom* const root( add_kin_atom( 0, aa, aav, 0, 1, atom_ptr ) );

	// nhydrogens_on_atm
	nH_polar( aa, aav ) =  0; // number of polar hydrogens
	nH_aromatic( aa, aav ) =  0; // number of aromatic hydrogens
	nH_apolar( aa, aav ) =  0; // number of apolar hydrogens
	int closest_j(0);
	float closest_d(1000.0f);
	for ( int i=1; i<= atom_count; ++i ) {
		if ( i<=nheavyatoms(aa,aav) ) {
			nhydrogens_on_atm( i, aa, aav ) = 0;
			for ( int j=1; j<= nbonded_neighbors(i,aa,aav); ++j ) {
				int const & j_atomno(bonded_neighbor(j,i,aa,aav));
				if ( atom_type_is_hydrogen( fullatom_type(j_atomno,aa,aav) ) ) {
					++nhydrogens_on_atm(i,aa,aav);
					hydrogens_on_atm( nhydrogens_on_atm(i,aa,aav),i,aa,aav) = j_atomno;
					std::cout << "detect hydrogen: " << nhydrogens_on_atm(i,aa,aav) << ' ' << i << ' '<<aa<<' '<<aav<< ' '<<hydrogens_on_atm( nhydrogens_on_atm(i,aa,aav),i,aa,aav)<< std::endl;
				}
			}
		} else if( fullatom_type( i,aa,aav ) <= MAX_REALTYPES ) {
			assert( atom_type_is_hydrogen( fullatom_type(i,aa,aav)) );

			//lin fill polar, apolar, aromatic hydrogens
			if( atom_type_is_polar_hydrogen( fullatom_type(i,aa,aav)) ) {
				nH_polar( aa, aav )++;
				Hpos_polar( nH_polar( aa, aav ), aa, aav ) = i;
			}
			if( atom_type_is_apolar_hydrogen( fullatom_type(i,aa,aav)) ) {
				nH_apolar( aa, aav )++;
				Hpos_apolar( nH_apolar( aa, aav ), aa, aav ) = i;
			}
			if( atom_type_is_aromatic_hydrogen( fullatom_type(i,aa,aav)) ) {
				nH_aromatic( aa, aav )++;
				Hpos_aromatic( nH_aromatic( aa, aav ), aa, aav ) = i;
			}

			// find closest bonded neighbor -- JSS I think this can go away unless you want to test atom_base
			closest_j=0; closest_d=1000.0f;
			for ( int k=1; k<= nbonded_neighbors( i, aa,aav); ++k ) {
				int const & j( bonded_neighbor( k,i,aa,aav ) );
				if ( dist(j,i) < closest_d ) {
					closest_d = dist(j,i);
					closest_j = j;
				}
			}
      if (atom_base( i, aa, aav ) != closest_j)
        std::cout << "Warning: unexpected Hbase("<<i<<","<< aa<<","<< aav<<")  in setup_ligand_aa: expected "
									<<closest_j<<" found "<<atom_base( i, aa, aav )<<std::endl;
      if( atom_type_is_polar_hydrogen( fullatom_type(i,aa,aav) ) ) {
        std::cout << "detect lig donor: " << i << ' ' << closest_j << std::endl;
      }
    }
  }

	//lin hbond acceptors
	//lin fill in abase2 of hybridized acceptor
	nacceptors( aa, aav ) = 0;
  for ( int i = 1; i <=  atom_count; ++i ) {
    int const acc_type = HBaccchemtype(fullatom_type(i,aa,aav));
    if (acc_type != hbacc_NO) {
      accpt_pos( 	++nacceptors( aa, aav ), aa, aav ) = i; // found a new acceptor
                                                          // atom_base was set; see if we have abase2
			closest_j=0; closest_d=1000.0f;  // find closest bonded neighbor that is not atom_base
			for ( int k=1; k<= nbonded_neighbors(i,aa,aav); ++k ) {
				int const & j( bonded_neighbor( k,i,aa,aav ) );
        if (j != atom_base(i,aa,aav))
					if ( dist(j,i) < closest_d ) {
						closest_d = dist(j,i);
						closest_j = j;
					}
      }
      if (closest_j != 0 ) {
        abase2( i, aa, aav ) = closest_j;
					std::cout << "detect lig " << (acc_type==hbacc_RING ? "Ring":"")
										<<" acceptor: " << i << ' ' << atom_base(i, aa, aav)
										<< ' ' << abase2(i, aa, aav) << std::endl;
        switch (acc_type) {
				case hbacc_SP3:
				case hbacc_RING:
					break; // we expect abase2 to be set already for these
				default:
					std::cout<<"Warning: setup_ligand_aa() found second base atom "<< closest_j<<" on atom "<<i
									 <<" with fullatom type "<<fullatom_type(i,aa,aav)<<" when no second base was expected"<<std::endl;
//					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
        }
      } else {
        assert(atom_base(i,aa,aav) > 0);
        abase2( i, aa, aav ) = atom_base( atom_base(i, aa, aav), aa, aav);
					std::cout << "detect lig " << (acc_type==hbacc_RING ? "Ring":"")
										<<" acceptor: " << i << ' ' << atom_base(i, aa, aav) << std::endl;
        switch (acc_type) {
				case hbacc_SP2:
        case hbacc_BB:
				case hbacc_NO:
					break; // we expect no abase2 for these
				default:
					std::cout<<"Warning: setup_ligand_aa() found no second base on atom "<<i
									 <<" with fullatom type "<<fullatom_type(i,aa,aav)<<" when second base was expected"<<std::endl;
//					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
        }
      }
    }
  }
}
