// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

// /////////////////////////////////////////////////////////////////////////////////////
// @pose_looping.cc
//
// @brief
// Routine to create a loop using pose
//
// @detailed:
// To use this mode, enter -pose_looping, -loop_name {name of loop file}
// For loop mode, loop_name is a file that contains gap_begin, gap_end,
// loop_begin, and loop_end, which are the insertion points for the loop
// If you do not want to define gap_begin and gap_end, set the value to 0 in
// the input file
//
// @authors: Monica Berrondo
//
// @last_modified: March 30 2006
// /////////////////////////////////////////////////////////////////////////////////////


// Rosetta Headers
#include "pose_looping.h"
#include "after_opts.h"
#include "domins_ns.h"
#include "files_paths.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "fullatom.h"
#include "jumping_loops.h"
#include "jumping_pairings.h" // get_loop_fraction
#include "jumping_util.h"
#include "loop_relax.h"
#include "monte_carlo.h"
#include "misc.h" // damn
#include "pose.h"
#include "pose_io.h"
#include "prof.h"
#include "random_numbers.h"
#include "runlevel.h"
#include "silent_input.h"
#include "score.h"

// ObjexxFCL Headers
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <list> // list class definition
#include <map> // map class definition


///////////////////////////////////////////////////////////
//
// @looping_main
//
// @brief
// makes calls to other functions depending on inputs and options
//
// @detailed
// Creates pose, copies coordinates in from misc
// Eventually to have three functionalities:
//   Create a simple loop (build_insertion_loop)
//   Do flexible docking
//   Create a molecular switch
//
// @param
//
// @global_read
//
// @global_write
//
// @remarks
//
// @authors: Monica Berrondo
//
// @last_modified: March 21 2006
////////////////////////////////////////////////////////////////

void
looping_main()
{
   using namespace pose_ns;
   using namespace silent_io;
   using namespace files_paths;
   using namespace misc;

   // reads in the values for loop_begin and loop_end from the command line
   int loop_begin, loop_end;
   int const max_loop_size( 12 );
   int const min_good_loops( 3 );
   std::string const loop_name(stringafteroption("loop_name"));
   std::cout << "file: " << loop_name << std::endl;
   std::string loop_file( start_path + loop_name);
   bool const fullatom( truefalseoption("fa_input") );

   // ideal_pos set to false because positions likely come from a pdb file
   bool const ideal_pos( false );
   bool const coords_init( true );
   int iter(20);

   set_pose_flag(true);

   // Generate a pose
   Pose start_pose,pose;

   // defines length and sequence of starting pose
   pose_from_misc( start_pose, fullatom, ideal_pos, coords_init);

   // copies the pose to a pose that will be worked with
   pose = start_pose;

   //initialize protein_insert
   std::map<std::string,std::string> params;
   get_loop_parameters(params, loop_name);

   loop_begin = atoi(params["loop_begin"].c_str());
   loop_end = atoi(params["loop_end"].c_str());

   int const min_loop_size ( 6 );
   int const nres ( pose.total_residue() );
   int const loop_size ( loop_end - loop_begin + 1);
   assert ( loop_size >= min_loop_size );
   int const cutpoint ( loop_begin + loop_size / 2 );

   // create jump tree, if at the ends of the protein, it only needs to be a
   // simple jump tree
   if (loop_begin == 1 || loop_end == nres)
           pose.simple_fold_tree(nres);
   else {
           pose.one_jump_tree(nres, loop_begin-1, loop_end+1, cutpoint);
           pose.set_allow_jump_move(false);
   }

   // build the loops
   build_insertion_loop(pose, max_loop_size, min_good_loops, loop_begin, loop_end, iter);

   // calls repack and adds sidechains back
   if ( fullatom ) {
           bool const repack_rotamers(true);
           pose.set_fullatom_flag(true, repack_rotamers);
   }

   //copy the coordinates back to misc for output
   pose.copy_to_misc();

   set_pose_flag( false );

   monte_carlo_reset();
}

///////////////////////////////////////////////////////////
//
// @build_insertion_loop
//
// @brief
// creates a loop of a certain size
//
// @detailed
// calls insert_loop
// window moves the size of the loop and position of start
//
// @param start_pose - in/out: pose to work with and return
// @param init_loop_begin - in: starting coordinates for loop
// @param init_loop_end - in: ending coordinates for loop
//
// @global_read
//
// @global_write
//
// @remarks
//
// @authors: Monica Berrondo
//
// @last_modified: March 21 2006
////////////////////////////////////////////////////////////////
void
build_insertion_loop(
   pose_ns::Pose & pose,
   int const max_loop_size,
   int const min_good_loops,
   int const min_loop_begin,
   int const min_loop_end,
   int const iter
   )
{
   using namespace pose_ns;

   // params
   const int min_breakout_good_loops( std::max( min_good_loops, 5 ) );
   const int min_loop_size(6);
   pose.score( score3 );
   float const start_score ( pose.get_0D_score( SCORE ) );
   float const start_vdw_score ( pose.get_0D_score( VDW ) );
   float const really_bad_score( 10000.0 );
   float best_score ( really_bad_score );
   int good_loop_count (0);

   Pose loop_pose, best_pose;
   loop_pose = pose;

   int cutpoint ((min_loop_begin - min_loop_end)/2);
   int const nres ( loop_pose.total_residue() );

   //initialize window
   bool const window(truefalseoption("window"));

   // if window is true, this goes in and inserts loops with a moving window
   // to find the best location and size of the loop
   if(window) {
     // loop over a range of possible loop positions, break out once one of the
     // loops closes and scores well
     for (int loop_size = std::min( min_loop_size, max_loop_size );
           loop_size <= max_loop_size; ++loop_size) {
       const FArray1D_float & loop_fraction( get_loop_fraction(nres));
       typedef std::list<std::pair< float,int > > Float_int_list;
       Float_int_list window_list;

       for (int ii=0; ii<=loop_size; ++ii) {
         const int loop_begin ( cutpoint - loop_size + ii + 1 );
         const int loop_end ( cutpoint + ii );
         assert ( loop_begin <= cutpoint+1 &&
                  loop_end       >= cutpoint &&
                  loop_end == loop_begin + loop_size - 1 );
         if ( loop_begin < min_loop_begin || loop_end > min_loop_end ) continue;

         float f(0);
         for (int i=loop_begin; i<=loop_end; ++i) {
             f += loop_fraction(i);
         }
         window_list.push_back(std::make_pair(f,loop_begin));
       }
       window_list.sort();
       window_list.reverse();

       // loop over different positions
       for(Float_int_list::const_iterator w_it=window_list.begin(),
             w_it_end = window_list.end(); w_it != w_it_end; ++w_it ) {
         const int loop_begin( w_it->second );
         const int loop_end( loop_begin + loop_size - 1 );

         //insert the loop
         std::cout << "inserting loop" << std::endl;
         insert_loop(loop_pose, loop_begin, loop_end, iter);

         float const score ( loop_pose.get_0D_score ( SCORE ) );
         float const vdw_score ( loop_pose.get_0D_score ( VDW ) );
         if (vdw_score < start_vdw_score + 0.5) {
           std::cout << "good loop " << vdw_score << ' ' << start_vdw_score << std::endl;
           ++good_loop_count;
         }
         if (best_score > score) {
             best_score = score;
             best_pose = loop_pose;
             std::cout << "best_accept: new-best= " << best_score << " start_score= " << start_score << std::endl;
         }

         if ( good_loop_count >= min_breakout_good_loops ) {
           std::cout << "seen " << good_loop_count << " good_loops; done\n";
           break;
         }
       } // slide loop

       if ( good_loop_count >= min_good_loops && best_score < really_bad_score-1 ) {
         std::cout << "closed the loop!!! score, start_score, vdw, old_vdw " <<
           best_pose.get_0D_score ( SCORE ) << ' ' << start_score << ' ' <<
           best_pose.get_0D_score ( VDW ) << ' ' <<
           pose.get_0D_score ( VDW ) << std::endl;
         pose = best_pose;
         break;
       }
     } // loop size
   } else {
     //insert the loop
     std::cout << "inserting loop" << std::endl;
     std::cout << "fold_tree: " << pose.fold_tree() << std::endl;
     insert_loop(loop_pose, min_loop_begin, min_loop_end, iter);

   } // slide loop
   pose = loop_pose;
}

///////////////////////////////////////////////////////////
//
// @insert_loop
//
// @brief
// inserts a loop by calling one_loop_build_fragment, no use of loops file,
// only information needed comes from the fragment files
//
// @detailed
// create a jump tree from the beginning to end of the loop with the cutpoint
// in the middle
// goes through an inner and outer cycle of trail and acceptance that inserts
// loop and does ccd closer to find the best loop
//
// @param  pose - in/out: pose that will be worked on and passed out
// @param  loop_begin - in: beginning coordinates for the loop to be inserted
// @param  loop_end - in: end coordinates for the loop to be inserted
//
// @global_read
//
// @global_write
//
// @remarks
//
// @authors: Monica Berrondo
//
// @last_modified: March 21 2006
////////////////////////////////////////////////////////////////
void
insert_loop(
  pose_ns::Pose & pose,
  int const loop_begin,
  int const loop_end,
  int const iter
  )
{
  using namespace files_paths;
  using namespace pose_ns;
  using namespace silent_io;

  /////////
  // params
  int const min_loop_size ( 6 );
  int const nres ( pose.total_residue() );
  int const loop_size ( loop_end - loop_begin + 1);
  assert ( loop_size >= min_loop_size );
  int const cutpoint ( loop_begin + loop_size / 2 - 1); // PB
  int ccd_times ( 5 );
  std::cout << "cutpoint: " << cutpoint << std::endl;

  assert( pose.fold_tree().get_is_cutpoint()(cutpoint));

  // cycles2 are outer cycles, cycles3 are inner cycles
  int const cycles2(iter);
  int const cycles3(loop_size*iter);
  int frag_size(3), frag_offset(0);
  std::string move ("loop_insertion");

  // idealize the bonds to avoid problems later on
  pose.insert_ideal_bonds(loop_begin, loop_end);

  // scoring and weight map
  Score_weight_map weight_map;

  weight_map.set_weight( ENV, 1.0 );
  weight_map.set_weight( PAIR, 1.0 );
  weight_map.set_weight( VDW, 1.0 );
  weight_map.set_weight( BARCODE, 1.0 );
  weight_map.set_weight( CHAINBREAK, 0.0 );
  weight_map.set_weight( CHAINBREAK_OVERLAP, 0.0 );
  weight_map.set_weight( CST_SCORE, 1.0 );
  score_set_cst_mode(3);

	if (!domins_ns::no_frags) {
	  // insert a fragment
	  one_loop_build_fragment(frag_size, pose, loop_begin, loop_end, frag_offset);
	}

  pose.score(weight_map);
  Monte_carlo mc(pose, weight_map, 2);
  // output the beginning values for the score and weight
  show_decoy_status(pose, move);

  bool do_ccd_moves( false );
  if(loop_begin != 1 && loop_end != nres)
    do_ccd_moves = true;

  // for domain_insertion, increase the ccd times
  if (domain_insertion) ccd_times = iter/2;

  float const final_weight(5.0);
  float const delta_weight(final_weight/cycles2);

  // starts the loop with c2=1, initializes the int overlap
  for(int c2=1,overlap; c2<=cycles2; ++c2) {
    //statistics
    int frag_accepts(0),
      frag_trials(0),
      ccd_accepts(0),
      ccd_trials(0);
    // set the current pose to the one with the lowest score
    pose = mc.low_pose();
    overlap = 0;
    if(loop_begin != 1 && loop_end != nres) {
      weight_map.set_weight( CHAINBREAK, c2 * delta_weight);
      if(c2 > cycles2/2) overlap = 1;
      weight_map.set_weight( CHAINBREAK_OVERLAP, overlap);
    }

    pose.score(weight_map);
    mc.set_weight_map(weight_map);
		mc.reset(pose);

    // inner loop
    for(int c3=1,size; c3 <=cycles3; ++c3) {
      size = 1;
      // ramdomly alternate between inserting fragments and doing ccd
      // moves
      if( ( !do_ccd_moves ||
          overlap == 0 ||
          ran3() * cycles2 > c2 ) &&
					!domins_ns::no_frags ) {

				PROF_START ( prof::INSERT_LOOP );
        choose_offset_frag( 3, pose, loop_begin, loop_end, 0 );
				PROF_STOP ( prof::INSERT_LOOP );

				move = "fragment_insertion";

        // determine whether the score is low enough to accept or not
        bool const accepted(mc.boltzmann(pose, move, false /*don't make a movie pdb*/));
        if( accepted ) ++frag_accepts;
        ++frag_trials;
				if (runlevel_ns::runlevel >= runlevel_ns::verbose) {
					mc.show_scores();
					show_decoy_status(pose, move);
				}
      } else {
				move = "ccd_moves";
        ccd_moves(ccd_times, pose, loop_begin, loop_end, cutpoint);
        bool const accepted( mc.boltzmann(pose, move, false /*don't make movie pdb*/) );
        if(accepted) ++ccd_accepts;
        ++ccd_trials;
				if (runlevel_ns::runlevel >= runlevel_ns::verbose) {
					mc.show_scores();
					show_decoy_status(pose, move);
				}
      }
    }
		move = "ccd_moves";
    ccd_moves(ccd_times, pose, loop_begin, loop_end, cutpoint);
    bool const accepted( mc.boltzmann(pose, move) );
    if(accepted) ++ccd_accepts;
    ++ccd_trials;

    // IMPORTANT: standard protocol is to have a recover low after inner loop
    pose = mc.low_pose();
		mc.show_scores();
		show_decoy_status(pose, move);

    // improve formatting of diagnostics
    // it's critical to pay attention to diagnostics when developing a new
    // protocol
    std::cout << "accepts: cycle: " << c2 <<
      " #frag_trials: " << frag_trials << " acceptance_rate: " <<
      ( frag_trials ? float( 100.0*frag_accepts ) / frag_trials : 0 ) <<
      " #ccd_trials: " << ccd_trials << " acceptance_rate: " <<
      ( ccd_trials ? float( 100.0*ccd_accepts) /ccd_trials : 0 ) << std::endl;

    jmp_show_chainbreaks( pose );
  }

  pose = mc.low_pose();
	mc.show_scores();

  // recover low, rescore with overlap - 1 for compatability with ccd closure
  if (loop_begin != 1 && loop_end != nres) {
    weight_map.set_weight(CHAINBREAK_OVERLAP, 1.0);
  }

  pose.score( weight_map );

}


////////////////////////////////////////////////////////////////////////////////
/// @begin one_loop_build_fragment
///
/// @brief choose a loop fragment based on the satisfation of distance constraints
///
/// @detailed
///
/// @param  frag_size - in: size of the fragment (default 3)
/// @param  pose - in/out: pose that will be worked on and passed out
/// @param  loop_begin - in: beginning coordinates for the loop to be inserted
/// @param  loop_end - in: end coordinates for the loop to be inserted
/// @param  frag_offset in:
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @authors Monica Berrondo
///
/// @last_modified March 22 2006
/////////////////////////////////////////////////////////////////////////////////
void
one_loop_build_fragment(
  const int frag_size,
  pose_ns::Pose & pose,
  const int loop_begin,
  const int loop_end,
  const int frag_offset
)
{
  using namespace fragments;

  const int loop_size ( loop_end - loop_begin + 1);
  const int actual_frag_size (loop_size < frag_size ? loop_size : frag_size);

  // choose a window
  const int pos (static_cast<int> (ran3() *  (loop_size - actual_frag_size + 1)));
  const int pose_pos ( loop_begin + pos);
  const int frag_pos ( loop_begin + pos + frag_offset);

  assert(1<=frag_pos && frag_pos <= fragments_nres);

  // choose a fragment
  const int size_bin(get_index_by_frag_size(frag_size));

  const int last_frag_pos(fragments_nres - frag_size + 1);
  const int shift(std::max(0, frag_pos - last_frag_pos));
  const int nn_num(static_cast<int> ((ran3() * align_depth(frag_pos-shift, size_bin))+1));

  // insert into pose arrays
  for (int k=shift; k<shift+actual_frag_size; ++k) {
    assert(k < frag_size);
    pose.set_phi (pose_pos+k-shift, align_phi(frag_pos-shift, nn_num,k,size_bin));
    pose.set_psi (pose_pos+k-shift, align_psi(frag_pos-shift, nn_num,k,size_bin));
    pose.set_omega (pose_pos+k-shift, align_omega(frag_pos-shift, nn_num,k,size_bin));
    pose.set_secstruct (pose_pos+k-shift, ss_type(frag_pos-shift, nn_num,k,size_bin));
  }
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_loop_parameters
///
/// @brief read loop parameters from the PDB1.prm file
///
/// @detailed
//    file format:
//    -definition             XX
//
//    eg.
//    gap_begin               230
//    gap_end                 250
//    loop_begin              230
//    loop_end                        250
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified March 28 2006
////////////////////////////////////////////////////////////////////////////////
void
get_loop_parameters(
  std::map<std::string, std::string> & params,
  std::string filename
)
{
  using namespace files_paths;

  //open the file for reading
  utility::io::izstream parm_file;
  std::string key, value;

  parm_file.open(filename);

  if (!parm_file) {
    std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
    std::cout << "ERROR: NO LOOP FILE DEFINED                     " << std::endl;
    std::cout << "Must define loop file for -pose_looping mode    " << std::endl;
    std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }

  // splits the line into words
  // puts the first word in the map
  // puts the second word in the value for the map
  while(!parm_file.eof()) {
    parm_file >> key >> value;
    params[key] = value;
  }
}


