// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 17157 $
//  $Date: 2007-09-12 14:04:55 -0700 (Wed, 12 Sep 2007) $
//  $Author: gfriedla $


// Rosetta Headers
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "files_paths.h"
#include "param_aa.h"
#include "pose_io.h"
#include "pose_movie.h"
#include "read_aaproperties.h"

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <iomanip>
#include <iostream>


namespace pose_ns {

////////////////////////////////////////////////////////////////////////////////
/// @begin Movie::Movie
///
/// @brief
/// initialize the new Movie object with Rosetta command line options
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
Movie::Movie()
{
	movie_ = truefalseoption("movie");
	trajectory_ = truefalseoption("trajectory");
	gz_ = truefalseoption("output_pdb_gz");
	one_file_ = true;
	frame_skip_ = intafteroption("movie_frame_skip", 1); // gfriedla
	if (frame_skip_ < 1) { // gfriedla
		std::cout << "WARNING: invalid value for frame_skip (" << frame_skip_ <<
			"). Defaulting to 1." << std::endl;
		frame_skip_ = 1;
	}
	sparse_ = truefalseoption("sparse_trajectory");
	if (sparse_) {
		trajectory_ = true;
	}
	model_ = 0;
	frame_ = 0;
	basename_ = "-";
	if (files_paths::query_defined) {
		basename_ = files_paths::code + files_paths::protein_name;
	}
	stringafteroption("moviefile", basename_, basename_);
	if (movie_ || trajectory_) {
		if ( basename_ == "-" ) {
			std::cout << "WARNING: unable to make movies/trajectories: "
			          << " no basename defined" << std::endl
			          << "either specify series_code and protein_name" << std::endl
			          << "or use -moviefile option" << std::endl;
			movie_ = false;
			trajectory_ = false;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Movie::update_full_coord
///
/// @brief
/// determine which atoms have moved since the last update
///
/// @detailed
/// If this function has never been called, all atoms are marked as having
/// moved.
///
/// @param[in] pose - Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Movie::update_full_coord(
	pose_ns::Pose const & pose
)
{
	const int nres = pose.total_residue();
	const FArray3D_float & full_coord = pose.full_coord();
	nupdates_ = 0;
	nupdates_bb_ = 0;

	// If this is the first time, we need to create the arrays
	if (updated_.size() == 0) {

		full_coord_ = full_coord;
		updated_.resize(nres);

		for (int seqpos = 1; seqpos <= nres; seqpos++) {
			const int aa = pose.res(seqpos);
			const int aav = pose.res_variant(seqpos);
			const int natoms = aaproperties_pack::natoms(aa, aav);
			updated_[seqpos].resize(natoms, true);
			nupdates_ += natoms;
			if (param_aa::is_protein(aa)) {
				nupdates_bb_ += 3;
			} else {
				nupdates_bb_ += natoms;
			}
		}

		return;
	}

	// If this is a subsequent time, then compare the atom locations
	for (int seqpos = 1; seqpos <= nres; seqpos++) {

		const int aa = pose.res(seqpos);
		const int aav = pose.res_variant(seqpos);
		const int natoms = aaproperties_pack::natoms(aa, aav);

		for (int i = 1; i <= natoms; i++) {

			if (full_coord_(1, i, seqpos) != full_coord(1, i, seqpos) ||
			    full_coord_(2, i, seqpos) != full_coord(2, i, seqpos) ||
			    full_coord_(3, i, seqpos) != full_coord(3, i, seqpos)) {

				full_coord_(1, i, seqpos) = full_coord(1, i, seqpos);
				full_coord_(2, i, seqpos) = full_coord(2, i, seqpos);
				full_coord_(3, i, seqpos) = full_coord(3, i, seqpos);

				updated_[seqpos][i] = true;
				nupdates_++;
				if (!param_aa::is_protein(aa) || i <= 3) {
					nupdates_bb_++;
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Movie::dump_updated_full_coord
///
/// @brief
/// write PDB ATOM entries for only the updated atoms
///
/// @detailed
///
/// @param[in] pose - Pose object
/// @param[in] out - output file stream
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Movie::dump_updated_full_coord(
	pose_ns::Pose const & pose,
	std::ostream & out
)
{
	const int nres = pose.total_residue();
	char chain_out = ' ';
	char insert_let_out = ' ';
	std::string atomname;
	std::string resname;
	int atom_number = 0;

	for (int seqpos = 1; seqpos <= nres; seqpos++) {

		const int aa = pose.res(seqpos);
		const int aav = pose.res_variant(seqpos);
		const int natoms = aaproperties_pack::natoms(aa,aav);

		name_from_num(aa,resname);

		for (int i = 1; i <= natoms; i++) {

			atom_number++;
			if (!updated_[seqpos][i]) continue;
			updated_[seqpos][i] = false;

			atom_name_from_atom_num(i,aa,aav,atomname);

			out << "ATOM  " << I(5, atom_number) << ' ' << atomname << ' '
			    << resname << ' ' << chain_out << I(4, seqpos)
				<< insert_let_out << "   " << F(8, 3, full_coord_(1,i,seqpos))
				<< F(8, 3, full_coord_(2,i,seqpos))
				<< F(8, 3, full_coord_(3,i,seqpos)) << '\n';
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Movie::write_frame
///
/// @brief
/// write a movie frame using the coordinates from a Pose object
///
/// @detailed
///
/// @param[in] pose - Pose object
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Movie::write_frame(
	pose_ns::Pose const & pose
)
{
	std::string filename;

	if (!movie_ && !trajectory_) return;

	frame_++;
	if (frame_ % frame_skip_ != 0) return; //gfriedla: don't print every frame if frame_skip flag was used

	if (movie_) {
		filename = basename_ + "movie.pdb" + (gz_ ? ".gz" : "");
		movie_stream_.open(filename);
		dump_full_coord(pose, movie_stream_);
		movie_stream_.close();
	}

	if (trajectory_) {

		if (!one_file_ || !sparse_) model_++;

		if (trajectory_stream_.filename().size() == 0) {
			filename = basename_ + (one_file_ ? "" : lead_zero_string_of(model_, 4))
			           + "T.pdb" + (gz_ ? ".gz" : "");
			trajectory_stream_.open(filename);
		}

		if (sparse_) {
			update_full_coord(pose);
			if (nupdates_) {
				model_++;
				trajectory_stream_ << "MODEL     " << std::setw(4) << model_
				                   << std::endl;
				trajectory_stream_ << "REMARK  99 " << frame_ << std::endl;
				trajectory_stream_ << "REMARK  98 " << pose.get_0D_score(SCORE) << std::endl;
				dump_updated_full_coord(pose, trajectory_stream_);
				trajectory_stream_ << "ENDMDL" << std::endl;
			}
		} else {
			trajectory_stream_ << "MODEL     " << std::setw(4) << model_
			                   << std::endl;
			trajectory_stream_ << "REMARK  99 " << frame_ << std::endl;
			trajectory_stream_ << "REMARK  98 " << pose.get_0D_score(SCORE) << std::endl;
			dump_full_coord(pose, trajectory_stream_);
			trajectory_stream_ << "ENDMDL" << std::endl;
		}

		if (!one_file_) {
			trajectory_stream_.close();
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Movie::close
///
/// @brief
/// close the movie file streams
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void Movie::close()
{
	movie_stream_.close();
	trajectory_stream_.flush();
	trajectory_stream_.close();
}

} // end pose_ns namespace
