// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 8705 $
//  $Date: 2006-06-06 16:15:21 -0700 (Tue, 06 Jun 2006) $
//  $Author: pbradley $


// Rosetta Headers
#include "pose_symmetric_docking.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "atom_tree_routines.h"
#include "design.h"
#include "files_paths.h"
#include "fragments.h"
#include "fold_loops.h"
#include "fullatom.h"
#include "fullatom_setup.h"
#include "jumping_util.h"
#include "minimize.h"
#include "misc.h"
#include "map_sequence.h"
#include "monte_carlo.h"
#include "loops.h"
#include "loop_class.h"
#include "loops_ns.h"
#include "loop_relax.h"
#include "nblist.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pose.h"
#include "pose_abinitio.h"
#include "pose_design.h"
#include "pose_docking.h"
#include "pose_io.h"
#include "pose_loops.h"
#include "pose_rotamer_trials.h"
#include "pose_rms.h"
#include "prof.h"
#include "read_aa_ss.h"
#include "refold.h"
#include "random_numbers.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "silent_input.h"
#include "smallmove.h"
#include "symmetric_design.h"
#include "util_basic.h"
#include "pose_vdw.h"
//Folding
#include "fold_abinitio.h"
#include "fragments_pose.h"
#include "gunn.h"
#include "jumping_util.h"
#include "torsion_bbmove_trials.h"

// Docking
#include "docking.h"
#include "docking_score.h"
#include "docking_ns.h"
#include "dock_structure.h"
#include "docking_minimize.h"
#include "docking_movement.h"

// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpressions.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>


///////////////////////////////////////////////////////////////////////////////
void
symm_move(
	pose_ns::Pose & pose,
	double const rot_mag, // in degrees
	double const trans_mag // in radians
)
{
	int const N( pose.symmetry_info().N() );
	assert(N);

	// Different moves depending on symm_type
	if ( pose.symmetry_info().symm_type() == "helix" ) {
		//For safety reset the degrees of freedom before move
		set_allow_dg_free_helix(pose);
		//Find the jumps
		int const jump_number( pose.num_jump() / N ); // the 1st symm-jump
		int const jump_number2( N+1 ); // the 1st symm-jump
		pose_ns::Jump jump( pose.get_jump( jump_number ) );
		pose_ns::Jump jump2( pose.get_jump( jump_number2 ) );

		// prob not necessary -- fold in any nonzero offsets to
		// translation,rotation
		jump.fold_in_rb_deltas();
		jump2.fold_in_rb_deltas();

		//Random moves: Random rotation of monomers and translation relative the
		//symmetry axes
		jump.reverse();
		jump.set_rotation
	    ( random_axis_rotation_matrix( rot_mag * gaussian() ) * jump.get_rotation() );
		jump.set_translation
			( numeric::xyzVector_double( trans_mag * gaussian(), 0.0, 0.0) +
				jump.get_translation() );
		jump.reverse();

		//Random move: Random translation that control the step between monomers in
		//the symmetry axis, z
		jump2.set_translation
	    ( numeric::xyzVector_double( 0.0, 0.0,  trans_mag * gaussian()) +
			      jump2.get_translation() );
		numeric::xyzMatrix_double Rz( Z_rot(   rot_mag * gaussian()/4 ) ); //Rotation pertubation 4X smaller
		jump2.set_rotation
			( Rz * jump2.get_rotation() );
		//Set jumps
		pose.set_jump( jump_number, jump );
		pose.set_jump( jump_number2, jump2 );

	} else if ( pose.symmetry_info().symm_type() == "cn" ) {

			//For safety reset the degrees of freedom before move
			set_allow_dg_free_cn(pose);
			int const num_jump_mon ( number_jump_monomer(pose) );

			int const jump_number( num_jump_mon + 1 ); // the 1st symm-jump
			pose_ns::Jump jump( pose.get_jump( jump_number ) );

		// prob not necessary -- fold in any nonzero offsets to
		// translation,rotation
		  jump.fold_in_rb_deltas();

		// jumps look from the nterminal residue to the cterminal residue
		// reverse the view since the pseudo rsd is c-terminal,
		// and this is the view that's guaranteed to be standard -- ie
		// z-axis is axis of symmetry, etc
		//
		// plus in this view rotation is about the C-alpha of the nterminal
		// residue, ie
		jump.reverse();

		jump.set_rotation
			( random_axis_rotation_matrix( rot_mag * gaussian() ) * jump.get_rotation() );

		jump.set_translation
			( numeric::xyzVector_double( trans_mag * gaussian(), 0.0, 0.0 ) +
				jump.get_translation() );

		// flip back to standard orientation -- n2c -- before setting in pose
		jump.reverse();

		// pose will replicate this jump to the N-1 other symmetry jumps
		pose.set_jump( jump_number, jump );

	}

	else if ( pose.symmetry_info().symm_type() == "dn" ) {
		//For safety reset the degrees of freedom before move
    set_allow_dg_free_dn(pose);
		 //Find the jumps. OBS!!! The basejump does probably not work!
		int const basejump ( number_jump_monomer( pose ) );
    int const jump_number( basejump +1 ); // the 1st symm-jump
    int const jump_number2( basejump + 2*N -1 ); // the 1st symm-jump
    pose_ns::Jump jump( pose.get_jump( jump_number ) );
    pose_ns::Jump jump2( pose.get_jump( jump_number2 ) );

    // prob not necessary -- fold in any nonzero offsets to
    // translation,rotation
    jump.fold_in_rb_deltas();
    jump2.fold_in_rb_deltas();

		//Random moves: Random rotation of monomers and translation relative the
		//symmetry axes
    jump.reverse();
    jump.set_rotation
      ( random_axis_rotation_matrix( rot_mag * gaussian() ) * jump.get_rotation() );
    jump.set_translation
      ( numeric::xyzVector_double( trans_mag * gaussian(), 0.0, 0.0) +
        jump.get_translation() );
    jump.reverse();


		static bool const dn_sixdim_search = truefalseoption( "dn_sixdim_search" );
		if ( dn_sixdim_search ){
			//let each monomer fully explore the available six dimensions.
			// the z-movement here is actually redundant here with the original
			// z- separation of the two dihedral rings, so turn that off.
			jump.set_translation
				( numeric::xyzVector_double( trans_mag * gaussian(), trans_mag * gaussian(), trans_mag * gaussian()) +
					jump.get_translation() );

		} else {
			//Random move: Random translation of distance between monomers in
			//z-direction
			jump2.set_translation
				( numeric::xyzVector_double( 0.0, 0.0, trans_mag * gaussian()) +
					jump2.get_translation() );

			static bool const dn_twist_rings = truefalseoption("dn_twist_rings");
			if (dn_twist_rings){
				float const angle (rot_mag * gaussian());
				numeric::xyzMatrix_double R( Z_rot(angle) ); //Rotate only around the Z-axis
				numeric::xyzVector_double origin( 0.0, 0.0, 0.0);
				FArray3D_float const & Epos = pose.Eposition();
				int const pseudo_pos = pose.fold_tree().upstream_jump_residue( jump_number2 );
				jump2.rotation_by_matrix( Epos( 1,1, pseudo_pos),
																	origin, R );
			}
		}

    pose.set_jump( jump_number, jump );
    pose.set_jump( jump_number2, jump2 );

  }

	else if ( pose.symmetry_info().symm_type() == "virus" ) {
		//For safety reset the degrees of freedom before move
		set_allow_dg_free_virus(pose);
		 //Find the jumps. OBS!!! The basejump probably does not work!
		int const basejump ( number_jump_monomer( pose ) );
		int const jump_number( basejump + 1 ); // the 1st symm-jump
		int const jump_number2( basejump + N + 1 ); // the 1st symm-jump

		pose_ns::Jump jump( pose.get_jump( jump_number ) );
		pose_ns::Jump jump2( pose.get_jump( jump_number2 ) );

		// prob not necessary -- fold in any nonzero offsets to
		// translation,rotation
		jump.fold_in_rb_deltas();
		jump2.fold_in_rb_deltas();

		//Random rotation: Rotation around three-fold axis and translation relative
		//this rotation axis and translation along the z axis of this sybsystem
		jump.reverse();
    jump.set_rotation
      ( random_axis_rotation_matrix( rot_mag * gaussian() ) * jump.get_rotation() );
    jump.set_translation
      ( numeric::xyzVector_double( trans_mag * gaussian(), 0.0, trans_mag * gaussian() ) +
        jump.get_translation() );
    jump.reverse();

		//Random rotation of the whole threefold system. Is this necessary!
		float const angle (rot_mag * gaussian());
		numeric::xyzMatrix_double R( Z_rot(angle) ); //Rotate only around the Z-axis
		jump2.set_rotation( R * jump2.get_rotation() );

		pose.set_jump( jump_number, jump );
		pose.set_jump( jump_number2, jump2 );

  }

}


///////////////////////////////////////////////////////////////////////////////
void
symm_pose_cn(
	pose_ns::Pose const & monomer_pose,
	bool const fullatom,
	int const N,
	int const anchor_rsd,
	numeric::xyzVector_float const & origin,
	numeric::xyzVector_float const & z_axis,
	pose_ns::Pose & pose // output pose
)
{
	// Setup for cyclic symmetry

	using namespace docking;
	using namespace pose_ns;
	typedef numeric::xyzVector_float Vec;
	typedef numeric::xyzMatrix_float Mat;

	int const njump_monomer( monomer_pose.num_jump() );
	int const nres_monomer( monomer_pose.total_residue() );

	assert( z_axis.is_normalized( 1e-2 ) );
	assert( !fullatom || monomer_pose.fullatom() );

	FArray3D_float Epos( monomer_pose.Eposition() );

	// transform coords
	// so that origin is at the origin, z-axis along z_axis
	static bool const trimer (docking::pose_symm_n_monomers == 3);
	float pseudo_z(0);
	if (trimer || (!docking_small_perturbation && !docking_local_refine) ) {
		Vec anchor_pos( &Epos(1,2,anchor_rsd) );
		Vec v( anchor_pos - origin );
		std::cout <<  "v after: " << v(1) << "\t" << v(2) << "\t" << v(3) << "\n";
		v = v - dot( v, z_axis ) * z_axis;
		assert( std::abs( dot( v, z_axis) ) < 1e-3 );
		Vec const x_axis( v.normalized() );
		Vec const y_axis( cross( z_axis, x_axis ) );
		Mat M( numeric::xyzMatrix_float::rows( x_axis, y_axis, z_axis ) );

		float min_z( 1000.0 );

		for ( int i=1; i<= nres_monomer; ++i ) {
			for ( int j=1; j<= 5; ++j ) {
				Vec w( &Epos(1,j,i));
				w = M*( w - origin );
				if ( i == anchor_rsd && j == 2 ) {
					assert( std::abs( w(2) ) < 1e-2 ); // y-coord = 0
					std::cout<< "w after: " << w(1) << "\t" << w(2) << "\t" << w(3) << "\n";
					std::cout<< "z axis: " << z_axis(1) << z_axis(2) << z_axis(3) << "\n";
				}
					for ( int k=1; k<= 3; ++k ) Epos(k,j,i) = w(k);
				min_z = std::min( w(3), min_z );
			}
		}
		pseudo_z = min_z - 20.0;
	}

	if (!trimer && (docking_small_perturbation || docking_local_refine) ) {
		float min_z( 1000.0 );
		for ( int i=1; i<= nres_monomer; ++i ) {
			 min_z = std::min( Epos(3,2,i), min_z );
			}
		pseudo_z = min_z - 20.0;
	}

	// setup a fold_tree for the symmetric pose
	//
	// the symmetric protein will consist of N copies
	// of monomer pose followed by N pseudo residues
	// for anchoring the monomers. There will be
	//
	// njump = N * ( njump_monomer + 1 )
	//
	// jumps in the symmetric pose. 1 --> njump_monomer * N
	// will be jumps from the monomer, and the last N will be
	// links to the monomers from the pseudo-rsds
	//

	Fold_tree f;
	int const nres( N * nres_monomer + N );
	int const njump( N * njump_monomer + 2*N - 1 );
	int num_fold_tree_cut_points;

	FArray2D_int const & jump_point_monomer
		( monomer_pose.fold_tree().get_jump_point() );
	FArray1D_int const & cuts_monomer
		( monomer_pose.fold_tree().get_fold_tree_cutpoint( num_fold_tree_cut_points ) );

	FArray2D_int jump_point( 2, njump );
	FArray1D_int cuts( njump );


	int jump_number(0);
	int const nres_real( N*nres_monomer );
	for ( int k=0; k<N; ++k ) {
		int const rsd_offset( nres_monomer * k );
		// first the jumps and cuts from the monomer
		for ( int j=1; j<= njump_monomer; ++j ) {
			jump_point( 1, ++jump_number ) = rsd_offset + jump_point_monomer( 1, j );
			jump_point( 2,   jump_number ) = rsd_offset + jump_point_monomer( 2, j );
			cuts( jump_number ) = rsd_offset + cuts_monomer( j );//OBS! CHECKOUT IF RES_OFFSET CORRECT...
		}
	}

	for ( int k=0; k<N; ++k ) {
		++jump_number;
    cuts(jump_number) = nres_monomer*(k+1);
    jump_point(1,jump_number) = nres_monomer*k + anchor_rsd;
    jump_point(2,jump_number) = nres_real + k + 1;
	}

	// N+1 -> N+(N-1) jumps between pseudo-residues
	// note that the first jump is by default the one
	// that will get packed b/c of setup of jump_clones
	// in Symmetry_info. If we move to setting the
	// domain_map so that only interactions with a single
	// peptide (the first) are scored, we need jump N+1 to
	// run from the first peptide to one of its neighbors...
	for ( int i=1; i<N; ++i ) {
	 ++jump_number;
		 cuts( jump_number ) = nres_real+i;
	//	cuts( N+i ) = nres_real+i;
	 // jump_number will be 1 for jump that goes from the first
	 // peptides pseudo residue to the one above it
	// int const jump_number( njump_monomer*N+N+i );
	 int const pseudo_pos1( nres_real+i );
	 int const pseudo_pos2( nres_real+i+1 );
	 jump_point(1,jump_number) = pseudo_pos1;
	 jump_point(2,jump_number) = pseudo_pos2;
	}

//	assert( jump_number == njump );
	// this builds the tree:
	f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );

	// reorder the tree so that it is rooted at 1st pseudo-rsd
	f.reorder( nres_real + 1);

	// put into pose
	pose.set_fold_tree( f );

	if ( fullatom ) pose.set_fullatom_flag( true, false /* dont repack */ );

	// copy some coordinates, torsions, sidechains if fullatom
	for ( int k=0; k<N; ++k ){
		pose.copy_segment( nres_monomer, monomer_pose, k*nres_monomer+1, 1 );
	}
	// create some coords at the end
	Pose pseudo_pose;

	if ( docking_pose_symm_subsystem ) {
    setup_pseudo_pose( pose_symm_n_monomers, pseudo_z,
      pseudo_pose );
    pose.copy_segment( docking_pose_symm_subsystem_size, pseudo_pose,
        nres_real+1, 1 );
       }
  else {
    setup_pseudo_pose( N, pseudo_z, pseudo_pose );
    pose.copy_segment( N, pseudo_pose, nres_real+1, 1 );
  }
	// setup the initial orientation
	// want to transform so that z_axis is parallel the z-axis
	// and xy_origin is along the z-axis
	//
	//

	Jump jump( Epos(1,1,anchor_rsd),
											pose.Eposition()(1,1,nres_real+1) );

	// declare symmetric pose:
	if ( docking_pose_symm_subsystem ) {
		pose.setup_symm_info( nres_monomer, njump_monomer, N, 2, "cn" );
	} else {
		pose.setup_symm_info( nres_monomer, njump_monomer, N, 1, "cn" );
	}

	pose.set_jump( N*njump_monomer + 1, jump );

	if (docking_small_perturbation && !docking_local_refine) {
		symm_move( pose, normal_perturbation, rotational_perturbation );
	}

	//symm_check( pose );
	// Setup for symmetrical packer
	if ( docking::docking_pose_symm_full ) {
   if ( docking_pose_symm_subsystem ) {
		param::MAX_CLONES() = 2;
    design_sym::num_clones = 1;
	 } else {
			param::MAX_CLONES() = docking::pose_symm_n_monomers;
			design_sym::num_clones = docking::pose_symm_n_monomers-1;
		}
	}
	// Set the allowed degrees of freedom
	set_allow_dg_free_cn(pose);
	// exit routine here
	return;
}

///////////////////////////////////////////////////////////////////////////////

void
symm_pose_helix(
	pose_ns::Pose & monomer_pose,
	bool const fullatom,
	int const N,
	int const anchor_rsd,
	float const alfa,
	float const step,
	float const radius,
	pose_ns::Pose & pose // output pose
	)
{
	//create symmetric pose for helical symmetry
	using namespace docking;
	using namespace misc;
	using namespace pose_ns;
	using numeric::constants::f::pi;
	using namespace numeric;

	Pose oligomer_pose, pseudo_pose;
	int nres_monomer = monomer_pose.total_residue();

	numeric::xyzVector_float x_axis(1.0,0.0,0);
	numeric::xyzVector_float delta (radius*x_axis);

	FArray3D_float Epos( monomer_pose.Eposition() );
	FArray3D_float fcoord( monomer_pose.full_coord() );

	// Translate monomer to origin
	for ( int i=1; i<=nres_monomer; ++i ) {
		int const natoms( aaproperties_pack::natoms( res(i), res_variant(i) ) );
		for ( int k=1; k<=3; ++k ) {
			for ( int j=1; j<= natoms; ++j ) {
				if ( j <= 5) {
					Epos(k,j,i) = monomer_pose.Eposition()(k,j,i)-monomer_pose.Eposition()(k,2,anchor_rsd) + delta(k);
				}
				fcoord(k,j,i) = monomer_pose.full_coord()(k,j,i)-monomer_pose.full_coord()(k,2,anchor_rsd) + delta(k);
			}
		}
	}
	monomer_pose.set_coords( false, Epos, fcoord );

	//Get jumps and cuts for a monomer
	FArray2D_int const & jump_point_monomer
		( monomer_pose.fold_tree().get_jump_point() );
	int num_fold_tree_cut_points;
	FArray1D_int const & cuts_monomer
		( monomer_pose.fold_tree().get_fold_tree_cutpoint(num_fold_tree_cut_points) );

	// Create the internal jumps and cuts in the system
	int const njump_monomer( monomer_pose.num_jump() );
	int const nres( N * nres_monomer );
	int const njump( N * njump_monomer );

	FArray2D_int jump_point( 2, njump );
	FArray1D_int cuts( njump );

	int jump_number(0);
	for ( int k=0; k<N; ++k ) {
		int const rsd_offset( nres_monomer * k );
		// first the jumps and cuts from the monomer
		for ( int j=1; j<= njump_monomer; ++j ) {
			jump_point( 1, ++jump_number ) = rsd_offset + jump_point_monomer( 1, j );
		  jump_point( 2,   jump_number ) = rsd_offset + jump_point_monomer( 2, j );
		  cuts( jump_number ) = cuts_monomer( j );
		}
	}
	Fold_tree f;
	// this builds the tree:
	f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );

	oligomer_pose.set_fold_tree( f );

	if (fullatom) {
		oligomer_pose.set_fullatom_flag( true, false /* dont repack */ );
	}
	// copy some coordinates, torsions, sidechains if fullatom
	for ( int k=0; k<N; ++k ){
		oligomer_pose.copy_segment( nres_monomer, monomer_pose, k*nres_monomer+1, 1 );
	}

	// setup the fold_tree
	Fold_tree ft;
	int const num_jump( 2*N-1 );
	FArray2D_int jumps_final(2,num_jump);
	FArray1D_int cuts_final(num_jump);

	// 1-N jumps from pseudo-residues to peptides
	for ( int i=1; i<= N; ++i ) {
		int const peptide_pos( anchor_rsd +nres_monomer*(i-1) );
		int const pseudo_pos( nres_monomer*N + i );
		jumps_final(1,i) = peptide_pos;
		jumps_final(2,i) = pseudo_pos;
		cuts_final(i) = i*nres_monomer;
	}

	// N+1 -> N+(N-1) jumps between pseudo-residues
	// note that the first jump is by default the one
	// that will get packed b/c of setup of jump_clones
	// in Symmetry_info. If we move to setting the
	// domain_map so that only interactions with a single
	// peptide (the first) are scored, we need jump N+1 to
	// run from the first peptide to one of its neighbors...
	for ( int i=1; i<N; ++i ) {
	 cuts_final( N+i ) = nres+i;
	 // jump_number will be 1 for jump that goes from the first
	 // peptides pseudo residue to the one above it
	 int const jump_number( N+i );
	 int const pseudo_pos1( nres+i );
	 int const pseudo_pos2( nres+i+1 );
	 jumps_final(1,jump_number) = pseudo_pos1;
	 jumps_final(2,jump_number) = pseudo_pos2;
	}

	int const nres_symm( nres+N);
	ft.tree_from_jumps_and_cuts( nres_symm, num_jump, jumps_final, cuts_final );
	ft.reorder(nres+1);

	// make the final symmetric pose!
	pose.set_fold_tree( ft );
	if ( fullatom ) pose.set_fullatom_flag( true, false /* dont repack */ );
	pose.copy_segment( nres, oligomer_pose, 1, 1 );

	// If we have a subsystem with create only docking_pose_symm_subsystem_size
	// pseudo residues
	if ( docking_pose_symm_subsystem ) {
		setup_pseudo_pose_helix(docking_pose_symm_subsystem_size, step, alfa, pseudo_pose);
    pose.copy_segment( docking_pose_symm_subsystem_size, pseudo_pose,
        nres+1, 1 );
       }
  else {
		setup_pseudo_pose_helix(N, step, alfa, pseudo_pose);
    pose.copy_segment( N, pseudo_pose, nres+1, 1 );
  }
	// declare symmetric
	pose.setup_symm_info( nres_monomer, 0, N, 2, "helix" );

	//Set the jumps in the system
	int const jump_nbr1( 1 ); // the 1st symm-jump
	int const jump_nbr2( N+1 );
	Jump jump1( pose.get_jump( jump_nbr1 ) );
	Jump jump2( pose.get_jump( jump_nbr2 ) );

  pose.set_jump( jump_nbr1, jump1 );
  pose.set_jump( jump_nbr2, jump2 );

	if ( docking_randomize1 || docking_randomize2 ) {
		jump1.reverse();
		jump1.set_rotation( random_reorientation_matrix() * jump1.get_rotation() );
		jump1.reverse();
		double const step_jump ( (ran3()-0.5) * 60 );
		std::cout<<"random step "<<step_jump<<std::endl;
		double const random_rot ( (ran3()-0.5) * 4.15 );
		numeric::xyzMatrix_double Rz( Z_rot(random_rot) );
		jump2.set_rotation( Rz * jump2.get_rotation() );
		jump2.set_translation
		      ( numeric::xyzVector_double( 0.0, 0.0, step_jump) +
					            jump2.get_translation() );
		pose.set_jump( jump_nbr1, jump1 );
		pose.set_jump( jump_nbr2, jump2 );
	}
	if (docking_small_perturbation && !docking_local_refine) {
		symm_move( pose, normal_perturbation, rotational_perturbation );
	}
	// Setup for symmetrical packer
	if ( docking_pose_symm_full ) {
   if ( docking_pose_symm_subsystem ) {
    param::MAX_CLONES() = docking_pose_symm_subsystem_size;
    design_sym::num_clones = docking_pose_symm_subsystem_size-1;
   } else {
      param::MAX_CLONES() = pose_symm_n_monomers;
      design_sym::num_clones = pose_symm_n_monomers-1;
    }
  }
	// Set the allowed degrees of freedom in the system
	set_allow_dg_free_helix(pose);
	}

///////////////////////////////////////////////////////////////////////////////
void
create_symmetric_pose(
	pose_ns::Pose & monomer_pose,
	bool const fullatom,
	int const N,
	int const anchor,
	numeric::xyzVector_float origin,
	numeric::xyzVector_float z_axis,
	std::string symmetry_type,
	pose_ns::Pose & pose // output pose
	)
	{
		using namespace docking;

		std::cout<<"Symmetry docking with "<<symmetry_type<<" symmetry"<<std::endl;

		if (symmetry_type == "cn" ) {
			if ( docking_pose_symm_subsystem ) {	//Subsystem with two monomers
				docking_pose_symm_subsystem_size = 3;
				symm_pose_cn( monomer_pose, fullatom, docking_pose_symm_subsystem_size, anchor, origin, z_axis, pose );
			} else {
				symm_pose_cn( monomer_pose, fullatom, N, anchor, origin, z_axis, pose );
			}
		}
		if ( symmetry_type == "helix" ) {
			  float alfa (15.92);	// These parameters need to be adjusted and randomized?
			  float step (30.5);
				float radius (120);
				if (docking::docking_local_refine || docking::docking_small_perturbation){
					//alfa = 15;
					//step = -2.568;
					//radius = 69.164;
				//	alfa = 15.3;
				//	step = 0.807;
				//	radius = 70.206;
				//		alfa = 15.453;
				//		step = 0.807;
				//		radius = 40+70.206;
			//	 alfa = 15.352;
			//	 step= +3.511;//0.571;
			//	 radius = 68.605;
				 alfa = 13.2768;
				 step= -2.632;
				radius = 79.205;

				}
		    if ( docking_pose_symm_subsystem ) { //Subsystem with two monomers
					docking_pose_symm_subsystem_size = 3;
					symm_pose_helix(monomer_pose, fullatom, docking_pose_symm_subsystem_size, anchor, alfa, step, radius, pose);
				} else {
					symm_pose_helix(monomer_pose, fullatom, N, anchor, alfa, step, radius, pose);
				}
	  }
		if ( symmetry_type == "dn" ) {
				float dist (50);	// These parameters need to be adjusted and randomized?
				float radius (50);
				float angle (0);
				if ( docking_pose_symm_subsystem ) { //Subsystem with four monomers
				// For the special case of dn we migh want to simulate only 3 monomers
					docking_pose_symm_subsystem_size = 4;
					if ( dn_with_3 ){
						docking_pose_symm_subsystem_size = 3;
					} else if ( dn_with_5 ) {
						docking_pose_symm_subsystem_size = 5;
					}
				if ( docking_local_refine || docking_small_perturbation ) {
					radius = realafteroption("dn_radius", 50);
					dist = realafteroption("dn_dist", 50);
					angle = realafteroption("dn_angle", 0);
				}
					symm_pose_dn(monomer_pose, fullatom, docking_pose_symm_subsystem_size , anchor, dist, radius, angle, pose);
				} else {
					if ( docking_local_refine || docking_small_perturbation ) {
						angle = angle - 180.0; // 180.0 is the default value
						pose_ns::Pose oligomer_pose;
						prepare_native_d2( pose, monomer_pose, oligomer_pose );
						pose = oligomer_pose;
					} else {
					symm_pose_dn(monomer_pose, fullatom, N, anchor, dist, radius, angle, pose);
					}
				}
			}

		if ( symmetry_type == "virus" ) {
			float dist (110);	// These parameters are probably ok. Small deviations from angle=0 are expected
			float radius(30);
			float angle(0);
			if (docking::docking_local_refine || docking::docking_small_perturbation){
				//dist=35.36;
				//radius=17.3-11.548;
				//angle=0;
	//			dist=3+68.16-32.361;
	//			radius=18.01-11.548;
	//			angle = -8;
	//			dist=70.7058-30.2305+40;
	//			radius=18.9299 - 11.547+20;
	//			angle=-8.26845;
	//				dist=68.7509-30.2305;
	//				radius=17.2985 - 11.547;
	//				angle=-11.3817;
				dist=70.3086-30.2305;
				radius=18.6009 - 11.547;
				angle=-8.86402;
			}
			if ( docking_pose_symm_subsystem ) { // We need 6 monomers at least in this system
				docking_pose_symm_subsystem_size = 6;
				symm_pose_virus(monomer_pose, fullatom, docking_pose_symm_subsystem_size, anchor, dist, radius, angle, pose);
			} else {
				symm_pose_virus(monomer_pose, fullatom, N, anchor, dist, radius, angle, pose);
			}
		}

	}
////////////////////////////////////////////////////////////////////////////////////////////
void
symm_pose_dn(
  pose_ns::Pose & monomer_pose,
  bool const fullatom,
  int const N,
  int const anchor,
  float const dist,
  float const radius,
	float const angle,
  pose_ns::Pose & pose // output pose
  )
{
	using namespace misc;
	using namespace docking;
  using namespace pose_ns;
  using numeric::constants::f::pi;
  using namespace numeric;

  pose_ns::Pose oligomer_pose, pseudo_pose;
  int nres_monomer = monomer_pose.total_residue();

  numeric::xyzVector_float z_axis, origin;
  numeric::xyzVector_float x_axis(1.0,0.0,0);
  numeric::xyzVector_float delta (radius*x_axis);

  FArray3D_float Epos( monomer_pose.Eposition() );
  FArray3D_float Epos2( monomer_pose.Eposition() );
  FArray3D_float fcoord( monomer_pose.full_coord() );
  FArray3D_float fcoord2( monomer_pose.full_coord() );

  for ( int i=1; i<=nres_monomer; ++i ) {
    int const natoms( aaproperties_pack::natoms( res(i), res_variant(i) ) );
    for ( int k=1; k<=3; ++k ) {
      for ( int j=1; j<= natoms; ++j ) {
        if ( j <= 5)
          Epos(k,j,i) = monomer_pose.Eposition()(k,j,i)-monomer_pose.Eposition()(k,2,anchor) + delta(k);
        fcoord(k,j,i) = monomer_pose.full_coord()(k,j,i)-monomer_pose.full_coord()(k,2,anchor) + delta(k);
        }
      }
    }
    monomer_pose.set_coords( false, Epos, fcoord );
		FArray2D_int const & jump_point_monomer
    ( monomer_pose.fold_tree().get_jump_point() );
		int num_fold_tree_cut_points;
  FArray1D_int const & cuts_monomer
    ( monomer_pose.fold_tree().get_fold_tree_cutpoint(num_fold_tree_cut_points) );
  int const njump_monomer( monomer_pose.num_jump() );
  int const nres( N * nres_monomer );
  int const njump( N * njump_monomer );
  FArray2D_int jump_point( 2, njump );
  FArray1D_int cuts( njump );

  int jump_number(0);

  for ( int k=0; k<N; ++k ) {
    int const rsd_offset( nres_monomer * k );
    // first the jumps and cuts from the monomer
      for ( int j=1; j<= njump_monomer; ++j ) {
        jump_point( 1, ++jump_number ) = rsd_offset + jump_point_monomer( 1, j );
        jump_point( 2,   jump_number ) = rsd_offset + jump_point_monomer( 2, j );
        cuts( jump_number ) = cuts_monomer( j );
      }
  }
  Fold_tree f;
  // this builds the tree:
  f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );

  oligomer_pose.set_fold_tree( f );
  oligomer_pose.set_fullatom_flag( true, false /* dont repack */ );

  // copy some coordinates, torsions, sidechains if
  // fullatom

  for ( int k=0; k<N; ++k ){
    oligomer_pose.copy_segment( nres_monomer, monomer_pose, k*nres_monomer+1, 1 );
  }
	// Create pseudo pose
	if ( docking_pose_symm_subsystem ) {
		setup_pseudo_pose_dn( pose_symm_n_monomers, pseudo_pose );
	} else {
		setup_pseudo_pose_dn(N, pseudo_pose);
	}

	// setup the fold_tree
  Fold_tree ft;
  int const num_jump(N*njump_monomer+2*N-1 );
  FArray2D_int jumps_final(2,num_jump);
  FArray1D_int cuts_final(num_jump);

  // Copy the internal monomer jumps and cuts

  int const internal_jumps (njump_monomer*N);
  for ( int i=1; i<= internal_jumps; ++i ) {
    int const jump (i+internal_jumps);
    jumps_final(1,jump) = jump_point(1,i);
    jumps_final(2,jump) = jump_point(2,i);
    cuts_final(jump) = cuts(i);
  }

// 1-N jumps from pseudo-residues to peptides
    for ( int i=1; i<= N; ++i ) {
      int const jump (i+internal_jumps);
      int const peptide_pos( anchor +nres_monomer*(i-1) );
      int const pseudo_pos(nres+i);
      jumps_final(1,jump) = peptide_pos;
      jumps_final(2,jump) = pseudo_pos;
      cuts_final(jump) = i*nres_monomer;
    }

	// We have three differents setups for dn and subsytem
	// 4 (standard), 3 and 5 monomers. This is controlled by the flags
	// dn_with_3/5.

	if ( docking_pose_symm_subsystem ) {
		int mon = N-1;
		if ( dn_with_5 ) mon = N-2;
	  for ( int i=1; i<mon; ++i ) {
			int const jump (i+internal_jumps);
			cuts_final( N+jump ) = nres+i;
			int const jump_number( N+jump );
			int const pseudo_pos1( nres+i );
			int const pseudo_pos2( nres+i+1 );
			jumps_final(1,jump_number) = pseudo_pos1;
			jumps_final(2,jump_number) = pseudo_pos2;
		}
		if ( dn_with_5 ) {
			jumps_final(1,8) = nres + 4;
			jumps_final(2,8) = nres + 5;
		}
	} else {

	  for ( int i=1; i<N/2; ++i ) {
			int const jump (i+internal_jumps);
			cuts_final( N+jump ) = nres+i;
			int const jump_number( N+jump );
			int const pseudo_pos1( nres+i );
			int const pseudo_pos2( nres+i+1 );
			jumps_final(1,jump_number) = pseudo_pos1;
			jumps_final(2,jump_number) = pseudo_pos2;
		}

	}

	if ( docking_pose_symm_subsystem ) {
			if (dn_with_5 ) {
				cuts_final(8) = nres + 3;
			}
	} else {

		for ( int i=N/2; i<N-1; ++i ) {
			int const jump (i+internal_jumps);
			cuts_final( N+jump ) = nres+i;
			int const jump_number( N+jump );
			int const pseudo_pos1( nres+i+1 );
			int const pseudo_pos2( nres+i+2 );
			jumps_final(1,jump_number) = pseudo_pos1;
			jumps_final(2,jump_number) = pseudo_pos2;
		}

	}

	int jumb_nbr_final (internal_jumps + 2*N-1);
  cuts_final( jumb_nbr_final ) = nres+N-1;
  jumps_final(1, jumb_nbr_final ) = nres+1;
	if ( docking_pose_symm_subsystem ) {
		int final_pseudo;
		if ( dn_with_3 ) {
			final_pseudo = 3;
			jumb_nbr_final = 5;
		}	else if ( dn_with_5) {
			final_pseudo = 5;
			jumb_nbr_final = 9;
		} else {
			final_pseudo = 4;
		}
		jumps_final(2, jumb_nbr_final ) = nres+final_pseudo;
	} else {
		jumps_final(2, jumb_nbr_final ) = nres+N/2+1;
	}

	int const nres_symm( nres+N);
  ft.tree_from_jumps_and_cuts( nres_symm, num_jump, jumps_final, cuts_final );
  ft.reorder(nres+1);
  // make the final symmetric pose!

  pose.set_fold_tree( ft );
  if ( fullatom ) pose.set_fullatom_flag( true, false /* dont repack */ );
		pose.copy_segment( nres, oligomer_pose, 1, 1 );
	if ( docking_pose_symm_subsystem ) {
		pose.copy_segment( docking_pose_symm_subsystem_size, pseudo_pose,
											nres+1, 1 );
	} else {
		pose.copy_segment( N, pseudo_pose, nres+1, 1 );
	}

	if ( dn_with_3 ) {
		pose.setup_symm_info( nres_monomer, internal_jumps, N, 3, "dn" );
	} else {
		// declare symmetric
		pose.setup_symm_info( nres_monomer, internal_jumps, N, 2, "dn" );
	}

  int const jump_nbr1( internal_jumps + 1 ); // the 1st symm-jump
  int const jump_nbr2( internal_jumps + 2*N-1 );

  Jump jump1( pose.get_jump( jump_nbr1 ) );
  Jump jump2( pose.get_jump( jump_nbr2 ) );

	if ( !docking_local_refine && !docking_small_perturbation ) {
		jump1.reverse();
		jump1.set_rotation
			( random_reorientation_matrix() * jump1.get_rotation() );
		jump1.reverse();
	}
  pose.set_jump( jump_nbr1, jump1 );
  pose.set_jump( jump_nbr2, jump2 );

  jump2.reverse();
	jump2.set_translation( numeric::xyzVector_double( 0.0, 0.0, dist ) + jump2.get_translation() );
  jump2.reverse();

  pose.set_jump( jump_nbr2, jump2 );

	float rot_angle (angle);
	if ( !docking_local_refine && !docking_small_perturbation ) {
		float const interval = 360.0/pose_symm_n_monomers;
		rot_angle = ran3()*interval;
	}

	rot_angle = realafteroption("dn_angle", rot_angle);
  numeric::xyzVector_double center( 0.0, 0.0, 0.0);
  Fold_tree ft_final( pose.fold_tree() );
  int const pseudo_pos = ft_final.upstream_jump_residue( jump_nbr2 );
  jump2.rotation_by_matrix( pose.Eposition()( 1,1, pseudo_pos),
                                center, Z_rot( rot_angle) );
	
  pose.set_jump( jump_nbr2, jump2 );

	set_allow_dg_free_dn(pose);

	// Setup for symmetrical packer
  if ( docking::docking_pose_symm_full ) {
    param::MAX_CLONES() = docking::pose_symm_n_monomers;
    design_sym::num_clones = docking::pose_symm_n_monomers-1;
  }
}
////////////////////////////////////////////////////////////////////////////////////////////

void
symm_pose_virus(
  pose_ns::Pose & monomer_pose,
  bool const fullatom,
  int const N,
  int const anchor,
  float const dist,
  float const radius,
	float const angle,
  pose_ns::Pose & pose // output pose
  )
{
	using namespace docking;
	using namespace misc;
	using namespace pose_ns;
	using numeric::constants::f::pi;
	using namespace numeric;

	int const njump_monomer( monomer_pose.num_jump() );
	int nres_monomer = monomer_pose.total_residue();

	//Setup pseudo pose

	Pose pseudo_pose;
	numeric::xyzVector_float rot_axis, monomer_pos;
	// Find the coordinates for the first monomer on the first face of the
	// icosahedron
	find_coordinates_virus(rot_axis,monomer_pos);

	FArray3D_float Epos( monomer_pose.Eposition() );
	FArray3D_float fcoord( monomer_pose.full_coord() );
	monomer_pose.set_coords( false, Epos, fcoord );
	numeric::xyzVector_float translate_vector (dist*rot_axis+monomer_pos);

	// Translate to origo and out to the first face of the icosahedron

	for ( int i=1; i<=nres_monomer; ++i ) {
	  int const natoms( aaproperties_pack::natoms( res(i), res_variant(i) ) );
	  for ( int k=1; k<=3; ++k ) {
		  for ( int j=1; j<= natoms; ++j ) {
		   if ( j <= 5) {
		     Epos(k,j,i) = monomer_pose.Eposition()(k,j,i)-monomer_pose.Eposition()(k,2,anchor) + translate_vector(k);
				}
		   fcoord(k,j,i) = monomer_pose.full_coord()(k,j,i)-monomer_pose.full_coord()(k,2,anchor) + translate_vector(k);
		  }
		}
	}
	monomer_pose.set_coords( false, Epos, fcoord );

	//Setup the pseudo_pose residues of the system
	setup_pseudo_pose_virus(N,pseudo_pose);
	// Setup the fold tree
	Fold_tree f;
	int const nres( N * nres_monomer + 2*N );
	int const njump( N * njump_monomer + 2*N );

	FArray2D_int const & jump_point_monomer
		( monomer_pose.fold_tree().get_jump_point() );
	int num_fold_tree_cut_points;
	FArray1D_int const & cuts_monomer
		( monomer_pose.fold_tree().get_fold_tree_cutpoint(num_fold_tree_cut_points) );

	FArray2D_int jump_point( 2, njump );
	FArray1D_int cuts( njump );


	int jump_number(0);
	int const nres_real( N*nres_monomer );
	for ( int k=0; k<N; ++k ) {
		int const rsd_offset( nres_monomer * k );
		// first the jumps and cuts from the monomer
		for ( int j=1; j<= njump_monomer; ++j ) {
			jump_point( 1, ++jump_number ) = rsd_offset + jump_point_monomer( 1, j );
			jump_point( 2,   jump_number ) = rsd_offset + jump_point_monomer( 2, j );
			cuts( jump_number ) = rsd_offset + cuts_monomer( j );//OBS! CHECKOUT IF RES_OFFSET CORRECT...
		}
	}

	// Jumps between pseudo residues and corresponding monomer
	for ( int k=0; k<N; ++k ) {
		++jump_number;
    cuts(jump_number) = nres_monomer*(k+1);
    jump_point(1,jump_number) = nres_monomer*k + anchor;
    jump_point(2,jump_number) = nres_real + k + 1;
	}

	// Jumps between threefold pseudoresidues and center pseudoresidues
	for ( int i=1; i<=N; ++i ) {
       ++jump_number;
			cuts( N+i ) = nres_real+i;
      int const pseudo_pos1( nres_real+i );
      int const pseudo_pos2( nres_real+i+N );
      jump_point(1,jump_number) = pseudo_pos1;
      jump_point(2,jump_number) = pseudo_pos2;
    }

	assert( jump_number == njump );

	// this builds the tree:
	f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );

	// reorder the tree so that it is rooted at 1st pseudo-rsd
	f.reorder( nres_real +  N + 1);

	// put into pose
	pose.set_fold_tree( f );
	if ( fullatom ) pose.set_fullatom_flag( true, false /* dont repack */ );

	// copy some coordinates, torsions, sidechains if fullatom
	for ( int k=0; k<N; ++k ){
		pose.copy_segment( nres_monomer, monomer_pose, k*nres_monomer+1, 1 );
	}

  pose.copy_segment( 2*N, pseudo_pose, nres_real+1, 1 );

	// declare symmetric pose. We score the first monomer which need to be in the
	// "middle"
	pose.setup_symm_info( nres_monomer, njump_monomer, N, 1, "virus" );

	int const jump_nbr1( 1 ); // the 1st symm-jump. rotation of monomers and translations around
														// threefold axis
  int const jump_nbr2( N+1 ); //The second jump. Rotation of the the three monomers together
  Jump jump1( pose.get_jump( jump_nbr1 ) );
  Jump jump2( pose.get_jump( jump_nbr2 ) );

	pose.set_jump( jump_nbr1, jump1 );
	numeric::xyzVector_double x, z, ori;
	return_xyz_vectors_virus(x, z, ori);
	// Set jumps
	jump1.reverse();
	jump1.set_translation
      ( numeric::xyzVector_double( radius, 0.0, 0.0 ) +
				        jump1.get_translation() );
//	if ( docking::docking_small_perturbation ) {
//		numeric::xyzMatrix_double Rz( Z_rot(-5.0) );
//		jump1.set_rotation( Rz * jump1.get_rotation() );
//	}
	jump1.reverse();
	numeric::xyzMatrix_double R( Z_rot(angle) );
	jump2.set_rotation( R * jump2.get_rotation() );

	pose.set_jump( jump_nbr1, jump1 );
	pose.set_jump( jump_nbr2, jump2 );

	// Randomize the starting positions
	if ( docking::docking_randomize1 || docking::docking_randomize2 ) {

		float const rot (30);

		jump1.reverse();
    jump1.set_rotation
      ( random_reorientation_matrix() * jump1.get_rotation() );
    jump1.reverse();

		float const angle_rand (2 * rot * (0.5-ran3()) );
		numeric::xyzMatrix_double Rz( Z_rot(angle_rand) );
		jump2.set_rotation( Rz * jump2.get_rotation() );

		pose.set_jump( jump_nbr1, jump1 );
		pose.set_jump( jump_nbr2, jump2 );

	}

	if (docking_small_perturbation && !docking_local_refine) {
		symm_move( pose, normal_perturbation, rotational_perturbation );
	}

	//Set the degrees of freedom to vary
	set_allow_dg_free_virus(pose);

  // Setup for symmetrical packer
  if ( docking::docking_pose_symm_full ) {
    param::MAX_CLONES() = docking::pose_symm_n_monomers;
    design_sym::num_clones = docking::pose_symm_n_monomers-1;
  }

}
void
setup_pseudo_pose(
	int const N,
	float const z,
	pose_ns::Pose & pose
)
{
	// Setup pseudo pose for cyclic symmetry
	using namespace docking;
	using numeric::constants::f::pi;

	float const radius( 40.0 );
	float const length(  1.5 );
	pose.simple_fold_tree( N );

	// Set coordinates for psudo glycines. The n-ca vector
	// points to the symmetry axis
	numeric::xyzVector_float n,ca,cb,c,o;
	ca(1) = radius;
	ca(2) = 0.0;
	ca(3) = z;

	cb(1) = radius;
	cb(2) = 0.0;
	cb(3) = z + length;

	n(1) = radius + length;
	n(2) = 0.0;
	n(3) = z;

	c(1) = radius;
	c(2) = length;
	c(3) = z;

	o(1) = radius-length;
	o(2) = length;
	o(3) = z;

	//Rotate the glycines around the z-axis
	FArray3D_float Epos(3, param::MAX_POS , N );
	for ( int i=1; i<=N; ++i ) {
		pose.set_res        ( i, param_aa::aa_gly );
		pose.set_res_variant( i, 1 );
		numeric::xyzMatrix_float const M( Z_rot_rad( ( i-1) * 2 * pi / N ) );
		for ( int k=1; k<= 3; ++k ){
			Epos(k,1,i) = ( M*n  )(k);
			Epos(k,2,i) = ( M*ca )(k);
			Epos(k,3,i) = ( M*cb )(k); // cbeta doesnt matter for gly
			Epos(k,4,i) = ( M*c  )(k);
			Epos(k,5,i) = ( M*o  )(k);
		}
	}

	pose.set_coords( false, Epos, FArray3D_float() );
	pose.set_fullatom_flag( true, false ); // only glycine anyway

}

///////////////////////////////////////////////////////////////////////////////
void
setup_pseudo_pose_helix(
	int const N,
	float const step,
	float  const alfa,
  pose_ns::Pose & pose
)
{
	//Setup psedu residues for helical symmetry

	using numeric::constants::f::pi;
	pose.simple_fold_tree( N );
	FArray3D_float Eposition(3,param::MAX_POS, N );

	// Set coordinates for psudoglycines
	float const length( 1.5 );
	numeric::xyzVector_float n,ca,c,o,cb;
	ca(1) = 0.0;
	ca(2) = 0.0;
	ca(3) = 0.0;

	n(1) = length;
	n(2) = 0.0;
	n(3) = 0.0;

	c(1) = 0.0;
	c(2) = length;
	c(3) = 0.0;

	o(1) = -length;
	o(2) = length;
	o(3) = 0.0;

	cb(1) = 0.0;
	cb(2) = 0.0;
	cb(3) = length;

	std::vector<int> place_pseudo;

  for (int i=1; i<=N;++i) place_pseudo.push_back(i);

	// Rotate and translate the glycines
	for ( int i=1; i<= N; ++i ) {
		pose.set_res        (i,param_aa::aa_gly);
		pose.set_res_variant(i,1);
		numeric::xyzMatrix_float const R( Z_rot_rad( ( i-1) * alfa/360 * 2 * pi ) );
			numeric::xyzVector_float v( 0.0, 0.0, (i-1)*step ); //Translate along z-axis step length
		int res ( place_pseudo[i-1] );
			for ( int k=1; k<= 3; ++k ) {
				Eposition(k,1,res) = (R * n  + v )(k);
				Eposition(k,2,res) = (R * ca + v )(k);
				Eposition(k,3,res) = (R * cb + v )(k);
				Eposition(k,4,res) = (R * c  + v )(k);
				Eposition(k,5,res) = (R * o  + v )(k);
			}
	}

	pose.set_coords( false, Eposition, FArray3D_float() );
	pose.set_fullatom_flag( true, false ); // build HA,HN
}
///////////////////////////////////////////////////////////////////////////////
void
setup_pseudo_pose_dn(
	int const N,
	pose_ns::Pose & pose
)
{
	// Setup pseudo pose for cyclic symmetry
	using namespace docking;
	using numeric::constants::f::pi;

	float const radius( 40.0 );
	float const length(  1.5 );
	pose.simple_fold_tree( N );

	// Set coordinates for psudo glycines. The n-ca vector
	// points to the symmetry axis
	numeric::xyzVector_float n,ca,cb,c,o;
	ca(1) = radius;
	ca(2) = 0.0;
	ca(3) = 0.0;

	cb(1) = radius;
	cb(2) = 0.0;
	cb(3) = length;

	n(1) = radius + length;
	n(2) = 0.0;
	n(3) = 0.0;

	c(1) = radius;
	c(2) = length;
	c(3) = 0.0;

	o(1) = radius-length;
	o(2) = length;
	o(3) = 0.0;

	//Rotate the glycines around the z-axis
	FArray3D_float Epos(3, param::MAX_POS , N );
	for ( int i=1; i<=N; ++i ) {
		pose.set_res        ( i, param_aa::aa_gly );
		pose.set_res_variant( i, 1 );
		numeric::xyzMatrix_float const M( Z_rot_rad( ( i-1) * 4 * pi / N ) );
		if ( i <= N/2 ) {
			for ( int k=1; k<= 3; ++k ){
				Epos(k,1,i) = ( M*n  )(k);
				Epos(k,2,i) = ( M*ca )(k);
				Epos(k,3,i) = ( M*cb )(k); // cbeta doesnt matter for gly
				Epos(k,4,i) = ( M*c  )(k);
				Epos(k,5,i) = ( M*o  )(k);
			}
		} else {
			numeric::xyzVector_float rot_axis(1.0,0.0,0.0);
			rot_axis.normalize();
			numeric::xyzMatrix_float const R( rotation_matrix(rot_axis, pi) );
			for ( int k=1; k<= 3; ++k ){
				for ( int l=1; l <= 5; ++l) {
					numeric::xyzVector_float Epos_temp( &Epos(1,l,i-N/2));
					Epos(k,l,i) = ( R* Epos_temp )(k);
				}
			}
		}
	}
	// Whe using a subsystem we need to select which of the 2-fold partners are
	// explicitly simulated
	if ( docking::docking_pose_symm_subsystem ) {
		int swap = 4;
		if ( docking::dn_with_3  ) swap = 3;
		for ( int k=1; k<= 3; ++k ){
			for ( int l=1; l <= 5; ++l) {
				Epos(k,l,swap) = Epos(k,l,N/2+1);
			}
		}
		if ( docking::dn_with_5  ) {
			swap = 5;
			for ( int k=1; k<= 3; ++k ){
				for ( int l=1; l <= 5; ++l) {
					Epos(k,l,swap) = Epos(k,l,N-1);
				}
			}
		}
	}

	pose.set_coords( false, Epos, FArray3D_float() );
	pose.set_fullatom_flag( true, false ); // only glycine anyway
}


///////////////////////////////////////////////////////////////////////////////
void
 find_coordinates_virus(
 numeric::xyzVector_float & rot_axis_save,
 numeric::xyzVector_float & monomer_pos_save
 )
{
	//Calculate the rotation axis of the first face of the icosahedron
	//Together with the position of the first monomer on the face

	using numeric::constants::f::pi;

	// Generate the icosahedron by specifying the vertices

	int size = 20;
	double golden = (1 + std::sqrt(5.0))/2;

	std::vector<numeric::xyzVector_float> vertices;
	vertices.push_back(numeric::xyzVector_float(-size*golden,0.0,-size));
	vertices.push_back(numeric::xyzVector_float(0.0,-size, -size*golden));
	vertices.push_back(numeric::xyzVector_float(-size,-size*golden,0.0));

	numeric::xyzVector_float v1( vertices[ 0 ]);
	numeric::xyzVector_float v2( vertices[ 1 ]);
	numeric::xyzVector_float v3( vertices[ 2 ]);
	numeric::xyzVector_float v4( (v2-v1)/2 +v1 );
	numeric::xyzVector_float v5( v3-v4 );
	numeric::xyzVector_float v6( v2-v1 );
	v5.normalize();
	float const dist ( std::tan(30.0/360.0*2.0*pi)*v6.length()/2 );
	numeric::xyzVector_float origin( v4+v5*dist );
	numeric::xyzVector_float v7 (origin - v1 );
	numeric::xyzVector_float v8 ( v3 - v1 );
	numeric::xyzVector_float v9 ( cross(v6,v8) );
	v9.normalize();
	numeric::xyzVector_float v10 ( (v3 - origin)/2 );
	float radius ( v10.length() );
	v10.normalize();
	numeric::xyzVector_float v11 ( cross(v9,v10) );
	v11.normalize();

	rot_axis_save = v9;
	monomer_pos_save =  v10*radius + origin;
}

///////////////////////////////////////////////////////////////////////////////
void
setup_pseudo_pose_virus(
  int const N,
  pose_ns::Pose & pose
)
{
// Setup the psudo residues for the T=1 symmetry

	using numeric::constants::f::pi;

	float const length( 1.5 );
	numeric::xyzVector_float n,ca,c,o,cb;

	// Generate the icosahedron by specifying the vertices

	int size = 20;
	double golden = (1 + std::sqrt(5.0))/2;

	std::vector<numeric::xyzVector_float> vertices;
	vertices.push_back(numeric::xyzVector_float(0.0,size, size*golden));
	vertices.push_back(numeric::xyzVector_float(0.0,size, -size*golden));
	vertices.push_back(numeric::xyzVector_float(0.0,-size, size*golden));
	vertices.push_back(numeric::xyzVector_float(0.0,-size, -size*golden));
	vertices.push_back(numeric::xyzVector_float(size,size*golden,0.0));
	vertices.push_back(numeric::xyzVector_float(size,-size*golden,0.0));
	vertices.push_back(numeric::xyzVector_float(-size,size*golden,0.0));
	vertices.push_back(numeric::xyzVector_float(-size,-size*golden,0.0));
	vertices.push_back(numeric::xyzVector_float(size*golden,0.0,size));
	vertices.push_back(numeric::xyzVector_float(size*golden,0.0,-size));
	vertices.push_back(numeric::xyzVector_float(-size*golden,0.0,size));
	vertices.push_back(numeric::xyzVector_float(-size*golden,0.0,-size));

	// Generate a list of which vertices make up the 20 faces

	typedef std::vector<int> vertices_faces;
	std::vector<vertices_faces> faces;
	vertices_faces v_face;

	//Set face 1
	v_face.push_back(12);
	v_face.push_back(4);
	v_face.push_back(8);
	faces.push_back(v_face);
	v_face.clear();

	//Set face 2
	v_face.push_back(4);
  v_face.push_back(6);
  v_face.push_back(8);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 3
	v_face.push_back(6);
  v_face.push_back(3);
  v_face.push_back(8);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 4
	v_face.push_back(3);
  v_face.push_back(11);
  v_face.push_back(8);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 5
	v_face.push_back(11);
  v_face.push_back(12);
  v_face.push_back(8);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 6
	v_face.push_back(12);
  v_face.push_back(2);
  v_face.push_back(4);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 7
	v_face.push_back(2);
  v_face.push_back(10);
  v_face.push_back(4);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 8
  v_face.push_back(6);
  v_face.push_back(4);
  v_face.push_back(10);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 9
  v_face.push_back(10);
  v_face.push_back(9);
  v_face.push_back(6);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 10
  v_face.push_back(3);
  v_face.push_back(6);
  v_face.push_back(9);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 11
  v_face.push_back(9);
  v_face.push_back(1);
  v_face.push_back(3);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 12
  v_face.push_back(11);
  v_face.push_back(3);
  v_face.push_back(1);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 13
  v_face.push_back(7);
  v_face.push_back(11);
  v_face.push_back(1);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 14
  v_face.push_back(7);
  v_face.push_back(12);
  v_face.push_back(11);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 15
	v_face.push_back(7);
  v_face.push_back(2);
  v_face.push_back(12);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 16
  v_face.push_back(2);
  v_face.push_back(5);
  v_face.push_back(10);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 17
  v_face.push_back(10);
  v_face.push_back(5);
  v_face.push_back(9);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 18
  v_face.push_back(9);
  v_face.push_back(5);
  v_face.push_back(1);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 19
  v_face.push_back(1);
  v_face.push_back(5);
  v_face.push_back(7);
  faces.push_back(v_face);
  v_face.clear();

	//Set face 20
  v_face.push_back(7);
  v_face.push_back(5);
  v_face.push_back(2);
  faces.push_back(v_face);
  v_face.clear();

	// Create the actual pseudo_pose. Each face has three monomers which are
	// related by a threefold symmetry axis. In total 60 monomers

  pose.simple_fold_tree( 2*N );
  FArray3D_float Eposition(3,param::MAX_POS, 2*N );

	// only create docking::docking_pose_symm_subsystem_size pseudoresidues.
	// Select those that are witihn the rifght subset
	std::vector<bool> place_pseudo;
	for (int j = 0; j<=60; ++j) {
		if (docking::docking_pose_symm_subsystem) {
			place_pseudo.push_back(false);
		} else {
			place_pseudo.push_back(true);
		}
	}
		if (docking::docking_pose_symm_subsystem) {
			place_pseudo[1] = true;
			place_pseudo[2] = true;
			place_pseudo[3] = true;
			place_pseudo[4] = true;
			place_pseudo[13] = true;
			place_pseudo[15] = true;
		}
// Use this code to genterate the full capsid in steps
//		place_pseudo[1] = true;
//	for (int j = 56; j<61; ++j) {
//		 place_pseudo[j] = true;
//		}
	// For each face generate the three glycines
	int i =1;
	int pseudo_res=0;
	for ( std::vector<vertices_faces>::const_iterator vf=faces.begin(); vf < faces.end(); ++vf ) {
		numeric::xyzVector_float v1( vertices[ (*vf)[0]-1 ]);
		numeric::xyzVector_float v2( vertices[ (*vf)[1]-1 ]);
		numeric::xyzVector_float v3( vertices[ (*vf)[2]-1 ]);
		numeric::xyzVector_float v4( (v2-v1)/2 +v1 );
		numeric::xyzVector_float v5( v3-v4 );
		numeric::xyzVector_float v6( v2-v1 );
		v5.normalize();
		float const dist ( std::tan(30.0/360.0*2.0*pi)*v6.length()/2 );
		numeric::xyzVector_float origin( v4+v5*dist );
		numeric::xyzVector_float v7 (origin - v1 );
		numeric::xyzVector_float v8 ( v3 - v1 );
		numeric::xyzVector_float v9 ( cross(v6,v8) );
		v9.normalize();
		numeric::xyzVector_float v10 ( (v3 - origin)/2 );
		float radius ( v10.length() );
		v10.normalize();
		numeric::xyzVector_float v11 ( cross(v9,v10) );
		v11.normalize();

		ca = v10*radius;
		n = v10*( radius + length);
		c = v10*radius + v11*length;
		o = v10*(radius - length) + v11*length;
		cb = v10*radius + v9*length;

			int nbr_added_per_face=0;
			for (int j=1; j<=3; ++j) {
				if (place_pseudo[ (i-1)*3 + j ]) {
					++pseudo_res;
					++nbr_added_per_face;
					pose.set_res(pseudo_res,param_aa::aa_gly);
					pose.set_res_variant(pseudo_res,1);
					float angle ( (j-1)*2.0*pi/3 );
					numeric::xyzMatrix_float R( rotation_matrix(v9, angle ) );
					for ( int k=1; k<= 3; ++k ) {
						Eposition(k,1,pseudo_res) = ( R * n )(k) + origin(k);
						Eposition(k,2,pseudo_res) = ( R * ca )(k) + origin(k);
						Eposition(k,3,pseudo_res) = ( R * cb )(k) + origin(k);
						Eposition(k,4,pseudo_res) = ( R * c )(k) + origin(k);
						Eposition(k,5,pseudo_res) = ( R * o )(k) + origin(k);
					}
				}
			}
			//Put pseudo residues in the center as well to be able to have rotations
			//around the symmetry axis of each face
			radius = 0.0;
			ca = v10*radius;
			n = v10*( radius + length);
			c = v10*radius + v11*length;
			o = v10*(radius - length) + v11*length;
			cb = v10*radius + v9*length;
			int counter=0;
			for (int j=1; j<=3; ++j) {
				if (place_pseudo[ (i -1)*3 + j ]) {
					int const pseudo_res2( pseudo_res + N + ++counter - nbr_added_per_face);
					pose.set_res(pseudo_res2,param_aa::aa_gly);
					pose.set_res_variant(pseudo_res2,1);
					float angle ( (j-1)*2.0*pi/3 );
					numeric::xyzMatrix_float R( rotation_matrix(v9, angle ) );
					for ( int k=1; k<= 3; ++k ) {
						Eposition(k,1,pseudo_res2) = ( R * n )(k) + origin(k);
						Eposition(k,2,pseudo_res2) = ( R * ca )(k) + origin(k);
						Eposition(k,3,pseudo_res2) = ( R * cb )(k) + origin(k);
						Eposition(k,4,pseudo_res2) = ( R * c )(k) + origin(k);
						Eposition(k,5,pseudo_res2) = ( R * o )(k) + origin(k);
					}
				}
			}
			++i;
		}
	pose.set_coords( false, Eposition, FArray3D_float() );
	pose.set_fullatom_flag( true,false ); // build HA,HN

}

/////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_coordsys_setup(
  pose_ns::Pose & monomer_pose,
	pose_ns::Pose & oligomer_pose,
	int & anchor,
	numeric::xyzVector_float & z_axis,
	numeric::xyzVector_float & origin,
	int const radius
)
{
	// Setup the coordinate system for the cyclic symmetry. This creates
	// a random starting position by randomly setting the z-axis of the system
	// together with the origin. For local pertubations of trimers we find the
	// z-axis and origin from the position of the monomers. For dimers we cannot
	// find the z-axis from this protocol. We assume instead that the dimer pose
	// have the z-axis along the absolute z (0,0,1) and skip this sections

	using namespace docking;

	int const nres_monomer( monomer_pose.total_residue() );
	int const trans_dist ( radius );
	anchor = monomer_pose.residue_center_of_mass(1, nres_monomer);
	//test
	//anchor =  111;

	// Generate random coordinate system
	if ( docking_randomize1 | docking_randomize2 ) {

		FArray3D_float const & Epos( monomer_pose.Eposition() );
		numeric::xyzVector_float anchor_vec ( &Epos(1,2,anchor) ),
			v( 0.5-ran3(), 0.5-ran3(), 0.5-ran3() ),	v2(0.5-ran3(), 0.5-ran3(), 0.5-ran3() ),
			v3(0.5-ran3(), 0.5-ran3(), 0.5-ran3() ), x_axis;

			v.normalize();
			v2.normalize();
			v3.normalize();
			z_axis = cross( v, v2 );
			x_axis =  cross( z_axis, v3 );
			z_axis.normalize();
			x_axis.normalize();
			origin = anchor_vec - trans_dist*x_axis;

		}

	else {
		// This does not work when you have more than 3 monomers!!! But then you
		// should use the subsystem flag
		if ( pose_symm_n_monomers == 2 ) {
			FArray3D_float const & Epos( oligomer_pose.Eposition() );
			numeric::xyzVector_float v1(&Epos(1,2,anchor)), v2(&Epos(1,2,anchor + nres_monomer)),
				u12, z(0.0, 0.0, 1.0);
				u12 = v2-v1;
				origin = (1.0f/2) * ( v1+v2 );
				z_axis = z;
		} else {
			if ( !docking_pose_symm_subsystem &&  pose_symm_n_monomers > 3 ) {
				std::cout << "Recreation of a system with pose_symmm_n_monomers > 3 "
					<< "without running as a subsystem only works when starting from a previous rosetta model" << std::endl;
			}
			FArray3D_float const & Epos( oligomer_pose.Eposition() );
			numeric::xyzVector_float v1(&Epos(1,2,anchor)), v2(&Epos(1,2,anchor + nres_monomer)),
				v3(&Epos(1,2,anchor+nres_monomer*2)), u12, u13, z;

			u12 = v2-v1;
			u13 = v3-v1;
			z = cross( u12, u13 );
			z_axis = z.normalize();
			origin = (1.0f/3) * ( v1+v2+v3 );
		}
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_protocol(pose_ns::Pose & oligomer_pose,
	int const nres_monomer)
{
	using namespace docking;

	std::cout << " pose_docking_symm_protocol excecuted " << "\n";

	pose_ns::Pose monomer_pose, pose, pose_save;
	pose_ns::Score_weight_map weight_map;

	int const N ( pose_symm_n_monomers );
	assert ( N );
  int anchor;
	std::string type ( symm_type );
	bool const fullatom ( get_docking_fullatom_flag() );
	numeric::xyzVector_float z_axis, origin;

	// Filter rigid body orients in centroid mode based on different criteria
	bool continue_fa ( false );
	while ( !continue_fa ) {

		// Create monomer pose from oligomer_pose
		create_monomer_pose(oligomer_pose, monomer_pose, nres_monomer);

		// Setup coordinate system this coordinate system is only used for cyclic
		// symmetry now
		pose_docking_symm_coordsys_setup(monomer_pose, oligomer_pose, anchor, z_axis, origin, 200);

		// Create symmetric pose

		create_symmetric_pose( monomer_pose, fullatom, N, anchor, origin, z_axis, type, pose );

		Pdb_info pdb_info;
		pdb_info.dimension( pose.total_residue());
		pdb_info.copy_segment( oligomer_pose.total_residue(), oligomer_pose.pdb_info(), 1, 1 );
		pose.set_pdb_information( pdb_info );

		if ( truefalseoption("dump_symmetrical_pose") ) {
			std::cout << "Dumping symmetrical pose object and exiting!..." << std::endl;
			pose.dump_pdb("symmetrical_pose.pdb");
			std::exit(0);
		}

		// Save pose
		pose_save = pose;

		// Go into centroid level
		pose.set_fullatom_flag( false );
		monomer_pose.set_fullatom_flag( false );

		// perturb
			pose_docking_symm_perturb_rigid_body(pose);

		// MC parameters are model specific
		int outer_cycle, inner_cycle;
		float trans_magnitude, rot_magnitude;
		if ( pose.symmetry_info().symm_type() == "helix" ) {
			outer_cycle = 10;
			inner_cycle = 50;
			trans_magnitude = 0.8; /* angstrom */
			rot_magnitude = 0.8; /* degree */
		} else {
			outer_cycle = 10;
			inner_cycle = 50;
			trans_magnitude = 0.7; /* angstrom */
			rot_magnitude = 0.5; /* degree */
		}

		// centroid MC search
		setup_score_weight_map( weight_map, score4d );
		pose_docking_symm_centroid_rigid_body_adaptive( pose, outer_cycle,
			inner_cycle, weight_map, trans_magnitude, rot_magnitude);

		// Should we move into fa?
		set_continue_fa (pose, continue_fa);
	}
	// fullatom mode
	if ( get_docking_fullatom_flag() ) {

		pose.set_fullatom_flag( true, false );
		// copy back the starting sidechain first
    pose.recover_sidechain( pose_save );
		// Initialize packer
//		if ( docking::docking_pose_symm_full )
//			setup_symmetric_packer(pose);

		pose_setup_packer();

	//	assert  ( symm_check(pose) ); does not work for all symmetries
		bool const interface_only(true), include_current(true);
		pose_docking_symm_repack( pose, interface_only, include_current );
		pose.score ( weight_map );

		if ( docking::dock_rtmin ) { //This is not tested!!!
      set_docking_interface_pack( interface_only );
      score_set_try_rotamers( true );
      score_set_minimize_rot( true );
      setup_score_weight_map( weight_map, score10d );
      pose.score( weight_map );
      score_set_minimize_rot( false );
      score_set_try_rotamers( false );
    }

		 if ( get_docking_mcm_flag() ) {
      pose_docking_symm_mcm_protocol( pose );
		} else if ( get_docking_minimize_flag() ) {
      setup_score_weight_map( weight_map, score10d_min );
      pose_docking_symm_minimize_trial( pose, "dfpmin_atol",
        weight_map, 0.02 );
    }
	}

	// Copy symmetric pose to a new pose without pseudo residues
	create_output_pose(oligomer_pose, pose);
	//oligomer_pose.score( score10d );
	//scores::symm_mon_score = oligomer_pose.get_0D_score(pose_ns::SCORE);
	return;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
void
set_continue_fa(
	pose_ns::Pose & pose,
	bool & continue_fa
)
{

	using namespace pose_ns;
  using namespace design_sym;

	if (pose.symmetry_info().symm_type() == "virus" ) {
		Symmetry_info const & s ( pose.symmetry_info() );
		int const total_residue( pose.total_residue() );

		pose_update_cendist( pose );
		FArray2D_float const & cendist ( pose.get_2D_score( CENDIST ) );
		int num_contacts ( 0 );
		for ( int i = 1; i <= total_residue; ++i ) {
			for (int j=1; j<=total_residue; ++j) {
				if ( !s.chi_independent( i ) && s.chi_independent( j ) ) {
					if ( cendist(i,j) < 64 ) ++num_contacts;
				}
			}
		}
		if ( num_contacts > 50 ) {
			continue_fa = true;
			std::cout<<"There are "<<num_contacts<<" contacts for the scoring monomer"<<std::endl;
			} else {
				std::cout<<"We didn't pass the filter godammit!, only "<< num_contacts <<" contacts" <<std::endl;
				Pose tmp_pose;
				pose=tmp_pose;
			}

	} else {
		continue_fa = true;
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_perturb_rigid_body(
	 pose_ns::Pose & monomer_pose,
	 pose_ns::Pose & pose
 ){
	pose_ns::Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4d );

	static bool const angle_perturb = truefalseoption("angle_perturb");
	if ( angle_perturb ) {
      pose_docking_symm_perturb_rigid_body_helix(pose);
			return;
	}

	pose_ns::Pose tmp_pose;
	tmp_pose = monomer_pose;
	tmp_pose.score( weight_map );
	float const monomer_vdw = tmp_pose.get_0D_score( pose_ns::DOCK_VDW );

	pose_docking_symm_perturb_rigid_body( pose, monomer_vdw );
}

/////////////////////////////////////////////////////////////////////////////////////////////////
// rhiju  Set default behavior of this  function to slide until vdw = 0.0
// rhiju  Overloaded version above should do have the same crazy behvaior that Ingemar was used to, though.
void
pose_docking_symm_perturb_rigid_body(
  pose_ns::Pose & pose,
	float const monomer_vdw
)
{

	// Translation of monomers along a symmetry axis to glancing contact.
	// for cyclic along an axis perpendicular to the rotation axis (x)
	// for helical symmetry along the axis perpindicular to the translational
	// symmetry axis. For the virus case there is a translation to along an
	// axis perpindicular to the threefold rotation axis of a face of the
	// icosahedron followed by a translation along the rotation axis

	if (docking::docking_local_refine)  return;

	using namespace pose_ns;
  using namespace docking;

	int const njump_monomer( number_jump_monomer( pose ) );
	int const N ( pose.symmetry_info().N() );

	//	pose.set_allow_bb_move( false );
	pose.set_allow_jump_move( false );
	pose.set_allow_jump_move( njump_monomer+1, true );

	// Two slide in contacts for helical setup and dn symmetry
	if (  pose.symmetry_info().symm_type() == "helix" ) pose.set_allow_jump_move( N+1, true );
	if (  pose.symmetry_info().symm_type() == "dn" ) pose.set_allow_jump_move( 2*N-1, true );
	float step_z_change( 0 );

	// Set weight_map using centroid mode scoring
	pose_ns::Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4d );
	pose.score( weight_map );
	int const c2n(-1);

	float vdw( pose.get_0D_score( pose_ns::DOCK_VDW ) );

	if ( !docking_local_refine || (docking_small_perturbation && pose.symmetry_info().symm_type() == "virus" )) {
		// Separation distance
		double sep(0.0);
		double sep2(0.0);
		bool dn_slide_x(true);
//		  std::cout << "monomer_vdw= " << monomer_vdw << " vdw= " << vdw <<
//								   " sep= " << sep << std::endl;

		// first try moving away from each other
		while ( vdw > monomer_vdw + 0.1 ) {
			dn_slide_x = true;
			sep += 0.5;
			if ( pose.symmetry_info().symm_type() == "dn" ) {
				if ( ran3() < 0.5 ) {
					sep -= 0.5;
					sep2 += 0.5;
					pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
					dn_slide_x = false;
				}
			}
			pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
			pose.score( weight_map );
			vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
			//			std::cout << "pos vdw= " << vdw << ' ' << "sep= " << sep << std::endl;
			//			 pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
		}
//	then try moving towards each other
		while ( vdw < monomer_vdw + 0.1 ) {
			dn_slide_x = true;
			sep -= 0.5;
			if ( pose.symmetry_info().symm_type() == "dn" ) {
				if ( ran3() < 0.5 ) {
					sep += 0.5;
					sep2 -= 0.5;
					pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
					dn_slide_x = false;
				}
			}
			pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
			pose.score( weight_map );
			vdw = pose.get_0D_score( pose_ns::DOCK_VDW );

			// Print output for debugging
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
				//				static int counter(0);
				//				std::cout << "neg vdw= " << vdw << ' ' << "sep= " << sep << ' ' << counter
				//			      << std::endl;
				//							pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}
			// If monomers do not touch slide them in along the z-axis until they meet by sliding along x.
			if ( pose.symmetry_info().symm_type() == "helix" && sep < -120 ) {
				pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, -sep );
				sep -= sep;
				pose.set_jump_rb_delta( N+1, 3, c2n, ++step_z_change );
				std::cout << "step_z_change= " << step_z_change << ' ' << "sep= " << sep << std::endl;
			}
				// In situations where the centroids dont clash but the rest of the backbone does we need to test for x < 0
				if (pose.symmetry_info().symm_type() == "cn" ) {
					Fold_tree f = pose.fold_tree();
					FArray2D_int const & jump_point( f.get_jump_point() );
					int const anchor = jump_point(1,njump_monomer+1);
					double const x_coord =  pose.Eposition()(1,2,anchor);
					if ( x_coord < 0) {
							sep += (int)(-x_coord+1);
							pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
							break;
					}
				}

		}
			if ( pose.symmetry_info().symm_type() == "dn" ) {
				float vdw_save (vdw);
				if ( dn_slide_x ) {
					float const sep2_start(sep2);
					while ( vdw < vdw_save + 0.1 ) {
						sep2 -= 0.5;
						pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
						pose.score( weight_map );
						vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
						//std::cout << " slide in 2: " << sep2 << std::endl;
						if ( sep2 - sep2_start < -20 ) {
							pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2_start );
							break;
						}
					}
				} else {
						float const sep_start(sep);
						while ( vdw < vdw_save + 0.1 ) {
						sep -= 0.5;
						pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
						pose.score( weight_map );
						vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
						//std::cout << " slide in: " << sep << std::endl;
						if ( sep - sep_start < -20 ) {
              pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep_start );
							break;
            }

					}
				}
				sep2 += 0.5;
				pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
		}

		  sep += 0.5;
			pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );

		// Translate the virus along the rotation axis until monomer of different
		// faces meet
		if (pose.symmetry_info().symm_type() == "virus" ) {

			pose.score( weight_map );
			vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
			float const start_vdw (vdw);
			sep = 0.0;

		// first try moving away from each other
			while ( vdw > start_vdw + 0.1 ) {
				sep += 0.5;
				pose.set_jump_rb_delta( njump_monomer+1, 3, c2n, sep );
				pose.score( weight_map );
				vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
			}
		//	then try moving towards each other
			while ( vdw < start_vdw + 0.1 ) {
				sep -= 0.5;
				pose.set_jump_rb_delta( njump_monomer+1,3 , c2n, sep );
				pose.score( weight_map );
				vdw = pose.get_0D_score( pose_ns::DOCK_VDW );

				// Print output for debugging
				static int counter(0);
				std::cout << "vdw= " << vdw << ' ' << "sep= " << sep << ' ' << counter
			      << std::endl;
				if (sep < -150 )
				{
						 std::cout<<"This is weird! sep < -150"<<std::endl;
						return;
				}
				//				pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}
			sep += 0.5;
			pose.set_jump_rb_delta( njump_monomer+1, 3, c2n, sep );
		}
	}

	if ( docking_local_refine) {
	    symm_move( pose, normal_perturbation, rotational_perturbation );
	}
}

void
pose_docking_symm_perturb_rigid_body(
  pose_ns::Pose & pose
)
{

	// Translation of monomers along a symmetry axis to glancing contact.
	// for cyclic along an axis perpendicular to the rotation axis (x)
	// for helical symmetry along the axis perpindicular to the translational
	// symmetry axis. For the virus case there is a translation to along an
	// axis perpindicular to the threefold rotation axis of a face of the
	// icosahedron followed by a translation along the rotation axis

	using namespace pose_ns;
  using namespace docking;

	if ( docking_local_refine) {
	    symm_move( pose, normal_perturbation, rotational_perturbation );
			return;
	}


	if ( pose.symmetry_info().symm_type() == "cn" ) slide_into_contact_cn( pose );
	else if ( pose.symmetry_info().symm_type() == "helix" ) slide_into_contact_helix( pose );
	else if ( pose.symmetry_info().symm_type() == "dn" ) slide_into_contact_dn( pose );
	else if ( pose.symmetry_info().symm_type() == "virus" ) slide_into_contact_virus( pose );
	else {
		std::cout << " No valid symmetry defined!!! " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

void
slide_into_contact_cn(
		pose_ns::Pose & pose
)
{
	using namespace pose_ns;
  using namespace docking;

	int const njump_monomer( number_jump_monomer( pose ) );
	//	int const N ( pose.symmetry_info().N() );

	pose.set_allow_jump_move( false );
	pose.set_allow_jump_move( njump_monomer+1, true );

	// Set weight_map using centroid mode scoring
  pose_ns::Score_weight_map weight_map;
  setup_score_weight_map( weight_map, score4d );
  pose.score( weight_map );
  int const c2n(-1);

	float vdw( pose.get_0D_score( pose_ns::DOCK_VDW ) );

  if ( !docking_local_refine ){
		double sep(0.0);

		// first try moving away from each other
    while ( pose.get_0D_score( pose_ns::DOCK_VDW ) > 0.1 ) {
      sep += 0.5;
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
      pose.score( weight_map );
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
            std::cout << "pos vdw= " << vdw << ' ' << "sep= " << sep << std::endl;
      //       pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}
    }

		//	then try moving towards each other
		while (  pose.get_0D_score( pose_ns::DOCK_VDW ) < 0.1 ) {
			sep -= 0.5;
			pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
			pose.score( weight_map );

			// Print output for debugging
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
					static int counter(0);
					std::cout << "neg vdw= " << vdw << ' ' << "sep= " << sep << ' ' << counter
							      << std::endl;
				//							pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}

			// In situations where the centroids dont clash but the rest of the backbone does we need to test for x < 0
				Fold_tree f = pose.fold_tree();
				FArray2D_int const & jump_point( f.get_jump_point() );
				int const anchor = jump_point(1,njump_monomer+1);
				double const x_coord =  pose.Eposition()(1,2,anchor);
				if ( x_coord < 0) {
						sep += (int)(-x_coord+1);
						pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
						break;
				}
		}
		sep += 0.5;
		pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );

	}
}

void
slide_into_contact_helix(
    pose_ns::Pose & pose
)
{

	using namespace pose_ns;
  using namespace docking;

	static bool const angle_perturb = truefalseoption("angle_perturb");
  if ( angle_perturb ) {
      pose_docking_symm_perturb_rigid_body_helix( pose );
      return;
  }

	int const njump_monomer( number_jump_monomer( pose ) );
	int const N ( pose.symmetry_info().N() );

	pose.set_allow_jump_move( false );
	pose.set_allow_jump_move( njump_monomer+1, true );
	// Two slide in contacts for helical setup
	pose.set_allow_jump_move( N+1, true );

	// Set weight_map using centroid mode scoring
  pose_ns::Score_weight_map weight_map;
  setup_score_weight_map( weight_map, score4d );
  pose.score( weight_map );
  int const c2n(-1);
	float step_z_change( 0 );

	float vdw( pose.get_0D_score( pose_ns::DOCK_VDW ) );

  if ( !docking_local_refine ){
		double sep(0.0);

		// first try moving away from each other
    while ( pose.get_0D_score( pose_ns::DOCK_VDW ) > 0.1 ) {
      sep += 0.5;
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
      pose.score( weight_map );
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
            std::cout << "pos vdw= " << vdw << ' ' << "sep= " << sep << std::endl;
      //       pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}
    }

		//	then try moving towards each other
		while (  pose.get_0D_score( pose_ns::DOCK_VDW ) < 0.1 ) {
			pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
			pose.score( weight_map );

			// Print output for debugging
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
					static int counter(0);
					std::cout << "neg vdw= " << vdw << ' ' << "sep= " << sep << ' ' << counter
							      << std::endl;
				//							pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}
			// If monomers do not touch slide them in along the z-axis until they meet by sliding along x.
			if ( pose.symmetry_info().symm_type() == "helix" && sep < -120 ) {
				pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, -sep );
				sep -= sep;
				pose.set_jump_rb_delta( N+1, 3, c2n, ++step_z_change );
				std::cout << "step_z_change= " << step_z_change << ' ' << "sep= " << sep << std::endl;
			}
		}
		sep += 0.5;
		pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
	}

}

void
slide_into_contact_dn(
    pose_ns::Pose & pose
)
{

	using namespace pose_ns;
  using namespace docking;

	int const njump_monomer( number_jump_monomer( pose ) );
  int const N ( pose.symmetry_info().N() );
	int const nres_monomer ( (pose.total_residue() - pose.symmetry_info().npseudo() )/N );


  pose.set_allow_jump_move( false );
  pose.set_allow_jump_move( njump_monomer+1, true );
	//Two slide in contacts for dn symmetry
	pose.set_allow_jump_move( 2*N-1, true );

  // Set weight_map using centroid mode scoring
  pose_ns::Score_weight_map weight_map;
  setup_score_weight_map( weight_map, score4d );
  pose.score( weight_map );

	// We are going to use score4 for subsystems because
	// subsystem scoring is not implemented in centroid
	pose_ns::Score_name score_component;
	if ( docking::docking_pose_symm_subsystem ) {
		pose_ns::Score_weight_map weight_map_subsys;
		setup_score_weight_map( weight_map_subsys, score4 );
		weight_map = weight_map_subsys;
		score_component = pose_ns::VDW;
		pose.score( weight_map );
		} else {
			score_component = pose_ns::DOCK_VDW;
		}

 int const c2n(-1);

  float vdw( pose.get_0D_score( score_component ) );

	if ( !docking_local_refine ){

		double sep(0.0);
    double sep2(0.0);
    bool dn_slide_x(true);

		float threshold;
		if ( docking::docking_pose_symm_subsystem ) {
			Pose monomer_pose;

			//			create_monomer_pose(pose, monomer_pose, nres_monomer);
			monomer_pose.simple_fold_tree( nres_monomer );
			monomer_pose.copy_segment( nres_monomer, pose, 1, 1 );

			monomer_pose.score( score4 );
			threshold = N*monomer_pose.get_0D_score( score_component );
			} else {
				threshold = 0;
			}

		// Introduce a random twist, rotate one ring relative to the other and make sure that the resulting rotation angle is within a specified range.
    static float const dn_twist_angle = realafteroption("dn_twist_angle",-1);
    if (dn_twist_angle > 0 ) {
			Fold_tree f( pose.fold_tree() );
			float const angle (dn_twist_angle * gaussian());
      numeric::xyzMatrix_double R( Z_rot(angle) ); //Rotate only around the Z-axis
      numeric::xyzVector_double origin( 0.0, 0.0, 0.0);
      int const pseudo_pos = f.upstream_jump_residue( 2*N-1 );
      Jump jump2 ( pose.get_jump(2*N-1) );
      jump2.rotation_by_matrix( pose.Eposition()( 1,1, pseudo_pos),
                                origin, R );
			pose.set_jump( 2*N-1, jump2 );
			int const anchor_pos = f.downstream_jump_residue( njump_monomer + 1 );
			numeric::xyzVector_float anchor( &(pose.Eposition()(1,2,anchor_pos ) ) );
			numeric::xyzVector_float anchor2( &(pose.Eposition()(1,2,anchor_pos+2*nres_monomer) ) );
			anchor2(3)=0.0;
      float rot_angle =180.0/numeric::constants::f::pi*acos( dot_product(anchor,anchor2)/(anchor.length()*anchor2.length()) );
			if ( anchor2(2) < 0 ) rot_angle = -rot_angle;
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
				std::cout << "angle rot_angle " << angle << " " <<rot_angle << std::endl;
			}
			float const interval = 360.0/pose_symm_n_monomers;
			if ( rot_angle > interval || rot_angle < 0 ){
				float reset_angle = -angle; // Default reset the angle
				// If dn_twist_angle is small we are using it as a reset swith ensuring
				// that we have not moved out of the acceptable interval and move it to
				// the edges
				if ( dn_twist_angle < 0.1 && rot_angle > interval ) reset_angle = interval -rot_angle;
				if ( dn_twist_angle < 0.1 && rot_angle < 0 ) reset_angle = -rot_angle;
				numeric::xyzMatrix_double RR( Z_rot(reset_angle) );
				jump2.rotation_by_matrix( pose.Eposition()( 1,1, pseudo_pos),
                                  origin, RR );
				pose.set_jump( 2*N-1, jump2 );
      }
			pose.score( weight_map );
		}
    // first try moving away from each other
    while ( pose.get_0D_score( score_component ) > threshold + 0.1 ) {
      dn_slide_x = true;
      sep += 0.5;
      if ( ran3() < 0.5 ) {
				sep -= 0.5;
        sep2 += 0.5;
        pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
        dn_slide_x = false;
      }
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
				std::cout << "sliding away sep " << sep << std::endl;
			}
      pose.score( weight_map );
    }

//		if ( docking::docking_pose_symm_subsystem ) {
//			pose.score( weight_map );
//			threshold = pose.get_0D_score( score_component );
//    } else {
//      threshold = 0;
//    }

		//  then try moving towards each other
    while ( pose.get_0D_score( score_component ) < threshold + 0.1 ) {
      dn_slide_x = true;
      sep -= 0.5;
      if ( ran3() < 0.5 ) {
				sep += 0.5;
        sep2 -= 0.5;
        pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
        dn_slide_x = false;
      }
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
      pose.score( weight_map );
			// Print output for debugging
			if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
								static int counter(0);
								vdw = pose.get_0D_score( score_component );
								std::cout << "neg vdw= " << vdw << " threshold " << threshold << " " << "sep= " << sep << ' ' << "sep2= " << sep2 << ' ' << counter
							      << std::endl;
				//							pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}

		}

		pose.score( weight_map );
		vdw = pose.get_0D_score( score_component );

		float vdw_save( vdw );
		if ( dn_slide_x ) {
			float const sep2_start(sep2);
			while ( vdw < vdw_save + 0.1 ) {
				sep2 -= 0.5;
				pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
				pose.score( weight_map );
				vdw = pose.get_0D_score( score_component );
				if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
					//					static int counter2(0);
					std::cout << " slide in 2: " << sep2 << " vdw_start " << vdw_save << " " << " vdw " << vdw <<std::endl;
	//				pose.dump_scored_pdb("slide_test2."+string_of(++counter2)+".pdb", weight_map);
				}
				if ( sep2 - sep2_start < -200 ) {
			//		std::cout << " resetting the score " << sep2_start << std::endl;
					pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2_start );
					break;
				}
			}
		} else {
			float const sep_start(sep);
			while ( vdw < vdw_save + 0.1 ) {
				sep -= 0.5;
				pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
				pose.score( weight_map );
				vdw = pose.get_0D_score( score_component );
				if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
					std::cout << " slide in: " << vdw << " sep : " << sep << std::endl;
				}
				if ( sep - sep_start < -200 ) {
					pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep_start );
					break;
        }
			}

		}
		sep2 += 0.5;
		pose.set_jump_rb_delta( 2*N-1, 3, c2n, sep2 );
		sep += 0.5;
		pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
	}
}

void
slide_into_contact_virus(
    pose_ns::Pose & pose
)
{

	using namespace pose_ns;
  using namespace docking;

	int const njump_monomer( number_jump_monomer( pose ) );
	//  int const N ( pose.symmetry_info().N() );

  pose.set_allow_jump_move( false );
  pose.set_allow_jump_move( njump_monomer+1, true );
	pose.set_allow_jump_move( njump_monomer+3, true );
  // Set weight_map using centroid mode scoring
  pose_ns::Score_weight_map weight_map;
  setup_score_weight_map( weight_map, score4d );
  pose.score( weight_map );
  int const c2n(-1);

  float vdw( pose.get_0D_score( pose_ns::DOCK_VDW ) );

  if ( !docking_local_refine || docking_small_perturbation ){
    double sep(0.0);

    // first try moving away from each other
    while ( pose.get_0D_score( pose_ns::DOCK_VDW ) > 0.1 ) {
      sep += 0.5;
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
      pose.score( weight_map );
      if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
            std::cout << "pos vdw= " << vdw << ' ' << "sep= " << sep << std::endl;
      //       pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
      }
    }

    //  then try moving towards each other
    while (  pose.get_0D_score( pose_ns::DOCK_VDW ) < 0.1 ) {
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
      pose.score( weight_map );

      // Print output for debugging
      if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
          static int counter(0);
          std::cout << "neg vdw= " << vdw << ' ' << "sep= " << sep << ' ' << counter
                    << std::endl;
        //              pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
      }

		}
		// Translate the virus along the rotation axis until monomer of different
		// faces meet
			pose.score( weight_map );
			vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
			float const start_vdw (vdw);
			float sep2 = 0.0;

		// first try moving away from each other
			while ( vdw > start_vdw + 0.1 ) {
				sep2 += 0.5;
				pose.set_jump_rb_delta( njump_monomer+1, 3, c2n, sep2 );
				pose.score( weight_map );
				vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
			}
		//	then try moving towards each other
			while ( vdw < start_vdw + 0.1 ) {
				sep2 -= 0.5;
				pose.set_jump_rb_delta( njump_monomer+1,3 , c2n, sep2 );
				pose.score( weight_map );
				vdw = pose.get_0D_score( pose_ns::DOCK_VDW );

				// Print output for debugging
				static int counter(0);
				std::cout << "vdw= " << vdw << ' ' << "sep2= " << sep2 << ' ' << counter
			      << std::endl;
				if (sep2 < -150 )
				{
						 std::cout<<"This is weird! sep < -150"<<std::endl;
						return;
				}
				//				pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}

		sep2 += 0.5;
    pose.set_jump_rb_delta( njump_monomer+1, 3, c2n, sep2 );
	  sep += 0.5;
		pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
	}
}


/////////////////////////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_perturb_rigid_body_helix(
	pose_ns::Pose & pose
)
{


	using namespace pose_ns;
  using namespace docking;
	using numeric::constants::f::pi;

	int const njump_monomer( number_jump_monomer( pose ) );
	int const N (pose.symmetry_info().N() );
	Jump jump( pose.get_jump( njump_monomer+1 ) );
	Jump jump2( pose.get_jump(  N+1 ) );


	pose.set_allow_bb_move( false );
  pose.set_allow_jump_move( false );
  pose.set_allow_jump_move( njump_monomer+1, true );
	pose.set_allow_jump_move( N+1, true );

	// Set weight_map using centroid mode scoring
	pose_ns::Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4d );
	pose.score( weight_map );
	int const c2n(-1);

		// Separation angle
		double sep(0.0);

 //  static int counter(0);
    // first try moving away from each other
    while (  pose.get_0D_score( pose_ns::DOCK_VDW ) > 0.1 ) {
      sep += 0.5;
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
      pose.score( weight_map );
    }
//  then try moving towards each other
    while ( pose.get_0D_score( pose_ns::DOCK_VDW ) < 0.1 ) {
      sep -= 0.5;
      pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
      pose.score( weight_map );
		}
		bool move_on (false);
		int counter(0);
		while (!move_on) {
			counter++;
			double const random_jump ( -ran3()*30.0f + sep );
			std::cout<<"random_jump "<<random_jump-sep<<std::endl;
			numeric::xyzMatrix_double const R( Z_rot_rad( 10.0/180.0*pi ) );
			pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, random_jump );
			jump2.set_rotation( R * jump2.get_rotation() );
			 pose.set_jump( N+1, jump2 );
			pose.score( weight_map );
			float vdw = pose.get_0D_score( pose_ns::DOCK_VDW );

			double angle_sep ( 10.0 );
			//	else try moving towards each other
			numeric::xyzMatrix_double const M( Z_rot_rad( -0.1/180.0*pi ) );
			while (  pose.get_0D_score( pose_ns::DOCK_VDW ) < 0.3 ) {
				angle_sep -= 0.1;
				jump2.set_rotation( M * jump2.get_rotation() );
				pose.set_jump( N+1, jump2 );
				pose.score( weight_map );
				vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
				// Print output for debugging
				if ( runlevel_ns::runlevel >= runlevel_ns::inform ) {
					static int counter(0);
					std::cout << "neg vdw= " << vdw << ' ' << "angle_sep= " << angle_sep << ' ' << counter
				    << std::endl;
				}
			 //	pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
			}
			// Calculate angle between monomers
			//first anchor point -- assume its the first jump.
  		Fold_tree f( pose.fold_tree() );
  		int const anchor_pos = f.downstream_jump_residue( njump_monomer + 1 );
			int const nres_monomer( pose.symmetry_info().nres_monomer_bb() );
			numeric::xyzVector_float anchor( &(pose.Eposition()(1,2,anchor_pos) ) );
			numeric::xyzVector_float anchor2( &(pose.Eposition()(1,2,anchor_pos+nres_monomer) ) );
			float angle =180.0/pi*acos( dot_product(anchor,anchor2)/(anchor.length()*anchor2.length()) );
			float a_low (13.85); //Hardcoded!!! Change
			float a_high (18.00); //Hardcoded!!! Change
			std::cout<<"The angle between monomers is: "<<angle<<std::endl;

			if ( fabs(angle) > a_low &&  fabs(angle) < a_high ) {
				move_on = true;
				numeric::xyzMatrix_double const M2( Z_rot_rad( 0.1/180.0*pi ) );
				jump2.set_rotation( M2 * jump2.get_rotation() );
				pose.set_jump( N+1, jump2 );
			} else {
				if ( counter > 100) {
					pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, random_jump );
					move_on = true;
				}
				numeric::xyzMatrix_double const M_reset( Z_rot_rad( -angle_sep/180.0*pi ) );
				jump2.set_rotation( M_reset * jump2.get_rotation() );
				pose.set_jump( N+1, jump2 );
			}
		}
}

//////////////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_centroid_rigid_body_adaptive(
  pose_ns::Pose & pose,
  int const outer_cycles,
  int const inner_cycles,
  const pose_ns::Score_weight_map & wt_map,
  float trans_mag,
  float rot_mag
)
{
	if (docking::docking_local_refine) return;

  for ( int i = 1; i <= outer_cycles; ++i ) {

    float accept_rate = pose_docking_symm_rigid_body_trial( pose, inner_cycles,
      wt_map, false/*fullatom*/, trans_mag, rot_mag );

		if ( accept_rate > 0.5 ) {
      trans_mag *= 1.1;
      rot_mag *= 1.1;
    } else {
      trans_mag *= 0.9;
      rot_mag *= 0.9;
    }
  }
}

////////////////////////////////////////////////////////////////////////////////
float
pose_docking_symm_rigid_body_trial(
  pose_ns::Pose & pose,
  int const cycles,
  const pose_ns::Score_weight_map & wt_map,
  bool  const repack,
  float const trans_mag,
  float const rot_mag
)
{
  using namespace pose_ns;

  float const temperature( 0.8 );
  Monte_carlo mc( pose, wt_map, temperature );
  int n_accepted = 0;

  for ( int i = 1; i <= cycles; ++i ) {
		symm_move( pose, trans_mag, rot_mag );
		if ( repack )	pose_docking_symm_repack( pose, true/*interface*/, true/*include_current*/ );
    if ( mc.boltzmann( pose ) ) ++n_accepted;
		pose.score( wt_map );
  }

  pose = mc.low_pose();

  std::cout << "pose_docking_rigid_body_trial -- "
            << (repack ? "fullatom ":"centroid ")
            << n_accepted << " out of " << cycles << " accepted\n";

	std::cout << "Acceptance rate: " << float(n_accepted)/float(cycles) << std::endl;
  return float(n_accepted)/float(cycles);

}

//////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_repack(
  pose_ns::Pose & pose,
  const bool pack_interface,
  const bool include_current
)
{

	if ( docking::norepack1 || docking::norepack2 ) {
		pose.set_allow_chi_move( false );
		return;
	}

  const int nres( pose.total_residue() );
  FArray1D_bool allow_repack( nres, true );

  pose.set_allow_bb_move( false );
  pose.set_allow_chi_move( true );
  pose.set_allow_jump_move( false );

  if ( pack_interface )
    pose_docking_symm_calc_interface( pose, allow_repack );

  pose.repack( allow_repack, include_current );

  return;
}

///////////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_calc_interface(
  pose_ns::Pose & pose,
  FArray1DB_bool & int_res
)
{
  using namespace pose_ns;
	using namespace design_sym;

	Symmetry_info const & s ( pose.symmetry_info() );
	int const total_residue( pose.total_residue() );

	pose_update_cendist( pose );
	FArray2D_float const & cendist ( pose.get_2D_score( CENDIST ) );

	// Set interface residues for one protamer
	for ( int i=1; i<=total_residue; ++i ) {
		int_res( i ) = false;
		for (int j=1; j<=total_residue; ++j) {
			if ( s.chi_independent( i ) && ! s.chi_independent( j ) ){
				if ( cendist(i,j) < 64 ) int_res( i ) = true;
			}
		}
	}

  return;

}

///////////////////////////////////////////////////////////////////////////////////
void
create_monomer_pose(pose_ns::Pose & oligomer_pose,
	pose_ns::Pose & monomer_pose,
	int const nres_monomer
)
{
	// copy some coordinates, torsions
	monomer_pose.simple_fold_tree( nres_monomer );
	// Is this necessary?
	monomer_pose.set_fullatom_flag( true, false );
	monomer_pose.copy_segment( nres_monomer, oligomer_pose, 1, 1 );
	return;
}

///////////////////////////////////////////////////////////////////////////////////
void
create_output_pose(pose_ns::Pose & oligomer_pose,
	pose_ns::Pose & pose
)
{
	pose_ns::Pose pose_out;
	pose_out = oligomer_pose;
	// Copy the same number of residues as in oligomer_pose. Not elegant!!!!
  int const nres_copy ( oligomer_pose.total_residue() );
	// Create simple fold tree
	pose_out.simple_fold_tree( nres_copy );
	pose_out.set_fullatom_flag( pose.fullatom(), false  /* we are going to copy sc info, dont repack */);

	// Copy residues from new pose to a temporary pose. Can we copt directly to oligomer_pose???
	pose_out.copy_segment( nres_copy, pose, 1, 1 );
	// In pose_out we do not have enough information to refold correctly. Create a new fold tree
	// by connecting residues by long unphysical bonds. This is only recommended if we do not wish
	// to do anything more with the strucure before decoy output.
	pose_out.update_backbone_bonds_and_torsions();
	 // Debug Is this correct?
	// pose_out.set_fullatom_flag( true, false );
	oligomer_pose = pose_out;
	//Copy symmetry info
//	int const nres_monomer ( pose.symmetry_info().nres_monomer() );
//	int const N (docking::docking_pose_symm_subsystem ? docking::docking_pose_symm_subsystem_size : pose.symmetry_info().N() );
//	std::string const symm_type ( pose.symmetry_info().symm_type() );
//	oligomer_pose.setup_symm_info( nres_monomer, 0, N, 1, symm_type );
  return;
}

/////////////////////////////////////////////////////////
void
pose_docking_symm_mcm_protocol(
	pose_ns::Pose & docking_pose
	)
{
	std::cout << std::endl;
  std::cout << "***** pose_docking_symm_mcm_protocol *****" << std:: endl;

  using namespace docking;

  //MC move
	float trans_magnitude = 0.2; /* angstrom */
	float rot_magnitude = 0.5; /* degree */

	if ( docking_pose.symmetry_info().symm_type() == "helix" ) {
		trans_magnitude = 0.8; /* angstrom */
		rot_magnitude = 0.8; /* degree */
	}


  // rb minimization
  const float loose_func_tol = 1.0;  /* absolute tolerance */
  const float tight_func_tol = 0.02; /* absolute tolerance */
  const std::string min_type = "dfpmin_atol";
  // monte carlo
  const float minimization_threshold = 15.0; /* score unit */
  // scorefxn to minimize
  pose_ns::Score_weight_map weight_map;
  setup_score_weight_map( weight_map, score10d_min );

  docking_store_cenmode_rms();

	mc_global_track::mc_score::score = docking_pose.score( score10d );
  if ( docking_mcm_filter(delta_before_mcm, score_before_mcm) ) {

    pose_docking_symm_minimize_trial( docking_pose, min_type, weight_map,
                                 loose_func_tol);

		mc_global_track::mc_score::score = docking_pose.score( score10d );
    if ( docking_mcm_filter(delta_after_one_min, score_after_one_min) ) {

      const int cycles = 4; /* 4 cycles mcm first */
      pose_docking_symm_monte_carlo_minimize( docking_pose, cycles, min_type,
        weight_map, trans_magnitude,rot_magnitude, minimization_threshold,
        loose_func_tol );

			mc_global_track::mc_score::score = docking_pose.score( score10d );
      if ( ! runlevel_ns::benchmark &&
				docking_mcm_filter(delta_after_five_mcm, score_after_five_mcm) ) {

        const int cycles = 45; /* 45 cycles of extensive mcm */
        pose_docking_symm_monte_carlo_minimize( docking_pose, cycles, min_type,
          weight_map, trans_magnitude, rot_magnitude, minimization_threshold,
          loose_func_tol );
        // minimize to a strict tolerance
        pose_docking_symm_minimize_trial( docking_pose, min_type, weight_map,
          tight_func_tol );

      } /* after_five_mcm */

    } /* after_one_min */

  } /* before_min */
}
////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_minimize_trial(
	pose_ns::Pose & docking_pose,
  const std::string & min_type,
  const pose_ns::Score_weight_map & wt_map,
  const float func_tol
)
{
  std::cout << std::endl;
  std::cout << "***** pose_docking_symm_minimize_trial *****" << std:: endl;

  using namespace pose_ns;

  const float temperature( 0.8 );
  Monte_carlo mc( docking_pose, wt_map, temperature );
  //pose_docking_set_rb_center ( docking_pose );
  pose_docking_symm_minimize_rb_position( docking_pose, min_type, wt_map, func_tol );

  mc.boltzmann( docking_pose );
  mc.show_scores();

  docking_pose = mc.low_pose();

  std::cout << "***** pose_docking_symm_minimize_trial *****" << std:: endl;
  std::cout << std::endl;
}

////////////////////////////////////////////////////////////////////////////
void
pose_docking_symm_monte_carlo_minimize(
  pose_ns::Pose & pose,
  const int cycles,
  const std::string & min_type,
  const pose_ns::Score_weight_map & wt_map,
  const float trans_magnitude,
  const float rot_magnitude,
  const float minimization_threshold,
  const float func_tol
)
{
  std::cout << std::endl;
  std::cout << "***** pose_docking_monte_carlo_minimize *****" << std::endl;

  using namespace pose_ns;

  // create a monte_carlo object
  const float temperature ( 0.8 );
  Monte_carlo mc ( pose, wt_map, temperature );

  // start MCM
  int n_accepted = 0; // diagnostic counter

  const bool interface_only = true;
  const bool include_current_sidechain = true;
	// for rotamer_trials
  const float energycut = get_energycut();
	const int nres = pose.total_residue();
  const int rottrial_cycles = 1;

  for( int i=1; i<=cycles; ++i ) {

   // pose_docking_set_rb_center( docking_pose );
		symm_move( pose, trans_magnitude, rot_magnitude );
    bool save_docking_interface_pack = get_docking_interface_pack();

    if ( i % 8 == 0) {
      // full repacking
      pose_docking_symm_repack( pose, interface_only,
                           include_current_sidechain );
      // rotamer_trials_minimization ?
      if ( docking::dock_rtmin ) {
        set_docking_interface_pack( interface_only );
				set_allow_chi_move_symm( pose );
        score_set_try_rotamers( true );
        score_set_minimize_rot( true );
        pose.score( wt_map );
        score_set_minimize_rot( false );
        score_set_try_rotamers( false );
      } else { // or just scoring with rotamer_trials off
        score_enable_rotamer_trials( false );
        pose.score( wt_map );
        score_enable_rotamer_trials( true );
      }
} else {
      //rotamer_trials for the whole protein
			set_docking_interface_pack( ! interface_only );
			set_allow_chi_move_symm( pose );
			score_set_try_rotamers( false );
			pose.score( wt_map );
	    score_set_try_rotamers( true );
			set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres,
        mc.best_pose().get_1D_score( pose_ns::RESENERGY ), rottrial_cycles);
      pose.score( wt_map );
      score_set_try_rotamers( false );
			pose.new_score_pose();
    }

    set_docking_interface_pack( save_docking_interface_pack );

    const float current_score ( pose.get_0D_score( SCORE ) );
    const float best_score ( mc.best_pose().get_0D_score( SCORE ) );

    if ( current_score - best_score < minimization_threshold ) {

      pose_docking_symm_minimize_rb_position( pose, min_type , wt_map,
                                         func_tol );
    }

    const bool accepted = mc.boltzmann( pose );

    if( accepted ) ++n_accepted;

    std::cout << "Cycle: " << i << " -- ";
    mc.show_scores();

  }

  std::cout << "pose_docking_mcm --" << n_accepted << " out of " << cycles
            << " monte carlo cycles accepted" << std::endl;

  pose = mc.low_pose();

	std::cout<<pose.show_scores()<<std::endl;
  std::cout << "***** pose_docking_monte_carlo_minimize *****" << std:: endl;
  std::cout << std::endl;
}

/////////////////////////////////////////////////////////////////////////////

void
pose_docking_symm_minimize_rb_position(
  pose_ns::Pose & pose,
  const std::string & type,
  const pose_ns::Score_weight_map & wt_map,
  const float func_tol
)
{
  using namespace pose_ns;

  if ( ! pose.fullatom() )
    error_stop("must be in fullatom mode for docking minimization");

//  minimize_set_func( 2 );
  minimize_set_tolerance( func_tol );
  //turn on only rigid-body/backbone moves

	if ( docking::minimize_backbone ) {
		pose.set_allow_bb_move( true );
		} else {
			 pose.set_allow_bb_move( false );
		}
  pose.set_allow_chi_move( false );
	pose.set_allow_jump_move( false );

	// Set allowed jumps
	if ( pose.symmetry_info().symm_type() == "helix" ) {
		 set_allow_dg_free_helix(pose);
		 int const N (pose.symmetry_info().N() );
		pose.set_allow_rb_move(6 ,N+1, false );
		 set_use_nblist( false );
	} else if (pose.symmetry_info().symm_type() == "virus" ) {
		set_allow_dg_free_virus(pose);
		// Don't minimize the rotation of threefold axis.
		// minimimizing this df does not work well need to find out why...
		int const N ( pose.symmetry_info().N() );
		pose.set_allow_jump_move( N + 1, false );
		set_use_nblist( false );

	} else {
		set_allow_dg_free_cn(pose);
		set_use_nblist( true );
	}
  //minimize and score
  pose.main_minimize( wt_map, type );
  pose.score( wt_map );

  return;
}

////////////////////////////////////////////////////////////////////////////////

void
set_allow_dg_free_cn(
 pose_ns::Pose & pose
 )
 {
    using namespace pose_ns;

		int const basejump ( number_jump_monomer( pose ) );

		pose.set_allow_jump_move( false );

    pose.set_allow_jump_move( basejump + 1, true );

    for (int i=1; i<= 6; ++i) {
      pose.set_allow_rb_move(i ,basejump + 1, true );
    }
		pose.set_allow_rb_move(2 ,basejump + 1, false );
    pose.set_allow_rb_move(3 ,basejump + 1, false );

}

/////////////////////////////////////////////////////////////////////////////////////
void
set_allow_dg_free_helix(
 pose_ns::Pose & pose
 )
 {
		using namespace pose_ns;

		int const N (pose.symmetry_info().N() );

		pose.set_allow_jump_move( N+1, true );
		pose.set_allow_jump_move( 1, true );

		for (int i=1; i<= 6; ++i) {
			pose.set_allow_rb_move(i ,1, false );
			pose.set_allow_rb_move(i ,N+1, false );
		}
		pose.set_allow_rb_move(1 ,1, true );
		pose.set_allow_rb_move(4 ,1, true );
		pose.set_allow_rb_move(5 ,1, true );
		pose.set_allow_rb_move(6 ,1, true );
		pose.set_allow_rb_move(3 ,N+1, true );
		pose.set_allow_rb_move(6 ,N+1, true );

}

///////////////////////////////////////////////////////////////////////////////////////////
void
set_allow_dg_free_dn(
 pose_ns::Pose & pose
 )
 {
    using namespace pose_ns;

    int const N (pose.symmetry_info().N() );
		int const basejump ( number_jump_monomer( pose ) );

    pose.set_allow_jump_move( basejump + 2*N - 1, true );
    pose.set_allow_jump_move( basejump + 1, true );

    for (int i=1; i<= 6; ++i) {
      pose.set_allow_rb_move(i ,basejump + 2*N - 1, false );
    }
    pose.set_allow_rb_move(3 ,basejump + 2*N - 1, true );
    pose.set_allow_rb_move(6 ,basejump + 2*N - 1, true );

}

/////////////////////////////////////////////////////////////////////////////////////////
void
set_allow_dg_free_virus(
 pose_ns::Pose & pose
 )
 {
    using namespace pose_ns;

    int const N (pose.symmetry_info().N() );
    int const basejump ( number_jump_monomer( pose ) );

    pose.set_allow_jump_move( basejump + 1, true );
    pose.set_allow_jump_move( basejump + N + 1, true );

    for (int i=1; i<= 6; ++i) {
			pose.set_allow_rb_move(i ,basejump + 1, true );
      pose.set_allow_rb_move(i ,basejump + N + 1, false );
    }
		pose.set_allow_rb_move(2 ,basejump + 1, false );
    pose.set_allow_rb_move(6 ,basejump + N + 1, true );

}

//////////////////////////////////////////////////////////////////////////
//// Loop stuff!																									////////
//////////////////////////////////////////////////////////////////////////

void
pose_docking_symm_loop_protocol(
  pose_ns::Pose & oligomer_pose,
  int const nres_monomer,
  pose_ns::Loops & loops
)
{
  using namespace docking;
  using namespace pose_ns;

  std::cout << " pose_docking_symm_loop_protocol excecuted " << "\n";

  pose_ns::Pose monomer_pose, pose, pose_save;
  pose_ns::Score_weight_map weight_map;

  int const N ( docking::pose_symm_n_monomers );
  assert ( N );
  int anchor;
  std::string type ( symm_type );
  bool const fullatom ( get_docking_fullatom_flag() );
  numeric::xyzVector_float z_axis, origin;

  // Create monomer pose from oligomer_pose
  create_monomer_pose(oligomer_pose, monomer_pose, nres_monomer);

  // Setup coordinate system  and create a new symmetric pose
  pose_docking_symm_coordsys_setup(monomer_pose, oligomer_pose, anchor, z_axis, origin, 50);

  // Order the loop in sequence order
  Loops ordered_loops = loops;
  ordered_loops.sequential_order();

  // Create fold tree for monomer
  pose_docking_symm_monomer_loop_tree( monomer_pose, anchor, ordered_loops);
  std::cout << monomer_pose.fold_tree() << std::endl;
  create_symmetric_pose( monomer_pose, fullatom, N, anchor, origin, z_axis, type, pose );

	// boinc stuff
	Pdb_info pdb_info;
  pdb_info.dimension( pose.total_residue());
  pdb_info.copy_segment( oligomer_pose.total_residue(), oligomer_pose.pdb_info(), 1, 1 );
  pose.set_pdb_information( pdb_info );

// Save pose
  pose_save = pose;

  const Fold_tree & f( pose.fold_tree() );
  int const num_fold_tree_cutpoint( f.get_num_fold_tree_cutpoint() );
  const FArray1D_int & cutpoint_map( f.get_cutpoint_map() );

  std::cout << f << std::endl;

	static bool const allow_rb_move = truefalseoption("allow_rb_move");

	if (allow_rb_move) {
		// Optimize rb position
  	int		outer_cycle = 10;
  	int		inner_cycle = 50;
  	float	trans_magnitude = 0.7; /* angstrom */
  	float	rot_magnitude = 0.5; /* degree */
		set_allow_dg_free_cn(pose);
		setup_score_weight_map( weight_map, score4d );
  	pose_docking_symm_centroid_rigid_body_adaptive( pose, outer_cycle,
  	inner_cycle, weight_map, trans_magnitude, rot_magnitude);
	}

  if ( loops.begin()->is_extended() )
      pose_loops_random_start( pose, loops );

  setup_score_weight_map( weight_map, score4L );
  weight_map.set_weight( CHAINBREAK, 1.0 );
  weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
  FArray1D_float cut_weight( num_fold_tree_cutpoint, 0.0 );
  weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );

  pose.score( weight_map );

  set_pose_loop_flag( true );
  pose.set_allow_jump_move( false );
  pose.set_allow_bb_move( false );
	pose.set_allow_chi_move( false );

//  int counter (0);
	for ( Loops::const_iterator it=ordered_loops.begin(),
            it_end=ordered_loops.end(); it != it_end; ++it ) {

    // score4L with chainbreak weight and cut_weight
		if ( it->start() != 1) {
    	cut_weight( cutpoint_map(it->cut()) ) = 1.0;
    	weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );
		}

		if ( it->is_extended() ) {// centroid building only if extend.
      Loops one_loop;
      one_loop.add_loop( it );
      pose_loops_set_allow_bb_move( pose, one_loop );
      std::cout << "perturb one_loop in centroid mode " << "\n" << one_loop;
      pose_perturb_one_loop_with_ccd( pose, weight_map, one_loop );
    }
    //pose.dump_pdb("after_loop_"+string_of(++counter)+".pdb");

  }

  if ( fullatom ) {
    pose.set_fullatom_flag(true,false);
    pose.recover_sidechain( pose_save );
    } else {
      return;
    }

  // repack
  int const nres ( pose.total_residue() );
  FArray1D_bool docking_repack(nres,false);
  FArray1D_bool loop_repack(nres,false);
  FArray1D_bool allow_repack(nres,false);
  pose_docking_symm_calc_interface( pose, docking_repack );
  pose_loops_select_loop_residues_symm( pose, loops, true/*w/ nbs*/, loop_repack );
  for( int i = 1; i <= nres; ++i ) {
    allow_repack(i) = docking_repack(i) || loop_repack(i);
  }
	pose.set_allow_chi_move(allow_repack);
  pose.repack( allow_repack, true/*include_current*/ );

	if (allow_rb_move) {
		// Do a short MCM to optimize rb
  	const float loose_func_tol = 1.0;  /* absolute tolerance */
  	//const float tight_func_tol = 0.02; /* absolute tolerance */
  	const std::string min_type = "dfpmin_atol";
  	// monte carlo
  	const float minimization_threshold = 15.0; /* score unit */
		float trans_magnitude = 0.7; /* angstrom */
    float rot_magnitude = 0.5; /* degree */
  	// scorefxn to minimize
  	setup_score_weight_map( weight_map, score10d_min );
		// Do a minimize first
		set_allow_dg_free_cn(pose);
		pose_docking_symm_minimize_trial( pose, min_type, weight_map,
                                 loose_func_tol);

  	static int const cycles = intafteroption("MC_cycles_fa",4);
  	pose_docking_symm_monte_carlo_minimize( pose, cycles, min_type,
  		weight_map, trans_magnitude,rot_magnitude, minimization_threshold,
  		loose_func_tol );

		//reset allow_jump
		pose.set_allow_jump_move( false );
		loop_repack = false;
		 allow_repack = false;
		pose_docking_symm_calc_interface( pose, docking_repack );
  	pose_loops_select_loop_residues_symm( pose, loops, true/*w/ nbs*/, loop_repack );
  	for( int i = 1; i <= nres; ++i ) {
    	allow_repack(i) = docking_repack(i) || loop_repack(i);
  	}
  	pose.set_allow_chi_move(allow_repack);

	}

  // initialize a fullatom weight_map with cut_weight
  setup_score_weight_map( weight_map, score10d_min );
  weight_map.set_weight( CHAINBREAK, 1.0 );
  weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
  weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );

  // dampen fa_rep weight
  float rep_weight_max = weight_map.get_weight( FA_REP );
  int rep_ramp_cycles = { 5 };
  float rep_ramp_step = (rep_weight_max - 0.02) / float(rep_ramp_cycles-1);
  bool no_ramp_rep = truefalseoption("no_ramp_rep");
  for ( int i = 1; i <= rep_ramp_cycles; i++ ) {
    if ( ! no_ramp_rep ) {
      float rep_weight = 0.02 + rep_ramp_step * float(i-1);
      std::cout << "rep_ramp_cycle/rep_weight/rep_weight_max: "
                << i << " / " << rep_weight << " / " << rep_weight_max << std::endl;
      weight_map.set_weight( FA_REP, rep_weight );
    }
    pose_refine_loops_with_ccd( pose, weight_map, ordered_loops, 1,
      ordered_loops.loop_size(), 1, true, true );
  }
  // Copy symmetric pose to a new pose without pseudo residues
  create_output_pose(oligomer_pose, pose);
  oligomer_pose.score( weight_map );

}

void
pose_docking_symm_monomer_loop_tree(
  pose_ns::Pose & pose,
  int const anchor,
  pose_ns::Loops & loops
)
{
   using namespace pose_ns;


  // add the loop jump into the current tree, delete some old edge accordingly
  Fold_tree f( pose.fold_tree() );
  f.clear();

  int const nres_monomer ( pose.total_residue() );
  int start (1);
  int last_loop_stop (0);
	for ( Loops::const_iterator it=loops.begin(), it_end=loops.end(), it_next;
         it != it_end; ++it ) {
        it_next = it;
        it_next++;
        int const loop_start =
          ( it->start() == 1 ) ? it->start() : it->start() - 1;
        int const loop_stop =
          ( it->stop() == nres_monomer ) ? it->stop() : it->stop() + 1;
        int loop_cutpoint ( it->cut() );

        int const loop_next_start =
          ( it_next == it_end ) ? nres_monomer : it_next->start()-1;

        last_loop_stop = loop_stop;
        if(  it->start() == 1 ){
          f.add_edge( loop_start, loop_stop, pose_param::PEPTIDE );
          f.add_edge( loop_stop, loop_next_start, pose_param::PEPTIDE );
          continue;
        }else if( it->stop() == nres_monomer ){
          f.add_edge( loop_start, loop_stop, pose_param::PEPTIDE );
          continue;
        }

        assert( loop_start < loop_stop );

        if ( loop_cutpoint == 0 ) { // random loop cutpoint
          loop_cutpoint = loop_start + int( (loop_stop - loop_start)*ran3() );
        }
        assert( loop_start <= loop_cutpoint && loop_cutpoint <= loop_stop );

        // make sure rb jumps do not reside in the loop region
        if ( anchor >= loop_start && anchor <= loop_stop ) {
          std::cout<<"anchor is in loop region! Not implemented..."<<std::endl;
          utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
        }
          int const num_jump = f.get_num_jump();
          f.add_edge( loop_start, loop_stop, num_jump+1 ); //loop_jump
          f.add_edge( start, loop_start, pose_param::PEPTIDE );
          f.add_edge( loop_start, loop_cutpoint, pose_param::PEPTIDE );
          f.add_edge( loop_cutpoint+1, loop_stop, pose_param::PEPTIDE );
        start = loop_stop;
    }

    if( loops.end()->stop() != nres_monomer && loops.begin()->start() != 1)
     f.add_edge( last_loop_stop, nres_monomer, pose_param::PEPTIDE );


  // reorder
  int root;
  if( loops.begin()->start() == 1 &&
    loops.begin()->stop() != nres_monomer )
    root = loops.begin()->stop()+1;
  else root = 1;
  f.reorder(root);
	pose.set_fold_tree(f);
}

void
pose_docking_symm_fragments_protocol(
  pose_ns::Pose & oligomer_pose,
	int const nres_monomer
)
{
	using namespace docking;
  using namespace pose_ns;

  std::cout << " pose_docking_symm_fragments_protocol excecuted " << "\n";

  pose_ns::Pose monomer_pose, pose;
  pose_ns::Score_weight_map weight_map;

  int const N ( docking::pose_symm_n_monomers );
  assert ( N );
  int anchor;
  std::string type ( symm_type );
//  bool const fullatom ( get_docking_fullatom_flag() );
  numeric::xyzVector_float z_axis, origin;

  // Create monomer pose from oligomer_pose
  create_monomer_pose(oligomer_pose, monomer_pose, nres_monomer);

  // Setup coordinate system  and create a new symmetric pose
  pose_docking_symm_coordsys_setup(monomer_pose, oligomer_pose, anchor, z_axis, origin, 50);

	 create_symmetric_pose( monomer_pose, false, N, anchor, origin, z_axis, type, pose );

	// now read frags
//	int const nres_real (pose.total_residue() - N );
//	read_fragments( nres_real );
	int const loop_begin (46);
	int const loop_end (55);
	int const loop_size (loop_end - loop_begin);

	// initialize torsions
  insert_init_frag( pose, loop_begin, loop_size );


      // setup symmetry info, allow_move

	pose.set_allow_bb_move( false );
  for ( int i=loop_begin; i<= loop_end; ++i ) pose.set_allow_bb_move( i,true);
	//	pose.set_allow_jump_move( false );

  // fold!
	monomer_pose.set_fullatom_flag(false);
  fold_fragments_docking( /*monomer_pose,*/ pose, false /* no chainbreak scoring */ );

	// Copy symmetric pose to a new pose without pseudo residues
	create_output_pose(oligomer_pose, pose);
	oligomer_pose.score( weight_map );

	oligomer_pose.dump_pdb("fold.pdb");

}

void
fold_fragments_docking(
											 /*	pose_ns::Pose & monomer_pose,*/
  pose_ns::Pose & pose,
  bool const score_chainbreaks // = true
)
{
  using namespace pose_ns;

  int number3merfrags = get_number3merfrags(); // Default 25
  int number9merfrags = get_number9merfrags(); // Default 200

  std::cout << "score_chainbreaks: " << score_chainbreaks << std::endl;

  float increasecycles = 1; //get_increasecycles(); // default is just 1

  // params
  int const score0_cycles(  static_cast< int > (200 * increasecycles) );
  int const score1_cycles(  static_cast< int > (200 * increasecycles) );
  int const score25_cycles( static_cast< int > (200 * increasecycles) );
  int const score3_cycles(  static_cast< int > (400 * increasecycles) );
  float const init_temp( 2.0 );
  choose_frag_set_top_N_frags(number9merfrags); // use top 25 frags

  // init score:
  Score_weight_map score0_weight_map( score0 );
  pose.score( score0_weight_map );
	Score_weight_map weight_map_docking;
	setup_score_weight_map( weight_map_docking, score4d );
  // create Monte_carlo object that we will use throughout
  Monte_carlo mc( pose, score0_weight_map, init_temp );
  mc.set_autotemp( true, init_temp );

  std::cout << "starting fragment insertions..." << std::endl;
  //pose.dump_pdb("start.pdb");

// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
//  insert secondary structure, bump check only
// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

  {
    int const size( 9 );

    for ( int j = 1; j <= score0_cycles; ++j ) {
			separate_oligomers(pose); //testing IA!!!!
      choose_fragment_pose( pose, size, true, true );
			pose_docking_symm_perturb_rigid_body(pose,0.0); //testing IA!!!
			pose_docking_symm_centroid_rigid_body_adaptive( pose, 5/*outer_cycle*/,
			    20 /*inner_cycle*/, score0_weight_map, 0.7/*trans_mag*/, 0.5/*rot_mag*/); //testing IA
      mc.boltzmann( pose );
      if ( start_sim( pose ) ) break; // all swapped one or more times
      if ( j == score0_cycles ) {
        std::cout << "WARNING:: extended chain may still remain!" << std::endl;
      }
    }
    ai_stats( "score0", pose );
  }

// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
//    regular moves   w/bumps,secondary structure, light strand pairing
//    no rg or cbeta yet
// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

  {
    int const size(9);

    Score_weight_map weight_map( score1 );
    if ( score_chainbreaks ) {
      weight_map.set_weight( CHAINBREAK, 0.25 );
      weight_map.set_weight( CHAINBREAK_OVERLAP, 0 );
    }
    mc.set_weight_map( weight_map );

    mc.set_temperature( init_temp );

    for ( int j = 1; j <= score1_cycles; ++j ) {
			separate_oligomers(pose); //testing IA!!!!
      choose_fragment_pose( pose, size, true, true );
			pose_docking_symm_perturb_rigid_body(pose,0.0); //testing IA!!!
      pose_docking_symm_centroid_rigid_body_adaptive( pose, 5/*outer_cycle*/,
          20 /*inner_cycle*/, weight_map, 0.7/*trans_mag*/, 0.5/*rot_mag*/); //testing IA
      mc.boltzmann( pose );
    }

    ai_stats( "score1", pose );
  }

// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
//   Add cbeta, rg (collapse chain)
//   alternate score2/score5 (high/low sheet weight)
//   convergence-checking enabled in monte_carlo
// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

  {
    int const nloop = 1;
    int const nloop2 = 10;
    int const size = 9;

    mc.set_temperature( init_temp );

    for ( int jj = 1; jj <= nloop; ++jj ) {
      for ( int kk = 1; kk <= nloop2; ++kk ) {

        float chainbreak_weight(0.0);
        Score_weight_map weight_map; // empty

        if ( mod( kk, 2 ) == 0 || kk > 7 ) {
          // score2
          chainbreak_weight = jj*kk*0.25;
          weight_map.set_weights( score2 );
        } else {
          // score5
          chainbreak_weight = jj*kk*0.05;
          weight_map.set_weights( score5 );
        }

        if ( score_chainbreaks ) {
          weight_map.set_weight( CHAINBREAK, chainbreak_weight );
          weight_map.set_weight( CHAINBREAK_OVERLAP, 0.0 );
        }
        mc.set_weight_map( weight_map );
        pose = mc.low_pose();

        for ( int j = 1; j <= score25_cycles; ++j ) {
					separate_oligomers(pose); //testing IA!!!!
          choose_fragment_pose( pose, size, true, true );
					pose_docking_symm_perturb_rigid_body(pose,0.0); //testing IA!!!
		      pose_docking_symm_centroid_rigid_body_adaptive( pose, 5/*outer_cycle*/,
          20 /*inner_cycle*/, weight_map, 0.7/*trans_mag*/, 0.5/*rot_mag*/); //testing IA

          mc.boltzmann( pose );
        }

        ai_stats( "score2/5", pose );
      }
    }
  }

// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
//   regular moves/smooth moves
//   score 3 ->all centroid based terms on
// XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

  {
    int const nloop = 3;
    int const size = 3;
    float const gunn_cutoff( 7.0 );

    for ( int kk = 1; kk <= nloop; ++kk ) {
      Score_weight_map weight_map( score3 );
      if ( score_chainbreaks ) {
        weight_map.set_weight( CHAINBREAK, kk*0.5+2.5);
        weight_map.set_weight( CHAINBREAK_OVERLAP, kk );
      }
      mc.set_weight_map( weight_map );

      pose = mc.low_pose();

      mc.set_temperature( init_temp );

      if ( kk == 2 ) choose_frag_set_top_N_frags(number3merfrags);
      for ( int j = 1; j <= score3_cycles; ++j ) {
        if ( kk == 1 ) {
					separate_oligomers(pose); //testing IA!!!!
          choose_fragment_pose( pose, size, true, true );
          mc.boltzmann( pose );
        } else {
					separate_oligomers(pose); //testing IA!!!!
          choose_fragment_gunn_pose( pose, size, gunn_cutoff );
					pose_docking_symm_perturb_rigid_body(pose,0.0); //testing IA!!!
          pose_docking_symm_centroid_rigid_body_adaptive( pose, 5/*outer_cycle*/,
          20 /*inner_cycle*/, weight_map, 0.7/*trans_mag*/, 0.5/*rot_mag*/); //testing IA
          mc.boltzmann( pose );
        }
      }                  // loop j
      ai_stats( "score3", pose );
    }                     // kk
  }

  score_set_evaluate_all_terms(true);
  pose = mc.low_pose();
  pose.score( score4 ); // output-score
  score_set_evaluate_all_terms(false);
}

///////////////////////////////////////////////////////////////////////////////

void
separate_oligomers(
  pose_ns::Pose & pose
)
{
	using namespace pose_ns;
  using namespace docking;

  int const njump_monomer( number_jump_monomer( pose ) );
	int const sep (-5);
	int const c2n (-1);

	pose.set_allow_jump_move( njump_monomer+1, true );
	pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );


}

void
pose_docking_symm_perturb_rigid_body_fold(
  pose_ns::Pose & monomer_pose,
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
  using namespace docking;

	int const njump_monomer( number_jump_monomer( pose ) );

	pose.set_allow_bb_move( false );
  pose.set_allow_jump_move( false );
  pose.set_allow_jump_move( njump_monomer+1, true );

	// Set weight_map using centroid mode scoring
	pose_ns::Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4d );
	pose.score( weight_map );
	int const c2n(-1);

	float vdw( pose.get_0D_score( pose_ns::DOCK_VDW ) );
	float monomer_vdw;
	{
		pose_ns::Pose tmp_pose;
		tmp_pose = monomer_pose;
		tmp_pose.score( weight_map );
	  monomer_vdw = tmp_pose.get_0D_score( pose_ns::DOCK_VDW );
	//	std::cout << tmp_pose.show_scores() << std::endl;
	}

	// Separation distance
	double sep(0.0);
//  std::cout << "monomer_vdw= " << monomer_vdw << " vdw= " << vdw <<
//						   " sep= " << sep << std::endl;
static int counter(0);
// first try moving away from each other
	 while ( vdw > 3*monomer_vdw + 0.3 ) {
    sep += 1.0;
    pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
		pose.score( weight_map );
    vdw = pose.get_0D_score( pose_ns::DOCK_VDW );
//		std::cout << "vdw= " << vdw << ' ' << "sep= " << sep << std::endl;
	//	pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
	}
//	then try moving towards each other
	while ( vdw < 3*monomer_vdw + 0.3 ) {
		sep -= 1.0;
		pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );
		pose.score( weight_map );
		vdw = pose.get_0D_score( pose_ns::DOCK_VDW );

		// Print output for debugging
		std::cout << "vdw= " << vdw << ' ' << "sep= " << sep << ' ' << counter
		          << std::endl;
  	// pose.dump_pdb("slide_test."+string_of(++counter)+".pdb");
  }
	  sep += 1.0;
	  pose.set_jump_rb_delta( njump_monomer+1, 1, c2n, sep );

}

//////////////////////////////////////////////////////////////////////////
void
calc_param_virus()
{
  using namespace misc;
  using numeric::constants::f::pi;
  using namespace numeric;
  using namespace misc;
  using namespace param;

  pose_ns::Pose monomer_pose, oligomer_pose;
  pose_from_pdb( oligomer_pose, stringafteroption( "s" ), true, false,true, 'K' );
  pose_from_pdb( monomer_pose, stringafteroption( "s" ), true, false);

  int nres_monomer (monomer_pose.total_residue() );
  int nres_oligomer (oligomer_pose.total_residue() );
  int anchor = monomer_pose.residue_center_of_mass(1, nres_monomer);
	std::cout<<" anchor "<<anchor<<std::endl;

  FArray3D_float Epos( oligomer_pose.Eposition() );
  FArray3D_float fcoord( oligomer_pose.full_coord() );

	xyzVector_double v1 (&Epos(1,2,anchor));
	xyzVector_double v2 (&Epos(1,2,anchor+nres_monomer));
	xyzVector_double v3 (&Epos(1,2,anchor+2*nres_monomer));
	xyzVector_double ori0 = (v1+v2+v3)/3;
	xyzVector_double x_new = v1-ori0;
	float radius = x_new.length();
	x_new.normalize();
	numeric::xyzVector_double x_vec,z_vec,ori;
	return_xyz_vectors_virus(x_vec,z_vec,ori);
	std::cout<<ori.length()<<std::endl;
	float angle =acos( dot_product(x_new,x_vec)/(x_new.length()*x_vec.length()) );
	std::cout<<" angle "<<angle*180/pi<<" size "<<ori0.length()<<" radius "<<radius<<std::endl;

  numeric::xyzVector_float translate_vector ((ori0-v1)/2);
	numeric::xyzVector_float z (z_vec);
	numeric::xyzMatrix_float R( rotation_matrix(z, -angle ) );
	translate_vector = R*translate_vector;


	for ( int i=1; i<=nres_oligomer; ++i ) {
    int const natoms( aaproperties_pack::natoms( res(i), res_variant(i) ) );
    for ( int k=1; k<=3; ++k ) {
      for ( int j=1; j<= natoms; ++j ) {
      numeric::xyzVector_float Ecor( &oligomer_pose.Eposition()(1,j,i) );
      numeric::xyzVector_float Fcor( & oligomer_pose.full_coord()(1,j,i) );
       if ( j <= 5){
         Epos(k,j,i) = ( R * Ecor )(k);
         }
       fcoord(k,j,i) = ( R * Fcor )(k);
      }
    }
  }

  oligomer_pose.set_coords( false, Epos, fcoord );
	oligomer_pose.dump_pdb("oligomer_pose.pdb");


}
////////////////////////////////////////////////////////////////////////////////
void
return_xyz_vectors_virus(
	numeric::xyzVector_double & x_vec,
	numeric::xyzVector_double & z_vec,
	numeric::xyzVector_double & ori
){
	//Calculate the rotation axis of the first face of the icosahedron
  //Together with the position of the first monomer on the face

  using numeric::constants::f::pi;

  // Generate the icosahedron by specifying the vertices

  int size = 20;
  double golden = (1 + std::sqrt(5.0))/2;

  std::vector<numeric::xyzVector_float> vertices;
  vertices.push_back(numeric::xyzVector_float(-size*golden,0.0,-size));
  vertices.push_back(numeric::xyzVector_float(0.0,-size, -size*golden));
  vertices.push_back(numeric::xyzVector_float(-size,-size*golden,0.0));

  numeric::xyzVector_float v1( vertices[ 0 ]);
  numeric::xyzVector_float v2( vertices[ 1 ]);
  numeric::xyzVector_float v3( vertices[ 2 ]);
  numeric::xyzVector_float v4( (v2-v1)/2 +v1 );
  numeric::xyzVector_float v5( v3-v4 );
  numeric::xyzVector_float v6( v2-v1 );
  v5.normalize();
  float const dist ( std::tan(30.0/360.0*2.0*pi)*v6.length()/2 );
  numeric::xyzVector_float origin( v4+v5*dist );
  numeric::xyzVector_float v7 (origin - v1 );
  numeric::xyzVector_float v8 ( v3 - v1 );
  numeric::xyzVector_float v9 ( cross(v6,v8) );
  v9.normalize();
  numeric::xyzVector_float v10 ( (v3 - origin)/2 );
  v10.normalize();
  numeric::xyzVector_float v11 ( cross(v9,v10) );
  v11.normalize();

	x_vec = v10;
  z_vec = v9;
	ori = origin;

}
///////////////////////////////////////////////////////////////////////////////////
int
number_jump_monomer(
	pose_ns::Pose & pose
){
	int num_jump_monomer( 0 );

	//Count the number of jumps that don't involve pseudoresidues.
	for (int n = 1; n <= pose.fold_tree().get_num_jump(); n++){
		int const jump_point1 = pose.fold_tree().upstream_jump_residue( n );
		int const jump_point2 = pose.fold_tree().downstream_jump_residue( n );

		if ( pose.pseudo( jump_point1 ) || pose.pseudo( jump_point2) ) continue;
		if ( !pose.symmetry_info().bb_independent( jump_point1 ) ) continue;
		if ( !pose.symmetry_info().bb_independent( jump_point2 ) ) continue;

		num_jump_monomer++;
	}

	return num_jump_monomer;
}
///////////////////////////////////////////////////////////////////////////////////////
void
set_allow_chi_move_symm(
	pose_ns::Pose & pose
){
	if ( get_docking_interface_pack() ) {
		int const nres ( pose.total_residue() );
		FArray1D_bool allow_repack( nres, false );
		pose_docking_symm_calc_interface( pose, allow_repack );
		for ( int i=1; i<=nres; ++i) {
			if ( allow_repack(i) ) pose.set_allow_chi_move(i, true);
		}
	} else {
		pose.set_allow_chi_move( true );
	}
}
///////////////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////////////
/// This function recapitulate the native configuaration of monomers based on
// for cyclic symmetry starting with at least 3 monomers
void
prepare_native()
{

  // Read starting structure into poses
  pose_ns::Pose monomer_pose,pose, oligomer_pose;
  pose_from_pdb( pose, stringafteroption( "s" ), true, false,true, 'A' );
  pose_from_pdb( monomer_pose, stringafteroption( "s" ), true, false);

  int N;
  intafteroption("monomers",6,N);
	create_full_symmetric_pose( monomer_pose, pose, N);
}

///////////////////////////////////////////////////////////////////////////////////////
void
prepare_native_cn(
	pose_ns::Pose & pose,
	pose_ns::Pose & monomer_pose,
	pose_ns::Pose & oligomer_pose
)
{
	int const N ( docking::pose_symm_n_monomers );
	oligomer_pose = pose;
	create_full_symmetric_pose( monomer_pose, oligomer_pose, N);
}

void
create_full_symmetric_pose(
  pose_ns::Pose & pose,
	int const N
)
{
	using namespace pose_ns;
	Symmetry_info const & s ( pose.symmetry_info() );
	int const N_sub = s.N_bb();
	int const total_residue = pose.total_residue_for_scoring();
	int const nres_monomer = total_residue/N_sub;
	Pose monomer_pose;
	create_monomer_pose(pose,monomer_pose,nres_monomer);
	if (!pose.fullatom()) monomer_pose.set_fullatom_flag(false);
	create_full_symmetric_pose(monomer_pose,pose,N);

}
void
create_full_symmetric_pose(
	pose_ns::Pose & monomer_pose,
	pose_ns::Pose & oligomer_pose,
	int const N
)
{
	using namespace misc;
  using numeric::constants::f::pi;
  using namespace numeric;

	int Mon (N);
	if ( truefalseoption("docking_pose_symm_subsystem") ) {
		Mon = 3;
	}

  int nres_monomer = monomer_pose.total_residue();
  int anchor = monomer_pose.residue_center_of_mass(1, nres_monomer);
  int anchor2 = anchor + nres_monomer;
  int anchor3 = anchor + nres_monomer*2;

  float dx = oligomer_pose.Eposition()(1,2,anchor) - oligomer_pose.Eposition()(1,2,anchor2);
  float dy = oligomer_pose.Eposition()(2,2,anchor) - oligomer_pose.Eposition()(2,2,anchor2);
  float dz = oligomer_pose.Eposition()(3,2,anchor) - oligomer_pose.Eposition()(3,2,anchor2);

  float dx2 = oligomer_pose.Eposition()(1,2,anchor2) - oligomer_pose.Eposition()(1,2,anchor3);
  float dy2 = oligomer_pose.Eposition()(2,2,anchor2) - oligomer_pose.Eposition()(2,2,anchor3);
  float dz2 = oligomer_pose.Eposition()(3,2,anchor2) - oligomer_pose.Eposition()(3,2,anchor3);

  float dist = dx*dx +dy*dy +dz*dz;
  dist = std::sqrt(dist);

  // Calculate radius
  float alfa = pi/N;
  float r = dist/(2*std::sin(alfa));

  std::cout<<"distance calculated: "<<r<<std::endl;
  std::cout<<"anchor1, anchor2 "<< anchor << " " << anchor2 << std::endl;

  // Find coordinate system
   numeric::xyzVector_float plane ( dx,dy,dz );
   numeric::xyzVector_float plane2 ( dx2,dy2,dz2 );
   xyzVector_float z_axis = cross(plane, plane2);
   z_axis.normalize();

  numeric::xyzVector_float x = cross(z_axis,plane);
  x.normalize();
  numeric::xyzVector_float x_axis = r*std::cos(alfa)*x + plane/2;
  numeric::xyzVector_float delta ( x_axis );
  x_axis.normalize();
  numeric::xyzVector_float y_axis = cross(z_axis,x_axis);
  numeric::xyzMatrix_float const R( numeric::xyzMatrix_float::rows( x_axis, y_axis, z_axis ) );

	 // Save coordinates into temporary coordinates
  FArray3D_float Epos( monomer_pose.Eposition() );
  FArray3D_float Epos2( monomer_pose.Eposition() );
  FArray3D_float fcoord( monomer_pose.full_coord() );
  FArray3D_float fcoord2( monomer_pose.full_coord() );

  // Create simple fold tree
  int num_fold_tree_cut_points;
  FArray2D_int const & jump_point_monomer
    ( monomer_pose.fold_tree().get_jump_point() );
  FArray1D_int const & cuts_monomer
    ( monomer_pose.fold_tree().get_fold_tree_cutpoint(num_fold_tree_cut_points) );

  int const njump_monomer( monomer_pose.num_jump() );
  int const nres( Mon * nres_monomer );
  int const njump( Mon * njump_monomer );

  FArray2D_int jump_point( 2, njump );
  FArray1D_int cuts( njump );

  int jump_number(0);
  for ( int k=0; k<Mon; ++k ) {
    int const rsd_offset( nres_monomer * k );
    // first the jumps and cuts from the monomer
    for ( int j=1; j<= njump_monomer; ++j ) {
      jump_point( 1, ++jump_number ) = rsd_offset + jump_point_monomer( 1, j );
      jump_point( 2,   jump_number ) = rsd_offset + jump_point_monomer( 2, j );
      cuts( jump_number ) = cuts_monomer( j );
    }
  }
  pose_ns::Fold_tree f;

  // this builds the tree:
  f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );

  // put into pose
  oligomer_pose.set_fold_tree( f );
	bool fullatom (false);
	if ( monomer_pose.fullatom() ) fullatom=true;
 oligomer_pose.set_fullatom_flag( fullatom, false );


	// Translate to origin and rotate coordinates into the absolute coordinate
  // system. Then translate radius distance out on x.
  numeric::xyzVector_float x_add(numeric::xyzVector_float(r,0,0));
  numeric::xyzVector_float anchor_pos( &Epos(1,2,anchor) );
  for ( int i=1; i<=nres_monomer; ++i ) {
       int const natoms
         ( aaproperties_pack::natoms( res(i), res_variant(i) ) );
       for ( int k=1; k<=3; ++k ) {
         for ( int j=1; j<= natoms; ++j ) {
						if (j <= 5 || fullatom) {
            	numeric::xyzVector_float E( &monomer_pose.Eposition()(1,j,i) );
            	numeric::xyzVector_float F( &monomer_pose.full_coord()(1,j,i) );
            	if ( j <= 5)
              	Epos(k,j,i) = (R*(E - anchor_pos))(k) + x_add(k);
            	fcoord(k,j,i) = (R*(F - anchor_pos))(k) + x_add(k);
						}
         	}
        }
      }


  // Create other monomers by rotation
  pose_ns::Pose & pose( monomer_pose );
  pose_ns::Pose & tmp_pose( monomer_pose );
  pose.set_fullatom_flag(true, false);
  pose.simple_fold_tree( nres_monomer );
  pose.set_coords( false, Epos, fcoord );
int hand = 1;
  for ( int k=1; k<=Mon; ++k ){
    numeric::xyzMatrix_float const M( Z_rot_rad( hand * (k-1) * 2 * pi / N ) );
    for ( int i=1; i<=nres_monomer; ++i ) {
       int const natoms
         ( aaproperties_pack::natoms( res(i), res_variant(i) ) );
        for ( int k=1; k<=3; ++k ) {
          for ( int j=1; j<= natoms; ++j ) {
            if ( j <= 5) {
              numeric::xyzVector_float E( &Epos(1,j,i) );
               Epos2(k,j,i) = ( M*E )(k);
            }
            numeric::xyzVector_float F( &fcoord(1,j,i) );
            fcoord2(k,j,i) = ( M*F )(k);
          }
        }
    }
    tmp_pose.set_coords( false, Epos2, fcoord2 );
    tmp_pose.set_fullatom_flag( true, false);
    oligomer_pose.copy_segment( nres_monomer, tmp_pose, (k-1)*nres_monomer+1, 1, false );
  }
    // output final transformed pose
    oligomer_pose.update_backbone_bonds_and_torsions();
		static const bool dump_single_pdb = truefalseoption("dump_single_pdb");
		if (dump_single_pdb) oligomer_pose.dump_pdb("final.pdb");
}

///////////////////////////////////////////////////////////////////////////////////////////////

void
prepare_native_dimer()
{
  // Read starting structure into poses
  pose_ns::Pose monomer_pose,dimer_pose, oligomer_pose;
  pose_from_pdb( dimer_pose, stringafteroption( "s" ), true, false,true, 'A' );
  pose_from_pdb( monomer_pose, stringafteroption( "s" ), true, false);

	prepare_native_dimer( dimer_pose, monomer_pose,
												oligomer_pose );

	// output final transformed pose
	oligomer_pose.dump_pdb("final.pdb");

}

void
prepare_native_dimer( pose_ns::Pose & dimer_pose, pose_ns::Pose & monomer_pose,
											pose_ns::Pose & oligomer_pose )
{
  using namespace misc;
  using numeric::constants::f::pi;
  using namespace numeric;

	bool const fullatom ( dimer_pose.fullatom() );
  int N (2);

  int nres_monomer = monomer_pose.total_residue();
  int anchor = monomer_pose.residue_center_of_mass(1, nres_monomer);
  int anchor2 = anchor + nres_monomer;

  float dx = dimer_pose.Eposition()(1,2,anchor) - dimer_pose.Eposition()(1,2,anchor2);
  float dy = dimer_pose.Eposition()(2,2,anchor) - dimer_pose.Eposition()(2,2,anchor2);
  float dz = dimer_pose.Eposition()(3,2,anchor) - dimer_pose.Eposition()(3,2,anchor2);

  // Save coordinates into temporary coordinates
  FArray3D_float Epos( monomer_pose.Eposition() );
  FArray3D_float Epos2( monomer_pose.Eposition() );
  FArray3D_float fcoord( monomer_pose.full_coord() );
  FArray3D_float fcoord2( monomer_pose.full_coord() );


  float dist = dx*dx +dy*dy +dz*dz;
  float r = std::sqrt(dist)/2;

  std::cout<<"distance calculated: "<<r<<std::endl;

// Find the Z axis in the native coordinates
  std::vector<numeric::xyzVector_float> ca_array;

  for ( int i=1; i<=nres_monomer; ++i ) {
            numeric::xyzVector_float E( &dimer_pose.Eposition()(1,1,i) );
            numeric::xyzVector_float F( &dimer_pose.full_coord()(1,1,i+nres_monomer) );
            ca_array.push_back( (E+F)/2 );
      }

  // Find the two average ca coordinates that aare furthest away and use that veector as z_axis
  numeric::xyzVector_float high, low;
  float distance = 0;
  float distance_save = 0;
  for ( int i=1; i<=nres_monomer; ++i ) {
    for ( int j=1; j<=nres_monomer; ++j ) {
      distance = distance_squared(ca_array[i-1],ca_array[j-1]);
      if ( distance > distance_save ) {
        high = ca_array[i-1];
        low = ca_array[j-1];
        distance_save = distance;
      }
    }
  }
  numeric::xyzVector_float z_axis = ( high-low ).normalize();
  std::cout << "z_axis caalculated to: " << z_axis << std::endl;
	if ( truefalseoption("invert_rot_axis") ) {
		z_axis = -z_axis;
	}

  // Find coordinate system
  numeric::xyzVector_float plane ( dx,dy,dz );
  numeric::xyzVector_float x_axis = plane;
  x_axis.normalize();
  numeric::xyzVector_float y_axis = cross(z_axis, x_axis);

  // construct rotation axis to convert the native coord system into the absolute coord system
  numeric::xyzMatrix_float const R( numeric::xyzMatrix_float::rows( x_axis, y_axis, z_axis ) );

  // Create simple fold tree
  int num_fold_tree_cut_points;
	FArray2D_int const & jump_point_monomer
    ( monomer_pose.fold_tree().get_jump_point() );
  FArray1D_int const & cuts_monomer
    ( monomer_pose.fold_tree().get_fold_tree_cutpoint(num_fold_tree_cut_points) );

  int const njump_monomer( monomer_pose.num_jump() );
  int const nres( N * nres_monomer );
  int const njump( N * njump_monomer );

  FArray2D_int jump_point( 2, njump );
  FArray1D_int cuts( njump );

  int jump_number(0);
  for ( int k=0; k<N; ++k ) {
    int const rsd_offset( nres_monomer * k );
    // first the jumps and cuts from the monomer
    for ( int j=1; j<= njump_monomer; ++j ) {
      jump_point( 1, ++jump_number ) = rsd_offset + jump_point_monomer( 1, j );
      jump_point( 2,   jump_number ) = rsd_offset + jump_point_monomer( 2, j );
      cuts( jump_number ) = cuts_monomer( j );
    }
  }
  pose_ns::Fold_tree f;

  // this builds the tree:
  f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );

  // put into pose
  oligomer_pose.set_fold_tree( f );
	if ( fullatom ) {
  	oligomer_pose.set_fullatom_flag( true, false );
	} else {
		oligomer_pose.set_fullatom_flag( false, false );
	}

// Translate to origin and rotate coordinates into the absolute coordinate
  // system. Then translate radius distance out on x.
  numeric::xyzVector_float x_add(numeric::xyzVector_float(r,0,0));
  numeric::xyzVector_float anchor_pos( &Epos(1,2,anchor) );
  for ( int i=1; i<=nres_monomer; ++i ) {
       int const natoms
         ( aaproperties_pack::natoms( res(i), res_variant(i) ) );
       for ( int k=1; k<=3; ++k ) {
         for ( int j=1; j<= natoms; ++j ) {

            numeric::xyzVector_float F( &monomer_pose.full_coord()(1,j,i) );
            fcoord(k,j,i) = (R*(F - anchor_pos))(k) + x_add(k);

            if ( j <= param::MAX_POS) {
							numeric::xyzVector_float E( &monomer_pose.Eposition()(1,j,i) );
              Epos(k,j,i) = (R*(E - anchor_pos))(k) + x_add(k);
						}

         }
        }
      }


  // Create other monomers by rotation
  pose_ns::Pose & pose( monomer_pose );
  pose_ns::Pose & tmp_pose( monomer_pose );
	if ( fullatom ) {
		pose.set_fullatom_flag(true, false);
	} else {
		pose.set_fullatom_flag(false, false);
	}
  pose.simple_fold_tree( nres_monomer );
	pose.set_coords( false, Epos, fcoord );
  int hand = 1;
  for ( int k=1; k<=N; ++k ){
    numeric::xyzMatrix_float const M( Z_rot_rad( hand * (k-1) * 2 * pi / N ) );
    for ( int i=1; i<=nres_monomer; ++i ) {
       int const natoms
         ( aaproperties_pack::natoms( res(i), res_variant(i) ) );
        for ( int k=1; k<=3; ++k ) {
          for ( int j=1; j<= natoms; ++j ) {
            if ( j <= 5) {
              numeric::xyzVector_float E( &Epos(1,j,i) );
               Epos2(k,j,i) = ( M*E )(k);
            }
            numeric::xyzVector_float F( &fcoord(1,j,i) );
            fcoord2(k,j,i) = ( M*F )(k);
          }
        }
    }
    tmp_pose.set_coords( false, Epos2, fcoord2 );
		if ( fullatom ) {
    	tmp_pose.set_fullatom_flag( true, false);
		 } else {
			tmp_pose.set_fullatom_flag( false, false);
		}
    oligomer_pose.copy_segment( nres_monomer, tmp_pose, (k-1)*nres_monomer+1, 1, false );
  }

	oligomer_pose.update_backbone_bonds_and_torsions();
}
//////////////////////////////////////////////////////////////////////////////
void
prepare_native_d2(
	pose_ns::Pose & pose,
	pose_ns::Pose & monomer_pose,
	pose_ns::Pose & oligomer_pose
)
{
	using namespace pose_ns;

	Pose dimer_pose, rot_monomer_pose;
	int const nres_monomer ( monomer_pose.total_residue() );
	prepare_native_dimer(pose,monomer_pose,dimer_pose);
	rot_monomer_pose.simple_fold_tree( nres_monomer );
  rot_monomer_pose.copy_segment( nres_monomer, dimer_pose, 1, 1, true );
	int const anchor = monomer_pose.residue_center_of_mass(1, nres_monomer);
	float dist, radius,angle;
	find_dist_radius_angle_d2(pose,anchor,nres_monomer,dist,radius,angle);
	symm_pose_dn(rot_monomer_pose, false /*fullatom*/, 4, anchor, dist, radius, angle, oligomer_pose);

}
//////////////////////////////////////////////////////////////////////////////
void
find_dist_radius_angle_d2(
	pose_ns::Pose & pose,
	int const anchor,
	int const nres_monomer,
	float & dist,
	float & radius,
	float & angle
)
{

	 FArray3D_float const & Epos( pose.Eposition() );
	 numeric::xyzVector_float v1(&Epos(1,2,anchor)), v2(&Epos(1,2,anchor + nres_monomer)),
        v3(&Epos(1,2,anchor+nres_monomer*2)),v4(&Epos(1,2,anchor+nres_monomer*3)), ori1, ori2, l1, l2;

	ori1 = ( v2 + v1 )/2;
	ori2 = ( v3 + v4 )/2;
	l1 = v2 - v1;
	l2 = ori2 - ori1;
	radius = l1.length()/2;
	dist = l2.length();
	v1= v1 - ori1;
	v3= v3 - l2 -ori1;
  angle =180.0/numeric::constants::f::pi*acos( dot_product(v1,v3)/(v1.length()*v3.length()) );
	std::cout << "d2 distance, radius and angle: " << dist << " " << radius << " " << angle << std::endl;

}
//////////////////////////////////////////////////////////////////////////////
void
pose_docking_hinge_protocol_symm(
	pose_ns::Pose & oligomer_pose,
	int const nres_monomer
)
{
	using namespace pose_ns;
	using namespace docking;

	int hinge_start = intafteroption( "hinge_start" );
	int hinge_end = intafteroption( "hinge_end" );
	bool const fix_rigid_body =  truefalseoption( "fix_rigid_body" );

	// Create pose objects
	Pose monomer_pose, pose;

	int const N ( docking::pose_symm_n_monomers );
  assert ( N );
  int anchor;
  std::string type ( symm_type );
  bool const fullatom ( get_docking_fullatom_flag() );
  numeric::xyzVector_float z_axis, origin;

	  // Create monomer pose from oligomer_pose
  create_monomer_pose(oligomer_pose, monomer_pose, nres_monomer);


	// if fragment insertion, do it on monomer
  if (truefalseoption("frag_insert_hinge") ) {
		static bool init = {false};
		if (!init) {
			read_fragments( monomer_pose.total_residue() );
			init =true;
		}
		// initialize torsions
		insert_init_frag( monomer_pose, hinge_start, hinge_end-hinge_start );
		monomer_pose.set_allow_bb_move( false );
		for ( int i=hinge_start; i<= hinge_end; ++i ) monomer_pose.set_allow_bb_move( i,true);
		// fold!
		monomer_pose.set_fullatom_flag(false);
		const float ncyc_frag_hinge = realafteroption("ncyc_frag_hinge", 0.2);
		fold_abinitio( monomer_pose, false /* no chainbreak scoring */,false, ncyc_frag_hinge );
		monomer_pose.set_fullatom_flag(fullatom);
	}
	// Setup coordinate system  and create a new symmetric pose
  pose_docking_symm_coordsys_setup(monomer_pose, oligomer_pose, anchor, z_axis, origin, 50);

	// Create symmetrical pose object
  create_symmetric_pose( monomer_pose, fullatom, N, anchor, origin, z_axis, type, pose );

	Score_weight_map weight_map;

	// For subsystems we score monomer 2
	int offset(0);
	if (docking::docking_pose_symm_subsystem) offset = monomer_pose.total_residue();
	hinge_start += offset;
	hinge_end   += offset;

	// set allow_move
	pose.set_allow_jump_move( false );
	int const num_jump_mon ( number_jump_monomer(pose) );
	pose.set_allow_jump_move( num_jump_mon + 1, ! fix_rigid_body );
	pose.set_allow_bb_move( false );
	for (int i=hinge_start; i<=hinge_end; ++i) {
		pose.set_allow_bb_move( i, true);
		pose.set_secstruct(i,'L');
	}

	int const nres (pose.total_residue());
	pose.set_allow_chi_move( false );
	Symmetry_info const & s ( pose.symmetry_info() );
	for( int i = 1; i <= nres; ++i ) {
    if ( s.chi_independent( i ) ) pose.set_allow_chi_move(i,true);
  }

	Pose pose_save;
	pose_save = pose;

	pose.set_fullatom_flag( false );
	monomer_pose.set_fullatom_flag( false );

	if ( ! get_docking_local_refine() ) {
		// perturb
		float helix_size, strand_size, other_size;
		real3afteroption("hinge_pert",0.0, helix_size, 25.0, strand_size,
			30.0, other_size);
		//pose_docking_perturb_hinge_region( pose, hinge_start, hinge_end,
		//	helix_size, strand_size, other_size );
		set_smallmove_size( helix_size, strand_size, other_size );
		pose_small_moves( pose, hinge_end-hinge_start+1 );
		pose_docking_symm_perturb_rigid_body(pose);
		pose_docking_calc_rmsd( pose );
		docking_store_initial_rms();

		// centroid MC search
		for (int i=hinge_start; i<=hinge_end; ++i) {
    pose.set_allow_bb_move( i, true);
		}
		setup_score_weight_map( weight_map, score4d );
		set_smallmove_size(0.0, 15.0, 18.0 );
		int const cycles( 1000 );
		float const trans_mag( 0.7 ), rot_mag( 0.5 );
		pose_docking_hinge_move_trial_symm( pose, cycles, weight_map, trans_mag,
			rot_mag );
		pose_docking_calc_rmsd( pose );
		docking_store_cenmode_rms();
	}
	if (fullatom) {
		// fullatom mode
		pose.set_fullatom_flag( true, false );
		// copy back the starting sidechain first
		pose.recover_sidechain( pose_save );
		// fullatom scorefxn
		setup_score_weight_map( weight_map, score10d );
		// repack
		FArray1D_bool allow_repack( pose.total_residue(), false );
		bool const include_current = true;
		pose_docking_symm_calc_interface( pose, allow_repack );
		identify_inter_hinge_residues( pose, allow_repack, hinge_start, hinge_end, nres_monomer );
		pose.repack( allow_repack, include_current );
		pose.score( weight_map );

		if ( truefalseoption("use_score10d_min") ) {
			setup_score_weight_map( weight_map, score10d_min );
		}
		pose_docking_hinge_mcm_trial_symm( pose, weight_map );
	}

	// Copy symmetric pose to a new pose without pseudo residues
  create_output_pose(oligomer_pose, pose);
  oligomer_pose.score( weight_map );

}


///////////////////////////////////////////////////////////////////////////////
///////////////////
// Very similar to pose_docking_symm_calc_interface
// Objective: To identify residues at the interface between two subdomains that are
// connected via a hinge. These interface residues as well as residues that interact with
// the hinge should be allowed to repack.
void
identify_inter_hinge_residues(
	pose_ns::Pose & pose,
  FArray1DB_bool & int_res,
  int hinge_start,
  int hinge_end,
  int nres_monomer
)
{
	using namespace pose_ns;
  using namespace design_sym;

	Symmetry_info const & s ( pose.symmetry_info() );

  pose_update_cendist( pose );
  FArray2D_float const & cendist ( pose.get_2D_score( CENDIST ) );
	int offset(0);
	if ( docking::docking_pose_symm_subsystem) offset = nres_monomer;

  for ( int i = offset+1; i <= offset+hinge_end; ++i ) {
    for ( int j = offset+hinge_start; j <= offset+nres_monomer; ++j ) {
      if ( cendist(i,j) <= 64 ) {
        if (!( s.chi_independent( i ) && s.chi_independent( j )) )
          std::cout << "WARNING: Changing the status of repacking in a non-controling monomer. " << i << " " << j << std::endl;
	        else int_res( i ) = true;
        }
	    }
	  }
	  return;
	}

//////////////////////////////////////////////////////////////////////////////
void
pose_docking_hinge_mcm_trial_symm(
	pose_ns::Pose & pose,
	const pose_ns::Score_weight_map & weight_map
)
{
	using namespace pose_ns;

	bool const separate = truefalseoption( "minimize_separately" );
	// params
//	minimize_set_func( 2 );
	minimize_set_tolerance( 0.05 );
	set_smallmove_size(0.0, 5.0, 6.0 );
	float const trans_mag( 0.2 );
	float const rot_mag( 0.5 );
	int const outer_cycles = intafteroption("outer_cycle_mcm_hinge", min(10, pose.get_total_insert() ) );
  int const inner_cycles = intafteroption("inner_cycle_mcm_hinge",10);
	float const init_temp = { 1.5 };
	float const last_temp = { 0.5 };
	float const gamma =
		std::pow( last_temp/init_temp, 1.0f/(outer_cycles*inner_cycles) );
	int n_trial(0), small_accepted(0), shear_accepted(0), jump_accepted(0);
	float temperature( init_temp );
	// mc obj
	Monte_carlo mc( pose, weight_map, init_temp );

	int const num_jump_mon ( number_jump_monomer(pose) );

	// should nb_list be used?
		bool nb_list = {true};
		static bool init = {false};
		if (!init) {
	if (truefalseoption("disable_nb_list_hinge") ) {
			nb_list = false;
			init =true;
		}
	}
		// Set allowed jumps and degrees of freedom
	if ( pose.symmetry_info().symm_type() == "helix" ) {
		 set_allow_dg_free_helix(pose);
		 int const N (pose.symmetry_info().N() );
		pose.set_allow_rb_move(6 ,N+1, false );
		 set_use_nblist( nb_list );
	} else if (pose.symmetry_info().symm_type() == "virus" ) {
		set_allow_dg_free_virus(pose);
		// Don't minimize the rotation of threefold axis.
		// minimimizing this df does not work well need to find out why...
		int const N ( pose.symmetry_info().N() );
		pose.set_allow_jump_move( N + 1, false );
		set_use_nblist( nb_list );

	} else {
		set_allow_dg_free_cn(pose);
		set_use_nblist( nb_list );
	}


	for ( int i = 1; i <= outer_cycles; ++i ) {
		int nmoves = static_cast<int>(outer_cycles/i);
		for ( int j = 1; j <= inner_cycles; ++j ) {
			temperature *= gamma;
			mc.set_temperature( temperature );
			n_trial++;;
			// small min trial
			pose_small_moves( pose, nmoves );
			if ( separate ) {
				pose_minimize_torsion( pose, weight_map, "dfpmin_atol" );
			} else {
				pose.main_minimize( weight_map, "dfpmin_atol" );
			}
			if( mc.boltzmann( pose ) ) { small_accepted++; }
			// shear min trial
			pose_shear_moves( pose, nmoves );
			if ( separate ) {
				pose_minimize_torsion( pose, weight_map, "dfpmin_atol" );
			} else {
				pose.main_minimize( weight_map, "dfpmin_atol" );
			}
			if( mc.boltzmann( pose ) ) { shear_accepted++; }
			// jump min trial
			if ( pose.get_allow_jump_move( num_jump_mon + 1 ) ) {
			//	pose_docking_set_rb_center( pose );
				symm_move( pose, trans_mag, rot_mag );
				if ( separate ) {
					pose_docking_symm_minimize_trial( pose, "dfpmin_atol",
						weight_map, 0.02 );
				} else {
					pose.main_minimize( weight_map, "dfpmin_atol" );
				}
				if ( mc.boltzmann( pose ) ) { jump_accepted++; }
			}
		} // inner cycle
		// repack every outer_cycle
		FArray1D_bool allow_repack( pose.total_residue(), false );
		pose_docking_symm_calc_interface( pose, allow_repack );
		pose.repack( allow_repack );
		pose.score( weight_map );
		mc.boltzmann( pose );

		std::cout << "Cycle: " << i << " -- ";
		mc.show_scores();
		pose = mc.low_pose();
	} // outer cycle

	std::cout << "pose_docking_hinge_protocol -- trial/accepted/ratio:\n"
						<< small_accepted << "/" << n_trial << "/"
						<< float(small_accepted)/float(n_trial) << " for small moves\n"
						<< shear_accepted << "/" << n_trial << "/"
						<< float(shear_accepted)/float(n_trial) << " for shear moves\n"
						<< jump_accepted  << "/" << n_trial << "/"
						<< float(jump_accepted) /float(n_trial) << " for jump  moves\n"
						<< std::endl;
}
//////////////////////////////////////////////////////////////////////////////
float
pose_docking_hinge_move_trial_symm(
	pose_ns::Pose & pose,
	int const cycles,
	const pose_ns::Score_weight_map & wt_map,
	float trans_mag,
	float rot_mag
)
{
	using namespace pose_ns;

	float const temperature( 0.8 );
	int const nmoves = static_cast<int> ( pose.get_total_insert() * 0.5 );
	Monte_carlo mc( pose, wt_map, temperature );
	int n_accepted = 0;

	int const num_jump_mon ( number_jump_monomer(pose) );
	for( int i = 1; i <= cycles; ++i ) {
		if ( ran3()>=0.5 || ! pose.get_allow_jump_move( num_jump_mon + 1 ) ) {
			pose_small_moves( pose, nmoves );
		}	else {
			symm_move( pose, trans_mag, rot_mag );
		}
		pose.score( wt_map );
		if( mc.boltzmann( pose ) ) ++n_accepted;
	}
	pose = mc.low_pose();

	std::cout << "pose_docking_hinge_move_trial_symm -- "
						<< n_accepted << " out of " << cycles << " accepted\n";

	return n_accepted / cycles;
}

void
setup_c1_symm( pose_ns::Pose & start_pose,
							 pose_ns::Pose & pose,
							 int const anchor_point /* = 1 */ )
{
	int const num_anchor = 1;
	FArray1D_int anchor(1, anchor_point );
	setup_c1_symm( start_pose, pose, anchor, num_anchor);
}

void
setup_c1_symm( pose_ns::Pose & start_pose,
							 pose_ns::Pose & pose,
							 FArray1D_int & anchor,
							 int const num_anchor )
{
	using namespace pose_ns;

	Pose pseudo_pose;

	float const length(  1.5 );
	float const z(-10.0);
	//Num_anchor should suffice, but
	// there's some fold_tree error if nres = 1.
  pseudo_pose.simple_fold_tree( num_anchor+1 );

  // Set coordinates for pseudo glycines.
	FArray3D_float Epos(3, param::MAX_POS , num_anchor+1 );
	for (int i=1; i<=num_anchor+1; ++i) {
		Epos(1,1,i) = length;
  	Epos(2,1,i) = 0.0;
  	Epos(3,1,i) = z;

		Epos(1,2,i) = 0.0;
  	Epos(2,2,i) = 0.0;
  	Epos(3,2,i) = z;

  	Epos(1,3,i) = 0.0;
  	Epos(2,3,i) = 0.0;
  	Epos(3,3,i) = z + length;

  	Epos(1,4,i) = 0.0;
  	Epos(2,4,i) = length;
  	Epos(3,4,i) = z;

  	Epos(1,5,i) = length;
  	Epos(2,5,i) = length;
  	Epos(3,5,i) = z;

		pseudo_pose.set_res( i, param_aa::aa_gly );
  	pseudo_pose.set_res_variant( i, 1 );
	}
  pseudo_pose.set_coords( false, Epos, FArray3D_float() );
  pseudo_pose.set_fullatom_flag( true, false ); // only glycine anyway


	Fold_tree f;
	int const total_residue = start_pose.total_residue();
  int const nres( total_residue + num_anchor );
	int const num_jump_start( start_pose.num_jump() );
  int const njump( num_jump_start + num_anchor );

	FArray2D_int const & jump_point_start
    ( start_pose.fold_tree().get_jump_point() );
  int num_fold_tree_cut_points;
  FArray1D_int const & cuts_start
    ( start_pose.fold_tree().get_fold_tree_cutpoint(num_fold_tree_cut_points) );

  FArray2D_int jump_point( 2, njump );
  FArray1D_int cuts( njump );

  int jump_number(0);
  for ( int j=1; j<= num_jump_start; ++j ) {
      jump_point( 1, ++jump_number ) = jump_point_start( 1, j );
      jump_point( 2,   jump_number ) = jump_point_start( 2, j );
      cuts( jump_number ) = cuts_start( j );
  }

  // Jumps between pseudo residues and corresponding monomer
	// 	cuts(jump_number+1) = total_residue;
	for ( int i=1; i<= num_anchor; ++i) {
		++jump_number;
		cuts(jump_number) = total_residue+i-1;
		jump_point(1,jump_number) = anchor(i);
		jump_point(2,jump_number) = total_residue+i;
	}

	f.tree_from_jumps_and_cuts( nres, njump, jump_point, cuts );

	// reorder the tree so that it is rooted at 1st pseudo-rsd
	f.reorder( nres);

	pose.set_fold_tree( f );
	// put into pose
	pose.set_fullatom_flag(true,false);
	pose.copy_segment(total_residue, start_pose, 1,1);
	pose.copy_segment(num_anchor,pseudo_pose,total_residue+1,1);

	param::MAX_CLONES() = 1;
	design_sym::num_clones = 0;

	// Setup symm_info
	pose.setup_symm_info(pose.total_residue(), pose.num_jump(), num_anchor, "c1");


	for ( int i=1; i<= num_anchor; ++i) {
		pose.set_allow_bb_move( total_residue+i, false );
	}

	pose.set_fullatom_flag( start_pose.fullatom() );
	if (start_pose.native_pose_exists() ){
		pose.set_native_pose( start_pose.native_pose() );
	}

}

void
c1_symm_test()
{
	using namespace pose_ns;
	Pose start_pose, pose;
	pose_from_pdb( start_pose, stringafteroption( "s" ), true, false);

	start_pose.score( score12 );
	std::cout << start_pose.show_scores() << std::endl;
  int const total_residue = start_pose.total_residue();

	int const num_anchor = 3;
	FArray1D_int anchor(3);
	anchor(1) = 20;
	anchor(2) =30;
	anchor(3) = 40;

	param::MAX_RES_assign_res( total_residue+num_anchor );

	setup_c1_symm( start_pose, pose, anchor, num_anchor );

	pose.score(score12);
	std::cout << pose.show_scores() << std::endl;
	pose.dump_pdb("pose.pdb");

	Score_weight_map weight_map;
	setup_score_weight_map(weight_map, score12);
	pose.score(weight_map);
	pose.set_allow_bb_move(true);
	pose.set_allow_jump_move(false);
	std::cout << pose.show_scores() << std::endl;
	std::string type ("dfpmin_atol");
	pose.main_minimize( weight_map, type );
	std::cout << pose.show_scores() << std::endl;
	pose.dump_pdb("after_relax.pdb");
	start_pose.main_minimize( weight_map, type );
	std::cout << start_pose.show_scores() << std::endl;
}
//vats May 15th 07//////////////////////////////////////////////////////////////////////////
void
initialize_looprlx_symm_pose(
	pose_ns::Pose & monomer_pose,
	bool const fullatom,
	pose_ns::Loops & one_loop,
	pose_ns::Pose & pose // output pose
	)
{
	using namespace pose_ns;
	using namespace docking;
	assert( docking::docking_local_refine );
	numeric::xyzVector_float origin( 0 ), z_axis( 0,0,1 );
	int N( docking::pose_symm_n_monomers );
	std::string type( docking::symm_type );
	int nres_monomer( monomer_pose.total_residue() );
	int anchor( monomer_pose.residue_center_of_mass(1, nres_monomer) );
	pose_docking_symm_monomer_loop_tree( monomer_pose, anchor, one_loop );
	create_symmetric_pose( monomer_pose, fullatom, N, anchor, origin, z_axis, type, pose );

}
///////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
//vats  SYMM LOOP RELAX
/////////////////////////////////////////////////////////////////////////////////

void
pose_symm_loop_relax(
	 pose_ns::Pose & oligomer_pose,
   int const nres_monomer
)
{

  using namespace docking;
  using namespace pose_ns;

  std::cout << " pose_docking_symm_loop_relax protocol excecuted " << "\n";

  pose_ns::Pose current_pose, save_monomer_fullatom_pose, monomer_pose, stage_oligomer_pose;
  pose_ns::Score_weight_map weight_map;

  int const N ( docking::pose_symm_n_monomers );
  assert ( N );
  int anchor;
  std::string type ( symm_type );
	//  bool const fullatom ( get_docking_fullatom_flag() );
  numeric::xyzVector_float z_axis, origin;
	Loops input_loops, output_loops;

  // Create monomer pose from oligomer_pose
  create_monomer_pose(oligomer_pose, current_pose, nres_monomer);
	save_monomer_fullatom_pose = current_pose;

	//Read loops file
	std::vector<int> ori_loops_begin, ori_loops_end;
	bool found_loops = read_loop_file( ori_loops_begin, ori_loops_end );
	if ( !found_loops ){
		std::cout << "Loops file not found, enter a valid loops file\n" << std::endl;
		std::exit( 0 );
	}

	verify_loop_file( current_pose, ori_loops_begin, ori_loops_end );

  // Setup coordinate system  and create a new symmetric pose
  pose_docking_symm_coordsys_setup(current_pose, oligomer_pose, anchor, z_axis, origin, 50);

	bool const full_atom( get_fullatom_loop_flag() ); //default is false

	int const loop_build_round( 1 ); // build full loops this many times
	int const n_loop_fail_tol( 1 );
	int loop_nfail( 0 );
	bool loop_done( false );

	static bool init = {false};
	if (!init) {
		read_fragments( current_pose.total_residue() );
		init =true;
	}


	while( !loop_done && loop_nfail <= n_loop_fail_tol ){
		loop_nfail++;
		output_loops.clear();
		for ( int i = 0; i < loop_build_round; ++i ){
			bool obligate( false );
			loop_done = build_random_loops( current_pose, ori_loops_begin, ori_loops_end, output_loops, full_atom, obligate );
			}
	}

	if ( save_monomer_fullatom_pose.total_residue() == current_pose.total_residue() ) {
		current_pose.set_fullatom_flag( true, false );
		current_pose.recover_sidechain( save_monomer_fullatom_pose );
		//		repack_loops( output_loops, current_pose );
	}

	create_symmetric_pose( current_pose, full_atom, N, anchor, origin, z_axis, type, stage_oligomer_pose );
	create_output_pose( oligomer_pose, stage_oligomer_pose );
	oligomer_pose.score( weight_map );


}
///////////////////////////////////////////////////////////////////////
void
copy_loops_file( std::vector < int > ori_loops_begin,
								 std::vector < int > ori_loops_end,
								 pose_ns::Loops & input_loops )
{
	int begin, end, cut;
	int const offset = 0;
	assert( ori_loops_begin.size() == ori_loops_end.size() );

	for ( int i = 0; i < int( ori_loops_begin.size() ); ++i ){
		begin = int( ori_loops_begin[i] );
		end = int( ori_loops_end[i] );
		cut = (begin-1) + int( ran3()*(end-(begin-1)+1) );
		input_loops.add_loop( begin, end, cut, offset, false );
	}
}
