// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14061 $
//  $Date: 2007-04-08 17:57:17 -0700 (Sun, 08 Apr 2007) $
//  $Author: rhiju $
#ifdef GL_GRAPHICS
#include "gl_graphics.h"
#define PROTEIN_GRAPHICS
#endif
#ifdef BOINC_GRAPHICS
#define PROTEIN_GRAPHICS
#endif
#ifdef PROTEIN_GRAPHICS
#include "protein_graphics.h"

#include "aaproperties_pack.h"
#include "enzyme.h" //needed for ligand aa stuff?
#include "etable.h"
#include "initialize.h"
#include "misc.h"
#include "namespace_low_pose.h"
#include "namespace_trajectory.h"
#include "param_aa.h"
#include "RotamerSet.h"
#include "read_aaproperties.h"

#ifdef MAC
#include <GLUT/glut.h>
#elif WIN32
#include "glut/glut.h"
#else
#include "GL/glut.h"
#endif

#include <numeric/xyzVector.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/string.functions.hh>

using pose_ns::Pose;
using numeric::xyzVector_float;

namespace protein_graphics {

  namespace privates {
    //rhiju parameters for cartoons
    const int NUM_SEGMENTS = 5;
    const float helix_half_width = 1.5;
    const float strand_half_width = 1.0;
    const float coilradius = 0.2;

		//lin parameters for spacefill
		float const ligand_sphere_opacity( 1.0 );
		float const protein_sphere_opacity( 1.0 );
		float const ligand_sphere_shininess( 1.0);
		float const protein_sphere_shininess( 1.0 );

		//lin parameters for ball and stick
		float const protein_wireframeScale( 0.1 );
		float const protein_stickScale( 0.2 );
		float const ligand_stickScale( 0.3 );
		float const protein_sphereScale( 0.2 );
		float const ligand_sphereScale( 0.3 );

 		//Lin parameters for cylinder

		// display list to draw spheres more quickly
		unsigned int sphereDisplayList = 0;

		//lin allow to show sidechain
		//		FArray1D_bool show_sc( param::MAX_RES()(), true );

		//lin allow to show packer
		bool show_repack = { false };
		int max_show_repack_fps = { 400 };

		bool text_aliasing = { true };
	}
	using namespace privates;


  class PdbInfoStandin {
  public:

    PdbInfoStandin(const FArray2DB_float & occupancy):
      m_occupancy(occupancy)
    {}

    inline const FArray2DB_float & occupancy() const { return m_occupancy; }

  private:

    const FArray2DB_float & m_occupancy;

  };

  class PoseStandin {
  public:

    PoseStandin(int total_residue,
  		bool fullatom,
	  	const FArray3DB_float & full_coord,
	  	const FArray3DB_float & Eposition,
		  const FArray1DB_int & res,
		  const FArray1DB_int & res_variant,
		  const FArray1DB_char & secstruct,
		  const FArray2DB_float & occupancy):
      m_total_residue(total_residue),
      m_fullatom(fullatom),
      m_full_coord(full_coord),
      m_Eposition(Eposition),
      m_res(res),
      m_res_variant(res_variant),
      m_secstruct(secstruct),
      m_pdb_info(occupancy)
    {}

    inline int total_residue() const { return m_total_residue; }
    inline bool fullatom() const { return m_fullatom; }
    inline const FArray3DB_float & full_coord() const { return m_full_coord; }
    inline const FArray3DB_float & Eposition() const { return m_Eposition; }
    inline const FArray1DB_int & res() const { return m_res; }
    inline const FArray1DB_int & res_variant() const { return m_res_variant; }
    inline const FArray1DB_char & secstruct() const { return m_secstruct; }
    inline const PdbInfoStandin & pdb_info() const { return m_pdb_info; }

  private:

    int m_total_residue;
    bool m_fullatom;
    const FArray3DB_float & m_full_coord;
    const FArray3DB_float & m_Eposition;
    const FArray1DB_int & m_res;
    const FArray1DB_int & m_res_variant;
    const FArray1DB_char & m_secstruct;
    const PdbInfoStandin m_pdb_info;
  };

  inline void glVertex3fxyz( const xyzVector_float & coord ) {
    glVertex3f(coord.x(), coord.y(), coord.z() );
  }

  inline void glColor3fxyz( const xyzVector_float & color ) {
    glColor3f(color.x(), color.y(), color.z());
  }

  void rainbow_color( float frac , float & red, float & green, float & blue , bool mute_color ) {

    float muting = .7;
    float my_color = frac;
    //	if (my_color > .5) {
    //		red = (2.0*my_color)-1;
    //		blue = 0.0;
    //	}
    //	if (my_color < .5) {
    //		blue = (1.0-2.0*my_color);
    //		red = 0.0;
    //	}
    //	if ((my_color > .25) && (my_color < .75)) {
    //		//		green = (2.0*my_color)-.5;
    //		green = (2.0*my_color);
    //	}
    //	else {
    //		green = 0.0;
    //	}

    red = my_color;
    blue = 1.0 - my_color ;
    if (my_color < .5) {
      green = 2.0*my_color;
    }
    else {
      green = 2.0-2.0*my_color;
    }

    if (mute_color) {
      float saturation = sqrt(red*red + green*green + blue*blue);
      red = muting*red/saturation;
      green = muting*green/saturation;
      blue = muting*blue/saturation;
    }
  }

  void get_residue_color(float i, float & red, float & green, float & blue, bool mute_color, int total_residue ) {
    float i_local = i;
    if (i > total_residue) i_local = total_residue;
    // if ( docking::docking_flag ) {
    // 	chain_color( static_cast<int>(i_local), red, green, blue );
    // } else {
    rainbow_color ( float(i_local) / float(total_residue), red, green, blue, mute_color);
    // }
  }

  std::map<int, xyzVector_float>  get_sidechain_color_rhiju() {
    using namespace param_aa;

    std::map<int, xyzVector_float> sidechain_color_rhiju;

    sidechain_color_rhiju[ aa_ala ] = xyzVector_float( 0.3, 0.3, 0.3); //gray
    sidechain_color_rhiju[ aa_cys ] = xyzVector_float( 0.7, 0.7, 0.0); //yellow
    sidechain_color_rhiju[ aa_asp ] = xyzVector_float( 0.7, 0.0, 0.0); //red
    sidechain_color_rhiju[ aa_glu ] = xyzVector_float( 0.7, 0.0, 0.0); //red
    sidechain_color_rhiju[ aa_phe ] = xyzVector_float( 0.3, 0.3, 0.3);
    sidechain_color_rhiju[ aa_gly ] = xyzVector_float( 0.7, 0.5, 0.0); //orange; this shouldn't happen from sidechain.
    sidechain_color_rhiju[ aa_his ] = xyzVector_float( 0.0, 0.0, 0.7); //blue
    sidechain_color_rhiju[ aa_ile ] = xyzVector_float( 0.3, 0.3, 0.3);
    sidechain_color_rhiju[ aa_lys ] = xyzVector_float( 0.0, 0.0, 0.7); //blue
    sidechain_color_rhiju[ aa_leu ] = xyzVector_float( 0.3, 0.3, 0.3);
    sidechain_color_rhiju[ aa_met ] = xyzVector_float( 0.3, 0.3, 0.3);
    sidechain_color_rhiju[ aa_asn ] = xyzVector_float( 0.0, 0.5, 0.0); //green
    sidechain_color_rhiju[ aa_pro ] = xyzVector_float( 0.3, 0.3, 0.3);
    sidechain_color_rhiju[ aa_gln ] = xyzVector_float( 0.0, 0.5, 0.0); //green
    sidechain_color_rhiju[ aa_arg ] = xyzVector_float( 0.0, 0.0, 0.7); //blue
    sidechain_color_rhiju[ aa_ser ] = xyzVector_float( 0.0, 0.5, 0.0); //green
    sidechain_color_rhiju[ aa_thr ] = xyzVector_float( 0.0, 0.5, 0.0); //green
    sidechain_color_rhiju[ aa_val ] = xyzVector_float( 0.3, 0.3, 0.3);
    sidechain_color_rhiju[ aa_trp ] = xyzVector_float( 0.3, 0.3, 0.3);
    sidechain_color_rhiju[ aa_tyr ] = xyzVector_float( 0.0, 0.5, 0.0); //green
    sidechain_color_rhiju[ aa_sep ] = xyzVector_float( 0.5, 0.5, 0.0); //orange
    sidechain_color_rhiju[ na_gua ] = xyzVector_float( 0.0, 0.0, 0.5); //blue
    sidechain_color_rhiju[ na_ade ] = xyzVector_float( 0.5, 0.5, 0.0); //yellow
    sidechain_color_rhiju[ na_cyt ] = xyzVector_float( 0.0, 0.5, 0.0); //green
    sidechain_color_rhiju[ na_thy ] = xyzVector_float( 0.5, 0.0, 0.0); //red

    sidechain_color_rhiju[ na_rgu ] = xyzVector_float( 0.0, 0.0, 0.5); //blue
    sidechain_color_rhiju[ na_rad ] = xyzVector_float( 0.5, 0.5, 0.0); //yellow
    sidechain_color_rhiju[ na_rcy ] = xyzVector_float( 0.0, 0.5, 0.0); //green
    sidechain_color_rhiju[ na_ura ] = xyzVector_float( 0.5, 0.0, 0.0); //red

		// This map might be initialized before all the ligand aa stuff is setup ... ah, threading!
		//	sidechain_color_rhiju[ ligand_aa_vector.at(1) ] = xyzVector_float( 1.0, 0.5, 0.0); //orange
		//	sidechain_color_rhiju[ ligand_aa_vector.at(2) ] = xyzVector_float( 0.5, 0.5, 0.0); //yellow

    return sidechain_color_rhiju;
  }

	///////////////////////////////////////////////////////////////////////////////
	numeric::xyzVector_float get_cpk_color( int type )
	{
		using namespace numeric;
		if ( type >= 7 && type <= 12 || type == 17 ) { // blue
			return xyzVector_float( 0.0, 0.0, 1.0);
		} else if ( type >= 13 && type <= 15 || type == 20 ) { // red
			return xyzVector_float( 1.0, 0.0, 0.0);
		} else if ( type == 16 ) { // yellow
			return xyzVector_float( 1.0, 1.0, 0.0);
		} else if ( type == 21 ) { // orange
			return xyzVector_float( 1.0, 0.5, 0.0);
		} else { //gray
			return xyzVector_float( 0.5, 0.5, 0.5 );
		}
		return  xyzVector_float( 0.0, 0.0, 0.0 );
	}

	///////////////////////////////////////////////////////////////////////////////
	numeric::xyzVector_float
	get_atom_color(
								 GraphicsState & gs,
								 int const atomno,
								 int const aa,
								 int const aav,
								 int const resnum = 0,
								 int const nres = 0
								 )
	{
		using namespace param_aa;
		using namespace numeric;

		float red,green,blue;

		static std::map<int, xyzVector_float> sidechain_color_rhiju = get_sidechain_color_rhiju();
		int type (0);

		switch ( gs.Color_mode ) {

		case CPK_COLOR:
			type = aaproperties_pack::fullatom_type(atomno,aa,aav);
			return get_cpk_color( type );

		case RAINBOW_COLOR:
			if ( (is_DNA(aa)  && atomno < 11) || (is_RNA(aa) && atomno < 12) ){
				rainbow_color( float(resnum)/ float(nres), red, green, blue, false /*mute_color*/);
				return xyzVector_float(red, green, blue);
			}
			rainbow_color( float(resnum)/ float(nres), red, green, blue, true /*mute_color*/);
			return xyzVector_float(red, green, blue);

		case RESIDUE_COLOR:
			if (sidechain_color_rhiju.count( aa ) )	return sidechain_color_rhiju[aa];
			return xyzVector_float( 1.0, 0.5, 0.0); //orange

		case RAINBOW_CPK_COLOR:
			type = aaproperties_pack::fullatom_type(atomno,aa,aav);
			if ( type >= 7 ) { //non carbone atoms
				return get_cpk_color( type );
			}
			if ( (is_DNA(aa)  && atomno < 11) || (is_RNA(aa) && atomno < 12) ){
				rainbow_color( float(resnum)/ float(nres), red, green, blue, false /*mute_color*/);
				return xyzVector_float(red, green, blue);
			}
			rainbow_color( float(resnum)/ float(nres), red, green, blue, true /*mute_color*/);
			return xyzVector_float(red, green, blue);

		case RESIDUE_CPK_COLOR:
			type = aaproperties_pack::fullatom_type(atomno,aa,aav);
			if ( type >= 7 ) { //non carbone atoms
				return get_cpk_color( type );
			}
			return sidechain_color_rhiju[aa];
		}
		return xyzVector_float( 1.0, 1.0, 1.0);
	}

  void draw_Calpha_trace( int start, int end, const PoseStandin & pose, float xwidth = 0.5) {

    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();
    const FArray2DB_float & occ = pose.pdb_info().occupancy();

    GLfloat currentrotation[16];
    glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
    xyzVector_float z(currentrotation[2],currentrotation[6],currentrotation[10]); // z pointing out of window in current view.
    const float z_offset = .1;
    const xyzVector_float z_halo = z * z_offset;

    xyzVector_float prev1( 0.0 ), prev2( 0.0 );
    bool last_bonded = false;
    for ( int i = start; i<end ; i++ ) {

      if(occ(2,i) <= 0.01) continue;
      xyzVector_float ca_pos1, ca_pos2;
      xyzVector_float ca_pos1tmp( &xyz_full(1,2,i) );
      xyzVector_float ca_pos2tmp( &xyz_full(1,2,i+1) );
      ca_pos1 = ca_pos1tmp;
      ca_pos2 = ca_pos2tmp;

      xyzVector_float bond;
      bond = ca_pos2 - ca_pos1;
      xyzVector_float width( cross( bond, z ));
      if ( width.length_squared() > 0.0001 )  width.normalize();
      width = width * xwidth;

      float red, green, blue;
      get_residue_color( i, red, green, blue, false, pose.total_residue() );
      if ( i > 1 && last_bonded) {
				glColor3f(red, green, blue);
				glBegin(GL_POLYGON);
				glVertex3fxyz ( prev1 );
				glVertex3fxyz ( ca_pos1 + width );
				glVertex3fxyz ( ca_pos1 - width );
				glVertex3fxyz ( prev2 );
				glEnd();

				//outline("halo")
				glColor3f(0,0,0);
				glBegin(GL_LINES);
				glVertex3fxyz ( prev1 );
				glVertex3fxyz ( ca_pos1 + width );
				glVertex3fxyz ( ca_pos1 - width );
				glVertex3fxyz ( prev2 );
				glEnd();
      }
      last_bonded = false;
      if (bond.length_squared() < 16) {
				last_bonded = true;
				//	float atom_size = 20;
				glColor3f(red, green, blue);
				glBegin(GL_POLYGON);
				glVertex3fxyz ( ca_pos1 + width );
				glVertex3fxyz ( ca_pos2 + width );
				glVertex3fxyz ( ca_pos2 - width );
				glVertex3fxyz ( ca_pos1 - width );
				glEnd();

				//outline("halo")
				glColor3f(0,0,0);
				glBegin(GL_LINES);
				glVertex3fxyz ( z_halo + ca_pos1 + width );
				glVertex3fxyz ( z_halo + ca_pos2 + width );
				glVertex3fxyz ( z_halo + ca_pos2 - width );
				glVertex3fxyz ( z_halo + ca_pos1 - width );
				glEnd();
      }
      prev1 = ca_pos2 + width;
      prev2 = ca_pos2 - width;
    }

  }

  void set_initial_polygon_vertices(xyzVector_float vec1, xyzVector_float vec2, GraphicsState & gs) {
    gs.previous_vertex1 = vec1;
    gs.previous_vertex2 = vec2;
    gs.previous_width_vector = 0.0;
  }

  void draw_next_polygon( xyzVector_float vec1, xyzVector_float vec2,
													float red, float green, float blue, int aa,
													GraphicsState & gs,
													bool darken_inside = false) {
    if (aa<1) return;

    //Give the cartoon some thickness?
    const bool show_thickness = true;
    xyzVector_float width_vector;

    float SHOWBONDCUTOFF2 = (5.0*5.0)/(privates::NUM_SEGMENTS*privates::NUM_SEGMENTS);
    if (param_aa::is_NA(aa)) SHOWBONDCUTOFF2 = (8.0*8.0)/(privates::NUM_SEGMENTS*privates::NUM_SEGMENTS);

    const float bond_length2 = ((vec1 + vec2)/2.0f - (gs.previous_vertex1 + gs.previous_vertex2)/2.0f).length_squared();
    if (!show_thickness) {
      glColor3f(red,green,blue);
      glBegin(GL_POLYGON);
      glVertex3fxyz ( gs.previous_vertex1 );
      glVertex3fxyz ( gs.previous_vertex2 );
      glVertex3fxyz ( vec2 );
      glVertex3fxyz ( vec1 );
      glVertex3fxyz ( gs.previous_vertex1 );
      glEnd();
    } else {
      width_vector = cross( (vec1 + vec2) - (gs.previous_vertex1 + gs.previous_vertex2),
														(vec1 + gs.previous_vertex1) - (vec2 + gs.previous_vertex2) );
      if (width_vector.length_squared() > 0.000001) width_vector.normalize();
      const float cartoon_width = 0.3f;
      width_vector *= -1.0f * cartoon_width;

      if (gs.previous_width_vector == 0.0f) gs.previous_width_vector = width_vector;

      if (bond_length2 < SHOWBONDCUTOFF2 ) {
				//outside
				glColor3f(red,green,blue);
				glBegin(GL_POLYGON);
				glVertex3fxyz ( gs.previous_vertex1 + gs.previous_width_vector );
				glVertex3fxyz ( gs.previous_vertex2 + gs.previous_width_vector );
				glVertex3fxyz ( vec2 + width_vector );
				glVertex3fxyz ( vec1 + width_vector );
				glEnd();

				//inside
				if (darken_inside) glColor3f(0.5*red,0.5*green,0.5*blue);
				glBegin(GL_POLYGON);
				glVertex3fxyz ( gs.previous_vertex1 - gs.previous_width_vector );
				glVertex3fxyz ( gs.previous_vertex2 - gs.previous_width_vector );
				glVertex3fxyz ( vec2 - width_vector );
				glVertex3fxyz ( vec1 - width_vector );
				glEnd();

				//edges
				glColor3f(0.7*red,0.7*green,0.7*blue);
				glBegin(GL_POLYGON);
				glVertex3fxyz ( gs.previous_vertex1 + gs.previous_width_vector );
				glVertex3fxyz ( vec1 + width_vector );
				glVertex3fxyz ( vec1 - width_vector );
				glVertex3fxyz ( gs.previous_vertex1 - gs.previous_width_vector );
				glEnd();
				glBegin(GL_POLYGON);
				glVertex3fxyz ( gs.previous_vertex2 + gs.previous_width_vector );
				glVertex3fxyz ( vec2 + width_vector );
				glVertex3fxyz ( vec2 - width_vector );
				glVertex3fxyz ( gs.previous_vertex2 - gs.previous_width_vector );
				glEnd();
      }

      // outline ("halo"):
      // z pointing out of window in current view.
      const bool show_halo = false;
      if (show_halo) {
				GLfloat currentrotation[16];
				glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
				xyzVector_float z(currentrotation[2],currentrotation[6],currentrotation[10]);
				const float z_offset = .01;
				const xyzVector_float z_halo = z * z_offset;

				glColor3f( 0.5*red, 0.5*green, 0.5*blue);
				glBegin(GL_LINES);
				glVertex3fxyz ( z_halo + gs.previous_vertex1 );
				glVertex3fxyz ( z_halo + vec1 );
				glEnd();
				glBegin(GL_LINES);
				glVertex3fxyz ( z_halo + gs.previous_vertex2 );
				glVertex3fxyz ( z_halo + vec2 );
				glEnd();
      }
    }

    gs.previous_vertex1 = vec1;
    gs.previous_vertex2 = vec2;
    gs.previous_width_vector = width_vector;
  }

  void get_direction( xyzVector_float & direction, int next_res, int prior_res, const FArray3DB_float & xyz_full,
											int const atomindex = 2 /*C_alpha*/ ) {
    xyzVector_float direction1( &xyz_full(1,atomindex,next_res) );
    xyzVector_float	direction2( &xyz_full(1,atomindex,prior_res) );
    direction = direction1 - direction2;
    if (direction.length_squared() > 0.00001) direction.normalize();
  }

	void get_normal( xyzVector_float & normal, int n, const FArray3DB_float & xyz_full,
									 int const atomindex = 2 /*C_alpha*/) {
    xyzVector_float vec1( &xyz_full(1,atomindex,n-1) );
    xyzVector_float	vec( &xyz_full(1,atomindex,n) );
    xyzVector_float	vec2( &xyz_full(1,atomindex,n+1) );

    normal = cross( (vec-vec1), (vec2-vec) );
    if (normal.length_squared() > 0.00001) normal.normalize();
  }

  void get_axis_and_tangent( xyzVector_float & axis, xyzVector_float & tangent, xyzVector_float & direction, xyzVector_float & normal) {
    //Magic linear combinations from Per Kraulis' molscript.
    const float HELIX_ALPHA = 0.5585;
    axis = std::cos(HELIX_ALPHA) * normal + std::sin(HELIX_ALPHA) * direction;

    const float HELIX_BETA = -0.1920;
    tangent = std::cos(HELIX_BETA) * direction + std::sin(HELIX_BETA) * normal;

    //	normal = cross(axis,tangent);
    //	normal.normalize();
  }

  xyzVector_float get_CA_segment( const xyzVector_float & prev_CA, const xyzVector_float & current_CA,
																	const xyzVector_float & prev_tangent, const xyzVector_float & tangent,
																	float p, float hermite_factor) {
    //Hermitean interpolation.
    const float p2 = p*p;
    const float p3 = p*p*p;

    const float h1 = 2*p3 - 3*p2 + 1;
    const float h2 =-2*p3 + 3*p2;
    const float h3 =   p3 - 2*p2 + p;
    const float h4 =   p3 -   p2;

    return (h1*prev_CA + h2*current_CA + h3*hermite_factor*prev_tangent + h4*hermite_factor*tangent);
    //return (1-p)*prev_CA + p*current_CA;
  }

  float get_half_width( const std::string & taper, float secstruct_half_width, float p ) {
    float half_width = secstruct_half_width;
    if (taper == "start"){
      half_width = privates::coilradius + (secstruct_half_width - privates::coilradius)*
				0.5 * ( - cos( 3.14159 * p )  + 1.0 );
    }
    if (taper == "end"){
      half_width = privates::coilradius + (secstruct_half_width - privates::coilradius)*
				0.5 * ( cos( 3.14159 * p )  + 1.0 );
    }
    //	if (taper == "strand_penultimate"){
    //		half_width = secstruct_half_width * (2 - p);
    //	}
    if (taper == "strand_ultimate"){
      half_width = privates::coilradius + (2*secstruct_half_width - privates::coilradius) * (1 - p);
    }

    return half_width;
  }

  void draw_secstruct_chunk( const xyzVector_float & prev_CA, const xyzVector_float & current_CA,
														 const xyzVector_float & prev_tangent, const xyzVector_float & tangent,
														 const xyzVector_float & prev_axis, const xyzVector_float & axis,
														 int n, char secstruct_res, const std::string & taper,
														 GraphicsState & gs, const PoseStandin & pose ) {

    const FArray1DB_int & res = pose.res();

    const float helix_hermite_factor = 4.7;
    const float strand_hermite_factor = 4.7;
    const float NA_hermite_factor = 7.7;

    float hermite_factor;
    float red( 0.0 ), green( 0.0 ), blue( 0.0 );

    float secstruct_half_width;
    bool darken_inside;
    if (secstruct_res == 'H'){
      secstruct_half_width = privates::helix_half_width;
      darken_inside = true;
      hermite_factor = helix_hermite_factor;
    } else if ( secstruct_res == 'E' ){
      secstruct_half_width = privates::strand_half_width;
      darken_inside = false;
      hermite_factor = strand_hermite_factor;
    } else {
      assert( secstruct_res == 'N' );
      secstruct_half_width = privates::strand_half_width;
      darken_inside = true;
      hermite_factor = NA_hermite_factor;
    }


    xyzVector_float axis_segment, CA_segment;
    for (int s = 1; s <= privates::NUM_SEGMENTS; s++ ){
      const float p = s / static_cast<float>(privates::NUM_SEGMENTS);
      axis_segment = p*axis + (1-p)*prev_axis;
      CA_segment = get_CA_segment( prev_CA, current_CA, prev_tangent, tangent, p, hermite_factor);
      get_residue_color( static_cast<float>(n - 1) + p, red, green, blue, false, pose.total_residue());

      const float half_width = get_half_width( taper, secstruct_half_width, p );
      draw_next_polygon(CA_segment - half_width*axis_segment, CA_segment + half_width*axis_segment,
												red,green,blue, res(n), gs, darken_inside);
    }

  }

  void priestle_smooth( const FArray3DB_float & xyz_full, FArray3DB_float & xyz_smooth,
												int start, int end, int total_residue ) {

    for (int i = 1; i<=total_residue; i++) {
      const int j = 2;
      for (int k = 1; k <= 3; k++ ){
				xyz_smooth(k,j,i) = xyz_full(k,j,i);
      }
    }

    for (int i = start+1; i<=end-1; i++) {
      int const j = 2;
      for (int k = 1; k <= 3; k++ ){
				xyz_smooth(k,j,i) = ((xyz_full(k,j,i-1) + xyz_full(k,j,i+1))/2 + xyz_full(k,j,i))/2;
      }
    }

  }

  void draw_helix( int start, int end, GraphicsState & gs, const PoseStandin & pose ) {

    const int total_residue = pose.total_residue();
    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();

    //Starting point.
    int prior_res( start-1 );
    if (prior_res < 1) prior_res = 1;
    xyzVector_float direction, normal, tangent, axis, current_CA;
    xyzVector_float prev_CA, prev_tangent, prev_axis;
    const int NUM_SEGMENTS( 5 );

    //For the starting point, need to figure out axis from next residue...
    int next_res = start + 1;
    if (next_res > total_residue-1)
      next_res = total_residue-1;
    get_direction( direction, next_res+1, next_res-1, xyz_full);
    get_normal( normal, next_res, xyz_full );
    get_axis_and_tangent( axis, tangent, direction, normal);

    get_direction(direction, start+1, prior_res, xyz_full);
    tangent = direction;
    current_CA = &xyz_full(1,2,start);
    if (start==1)
      set_initial_polygon_vertices( current_CA - privates::helix_half_width*axis, current_CA + privates::helix_half_width*axis, gs);

    //previous residue's helix geometry
    prev_CA = current_CA;
    prev_tangent = tangent;
    prev_axis = axis;
    std::string taper = "start";

    // Draw the body of the helix.
    for (int n = start+1; n<=end-1; n++){
      //new residue's helix geometry
      get_direction( direction, n+1, n-1, xyz_full);
      get_normal( normal, n, xyz_full );
      get_axis_and_tangent( axis, tangent, direction, normal);
      current_CA = &xyz_full(1,2,n);
      draw_secstruct_chunk( prev_CA, current_CA, prev_tangent, tangent, prev_axis, axis, n,
														'H', taper, gs, pose);

      //previous residue's helix geometry
      prev_CA = current_CA;
      prev_tangent = tangent;
      prev_axis = axis;
      taper = "none";
    }

    //last piece.
    //	next_res = end + 1;
    //	if (next_res > nres)  next_res = nres;
    //	get_direction( direction, next_res, end - 1, xyz_full);
    get_direction( direction, end, end - 1, xyz_full);
    tangent = direction;
    current_CA = &xyz_full(1,2,end);
    taper = "end";
    draw_secstruct_chunk( prev_CA, current_CA, prev_tangent, tangent, prev_axis, axis, end,
													'H', taper, gs, pose);

  }

  void draw_strand( int start, int end, GraphicsState & gs, const PoseStandin & pose ) {

    const int total_residue = pose.total_residue();
    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();

    //Pre-smooth?
    FArray3D_float xyz_smooth(3, param::MAX_POS, total_residue);
    priestle_smooth( xyz_full, xyz_smooth, start, end, total_residue );

    //Starting point.
    int prior_res( start-1 );
    if (prior_res < 1) prior_res = 1;
    xyzVector_float direction, normal, tangent, axis, current_CA;
    xyzVector_float prev_CA, prev_direction, prev_normal;
    const int NUM_SEGMENTS( 5 );

    //For the starting point, need to figure out axis from next residue...
    int next_res = start + 1;
    if (next_res > total_residue-1) next_res = total_residue-1;
    get_direction( direction, next_res+1, next_res-1, xyz_smooth);
    get_normal( normal, next_res, xyz_smooth );

    current_CA = &xyz_smooth(1,2,start);
    if (start==1)
      set_initial_polygon_vertices( current_CA - privates::strand_half_width*normal, current_CA + privates::strand_half_width*normal, gs);

    //previous residue's strand geometry
    prev_CA = current_CA;
    prev_normal = normal;
    prev_direction = direction;
    std::string taper = "start";

    // Draw the body of the strand.
    for (int n = start+1; n<=end-1; n++){

      //		if (n == end-1) taper = "strand_penultimate";

      //new residue's strand geometry
      get_direction( direction, n+1, n-1, xyz_smooth);
      get_normal( normal, n, xyz_full );
      if ( dot(normal, prev_normal) < 0.0 ) normal *= -1.0;
      current_CA = &xyz_smooth(1,2,n);
      draw_secstruct_chunk( prev_CA, current_CA, prev_direction, direction, prev_normal, normal, n,
														'E', taper, gs, pose);

      //previous residue's strand geometry
      prev_CA = current_CA;
      prev_normal = normal;
      prev_direction = direction;
      taper = "none";
    }

    //last piece.
    next_res = end + 1;
    if (next_res > total_residue)  next_res = total_residue;
    get_direction( direction, next_res, end - 1, xyz_smooth);
    tangent = direction;
    current_CA = &xyz_smooth(1,2,end);
    taper = "strand_ultimate";
    draw_secstruct_chunk( prev_CA, current_CA, prev_direction, direction, prev_normal, normal, end,
													'E', taper, gs, pose);
  }

  void draw_NA( const int start, const int end, GraphicsState & gs, const PoseStandin & pose ) {

    const int total_residue = pose.total_residue();
    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();

    if (end <= 1) return;

    if (!param_aa::is_NA( pose.res()(start) )) return;

    int atomindex( 6 ); //C4*

    //Pre-smooth?
    //	FArray3D_float xyz_smooth(3, param::MAX_POS, total_residue);
    //	priestle_smooth( xyz_full, xyz_smooth, start, end, total_residue);

    //Starting point.
    xyzVector_float direction, normal, tangent, axis, current_CA;
    xyzVector_float prev_CA, prev_direction, prev_normal;
    const int NUM_SEGMENTS( 5 );

    //For the starting point, need to figure out axis from next residue...
    int next_res = start + 1;
    if (next_res > total_residue-1) next_res = total_residue-1;
    get_direction( direction, next_res + 1, start, xyz_full, atomindex);
    get_normal( normal, next_res, xyz_full, atomindex );

    current_CA = &xyz_full(1,atomindex,start);
		if (start==1)
			set_initial_polygon_vertices( current_CA - privates::strand_half_width*normal, current_CA + privates::strand_half_width*normal, gs);

    //previous residue's strand geometry
    prev_CA = current_CA;
    prev_normal = normal;
    prev_direction = direction;
    std::string taper = "none";

    // Draw the body of the strand.
    for (int n = start+1; n<=end-1; n++){
      //new residue's strand geometry
      get_direction( direction, n+1, n-1, xyz_full, atomindex);
      get_normal( normal, n, xyz_full, atomindex );
      //		if ( dot(normal, prev_normal) < 0.0 ) normal *= -1.0;
      current_CA = &xyz_full(1,atomindex,n);
      draw_secstruct_chunk( prev_CA, current_CA, prev_direction, direction, prev_normal, normal, n,
														'N', taper, gs, pose);

      //previous residue's strand geometry
      prev_CA = current_CA;
      prev_normal = normal;
      prev_direction = direction;
    }

    //last piece.
    next_res = end + 1;
		if (next_res > total_residue)  next_res = total_residue;
		get_direction( direction, next_res, end - 1, xyz_full, atomindex);
    tangent = direction;
    current_CA = &xyz_full(1,atomindex,end);

    draw_secstruct_chunk( prev_CA, current_CA, prev_direction, direction, prev_normal, normal, end,
													'N', taper, gs, pose);
  }

  void draw_coil_chunk( const xyzVector_float & prev_CA, const xyzVector_float & current_CA,
												const xyzVector_float & prev_tangent, const xyzVector_float & tangent,
												int n,
												GraphicsState & gs, const PoseStandin & pose ) {

    const FArray1DB_int & res = pose.res();

    const int NUM_SEGMENTS = 10;
    const float coil_hermite_factor = 5.0;
    float red( 0.0 ), green( 0.0 ), blue( 0.0 );

    xyzVector_float axis_segment, CA_segment, prev_CA_segment, bond, prev_bond;

    GLfloat currentrotation[16];
    glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
    xyzVector_float z(currentrotation[2],currentrotation[6],currentrotation[10]); // z pointing out of window in current view.

    prev_CA_segment = prev_CA;
    prev_bond = current_CA - prev_CA;
    for (int s = 1; s <= NUM_SEGMENTS; s++ ){
      const float p = s / static_cast<float>(NUM_SEGMENTS);
      CA_segment = get_CA_segment( prev_CA, current_CA, prev_tangent, tangent, p, coil_hermite_factor);
      get_residue_color ( static_cast<float>(n) + p, red, green, blue, false, pose.total_residue());

      //Need to replace the following with a real cylinder.
      bond = CA_segment - prev_CA_segment;
      //		if (bond.length_squared() > 4.0) continue;
      xyzVector_float	width =  cross( bond, z );
      //		xyzVector_float	width =  cross( bond, prev_bond );
      if (width.length_squared() > 0.0001 )  width.normalize();
      width = (width - 0.5f*z).normalized(); //nice shadow effect
      width *= privates::coilradius;

      draw_next_polygon( CA_segment + width, CA_segment - width, red, green, blue, res(n) , gs );

      prev_CA_segment = CA_segment;
      prev_bond = bond;
    }

  }

  void draw_coil( int start, int end, GraphicsState & gs, const PoseStandin & pose ) {

    const int total_residue = pose.total_residue();
    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();

    // Draw the body of the coil.
    xyzVector_float prev_CA, prev_direction, direction, current_CA;
    prev_CA = &xyz_full(1,2,start);
    get_direction( prev_direction, start+1, start, xyz_full);

    if (start==1)
      set_initial_polygon_vertices( prev_CA, prev_CA, gs );
    for (int n = start; n <= end-1; n++){
      if (n < end-1 && end < total_residue)
				get_direction( direction, n+2, n, xyz_full);
      else
				get_direction( direction, n+1, n, xyz_full);
      current_CA = &xyz_full(1,2,n+1);
      draw_coil_chunk( prev_CA, current_CA, prev_direction, direction, n, gs, pose);
      //previous residue's coil geometry
      prev_CA = current_CA;
      prev_direction = direction;
    }
  }

  void draw_segment( int start_segment, int end_segment, const char prev_secstruct,
										 GraphicsState & gs, const PoseStandin & pose ) {

    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();

    const int size_segment = end_segment - start_segment + 1;

    if (start_segment >= end_segment) return;  //not drawable.

    //	if (graphics::current_label == "Native") std::cerr << "DRAW_SEGMENT " << start_segment << " " << end_segment << " " << prev_secstruct << " " << total_residue << std::endl;

    if (prev_secstruct=='H' && size_segment >= 4)
      draw_helix( start_segment, end_segment, gs, pose );
    else if (prev_secstruct=='E' && size_segment >= 2)
      draw_strand( start_segment, end_segment, gs, pose );
    else if (prev_secstruct=='N' )
      draw_NA( start_segment, end_segment, gs, pose );
    //		draw_coil( start_segment, end_segment, pose );
    else //if ( prev_secstruct=='H' || prev_secstruct =='L' || prev_secstruct=='E' )
      draw_coil( start_segment, end_segment, gs, pose );
  }

  bool check_occupancy(int i, const FArray1DB_int & res, const FArray2DB_float & occ) {
    using namespace param_aa;

    int atomindex( 2 ); //C-alpha
    if (is_NA( res(i) )) {
      static const int c4star = LookupByName( res(i), 1, " C4*");
      atomindex = c4star;
    }
    return ( occ( atomindex, i ) > 0.01 );
  }

  bool check_chainbreak(int i, const FArray1DB_int & res, const FArray3DB_float & xyz_full) {
    using namespace param_aa;

    if (i==1) return false;

    if ( is_NA(res(i)) && !is_NA(res(i-1)) ) return true;
    if ( is_NA(res(i-1)) && !is_NA(res(i)) ) return true;

    float CHAINBREAK_CUTOFF2 = 4.5*4.5;
    if ( is_NA(res(i))) CHAINBREAK_CUTOFF2 = 7.5*7.5;

    int atomindex = 2;
    if (is_NA( res(i) )) {
      static const int c4star = LookupByName( res(i), 1, " C4*");
      atomindex = c4star;
    }

    xyzVector_float vec( &xyz_full(1,atomindex,i) );
    xyzVector_float vec_prev( &xyz_full(1,atomindex, i-1) );
    const float dist2 = (vec-vec_prev).length_squared();
    if ( dist2 > CHAINBREAK_CUTOFF2) {
      return true;
    }

    return false;
  }


  void draw_secstruct(GraphicsState & gs, const PoseStandin & pose, int const begin, int const end, bool draw_cartoon=true/* if false, draw cylinder*/) {

 		using namespace protein_graphics;
		// aliases
    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();
    const FArray1DB_char & secstruct = pose.secstruct();
    const FArray2DB_float & occ = pose.pdb_info().occupancy();
    const FArray1DB_int & res = pose.res();

		draw_cartoon = true; // currently,  draw cylinder is not ready

		//////////////////////////////////////////////////////////////////////
    //Parse secondary structure elements, and then draw them separately.
    //////////////////////////////////////////////////////////////////////
    //Need to make sure we start at an occupied residue.
    int start_draw = begin;
    for (start_draw = begin;  start_draw<=end; start_draw++ )
      if (check_occupancy(start_draw, res, occ))
				break;

    char prev_secstruct = secstruct( start_draw );
    if ( param_aa::is_NA( res(1)) )
      prev_secstruct = 'N';

    int start_segment( start_draw );
    bool is_chainbreak = false;

    for (int i = start_draw+1; i <= end; i++){
      char current_secstruct = secstruct(i);
      const int aa = res(i);
      if (aa<1)
				return;
      if (param_aa::is_NA(aa))
				current_secstruct = 'N';

      if (current_secstruct != prev_secstruct || !check_occupancy(i,res,occ) || check_chainbreak(i,res,xyz_full))  {

				int const end_segment = i - 1;

				draw_segment( start_segment, end_segment, prev_secstruct, gs, pose );
				start_segment = i;

				if ( !check_occupancy(i,res,occ) ) {
					start_segment++;
				} else if ( !check_chainbreak(i,res,xyz_full) ) {
					draw_segment( end_segment, end_segment+1, 'L', gs, pose ); //connector region
				}
      }
      prev_secstruct = current_secstruct;
    }

    if (check_occupancy(end, res, occ)) {
      draw_segment( start_segment, end, prev_secstruct, gs, pose );
		}

  }

  void draw_backbone( GraphicsState & gs, const PoseStandin & pose ) {

    // aliases
    const int total_residue = pose.total_residue();
		const FArray1DB_int & res = pose.res();

		if (int(res.size1()) < total_residue)
      return; //Are we initialized yet?

		switch( gs.BBdisplay_state ) {
		case SHOW_BACKBONE:
			draw_Calpha_trace( 1, total_residue, pose );
			return;
		case SHOW_CARTOON:
			draw_secstruct( gs, pose, 1, total_residue );
			return;
		case SHOW_CYLINDER:
			draw_secstruct( gs, pose, 1, total_residue );
			return;
		case SHOW_BB_SPACEFILL:
			return;
		case SHOW_NOBB:
			return;
		}
	}

  void draw_sidechains(GraphicsState & gs, const PoseStandin & pose) {
    // This code is getting a bit complicated because I wanted to do nucleic
    // acids a bit different from proteins.
    // Also this will probably go haywire for DNA (I was using an RNA test case).
		using namespace protein_graphics;
		using namespace param_aa;

    const int total_residue = pose.total_residue();
    const FArray3DB_float & xyz_full = pose.full_coord();
    const FArray1DB_int & res = pose.res();
    const FArray1DB_int & res_variant = pose.res_variant();
    const FArray2DB_float & occ = pose.pdb_info().occupancy();

    GLfloat currentrotation[16];
    glGetFloatv(GL_MODELVIEW_MATRIX, currentrotation);
    xyzVector_float z(currentrotation[2],currentrotation[6],currentrotation[10]); // z pointing out of window in current view.

    const float z_offset = .1;
    xyzVector_float zsmalloffset = z * z_offset;

    // sidechain atoms
    for ( int i = 1; i <= total_residue; i++ ) {
			//			if(!is_ligand(res(i))&&occ(2,i) <= 0.01) continue;
			const int aa( res(i) );
      const int aav( res_variant(i) );

			if ( !is_ligand(aa) && !get_show_sc(i) ) continue;
      if ( aa == param_aa::aa_gly ) continue;
			if ( !is_ligand(aa) && gs.SCdisplay_state != SHOW_STICK
					 && gs.SCdisplay_state != SHOW_WIREFRAME ) continue;
			if ( is_ligand(aa) && gs.LIGdisplay_state != SHOW_LIG_STICK
					 && gs.LIGdisplay_state != SHOW_BALL_AND_STICK ) continue;
      if ( aav != 1 ) continue; //chu no variant yet
      // color selection
      float red( 0.0 ), green( 0.0 ), blue( 0.0 );
      get_residue_color( i, red, green, blue, false, total_residue );
      glColor3f(red, green, blue);

      // loop through each heavy atom
			float xwidth = protein_stickScale;
			float ligand_width = ligand_stickScale; //for ligand
      std::map< int, xyzVector_float> prev1, prev2;

      std::vector< int > atom1_list, atom2_list;
      int atom2_begin = aaproperties_pack::first_scatom(aa,aav);
      int atom2_end = aaproperties_pack::nheavyatoms(aa,aav);

			if( !is_ligand(aa) && gs.SCdisplay_state == SHOW_WIREFRAME ) {
				xwidth = protein_wireframeScale;
			}

			//This is getting too complicated, man!
			if (param_aa::is_NA(aa)){
				static const int n1 = LookupByName( aa, 1, " N1 " ); //don't draw backbone
				if ( gs.BBdisplay_state == SHOW_CARTOON ) {
					if ( gs.SCdisplay_state != SHOW_NOSC ) {
						atom2_begin = n1;
					} else {
						continue;
					}
				} else {
					atom2_begin = 1;
					if ( gs.SCdisplay_state == SHOW_NOSC ) atom2_end = n1 - 1;
				}
			}


			/////////////////////////////
			// Side chain bonds.
			/////////////////////////////
			int const num_atoms = aaproperties_pack::natoms( aa, aav );
			FArray2D_bool is_bonded(num_atoms, num_atoms, false );

      for ( int atom1 = atom2_begin; atom1 <= atom2_end; ++atom1 ) {

				if(!is_ligand(res(i)) && occ(atom1,i) <= 0.01) continue;
				int const num_neighbors = aaproperties_pack::nbonded_neighbors(atom1,aa,aav);

				for (int k = 1; k <= num_neighbors; k++){

					int const atom2 = aaproperties_pack::bonded_neighbor(k, atom1, aa, aav );
					if(!is_ligand(res(i)) && occ(atom2,i) <= 0.01) continue;

					//don't draw backbone for nucleic acids.
					if (param_aa::is_NA( aa ) && gs.BBdisplay_state == SHOW_CARTOON
							&& aaproperties_pack::atom_name(atom2,aa,aav)==" C1*" ) continue;

					// Don't draw hydrogens
					if (aaproperties_pack::fullatom_type( atom2, aa, aav) > 21) continue;

					//Don't add bond if its already there!
					if ( is_bonded( atom2, atom1) ) continue;

					is_bonded( atom1, atom2) = true;
					atom1_list.push_back( atom1 );
					atom2_list.push_back( atom2 );
				}
			}


			/////////////////////////////
			// Other bonds
			/////////////////////////////

      bool link_ahead = false;

      if (  gs.BBdisplay_state == SHOW_CARTOON && gs.SCdisplay_state != SHOW_NOSC ){
				/////////////////////////////
				//Add connection to backbone.
				/////////////////////////////
				if (param_aa::is_protein( aa ) && !(aa == param_aa::aa_gly) ){
					atom1_list.push_back( 2 ); //C-alpha
					atom2_list.push_back( 5 ); //C-beta
				}
				if (param_aa::is_NA( aa )) {
					static const int c4star( LookupByName( aa, 1, " C4* " ) );
					atom1_list.push_back( c4star );
					//I had a slick way of figuring out the connection
					// point before, but I accidentally erased the code, and can't remember it.
					int baseconnect;
					if (aa == param_aa::na_rgu || aa == param_aa::na_gua || aa == param_aa::na_ade || aa == param_aa::na_rad ) {
						baseconnect = LookupByName( aa, 1, " N9 ");
					} else {
						baseconnect = LookupByName( aa, 1, " N1 ");
					}
					atom2_list.push_back( baseconnect );
				}
      } else {
				///////////////////////////////
				//Link ahead for nucleic acids.
				///////////////////////////////
				if ( i < total_residue && param_aa::is_NA( res(i) ) && param_aa::is_NA( res(i+1)) ) {
					// add the connection to the next residue
					static const int o3( LookupByName( res(i), 1, " O3*" ) );
					static const int p ( LookupByName( res(i+1), 1, " P  " ) );
					atom1_list.push_back( o3 );
					atom2_list.push_back( p  );
					link_ahead = true;
				}
      }


      assert( atom1_list.size() == atom2_list.size() );

      for ( unsigned k = 0; k < atom1_list.size(); ++k ) {
				const int atom1 = atom1_list[k];
				const int atom2 = atom2_list[k];

				// colors
				//			get_residue_color( i, red, green, blue, true /*mute_color*/ );
				xyzVector_float
					atom1_color, atom2_color;
				atom1_color = get_atom_color( gs,atom1,aa,aav,i,total_residue);
				atom2_color = get_atom_color( gs,atom2,aa,aav,i,total_residue);

				xyzVector_float atom_pos1( &xyz_full(1,atom1,i));
				xyzVector_float atom_pos2( &xyz_full(1,atom2,i));

				if ( link_ahead && k == atom1_list.size()-1) {
					atom_pos2 = &(xyz_full(1,atom2,i+1));
				}

				//////////////////////////////
				// Check for chainbreaks.
				//////////////////////////////
				xyzVector_float bond;
				bond = atom_pos2 - atom_pos1;
				float BOND_LENGTH_CUTOFF2 = 4.0*4.0;
				if (param_aa::is_NA(aa)) BOND_LENGTH_CUTOFF2 = 8.0*8.0;
				if ( bond.length_squared() > BOND_LENGTH_CUTOFF2 ) break;


				//////////////////////////////
				// Set width.
				//////////////////////////////
				xyzVector_float width( cross( bond, z ));
				if ( width.length_squared() > 0.0001 )  width.normalize();
				if( param_aa::is_ligand(aa) ) {
					width = width * ligand_width;
				} else {
					width = width* xwidth;
				}

				//Slight tweak for nucleic acids.
				if (param_aa::is_NA(aa)) {
					if (atom1 < 12 && atom2 < 12) {
						width *= 2.0; //thicker backbone
					} else {
						width *=0.5; //really thin bases
						if ( gs.SCdisplay_state == SHOW_NOSC ) continue; //don't draw at all!
					}
				}

				glColor3fxyz( atom1_color );
				//							if ( atom2 > 5 && ! (aa == param_aa::aa_pro && atom1 == 1) && param_aa::is_protein(aa) ) {
				if ( prev1.count( atom1 ) ) {
					glBegin(GL_POLYGON);
					glVertex3fxyz ( prev1[atom1] );
					glVertex3fxyz ( atom_pos1 + width );
					glVertex3fxyz ( atom_pos1 - width );
					glVertex3fxyz ( prev2[atom1] );
					glEnd();
				}
				glBegin(GL_POLYGON);
				glVertex3fxyz ( atom_pos1 + width );
				glVertex3fxyz ( atom_pos1 - width );
				glColor3fxyz( atom2_color );
				glVertex3fxyz ( atom_pos2 - width );
				glVertex3fxyz ( atom_pos2 + width );
				glEnd();


				if ( param_aa::is_NA( aa ) && atom1 <= 12 && atom2 <= 12 ) {
					glColor3f(0,0,0);
					glBegin(GL_LINES);
					glVertex3fxyz ( zsmalloffset + atom_pos1 + width );
					glVertex3fxyz ( zsmalloffset + atom_pos2 + width );
					glVertex3fxyz ( zsmalloffset + atom_pos2 - width );
					glVertex3fxyz ( zsmalloffset + atom_pos1 - width );
					glEnd();
				}

				prev1[atom2] = ( atom_pos2 + width );
				prev2[atom2] = ( atom_pos2 - width );
      }
    }

  }

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void
	draw_sphere( GraphicsState & gs, const PoseStandin & pose ){

		//lin currently only ligand
		//lin should be easy to extend to non ligand
		using namespace protein_graphics;
		using namespace param_aa;
		using namespace aaproperties_pack;

    const int nres = pose.total_residue();
    const FArray3DB_float & xyz_full = pose.full_coord();
    const FArray1DB_int & res = pose.res();
    const FArray1DB_int & res_variant = pose.res_variant();
    const FArray2DB_float & occ = pose.pdb_info().occupancy();

 		glPushAttrib(GL_ENABLE_BIT);
		glEnable(GL_LIGHTING);

		GLUquadricObj *sphereObj = gluNewQuadric();
		// Create a glu quadric object
		gluQuadricDrawStyle(sphereObj, GLU_FILL);
		gluQuadricNormals(sphereObj, GLU_SMOOTH);

		// sidechain atoms
		for ( int i = 1; i <= nres; i++ ) {
			int const aa( res(i) );
			if( !is_ligand(aa) && !get_show_sc(i) ) continue;
			if( !is_ligand(aa) && gs.BBdisplay_state != SHOW_BB_SPACEFILL
					&& gs.SCdisplay_state != SHOW_SC_SPACEFILL ) continue;
			if( is_ligand(aa) && gs.LIGdisplay_state != SHOW_BALL_AND_STICK
					&& gs.LIGdisplay_state != SHOW_LIG_SPACEFILL ) continue;
			int const aav( res_variant(i) );

			// loop through each heavy atom
			int atom_begin = 1 ;
			int atom_end = natoms(aa,aav);
			//int atom_end = nheavyatoms(aa,aav);
			if( !is_ligand(aa) ) {
				if( gs.BBdisplay_state != SHOW_BB_SPACEFILL )
					atom_begin = first_scatom(aa,aav);
				if( gs.SCdisplay_state != SHOW_SC_SPACEFILL )
					atom_end = first_scatom(aa,aav) - 1 ;
			}

			for ( int atom = atom_begin; atom <= atom_end; ++atom ) {

				if( !is_ligand(aa) && occ(atom,i) <= 0.01 ) continue;

				xyzVector_float atom_pos( &xyz_full(1,atom,i));

				// Figure out the material based on the atom type.
				xyzVector_float atom_color
					( get_atom_color( gs,atom,aa,aav,i,nres) );

				float const sphere_opacity ( is_protein(aa) ? protein_sphere_opacity :
																		 ligand_sphere_opacity );
				float const sphere_shininess ( is_protein(aa) ? protein_sphere_shininess :
																			 ligand_sphere_shininess );
				//GLfloat mat_shininess[] = { sphere_shininess };
				GLfloat atom_material[4] = {
					atom_color[0],
					atom_color[1],
					atom_color[2],
					1.0,
				};
				GLfloat specular_material[4] = {
					atom_material[0] * sphere_opacity,
					atom_material[1] * sphere_opacity,
					atom_material[2] * sphere_opacity,
					sphere_opacity,
				};

				// Highlight the nonprotein
				if ( ! is_protein(aa) ) {
					for (int color = 0 ; color < 3 ; color++) {
						atom_material[color] *= 1.5;
						if (atom_material[color] > 1.0)
							atom_material[color] = 1.0;
					}
					if (atom_material[3] < 0.1)
						atom_material[3] = 0.1;
				}
				if (atom_material[3] < 0.1) continue;
				glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,	 atom_material);
				glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,	 atom_material);
				glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, specular_material);
				glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &sphere_shininess);

				glPushMatrix();
				glTranslatef(atom_pos(1), atom_pos(2), atom_pos(3));

				// draw a sphere using the sphere display list
				if (privates::sphereDisplayList == 0) {
					const int SPHERE_SLICES(16), SPHERE_STACKS(16);
					float sphereRadius(1.0);
					privates::sphereDisplayList = glGenLists(1);
					assert(privates::sphereDisplayList != 0);

					glNewList(privates::sphereDisplayList, GL_COMPILE);
					gluSphere(sphereObj, sphereRadius, SPHERE_SLICES, SPHERE_STACKS);
					glEndList();
				}
				if( privates::sphereDisplayList != 0) {
					float sphereRadius, sphereScale(1.0);
					const float scale_for_display_list(1.0);
					if( is_ligand(aa) && gs.LIGdisplay_state == SHOW_BALL_AND_STICK ) {
						sphereScale = ligand_sphereScale * scale_for_display_list ;
					} else {
						sphereScale = 1.0 * scale_for_display_list ;
					}
					sphereRadius = sphereScale * etable::fa_lj_radius(fullatom_type(atom,aa,aav));
					glScalef(sphereRadius, sphereRadius, sphereRadius);
				}
				glCallList(privates::sphereDisplayList);

				glPopMatrix();
			}
		}

// 		// clean up
		glPopAttrib();
		gluDeleteQuadric(sphereObj);
		glDisable(GL_LIGHTING);// Turn lighting off
	}


  void draw_pose(const PoseStandin & pose,
								 GraphicsState & gs,
								 bool fullatom, bool centered) {

    const int total_residue = pose.total_residue();
    const FArray3DB_float & xyz_full = pose.fullatom() ? pose.full_coord() : pose.Eposition();
    const FArray1DB_int & res = pose.res();
    const FArray2DB_float & occ = pose.pdb_info().occupancy();

		GLfloat light_position[] = { 1.0, 1.0, 1.0, 0.0 };
		glEnable(GL_LIGHT0);
		glEnable(GL_DEPTH_TEST);
		glShadeModel(GL_SMOOTH);
		glLightfv(GL_LIGHT0, GL_POSITION, light_position);

    //Pre-screen.
		if ( xyz_full.size3() < total_residue) return;
    for ( int j = 1; j <= total_residue; ++j ) {
      if (res(j)<1 || res(j) > param::MAX_AA()() ) return; //out of here.
    }

    if(centered) {
      xyzVector_float avg_vec(0.0f, 0.0f, 0.0f);

      for ( int i = 1; i<=3; i++ ) {
				float avg = 0.0f;
				float total_displayed_res = 0.0f;
				for ( int j = 1; j <= total_residue; ++j ) {

					int atomindex = 2; //C-alpha
					if (param_aa::is_NA( res(j) )) {
						static const int c4star = LookupByName( res(j), 1, " C4*");
						atomindex = c4star;
					}

					// properly display structures with missing densities
					if( occ( atomindex, j ) > 0.01 ){
						if (param_aa::is_NA( res(j))) {
							if (!pose.fullatom() ) return;  //problem!
							avg += xyz_full(i,atomindex,j); //need to look into fullcoord
						} else {
							avg += xyz_full(i,2,j);
						}
						total_displayed_res++;
					}

				}

				avg_vec(i) = avg / total_displayed_res;
      }

      glPushMatrix();
      glTranslatef(-avg_vec.x(), -avg_vec.y(), -avg_vec.z());
    }

    if ( total_residue > 0 ) {
      draw_backbone( gs, pose );
			draw_sphere( gs, pose );
		}
    if ( fullatom )
      draw_sidechains( gs, pose );

    if(centered) {
      glPopMatrix();
    }
		glDisable(GL_LIGHT0);// Turn lighting off
  }

  void draw_pose(const Pose & pose,
								 GraphicsState & gs,
								 bool fullatom, bool centered) {

    PoseStandin standin(pose.total_residue(),
												pose.fullatom(),
												pose.full_coord(),
												pose.Eposition(),
												pose.res(),
												pose.res_variant(),
												pose.secstruct(),
												pose.pdb_info().occupancy());

    draw_pose(standin, gs, fullatom, centered);
  }

  void draw_pose(int const total_residue,
		 const FArray3DB_float & full_coord,
		 const FArray3DB_float & Eposition,
		 const FArray1DB_int & res,
		 const FArray1DB_int & res_variant,
		 const FArray1DB_char & secstruct,
		 const FArray2DB_float & occupancy,
		 protein_graphics::GraphicsState & gs,
		 bool fullatom, bool centered) {

    //I'm not entirely sure, but I think the conversion of Eposition
    // to a FArray3D_float may involve a copy, which isn't very efficient.
    // Should be possible to avoid this.
    PoseStandin standin(total_residue,
												fullatom,
												full_coord,
												Eposition,
												res,
												res_variant,
												secstruct,
												occupancy);

    draw_pose(standin, gs, fullatom, centered);
  }

	/////////////////////////////////////////////////////////////////////
	void
	set_show_sc_array( FArray1D_bool & allow_sc )
	{
		////////////////////
		////////////////////
		// Temporary hack.
		////////////////////
		////////////////////
		return;
		////////////////////
		////////////////////

// 		assert(int(privates::show_sc.size1()) >= int(allow_sc.size1()) );
// 		for(int i=1;i<=int(allow_sc.size1());i++) {
// 			privates::show_sc(i)=allow_sc(i);
// 		}
// #ifdef GL_GRAPHICS
// 		std::string outstring("  ");
// 		GRAPHICS_LOG( "Show sidechain::" );
// 		for(int i=1;i<=int(privates::show_sc.size1());i++)
// 			if(	privates::show_sc(i) ) outstring = outstring + string_of(i) + " ";
// 		GRAPHICS_LOG( outstring );
// #endif
	}

	/////////////////////////////////////////////////////////////////////
	bool
	get_show_sc( int const ii ) {

		////////////////////
		////////////////////
		// Temporary hack.
		////////////////////
		////////////////////
		return true;
		////////////////////
		////////////////////

		//		return privates::show_sc(ii);
	}
	/////////////////////////////////////////////////////////////////////
	void
	set_show_repack( bool state, int show_repack_fps ){
		privates::show_repack=state;
		privates::max_show_repack_fps=show_repack_fps;
	}

	/////////////////////////////////////////////////////////////////////
	bool
	get_show_repack( int count ){
		if( privates::show_repack ) {
			return count % privates::max_show_repack_fps == 1;
		} else {
			return false;
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void
	plot_timeseries(
	std::vector<float> const & data,
	bool const vertical,
	float const min,
	float const max
									)
	{
		using namespace protein_graphics;

		unsigned int total_steps = data.size();
		if ( data.size() > 0) {
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();

			if (vertical) {
				gluOrtho2D( min, max, 0, total_steps-1  );
			} else {
				gluOrtho2D( 0, total_steps-1, min, max );
			}
			glMatrixMode(GL_MODELVIEW);
			//  glColor3f(0.1f,0.7f,0.7f); //light blue
			//  glColor3f(1.0f,0.0f,0.0f); // red
			glDisable( GL_DEPTH_TEST );
			glLoadIdentity();
			glColor3f(.5,.5,.5);
			float offset = 0; // .1;

			glColor3f(0.0f,0.5f,0.4f); //light blue
			float min_point = 1e12;
			glBegin( GL_LINE_STRIP );
			for (unsigned int step = 0; step < total_steps; step++) {
				float x, y;
				float point = data[step];
				if ( point < min_point ) min_point = point;
				if (vertical) {
					x = point;
					y = (total_steps-1) - step;
				} else {
					x = step;
					y = point;
				}
				glVertex2f( x, y ) ;
			}
			glEnd();
			glLoadIdentity();

			if (vertical) {
				if (get_native_exists() && max > 0) {
					glLoadIdentity();
					float width = max-min;
					glTranslatef(min+width*.04,float(total_steps)*.02,0);
					float yscale = .0005*float(total_steps-1.00);
					float xscale = .001*width;
					writeStrokeString( std::string ("RMSD"), xscale, yscale );
				}
			} else {
				float height = max - min;
				glLoadIdentity();
				glTranslatef(float(total_steps)*.02,min+height*.055,0);
				float xscale = .00020*float(total_steps-1);
				float yscale = .00120*height;
				writeStrokeString( std::string ("Accepted Energy"), xscale, yscale );
			}
			glEnable( GL_DEPTH_TEST );
			glLoadIdentity();
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void
	plot_2D(
					std::vector<float> const & xdata,
					std::vector<float> const & ydata,
					float const xmin,
					float const xmax,
					float const ymin,
					float const ymax
					)
	{

		unsigned int total_steps = xdata.size();

		if (ydata.size() < total_steps) {
			total_steps = ydata.size();
		}
		if ( total_steps > 0 ) {
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			gluOrtho2D( xmin, xmax, ymin, ymax );
			glMatrixMode(GL_MODELVIEW);
			glDisable( GL_DEPTH_TEST );
			glLoadIdentity();

			glPointSize( 2.0 );
			glBegin( GL_POINTS );
			for (unsigned int step = 0; step < total_steps; step++) {
				float x = xdata[step];
				float y = ydata[step];
				glVertex2f( x, y ) ;
			}
			glEnd();

			int decoys = trajectory::low_rmsd.size();
			if (decoys > 0) {
				glPointSize( 4.0 );
				glBegin( GL_POINTS );
				glColor3f(0.6f,0.13f,0.0f); //rust
				for (int step = 0; step < decoys; step++) {
					float x = trajectory::low_rmsd[step];
					float y = trajectory::low_energy[step];
					glVertex2f( x, y ) ;
				}
				glEnd();
			}

			// cross hairs
			glColor3f(.5,.5,.5);
			glBegin( GL_LINES );
			glVertex2f( xdata[total_steps-1], ymax ) ;
			glVertex2f( xdata[total_steps-1], ymin );
			glVertex2f( xmin, ydata[total_steps-1] );
			glVertex2f( xmax, ydata[total_steps-1] );
			glEnd();

			glEnable( GL_DEPTH_TEST );
			glLoadIdentity();

		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////////
	void
	get_bounds(
						 std::vector< float > const & t,
						 float & mn,
						 float & mx
					 )
	{
		if ( !t.size() ) return;

		// This is sort of howit was before:
		//	mn = t[0].score;
		//	mx = t[0].score;
		//Look back at most 100 steps.
		//	int const i_start = std::max( t.size() - 100, 1);
		//	for ( int i=i_start, i_end= t.size(); i< i_end; ++i ) {
		//		mn = std::min( mn, t[i].score);
		//		mx = std::max( mx, t[i].score);
		//	}

		//How about using some estimate of the variance. This
		// will help us figure out if a big jump occurred recently.
		int const NUM_LOOK_BACK = 10000;
		int const i_start = std::max( static_cast<int>(t.size()) - NUM_LOOK_BACK, 0);
		int const i_end = t.size() - 1;
		int const numpoints = i_end - i_start + 1;

		float x (0.0);
		for ( int i=i_start+1; i<= i_end; ++i ) {
			float const score_jump = std::abs(t[i+1] - t[i]);
			if (score_jump < 10.0 )   x += score_jump;
		}
		float const avg_jump 	= x / numpoints; //This should be of order 1 (for energy).

		//OK, now look back -- if there was a huge jump at some point,
		// ignore points in the trajectory before then.
		mn = t[ i_end ];
		mx = mn;
		int count(0);
		for ( int i = i_end-1; i >= i_start; --i ) {
			if (std::abs(t[i+1] - t[i]) > 50.0*avg_jump) break;
			mn = std::min( mn, t[i]);
			mx = std::max( mx, t[i]);
			count++;
		}

		//	mn = mean - 4*rms;
		//	mx = mean + 4*rms;

		//	std::cerr << "MIN/MAX: " << mn << " " << mx << " " << avg_jump << " " << numpoints << " " <<
		//		count << " "  << t[i_end] <<  std::endl;

	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	void
	writeStrokeString( const std::string & text_string, float xscale, float yscale) {
		char const *temp_cstring = text_string.c_str();
		glPushMatrix();
		glScalef(xscale,yscale,1.0);
		glDisable( GL_BLEND );
		if ( protein_graphics::text_aliasing ) {
			glEnable( GL_BLEND );
		}
		glDisable( GL_DEPTH_TEST );
		int len = (int) strlen(temp_cstring);
		for (int i = 0; i < len; i++) {
			glutStrokeCharacter(GLUT_STROKE_ROMAN, temp_cstring[i]);
		}
		glPopMatrix();
		glEnable( GL_BLEND );
		glEnable( GL_DEPTH_TEST );

	}

	void
	graphics_new_packer_to_misc(
			 RotamerSet const & rotamer_set,
			 FArray1D_int const & state_on_node ) {

		for (int ii = 1;ii <= rotamer_set.nmoltenres(); ++ii){
			int iiresid = rotamer_set.moltenres_2_resid(ii);
			if( iiresid > 0 ){
				int const rot =
					state_on_node(ii) + rotamer_set.rotindex_offsets(iiresid);
				misc::res(iiresid)=rotamer_set.report_aa(rot);
				misc::res_variant(iiresid)=rotamer_set.report_aav(rot);
				for ( int i=1; i <= param::MAX_ATOM(); ++i ) {
					for ( int j=1; j<=3; ++j) {
						misc::full_coord(j,i,iiresid) = rotamer_set.get_rotcoord(j,i,rot);
					}
				}
			}
		}
	}
}
#endif
