// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-

#ifndef INCLUDED_rb_class
#define INCLUDED_rb_class

#include "pose.h"
#include "loop_class.h"
#include <vector>

//////////////////////////////////////////////////////////////
//sraman Oct 17th 2007
//One rigid body segment class.
//Each rigid body segment is instantiated by segment_begin, segment_end.
//In addition, it also requires an anchor_position. This is flexible jump anchor
//position. The rigid body perturbations are effected through this jump.
//Following the rigid-body perturbation, the loops have to be closed. The loop
//stub positions in the N and C-terminals have to be specified as well. This could be
//automated later based on dssp ss description for identifying a residue as loop.
//seg_begin, seg_end, anchor_pos, loop_stub_n, loop_stub_c are specified as an rb_file
//read in during the simulation (similar to a loop file)
//////////////////////////////////////////////////////////////

namespace pose_ns {
  class Rb {

    int seg_begin_;
    int seg_end_;
    int anchor_pos_;
		int loop_stub_n_;
		int loop_stub_c_;

  public:
  Rb():
    seg_begin_(0),
      seg_end_(0),
		anchor_pos_(0),
		loop_stub_n_(0),
		loop_stub_c_(0)
	{}

    Rb(
       int const seg_begin, int const seg_end,
       int const anchor_pos, int loop_stub_n, int loop_stub_c
       ):
    seg_begin_( seg_begin ),
      seg_end_( seg_end ),
		anchor_pos_( anchor_pos ),
		loop_stub_n_( loop_stub_n ),
		loop_stub_c_( loop_stub_c )
      {}


    inline int seg_begin() const { return seg_begin_; }
    inline int seg_end() const { return seg_end_; }
    inline int anchor_pos() const { return anchor_pos_; }
		inline int loop_stub_n() const { return loop_stub_n_; }
		inline int loop_stub_c() const { return loop_stub_c_; }

    friend std::ostream & operator<<(std::ostream & is, const Rb & rb);

  };

////////////////////////////////////////////////////////////////
//sraman
//A list of rigid body segments.
//Methods to add, delete, pick segments
//set up one segment fold tree
/////////////////////////////////////////////////////////////////

  class RbSegments {
  public:
    typedef std::vector< Rb > Rb_list;
    typedef Rb_list::iterator iterator;
    typedef Rb_list::const_iterator const_iterator;

  private:
    Rb_list rb_list;

  public:
    //constructor
    RbSegments(){};

    //copy constructor
		RbSegments( const RbSegments & src ):rb_list( src.rb_list ){}
    //operator
    RbSegments & operator =( RbSegments const & src ) {
      rb_list = src.rb_list;
      return *this;
    }
		std::map < std::pair < int, int >, int > segment_anchor_map; //stores segments and their corresponding anchor positions
    friend std::ostream & operator<<( std::ostream & os, const RbSegments & rbsegments );

    inline int num_rb() const { return rb_list.size(); }

		//Following functions assume that there is only one segment in the rb_list. This is needed because rb_list is declared as private
		inline int seg_start() const;
		inline int seg_stop() const;
		inline int anchor_res() const;
		inline int n_term_loop() const;
		inline int c_term_loop() const;

		//		inline int get_anchor_position( int seg_begin, int seg_end ) { return segment_anchor_map[ std::pair < int, int >( seg_begin, seg_end ) ]; }

	void
	one_segment_fold_tree(
		pose_ns::Fold_tree & f,
		int const total_residue
	);

  void
  read_segments_from_file(
		std::string const filename
  );

  void
  add_segment(
  int const seg_begin,
  int const seg_end,
  int const anchor_pos,
	int const loop_stub_n,
	int const loop_stub_c
  );

	void
	add_segment(
	const RbSegments::iterator & it
	);

	bool
	verify_segment(
  int const seg_begin,
  int const seg_end,
  int const anchor_pos,
	int const loop_stub_n,
	int const loop_stub_c
  );

	void
	delete_segment(
	 int const seg_begin,
	 int const seg_end
	 );

	void
	delete_segment(
	 const RbSegments::iterator & it
	 );

	iterator one_random_segment();

		//	void
		//	sequential_order();

  };

	///////////////////////////////////////////////////////////////////////////
	//inline function definitions


	inline int RbSegments::seg_start() const
	{
		Rb rb( rb_list.at( 0 ) );
		return rb.seg_begin();
	}


	inline int RbSegments::seg_stop() const
	{
		Rb rb( rb_list.at( 0 ) );
		return rb.seg_end();
	}

	inline int RbSegments::anchor_res() const
	{
		Rb rb( rb_list.at( 0 ) );
		return rb.anchor_pos();
	}

	inline int RbSegments::n_term_loop() const
	{
		Rb rb( rb_list.at( 0 ) );
		return rb.loop_stub_n();
	}

	inline int RbSegments::c_term_loop() const
	{
		Rb rb( rb_list.at( 0 ) );
		return rb.loop_stub_c();
	}

}

#endif
