// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-

#include "after_opts.h"
#include "are_they_neighbors.h"
#include "files_paths.h"
#include "fold_loops.h"
#include "loop_class.h"
#include "loop_relax.h"
#include "misc.h"
#include "pose.h"
#include "pose_idealize.h"
#include "pose_io.h"
#include "pose_loops.h"
#include "random_numbers.h"
#include "rb_class.h"
#include "rb_relax.h"
#include "score.h"
#include "score_data.h"


// C++ Headers
#include <cassert>
#include <iostream>
#include <cassert>
#include <algorithm>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/ocstream.hh>
#include <utility/file/file_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin rb_relax
///
/// @brief rigid-body perturbation of a segment, close loops and relax the segment
///
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
////////////////////////////////////////////////////////////////////////////////

bool
rb_relax()
{
	using namespace pose_ns;
	using namespace files_paths;

	Pose pose, save_fullatom_pose;
	RbSegments rb_segments;

	bool const ideal_pose( false );
	bool const init_coord( true );

	set_pose_flag( true );
	pose_from_misc( pose, files_paths::input_fa, ideal_pose, init_coord );

	std::cout << "Inside rb_relax " << std::endl;

	bool const fullatom( get_fullatom_rb_flag() );

	if ( pose.fullatom() ) { //store fullatom pose to recover sidechains later
		save_fullatom_pose = pose;
		pose.set_fullatom_flag( false, false );
	}

	std::string rb_file;
	stringafteroption( "rb_file", "", rb_file );
	if ( rb_file == "" ) {
		std::cout << "No rb_file specified"
							<< "Returning without rb_relax"
							<< std::endl;
		return false;
	}

	rb_segments.read_segments_from_file( rb_file );
	segment_rb_move( pose, rb_segments, fullatom );

  return true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin segment_rb_move
///
/// @brief For each segment in the RbSegments object sets up a fold tree containing
/// two jumps and two chainbreaks
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
////////////////////////////////////////////////////////////////////////////////

void
segment_rb_move(
								pose_ns::Pose & pose,
								pose_ns::RbSegments & rb_segments,
								bool const fullatom
								)
{
	using namespace pose_ns;

	pose_ns::Fold_tree f;
	int const nres ( pose.total_residue() );

	while ( rb_segments.num_rb() > 0 ) {

		RbSegments::iterator it ( rb_segments.one_random_segment() ); //pick one random segment
		RbSegments tmp_segment;//one segment object to make one segment fold tree
		tmp_segment.add_segment( it );
		rb_segments.delete_segment( it ); //remove this segment from the list

		tmp_segment.one_segment_fold_tree( f, nres );
		pose.set_fold_tree( f );//set up fold tree for one segment

		perturb_one_segment_and_close_loops( pose, tmp_segment, fullatom ); //rigid body move and loop closure
		//		pose.simple_fold_tree( nres ); //reset fold tree

		int start_pos( tmp_segment.n_term_loop() );
		int end_pos( tmp_segment.c_term_loop() );
		refine_segment( pose, start_pos, end_pos ); //do small and shear moves to let the perturbation settle
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin perturb_one_segment_and_close_loops
///
/// @brief Perturbs segment and closes loops on both sides by ccd closure
///
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
////////////////////////////////////////////////////////////////////////////////


void
perturb_one_segment_and_close_loops(
																		pose_ns::Pose & pose,
																		pose_ns::RbSegments & tmp_segment,
																		bool const fullatom
																		)

{
	using namespace pose_ns;

	int seg_begin( tmp_segment.seg_start() );
	int seg_end( tmp_segment.seg_stop() );

	//perturb one segment
	perturb_segment( pose, seg_begin, seg_end, fullatom );

	pose.dump_pdb("perturb_segment.pdb");

	bool nterm_loop_built ( false );
	bool cterm_loop_built ( false );

	//now to close loops on both sides with ccd closure
	while ( !nterm_loop_built || !cterm_loop_built ) {

		if ( ( ran3() < 0.5 && !nterm_loop_built )  || cterm_loop_built ) { //start with n or c-term loop closure stochastically
			int loop_begin( tmp_segment.n_term_loop() );
			int loop_end( seg_begin );
			int cutpoint( seg_begin );
			std::cout << "nterm loop " << loop_begin << " " << loop_end << " " << cutpoint << std::endl;
			build_loop_with_ccd_closure( pose, loop_begin, loop_end, cutpoint, fullatom );
			nterm_loop_built = true;
			std::cout << "Built n-term loop" << std::endl;
			pose.dump_pdb("built_nterm_loop.pdb");
		} else if ( !cterm_loop_built || nterm_loop_built ) {
			int loop_begin( seg_end );
			int loop_end( tmp_segment.c_term_loop() );
			int cutpoint( seg_end );
			std::cout << "cterm loop " << loop_begin << " " << loop_end << " " << cutpoint << std::endl;
			build_loop_with_ccd_closure( pose, loop_begin, loop_end, cutpoint, fullatom );
			cterm_loop_built = true;
			std::cout << "Built c-term loop" << std::endl;
			pose.dump_pdb("built_cterm_loop.pdb");
		}
		std::cout << "n-term loop built " << nterm_loop_built << " c-term loop built " << cterm_loop_built << std::endl;

	}
	pose.dump_pdb("perturb_segment_loop_built.pdb");

}

////////////////////////////////////////////////////////////////////////////////
/// @begin random_gaussian_perturbation
///
/// @brief Perturbs all six degrees of freedom based on a gaussian distribution
/// IMPORTANT : In the present implementation, it assumes that jump 1 is the flexible jump
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
////////////////////////////////////////////////////////////////////////////////

void
random_gaussian_perturbation(
														 pose_ns::Pose & pose
														 )
{

	const float trans_delta( 0.4 );
	const float rot_delta( 0.4 );
	const int jump( 1 ); //this assumes that jump 1 is the flexible jump
	//this has to be changed for a more generalized case


	for( int i=1; i<=3; ++i ) { // first 3 are translation
		 const float rb_tmp( pose.get_jump_rb_delta( jump ,i/*rg*/,1/*dir*/) );
		 pose.set_jump_rb_delta(jump,i,1,rb_tmp+trans_delta*gaussian());
	}

	for( int i=4; i<=6; ++i ) { // next 3 are rotations
		const float rb_tmp( pose.get_jump_rb_delta(jump,i,1) );
		pose.set_jump_rb_delta(jump,i,1,rb_tmp+rot_delta*gaussian());
	}

	pose.dump_pdb("gaussian.pdb");

}

////////////////////////////////////////////////////////////////////////////////
/// @begin perturb_segment
///
/// @brief sets up score function for perturbation and calls gaussian_perturbation
///
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
////////////////////////////////////////////////////////////////////////////////

void
perturb_segment(
								pose_ns::Pose & pose,
								int & seg_begin,
								int & seg_end,
								bool const fullatom
								)
{
	using namespace pose_ns;
	Score_weight_map weight_map;
	int ncycles( 1 ); //choose a good setting for this

	FArray1D_bool allow_repack( pose.total_residue(), false );
	if ( fullatom )	select_repack_regions( pose, seg_begin, seg_end, allow_repack );

	pose.set_allow_jump_move( false );

	if ( fullatom ) {
		weight_map.set_weights( score12 );
	} else {//centroid mode : a simple combination of terms.
		weight_map.set_weight( ENV, 1.0 );
		weight_map.set_weight( PAIR, 1.0 );
		weight_map.set_weight( VDW, 1.0 );
		weight_map.set_weight( CB, 1.0 );
	}
	weight_map.set_weight( CST_SCORE, 1.0 );
	score_set_cst_mode( 3 );

	float const init_temp( 4.0 ); //is this a little too high ??
	float const final_temp( 1.0 );
	float const step( ( final_temp  - init_temp )/ncycles );

	Monte_carlo mc( pose, weight_map, init_temp );

	pose.score( weight_map );
	pose = mc.low_pose();
	mc.set_weight_map( weight_map );
	mc.reset( pose );
	show_decoy_status( pose );
	pose.dump_pdb("before_rb_perturb.pdb");
	for ( int i = 1; i<= ncycles; ++i ) {
		pose.set_allow_jump_move( 1, true ); //CAUTION: assuming that only jump 1 is flexible
		random_gaussian_perturbation ( pose ); //This should be substituted with family-specific distribution and anisotropy
		if ( fullatom ) pose.repack ( allow_repack, true );
		pose.main_minimize( weight_map, "linmin" );
		bool const accepted( mc.boltzmann( pose ) );
		std::cout << "Move accepted/rejected " << accepted << std::endl;
		mc.set_temperature( init_temp - i*step );
		pose.set_allow_jump_move( false );
	}

	//this is a big hack to ensure acceptance ******************* HACK !!!!!! *********************
	mc.reset( pose );
	pose = mc.low_pose();


}

////////////////////////////////////////////////////////////////////////////////
/// @begin select_repack_regions
///
/// @brief
///
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
///////////////////////////////////////////////////////////////////////////////
void
select_repack_regions(
											pose_ns::Pose & pose,
											int & seg_begin,
											int & seg_end,
											FArray1D_bool & allow_repack
											)
{

	using namespace pose_ns;

	for ( int i = seg_begin; i <= seg_end; ++i )
		allow_repack( i ) = true;

	//now to turn on the neighbors for repack
	const FArray3D_float & full_coord( pose.full_coord() );
	const FArray1D_int & res( pose.res() );
	float dis2;
	bool neighbors;

	for ( int i = seg_begin; i <= seg_end; ++i ) {
		for ( int j = 1; j <= pose.total_residue(); ++j ) {
			are_they_neighbors( res( i ), res( j ), full_coord( 1, 1, i ), full_coord( 1, 1, j ), dis2, neighbors );
			if ( neighbors ) allow_repack( j ) = true;
		}
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_fullatom_rb_flag
///
/// @brief
///
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
///////////////////////////////////////////////////////////////////////////////

bool
get_fullatom_rb_flag()

{
	static bool init = { false };
	static bool fullatom_rb = { false };

	if ( !init ) {
		if ( truefalseoption("fullatom_rb") ) fullatom_rb = true;
		init = true;
	}
	return fullatom_rb;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin refine_allow_segment_move_map
///
/// @brief
///
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
///////////////////////////////////////////////////////////////////////////
void
refine_segment_allow_move_map(
															pose_ns::Pose & pose,
															int const & start,
															int const & stop
															)
{
	using namespace pose_ns;
	int const nres( pose.total_residue() );

	for ( int i = 1; i <= nres; ++i ) {
		if ( i >= start && i <= stop ) {
			pose.set_allow_bb_move( i, true );
		} else {
			pose.set_allow_bb_move( i, false );
		}
	}
}
////////////////////////////////////////////////////////////////////////////////
/// @begin refine_segment
/// This essentially mimics a fullatom relax of the perturbed region. This calls Chu's
/// loop_refinement routine. The cutpoints are at the end of the segment. The loop closure
/// is done after every small/shear move.
/// @brief
///
/// @detailed
///
/// @param
/// @param
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Oct 17th 2007
////////////////////////////////////////////////////////////////////////////
void
refine_segment(
							 pose_ns::Pose & pose,
							 int & start_pos,
							 int & end_pos
							 )
{
	using namespace pose_ns;

	int nres( pose.total_residue() );

	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4L );
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );

	int cutpoint( 0 );
	int const offset( 0 );
	bool const extended( false );

	pose_ns::Fold_tree f;
	pose_ns::Loops one_loop;

	if ( ran3() > 0.5 )
		cutpoint = start_pos;
	else cutpoint = end_pos;

	cutpoint = start_pos;

	std::cout << "Segment refine Cut point " << cutpoint << std::endl;

	one_loop.add_loop( start_pos, end_pos, start_pos, offset, extended );
	pose_loops_build_fold_tree( nres, one_loop, f );
	pose.set_fold_tree( f );
	pose.set_allow_jump_move( false );
	pose_loops_set_allow_bb_move( pose, one_loop );

	pose.dump_pdb("after_new_fold_tree.pdb");
	std::cout << "New fold tree for segment refinement " << f << std::endl;

	pose_refine_loops_with_ccd( pose, weight_map, one_loop );
	pose.dump_pdb("after_pose_refine.pdb");


}
