// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13191 $
//  $Date: 2007-03-01 08:44:35 -0800 (Thu, 01 Mar 2007) $
//  $Author: leaverfa $


// Rosetta Headers
#include "recover.h"
#include "aa_name_conversion.h"
#include "current_pose.h"
#include "disulfides.h"
#include "docking_movement.h"
#include "fullatom.h"
#include "fullatom_energy.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "loops.h"
#include "maps.h"
#include "misc.h"
#include "monte_carlo.h"
#include "namespace_low_pose.h"
#include "nblist.h"
#include "param.h"
#include "refold.h"
#include "score.h"
#include "status.h"
#include "template_pack.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Dp.hh>


////////////////////////////////////////////////////////////////////////////////
/// @begin update_sequence
///
/// @brief
///car update all sequence-dependent arrays to be consistent
///car with 'res' and 'res_variant' arrays.
///
/// @detailed
///
/// @global_read
///   res         misc.h
///   res_variant misc.h
///
/// @global_write
///   residue3    misc.h
///   residue1    misc.h
///
/// @remarks
///car CALL THIS FUNCTION EVERY TIME THE SEQUENCE CHANGES!!
///
/// multiple indirect changes to global variables
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_sequence(
	bool const update_allatom_list // = true
)
{
	using namespace misc;

	for ( int i = 1; i <= total_residue; ++i ) {
		name_from_num( res(i), residue3(i) );
		res1_from_num( res(i), residue1(i) );
	}

	setup_atom_type(); // for vdw_compute
	disulfides::BOUNDARY::identify_cys(); // locate cysteines for disulfides
	if ( update_allatom_list ) {
		if ( pose_flag() || refold_check_current_pose() ) return;
		setup_allatom_list(); // for nblist and derivative
		score_set_new_pose(); //Objexx:SGM As per Brian Kuhlman
	}
}

///////////////////////////////////////////////////////////////////////////////
bool
same_sequence(
	int const nres,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray1D_int const & new_res,
	FArray1D_int const & new_res_variant
	)
{
	bool match( true );
	for ( int i=1; i<= nres; ++i ) {
		if ( res(i) != new_res(i) ||
				 res_variant(i) != new_res_variant(i) ) {
			match = false;
			break;
		}
	}
	return match;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin resetphipsi
///
/// @brief copies the best_pose (except coordinates) back into current_pose
///
/// @detailed resetphipsi effectively 'unmakes' a backbone modification
///
/// @global_read
///   best_pose  (except coordinates)  block misc.h
///
/// @global_write
///   current_pose (except coordiantes) block misc.h
///
/// @remarks
///car   coordinates are not updated because in general, the current coordinates
///car   are not updated until the structure is ready for evaluation; coordinates
///car   are generated from the best coordinates and the current angles
///car
///car   also resets frag_begin,frag_end, and new_rotamer arrays to indicate
///car   that best_pose and current_pose are identical
///
///car indirect changes to global variables
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
resetphipsi()
{
	using namespace misc;

	phi = best_phi;
	psi = best_psi;
	omega = best_omega;
	secstruct = best_secstruct;
	name = best_name;
	if ( ! same_sequence( total_residue, res, res_variant,
												best_res, best_res_variant ) ) {
		res = best_res;
		res_variant = best_res_variant;
		update_sequence();
	}
	reset_move_map();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_best_pose
///
/// @brief copies the best_pose into the current_pose
///
/// @detailed
///
/// @global_read
///   best_pose  block misc.h
///   best_hetero_atom_coord ligand.h
///
/// @global_write
///   current_pose  block misc.h
///   hetero_atom_coord ligand.h
///
/// @remarks
///car  retrieve_best_pose is a rapid method of regenerating coordinates
///car  for the best_pose; it is called when then starting coordinates are
///car  needed in order to make a new move (rather than just the starting
///car  torsion angles
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_best_pose()
{
	using namespace ligand;
	using namespace misc;
	using namespace param;

	resetphipsi();

	Eposition = Ebest_position;
	centroid = best_centroid;
	centroid_parm_eachres = best_centroid_parm_eachres;

//car recover full_coord
	if ( get_fullatom_flag() ) {
		full_coord = best_full_coord;
	}

//mj recover ligand coordinates
	if ( get_ligand_flag() ) {
		ligand_one->recover_best_coordinates();
	}

//car retrieve relative rotation and translation of chains
	docking_retrieve_best();
	loop_retrieve_best_pose();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin recover_LOW
///
/// @brief restore monte carlo search to the lowest scoring structure
///
/// @detailed
///car recover structure stored in LOW arrays, updates all current arrays and
///car internal arrays of scoring functions, and resets monte carlo
///
/// @param[in]   scoring_function - in -  external
///
/// @global_read
///  total_residue misc.h
///
/// @global_write
///  score  misc.h
///
/// @remarks
///car many indirect changes to global variables
///
/// @references
///
/// @authors  car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
recover_LOW( Scoring_Function scoring_function )
{
	using namespace misc;

	retrieve_low_pose();

//car complete refold here isn't strictly necessary, but include to
//car help eliminate accumulated roundoff errors
	monte_carlo_accept_best(); // put rotamers in best before refold
	refold(1,total_residue);

//car tell scorefxns there's a new structure in arrays...
	score_set_new_pose();
	score_enable_rotamer_trials(false); // disable rotamer trials
	mc_global_track::mc_score::score = scoring_function();
	score_enable_rotamer_trials(true);
	save_status_info("recover_LOW",0,0);
	monte_carlo_reset(); // forces low and best accept
}

////////////////////////////////////////////////////////////////////////////////
/// @begin new_score_function
///
/// @brief switch to a new scoring function in monte carlo search
///
/// @detailed
///car resets monte carlo for a new scoring function without recovering
///car the low_pose
///
/// @param[in]   scoring_function - in - external
///
/// @global_read
///   total_residue misc.h
///
/// @global_write
///   score  misc.h
///   low_score misc.h
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
new_score_function( Scoring_Function scoring_function )
{

//car get low pose, score it and save score
	retrieve_low_pose();

//car no refold or repack ...

	score_enable_rotamer_trials(false); // disable

//car tell scorefxns there's a new structure in arrays...
	score_set_new_pose();
	mc_global_track::mc_score::low_score = scoring_function(); // save its score

//car return to best pose
//car best pose still unchanged (both bb and sidechains)
	retrieve_best_pose();

	score_set_new_pose();
	mc_global_track::mc_score::score = scoring_function();
	score_enable_rotamer_trials(true);

// since scorefxn weight may have changed and we're not doing a mc reset
// need to ensure that all scoring functions that need updating are current
	mc_update_scorefxns();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_low_pose
///
/// @brief copy the low_pose into the current_pose
///
/// @detailed
///
/// @global_read
///  low_pose  block static
///  total_residue misc.h
///  low_hetero_atom_coord namespace_low_pose.h
///
/// @global_write
///  current_pose block misc.h
///  hetero_atom_coord ligand.h
///
/// @remarks
///
/// @references
///
/// @authors car 8/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_low_pose()
{
	using namespace ligand;
	using namespace low_pose;
	using namespace misc;
	using namespace param;

	phi = low_phi;
	psi = low_psi;
	omega = low_omega;
	secstruct = low_secstruct;
	name = low_name;
	Eposition = Elow_position;
	centroid = low_centroid;
	centroid_parm_eachres = low_centroid_parm_eachres;
	if ( ! same_sequence( total_residue, res, res_variant,
												low_res, low_res_variant ) ) {
		res = low_res;
		res_variant = low_res_variant;
		update_sequence();
	}

//car recover full_coord
	if ( get_fullatom_flag() ) {
		full_coord = low_full_coord;
	}

//mj recover ligand coordinates
	if ( get_ligand_flag() ) {
		for ( size_t k = 0; k < ligand::ligand_one->atom_vector.size(); ++k ) {
			if (ligand::ligand_one->atom_vector.size()==low_hetero_atom_coord.size()){
				ligand::ligand_one->atom_vector[k]->set_coordinates(low_hetero_atom_coord[k]);
			}else{
			std::cerr << "low_hetero_atom_coord and atom_vector have different sizes." << std::endl;
			}
		}
	}

	docking_retrieve_low();
	loop_retrieve_low_pose();

	maps_set_new_pose();
}
