// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14630 $
//  $Date: 2007-04-29 16:06:13 -0700 (Sun, 29 Apr 2007) $
//  $Author: rhiju $


// Rosetta Headers
#include "refold.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "angles.h"
//#include "bonds.h"
#include "current_pose.h"
#include "files_paths.h"
#include "fullatom.h"
#include "input_pdb.h"
#include "jumping_refold.h"
#include "jumping_util.h"
#include "loops.h"
#include "maps.h"
#include "misc.h"
#include "native.h"
#include "namespace_fullatom_flag.h"
#include "orient_rms.h"
#include "param.h"
#include "pose.h"
#include "recover.h"
#include "refold_ns.h"
#include "runlevel.h"
#include "start.h"
#include "termini.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>


////////////////////////////////////////////////////////////////////////////////
//car  based on GL_angles.cc obtained from Charlie E. M. Strauss (2001)
//
// functions: lowest level to highest
//     refold_coord_sys()      define coordinate system for building next atom
//     build_atom()         build atom from dihedral, bond_angle, bond-length
//
//     get_GL_matrix()      defines transform of three points into three points
//     GL_rotate()          apply GL rotation and offset matrix to vector list
//     GL_rot()             apply GL rot/off matrix to a single vector
//
//     build_residue_n2c(ires,...) fold atoms defined by phi,psi,omega of ires
//                                    c,cb,cen,n(i+1),o,ca(i+1)
//     build_residue_c2n(ires,...) fold atoms defined by phi,psi,omega of ires
//                                    ca,o,n,cb,cen,c(i-1)
//
//     build_backbone(dir,stub,end ...) generate coordinates from angles for
//                               1->end in direction specified, building
//                               off stub provided
//
//     refold_update_coords(dir,begin,end,reg_end...)
//                               generate new position array from reg_begin
//                               to reg_end; begin->end generated from torsion
//                               angles; outside this region, coords are
//                               copied or transformed
//
//     refold(begin,end)         completely update coordinates using
//                               misc.h variables
////////////////////////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////////////////////////
/// @begin refold_set_direction
///
/// @brief force refold to use a desired folding direction, or autoselect
///
/// @detailed
///
/// @param[in]   dir_x - in - 1 = n2c, -1 = c2n, 0=autoselect
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_set_direction( int dir_x )
{
// force refold direction is a signal to refold that tells it
// which direction it should fold the chain.
//    1--fold N->C  (ie transform coord of residues C-term to insertion)
//    0--Automatic: choose shortest direction for update.  this is default
//   -1--fold C->N   (ie transform coord of residues N-term to insertion)
	refold_ns::refold_mat::force_refold_dir = dir_x;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_get_dir
///
/// @brief retrieve value of refolding direction dir_x static variable
///
/// @detailed
///
/// @param[out]   dir_x - out -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_get_dir( int & dir_x )
{
// method returns direction of last refold

	dir_x = refold_ns::refold_mat::refold_dir;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold
///
/// @brief update all coordinates in position,centoid,and full_coord
///  arrays using info in phi,psi,omega, best_arrays, and folding segments
///
/// @detailed
///
/// @param[in]   begin_res - in - first res for which angles differ from best_angles
/// @param[in]   end_res - in - last residue for which angles differ from best_anglest
///
/// @global_read
///   misc.h
/// @global_write
///   misc.h
///
/// @remarks
///
///car for full_coord array, refold generates coordinates as follows:
///car N,CA,C,O  - transformed from best_fullcoord or copied from position
///car HN  transformed from best_fullcoord or built from scratch
///car sidechain/HA atoms - transformed from best_full_coord
///
///car handles special cases of folding discontiguous segments
///
///car handles special case of fixed regions in the chain that should
///car not be refolded from angles, but copied directly from the best_arrays
///car WARNING!!  Assumes that there is only one contiguous variable region
///car per segment
///
///   1) region begin_res->end_res is refolded from scratch using the phi-psi
///      array using begin_res-1 in the best arrays as an anchor point.
///   2) the shorter chain portion 1-begin_res or begin_res->end is rotated and
///      offset to align with the newly folded section.
///   3) residues neither refolded nor transformed are copied from best_position
///      to position.
///   4) rotamers are updated to the new coordinates using best_full_coord
///      as the rotamer source
///
///------------------------------------------------------------------------------
/// INITIALIZING THIS ROUTINE:
///  1) call refold(1,total_residue)
///  2) force a monte_carlo accept to set the best_position array.
///
/// Notes on function:
/// 1) begin_res->end_res tell this function the range of residues for which
///    the phi psi angles have changed.  Outside of this range it is assumed
///    the the phi-psi angles have not changed since the last time a
///    monte-carlo accept occured.  (NOT the last time refold was called.)
///
/// 2) scoring functions assume that the best_position and best_centroid position
///    arrays are consistent with the phi psi array except as indicated by
///    pair_moved, res_moved.  refold must notify maps about all changes that
///    are made to the position array
///
/// 3) best_position is the source, position is the destination of all
///    coordinates. best_position is _not_ changed by this function
///    only by monte_carlo when a move is accepted. if this function is
///    called 2x (w/out a monte_carlo accept) the position array will reflect
///    only the phi-psi changes in the last call.
///------------------------------------------------------------------------------
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold(
	int begin_res,
	int end_res
)
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;
	using namespace termini_ns;

//car local
	int seg_begin,seg_end;
	int fold_begin,fold_end,dir;
	int seg_length;
	FArray1D_bool allow_insert( MAX_RES()() );
	float orient_rms;
	FArray3D_float refold_orient_position( 3, MAX_POS, MAX_RES()() );

	if ( refold_check_current_pose() ) {
		std::cout << "standard refold is not compatible with pose" <<
			std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//car idiot hunt:
	if ( begin_res > end_res ) {
		if (runlevel > standard) {
			std::cout << "WARNING!! refold called with begin_res>end_res"
								<< std::endl;
			std::cout << "          retrieving best pose..." << std::endl;
		}
		retrieve_best_pose();
		return;
	}

//rhiju Allow centroids to move, too. For now, assign centroid positions based
//  on sequence and Rosetta's default parameters.
	if ( get_use_native_centroid_flag() ) {
		//  place centroids at actual centers of mass (relative to backbone) of
		//      side chains from native structure.
		get_centroid_parm_eachres_fromnative( centroid_parm_eachres);
	}	else {
		// default: place centroids according to the average locations of each
		// amino acid type's centroid relative to backbone in PDB.
		get_centroid_parm_eachres_default( centroid_parm_eachres, res);
	}


	bool const fullatom = get_fullatom_flag();
	int seg1 = identify_segment(1);
	int seg2 = identify_segment(total_residue);

//car fold seg1->seg2
	retrieve_allow_insert(allow_insert,total_residue);
	for ( int i = seg1; i <= seg2; ++i ) {
		retrieve_segment_ends(i,seg_begin,seg_end);
		fold_begin = std::max(seg_begin,begin_res);
		fold_end = std::min(end_res,seg_end);

//car look for variable residues
//car WARNING:: assume there is only one contiguous variable region per segment
//car WARNING:: !!not the best way to handle this problem!!
		while ( !allow_insert(fold_begin) ) {
			++fold_begin;
			if ( fold_begin >= seg_end ) goto L300;
			if ( fold_begin > total_residue ) {
				std::cout << "STOP: Error in refold, unable to find fold_begin" << std::endl;
				std::cout << "fold_begin:" << SS( fold_begin ) <<
				 "total_residue:" << SS( total_residue ) <<
				 "seg_end:" << SS( seg_end ) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
L300:
		while ( !allow_insert(fold_end) ) {
			--fold_end;
			if ( fold_end <= seg_begin ) goto L200;
			if ( fold_end < 1 ) {
				std::cout << "STOP: Error in refold, unable to find fold_end" << std::endl;
				std::cout << "fold_end:" << SS( fold_end ) <<
				 "seg_begin:" << SS( seg_begin ) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

L200:
		refold_select_dir(i,fold_begin,fold_end,dir);

 		refold_update_coords(dir,fold_begin-seg_begin+1,fold_end-seg_begin+1,
			seg_end-seg_begin+1, seg_begin, phi(seg_begin),psi(seg_begin),omega(seg_begin),
			res(seg_begin),res_variant(seg_begin),is_N_terminus(seg_begin),
			is_C_terminus(seg_begin),Ebest_position(1,1,seg_begin),
			Eposition(1,1,seg_begin),best_centroid(1,seg_begin),centroid(1,seg_begin),
			fullatom,best_full_coord(1,1,seg_begin),full_coord(1,1,seg_begin),
			centroid_parm_eachres(1,-1,seg_begin));

		maps_set_new_phipsi(fold_begin,fold_end,seg_begin,seg_end,dir);

//glb reorient chain to native position if idealizing multiple chains simultaneously
		if ( refold_get_orient_flag() ) {
			refold_get_orient_position(refold_orient_position);
			seg_length = seg_end - seg_begin + 1;
			orient_region( seg_length, Eposition(1,1,seg_begin),
			 full_coord(1,1,seg_begin), refold_orient_position(1,1,seg_begin),
			 1, 1, seg_length, 1, 1, seg_length, orient_rms );
		}
		calculate_segment_overlap(i,fold_begin,fold_end,seg_begin,seg_end,dir);

	}

	if ( fullatom ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( std::abs( Eposition(1,2,i) - full_coord(1,2,i) ) > .001 ) {
				std::cout << "WARNING!!! position and full_coord disagree!" << std::endl;
				std::cout << SS( i ) << SS( Eposition(1,2,i) ) <<
				 SS( full_coord(1,2,i) ) << std::endl;
				std::cout << SS( i ) << SS( Ebest_position(1,2,i) ) <<
				 SS( best_full_coord(1,2,i) ) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_set_orient_flag
///
/// @brief - set true to reorient chain to coords in refold_orient_Eposition
///         during refold (glb)
///
/// @detailed
///
/// @param  [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_set_orient_flag( bool const yes_no )
{
	refold_ns::orient_flag_bool::refold_orient_flag = yes_no;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_get_orient_flag
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
refold_get_orient_flag()
{
	return refold_ns::orient_flag_bool::refold_orient_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_select_dir
///
/// @brief determine folding direction for a particular segment
///
/// @detailed selects folding direction for a segment based on region to be
/// generated from angles, the setting of the force_refold_dir private
/// static, and loop setting for segmental folding
///
/// @param[in]   seg_num - in - segment number
/// @param[in]   fold_begin - in - first residue to generate from angles
/// @param[in]   fold_end - in - last residue to generate from angles
/// @param[out]   dir - out - folding direction
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_select_dir(
	int const seg_num,
	int const fold_begin,
	int const fold_end,
	int & dir
)
{
	using namespace refold_ns::refold_mat;
	using namespace refold_ns::refold_segments;

//car parameters
	int const n2c = { 1 };
	int const c2n = { -1 };

	dir = force_refold_dir;

	if ( dir == 0 ) dir = seg_dir(seg_num);

	if ( dir == 0 ) {
		dir = ( (seg_end(seg_num) - fold_end) <=
		 (fold_begin - seg_begin(seg_num))  ? n2c : c2n );
	}
	refold_dir = dir; // save in namespace
}

/////////////////////////////////////////////////////////////////////////
/// @begin refold_set_segments
///
/// @brief set up segments for folding
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 3/16/2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////
void
refold_set_segments(
	int const first_seg,
	int const last_seg,
	FArray1Da_int begin,
	FArray1Da_int end,
	FArray1Da_int dir
)
{
	using namespace refold_ns::refold_segments;

	begin.dimension( SRange( first_seg, last_seg ) );
	end.dimension( SRange( first_seg, last_seg ) );
	dir.dimension( SRange( first_seg, last_seg ) );

	for ( int i = first_seg; i <= last_seg; ++i ) {
		seg_begin(i) = begin(i);
		seg_end(i) = end(i);
		seg_dir(i) = dir(i);
		if ( seg_dir(i) > 1 || seg_dir(i) < -1 ) {
			std::cerr << "ERROR: folding direction must be -1, 0, or 1 ! \n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for ( int j = seg_begin(i), je = seg_end(i); j <= je; ++j ) {
			seg_map(j) = i;
		}
	}
	first_segment = first_seg;
	last_segment = last_seg;
}




/////////////////////////////////////////////////////////////////////////
/// @begin refold_adjust_segment_ends
///
/// @brief adjust segment ends to handle special cases
///
/// @detailed  changes segment end values without modifying the seg_map
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// Segments currently are required to have a fixed anchor residue
/// and only one movable region. Consequently, there are special cases where
/// a single residue may have to serve as an anchor for two adjacent segments
/// e.g. residue 1-4 is a loop, residue 5 fixed, residue 6-10 is a loop.
/// To fold segment 1, residues 1-5 must be folded. To fold segment 2,
/// residues 5-10 must be folded.  Residue 5 'belongs' to only 1 segment (ie
/// to fold residue 5, only segment 1 or segment 2 must be folded) so segmap(5)
/// has a single value, but segment 1 and segment 2 overlap in the sense that
/// seg_end(1)=seg_begin(2)
///
/// @references
///
/// @authors car 4/4/2004
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////
void
refold_adjust_segment_ends(
	int const first_seg,
	int const last_seg,
	FArray1Da_int begin,
	FArray1Da_int end,
	FArray1Da_int dir
)
{
	using namespace refold_ns::refold_segments;

	begin.dimension( SRange( first_seg, last_seg ) );
	end.dimension( SRange( first_seg, last_seg ) );

	if ( first_segment != first_seg || last_segment != last_seg ) {
			std::cerr << "mismatch in refold segments \n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	for ( int i = first_seg; i <= last_seg; ++i ) {
		seg_begin(i) = begin(i);
		seg_end(i) = end(i);
		seg_dir(i) = dir(i);
		if ( seg_dir(i) > 1 || seg_dir(i) < -1 ) {
			std::cerr << "ERROR: folding direction must be -1, 0, or 1 ! \n";
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin identify_segment
///
/// @brief
//car return the segment number that should be refolded in order to generate
//car coordinates for residue
///
/// @detailed
///
/// @param  residue - [in/out]? -
/// @param  seg_num - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
identify_segment(
	int const residue
)
{

	using namespace refold_ns::refold_segments;
	using namespace misc;

	if ( residue < 1 || residue > total_residue) return -1;
	return seg_map(residue);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_segment_ends
///
/// @brief
///car returns seg_begin, seg_end which indicate the complete region
///car that should be folded to generate coordinates for segment i
///
/// @detailed
///
/// @param  seg_num - [in/out]? -
/// @param  begin - [in/out]? -
/// @param  end - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_segment_ends(
	int const seg_num,
	int & begin,
	int & end
)
{

	using namespace misc;
	using namespace refold_ns::refold_segments;

	if ( seg_num < first_segment ||seg_num > last_segment ) {
		begin = total_residue + 1;
		end = 0;
		return;
	}

	begin = seg_begin(seg_num);
	end = seg_end(seg_num);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_set_orient_position
///
/// @brief sets the thether array for use in other places
///
/// @detailed
///
/// @param[in]   thether_pos - in - coords of orient
///
/// @global_read - none
///
/// @global_write - none
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_set_orient_position(
	int const total_residue,
	FArray3Da_float orient_position
)
{
	using namespace param;
	using namespace refold_ns::refold_orient_pos;

	orient_position.dimension( 3, MAX_POS, total_residue );

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= MAX_POS; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				refold_orient_position(k,j,i) = orient_position(k,j,i);
			}
		}
	}
	orient_total_residue = total_residue;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_get_orient_position
///
/// @brief get the thether array
///
/// @detailed
///
/// @param[out]   orient_position - out - coords of orient
///
/// @global_read - none
///
/// @global_write - none
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_get_orient_position( FArray3Da_float orient_position /* native backbone */ )
{
	using namespace param;
	using namespace refold_ns::refold_orient_pos;

	orient_position.dimension( 3, MAX_POS, MAX_RES() );

	for ( int i = 1; i <= orient_total_residue; ++i ) {
		for ( int j = 1; j <= MAX_POS; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				orient_position(k,j,i) = refold_orient_position(k,j,i);
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin use_st_bond
///
/// @brief
///
/// @detailed
///c
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
use_st_bond()
{
	// we don't want to ever use st-bond stuff if we are
	// refolding a pose. There could be st-bond stuff if we
	// started in misc-world and then started dealing with a pose
	return !refold_check_current_pose() && refold_ns::st_bond_switch::st_bond;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_update_coords
///
/// @brief  updates coordinates from reference coordinates and torsion angles
///
/// @detailed generates position array and full_coord array; begin_frag to end_frag
/// is generated from torsion angles; if dir=1 (n2c),1 to begin_frag-1 is
/// copied from the position array, and end_frag+1 to end_region is generated
/// by rotating the coordinates in best_position to overlap at end_frag.
/// if dir = -1 (c2n), end_region to end_frag+1 is copied from the best_position
/// array and 1 to begin_frag-1 is rotated to superimpose at begin_frag.
///
/// @param[in]   dir - in -  direction to fold
/// @param[in]   begin_frag - in - start of region to generate from angles
/// @param[in]   end_frag - in - end of region to generate from angles
/// @param[in]   end_region - in - end of region to generate coordinates for
/// @param[in]   phi - in - phi angles for 1 to end_region
/// @param[in]   psi - in - psi angles for 1 to end_region
/// @param[in]   omega - in - omega angles for 1 to end_region
/// @param[in]   res - in - sequence of 1 to end_region
/// @param[in]   resv - in - sequence variants of 1 to end_region
/// @param[in]   Ebest_position - in - reference backbone coords of 1 to end_region
/// @param[out]   Eposition - out - computed backbone coords of 1 to end_region
/// @param[in]   best_centroid - in - reference centroid coords of 1 to end_region
/// @param[out]   centroid - out - updated centroid coords of 1 to end_region
/// @param[in]   fullatom - in - should full_coord array be updated?
/// @param[in]   best_full_coord - in - reference fullatom coords of 1 to end_region
/// @param[out]   full_coord - out - updated  fullatom coords of 1 to end_region
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  note that begin_frag,end_frag must be numbered with respect to the
///  segment being passed into this function
///
///  note that HN of the first residue of a segment passed to refold_update_coord
///  is place 1A out along N CA bond!!
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_update_coords(
	int dir,
	int begin_frag,
	int end_frag,
	int end_region,
	int seg_begin,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega,
	FArray1Da_int res, // residue types
	FArray1Da_int resv, // residue variants
	FArray1Da_bool N_term, // Nterm
	FArray1Da_bool C_term, // Cterm
	FArray3Da_float Ebest_position,
	FArray3Da_float Eposition,
	FArray2Da_float best_centroid,
	FArray2Da_float centroid,
	bool fullatom,
	FArray3Da_float best_full_coord,
	FArray3Da_float full_coord
)
{
	//This is a wrapper around the previous refold_update_coords...
	// If you want to move the centroids from their standard positions,
	// call next function directly with "centroid_parm_eachresidue". Otherwise,
	// we here assign default centroids, based on standard Rosetta parameters..
	FArray3D_float centroid_parm_eachres( 5, DRange(-1, 1), end_region, 0.0f);
	get_centroid_parm_eachres_default( centroid_parm_eachres, res);
	refold_update_coords( dir, begin_frag, end_frag, end_region, seg_begin,
		phi, psi, omega, res, resv, N_term, C_term, Ebest_position, Eposition,
		best_centroid, centroid, fullatom, best_full_coord, full_coord,
		centroid_parm_eachres);
}
void
refold_update_coords(
	int dir,
	int begin_frag,
	int end_frag,
	int end_region,
	int seg_begin,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega,
	FArray1Da_int res, // residue types
	FArray1Da_int resv, // residue variants
	FArray1Da_bool N_term, // Nterm
	FArray1Da_bool C_term, // Cterm
	FArray3Da_float Ebest_position,
	FArray3Da_float Eposition,
	FArray2Da_float best_centroid,
	FArray2Da_float centroid,
	bool fullatom,
	FArray3Da_float best_full_coord,
	FArray3Da_float full_coord,
	FArray3Da_float centroid_parm_eachres
)
{
	using namespace aaproperties_pack;
	using namespace param;

	phi.dimension( end_region );
	psi.dimension( end_region );
	omega.dimension( end_region );
	res.dimension( end_region );
	resv.dimension( end_region );
	N_term.dimension( end_region );
	C_term.dimension( end_region );
	Ebest_position.dimension( 3, MAX_POS, end_region );
	Eposition.dimension( 3, MAX_POS, end_region );
	best_centroid.dimension( 3, end_region );
	centroid.dimension( 3, end_region );
	best_full_coord.dimension( 3, MAX_ATOM(), end_region );
	full_coord.dimension( 3, MAX_ATOM(), end_region );
	centroid_parm_eachres.dimension( 5, DRange(-1, 1), end_region);

	int const n2c = { 1 };
	// int const c2n = { -1 };
	// c2n removed to avoid compiler warning since it is not used
	// when not in debug mode (see assert statement below)

	int const c = { 4 };
	int const n = { 1 };
	int const ca = { 2 };

// local
	FArray2D_float Mgl( 4, 4 );
	FArray1D_float c_xyz( 3 );
	FArray1D_float n_xyz( 3 );
	FArray1D_float ca_xyz( 3 );
	int fold_start, fold_end, update_start, update_end;

	if ( dir == n2c ) {
		fold_start = begin_frag;
		fold_end = end_frag;
		update_start = 1;
		update_end = end_region;
	} else {
		assert( ( dir == -1 ) ); //!!! should be c2n instead of -1!!!
		fold_end = begin_frag;
		fold_start = end_frag;
		update_end = 1;
		update_start = end_region;
	}

//car if there is no rebuilding to do, make sure we copy, not rotate
	bool refold_is_needed = true;
	if ( begin_frag > end_region || end_frag < 1 || begin_frag > end_frag ) {
		fold_end = update_end + dir;
		fold_start = update_end + dir;
		refold_is_needed = false;
	}

//car copy all residues that are not going to move
	for ( int i = update_start, e = (fold_start-dir)*dir; i*dir <= e; i += dir ) {
		copyXYZ(1,5,Ebest_position(1,1,i),Eposition(1,1,i));
		copyXYZ(1,1,best_centroid(1,i),centroid(1,i));
	}
	if ( fullatom ) {
		for ( int i = update_start, e = (fold_start-dir)*dir; i*dir <= e; i += dir ) {
			copyXYZ(1,natoms(res(i),resv(i)),best_full_coord(1,1,i),
			 full_coord(1,1,i));
		}
	}

	if ( ! refold_is_needed) return; //chu  --- car: why is this needed?

//car build region from angles
	if ( end_frag - begin_frag >= 0 ) {
//car find stub atoms for newly folded region
		int stub_res;
		if ( dir == n2c ) {
			stub_res = fold_start - 1;
		} else {
			stub_res = fold_start;
		}
		if ( stub_res == 0 || stub_res == end_region ) {
			refold_default_stub(c_xyz,n_xyz,ca_xyz,dir);
		} else {
			for ( int i = 1; i <= 3; ++i ) {
				c_xyz(i) = Ebest_position(i,c,stub_res);
				n_xyz(i) = Ebest_position(i,n,stub_res+1);
				ca_xyz(i) = Ebest_position(i,ca,stub_res+1);
			}
		}

//car rebuild residues with new phi/psi/omg
		build_backbone(dir,c_xyz,n_xyz,ca_xyz,end_frag-begin_frag+1,phi(begin_frag),
		 psi(begin_frag),omega(begin_frag),res(begin_frag),
		 Eposition(1,1,begin_frag),centroid(1,begin_frag),
		 c_xyz,n_xyz,ca_xyz,seg_begin+begin_frag-1, centroid_parm_eachres(1,-1,begin_frag) );
	}
//car rotate the remainder of chain to graft it onto end we just built
	if ( ( update_end - fold_end - dir ) * dir >= 0 ) {
		int const stub_res = std::max( fold_end, fold_end + dir ); // begin_frag or end_frag+1
//car calculate rotation matrix
		get_GL_matrix(Ebest_position(1,c,stub_res-1),Ebest_position(1,n,stub_res),
		 Ebest_position(1,ca,stub_res), /* where we are now */ c_xyz,n_xyz,ca_xyz,
		 /* where we want to go */ Mgl); // output transform matrix

//car rotate
		for ( int i = fold_end+dir, e = update_end*dir; i*dir <= e; i += dir ) {
			GL_rotate(5,1,Mgl,Ebest_position(1,1,i),Eposition(1,1,i));
			GL_rotate(1,1,Mgl,best_centroid(1,i),centroid(1,i));
		}
		if ( fullatom ) {
			for ( int i = fold_end+dir, e = update_end*dir; i*dir <= e; i += dir ) {
				GL_rotate(natoms(res(i),resv(i)),1,Mgl,best_full_coord(1,1,i),
				 full_coord(1,1,i));
			}
		}
	}

//car fill in rest of fullcoord array in build region
//car note: do this last because placing the terminal HN relies on
//car coordinates outside begin_frag,end_frag and we want to ensure that
//car they are updated.
	if ( fullatom ) build_fullcoord(Eposition,begin_frag,end_frag,res,resv,
	 N_term,C_term,phi(begin_frag),best_full_coord,full_coord);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin build_backbone
///
/// @brief  generate coordinates for residues 1 to end_frag from torsion angles
///
/// @detailed
///car given starting position c_xyz,n_xyz,ca_xyz
///car ending position is c2_xyz,n2_xyz,ca2_xyz
///car coordinates from 1-end_frag in Eposition and centroid are changed
///
/// @param[in]   dir - in - direction to fold: n2c or c2n
/// @param[in]   c1_xyz - in - coords of starting stub carbon
/// @param[in]   n1_xyz - in - coords of starting stub nitrogen
/// @param[in]   ca1_xyz - in - coords of starting stub calpha
/// @param[in]   end_frag - in - logical size of arrays, number of residues to fold
/// @param[in]   phi - in - phi angles of 1 to end_frag
/// @param[in]   psi - in - psi angles of 1 to end_frag
/// @param[in]   omega - in - omega angles of 1 to end_frag
/// @param[in]   res - in - residue type of 1 to end_frag
/// @param[out]   Eposition - out - backbone coordinates
/// @param[out]   centroid - out - centroid coordinates
/// @param[out]   c2_xyz - out -  coords of ending stub carbon
/// @param[out]   n2_xyz - out - coords of ending stub nitrogen
/// @param[out]   ca2_xyz - out - coords of ending stub calpha
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors 10/13/2003  car
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_backbone(
	int dir,
	FArray1Da_float c1_xyz,
	FArray1Da_float n1_xyz,
	FArray1Da_float ca1_xyz,
	int end_frag,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega,
	FArray1Da_int res,
	FArray3Da_float Eposition,
	FArray2Da_float centroid,
	FArray1Da_float c2_xyz,
	FArray1Da_float n2_xyz,
	FArray1Da_float ca2_xyz,
	int const begin_offset
)
{
	//This is a wrapper around the previous refold_update_coords...
	// If you want to move the centroids from their standard positions,
	// call next function directly with "centroid_parm_eachresidue". Otherwise,
	// we here assign default centroids, based on standard Rosetta parameters..
	FArray3D_float centroid_parm_eachres( 5, DRange(-1, 1), end_frag, 0.0f);
	get_centroid_parm_eachres_default( centroid_parm_eachres, res);
	build_backbone(dir, c1_xyz, n1_xyz, ca1_xyz, end_frag, phi, psi, omega, res, Eposition,
							 centroid, c2_xyz, n2_xyz, ca2_xyz, begin_offset, centroid_parm_eachres);
}


void
build_backbone(
	int dir,
	FArray1Da_float c1_xyz,
	FArray1Da_float n1_xyz,
	FArray1Da_float ca1_xyz,
	int end_frag,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega,
	FArray1Da_int res,
	FArray3Da_float Eposition,
	FArray2Da_float centroid,
	FArray1Da_float c2_xyz,
	FArray1Da_float n2_xyz,
	FArray1Da_float ca2_xyz,
	int const begin_offset,
	FArray3Da_float centroid_parm_eachres
)
{
	using namespace param;
	using namespace refold_ns;
	using namespace start;

	c1_xyz.dimension( 3 );
	n1_xyz.dimension( 3 );
	ca1_xyz.dimension( 3 );
	phi.dimension( end_frag );
	psi.dimension( end_frag );
	omega.dimension( end_frag );
	res.dimension( end_frag );
	Eposition.dimension( 3, MAX_POS, end_frag );
	centroid.dimension( 3, end_frag );
	c2_xyz.dimension( 3 );
	n2_xyz.dimension( 3 );
	ca2_xyz.dimension( 3 );
	centroid_parm_eachres.dimension( 5, DRange(-1, 1), end_frag);

//car fold 1 to end_frag from torsion angles in direction dir
//car given starting position c_xyz,n_xyz,ca_xyz
//car ending position is c2_xyz,n2_xyz,ca2_xyz
//car coordinates from 1-end_frag in Eposition and centroid are changed

// local
	FArray2D_float X( 3, 3 );
	FArray3D_float temp_pos( 3, MAX_POS, DRange( 0, 1 ) ); // to handle end residues
	FArray2D_float M( 3, 3 );

	///////////////////////
	// get bonds from pose?
	bonds_class::Bonds const * pose_bonds(0);
	if ( refold_check_current_pose() ) {
		pose_ns::Pose const & pose( refold_get_current_pose() );
		if ( !pose.ideal_backbone() ) {
			pose_bonds = &(pose.bonds());
		}
	}

	if ( end_frag < 1 ) return;
	init_fold();
	if ( use_st_bond() ) init_start_fold();

	if ( dir == n2c ) {

//car n,ca of first
		for ( int i = 1; i <= 3; ++i ) {
			Eposition(i,n,1) = n1_xyz(i);
			Eposition(i,ca,1) = ca1_xyz(i);
		}

//car define initial bond vectors
		subvec(n1_xyz,c1_xyz,X(1,om));
		subvec(ca1_xyz,n1_xyz,X(1,ph));

		if ( pose_bonds ) {
			for ( int i = 1; i < end_frag; ++i ) { // build c,cb,n(i+1),o,ca(i+1)
				int const ii = begin_offset + i - 1;
				build_residue_n2c
					( X, Eposition(1,1,i), centroid(1,i),
						pose_bonds->cT()(1,n2c,ii),	pose_bonds->cT()(1,n2c,ii+1),
						pose_bonds->sT()(1,n2c,ii), pose_bonds->sT()(1,n2c,ii+1),
						pose_bonds->D ()(1,n2c,ii), pose_bonds->D ()(1,n2c,ii+1),
						pose_bonds->cb_phi_offset()(n2c,ii),
						pose_bonds->ox_psi_offset()(n2c,ii),
						centroid_parm_eachres(1,n2c,i),
						phi(i),psi(i),omega(i) );
			}
		} else if ( use_st_bond() ) {
			for ( int i = 1; i < end_frag; ++i ) { // build c,cb,n(i+1),o,ca(i+1)
				int const ii = begin_offset + i - 1;
				/*chu ATTENTION: at this point use_st_bond assumes begin_offset is
					the offset number from RESIDUE 1, NOT the SEGMENT begin residue.
					so when multiple segments are to be refolded, use_st_bond will insert
					wrong bond lengths/angles and cause problem for refolding*/
				build_residue_n2c(X,Eposition(1,1,i),centroid(1,i),
				 st_cT(1,n2c,ii),st_cT(1,n2c,ii+1),st_sT(1,n2c,ii), st_sT(1,n2c,ii+1),
				 st_D(1,n2c,ii),st_D(1,n2c,ii+1),
				 st_cb_phi_offset(n2c,ii),st_ox_psi_offset(n2c,ii),
				 centroid_parm_eachres(1,n2c,i),
				 phi(i),psi(i),omega(i));
			}
		} else {
			for ( int i = 1; i < end_frag; ++i ) { // build c,cb,n(i+1),o,ca(i+1)
				build_residue_n2c(X,Eposition(1,1,i),centroid(1,i),
				 cT(1,n2c),cT(1,n2c),sT(1,n2c),sT(1,n2c),
				 D(1,n2c),D(1,n2c),
				 cb_phi_offset(n2c),ox_psi_offset(n2c),
				 centroid_parm_eachres(1,n2c,i),
				 phi(i),psi(i),omega(i));
			}
		}


//car build last residue into temporary array and copy out
//car only the atoms we want
		for ( int i = 1; i <= 3; ++i ) {
			temp_pos(i,ca,0) = Eposition(i,ca,end_frag);
		}

		if ( pose_bonds ) {
			int const ii = begin_offset + end_frag - 1;
			build_residue_n2c
				( X, temp_pos(1,1,0), centroid(1,end_frag),
					pose_bonds->cT()(1,n2c,ii), pose_bonds->cT()(1,n2c,ii+1),
					pose_bonds->sT()(1,n2c,ii), pose_bonds->sT()(1,n2c,ii+1),
					pose_bonds->D ()(1,n2c,ii), pose_bonds->D ()(1,n2c,ii+1),
					pose_bonds->cb_phi_offset()(n2c,ii),
					pose_bonds->ox_psi_offset()(n2c,ii),
					centroid_parm_eachres(1,n2c,end_frag),
					phi(end_frag),psi(end_frag),omega(end_frag) );
		} else if ( use_st_bond() ) {
			int const ii = begin_offset + end_frag - 1;
			build_residue_n2c(X,temp_pos(1,1,0),centroid(1,end_frag),
			 st_cT(1,n2c,ii),st_cT(1,n2c,ii+1),st_sT(1,n2c,ii),st_sT(1,n2c,ii+1),
			 st_D(1,n2c,ii),st_D(1,n2c,ii+1),
			 st_cb_phi_offset(n2c,ii),st_ox_psi_offset(n2c,ii),
			 centroid_parm_eachres(1,n2c,end_frag),
			 phi(end_frag),psi(end_frag),omega(end_frag));
		} else {
			build_residue_n2c(X,temp_pos(1,1,0),centroid(1,end_frag),
			 cT(1,n2c),cT(1,n2c),sT(1,n2c),sT(1,n2c),
			 D(1,n2c),D(1,n2c),
			 cb_phi_offset(n2c),ox_psi_offset(n2c), // chu,bugfixed
			 centroid_parm_eachres(1,n2c,end_frag),
			 phi(end_frag), psi(end_frag),omega(end_frag));
		}

		for ( int i = 1; i <= 3; ++i ) {
			Eposition(i,c,end_frag) = temp_pos(i,c,0);
			Eposition(i,cb,end_frag) = temp_pos(i,cb,0);
			Eposition(i,ox,end_frag) = temp_pos(i,ox,0);
			c2_xyz(i) = temp_pos(i,c,0);
			n2_xyz(i) = temp_pos(i,n,1);
			ca2_xyz(i) = temp_pos(i,ca,1);
		}

	} else { // dir == c2n

//car c  of last
		for ( int i = 1; i <= 3; ++i ) {
			Eposition(i,c,end_frag) = c1_xyz(i);
		}
//car define initial bond vectors
		subvec(c1_xyz,n1_xyz,X(1,om));
		subvec(n1_xyz,ca1_xyz,X(1,ph));

		if ( pose_bonds ) {
			for ( int i = end_frag; i >= 2; --i ) {
				int const ii = begin_offset + i - 1;
				build_residue_c2n
					( X, Eposition(1,1,i-1), centroid(1,i),
						pose_bonds->cT()(1,c2n,ii), pose_bonds->cT()(1,c2n,ii-1),
						pose_bonds->sT()(1,c2n,ii), pose_bonds->sT()(1,c2n,ii-1),
						pose_bonds->D ()(1,c2n,ii), pose_bonds->D ()(1,c2n,ii-1),
						pose_bonds->cb_phi_offset()(c2n,ii),
						pose_bonds->ox_psi_offset()(c2n,ii),
						centroid_parm_eachres(1,c2n,i),
						phi(i),psi(i),omega(i) );
			}
		} else if ( use_st_bond() ) {
			for ( int i = end_frag; i >= 2; --i ) {
				int const ii = begin_offset + i - 1;
				build_residue_c2n(X,Eposition(1,1,i-1),centroid(1,i),
				 st_cT(1,c2n,ii),st_cT(1,c2n,ii-1),st_sT(1,c2n,ii),st_sT(1,c2n,ii-1),
				 st_D(1,c2n,ii),st_D(1,c2n,ii-1),
				 st_cb_phi_offset(c2n,ii),st_ox_psi_offset(c2n,ii),
				 centroid_parm_eachres(1,c2n,i),
				 phi(i),psi(i),omega(i));
			}
		} else {
			for ( int i = end_frag; i >= 2; --i ) {
				build_residue_c2n(X,Eposition(1,1,i-1),centroid(1,i),
				 cT(1,c2n),cT(1,c2n),sT(1,c2n),sT(1,c2n),
				 D(1,c2n), D(1,c2n),
				 cb_phi_offset(c2n),ox_psi_offset(c2n),
				 centroid_parm_eachres(1,c2n,i),
				 phi(i),psi(i),omega(i));
			}
		}

		for ( int i = 1; i <= 3; ++i ) {
			temp_pos(i,c,1) = Eposition(i,c,1);
		}

		if ( pose_bonds ) {
			int const ii = begin_offset + 1 - 1;
			build_residue_c2n
				( X, temp_pos(1,1,0), centroid(1,1),
					pose_bonds->cT()(1,c2n,ii), pose_bonds->cT()(1,c2n,ii-1),
					pose_bonds->sT()(1,c2n,ii), pose_bonds->sT()(1,c2n,ii-1),
					pose_bonds->D ()(1,c2n,ii), pose_bonds->D ()(1,c2n,ii-1),
					pose_bonds->cb_phi_offset()(c2n,ii),
					pose_bonds->ox_psi_offset()(c2n,ii),
					centroid_parm_eachres(1,c2n,1),
					phi(1),psi(1),omega(1) );
		} else if ( use_st_bond() ) {
			int const ii = begin_offset + 1 - 1;
			build_residue_c2n(X,temp_pos(1,1,0),centroid(1,1),
			 st_cT(1,c2n,ii),st_cT(1,c2n,ii-1),st_sT(1,c2n,ii),st_sT(1,c2n,ii-1),
			 st_D(1,c2n,ii),st_D(1,c2n,ii-1),
			 st_cb_phi_offset(c2n,ii),st_ox_psi_offset(c2n,ii),
			 centroid_parm_eachres(1,c2n,1),
			 phi(1),psi(1),omega(1));
		} else {
			build_residue_c2n(X,temp_pos(1,1,0),centroid(1,1),
			 cT(1,c2n),cT(1,c2n),sT(1,c2n),sT(1,c2n),
			 D(1,c2n), D(1,c2n),
			 cb_phi_offset(c2n),ox_psi_offset(c2n),
			 centroid_parm_eachres(1,c2n,1),
			 phi(1),psi(1),omega(1));
		}

		for ( int i = 1; i <= 3; ++i ) {
			Eposition(i,n,1) = temp_pos(i,n,1);
			Eposition(i,ca,1) = temp_pos(i,ca,1);
			Eposition(i,cb,1) = temp_pos(i,cb,1);
			Eposition(i,ox,1) = temp_pos(i,ox,1);
			c2_xyz(i) = temp_pos(i,c,0);
			n2_xyz(i) = temp_pos(i,n,1);
			ca2_xyz(i) = temp_pos(i,ca,1);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin build_residue_n2c
///
/// @brief generate coodinates for one residue from torsion angles considering
/// chain in n2c direction
///
/// @detailed
///car given initial orientation info in X
///car and starting atom position( ca(1) in Eposition)
///car build one residue from phi,psi,omega
///car one residue is defined by phi,psi,omega
///car for n->c direction this means determining coords of
///car   c(1),cb(1),cen(1),n(2),o(1),ca(2)
///
/// @param  X - [in/out]? - bond vectors of chain stub
///car        X(1,1) = last phi vector in n-c direction
///car        X(1,2) = last psi vector
///car        X(1,3) = last omega vector
///car   X(1,1) and x(1,3) must be defined before calling
///car   values are modified!!
/// @param  Eposition - [in/out]? - backbone coordinates of residue i,i+1
///        ca(1) is input, c(1),cb(1),cen(1),n(2),o(1),ca(2) are output
/// @param[out]   centroid - out - centroid coordinates of residue i
/// @param[in]   phi - in - phi of residue 1
/// @param[in]   psi - in - psi of residue 1
/// @param[in]   omega - in - omega of residue 1
/// @param[in]   res - in - residue type of residue1
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car   X(1,1) and x(1,3) must be defined before calling
///car   values are modified!!
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_residue_n2c(
	FArray2Da_float X,
	FArray3Da_float Eposition, // coords of res i, i+1
	FArray1Da_float centroid, // centroid of res i
	FArray1Da_float cT_1,
	FArray1Da_float cT_2,
	FArray1Da_float sT_1,
	FArray1Da_float sT_2,
	FArray1Da_float D_1,
	FArray1Da_float D_2,
	float const cb_phi_offset,
	float const ox_psi_offset,
	FArray1Da_float centroid_parm,
	float const phi,
	float const psi,
	float const omega
)
{
	using namespace param;
	using namespace refold_ns;
	using numeric::conversions::degrees;

	X.dimension( 3, 3 );
	Eposition.dimension( 3, MAX_POS, 2 );
	centroid.dimension( 3 );
	cT_1.dimension( 5 );
	cT_2.dimension( 5 );
	sT_1.dimension( 5 );
	sT_2.dimension( 5 );
	D_1.dimension( 5 );
	D_2.dimension( 5 );
	centroid_parm.dimension( 5 );


	int const n = { 1 };
	int const ca = { 2 };
	int const cb = { 3 };
	int const c = { 4 };
	int const ox = { 5 };
//	int const cen = { 6 };

	int const ph = { 1 };
	int const ps = { 2 };
	int const om = { 3 };

//	int const n2c = { 1 };
//	int const c2n = { -1 };

// local
	FArray2D_float M( 3, 3 );
	FArray1D_float junk( 3 );

//car c
	refold_coord_sys(X(1,om),X(1,ph),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,ca,1),cT_1(c),sT_1(c),phi,D_1(c),
	 Eposition(1,c,1),X(1,ps));

//car cb
	build_atom( M, Eposition(1,ca,1), cT_1(cb), sT_1(cb), phi + cb_phi_offset, D_1(cb),
	 Eposition(1,cb,1), junk );
//car cen
	build_atom( M, Eposition(1,ca,1), centroid_parm(3), centroid_parm(4),
	 phi + degrees( centroid_parm(2) ), centroid_parm(1), centroid, junk );

//car n(i+1)
	refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,c,1),cT_2(n),sT_2(n),psi,D_2(n),
	 Eposition(1,n,2),X(1,om));

//car o(i)
	build_atom(M,Eposition(1,c,1),cT_1(ox),sT_1(ox),psi+ox_psi_offset,D_1(ox),
	 Eposition(1,ox,1),junk);

//car ca(i+1)
	refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,n,2),cT_2(ca),sT_2(ca),omega,D_2(ca),
	 Eposition(1,ca,2),X(1,ph));

}

/////////////////////////////////////////////////////////////////////////////////
/// @begin build_residue_c2n
///
/// @brief generate coodinates for one residue from torsion angles considering
/// chain in c2n direction
///
/// @detailed
///car given initial orientation info in X
///car and starting atom position ( c(1) in Eposition)
///car build one residue from phi,psi,omega
///car one residue is defined by phi,psi,omega
///car for c->n direction this means determining coords of
///car ca(1),o(1),n(1),cb(1),cen(1),c(0)
///
/// @param  X - [in/out]? - bond vectors of chain stub
///car        X(1,1) = last phi vector in c-n direction
///car        X(1,2) = last psi vector
///car        X(1,3) = last omega vector
///car   X(1,1) and x(1,3) must be defined before calling
///car   values are modified!!
/// @param  Eposition - [in/out]? - backbone coordinates of residue i-1,i
///       c(1) is input, ca(1),o(1),n(1),cb(1),cen(1),c(0)
/// @param[out]   centroid - out - centroid coordinates of residue i
/// @param[in]   phi - in - phi of residue 1
/// @param[in]   psi - in - psi of residue 1
/// @param[in]   omega - in - omega of residue 1
/// @param[in]   res - in - residue type of residue1
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car   X(1,1) and x(1,3) must be defined before calling
///car   values are modified!!
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_residue_c2n(
	FArray2Da_float X,
	FArray3Da_float Eposition, // coords of res i-1,i
	FArray1Da_float centroid, // centroid of res i
	FArray1Da_float cT_1,
	FArray1Da_float cT_0,
	FArray1Da_float sT_1,
	FArray1Da_float sT_0,
	FArray1Da_float D_1,
	FArray1Da_float D_0,
	float const cb_phi_offset,
	float const ox_psi_offset,
	FArray1Da_float centroid_parm,
	float const phi,
	float const psi,
	float const omega
)
{
	using namespace param;
	using namespace refold_ns;
	using numeric::conversions::degrees;

	X.dimension( 3, 3 );
	Eposition.dimension( 3, MAX_POS, SRange( 0, 1 ) );
	centroid.dimension( 3 );
	cT_1.dimension( 5 );
	cT_0.dimension( 5 );
	sT_1.dimension( 5 );
	sT_0.dimension( 5 );
	D_1.dimension( 5 );
	D_0.dimension( 5 );
	centroid_parm.dimension( 5 );

	int const n = { 1 };
	int const ca = { 2 };
	int const cb = { 3 };
	int const c = { 4 };
	int const ox = { 5 };
//	int const cen = { 6 };

	int const ph = { 1 };
	int const ps = { 2 };
	int const om = { 3 };

//	int const n2c = { 1 };
//	int const c2n = { -1 };

// local
	FArray2D_float M( 3, 3 );
	FArray1D_float junk( 3 );

//car ca
	refold_coord_sys(X(1,ph),X(1,om),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,c,1),cT_1(ca),sT_1(ca),omega,D_1(ca),
	 Eposition(1,ca,1),X(1,ps));

//car ox
	build_atom(M,Eposition(1,c,1),cT_1(ox),sT_1(ox),omega+ox_psi_offset,D_1(ox),
	 Eposition(1,ox,1),junk);

//car n
	refold_coord_sys(X(1,om),X(1,ps),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,ca,1),cT_1(n),sT_1(n),psi,D_1(n),
	 Eposition(1,n,1),X(1,ph));

//car cb
	build_atom(M,Eposition(1,ca,1),cT_1(cb),sT_1(cb),psi+cb_phi_offset,D_1(cb),
	 Eposition(1,cb,1),junk);
//car cen
	build_atom( M, Eposition(1,ca,1), centroid_parm(3), centroid_parm(4),
	 psi + degrees( centroid_parm(2) ), centroid_parm(1), centroid, junk );

//car c(i-1)
	refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,n,1),cT_0(c),sT_0(c),phi,D_0(c),
	 Eposition(1,c,0),X(1,om));

}

////////////////////////////////////////////////////////////////////////////////
/// @begin GL_rotate
///
/// @brief Applies a rotation matrix to a list of 3D vectors
///
/// @detailed
///
/// @param[in]   last - in - number of vectors to rotate
/// @param[in]   stride - in - step size for iterating through vector list
/// @param[in]   Mgl - in - rotation matrix (4x4)
/// @param[in]   V - in - starting vectors
/// @param[out]   V_out - out - rotated vectors
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     The rotation
///     matrix is NOT applied to every vector in the list; it is only
///     applied to every STRIDE'th entry starting at 1, and
///     ending on or before last e.g. if last = 9,
///     stride=4, then vecs 1,5 get rotated.  why STRIDE? often Stride
///     = 5 or -5 and so we will be rotating one KIND of atom c
///
///     no in-place rotation!!
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
GL_rotate(
	int last,
	int stride,
	FArray2Da_float Mgl,
	FArray2Da_float V,
	FArray2Da_float V_out
)
{
	Mgl.dimension( 4, 4 );
	V.dimension( 3, last );
	V_out.dimension( 3, last );

	assert( ( stride >= 0 ) ); // Or loop test should be >=
	for ( int i = 1; i <= last; i += stride ) {
		GL_rot( Mgl, V(1,i), V_out(1,i) );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin GL_rot
///
/// @brief apply a rotation to a 3d vector
///
/// @detailed
///
/// @param[in]   Mgl - in - 4X4 rotation matrix
/// @param[in]   V - in - input vector
/// @param[out]   X - out - rotated vector
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
///cems     A 4x4 matrix multiply times a vector v(4) except it is assumed
///cems     that the 4th row of Mgl is (0,0,0,1) and it is assumed that V(4)
///cems     is 1
///cems     NOTE: this function will work fine if it is handed Mgl(3,4) and V(3)
///cems     history lesson: the gl suffix refers to SGI GL
///
/// @references
///
/// @authors car 10/03/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
GL_rot(
	FArray2Da_float Mgl,
	FArray1Da_float V,
	FArray1Da_float X
)
{
	Mgl.dimension( 4, 4 );
	V.dimension( 4 );
	X.dimension( 3 );

	for ( int i = 1; i <= 3; ++i ) {
		X(i) = Mgl(i,1)*V(1) + Mgl(i,2)*V(2) + Mgl(i,3)*V(3) + Mgl(i,4);
	}

}

void
GL_rot_in_place(
	FArray2Da_float Mgl,
	FArray1Da_float X
)
{
	Mgl.dimension( 4, 4 );
	static FArray1D_float V(3);
	X.dimension( 3 );

	for ( int i = 1; i <= 3; ++i ) {
		V(i) = X(i);
	}
	for ( int i = 1; i <= 3; ++i ) {
		X(i) = Mgl(i,1)*V(1) + Mgl(i,2)*V(2) + Mgl(i,3)*V(3) + Mgl(i,4);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin build_atom
///
/// @brief compute coordinates of an atom from a defined coordinate system
///       and spherical coordinates defining atoms position
///
/// @detailed
///
/// @param[in]   M - in - rotation matrix defining coordinate system
/// @param[in]   ori - in - origin of coordinate system (ie atom i-1)
/// @param[in]   cT - in - cos(theta), spherical coords of atom to be built
/// @param[in]   sT - in - sin(theta), spherical coords of atom to be built
/// @param[in]   P - in - phi, spherical coords of atom to be built
/// @param[in]   D - in - bond length, spherical coords of atom to be built
/// @param[out]   Xo - out - output coordinates of atom
/// @param[out]   N - out - vector from ori->Xo (for next coord system)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     the new bond makes an angle theta to the x-axis and the X-axis is
///     rotated by phi wrt to the B-axis P is the point where the bond
///     starts the length of the bond is D.  Xo is the output new position
///     and N is the output new bond direction vector
///car   ori          origin of coordinate system (atom i-1)
///car   M            rotation matrix defining coordinate system
///car   B,X          vectors defining the coordinate system
///car                X is colinear with x-axis; (i-1) -> (i-2) vector
///car                B lies in xy plane & dotprod(y,b)>0 ; (i-2)->(i-3) vector
///car   D            bond length
///car   theta        angle of (i-1)->i w/ x axis;(pi -  bond angle)
///car   phi          dihedral
///
///car   example: to build C2
///car   coord system   center CA2
///car                x vector N2->CA2
///car                b vector C1->N2
///car   D            CA2-C2 bond length
///car   theta        180 - (N2-CA2-C2 bond angle), angle w/ x axis
///car   phi          C1-N2-CA2-C2 dihedral (ie omega(1))
///car                     NOTE THAT ANGLES ARE IN RADIANS
///
///car   cT,sT    cos(theta), sin(theta)
///car   cP,sP    cos(phi),sin(phi)  OR cos(psi),sin(psi)
///car   D        distance   (bond length)
///
///
///car   Xo       output coordinates
///car   N        (i-1) -> i vector (for next coord system)
///
/// @references
///
/// @authors car 10/03/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_atom(
	FArray2Da_float M,
	FArray1Da_float ori,
	float const cT,
	float const sT,
	float const P,
	float const D,
	FArray1Da_float Xo,
	FArray1Da_float N
)
{
	using numeric::conversions::radians;

	M.dimension( 3, 3 );
	ori.dimension( 3 );
	Xo.dimension( 3 );
	N.dimension( 3 );

	float temp = radians( P );
	float cP = std::cos(temp);
	float sP = std::sin(temp);

	float cTD = cT * D;
	temp = sT * D;
	float cPsTD = cP * temp;
	float sPsTD = sP * temp;

	for ( int i = 1; i <= 3; ++i ) {
		N(i) = M(i,1)*cTD + M(i,2)*cPsTD + M(i,3)*sPsTD;
		Xo(i) = N(i) + ori(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_coord_sys
///
/// @brief computes a generalized coordsys matrix from two plane-defining vectors
///
/// @detailed
///
/// @param[in]   B - in - vector colinear with x-axis
/// @param[in]   X0 - in - vector in x-y plan
/// @param[out]   X - out - normalized x-axis basis vector
/// @param[out]   Y - out - normalized y-axis basis vector
/// @param[out]   Z - out - normalized z-axis basis vector
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     B and X0 are plane defining vectors.  not nessesarily
///     perpendicular or normalized X is the x-axis basis vector
///     (normalized x), the Y-axis is in the B-X plane and has a positive
///     dot product on the B-axis
///
///     (X,Y,Z) is the output rotation matrix.  to use this with a 3x3
///     matrix M call: refold_coord_sys(B,X0,M(1,1),M(1,2),M(1,3))
///
/// @references
///
/// @authors 10/03/2003 car
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_coord_sys(
	FArray1Da_float B,
	FArray1Da_float X0,
	FArray1Da_float X,
	FArray1Da_float Y,
	FArray1Da_float Z
)
{
	B.dimension( 3 );
	X0.dimension( 3 );
	X.dimension( 3 );
	Y.dimension( 3 );
	Z.dimension( 3 );

	cros(B,X0,Z); // Z axis  6 multiplies 3 adds
	unitvec(X0,X); // 7 mult 1 sqrt
	unitvec(Z,Z); // 7 mult 1 sqrt
	cros(Z,X,Y); // Y axis  6 multiples 3 adds

// Y is normalized since X and Z were

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_GL_matrix
///
/// @brief find GL rotation matrix that transforms P1,P2,P3 to Q1,Q2,Q3
///
/// @detailed
///
/// @param[in]   P1 - in - point 1 in original coord system
/// @param[in]   P2 - in - point 2 in original coord system
/// @param[in]   P3 - in - point 3 in original coord system
/// @param[in]   Q1 - in - point 1 in desired coord system
/// @param[in]   Q2 - in - point 2 in desired coord syste
/// @param[in]   Q3 - in - point 3 in desired coord syste
/// @param[out]   Mgl - out - GL rotation matrix
///
/// @global_read
///
/// @global_write
///
/// @remarks
///     the output rotation is the
///     matrix (Mgl(1,1),Mgl(1,2),Mgl(1,3)), the offset vector added
///     after rotation is M(1,4)
///     history lesson: the gl suffix refers to SGI GL   c
///
/// PB: appears that P2 is mapped to Q2
/// and the P2->P3 vector is mapped to the Q2->Q3 vector
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_GL_matrix(
	FArray1Da_float P1,
	FArray1Da_float P2,
	FArray1Da_float P3,
	FArray1Da_float Q1,
	FArray1Da_float Q2,
	FArray1Da_float Q3,
	FArray2Da_float Mgl
)
{
	P1.dimension( 3 );
	P2.dimension( 3 );
	P3.dimension( 3 );
	Q1.dimension( 3 );
	Q2.dimension( 3 );
	Q3.dimension( 3 );
	Mgl.dimension( 4, 4 );

// internal
	FArray1D_float Bp( 3 );
	FArray1D_float Xp( 3 );
	FArray1D_float Bq( 3 );
	FArray1D_float Xq( 3 );
	FArray2D_float Mq( 3, 3 );
	FArray2D_float Mp( 3, 3 );

	subvec(P3,P2,Xp);
	subvec(Q3,Q2,Xq);
	subvec(P2,P1,Bp);
	subvec(Q2,Q1,Bq);

	refold_coord_sys(Bp,Xp,Mp(1,1),Mp(1,2),Mp(1,3));
	refold_coord_sys(Bq,Xq,Mq(1,1),Mq(1,2),Mq(1,3));

	for ( int i = 1; i <= 3; ++i ) {

		Mgl(i,1) = 0.0;
		Mgl(i,2) = 0.0;
		Mgl(i,3) = 0.0;

		for ( int j = 1; j <= 3; ++j ) {
			Mgl(i,1) += Mp(1,j)*Mq(i,j);
			Mgl(i,2) += Mp(2,j)*Mq(i,j);
			Mgl(i,3) += Mp(3,j)*Mq(i,j);

		}
		Mgl(i,4) = Q2(i) - Mgl(i,1)*P2(1) - Mgl(i,2)*P2(2) - Mgl(i,3)*P2(3);

		Mgl(4,i) = 0.0;
	}

	Mgl(4,4) = 1.0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin refold_default_stub
///
/// @brief copies default template into three atoms
///
/// @detailed
///
/// @param[out]   c_xyz - out - default stub C coords
/// @param[out]   n_xyz - out - default stub N coords
/// @param[out]   ca_xyz - out - default stub CA coords
///
/// @global_read
///
/// @global_write
///
/// @remarks should be called only from refold to build the coords, but not
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified chu 11/01/2004
/////////////////////////////////////////////////////////////////////////////////
void
refold_default_stub(
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	int const dir // n2c or c2n
)
{
	using namespace refold_ns;
	using namespace start;
	using namespace misc; //PB: for total_residue

	c_xyz.dimension( 3 );
	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );

//chu  currently assume the default stub is either at N-terminal or C-terminal
//     when start_bond information is used. This might not be true for the loop
//     mode.
	if ( refold_check_current_pose() &&
			 !refold_get_current_pose().ideal_backbone() ) {
		std::cout << "refold_default_stub not compatible w/ non-ideal pose" <<
			std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else if ( use_st_bond() ) {
		for ( int i = 1; i <= 3; ++i ) {
			c_xyz(i) = st_default_template(i,1,dir);
			n_xyz(i) = st_default_template(i,2,dir);
			ca_xyz(i) = st_default_template(i,3,dir);
		}
	} else {
		for ( int i = 1; i <= 3; ++i ) {
			c_xyz(i)  = default_template(i,1);
			n_xyz(i)  = default_template(i,2);
			ca_xyz(i) = default_template(i,3);
		}
	}
}
////////////////////////////////////////////////////////////////////////////////
/// @begin refold_default_template_stub
///
/// @brief copies default idealized template into three atoms
///
/// @detailed
///
/// @param[out]   c_xyz - out - default stub C coords
/// @param[out]   n_xyz - out - default stub N coords
/// @param[out]   ca_xyz - out - default stub CA coords
///
/// @global_read
///
/// @global_write
///
/// @remarks this function is part of "refold_default_stub" with only
///          idealized template being copied. Should be called from moves
///          with fragment insertion, i.e. chuck_move.This will allow
///          use_input_bond be compatible with fragment insertions.
///
/// @references
///
/// @authors chu 11/01/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
refold_default_template_stub(
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz
)
{
	using namespace refold_ns;
	using namespace start;

	c_xyz.dimension( 3 );
	n_xyz.dimension( 3 );
	ca_xyz.dimension( 3 );

//chu  retrieve only the default idealized template.
	for ( int i = 1; i <= 3; ++i ) {
		c_xyz(i)  = default_template(i,1);
		n_xyz(i)  = default_template(i,2);
		ca_xyz(i) = default_template(i,3);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_fold
///
/// @brief precompute constants used in building atom coordinates from torsion
///       angles
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
init_fold()
{
	using namespace refold_ns;
	using numeric::conversions::radians;

	static bool init = { false };

	if ( init ) return;

	float tmp;

	init = true;

	tmp = radians( 180.0f - angle(n,n2c) );
	cT(n,n2c) = std::cos(tmp);
	sT(n,n2c) = std::sin(tmp);

	tmp = radians( 180.0f - angle(ca,n2c) );
	cT(ca,n2c) = std::cos(tmp);
	sT(ca,n2c) = std::sin(tmp);

	tmp = radians( 180.0f - angle(c,n2c) );
	cT(c,n2c) = std::cos(tmp);
	sT(c,n2c) = std::sin(tmp);

	tmp = radians( 180.0f - angle(ox,n2c) );
	cT(ox,n2c) = std::cos(tmp);
	sT(ox,n2c) = std::sin(tmp);

	tmp = radians( 180.0f - angle(cb,n2c) );
	cT(cb,n2c) = std::cos(tmp);
	sT(cb,n2c) = std::sin(tmp);


	tmp = radians( 180.0f - angle(n,c2n) );
 	cT(n,c2n) = std::cos(tmp);
	sT(n,c2n) = std::sin(tmp);

	tmp = radians( 180.0f - angle(ca,c2n) );
	cT(ca,c2n) = std::cos(tmp);
	sT(ca,c2n) = std::sin(tmp);

	tmp = radians( 180.0f - angle(c,c2n) );
	cT(c,c2n) = std::cos(tmp);
	sT(c,c2n) = std::sin(tmp);

	tmp = radians( 180.0f - angle(ox,c2n) );
	cT(ox,c2n) = std::cos(tmp);
	sT(ox,c2n) = std::sin(tmp);

	tmp = radians( 180.0f - angle(cb,c2n) );
	cT(cb,c2n) = std::cos(tmp);
	sT(cb,c2n) = std::sin(tmp);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_start_bond_info
///
/// @brief
///
/// @detailed
///
/// @param - coord_fail - [in/out]? - true if there is no coordinates read in
/// @param - total_residue - [in/out]? -
/// @param - res - [in/out]? -
/// @param - Eposition - [in/out]? -
/// @param - phi - [in/out]? -
/// @param - psi - [in/out]? -
/// @param - omega - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_start_bond_info(
	bool const coord_fail, // true if there is no coordinates read in
	int const total_residue,
	FArray1Da_int res,
	FArray3Da_float Eposition,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray1Da_float omega
)
{
	using namespace files_paths;
	using namespace param;

	static bool init = { false };

	res.dimension( MAX_RES() );
	Eposition.dimension( 3, MAX_POS, MAX_RES() );
	phi.dimension( MAX_RES() );
	psi.dimension( MAX_RES() );
	omega.dimension( MAX_RES() );

//chu should be initialized only when start_coord is ready
	if ( ! init ) {
		init = true;
		refold_ns::st_bond_switch::st_bond = truefalseoption("use_input_bond");
			if ( refold_ns::st_bond_switch::st_bond && require_frags ) {
				std::cout << "WARNING:: require_frags T, use_input_bond T" << std::endl;
				std::cout << "fragment insertion should be used with caution when input_bond is used" << std::endl;
			}
	}

	if ( ! refold_ns::st_bond_switch::st_bond ) return;
	if ( coord_fail ) {
		std::cout << "ERROR: use_input_bond is TRUE, but no input coord" << std::endl;
		std::cout << "STOP" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	calculate_start_bond_info( total_residue, Eposition, phi, psi, omega );

}
///////////////////////////////////////////////////////////////////////////////
void
calculate_start_bond_info(
	int const total_residue,
	FArray3DB_float const & Eposition,
	FArray1DB_float const & phi,
	FArray1DB_float const & psi,
	FArray1DB_float const & omega
)
{
	using namespace refold_ns;
	using namespace start;

	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );
	FArray1D_float pseudo_n( 3 );
	FArray1D_float pseudo_ca( 3 );
	FArray1D_float pseudo_c( 3 );

	init_fold(); // need cT and sT

	int i = 0; // add a pseudo position before N-terminal
	for ( int j = 1; j <= 5; ++j ) {
		st_D(j,n2c,i) = D(j,n2c);
		st_D(j,c2n,i) = D(j,c2n);
		st_angle(j,n2c,i) = angle(j,n2c);
		st_angle(j,c2n,i) = angle(j,c2n);
	}
//chu   n2c direction  the first residue is special
	i = 1;
	subvec(Eposition(1,ca,i),Eposition(1, c,i),X(1,ps));
	subvec(Eposition(1, n,i),Eposition(1,ca,i),X(1,ph));
	refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,n,1),cT(c,c2n),sT(c,c2n),phi(i),D(c,c2n),
	 pseudo_c,X(1,om)); // generate a pseduo C at position zero
//     save them in the default template if refolding from seqpos 1
	vector_copy(pseudo_c,st_default_template(1,1,n2c));
	vector_copy(Eposition(1, n,i),st_default_template(1,2,n2c));
	vector_copy(Eposition(1,ca,i),st_default_template(1,3,n2c));

//     bond_length
	st_D( n,n2c,i) = D(n,n2c); // use template D
	st_D(ca,n2c,i) = vec_dist(Eposition(1,ca,i),Eposition(1, n,i));
	st_D(cb,n2c,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
	st_D( c,n2c,i) = vec_dist(Eposition(1, c,i),Eposition(1,ca,i));
	st_D(ox,n2c,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));
//     bond_angle
	st_angle( n,n2c,i) = angle( n,n2c); // use template angle
	st_angle(ca,n2c,i) = angle(ca,n2c); // use template angle
	st_angle(cb,n2c,i) =
	 vec_angle( Eposition(1, n,i), Eposition(1,ca,i), Eposition(1,cb,i) );
	st_angle( c,n2c,i) =
	 vec_angle( Eposition(1, n,i), Eposition(1,ca,i), Eposition(1, c,i) );
	st_angle(ox,n2c,i) =
	 vec_angle( Eposition(1,ca,i), Eposition(1, c,i), Eposition(1,ox,i) );
//     offsets
	st_cb_phi_offset(n2c,i) = dihedral(pseudo_c,Eposition(1, n,i),
	 Eposition(1,ca,i),Eposition(1,cb,i)) - phi(i);
	st_ox_psi_offset(n2c,i) = dihedral(Eposition(1, n,i),Eposition(1,ca,i),
	 Eposition(1, c,i),Eposition(1,ox,i)) - psi(i);

//chu   the rest residues
	for ( int i = 2; i <= total_residue; ++i ) {
//     bond_length
		st_D( n,n2c,i) = vec_dist(Eposition(1, n,i),Eposition(1, c,i-1));
		st_D(ca,n2c,i) = vec_dist(Eposition(1,ca,i),Eposition(1, n,i));
		st_D(cb,n2c,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
		st_D( c,n2c,i) = vec_dist(Eposition(1, c,i),Eposition(1,ca,i));
		st_D(ox,n2c,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));
//     bond_angle
		st_angle( n,n2c,i) =
		 vec_angle( Eposition(1,ca,i-1), Eposition(1, c,i-1), Eposition(1, n,i) );
		st_angle(ca,n2c,i) =
		 vec_angle( Eposition(1, c,i-1), Eposition(1, n,i), Eposition(1,ca,i) );
		st_angle(cb,n2c,i) =
		 vec_angle( Eposition(1, n,i), Eposition(1,ca,i), Eposition(1,cb,i) );
		st_angle( c,n2c,i) =
		 vec_angle( Eposition(1, n,i), Eposition(1,ca,i), Eposition(1, c,i) );
		st_angle(ox,n2c,i) =
		 vec_angle( Eposition(1,ca,i), Eposition(1, c,i), Eposition(1,ox,i) );
//     offsets
		st_cb_phi_offset(n2c,i) = dihedral(Eposition(1, c,i-1),Eposition(1, n,i),
		 Eposition(1,ca,i),Eposition(1,cb,i)) - phi(i);
		st_ox_psi_offset(n2c,i) = dihedral(Eposition(1, n,i),Eposition(1,ca,i),
		 Eposition(1, c,i),Eposition(1,ox,i)) - psi(i);
	}

	i = total_residue + 1; // add a pseudo position after C-terminal
	for ( int j = 1; j <= 5; ++j ) {
		st_D(j,n2c,i) = D(j,n2c);
		st_D(j,c2n,i) = D(j,c2n);
		st_angle(j,n2c,i) = angle(j,n2c);
		st_angle(j,c2n,i) = angle(j,c2n);
	}

//chu   c2n direction, the last residue is special
	i = total_residue;

	subvec(Eposition(1,ca,i),Eposition(1, n,i),X(1,ph));
	subvec(Eposition(1, c,i),Eposition(1,ca,i),X(1,ps));
	refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
	build_atom(M,Eposition(1,c,1),cT(n,n2c),sT(n,n2c),psi(i),D(n,n2c),
	 pseudo_n,X(1,om)); // generate a pseudo N at total_res+1
	refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
	build_atom(M,pseudo_n,cT(ca,n2c),sT(ca,n2c),omega(i),D(ca,n2c),
	 pseudo_ca,X(1,ph)); // generate a pseudo CA at total_res+1
//     save them in the default template
	vector_copy(Eposition(1,c,i),st_default_template(1,1,c2n));
	vector_copy(pseudo_n ,st_default_template(1,2,c2n));
	vector_copy(pseudo_ca,st_default_template(1,3,c2n));

	st_D( c,c2n,i) = D( c,c2n); // use template
	st_D(ox,c2n,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));
	st_D(ca,c2n,i) = vec_dist(Eposition(1,ca,i),Eposition(1, c,i));
	st_D(cb,c2n,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
	st_D( n,c2n,i) = vec_dist(Eposition(1, n,i),Eposition(1,ca,i));

	st_angle( c,c2n,i) = angle( c,c2n); // use template
	st_angle(ca,c2n,i) = vec_angle(pseudo_n,Eposition(1,c,i),Eposition(1,ca,i));
	st_angle(ox,c2n,i) = vec_angle(pseudo_n,Eposition(1,c,i),Eposition(1,ox,i));
	st_angle(cb,c2n,i) =
	 vec_angle(Eposition(1, c,i),Eposition(1,ca,i),Eposition(1,cb,i));
	st_angle( n,c2n,i) =
	 vec_angle(Eposition(1, c,i),Eposition(1,ca,i),Eposition(1, n,i));

	st_cb_phi_offset(c2n,i) = dihedral(Eposition(1,cb,i),Eposition(1,ca,i),
	 Eposition(1,c,i),pseudo_n) - psi(i);
	st_ox_psi_offset(c2n,i) = dihedral(Eposition(1,ox,i),Eposition(1, c,i),
	 pseudo_n,pseudo_ca) - omega(i);

	for ( int i = total_residue - 1; i >= 1; --i ) {
		st_D( c,c2n,i) = vec_dist(Eposition(1, c,i),Eposition(1, n,i+1));
		st_D(ox,c2n,i) = vec_dist(Eposition(1,ox,i),Eposition(1, c,i));
		st_D(ca,c2n,i) = vec_dist(Eposition(1,ca,i),Eposition(1, c,i));
		st_D(cb,c2n,i) = vec_dist(Eposition(1,cb,i),Eposition(1,ca,i));
		st_D( n,c2n,i) = vec_dist(Eposition(1, n,i),Eposition(1,ca,i));

		st_angle( c,c2n,i) =
		 vec_angle( Eposition(1,ca,i+1), Eposition(1, n,i+1), Eposition(1, c,i) );
		st_angle(ca,c2n,i) =
		 vec_angle( Eposition(1, n,i+1), Eposition(1, c,i), Eposition(1,ca,i) );
		st_angle(ox,c2n,i) =
		 vec_angle( Eposition(1, n,i+1), Eposition(1, c,i), Eposition(1,ox,i) );
		st_angle(cb,c2n,i) =
		 vec_angle( Eposition(1, c,i), Eposition(1,ca,i), Eposition(1,cb,i) );
		st_angle( n,c2n,i) =
		 vec_angle( Eposition(1, c,i), Eposition(1,ca,i), Eposition(1, n,i) );

		st_cb_phi_offset(c2n,i) = dihedral(Eposition(1,cb,i),Eposition(1,ca,i),
		 Eposition(1,c,i),Eposition(1,n,i+1)) - psi(i);
		st_ox_psi_offset(c2n,i) = dihedral(Eposition(1,ox,i),Eposition(1, c,i),
		 Eposition(1,n,i+1),Eposition(1,ca,i+1)) - omega(i);
	}

	init_start_fold_angles();

	std::cout << "ATTENTION: new_bond_info updated and used for refolding now!\n";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin init_start_fold
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
init_start_fold()
{

	static bool s_init = { false };
	if ( s_init ) return;
	s_init = true;

	init_start_fold_angles();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin init_start_fold_angles
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bqian 12/21/2004
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
init_start_fold_angles()
{
	using namespace misc;
	using namespace refold_ns;
	using namespace start;
	using numeric::conversions::radians;

	float tmp;

	for ( int i = 1; i <= total_residue; ++i ) {
		tmp = radians( 180.0f - st_angle(n,n2c,i) );
		st_cT(n,n2c,i) = std::cos(tmp);
		st_sT(n,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(ca,n2c,i) );
		st_cT(ca,n2c,i) = std::cos(tmp);
		st_sT(ca,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(c,n2c,i) );
		st_cT(c,n2c,i) = std::cos(tmp);
		st_sT(c,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(ox,n2c,i) );
		st_cT(ox,n2c,i) = std::cos(tmp);
		st_sT(ox,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(cb,n2c,i) );
		st_cT(cb,n2c,i) = std::cos(tmp);
		st_sT(cb,n2c,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(n,c2n,i) );
		st_cT(n,c2n,i) = std::cos(tmp);
		st_sT(n,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(ca,c2n,i) );
		st_cT(ca,c2n,i) = std::cos(tmp);
		st_sT(ca,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(c,c2n,i) );
		st_cT(c,c2n,i) = std::cos(tmp);
		st_sT(c,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(ox,c2n,i) );
		st_cT(ox,c2n,i) = std::cos(tmp);
		st_sT(ox,c2n,i) = std::sin(tmp);

		tmp = radians( 180.0f - st_angle(cb,c2n,i) );
		st_cT(cb,c2n,i) = std::cos(tmp);
		st_sT(cb,c2n,i) = std::sin(tmp);
	}

	int i = 0;
	for ( int j = 1; j <= 5; ++j ) {
		st_cT(j,n2c,i) = cT(j,n2c);
		st_sT(j,c2n,i) = sT(j,c2n);
	}

	i = total_residue + 1;
	for ( int j = 1; j <= 5; ++j ) {
		st_cT(j,n2c,i) = cT(j,n2c);
		st_sT(j,c2n,i) = sT(j,c2n);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin backbone_fold
///
/// @brief
///  fold n,ca,c only using given torsion angles
///  nres + 1 c folded using psi of zero.
///
/// @detailed
///   for use by wobble, chuck, etc
///
/// @param  nres - [in] -
/// @param  phi - [in] -
/// @param  psi - [in] -
/// @param  omega - [in] -
/// @param  xyz - [out] -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   return array xyz must be declared at least nres+1
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
backbone_fold(int const nres, FArray1Da_float phi,FArray1Da_float psi, FArray1Da_float omega, FArray3Da_float xyz)
{
	using namespace param;
	using namespace refold_ns;

	xyz.dimension( 3, MAX_POS, nres+1 );
	phi.dimension( nres );
	psi.dimension( nres );
	omega.dimension( nres );

	FArray2D_float X( 3, 3 );
	FArray2D_float M( 3, 3 );
	FArray1D_float c_xyz( 3 );

// for backcompatibility to angles stub
	FArray1D_double stub_n( 3 );
	FArray1D_double stub_ca( 3 );
	FArray1D_double stub_c( 3 );

	//chu copy from the idealized template stub
	refold_default_template_stub(c_xyz, xyz(1,n,1),xyz(1,ca,1));

//car define initial bond vectors
	subvec(xyz(1,n,1), c_xyz, X(1,om));
	subvec(xyz(1,ca,1), xyz(1,n,1), X(1,ph));

	for ( int ires = 1; ires <= nres; ++ires ) {

//car c
		refold_coord_sys(X(1,om),X(1,ph),M(1,1),M(1,2),M(1,3));
		build_atom(M,xyz(1,ca,ires),cT(c,n2c),sT(c,n2c),phi(ires),D(c,n2c),
							 xyz(1,c,ires),X(1,ps));

//car n(i+1)
		refold_coord_sys(X(1,ph),X(1,ps),M(1,1),M(1,2),M(1,3));
		build_atom(M,xyz(1,c,ires),cT(n,n2c),sT(n,n2c),psi(ires),D(n,n2c),
							 xyz(1,n,ires+1),X(1,om));

//car ca(i+1)
		refold_coord_sys(X(1,ps),X(1,om),M(1,1),M(1,2),M(1,3));
		build_atom(M,xyz(1,n,ires+1),cT(ca,n2c),sT(ca,n2c),omega(ires),D(ca,n2c),
							 xyz(1,ca,ires+1),X(1,ph));
	}

//car  fold last c with phi = 0.0
	int stubres = nres + 1;
	refold_coord_sys(X(1,om),X(1,ph),M(1,1),M(1,2),M(1,3));
	build_atom(M,xyz(1,ca,stubres),cT(c,n2c),sT(c,n2c),0.0,D(c,n2c),
						 xyz(1,c,stubres),X(1,ps));

//car copy stub and save
	for ( int i = 1; i <= 3; ++i ) { // copy to double-precision namespace
		stub_n(i) = xyz(i,n,stubres);
		stub_ca(i) = xyz(i,ca, stubres);
		stub_c(i) = xyz(i,c,stubres);
	}
	angles_save_final_stub(stub_n,stub_ca,stub_c);

}




////////////////////////////////////////////////////////////////////////////////
/// @begin get_centroid_parm_eachres_default
///
/// @brief
/// Assign centroid parameters (angles and distances of centroid
///  relative to C-alpha for each residue) using default values.
///
/// @return
///
/// @param  res - [in] -
/// @param  centroid_parm_eachres - [out] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  rhiju
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
get_centroid_parm_eachres_default( FArray3Da_float centroid_parm_eachres, FArray1Da_int res)
{
	using namespace refold_ns;

 	int const numparams   = centroid_parm_eachres.size1();
 	int const numresidues = centroid_parm_eachres.size3();

 	for ( int i = 1; i <= numparams; ++i ) {
		for ( int j = 1; j <= numresidues; ++j ) {
 			int res_j = res(j);
			// If residue number is not between 1 and 20, no amino acid there.
			// Use alanine as a placeholder.
			if (res_j < 1)  res_j = 1;
			if (res_j > 20) res_j = 1;
 			for ( int k = -1; k <= 1; k += 2 ) {
// Note the weird switch in indices: k,j from res_j,k. We need to have the third index of
// centroid_parm_eachres be the residue position (j) for use in build_backbone and refold_update_coords.
// Unfortunately, the pre-existing centroid_parm had the residue type (res_j) in the second index.
 				centroid_parm_eachres( i, k, j) = centroid_parm( i, res_j, k);
 			}
 		}
 	}

	return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin get_centroid_parm_eachres_fromnative
///
///
/// @brief
/// Assign centroid parameters (angles and distances of centroid
///  relative to C-alpha for each residue) using default values.
///
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
///
/// @authors
///  rhiju
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
get_centroid_parm_eachres_fromnative( FArray3Da_float centroid_parm_eachres)
{
	using namespace native;

 	int const numparams   = centroid_parm_eachres.size1();
 	int const numresidues = centroid_parm_eachres.size3();

	for ( int i = 1; i <= numresidues; ++i ) {
		for ( int j = 1; j <= numparams; ++j ) {
			for ( int k = -1; k <= 1; k += 2 ) {
				centroid_parm_eachres(j, k, i) = native_centroid_parm_eachres(j, k, i);
			}
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin compute_centroid_parm_eachres
///
/// @brief
/// Assign centroid parameters (angles and distances of centroid
///  relative to C-alpha for each residue) based on heavy atom
///  sidechain positions in the inputted fullcoords (called by, e.g. refold)..
///
///
/// @return
///    centroid_parm_eachres -- parameters necessary to place
///    centroids relative to backbone for each residue in the protein.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///  rhiju
/// @last_modified 15-Dec-2005
/////////////////////////////////////////////////////////////////////////////////

FArray3Da_float
compute_centroid_parm_eachres(
	 FArray1Da_int res,
	 FArray1Da_int res_variant,
	 FArray3Da_float full_coord )
{
	using namespace misc;
	using namespace native;
	using namespace aaproperties_pack;
	using namespace param;
	using namespace refold_ns;
	using numeric::xyzVector_float;
	using numeric::sin_cos_range;

 	int const numparams = 5;
	int aa1, aav1, num_heavy, total_heavy_atoms, dir;
	xyzVector_float coord_N, coord_Ca, coord_C, coord_heavy, coord_avg;
	xyzVector_float xvec, yvec, zvec, coord_avg_norm;
	FArray3D_float centroid_parm_eachres(5, DRange(-1, 1), MAX_RES(), 0.0f );
	float projectontoy, projectontoz, costheta;

	bool const printstuff = false;

	for ( int i = 1; i <= total_residue; ++i ) {
		aa1  = res(i);
		aav1 = res_variant(i);
		num_heavy = nheavyatoms(aa1,aav1);

			//As a default just assign the standard Rosetta parameters.
		for ( int j = 1; j <= numparams; ++j ) {
			for ( int k = -1; k <= 1; k += 2 ) {
// Note the weird switch in indices: k,i from aa1,k. We need to have the third index of
// centroid_parm_eachres be the residue position (j) for use in build_backbone and refold_update_coords.
// Unfortunately, the pre-existing centroid_parm had the residue type (aa1) in the second index.
				if (aa1<1)  aa1=1;
				if (aa1>20) aa1=1;
				centroid_parm_eachres(j, k, i) = centroid_parm(j, aa1, k);
			}
		}

		coord_N  =  &full_coord(1, 1, i);
		coord_Ca =  &full_coord(1, 2, i);
		coord_C  =  &full_coord(1, 3, i);

		coord_avg = 0;
		total_heavy_atoms = num_heavy - 4;
		if (total_heavy_atoms > 0) { //Not glycine
			for ( int atom1 = 5; atom1 <= num_heavy; ++atom1 ) {
				coord_heavy = &full_coord(1, atom1, i);
				coord_avg += coord_heavy - coord_Ca;
				if (printstuff) std::cout << i << " " << aa1 << " " << aav1 << " " << num_heavy << " " <<
													coord_Ca[1] << " " << coord_heavy[1] << std::endl;
			}
			coord_avg /= float(total_heavy_atoms);
			if (printstuff) std::cout << "--------------------------------------------------" << std::endl;

			coord_avg_norm = coord_avg;
			coord_avg_norm = coord_avg_norm.normalize();

			//For n2c direction...
			dir = +1;
			xvec = (coord_Ca - coord_N).normalize();
			yvec = (coord_C - coord_Ca);
			zvec = cross( xvec, yvec).normalize();
			yvec = cross( zvec, xvec);
			zvec = zvec;
			//distance
			centroid_parm_eachres( 1, dir, i) = coord_avg.length();
			//phi
			projectontoy = dot_product( coord_avg_norm, yvec);
			projectontoz = dot_product( coord_avg_norm, zvec);
			centroid_parm_eachres( 2, dir, i) = std::atan2( projectontoz, projectontoy); //Yup, radians.
			//costheta
			costheta =  dot_product( coord_avg_norm, xvec);
			centroid_parm_eachres( 3, dir, i) = costheta;
			//sintheta
			centroid_parm_eachres( 4, dir, i) = std::sqrt( 1 - costheta*costheta);
			//Theta goes at end
			centroid_parm_eachres( 5, dir, i) = std::acos( sin_cos_range( costheta) ); // Again, radians


			//For c2n direction...
			dir = -1;
			xvec = (coord_Ca - coord_C).normalize();
			yvec = (coord_N - coord_Ca);
			zvec = cross( xvec, yvec).normalize();
			yvec = cross( zvec, xvec);
			zvec = zvec;
			//distance
			centroid_parm_eachres( 1, dir, i) = coord_avg.length();
			//phi
			projectontoy = dot_product( coord_avg_norm, yvec);
			projectontoz = dot_product( coord_avg_norm, zvec);
			centroid_parm_eachres( 2, dir, i) = std::atan2( projectontoz, projectontoy); //Yup, radians.
			//costheta
			costheta =  dot_product( coord_avg_norm, xvec);
			centroid_parm_eachres( 3, dir, i) = costheta;
			//sintheta
			centroid_parm_eachres( 4, dir, i) = std::sqrt( 1 - costheta*costheta);
			//Theta goes at end
			centroid_parm_eachres( 5, dir, i) = std::acos( sin_cos_range( costheta) ); // Again, radians
		}

	}

	if (printstuff) std::cout << "--------------------------------------------------" << std::endl;
	if (printstuff) std::cout << "--------------------------------------------------" << std::endl;

	return centroid_parm_eachres;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin get_use_native_centroid_flag
///
/// @brief
///  Look for "use_native_centroid" in command line.
///
/// @return
///    use_native_centroid
///
/// @authors
///  rhiju
/////////////////////////////////////////////////////////////////////////////////
bool get_use_native_centroid_flag() {
	static bool init = {false};
	static bool use_native_centroid = {false};

	if (!init) {
		use_native_centroid = truefalseoption("use_native_centroid");
		init = true;
	}

	return use_native_centroid;
}


////////////////////////////////////////////////////////////////
void
build_actual_centroid(int const aa,
											FArray2Da_float full_coord,
											FArray1Da_float centroid)
{
	using namespace aaproperties_pack;
	using namespace numeric;
	using namespace param;

	full_coord.dimension(3, MAX_ATOM() );
	centroid.dimension( 3 );

	int const num_heavy = nheavyatoms(aa,1);
	int const	total_heavy_atoms = num_heavy - 4;
	xyzVector_float coord_heavy, coord_avg;

	coord_avg = 0.0;

	if (total_heavy_atoms > 0) { //Not glycine
		for ( int atom1 = 5; atom1 <= num_heavy; ++atom1 ) {
			coord_heavy = &full_coord(1, atom1);
			coord_avg += coord_heavy;
		}
		coord_avg /= float(total_heavy_atoms);
	} else {
		coord_avg = &full_coord(1, 2); //CA
	}

	for (int k = 1; k<=3; ++k)	centroid(k) = coord_avg[k-1];
}
