// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   remodel_functions.hh
/// @brief  Pose manipulation functions for Possu.
/// @brief  Clipped out of epigraft.hh and altered to use std::vector.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_remodel_functions_HH_
#define INCLUDED_remodel_functions_HH_

// Rosetta headers
#include <aaproperties_pack.h>
#include <param.h>
#include <PackerTask.h>
#include <pack.h>
#include <enzyme_ns.h>
// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray3D.hh>

// C++ headers
#include <utility/vector1.hh>


/* begin code due to extractiong from epigraft.hh */

typedef int Integer;
typedef float Real;
//typedef pose_ns::Pose Pose;
using pose_ns::Pose;

/* end code due to extractiong from epigraft.hh */


/// @brief repetitive pose preparation
inline
void
prepare_pose(
	Pose & p,
	int const & nres
)
{
	p.simple_fold_tree( nres );
	p.set_fullatom_flag( true, false ); // force booleans: fullatom, repack
}


/// @brief transfer only partial identity information (res, res_variant, secstruct, name) between two Poses
/// @note  this function exists without transferring dihedrals and coordinates because Pose::set_coords
/// @note  does internal recalculations to dihedrals and requires all coordinates to exist
inline
void
transfer_identity_info_between_Poses(
	Pose const & p1,
	utility::vector1< bool > const & keep,
	Pose & p2,
	int const & p2_begin,
	bool const & keep_pdb_info = false
)
{
	for ( Integer residue = 1, last_residue = p1.total_residue(), p2_count = p2_begin; residue <= last_residue; ++residue ) {
		if ( keep[ residue ] ) {

			p2.set_res        ( p2_count, p1.res( residue ) );
			p2.set_res_variant( p2_count, p1.res_variant( residue ));
			p2.set_secstruct  ( p2_count, p1.secstruct( residue ) );
			p2.set_name       ( p2_count, p1.name( residue ) );

			++p2_count;
		}
	}

	if ( keep_pdb_info ) { // outside prior loop for speed
		for ( Integer residue = 1, last_residue = p1.total_residue(), p2_count = p2_begin; residue <= last_residue; ++residue ) {
			if ( keep[ residue ] ) {
				p2.pdb_info().set_pdb_chain( p2_count, p1.pdb_info().res_chain( residue ) );
				p2.pdb_info().set_pdb_res( p2_count, p1.pdb_info().pdb_res_num( residue ) );
				p2.pdb_info().set_pdb_insert_let( p2_count, p1.pdb_info().pdb_insert_let( residue ) );

				++p2_count;
			}
		}
	}
}


/// @brief fill full_coord from Pose, 'index' specifies the first residue in the full_coord
/// @brief to be filled and is required!  index is modified so that it ends up at the _next_
/// @brief residue to be filled
inline
void
fill_full_coord_from_Pose(
	Pose const & p,
	utility::vector1< bool > const & keep,
	FArray3D_float & full_coord,
	int & index
)
{
	FArray3D_float const & p_full_coord = p.full_coord();

	for ( Integer residue = 1, last_residue = p.total_residue(); residue <= last_residue; ++residue ) {
		if ( keep[ residue ] ) {
		    int const aa( p.res( residue ) );
			int const aav( p.res_variant( residue ) );
			for ( Integer atom = 1, last_atom = aaproperties_pack::natoms( aa, aav ); atom <= last_atom; ++atom ) {
				for ( Integer i = 1; i <= 3; ++i ) {
					full_coord( i, atom, index ) = p_full_coord( i, atom, residue );
				}
			}

			++index;
		}
	}
}


/// @brief make Eposition from full_coord
/// @note  in Rosetta full_coord is somtimes dimensioned with MAX_RES... which means we
/// @note  have to pass "nres" and can't use FArray's internal size function because
/// @note  the two might not be in sync
inline
void
full_coord_to_Eposition(
	int const & nres,
	FArray3D_float & full_coord,
	FArray3D_float & Eposition
)
{
	Eposition.dimension( 3, param::MAX_POS, nres );

	for ( Integer residue = 1; residue <= nres; ++residue ) {
		for ( Integer i = 1; i <= 3; ++ i ) {
				Eposition(i, 1, residue) = full_coord(i, 1, residue); // N
				Eposition(i, 2, residue) = full_coord(i, 2, residue); // CA
				Eposition(i, 4, residue) = full_coord(i, 3, residue); // C
				Eposition(i, 5, residue) = full_coord(i, 4, residue); // O
				Eposition(i, 3, residue) = full_coord(i, 5, residue); // CB
		}
	}
}


inline
void
fragment_Pose(
	Pose const & p,
	utility::vector1< bool > const & keep,
	Pose & fragmented,
	bool const & keep_pdb_info = false
)
{
	// count number of residue
	Integer total_residue = 0;
	for ( Integer residue = 1, last_residue = keep.size(); residue <= last_residue; ++residue ) {
		if ( keep[ residue ] ) {
			total_residue++;
		}
	}

	prepare_pose( fragmented, total_residue );

	// transfer non-coordinate info
	transfer_identity_info_between_Poses( p, keep, fragmented, 1, keep_pdb_info );

	// make full_coord
	int counter = 1;
	FArray3D_float full_coord( 3, param::MAX_ATOM(), fragmented.total_residue() );
	fill_full_coord_from_Pose( p, keep, full_coord, counter );

	// make Eposition
	FArray3D_float Eposition( 3, param::MAX_POS, fragmented.total_residue() );
	full_coord_to_Eposition( fragmented.total_residue(), full_coord, Eposition );

	fragmented.set_coords( false, Eposition, full_coord, false ); // internally recomputes all angles
}

void
design_using_design_matrix(
                           pose_ns::Pose & pose_in,
                           FArray2D_bool const & design_matrix_local
                           )
{
  //
  //fill misc arrays from this pose
  //
  //  Note: must do this (b/c sets total_residue) before setting design_matrix
  //
  pose_to_misc( pose_in );

  //
  //Set design globals for design_matrix to control which aa allowed at which positions.
  //

  design::use_design_matrix = true;
  design::design_matrix = false; //design_matrix( MAX_AA(), MAX_RES() );
  design::design_commands::try_both_his_tautomers = true;
  //trigger jk's additional call to optimizeH to build "similar" rotamers ( see RotamerSet.cc )
  //this not set in options.cc b/c input_fa not set in pose1 mode.
  //this adds just a handful of additional rotamers in each nglyco site design.
  design::active_rotamer_options.use_input_sc = true;
  //if ( truefalseoption( "large_rotamer_set" ) ) select_rotamer_set( "large" );
  //if ( truefalseoption( "favor_native_residue" ) ) {
   // set_favor_native_residue( true );
   // float native_bonus_tmp;
   // realafteroption("favor_native_residue",-1.5, native_bonus_tmp);
   // set_native_bonus( native_bonus_tmp );
 // }

  int const nres = pose_in.total_residue();

  for ( int ires = 1 ; ires <= nres ; ++ires ) {
    for ( int aa = 1; aa <= param::MAX_AA(); aa++ ) {
      design::design_matrix( aa, ires ) = design_matrix_local( aa, ires );
    }
  }

  //
  //variables to set for the call to pack_rotamers
  //
  std::string pack_mode( "design" );
  bool make_output_file( false );
  FArray1D_bool allow_repack_local( param::MAX_RES()(), false );
  bool include_current( true );
  bool include_extra ( false );
  FArray2D_int extra_rot( param::MAX_CHI, param::MAX_RES()() ); // dummy variable in this instance
  FArray2D_float extra_chi( param::MAX_CHI, param::MAX_RES()() ); // dummy variable in this instance

  //yl, Create PackerTask and setup values before pass into pack_rotamers
  PackerTask Task(pose_in);
  Task.set_task( pack_mode, make_output_file, allow_repack_local,
                 include_current, include_extra, extra_rot, extra_chi);
  //bk set variables that specify which residues to vary
  Task.setup_residues_to_vary();

	if( enzyme::ligand_weight_scale != 1.0f ) {
		for( int seqpos1=1; seqpos1<=pose_in.total_residue();seqpos1++ ) {
			if( !param_aa::is_ligand( pose_in.res(seqpos1) ) ) continue;
			for( int seqpos2=1; seqpos2<=pose_in.total_residue();seqpos2++ ) {
				if( param_aa::is_ligand( pose_in.res(seqpos2) ) ) continue;
				Task.set_residue_weight( enzyme::ligand_weight_scale, seqpos1, seqpos2 );
			}
		}
	}

  //
  //design using misc arrays
  //
  pack_rotamers( pose_in, Task );

  //
  //retrieve design from misc.
  //
  // yab: merge into mainline; below isn't necessary anymore
//  bool const fullatom( true );
//  bool const ideal_pose( false );
//  bool const coords_init( true );
//  pose_from_misc( pose_out, fullatom, ideal_pose, coords_init );
}

void 
dump_fragments(
							 std::string const & filename,
							 int const size,
							 int const first_window,
							 int const last_window,
							 int const total_residue
)
{
	using namespace fragments;	
	utility::io::ozstream out( filename.c_str() );
	int const size_bin ( get_index_by_frag_size(size) );

	int const end_window( std::min( last_window, int(total_residue - size + 1 )));

	for ( int pos= first_window; pos<= end_window; ++pos ) {
		int const depth( align_depth( pos, size_bin ) );
		out << " position:" << I(13,pos) << " neighbors:" << I(13,depth);
		out << '\n';
		out << '\n';
		for ( int i=1; i<= depth; ++i ) {
			for ( int j=0; j< size; ++j ) {
				out << 
					' ' << align_name  ( pos, i, size_bin) << 
					' ' << align_chain ( pos, i, size_bin) << 

					' ' << I(5,align_resseq( pos, i, size_bin)+j)  <<  

					' ' << align_res_id( pos, i, j, size_bin) << 
					' ' << ss_type( pos, i, j, size_bin )
 					 << F(9,3,align_phi  ( pos, i, j, size_bin ))
					 << F(9,3,align_psi  ( pos, i, j, size_bin ))
					 << F(9,3,align_omega( pos, i, j, size_bin ));
				out << '\n';
			}
			out << '\n';
		}
	} 

	return;
}

#endif /*INCLUDED_remodel_functions_HH_*/
