// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "orient_rms.h"
#include "rms.h"
#include "maxsub.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>

// Utility headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <complex>
#include <cstdlib>
#include <iostream>


// Namespaces
namespace rms_obj {
	FArray2D_double xm( 3, 3 ); // note zero origin
	double xre;
	double xrp;
	FArray1D_double xse( 3 );
	FArray1D_double xsp( 3 );
	int count;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rms_FArray
///
/// @brief
///    Wrapper function that calculates RMS between two arbitrary arrays
/// @detailed
///
/// @param  vector_list_1 - [in]? -
/// @param  vector_list_2 - [in]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  M. Tyka
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
rms_FArray(
	FArray2D_float &vector_list_1,
  FArray2D_float &vector_list_2
)
{

  // convert to double arrays
	FArray2D_double fvector_list_1( vector_list_1  );
	FArray2D_double fvector_list_2( vector_list_2  );

  return rms_FArray(fvector_list_1,fvector_list_2);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rms_FArray
///
/// @brief
///    Wrapper function that calculates RMS between two arbitrary arrays
/// @detailed
///
/// @param  vector_list_1 - [in]? -
/// @param  vector_list_2 - [in]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  M. Tyka
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
rms_FArray(
	FArray2D_double &vector_list_1,
  FArray2D_double &vector_list_2
)
{

  // check array dimensions
  if( vector_list_1.size1() != 3 ){
    std::cout << "ERROR: vector_list_1 must be an FArray2D_double(3,N)" << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }
  if( vector_list_1.size1() != 3 ){
    std::cout << "ERROR: vector_list_2 must be an FArray2D_double(3,N)" << std::endl;
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }
  if( vector_list_1.size2() != vector_list_2.size2() ){
		std::cout << "Length mismatch in rms.cc:rms_FArray(..): vector_list_1: " << vector_list_1.size2() << "  vector_list_2: " << vector_list_2.size2() ;
		return 0.0;
	}

	// calc rms
	FArray1D_double const ww( vector_list_1.size2(), 1.0 );
	FArray2D_double uu( 3, 3 );
	double ctx;
	findUU( vector_list_1, vector_list_2, ww, vector_list_1.size2(), uu, ctx );
	float fast_rms;
	calc_rms_fast( fast_rms, vector_list_1, vector_list_2, ww, vector_list_1.size2(), ctx );
	return fast_rms;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rms_setup
///
/// @brief
///
/// @detailed
///  precomputes moments and cross moments of two aligned point vectors
///  this function is just a middle man calling others to do its work
///  in particular I note that if you are going to be calling this with
///  various pairings of the same arrays, then its only the xm array
///  that is coing to change, and you might want to rewrite this for
///  better effiency.
///
/// @param  n - [in/out]? -
/// @param  xe - [in/out]? -
/// @param  xp - [in/out]? -
/// @param  xm - [in/out]? -
/// @param  xre - [in/out]? -
/// @param  xrp - [in/out]? -
/// @param  xse - [in/out]? -
/// @param  xsp - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors copyright 2001 Charlie E. M. Strauss 2001
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rms_setup(
	int & n,
	FArray2Da_double xe,
	FArray2Da_double xp,
	FArray3Da_double xm,
	FArray1Da_double xre,
	FArray1Da_double xrp,
	FArray2Da_double xse,
	FArray2Da_double xsp
)
{
	xe.dimension( 3, n );
	xp.dimension( 3, n );
	xm.dimension( 3, 3, SRange( 0, n ) );
	xre.dimension( SRange( 0, n ) );
	xrp.dimension( SRange( 0, n ) );
	xse.dimension( 3, SRange( 0, n ) );
	xsp.dimension( 3, SRange( 0, n ) );


	prepare_cross_moments(n,xe,xp,xm);
	prepare_rsums(n,xe,xse);
	prepare_rsums(n,xp,xsp);
	prepare_radii(n,xe,xre);
	prepare_radii(n,xp,xrp);
}


//    // the headache here is dealing with the center of mass in a sliding way.

////////////////////////////////////////////////////////////////////////////////
/// @begin rms2_fast2
///
/// @brief
///   computes the rms between two non-weighted point vectors.
///   intended to be applied to a windowed region of a large array of points.
///
/// @detailed
///   the vectors are not handed in.
///   instead their pre-computed sliding moments are handed in
///   plus the start(ii) and stop(jj) atoms defining the window in the larger
///   array
///
///  assumes caller is not an idiot:  jj > ii > 0
///   det is an out value of the determinant of the cross moment matrix
///   returned value is the rms
///
///   most of this is double for good reasons.  first there are some large
///   differences of small numbers.  and second the rsymm_eignen() function can
///   internally have numbers larger than the largest double number.
///   (you could do some fancy foot work to rescale things if you really
///   had a problem with this.)
///
/// @param  ii - [in/out]? -
/// @param  jj - [in/out]? -
/// @param  xm - [in/out]? - sliding cross moments array
/// @param  xre - [in/out]? - liding radius array for the first set of points
/// @param  xrp - [in/out]? - sliding radius array for the second set of points
/// @param  xse - [in/out]? - liding sum_coord array for first
/// @param  xsp - [in/out]? - liding sum_coord array for the second
/// @param  det - [in/out]? -
/// @param  ev - [in/out]?-
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
///@author charlie strauss 2001
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
double
rms2_fast2(
	int & ii,
	int & jj,
	FArray3Da_double xm, // note zero origin
	FArray1Da_double xre,
	FArray1Da_double xrp,
	FArray2Da_double xse,
	FArray2Da_double xsp,
	FArray1Da_double ev
)
{
	xm.dimension( 3, 3, SRange( 0, jj ) );
	xre.dimension( SRange( 0, jj ) );
	xrp.dimension( SRange( 0, jj ) );
	xse.dimension( 3, SRange( 0, jj ) );
	xsp.dimension( 3, SRange( 0, jj ) );
	ev.dimension( 3 );

	int kk = ii - 1;
	int n = jj - kk;

	return rms2_fast3(n,xm(1,1,kk),xre(kk),xrp(kk),xse(1,kk),xsp(1,kk),
	 xm(1,1,jj),xre(jj),xrp(jj),xse(1,jj),xsp(1,jj));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rms2_fast3
///
/// @brief
///   computes the rms between two non-weighted point vectors.
///   intended to be applied to a windowed region of a large array of points.
///
/// @detailed
///
///   the vectors are not handed in.
///   instead their pre-computed sliding moments are handed in
///   plus the start(ii) and stop(jj) atoms defining the window in the larger
///   array
///   xm is the sliding cross moments array
///   xre is the sliding radius array for the first set of points
///   xrp is the sliding radius array for the second set of points
///   xse is the sliding sum_coord array for first
///   xsp is the sliding sum_coord array for the second
///
///  assumes caller is not an idiot:  jj > ii > 0
///
///  det is an out value of the determinant of the cross moment matrix
///  returned value is the rms
///
///  most of this is double for good reasons.  first there are some large
///  differences of small numbers.  and second the rsymm_eignen() function can
///  internally have numbers larger than the largest double number.
///  (you could do some fancy foot work to rescale things if you really
///  had a problem with this.)
///
/// @param  n - [in/out]? -
/// @param  xmi - [in/out]? -
/// @param  xrei - [in/out]? -
/// @param  xrpi - [in/out]? -
/// @param  xsei - [in/out]? -
/// @param  xspi - [in/out]? -
/// @param  xmj - [in/out]? -
/// @param  xrej - [in/out]? -
/// @param  xrpj - [in/out]? -
/// @param  xsej - [in/out]? -
/// @param  xspj - [in/out]? -
///
///\RETURN
///
///\GLOBAL_READ
///
///\GLOBAL_WRITE
///
/// @remarks
///
/// @references
///
/// @authors charlie strauss 2001
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
double
rms2_fast3(
	int & n,
	FArray2Da_double xmi, // note zero origin
	double & xrei,
	double & xrpi,
	FArray1Da_double xsei,
	FArray1Da_double xspi,
	FArray2Da_double xmj, // note zero origin
	double & xrej,
	double & xrpj,
	FArray1Da_double xsej,
	FArray1Da_double xspj
)
{

	xmi.dimension( 3, 3 );
	xsei.dimension( 3 );
	xspi.dimension( 3 );
	xmj.dimension( 3, 3 );
	xsej.dimension( 3 );
	xspj.dimension( 3 );


	// local
	double det;
	int i,j,k;
	double temp1,invmass;
	FArray1D_double come( 3 );
	FArray1D_double comp( 3 );
	FArray1D_double ev( 3 );
	FArray2D_double m_moment( 3, 3 );
	FArray2D_double rr_moment( 3, 3 );
	double rms_ctx,rms2_sum;
	double handedness;


// compute center of mass

	invmass = 1.0/(n); // 1 over the mass
	come(1) = (xsej(1)-xsei(1))*invmass; // x_com
	come(2) = (xsej(2)-xsei(2))*invmass; // y_com
	come(3) = (xsej(3)-xsei(3))*invmass; // z_com

	comp(1) = (xspj(1)-xspi(1))*invmass;
	comp(2) = (xspj(2)-xspi(2))*invmass;
	comp(3) = (xspj(3)-xspi(3))*invmass;

//       Make cross moments matrix

	for ( k = 1; k <= 3; ++k ) {
//        // for ( j = 1; j <= 3; ++j ) {
//             //  Ye Olde moments trick:   sum (x-x0)(y-y0) = sum (xy-x0y0)
		m_moment(k,1) = (xmj(k,1)-xmi(k,1))*invmass - come(k)*comp(1); // flopped
		m_moment(k,2) = (xmj(k,2)-xmi(k,2))*invmass - come(k)*comp(2);
		m_moment(k,3) = (xmj(k,3)-xmi(k,3))*invmass - come(k)*comp(3);
//        // }
	}

	det = det3(m_moment);

	if ( std::abs(det) <= 1.0E-24 ) {
//     //std::cout << "Warning:degenerate cross moments: det=" << det << std::endl;
//      // returning a zero rms, to avoid any chance of
//      // Floating Point Errors.

		return 0.0;

	}
//        // get handedness  of frame from determinant
	handedness = sign(1.0,det);
//       // weird but documented "feature" of sign(a,b) (but not SIGN) is
//       // that if fails if a < 0

//           //  multiply cross moments by itself

	for ( i = 1; i <= 3; ++i ) {
		for ( j = i; j <= 3; ++j ) {
			rr_moment(j,i) = rr_moment(i,j) =
			 m_moment(1,i)*m_moment(1,j) +
			 m_moment(2,i)*m_moment(2,j) +
			 m_moment(3,i)*m_moment(3,j);
		}
	}


//            //  compute eigen values of cross-cross moments

	rsym_eigenval(rr_moment,ev);

	if ( handedness < 0 ) {

//      // reorder eigen values  so that ev(3) is the smallest eigenvalue

		if ( ev(2) > ev(3) ) {
			if ( ev(3) > ev(1) ) {
				temp1 = ev(3);
				ev(3) = ev(1);
				ev(1) = temp1;
			}
		} else {
			if ( ev(2) > ev(1) ) {
				temp1 = ev(3);
				ev(3) = ev(1);
				ev(1) = temp1;
			} else {
				temp1 = ev(3);
				ev(3) = ev(2);
				ev(2) = temp1;
			}
		}
	}
//                 // ev(3) is now the smallest eigen value.  the other two are not
//                 //  sorted.


//            // now we must catch the special case of the rotation with inversion.
//            // we cannot allow inversion rotations.
//            // fortunatley, and curiously, the optimal non-inverted rotation
//            // matrix
//            // will have the similar eigen values.
//            // we just have to make a slight change in how we handle things
//            // depending on determinant

	rms_ctx = std::sqrt(std::abs(ev(1))) + std::sqrt(std::abs(ev(2))) +
	 handedness*std::sqrt(std::abs(ev(3)));

//            // the abs() are theoretically unneccessary since the eigen values
//            // of a real symmetric
//            // matrix are non-negative.  in practice sometimes small eigen vals
//            // end up just negative
	temp1 = ( come(1) * come(1) ) + ( come(2) * come(2) ) + ( come(3) * come(3) ) +
	 ( comp(1) * comp(1) ) + ( comp(2) * comp(2) ) + ( comp(3) * comp(3) );

	rms2_sum = (xrej-xrei + xrpj-xrpi )*invmass - temp1;

//            // and combine the outer and cross terms into the final calculation.
//            //  (the abs() just saves us a headache when the roundoff error
//            // accidantally makes the sum negative)

	return std::sqrt(std::abs(rms2_sum-2.0*rms_ctx));
}




////////////////////////////////////////////////////////////////////////////////
/// @begin rms2_fast
///
/// @brief  computes the rms between two non-weighted point vectors.
///
/// @detailed
///   intended to be applied to a windowed region of a large array of points.
///
///
///   the vectors are not handed in.
///   instead their pre-computed sliding moments are handed in
///   plus the start(ii) and stop(jj) atoms defining the window in the larger
///   array
///
///   most of this is double for good reasons.  first there are some large
///   differences of small numbers.  and second the rsymm_eignen() function can
///   internally have numbers larger than the largest double number.
///   (you could do some fancy foot work to rescale things if you really
///   had a problem with this.)
///
/// @param  ii - [in/out]? -  assumes caller is not an idiot:  jj > ii > 0
/// @param  jj - [in/out]? -
/// @param  xm - [in/out]? -  xm is the sliding cross moments array
/// @param  xre - [in/out]? - xre is the sliding radius array for the first set of points
/// @param  xrp - [in/out]? - xrp is the sliding radius array for the second set of points
/// @param  xse - [in/out]? - xse is the sliding sum_coord array for first
/// @param  xsp - [in/out]? - xsp is the sliding sum_coord array for the second
/// @param  det   [in/out]? - det is an out value of the determinant of the cross moment matrix
///   returned value is the rms
/// @param  ev - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors charlie strauss
///
/// @last_modified 2001
/////////////////////////////////////////////////////////////////////////////////
double
rms2_fast(
	int & ii,
	int & jj,
	FArray3Da_double xm, // note zero origin
	FArray1Da_double xre,
	FArray1Da_double xrp,
	FArray2Da_double xse,
	FArray2Da_double xsp,
	double & det,
	FArray1Da_double ev
)
{

	xm.dimension( 3, 3, SRange( 0, jj ) );
	xre.dimension( SRange( 0, jj ) );
	xrp.dimension( SRange( 0, jj ) );
	xse.dimension( 3, SRange( 0, jj ) );
	xsp.dimension( 3, SRange( 0, jj ) );
	ev.dimension( 3 );


	// local
	int i,j,k;
	double temp1,invmass;
	FArray1D_double come( 3 );
	FArray1D_double comp( 3 );
	FArray2D_double m_moment( 3, 3 );
	FArray2D_double rr_moment( 3, 3 );
	double rms_ctx,rms2_sum;
	double handedness;


// compute center of mass

	invmass = 1.0/(jj-ii+1.0); // 1 over the mass
	come(1) = (xse(1,jj)-xse(1,ii-1))*invmass; // x_com
	come(2) = (xse(2,jj)-xse(2,ii-1))*invmass; // y_com
	come(3) = (xse(3,jj)-xse(3,ii-1))*invmass; // z_com

	comp(1) = (xsp(1,jj)-xsp(1,ii-1))*invmass;
	comp(2) = (xsp(2,jj)-xsp(2,ii-1))*invmass;
	comp(3) = (xsp(3,jj)-xsp(3,ii-1))*invmass;

//       Make cross moments matrix

	for ( k = 1; k <= 3; ++k ) {
//        // for ( j = 1; j <= 3; ++j ) {
//             //  Ye Olde moments trick:   sum (x-x0)(y-y0) = sum (xy-x0y0)
		m_moment(k,1) = (xm(k,1,jj)-xm(k,1,ii-1))*invmass - come(k)*comp(1);
		 // flopped
		m_moment(k,2) = (xm(k,2,jj)-xm(k,2,ii-1))*invmass - come(k)*comp(2);
		m_moment(k,3) = (xm(k,3,jj)-xm(k,3,ii-1))*invmass - come(k)*comp(3);
//        // }
	}

	det = det3(m_moment);

	if ( std::abs(det) <= 1.0E-24 ) {
//               //std::cout << "Warning:degenerate cross moments: det=" << det << std::endl;
//                // returning a zero rms, to avoid any chance of
//                // Floating Point Errors.

		return 0.0;

	}
//             // get handedness  of frame from determinant
	handedness = sign(1.0,det);
//             // weird but documented "feature" of sign(a,b) (but not SIGN) is
//             // that if fails if a < 0

//           //  multiply cross moments by itself

	for ( i = 1; i <= 3; ++i ) {
		for ( j = i; j <= 3; ++j ) {
			rr_moment(j,i) = rr_moment(i,j) =
			 m_moment(1,i)*m_moment(1,j) +
			 m_moment(2,i)*m_moment(2,j) +
			 m_moment(3,i)*m_moment(3,j);
		}
	}


//            //  compute eigen values of cross-cross moments

	rsym_eigenval(rr_moment,ev);

	if ( handedness < 0 ) {

//         // reorder eigen values  so that ev(3) is the smallest eigenvalue

		if ( ev(2) > ev(3) ) {
			if ( ev(3) > ev(1) ) {
				temp1 = ev(3);
				ev(3) = ev(1);
				ev(1) = temp1;
			}
		} else {
			if ( ev(2) > ev(1) ) {
				temp1 = ev(3);
				ev(3) = ev(1);
				ev(1) = temp1;
			} else {
				temp1 = ev(3);
				ev(3) = ev(2);
				ev(2) = temp1;
			}
		}
	}
//        // ev(3) is now the smallest eigen value.  the other two are not
//        //  sorted.


//       // now we must catch the special case of the rotation with inversion.
//       // we cannot allow inversion rotations.
//       // fortunatley, and curiously, the optimal non-inverted rotation
//       // matrix
//       // will have the similar eigen values.
//       // we just have to make a slight change in how we handle things
//       // depending on determinant


	rms_ctx = std::sqrt(std::abs(ev(1))) + std::sqrt(std::abs(ev(2))) +
	 handedness*std::sqrt(std::abs(ev(3)));


//        // the abs() are theoretically unneccessary since the eigen values
//        // of a real symmetric
//        // matrix are non-negative.  in practice sometimes small eigen vals
//        // end up just negative
	temp1 = ( come(1) * come(1) ) + ( come(2) * come(2) ) + ( come(3) * come(3) ) +
	 ( comp(1) * comp(1) ) + ( comp(2) * comp(2) ) + ( comp(3) * comp(3) );

	rms2_sum = (xre(jj)-xre(ii-1) + xrp(jj)-xrp(ii-1) )*invmass - temp1;

//        // and combine the outer and cross terms into the final calculation.
//        //  (the abs() just saves us a headache when the roundoff error
//        // accidantally makes the sum negative)

	return std::sqrt(std::abs(rms2_sum-2.0*rms_ctx));
}




////////////////////////////////////////////////////////////////////////////////
/// @begin prepare_cross_moments
///
/// @brief  computes a 3x3 matrix of cross moments between the x,y,z components of
///  the two input vectors.
///
/// @detailed
///  the output is the running sum of these matricies
///
/// @param  nover - [in/out]? -
/// @param  xe - [in/out]? -
/// @param  xp - [in/out]? -
/// @param  xm - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors charlie E. M. Strauss 2001
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
prepare_cross_moments(
	int & nover,
	FArray2Da_double xe,
	FArray2Da_double xp,
	FArray3Da_double xm
)
{
	xe.dimension( 3, nover );
	xp.dimension( 3, nover );
	xm.dimension( 3, 3, SRange( 0, nover ) ); // note zero origin


//    // init array we will making into a running sum
	for ( int k = 1; k <= 3; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			xm(j,k,0) = 0.0;
		}
	}

	for ( int i = 1; i <= nover; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			double const xp_ji = xp(j,i);
			for ( int k = 1; k <= 3; ++k ) {
				xm(k,j,i) = xm(k,j,i-1) + xp_ji*xe(k,i); // flopped
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin prepare_radii
///
/// @brief this computes the running sum of the x
///
/// @detailed the output is an array containing the running sum of the radii squared
///
/// @param  n - [in/out]? -
/// @param  xu - [in/out]? -
/// @param  xr - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
prepare_radii(
	int & n,
	FArray2Da_double xu,
	FArray1Da_double xr
)
{
	xu.dimension( 3, n );
	xr.dimension( SRange( 0, n ) );


//    // local vars

	xr(0) = 0.0;
	for ( int i = 1; i <= n; ++i ) {
		xr(i) = xr(i-1) +
		 ( xu(1,i) * xu(1,i) ) + ( xu(2,i) * xu(2,i) ) + ( xu(3,i) * xu(3,i) );
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin prepare_rsums
///
/// @brief  this computes the running sum of the x
///
/// @detailed the output is an array containing the running sum of the radii squared
///
/// @param  n - [in/out]? -
/// @param  xu - [in/out]? -
/// @param  xs - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
prepare_rsums(
	int & n,
	FArray2Da_double xu,
	FArray2Da_double xs
)
{
	xu.dimension( 3, n );
	xs.dimension( 3, SRange( 0, n ) );


	xs(1,0) = 0.0;
	xs(2,0) = 0.0;
	xs(3,0) = 0.0;
	for ( int i = 1; i <= n; ++i ) {
		xs(1,i) = xs(1,i-1)+xu(1,i);
		xs(2,i) = xs(2,i-1)+xu(2,i);
		xs(3,i) = xs(3,i-1)+xu(3,i);
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin clear_rms
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
clear_rms()
{
	using namespace rms_obj;

	// init array we will making into a running sum
	for ( int j = 1; j <= 3; ++j ) {
		for ( int k = 1; k <= 3; ++k ) {
			xm(j,k) = 0.0;
		}
	}
	xre = 0.0;
	xrp = 0.0;

	xse(1) = 0.0;
	xse(2) = 0.0;
	xse(3) = 0.0;
	xsp(1) = 0.0;
	xsp(2) = 0.0;
	xsp(3) = 0.0;
	count = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_rms
///
/// @brief  computes a 3x3 matrix of cross moments between the x,y,z components of
///  the two input vectors.
///
/// @detailed  the output is the running sum of these matricies
///
/// @param  i - [in/out]? -
/// @param  xp - [in/out]? -
/// @param  xe - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
add_rms(
	int i,
	FArray2Da_double xp,
	FArray2Da_double xe
)
{
	using namespace rms_obj;

	xp.dimension( 3, i );
	xe.dimension( 3, i );


	++count;
	for ( int j = 1; j <= 3; ++j ) {
		for ( int k = 1; k <= 3; ++k ) {
			xm(k,j) += xp(j,i) * xe(k,i); // flopped
		}
	}

	xre += ( xe(1,i) * xe(1,i) ) + ( xe(2,i) * xe(2,i) ) + ( xe(3,i) * xe(3,i) );
	xrp += ( xp(1,i) * xp(1,i) ) + ( xp(2,i) * xp(2,i) ) + ( xp(3,i) * xp(3,i) );

	xse(1) += xe(1,i);
	xse(2) += xe(2,i);
	xse(3) += xe(3,i);
	xsp(1) += xp(1,i);
	xsp(2) += xp(2,i);
	xsp(3) += xp(3,i);

}


////////////////////////////////////////////////////////////////////////////////
/// @begin rmsfitca3
///
/// @brief
///
/// @detailed
///   This function gets its alignment info via a namespace!
///   Alignment (rotation matrix) and rms(esq) are computed on the basis
///   of residues previously designated by calls to add_rms().
///   However, the rotation is applied to all Npoints of XX0,yy0 with the
///   results returned in xx,yy.
///
///   most of this is double for good reasons.
///   first there are some large differences of small numbers.
///   second the rsymm_eignen() function can internally have numbers
///   larger than the largest double number.  (you could do some fancy foot work
///   to rescale m_moment if you really had a problem with this.)
///
/// @param  npoints - [in/out]? -
/// @param  xx0 - [in/out]? -
/// @param  xx - [in/out]? -
/// @param  yy0 - [in/out]? -
/// @param  yy - [in/out]? -
/// @param  esq - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   NOTE: det is a double precision real
///   NOTE: (xx,yy) can be same arrays as (xx_0,yy_0) if desired
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
rmsfitca3(
	int npoints,
	FArray2Da_double xx0,
	FArray2Da_double xx,
	FArray2Da_double yy0,
	FArray2Da_double yy,
	double & esq
)
{
	using namespace rms_obj;

	xx0.dimension( 3, npoints );
	xx.dimension( 3, npoints );
	yy0.dimension( 3, npoints );
	yy.dimension( 3, npoints );

	// local
	double det;
	double temp1,mass;
	FArray1D_double come( 3 );
	FArray1D_double comp( 3 );
	FArray1D_double ev( 3 );
	FArray2D_double m_moment( 3, 3 );
	FArray2D_double rr_moment( 3, 3 );
	double rms_ctx,rms2_sum;
	double handedness;

	FArray2D_double r( 3, 3 );
	FArray1D_double t( 3 );


// compute center of mass

	mass = count;
	come(1) = xse(1)/mass; // x_com
	come(2) = xse(2)/mass; // y_com
	come(3) = xse(3)/mass; // z_com

	comp(1) = xsp(1)/mass;
	comp(2) = xsp(2)/mass;
	comp(3) = xsp(3)/mass;


//       Make cross moments matrix

	for ( int k = 1; k <= 3; ++k ) {
		for ( int j = 1; j <= 3; ++j ) {
			m_moment(k,j) = xm(k,j)/mass - come(k)*comp(j); // flopped com
		}
	}

	det = det3(m_moment); // get handedness  of frame from determinant

	if ( std::abs(det) <= 1.0E-24 ) {
//   //std::cout << "Warning:degenerate cross moments: det=" << det << std::endl;
//   // might think about returning a zero rms, to avoid any chance of
//   // Floating Point Errors?

		esq = 0.0;
		return;

	}
	handedness = sign(1.0,det);
//    // weird but documented "feature" of sign(a,b) (but not SIGN) is
//    // that if fails if a < 0

//    //  multiply cross moments by itself

	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = i; j <= 3; ++j ) {
			rr_moment(j,i) = rr_moment(i,j) =
			 m_moment(1,i)*m_moment(1,j) +
			 m_moment(2,i)*m_moment(2,j) +
			 m_moment(3,i)*m_moment(3,j);
		}
	}

//    // compute eigen values of cross-cross moments

	rsym_eigenval(rr_moment,ev);


//    // reorder eigen values  so that ev(3) is the smallest eigenvalue

	if ( ev(2) > ev(3) ) {
		if ( ev(3) > ev(1) ) {
			temp1 = ev(3);
			ev(3) = ev(1);
			ev(1) = temp1;
		}
	} else {
		if ( ev(2) > ev(1) ) {
			temp1 = ev(3);
			ev(3) = ev(1);
			ev(1) = temp1;
		} else {
			temp1 = ev(3);
			ev(3) = ev(2);
			ev(2) = temp1;
		}
	}

//     // ev(3) is now the smallest eigen value.  the other two are not
//     //  sorted.  This is prefered order for computing the rotation matrix

	rsym_rotation(m_moment,rr_moment,ev,r);

//            // now we rotate and offset all npoints


	for ( int i = 1; i <= npoints; ++i ) {
		for ( int k = 1; k <= 3; ++k ) { // remove center of mass
			yy(k,i) = yy0(k,i)-come(k);
			xx(k,i) = xx0(k,i)-comp(k);
		}
		for ( int j = 1; j <= 3; ++j ) { // compute rotation
//               // temp1 = 0.0;
//               // for ( k = 1; k <= 3; ++k ) {
//               //  temp1 += r(j,k)*yy(k,i);
//               // }
//               // t(j) = temp1;
			t(j) = r(j,1)*yy(1,i) + r(j,2)*yy(2,i) + r(j,3)*yy(3,i);

		}
		yy(1,i) = t(1);
		yy(2,i) = t(2);
		yy(3,i) = t(3);
	}

//            // now we must catch the special case of the rotation with inversion.
//            // fortunatley, and curiously, the optimal non-inverted rotation
//            // matrix will have a similar relation between rmsd and the eigen values.
//            // we just have to make a slight change in how we handle things
//            // depending on determinant

	rms_ctx = std::sqrt(std::abs(ev(1))) + std::sqrt(std::abs(ev(2))) +
	 handedness*std::sqrt(std::abs(ev(3)));

//            // the abs() are theoretically unneccessary since the eigen values
//            // of a real symmetric matrix are non-negative.
//            // in practice sometimes small eigen vals end up as tiny negatives.

	temp1 = ( come(1) * come(1) ) + ( come(2) * come(2) ) + ( come(3) * come(3) ) +
	 ( comp(1) * comp(1) ) + ( comp(2) * comp(2) ) + ( comp(3) * comp(3) );

	rms2_sum = (xre + xrp)/mass - temp1;

//            // and combine the outer and cross terms into the final calculation.
//            //  (the abs() just saves us a headache when the roundoff error
//            // accidantally makes the sum negative)

	esq = std::sqrt(std::abs(rms2_sum-2.0*rms_ctx));

}


////////////////////////////////////////////////////////////////////////////////
/// @begin rmsfitca2
///
/// @brief computes the rms between two weighted point vectors.
///
/// @detailed
///   xx_0,yy_0 are the input vectors of of points and ww is their weights
///   xx,yy are by product output vectors of the same points offset to remove center of mass
///   det is an out value of the determinant of the cross moment matrix
///   returned value is the rms
///
///   most of this is double for good reasons.  first there are some large
///   differences of small numbers.  and second the rsymm_eignen() function can
///   internally have numbers larger than the largest double number.
///   (you could do some fancy foot work to rescale things if you really
///   had a problem with this.)
///
/// @param  npoints - [in/out]? -
/// @param  xx - [in/out]? -
/// @param  yy - [in/out]? -
/// @param  ww - [in/out]? -
/// @param  natsel - [in/out]? -
/// @param  esq - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   det is a double precision real
///  (xx,yy) can be same arrays as (xx_0,yy_0) if desired
///
/// @references
///
/// @authors charlie strauss 2001
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rmsfitca2(
	int npoints,
	FArray2Da_double xx,
	FArray2Da_double yy,
	FArray1Da_double ww,
	int & natsel,
	double & esq
)
{
	xx.dimension( 3, npoints );
	yy.dimension( 3, npoints );
	ww.dimension( npoints );


	double det;
	int i,j,k;
	double temp1,temp3;
	FArray1D_double ev( 3 );
	FArray2D_double m_moment( 3, 3 );
	FArray2D_double rr_moment( 3, 3 );
	double rms_ctx;
	double rms_sum;
	double handedness;
	FArray1D_double t( 3 );

	FArray2D_double R( 3, 3 );
	double XPC, YPC, ZPC, XEC, YEC, ZEC;
//       //COMMON /TRANSFORM/ XPC,YPC,ZPC,XEC,YEC,ZEC,R

// align center of mass to origin

	COMAS(xx,ww,npoints,XPC,YPC,ZPC);
	COMAS(yy,ww,npoints,XEC,YEC,ZEC);

	temp3 = 0.0;
	for ( i = 1; i <= npoints; ++i ) {
		temp3 += ww(i);
	}

//       Make cross moments matrix   INCLUDE THE WEIGHTS HERE
	for ( k = 1; k <= 3; ++k ) {
		for ( j = 1; j <= 3; ++j ) {
			temp1 = 0.0;
			for ( i = 1; i <= npoints; ++i ) {
				temp1 += ww(i)*yy(k,i)*xx(j,i);
			}
			m_moment(k,j) = temp1    /(temp3); // rescale by temp3
		}
	}

	det = det3(m_moment); // will get handedness  of frame from determinant

	if ( std::abs(det) <= 1.0E-24 ) {
//     //  std::cout << "Warning:degenerate cross moments: det=" << det << std::endl;
//     // might think about returning a zero rms, to avoid any chance of Floating Point Errors?

		esq = 0.0;
		return;

	}
	handedness = sign(1.0,det);
//  // weird but documented fortran "feature" of sign(a,b) (but not SIGN) is that if fails if a < 0

//  //  multiply cross moments by itself

	for ( i = 1; i <= 3; ++i ) {
		for ( j = i; j <= 3; ++j ) {
			rr_moment(j,i) = rr_moment(i,j) = // well it is symmetric afterall
			 m_moment(1,i)*m_moment(1,j) +
			 m_moment(2,i)*m_moment(2,j) +
			 m_moment(3,i)*m_moment(3,j);
		}
	}

//            //  compute eigen values of cross-cross moments

	rsym_eigenval(rr_moment,ev);

//               // reorder eigen values  so that ev(3) is the smallest eigenvalue

	if ( ev(2) > ev(3) ) {
		if ( ev(3) > ev(1) ) {
			temp1 = ev(3);
			ev(3) = ev(1);
			ev(1) = temp1;
		}
	} else {
		if ( ev(2) > ev(1) ) {
			temp1 = ev(3);
			ev(3) = ev(1);
			ev(1) = temp1;
		} else {
			temp1 = ev(3);
			ev(3) = ev(2);
			ev(2) = temp1;
		}
	}

//                 // ev(3) is now the smallest eigen value.  the other two are not
//                 //  sorted.  this is prefered order for rotation matrix


	rsym_rotation(m_moment,rr_moment,ev,R);

//$$$             for ( i = 1; i <= npoints; ++i ) {
//$$$               for ( j = 1; j <= 3; ++j ) {
//$$$                 temp1 = 0.0;
//$$$                for ( k = 1; k <= 3; ++k ) {
//$$$                  temp1 += R(j,k)*yy(k,i);
//$$$                }
//$$$                t(j) = temp1;
//$$$               }
//$$$               yy(1,i) = t(1);
//$$$               yy(2,i) = t(2);
//$$$               yy(3,i) = t(3);
//$$$             }

	for ( i = 1; i <= npoints; ++i ) {
		for ( j = 1; j <= 3; ++j ) { // compute rotation
			t(j) = R(j,1)*yy(1,i) + R(j,2)*yy(2,i) + R(j,3)*yy(3,i);
		}
		yy(1,i) = t(1);
		yy(2,i) = t(2);
		yy(3,i) = t(3);
	}
//   // now we must catch the special case of the rotation with inversion.
//   // we cannot allow inversion rotations.
//   // fortunatley, and curiously, the optimal non-inverted rotation matrix
//   // will have the similar eigen values.
//   // we just have to make a slight change in how we handle things depending on determinant


	rms_ctx = std::sqrt(std::abs(ev(1))) + std::sqrt(std::abs(ev(2))) +
	 handedness*std::sqrt(std::abs(ev(3)));

	rms_ctx *= temp3;

//   // the abs() are theoretically unneccessary since the eigen values of a real symmetric
//   // matrix are non-negative.  in practice sometimes small eigen vals end up just negative
	rms_sum = 0.0;
	for ( i = 1; i <= npoints; ++i ) {
		for ( j = 1; j <= 3; ++j ) {
			rms_sum += ww(i)*( ( yy(j,i) * yy(j,i) ) + ( xx(j,i) * xx(j,i) ) );
		}
	}
	// rms_sum = rms_sum; //   /temp3   (will use natsel instead)

//  // and combine the outer and cross terms into the final calculation.
//  //  (the abs() just saves us a headache when the roundoff error accidantally makes the sum negative)

	esq = std::sqrt( std::abs( rms_sum - ( 2.0 * rms_ctx ) ) / natsel );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin rsym_rotation
///
/// @brief finds a (proper) rotation matrix that minimizes the rms.
///
/// @detailed this computes the rotation matrix based on the eigenvectors of m that gives the
/// the mimimum rms.  this is determined using mm the cross moments matrix.
///
/// for best results the third eigen value should be the
/// smallest eigen value!
///
///
/// @param  mm - [in/out]? -
/// @param  m - [in/out]? -
/// @param  ev - [in/out]? -
/// @param  rot - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors charlie strauss (cems) 2001
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rsym_rotation(
	FArray2Da_double mm,
	FArray2Da_double m,
	FArray1Da_double ev,
	FArray2Da_double rot
)
{
	mm.dimension( 3, 3 );
	m.dimension( 3, 3 );
	ev.dimension( 3 );
	rot.dimension( 3, 3 );


	FArray2D_double temp( 3, 3 );
	FArray2D_double mvec( 3, 3 );

	rsym_evector(m,ev,mvec);

	for ( int i = 1; i <= 2; ++i ) { // dont need no stinkin third component
		double norm = 1.0 / std::sqrt(std::abs(ev(i) )); // abs just fixes boo boos
		// if one was nervous here one could explicitly
		// compute the norm of temp.
		for ( int j = 1; j <= 3; ++j ) {
			temp(j,i) = 0.0;
			for ( int k = 1; k <= 3; ++k ) {
				temp(j,i) += mvec(k,i)*mm(j,k);
			}
			temp(j,i) *= norm;
		}
	}

	temp(1,3) =  temp(2,1)*temp(3,2) - temp(2,2)*temp(3,1);
	temp(2,3) = -temp(1,1)*temp(3,2) + temp(1,2)*temp(3,1);
	temp(3,3) =  temp(1,1)*temp(2,2) - temp(1,2)*temp(2,1);

	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			rot(j,i) = 0.0;
			for ( int k = 1; k <= 3; ++k ) {
				rot(j,i) += temp(i,k)*mvec(j,k);
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin det3
///
/// @brief determinant of a 3x3 matrix
///
/// @detailed
/// cute factoid: det of a 3x3 is the dot product of one row with the cross
/// product of the other two. This explains why a right hand coordinate system
/// has a positive determinant. cute huh?
///
/// @param  m - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors charlie strauss 2001
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
double
det3( FArray2Da_double m )
{

	m.dimension( 3, 3 );

	return
	 m(1,3)*( m(2,1)*m(3,2) - m(2,2)*m(3,1) ) -
	 m(2,3)*( m(1,1)*m(3,2) - m(1,2)*m(3,1) ) +
	 m(3,3)*( m(1,1)*m(2,2) - m(1,2)*m(2,1) );
}



////////////////////////////////////////////////////////////////////////////////
/// @begin rsym_eigenval
///
/// @brief computes the eigen values of a real symmetric 3x3 matrix
///
/// @detailed
/// the method used is a deterministic analytic result that I hand factored.(whew!)
/// Amusingly, while I suspect this factorization is not yet optimal in the
/// number of calcs required
/// I cannot find a signifcantly better one.
///  (if it were optimal I suspect I would not have to compute imaginary numbers that
///  I know must eventually cancel out to give a net real result.)
/// this method relys on the fact that an analytic factoring of an order 3 polynomial exists.
/// m(3,3) is a 3x3 real symmetric matrix: only the upper triangle is actually used
///  ev(3) is a real vector of eigen values, not neccesarily in sorted order.
///
/// @param  m - [in/out]? -
/// @param  ev - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors charlie strauss 2001
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rsym_eigenval(
	FArray2Da_double m,
	FArray1Da_double ev
)
{
	m.dimension( 3, 3 );
	ev.dimension( 3 );



	double xx,yy,zz,xy,xz,yz;
	double a,b,c,s0;
	std::complex< double > f1,f2,f3,f4,f5;

	static std::complex< double > const unity = std::complex< double >(1.0,0.0);
	static std::complex< double > const sqrt_3i =
	 std::sqrt( 3.0 ) * std::complex< double >(0.0,1.0);

	// first, for lexical sanity, name some temporary variables
	xx = m(1,1);
	yy = m(2,2);
	zz = m(3,3);
	xy = m(1,2);
	xz = m(1,3);
	yz = m(2,3);

	// coefficients of characterisitic polynomial
	a = xx+yy+zz;
	b = -xx*zz-xx*yy-yy*zz+xy*xy+xz*xz+yz*yz;
	c = xx*yy*zz-xz*xz*yy-xy*xy*zz-yz*yz*xx+2*xy*xz*yz;

	// For numerical dynamic range we rescale the variables here
	// with rare exceptions this is unnessary but it doesn't add much to the calculation time.
	// this also allows use of double if desired. cems
	//  for complex< float > the rescaling trigger  should be  1e15  (or less)
	//  for complex< double > the rescaling trigger should be  1e150 (or less)
	// note we completely ignore the possiblity of needing rescaling to avoid
	// underflow due to too small numbers. left as an excercise to the reader.

	double norm = std::max(std::abs(a),std::max(std::abs(b),std::abs(c)));
	if ( norm > 1.0E50 ) {   // rescaling trigger
		a /= norm;
		b /= norm * norm;
		c /= norm * norm * norm;
	} else {
		norm = 1.0;
	}
	// we undo the scaling by de-scaling the eigen values at the end

	// Power constants
	double const a2 = a * a;
	double const a3 = a * a * a;

	// eigenvals are the roots of the characteristic polymonial  0 = c + b*e + a*e^2 - e^3
	// solving for the three roots now:
	// dont try to follow this in detail: its just a tricky
	// factorization of the formulas for cubic equation roots.

	s0 = ( -12.0 * ( b * b * b ) ) - ( 3.0 * ( b * b ) * a2 ) +
	 ( 54.0 * c * b * a ) + ( 81.0 * ( c * c ) ) + ( 12.0 * c * a3 );
	 // butt ugly term

	f1 = b*a/6.0 + c/2.0 + a3/27.0 + std::sqrt(s0*unity)/18.0;

	f2 = std::pow( f1, (1.0/3.0) ); // note f1 is a complex number

	f3 = (-b/3.0 - a2/9.0)/f2;

	f4 = f2-f3;

	f5 = sqrt_3i * (f2+f3); // just our imaginary friend, mr. i

	s0 = a/3.0;
	ev(1) = f4.real();
	 // note implicitly take real part, imag part "should" be zero
	ev(1) = norm*(ev(1)+s0 );
	 // do addition after type conversion in previous line.
	ev(2) = (-f4+f5).real(); // note real part, imag part is zero
	ev(2) = norm*(ev(2)*0.5 + s0 );
	ev(3) = (-f4-f5).real(); // note real part, imag part is zero
	ev(3) = norm*(ev(3)*0.5 + s0);

}




////////////////////////////////////////////////////////////////////////////////
/// @begin rsym_evector
///
/// @brief
/// Author: charlie strauss (cems) 2001
/// given original matrix plus its eigen values
/// compute the eigen vectors.
/// USAGE notice: for computing min rms rotations be sure to call this
/// with the lowest eigen value in Ev(3).
///
/// The minimal factorization of the eigenvector problem I have derived below has a puzzling
/// or, rather, interesting assymetry.  Namely, it doesn't matter what
///  either ZZ=M(3,3) is or what Ev(3) is!
///  One reason for this is that of course all we need to  know is contained in the
///  M matrix in the firstplace, so the eigen values overdetermine the problem
///  We are just exploiting the redundant info in the eigen values to hasten the solution.
///  What I dont know is if infact there exists any symmetic form using the eigen values.
///
/// we deliberately introduce another assymetry for numerical stability, and
/// to force proper rotations  (since eigen vectors are not unique within a sign change).
///   first, we explicitly numerically norm the vectors to 1 rather than assuming the
///   algebra will do it with enough accuracy. ( and its faster to boot!)
///   second, we explicitly compute the third vector as being the cross product of the
///   previous two rather than using the M matrix and third eigen value.
///   If you arrange the eigen values so that the third eigen value is the lowest
///   then this guarantees a stable result under the case of either very small
///   eigen value or degenerate eigen values.
///   this norm, and ignoring the third eigen value also gaurentee us the even if the
///   eigen vectors are not perfectly accurate that, at least, the matrix
///   that results is a pure orthonormal rotation matrix, which for most applications
///   is the most important form of accuracy.
///
/// @detailed
///
/// @param  m - [in/out]? -
/// @param  ev - [in/out]? -
/// @param  mvec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rsym_evector(
	FArray2Da_double m,
	FArray1Da_double ev,
	FArray2Da_double mvec
)
{
	m.dimension( 3, 3 );
	ev.dimension( 3 );
	mvec.dimension( 3, 3 );

	// local
	double xx,yy,xy,zx,yz; //zz
	double e1,e2,e3,znorm;


	// first, for sanity only, name some temporary variables
	//zz = m(3,3);
	xy = m(1,2);
	zx = m(1,3);
	yz = m(2,3);

	if ( ev(1) != ev(2) ) {  // test for degenerate eigen values

		for ( int i = 1; i <= 2; ++i ) {
		 // only computer first two eigen vectors using this method
			// note you could compute all three this way if you wanted to,
			// but you would run into problems with degenerate eigen values.

			xx = m(1,1)-ev(i);
			yy = m(2,2)-ev(i);
			// I marvel at how simple this is when you know the eigen values.
			e1 = xy*yz-zx*yy;
			e2 = xy*zx-yz*xx;
			e3 = xx*yy-xy*xy;

			znorm = std::sqrt( ( e1 * e1 ) + ( e2 * e2 ) + ( e3 * e3 ) );

			mvec(1,i) = e1/znorm;
			mvec(2,i) = e2/znorm;
			mvec(3,i) = e3/znorm;

		}

		// now compute the third eigenvector
		mvec(1,3) =  mvec(2,1)*mvec(3,2) - mvec(2,2)*mvec(3,1);
		mvec(2,3) = -mvec(1,1)*mvec(3,2) + mvec(1,2)*mvec(3,1);
		mvec(3,3) =  mvec(1,1)*mvec(2,2) - mvec(1,2)*mvec(2,1);

		// pathologically nervous people would explicitly normalize this vector too.

		return;

	} else {

		if ( ev(2) != ev(3) ) {
			std::cout << " hey is this the right thing to be doing??? " << std::endl;

			for ( int i = 2; i <= 3; ++i ) {
			 // Okay, since 1 and 2 are degenerate we will use 2 and 3 instead.

				xx = m(1,1)-ev(i);
				yy = m(2,2)-ev(i);
				// I marvel at how simple this is when you know the eigen values.
				e1 = xy*yz-zx*yy;
				e2 = xy*zx-yz*xx;
				e3 = xx*yy-xy*xy;
				 // yes you sharp eyed person, its not quite symmetric here too.
				//                   life is odd.

				znorm = std::sqrt( ( e1 * e1 ) + ( e2 * e2 ) + ( e3 * e3 ) );

				mvec(1,i) = e1/znorm;
				mvec(2,i) = e2/znorm;
				mvec(3,i) = e3/znorm;

			}

			// now compute the third eigenvector
			mvec(1,1) =  mvec(2,2)*mvec(3,3) - mvec(2,3)*mvec(3,2);
			mvec(2,1) = -mvec(1,2)*mvec(3,3) + mvec(1,3)*mvec(3,2);
			mvec(3,1) =  mvec(1,2)*mvec(2,3) - mvec(1,3)*mvec(2,2);

			// pathologically nervous people would explicitly normalize this vector too.

			return;

		} else {

			std::cout << "warning: all eigen values are equal" << std::endl;

			for ( int i = 1; i <= 3; ++i ) {
				mvec(1,i) = 0.0;
				mvec(2,i) = 0.0;
				mvec(3,i) = 0.0;
				mvec(i,i) = 1.0;
			}
			return;
		}
	}

}
