// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.10 $
//  $Date: 2005/10/26 20:30:45 $
//  $Author: flo $


//Rosetta headers
#include "rotamer_explosion.h"
#include "aaproperties_pack.h"
#include "design.h"
#include "hbonds.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "misc.h"
#include "pack.h"
#include "pack_geom_inline.h" //distance function
#include "param_pack.h"
#include "pose.h" //need pose to screen exploded rotamers
#include "rotamer_functions.h" // get_rot_coord
#include "RotamerSet.h"
#include "template_pack.h"
#include "minimize.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>

//C++ headers
#include <string>
#include <ctime>

namespace rotamer_explosion {

	pose_ns::Pose *rotmin_pose = { NULL };
	pose_ns::Score_weight_map *rotmin_score_weight_map = { NULL };
	std::map<int,bool> rotamers_to_minimize;
	std::map<int,float> rotamer_lig_energies;

}


////////////////////////////////////////////////////////////////////////////////
// @begin screen_exploded_rotamers_ligand
//
// @brief
// Filters rotamers for hydrogen bonding to the ligand
// Used with "rotamer_explosion" to allow oversampling of polar
// sidechains.
//
// @references
// Ashworth J, Havranek J, Duarte C, Sussman D, Monnat R, Stoddard B, Baker D.
// Nature 441, 656-659 (2006)
//
// @authors
// flo, july 2007
//
////////////////////////////////////////////////////////////////////////////////

bool
screen_exploded_rotamers_ligand(
	int const seqpos,
	int const aa,
	int const aav,
	FArray1DB_float & temp_chi,
	DesignMap const & design_map
)
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace ligand;
	using namespace param; // MAX's
	using namespace param_aa; // is_NA
	using namespace param_pack;
	using namespace template_pack; //neighbors
	using namespace hbonds;
	using namespace rotamer_explosion; //rotmin_pose
	float dummy1 = 0.0;
	float dummy2 = 0.0;
	float dummy3 = 0.0;
	int aa_rotmin_pose;
	int aav_rotmin_pose;

	//don't bother with hydrophobic aas
	if( (aa == 5) || (aa == 6) || (aa == 8) || (aa == 10) || (aa == 11) || (aa == 13) || (aa == 18) ) return false;

	design::active_rotamer_options.rot_explode_debug_number_screens++;

	FArray2D_float rot_coord( 3, MAX_ATOM() );
	FArray2D_float minrot_pose_coord( 3, MAX_ATOM() );

	get_rot_coord( full_coord, aa, aav, temp_chi, rot_coord, seqpos );

	float bumpenergy = 0.0;
	bump_check( aa, aav, seqpos, rot_coord, full_coord, res, res_variant,
	 design_map, total_residue, bumpenergy );
	if ( bumpenergy > max_rot_bumpenergy ) return false;

	// check sidechain rotamer against all ligands present

	for(std::vector<int>::const_iterator ligpos( ligand_residue_numbers.begin() );
			ligpos != ligand_residue_numbers.end(); ligpos++ ) {

		int ligaa = res( *ligpos );
		int ligv = res_variant( *ligpos );

		if ( get_ligand_flexible_flag() ) {
			std::cout << "screen_exploded_rotamers_ligand does not yet support flexible ligands" << std::endl;;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			return false;
		}

		float sc_lig_hbE = 0.0;
		float & xyz_lig( full_coord(1,1,*ligpos) );

		get_hbE(true,aa,ligaa,aav,ligv,seqpos,*ligpos,neighbors(seqpos),
			neighbors(*ligpos),rot_coord,xyz_lig,sc_lig_hbE,dummy1,
			dummy2,dummy3);

		if ( sc_lig_hbE < design::active_rotamer_options.explode_cutoff ) {
			//std::cout << "floscreen chi1,2,3,4: " << F(6,2,temp_chi(1)) << "," << F(6,2,temp_chi(2)) << "," << F(6,2,temp_chi(3)) << "," << F(6,2,temp_chi(4)) << ", lig:" << *ligpos << " res:" << seqpos << " aa:" << aa << " hbE:" << F(5,2,sc_lig_hbE)<< ", min to chi1,2,3,4: ";


			//note: the code in the following if-block is still under development. it is functional but could be made a lot faster
			if ( design::active_rotamer_options.rot_explode_rtmin ) {

				//do minimization if rotamer seems to have some hbond energy
				//first save rotamer in rotmin_pose
				aa_rotmin_pose = rotmin_pose->res(seqpos);  //need to put back later
				aav_rotmin_pose = rotmin_pose->res_variant(seqpos); //need to put back later
				for ( int atom = 1; atom <= aaproperties_pack::natoms( aa, aav ); ++atom ) {
					for ( int i = 1; i <= 3; ++i ) {
						minrot_pose_coord( i, atom ) = rotmin_pose->full_coord()( i, atom, seqpos );
					}
				}

				//then put in new rotamer
				rotmin_pose->copy_sidechain( seqpos, aa, aav, rot_coord, true);
				//now do minimization
				rotmin_pose->set_allow_chi_move(seqpos,true);

				//clock_t minstart = clock();

				//minimize_set_tolerance( 0.0000001 );
				rotmin_pose->main_minimize( *rotmin_score_weight_map , "linmin" );
				//clock_t minend = clock();
				//double mintime = double (minend - minstart)/CLOCKS_PER_SEC;


				//std::cout << F(6,2,rotmin_pose->chi(1,seqpos)) << "," << F(6,2,rotmin_pose->chi(2,seqpos)) << "," << F(6,2,rotmin_pose->chi(3,seqpos)) << "," << F(6,2,rotmin_pose->chi(4,seqpos)) << " new hbE:" << F(5,2,new_sc_lig_hbE) ;

				//update chis
				for (int chicount = 1; chicount <= MAX_CHI; chicount++) {
					temp_chi(chicount) = rotmin_pose->chi(chicount,seqpos);
				}
				//and finally restore original pose
				rotmin_pose->set_allow_chi_move(seqpos,false);
				rotmin_pose->copy_sidechain( seqpos, aa_rotmin_pose, aav_rotmin_pose, minrot_pose_coord, true);

			}

			//std::cout << " bumpE:" << F(6,2,bumpenergy) << " #scr:" << design::active_rotamer_options.rot_explode_debug_number_screens << std::endl ;
		}
		// the rotamer just needs to pass once
		if ( sc_lig_hbE < design::active_rotamer_options.explode_cutoff ) {
			return true;
		}


	}	 // residue loop
	return false;
}



////////////////////////////////////////////////////////////////////////////////
// @begin prescreen_base_rotamer_ligand_contacts
//
// @brief
// Screens a certain rotamer for whether any polar atoms are within a certain
// distance of ligand polar atoms
// Used with "rotamer_explosion" to allow oversampling of polar
// sidechains.
//
// @references
// Ashworth J, Havranek J, Duarte C, Sussman D, Monnat R, Stoddard B, Baker D.
// Nature 441, 656-659 (2006)
//
// @authors
// flo, aug 2007
//
////////////////////////////////////////////////////////////////////////////////

bool
prescreen_base_rotamer_ligand_contacts(
	int const seqpos,
	int const aa,
	int const aav,
	FArray1Da_float base_chi
)
{

	using namespace aaproperties_pack;
	using namespace misc;
	using namespace ligand;
	using namespace param; // MAX's
	using namespace param_aa; // is_NA

	//don't bother with hydrophobic aas
	if( (aa == aa_phe) || (aa == aa_gly) || (aa == aa_ile) || (aa == aa_leu) || (aa == aa_met) || (aa == aa_pro) || (aa == aa_val) ) return false;


	// 1. build the base rotamer at this position
	FArray2D_float rot_coord( 3, MAX_ATOM() );
	get_rot_coord( full_coord, aa, aav, base_chi, rot_coord, seqpos );


	//2. loop over all ligands present
	FArray1D_float CurDon_coord ( 3 );
	FArray1D_float CurAcc_coord ( 3 );
	for(std::vector<int>::const_iterator ligpos( ligand_residue_numbers.begin() );
			ligpos != ligand_residue_numbers.end(); ligpos++ ) {

			int ligaa = res( (*ligpos) );
			int ligv = res_variant( (*ligpos) );

			if ( get_ligand_flexible_flag() ) {
				std::cout << "rotamer explosion does not yet support flexible ligands" << std::endl;;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				return false;
			}

			else{

 				//3a. loop over all base rotamer donors and ligand acceptors
				for ( int rot_don = 2; rot_don <= nH_polar(aa,aav); rot_don++) { //start at 2 because 1 is always backbone H
					int CurDon = atom_base( Hpos_polar(rot_don, aa, aav), aa, aav);
					for(int ii =1; ii<=3;ii++) {CurDon_coord(ii) = rot_coord(ii,CurDon);}

					for (int lig_acc = 1; lig_acc <= nacceptors(ligaa,ligv); lig_acc++ ){
						int CurAcc = accpt_pos(lig_acc,ligaa,ligv);
						for(int ii =1; ii<=3;ii++) {CurAcc_coord(ii) = full_coord(ii,CurAcc,*ligpos);}

						float CurDist2 = 100.0;
						distance2_bk(CurDon_coord,CurAcc_coord,CurDist2);
						//std::cout << aa << " at pos " << seqpos << ", atom " << CurDon << "(" << CurDon_coord(1) << "," << CurDon_coord(2) << "," << CurDon_coord(3) << ") is " << CurDist2 << " away from " << ligaa << " atom " << CurAcc << std::endl;
						if(CurDist2 < 20.0) {return true;}
					}
				}

				//3b. loop over all base rotamer acceptors and ligand donors
				for ( int rot_acc = 2; rot_acc <= nacceptors(aa,aav); rot_acc++) { //start at 2 because 1 is always backbone O
					int CurAcc = accpt_pos(rot_acc,aa,aav);
					for(int ii =1; ii<=3;ii++) CurAcc_coord(ii) = rot_coord(ii,CurAcc) ;

					for (int lig_don = 1; lig_don <= nH_polar(ligaa,ligv); lig_don++ ){
						int CurDon = atom_base( Hpos_polar(lig_don, ligaa, ligv), ligaa, ligv);
						for(int ii =1; ii<=3;ii++) CurDon_coord(ii) = full_coord(ii,CurDon,*ligpos) ;

						float CurDist2 = 100.0;
						distance2_bk(CurDon_coord,CurAcc_coord,CurDist2);
						//std::cout << aa << " at pos " << seqpos << ", atom " << CurAcc << " is " << CurDist2 << " away from " << ligaa << " atom " << CurDon << " (" << CurDon_coord(1) << "," << CurDon_coord(2) << "," << CurDon_coord(3) << ")" << std::endl;
						if(CurDist2 < 20.0) {return true;}
					}
				}

				//4. if no potential hydrogen-bonding pairs are near each other, rotamer fails prescreen
				//std::cout << aa << " at position " << seqpos << "with chi 1,2,3,4 " << base_chi(1) << "," << base_chi(2) << "," << base_chi(3) << "," << base_chi(4) << " cannot make a hydrogen-bond to the ligand." << std::endl;
				return false;
			}
		}
		return false;
}



////////////////////////////////////////////////////////////////////////////////
// @begin screen_rotamer_set_for_ligand_contacts
//
// @brief
// screens a rotamer set for ligand contacts. For each residue at each position,
// the best three rotamers (one for each chi1 bin)  are saved in map best_rotamers
//
//
// @authors
// flo, november 2007
//
////////////////////////////////////////////////////////////////////////////////
void screen_rotamer_set_for_ligand_contacts(
	const FArray3D_float & full_coord,
	RotamerSet const & rotamer_set,
	std::map<int,bool> & best_rotamers,
	int screen_start_rotamer
)
{

	using namespace aaproperties_pack;
	using namespace misc;
	using namespace ligand;
	using namespace param; // MAX's
	using namespace param_aa; // is_NA
	using namespace param_pack; // packer_wts
	using namespace template_pack; //neighbors


	//FArray3D_float temp_fcoord = pose.full_coord();
	FArray1D_float dummyact1(3,0.0);  //need these two arrays for the sc_energy function to work, but since we're only interested in ligand interactions
	FArray1D_float dummyact2(3,0.0);  //no pairterm exists, so we don't need the actual coordinates
	float dummy1 = 0.0;
	float dummy2 = 0.0;
	float dummy3 = 0.0;

	FArray3D_float BestE (MAX_RES(), MAX_AA_PLUS(), 4, 999.9);  //4 is the number of rotamer bins,
	FArray3D_int BestE_rot ( MAX_RES(), MAX_AA_PLUS(), 4, 0 );

	for(std::vector<int>::const_iterator ligpos( ligand_residue_numbers.begin() );
			ligpos != ligand_residue_numbers.end(); ligpos++ ) {

		if ( get_ligand_flexible_flag() ) {
			std::cout << "best rotamer minimization does not yet support flexible ligands" << std::endl;;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		else {

			FArray2D_float ligcoord( 3, MAX_ATOM() );  //coordinates of this ligand
			int const ligaa = res(*ligpos);
			int const ligaav = res_variant(*ligpos);

			//1.first get the coordinates of the ligand
			int ligatoms=aaproperties_pack::natoms( res(*ligpos), res_variant(*ligpos) );
			for(int la=1; la<=ligatoms; la++){
				for(int k=1; k<=3; k++){
					ligcoord(k,la) = full_coord(k,la,*ligpos);
				}
			}

			//2.then go through all the rotamers in the set and calculate their energy with the ligand
			for (int rot = screen_start_rotamer; rot <= rotamer_set.nrotamers(); rot++) {

				int const rotres = rotamer_set.report_seqpos(rot);
				int const rotaa = rotamer_set.report_aa(rot);
				int const rotaav = rotamer_set.report_aav(rot);
				int const rotbin_chi1 = rotamer_set.get_rrot(1,rot);
				if(rotres == *ligpos) { continue; }   //just in case...
				FArray2D_float rotcoord = rotamer_set.get_rotcoord(rot); //get rotamer coordinates


				float solvE,atrE,repE,elecE,pairE,plane_totalE,schbE;  //couple of variables
				get_sc_scE(rotaa,rotaav,rotcoord,ligaa,ligaav,ligcoord,dummyact1,dummyact2,
								 rotres,*ligpos,solvE,atrE,repE,pairE,plane_totalE,elecE);  //compute sclig energy

			//if ( geometric_sol::geometric_sol_flag ) {
			//	solvE = 0.;
				// any polar groups on the rotamer sidechain occluded by backbone
			//	solvE += res_res_geometric_sol(rotres,bbres,aarot,aavrot,res(bbres),res_variant(bbres),
			//		rotcoord,xyz_11b,false,true,true,false);
				// any polar groups on the backbone occluded by rotamer sidechain
			//		solvE += res_res_geometric_sol(bbres,rotres,res(bbres),res_variant(bbres),aarot,aavrot,
			//		xyz_11b,rotcoord,true,false,false,true);
			//}

				get_hbE(true,rotaa,ligaa,rotaav,ligaav,rotres,*ligpos,neighbors(rotres),
							neighbors(*ligpos),rotcoord,ligcoord,schbE,dummy1,dummy2,dummy3);  //compute hb energy

				float rotligE = pack_wts.Wsol() * solvE + pack_wts.Watr() * atrE + pack_wts.Wrep() * repE +
					+ pack_wts.Wpair() * pairE + pack_wts.Wplane_total() * plane_totalE + pack_wts.Whbond_sc() * schbE;

				rotamer_explosion::rotamer_lig_energies[rot] = rotligE;

				if(rotligE < BestE(rotres,rotaa,rotbin_chi1) ) {
					BestE(rotres,rotaa,rotbin_chi1) = rotligE;
					best_rotamers[rot] = true;
					best_rotamers.erase( BestE_rot(rotres,rotaa,rotbin_chi1) );
					BestE_rot(rotres,rotaa,rotbin_chi1) = rot;
 				//std::cout << "new best rotamer for rotbin " << rotbin_chi1 << ", rot number " << rot << " with E " << BestE(rotres,rotaa,rotbin_chi1) << std::endl;
				}

			//			std::cout <<"floromin rot " << rot <<" (" << rotres << "," << rotaa << ") has totalE/atrE of " << rotligE << ", " << pack_wts.Watr() * atrE << std::endl;

			}
		}
	}
}
