/// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 32323 $
//  $Date: 2009-08-25 13:16:13 -0700 (Tue, 25 Aug 2009) $
//  $Author: johnk $


// Rosetta Headers
#include "rotamer_trials.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "atom_is_backbone.h"
#include "constraints.h"
#include "count_pair.h"
#include "current_pose.h"
#include "DesignMap.h"
#include "design.h"
#include "disulfides.h"
#include "dna.h"
#include "dna_classes.h"
#include "dna_ns.h"
#include "docking.h"
#include "docking_minimize.h"
#include "etable_manager.h"
#include "enzyme.h"
#include "enzyme_ns.h"
#include "files_paths.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "fullatom_setup.h"
#include "gb_elec.h"
#include "geometric_solvation.h"
#include "gl_graphics.h"
#include "hbonds.h"
#include "hbonds_ns.h"
#include "interface.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "pack.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_torsion.h"
#include "pdbstatistics_pack.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_rotamer_trials.h"
#include "prof.h"
#include "random_numbers.h"
#include "read_aaproperties.h"
#include "rotamer_descriptor.h"
#include "rotamer_functions.h"
#include "rotamer_trial_energies.h"
#include "rotamer_trie_calc_energies.h"
#include "RotamerSet.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "symmetric_design.h"
#include "template_pack.h"
#include "termini.h"
#include "util_vector.h"
#include "wobble.h"
#include "water.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
//#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <vector>


// Namespaces
namespace energy_cut {
	float energycut = { 0.01 };
}

bool const rottrial_verbose( false );

////////////////////////////////////////////////////////////////////////////////
/// @begin           rotamer_trials()
///
/// @brief
///car simple wrapper to call rotamer trials
///car the fullatom energy arrays are updated, the rosetta scores ARE NOT!!
///
/// @detailed
///
/// @global_read     full_coord                float       misc.h     current_pose
/// @global_read     res_variant               int    misc.h     current_pose
/// @global_read     total_residue             int    misc.h     ints
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rotamer_trials()
{
	using namespace misc;
	using namespace param;
	using namespace files_paths; // for use_conformer
	using namespace template_pack;
	using namespace water;

	DesignMap design_map( total_residue, res );

	retrieve_allow_rottrial(design_map,total_residue);
	//---dummy_seq_select = false;
	//design_map.disable_all();

	if ( ! rotamer_trial_energies::use_trials_trie_flag ||
		design::explicit_h2o || water::use_hetero_h2o ||
		design::optE || design::fast_ala_scan || design::minimize_rot ||
			 ( score_check_current_pose() &&
				 score_get_current_pose().symmetric() ) )
	{
		fullatom_rotamer_trials(full_coord,res_variant,design_map,"single");
	}
	else
	{
		fullatom_rotamer_trials_trie(full_coord,res_variant,design_map,"single");
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin           set_allow_rotamer_trials(setting,total_residue)
///
/// @brief
///
/// @detailed
///
/// @param[in]        setting                   bool
/// @param[in]        total_residue             int
///
/// @global_write    allow_rottrial            bool    maps_ns.h     protein_maps
///
/// @remarks
///
/// @references
///

/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_allow_rotamer_trials(
	FArray1Da_bool setting,
	int & total_residue
)
{
	using namespace protein_maps;

	setting.dimension( total_residue );

	for ( int i = 1; i <= total_residue; ++i ) {
		allow_rottrial(i) = setting(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin           retrieve_allow_rottrial(allow,total_residue)
///
/// @brief
///car allow_insert indicates residues fixed in torsion space
///car not necessarily fixed in cartesian space
///
/// @detailed
///
/// @param[in]        allow                     bool
/// @param[in]        total_residue             int
///
/// @param[out]       allow                     bool
///
/// @global_read     allow_rottrial            bool    maps_ns.h     protein_maps
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_allow_rottrial(
												DesignMap & design_map,
	int & total_residue
)
{
	using namespace protein_maps;
	using namespace files_paths;
	using namespace misc;
	using namespace param;

	//---	allow.dimension( total_residue );

	for ( int i = 1; i <= total_residue; ++i ) {
		//---allow(i) = allow_rottrial(i);
		if(allow_rottrial(i)){
			design_map.set(i,res(i));
		}
		else{
			design_map.fix_completely(i);
		}
	}

	// jk read a resfile here too, if one is specified on the command-line...
	if ( files_paths::resfile == "none" ) return;

	std::cout << "Loading rotamer trials resfile " << files_paths::resfile << std::endl;

	data_x.clear();
	if ( files_paths::resfile == "auto" ) {
		data_x.open( start_path + start_file + ".resfile" );
	} else {
		data_x.open( files_paths::resfile );
	}
	if ( !data_x ) {
		std::cout << "Open failed for file: " << data_x.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//ja key on " start" instead of starting at arbitrary line #26
	std::string header = "header";
	while ( header != " start" && !data_x.eof() ) {
		data_x >> bite( 6, header ) >> skip;
//	std::cout << header << std::endl;
	}

	if ( header != " start" ) {
		std::cout << "Error: could't find start of resfile" << std::endl;
	}

	int seqpos;
	std::string charid;
	FArray1D_char aachar( MAX_AA(), ' ' );

	for ( int ii = 1; ii <= total_residue; ++ii ) {
		data_x >> skip(3) >> bite(4,seqpos) >> skip(6) >> bite(5,charid) >> skip(2);
		for ( int i = 1, ie = MAX_AA(); i <= ie; ++i ) {
			data_x >> bite( aachar(i) );
			// allow comments after this point
			if ( aachar(i) == '#' ) { aachar(i) = ' '; break; }
		}
		data_x >> skip;
		if ( ( charid == "NATRO" ) || ( charid == "NATVA" ) ) {
			design_map.fix_completely( seqpos );
		}
	}
	data_x.close();
	data_x.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           bool function get_minimize_rot_flag()
///
/// @brief
///
/// @detailed
///
/// @return S         minimize_rot
///
/// @global_read     minimize_rot              bool    design.h   design_commands
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_minimize_rot_flag()
{
	return design::minimize_rot;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin score_set_minimize_rot
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
score_set_minimize_rot( bool const setting )
{
	design::minimize_rot = setting;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_energycut
///
/// @brief
///
/// @detailed
///
/// @param  cutoff - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_energycut( float const cutoff )
{
	energy_cut::energycut = cutoff;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_energycut
///
/// @brief
///
/// @detailed
///
/// @param  cutoff - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
get_energycut()
{
	return energy_cut::energycut;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin           fullatom_rotamer_trials(xyz_out,res_variant_out,allow_rottrial)
///
/// @brief
///bk tries new rotamers at residues that gained energy when the last
///bk fragment insertion was made
///
/// @detailed
///bk !!!!!  must call fullatom_energy before this function to
///bk !!!!!  fill global variables needed by rotamer_trials
///
///car exits with the fullatom_energy arrays current
///car returns coordinates with updated rotamers in place
///
/// @param[in]        res_variant_out            int
/// @param[in]        allow_rottrial            bool
/// @param[in]        xyz_out                   float
/// @param[out]       res_variant_out            int
/// @param[out]       xyz_out                   float
///
/// @global_read     aa_ala                    int    param_aa.h
/// @global_read     aa_cys                    int    param_aa.h
/// @global_read     aa_gly                    int    param_aa.h
/// @global_read     res_variant                int    misc.h
/// @global_read     res                       int    misc.h
/// @global_read     atr_pair                  float       fullatom_energies.h energies
/// @global_read     atrenergy                 float       fullatom_energies.h energies
/// @global_read     best_resenergy            float       fullatom_energies.h best_energies
/// @global_read     fullatome                 float       fullatom_energies.h totals
/// @global_read     fullatome_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     hbenergy                  float       fullatom_energies.h energies
/// @global_read     hbond_pair                float       fullatom_energies.h energies
/// @global_read     inform                    int    runlevel.h
/// @global_read     MAX_ATOM                  int    param.h
/// @global_read     minimize_rot              bool    design.h   design_commands
/// @global_read     neighbors                 int    template_pack.h template
/// @global_read     total_residue             int    misc.h
/// @global_read     opte                      bool    design.h   design_commands
/// @global_read     opte_ctsa                 bool    design.h   design_commands
/// @global_read     fast_ala_scan             bool    design.h   design_commands
/// @global_read     pair_energy               float       fullatom_energies.h energies
/// @global_read     pair_pair                 float       fullatom_energies.h energies
/// @global_read     rep_pair                  float       fullatom_energies.h energies
/// @global_read     repenergy                 float       fullatom_energies.h energies
/// @global_read     resenergy                 float       fullatom_energies.h energies
/// @global_read     runlevel                  int    runlevel.h run_level_status
/// @global_read     sol_pair                  float       fullatom_energies.h energies
/// @global_read     solenergy                 float       fullatom_energies.h energies
/// @global_read     full_coord                       float       misc.h template
///
/// @global_write    atrenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_write    hbenergy_minus_current    float       rotamer_trial_energies.h minus_energies
/// @global_write    pair_energy_minus_current float       rotamer_trial_energies.h minus_energies
/// @global_write    repenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_write    solenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_write    gb_elecenergy_minus_current   float       rotamer_trial_energies.h minus_energies
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fullatom_rotamer_trials(
	FArray3Da_float xyz_out,
	FArray1Da_int res_variant_out,
	DesignMap const & design_map,
	std::string const & seq_opt
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace files_paths;
	using namespace fullatom_energies;
	using namespace interface;
	using namespace ligand;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace rotamer_trial_energies;
	using namespace runlevel_ns;
	using namespace template_pack;
	using namespace param_pack;
	using namespace files_paths; // for use_conformer
	using namespace design_sym; // for symmetric rottrialing
	using namespace disulfides::BOUNDARY;

	initialize_fullatom();

	xyz_out.dimension( 3, MAX_ATOM(), star );
	res_variant_out.dimension( MAX_RES() );

//bk energycut: if energy difference is better than this value, don't
//bk repack this position
//db energycut can now be set externally using set_energycut to allow
//db annealing during fullatom_relax

	int first_res,last_res; // first and last residue that changes
	FArray1D_int repack_list( MAX_RES()() );
	int looplow, loophigh, nrotaa, aav;
	float deltaE;
	FArray2D_int extra_rot( MAX_CHI, MAX_RES()() );
	FArray2D_float extra_chi( MAX_CHI, MAX_RES()() );
	bool inc_current;
	FArray1D_bool repack( MAX_RES()() );
	bool rot_min_switch; // whether to use rot minimization
	bool inc_extra; // additional rotamers to try ?
	FArray1D_bool new_rottrial_rotamer( total_residue );
	bool rot_chosen( false ); // for optE

	///////////////////////
	// pose/symmetry stuff:
	//
	// currently the symmetry code only works with poses but this restriction
	// could easily be relaxed.
	//
	bool const pose_rottrial( score_check_current_pose() );
	bool symm_rottrial( false );
	if ( pose_rottrial && score_get_current_pose().symmetric() ) {
		setup_symmetric_packing( total_residue,
														 score_get_current_pose().symmetry_info() );
		if ( num_clones > 0 ) symm_rottrial = true;
		//		std::cout << "symm_rottrial: num_clones= " << num_clones << std::endl;
	}

//bk set repack flag true if change in energy is larger than energycut
//chu if rotamer minimization is applied, escape this energy criteria.
	float const energycut = get_energycut();

	int position = 0;
	int numpositions = 0;
	int rottrial_cycles = 1; // default is to go through once

	// jk Instantiate a RotamerSet
	RotamerSet rotamer_set;

	//jjh get Born radii
	if ( gen_born ) {
		PROF_START( prof::TEST1 );
		gb_get_all_born_radii();
		PROF_STOP ( prof::TEST1 );
	}

	if ( !optE && !fast_ala_scan && !pose_rottrial ) { // see below for setup for optE mode
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			deltaE = resenergy(seqpos)-best_resenergy(seqpos);
			rot_min_switch = (deltaE > energycut || minimize_rot);
			if ( rot_min_switch && res(seqpos) != aa_gly &&
					 //--- res(seqpos) != aa_ala && allow_rottrial(seqpos) )
					 res(seqpos) != aa_ala && design_map.repack_residue(seqpos) ) {
				if ( !( get_norepack_disulf() && cys_res_in_disulf(seqpos) ) ) {
					++position;
					repack_list(position) = seqpos;
				}
			}
		}
		numpositions = position;
		if ( runlevel >= inform ) std::cout <<
		 "rotamer trial at this many positions " << numpositions << std::endl;

//bk make a list that determines the order residues are repacked
//bk (order is random).

		random_order(repack_list,numpositions);
	} else if ( pose_rottrial ) {
		pose_setup_repack_list( score_get_current_pose(), repack_list,
														numpositions, rottrial_cycles );
		if ( runlevel >= inform ) std::cout <<
			"rotamer trial at this many positions " << numpositions << std::endl;
		GRAPHICS_LOG( "rotamer trials at "+string_of(numpositions)+" positions.");
	}

//bk if this function is being used to output energies for optimizing weights
//bk then just set it to repack residues 5 to the lesser of 75 or (total_residue-5)[bk]
//   or (total_residue-5 only) [ctsa], don't include cysteines [cmd] unless it's for dna
	if ( optE || fast_ala_scan ) {
		position = 0;

		if ( read_optE_poslist ) {
			// read the list of positions to include from a file

			std::string inposfile = start_path + start_file + "." + opte_ext;
			std::ifstream inposlist( inposfile.c_str() );

			if ( !inposlist ) {
				std::cout << "Error! Could not open optE position list file: " <<
					inposfile << std::endl;
				std::exit( EXIT_FAILURE );
			}

			std::string line;
			while ( std::getline( inposlist, line ) ) {
				std::istringstream line_stream (line);
				int seqpos;
				line_stream >> seqpos >> skip;

				if ( ( seqpos > 0 ) && ( seqpos <= MAX_RES() ) ) {
					++position;
					repack_list(position) = seqpos;
				} else {
					std::cout << "seqpos out of range: " <<
						seqpos << std::endl;
					std::exit( EXIT_FAILURE );
				}
			}
			inposlist.close();

		} else if ( dna_interface ) {
			for ( std::list< ResInfo >::iterator
				    it( dna_variables::interface_list.begin() );
			      it != dna_variables::interface_list.end(); ++it ) {
				if ( it->design ) {
					++position;
					repack_list(position) = it->pos;
				}
			}
		} else if ( fast_ala_scan ) {
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				++position;
				repack_list(position) = seqpos;
			}

//mj for ligands

		} else if ( get_ligand_flag() ) {
			detect_ligand_interface((*ligand::ligand_one), false, total_residue, res, full_coord);
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if ( int_res(seqpos) ) {
					++position;
					repack_list(position) = seqpos;
				}
			}
//mj end mj

		} else {
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if ( optE_inter ) {
					if ( interface_residue(seqpos) && res(seqpos) != (aa_cys) ) {
						repack(seqpos) = true;
						++position;
						repack_list(position) = seqpos;
					}
				} else {
//					if ( seqpos > 5 && seqpos < total_residue-5 && res(seqpos) != aa_cys &&
					if ( seqpos > 5 && seqpos < total_residue-5 &&
					 ( optE_ctsa || seqpos <= 75 ) ) {
						++position;
						repack_list(position) = seqpos;
					}
				}
			}
		}
		numpositions = position;

		if ( position > 0 ) {
			first_res = repack_list(1);
			last_res = repack_list(position);
		} else {
			first_res = 0;
			last_res = 0;
		}
	}

	// jk turn on soft rep and change weights, if desired
	enable_packing_etables(total_residue,res,res_variant,full_coord);

	inc_extra = get_unboundrot_flag();
	if ( inc_extra ) retrieve_unbound_rotamers(extra_rot,extra_chi);
//bk go through the sequence positions that will be repacked, and choose
//bk the lowest energy rotamer at every position
	for ( int cycle=1; cycle<= rottrial_cycles; ++cycle ) {
		new_rottrial_rotamer = false;
		for ( int position = 1; position <= numpositions; ++position ) {
			int seqpos( repack_list(position) );
			int aa( res(seqpos) );

//bk for future calculations, it will be useful to know the energy of
//bk the protein with the current rotamer removed.

			if ( !symm_rottrial ) {
				for ( int i = 1; i <= total_residue; ++i ) {
					if ( i != seqpos ) {
						atrenergy_minus_current(i) = atrenergy(i)-0.5*atr_pair(seqpos,i);
						repenergy_minus_current(i) = repenergy(i)-0.5*rep_pair(seqpos,i);
						solenergy_minus_current(i) = solenergy(i)-0.5*sol_pair(seqpos,i);
						pair_energy_minus_current(i) = pair_energy(i)-0.5*pair_pair(seqpos,i);
						plane_energy_minus_current(i) = plane_energy(i)-0.5*plane_plane(seqpos,i);
						hbenergy_minus_current(i) = hbenergy(i)-0.5*hbond_pair(seqpos,i);
						gb_elecenergy_minus_current(i) =
							gb_elecenergy(i)  - 0.5*gb_elec_pair(seqpos, i);
						cstenergy_minus_current(i) = cstenergy(i)-0.5*cst_pair(seqpos,i);
						h2oenergy_minus_current(i) = h2oenergy(i)-0.5*h2o_pair(seqpos,i);
						h2ohbenergy_minus_current(i) =
							h2ohbenergy(i)-0.5*h2o_hb_pair(seqpos,i);
					} else {
						atrenergy_minus_current(i) = 0.0;
						repenergy_minus_current(i) = 0.0;
						solenergy_minus_current(i) = 0.0;
						pair_energy_minus_current(i) = 0.0;
						plane_energy_minus_current(i) = 0.0;
						hbenergy_minus_current(i) = 0.0;
						gb_elecenergy_minus_current(i) = 0.0;
						cstenergy_minus_current(i) = 0.0;
						h2oenergy_minus_current(i) = 0.0;
						h2ohbenergy_minus_current(i) = 0.0;
					}
				}
			} else {
				// symmetric rottrial
				for ( int i = 1; i <= total_residue; ++i ) {
					if ( clone_follows(i) ) continue; // clone position
					if ( i==seqpos ) {
						atrenergy_minus_current(i) = 0.0;
						repenergy_minus_current(i) = 0.0;
						solenergy_minus_current(i) = 0.0;
						pair_energy_minus_current(i) = 0.0;
						plane_energy_minus_current(i) = 0.0;
						hbenergy_minus_current(i) = 0.0;
						gb_elecenergy_minus_current(i) = 0.0;
						cstenergy_minus_current(i) = 0.0;
					} else {
						atrenergy_minus_current(i) = atrenergy(i);
						repenergy_minus_current(i) = repenergy(i);
						solenergy_minus_current(i) = solenergy(i);
						pair_energy_minus_current(i) = pair_energy(i);
						plane_energy_minus_current(i) = plane_energy(i);
						hbenergy_minus_current(i) = hbenergy(i);
						gb_elecenergy_minus_current(i) = gb_elecenergy(i);
						cstenergy_minus_current(i) = cstenergy(i);
						for ( int k=0; k<= num_clones; ++k ) {
							int const clone_pos( k==0 ? seqpos : clone_list(seqpos,k) );
							atrenergy_minus_current(i) -= 0.5 * atr_pair(clone_pos,i);
							repenergy_minus_current(i) -= 0.5 * rep_pair(clone_pos,i);
							solenergy_minus_current(i) -= 0.5 * sol_pair(clone_pos,i);
							pair_energy_minus_current(i) -= 0.5 * pair_pair(clone_pos,i);
							plane_energy_minus_current(i) -= 0.5 * plane_plane(clone_pos,i);
							hbenergy_minus_current(i) -= 0.5 * hbond_pair(clone_pos,i);
							gb_elecenergy_minus_current(i) -=0.5 * gb_elec_pair(clone_pos,i);
							cstenergy_minus_current(i) -= 0.5*cst_pair(clone_pos,i);
						} // clones
					} // i == seqpos
				} // i
			} // symm_rottrial?


//bk different modes change which amino acids to run over
//bk optE is a mode that sends rotamers energies to an output file so that
//bk they can be used to calibrate the weights on different energy terms

			if ( optE ) {
				looplow = 1;
				loophigh = 20;
//				if ( dna_rotamers_exist && is_DNA(aa) ) {
//					looplow = 22;
//					loophigh = 25;
//				}
			} else if ( seq_opt == "multi" ) {
				looplow = 1;
				loophigh = 20;
			} else if ( fast_ala_scan ) {
				looplow = 1;
				loophigh = 1;
			} else {
				looplow = aa;
				loophigh = aa;
			}

//bk inc_current tells get_rotamers if the current sidechain has already
//bk been accounted for.  If inc_current is false than get_rotamers should
//bk add the current rotamer to the rotamer list.
			if ( optE || fast_ala_scan ) {
				inc_current = false;
			} else {
				// also minimize the current rotamer
				if ( minimize_rot ) inc_current = false;
				else inc_current = true;
			}

			//psh the res_variant() reflects the variant type of the input structure. This has a potential
			//psh problem when the wildtype has an amino acid variant at the seqpos. (e.g. HIS variant 2)
			//psh modification below forces the aav to be 1 so it won't change the desired "temporary" behavior
			//psh For conformers, since it defaults to both his tautomer types, the aav will be updated from rotamers namespace
			if (use_conformer){
				aav=1; // conformers lacks rotamers of other variant types so force to 1 here.  They are taken care of elsewhere.
			}
			else { // to preserve the behavior of terminus amino acids and water rotamers among others
				aav = res_variant(seqpos);
			}
			//bk dummy_protamers is passed to get_rotamers for the bump check, all are
			//bk set to true, meaning don't do a bump check with any sidechain atoms
			//psh this design map is hijacked for conformer lib optE purpose since
			//psh it doesn't really serve any purpose.  the bumpcheck bypass is taken
			//psh care of by the mode = optE argument

      PackerTask Task( misc::total_residue, misc::res );

			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				for ( int rotaa(looplow); rotaa <= loophigh; ++rotaa ) {
					Task.get_designmap().set(seqpos,rotaa);
				}
			}
			// loop over available amino acids
			for ( int rotaa( looplow ); rotaa <= loophigh; ++rotaa ) {

				rotamer_set.reset();
				nrotaa = 0;

				if (use_conformer && confLibHandler.isConformerKeyValid(confLibHandler.conformerKeyFromCoord(seqpos,rotaa))) {
					rotamer_set.buildBDAconformers(total_residue, res, res_variant, full_coord, Task.get_designmap(), seqpos, rotaa);
					// initialize rotactcoord
					for (int i = 1; i<= rotamer_set.nrotamers(); i++){
						put_wcentroid(rotamer_set.get_rotcoord(i),rotamer_set.get_rotactcoord(i),rotamer_set.report_aa(i));
					}

				} else {

					if ( rotaa == aa_cys && aa != rotaa ) {
						if ( !dna_interface && !get_ligand_flag() ) goto L85;
					}
					if ( seq_opt == "multi" && ( !design_map.get(seqpos,rotaa) ) ) goto L85;

					if ( inc_extra ) {
						rotamer_set.add_extra_rot_to_seqpos(rotaa,aav,seqpos,template_pack::phi,
						 template_pack::psi,full_coord,extra_chi,extra_rot,nrotaa);
					}

					// Restrict "terminus" variants to the termini, His tautomers to places where His is allowed
					if ( ! termini_ns::is_N_terminus(seqpos) ) {
						Task.get_designmap().disallow_variant_type_at_residue(aaproperties_pack::aav_Nterm, seqpos);
					}
					if ( ! termini_ns::is_C_terminus(seqpos) ) {
						Task.get_designmap().disallow_variant_type_at_residue(aaproperties_pack::aav_Cterm, seqpos);
					}
					if ( ! design_map.get(seqpos,aa_his) ) {
						Task.get_designmap().disallow_variant_type_at_residue(aaproperties_pack::aav_his_taut, seqpos);
					}

					// aav loop added to process variant types properly, if any
					for ( aav = 1; aav <= nvar(rotaa); ++aav ) {

						// jk jumpout if this aav corresponds to a variant we're not using
						bool jumpout( true );
						for ( int vtype = 1; vtype <= number_aav_type; ++vtype ) {
							if ( variant_type( vtype, rotaa, aav ) &&
									 Task.get_designmap().variant_type_is_allowed(vtype,seqpos) ) {
								jumpout = false;
								break;
							}
						}
						if ( jumpout ) continue;

						if ( !optE && param_aa::is_DNA( rotaa ) && design::hydrate_dna ) {
							rotamer_set.get_rotamers_seqpos_aa_aav( total_residue,res, res_variant, full_coord, rotaa, aav, seqpos, nrotaa,
								inc_current, "optimizeH", Task );
						} else {
							rotamer_set.get_rotamers_seqpos_aa_aav( total_residue, res, res_variant, full_coord,rotaa, aav, seqpos, nrotaa,
								inc_current, "optE", Task);
						}
					}
//					if ( param_aa::is_DNA( rotaa ) ) {
//						std::cout << "rottrial DNA: " << seqpos << " nrotamers = " <<
//							rotamer_set.nrotamers() << std::endl;
//					}
				}

	//jjh   Get the born radii for the rotamers

				if ( gen_born ) {

					// build dummy placeholders
					for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
						Task.get_designmap().set(seqpos,aa);
					}
					gb_build_placeholders(total_residue, full_coord, res, res_variant, Task.get_designmap());

	// get Born radii using dummy placeholders for all (for testing only)
					// in fact this should not use dummy placeholders at all!
					for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
						Task.get_designmap().fix_completely(seqpos);
						Task.get_designmap().require_base_rotamer( seqpos );
					}
					for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
						if ( Task.get_designmap().repack_residue(seqpos) ) {
							std::cout << "aagh! using placeholder!" << std::endl;
							exit( EXIT_FAILURE );
						}
					}
					gb_get_rotamers_born_radii(total_residue, full_coord, res,
					 res_variant, rotamer_set, Task.get_designmap());
				}

				// this is a RotamerSet for a single amino acid at a single position
				rot_chosen = false; // for optE
				for ( int rot(1); rot <= rotamer_set.nrotamers(); ++rot ) {
					int const rotaav( rotamer_set.report_aav(rot) );


					if ( symm_rottrial ) {
						// copies the rotamer coords into full_coord and actcoord
						create_symmetric_rotamers( seqpos, rotaa, rotaav,
																			 rotamer_set.get_rotcoord(rot),
																			 rotamer_set.get_rotactcoord(rot),
																			 full_coord);
					}

					get_trial_energies(
						rotamer_set.get_rperc(rot), rotamer_set.get_rotcoord(rot),
						rotamer_set.get_rot_born_radius(rot), seqpos, rotaa,
						rotaav, rotamer_set.get_rotactcoord(rot),
						total_residue,res,res_variant,full_coord,
						symm_rottrial );

					if ( minimize_rot ) { //chu: minimize this rotamer
						PROF_START(prof::RTMIN);
						minimize_trial_energies(
							rotamer_set.get_rchi(rot), rotamer_set.get_rrot(rot), seqpos, rotaa,
							rotaav, rotamer_set.get_rotcoord(rot),
							rotamer_set.get_rot_born_radius(rot), rotamer_set.get_rotactcoord(rot),
							0.1, "dfpmin", total_residue, res, res_variant, full_coord);
						PROF_STOP(prof::RTMIN);
					}

					if ( optE || fast_ala_scan ) {
						if ( optE_rep_cut ) {
//ja for the sake of downstream memory and efficiency, skip rotamers with
//ja HUGE clashes that will /never/ be optimized to be 'the best' by optE
//ja (Must be careful with this though, since optE will also attentuate Wrep if
//ja necessary for recovery.)
							float rot_repE( 2*repenergy_trial(seqpos) );
							if ( rot_repE < optE_rep_cut_val || !rot_chosen ) {
								rot_chosen = true; // optE needs at least one rotamer per aa
							}
							else continue; // skip to next rotamer
						}

						if ( fast_ala_scan ) {
							// jk Alanine scanning
							float Ediff = fullatomE_trial - fullatomE;
							fast_ala_scan_x << seqpos << ' ' << Ediff << std::endl;
							continue;
						}

						int rotnum( rotamer_set.report_rotnum(rot) ); // redundant?
						int nb( neighbors(seqpos) );
						// rmsd to native rotamer--additional success criterion for optE
						float rmsd( sidechain_rmsd( aa, res_variant(seqpos),
							           full_coord(1,1,seqpos), rotaa, rotaav,
							           rotamer_set.get_rotcoord(rot) ) );
						if ( optE_ctsa ) {
							//// superficial output format change
							write_rot_energies_ctsa( seqpos, rotaa, rotnum, aa, nb );
						} else if ( get_ligand_flag() ) {
							//// modified output for lig interface, electrostatics included
							write_rot_energies_lig( seqpos, rotaa, rotnum, aa, nb );
						} else if ( optE_jk ) {
							opte_output.write_rot_ene( seqpos, rotaa, aa, nb, rmsd );
						} else {
							write_rot_energies( seqpos, rotaa, rotnum, aa, nb );
						}
					} else {
	//					if ( ( fullatomE_trial < fullatomE ) || ( rot == 1 ) )
						if ( fullatomE_trial < fullatomE ) {

							new_rottrial_rotamer( seqpos ) = true;
							reset_energy_coord_from_trial(
								total_residue,res, res_variant,full_coord,
								rotamer_set.get_rotcoord(rot), rotamer_set.get_rchi(rot),
								rotamer_set.get_rrot(rot), seqpos, rotamer_set.get_rotactcoord(rot),
								rotamer_set.get_rot_born_radius(rot),
								rotaav, rotaa, symm_rottrial );
							maps_set_new_rotamer(seqpos);
							if ( symm_rottrial ) {
								for ( int k=1; k<= num_clones; ++k ) {
									new_rottrial_rotamer( clone_list( seqpos, k ) ) = true;
									maps_set_new_rotamer( clone_list( seqpos, k ) ); //unnec4pose
								}
							}
						} else if ( symm_rottrial ) {
							// reset coords at clone positions
							create_symmetric_rotamers( seqpos, rotaa, res_variant(seqpos),
																				 full_coord(1,1,seqpos),
																				 actcoord(1,seqpos),
																				 full_coord);
						}
					}
				} // rotamers
	L85:;
			} // amino acids
	} // seq positions

	if ( cycle < rottrial_cycles ) {
		// update the repack_list
		FArray1D_bool allow_move( total_residue );
		for ( int i=1; i<= total_residue; ++i ) {
			if ( pose_rottrial ) {
				allow_move(i) = score_get_current_pose().get_allow_chi_move( i ) ||
					design::hydrate_dna &&
					param_aa::is_DNA( score_get_current_pose().res(i) );
			} else {
				allow_move(i) = design_map.repack_residue(i);
			}
		}
		update_repack_list( repack_list, numpositions, total_residue,
												new_rottrial_rotamer, allow_move, neighborlist );
		random_order(repack_list,numpositions);
		GRAPHICS_LOG( "rottrial cycle= "+string_of( cycle )+" numpositions= "+
									string_of(numpositions) );
	}

	} // cycle

	// jk disable soft rep
	disable_packing_etables(total_residue,res,res_variant,full_coord);

	if ( pose_rottrial ) {
		// update pose coordinates
		score_get_current_pose().after_rotamer_trials( total_residue,
																									 res_variant, full_coord );
		if ( symm_rottrial ) {
			reset_symmetric_packing();
		}
	}

//bk write the new protein coordinates into an output argument so that
//bk they can be passed to rosetta
	for ( int k = 1, max_atom = MAX_ATOM(); k <= total_residue; ++k ) {
		for ( int j = 5; j <= max_atom; ++j ) {
			for ( int i = 1; i <= 3; ++i ) {
				xyz_out(i,j,k) = full_coord(i,j,k);
			}
		}
	}
	for ( int k = 1; k <= total_residue; ++k ) {
		res_variant_out(k) = res_variant(k);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin fullatom_rotamer_trials_trie(xyz_out,res_variant_out,allow_rottrial)
///
/// @brief
///bk tries new rotamers at residues that gained energy when the last
///bk fragment insertion was made
///apl uses the Trie vs Background algorithm, roughly twice as fast.
///
/// @detailed
///bk !!!!!  must call fullatom_energy before this function to
///bk !!!!!  fill global variables needed by rotamer_trials
///
///car exits with the fullatom_energy arrays current
///car returns coordinates with updated rotamers in place
///
/// @param[in]        res_variant_out            int
/// @param[in]        allow_rottrial            bool
/// @param[in]        xyz_out                   float
/// @param[out]       res_variant_out            int
/// @param[out]       xyz_out                   float
///
/// @global_read     aa_ala                    int    param_aa.h
/// @global_read     aa_cys                    int    param_aa.h
/// @global_read     aa_gly                    int    param_aa.h
/// @global_read     res_variant                int    misc.h template
/// @global_read     res                       int    misc.h template
/// @global_read     atr_pair                  float       fullatom_energies.h energies
/// @global_read     atrenergy                 float       fullatom_energies.h energies
/// @global_read     best_resenergy            float       fullatom_energies.h best_energies
/// @global_read     fullatome                 float       fullatom_energies.h totals
/// @global_read     fullatome_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     hbenergy                  float       fullatom_energies.h energies
/// @global_read     hbond_pair                float       fullatom_energies.h energies
/// @global_read     inform                    int    runlevel.h
/// @global_read     MAX_ATOM                  int    param.h
/// @global_read     minimize_rot              bool    design.h   design_commands
/// @global_read     neighbors                 int    template_pack.h template
/// @global_read     total_residue                      int    misc.h template
/// @global_read     opte                      bool    design.h   design_commands
/// @global_read     opte_ctsa                 bool    design.h   design_commands
/// @global_read     pair_energy               float       fullatom_energies.h energies
/// @global_read     pair_pair                 float       fullatom_energies.h energies
/// @global_read     rep_pair                  float       fullatom_energies.h energies
/// @global_read     repenergy                 float       fullatom_energies.h energies
/// @global_read     resenergy                 float       fullatom_energies.h energies
/// @global_read     runlevel                  int    runlevel.h run_level_status
/// @global_read     sol_pair                  float       fullatom_energies.h energies
/// @global_read     solenergy                 float       fullatom_energies.h energies
/// @global_read     full_coord                       float       misc.h template
///
/// @global_write    atrenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_write    hbenergy_minus_current    float       rotamer_trial_energies.h minus_energies
/// @global_write    pair_energy_minus_current float       rotamer_trial_energies.h minus_energies
/// @global_write    repenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_write    solenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_write    gb_elecenergy_minus_current   float       rotamer_trial_energies.h minus_energies
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fullatom_rotamer_trials_trie(
	FArray3Da_float xyz_out,
	FArray1Da_int res_variant_out,
	DesignMap const & design_map,
	std::string const & seq_opt
)
{
	using namespace design;
	using namespace fullatom_energies;
	using namespace interface;
	using namespace ligand;
	using namespace param;
	using namespace param_aa;
	using namespace rotamer_trial_energies;
	using namespace runlevel_ns;
	using namespace misc;
	using namespace param_pack;
	using namespace disulfides::BOUNDARY;

	initialize_fullatom();

	xyz_out.dimension( 3, MAX_ATOM(), star );
	res_variant_out.dimension( MAX_RES() );

//bk energycut: if energy difference is better than this value, don't
//bk repack this position
//db energycut can now be set externally using set_energycut to allow
//db annealing during fullatom_relax

	int first_res,last_res; // first and last residue that changes
	FArray1D_int repack_list( MAX_RES()() );
	int looplow, loophigh, nrotaa, aav;
	float deltaE;
	FArray2D_int extra_rot( MAX_CHI, MAX_RES()() );
	FArray2D_float extra_chi( MAX_CHI, MAX_RES()() );
	bool inc_current;
	FArray1D_bool repack( MAX_RES()() );
	bool rot_min_switch; // whether to use rot minimization
	bool inc_extra; // additional rotamers to try ?

//bk set repack flag true if change in energy is larger than energycut
//chu if rotamer minimization is applied, escape this energy criteria.
	float const energycut = get_energycut();

	int position = 0;
	int numpositions = 0;

	bool const pose_rottrial( score_check_current_pose() );

	// jk Instantiate a RotamerSet
	RotamerSet rotamer_set;

//jjh get Born radii
	if ( gen_born ) {
		gb_get_all_born_radii();
	}

	if ( !optE && !pose_rottrial ) { // see below for setup for optE mode
		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			deltaE = resenergy(seqpos)-best_resenergy(seqpos);
			rot_min_switch = (deltaE > energycut || minimize_rot);
			if ( rot_min_switch && res(seqpos) != aa_gly &&
					 //---res(seqpos) != aa_ala && allow_rottrial(seqpos) ) {
					 res(seqpos) != aa_ala && design_map.repack_residue(seqpos) ) {
				if ( !( get_norepack_disulf() && cys_res_in_disulf(seqpos) ) ) {
					++position;
					repack_list(position) = seqpos;
				}
			}
		}
		numpositions = position;

		if ( runlevel >= inform ) std::cout <<
		 "rotamer trial at this many positions " << numpositions << std::endl;

//bk make a list that determines the order residues are repacked
//bk (order is random).

		random_order(repack_list,numpositions);
	} else if ( pose_rottrial ) {
		int rottrial_cycles; // doesnt work with TRIE code yet
		pose_setup_repack_list( score_get_current_pose(), repack_list,
														numpositions, rottrial_cycles );
		if ( rottrial_cycles > 1 ) {
			std::cout << "Phil needs to add multiple-cycle compatibility " <<
				"to rotamer_trials with the trie" << std::endl;
			std::exit( EXIT_FAILURE );
		}
		if ( runlevel >= inform ) std::cout <<
			"rotamer trial at this many positions " << numpositions << std::endl;
	} // ! optE

//bk if this function is being used to output energies for optimizing weights
//bk then just set it to repack residues 5 to the lesser of 75 or (total_residue-5)[bk]
//   or (total_residue-5 only) [ctsa], don't include cysteines [cmd] unless it's for dna
	if ( optE ) {
		position = 0;

		if ( dna_interface ) {
			for ( std::list< ResInfo >::iterator
            it( dna_variables::interface_list.begin() );
			      it != dna_variables::interface_list.end(); ++it ) {
				if ( it->design ) {
					++position;
					repack_list(position) = it->pos;
				}
			}

//mj for ligands

		} else if ( get_ligand_flag() ) {
			detect_ligand_interface((*ligand::ligand_one), false, total_residue, res, full_coord);
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if ( int_res(seqpos) ) {
					++position;
					repack_list(position) = seqpos;
				}
			}
//mj end mj

		} else {
			for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
				if ( optE_inter ) {
					if ( interface_residue(seqpos) && res(seqpos) != (aa_cys) ) {
						repack(seqpos) = true;
						++position;
						repack_list(position) = seqpos;
					}
				} else {
//					if ( seqpos > 5 && seqpos < total_residue-5 && res(seqpos) != aa_cys &&
					if ( seqpos > 5 && seqpos < total_residue-5 &&
					 ( optE_ctsa || seqpos <= 75 ) ) {
						++position;
						repack_list(position) = seqpos;
					}
				}
			}
		}
		numpositions = position;

		if ( position > 0 ) {
			first_res = repack_list(1);
			last_res = repack_list(position);
		} else {
			first_res = 0;
			last_res = 0;
		}
	}

//bk dummy_protamers is passed to get_rotamers for the bump check, all are
//bk set to true, meaning don't do a bump check with any sidechain atoms
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );
	PackerTask Task( pose );
	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		//---		dummy_protamers(seqpos) = true;
		Task.get_designmap().set(seqpos,res(seqpos));
	}

	// jk turn on soft rep and change weights, if desired
	enable_packing_etables(total_residue,res,res_variant,full_coord);

//apl create rotamer trie for each residue
	std::vector< rotamer_trie* > bg_tries(total_residue + 1);
	for (int ii = 1; ii <= total_residue; ++ii)
	{
		create_background_rotamer_trie(ii, bg_tries[ii]);
		//std::cerr << "res " << ii << "trie:" << std::endl;
		//std::cerr << *(bg_tries[ii]) << std::endl;
	}

	inc_extra = get_unboundrot_flag();
	if ( inc_extra ) retrieve_unbound_rotamers(extra_rot,extra_chi);
//bk go through the sequence positions that will be repacked, and choose
//bk the lowest energy rotamer at every position
	for ( int position = 1; position <= numpositions; ++position ) {
		int seqpos = repack_list(position);

//bk for future calculations, it will be useful to know the energy of
//bk the protein with the current rotamer removed.

		// for ( int i = 1; i <= total_residue; ++i ) {
		// 	if ( i != seqpos ) {
		// 		atrenergy_minus_current(i) = atrenergy(i)-0.5*atr_pair(seqpos,i);
		// 		repenergy_minus_current(i) = repenergy(i)-0.5*rep_pair(seqpos,i);
		// 		solenergy_minus_current(i) = solenergy(i)-0.5*sol_pair(seqpos,i);
		// 		pair_energy_minus_current(i) = pair_energy(i)-0.5*pair_pair(seqpos,i);
		// 		plane_energy_minus_current(i) = plane_energy(i)-0.5*plane_plane(seqpos,i);
		// 		hbenergy_minus_current(i) = hbenergy(i)-0.5*hbond_pair(seqpos,i);
		// 		gb_elecenergy_minus_current(i) =
		// 			gb_elecenergy(i)  - 0.5*gb_elec_pair(seqpos, i);
		// 		cstenergy_minus_current(i) = cstenergy(i)-0.5*cst_pair(seqpos,i);
		// 	} else {
		// 		atrenergy_minus_current(i) = 0.0;
		// 		repenergy_minus_current(i) = 0.0;
		// 		solenergy_minus_current(i) = 0.0;
		// 		pair_energy_minus_current(i) = 0.0;
		// 		plane_energy_minus_current(i) = 0.0;
		// 		hbenergy_minus_current(i) = 0.0;
		// 		gb_elecenergy_minus_current(i) = 0.0;
		// 		cstenergy_minus_current(i) = 0.0;
		// 	}
		// }

//bk different modes change which amino acids to run over
//bk optE is a mode that sends rotamers energies to an output file so that
//bk they can be used to calibrate the weights on different energy terms
		if ( optE ) {
			looplow = 1;
			loophigh = 20;
		} else if ( seq_opt == "multi" ) {
			looplow = 1;
			loophigh = 20;
		} else {
			looplow = res(seqpos);
			loophigh = res(seqpos);
		}

//bk inc_current tells get_rotamers if the current sidechain has already
//bk been accounted for.  If inc_current is false than get_rotamers should
//bk add the current rotamer to the rotamer list.
		if ( optE ) {
			inc_current = false;
		} else {
			if ( minimize_rot ) {
				inc_current = false; // also minimize the current rotamer
			} else {
				inc_current = true;
			}
		}

		aav = res_variant(seqpos);//1; // temporary: this only allows variant 1 for each
		// amino acid
		for ( int aa = looplow; aa <= loophigh; ++aa ) {
			rotamer_set.reset();
			nrotaa = 0;
			if ( aa == aa_cys && res(seqpos) != aa ) {
				if ( !dna_interface && !get_ligand_flag() ) continue;
			}
			//---			if ( seq_opt == "multi" && ( !des_select(aa,seqpos) ) ) continue;
			if ( seq_opt == "multi" && ( !design_map.get(seqpos,aa) ) ) continue;
			if ( inc_extra ) {
				rotamer_set.add_extra_rot_to_seqpos(aa,aav,seqpos,template_pack::phi,
					template_pack::psi,full_coord,extra_chi,extra_rot,nrotaa);
			}

			rotamer_set.get_rotamers_seqpos_aa_aav(total_residue, res, res_variant, full_coord, aa,aav,seqpos,nrotaa,inc_current,"optE",Task);

			//std::cerr << "SEQPOS Rotamers: " << seqpos << " " << rotamer_set.nrotamers() << std::endl;

			if ( rotamer_set.nrotamers() == 0 )
			{
				//found no rotamers -- keep the original rotamer
				continue;
			}

			rotamer_trie * rt( 0 );
			create_rotamer_trie(rotamer_set, seqpos, rt);
			//std::cerr << "rt" << std::endl;
			//std::cerr << *rt << std::endl;

			//jjh   Get the born radii for the rotamers
 			if ( gen_born )
			{
				// build dummy placeholders
				for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
					Task.get_designmap().set(seqpos,res(seqpos));
				}
				gb_build_placeholders(total_residue, full_coord, res, res_variant,Task.get_designmap());

				// get Born radii using dummy placeholders for all (for testing only)
				for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
					Task.get_designmap().fix_completely(seqpos);
				}
				gb_get_rotamers_born_radii(total_residue, full_coord, res, res_variant, rotamer_set,
																	 Task.get_designmap());
 			}

			trie_trials_energies( rt, seqpos, aa, aav, bg_tries, rotamer_set );
			delete rt; rt=0;
		} // amino acids
	} // seq positions

	// jk disable soft rep
	disable_packing_etables(total_residue,res,res_variant,full_coord);

	if ( pose_rottrial ) {
		// update pose coordinates
		score_get_current_pose().after_rotamer_trials( total_residue, res_variant,
																									 full_coord );
	}

//bk write the new protein coordinates into an output argument so that
//bk they can be passed to rosetta
	for ( int k = 1, max_atom = MAX_ATOM(); k <= total_residue; ++k ) {
		for ( int j = 5; j <= max_atom; ++j ) {
			for ( int i = 1; i <= 3; ++i ) {
				xyz_out(i,j,k) = full_coord(i,j,k);
			}
		}
	}
	for ( int k = 1; k <= total_residue; ++k ) {
		res_variant_out(k) = res_variant(k);
	}

	for (int ii = 1; ii <= total_residue; ++ii)
	{	delete bg_tries[ii];}

}

void
trie_trials_energies(
	rotamer_trie * rt,
	int seqpos,
	int seqpos_aa,
	int seqpos_aav,
	std::vector< rotamer_trie* > & bg_tries,
	RotamerSet const & rotamer_set
)
{
	using namespace design;
	using namespace fullatom_energies;
	using namespace hbonds;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace rotamer_trial_energies;
	using namespace runlevel_ns;
	using namespace misc;
	using namespace template_pack;
	using namespace aaproperties_pack;
	using namespace ligand;
	using namespace enzyme;
	using namespace disulfides::BOUNDARY;
	using namespace water;

	float dunE, Paa_phipsiE, unfE, intra_atrE, intra_repE;
	float lig_solE,lig_couE,lig_h2oE,lig_hbE,lig_atrE,lig_repE,lig_virE,lig_E;
	float solE,atrE,repE,pairE,elecE,plane_totalE,schbE,srbbhbE,lrbbhbE;
	float sc_bbhbE, gb_elecE, cstE;
	float h2olj_E, h2ohb_E;
	float tmp1, tmp2;

	float best_energy;
	int best_rotamer = -1;
	bool cst_exist = classical_constraints::BOUNDARY::get_constraints_exist();

	//set best_energy to current rotamer assignment;

	best_energy = dunenergy(seqpos) - unfenergy(seqpos) + probenergy(seqpos) + intraresenergy(seqpos) + h2oenergy(seqpos) + h2ohbenergy(seqpos);
	int li = atr_pair.index(1, seqpos);
	for (int ii = 1; ii <= total_residue; ++ii)
	{
		best_energy += atr_pair[ li ] + rep_pair[ li ] + sol_pair[ li ] + hbond_pair[ li ]
			+ pair_pair[ li ] + gb_elec_pair[ li ] + cst_pair[ li ];
		++li;
	}

	//zero trial arrays
	gb_elec_trial_pair = 0;
	pair_trial_pair = 0;
	cst_trial_pair = 0;

	//componentE has four rows: 1) LJ atr, 2) LJ rep 3) LK Solv 4) Kortemme's Hbond
	FArray3D_float trie_trials_componentE( 4, rotamer_set.nrotamers(), MAX_RES(), 0. );
	FArray1D_float trie_trials_sumE( rotamer_set.nrotamers(), 0. );

	//apl Assumption: - during rotamer trials, the status of the disulfide flag will not change
	//apl I'm keeping a local copy of the flag for speed.
	bool const local_disulfide_flag = get_disulf_flag();

	for ( int ii = 1; ii <= total_residue; ++ii)
	{
		if (ii == seqpos) continue;

		int iiaa = res( ii );
		int iiaav = res_variant( ii );
		if (neighborlist( ii, seqpos ) )
		{
			//disulfides must be handled separately from all other residue/residue interactions
			if ( local_disulfide_flag && seqpos_aa == aa_cys && iiaa == aa_cys &&
				 cys_pair_in_disulf(seqpos, ii ) )
			{
				//apl disulfides are rare, if at all present
				//apl the presence of this code should have little effect on performance
				//apl yet makes trie_trials compatable with disulfide predictions.

				float & xyz_11s = full_coord(1,1,ii);
				for (int kk = 1; kk <= rotamer_set.nrotamers(); ++kk)
				{
					FArray2Da_float kkrotcoord(rotamer_set.get_rotcoord(kk));
					kkrotcoord.dimension(3,MAX_ATOM() );

					FArray1Da_float kkrotactcoord(rotamer_set.get_rotactcoord(kk));
					kkrotactcoord.dimension(3);

					get_res_resE(seqpos_aa, iiaa, seqpos_aav, iiaav, kkrotcoord, xyz_11s,kkrotactcoord,
						actcoord(1,ii), seqpos, ii,
						solE, atrE, repE, pairE, plane_totalE, elecE);

					trie_trials_componentE( componentE_ljatr_row, kk, ii ) = pack_wts.Watr() * atrE;
					trie_trials_componentE( componentE_ljrep_row, kk, ii ) = pack_wts.Wrep() * repE;
					trie_trials_componentE( componentE_solvE_row, kk, ii ) = pack_wts.Wsol() * solE;

					//apl note: I've discarded pairE and plane_totalE
					//apl pairE is replaced by the disulfide-specific pairE function,
					//apl while planeE still seems stubbed-out.

					get_hbE(true, seqpos_aa, iiaa, seqpos_aav, iiaav, seqpos, ii, neighbors(seqpos),
						neighbors(ii), kkrotcoord, xyz_11s, schbE, srbbhbE,lrbbhbE,
						sc_bbhbE);

					trie_trials_componentE( componentE_hbond_row, kk, ii ) =
						pack_wts.Whbond_sc() * schbE + pack_wts.Whbond_bb() * ( lrbbhbE + srbbhbE ) + pack_wts.Whbond_bb_sc() * sc_bbhbE;

					trie_trials_sumE( kk ) += pack_wts.Watr() * atrE + pack_wts.Wrep() * repE +  pack_wts.Wsol() * solE + pack_wts.Whbond_sc() * schbE
						+ pack_wts.Whbond_bb() * ( lrbbhbE + srbbhbE ) + pack_wts.Whbond_bb_sc() * sc_bbhbE;

				}

			}
			else
			{
				//apl Trie Vs Background speeds up rotamer trials by a factor of 2
				rt->trie_vs_background( *( bg_tries[ii]), trie_trials_componentE(1,1,ii),
					trie_trials_sumE);
			}
		}
	}

	unfE = pack_wts.Waa(seqpos_aa);
	for ( int ii = 1; ii <= rotamer_set.nrotamers(); ++ii)
	{
		//apl rotamer internal energies
		float const neg_Wdun_log_rperc = -pack_wts.Wdun() * std::log( rotamer_set.get_rperc(ii) );
		dunE = neg_Wdun_log_rperc;
		get_Paa_ppE(template_pack::phi(seqpos),template_pack::psi(seqpos),seqpos_aa,
      Paa_phipsiE,tmp1,tmp2);
		Paa_phipsiE *= pack_wts.Wone();
		get_intra_resE(seqpos_aa,seqpos_aav,rotamer_set.get_rotcoord(ii),intra_atrE,intra_repE);
		intra_repE *= pack_wts.Wintra();

		float h2oElj = 0.0;
		float h2oEhb = 0.0;
		if ( use_hetero_h2o ) {
			get_sc_hetero_h2oE(seqpos_aa,seqpos_aav,rotamer_set.get_rotcoord(ii),h2oElj);
			get_sc_hetero_h2oEhb(seqpos_aa,seqpos_aav,rotamer_set.get_rotcoord(ii),h2oEhb);
		}
		h2olj_E = pack_wts.Wh2o_lj() * h2oElj;
		h2ohb_E = pack_wts.Wh2o_hb() * h2oEhb;

		trie_trials_sumE(ii) += dunE + Paa_phipsiE + intra_repE - unfE + h2olj_E + h2ohb_E;

		//apl rotamer/ligand energies
		//mj check on ligand flag
		if ( get_ligand_flag() )
		{

			//mj compute virtual energies
			get_sc_ha_virtualE(seqpos, seqpos_aa, seqpos_aav, rotamer_set.get_rotcoord(ii),
				lig_virE, false, true, true,(*ligand::ligand_one));

			//mj compute vdw atr, vdw rep
			get_sc_haE(seqpos, seqpos_aa, seqpos_aav, rotamer_set.get_rotcoord(ii),
				lig_solE,lig_atrE,lig_repE,false,true,(*ligand::ligand_one));

			//mj compute h bond energies
			get_sc_ha_hbondE(seqpos, seqpos_aa, seqpos_aav, rotamer_set.get_rotcoord(ii),
                       false,true,(*ligand::ligand_one),lig_hbE); // no deriv

			//mj compute h2o energies
			get_sc_ha_h2oE(seqpos, seqpos_aa, seqpos_aav, rotamer_set.get_rotcoord(ii),
				lig_h2oE,(*ligand::ligand_one));

			//mj compute electrostatics
			if ( !gen_born )
			{
				get_sc_ha_coulombE(seqpos, seqpos_aa, seqpos_aav, rotamer_set.get_rotcoord(ii), ligand::ligand_one,
					lig_couE,false,true);
			}
			else
			{
        FArray2D_float het_atom_coord;
        FArray1D_float het_atom_charge;
        FArray1D_int het_atom_type;
        ligand::ligand_one->get_FArray2D_of_coordinates(ligand::ligand_one->atom_vector,het_atom_coord);
        ligand::ligand_one->get_FArray1D_of_charge(ligand::ligand_one->atom_vector, het_atom_charge);
        ligand::ligand_one->get_FArray1D_of_atom_type(ligand::ligand_one->atom_vector, het_atom_type);
				//jjh get GB electrostatics between protein and ligand
				lig_couE = gb_get_ligand_elecE( natoms( seqpos_aa, seqpos_aav ), MAX_ATOM(),
					rotamer_set.get_rotcoord(ii), rotamer_set.get_rot_born_radius(ii),
					atomic_charge( 1, seqpos_aa, seqpos_aav ), fullatom_type( 1, seqpos_aa, seqpos_aav),
					ligand::ligand_one->atom_vector.size(), HETERO_ATOM_MAX(), het_atom_coord( 1, 1 ),
					ligand::ligand_one->hetero_born_radius( 1 ), het_atom_charge( 1 ),
					het_atom_type( 1 ), false);
				//mjdebug
				//std::cout << "===> rot_trails: " << SS( lig_couE) << SS( aa1) << SS( aav1) <<  std::endl;
				//mjend
			}

			//mj save energies
			trie_trials_componentE( componentE_ljatr_row, ii, seqpos) += pack_wts.Wlig_atr()*lig_atrE;
			trie_trials_componentE( componentE_ljrep_row, ii, seqpos) += pack_wts.Wlig_rep()*lig_repE;
			trie_trials_componentE( componentE_solvE_row, ii, seqpos) += pack_wts.Wlig_sol()*lig_solE;
			trie_trials_componentE( componentE_hbond_row, ii, seqpos) += pack_wts.Wlig_hb()*lig_hbE;
			gb_elec_trial_pair(seqpos) += pack_wts.Wlig_cou()*lig_couE;
			pair_trial_pair(seqpos) += pack_wts.Wlig_vir()*lig_virE;

			//mj add energies
			lig_E = pack_wts.Wlig_sol()*lig_solE + pack_wts.Wlig_atr()*lig_atrE + pack_wts.Wlig_rep()*lig_repE +
				pack_wts.Wlig_cou()*lig_couE + pack_wts.Wlig_vir()*lig_virE + pack_wts.Wlig_hb()*lig_hbE;

			trie_trials_sumE( ii ) +=  lig_E;

			//mj print energies
			//if ( runlevel >= verbose ) std::cout << "ligand energies: " <<
			//pack_wts.Wlig_sol()*lig_solE << ' ' <<
			//pack_wts.Wlig_atr()*lig_atrE << ' ' <<
			//pack_wts.Wlig_rep()*lig_repE << ' ' <<
			//pack_wts.Wlig_hb()*lig_hbE << ' ' <<
			//pack_wts.Wlig_cou()*lig_couE << ' ' <<
			//pack_wts.Wlig_vir()*lig_virE << ' ' <<
			//lig_E << std::endl;

		}
		//end ligand


		//apl I now go back and compute the rotamer-pair energy terms which the trie does not.
		//apl These terms include 1) pairE energies, 2) gen born and 3) cstE
		pair_trial_pair = 0;
		for ( int jj = 1; jj <= total_residue; ++jj)
		{
			int jjaa = res( jj );
			int jjaav = res_variant( jj );

			if (jj != seqpos )
			{
				if ( neighborlist( jj, seqpos) )
				{
					//again, disulfides present special cases
					if ( local_disulfide_flag && seqpos_aa == aa_cys && jjaa == aa_cys &&
						cys_pair_in_disulf(seqpos, jj ) )
					{
						float cys_pairE = getTotWtdScThisDisulf_pack(seqpos, rotamer_set.get_rotcoord(ii), jj, full_coord(1,1,jj) );
						pair_trial_pair( jj ) = cys_pairE;
						trie_trials_sumE( ii ) += cys_pairE;

					}
					else
					{
						get_pairtermE(seqpos_aa, res(jj), seqpos, jj, rotamer_set.get_rotactcoord(ii),
							actcoord(1,jj),pairE,tmp1);
						pairE *= pack_wts.Wpair();
						pair_trial_pair( jj ) = pairE;
						trie_trials_sumE(ii) += pairE;
					}

					if ( gen_born )
					{
						gb_elecE = gb_get_res_res_elecE( seqpos, seqpos_aa, seqpos_aav,
							rotamer_set.get_rotcoord(ii), rotamer_set.get_rot_born_radius(ii),
							jj, jjaa, jjaav, full_coord(1,1,jj), born_radius(1, jj));
						gb_elecE *= pack_wts.Wgb_elec();
						gb_elec_trial_pair( jj ) = gb_elecE;
						trie_trials_sumE( ii ) += gb_elecE;
					}

				} //end if neighbors
			}
			else // seqpos == jj, gen born energies apply
			{
				if ( gen_born )
				{
					gb_elecE = gb_get_res_res_elecE( seqpos, seqpos_aa, seqpos_aav,
						rotamer_set.get_rotcoord(ii), rotamer_set.get_rot_born_radius(ii),
						seqpos, seqpos_aa, seqpos_aav, rotamer_set.get_rotcoord(ii),
						rotamer_set.get_rot_born_radius(ii));
					gb_elecE *= pack_wts.Wgb_elec();
					gb_elec_trial_pair( jj ) += gb_elecE; // may contain ligand/rotamer gb energy, use +=
					trie_trials_sumE( ii ) += gb_elecE;
				}

			}


			if ( cst_exist )
			{
				//car add cst E;  may be intraresidue, may be long range

				//apl Fixing a bug: intraresidue constraints should use
				//apl rotcoord for the rotamer being considered, not full_coord(1,1,seqpos)
				//apl 10/23/06 fixing my own reference-initialization bug; thanks Phil!

				if ( seqpos == jj ) {
					cstE = classical_constraints::BOUNDARY::get_res_res_cstE(seqpos, jj, seqpos_aa, seqpos_aav,
					  jjaa, jjaav, rotamer_set.get_rotcoord(ii), rotamer_set.get_rotcoord(ii));
				} else {
					cstE = classical_constraints::BOUNDARY::get_res_res_cstE(seqpos, jj, seqpos_aa, seqpos_aav,
					  jjaa, jjaav, rotamer_set.get_rotcoord(ii), full_coord(1,1,jj));
				}
				cst_trial_pair( jj ) = cstE;
				trie_trials_sumE( ii ) += cstE;
				//no packer-specific weight, scoring function weight applied in
				//get_res_res_cstE

			}
		}

		if ( trie_trials_sumE( ii ) < best_energy )
		{
			best_rotamer = ii;
			best_energy  = trie_trials_sumE(ii);

			trie_best_pair_trial_pair = pair_trial_pair;
			trie_best_gb_elec_trial_pair = gb_elec_trial_pair;
			trie_best_cst_trial_pair = cst_trial_pair;

			trie_best_dun_trial = dunE;
			trie_best_prob_trial = Paa_phipsiE;
			trie_best_unf_trial = unfE;
			trie_best_intrares_trial = intra_repE;
			trie_best_h2olj_trial = h2olj_E;
			trie_best_h2ohb_trial = h2ohb_E;

		}
	}

	//std::cerr << "res " << seqpos <<   "best rotamer: " << best_rotamer << " best_energy " << best_energy << std::endl;

	if (best_rotamer == -1) return; //didn't find a better rotamer than the one originally present

	set_best_rotamer_trie_trials(seqpos, best_rotamer, seqpos_aa, seqpos_aav, bg_tries, trie_trials_componentE,
		rotamer_set.get_rotcoord(best_rotamer), rotamer_set.get_rotactcoord(best_rotamer),
		rotamer_set.get_rchi(best_rotamer), rotamer_set.get_rrot(best_rotamer));
	return;
}

void
set_best_rotamer_trie_trials(
	int seqpos,
	int best_rotamer,
	int rot_aa,
	int rot_aav,
	std::vector< rotamer_trie* > & bg_tries,
	FArray3D_float & trie_trials_componentE,
	FArray2Da_float rotcoord,
	FArray1Da_float rotactcoord,
	FArray1Da_float rotchi,
	FArray1Da_int rotid
)
{
	using namespace design;
	using namespace fullatom_energies;
	using namespace hbonds;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace rotamer_trial_energies;
	using namespace runlevel_ns;
	using namespace misc;
	using namespace template_pack;
	using namespace aaproperties_pack;
	using namespace ligand;
	using namespace enzyme;

	rotcoord.dimension(3,MAX_ATOM() );
	rotactcoord.dimension(3);
	rotchi.dimension(4);
	rotid.dimension(4);

	//apl copy information for best rotamer into
	//apl [atr,rep,sol,hb,pair]energy and [atr,rep,sol,hb,pair]_pair arrays.

	//apl start with this residue
	//apl these "pair" energies don't get chopped in half and handed to either side,
	//apl since both sides are the same residue - seqpos.  Thus no " / 2" at the end
	atrenergy(seqpos) = trie_trials_componentE( componentE_ljatr_row, best_rotamer, seqpos);
	atr_pair(seqpos, seqpos) = atrenergy(seqpos);
	repenergy(seqpos) = trie_trials_componentE( componentE_ljrep_row, best_rotamer, seqpos);
	rep_pair(seqpos, seqpos) = repenergy( seqpos );
	hbenergy( seqpos) = trie_trials_componentE( componentE_hbond_row, best_rotamer, seqpos);
	hbond_pair( seqpos, seqpos) = hbenergy( seqpos );
	solenergy(seqpos) = trie_trials_componentE( componentE_solvE_row, best_rotamer, seqpos);
	sol_pair( seqpos, seqpos ) = solenergy( seqpos );

	gb_elecenergy(seqpos) = trie_best_gb_elec_trial_pair( seqpos );
	gb_elec_pair(seqpos, seqpos) = gb_elecenergy(seqpos);
	cstenergy(seqpos) = trie_best_cst_trial_pair( seqpos );
	cst_pair(seqpos, seqpos) = cstenergy( seqpos );
	pair_energy(seqpos) = trie_best_pair_trial_pair( seqpos );
	pair_pair(seqpos, seqpos) = pair_energy( seqpos );

	dunenergy(seqpos) = trie_best_dun_trial;
	unfenergy(seqpos) = trie_best_unf_trial;
	probenergy(seqpos) = trie_best_prob_trial;
	intraresenergy(seqpos) = trie_best_intrares_trial;
	h2oenergy(seqpos) = trie_best_h2olj_trial;
	h2ohbenergy(seqpos) = trie_best_h2ohb_trial;

	fullatomE = 0;

	{ for ( int ii = 1; ii <= total_residue; ++ii ) {
		if (ii == seqpos) continue;

		float const trials_atr = trie_trials_componentE( componentE_ljatr_row, best_rotamer, ii );
		atrenergy(ii) += (trials_atr - atr_pair( seqpos, ii)) / 2;
		atr_pair( seqpos, ii) = trials_atr;
		atr_pair( ii, seqpos) = trials_atr;
		atrenergy(seqpos) += trials_atr / 2;

		float const trials_rep = trie_trials_componentE( componentE_ljrep_row, best_rotamer, ii );
		repenergy(ii) += (trials_rep - rep_pair( seqpos, ii)) / 2;
		rep_pair( seqpos, ii) = trials_rep;
		rep_pair( ii, seqpos) = trials_rep;
		repenergy(seqpos) += trials_rep / 2;

		float const trials_sol = trie_trials_componentE( componentE_solvE_row, best_rotamer, ii );
		solenergy(ii) += ( trials_sol - sol_pair( seqpos, ii)) / 2;
		sol_pair( seqpos, ii) = trials_sol;
		sol_pair( ii, seqpos) = trials_sol;
		solenergy(seqpos) += trials_sol / 2;

		float const trials_hbe = trie_trials_componentE( componentE_hbond_row, best_rotamer, ii );
		hbenergy( ii) += (trials_hbe - hbond_pair( seqpos, ii)) / 2;
		hbond_pair( seqpos, ii) = trials_hbe;
		hbond_pair( ii, seqpos) = trials_hbe;
		hbenergy(seqpos) += trials_hbe / 2;

		float const trials_pair = trie_best_pair_trial_pair( ii );
		pair_energy(ii) += (trials_pair - pair_pair( seqpos, ii)) / 2;
		pair_pair( seqpos, ii) = trials_pair;
		pair_pair( ii, seqpos) = trials_pair;
		pair_energy(seqpos) += trials_pair / 2;

		float const trials_gb = trie_best_gb_elec_trial_pair( ii );
		gb_elecenergy(ii) += (trials_gb - gb_elec_pair( seqpos, ii)) / 2;
		gb_elec_pair( seqpos, ii ) = trials_gb;
		gb_elec_pair( ii, seqpos ) = trials_gb;
		gb_elecenergy( seqpos ) += trials_gb / 2;

		float const trials_cst = trie_best_cst_trial_pair( ii );
		cstenergy( ii ) += (trials_cst - cst_pair(seqpos, ii)) / 2;
		cst_pair( seqpos, ii ) = trials_cst;
		cst_pair( ii, seqpos ) = trials_cst;
		cstenergy( seqpos ) += trials_cst / 2;

		resenergy(ii) = atrenergy(ii) + repenergy(ii) + solenergy(ii) + hbenergy(ii) +
			pair_energy(ii) + gb_elecenergy(ii) + cstenergy(ii) +
			dunenergy(ii) - unfenergy(ii) + probenergy(ii) +  intraresenergy(ii) +
			h2oenergy(ii) + h2ohbenergy(ii);
		fullatomE += resenergy(ii);
	} } //Objexx:SGM Extra {} scope is a VC++ work-around

	resenergy(seqpos) = atrenergy(seqpos) + repenergy(seqpos) + solenergy(seqpos) + hbenergy(seqpos) +
		pair_energy(seqpos) + gb_elecenergy(seqpos) + cstenergy(seqpos) +
		trie_best_dun_trial - trie_best_unf_trial +
		trie_best_prob_trial + trie_best_intrares_trial +
		trie_best_h2olj_trial + trie_best_h2ohb_trial;

	fullatomE += resenergy(seqpos);

	//bk reset coordinates
	int li_xyz = full_coord.index(1,1,seqpos);
	int li_rotcoord = rotcoord.index(1,1);
	for ( int ii = 1, iie = natoms(rot_aa,rot_aav); ii <= iie; ++ii ) {
		for ( int jj = 1; jj <= 3; ++jj ) {
			full_coord[li_xyz] = rotcoord[li_rotcoord];
			++li_xyz;
			++li_rotcoord;
		}
	}
	for ( int ii = 1; ii <= 3; ++ii ) {
		actcoord(ii,seqpos) = rotactcoord(ii);
	}
	for ( int ii = 1, iie = nchi(rot_aa,rot_aav); ii <= iie; ++ii ) {
		chiarray(ii,seqpos) = rotchi(ii);
		rotarray(ii,seqpos) = rotid(ii);
	}

	bg_tries[seqpos]->update_coordinates(full_coord(1,1,seqpos), rot_aa, rot_aav);
	maps_set_new_rotamer(seqpos);
	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           get_trial_energies(rperc,rotcoord,seqpos1,aa1,aav1,rotactcoord)
///
/// @brief
///
/// @detailed
///
/// @param[in]        aa1                       int
/// @param[in]        aav1                      int
/// @param[in]        rotactcoord               float
/// @param[in]        rotcoord                  float
/// @param[in]        rperc                     float
/// @param[in]        seqpos1                   int
///
/// @global_read     aa_cys                    int    param_aa.h
/// @global_read     res_variant                int    misc.h template
/// @global_read     res                       int    misc.h template
/// @global_read     actcoord                  float       template_pack.h template
/// @global_read     atrenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_read     dunenergy                 float       fullatom_energies.h energies
/// @global_read     intraresenergy            float       fullatom_energies.h energies
/// @global_read     hbenergy_minus_current    float       rotamer_trial_energies.h minus_energies
/// @global_read     neighborlist              bool    template_pack.h template
/// @global_read     neighbors                 int    template_pack.h template
/// @global_read     total_residue                      int    misc.h template
/// @global_read     opte                      bool    design.h   design_commands
/// @global_read     pair_energy_minus_current float       rotamer_trial_energies.h minus_energies
/// @global_read     phi                       float       misc.h template
/// @global_read     probenergy                float       fullatom_energies.h energies
/// @global_read     psi                       float       misc.h template
/// @global_read     repenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_read     runlevel                  int    runlevel.h run_level_status
/// @global_read     solenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_read     gb_elecenergy_minus_current   float       rotamer_trial_energies.h minus_energies
/// @global_read     unfenergy                 float       fullatom_energies.h energies
/// @global_read     protonation_energy        float       fullatom_energies.h energies
/// @global_read     verbose                   int    runlevel.h
/// @global_read     pack_wts                       float       param_pack.h packer_weight class instance
/// @global_read     full_coord                       float       misc.h template
///
/// @global_write    atr_trial_pair            float       rotamer_trial_energies.h trial_energies
/// @global_write    atrenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_write    bbhbenergy_trial          float       rotamer_trial_energies.h trial_energies
/// @global_write    dun_trial                 float       rotamer_trial_energies.h trial_energies
/// @global_write    intrares_trial            float       rotamer_trial_energies.h trial_energies
/// @global_write    fullatome_trial           float       rotamer_trial_energies.h trial_energies
/// @global_write    hbenergy_trial            float       rotamer_trial_energies.h trial_energies
/// @global_write    hbond_trial_pair          float       rotamer_trial_energies.h trial_energies
/// @global_write    pair_energy_trial         float       rotamer_trial_energies.h trial_energies
/// @global_write    pair_trial_pair           float       rotamer_trial_energies.h trial_energies
/// @global_write    prob_trial                float       rotamer_trial_energies.h trial_energies
/// @global_write    rep_trial_pair            float       rotamer_trial_energies.h trial_energies
/// @global_write    repenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_write    resenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_write    sc_bbhbenergy_trial       float       rotamer_trial_energies.h trial_energies
/// @global_write    schbenergy_trial          float       rotamer_trial_energies.h trial_energies
/// @global_write    sol_trial_pair            float       rotamer_trial_energies.h trial_energies
/// @global_write    solenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_write    unf_trial                 float       rotamer_trial_energies.h trial_energies
/// @global_write    protonation_trial         float       rotamer_trial_energies.h trial_energies
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_trial_energies(
	float const rperc,
	FArray2Da_float rotcoord,
	FArray1Da_float rot_born_radius,
	int const seqpos1,
	int const aa1,
	int const aav1,
	FArray1Da_float rotactcoord,
	int total_residue,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3D_float const & full_coord,
	bool const symm_rottrial // = false
)
{
	using namespace design;
	using namespace disulfides::BOUNDARY;
	using namespace fullatom_energies;
	using namespace hbonds;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace pH_ns;
	using namespace rotamer_trial_energies;
	using namespace runlevel_ns;
	using namespace template_pack;
	using namespace aaproperties_pack;
	using namespace ligand;
	using namespace enzyme;
	using namespace water;
	using namespace design_sym; // for symmetric rottrialing
	using namespace disulfides::BOUNDARY;

	rotcoord.dimension( 3, MAX_ATOM() );
	rot_born_radius.dimension( MAX_ATOM() );
	rotactcoord.dimension( 3 );

	int aa2, aav2;
	float solE,atrE,repE,elecE, pairE,plane_totalE,schbE,srbbhbE,lrbbhbE,Paa_phipsiE;
	float sc_bbhbE, gb_elecE, cstE;
	float intra_atrE,intra_repE;
	float lig_solE,lig_couE,lig_h2oE,lig_hbE,lig_atrE,lig_repE,lig_virE,lig_E;
	float tmp1, tmp2;
	float const neg_Wdun_log_rperc = -pack_wts.Wdun() * std::log( rperc );
	bool cst_exist = classical_constraints::BOUNDARY::get_constraints_exist();
	cst_trial_pair = 0.0;

	float const Waa_aa1 = pack_wts.Waa(aa1);
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( i == seqpos1 ) {
			dun_trial(i) = neg_Wdun_log_rperc;
			get_Paa_ppE(template_pack::phi(i),template_pack::psi(i),aa1,Paa_phipsiE,tmp1,tmp2);
			prob_trial(i) = pack_wts.Wone() * Paa_phipsiE;
			unf_trial(i) = Waa_aa1;
			if ( get_pH_packing_flag() ) {
				protonation_trial(i) = dGprotonation_table(aa1,aav1);
			} else {
				protonation_trial(i) = 0.0;
			}
			get_intra_resE(aa1,aav1,rotcoord,intra_atrE,intra_repE);
			intrares_trial(i) = pack_wts.Wintra() * intra_repE;

			float h2oElj = 0.0;
			float h2oEhb = 0.0;
			if ( use_hetero_h2o ) {
				get_sc_hetero_h2oE(aa1,aav1,rotcoord,h2oElj);
				get_sc_hetero_h2oEhb(aa1,aav1,rotcoord,h2oEhb);
			}
			h2olj_trial(i) = pack_wts.Wh2o_lj() * h2oElj;
			h2ohb_trial(i) = pack_wts.Wh2o_hb() * h2oEhb;

		} else {
			dun_trial(i) = dunenergy(i);
			prob_trial(i) = probenergy(i);
			unf_trial(i) = unfenergy(i);
			protonation_trial(i) = protonation_energy(i);
			intrares_trial(i) = intraresenergy(i);
			h2olj_trial(i) = h2oenergy(i);
			h2ohb_trial(i) = h2ohbenergy(i);
		}
		atrenergy_trial(i) = atrenergy_minus_current(i);
		repenergy_trial(i) = repenergy_minus_current(i);
		solenergy_trial(i) = solenergy_minus_current(i);
		pair_energy_trial(i) = pair_energy_minus_current(i);
		plane_energy_trial(i) = plane_energy_minus_current(i);
		hbenergy_trial(i) = hbenergy_minus_current(i);
		cstenergy_trial(i) = cstenergy_minus_current(i);
		gb_elecenergy_trial(i) = gb_elecenergy_minus_current(i);
		h2oenergy_trial(i) = h2oenergy_minus_current(i);
		h2ohbenergy_trial(i) = h2ohbenergy_minus_current(i);

//bk   used for calibrating weights
		if ( optE ) {
			bbhbenergy_trial(i) = hbenergy_trial(i);
			schbenergy_trial(i) = hbenergy_trial(i);
			sc_bbhbenergy_trial(i) = hbenergy_trial(i);
			coulenergy_trial(i) = 0.0;
			lig_couenergy_trial(i) = 0.0;
			lig_virenergy_trial(i) = 0.0;
			lig_atrenergy_trial(i) = 0.0;
			lig_repenergy_trial(i) = 0.0;
			lig_solenergy_trial(i) = 0.0;
			lig_hb_energy_trial(i) = 0.0;
		}
	}

//mj check on ligand flag
	if ( get_ligand_flag() ) {

//mj compute virtual energies
		get_sc_ha_virtualE(seqpos1,aa1,aav1,rotcoord,lig_virE,false,true,true,(*ligand::ligand_one));

//mj compute vdw atr, vdw rep
		get_sc_haE(seqpos1,aa1,aav1,rotcoord,lig_solE,lig_atrE,lig_repE,false,true,(*ligand::ligand_one));

//mj compute h bond energies
		get_sc_ha_hbondE(seqpos1,aa1,aav1,rotcoord,false,true,(*ligand::ligand_one),lig_hbE); // no deriv

//mj compute h2o energies
		get_sc_ha_h2oE(seqpos1,aa1,aav1,rotcoord,lig_h2oE,(*ligand::ligand_one));

//mj compute electrostatics
		if ( !gen_born )
			get_sc_ha_coulombE(seqpos1,aa1,aav1,rotcoord,ligand::ligand_one,lig_couE,false,true);
		else {
        FArray2D_float het_atom_coord;
        FArray1D_float het_atom_charge;
        FArray1D_int het_atom_type;
        ligand::ligand_one->get_FArray2D_of_coordinates(ligand::ligand_one->atom_vector,het_atom_coord);
        ligand::ligand_one->get_FArray1D_of_charge(ligand::ligand_one->atom_vector, het_atom_charge);
        ligand::ligand_one->get_FArray1D_of_atom_type(ligand::ligand_one->atom_vector, het_atom_type);
//jjh get GB electrostatics between protein and ligand
			lig_couE = gb_get_ligand_elecE( natoms( aa1, aav1 ), MAX_ATOM(),
																			rotcoord, rot_born_radius,
																			atomic_charge( 1, aa1, aav1 ), fullatom_type( 1, aa1, aav1),
																			ligand::ligand_one->atom_vector.size(), HETERO_ATOM_MAX(), het_atom_coord( 1, 1 ),
																			ligand::ligand_one->hetero_born_radius( 1 ), het_atom_charge( 1 ),
																			het_atom_type( 1 ), false);

			//mjdebug
			//std::cout << "===> rot_trails: " << SS( lig_couE) << SS( aa1) << SS( aav1) <<  std::endl;
			//mjend
		}

//mj add energies
		lig_E = pack_wts.Wlig_sol()*lig_solE + pack_wts.Wlig_atr()*lig_atrE + pack_wts.Wlig_rep()*lig_repE +
			pack_wts.Wlig_cou()*lig_couE + pack_wts.Wlig_vir()*lig_virE + pack_wts.Wlig_hb()*lig_hbE;

//mj print energies
//  		if ( runlevel >= verbose ) std::cout << "ligand energies: " <<
//  		 pack_wts.Wlig_sol()*lig_solE << ' ' <<
//  		 pack_wts.Wlig_atr()*lig_atrE << ' ' <<
//  		 pack_wts.Wlig_rep()*lig_repE << ' ' <<
//  		 pack_wts.Wlig_hb()*lig_hbE << ' ' <<
//  		 pack_wts.Wlig_cou()*lig_couE << ' ' <<
//  		 pack_wts.Wlig_vir()*lig_virE << ' ' <<
//  		 lig_E << std::endl;

//mj save energies
		if ( ! optE ) {
			atrenergy_trial(seqpos1) += pack_wts.Wlig_atr()*lig_atrE;
			repenergy_trial(seqpos1) += pack_wts.Wlig_rep()*lig_repE;
			solenergy_trial(seqpos1) += pack_wts.Wlig_sol()*lig_solE;
			gb_elecenergy_trial(seqpos1) += pack_wts.Wlig_cou()*lig_couE;
			pair_energy_trial(seqpos1) += pack_wts.Wlig_vir()*lig_virE;
			hbenergy_trial(seqpos1) += pack_wts.Wlig_hb()*lig_hbE;
		} else {
			lig_atrenergy_trial(seqpos1) += pack_wts.Wlig_atr()*lig_atrE;
			lig_repenergy_trial(seqpos1) += pack_wts.Wlig_rep()*lig_repE;
			lig_solenergy_trial(seqpos1) += pack_wts.Wlig_sol()*lig_solE;
			lig_couenergy_trial(seqpos1) += pack_wts.Wlig_cou()*lig_couE;
			lig_virenergy_trial(seqpos1) += pack_wts.Wlig_vir()*lig_virE;
			lig_hb_energy_trial(seqpos1) += pack_wts.Wlig_hb()*lig_hbE;
		}
	}
//mj end ligand

	for ( int seqpos2 = 1; seqpos2 <= total_residue; ++seqpos2 ) {
		aa2 = res(seqpos2);
		aav2 = res_variant(seqpos2);

		if ( seqpos2 == seqpos1 ) {
			atr_trial_pair(seqpos2) = 0.0;
			rep_trial_pair(seqpos2) = 0.0;
			sol_trial_pair(seqpos2) = 0.0;
			pair_trial_pair(seqpos2) = 0.0;
			plane_trial_plane(seqpos2) = 0.0;
			hbond_trial_pair(seqpos2) = 0.0;

//jjh GB electrostatics has i=j interaction
			gb_elecE = 0.0;
			if ( gen_born ) {
				gb_elecE = gb_get_res_res_elecE( seqpos1, aa1, aav1,
					rotcoord, rot_born_radius,
					seqpos1, aa1, aav1, rotcoord, rot_born_radius);
			}
			gb_elecenergy_trial(seqpos1) +=
			  ( gb_elec_trial_pair(seqpos1) = pack_wts.Wgb_elec() * gb_elecE );

			// h2o has self-energy (but not in the hb term)
			h2oenergy_trial(seqpos1) +=
			  ( h2o_trial_pair (seqpos1) = pack_wts.Wh2o_intra() * nh2o(aa1,aav1) );
			h2o_hb_trial_pair(seqpos1) = 0.0;

			// jk geometric solvation term has i=j interaction
			if ( geometric_sol::geometric_sol_flag ) {
				sol_trial_pair(seqpos1) = res_res_geometric_sol(seqpos1,seqpos1,aa1,aav1,aa1,aav1,
																			 rotcoord,rotcoord,true,true,true,true);
			}

		} else {

			float const & xyz_11s( full_coord(1,1,seqpos2) );

			if ( neighborlist(seqpos2,seqpos1) ) {
				get_res_resE(aa1,aa2,aav1,aav2,rotcoord,xyz_11s,rotactcoord,
				 actcoord(1,seqpos2),seqpos1,seqpos2,solE,atrE,repE,pairE, plane_totalE, elecE);
				get_hbE(true,aa1,aa2,aav1,aav2,seqpos1,seqpos2,neighbors(seqpos1),
				 neighbors(seqpos2),rotcoord,xyz_11s,schbE,srbbhbE,lrbbhbE,
				 sc_bbhbE);

				// jk geometric solvation term
				if ( geometric_sol::geometric_sol_flag ) {
					// any polar groups on the rotamer sidechain occluded by fixed sidechain
					solE += res_res_geometric_sol(seqpos1,seqpos2,aa1,aav1,aa2,aav2,
																				rotcoord,xyz_11s,true,true,true,true);
					// any polar groups on fixed sidechain occluded by rotamer sidechain
					solE += res_res_geometric_sol(seqpos2,seqpos1,aa2,aav2,aa1,aav1,
																				xyz_11s,rotcoord,true,true,true,true);
				}

				gb_elecE = 0.0;
				if ( gen_born ) {
					gb_elecE = gb_get_res_res_elecE(
						seqpos1,aa1,aav1,rotcoord,rot_born_radius,
						seqpos2,aa2,aav2,full_coord(1,1,seqpos2),born_radius(1,seqpos2));
				}

				float h2oE = 0.0;
				float h2ohbE = 0.0;
				if ( water_exists() ) {
					get_h2oE(aa1,aa2,aav1,aav2,rotcoord,full_coord(1,1,seqpos2),seqpos1,
									 seqpos2,h2oE,h2ohbE);
				}

				if ( !symm_rottrial ) {

					float temp = 0.5 * pack_wts.Watr() * atrE;
					atrenergy_trial(seqpos1) += temp;
					atrenergy_trial(seqpos2) += temp;
					atr_trial_pair(seqpos2) = temp + temp;

					temp = 0.5 * pack_wts.Wrep() * repE;
					repenergy_trial(seqpos1) += temp;
					repenergy_trial(seqpos2) += temp;
					rep_trial_pair(seqpos2) = temp + temp;

					temp = 0.5 * pack_wts.Wsol() * solE;
					solenergy_trial(seqpos1) += temp;
					solenergy_trial(seqpos2) += temp;
					sol_trial_pair(seqpos2) = temp + temp;

					temp = 0.5 * pack_wts.Wgb_elec() * gb_elecE;
					gb_elecenergy_trial(seqpos1) += temp;
					gb_elecenergy_trial(seqpos2) += temp;
					gb_elec_trial_pair(seqpos2) = temp + temp;

					temp = 0.5 * pack_wts.Wh2o() * pack_wts.Wh2o_lj() * h2oE;
					h2oenergy_trial(seqpos1) += temp;
					h2oenergy_trial(seqpos2) += temp;
					h2o_trial_pair(seqpos2) = temp + temp;

					temp = 0.5 * pack_wts.Wh2o() * h2ohbE;
					h2ohbenergy_trial(seqpos1) += temp;
					h2ohbenergy_trial(seqpos2) += temp;
					h2o_hb_trial_pair(seqpos2) = temp + temp;

					temp = 0.5 * pack_wts.Wpair() * pairE;
					pair_energy_trial(seqpos1) += temp;
					pair_energy_trial(seqpos2) += temp;
					pair_trial_pair(seqpos2) = temp + temp;

					temp = 0.5 * pack_wts.Wplane_total() * plane_totalE;
					plane_energy_trial(seqpos1) += temp;
					plane_energy_trial(seqpos2) += temp;
					plane_trial_plane(seqpos2) = temp + temp;


//chu add in disulf pair terms
					if ( get_disulf_flag() && aa1 == aa_cys && aa2 == aa_cys &&
					 cys_pair_in_disulf(seqpos1,seqpos2) ) {
						temp =
						 0.5 * getTotWtdScThisDisulf_pack(seqpos1,rotcoord,seqpos2,xyz_11s);
						pair_energy_trial(seqpos1) += temp;
						pair_energy_trial(seqpos2) += temp;
						pair_trial_pair(seqpos2) += temp + temp;
					}

					temp = 0.5 * ( pack_wts.Whbond_sc() * schbE +
					 pack_wts.Whbond_bb() * ( lrbbhbE + srbbhbE ) +
					 pack_wts.Whbond_bb_sc() * sc_bbhbE );
					hbenergy_trial(seqpos1) += temp;
					hbenergy_trial(seqpos2) += temp;
					hbond_trial_pair(seqpos2) = temp + temp;

				} else { // symm_rottrial true
					assert( !water_exists() ); // water energies not here yet
					// symmetric case -- all the 1D trial energies go into the
					// independent positions:
					int const seqpos2_indep
						( clone_follows(seqpos2) ? clone_follows(seqpos2) : seqpos2 );
					assert( clone_follows( seqpos2_indep ) == 0 );

					// get weighted energies
					float const
						WatrE        ( pack_wts.Watr() * atrE ),
						WrepE        ( pack_wts.Wrep() * repE ),
						WsolE        ( pack_wts.Wsol() * solE ),
						WpairE       ( pack_wts.Wpair() * pairE ),
						WhbondE      ( pack_wts.Whbond_sc() * schbE +
													 pack_wts.Whbond_bb() * ( lrbbhbE + srbbhbE ) +
													 pack_wts.Whbond_bb_sc() * sc_bbhbE ),
						Wgb_elecE    ( pack_wts.Wgb_elec() * gb_elecE ),
						Wplane_totalE( pack_wts.Wplane_total() * plane_totalE );

					assert( std::abs( Wplane_totalE ) < 1e-2 );

					// increment 1d energies
					atrenergy_trial    ( seqpos1 ) += 0.5 * WatrE;
					repenergy_trial    ( seqpos1 ) += 0.5 * WrepE;
					solenergy_trial    ( seqpos1 ) += 0.5 * WsolE;
					pair_energy_trial  ( seqpos1 ) += 0.5 * WpairE;
					hbenergy_trial     ( seqpos1 ) += 0.5 * WhbondE;
					gb_elecenergy_trial( seqpos1 ) += 0.5 * Wgb_elecE;
					plane_energy_trial ( seqpos1 ) += 0.5 * Wplane_totalE;

					if ( seqpos2_indep != seqpos1 ) {
						atrenergy_trial    ( seqpos2_indep ) += 0.5 * WatrE;
						repenergy_trial    ( seqpos2_indep ) += 0.5 * WrepE;
						solenergy_trial    ( seqpos2_indep ) += 0.5 * WsolE;
						pair_energy_trial  ( seqpos2_indep ) += 0.5 * WpairE;
						hbenergy_trial     ( seqpos2_indep ) += 0.5 * WhbondE;
						gb_elecenergy_trial( seqpos2_indep ) += 0.5 * Wgb_elecE;
						plane_energy_trial ( seqpos2_indep ) += 0.5 * Wplane_totalE;
					}

					// trial_pair terms
					atr_trial_pair    ( seqpos2 ) = WatrE;
					rep_trial_pair    ( seqpos2 ) = WrepE;
					sol_trial_pair    ( seqpos2 ) = WsolE;
					pair_trial_pair   ( seqpos2 ) = WpairE;
					hbond_trial_pair  ( seqpos2 ) = WhbondE;
					gb_elec_trial_pair( seqpos2 ) = Wgb_elecE;
					plane_trial_plane ( seqpos2 ) = Wplane_totalE;

					//////////////////////////////
					//chu add in disulf pair terms
					// need to think more about this in the symmetric case -- PB
					if ( get_disulf_flag() && aa1 == aa_cys && aa2 == aa_cys &&
							 cys_pair_in_disulf(seqpos1,seqpos2) ) {
						assert( false );
						float const temp =
							0.5*getTotWtdScThisDisulf_pack(seqpos1,rotcoord,seqpos2,xyz_11s);
						pair_energy_trial(seqpos1) += temp;
						if ( seqpos2_indep != seqpos1 ) {
							pair_energy_trial( seqpos2_indep) += temp;
						}
						pair_trial_pair(seqpos2) += temp + temp;
					}
				} // if ( !symm_rottrial )...


				if ( optE ) {
					float temp;
					assert(!symm_rottrial);
					temp = 0.5 * pack_wts.Whbond_sc() * schbE;
					schbenergy_trial(seqpos1) += temp;
					schbenergy_trial(seqpos2) += temp;
					temp = 0.5 * pack_wts.Whbond_bb() * ( lrbbhbE + srbbhbE );
					bbhbenergy_trial(seqpos1) += temp;
					bbhbenergy_trial(seqpos2) += temp;
					temp = 0.5 * pack_wts.Whbond_bb_sc() * sc_bbhbE;
					sc_bbhbenergy_trial(seqpos1) += temp;
					sc_bbhbenergy_trial(seqpos2) += temp;
				}

			} else {
				atr_trial_pair(seqpos2) = 0.0;
				rep_trial_pair(seqpos2) = 0.0;
				sol_trial_pair(seqpos2) = 0.0;
				gb_elec_trial_pair(seqpos2) = 0.0;
				h2o_trial_pair(seqpos2) = 0.0;
				h2o_hb_trial_pair(seqpos2) = 0.0;
				pair_trial_pair(seqpos2) = 0.0;
				plane_trial_plane(seqpos2) = 0.0;
				hbond_trial_pair(seqpos2) = 0.0;
			}               // are they neighbors
		}                  // does seqpos1 = seqpos2

		if ( cst_exist ) {
			assert( !symm_rottrial ); // need to think about this
			//car add cst E;  may be intraresidue, may be long range
			//apl Fixing a bug: intraresidue constraints should use
			//apl rotcoord for the rotamer being considered
			float const & seqpos2_coord = ( seqpos1 == seqpos2 ) ? rotcoord(1,1) : full_coord(1,1,seqpos2); //apl fixing reference initialization bug
			cstE = classical_constraints::BOUNDARY::get_res_res_cstE(seqpos1,seqpos2,aa1,aav1,aa2,aav2,rotcoord, seqpos2_coord);
			float temp = 0.5*cstE;
			cstenergy_trial(seqpos1) = cstenergy_trial(seqpos1) + temp;
			cstenergy_trial(seqpos2) = cstenergy_trial(seqpos2) + temp;
			cst_trial_pair(seqpos2) = cstE;
			//no packer-specific weight, scoring function weight applied in
			//get_res_res_cstE
		}

	}

//bk add up new residue energies
	fullatomE_trial = 0.0;
	if ( !symm_rottrial ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			resenergy_trial(i) = atrenergy_trial(i) + repenergy_trial(i) +
				solenergy_trial(i) + dun_trial(i) + hbenergy_trial(i) +
				prob_trial(i) + pair_energy_trial(i) + plane_energy_trial(i) -
				unf_trial(i) + protonation_trial(i) + intrares_trial(i) +
				gb_elecenergy_trial(i) + cstenergy_trial(i) +
				h2oenergy_trial(i) + h2ohbenergy_trial(i);
			fullatomE_trial += resenergy_trial(i);
		}
	} else {
		// symmetric case
		for ( int i = 1; i <= total_residue; ++i ) {
			if ( clone_follows(i) ) {
				resenergy_trial(i) = 0.0;
			} else {
				resenergy_trial(i) = atrenergy_trial(i) + repenergy_trial(i) +
					solenergy_trial(i) + dun_trial(i) + hbenergy_trial(i) +
					prob_trial(i) + pair_energy_trial(i) + plane_energy_trial(i) -
					unf_trial(i) + protonation_trial(i) + intrares_trial(i) +
					gb_elecenergy_trial(i) + cstenergy_trial(i);
				fullatomE_trial += ( num_clones + 1 ) * resenergy_trial(i);
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin           reset_energy_coord_from_trial(rotcoord,rotchi,rotid,seqpos,rotactcoord,rot_aav,rot_aa)
///
/// @brief
///bk call when a rotamer trial has been accepted
///
/// @detailed
///
/// @param[in]        rot_aav                   int
/// @param[in]        rotactcoord               float
/// @param[in]        rotchi                    float
/// @param[in]        rotcoord                  float
/// @param[in]        rotid                     int
/// @param[in]        seqpos                    int
/// @param[in]        rot_aa                    int   amino acid #
///
/// @global_read     atr_trial_pair            float       rotamer_trial_energies.h trial_energies
/// @global_read     atrenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     dun_trial                 float       rotamer_trial_energies.h trial_energies
/// @global_read     intrares_trial            float       rotamer_trial_energies.h trial_energies
/// @global_read     hbenergy_trial            float       rotamer_trial_energies.h trial_energies
/// @global_read     hbond_trial_pair          float       rotamer_trial_energies.h trial_energies
/// @global_read     MAX_ATOM                  int    param.h
/// @global_read     MAX_CHI                   int    param.h
/// @global_read     total_residue                      int    misc.h template
/// @global_read     pair_energy_trial         float       rotamer_trial_energies.h trial_energies
/// @global_read     pair_trial_pair           float       rotamer_trial_energies.h trial_energies
/// @global_read     prob_trial                float       rotamer_trial_energies.h trial_energies
/// @global_read     rep_trial_pair            float       rotamer_trial_energies.h trial_energies
/// @global_read     repenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     resenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     sol_trial_pair            float       rotamer_trial_energies.h trial_energies
/// @global_read     solenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     unf_trial                 float       rotamer_trial_energies.h trial_energies
///
/// @global_write    res_variant                int    misc.h template
/// @global_write    actcoord                  float       template_pack.h template
/// @global_write    atr_pair                  float       fullatom_energies.h energies
/// @global_write    atrenergy                 float       fullatom_energies.h energies
/// @global_write    chiarray                  float       template_pack.h template
/// @global_write    dunenergy                 float       fullatom_energies.h energies
/// @global_write    intrareseenergy           float       fullatom_energies.h energies
/// @global_write    fullatome                 float       fullatom_energies.h totals
/// @global_write    hbenergy                  float       fullatom_energies.h energies
/// @global_write    hbond_pair                float       fullatom_energies.h energies
/// @global_write    pair_energy               float       fullatom_energies.h energies
/// @global_write    pair_pair                 float       fullatom_energies.h energies
/// @global_write    probenergy                float       fullatom_energies.h energies
/// @global_write    rep_pair                  float       fullatom_energies.h energies
/// @global_write    repenergy                 float       fullatom_energies.h energies
/// @global_write    resenergy                 float       fullatom_energies.h energies
/// @global_write    rotarray                  int    template_pack.h template
/// @global_write    sol_pair                  float       fullatom_energies.h energies
/// @global_write    solenergy                 float       fullatom_energies.h energies
/// @global_write    unfenergy                 float       fullatom_energies.h energies
/// @global_write    full_coord                       float       misc.h template
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_energy_coord_from_trial(
	int total_residue,
	FArray1D_int & res,
	FArray1D_int & res_variant,
	FArray3D_float & full_coord,
	FArray2Da_float rotcoord,
	FArray1Da_float rotchi,
	FArray1Da_int rotid,
	int seqpos,
	FArray1Da_float rotactcoord,
	FArray1Da_float rot_born_radius,
	int rot_aav,
	int rot_aa,
	bool const symm_rottrial // = false
)
{

	using namespace aaproperties_pack;
	using namespace fullatom_energies;
	using namespace param;
	using namespace rotamer_trial_energies;
	using namespace template_pack;
	using namespace design_sym; // for symmetric rottrialing

	rotcoord.dimension( 3, MAX_ATOM() );
	rotchi.dimension( 4 );
	rotid.dimension( 4 );
	rotactcoord.dimension( 3 );

//bk reset amino acid
	res(seqpos) = rot_aa;

//bk reset amino acid variant
	res_variant(seqpos) = rot_aav;

//bk reset coordinates
	for ( int j = 1, je = natoms(rot_aa,rot_aav); j <= je; ++j ) {
		for ( int i = 1; i <= 3; ++i ) {
			full_coord(i,j,seqpos) = rotcoord(i,j);
		}
	}
	for ( int i = 1; i <= 3; ++i ) {
		actcoord(i,seqpos) = rotactcoord(i);
	}
	for ( int i = 1, ie = nchi(rot_aa,rot_aav); i <= ie; ++i ) {
		chiarray(i,seqpos) = rotchi(i);
		rotarray(i,seqpos) = rotid(i);
		if ( symm_rottrial ) {
			for ( int k=1; k<= num_clones; ++k ) {
				int const clone_pos( clone_list(seqpos,k));
				chiarray(i,clone_pos) = rotchi(i);
				rotarray(i,clone_pos) = rotid(i);
			}
		}
	}

	if ( param_pack::gen_born ) {
		for ( int j = 1, je = natoms(rot_aa,rot_aav); j <= je; ++j ) {
			born_radius(j,seqpos) = rot_born_radius(j);
		}
	}

//bk reset energies
	fullatomE = 0.0;
	for ( int i = 1; i <= total_residue; ++i ) {
		atrenergy(i) = atrenergy_trial(i);
		repenergy(i) = repenergy_trial(i);
		solenergy(i) = solenergy_trial(i);
		gb_elecenergy(i) = gb_elecenergy_trial(i);
		h2oenergy  (i) =   h2oenergy_trial(i);
		h2ohbenergy(i) = h2ohbenergy_trial(i);
		assert( water_exists() || std::abs(h2oenergy(i)) +
						std::abs(h2ohbenergy(i)) < 1e-2 );
		resenergy(i) = resenergy_trial(i);
		pair_energy(i) = pair_energy_trial(i);
		plane_energy(i) = plane_energy_trial(i);
		hbenergy(i) = hbenergy_trial(i);
		unfenergy(i) = unf_trial(i);
		protonation_energy(i) = protonation_trial(i);
		dunenergy(i) = dun_trial(i);
		probenergy(i) = prob_trial(i);
		intraresenergy(i) = intrares_trial(i);
		cstenergy(i) = cstenergy_trial(i);
		h2oenergy(i) = h2olj_trial(i);
		h2ohbenergy(i) = h2ohb_trial(i);

		if ( rottrial_verbose ) {
			std::cout << "rottrial-accept: " << i << ' ' << resenergy(i) << ' ' <<
				atrenergy(i) << ' ' <<
				repenergy(i) << ' ' <<
				solenergy(i) << ' ' <<
				gb_elecenergy(i) << ' ' <<
				h2oenergy(i) << ' ' <<
				h2ohbenergy(i) << ' ' <<
				resenergy(i) << ' ' <<
				pair_energy(i) << ' ' <<
				plane_energy(i) << ' ' <<
				hbenergy(i) << ' ' <<
				unfenergy(i) << ' ' <<
				protonation_energy(i) << ' ' <<
				dunenergy(i) << ' ' <<
				probenergy(i) << ' ' <<
				intraresenergy(i) << ' ' <<
				cstenergy(i) << std::endl;
		}

//car save new total in global variable for future comparisons
		if ( !symm_rottrial ) {
			fullatomE += resenergy(i);
		} else {
			// symmetric
			if ( !clone_follows(i) ) fullatomE += (num_clones+1) * resenergy(i);
		}
		atr_pair(seqpos,i) = atr_trial_pair(i);
		atr_pair(i,seqpos) = atr_trial_pair(i);
		rep_pair(seqpos,i) = rep_trial_pair(i);
		rep_pair(i,seqpos) = rep_trial_pair(i);
		sol_pair(seqpos,i) = sol_trial_pair(i);
		sol_pair(i,seqpos) = sol_trial_pair(i);
		gb_elec_pair(seqpos,i) = gb_elec_trial_pair(i);
		gb_elec_pair(i,seqpos) = gb_elec_trial_pair(i);
		h2o_pair   (seqpos,i) = h2o_trial_pair   (i);
		h2o_pair   (i,seqpos) = h2o_trial_pair   (i);
		h2o_hb_pair(seqpos,i) = h2o_hb_trial_pair(i);
		h2o_hb_pair(i,seqpos) = h2o_hb_trial_pair(i);
		assert( water_exists() || std::abs( h2o_trial_pair(i) ) +
						std::abs( h2o_hb_trial_pair(i) ) < 1e-2 );
		pair_pair(seqpos,i) = pair_trial_pair(i);
		pair_pair(i,seqpos) = pair_trial_pair(i);
		plane_plane(seqpos,i) = plane_trial_plane(i);
		plane_plane(i,seqpos) = plane_trial_plane(i);
		hbond_pair(seqpos,i) = hbond_trial_pair(i);
		hbond_pair(i,seqpos) = hbond_trial_pair(i);
		cst_pair(seqpos,i) = cst_trial_pair(i);
		cst_pair(i,seqpos) = cst_trial_pair(i);

		if ( symm_rottrial ) {
			// symmetric rottrials
			//
			// have to update pair energies involving seqpos' clone positions
			// these will not be exactly correct, which is OK inside rottrials
			// since we sum over clones anyhow. But when done rottrialing we
			// should probably force a complete rescore

			int const i_base( clone_follows(i) ? clone_follows(i) : i );
			for ( int k=1, kk=0; k<= num_clones; ++k, ++kk ) {
				int const seqpos_clone( clone_list(seqpos,k) );
				int const i_clone( kk ? clone_list(i_base,kk) : i_base );
				if ( i == i_clone ) { // just increment kk on this go-round
					--k;
					continue;
				}
				assert( k<num_clones || kk == num_clones ||
								clone_list(i_base,num_clones) == i ); // sanity
				atr_pair    (seqpos_clone,i) = atr_pair    (seqpos,i_clone);
   			rep_pair    (seqpos_clone,i) = rep_pair    (seqpos,i_clone);
    		sol_pair    (seqpos_clone,i) = sol_pair    (seqpos,i_clone);
    		cst_pair    (seqpos_clone,i) = cst_pair    (seqpos,i_clone);
    		pair_pair   (seqpos_clone,i) = pair_pair   (seqpos,i_clone);
    		hbond_pair  (seqpos_clone,i) = hbond_pair  (seqpos,i_clone);
    		plane_plane (seqpos_clone,i) = plane_plane (seqpos,i_clone);
    		gb_elec_pair(seqpos_clone,i) = gb_elec_pair(seqpos,i_clone);
				atr_pair    (i,seqpos_clone) = atr_pair    (seqpos,i_clone);
   			rep_pair    (i,seqpos_clone) = rep_pair    (seqpos,i_clone);
    		sol_pair    (i,seqpos_clone) = sol_pair    (seqpos,i_clone);
    		cst_pair    (i,seqpos_clone) = cst_pair    (seqpos,i_clone);
    		pair_pair   (i,seqpos_clone) = pair_pair   (seqpos,i_clone);
    		hbond_pair  (i,seqpos_clone) = hbond_pair  (seqpos,i_clone);
    		plane_plane (i,seqpos_clone) = plane_plane (seqpos,i_clone);
    		gb_elec_pair(i,seqpos_clone) = gb_elec_pair(seqpos,i_clone);
			} // num_clones
		} // symm_rottrial
	} // i=1,total_residue
}


////////////////////////////////////////////////////////////////////////////////
/// @begin           random_order(list,npos)
///
/// @brief
///ctsa  in-place random shuffle on an integer array
///
/// @detailed
///
/// @param[in]        list                      int
/// @param[in]        npos                      int
///
/// @param[out]       list                      int
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
random_order(
	FArray1Da_int list,
	int const npos
)
{

	if ( npos <= 1 ) return;

	list.dimension( star );

	for ( int i = npos; i >= 2; --i ) {
		int const r = static_cast< int >( ran3() * i ) + 1;
		int const iswap = list(r);
		list(r) = list(i);
		list(i) = iswap;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           write_rot_energies(seqpos,aa,rot,aanat,nb)
///
/// @brief
///
/// @detailed
///
/// @param[in]        aa                        int
/// @param[in]        aanat                     int
/// @param[in]        nb                        int
/// @param[in]        rot                       int
/// @param[in]        seqpos                    int
///
/// @global_read     atrenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     bbhbenergy_trial          float       rotamer_trial_energies.h trial_energies
/// @global_read     dun_trial                 float       rotamer_trial_energies.h trial_energies
/// @global_read     intrares_trial            float       rotamer_trial_energies.h trial_energies
/// @global_read     opte_x                    int         files_paths.h
/// @global_read     pair_energy_trial         float       rotamer_trial_energies.h trial_energies
/// @global_read     prob_trial                float       rotamer_trial_energies.h trial_energies
/// @global_read     protein_chain             char        misc.h     chars
/// @global_read     repenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     sc_bbhbenergy_trial       float       rotamer_trial_energies.h trial_energies
/// @global_read     schbenergy_trial          float       rotamer_trial_energies.h trial_energies
/// @global_read     solenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     start_file                std::string files_paths.h file_path_char
/// @global_read     unf_trial                 float       rotamer_trial_energies.h trial_energies
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
write_rot_energies(
	int const seqpos,
	int const aa,
	int const rot,
	int const aanat,
	int const nb
)
{
	using namespace files_paths;
	using namespace rotamer_trial_energies;

	opte_x << start_file.substr(0,4) << protein_chain << ' ' <<
		I(4,seqpos) <<' '<< I(2,aa) <<' '<< I(3,rot) <<' '<< I(2,aanat) <<' '<<
		I( 1, 0 ) << ' ' << I( 2, nb ) << ' ' << I( 2, 0 ) << ' ' <<
		E( 10, 4, 2*atrenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*repenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*pair_energy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*solenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*plane_energy_trial(seqpos) ) << ' ' <<
		E( 10, 4, dun_trial(seqpos) ) << ' ' <<
		E( 10, 4, intrares_trial(seqpos) ) << ' ' <<
		E( 10, 4, unf_trial(seqpos) ) << ' ' <<
		E( 10, 4, protonation_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*(sc_bbhbenergy_trial(seqpos)+bbhbenergy_trial(seqpos)) ) << ' ' <<
		E( 10, 4, 2*schbenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*gb_elecenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, prob_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*cstenergy_trial(seqpos) ) << std::endl;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin           minimize_trial_energies(chi,rotid,seqpos,aa,aav,rotcoord,rotactcoord,func_tol,min_type)
///
/// @brief
///chu  main function for side chain minimization in rotamer trial
///
/// @detailed
///
/// @param[in]        aa                        int
/// @param[in]        aav                       int
/// @param[in]        chi                       float
/// @param[in]        func_tol                  float
/// @param[in]        min_type                  std::string
/// @param[in]        rotactcoord               float
/// @param[in]        rotcoord                  float
/// @param[in]        rotid                     int
/// @param[in]        seqpos                    int
///
/// @param[out]       aa                        int
/// @param[out]       aav                       int
/// @param[out]       chi                       float
/// @param[out]       rotactcoord               float
/// @param[out]       rotcoord                  float
/// @param[out]       rotid                     int
/// @param[out]       seqpos                    int
///
/// @global_read     fullatome_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     inform                    int    runlevel.h
/// @global_read     MAX_ATOM                  int    param.h
/// @global_read     MAX_CHI                   int    param.h
/// @global_read     nchi                      int    aaproperties_pack.h properties_1
/// @global_read     total_residue             int    misc.h template
/// @global_read     phi                       float       misc.h template
/// @global_read     psi                       float       misc.h template
/// @global_read     runlevel                  int    runlevel.h run_level_status
/// @global_read     sc_rotactcoord            float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotcoord               float       rotamer_functions.h rotamer_minimization
///
/// @global_write    sc_aa                     int    rotamer_functions.h rotamer_minimization
/// @global_write    sc_aav                    int    rotamer_functions.h rotamer_minimization
/// @global_write    sc_chi                    float       rotamer_functions.h rotamer_minimization
/// @global_write    sc_nfree                  int    rotamer_functions.h rotamer_minimization
/// @global_write    sc_rotid                  int    rotamer_functions.h rotamer_minimization
/// @global_write    sc_seqpos                 int    rotamer_functions.h rotamer_minimization
/// @global_write    start_chi                 float       rotamer_functions.h rotamer_minimization
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
minimize_trial_energies(
	FArray1Da_float chi,
	FArray1Da_int rotid,
	int const seqpos,
	int const aa,
	int const aav,
	FArray2Da_float rotcoord,
	FArray1Da_float rot_born_radius,
	FArray1Da_float rotactcoord,
	float func_tol,
	std::string const & min_type,
	int total_residue,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray3D_float const & full_coord
	)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace rotamer_minimization;
	using namespace rotamer_trial_energies;
	using namespace runlevel_ns;

	chi.dimension( MAX_CHI );
	rotid.dimension( MAX_CHI );
	rotcoord.dimension( 3, MAX_ATOM() );
	rot_born_radius.dimension( MAX_ATOM() );
	rotactcoord.dimension( 3 );

//chu   local
	FArray1D_float min_chi( MAX_CHI );
	FArray1D_float dE_dchi( MAX_CHI );
	FArray1D_float dp_dchi( MAX_CHI );
	float start_fullatomE_trial, func_return_val, prob, dp_dphi, dp_dpsi;

	for ( int i = 1; i <= nchi(aa,aav); ++i ) {
		if ( ( chi(i) != chi(i) ) || ( std::abs(chi(i) ) > 10000. ) ) {
#ifndef BOINC
			std::cout << "ABORT BAD STARTING CHI " << chi(i) << " IN MINIMIZE_TRIAL_ENERGIES" << std::endl;
			std::cout << "SEQPOS " << seqpos << " AA " << aa << " AAV " << aav << " CHI " << i << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
#endif
		}
	}

	// jk turn off soft_rep for the minimization step, if it was on
	disable_packing_etables(total_residue, res, res_variant, full_coord, true);

//chu   num of chi angles to be minimized;
	sc_nfree = nchi(aa,aav);
//      if ( aa == aa_ser || aa == aa_thr || aa == aa_tyr ) --sc_nfree; // no rotatable H
	if ( sc_nfree == 0 ) return;
//chu   save side chain information
	sc_seqpos = seqpos;
	sc_aa = aa;
	sc_aav = aav;
	start_chi = chi;
	sc_chi = chi;
	sc_rotid = rotid;
	sc_rot_born_radius = rot_born_radius;

//chu   minimize
	minimize_set_func(5);
	bool gfrag = true;
	int iter = 0;

	pack_min_chi(min_chi);
	start_fullatomE_trial = fullatomE_trial;
	if ( min_type == "linmin" ) {
		dfunc(min_chi,dE_dchi,sc_nfree);
		linmin(min_chi,dE_dchi,sc_nfree,func_return_val,gfrag);
	} else if ( min_type == "frpmin" ) {
		frprmn(min_chi,sc_nfree,func_tol,iter,func_return_val,gfrag);
	} else if ( min_type == "dfpmin" ) { //chu: use absolute tol
		dfpmin_atol(min_chi,sc_nfree,func_tol,iter,func_return_val,gfrag);
	} else {
		std::cout << "min_type not defined in rotamer trial" << std::endl;
		std::cout << "min_type: " << min_type << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( gfrag ) {

		for ( int i = 1; i <= aaproperties_pack::nchi(sc_aa,sc_aav); ++i ) {
			if ( ( min_chi(i) != min_chi(i) ) || ( std::abs(min_chi(i) ) > 10000. ) ) {
				if ( runlevel > standard ) {
#ifndef BOINC
					std::cout << "WARNING: FOUND BAD MIN_CHI " << min_chi(i) << " IN MIN_TRIAL_ENERGIES" << std::endl;
					std::cout << "WARNING: RESTORING START_CHI " << start_chi(i) << std::endl;
#endif
				}
				min_chi(i) = start_chi(i);
			}
		}

		unpack_min_chi(min_chi);
		if ( runlevel > chat ) {
			std::cout << "seqpos" << SS( sc_seqpos ) << "res" << SS( sc_aa ) << std::endl;
			std::cout << "start_chi:";
			for ( int j = 1; j <= MAX_CHI; ++j ) {
				std::cout << SS( chi(j) );
			} std::cout << std::endl;
			std::cout << "end_chi:";
			for ( int j = 1; j <= MAX_CHI; ++j ) {
				std::cout << SS( sc_chi(j) );
			} std::cout << std::endl;
			std::cout << "start_energy:" << SS( start_fullatomE_trial ) << std::endl;
			std::cout << "end_energy:" << SS( func_return_val ) << std::endl;
		}
		if ( (start_fullatomE_trial-func_return_val) < -1.0 ) {
			if ( runlevel > chat ) {
				std::cout << "WARNING!!! " << std::endl;
				std::cout << "Energy increases in rotamer_trial minimization !!!" << std::endl;
			}
		}
	} else {
		std::cout << "gfrag is false!!!" << std::endl;
		std::cout << "seq_pos:" << SS( sc_seqpos ) << "reside:" << SS( sc_aa ) <<
		 std::endl;
		std::cout << "chi angles:";
		for ( int j = 1; j <= MAX_CHI; ++j ) {
			std::cout << SS( sc_chi(j) );
		} std::cout << std::endl;
		std::cout << "func:" << SS( func_return_val ) << std::endl;
		std::cout << "rotamer minimization failed!!!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
//chu   return the minimized rotamer
	assert( seqpos == sc_seqpos );
	assert( aa == sc_aa );
	assert( aav == sc_aav );
	chi = sc_chi;
	rotid = sc_rotid;
	rotcoord = sc_rotcoord;
	rot_born_radius = sc_rot_born_radius;
	rotactcoord = sc_rotactcoord;

	// jk turn on soft rep and change weights, if desired
	enable_packing_etables(total_residue,res,res_variant,full_coord,true);

//chu   make sure fullatomE_trial has the minimized energy
	get_rotamer_probability(aa,aav,template_pack::phi(seqpos),
   template_pack::psi(seqpos),seqpos,total_residue,chi,rotid,
	 prob,dp_dphi,dp_dpsi,dp_dchi);

	get_trial_energies(prob,rotcoord,rot_born_radius,seqpos,aa,aav,rotactcoord,total_residue,res,res_variant, full_coord);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           float func_scvdw(min_chi,gfrag)
///
/// @brief
///chu  function for side chain minimization in rotamer trials
///
/// @detailed
///
/// @param[in]        gfrag                     bool
/// @param[in]        min_chi                   float
///
/// @return S         fullatome_trial
///
/// @global_read     fullatome_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     total_residue                      int    template_pack.h template
/// @global_read     phi                       float       template_pack.h template
/// @global_read     psi                       float       template_pack.h template
/// @global_read     sc_aa                     int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_aav                    int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_chi                    float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotactcoord            float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotcoord               float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotid                  int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_seqpos                 int    rotamer_functions.h rotamer_minimization
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
func_scvdw(
	FArray1Da_float min_chi,
	bool & gfrag
)
{

	if ( !gfrag ) return 0.0;

	using namespace misc;
	using namespace param;
	using namespace rotamer_minimization;
	using namespace rotamer_trial_energies;
	using namespace runlevel_ns;

	min_chi.dimension( star );

	float func_scvdw = 0.0; // Return value

	FArray1D_float dp_dchi( MAX_CHI );
	float prob, dp_dphi, dp_dpsi;

	for ( int i = 1; i <= aaproperties_pack::nchi(sc_aa,sc_aav); ++i ) {
		if ( ( min_chi(i) != min_chi(i) ) || ( std::abs(min_chi(i) ) > 10000. ) ) {
#ifndef BOINC
			if ( runlevel > standard ) {
				std::cout << "WARNING: FOUND BAD MIN_CHI " << min_chi(i) << " IN FUNC_SCVDW" << std::endl;
				std::cout << "WARNING: RESTORING START_CHI " << start_chi(i) << std::endl;
			}
#endif
			min_chi(i) = start_chi(i);
		}
	}

	unpack_min_chi(min_chi);
	get_rotamer_probability(sc_aa,sc_aav,template_pack::phi(sc_seqpos),
   template_pack::psi(sc_seqpos),sc_seqpos,total_residue,sc_chi,
   sc_rotid,prob,dp_dphi,dp_dpsi,dp_dchi);

//jjh temporarily off
	get_trial_energies(prob,sc_rotcoord,sc_rot_born_radius,sc_seqpos,sc_aa,sc_aav,sc_rotactcoord,total_residue,res,res_variant, full_coord);

//chu   rotamer minimization uses the fullatom-energy as the function value,
//     but not the scorefxn as in func_vdw(full protein minimization) and
//     func_dvdw(docking).
	func_scvdw = fullatomE_trial;
//      std::cout << "FUNC VALUE:" << SS( func_scvdw ) << "min_chi" << SS( min_chi(1) ) << std::endl;

	return func_scvdw;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin           dfunc_scvdw(min_chi,de_dchi,nfree)
///
/// @brief
///chu   this function evaluates the derivatives of the van der Waals
//     energy and hbond energy relative to the side chain dihedral angles.
///
/// @detailed
///
///
///     After reading this paper, looking at DB's comments and Rosetta code,
///     I believe there are some TYPOS in DB's comments in minimize.cc. Here is
///     the corrected version:
///
///     dE/dchi(or dE/dphi) = dE/dr * dr/dchi //// By the chain rule
///
///     dE/dr can be read from the precalculated table and accumulated.
///
///     dr/dchi = Eab x (V-Vb).(V-V')/|V'-V|
///
///     Eab is the unit vector for the rotatable bond
///         pointing to the moving side, e.g. for chi1, CA-->CB.
///     Vb is the end atom of the rotatable bond, e.g., CB for chi1.
///     V is the atom whose position is depedent on this chi.
///     V' is the atom who is interacting with V, but not dependent on this chi.
///     Obviously, |V'-V| = r.
///
///     So Eab x (V-Vb) is the displacement of V upon a rotation dchi around the
///     unit vector Eab. Then the dot product tells what is the effective
///     displacement along the direction (V-V').
///
///
///     dE/dchi = 1/r * dE/dr * [ Eab x (V-Vb) . (V-V')]
///             = (1/r * dE/dr) * [ (Eab x V - Eab x Vb) . (V-V') ]
///             = (1/r * dE/dr) * [ -Eab x V . V' - Eab x Vb . (V-V') ]
///             = (1/r * dE/dr) * [ -Eab . (V x V') - Eab x Vb . (V-V') ]
///             = -Eab . [ 1/r * dE/dr * (V x V') ]
///                      -Eab x Vb . [ 1/r * dE/dr * (V-V') ]
///
///     The above formula contains the summation over all the V on one side
///     of the rotatable angle and all the V' on the other side.
///
///db   note that Eab and Vb are different for each torsion angle, but V'
///db   and V are the same. So a recursive method is used to speed up
///     the calculation. In this algorithm, V is the set of atoms whose
///     postions are dependent on only one rotatable angle, e.g. rigid-body
///     region between two angles, and V' is the complementary set.
///
///     In the code below,
///     F1: accumulated  [1/r * dE/dr * (V x V')]
///     F2: accumulated  [1/r * dE/dr * (V-V') ]
///     Eab: unit ;  Vb: end_atom
///     V is defined in chi_rigid_atom_list
///
/// @param[in]        de_dchi                   float
/// @param[in]        min_chi                   float
/// @param[in]        nfree                     int
///
/// @param[out]       de_dchi                   float
///
/// @global_read     runlevel                  int    runlevel.h run_level_status
/// @global_read     sc_aa                     int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotcoord               float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_seqpos                 int    rotamer_functions.h rotamer_minimization
/// @global_read     verbose                   int    runlevel.h
///
/// @remarks
///
/// @references
///chu   Abe, Braun, Noguti and Go,  Comput.Chem.(1984)8,239-247
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dfunc_scvdw(
	FArray1Da_float min_chi,
	FArray1Da_float dE_dchi,
	int nfree
)
{
	using namespace param;
	using namespace rotamer_minimization;
	using namespace runlevel_ns;
	using numeric::conversions::radians;

	min_chi.dimension( nfree );
	dE_dchi.dimension( nfree );

	// local
	FArray2D_float f1_hbond_deriv( 3, MAX_CHI );
	FArray2D_float f2_hbond_deriv( 3, MAX_CHI );
	FArray1D_float f1_LJ_solv_deriv( 3 );
	FArray1D_float f2_LJ_solv_deriv( 3 );
	FArray1D_float f1_all_deriv( 3, 0.0 );
	FArray1D_float f2_all_deriv( 3, 0.0 );
	FArray1D_float unit( 3 );
	FArray1D_float endfull_coord( 3 );
	FArray1D_float product( 3 );
	// variables for checking numerical derivatives
	FArray1D_float min_chi_pert( nfree );
	FArray1D_float f_deriv( nfree );
	FArray1D_float atr_deriv( nfree );
	FArray1D_float rep_deriv( nfree );
	FArray1D_float solv_deriv( nfree );
	FArray1D_float hbond_deriv( nfree );
	FArray1D_float sub_deriv( nfree );
	float f_plus,f_minus,atr_plus,atr_minus,rep_plus,rep_minus,solv_plus,
	 solv_minus,hbond_plus,hbond_minus,epsilon;
	bool gfrag( true );

	for ( int i = 1; i <= aaproperties_pack::nchi(sc_aa,sc_aav); ++i ) {
		if ( ( min_chi(i) != min_chi(i) ) || ( std::abs(min_chi(i) ) > 10000. ) ) {
#ifndef BOINC
			if ( runlevel > standard ) {
				std::cout << "WARNING: FOUND BAD MIN_CHI " << min_chi(i) << " IN DFUNC_SCVDW" << std::endl;
				std::cout << "WARNING: RESTORING START_CHI " << start_chi(i) << std::endl;
			}
#endif
			min_chi(i) = start_chi(i);
		}
	}

	unpack_min_chi(min_chi);

	// precalculate hbond derivatives
	get_hbond_deriv(f1_hbond_deriv,f2_hbond_deriv);

	for ( int ch = nfree; ch >= 1; --ch ) {
		dE_dchi(ch) = 0.0;
		int end_atom;
		make_chi_unit_vector(ch,unit,end_atom,sc_rotcoord);
		get_atom_from_sc_rotcoord(end_atom,endfull_coord);
		get_LJ_solv_deriv(ch,f1_LJ_solv_deriv,f2_LJ_solv_deriv);
		for ( int j = 1; j <= 3; ++j ) { // pairwise deriv
			f1_all_deriv(j) += f1_LJ_solv_deriv(j);
			f2_all_deriv(j) += f2_LJ_solv_deriv(j);
		}
		for ( int j = 1; j <= 3; ++j ) { // hbond deriv
			f1_all_deriv(j) += f1_hbond_deriv(j,ch);
			f2_all_deriv(j) += f2_hbond_deriv(j,ch);
		}
		cros(unit,endfull_coord,product);
		dE_dchi(ch) = -radians( dotprod(unit,f1_all_deriv) + dotprod(product,f2_all_deriv) );
	} // each chi minimized

	// check derivatives numerically
	if ( runlevel > verbose ) {
		for ( int i = 1; i <= nfree; ++i ) {
			min_chi_pert(i) = min_chi(i);
		}

		for ( int i = 1; i <= nfree; ++i ) {
			gfrag = true;
			epsilon = 0.01;
			min_chi_pert(i) = min_chi(i) - epsilon;
			angle_in_range(min_chi_pert(i));
			f_minus = func(gfrag,min_chi_pert);
			sum_trial_energy(atr_minus,rep_minus,solv_minus,hbond_minus);
			min_chi_pert(i) = min_chi(i) + epsilon;
			angle_in_range(min_chi_pert(i));
			f_plus = func(gfrag,min_chi_pert);
			sum_trial_energy(atr_plus,rep_plus,solv_plus,hbond_plus);
			f_deriv(i) = 0.5 * (f_plus - f_minus) / epsilon;
			atr_deriv(i) = 0.5 * (atr_plus - atr_minus) / epsilon;
			rep_deriv(i) = 0.5 * (rep_plus - rep_minus) /epsilon;
			solv_deriv(i) = 0.5 * (solv_plus - solv_minus) / epsilon;
			hbond_deriv(i) = 0.5 * (hbond_plus - hbond_minus) /epsilon;
			sub_deriv(i) = 0.0 + atr_deriv(i) + rep_deriv(i) + solv_deriv(i) +
			 hbond_deriv(i);

			min_chi_pert(i) = min_chi(i);
		}

		std::cout << "seqpos: " << sc_seqpos << " aa: " << sc_aa << std::endl;

		std::cout << "  min_chi:";
		for ( int i = 1; i <= nfree; ++i ) {
			std::cout << F( 12, 4, min_chi(i) );
		} std::cout << std::endl;

		std::cout << "  dE_dchi:";
		for ( int i = 1; i <= nfree; ++i ) {
			std::cout << F( 12, 4, dE_dchi(i) );
		} std::cout << std::endl;

//		std::cout << "  f_deriv:";
//		for ( int i = 1; i <= nfree; ++i ) {
//			std::cout << F( 12, 4, f_deriv(i) );
//		} std::cout << std::endl;
//
//		std::cout << "  f_ratio:";
//		for ( int i = 1; i <= nfree; ++i ) {
//			std::cout << F( 12, 4,
//			 f_deriv(i) / sign(std::max(std::abs(dE_dchi(i)),0.0001),dE_dchi(i)) );
//		} std::cout << std::endl;
//
//		std::cout << "sub_deriv:";
//		for ( int i = 1; i <= nfree; ++i ) {
//			std::cout << F( 12, 4, sub_deriv(i) );
//		} std::cout << std::endl;
//
//		std::cout << "sub_ratio:";
//		for ( int i = 1; i <= nfree; ++i ) {
//			std::cout << F( 12, 4,
//			 sub_deriv(i)/sign(std::max(std::abs(dE_dchi(i)),0.0001),dE_dchi(i)) );
//		} std::cout << std::endl;

		for ( int i = 1; i <= nfree; ++i ) {
			std::cout << "Numerical/Analytical:" <<
			 F( 12, 4, sub_deriv(i) ) << F( 12, 4, dE_dchi(i) ) << std::endl;
		} std::cout << std::endl;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           get_hbond_deriv(f1_hbond_deriv,f2_hbond_deriv)
///
/// @brief
///chu   calculate the hbond derivative for side-chain minimization in
///     rotamer_trial.
///
/// @detailed
///
/// @param[in]        f1_hbond_deriv            float
/// @param[in]        f2_hbond_deriv            float
///
/// @param[out]       f1_hbond_deriv            float
/// @param[out]       f2_hbond_deriv            float
///
/// @global_read     res_variant                int    template_pack.h template
/// @global_read     res                       int    template_pack.h template
/// @global_read     chi_rigid_atom_list       int    aaproperties_pack.h properties_1
/// @global_read     length_chi_rigid_atom_list int    aaproperties_pack.h properties_1
/// @global_read     lrbb_hbtype               int    hbonds.h
/// @global_read     MAX_ATOM                  int    param.h
/// @global_read     MAX_CHI                   int    param.h
/// @global_read     neighborlist              bool    template_pack.h template
/// @global_read     neighbors                 int    template_pack.h template
/// @global_read     total_residue                      int    template_pack.h template
/// @global_read     sc_aa                     int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_aav                    int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_hbtype                 int    hbonds.h
/// @global_read     sc_rotcoord               float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_seqpos                 int    rotamer_functions.h rotamer_minimization
/// @global_read     scbb_hbtype               int    hbonds.h
/// @global_read     srbb_hbtype               int    hbonds.h
/// @global_read     pack_wts.Whbond_bb()                 float       param_pack.h packer_weight_parameters
/// @global_read     pack_wts.Whbond_bb_sc()              float       param_pack.h packer_weight_parameters
/// @global_read     pack_wts.Whbond_sc()                 float       param_pack.h packer_weight_parameters
/// @global_read     full_coord                       float       template_pack.h template
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_hbond_deriv(
	FArray2DB_float & f1_hbond_deriv,
	FArray2DB_float & f2_hbond_deriv
)
{
	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace param;
	using namespace param_pack;
	using namespace rotamer_minimization;
	using namespace misc;
	using namespace template_pack;

//     arguments for get_hbE
	int aa,aav;
	FArray1D_float hbenergies_l( MAX_HB_PER_ATM );
	float schbE, srbbhbE, lrbbhbE, sc_bbhbE;
	FArray1D_int hbdonh_atm_l( MAX_HB_PER_ATM );
	FArray1D_int hbdon_res_l( MAX_HB_PER_ATM );
	FArray1D_int hbact_atm_l( MAX_HB_PER_ATM );
	FArray1D_int hbact_res_l( MAX_HB_PER_ATM );
	FArray1D_int hbtype_l( MAX_HB_PER_ATM );
	int nhbonds_l;
	FArray1D_bool hbchk1_l( 2 ); // unused
	FArray1D_bool hbchk2_l( 2 );
	FArray3D_float hbderiv_l( 3, 2, MAX_HB_PER_ATM );
//     local
	int dres,dhatm,ares,aatm,atom;
	float weight;
	bool d_is_bb, a_is_bb;
	FArray2D_float f1( 3, MAX_ATOM()(), 0.0 );
	FArray2D_float f2( 3, MAX_ATOM()(), 0.0 );
	FArray2D_float f1_ligand( 3, MAX_ATOM()() );
	FArray2D_float f2_ligand( 3, MAX_ATOM()() );
	float hbondE;

	bool const fullatom = true;
  HBDerivType const deriv_type = hbonds::hbderiv_ABE_GO;
	f1_hbond_deriv = 0.0;
	f2_hbond_deriv = 0.0;

//==============================================================================
//mj ligand

	if ( get_ligand_flag() ) {
		get_sc_ha_hbondE(sc_seqpos,sc_aa,sc_aav,sc_rotcoord,false,true,(*ligand::ligand_one),
                     hbondE, deriv_type, f1_ligand,f2_ligand);

		//// add ligand contribution to the accumulator
		for ( int l = 0, le = 3 * MAX_ATOM()(); l < le; ++l ) {
			f1[ l ] += f1_ligand[ l ]; // f1(j,i) += f1_ligand(j,i);
			f2[ l ] += f2_ligand[ l ]; // f2(j,i) += f2_ligand(j,i);
		}
	}

//==============================================================================

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( seqpos == sc_seqpos ) goto L100; // same res
		if ( !neighborlist(seqpos,sc_seqpos) ) goto L100; // not neighbor
		aa = res(seqpos);
		aav = res_variant(seqpos);
		get_hbE(fullatom,sc_aa,aa,sc_aav,aav,sc_seqpos,seqpos,neighbors(sc_seqpos),
		 neighbors(seqpos),sc_rotcoord,full_coord(1,1,seqpos),schbE,srbbhbE,lrbbhbE,
		 sc_bbhbE,nhbonds_l,hbenergies_l,hbdonh_atm_l,hbdon_res_l,hbact_atm_l,
		 hbact_res_l,hbtype_l,hbchk1_l,hbchk2_l,deriv_type,hbderiv_l);

		if ( nhbonds_l == 0 ) goto L100; // no hbond
		for ( int i = 1; i <= nhbonds_l; ++i ) {
			dres = hbdon_res_l(i);
			dhatm = hbdonh_atm_l(i);
			ares = hbact_res_l(i);
			aatm = hbact_atm_l(i);
			d_is_bb = atom_is_backbone(dhatm,res(dres),res_variant(dres));
			a_is_bb = atom_is_backbone(aatm,res(ares),res_variant(ares));

			if ( !allow_hbond(dres,d_is_bb,ares,a_is_bb) ) goto L80;

			if ( dres == sc_seqpos ) { // minimized res is donor
				atom = dhatm;
				weight = 1.0;
			} else {                // minimized res is acceptor
				atom = aatm;
				weight = -1.0;
			}

			switch(hbtype_l(i)) { //JSS hack for now
        case hbe_BBTURN:
        case hbe_BBHELIX:
					weight *= pack_wts.Whbond_bb();
          break;
        case hbe_BBOTHER:
					weight *= pack_wts.Whbond_bb();
          break;
        case hbe_SP2B:
        case hbe_SP3B:
        case hbe_RINGB:
        case hbe_BSC:
					weight *= pack_wts.Whbond_bb_sc();
          break;
        case hbe_SP2SC:
        case hbe_SP3SC:
        case hbe_RINGSC:
					weight *= pack_wts.Whbond_sc();
         break;
        default:
          std::cout<<"Warning: energy from unexpected HB type ignored "<< hbtype_l(i) <<std::endl;
          break;
			}

			for ( int k = 1; k <= 3; ++k ) { // accumulate f1 & f2
				f1(k,atom) += weight * hbderiv_l(k,1,i);
				f2(k,atom) += weight * hbderiv_l(k,2,i);
			}
L80:; // skip this hbond
		}
L100:; // skip this seqpos
	}

  assert(deriv_type == hbonds::hbderiv_ABE_GO);
  //     convert deriv by atom to deriv by chi_rigid_atom_list
	for ( int k = 1; k <= MAX_CHI; ++k ) {
		for ( int i = 1, ie = length_chi_rigid_atom_list(k,sc_aa,sc_aav); i <= ie; ++i ) {
			atom = chi_rigid_atom_list(i,k,sc_aa,sc_aav);
			for ( int j = 1; j <= 3; ++j ) {
				f1_hbond_deriv(j,k) += f1(j,atom);
				f2_hbond_deriv(j,k) += f2(j,atom);
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           get_lj_solv_deriv(chi,f1,f2)
///
/// @brief
///
/// @detailed
///
/// @param[in]        chi                       int
/// @param[in]        f1                        float
/// @param[in]        f2                        float
///
/// @param[out]       f1                        float
/// @param[out]       f2                        float
///
/// @global_read     res_variant                int    template_pack.h template
/// @global_read     res                       int    template_pack.h template
/// @global_read     chi_rigid_atom_list       int    aaproperties_pack.h properties_1
/// @global_read     length_chi_rigid_atom_list int    aaproperties_pack.h properties_1
/// @global_read     neighborlist              bool    template_pack.h template
/// @global_read     total_residue                      int    template_pack.h template
/// @global_read     sc_aa                     int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_aav                    int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotcoord               float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_seqpos                 int    rotamer_functions.h rotamer_minimization
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_LJ_solv_deriv(
	int & chi,
	FArray1DB_float & f1,
	FArray1DB_float & f2
)
{
	using namespace aaproperties_pack;
	using namespace rotamer_minimization;
	using namespace misc;
	using namespace template_pack;

//chu   local
	static FArray1D_float ifull_coord( 3 );

	f1 = 0.0;
	f2 = 0.0;

//==============================================================================
//mj ligand

	if ( get_ligand_flag() ) {
		get_sc_haD_chi(sc_seqpos,sc_aa,sc_aav,sc_rotcoord,chi,f1,f2,(*ligand::ligand_one));
	}

//==============================================================================

	for ( int i = 1, l = chi_rigid_atom_list.index(i,chi,sc_aa,sc_aav),
	 ie = length_chi_rigid_atom_list(chi,sc_aa,sc_aav); i <= ie; ++i, ++l ) {
		int iatom = chi_rigid_atom_list[ l ]; // chi_rigid_atom_list(i,chi,sc_aa,sc_aav);
		get_atom_from_sc_rotcoord(iatom,ifull_coord);
//		accumulate_local_f1_f2(iatom,sc_aa,sc_aav,sc_seqpos,ifull_coord,f1,f2);
		for ( int jseqpos = 1, ln = neighborlist.index(jseqpos,sc_seqpos);
		 jseqpos <= total_residue; ++jseqpos, ++ln ) {
			if ( sc_seqpos != jseqpos ) {
				if ( neighborlist[ ln ] ) { // neighborlist(jseqpos,sc_seqpos)
					int jaa = res(jseqpos);
					int jaav = res_variant(jseqpos);
					accumulate_nb_f1_f2(iatom,sc_aa,sc_aav,sc_seqpos,sc_rotcoord,jaa,jaav,
					 jseqpos,f1,f2);
				} // neighbors
			} // local / nb
		} // seqpos
	} // chi_rigid_atom_list

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           get_rotcoord_from_fullcoord(seqpos,coord)
///
/// @brief
///
/// @detailed
///
/// @param[in]        coord                     float
/// @param[in]        seqpos                    int
///
/// @param[out]       coord                     float
///
/// @global_read     MAX_ATOM                  int    param.h
/// @global_read     full_coord                       float       template_pack.h template
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
get_rotcoord_from_fullcoord(
	int seqpos,
	FArray2DB_float & coord // Dims: ( 3, MAX_ATOM )
)
{
	using namespace misc;

	assert( coord.size_bounded() );
	for ( int lc = 0, lfull_coord = full_coord.index( 1, 1, seqpos ), e = coord.size();
	 lc < e; ++lc, ++lfull_coord ) {
		coord[ lc ] = full_coord[ lfull_coord ];
	}

//Objexx: Replaced by above for speed
//	for ( int j = 1, max_atom = MAX_ATOM(); j <= max_atom; ++j ) {
//		for ( int i = 1; i <= 3; ++i ) {
//			coord(i,j) = full_coord(i,j,seqpos);
//		}
//	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin           accumulate_nb_f1_f2(iatom,iaa,iaav,iseqpos,icoord,jaa,jaav,jseqpos,f1,f2)
///
/// @brief
///chu   Here the deriv calculation is dependent on whether iatom is a hydrogen
///     because the repulsive energy for hydrogens are evaluated only when
///     the distance between two heavy atoms is within a certain cutoff.
///
/// @detailed
///
/// @param[in]        f1                        float
/// @param[in]        f2                        float
/// @param[in]        iaa                       int
/// @param[in]        iaav                      int
/// @param[in]        iatom                     int
/// @param[in]        icoord                    float
/// @param[in]        iseqpos                   int
/// @param[in]        jaa                       int
/// @param[in]        jaav                      int
/// @param[in]        jseqpos                   int
///
/// @global_read     fullatom_type             int    aaproperties_pack.h
/// @global_read     atom_base                 int    aaproperties_pack.h
/// @global_read     hydrogen_interaction_cutoff float       pdbstatistics_pack.h
/// @global_read     nheavyatoms               int    aaproperties_pack.h
/// @global_read     pack_wts.Watr()                      float       param_pack.h packer_weight_parameters
/// @global_read     pack_wts.Wrep()                      float       param_pack.h packer_weight_parameters
/// @global_read     pack_wts.Wsol()                      float       param_pack.h packer_weight_parameters
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
accumulate_nb_f1_f2(
	int iatom,
	int iaa,
	int iaav,
	int iseqpos,
	FArray2DB_float & icoord,
	int jaa,
	int jaav,
	int jseqpos,
	FArray1DB_float & f1,
	FArray1DB_float & f2
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_pack;
	using namespace pdbstatistics_pack;


// local //Objexx: static arrays for speed
	static FArray1D_float ifull_coord( 3 );
	static FArray1D_float jfull_coord( 3 );
	static FArray2D_float jcoord( 3, MAX_ATOM() );
	float dlj,drep,datr,dsol,dE_dr,dis_ij,cp_weight;

	int iatype = fullatom_type(iatom,iaa,iaav);

	{ // positon of iatom
		int lic = icoord.index(1,iatom);
		ifull_coord(1) = icoord[   lic ]; // icoord(1,iatom);
		ifull_coord(2) = icoord[ ++lic ]; // icoord(2,iatom);
		ifull_coord(3) = icoord[ ++lic ]; // icoord(3,iatom);
	}

	bool const iatom_is_H = ( iatom > nheavyatoms(iaa,iaav) );
	int iatom_base;
	if ( iatom_is_H ) {
		iatom_base = atom_base(iatom,iaa,iaav);
	} else {
		iatom_base = iatom;
	}
	for ( int jatom = 1, lfa = fullatom_type.index(jatom,jaa,jaav),
	 ljc = jcoord.index(1,jatom), jatome = nheavyatoms(jaa,jaav);
	 jatom <= jatome; ++jatom, ++lfa, ljc+=3 ) {
		if ( count_pair(iseqpos,iatom_base,iaa,iaav,jseqpos,jatom,jaa,jaav,true,
		 cp_weight) ) {

			get_rotcoord_from_fullcoord(jseqpos,jcoord);

			if ( !iatom_is_H ) { // iatom is heavy atom
				int const jatype = fullatom_type[ lfa ]; // fullatom_type(jatom,jaa,jaav);
				// positon of jatom
				jfull_coord(1) = jcoord[ ljc   ]; // jcoord(1,jatom);
				jfull_coord(2) = jcoord[ ljc+1 ]; // jcoord(2,jatom);
				jfull_coord(3) = jcoord[ ljc+2 ]; // jcoord(3,jatom);
				datr = 0.0;
				drep = 0.0;
				dsol = 0.0;
				pairderiv(ifull_coord,jfull_coord,iatype,jatype,dlj,datr,drep,dsol);
				dE_dr = ( datr*pack_wts.Watr() + drep*pack_wts.Wrep() + dsol*pack_wts.Wsol() ) * cp_weight;
				do_f1_f2_formula(ifull_coord,jfull_coord,dE_dr,f1,f2);
				//// heavy atom --> heavy atom
				dis_ij = vec_dist(ifull_coord,jfull_coord);
				if ( dis_ij < std::sqrt(hydrogen_interaction_cutoff) ) {
					pairderiv_heavyatom_to_h(iatom,iaa,iaav,iseqpos,icoord,
					 jatom,jaa,jaav,jseqpos,jcoord,f1,f2); // heavyatom --> H
				}
			} else { // iatom is H
				pairderiv_h_to_all(iatom,iatom_base,iaa,iaav,iseqpos,icoord,
				 jatom,jaa,jaav,jseqpos,jcoord,f1,f2);
				 //// H --> heavy atom and its attched hydrogens
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           pairderiv_h_to_all(hatom1,atom1,aa1,aav1,coord1,atom2,aa2,aav2,coord2,f1,f2)
///
/// @brief
///chu   Hydrogen --> a heavy atom and its attached hydrogens
///
/// @detailed
///
/// @param[in]        aa1                       int
/// @param[in]        aa2                       int
/// @param[in]        aav1                      int
/// @param[in]        aav2                      int
/// @param[in]        atom1                     int
/// @param[in]        atom2                     int
/// @param[in]        coord1                    float
/// @param[in]        coord2                    float
/// @param[in]        f1                        float
/// @param[in]        f2                        float
/// @param[in]        hatom1                    int
///
/// @global_read     fullatom_type             int    aaproperties_pack.h properties_1
/// @global_read     hydrogen_interaction_cutoff float       pdbstatistics_pack.h
/// @global_read     hydrogens_on_atm          int    aaproperties_pack.h properties_4
/// @global_read     nhydrogens_on_atm         int    aaproperties_pack.h properties_4
/// @global_read     pack_wts.Wrep()                      float       param_pack.h packer_weight_parameters
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
pairderiv_h_to_all(
	int const hatom1,
	int const atom1,
	int const aa1,
	int const aav1,
	int const res1,
	FArray2DB_float & coord1,
	int const atom2,
	int const aa2,
	int const aav2,
	int const res2,
	FArray2DB_float & coord2,
	FArray1DB_float & f1,
	FArray1DB_float & f2
)
{
	using namespace aaproperties_pack;
	using namespace param_pack;
	using namespace pdbstatistics_pack;

	float & coord2_a2( coord2(1,atom2) );
	float dis2 = vec_dist2(coord1(1,atom1),coord2_a2);
	if ( dis2 >= hydrogen_interaction_cutoff ) return; // > cutoff

	float cp_weight;
	float drep = 0.0;
	float & coord1_h1( coord1(1,hatom1) );
	int & fa_type_h1( fullatom_type(hatom1,aa1,aav1) );
	if ( count_pair(res1,hatom1,aa1,aav1,res2,atom2,aa2,aav2,true,cp_weight) )
	 pairderiv_hydrogens(coord1_h1,coord2_a2,
	 fa_type_h1,fullatom_type(atom2,aa2,aav2),drep,cp_weight);
	do_f1_f2_formula(coord1_h1,coord2_a2,drep*pack_wts.Wrep(),f1,f2);
	for ( int n2 = 1, l = hydrogens_on_atm.index(n2,atom2,aa2,aav2),
	 n2e = nhydrogens_on_atm(atom2,aa2,aav2); n2 <= n2e; ++n2, ++l ) {
		int hatom2 = hydrogens_on_atm[ l ]; // hydrogens_on_atm(n2,atom2,aa2,aav2)
		drep = 0.0;
		float & coord2_h2( coord2(1,hatom2) );
		if ( count_pair(res1,hatom1,aa1,aav1,res2,hatom2,aa2,aav2,true,cp_weight) )
		 pairderiv_hydrogens(coord1_h1,coord2_h2,fa_type_h1,
		 fullatom_type(hatom2,aa2,aav2),drep,cp_weight);
		do_f1_f2_formula(coord1_h1,coord2_h2,drep*pack_wts.Wrep(),f1,f2);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           pairderiv_heavyatom_to_h(atom1,aa1,aav1,coord1,atom2,aa2,aav2,coord2,f1,f2)
///
/// @brief
///chu   heavyatom(no attached H) --> the attached Hydrogens on another heavyatom.
///
/// @detailed
///
/// @param[in]        aa1                       int
/// @param[in]        aa2                       int
/// @param[in]        aav1                      int
/// @param[in]        aav2                      int
/// @param[in]        atom1                     int
/// @param[in]        atom2                     int
/// @param[in]        coord1                    float
/// @param[in]        coord2                    float
/// @param[in]        f1                        float
/// @param[in]        f2                        float
///
/// @global_read     fullatom_type             int    aaproperties_pack.h properties_1
/// @global_read     hydrogens_on_atm          int    aaproperties_pack.h properties_4
/// @global_read     nhydrogens_on_atm         int    aaproperties_pack.h properties_4
/// @global_read     pack_wts.Wrep()                      float       param_pack.h packer_weight_parameters
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pairderiv_heavyatom_to_h(
	int const atom1,
	int const aa1,
	int const aav1,
	int const res1,
	FArray2DB_float & coord1,
	int const atom2,
	int const aa2,
	int const aav2,
	int const res2,
	FArray2DB_float & coord2,
	FArray1DB_float & f1,
	FArray1DB_float & f2
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_pack;

	float cp_weight;
	float & coord1_a1( coord1(1,atom1) );
	int const fa_type_a1 = fullatom_type(atom1,aa1,aav1);
	for ( int n2 = 1, l = hydrogens_on_atm.index(n2,atom2,aa2,aav2),
	 n2e = nhydrogens_on_atm(atom2,aa2,aav2); n2 <= n2e; ++n2, ++l ) {
		int hatom2 = hydrogens_on_atm[ l ]; // hydrogens_on_atm(n2,atom2,aa2,aav2);
		float drep = 0.0;
		float & coord2_h2( coord2(1,hatom2) );
		if ( count_pair(res1,atom1,aa1,aav1,res2,hatom2,aa2,aav2,true,cp_weight) )
		 pairderiv_hydrogens(coord1_a1,coord2_h2,fa_type_a1,
		 fullatom_type(hatom2,aa2,aav2),drep,cp_weight);
		do_f1_f2_formula(coord1_a1,coord2_h2,drep*pack_wts.Wrep(),f1,f2);
	}

//mj test
//
//	for ( int n1 = 1, n1e = nhydrogens_on_atm(atom1,aa1,aav1); n1 <= n1e; ++n1 ) {
//		hatom1 = hydrogens_on_atm(n1,atom1,aa1,aav1);
//		drep = 0.0;
//		pairderiv_hydrogens(coord1(1,hatom1),coord2(1,atom2),
//		 fullatom_type(hatom1,aa1,aav1),fullatom_type(atom2,aa2,aav2),drep);
//		do_f1_f2_formula(coord1(1,hatom1),coord2(1,atom2),drep*pack_wts.Wrep(),f1,f2);
//	}
//
//	for ( int n1 = 1, n1e = nhydrogens_on_atm(atom1,aa1,aav1),
//	 n2e = nhydrogens_on_atm(atom2,aa2,aav2); n1 <= n1e; ++n1 ) {
//		for ( int n2 = 1; n2 <= n2e; ++n2 ) {
//			hatom1 = hydrogens_on_atm(n1,atom1,aa1,aav1);
//			hatom2 = hydrogens_on_atm(n2,atom2,aa2,aav2);
//			drep = 0.0;
//			pairderiv_hydrogens(coord1(1,hatom1),coord2(1,hatom2),
//			 fullatom_type(hatom1,aa1,aav1),fullatom_type(hatom2,aa2,aav2),drep);
//			do_f1_f2_formula(coord1(1,hatom1),coord2(1,hatom2),drep*pack_wts.Wrep(),f1,f2);
//		}
//	}
//
//mj end test


}

////////////////////////////////////////////////////////////////////////////////
/// @begin           do_f1_f2_formula(ifull_coord,jfull_coord,de_dr,f1,f2)
///
/// @brief
///
/// @detailed
///
/// @param[in]        de_dr                     float
/// @param[in]        f1                        float
/// @param[in]        f2                        float
/// @param[in]        ifull_coord                      float
/// @param[in]        jfull_coord                      float
///
/// @param[out]       f1                        float
/// @param[out]       f2                        float
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
do_f1_f2_formula(
	FArray1Da_float ifull_coord,
	FArray1Da_float jfull_coord,
	float dE_dr,
	FArray1DB_float & f1,
	FArray1DB_float & f2
)
{
	if ( dE_dr == 0.0 ) return;

//Objexx: Only pass these to argument arrays so don't need to set dimensions
//	ifull_coord.dimension( 3 );
//	jfull_coord.dimension( 3 );

//     local  //Objexx: static for speed
	static FArray1D_float minus( 3 );
	static FArray1D_float product( 3 );

	subvec(ifull_coord,jfull_coord,minus);
	float const dis = std::sqrt(
	 ( minus(1) * minus(1) ) + ( minus(2) * minus(2) ) + ( minus(3) * minus(3) ) );
	cros(ifull_coord,jfull_coord,product);
	float const dE_dr_dis = dE_dr / dis;
	for ( int j = 1; j <= 3; ++j ) {
		f1(j) += product(j) * dE_dr_dis;
		f2(j) += minus(j) * dE_dr_dis;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           pairderiv_hydrogens(atom1,atom2,attype1,attype2,repulse)
///
/// @brief
///jg     uses lookup table to find the pairwise derivatives between two atoms,
///
/// @detailed
///chu     note this is the partial version of parideriv above and should be only
///       used for hydrogens derivative calculation. LJ repulsive only.
///
/// @param[in]        atom1                     float
/// @param[in]        atom2                     float
/// @param[in]        attype1                   int
/// @param[in]        attype2                   int
/// @param[in]        repulse                   float
///
/// @param[out]       repulse                   float
///
/// @global_read     dljrep                    float       pdbstatistics_pack.h pdbstatistics
/// @global_read     fa_bins_per_a2            int    pdbstatistics_pack.h
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
pairderiv_hydrogens(
	FArray1Da_float atom1,
	FArray1Da_float atom2,
	int const attype1,
	int const attype2,
	float & repulse,
	float const cp_weight
)
{
	using namespace pdbstatistics_pack;


	float const a_1 = atom1[ 0 ] - atom2[ 0 ]; // atom1(1) - atom2(1)
	float const a_2 = atom1[ 1 ] - atom2[ 1 ]; // atom1(2) - atom2(2)
	float const a_3 = atom1[ 2 ] - atom2[ 2 ]; // atom1(3) - atom2(3)
	float const d2 = ( ( a_1 * a_1 ) + ( a_2 * a_2 ) + ( a_3 * a_3 ) );

//$$$   if ( d2 >= safe_max_dis2 || d2 == 0.0 ) {
//$$$      return;
//$$$   }

//ctsa
//ctsa  get bins and interpolation fraction
//ctsa
	float const d2_bin = d2 * fa_bins_per_A2;
	int const disbin1 = static_cast< int >( d2_bin ) + 1;
//	int const disbin2 = disbin1 + 1;
	float const frac = d2_bin - ( disbin1 - 1 );

//ctsa
//ctsa   tables have been hacked so that if disbin2 = lastbin, all values = 0.
//ctsa

	int l = pCurrentEtable->dljrep.index(disbin1,attype2,attype1);
	float const repulse1 = pCurrentEtable->dljrep[   l ]; // dljrep(disbin1,attype2,attype1);
	float const repulse2 = pCurrentEtable->dljrep[ ++l ]; // dljrep(disbin2,attype2,attype1);

	repulse += ( ( ( 1 - frac ) * repulse1 ) + ( frac * repulse2 ) ) * cp_weight;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           get_atom_from_sc_rotcoord(atom,coord)
///
/// @brief
///
/// @detailed
///
/// @param[in]        atom                      int
/// @param[in]        coord                     float
///
/// @param[out]       coord                     float
///
/// @global_read     sc_rotcoord               float       rotamer_functions.h rotamer_minimization
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_atom_from_sc_rotcoord(
	int const atom,
	FArray1DB_float & coord
)
{
	using namespace rotamer_minimization;

	for ( int i = 1, l = sc_rotcoord.index(i,atom); i <= 3; ++i, ++l ) {
		coord(i) = sc_rotcoord[ l ]; // sc_rotcoord(i,atom);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin           make_chi_unit_vector(torsion,vec,end_atom,rotcoord)
///
/// @brief
///
/// @detailed
///chu   return the unit vector along the chi torsion angle
///chu   used only in rotamer_trial with side-chain minimization!!!
///
/// @param[in]        end_atom                  int
/// @param[in]        rotcoord                  float
/// @param[in]        torsion                   int
/// @param[in]        vec                       float
///
/// @param[out]       end_atom                  int
/// @param[out]       vec                       float
///
/// @global_read     chi_atoms                 int    aaproperties_pack.h properties_1
/// @global_read     sc_aa                     int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_aav                    int    rotamer_functions.h rotamer_minimization
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_chi_unit_vector(
	int const torsion,
	FArray1DB_float & vec,
	int & end_atom,
	FArray2DB_float & rotcoord
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace rotamer_minimization;

	int l_atoms = chi_atoms.index(2,torsion,sc_aa,sc_aav);
	int start_atom = chi_atoms[ l_atoms ]; // chi_atoms(2,torsion,sc_aa,sc_aav);
	end_atom = chi_atoms[ ++l_atoms ]; // chi_atoms(3,torsion,sc_aa,sc_aav);

	if ( start_atom == 0 || end_atom == 0 ) {
		std::cout << "Invalid value of torsion: " << "aa->" << SS( sc_aa ) <<
		 " chi->" << SS( torsion ) << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	vec(1) = rotcoord(1,end_atom) - rotcoord(1,start_atom);
	vec(2) = rotcoord(2,end_atom) - rotcoord(2,start_atom);
	vec(3) = rotcoord(3,end_atom) - rotcoord(3,start_atom);

	float const temp = 1.0 / std::sqrt(
	 ( vec(1) * vec(1) ) + ( vec(2) * vec(2) ) + ( vec(3) * vec(3) ) );

	for ( int k = 1; k <= 3; ++k ) {
		vec(k) *= temp;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin           pack_min_chi(min_chi)
///
/// @brief
///
/// @detailed
///
/// @param[in]        min_chi                   float
///
/// @param[out]       min_chi                   float
///
/// @global_read     MAX_CHI                   int    param.h
/// @global_read     sc_chi                    float       rotamer_functions.h rotamer_minimization
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
pack_min_chi( FArray1DB_float & min_chi )
{
	using namespace rotamer_minimization;

	min_chi = sc_chi;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin           unpack_min_chi(min_chi)
///
/// @brief
///
/// @detailed
///
/// @param[in]        min_chi                   float
///
/// @global_read     sc_aa                     int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_aav                    int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_nfree                  int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotcoord               float       rotamer_functions.h rotamer_minimization
/// @global_read     sc_rotid                  int    rotamer_functions.h rotamer_minimization
/// @global_read     sc_seqpos                 int    rotamer_functions.h rotamer_minimization
/// @global_read     full_coord                       float       template_pack.h template
///
/// @global_write    sc_chi                    float       rotamer_functions.h rotamer_minimization
/// @global_write    sc_rotactcoord            float       rotamer_functions.h rotamer_minimization
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
unpack_min_chi( FArray1Da_float min_chi )
{
	using namespace rotamer_minimization;
	using namespace misc;

	min_chi.dimension( star );

	for ( int j = 1; j <= sc_nfree; ++j ) {
		if ( std::abs(min_chi(j)) >= 180.0 ) {
			angle_in_range(min_chi(j));
		}
		sc_chi(j) = min_chi(j);
	}

	rotamer_from_chi(sc_chi,sc_aa,sc_rotid);
	get_rot_coord(full_coord,sc_aa,sc_aav,sc_chi,sc_rotcoord,sc_seqpos);
	put_wcentroid(sc_rotcoord,sc_rotactcoord,sc_aa);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           sum_trial_energy(atr_e,rep_e,solv_e,hbond_e)
///
/// @brief
///chu   only for numerical derivatives check in rotamer minimization
///
/// @detailed
///
/// @param[in]        atr_e                     float
/// @param[in]        hbond_e                   float
/// @param[in]        rep_e                     float
/// @param[in]        solv_e                    float
///
/// @param[out]       atr_e                     float
/// @param[out]       hbond_e                   float
/// @param[out]       rep_e                     float
/// @param[out]       solv_e                    float
///
/// @global_read     atrenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     hbenergy_trial            float       rotamer_trial_energies.h trial_energies
/// @global_read     total_residue                      int    template_pack.h template
/// @global_read     repenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     solenergy_trial           float       rotamer_trial_energies.h trial_energies
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
sum_trial_energy(
	float & atr_E,
	float & rep_E,
	float & solv_E,
	float & hbond_E
)
{

	using namespace rotamer_trial_energies;
	using namespace misc;

	atr_E = 0.0;
	rep_E = 0.0;
	solv_E = 0.0;
	hbond_E = 0.0;

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		atr_E += atrenergy_trial(seqpos);
		rep_E += repenergy_trial(seqpos);
		solv_E += solenergy_trial(seqpos);
		hbond_E += hbenergy_trial(seqpos);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin           write_rot_energies_ctsa(seqpos,aa,rot,aanat,nb)
///
/// @brief
/// ctsa -slightly reformatted opte output for my parameterization code
///
/// @detailed
///
/// @param[in]        aa                        int
/// @param[in]        aanat                     int
/// @param[in]        nb                        int
/// @param[in]        rot                       int
/// @param[in]        seqpos                    int
///
/// @global_read     atrenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     bbhbenergy_trial          float       rotamer_trial_energies.h trial_energies
/// @global_read     dun_trial                 float       rotamer_trial_energies.h trial_energies
/// @global_read     intrares_trial            float       rotamer_trial_energies.h trial_energies
/// @global_read     opte_x                    int         files_paths.h
/// @global_read     pair_energy_trial         float       rotamer_trial_energies.h trial_energies
/// @global_read     prob_trial                float       rotamer_trial_energies.h trial_energies
/// @global_read     protein_chain             char        misc.h     chars
/// @global_read     repenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     sc_bbhbenergy_trial       float       rotamer_trial_energies.h trial_energies
/// @global_read     schbenergy_trial          float       rotamer_trial_energies.h trial_energies
/// @global_read     solenergy_trial           float       rotamer_trial_energies.h trial_energies
/// @global_read     start_file                std::string files_paths.h file_path_char
/// @global_read     unf_trial                 float       rotamer_trial_energies.h trial_energies
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
write_rot_energies_ctsa(
	int const seqpos,
	int const aa,
	int const rot,
	int const aanat,
	int const nb
)
{

	using namespace files_paths;
	using namespace rotamer_trial_energies;

	opte_x << start_file.substr(0,4) << protein_chain << ' ' <<
		I( 4, seqpos ) << ' ' << I( 2, aa ) << ' ' << I( 3, rot ) << ' ' <<
		I( 2, aanat ) << ' ' << I( 1, 0 ) << ' ' << I( 2, nb ) << ' ' <<
		I( 2, 0 ) << ' ' <<
		E( 10, 4, 2*atrenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*repenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*pair_energy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*plane_energy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*solenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, dun_trial(seqpos) ) << ' ' <<
		E( 10, 4, intrares_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*bbhbenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*sc_bbhbenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*schbenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*gb_elecenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, prob_trial(seqpos) ) << ' ' <<
		E( 10, 4, unf_trial(seqpos) ) << ' ' <<
		E( 10, 4, protonation_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*cstenergy_trial(seqpos) ) << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin write_rot_energies_lig
///
/// @brief
/// ccmd -slightly reformatted opte output for LIG parameterization
///
/// @detailed
///
/// @param  seqpos - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  rot - [in/out]? -
/// @param  aanat - [in/out]? -
/// @param  nb - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
write_rot_energies_lig(
	int const seqpos,
	int const aa,
	int const rot,
	int const aanat,
	int const nb
)
{

	using namespace files_paths;
	using namespace rotamer_trial_energies;

	opte_x << start_file.substr(0,4) << protein_chain << ' ' <<
		I( 4, seqpos ) << ' ' << I( 2, aa ) << ' ' << I( 3, rot ) << ' ' <<
		I( 2, aanat ) << ' ' << I( 1, 0 ) << ' ' << I( 2, nb ) << ' ' <<
		I( 2, 0 ) << ' ' <<
		E( 10, 4, 2*atrenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*repenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*pair_energy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*solenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, dun_trial(seqpos) ) << ' ' <<
		E( 10, 4, intrares_trial(seqpos) ) << ' ' <<
		E( 10, 4, unf_trial(seqpos) ) << ' ' <<
		E( 10, 4,  protonation_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*(sc_bbhbenergy_trial(seqpos)+bbhbenergy_trial(seqpos)) ) << ' ' <<
		E( 10, 4, 2*schbenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, 2*gb_elecenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, prob_trial(seqpos) ) << ' ' <<
		E( 10, 4, lig_repenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, lig_atrenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, lig_solenergy_trial(seqpos) ) << ' ' <<
		E( 10, 4, lig_couenergy_trial(seqpos) ) << ' ' <<
		//		E( 10, 4, lig_virenergy_trial(seqpos) ) << ' ' << ! we hardly want to fit virtual energy ;-)
		E( 10, 4, lig_hb_energy_trial(seqpos) ) << std::endl;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin set_dummy_seq_array
///
/// @brief
///jjh This just initializes the dummy argument for rotamer_trials
///jjh It's ok to omit this function, but ftnchek complains
///
/// @detailed
///
/// @param  dummy_seq_select - [in/out]? -
/// @param  max_aa - [in/out]? -
/// @param  max_res - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_dummy_seq_array(
	FArray2Da_bool dummy_seq_select,
	int max_aa,
	int max_res
)
{
	dummy_seq_select.dimension( max_aa, max_res );

	dummy_seq_select = false;
}


///////////////////////////////////////////////////////////////////////////////
void
update_repack_list(
	FArray1D_int & repack_list,
	int & numpositions,
	int const total_residue,
	FArray1D_bool const & new_rotamer,
	FArray1D_bool const & allow_move,
	FArray2D_bool const & neighborlist
)
{
	FArray1D_bool included(total_residue,false);
	numpositions = 0;
	for ( int i=1; i<= total_residue; ++i ) {
		if ( new_rotamer(i) ) {
			for ( int j=1,l=neighborlist.index(1,i); j<= total_residue; ++j,++l ) {
				assert ( neighborlist(j,i) == neighborlist[l] ); // TAKE OUT
				if ( ( j==i || neighborlist[l] ) && !included(j) && allow_move(j) ) {
					++numpositions;
					repack_list(numpositions) = j;
					included(j) = true;
				}
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
void
create_symmetric_rotamers(
	int const seqpos,
	int const aa,
	int const aav,
	FArray2Da_float rotcoord,
	FArray1Da_float rotactcoord,
	FArray3D_float const & full_coord
)
{

	using namespace aaproperties_pack;
	using namespace param;
	using namespace template_pack;
	using namespace design_sym;

	rotcoord.dimension(3,MAX_ATOM()());
	rotactcoord.dimension(3);

	FArray2D_float Mgl(4,4);

	assert( clone_follows(seqpos) == 0 );
	for ( int i=1; i<= num_clones; ++i ) {

		int const clone_pos( clone_list( seqpos, i ) );

		// calculate transform
		get_sym_rotamer_transform( full_coord(1,1,seqpos),
															 full_coord(1,1,clone_pos), Mgl );

		transfer_sym_rotamer( aa, aav, Mgl, rotcoord, rotactcoord,
													full_coord(1,1,clone_pos), actcoord(1,clone_pos) );
	}
}


///////////////////////////////////////////////////////////////////////////////
// call this routine right after
//
void
debug_rotamer_trials(
	pose_ns::Pose & pose,
	Scoring_Function score_fxn
)
{
	if ( !score_get_try_rotamers() ) return;
	static bool init( false ), debug( false );
	if ( !init ) {
		init = true;
		debug = truefalseoption("debug_rotamer_trials");
	}
	if ( !debug ) return;

	using fullatom_energies::resenergy;
	using fullatom_energies::atrenergy;
	using fullatom_energies::repenergy;
	using fullatom_energies::pair_energy;
	using fullatom_energies::hbenergy;
	using fullatom_energies::dunenergy;
	using fullatom_energies::protonation_energy;
	using fullatom_energies::cstenergy;
	using fullatom_energies::gb_elecenergy;
	using fullatom_energies::pair_pair;
	using fullatom_energies::h2oenergy;
	using fullatom_energies::h2ohbenergy;
	//using namespace design_sym;

	// copies
	FArray1D_int nb_count_pre( template_pack::neighbors );
	FArray1D_float resE( resenergy ), atrE( atrenergy ), repE( repenergy ),
		pairE( pair_energy ), hbE( hbenergy ), gbE( gb_elecenergy ),
		protE( protonation_energy ), cstE( cstenergy ), dunE( dunenergy ),
		h2oE( h2oenergy ), h2ohbE( h2ohbenergy );

	float const rt_fullatomE( fullatom_energies::fullatomE );

	// rescore with rotamer trials off
	bool const save_enable( score_get_try_rotamers() );
	score_enable_rotamer_trials( false );
	pose.new_score_pose();
	pose.score( score_fxn );
	score_enable_rotamer_trials( save_enable );

	// compare energies
	float const fE( fullatom_energies::fullatomE );
	bool nb_mismatch( false );
	bool const symm_rottrial( pose.symmetric() );
	for ( int i=1; i<= pose.total_residue(); ++i ) {
		if ( nb_count_pre(i) != template_pack::neighbors(i) ) {
			nb_mismatch=true;
		}
		if ( i <= pose.total_residue() ) {
			if ( symm_rottrial && !pose.symmetry_info().chi_independent(i)) continue;
			float const dev
				( std::abs(     resenergy(i) -  resE(i) ) +
					std::abs(     atrenergy(i) -  atrE(i) ) +
					std::abs(      hbenergy(i) -   hbE(i) ) +
					std::abs( gb_elecenergy(i) -   gbE(i) ) +
					std::abs(   pair_energy(i) - pairE(i) ) );
			if ( dev > 0.1 ) {
				std::cout << i <<
					" resE_delta: " << F(9,3,resenergy(i) - resE(i) ) <<
					" atrE_delta: " << F(9,3,atrenergy(i) - atrE(i) ) <<
					" repE_delta: " << F(9,3,repenergy(i) - repE(i) ) <<
					" hbE_delta: " << F(9,3,hbenergy(i) - hbE(i) ) <<
					" gbE_delta: " << F(9,3,gb_elecenergy(i) - gbE(i) ) <<
					" protE_delta: " << F(9,3,protonation_energy(i) - protE(i) ) <<
					" dunE_delta: " << F(9,3,dunenergy(i) - dunE(i) ) <<
					" cstE_delta: " << F(9,3,cstenergy(i) - cstE(i) ) <<
					" h2oE_delta: " << F(9,3,h2oenergy(i) - h2oE(i) ) <<
					" h2ohbE_delta: " << F(9,3,h2ohbenergy(i) - h2ohbE(i) ) <<
					" pairE_delta: " << F(9,3,pair_energy(i) - pairE(i) ) << std::endl;
			}
		}
	}

	std::cout << "rottrial_fullatomE_delta: " << F(9,3,fE - rt_fullatomE ) <<
		" fullatomE: " << F(9,3,fE) << F(9,3,rt_fullatomE) <<
		" nb-mismatch: " << nb_mismatch <<
		std::endl;
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////



