#ifndef INCLUDED_ObjexxFCL_FArray1Da_HH
#define INCLUDED_ObjexxFCL_FArray1Da_HH


// FArray1Da: Fortran-Compatible 1D Argument Array
//
// Project: Objexx Fortran Compatibility Library (ObjexxFCL)
//
// Version: 2.6.2
//
// Language: C++
//
// Copyright (c) 2007 Objexx Engineering, Inc. All Rights Reserved.
// Use of this source code or any derivative of it is restricted by license.
// Licensing is available from Objexx Engineering, Inc.:   http://objexx.com   Objexx@objexx.com


// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.fwd.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Dp.hh>
#include <ObjexxFCL/StaticIndexRange.hh>


namespace ObjexxFCL {


/// @brief FArray1Da: Fortran-Compatible 1D Argument Array
template< typename T >
class FArray1Da :
	public FArray1DB< T >
{


private: // Types


	typedef  FArray1DB< T >  Super;
	typedef  typename Super::real_FArray  real_FArray;
	typedef  typename Super::proxy_FArray  proxy_FArray;
	typedef  typename Super::arg_FArray  arg_FArray;
	typedef  internal::ProxySentinel  ProxySentinel;


public: // Types


	typedef  typename Super::Base  Base;
	typedef  typename Base::Section  Section;
	typedef  typename Super::IR  SIR;
	typedef  StaticIndexRange  IR;

	// STL Style
	typedef  typename Base::value_type  value_type;
	typedef  typename Base::reference  reference;
	typedef  typename Base::const_reference  const_reference;
	typedef  typename Base::pointer  pointer;
	typedef  typename Base::const_pointer  const_pointer;
	typedef  typename Base::size_type  size_type;
	typedef  typename Base::difference_type  difference_type;

	// C++ Style
	typedef  typename Base::Value  Value;
	typedef  typename Base::Reference  Reference;
	typedef  typename Base::ConstReference  ConstReference;
	typedef  typename Base::Pointer  Pointer;
	typedef  typename Base::ConstPointer  ConstPointer;
	typedef  typename Base::Size  Size;
	typedef  typename Base::Difference  Difference;

	using Super::array_;
	using Super::array_size_;
	using Super::npos;
	using Super::sarray_;
	using Super::shift_;
	using Super::shift_set;
	using Super::size_set;


public: // Creation


	/// @brief Default Constructor
	inline
	FArray1Da() :
		Super( ProxySentinel() )
	{}


	/// @brief Copy Constructor
	inline
	FArray1Da( FArray1Da const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.I_ )
	{
		shift_set( a.shift_ );
	}


	/// @brief Real Constructor
	inline
	FArray1Da( real_FArray const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.I_ )
	{
		shift_set( a.shift_ );
	}


	/// @brief Proxy Constructor
	inline
	FArray1Da( proxy_FArray const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.I_ )
	{
		shift_set( a.shift_ );
	}


	/// @brief Super Constructor
	inline
	FArray1Da( Super const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.I() )
	{
		shift_set( a.shift_ );
	}


	/// @brief Base Constructor
	inline
	FArray1Da( Base const & a ) :
		Super( a, ProxySentinel() ),
		I_( a.size() )
	{
		shift_set( 1 );
	}


	/// @brief Section Constructor
	inline
	FArray1Da( Section const & s ) :
		Super( s, ProxySentinel() ),
		I_( s.size() )
	{
		shift_set( 1 );
	}


	/// @brief Value Constructor
	inline
	FArray1Da( value_type const & t ) :
		Super( t, ProxySentinel() ),
		I_( star ) // Unbounded
	{
		shift_set( 1 );
	}


	/// @brief Copy + IndexRange Constructor
	inline
	FArray1Da( FArray1Da const & a, IR const & I_a ) :
		Super( a, ProxySentinel() ),
		I_( I_a )
	{
		dimension_argument();
	}


	/// @brief Super + IndexRange Constructor
	inline
	FArray1Da( Super const & a, IR const & I_a ) :
		Super( a, ProxySentinel() ),
		I_( I_a )
	{
		dimension_argument();
	}


	/// @brief Base + IndexRange Constructor
	inline
	FArray1Da( Base const & a, IR const & I_a ) :
		Super( a, ProxySentinel() ),
		I_( I_a )
	{
		dimension_argument();
	}


	/// @brief Section + IndexRange Constructor
	inline
	FArray1Da( Section const & s, IR const & I_a ) :
		Super( s, ProxySentinel() ),
		I_( I_a )
	{
		dimension_argument();
	}


	/// @brief Value + IndexRange Constructor
	inline
	FArray1Da( value_type const & t, IR const & I_a ) :
		Super( t, ProxySentinel() ),
		I_( I_a )
	{
		dimension_argument();
	}


	/// @brief Destructor
	inline
	virtual
	~FArray1Da()
	{}


public: // Assignment


	/// @brief Copy Assignment
	inline
	FArray1Da &
	operator =( FArray1Da const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension( a );
			Base::operator =( a );
		}
		return *this;
	}


	/// @brief Super Assignment
	inline
	FArray1Da &
	operator =( Super const & a )
	{
		if ( this != &a ) {
			if ( ! equal_dimension( a ) ) dimension( a );
			Base::operator =( a );
		}
		return *this;
	}


	/// @brief Super Assignment Template
	template< typename U >
	inline
	FArray1Da &
	operator =( FArray1DB< U > const & a )
	{
		if ( ! equal_dimension( a ) ) dimension( a );
		Base::operator =( a );
		return *this;
	}


	/// @brief += Array Template
	template< typename U >
	inline
	FArray1Da &
	operator +=( FArray1DB< U > const & a )
	{
		Super::operator +=( a );
		return *this;
	}


	/// @brief -= Array Template
	template< typename U >
	inline
	FArray1Da &
	operator -=( FArray1DB< U > const & a )
	{
		Super::operator -=( a );
		return *this;
	}


	/// @brief = Value
	inline
	FArray1Da &
	operator =( value_type const & t )
	{
		Super::operator =( t );
		return *this;
	}


	/// @brief += Value
	inline
	FArray1Da &
	operator +=( value_type const & t )
	{
		Super::operator +=( t );
		return *this;
	}


	/// @brief -= Value
	inline
	FArray1Da &
	operator -=( value_type const & t )
	{
		Super::operator -=( t );
		return *this;
	}


	/// @brief *= Value
	inline
	FArray1Da &
	operator *=( value_type const & t )
	{
		Super::operator *=( t );
		return *this;
	}


	/// @brief /= Value
	inline
	FArray1Da &
	operator /=( value_type const & t )
	{
		Super::operator /=( t );
		return *this;
	}


public: // Subscript


	/// @brief array( i ) const
	inline
	value_type const &
	operator ()( int const i ) const
	{
		assert( I_.contains( i ) );
		return sarray_[ i ];
	}


	/// @brief array( i )
	inline
	value_type &
	operator ()( int const i )
	{
		assert( I_.contains( i ) );
		return sarray_[ i ];
	}


	/// @brief Section Starting at array( i )
	inline
	Section const
	a( int const i ) const
	{
		assert( I_.contains( i ) );
		return Section( ( array_size_ != npos ) ? array_size_ - ( i - shift_ ) : npos, sarray_ + i );
	}


	/// @brief Linear Index
	inline
	size_type
	index( int const i ) const
	{
		return ( i - shift_ );
	}


public: // Predicate


	/// @brief Dimensions Initialized?
	inline
	bool
	dimensions_initialized() const
	{
		return true;
	}


	/// @brief Contains Indexed Element?
	inline
	bool
	contains( int const i ) const
	{
		return I_.contains( i );
	}


	/// @brief Initializer Active?
	inline
	bool
	initializer_active() const
	{
		return false;
	}


public: // Inspector


	/// @brief IndexRange
	inline
	IR const &
	I1() const
	{
		return I_;
	}


	/// @brief Lower Index
	inline
	int
	l1() const
	{
		return I_.l();
	}


	/// @brief Upper Index
	inline
	int
	u1() const
	{
		return I_.u();
	}


	/// @brief Size
	inline
	size_type
	size1() const
	{
		return I_.size();
	}


	/// @brief IndexRange
	inline
	IR const &
	I() const
	{
		return I_;
	}


	/// @brief Lower Index
	inline
	int
	l() const
	{
		return I_.l();
	}


	/// @brief Upper Index
	inline
	int
	u() const
	{
		return I_.u();
	}


public: // Modifier


	/// @brief Clear
	inline
	FArray1Da &
	clear()
	{
		Super::clear();
		I_.clear();
		return *this;
	}


	/// @brief Dimension by IndexRange
	inline
	FArray1Da &
	dimension( IR const & I_a )
	{
		I_.assign_value_of( I_a );
		dimension_argument();
		return *this;
	}


	/// @brief Dimension by Array
	template< typename U >
	inline
	FArray1Da &
	dimension( FArray1DB< U > const & a )
	{
		I_.assign_value_of( a.I() );
		dimension_argument();
		return *this;
	}


	/// @brief Attach to Argument Array
	inline
	FArray1Da &
	attach( FArray1Da const & a )
	{
		Base::attach( a );
		I_.assign_value_of( a.I_ );
		return *this;
	}


	/// @brief Attach to Real Array
	inline
	FArray1Da &
	attach( real_FArray const & a )
	{
		Base::attach( a );
		I_.assign_value_of( a.I_ );
		return *this;
	}


	/// @brief Attach to Proxy Array
	inline
	FArray1Da &
	attach( proxy_FArray const & a )
	{
		Base::attach( a );
		I_.assign_value_of( a.I_ );
		return *this;
	}


	/// @brief Attach to Super Array
	inline
	FArray1Da &
	attach( Super const & a )
	{
		Base::attach( a );
		I_.assign_value_of( a.I() );
		return *this;
	}


	/// @brief Attach to Base Array
	inline
	FArray1Da &
	attach( Base const & a )
	{
		Base::attach( a, 1 );
		I_ = a.size();
		return *this;
	}


	/// @brief Attach to Section
	inline
	FArray1Da &
	attach( Section const & s )
	{
		Base::attach( s, 1 );
		I_ = s.size();
		return *this;
	}


	/// @brief Attach to Value
	inline
	FArray1Da &
	attach( value_type const & t )
	{
		Base::attach( t, 1 );
		I_ = star; // Unbounded
		return *this;
	}


	/// @brief Detach from Source Array
	inline
	FArray1Da &
	detach()
	{
		Base::detach();
		I_.clear();
		return *this;
	}


protected: // Functions


	/// @brief Dimension by IndexRange
	inline
	void
	dimension_assign( SIR const & I_a )
	{
		I_.assign_value_of( I_a );
		dimension_argument();
	}


private: // Functions


	/// @brief Setup for Current IndexRange Dimensions
	inline
	void
	dimension_argument()
	{
		if ( I_.bounded_value() ) { // Bounded
			size_set( I_.size() );
		} else if ( array_size_ != npos ) { // Unbounded with bounded data array
			// Infer upper index and size
			I_.u( I_.lz() + array_size_ - 1 );
			size_set( I_.size() );
		} else { // Unbounded with unbounded data array
			size_set( npos );
		}
		shift_set( I_.lz() );
	}


private: // Data


	/// @brief Index range
	IR I_;


}; // FArray1Da


} // namespace ObjexxFCL


#endif // INCLUDED_ObjexxFCL_FArray1Da_HH
