// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/exit.hh
/// @brief  Program exit functions and macros
/// @author David Kim (dekim@u.washington.edu)
/// @author Sergey Lyskov (Sergey.Lyskov@jhu.edu)
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
///
/// @note The point of these is:
///  @li  Show the file and line number where the exit originates
///  @li  Optionally show a message about why the exit occurred
///  @li  Provide a core dump on Linux/UNIX so a post-mortem backtrace
///       can be performed
///  @li  Provide macro functions to add the file and line for you
/// @note Break on utility::exit when debugging to allow you to
///       get a backtrace from the point of exit


#ifndef INCLUDED_utility_exit_HH
#define INCLUDED_utility_exit_HH


// C++ headers
#include <string>


/// @brief Macro function wrappers for utility::exit
///
/// @note Convenience macros that fills in the file and line
/// @note These have to be macros to get the file and line from the point of call
/// @note Don't use variadic macros to reduce the "overloads": They aren't standard C++


/// @brief Exit with file + line
#define utility_exit() utility::exit( __FILE__, __LINE__ )


/// @brief Exit with file + line + message
///
/// @note The m argument is a message string
#define utility_exit_with_message(m) utility::exit( __FILE__, __LINE__, m )


/// @brief Exit with file + line + status
///
/// @note The s argument is a status value
#define utility_exit_with_status(s) utility::exit( __FILE__, __LINE__, s )


/// @brief Exit with file + line + message + status
///
/// @note The m argument is a message string
/// @note The s argument is a status value
#define utility_exit_with_message_status(m,s) utility::exit( __FILE__, __LINE__, m, s )


namespace utility {


/// @brief Exit with file + line + message + optional status
void
exit(
	std::string const & file,
	int const line,
	std::string const & message,
	int const status = 1
);


/// @brief Exit with file + line + optional status
inline
void
exit(
	std::string const & file,
	int const line,
	int const status = 1
)
{
	utility::exit( file, line, std::string(), status );
}


/// @brief Exit with file + line + status
///
/// @note  Deprecated: For backwards compatibility with earlier version
inline
void
exit(
	int const status,
	std::string const & file,
	int const line
)
{
	utility::exit( file, line, std::string(), status );
}


} // namespace utility


#endif // INCLUDED_utility_exit_HH
