// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 30396 $
//  $Date: 2009-06-10 14:48:53 -0700 (Wed, 10 Jun 2009) $
//  $Author: aroop $


// Rosetta Headers
#include "status.h"
#include "after_opts.h"
#include "antibody_modeling.h"
#include "cenlist.h"
#include "constraints.h"
#include "counters.h"
#include "decoystats.h"
#include "decoystats_interface.h"
#include "decoystats_ns.h"
#include "design.h"
#include "dipolar.h"
#include "disulfides.h"
#include "dock_loops.h"
#include "dock_pivot.h"
#include "dock_structure.h"
#include "docking.h"
#include "dock_loop_ensemble.h"
#include "docking_ns.h"
#include "domins_ns.h"
#include "dummy_model.h"
#include "files_paths.h"
#include "filters.h"
#include "force_barcode.h"
#include "fullatom.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "initialize.h"
#include "ligand.h"
#include "ligand_ns.h"
#include "loops.h"
#include "loop_relax.h"
#include "maxsub.h"
#include "misc.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "packing_measures.h"
#include "param.h"
#include "param_pack.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "pose_benchmark.h"
#include "ramachandran.h"
#include "relax.h"
#include "relax_structure.h"
#include "saxs_model.h"
#include "shotgun.h"
#include "score_ns.h"
#include "score.h"
#include "start.h"
#include "structure.h"
#include "taboo_search.h"
#include "void.h"

#include "monte_carlo.h" // yab: misc removal

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/Time_Date.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <cstdlib>
#include <ctime>
#include <iostream>


// Namespaces
namespace status_info {
	bool status_init;
	bool fastatus_init;
}
namespace scorefile_info {
	float rms_to_start;
	int nalign;
	float maxsub_rms;
	float logEvalue;
	float rms_local;
	float splicemsd;
	float ngap;
	float cgap;
	float decoy_time;
	float chain_break_score;
	float nsd_to_dummy_model;
	std::string filter_tag("tag");
}

// an easy way to add new score columns...
namespace score_file_ns {
	bool use_new_columns = false;
	std::map<std::string,float> new_float_cols;
	std::map<std::string,int> new_int_cols;
	std::map<std::string,std::string> new_string_cols;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin save_status_info
///
/// @brief save data corresponding to the most recent conformational move
///     for later use in status and scorefile output
///
/// @detailed
///
/// @param[in]   type  - in - type of move being made
/// @param[in]   start - in - residue number where most recent change has been made
/// @param[in]   size  - in - size of fragment insert in most recent change
///
/// @global_read none
///
/// @global_write counters.h
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
save_status_info(
	std::string const & type,
	int start,
	int size
)
{
	using namespace counters;

	insert_pt = start;
	insert_size = size;
	move_type = type;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_status_info
///
/// @brief erase the most recent move type and location
///
/// @detailed
///
/// @global_read
///
/// @global_write counters.h
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_status_info()
{
	using namespace counters;

	insert_pt = 0;
	insert_size = 0;
	move_type = "unknown";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_status_file
///
/// @brief Open a 'status file' to record scores at intermediate times
///     during the calculations
///
/// @detailed The status fil is activated with the -status flag.  This function
///     opens the file and calls the header function
///
/// @param[in]   fullatom - in - flag to indicate fullatom mode
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
utility::io::ozstream *
initialize_status_file( bool fullatom )
{
	using namespace files_paths;
	using namespace status_info;

//car local
	std::string fullname;
	std::string filename;

	if ( !use_status ) {
		return 0;
	}

	if ( fullatom ) {
		if ( fastatus_init ) return &fa_status_x;
	} else {
		if ( status_init ) return &status_x;
	}

	retrieve_decoy_name(filename);
	status_file = filename;
	if ( status_file == "initialize" ) {
		return 0;
	}

	utility::io::ozstream * iunit_p;

	if ( fullatom ) {
		fastatus_init = true;
		fullname = status_path + status_file + ".fastatus";
		std::cout << "fullatom status file: " << fullname << std::endl;
		fa_status_x.clear();
		fa_status_x.open( fullname );
		if ( !fa_status_x ) {
			std::cout << "Open failed for file: " << fullname << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		iunit_p = &fa_status_x;
	} else {
		status_init = true;
		fullname = status_path + status_file + ".status";
		std::cout << "status file: " << fullname << std::endl;
		status_x.clear();
		status_x.open( fullname );
		if ( !status_x ) {
			std::cout << "Open failed for file: " << fullname << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		iunit_p = &status_x;
	}
	utility::io::ozstream & iunit( *iunit_p );

	iunit << "mc" << "  nlow" << " inpos";
	iunit << "   temp";
	score_header_output(iunit,fullatom); iunit << "\n";

	return iunit_p;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin close_status_file
///
/// @brief closes the status file
///
/// @detailed
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
close_status_file()
{
	using namespace files_paths;
	using namespace status_info;

	fa_status_x.close();
	fa_status_x.clear();
	status_x.close();
	status_x.clear();

	status_init = false;
	fastatus_init = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_status_file
///
/// @brief output scores for the current structure to the status file
///
/// @detailed it is appropriate to call this function anywhere you would like
///     information about the state of the protein structure.  Call scorefxn()
///     before calling this function.
///
/// @param[in]   cycle_number - in - Monte Carlo cycle number
/// @param[in]   monte_carlo_accept - in - Was the last move accepted? (see codes in
///     monte_carlo.f (insert hyperlink)
/// @param[in]   temperature - in - for simulated annealing runs
///
/// @global_read move_type, monte_carlo info
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
output_status_file(
	int monte_carlo_accept,
	float temperature
)
{
	using namespace counters;
	using namespace files_paths;

	bool full_atom = get_fullatom_flag();
	if ( !use_status ) return;
	if ( move_type == "initialize" ) return;

	utility::io::ozstream * iunit_p = initialize_status_file( full_atom );

	if ( !iunit_p ) return; // failed to open file

	utility::io::ozstream & iunit( *iunit_p );

	update_scorefile_info();

	iunit << I( 2, monte_carlo_accept ) << ' ' <<
	 I( 5, n_low_accept ) << ' ' << I( 5, insert_pt );
	iunit << ' ' << F( 6, 1, temperature );
	score_output(iunit,move_type,full_atom); iunit << "\n";

}

///////////////////////////////////////////////////////////////////////////////
/// @begin score_header_output
///
/// @brief print the scorefile/statusfile header
///
/// @detailed see also score_output
///
/// @param[out]   iunit - in - output unit number
/// @param[in]   fullatom - in - flag
///
/// @global_read mode flags
///
/// @global_write none
///
/// @remarks
///
/// jjg 7/17/2001
/// car 11/27/01
///
///     functions for outputting scores
///       score_header_output  - print the header
///       score_output         - print the scores
///
///     these functions should be called for all cases where horizontal
///       output of scores is desired
///
///     the format is designed to be used by Splus (or R)
///     For example, to read in the matrix of data and plot rms vs score, do:
///
///       Splus> mydata _ read.table("aa1brs_0001.fastatus",header=T)
///       Splus> plot(mydata$rms,mydata$score)
///
///     Note that Splus likes a consistant number of columns, so don't put
///     spaces in the status line tags
///
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified Aroop 07/19/2007
///////////////////////////////////////////////////////////////////////////////
void
score_header_output(
	std::ostream & iunit,
	bool fullatom
)
{
	using namespace files_paths;
	using namespace relax_options;

//jg local
	bool const native_exists = get_native_exists();
	bool const dock_flag = (multi_chain && !design::dna_interface &&
													!antibody_modeler);
	bool const loop_flag = get_loop_flag();
	bool const pose_loop_flag = get_pose_loop_flag();
	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag();
	bool const decoystats_flag = get_decoystats_flag();
	bool const domins_flag = domain_insertion;
	bool const antibody_flag = antibody_modeler;

// score
	iunit << A( 10, "score" );

// antibody modeler
	if ( antibody_flag ) {
		iunit << A( 6, "rmsg"); // aroop cdr_h3 rms global
		// hydrophobic score
		iunit << A( 10, "hydrphbc" );
		// L chain rms
		iunit << A( 6, "rms" );
		// cdr h3 base rmsg: B for Base
		iunit << A( 6, "Brms");
		// cdr h3 non-base rmsg: R for Rest of the loop
		iunit << A( 6, "Rrms");
		// cutpoint separation
		iunit << A( 6, " Gap");
		// Does decoy match base dictated by sequence
		iunit << A( 4, "H3");
		// Nitrogen Oxygen distance in kink
		iunit << A( 6, "N--O");
		// Number of Hydrogen bonds in CDR
		iunit << A( 4, "HH");
		// Does decoy match stem dictated by sequence
		iunit << A( 4, "St");
	}

// docking
	if ( dock_flag ) {
		iunit << A( 7, "rms" );
		if( docking::dle_loops_flag )
			iunit <<
				' ' << A( 6, "rmsg"); // aroop output loop rms global header
		if( docking::docking_ensemble_flag ){
			iunit <<
					' ' << A( 6, "c_num1") <<
					' ' << A( 6, "c_num2") <<
					' ' << A( 6, "nscore"); //sid output conformer# from ensemble
					}
		iunit <<
			A( 7, "Fnat" ) <<
			A( 7, "Fnon" ) <<
			A( 7, "I_rms") <<
			A( 7, "I_sc");

		if ( fullatom ) iunit << " ncont";
		if ( get_flexbb_docking_flag() )
			iunit<<	A( 7, "p1_rms" ) << A( 7, "p2_rms" );
		iunit <<
		 A( 7, "trans" ) <<
		 A( 7, "rot" );
		iunit <<
		 A( 7, "st_rms" ) <<
		 A( 7, "cenrms" );
		if ( fullatom ) {
			iunit << A( 10, "sc_before" );
			iunit << A( 10, "sc_after1" );
			iunit << A( 10, "sc_after5" );
		}
		iunit <<
		 A( 8, "d_env" ) <<
		 A( 8, "d_pair" ) <<
		 A( 8, "contact" ) <<
		 A( 8, "d_vdw" ) <<
		 A( 8, "d_sc" ) <<
		 A( 8, "d_fab" );
	}

// ligand
	if ( get_ligand_flag() ) {
		iunit <<
		 A( 8, "ligsum" ) <<
		 A( 8, "ligrep" ) <<
		 A( 8, "ligatr" ) <<
		 A( 8, "ligsol" ) <<
		 A( 8, "ligcou" ) <<
		 A( 8, "lighb" ) <<
		 A( 8, "ligvir" ) <<
		 A( 8, "ligh2o" ) <<
		 A( 8, "ligsas" ) <<
     A( 8, "liginta" ) <<
     A( 8, "ligintr" ) <<
     A( 8, "ligints" ) <<
     A( 8, "ligintd" ) <<
     A( 8, "ligintc" ) <<
     A( 8, "liginth" ) <<
		 A( 8, "cont" ) <<
		 A( 4, "saD" ) <<
		 A( 4, "unD" ) <<
		 A( 4, "saA" ) <<
		 A( 4, "unA" ) <<
		 A( 8, "l1_nbrs" );
	}

// backbone
	iunit <<
	 A( 8, "env" ) <<
	 A( 8, "pair" ) <<
	 A( 8, "vdw" ) <<
	 A( 8, "hs" ) <<
	 A( 8, "ss" ) <<
	 A( 8, "sheet" ) <<
	 A( 8, "cb" ) <<
	 A( 8, "rsigma" ) <<
	 A( 8, "hb_srbb" ) <<
	 A( 8, "hb_lrbb" ) <<
	 A( 8, "rg" );

	if ( domins_flag ) {
		iunit << A( 8, "cen_sco" ) <<
			A( 8, "chbreak" ) <<
			A( 8, "st_rms" ) <<
			A( 8, "cen_rms" ) <<
			A( 8, "ins_rms" ) <<
			A( 8, "lp_rms" ) <<
			A( 8, "rms" ) <<
			A( 8, "contact" ) <<
			A( 8, "reject" );
	}

	if ( structure::BOUNDARY::get_separate_centroidpackscore_flag() ) {
		iunit << A( 8, "cpack" );
	}

	if ( !dock_flag || get_flexbb_docking_flag() )
		if ( !domins_flag ) {
			iunit <<
			A( 8, "co" ) <<
			A( 8, "contact" ) <<
			A( 8, "rama" );
		}

// fullatom
	if ( fullatom ) {
		iunit <<
			A( 9, "bk_tot" ) <<
			A( 9, "fa_atr" ) <<
			A( 9, "fa_rep" ) <<
			A( 9, "fa_sol" ) <<
			A( 9, "h2o_sol" ) <<
			A( 9, "hbsc" ) <<
			A( 9, "fa_dun" ) <<
			A( 9, "fa_intra" ) <<
			A( 9, "fa_pair" ) <<
			A( 9, "fa_plane" ) <<
			A( 9, "fa_prob" ) <<
			A( 9, "fa_h2o" ) <<
			A( 9, "h2o_hb" );

		iunit << A( 9, "gsolt" );
		iunit << A( 9, "sasa" );
		//iunit << A( 9, "sasaprob" ); //sheffler

//vatsan The sasa term will offset the rms column in the scorefile
//$$$	if ( dock_flag ) iunit <<
//$$$	 A( 10, "d_Eatr" ) <<
//$$$	 A( 10, "d_Erep" ) <<
//$$$	 A( 10, "d_Eatr_lr" ) <<
//$$$	 A( 10, "d_Erep_lr" );
		if ( dock_flag && !get_simple_elec() ) iunit << A( 10, "d_elec" );

    if ( get_simple_elec() )
      iunit << A( 10, "fa_elec" );
		if ( get_pose_symmetry_docking_flag() ) iunit << A( 15, "symm_mon_score" );
    if ( 0 ) //dock_flag ) // SJF For Ora's FunHunt
      iunit << A( 10, "scr10d_min1" ) << A(10, "scr10d_min5" );
	}

// constraints
	if ( classical_constraints::BOUNDARY::get_constraints_exist() ) iunit <<
	 A( 8, "pc" ) <<
	 A( 8, "pc_viol" );
	if ( get_dipolar_exist() ) iunit <<
	 A( 8, "dipolar" ) << ' ' <<
	 A( 7, "projection" );

	if ( barcode_exist() ) iunit <<
													 A( 8, "barcst" ) <<
													 A( 8, "barenrg");

// taboo_score
//
	if ( get_taboo_mode_exist() ) iunit <<
	 A( 8, "taboo");

	if ( score_get_vary_omega()) iunit << A( 9, "omega_sc") ;

// rms
	if ( native_exists ) {
		if ( !dock_flag && !domins_flag ) {
			iunit <<
			 A( 4, "rms" ) <<
			 A( 8, "rms_min" );
			if ( require_start ) iunit << A( 6, "srms" );
			if ( record_rms_before_relax ) iunit << A( 6, "irms" );
			iunit <<
			 A( 4, "mxn" ) <<
			 A( 6, "mxrms" ) <<
			 A( 6, "mxlgE" ) <<
			 A( 6, "%cont" ) <<
			 A( 6, "mm11" ) <<
			 A( 6, "mm22" ) <<
			 A( 6, "mm33" ) <<
			 A( 6, "mm43" ) <<
			 A( 6, "mm74" ) <<
			 A( 7, "gdtmm" );
		}
		if ( dock_flag && loop_flag ) iunit << A( 7, "lp_rms" );
		if ( loop_flag ) iunit << A( 6, "rms_l" );
	}

	// maxsub to server_models
	if (get_maxsub_to_server_models_flag())
		maxsub_to_server_models_output_header( iunit );

	if ( loop_flag ) iunit << ' ' <<
		A( 5, "splice" ) <<
		A( 6, "ngap" ) <<
		A( 6, "cgap" );
	if( get_looprlx_exist() || pose_loop_flag ) iunit <<
		A( 8, "cbreak" );
	if( files_paths::use_homolog_env_stats ) iunit <<
		A( 8, "homenv" );
	if( files_paths::use_homolog_pair_stats ) iunit <<
		A( 8, "hompair" );

//bs disulfides
	if ( disulf_flag ) {
		iunit <<
		 A( 5, "Ncen" ) <<
		 A( 9, "censc" ) <<
		 A( 9, "cbsc" ) <<
		 A( 9, "bbdih" ) <<
		 A( 9, "cacbcb" ) <<
		 A( 9, "cacbcbca" );
		if ( fullatom ) {
			iunit <<
			 A( 4, "Nfa" ) <<
			 A( 9, "dist" ) <<
			 A( 9, "csang" ) <<
			 A( 9, "dihed" ) <<
			 A( 9, "cadih" );
		}
	}

//bs jumping
	//if ( get_jumping_flag() ) {
	//	iunit << A( 9, "jmpclose" );
	//}

	if ( get_pH_packing_flag() ) iunit << A( 9, "pH" );

	if ( ( ! get_interface_ds_flag() ) && decoystats::tight_core_analysis )
		iunit << A( 17, "tight_core_atoms" );

//car time per decoy  (includes repeats due to failure)
	if ( use_timer ) iunit << A( 6, "time" );
	if ( count_attempts ) iunit << A(8, "attmpt");

//pb decoystats stuff at the end
	if ( decoystats_flag ) decoystats_score_header_output(iunit);

	if (get_output_flavor())
		iunit << A(60,"flavor") ;

	if (get_dummy_model_flag())
		iunit << A(6, "nsd") << A(10, "nsdscore");

	if (get_saxs_model_flag())
		iunit << A(6, "saxs");

	if (get_output_filter())
		iunit << A(14,"filter") ;

	if (get_auto_relax_score_filter())
		iunit << ' ' << A(8,"rlxfilt1") << ' ' << A(8,"rlxfilt2") ;

	if ( get_docking_silent_input_flag() ) {
		iunit << " dock_silent_input_tag";
	}

	// packing scores
	if ( score_file_ns::use_new_columns ) {
		using namespace std;
		using namespace score_file_ns;
		for ( map<string,float>::iterator ii = new_float_cols.begin();
					ii != new_float_cols.end(); ii++ )
						iunit << " " << A(max( 9 , (int)ii->first.size()), ii->first );
		for ( map<string,int>::iterator ii = new_int_cols.begin();
					ii != new_int_cols.end(); ii++ )
						iunit << " " << A(max( 6 , (int)ii->first.size()), ii->first );
		for ( map<string,string>::iterator ii = new_string_cols.begin();
					ii != new_string_cols.end(); ii++ )
						iunit << " " << A(max( (int)ii->second.size() , (int)ii->first.size()), ii->first );
	}

	//user defined tag
	if ( get_user_score_tag() != "")  {
		iunit << ' ' << A(get_user_score_tag_length(), "user_tag");
	}

// description
	iunit << " description" ; //<< '\n'; //std::endl;

}

///////////////////////////////////////////////////////////////////////////////
/// @begin score_output
///
/// @brief output the current scores
///
/// @detailed outputs the scores which are appropriate for the current mode and
///           flags. see also score_header_output
///
/// @param[out]   iunit - in - output unit
/// @param[in]   tag - in - label for the score line
/// @param[in]   fullatom - in - flag
///
/// @global_read scores!
///
/// @global_write none
///
/// @remarks
/// jjg 7/17/2001
/// car 11/27/01
///
///     functions for outputting scores
///       score_header_output  - print the header
///       score_output         - print the scores
///
///     these functions should be called for all cases where horizontal
///       output of scores is desired
///
///     the format is designed to be used by Splus (or R)
///     For example, to read in the matrix of data and plot rms vs score, do:
///
///       Splus> mydata _ read.table("aa1brs_0001.fastatus",header=T)
///       Splus> plot(mydata$rms,mydata$score)
///
///     Note that Splus likes a consistant number of columns, so don't put
///     spaces in the status line tags
///
///
///     note: no score evaluations should be done here!!!!  this function
///     must run quickly to avoid tying up the NFS file handle
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified Aroop Sircar 07/19/2007
///////////////////////////////////////////////////////////////////////////////
void
score_output(
	std::ostream & iunit,
	std::string const & tag,
	bool fullatom
)
{
	using namespace files_paths;
	//using namespace ligand;
	//using namespace misc;
	using namespace scorefile_info;
	using namespace scores;
	//using namespace scorefxns;
	using namespace relax_options;

	using namespace mc_global_track::mc_score; // yab: misc removal
	using namespace mc_global_track::diagnose; // yab: misc removal

	using namespace disulfides::BOUNDARY;

//jg local
	bool const dock_flag = (multi_chain && !design::dna_interface &&
													!antibody_modeler);
	bool const loop_flag = get_loop_flag();
	bool const pose_loop_flag = get_pose_loop_flag();
	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag();
	bool const native_exists = get_native_exists();
	bool const decoystats_flag = get_decoystats_flag();
	bool const domins_flag = domain_insertion;
	bool const antibody_flag = antibody_modeler;

	if ( has_prefix( tag, "native" ) ) rms_to_start = 0.0;

// score
	iunit << F( 10, 2, score );

// antibody modeler
	if ( antibody_flag ) {
		iunit << F( 6, 2, antibody_modeling_h3_rmsg()); // aroop cdr_h3 rms global
		// hydrophobic score
		iunit << F( 10, 2, dle_hydrophobic());
		// L chain rms
		iunit << F( 6, 2, antibody_modeling_L_rms());
		// cdr h3 base rmsg
		iunit << F( 6, 2, antibody_modeling_h3_base_rmsg());
		// cdr h3 non-base rmsg
		iunit << F( 6, 2, antibody_modeling_h3_non_base_rmsg());
		// separation at cutpoint
		iunit << F( 6, 2, dle_cutpoint_separation());
		// Does decoy match base dictated by sequence
		iunit << I( 4, antibody_modeling_h3_match());
		// Nitrogen Oxygen distance in kink
		iunit << F( 6, 2,antibody_modeling_h3_NE1_n_plus_1_O_n_minus_2_distance());
		// Number of hydrogen bonds in cdr
		iunit << I( 4, antibody_modeling_h3_hydrogen_ladder());
		// Does decoy match stem dictated by sequence
		iunit << I( 4, antibody_modeling_h3_stem_match());
	}

// docking
	if ( dock_flag ) {
		iunit <<
			' ' << F( 6, 2, rms_err );
		// aroop output loop rms global
		if( docking::dle_loops_flag )
			iunit <<
				' ' << F( 6, 2, dle_first_loop_rmsg());
		if( docking::docking_ensemble_flag ){
			using namespace docking;
			iunit <<
				' ' << I( 6, 2, current_conformer(1)) <<
				' ' << I( 6, 2, current_conformer(2));
			if(ensemble_read(1) || ensemble_read(2)){
				iunit <<' ' << F(6, 2, (score - FA_ref_energy(1,current_conformer(1)) - FA_ref_energy(2,current_conformer(2)))); // sid output conformer # from ensemble
				} else {
					iunit <<' ' << F(6, 2, score); // sid output conformer # from ensemble
					}
				}
		iunit <<
			' ' << F( 6, 3, get_docking_frac_nat_contact() ) <<
			' ' << F( 6, 3, get_docking_frac_nonnat_contact() ) <<
			' ' << F( 6, 2, get_docking_interf_rmsd() ) <<
			' ' << F( 6, 2, get_docking_interf_energy() );
		if ( fullatom ) iunit << ' ' << I( 5, get_countcont() );
		if ( get_flexbb_docking_flag() )
			iunit << ' ' << F( 6, 2, calc_docking_partner_rmsd(1))
						<< ' ' << F( 6, 2, calc_docking_partner_rmsd(2));
		iunit <<
		 ' ' << F( 6, 2, get_docking_translation_dist() ) <<
		 ' ' << F( 6, 2, get_docking_rotation_angle() ) <<
		 ' ' << F( 6, 2, docking_get_initial_rms() ) <<
		 ' ' << F( 6, 2, docking_get_cenmode_rms() );
		if ( fullatom ) {
			iunit <<
			 ' ' << F( 9, 2, docking_get_score_before_mcm() ) <<
			 ' ' << F( 9, 2, docking_get_score_after_one_min() ) <<
			 ' ' << F( 9, 2, docking_get_score_after_5_mcm() );
		}
		iunit <<
		 ' ' << F( 7, 2, docking_env_score ) <<
		 ' ' << F( 7, 2, docking_pair_score ) <<
		 ' ' << F( 7, 2, docking_contact_score ) <<
		 ' ' << F( 7, 2, docking_vdw_score ) <<
		 ' ' << F( 7, 2, docking_site_constraint_score ) <<
		 ' ' << F( 7, 2, docking_fab_score );
	}

// ligand
	if ( get_ligand_flag() ) {
		int don_sat, don_non, acc_sat, acc_non, lig_contacts;
		count_satisfied_ha_hbond( don_sat, don_non, acc_sat, acc_non,(*ligand::ligand_one) );
		calc_number_of_ligand_contacts( lig_contacts ,(*ligand::ligand_one));
		iunit <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_sumE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_repE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_atrE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_solE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_couE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_hbE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_virE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_h2oE ) <<
		 ' ' << F( 7, 2, ligand::ligand_one->lig_sasE ) <<
     ' ' << F( 7, 2, ligand::ligand_one->lig_int_atrE ) <<
     ' ' << F( 7, 2, ligand::ligand_one->lig_int_repE ) <<
     ' ' << F( 7, 2, ligand::ligand_one->lig_int_solvE ) <<
     ' ' << F( 7, 2, ligand::ligand_one->lig_int_dihedE ) <<
     ' ' << F( 7, 2, ligand::ligand_one->lig_int_coulombE ) <<
     ' ' << F( 7, 2, ligand::ligand_one->lig_int_hbE  );
		iunit << ' ' << I( 7, lig_contacts );
		iunit <<
		 ' ' << I( 3, don_sat ) <<
		 ' ' << I( 3, don_non ) <<
		 ' ' << I( 3, acc_sat ) <<
		 ' ' << I( 3, acc_non ) <<
		 ' ' << I( 7, ligand::ligand_one->lig_neighbors );
}

// backbone
	iunit <<
	 ' ' << F( 7, 2, env_score ) <<
	 ' ' << F( 7, 2, pair_score ) <<
	 ' ' << F( 7, 2, vdw_score ) <<
	 ' ' << F( 7, 2, hs_score ) <<
	 ' ' << F( 7, 2, ss_score ) <<
	 ' ' << F( 7, 2, sheet_score ) <<
	 ' ' << F( 7, 2, cb_score ) <<
	 ' ' << F( 7, 2, rsigma_score ) <<
	 ' ' << F( 7, 2, hb_srbb_score ) <<
	 ' ' << F( 7, 2, hb_lrbb_score ) <<
	 ' ' << F( 7, 2, rg );

	if ( domins_flag ) {
		iunit <<
			' ' << F( 7, 2, domins_ns::centroid_score ) <<
			' ' << F( 7, 2, chainbreak_score ) <<
			' ' << F( 7, 2, domins_ns::start_rmsd ) <<
			' ' << F( 7, 2, domins_ns::centroid_rmsd ) <<
			' ' << F( 7, 2, domins_ns::insert_rmsd ) <<
			' ' << F( 7, 2, domins_ns::loop_rmsd ) <<
			' ' << F( 7, 2, domins_ns::rmsd_to_native ) <<
			' ' << F( 7, 2, domain_contact_score ) <<
			' ' << I( 7, domins_ns::centroid_rejected );
	}

	if ( structure::BOUNDARY::get_separate_centroidpackscore_flag() ) {
		iunit <<
			' ' << F( 7, 2, centroid_pack_score );
	}

	if ( !dock_flag || get_flexbb_docking_flag() ) {
		if ( !domins_flag ) {
			iunit <<
			 ' ' << F( 7, 2, co ) <<
			 ' ' << F( 7, 2, contact_prediction ) <<
			 ' ' << F( 7, 2, ramachandran_score );
		}
	}

// fullatom
	if ( fullatom ) {

		iunit <<
			' ' << F( 8, 2, get_fullatom_totalE() ) <<
			' ' << F( 8, 2, fa_atr_score ) <<
			' ' << F( 8, 2, fa_rep_score ) <<
			' ' << F( 8, 2, fa_solv_score ) <<
			' ' << F( 8, 2, fa_h2o_solv_score ) <<
			' ' << F( 8, 2, hb_sc_score ) <<
			' ' << F( 8, 2, fa_dun_score ) <<
			' ' << F( 8, 2, fa_intrares_score ) <<
			' ' << F( 8, 2, fa_pair_score ) <<
			' ' << F( 8, 2, fa_plane_score ) <<
			' ' << F( 8, 2, fa_prob1b_score ) <<
			' ' << F( 8, 2, fa_h2o_score ) <<
			' ' << F( 8, 2, fa_h2o_hb_score );

		iunit << ' ' << F( 8, 2, get_ex_gsolt() );
		iunit << ' ' << F( 8, 2, get_ex_sasa_gsolt() );
		//iunit << ' ' << F( 8, 3, retrieve_avg_sasa_prob() ); //sheffler

    if( get_simple_elec() )
      iunit << ' ' << F( 9, 3, fa_elec_score );

		if ( dock_flag ) {
      if( !get_simple_elec() )
			  iunit << ' ' << F( 9, 3, docking_warshel_elec_score );
			if ( get_pose_symmetry_docking_flag() ) {
			        iunit << ' ' << F( 9, 3, symm_mon_score );
		  }
		}
    if ( 0 ) //dock_flag ) // SJF For Ora's FunHunt
      iunit << ' ' << F( 9, 3, docking::docking_score10d_min_after1 ) << ' ' << F( 9, 3, docking::docking_score10d_min_after5 );
	}

// constraints
	if ( classical_constraints::BOUNDARY::get_constraints_exist() ) {
		iunit <<
		 ' ' << F( 7, 2, pc_score ) <<
		 ' ' << F( 7, 2, pc_viol );
	}

	if ( get_dipolar_exist() ) {
		iunit <<
		 ' ' << F( 7, 2, dipolar_score ) <<
		 ' ' << F( 7, 2, projection_score );
	}

	if ( barcode_exist() ) {
		iunit <<
			' ' << F( 7, 2, barscore) <<
			' ' << F( 7, 2, bar_energy_score);
	}

// taboo
//
	if (get_taboo_mode_exist() ) {
		iunit <<
			' ' << F( 7, 2, taboo_best);
	}

	if (score_get_vary_omega() ) {
		iunit <<
			' ' << F( 8, 2, omega_score);
	}
// rms
	if ( native_exists ) {
		if ( !dock_flag && !domins_flag ) {
			iunit <<
			 ' ' << F( 5, 2, rms_err ) <<
			 ' ' << F( 5, 2, rms_min );
			if ( require_start ) iunit << ' ' << F( 5, 2, rms_to_start );
			if ( record_rms_before_relax ) iunit << ' ' << F(5, 2, rms_before_relax);
			iunit <<
			 ' ' << I( 3, nalign ) <<
			 ' ' << F( 5, 2, maxsub_rms ) <<
			 ' ' << F( 5, 1, logEvalue ) <<
			 ' ' << F( 5, 2, fraction_native_contacts() );

			float mm11, mm22, mm33, mm43, mm74, gdtmm;
			get_gdtmm_scores( mm11, mm22, mm33, mm43, mm74, gdtmm );
			iunit << F( 6, 3, mm11 )
				<< F( 6, 3, mm22 )
				<< F( 6, 3, mm33 )
				<< F( 6, 3, mm43 )
				<< F( 6, 3, mm74 )
				<< F( 7, 3, gdtmm );
		}

		if ( loop_flag && dock_flag ) iunit << ' ' <<
		 F( 6, 2, get_dock_loop_rmsd() );
		if ( loop_flag ) iunit << ' ' << F( 5, 2, rms_local );
	}

	// maxsub to server_models
	if (get_maxsub_to_server_models_flag()){
		maxsub_to_server_models_output( iunit );
	}

	if ( loop_flag ) {
		iunit <<
		 ' ' << I( 5, static_cast< int >( splicemsd ) ) <<
		 ' ' << F( 5, 2, ngap ) <<
		 ' ' << F( 5, 2, cgap );
	}


	if( get_looprlx_exist() ) {
		iunit << ' ' << F( 7, 2, chain_break_score );
	} else if ( pose_loop_flag ) {
		iunit << ' ' << F( 7, 2, scores::current_scores::chainbreak_score );
	}

//bqian homolog stats scores
	if( files_paths::use_homolog_env_stats )
		iunit << ' ' << F( 7, 2, scores::current_scores::homolog_env_score );
	if( files_paths::use_homolog_pair_stats )
		iunit << ' ' << F( 7, 2, scores::current_scores::homolog_pair_score );

//bs disulfides
	if ( disulf_flag ) {
		iunit <<
		 ' ' << I( 4, get_n_disulf_cent_natcut() ) <<
		 ' ' << F( 8, 2, disulf_cendist_score ) <<
		 ' ' << F( 8, 2, disulf_cbdist_score ) <<
		 ' ' << F( 8, 2, disulf_bb_dih_score ) <<
		 ' ' << F( 8, 2, disulf_cacbcb_score ) <<
		 ' ' << F( 8, 2, disulf_cacbcbca_score );
		if ( fullatom ) {
			iunit <<
			 ' ' << I( 3, get_n_disulf_fa() ) <<
			 ' ' << F( 8, 2, get_fullatom_disulf_dist_sc() ) <<
			 ' ' << F( 8, 2, get_fullatom_disulf_csang_sc() ) <<
			 ' ' << F( 8, 2, get_fullatom_disulf_dihedral_sc() ) <<
			 ' ' << F( 8, 2, get_fa_disulf_ca_dihed_sc() );
		}
	}

//pb jumping chainbreak score (saved from the last evaluatation)
	//if ( get_jumping_flag() ) {
	//	iunit << ' ' << F( 8, 3, get_jump_close() );
	//}

	if ( get_pH_packing_flag() ) iunit << ' ' << F( 8, 2, get_current_pH() );

	if ( ( ! get_interface_ds_flag() ) && decoystats::tight_core_analysis )
		iunit << I( 16, find_tight_core_packing(false) );

//car time
	if ( use_timer ) iunit << ' ' << F( 5, 1, decoy_time );

	if ( count_attempts ) iunit << ' ' << I( 7, get_decoy_attempt_counter() );

	if ( decoystats_flag ) decoystats_score_output(iunit);

	if (get_output_flavor())
			barcode_flavor_output(iunit);

	if (get_dummy_model_flag())
		iunit << ' ' << F(5, 3, nsd_to_dummy_model ) <<
			' ' << F(9, 3, dummy_model_score);

	if (get_saxs_model_flag())
		iunit << ' ' << F(5, 3, saxs_model_score );

	if (get_output_filter())
		iunit << A(14, filter_tag) ;

	if (get_auto_relax_score_filter()){
		iunit <<
			' ' << F(8, 2, get_relax_score_filter1() ) <<
			' ' << F(8, 2, get_relax_score_filter2() );
	}

	if ( get_docking_silent_input_flag() ) {
		iunit << ' ' << A( 21, "in_" + docking_silent_input_get_current_tag( false ));
	}

// additional scores from vars in score_file_ns
	if ( score_file_ns::use_new_columns ) {
		using namespace std;
		using namespace score_file_ns;
		for ( map<string,float>::iterator ii = new_float_cols.begin();
					ii != new_float_cols.end(); ii++ )
						iunit << " " << F( max(9,(int)ii->first.size()), 3,  ii->second );
		for ( map<string,int>::iterator ii = new_int_cols.begin();
					ii != new_int_cols.end(); ii++ )
						iunit << " " << I( max(6,(int)ii->first.size()) , ii->second );
		for ( map<string,string>::iterator ii = new_string_cols.begin();
					ii != new_string_cols.end(); ii++ )
						iunit << " " << A( max((int)ii->second.size(),(int)ii->first.size()) , ii->second );
  }

	//user defined tag
	if ( get_user_score_tag() != "")  {
		iunit << ' ' << get_user_score_tag();
	}

// description
	iunit << ' ' << tag ; //<< '\n'; //std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin update_scorefile_info
///
/// @brief update info to be written in scorefile
///
/// @detailed Call various functions which measure decoy quality, usually
///      compared to native (rms, maxsub).  This function is called before
///      decoy or status file output
///
/// @global_read position arrays, native position arrays, mode flags
///
/// @global_write various 'scores' and private common block
///
/// @remarks
///
/// @references
///
/// @authors Jeff Gray
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_scorefile_info()
{
	using namespace files_paths;
	using namespace misc;
	using namespace scorefile_info;
	using namespace scorefxns;
	using namespace scores;
	using namespace start;


	calc_rms();
	if ( multi_chain && !design::dna_interface && !antibody_modeler ) {
		calc_docking_frac_nat_contact();
		calc_docking_interf_rmsd();
		calc_docking_number_of_contacts();
		calc_docking_hb_stats();
	}
	if ( require_start && start_initialized && ( !multi_chain || design::dna_interface || antibody_modeler ) )
	 fast_rms_x(position,start_position,rms_to_start);
	if (get_maxsub_to_server_models_flag()) maxsub_to_server_models();
	maxsub_native(Eposition,nalign,maxsub_rms,logEvalue);
	if ( ramachandran_weight == 0.0 ) ramachandran_score =
	 eval_rama_score_all(phi,psi,res,secstruct,total_residue);
	if ( get_loop_flag() ) {
		splicemsd = get_loop_splicemsd();
		loop_local_rms(0,position,rms_local);
		loop_gaps(ngap,cgap);
	}
	if( get_looprlx_exist() )
		chain_break_score = get_total_chain_break_score();
//car these probably havenot been called by the scorefxn
  contact_order(co, res, total_residue);
	structure::functions::radius_gyration(rg, total_residue);

//car figure out time/decoy in minutes
	if ( use_timer ) {
		double const start_minutes = retrieve_timer_start();
		if ( start_minutes < 0.0 ) { // Uninitialized
			decoy_time = 0.0;
		} else {
			decoy_time = minutes() - start_minutes;
		}
	}

	nsd_to_dummy_model = calc_NSD_to_dummy_model();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin minutes
///
/// @brief Minutes since 00:00:00 on January 1, 0000 in decimal to nearest second
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks Ignores leap days in earlier years
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
double
minutes()
{
	std::time_t current_time = std::time( NULL );
	std::tm * timeinfo = std::localtime( &current_time );
	return
	 ( ( ( ( 1900 + timeinfo->tm_year ) * 365 ) + timeinfo->tm_yday ) * 24.0 * 60.0 ) +
	 ( 60.0 * timeinfo->tm_hour ) + timeinfo->tm_min + ( timeinfo->tm_sec / 60.0 );
}

void
save_filter_tag( std::string const & tag){
	scorefile_info::filter_tag = tag;
}

std::string
get_filter_tag(){
	return scorefile_info::filter_tag;
}

bool
get_output_filter()
{
	static bool init = {false};
	static bool output_filter;

	if (!init) {
		output_filter = truefalseoption("output_filter");
		init = true;
	}

	return output_filter;
}

int
get_user_score_tag_length()
{
	static bool init = {false};
	static int user_defined_score_tag_length;

	if (!init) {
		std::string user_defined_score_tag;
		stringafteroption("user_tag", "", user_defined_score_tag );
		user_defined_score_tag_length = user_defined_score_tag.length();

		init = true;
	}

	return (user_defined_score_tag_length);
}

std::string
get_user_score_tag()
{
	static bool init = {false};
	static std::string user_defined_score_tag;

	if (!init) {
		stringafteroption("user_tag", "", user_defined_score_tag );
		init = true;
	}

	return (user_defined_score_tag);
}
