// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19092 $
//  $Date: 2007-12-12 17:49:43 -0800 (Wed, 12 Dec 2007) $
//  $Author: johnk $


// Rosetta Headers
#include "structure.h"
#include "after_opts.h"
#include "barcode_stats.h"
#include "cenlist.h"
#include "current_pose.h"
#include "disulfides.h"
#include "files_paths.h"
#include "force_barcode.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "pose.h"
#include "pose_vdw.h"
#include "random_numbers.h"
#include "read_paths.h"
#include "score.h"
#include "sheet_filter.h"
#include "util_vector.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpressions.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.i.hh>
//#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/file/file_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/ocstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>


// yab: 20070307
// The HomologStats usage and internals needs to be cleaned up.

namespace structure {

namespace functions {

////////////////////////////////////////////////////////////////////////////////
/// @begin radius_gyration
///
/// @brief
///
/// @detailed
///
/// @param  rg - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
radius_gyration(
	float & rg,
	int total_residue
)
{
	using namespace cenlist_ns;

	double rg_sum = 0.0;
	for ( int j = 1; j <= total_residue; ++j ) {
		for ( int i = 1, l = cendist.index(i,j); i < j; ++i, ++l ) {
			rg_sum += cendist[ l ]; // cendist(i,j)
		}
	}
	if ( total_residue != 0 ) {
		int const num = 2 * ( ( total_residue * total_residue )/2 - total_residue/2 );
		rg_sum = std::sqrt( rg_sum / num );
	}
	rg = rg_sum;
}

}

////////////////
/// Boundary ///
////////////////
// yab:
// functions here act as an interface between the classic monolithic
// Rosetta world and the new object world until misc removal is
// complete and Rosetta's computational pipeline is detangled.
namespace BOUNDARY {

SecondaryStructureEval * sse_object_;
HomologStats * cached_homolog_stats_ = NULL;

bool called_once_ = false;

void
init_module()
{
	using namespace misc;

	// setup options, note that default values in *option calls are a little bit
	// redundant, as those values are initialized on instantiation of object
	// note that a few other options are set in the setup_* functions defined below


	static bool init = false; // we have to cache these values for now, otherwise people are going to yell
	                          // about the constant output if they have multiple pdb files -- the
	                          // Rosetta options output is something that needs to be revamped, as
	                          // the way it's currently done is a little clunky

	static bool ncp;
	static float cpw;
	static bool sscp;
	static float cbw;
	static float lspp;
	static float rsdwth;
	static float rsdwtl;
	static float rsdwts;
	static float sdd;
	static bool ssdc;
	static float sss;
	static float msdc;
	static bool isbic;
	static bool hsf;

	if (!init) {
		ncp = truefalseoption("new_centroid_packing");
		realafteroption("centroid_pack_weight", 2.0, cpw);
		sscp = truefalseoption("separate_centroid_pack_score");
		realafteroption("cb_weight", 1.0, cbw);
		realafteroption("localstrandpair_penalty", 0.0, lspp);
		realafteroption("rsd_wt_helix", 1.0, rsdwth);
		realafteroption("rsd_wt_loop", 1.0, rsdwtl);
		realafteroption("rsd_wt_strand", 1.0, rsdwts);
		realafteroption("strand_dist_cutoff", 6.5, sdd);
		ssdc = truefalseoption("stretch_strand_dist_cutoff");
		realafteroption("seq_sep_scale", 20.0, sss);
		realafteroption("max_strand_dist_cutoff", 12.0, msdc);
		isbic = truefalseoption("ignore_sspair_barcode_in_score");
		hsf = truefalseoption("handedness_score");

		sse_object_ = new SecondaryStructureEval(&position, &Eposition, &res, &secstruct, &total_residue);
		cached_homolog_stats_ = new HomologStats(sse_object_);

		init = true;
	} else {
		cached_homolog_stats_->set_parent_sse_object(NULL);
		delete sse_object_;
		sse_object_ = new SecondaryStructureEval(&position, &Eposition, &res, &secstruct, &total_residue);
	}

	sse_object_->set_homolog_stats(cached_homolog_stats_);

	sse_object_->set_new_centroid_packing(ncp);
	sse_object_->set_centroid_packing_weight(cpw);
	sse_object_->set_separate_centroidpackscore(sscp);
	sse_object_->set_cb_weight(cbw);
	sse_object_->set_localstrandpair_penalty(lspp);
	sse_object_->set_rsd_wt_helix(rsdwth);
	sse_object_->set_rsd_wt_loop(rsdwtl);
	sse_object_->set_rsd_wt_strand(rsdwts);
	sse_object_->set_strand_dist_cutoff(sdd);
	sse_object_->set_stretch_strand_dist_cutoff(ssdc);
	sse_object_->set_seq_sep_scale(sss);
	sse_object_->set_max_strand_dist_cutoff(msdc);
	sse_object_->set_ignore_sspair_barcode_in_score(isbic);
	sse_object_->set_handedness_score_flag(hsf);

	called_once_ = true;
}

bool
init_module_called_once()
{
	return called_once_;
}

void
remove_module()
{
	delete sse_object_; sse_object_ = NULL;
	delete cached_homolog_stats_; cached_homolog_stats_ = NULL;
	return;
}

void
identify_ss()
{
	sse_object_->identify_ss();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_ss
///
/// @brief
/// THIS FUNCTION EVALUATES THE ENERGY OF A STRUCTURE BASED ON
///      THE TERTIARY PROBABILITY FILES
///
///  P(STRUCTURE) OF SECONDARY STRUCTURE PAIRS
/// @detailed
///
/// @param  lowstrand - [in/out]? - car lowstrand = 0 no distance terms,
///                    only orientational terms (distance dependence in rsigma)
///                     these are passed on to sspair_score
/// @param  cutoff - [in/out]? - don't compute interactions for pairs with sequence
///                 separation less than cutoff
/// @param  hs_score - [in/out]? -
/// @param  ss_score - [in/out]? -
/// @param  rsigma_score - [in/out]? -
/// @param  sheet_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
evaluate_ss(
	int & lowstrand,
	int & cutoff,
	float & hs_score,
	float & ss_score,
	float & rsigma_score,
	float & sheet_score
)
{
	sse_object_->evaluate_ss(lowstrand, cutoff, hs_score, ss_score, rsigma_score, sheet_score);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_envpair
///
/// @brief
/// THIS FUNCTION EVALUATES THE ENERGY OF A STRUCTURE BASED ON
///      THE TERTIARY PROBABILITY FILES
///car env_score, pair_score and cb_score
///
/// @detailed
///
/// @param  env_score - [in/out]? -
/// @param  pair_score - [in/out]? -
/// @param  cb_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car NOTE:  ASSUMES THE CENDIST ARRAY IS CURRENT!!!!
///
///car NOTE:  if disulfide scores are being used, disulfides must
///car be identified before calling this function
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
evaluate_envpair(
	float & env_score,
	float & pair_score,
	float & cb_score,
	float & centroid_pack_score
)
{
	sse_object_->evaluate_envpair(env_score, pair_score, cb_score, centroid_pack_score);
}

bool
get_separate_centroidpackscore_flag()
{
	return sse_object_->get_separate_centroidpackscore();
}

void
setup_SS_default_weight()
{
	static bool init = false;
	static float dpw, daw;
	if (!init){
		realafteroption("parallel_weight", 1.0, dpw);
		realafteroption("antiparallel_weight", 1.0, daw);
		init = true;
	}
	sse_object_->set_default_parallel_wt(dpw);
	sse_object_->set_default_antiparallel_wt(daw);
	sse_object_->setup_SS_default_weight();
}


void
setup_SS_weight()
{
	static bool init = false;
	static bool rand_SS_wt = false;
	static bool random_parallel_antiparallel = false;
	if (!init){
		rand_SS_wt = truefalseoption("rand_SS_wt");
		random_parallel_antiparallel = truefalseoption("random_parallel_antiparallel");
		init = true;
	}
	sse_object_->setup_SS_weight( rand_SS_wt, random_parallel_antiparallel);
}


void
setup_rsd_weight()
{
	static bool init = false;
	static bool rand_envpair_res_wt = false;
	if (!init) {
		rand_envpair_res_wt = truefalseoption("rand_envpair_res_wt");
		init = true;
	}
	sse_object_->setup_rsd_weight(rand_envpair_res_wt);
}

void
reset_rsd_wt()
{
	sse_object_->reset_rsd_weight();
}

void read_homolog_env_stats_file(
)
{
	std::string homolog_env_filename("-");
	stringafteroption( "env_stats_file",  "-",  homolog_env_filename  );

	cached_homolog_stats_->read_homolog_env_stats_file(homolog_env_filename);
}

void read_homolog_pair_stats_file(
)
{
	std::string homolog_pair_filename("-");
	stringafteroption( "pair_stats_file",  "-",  homolog_pair_filename  );

	cached_homolog_stats_->read_homolog_pair_stats_file(homolog_pair_filename);
}

void
evaluate_homolog_env(
	float & env_score
)
{
	sse_object_->evaluate_homolog_env(env_score);
}

void
evaluate_homolog_pair(
	float & pair_score
)
{
	sse_object_->evaluate_homolog_pair(pair_score);
}

void
SSpair_score(
	int lowstrand,
	int cutoff,
	float & ss_score,
	float & rsigma_score
)
{
	sse_object_->SSpair_score(lowstrand, cutoff, ss_score, rsigma_score);
}

int
get_total_strands()
{
	return sse_object_->get_total_strands();
}

FArray2D_int const &
get_SS_strand_end()
{
	return sse_object_->get_SS_strand_end();
}

FArray2D_float const &
get_strand_strand_score()
{
	return sse_object_->get_strand_strand_score();
}

int
get_total_SS_dimer()
{
	return sse_object_->get_total_SS_dimer();
}

FArray1D_int const &
get_SS_resnum()
{
	return sse_object_->get_SS_resnum();
}

FArray1D_int const &
get_SS_strand()
{
	return sse_object_->get_SS_strand();
}

FArray1D_float const &
get_pdb_cen10_prob()
{
	return sse_object_->get_pdb_cen10_prob();
}

FArray2D_float const &
get_pdb_pair_prob()
{
	return sse_object_->get_pdb_pair_prob();
}

} // namespace BOUNDARY

////////////////////////////////////////
/// Public: Construction/Destruction ///
////////////////////////////////////////

SecondaryStructureEval::SecondaryStructureEval(
	FArray2D_float const * position,
	FArray3Dp_float const * Eposition,
	FArray1D_int const * res,
	FArray1D_char const * secstruct,
	int const * total_residue
)
{
	position_ = position;
	Eposition_ = Eposition;
	res_ = res;
	secstruct_ = secstruct;
	total_residue_ = total_residue;

	initialize();
}

SecondaryStructureEval::SecondaryStructureEval() {}

SecondaryStructureEval::~SecondaryStructureEval() {
}

/////////////////////////////////////////////////////
/// Public: Load and Cache required external data ///
/////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin load_phi_theta_bins_from_file
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::load_and_cache_phi_theta_bins_from_file()
{
	int isep,iph,itheta,idot;
	double totn,tot;

	FArray3D_float pts_HS( 36, 36, 3 );
	FArray3D_float pts_SS( 36, 36, 3 );
	FArray1D_float pts_correct( 3 );
//------------------------------------------------------------------------------

	for ( itheta = 1; itheta <= 36; ++itheta ) {
//        radians = pi/180.0;
//        angle1 = (itheta+17)*5*radians;
//        angle2 = (itheta+18)*5*radians;
//        std::cout << SS( angle1 ) << SS( angle2 ) << std::endl;
//        cached_iptsn(itheta) = 10000*std::abs(std::sin(angle1)-std::sin(angle2));
//        std::cout << SS( itheta ) << SS( cached_iptsn(itheta) ) << std::endl;
//  FOR PRESMOOTHED/BINNED COUNTS FROM INGO USE NO THETA NORM
		cached_iptsn(itheta) = 100;
	}

	utility::io::izstream & HS_stream( open_data_file( "phi.theta.36.HS.resmooth" ) );
	for ( isep = 1; isep <= 3; ++isep ) {
		for ( itheta = 1; itheta <= 36; ++itheta ) {
			for ( iph = 1; iph <= 36; ++iph ) {
				HS_stream >> pts_HS(itheta,iph,isep) >> skip;
			}
		}
	}
	HS_stream.close();
	HS_stream.clear();

	for ( isep = 1; isep <= 3; ++isep ) {
		tot = 0.0;
		totn = 0.0;
		for ( iph = 1; iph <= 36; ++iph ) {
			for ( itheta = 1; itheta <= 36; ++itheta ) {
//  SMALL COUNTS CORRECTION
				pts_HS(itheta,iph,isep) += cached_iptsn(itheta)*0.000001f;
				tot += pts_HS(itheta,iph,isep);
				totn += cached_iptsn(itheta);
			}
		}
		for ( iph = 1; iph <= 36; ++iph ) {
			for ( itheta = 1; itheta <= 36; ++itheta ) {
				cached_pts(1,isep,iph,itheta) = -std::log(pts_HS(itheta,iph,isep)/tot) +
				 std::log(cached_iptsn(itheta)/totn);
			}
		}
	}

	utility::io::izstream & SS_stream( open_data_file( "phi.theta.36.SS.resmooth" ) );
	for ( isep = 1; isep <= 3; ++isep ) {
		for ( itheta = 1; itheta <= 36; ++itheta ) {
			for ( iph = 1; iph <= 36; ++iph ) {
				SS_stream >> pts_SS(itheta,iph,isep) >> skip;
			}
		}
		if ( isep == 1 ) SS_stream.seek_beg();
	}
	SS_stream.close();
	SS_stream.clear();

	for ( isep = 1; isep <= 3; ++isep ) {
		tot = 0.0;
		totn = 0.0;
		for ( iph = 1; iph <= 36; ++iph ) {
			for ( itheta = 1; itheta <= 36; ++itheta ) {
//  SMALL COUNTS CORRECTION
				pts_SS(itheta,iph,isep) += cached_iptsn(itheta)*0.000001f;
				tot += pts_SS(itheta,iph,isep);
				totn += cached_iptsn(itheta);
			}
		}
		for ( iph = 1; iph <= 36; ++iph ) {
			for ( itheta = 1; itheta <= 36; ++itheta ) {
				cached_pts(2,isep,iph,itheta) = -std::log(pts_SS(itheta,iph,isep)/tot) +
				 std::log(cached_iptsn(itheta)/totn);
			}
		}
	}

	tot = 0.0;
	totn = 0.0;
	for ( idot = 1; idot <= 6; ++idot ) {
		tot += ids(idot);
		totn += idsn(idot);
	}
	for ( idot = 1; idot <= 6; ++idot ) {
		if ( ids(idot) != 0 ) {
			cached_ds(idot) = -std::log(ids(idot)/tot) + std::log(idsn(idot)/totn);
		} else {
			cached_ds(idot) = 0.0;
		}
	}

	for ( isep = 1; isep <= 3; ++isep ) {
		pts_correct(isep) = 0.0;
		for ( iph = 1; iph <= 36; ++iph ) {
			for ( itheta = 1; itheta <= 36; ++itheta ) {
				if ( cached_pts(2,isep,iph,itheta) > pts_correct(isep) )
				 pts_correct(isep) = cached_pts(2,isep,iph,itheta);
			}
		}
	}

	// set cache state
	cache_initialized = true;

//Objexx: Unused
//	float dot_correct = 0.0;
//	for ( idot = 1; idot <= 6; ++idot ) {
//		if ( ds(idot) > dot_correct ) dot_correct = ds(idot);
//	}

}


void
SecondaryStructureEval::read_homolog_env_stats_file(
	std::string homolog_env_filename
)
{
	homolog_stats->read_homolog_env_stats_file(homolog_env_filename);
}

void
SecondaryStructureEval::read_homolog_pair_stats_file(
	std::string homolog_pair_filename
)
{
	homolog_stats->read_homolog_pair_stats_file(homolog_pair_filename);
}

/////////////////////////////////////////////////
/// Public: Scoring for HomologStats Instance ///
/////////////////////////////////////////////////
void
SecondaryStructureEval::evaluate_homolog_env(
	float & env_score
)
{
	homolog_stats->evaluate_homolog_env(env_score);
}

void
SecondaryStructureEval::evaluate_homolog_pair(
	float & pair_score
)
{
	homolog_stats->evaluate_homolog_pair(pair_score);
}

FArray1D_float const &
SecondaryStructureEval::get_pdb_cen10_prob()
{
	return homolog_stats->pdb_cen10_prob;
}

FArray2D_float const &
SecondaryStructureEval::get_pdb_pair_prob()
{
	return homolog_stats->pdb_pair_prob;
}

/////////////////////////////////
/// Public: Structure Scoring ///
/////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_ss
///
/// @brief
/// THIS FUNCTION EVALUATES THE ENERGY OF A STRUCTURE BASED ON
///      THE TERTIARY PROBABILITY FILES
///
///  P(STRUCTURE) OF SECONDARY STRUCTURE PAIRS
/// @detailed
///
/// @param  lowstrand - [in/out]? - car lowstrand = 0 no distance terms,
///                    only orientational terms (distance dependence in rsigma)
///                     these are passed on to sspair_score
/// @param  cutoff - [in/out]? - don't compute interactions for pairs with sequence
///                 separation less than cutoff
/// @param  hs_score - [in/out]? -
/// @param  ss_score - [in/out]? -
/// @param  rsigma_score - [in/out]? -
/// @param  sheet_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::evaluate_ss(
	int & lowstrand,
	int & cutoff,
	float & hs_score,
	float & ss_score,
	float & rsigma_score,
	float & sheet_score
)
{
	identify_ss();
	ss_score = 0.0;
	hs_score = 0.0;
	sheet_score = 0.0;
	rsigma_score = 0.0;
	if ( strands.total_SS_dimer >= 1 && helices.total_HH_dimer >= 1 ) {
		HSpair_score(hs_score);
	}
	if ( strands.total_strands > 1 || strands.total_SS_dimer >= 1 ) {
		SSpair_score(lowstrand,cutoff,ss_score,rsigma_score);
	}
	if ( strands.total_SS_dimer >= 1 ) {
		sheets_from_dimers(sheet_score); // must follow SSpair_score
		}

	sheet_score *= 2.019f;
	hs_score *= 0.090f;
	ss_score *= 0.498 * .75;
	rsigma_score *= 0.1f;

	ss_score *= get_SS_wt();
	sheet_score *= get_sheet_wt();
	hs_score *= get_HS_wt();
	rsigma_score *= get_rsigma_wt();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_envpair
///
/// @brief
/// THIS FUNCTION EVALUATES THE ENERGY OF A STRUCTURE BASED ON
///      THE TERTIARY PROBABILITY FILES
///car env_score, pair_score and cb_score
///
/// @detailed
///
/// @param  env_score - [in/out]? -
/// @param  pair_score - [in/out]? -
/// @param  cb_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car NOTE:  ASSUMES THE CENDIST ARRAY IS CURRENT!!!!
///
///car NOTE:  if disulfide scores are being used, disulfides must
///car be identified before calling this function
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::evaluate_envpair(
	float & env_score,
	float & pair_score,
	float & cb_score,
	float & centroid_pack_score
)
{
	using namespace cenlist_ns;
	using namespace param;
	using namespace param_aa;

	int const & total_residue = *total_residue_; // yab: misc removal
	FArray1D_int const & res = *res_; // yab: misc removal

	int res_id, icon;
	int interp1;
	float interp2;
	float k;
	float cb_score6;
	float cb_score12;

	if ( score_check_current_pose() ) {
		pose_update_cendist( score_get_current_pose() );
	}

	compute_CB_environment(); // function is hiding in vdw_compute.cc

	// make working copy of rsd_wt
	FArray1D_float rsd_wt(get_rsd_wt()); // rsd_wt is 1 unless command line flag envpair_wt is set

	// Testing an idea -- decrease weight on helices and loops to increase
	// their effective "temperature" and get more diverse structures.
	reweight_rsd_wt_for_secstruct(rsd_wt, 'H',  get_rsd_wt_helix());
	reweight_rsd_wt_for_secstruct(rsd_wt, 'E',  get_rsd_wt_strand());
	reweight_rsd_wt_for_secstruct(rsd_wt, 'L',  get_rsd_wt_loop());

	env_score = 0.0;
	cb_score6 = 0.0;
	cb_score12 = 0.0;
	cb_score = 0.0;

	FArray1D_float env_res( total_residue, 0.0 );
	FArray1D_float cb_score6_res( total_residue, 0.0 );
	FArray1D_float cb_score12_res( total_residue, 0.0 );
	for ( int i = 1; i <= total_residue; ++i ) {

//cmd  Several checks added to only consider amino acids, not DNA/RNA
		res_id = res(i);
		if ( is_protein(res_id) ) {

			if ( fcen10(i) >= 31.0f ) fcen10(i) = 30.9999f;

			env_res( i ) += env_log(res(i),static_cast< int >(fcen10(i))) *rsd_wt(i);

			if ( fcen12(i) < 1 ) {
				fcen12(i) = 1;
			} else if ( fcen12(i) >= 45.0f ) {
				fcen12(i) = 44.9999f;
			}
			if ( fcen6(i) >= 45.0f ) fcen6(i) = 44.9999f;

// interp1 rounds down to nearest (non-negative) integer.
			interp1 = static_cast< int >( fcen6(i) );
			 // note cen6 is always at least 1.0

// fraction remainder after nearest lower integer is removed
			interp2 = fcen6(i) - interp1;

//  use interp2 to linearly interpolate the two nearest bin values
			cb_score6_res( i ) = (( 1.0f - interp2 ) * cbeta_den6(interp1) +
			 interp2 * cbeta_den6(interp1+1)) * rsd_wt(i);

			interp1 = static_cast< int >( fcen12(i) );
			 // note cen12 is always at least 1.0
			interp2 = fcen12(i) - interp1;
			cb_score12_res( i ) = (( 1.0f - interp2 ) * cbeta_den12(interp1) +
			 interp2 * cbeta_den12(interp1+1)) * rsd_wt(i);

		} // amino acid check

	}

//     PAIR SCORE

	static double const cen_dist5_pad_plus  = cen_dist5_pad  + 25.0;
	static double const cen_dist7_pad_plus  = cen_dist7_pad  + 56.25;
	static double const cen_dist10_pad_plus = cen_dist10_pad + 100.0;

	static double const cen_dist5_pad_minus  = cen_dist5_pad  - 25.0;
	static double const cen_dist7_pad_minus  = cen_dist7_pad  - 56.25;
	static double const cen_dist10_pad_minus = cen_dist10_pad - 100.0;
	static double const cen_dist12_pad_minus = cen_dist12_pad - 144.0;

	static double const cen_dist5_pad_hinv  = 0.5 / cen_dist5_pad;
	static double const cen_dist7_pad_hinv  = 0.5 / cen_dist7_pad;
	static double const cen_dist10_pad_hinv = 0.5 / cen_dist10_pad;
	static double const cen_dist12_pad_hinv = 0.5 / cen_dist12_pad;

	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag();
	pair_score = 0.0;

	bool new_centroid_packing_flag = get_new_centroid_packing();
	centroid_pack_score  = 0.0;

	FArray1D_float pair_res( total_residue, 0.0 );
	FArray1D_float centroid_pack_res( total_residue, 0.0 );
	for ( int i = 1; i <= total_residue; ++i ) {
		res_id = res(i);

		if ( is_protein(res_id) || is_nonnatural(res_id) ) {
			for ( int kk = 1, lcl = cen_list.index(kk,i), kke = cen12up(i);
			 kk <= kke; ++kk, ++lcl ) {
				int j = cen_list[ lcl ]; // cen_list(kk,i)
				int const res_j = res(j);

//CAR  no pair score if a disulfide
				if ( disulf_flag && res_id == 2 && res_j == 2 &&
				 disulfides::BOUNDARY::cys_pair_in_disulf(i,j) ) goto L200;

				if ( j - i > 8 ) {
					k = cendist(i,j);

//$$$  we now try to find which bin the pair distance lies in
//$$$  I note this could in principle be calculated and updatded
//$$$  just like cen_dist is if there is a need for speed.
//$$$  this function interpolates between bins.
//$$$  An important(!) requirement on pair_log is that the
//$$$  value should approach zero as the radius increases.
//$$$  this fact permits us not to have to compute and score pairs are larger
//$$$  than cen_dist > cutoff.

					icon = 5;
					if ( k > cen_dist10_pad_plus ) {
						icon = 4;
						interp2 = ( k + cen_dist12_pad_minus ) * cen_dist12_pad_hinv;
					} else {
						if ( k > cen_dist7_pad_plus ) {
							icon = 3;
							interp2 = ( k + cen_dist10_pad_minus ) * cen_dist10_pad_hinv;
						} else {
							if ( k > cen_dist5_pad_plus ) {
								icon = 2;
								interp2 = ( k + cen_dist7_pad_minus ) * cen_dist7_pad_hinv;
							} else {
								icon = 1;
								interp2 = ( k + cen_dist5_pad_minus ) * cen_dist5_pad_hinv;
							}
						}
					}
					if ( interp2 < 0.0 ) interp2 = 0.0;

// note in theory this will never happen but in practice round off
// error can cause problem
					if ( interp2 > 1.0 ) interp2 = 1.0;

// handle last bin specially since icon+1 would be past array end
					float pair_contribution = 0.0;
					if ( icon != 5 ) {
						int const lpl = pair_log.index(icon,res_id,res_j);
						pair_contribution = ( ( 1.0f - interp2 ) * pair_log[ lpl ] +
						 interp2 * pair_log[ lpl+1 ] );
					} else {
						pair_contribution = ( 1.0f - interp2 ) * pair_log(icon,res_id,res_j);
					}
					pair_res( i ) += pair_contribution * rsd_wt(i)/2;
					pair_res( j ) += pair_contribution * rsd_wt(i)/2;

					// Adding a term that should help reproduce pairwise correlation function between centroids
					//      as observed in the PDB.
					if (new_centroid_packing_flag) {
						int kbin = static_cast <int> (sqrt(k)*10 + 1);
						if (kbin>120) kbin = 120;
						if (kbin<1)   kbin = 1;
						centroid_pack_res( i ) += centroid_pack_correction( kbin)/2;
						centroid_pack_res( i ) += centroid_pack_correction( kbin)/2;
					}

				}
L200:; // escape for cys-cys disulfide paires
			}
		} // escape for non-AA's
	}


	// handle symmetry
	FArray1D_float rsd_wt_symmetry( total_residue, 1.0f );
	if ( score_check_current_pose() && score_get_current_pose().symmetric() ) {
		pose_ns::Symmetry_info const & s
			( score_get_current_pose().symmetry_info() );
		int const N( s.N_bb() );
		for ( int i=1; i<= total_residue; ++i ) {
			if ( s.bb_independent(i) ) {
				rsd_wt_symmetry(i) = N;
			} else {
				rsd_wt_symmetry(i) = 0.0;
			}
		}
	}
	for (int i = 1; i <= total_residue; i++){
		env_score           += env_res( i )         * rsd_wt_symmetry( i );
		cb_score6           += cb_score6_res( i )   * rsd_wt_symmetry( i );
		cb_score12          += cb_score12_res( i )  * rsd_wt_symmetry( i );
		centroid_pack_score += centroid_pack_res(i) * rsd_wt_symmetry( i );
		pair_score          += pair_res( i )        * rsd_wt_symmetry( i );
	}

	env_score *= 2.019f;
	pair_score *= 2.019f;
	cb_score = 2.667f * ( cb_score6 + cb_score12 ) * 0.3f;
	cb_score *= get_cb_weight();

	if (new_centroid_packing_flag) {
		float centroid_pack_weight = get_centroid_packing_weight();
		centroid_pack_score *= centroid_pack_weight;

		if ( !get_separate_centroidpackscore() ) //If user desires, output centroid packing score in a separate score column ("cpack")
			cb_score += centroid_pack_score; // Default: add centroid packing score to cb_score.
	}
}

/////////////////////////
/// Public: Accessors ///
/////////////////////////
void
SecondaryStructureEval::set_localstrandpair_penalty(
	float val
)
{
	localstrandpair_penalty = val;
}

float
SecondaryStructureEval::get_localstrandpair_penalty()
{
	return localstrandpair_penalty;
}

void
SecondaryStructureEval::set_handedness_score_flag(
	bool flag
)
{
	handedness_score_flag = flag;
}

bool
SecondaryStructureEval::get_handedness_score_flag()
{
	return handedness_score_flag;
}

void
SecondaryStructureEval::set_ignore_sspair_barcode_in_score(
	bool flag
)
{
	ignore_sspair_barcode_in_score = flag;
}

bool
SecondaryStructureEval::get_ignore_sspair_barcode_in_score()
{
	return ignore_sspair_barcode_in_score;
}

void
SecondaryStructureEval::set_max_strand_dist_cutoff(
	float val
)
{
	max_strand_dist_cutoff = val;
}

float
SecondaryStructureEval::get_max_strand_dist_cutoff()
{
	return max_strand_dist_cutoff;
}

void
SecondaryStructureEval::set_seq_sep_scale(
	float val
)
{
	seq_sep_scale = val;
}

float
SecondaryStructureEval::get_seq_sep_scale()
{
	return seq_sep_scale;
}

void
SecondaryStructureEval::set_stretch_strand_dist_cutoff(
	bool flag
)
{
	stretch_strand_dist_cutoff = flag;
}

bool
SecondaryStructureEval::get_stretch_strand_dist_cutoff()
{
	return stretch_strand_dist_cutoff;
}

void
SecondaryStructureEval::set_strand_dist_cutoff(float val)
{
	strand_dist_cutoff = val;
}

float
SecondaryStructureEval::get_strand_dist_cutoff()
{
	return strand_dist_cutoff;
}

void
SecondaryStructureEval::set_cb_weight(
	float val
)
{
	cb_weight = val;
}

float
SecondaryStructureEval::get_cb_weight()
{
	return cb_weight;
}

void
SecondaryStructureEval::set_separate_centroidpackscore(
	bool flag
)
{
	separate_centroidpackscore = flag;
}

bool
SecondaryStructureEval::get_separate_centroidpackscore()
{
	return separate_centroidpackscore;
}

void
SecondaryStructureEval::set_centroid_packing_weight(
	float val
)
{
	centroid_packing_weight = val;
}

float
SecondaryStructureEval::get_centroid_packing_weight()
{
	return centroid_packing_weight;
}

void
SecondaryStructureEval::set_new_centroid_packing(
	bool flag
)
{
	new_centroid_packing = flag;
}

bool
SecondaryStructureEval::get_new_centroid_packing()
{
	return new_centroid_packing;
}

void
SecondaryStructureEval::set_homolog_stats(
	HomologStats * hs
)
{
	homolog_stats = hs;
	homolog_stats->set_parent_sse_object(this);
}

HomologStats *
SecondaryStructureEval::get_homolog_stats(
)
{
	return homolog_stats;
}

////////////////////////////////////////////////////////////
/// Public: Accessors, Get Only, Access to Inner Classes ///
////////////////////////////////////////////////////////////

int
SecondaryStructureEval::get_total_strands()
{
	return strands.total_strands;
}

FArray2D_int const &
SecondaryStructureEval::get_SS_strand_end()
{
	return strands.SS_strand_end;
}

FArray2D_float const &
SecondaryStructureEval::get_strand_strand_score()
{
	return strands.strand_strand_score;
}

int
SecondaryStructureEval::get_total_SS_dimer()
{
	return strands.total_SS_dimer;
}

FArray1D_int const &
SecondaryStructureEval::get_SS_resnum()
{
	return strands.SS_resnum;
}

FArray1D_int const &
SecondaryStructureEval::get_SS_strand()
{
	return strands.SS_strand;
}

/////////////////////////
/// Public: SS Weight ///
/////////////////////////
void
SecondaryStructureEval::reset_SS_weight()
{
	// set default weights
	SS_wt = 1.0;
	HS_wt = 1.0;
	sheet_wt = 1.0;
	rsigma_wt = 1.0;
	parallel_weight = get_default_parallel_wt();
	antiparallel_weight = get_default_antiparallel_wt();
}

void
SecondaryStructureEval::setup_SS_weight(
	bool use_ss_weight,
	bool randomize_parallel_antiparallel_weight
)
{
	use_ss_wt = use_ss_weight;
	random_parallel_antiparallel = randomize_parallel_antiparallel_weight;

	// set default weights
	reset_SS_weight();

	// if flagged, change
	if (use_ss_wt) {
		HS_wt=ran3()+ 0.5;
		SS_wt=ran3()+ 0.5;
		if (SS_wt > 1.2) SS_wt=1.2;
		sheet_wt=ran3() + 0.5;

		rsigma_wt=ran3() + 0.5;
		std::cout << "SS_WT: " << SS(SS_wt) << SS(HS_wt) << SS(sheet_wt) << SS(rsigma_wt) << std::endl;
	}

	// if flagged, change
	if (random_parallel_antiparallel) {
		//Choose whether to weight up parallel or antiparallel
		float randomnumber = ran3();
		float randomweight = 10.0 * ran3(); //Pretty drastic reweighting...
		if (randomnumber<0.5){
			parallel_weight = randomweight;
			antiparallel_weight = 1.0/randomweight;
		} else {
			parallel_weight = 1.0/randomweight;
			antiparallel_weight = randomweight;
		}
		std::cout << "New weight for long-range     parallel beta parings: " << parallel_weight << std::endl;
		std::cout << "New weight for long-range antiparallel beta parings: " << antiparallel_weight << std::endl;
	}
}

void
SecondaryStructureEval::setup_SS_default_weight()
{
	setup_SS_weight(false, false);
}

float
SecondaryStructureEval::get_HS_wt()
{
	return HS_wt;
}

float
SecondaryStructureEval::get_SS_wt()
{
	return SS_wt;
}

float
SecondaryStructureEval::get_rsigma_wt()
{
	return rsigma_wt;
}

float
SecondaryStructureEval::get_sheet_wt()
{
	return sheet_wt;
}

float
SecondaryStructureEval::get_parallel_wt()
{
	return parallel_weight;
}

float
SecondaryStructureEval::get_antiparallel_wt()
{
	return antiparallel_weight;
}

float
SecondaryStructureEval::get_default_parallel_wt()
{
	return default_parallel_weight;
}

float
SecondaryStructureEval::get_default_antiparallel_wt()
{
	return default_antiparallel_weight;
}

void
SecondaryStructureEval::set_default_parallel_wt(float val)
{
	default_parallel_weight = val;
}

void
SecondaryStructureEval::set_default_antiparallel_wt(float val)
{
	default_antiparallel_weight = val;
}

//////////////////////////
/// Public: RSD Weight ///
//////////////////////////

void
SecondaryStructureEval::reset_rsd_weight()
{
	int const & total_residue = *total_residue_; // yab: misc removal

	// set default weight
	rsd_wt.dimension(total_residue);
	rsd_wt = 1.0; // all values in FArray as 1.0
}

//////////////////////////////////////////////////////////////////////////////
// yab:
// setup_rsd_weight ALSO ensures that rsd_wt array is dimensioned properly,
// i.e. when total_residue is changed!
// There's some potentially dangerous behavior here, e.g. if a new structure
// is loaded, total residue changes (or even doesn't change), and setup_rsd_wt
// is not called.
// This and the associated routines, such as get_rsd_wt, are cleaned up over
// the classic Rosetta version but be careful.
//////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::setup_rsd_weight(bool use_rsd_weight)
{
	use_rsd_wt = use_rsd_weight;

	int const & total_residue = *total_residue_; // yab: misc removal

	// set default weight
	reset_rsd_weight();

	// if flagged, change
	if (use_rsd_wt) {
		for (int i =1; i <= total_residue; ++i){
			rsd_wt(i) = ran3() + 0.5;    // should switch to gaussian later//
			if (rsd_wt(i) > 1.2) rsd_wt(i)=1.2;
		}
	}
}

void
SecondaryStructureEval::setup_rsd_default_weight()
{
	setup_rsd_weight(false);
}

FArray1D_float const &
SecondaryStructureEval::get_rsd_wt()
{
	// yab:
	// The following if/else block is here to reproduce old behavior.
	// There is potential for inconsistent behavior here, especially since
	// in one case, the array is redimensioned and in the other it's not.
	// Something needs to be changed elsewhere so that this old behavior is
	// not needed any more.
	if (use_rsd_wt) {
		int const & total_residue = *total_residue_;
		assert( rsd_wt.size() == static_cast <FArray1D_float::size_type> (total_residue) );
	} else {
		reset_rsd_weight();
	}
	return rsd_wt;
}

void
SecondaryStructureEval::set_rsd_wt_helix(
	float val
)
{
	rsd_wt_helix = val;
}

void
SecondaryStructureEval::set_rsd_wt_loop(
	float val
)
{
	rsd_wt_loop = val;
}

void
SecondaryStructureEval::set_rsd_wt_strand(
	float val
)
{
	rsd_wt_strand = val;
}

float
SecondaryStructureEval::get_rsd_wt_helix()
{
	return rsd_wt_helix;
}

float
SecondaryStructureEval::get_rsd_wt_loop()
{
	return rsd_wt_loop;
}

float
SecondaryStructureEval::get_rsd_wt_strand()
{
	return rsd_wt_strand;
}

//////////////////////////////////////////////////////////////////////////////////////
/// Public: Accessor for Mirrors ; these functions are temporary and will go away! ///
///         Functionality is here only for HomologStats                            ///
//////////////////////////////////////////////////////////////////////////////////////
int const &
SecondaryStructureEval::get_total_residue()
const
{
	return *total_residue_;
}

FArray1D_int const &
SecondaryStructureEval::get_res()
const
{
	return *res_;
}

///////////////////////////
/// Private: RSD Weight ///
///////////////////////////
void
SecondaryStructureEval::reweight_rsd_wt_for_secstruct(
			FArray1D_float & rsd_wt,
			char secstruct_to_reweight,
			float reweightfactor
)
{
	int const & total_residue = *total_residue_; // yab: misc removal
	FArray1D_char const & secstruct = *secstruct_; // yab: misc removal

	for ( int i = 1; i <= total_residue; ++i ) {
		if (secstruct(i) == secstruct_to_reweight) {
			rsd_wt(i) *= reweightfactor;
		}
	}
}

/////////////////////////////////////
/// Private: Structure Scoring    ///
/////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin identify_ss
///
/// @brief
///car find all strand dimers and helix dimers
///
/// @detailed
///car strand dimer is first residue of two consecutive E's
///car helix dimer is second H in four consecutive H's    XHHHHX
///                             this is a helical position ^
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::identify_ss()
{
	using namespace param;

	FArray1D_char const & secstruct = *secstruct_; // yab: misc removal
	int const & total_residue = *total_residue_; // yab: misc removal

	FArray1D_int dimer_type( total_residue ); // what type of dimer each position is
	// 0-none, 1-SS 2-HH

//car find SSdimers and HHdimers
//car find N-terminal ends of SS elements
//car save a map of the dimers to aid finding C-termini later
	int lastH = 1;
	int lastE = 1;
	int lastL = 1;
	helices.total_HH_dimer = 0;
	strands.total_SS_dimer = 0;
	for ( int i = 1; i <= total_residue; ++i ) {
		char const c = secstruct(i);
		if ( c == 'L' ) {
			lastL = i;
		} else if ( c == 'E' ) {
			lastE = i;
		} else if ( c == 'H' ) {
			lastH = i;
		}

		dimer_type(i) = 0;
		if ( i <= total_residue-1 ) {
			if ( secstruct(i) == 'E' && secstruct(i+1) == 'E' ) {
				++strands.total_SS_dimer;
				strands.SS_resnum(strands.total_SS_dimer) = i;
				strands.SS_strand_end(1, strands.total_SS_dimer) = std::max(lastH,lastL);
				dimer_type(i) = 1;
			}
		}
		if ( i >= 2 && i <= total_residue-2 ) {
			if ( secstruct(i-1) == 'H' && secstruct(i) == 'H' && secstruct(i+1) == 'H' &&
			 secstruct(i+2) == 'H' ) {
				++helices.total_HH_dimer;
				helices.HH_resnum(helices.total_HH_dimer) = i;
				helices.HH_helix_end(1, helices.total_HH_dimer) = std::max(lastE,lastL);
				dimer_type(i) = 2;
			}
		}
	}

	lastH = total_residue;
	lastE = total_residue;
	lastL = total_residue;
	int iHH = helices.total_HH_dimer;
	int iSS = strands.total_SS_dimer;

	for ( int i = total_residue; i >= 1; --i ) {
		char const c = secstruct(i);
		if ( c == 'L' ) {
			lastL = i;
		} else if ( c == 'E' ) {
			lastE = i;
		} else if ( c == 'H' ) {
			lastH = i;
		}

		if ( dimer_type(i) == 1 ) {
			strands.SS_strand_end(2, iSS) = std::min(lastH, lastL);
			--iSS;
		} else if ( dimer_type(i) == 2 ) {
			helices.HH_helix_end(2, iHH) = std::min(lastE, lastL);
			--iHH;
		}
	}

//car assign strand numbers to ss_dimers
	if ( strands.total_SS_dimer > 0 ) {
		strands.SS_strand(1) = 1; // first ss-dimer belongs to strand 1
		strands.total_strands = 1;

		for ( int i = 2; i <= strands.total_SS_dimer; ++i ) {
			if ( strands.SS_strand_end(1,i) == strands.SS_strand_end(1,i-1) ) {
				strands.SS_strand(i) = strands.SS_strand(i-1); // same strand
			} else {
				++strands.total_strands;
				strands.SS_strand(i) = strands.total_strands;
			}
		}
	} else {
		strands.total_strands = 0;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin sheets_from_dimers
///
/// @brief
///
/// @detailed
///js This function takes a set of dimer neighbors, and determines how
///js many sheets there, and how many strands are in each sheet
///js This information is then used to calculate the "poker hand" score,
///js which reflects to probability of that distribution of strands and
///js sheets.
///js In current version, it seems to simply penalize sheets with fewer
///js strands compared to those with more strands.
///
///js This function looks at a list of dimers, which contains up to
///js two neighbors for each dimer.  In priniciple these neighbors would
///js be hydrogen bond partners in neighboring strands.  This function
///js will take this list, however it is made.
///
///js Currently, dimer neighbors are defined somewhat arbitrarily.
///js If I understand the code correctly, the first and last dimers in
///js sequence that follow the current dimer, and that are within 6.5
///js angstroms, are the neighbors.  There is no orientation dependence
///js on what counts as a strand neighbor.
///
///js A sheet is then loosely defined by all the strands that are connected
///js by one of these neighbors.  This amounts to "single-linkage clustering."
///js A sheet is determined by the set of strands that can be linked by
///js the dimer neighbors.  Note that a one neighbor is enough to say that
///js two strands are in a sheet.
///
///js The final score is put into sheet_score, and is determined by the
///js number of sheets of each size.
///
///js   Basic strategy:  go through all dimers, finding neigboring strands
///js   Because each dimer can only have two neighbors we can first search
///js   all the way down one side, and then the other.  This will give a set
///js   of dimers that are connected.  The strands in which these dimers reside
///js   are then considered connected into sheets.  By going through all
///js   dimers with the proper bookkeeping, we can determine how many sheets
///js   there are, and how many strands in each.
///
/// @param  sheet_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::sheets_from_dimers( float & sheet_score )
{
	using namespace param;

	int const & total_residue = *total_residue_; // yab: misc removal

	static FArray1D_bool searched( MAX_RES() );
	static FArray1D_int strand_sheet( MAX_RES() );
	 // 40 is the maximum number of strands
	static FArray1D_int num_of_strands( MAX_RES() );
	 // 11 is the maxmumber number of sheets
	static FArray2D_int strand_sheet_list( MAX_RES(), MAX_RES() );
	static FArray1D_float const m_term( 4, m_term_initializer );

	for ( int current_dimer = 1; current_dimer <= strands.total_SS_dimer;
	 ++current_dimer ) {
//js      Set all dimers as unchecked.
		searched(current_dimer) = false;
//js      set all sheet locations as null
//js         dimer_sheet(current_dimer) = 0
	}
	for ( int current_strand = 1; current_strand <= strands.total_strands;
	 ++current_strand ) {
//js      Set the sheets of all strands as null
		strand_sheet(current_strand) = 0;
	}
	for ( int current_sheet = 1; current_sheet <= total_residue;
	 ++current_sheet ) {
		num_of_strands(current_sheet) = 0;
	}

//js Find the neighbors of each dimer.  Some will be found during
//js the search, some will be initial nodes.  That is why we keep
//js track of whether it has been searched.
	int num_of_sheets = 0;
	int current_sheet = 0;
	for ( int current_dimer = 1; current_dimer <= strands.total_SS_dimer;
	 ++current_dimer ) {
		if ( !searched(current_dimer) ) {
//js we need to check this one
			searched(current_dimer) = true;
			int current_strand = strands.SS_strand(current_dimer);
			// place node strand in sheet
			if ( strand_sheet(current_strand) == 0 ) { // it is not in a sheet, so:
				// make new sheet
				++num_of_sheets;
				current_sheet = num_of_sheets;

				// place strand in current sheet
				strand_sheet(current_strand) = current_sheet;
				++num_of_strands(current_sheet);
				strand_sheet_list(num_of_strands(current_sheet),current_sheet) =
				 current_strand;
			} else {
				current_sheet = strand_sheet(current_strand);
			}
			for ( int direction = 1; direction <= 2; ++direction ) {
			 // the two directions of searching
				int neighbor = strands.dimer_neighbor(direction,current_dimer);
				while ( neighbor != 0 ) {
//               if ( neighbor != 0 ) {
//js                  if ( !searched(neighbor) ) {
					searched(neighbor) = true;

					current_strand = strands.SS_strand(neighbor);
					if ( strand_sheet(current_strand) == 0 ) {
// js  if neighbor strand is not in a sheet already, put it in the working sheet
						strand_sheet(current_strand) = current_sheet;
						++num_of_strands(current_sheet);

						strand_sheet_list(num_of_strands(current_sheet),current_sheet) =
						 current_strand;
					} else if ( strand_sheet(current_strand) != current_sheet ) {
// js if neighbor strand is  already in a different sheet, merge the sheets.
// js the sheet of the new strand must have a lower sheet number, so give the
// js strands of the current sheet to its sheet
						int const new_sheet = strand_sheet(current_strand);
						int & num_of_strands_new_sheet( num_of_strands(new_sheet) );
						for ( int merge = 1,
						 lss = strand_sheet_list.index(merge,current_sheet),
						 mergee = num_of_strands(current_sheet);
						 merge <= mergee; ++merge, ++lss ) {
							++num_of_strands_new_sheet;
							int const strand_sheet_list_mc = strand_sheet_list[ lss ];
							 // strand_sheet_list(merge,current_sheet)
							strand_sheet_list(num_of_strands_new_sheet,new_sheet) =
							 strand_sheet_list_mc;
							strand_sheet(strand_sheet_list_mc) = new_sheet;
						}
						--num_of_sheets;
// rhiju After merging one sheet with another, need to erase traces
// rhiju of sheet that got eaten up, and reorder other sheets.
						num_of_strands(current_sheet) = 0;
						for (int shiftsheet = current_sheet; shiftsheet <= num_of_sheets;
								 ++shiftsheet){
							num_of_strands(shiftsheet) = num_of_strands(shiftsheet+1);
							for (int i = 1; i <= num_of_strands(shiftsheet); ++i){
								int strandtoshift =	strand_sheet_list(i,shiftsheet+1);
								strand_sheet_list(i,shiftsheet) = strandtoshift;
								strand_sheet(strandtoshift) = shiftsheet;
							}
						}
						current_sheet = new_sheet;
					}
//js                  }
					neighbor = strands.dimer_neighbor(direction,neighbor);
				}
			}
		}
	}

//js calculate score, based on the number of sheets of each size
	double sheet_score_sum = 0.0;
	for ( int current_sheet = 1; current_sheet <= num_of_sheets; ++current_sheet ) {
		sheet_score_sum += m_term( std::min( num_of_strands(current_sheet), 4 ) );
	}
	sheet_score = sheet_score_sum;
	//	sheet_score *+ get_sheet_wt();

	if ( get_handedness_score_flag() && files_paths::use_filter(files_paths::sheet_type) ){
		int result = 0;
		sheet_filter::SheetFilter sf(position_, secstruct_, total_residue_);
		sf.compute_result(result); // Trigger evaluation if Ingo's sheet filter, including handedness checks.
		sheet_score += sf.get_handedness_score();
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin HSpair_score
///
/// @brief
/// THIS FUNCTION LOOKS AT 4-MERS OF HELICES AND 2-MERS OF STRANDS
///
/// @detailed
///
/// @param  hs_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::HSpair_score( float & hs_score )
{
	using namespace cenlist_ns;

	FArray3Dp_float const & Eposition = *Eposition_;

	static FArray1D_float pt1( 3 );
	static FArray1D_float pt2( 3 );
	static FArray1D_float pt3( 3 );
	static FArray1D_float pt4( 3 );
	static FArray1D_float vdist( 3 );
	static FArray1D_float cen1( 3 );
	static FArray1D_float cen2( 3 );

	int ss1,ss2,itemp;
	int iph,ith,iseqsep;
//	int idist;
	float ph, th, dist;
//------------------------------------------------------------------------------
	if ( score_check_current_pose() ) {
		pose_update_cendist( score_get_current_pose() );
	}

	hs_score = 0.0;

	for ( ss1 = 1; ss1 <= helices.total_HH_dimer; ++ss1 ) {
		int const HH_resnum_ss1 = helices.HH_resnum(ss1);
		int const HH_helix_end_1ss1 = helices.HH_helix_end(1,ss1);
		int const HH_helix_end_2ss1 = helices.HH_helix_end(2,ss1);
		helix_end(HH_resnum_ss1,pt1,pt2);
		for ( ss2 = 1; ss2 <= strands.total_SS_dimer; ++ss2 ) {
			int const SS_resnum_ss2 = strands.SS_resnum(ss2);
			if ( cendist(HH_resnum_ss1,SS_resnum_ss2) < 144.0 ) {
				int l3 = Eposition.index(1,1,SS_resnum_ss2);
				pt3(1) = Eposition[   l3 ]; // Eposition(1,1,SS_resnum_ss2)
				pt3(2) = Eposition[ ++l3 ]; // Eposition(2,1,SS_resnum_ss2)
				pt3(3) = Eposition[ ++l3 ]; // Eposition(3,1,SS_resnum_ss2)
				int l4 = Eposition.index(1,4,SS_resnum_ss2+1);
				pt4(1) = Eposition[   l4 ]; // Eposition(1,4,SS_resnum_ss2+1)
				pt4(2) = Eposition[ ++l4 ]; // Eposition(2,4,SS_resnum_ss2+1)
				pt4(3) = Eposition[ ++l4 ]; // Eposition(3,4,SS_resnum_ss2+1)

				dist_pair(pt1,pt2,pt3,pt4,dist,cen1,cen2,vdist);
				if ( dist <= 12.0 ) {
					spherical(pt2,pt4,ph,th,cen1,cen2,vdist);

					if ( ph < -180.0 ) {
						ph += 360.0f;
					} else if ( ph > 180.0 ) {
						ph -= 360.0f;
					}

					iph = static_cast< int >(1+(ph+180.0f)/10);
					if ( iph > 36 ) {
						iph = 36;
					} else if ( iph < 1 ) {
						iph = 1;
					}

					ith = static_cast< int >(1+(th/5));
					if ( ith > 36 ) {
						ith = 36;
					} else if ( ith < 1 ) {
						ith = 1;
					}

//car bin distance, but never used
//					if ( dist > 6.0 && dist < 8.0 ) {
//						idist = 2;
//					} else if ( dist > 8.0 && dist < 10.0 ) {
//						idist = 3;
//					} else if ( dist > 10.0 && dist < 12.0 ) {
//						idist = 4;
//					} else {
//						idist = 1; // dist <= 6.0 (or eq 8.0 or 10.0 or 12.0)
//					}

					itemp = std::min(
					 std::abs( strands.SS_strand_end(2,ss2) - HH_helix_end_1ss1 ) + 1,
					 std::abs( HH_helix_end_2ss1 - strands.SS_strand_end(1,ss2) ) + 1 );
					if ( itemp >= 2 && itemp <= 10 ) {
						iseqsep = 2;
					} else {
						if ( itemp > 10 ) {
							iseqsep = 3;
						} else {
							iseqsep = 1;
						}
					}

					hs_score += pts(1,iseqsep,iph,ith);
//        std::cout <<
//         I( 3, HH_resnum(ss1) ) << I( 3, SS_resnum(ss2) ) <<
//         F( 8, 2, dist ) << F( 8, 2, ph ) << I( 2, iph ) <<
//         F( 8, 2, th ) << I( 2, ith ) <<
//         F( 8, 2, pts(1,iseqsep,iph,ith) ) << std::endl;

				}
				//				hs_score *= get_HS_wt();
//car bin distance, but pointless cause dist score is commented out
//				if ( dist < 12.0 ) {
//					if ( dist > 6.5 && dist < 8.5 ) {
//						idist = 2;
//					} else if ( dist > 8.5 && dist < 10.5 ) {
//						idist = 3;
//					} else if ( dist > 10.5 && dist < 12.0 ) {
//						idist = 4;
//					} else {
//						idist = 1; // dist <= 6.5 or == 8.5 or == 10.5
//					}
//					hs_score += ssdist(idist,1);
//				}
			}
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SSpair_score
///
/// @brief
///car score the interaction of strand dimer pairs.
///
/// @detailed
///car score the interaction of strand dimer pairs.
///car score composed of
///car   1. phithetascore
///car   2. distscore
///car   3. dotscore
///car for all terms, score < 0 implies there are counts in the bin
///
///car  define V1 = vector from N of res 1 to C' of res 2 in dimer 1
///car         V2 =                                          dimer 2
///car         M  = vector connecting midpoint of V1 and V2
///
///car  phithetascore depends on....
///car  theta = angle between V1 and V2
///car  phi_ss = angle by which V2 is rotated out of the M-V1 plane
///car           around V1
///car
///car  strand_seqsep = sequence separation of the _strands_!!
///car           binned into three groups: same strand,
///car           sequence-adjacent strands (2-10 res separation) and non-
///car           sequence-adjacent strands (>10 residues separation)
///car these values are binned to give iphi,itheta,istrand_seqsep and
///car          score looked up in pts array in structure.h
///car  the first element of the pts array is a key for HS versus SS terms
///car         2 is SS
///
///
///car note that there is also 'dimer_seqsep' - the distance between the two
///car dimers;  if dimer_seqsep < cutoff, then the total score for a pair is 0
///
///
///car  sigma  = angle between V1 and M (ie the shear between V1 and V2)
///car           THIS IS USED IN DISTANCE_SIGMA!!
///
///car distscore depends on....
///car  dist = length of M
///car  vdist = unit vector along M, calculated by function dist_pair
///car     score is a lookup in ssdist, which is presumably binned by distances
///car     but only the bin 1 is used in this function?
///car     here if dist<6.5 angstroms, distscore is assigned the bin 1 value
///car     if the two dimers are greater in sequence separation (ie dimer_seqsep the
///car     distance between the two dimers) than cutoff
///car     (sometimes don't want to reward local pairs- see score2 and score5)
///car
///car  since rsigma also scores distances between dimer pairs (along with
///car    the shear, distscore is not used if lowstrand = 0.  (ie lowstrand
///car    should always be 0 is distance_sigma score is being used)
///car
///
///car  dotscore ....  ('dotproduct' score)
///car  is calculated by pair_dp  - scores whether the Hbonding groups actually
///car         point at each other (ie rotation around V1 and V2 axes)
///car    CO11 = unit vector along C'=O bond of res 1 of dimer 1
///car    CO21 = unit vector along O=C' bond of res 2 of dimer 1
///car    CO12 = unit vector along C'=O bond of res 1 of dimer 2
///car    CO21 = unit vector along O=C' bond of res 2 of dimer 2
///car    dpall = (|M.CO11|+|M.CO21|+|M.CO21|+|M.CO22|)/2,  when binned,
///car          becomes idot, this describes the extent to which all CO vectors
///car          are coplaner
///car    sign1 = which direction do COs on dimer 1 point?
///car            determined by the sign of (M.CO11+M.CO21)
///car    sign2 =                                 2
///car            determined by the sign of (M.CO12+M.CO22)
///car            sign1/2 check that not only are CO's coplanar, they point
///car            in the correct direction
///car  is calculated by pair_dp  - scores whether the Hbonding groups actually
///car         point at each other (ie rotation around V1 and V2 axes)
///car  dotscore is a lookup from ds array
///car  dotscore  only evaluated and added to total if there's a
///car  good phithetascore
///
///car  scoring strategy:
///car     want to ensure that each dimer interacts only with a dimers from
///car     a single strand on a single side (ie a strand can only have one
///car     neighbor); consequently scores for each dimer are stored (in the
///car     dimer_pair_score array) and { the list of dimer-pairs that interact
///car     is trimmed to remove conflicting interactions, keeping the
///car     best-scoring interactions.  This applies only to favorable interactions
///car     bad interactions never conflict with each other
///
///
///car ss_orient is save to note pairs with goo phitheta and dpall score
///car   this was used in strand_sheet, but isn't currently used.
///
/// @param  lowstrand - [in/out]? -
/// @param  cutoff - [in/out]? -
/// @param  ss_score - [in/out]? -
/// @param  rsigma_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::SSpair_score(
	int lowstrand,
	int cutoff,
	float & ss_score,
	float & rsigma_score
)
{
	using namespace cenlist_ns;
	using namespace param;

	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal
	int const & total_residue = *total_residue_; // yab: misc removal

	static Dimension const MAX_dimer_pairs( MAX_RES() * 20 );
	static FArray1D_int pair_list1( MAX_dimer_pairs );
	static FArray1D_int pair_list2( MAX_dimer_pairs );
	static FArray1D_float theta_list( MAX_dimer_pairs );
	static FArray2D_int signdimer1( MAX_RES(), MAX_RES() );
	static FArray2D_int signdimer2( MAX_RES(), MAX_RES() );
	static FArray1D_float pt1( 3 );
	static FArray1D_float pt2( 3 );
	static FArray1D_float pt3( 3 );
	static FArray1D_float pt4( 3 );
	static FArray1D_float vdist( 3 );
	static FArray1D_float cen1( 3 );
	static FArray1D_float cen2( 3 );
	static FArray1D_float unit_vdist( 3 );
	static FArray2D_float dimer_pair_score( MAX_RES(), MAX_RES() );
	 // dimer to dimer score (<6.5A)
	static FArray1D_float const SS_penalty( 11, SS_penalty_initializer );

//car local
	int const dimer_pair_score_size1 = dimer_pair_score.size1();
	int const ss_orient_size1 = strands.ss_orient.size1();
	int dimer_seqsep,strand_seqsep,sign1,sign2,score_exists;
	int iphi,itheta,istrand_seqsep,idot,pos1,dimer1,dimer2;
	int total_pairs;
	int idist, isig;
	float phi_ss, theta, dist, dpall, distscore, phithetascore, dotscore, minscore;
	float tempscore_rsigma, sig;
	float const ssdist_12 = ssdist(1,2);

	if ( score_check_current_pose() ) {
		pose_update_cendist( score_get_current_pose() );
	}

	// do barcode sspair constraints exist?
	bool const barcode_sspair_constraints_exist
		( barcode_exist() && barcode_feature_set().sspair_constraints_exist() && !get_ignore_sspair_barcode_in_score());

	if ( barcode_sspair_constraints_exist ) {
		reset_barcode_sspair_limits();
	}

//car initialize
	total_pairs = 0;
	for ( int ss1 = 1; ss1 <= strands.total_SS_dimer; ++ss1 ) {
		strands.dimer_neighbor(1,ss1) = 0;
		strands.dimer_neighbor(2,ss1) = 0;
	}
	for ( int ss2 = 1; ss2 <= strands.total_SS_dimer; ++ss2 ) {
		for ( int ss1 = 1, l = dimer_pair_score.index(ss1,ss2);
		 ss1 <= strands.total_SS_dimer; ++ss1, ++l ) {
			//Objexx: Assumes all of these arrays have same dimensions
			dimer_pair_score[ l ] = 0.0; // dimer_pair_score(ss1,ss2)
			signdimer1[ l ] = 0;         // signdimer1(ss1,ss2)
			signdimer2[ l ] = 0;         // signdimer2(ss1,ss2)
			strands.ss_orient[ l ] = 0;          // ss_orient(ss1,ss2)
		}
	}
	ss_score = 0.0;

	//// some diagnostics routines use this function to calculate
	// pairing data for a list of structures:
	// decide whether we should do some book-keeping:
	bool const store_ss_scores( get_store_ss_scores() );
	if ( store_ss_scores ) {
		// declared in structure::secstruct_ns
		// this is a little slow, re-dimensioning each time
		// requires re-allocation of space?
		strands.strand_strand_score.dimension(strands.total_strands, strands.total_strands);
		strands.strand_strand_score = 0.0; // array initialization
	}

//car ss1 is the first dimer in the possible pair
	for ( int ss1 = 1; ss1 < strands.total_SS_dimer; ++ss1 ) {

		int const SS_resnum_ss1 = strands.SS_resnum(ss1);
		int const SS_strand_end_1ss1 = strands.SS_strand_end(1,ss1);
		int const SS_strand_end_2ss1 = strands.SS_strand_end(2,ss1);
		int & dimer_neighbor_1ss1( strands.dimer_neighbor(1,ss1) );
		int & dimer_neighbor_2ss1( strands.dimer_neighbor(2,ss1) );
		int const SS_strand_ss1 = strands.SS_strand(ss1);

//  SS_resnum(ss1) IS THE SEQUENCE POSITION OF STRAND ss1
		pos1 = SS_resnum_ss1;
		int lE = Eposition.index(1,1,pos1);
		pt1(1) = Eposition[   lE ]; // Eposition(1,1,pos1); // N of res1 in dimer1
		pt1(2) = Eposition[ ++lE ]; // Eposition(2,1,pos1);
		pt1(3) = Eposition[ ++lE ]; // Eposition(3,1,pos1);
		++pos1;
		lE = Eposition.index(1,4,pos1);
		pt2(1) = Eposition[   lE ]; // Eposition(1,4,pos1); // C' of res2 in dimer1
		pt2(2) = Eposition[ ++lE ]; // Eposition(2,4,pos1);
		pt2(3) = Eposition[ ++lE ]; // Eposition(3,4,pos1);

//car ss2 is second dimer, always C term to first
		for ( int ss2 = ss1+1, l = strands.ss_orient.index(ss1,ss2);
		 ss2 <= strands.total_SS_dimer; ++ss2, l+=ss_orient_size1 ) {

			int const SS_resnum_ss2 = strands.SS_resnum(ss2);

			if ( cendist(SS_resnum_ss1,SS_resnum_ss2) < 144.0 ) {

//  IF TWO DIMERS HAVE A GOOD PHI/THETA AND DOT PRODUCT, ss_orient=1
				strands.ss_orient[ l ] = 0; // ss_orient(ss1,ss2)

//car calculate the sequence separation between the two dimers
//car note that this is not the strand separation that phitheta is
//car  conditioned on;  the dimer separation is used to decide which
//car pair to count in ss_score based on the value of cutoff

				dimer_seqsep = SS_resnum_ss2 - SS_resnum_ss1;
//              if ( SS_resnum_ss2 - SS_resnum_ss1 > 5 ) {

				int const SS_strand_end_1ss2 = strands.SS_strand_end(1,ss2);
				if ( SS_strand_end_1ss1 != SS_strand_end_1ss2 ) { // not in same strand
					pos1 = SS_resnum_ss2;
					lE = Eposition.index(1,1,pos1);
					pt3(1) = Eposition[   lE ]; // Eposition(1,1,pos1); // N of res1 in dimer2
					pt3(2) = Eposition[ ++lE ]; // Eposition(2,1,pos1);
					pt3(3) = Eposition[ ++lE ]; // Eposition(3,1,pos1);
					++pos1;
					lE = Eposition.index(1,4,pos1);
					pt4(1) = Eposition[   lE ]; // Eposition(1,4,pos1); // C' of res2 in dimer2
					pt4(2) = Eposition[ ++lE ]; // Eposition(2,4,pos1);
					pt4(3) = Eposition[ ++lE ]; // Eposition(3,4,pos1);

//car find vector and distance between midpoints
					dist_pair(pt1,pt2,pt3,pt4,dist,cen1,cen2,vdist);

					float strand_dist_cutoff = get_strand_dist_cutoff(); // Default is 6.5 Angstroms

					bool const stretch_strand_dist_cutoff = get_stretch_strand_dist_cutoff();
					if (stretch_strand_dist_cutoff)		{
						float const seq_sep_scale  = get_seq_sep_scale();
						float const max_strand_dist_cutoff  = get_max_strand_dist_cutoff();
						strand_dist_cutoff = 6.5 +
							(max_strand_dist_cutoff - 6.5) *(dimer_seqsep / seq_sep_scale);
						if (strand_dist_cutoff > max_strand_dist_cutoff) strand_dist_cutoff = max_strand_dist_cutoff;
					}

					if ( dist < strand_dist_cutoff ) {
//car find phi and theta
						spherical(pt2,pt4,phi_ss,theta,cen1,cen2,vdist);

//car find sequence separation between STRANDS  (not between dimers)
						strand_seqsep = SS_strand_end_1ss2 - SS_strand_end_2ss1 + 1;

//car bin all these values...
						if ( strand_seqsep >= 2 && strand_seqsep <= 10 ) {
							istrand_seqsep = 2;
						} else {
							if ( strand_seqsep > 10 ) {
								istrand_seqsep = 3;
							} else {
								istrand_seqsep = 1;
							}
						}

						if ( phi_ss > 180.0 ) {
							phi_ss -= 360.0f;
						} else if ( phi_ss < -180.0 ) {
							phi_ss += 360.0f;
						}

						iphi = static_cast< int >(1+(phi_ss+180.0)/10);
						if ( iphi > 36 ) {
							iphi = 36;
						} else if ( iphi < 1 ) {
							iphi = 1;
						}

						itheta = static_cast< int >(1+(theta/5));
						if ( itheta > 36 ) {
							itheta = 36;
						} else if ( itheta < 1 ) {
							itheta = 1;
						}

//car find dp_all and sign1,sign2  (convert vdist to unit vec first)
						double const dist_inv = 1.0 / dist;
						unit_vdist(1) = vdist(1) * dist_inv;
						unit_vdist(2) = vdist(2) * dist_inv;
						unit_vdist(3) = vdist(3) * dist_inv;
						pair_dp(SS_resnum_ss1,SS_resnum_ss2,dpall,unit_vdist,sign1,sign2);

//car evaluate the first two scoring terms:
						phithetascore = pts(2,istrand_seqsep,iphi,itheta);
						distscore = 0.0;

//rhiju Extra rewards/bonuses for parallel and antiparallel terms. Only affects
//         long-range pairings.
						if (theta<90 & istrand_seqsep==3) phithetascore *= get_parallel_wt();
						if (theta>90 & istrand_seqsep==3) phithetascore *= get_antiparallel_wt();

//car save the total score for the pair and the signs
//car dimer_pair_score is the total for the dimer pair
//     LOCAL STRANDS DON'T GET SO MUCH OF A SCORE BONUS
//car add terms to pair only if pair separation > cutoff
//Objexx: Assumes these arrays have same dimensions as ss_orient
						if ( dimer_seqsep >= cutoff ) {
							if ( lowstrand > 0.5 ) distscore = ssdist_12;
							 dimer_pair_score[ l ] += phithetascore + distscore;
							 // dimer_pair_score(ss1,ss2)
						}
						signdimer1[ l ] = sign1; // signdimer1(ss1,ss2)
						signdimer2[ l ] = sign2; // signdimer2(ss1,ss2)

						if ( phithetascore < 0.0 ) {
//car bin dpall to get idot
//km changed bins 5 and 6
							if ( dpall > 0.33 && dpall <= 0.66 ) {
								idot = 2;
							} else if ( dpall > 0.66 && dpall <= 1.00 ) {
								idot = 3;
							} else if ( dpall > 1.00 && dpall <= 1.33 ) {
								idot = 4;
							} else if ( dpall > 1.33 && dpall <= 1.60 ) {
								idot = 5;
							} else if ( dpall > 1.60 && dpall <= 1.80 ) {
								idot = 6;
							} else if ( dpall > 1.80 && dpall <= 2.00 ) {
								idot = 5;
							} else {
								idot = 1;
							}

//car lookup the dotscore
							dotscore = ds( idot );

//car again, if the distance between pairs is too small, don't add this
//car term to the total
//     LOCAL STRANDS DON'T GET SO MUCH OF A SCORE BONUS
							if ( dimer_seqsep >= cutoff )
							 dimer_pair_score[ l ] += dotscore; // dimer_pair_score(ss1,ss2)

//car note that ss_orient is 1 if phithetascore<0 and dotscore<0 and dist<6.5
//car but these terms not in dimer_pair_score if dimer_seqsep < cutoff
							if ( dotscore < 0.0 ) strands.ss_orient[ l ] = 1; // ss_orient(ss1,ss2)

//js quick fix for rsigma dependence on sign1,sign2 by evaluating here.
//js
							sigma(pt2,cen1,vdist,sig);
							if ( sig > 179.0 ) {
								sig = 179.0;
							} else if ( sig < 0.0 ) {
								sig = 0.0;
							}
							isig = static_cast< int >( sig / 10 ) + 1;
							if ( dist > 6.4 ) {
								dist = 6.4;
							} else if ( dist < 3.5 ) {
								dist = 3.5;
							}
							idist = static_cast< int >( ( dist - 3.5 ) / 0.25 ) + 1;

							//tempscore_rsigma = rsigma_dot(idist,isig,sign1,sign2);

							// The definition of dimer signs (sign1,sign2)
							// appears inverted (1 should be 2, vice versa).
							tempscore_rsigma = rsigma_dot(idist, isig, 3 - sign1, 3 - sign2);


							// Modify sigma potential to no longer give an rsigma bonus
							//  to strands in wrong register.
							if (sign1 == 1 & sign2 == 1 & sig < 110. & sig > 70.) tempscore_rsigma = 0.0;
							if (sign1 == 1 & sign2 == 2 & (sig < 75. | sig > 95.)) tempscore_rsigma = 0.0;;
							if (sign1 == 2 & sign2 == 1 & (sig < 90. | sig > 110.)) tempscore_rsigma = 0.0;
							if (sign1 == 2 & sign2 == 2 & sig < 120. & sig > 80.) tempscore_rsigma = 0.0;

							if ( tempscore_rsigma > 0.0 ) {
//car add in all unfavorable rsigma scores (these pairs may not contribute
//car to ss_score if they have a favorable dimer_pair_score)
//car note there are no positive scores in structure_db.cc as of 7/03
								rsigma_score += tempscore_rsigma;
							} else {
//car and favorable ones if phitheta and dot score favorable & dist<6.5
//car note that rsigma is not subject to the favorable-dimer-interactions-must
//car be-consistent-pairwise-strands rule that is applied to the ss_score
//car below
								if ( strands.ss_orient[ l ] == 1 ) rsigma_score += tempscore_rsigma;
								 // ss_orient(ss1,ss2)
							}

							//////////////////////////////
							// barcode sspair constraints:
							//
							if ( barcode_sspair_constraints_exist ) {
								// these are scored regardless of dimer_seqsep
								// (eg to penalize hairpins even in score5)
								// but only if the interaction is favorable
								// ie dimer_score < 0.0
								//
								// the score is equal to
								// -1 * weight * dimer_score
								//
								// the -1 is so that a positive weight in the barcode file
								// will penalize occurrence of the feature
								//
								float tmp_distscore( lowstrand > 0.5 ? ssdist_12 : 0.0 );
								// reconstruct dimer_score w/o the dimer_seqsep cutoff:
								float const dimer_score
									( phithetascore + tmp_distscore + dotscore );
								if ( dimer_score < 0.0 ) {
									// score is added directly to sspair score

									// offset the residue position for the second dimer
									// if they are anti-parallel. This has the effect that
									// if the dimers are aligned, then the residues passed
									// in will in fact be aligned as well.
									int const dimer2_offset( theta > 90.0 ? 1 : 0 );

									float const score_delta( barcode_sspair_constraint_score(
										SS_resnum_ss1, SS_resnum_ss2+dimer2_offset, theta,
										dimer_score ));

									ss_score += score_delta;

									if ( false && std::abs( score_delta ) > 0.1 )
										std::cout << "Apply score: " << score_delta << ' ' <<
											SS_resnum_ss1 << ' ' << SS_resnum_ss2 << std::endl;
								}
							}

							// rhiju Allow penalty for locally paired strands to be set by user.
							int sequence_separation = std::abs( strands.SS_resnum(ss2) - strands.SS_resnum(ss1) );
							if ( sequence_separation <= 11)	{
								dimer_pair_score[ l] += get_localstrandpair_penalty() *
									SS_penalty(sequence_separation);
							}

//car old code...
//     if ( istrand_seqsep <= 2 && dimer_pair_score(ss1,ss2) < 0 )
//      dimer_pair_score(ss1,ss2) *= 0.8f;
//car now add  pairs that interact favorably to a list for later trimming...
//car pair_list1 is the first dimer of the pair, pair_list2 is the second dimer
//car total_pairs is the length of the list
//car pairs that interact unfavorably count toward the ss_score
//car and then are zeroed out.  dimer_pair_score=0 later is used to indicate
//car that a pair is disallowed (because some other pair has already been
//car counted)

							if ( dimer_pair_score[ l ] < 0.0 ) { // dimer_pair_score(ss1,ss2)
//car here we should also build a list of what dimers ss1 and ss2 are close
//car to so that we can speed up the check for disallowed pairs at the end
//car of this function

								++total_pairs;
								if ( total_pairs > MAX_dimer_pairs() ) {
									std::cout << "increase MAX_dimer_pairs in " <<
									 "SS_pair_score" << std::endl;
									utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
								}
								pair_list1(total_pairs) = ss1;
								pair_list2(total_pairs) = ss2;
								// store for later book-keeping
								if ( store_ss_scores ) theta_list(total_pairs) = theta;
							} else {
								ss_score += dimer_pair_score[ l ]; // dimer_pair_score(ss1,ss2)
								dimer_pair_score[ l ] = 0.0; // dimer_pair_score(ss1,ss2)
								dimer_pair_score(ss2,ss1) = 0.0;
							}
						}      // good phi/th

//js collect data on neighbors of dimers for making sheets later
//js somewhat odd method of selecting the two neighbors for determining what
//js makes a sheet.  Selects the first and last dimer that follow the current
//js dimer in sequence.  Not completely sure it is even symmetric.
//js There has been discussion of replacing this with something like
//js a) a check that these dimers are also well oriented according to SS_strand_score
//js b) or possibly using the good two that are selected with later.


						if ( dist <= 5.5 ) { // && ss_orient(i,j) == 1} ) {
							if ( dimer_neighbor_1ss1 == 0 ) {
								dimer_neighbor_1ss1 = ss2;
							} else if ( strands.SS_strand(dimer_neighbor_1ss1) != strands.SS_strand(ss2) ) {
								dimer_neighbor_2ss1 = ss2;
							}
							if ( ( dimer_neighbor_1ss1 != ss2 ) &&
							 ( dimer_neighbor_2ss1 != ss2 ) ) {
								if ( strands.dimer_neighbor(1,ss2) == 0 ) {
									strands.dimer_neighbor(1,ss2) = ss1;
								} else if ( strands.SS_strand(strands.dimer_neighbor(1,ss2)) != SS_strand_ss1 ) {
									strands.dimer_neighbor(2,ss2) = ss1;
								}
							}
						}      // within sheet distance cutoff (5.5 Angstrom)
					}         // within distance cutoff (6.5 Angstrom)
				}            // not the same strand
//     }  // more than 5 residues apart
			}               // centroids within 12 Angstroms
		}                  // index of 2nd strand position
	}                     // index of 1st strand position

//car symmetrize the score and sign matrices...
	for ( int ss1 = 1; ss1 <= strands.total_SS_dimer; ++ss1 ) {
		for ( int ss2 = ss1+1,
		 l = dimer_pair_score.index(ss1,ss2),
		 lT = dimer_pair_score.index(ss2,ss1);
		 ss2 <= strands.total_SS_dimer; ++ss2, l+=dimer_pair_score_size1, ++lT ) {
			//Objexx: Assumes arrays have same dimensions
			dimer_pair_score[ lT ] = dimer_pair_score[ l ];
			signdimer1[ lT ] = signdimer2[ l ];
			signdimer2[ lT ] = signdimer1[ l ];
		}
	}
//Objexx: Replaced by above for speed
//	for ( int ss1 = 1; ss1 <= total_SS_dimer; ++ss1 ) {
//		for ( int ss2 = ss1+1; ss2 <= total_SS_dimer; ++ss2 ) {
//			dimer_pair_score(ss2,ss1) = dimer_pair_score(ss1,ss2);
//			signdimer1(ss2,ss1) = signdimer2(ss1,ss2);
//			signdimer2(ss2,ss1) = signdimer1(ss1,ss2);
//		}
//	}

//car okay, we've now scored all dimer pairs and we have to figure out
//car if there are inconsistent pairs on our list (ie each dimer can
//car interact with at most two strands, one on each side; note that
//car a dimer can interact with multiple dimers on a single side, as long
//car as they're in the same strand)
//car
//car What follows below is very inefficient, but here's the idea:
//car  loop through the list of favorable pairs to find the best score;
//car  this pair is 'allowed' and added to the total.  This pair now
//car  defines which strand is to one side of dimer1 and which strand is
//car  to one side of dimer2 (the side is defined by signdimer).  So
//car  now we loop through all possible dimer pairs and if they are
//car  not in
//car  then mark them as disallowed by setting the score for that pair
//car   (ie dimer_pair_score) to zero.
//car  Now repeat this for the next-best scoring pair that remains in the
//car  the favorable pairs list.  When there are no more allowed pairs,
//car  escape.
//car this will be sped up enormously by
//car   1. sort the allowed pairs list so we just have to loop through it 1x
//car   2. save a list (above) of every dimer that a particular dimer is
//car      near so that marking disallowed pairs is fast

//  TOTAL SS_SCORE (DONT LET A DIMER INTERACT WITH MORE THAN 2 OTHERS)
//    UNLESS THE SCORE IS UNFAVORABLE, THEN IT STILL GETS THE PENALTY
//car    (note the penalties have already been added in)

	score_exists = 1; // ie there are still pairs on the list that are allowed
	while ( score_exists == 1 ) {
		minscore = 99999.0; // flag the min score so we can tell if found a min
		dimer1 = 0;
		dimer2 = 0;
		int best_pair = 0;
		for ( int ss1 = 1; ss1 <= total_pairs; ++ss1 ) {
		 // loop through the favorable pairs list

//car if the pair is still allowed (ie dimer_pair_score hasn't been set to zero)
//car and it's a better score, save it
			float const dp_score = dimer_pair_score( pair_list1(ss1), pair_list2(ss1) );
			if ( dp_score < minscore && dp_score != 0.0 ) {
				minscore = dp_score;
				dimer1 = pair_list1(ss1);
				dimer2 = pair_list2(ss1);
				best_pair = ss1;
				sign1 = signdimer1(dimer1,dimer2);
				sign2 = signdimer2(dimer1,dimer2);
			}
//       std::cout << SS( dimer1 ) << SS( dimer2 ) << SS( minscore ) << std::endl;
		}

		if ( minscore == 99999.0 ) {

//car aha! no allowed pairs found, so we must be done
//car  set score_exists so we'll know
			score_exists = 0;
		} else {

//car otherwise, add the pair score to the total
			ss_score += minscore;

//DB  BONUS FOR NONLOCAL PAIRS!!
//car this was added to correct for an excess of local strand pairs
//car and really should probably go above where each dimer is calculated...
//car  ask DB about this...
			//// added check for dimers, had initialization problems here before
			if ( dimer1 <= 0 || dimer2 <= 0 ) {
				std::cout << "ABORT: failed dimer1/2 initialization" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			if ( std::abs( strands.SS_resnum(dimer2) - strands.SS_resnum(dimer1) ) > 11 &&
			 std::abs( strands.SS_strand_end(1,dimer2) - strands.SS_strand_end(2,dimer1) ) > 8 )
			 ss_score -= 0.2;

//car mark this particular pair as disallowed (ie we've used it already)
			dimer_pair_score(dimer1,dimer2) = 0.0;
			dimer_pair_score(dimer2,dimer1) = 0.0;

			int const SS_strand_dimer1 = strands.SS_strand(dimer1);
			int const SS_strand_dimer2 = strands.SS_strand(dimer2);

			// should we do some book-keeping? ////////////////////////
			if ( store_ss_scores ) {
				// DANGER: some of these arrays are not even setup unless
				//         store_ss_scores == true
				assert( SS_strand_dimer1 < SS_strand_dimer2 );

				// record this pair interaction in the array of strand-strand scores
				// NOTE: this is not quite right -- neglecting the -0.2 bonus above!
				if ( theta_list( best_pair)  > 90.0 ) {
					// antiparallel ==>  first index < second index
					// this must sync with anybody using this array!!
					strands.strand_strand_score( SS_strand_dimer1, SS_strand_dimer2 ) +=
						minscore;
				} else {
					// parallel     ==>  first index > second index
					strands.strand_strand_score( SS_strand_dimer2, SS_strand_dimer1 ) +=
						minscore;
				}

				// store the residue-residue pairing
				store_residue_pairing( strands.SS_resnum( dimer1), strands.SS_resnum( dimer2), minscore, total_residue );
			} // end of book-keeping /////////////////////////////////////////


// ARE THERE OTHER DIMERS INTERACTING WITH THE BEST PAIR?
			for ( int ss2 = 1; ss2 <= strands.total_SS_dimer; ++ss2 ) { // loop through all dimers
//    DIMER1 CAN ONLY INTERACT WITH other dimers of the dimer2 STRAND
//   on the side that dimer2 interacts

//Objexx: Assumes arrays are same dimensions

//car if dimer1 and ss2 interact favorably
//car and ss2 and dimer2 are in different strands
//car and ss2 is on the same side of dimer1 as dimer2 is...
//car then mark this pair as dissallowed
				if ( dimer_pair_score(dimer1,ss2) < 0.0 &&
				 strands.SS_strand(ss2) != SS_strand_dimer2 &&
				 signdimer1(dimer1,ss2) == sign1 ) {
					dimer_pair_score(dimer1,ss2) = 0.0;
					dimer_pair_score(ss2,dimer1) = 0.0;
				}

//car repeat the process for dimer2
//    SAME IS TRUE OF DIMER2
				if ( dimer_pair_score(dimer2,ss2) < 0.0 &&
				 strands.SS_strand(ss2) != SS_strand_dimer1 &&
				 signdimer1(dimer2,ss2) == sign2 ) {
					dimer_pair_score(dimer2,ss2) = 0.0;
					dimer_pair_score(ss2,dimer2) = 0.0;
				}
			}    // loop over ss2, all dimers
		}      // if we found an allowed pair still on the favorable list
	}      // while there's still something on the favorable list

	//	ss_score *= get_SS_wt();
	//rsigma_score *+ get_rsigma_wt();
}



///////////////////////////////////
/// Private: Structure Geometry ///
///////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin helix_end
///
/// @brief
///     This function reads in two points in sequence and
///     returns two points in space, the endpoints of the axis
///     through an alpha-helix
///
/// @detailed
///
/// @param  pos1 - [in/out]? -
/// @param  p1 - [in/out]? -
/// @param  p2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::helix_end(
	int pos1,
	FArray1DB_float & p1,
	FArray1DB_float & p2
)
{
	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal

	int const s1 = pos1-1;
	int const s2 = pos1;
	int const s3 = pos1+1;
	int const s4 = pos1+2;

	static double const eleven_inv = 1.0 / 11.0;

	for ( int i = 1; i <= 3; ++i ) {
		double const Epos_sum =
		 Eposition(i,2,s1) + Eposition(i,4,s1) +
		 Eposition(i,1,s2) + Eposition(i,2,s2) + Eposition(i,4,s2) +
		 Eposition(i,1,s3) + Eposition(i,2,s3) + Eposition(i,4,s3) +
		 Eposition(i,1,s4) + Eposition(i,2,s4);
		p1(i) = ( Epos_sum + Eposition(i,1,s1) ) * eleven_inv;
		p2(i) = ( Epos_sum + Eposition(i,4,s4) ) * eleven_inv;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pair_dp
///
/// @brief
///
/// @detailed
///car calculate sum of dot product of the co vectors of strand dimers ss1 and ss2
///car with the vector connecting the midpoints of the dimer vectors (vdist)
///car also determine return the sign of the dot products for each dimer
///car to determine which direction the CO groups point
///
/// @param  ss1 - [in/out]? -
/// @param  ss2 - [in/out]? -
/// @param  dp - [in/out]? -
/// @param  vdist - [in/out]? -
/// @param  sign1 - [in/out]? -
/// @param  sign2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::pair_dp(
	int ss1,
	int ss2,
	float & dp,
	FArray1DB_float const & vdist,
	int & sign1,
	int & sign2
)
{
	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal

//car parameters
//	static float const dist_co = { 1.231015f }; // length of C=O bond
	static float const dist_co_inv = { 1.0f / 1.231015f };
	int const Ep_s12 = Eposition.size1() * Eposition.size2();

//car local
	static FArray1D_float temp( 3 );

	dp = 0.0;
	float dp1 = 0.0;
	float dp2 = 0.0;
	float sdp1 = 0.0;
	float sdp2 = 0.0;

	for ( int i = ss1, l0 = Eposition.index(1,4,i); i <= ss1+1; ++i, l0+=Ep_s12 ) {
		if ( i == ss1+1 ) {
			for ( int j = 1, l = l0; j <= 3; ++j, ++l ) {
				temp(j) = -( Eposition[ l+3 ] - Eposition[ l ] ) * dist_co_inv; // 5=O, 4=C
				 //       -( Eposition(j,5,i) - Eposition(j,4,i) )
			}
		} else {
			for ( int j = 1, l = l0; j <= 3; ++j, ++l ) {
				temp(j) = ( Eposition[ l+3 ] - Eposition[ l ] ) * dist_co_inv; // 5=O, 4=C
				 //       ( Eposition(j,5,i) - Eposition(j,4,i) )
			}
		}
		if ( vdist(1) != 10.0 ) {   // why is this checked?  (car)
			float const tempdot = dotprod(temp,vdist);
			dp1 += std::abs(tempdot);
			sdp1 += tempdot;
		}
	}
	dp1 *= 0.5;

	for ( int i = ss2, l0 = Eposition.index(1,4,i); i <= ss2+1; ++i, l0+=Ep_s12 ) {
		if ( i == ss2+1 ) {
			for ( int j = 1, l = l0; j <= 3; ++j, ++l ) {
				temp(j) = -( Eposition[ l+3 ] - Eposition[ l ] ) * dist_co_inv; // 5=O, 4=C
				 //       -( Eposition(j,5,i) - Eposition(j,4,i) )
			}
		} else {
			for ( int j = 1, l = l0; j <= 3; ++j, ++l ) {
				temp(j) = ( Eposition[ l+3 ] - Eposition[ l ] ) * dist_co_inv; // 5=O, 4=C
				 //       ( Eposition(j,5,i) - Eposition(j,4,i) )
			}
		}
		if ( vdist(1) != 10.0 ) {
			float const tempdot = dotprod(temp,vdist);
			dp2 += std::abs(tempdot);
			sdp2 += tempdot;
		}
	}
	dp2 *= 0.5;

	if ( vdist(1) == 10.0 ) {
		dp = 10.0;
	} else {
		dp = dp1 + dp2;
//js These signs tell whether the first c=o bond vector of a dimer points
//js at the other dimer.  sign1 = 1 means that the first c=o bond of dimer1
//js points at dimer2.  sign2 = 1 means that dimer2 points at dimer1.  When
//js sign1 or sign2 equals 2, that dimer points away from the other dimer.
		sign1 = ( sdp1 > 0.0 ? 2 : 1 );
		sign2 = ( sdp2 < 0.0 ? 2 : 1 );
	}
}

/////////////////////
/// Private: Math ///
/////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin dist_pair
///
/// @brief
///car find the vector connecting the midpoints of two line segments
///car defined by a1,a2 and a3,a4
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  a3 - [in/out]? -
/// @param  a4 - [in/out]? -
/// @param  dist - [in/out]? - length of v21
/// @param  cen1 - [in/out]? -
/// @param  cen2 - [in/out]? -
/// @param  v21 - [in/out]? - vector connecting midpoints
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::dist_pair(
	FArray1DB_float const & a1,
	FArray1DB_float const & a2,
	FArray1DB_float const & a3,
	FArray1DB_float const & a4,
	float & dist, // length of v21
	FArray1DB_float & cen1,
	FArray1DB_float & cen2,
	FArray1DB_float & v21 // vector connecting midpoints
)
{

//car find midpoint coordinates
	cen1(1) = ( a1(1) + a2(1) )*0.5f;
	cen1(2) = ( a1(2) + a2(2) )*0.5f;
	cen1(3) = ( a1(3) + a2(3) )*0.5f;

	cen2(1) = ( a3(1) + a4(1) )*0.5f;
	cen2(2) = ( a3(2) + a4(2) )*0.5f;
	cen2(3) = ( a3(3) + a4(3) )*0.5f;

//car find distance between midpoint coordinates
	v21(1) = cen2(1) - cen1(1);
	v21(2) = cen2(2) - cen1(2);
	v21(3) = cen2(3) - cen1(3);
	dist = std::sqrt(
	 v21(1) * v21(1) +
	 v21(2) * v21(2) +
	 v21(3) * v21(3) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin sigma
///
/// @brief
///car find angle sigma between vectors cen1->a2 and v21
///
/// @detailed
///
/// @param  a2 - [in/out]? - endpoints of v1
/// @param  cen1 - [in/out]? -
/// @param  v21 - [in/out]? -
/// @param  sig - [in/out]? - sigma
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::sigma(
	FArray1DB_float const & a2, // endpoints of v1
	FArray1DB_float const & cen1,
	FArray1DB_float const & v21,
	float & sig // sigma
)
{
	using numeric::conversions::to_degrees;
	using numeric::sin_cos_range;

//car this could be done in spherical

//car local
	static FArray1D_float v1( 3 );
	static FArray1D_float uz( 3 );
	static FArray1D_float u21( 3 );

	subvec(a2,cen1,v1);
	unitvec(v1,uz);
	unitvec(v21,u21);
	float const u21z = dotprod(u21,uz);

	sig = 200.0; // why 200? should be 0 or 180
	if ( std::abs(u21z) <= 1.0 ) { //Objexx:SGM This logic is hackery that should be cleaned up
		if ( std::abs(u21z) < 1.0 ) sig = std::acos( sin_cos_range( u21z ) );
		to_degrees( sig );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin spherical
///
/// @brief
///car find the angle (theta) between two vectors v1 and v2
///car and the angle (phi) by which v2 is rotated out of the plane
///car defined by v1 and v21 around v1 (ie v1 is the z-axis)
///
/// @detailed
///car define a coordinate system with Z axis along cen1->a2 (v1),
///car xz plane defined by cen1->a2 and v21. and origin at cen1.
///car Find the spherical coordinates phi,and theta of point a4 if
///car vector cen2->a4 was moved such that cen2=cen1
///
/// @param  a2 - [in/out]? -
/// @param  a4 - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  theta - [in/out]? -
/// @param  cen1 - [in/out]? -
/// @param  cen2 - [in/out]? -
/// @param  v21 - [in/out]? - vector between midpoints
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::spherical(
	FArray1DB_float const & a2,
	FArray1DB_float const & a4,
	float & phi,
	float & theta,
	FArray1DB_float const & cen1,
	FArray1DB_float const & cen2,
	FArray1DB_float const & v21 // vector between midpoints
)
{
	using numeric::conversions::degrees;
	using numeric::sin_cos_range;

//car local
	float v2x,v2y,v2z,r1;
	float u21z;
	float length;
	static FArray1D_float v1( 3 );
	static FArray1D_float v2( 3 );
	static FArray1D_float ux( 3 );
	static FArray1D_float uy( 3 );
	static FArray1D_float uz( 3 );
	static FArray1D_float u21( 3 );

//car v1 vector from center to end of dimer1 vector
//car v2                                   2
	v1(1) = a2(1) - cen1(1);
	v1(2) = a2(2) - cen1(2);
	v1(3) = a2(3) - cen1(3);
	v2(1) = a4(1) - cen2(1);
	v2(2) = a4(2) - cen2(2);
	v2(3) = a4(3) - cen2(3);

//car find unit vector along v1 = uz
	length = std::sqrt(
	 v1(1) * v1(1) +
	 v1(2) * v1(2) +
	 v1(3) * v1(3) );
	if ( length == 0.0 ) length = 0.0001;
	length = 1.0/length;
	uz(1) = v1(1)*length;
	uz(2) = v1(2)*length;
	uz(3) = v1(3)*length;

//car find unit vector perpendicular v21 v1 plane
	uy(1) = uz(2)*v21(3) - uz(3)*v21(2); // uy = uz x v21
	uy(2) = uz(3)*v21(1) - uz(1)*v21(3);
	uy(3) = uz(1)*v21(2) - uz(2)*v21(1);

	length = std::sqrt(
	 uy(1) * uy(1) +
	 uy(2) * uy(2) +
	 uy(3) * uy(3) );
	if ( length == 0.0 ) length = 0.0001;
	length = 1.0/length;
	uy(1) *= length; // uy=uy/|uy|
	uy(2) *= length;
	uy(3) *= length;

//car third unit vector to define coordinate system
	ux(1) = uy(2)*uz(3) - uy(3)*uz(2); // uz = uy x uz
	ux(2) = uy(3)*uz(1) - uy(1)*uz(3);
	ux(3) = uy(1)*uz(2) - uy(2)*uz(1);

	length = std::sqrt(
	 ux(1) * ux(1) +
	 ux(2) * ux(2) +
	 ux(3) * ux(3) );
	if ( length == 0.0 ) length = 0.0001;
	length = 1.0/length;
	ux(1) *= length; // ux=ux/|ux|
	ux(2) *= length;
	ux(3) *= length;


//car find projection of v2 onto each of these azes
	v2x = v2(1)*ux(1) + v2(2)*ux(2) + v2(3)*ux(3); // v2x=v2.ux
	v2y = v2(1)*uy(1) + v2(2)*uy(2) + v2(3)*uy(3); // v2y=v2.uy
	v2z = v2(1)*uz(1) + v2(2)*uz(2) + v2(3)*uz(3); // v2z=v2.uz

//car and length of v2
	r1 = std::sqrt(
	 v2(1) * v2(1) +
	 v2(2) * v2(2) +
	 v2(3) * v2(3) );

//car unit vector along v21
	length = std::sqrt(
	 v21(1) * v21(1) +
	 v21(2) * v21(2) +
	 v21(3) * v21(3) );
	if ( length == 0.0 ) length = 0.0001;
	length = 1.0/length;
	u21(1) = v21(1)*length; // u21=v21/|v21|
	u21(2) = v21(2)*length;
	u21(3) = v21(3)*length;

//car projection of u21 on uz
	u21z = u21(1)*uz(1) + u21(2)*uz(2) + u21(3)*uz(3); // u21z=u21.uz


//car the same thing in function calls (is this expensive?)
//$$$      unitvec(v1,uz);
//$$$      cros(uz,v21,uy);
//$$$      unitvec(uy,uy);
//$$$      cros(uy,uz,ux);
//$$$      unitvec(ux,ux);
//$$$
//$$$      v2x = dotprod(v2,ux);
//$$$      v2y = dotprod(v2,uy);
//$$$      v2z = dotprod(v2,uz);
//$$$      r1 = std::sqrt( ( v2(1) * v2(1) )+( v2(2) * v2(2) )+( v2(3) * v2(3) ) );
//$$$
//$$$      unitvec(v21,u21);
//$$$      u21z = dotprod(u21,uz);


	phi = 200.0; // why set to 200?  if v2y = 0, v2 lies in xz plane and phi
	// is zero; if v2x=0, v2 lies in yz plane and phi is 90 or -90
	if ( v2y != 0.0 && v2x != 0.0 ) {
		phi = degrees( std::atan2( v2y, v2x ) );
	}

	float const v2z_r1 = v2z / r1;
	if ( r1 != 0.0 && std::abs( v2z_r1 ) <= 1.0 ) {
		theta = degrees( std::acos( sin_cos_range( v2z_r1 ) ) );
	} else if ( v2z_r1 > 1.0 ) {
		theta = 0.0;
	} else {
		theta = 180.0;
	}
}

///////////////////////////////////////////
/// Private: Methods, Data Initializers ///
///////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin SS_penalty_initializer
///
/// @brief
///  Penalty for pairing strand dimers that are close in sequence.
///
/// @detailed
///  Inferred from the log ratio of pairing probabilities of strands
///  in the PDB vs. in Rosetta decoys. Calculated as a function of
///  strand separation.
///
/// @param  SS-penalty - [in/out]? -
///
/// @authors rhiju
///
/// @last_modified Feb. 15, 2006
/////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::SS_penalty_initializer(
	FArray1D_float & SS_penalty
)
{
	// For strand separations less than 5, statistics become small, so
	// set penalty to be a constant.
		 		SS_penalty( 1) = 1.13386;
		 		SS_penalty( 2) = 1.13386;
		 		SS_penalty( 3) = 1.13386;
		 		SS_penalty( 4) = 1.13386;
		 		SS_penalty( 5) = 1.13386;
		 		SS_penalty( 6) = 0.70241;
		 		SS_penalty( 7) = 0.57908;
		 		SS_penalty( 8) = 0.44451;
		 		SS_penalty( 9) = 0.31653;
		 		SS_penalty(10) = 0.22074;
		 		SS_penalty(11) = 0.14869;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin m_term_initializer
///
/// @brief
///
/// @detailed
///
/// @param  m_term - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
SecondaryStructureEval::m_term_initializer(
	FArray1D_float & m_term
)
{
	m_term(1) = 1.87f;
	m_term(2) = .61f;
	m_term(3) = .74f;
	m_term(4) = .17f; // score for 4 or more
}

/* struct catalog */
void
SecondaryStructureEval::idsn_initializer(
	FArray1D_int & idsn
)
{
	// triangle-2 random numbers
	//     data idsn/56,167,278,278,167,56/
	// sort of triangle-4 random numbers
	int i = 0;
	idsn( ++i ) = 5596;
	idsn( ++i ) = 16581;
	idsn( ++i ) = 27823;
	idsn( ++i ) = 27823;
	idsn( ++i ) = 16581;
	idsn( ++i ) = 5596;
}

void
SecondaryStructureEval::ids_initializer(
	FArray1D_int & ids
)
{
	int i = 0;
	ids( ++i ) = 1;
	ids( ++i ) = 48;
	ids( ++i ) = 368;
	ids( ++i ) = 2378;
	ids( ++i ) = 7141;
	ids( ++i ) = 8904;
}

void
SecondaryStructureEval::ssdist_initializer(
	FArray2D_float & ssdist
)
{
	//  DATA 0-12A, GOES BACK TO PROTEINS 1999 PAPER
	//     data ssdist/2.652527,0.7284873,0.0176830,-0.2566608,
	//    #           -1.471609,0.0104174,0.0679096, 0.4667910/

	//  DATA 0-12A+, CREATED JANUARY 30, 1999
	FArray1Da_float ssdist1d( ssdist, ssdist.size() ); // 1D view
	int i = 0;
	ssdist1d( ++i ) = 2.3962;
	ssdist1d( ++i ) = 0.56921;
	ssdist1d( ++i ) = -0.20262;
	ssdist1d( ++i ) = -0.55172;
	ssdist1d( ++i ) = -1.6408;
	ssdist1d( ++i ) = -0.63196;
	ssdist1d( ++i ) = -0.57115;
	ssdist1d( ++i ) = -0.26221;
}

void
SecondaryStructureEval::cbeta_den6_initializer(
	FArray1D_float & cbeta_den6
)
{
	// Inactive DATA intializer from original Fortran code
	//      data cbeta_den6/0.6189,0.6189,0.3332,0.2583,0.2286,
	//     #                0.2838,0.3604,0.4742,0.6065,0.6065,
	//     #                0.6065,0.6065,0.6065,0.6065,0.6065,
	//     #                0.6065,0.6065,0.6065,0.6065,0.6065,
	//     #                0.6065,0.6065,0.6065,0.6065,0.6065,
	//     #                0.6065,0.6065,0.6065,0.6065,0.6065,
	//     #                0.6065,0.6065,0.6065,0.6065,0.6065,
	//     #                0.6065,0.6065,0.6065,0.6065,0.6065,
	//     #                0.6065,0.6065,0.6065,0.6065,0.6065/
	int i = 0;
	cbeta_den6( ++i ) = 0.6189;
	cbeta_den6( ++i ) = 0.6198;
	cbeta_den6( ++i ) = 0.3573;
	cbeta_den6( ++i ) = 0.2799;
	cbeta_den6( ++i ) = 0.2288;
	cbeta_den6( ++i ) = 0.2755;
	cbeta_den6( ++i ) = 0.3067;
	cbeta_den6( ++i ) = 0.4102;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
	cbeta_den6( ++i ) = 0.6065;
}

void
SecondaryStructureEval::cbeta_den12_initializer(
	FArray1D_float & cbeta_den12
)
{
	// Inactive DATA intializer from original Fortran code
	//      data cbeta_den12/0.1383,0.1383,0.1383,0.1309,0.1157,
	//     #                 0.1069,0.1013,0.0944,0.0882,0.0835,
	//     #                 0.0771,0.0757,0.0740,0.0731,0.0704,
	//     #                 0.0703,0.0688,0.0694,0.0709,0.0720,
	//     #                 0.0729,0.0744,0.0762,0.0791,0.0815,
	//     #                 0.0825,0.0861,0.0884,0.0922,0.0942,
	//     #                 0.0963,0.1022,0.1037,0.1091,0.1107,
	//     #                 0.1159,0.1215,0.1315,0.1323,0.1427,
	//     #                 0.1506,0.1564,0.1564,0.1564,0.1564/
	int i = 0;
	cbeta_den12( ++i ) = 0.1383;
	cbeta_den12( ++i ) = 0.1383;
	cbeta_den12( ++i ) = 0.1383;
	cbeta_den12( ++i ) = 0.1383;
	cbeta_den12( ++i ) = 0.1383;
	cbeta_den12( ++i ) = 0.1309;
	cbeta_den12( ++i ) = 0.1157;
	cbeta_den12( ++i ) = 0.1069;
	cbeta_den12( ++i ) = 0.1013;
	cbeta_den12( ++i ) = 0.0944;
	cbeta_den12( ++i ) = 0.0882;
	cbeta_den12( ++i ) = 0.0835;
	cbeta_den12( ++i ) = 0.0771;
	cbeta_den12( ++i ) = 0.0757;
	cbeta_den12( ++i ) = 0.074;
	cbeta_den12( ++i ) = 0.0731;
	cbeta_den12( ++i ) = 0.0704;
	cbeta_den12( ++i ) = 0.0703;
	cbeta_den12( ++i ) = 0.0688;
	cbeta_den12( ++i ) = 0.0694;
	cbeta_den12( ++i ) = 0.0709;
	cbeta_den12( ++i ) = 0.072;
	cbeta_den12( ++i ) = 0.0729;
	cbeta_den12( ++i ) = 0.0744;
	cbeta_den12( ++i ) = 0.0762;
	cbeta_den12( ++i ) = 0.0791;
	cbeta_den12( ++i ) = 0.0815;
	cbeta_den12( ++i ) = 0.0825;
	cbeta_den12( ++i ) = 0.0861;
	cbeta_den12( ++i ) = 0.0884;
	cbeta_den12( ++i ) = 0.0922;
	cbeta_den12( ++i ) = 0.0942;
	cbeta_den12( ++i ) = 0.0963;
	cbeta_den12( ++i ) = 0.1022;
	cbeta_den12( ++i ) = 0.1037;
	cbeta_den12( ++i ) = 0.1091;
	cbeta_den12( ++i ) = 0.1107;
	cbeta_den12( ++i ) = 0.1159;
	cbeta_den12( ++i ) = 0.1215;
	cbeta_den12( ++i ) = 0.1315;
	cbeta_den12( ++i ) = 0.1323;
	cbeta_den12( ++i ) = 0.1427;
	cbeta_den12( ++i ) = 0.1506;
	cbeta_den12( ++i ) = 0.1564;
	cbeta_den12( ++i ) = 0.1564;
}

void
SecondaryStructureEval::hs_dp_initializer(
	FArray1D_float & hs_dp
)
{
	int i = 0;
	hs_dp( ++i ) = 0.416;
	hs_dp( ++i ) = -0.412;
	hs_dp( ++i ) = -0.542;
	hs_dp( ++i ) = -0.489;
	hs_dp( ++i ) = -0.351;
	hs_dp( ++i ) = -0.104;
	hs_dp( ++i ) = 0.211;
	hs_dp( ++i ) = 0.494;
	hs_dp( ++i ) = 0.942;
	hs_dp( ++i ) = 1.897;
}

void
SecondaryStructureEval::rsigma_dot_initializer(
	FArray4D_float & rsigma_dot
)
{
	// section 12
	//js --------------------------
	//js new rsigma stats that take into account whether the first
	//js c=o bond vector points towards away from the other dimer
	rsigma_dot(  1,  1,  1,  2 ) = -1.038100;
	rsigma_dot(  1,  2,  1,  2 ) = -1.038100;
	rsigma_dot(  1,  3,  1,  2 ) = -1.038100;
	rsigma_dot(  1,  4,  1,  2 ) = -1.038100;
	rsigma_dot(  1,  5,  1,  2 ) = -1.038100;
	rsigma_dot(  1,  6,  1,  2 ) = -1.038100;
	rsigma_dot(  1,  7,  1,  2 ) = -1.038100;
	rsigma_dot(  1,  8,  1,  2 ) = -2.984000;
	rsigma_dot(  1,  9,  1,  2 ) = -3.746110;
	rsigma_dot(  1, 10,  1,  2 ) = -3.746110;
	rsigma_dot(  1, 11,  1,  2 ) = -2.647500;
	rsigma_dot(  1, 12,  1,  2 ) = -1.038100;
	rsigma_dot(  1, 13,  1,  2 ) = -1.038100;
	rsigma_dot(  1, 14,  1,  2 ) = -1.038100;
	rsigma_dot(  1, 15,  1,  2 ) = -1.038100;
	rsigma_dot(  1, 16,  1,  2 ) = -1.038100;
	rsigma_dot(  1, 17,  1,  2 ) = -1.038100;
	rsigma_dot(  1, 18,  1,  2 ) = -1.038100;
	rsigma_dot(  2,  1,  1,  2 ) = -0.973500;
	rsigma_dot(  2,  2,  1,  2 ) = -0.973500;
	rsigma_dot(  2,  3,  1,  2 ) = -0.973500;
	rsigma_dot(  2,  4,  1,  2 ) = -0.973500;
	rsigma_dot(  2,  5,  1,  2 ) = -0.973500;
	rsigma_dot(  2,  6,  1,  2 ) = -0.973500;
	rsigma_dot(  2,  7,  1,  2 ) = -2.072100;
	rsigma_dot(  2,  8,  1,  2 ) = -0.973500;
	rsigma_dot(  2,  9,  1,  2 ) = -4.943820;
	rsigma_dot(  2, 10,  1,  2 ) = -5.178220;
	rsigma_dot(  2, 11,  1,  2 ) = -4.823670;
	rsigma_dot(  2, 12,  1,  2 ) = -2.919400;
	rsigma_dot(  2, 13,  1,  2 ) = -2.583000;
	rsigma_dot(  2, 14,  1,  2 ) = -0.973500;
	rsigma_dot(  2, 15,  1,  2 ) = -0.973500;
	rsigma_dot(  2, 16,  1,  2 ) = -0.973500;
	rsigma_dot(  2, 17,  1,  2 ) = -0.973500;
	rsigma_dot(  2, 18,  1,  2 ) = -0.973500;
	rsigma_dot(  3,  1,  1,  2 ) = -0.912900;
	rsigma_dot(  3,  2,  1,  2 ) = -0.912900;
	rsigma_dot(  3,  3,  1,  2 ) = -0.912900;
	rsigma_dot(  3,  4,  1,  2 ) = -0.912900;
	rsigma_dot(  3,  5,  1,  2 ) = -0.912900;
	rsigma_dot(  3,  6,  1,  2 ) = -0.912900;
	rsigma_dot(  3,  7,  1,  2 ) = -2.011500;
	rsigma_dot(  3,  8,  1,  2 ) = -4.048400;
	rsigma_dot(  3,  9,  1,  2 ) = -5.981810;
	rsigma_dot(  3, 10,  1,  2 ) = -7.006470;
	rsigma_dot(  3, 11,  1,  2 ) = -6.696730;
	rsigma_dot(  3, 12,  1,  2 ) = -4.131780;
	rsigma_dot(  3, 13,  1,  2 ) = -0.912900;
	rsigma_dot(  3, 14,  1,  2 ) = -0.912900;
	rsigma_dot(  3, 15,  1,  2 ) = -0.912900;
	rsigma_dot(  3, 16,  1,  2 ) = -0.912900;
	rsigma_dot(  3, 17,  1,  2 ) = -0.912900;
	rsigma_dot(  3, 18,  1,  2 ) = -0.912900;
	rsigma_dot(  4,  1,  1,  2 ) = -0.855700;
	rsigma_dot(  4,  2,  1,  2 ) = -0.855700;
	rsigma_dot(  4,  3,  1,  2 ) = -0.855700;
	rsigma_dot(  4,  4,  1,  2 ) = -0.855700;
	rsigma_dot(  4,  5,  1,  2 ) = -0.855700;
	rsigma_dot(  4,  6,  1,  2 ) = -2.465200;
	rsigma_dot(  4,  7,  1,  2 ) = -3.420700;
	rsigma_dot(  4,  8,  1,  2 ) = -3.900270;
	rsigma_dot(  4,  9,  1,  2 ) = -6.315330;
	rsigma_dot(  4, 10,  1,  2 ) = -8.116270;
	rsigma_dot(  4, 11,  1,  2 ) = -8.329380;
	rsigma_dot(  4, 12,  1,  2 ) = -5.298390;
	rsigma_dot(  4, 13,  1,  2 ) = -1.954400;
	rsigma_dot(  4, 14,  1,  2 ) = -0.855700;
	rsigma_dot(  4, 15,  1,  2 ) = -0.855700;
	rsigma_dot(  4, 16,  1,  2 ) = -0.855700;
	rsigma_dot(  4, 17,  1,  2 ) = -0.855700;
	rsigma_dot(  4, 18,  1,  2 ) = -0.855700;
	rsigma_dot(  5,  1,  1,  2 ) = -0.801700;
	rsigma_dot(  5,  2,  1,  2 ) = -0.801700;
	rsigma_dot(  5,  3,  1,  2 ) = -0.801700;
	rsigma_dot(  5,  4,  1,  2 ) = -0.801700;
	rsigma_dot(  5,  5,  1,  2 ) = -0.801700;
	rsigma_dot(  5,  6,  1,  2 ) = -1.900300;
	rsigma_dot(  5,  7,  1,  2 ) = -3.634900;
	rsigma_dot(  5,  8,  1,  2 ) = -5.119160;
	rsigma_dot(  5,  9,  1,  2 ) = -6.381410;
	rsigma_dot(  5, 10,  1,  2 ) = -9.110860;
	rsigma_dot(  5, 11,  1,  2 ) = -9.615860;
	rsigma_dot(  5, 12,  1,  2 ) = -6.566870;
	rsigma_dot(  5, 13,  1,  2 ) = -2.411100;
	rsigma_dot(  5, 14,  1,  2 ) = -1.900300;
	rsigma_dot(  5, 15,  1,  2 ) = -0.801700;
	rsigma_dot(  5, 16,  1,  2 ) = -0.801700;
	rsigma_dot(  5, 17,  1,  2 ) = -0.801700;
	rsigma_dot(  5, 18,  1,  2 ) = -0.801700;
	rsigma_dot(  6,  1,  1,  2 ) = -0.750400;
	rsigma_dot(  6,  2,  1,  2 ) = -0.750400;
	rsigma_dot(  6,  3,  1,  2 ) = -0.750400;
	rsigma_dot(  6,  4,  1,  2 ) = -0.750400;
	rsigma_dot(  6,  5,  1,  2 ) = -0.750400;
	rsigma_dot(  6,  6,  1,  2 ) = -2.359800;
	rsigma_dot(  6,  7,  1,  2 ) = -4.511580;
	rsigma_dot(  6,  8,  1,  2 ) = -5.325090;
	rsigma_dot(  6,  9,  1,  2 ) = -6.768980;
	rsigma_dot(  6, 10,  1,  2 ) = -8.613260;
	rsigma_dot(  6, 11,  1,  2 ) = -9.405770;
	rsigma_dot(  6, 12,  1,  2 ) = -7.032650;
	rsigma_dot(  6, 13,  1,  2 ) = -3.885880;
	rsigma_dot(  6, 14,  1,  2 ) = -3.315300;
	rsigma_dot(  6, 15,  1,  2 ) = -0.750400;
	rsigma_dot(  6, 16,  1,  2 ) = -0.750400;
	rsigma_dot(  6, 17,  1,  2 ) = -0.750400;
	rsigma_dot(  6, 18,  1,  2 ) = -0.750400;
	rsigma_dot(  7,  1,  1,  2 ) = -0.701600;
	rsigma_dot(  7,  2,  1,  2 ) = -0.701600;
	rsigma_dot(  7,  3,  1,  2 ) = -0.701600;
	rsigma_dot(  7,  4,  1,  2 ) = -0.701600;
	rsigma_dot(  7,  5,  1,  2 ) = -0.701600;
	rsigma_dot(  7,  6,  1,  2 ) = -2.311000;
	rsigma_dot(  7,  7,  1,  2 ) = -4.365150;
	rsigma_dot(  7,  8,  1,  2 ) = -5.446520;
	rsigma_dot(  7,  9,  1,  2 ) = -6.795160;
	rsigma_dot(  7, 10,  1,  2 ) = -7.145720;
	rsigma_dot(  7, 11,  1,  2 ) = -7.527050;
	rsigma_dot(  7, 12,  1,  2 ) = -6.885740;
	rsigma_dot(  7, 13,  1,  2 ) = -4.508250;
	rsigma_dot(  7, 14,  1,  2 ) = -1.800200;
	rsigma_dot(  7, 15,  1,  2 ) = -2.647500;
	rsigma_dot(  7, 16,  1,  2 ) = -0.701600;
	rsigma_dot(  7, 17,  1,  2 ) = -0.701600;
	rsigma_dot(  7, 18,  1,  2 ) = -0.701600;
	rsigma_dot(  8,  1,  1,  2 ) = -0.655100;
	rsigma_dot(  8,  2,  1,  2 ) = -0.655100;
	rsigma_dot(  8,  3,  1,  2 ) = -0.655100;
	rsigma_dot(  8,  4,  1,  2 ) = -0.655100;
	rsigma_dot(  8,  5,  1,  2 ) = -0.655100;
	rsigma_dot(  8,  6,  1,  2 ) = -3.699600;
	rsigma_dot(  8,  7,  1,  2 ) = -4.089060;
	rsigma_dot(  8,  8,  1,  2 ) = -5.417250;
	rsigma_dot(  8,  9,  1,  2 ) = -6.293430;
	rsigma_dot(  8, 10,  1,  2 ) = -6.663890;
	rsigma_dot(  8, 11,  1,  2 ) = -6.788470;
	rsigma_dot(  8, 12,  1,  2 ) = -6.368800;
	rsigma_dot(  8, 13,  1,  2 ) = -4.732610;
	rsigma_dot(  8, 14,  1,  2 ) = -1.753700;
	rsigma_dot(  8, 15,  1,  2 ) = -0.655100;
	rsigma_dot(  8, 16,  1,  2 ) = -0.655100;
	rsigma_dot(  8, 17,  1,  2 ) = -0.655100;
	rsigma_dot(  8, 18,  1,  2 ) = -0.655100;
	rsigma_dot(  9,  1,  1,  2 ) = -0.610600;
	rsigma_dot(  9,  2,  1,  2 ) = -0.610600;
	rsigma_dot(  9,  3,  1,  2 ) = -0.610600;
	rsigma_dot(  9,  4,  1,  2 ) = -0.610600;
	rsigma_dot(  9,  5,  1,  2 ) = -1.709200;
	rsigma_dot(  9,  6,  1,  2 ) = -3.906460;
	rsigma_dot(  9,  7,  1,  2 ) = -3.906460;
	rsigma_dot(  9,  8,  1,  2 ) = -5.355550;
	rsigma_dot(  9,  9,  1,  2 ) = -6.070210;
	rsigma_dot(  9, 10,  1,  2 ) = -5.971910;
	rsigma_dot(  9, 11,  1,  2 ) = -6.070210;
	rsigma_dot(  9, 12,  1,  2 ) = -5.852370;
	rsigma_dot(  9, 13,  1,  2 ) = -4.901080;
	rsigma_dot(  9, 14,  1,  2 ) = -2.807800;
	rsigma_dot(  9, 15,  1,  2 ) = -2.220100;
	rsigma_dot(  9, 16,  1,  2 ) = -0.610600;
	rsigma_dot(  9, 17,  1,  2 ) = -1.709200;
	rsigma_dot(  9, 18,  1,  2 ) = -0.610600;
	rsigma_dot( 10,  1,  1,  2 ) = -0.568100;
	rsigma_dot( 10,  2,  1,  2 ) = -0.568100;
	rsigma_dot( 10,  3,  1,  2 ) = -0.568100;
	rsigma_dot( 10,  4,  1,  2 ) = -2.177500;
	rsigma_dot( 10,  5,  1,  2 ) = -2.765300;
	rsigma_dot( 10,  6,  1,  2 ) = -4.002050;
	rsigma_dot( 10,  7,  1,  2 ) = -4.802170;
	rsigma_dot( 10,  8,  1,  2 ) = -5.598500;
	rsigma_dot( 10,  9,  1,  2 ) = -5.809810;
	rsigma_dot( 10, 10,  1,  2 ) = -5.809810;
	rsigma_dot( 10, 11,  1,  2 ) = -5.732850;
	rsigma_dot( 10, 12,  1,  2 ) = -5.572010;
	rsigma_dot( 10, 13,  1,  2 ) = -4.858520;
	rsigma_dot( 10, 14,  1,  2 ) = -3.935360;
	rsigma_dot( 10, 15,  1,  2 ) = -1.666700;
	rsigma_dot( 10, 16,  1,  2 ) = -1.666700;
	rsigma_dot( 10, 17,  1,  2 ) = -0.568100;
	rsigma_dot( 10, 18,  1,  2 ) = -1.666700;
	rsigma_dot( 11,  1,  1,  2 ) = -0.527200;
	rsigma_dot( 11,  2,  1,  2 ) = -0.527200;
	rsigma_dot( 11,  3,  1,  2 ) = -0.527200;
	rsigma_dot( 11,  4,  1,  2 ) = -1.625900;
	rsigma_dot( 11,  5,  1,  2 ) = -3.746110;
	rsigma_dot( 11,  6,  1,  2 ) = -4.190800;
	rsigma_dot( 11,  7,  1,  2 ) = -4.761350;
	rsigma_dot( 11,  8,  1,  2 ) = -5.897880;
	rsigma_dot( 11,  9,  1,  2 ) = -5.850250;
	rsigma_dot( 11, 10,  1,  2 ) = -5.789930;
	rsigma_dot( 11, 11,  1,  2 ) = -5.371430;
	rsigma_dot( 11, 12,  1,  2 ) = -5.608640;
	rsigma_dot( 11, 13,  1,  2 ) = -5.306360;
	rsigma_dot( 11, 14,  1,  2 ) = -4.288440;
	rsigma_dot( 11, 15,  1,  2 ) = -3.092200;
	rsigma_dot( 11, 16,  1,  2 ) = -0.527200;
	rsigma_dot( 11, 17,  1,  2 ) = -0.527200;
	rsigma_dot( 11, 18,  1,  2 ) = -0.527200;
	rsigma_dot( 12,  1,  1,  2 ) = -0.488000;
	rsigma_dot( 12,  2,  1,  2 ) = -0.488000;
	rsigma_dot( 12,  3,  1,  2 ) = -0.488000;
	rsigma_dot( 12,  4,  1,  2 ) = -3.623500;
	rsigma_dot( 12,  5,  1,  2 ) = -4.722120;
	rsigma_dot( 12,  6,  1,  2 ) = -4.692710;
	rsigma_dot( 12,  7,  1,  2 ) = -5.197550;
	rsigma_dot( 12,  8,  1,  2 ) = -5.544260;
	rsigma_dot( 12,  9,  1,  2 ) = -5.740290;
	rsigma_dot( 12, 10,  1,  2 ) = -5.478450;
	rsigma_dot( 12, 11,  1,  2 ) = -5.347830;
	rsigma_dot( 12, 12,  1,  2 ) = -5.363220;
	rsigma_dot( 12, 13,  1,  2 ) = -5.103140;
	rsigma_dot( 12, 14,  1,  2 ) = -4.722120;
	rsigma_dot( 12, 15,  1,  2 ) = -3.984530;
	rsigma_dot( 12, 16,  1,  2 ) = -3.196100;
	rsigma_dot( 12, 17,  1,  2 ) = -0.488000;
	rsigma_dot( 12, 18,  1,  2 ) = -0.488000;

	// section 11
	rsigma_dot(  1,  1,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  2,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  3,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  4,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  5,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  6,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  7,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  8,  1,  1 ) = -0.552300;
	rsigma_dot(  1,  9,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 10,  1,  1 ) = -1.651000;
	rsigma_dot(  1, 11,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 12,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 13,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 14,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 15,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 16,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 17,  1,  1 ) = -0.552300;
	rsigma_dot(  1, 18,  1,  1 ) = -0.552300;
	rsigma_dot(  2,  1,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  2,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  3,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  4,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  5,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  6,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  7,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  8,  1,  1 ) = -0.487800;
	rsigma_dot(  2,  9,  1,  1 ) = -0.487800;
	rsigma_dot(  2, 10,  1,  1 ) = -1.586400;
	rsigma_dot(  2, 11,  1,  1 ) = -2.097200;
	rsigma_dot(  2, 12,  1,  1 ) = -1.586400;
	rsigma_dot(  2, 13,  1,  1 ) = -0.487800;
	rsigma_dot(  2, 14,  1,  1 ) = -0.487800;
	rsigma_dot(  2, 15,  1,  1 ) = -0.487800;
	rsigma_dot(  2, 16,  1,  1 ) = -0.487800;
	rsigma_dot(  2, 17,  1,  1 ) = -0.487800;
	rsigma_dot(  2, 18,  1,  1 ) = -0.487800;
	rsigma_dot(  3,  1,  1,  1 ) = -0.427200;
	rsigma_dot(  3,  2,  1,  1 ) = -0.427200;
	rsigma_dot(  3,  3,  1,  1 ) = -0.427200;
	rsigma_dot(  3,  4,  1,  1 ) = -0.427200;
	rsigma_dot(  3,  5,  1,  1 ) = -0.427200;
	rsigma_dot(  3,  6,  1,  1 ) = -1.525800;
	rsigma_dot(  3,  7,  1,  1 ) = -2.036600;
	rsigma_dot(  3,  8,  1,  1 ) = -2.036600;
	rsigma_dot(  3,  9,  1,  1 ) = -0.427200;
	rsigma_dot(  3, 10,  1,  1 ) = -2.373100;
	rsigma_dot(  3, 11,  1,  1 ) = -1.525800;
	rsigma_dot(  3, 12,  1,  1 ) = -2.992100;
	rsigma_dot(  3, 13,  1,  1 ) = -2.373100;
	rsigma_dot(  3, 14,  1,  1 ) = -0.427200;
	rsigma_dot(  3, 15,  1,  1 ) = -0.427200;
	rsigma_dot(  3, 16,  1,  1 ) = -0.427200;
	rsigma_dot(  3, 17,  1,  1 ) = -0.427200;
	rsigma_dot(  3, 18,  1,  1 ) = -0.427200;
	rsigma_dot(  4,  1,  1,  1 ) = -0.370000;
	rsigma_dot(  4,  2,  1,  1 ) = -0.370000;
	rsigma_dot(  4,  3,  1,  1 ) = -0.370000;
	rsigma_dot(  4,  4,  1,  1 ) = -0.370000;
	rsigma_dot(  4,  5,  1,  1 ) = -0.370000;
	rsigma_dot(  4,  6,  1,  1 ) = -0.370000;
	rsigma_dot(  4,  7,  1,  1 ) = -2.315900;
	rsigma_dot(  4,  8,  1,  1 ) = -2.567200;
	rsigma_dot(  4,  9,  1,  1 ) = -2.767900;
	rsigma_dot(  4, 10,  1,  1 ) = -3.588900;
	rsigma_dot(  4, 11,  1,  1 ) = -3.414500;
	rsigma_dot(  4, 12,  1,  1 ) = -4.033580;
	rsigma_dot(  4, 13,  1,  1 ) = -3.804010;
	rsigma_dot(  4, 14,  1,  1 ) = -0.370000;
	rsigma_dot(  4, 15,  1,  1 ) = -0.370000;
	rsigma_dot(  4, 16,  1,  1 ) = -0.370000;
	rsigma_dot(  4, 17,  1,  1 ) = -0.370000;
	rsigma_dot(  4, 18,  1,  1 ) = -0.370000;
	rsigma_dot(  5,  1,  1,  1 ) = -0.316000;
	rsigma_dot(  5,  2,  1,  1 ) = -0.316000;
	rsigma_dot(  5,  3,  1,  1 ) = -0.316000;
	rsigma_dot(  5,  4,  1,  1 ) = -0.316000;
	rsigma_dot(  5,  5,  1,  1 ) = -0.316000;
	rsigma_dot(  5,  6,  1,  1 ) = -1.925400;
	rsigma_dot(  5,  7,  1,  1 ) = -3.683200;
	rsigma_dot(  5,  8,  1,  1 ) = -4.122610;
	rsigma_dot(  5,  9,  1,  1 ) = -3.812460;
	rsigma_dot(  5, 10,  1,  1 ) = -3.683200;
	rsigma_dot(  5, 11,  1,  1 ) = -3.360500;
	rsigma_dot(  5, 12,  1,  1 ) = -4.969910;
	rsigma_dot(  5, 13,  1,  1 ) = -4.869830;
	rsigma_dot(  5, 14,  1,  1 ) = -3.149200;
	rsigma_dot(  5, 15,  1,  1 ) = -0.316000;
	rsigma_dot(  5, 16,  1,  1 ) = -0.316000;
	rsigma_dot(  5, 17,  1,  1 ) = -0.316000;
	rsigma_dot(  5, 18,  1,  1 ) = -0.316000;
	rsigma_dot(  6,  1,  1,  1 ) = -0.264700;
	rsigma_dot(  6,  2,  1,  1 ) = -0.264700;
	rsigma_dot(  6,  3,  1,  1 ) = -0.264700;
	rsigma_dot(  6,  4,  1,  1 ) = -0.264700;
	rsigma_dot(  6,  5,  1,  1 ) = -0.264700;
	rsigma_dot(  6,  6,  1,  1 ) = -2.829600;
	rsigma_dot(  6,  7,  1,  1 ) = -5.680760;
	rsigma_dot(  6,  8,  1,  1 ) = -5.076840;
	rsigma_dot(  6,  9,  1,  1 ) = -4.271990;
	rsigma_dot(  6, 10,  1,  1 ) = -4.234950;
	rsigma_dot(  6, 11,  1,  1 ) = -3.761170;
	rsigma_dot(  6, 12,  1,  1 ) = -5.567960;
	rsigma_dot(  6, 13,  1,  1 ) = -6.108200;
	rsigma_dot(  6, 14,  1,  1 ) = -4.956010;
	rsigma_dot(  6, 15,  1,  1 ) = -1.874100;
	rsigma_dot(  6, 16,  1,  1 ) = -0.264700;
	rsigma_dot(  6, 17,  1,  1 ) = -0.264700;
	rsigma_dot(  6, 18,  1,  1 ) = -0.264700;
	rsigma_dot(  7,  1,  1,  1 ) = -0.215900;
	rsigma_dot(  7,  2,  1,  1 ) = -0.215900;
	rsigma_dot(  7,  3,  1,  1 ) = -0.215900;
	rsigma_dot(  7,  4,  1,  1 ) = -0.215900;
	rsigma_dot(  7,  5,  1,  1 ) = -0.215900;
	rsigma_dot(  7,  6,  1,  1 ) = -5.011660;
	rsigma_dot(  7,  7,  1,  1 ) = -7.313420;
	rsigma_dot(  7,  8,  1,  1 ) = -6.580620;
	rsigma_dot(  7,  9,  1,  1 ) = -4.726730;
	rsigma_dot(  7, 10,  1,  1 ) = -4.748470;
	rsigma_dot(  7, 11,  1,  1 ) = -2.923900;
	rsigma_dot(  7, 12,  1,  1 ) = -5.700670;
	rsigma_dot(  7, 13,  1,  1 ) = -6.799280;
	rsigma_dot(  7, 14,  1,  1 ) = -6.059410;
	rsigma_dot(  7, 15,  1,  1 ) = -2.161800;
	rsigma_dot(  7, 16,  1,  1 ) = -0.215900;
	rsigma_dot(  7, 17,  1,  1 ) = -0.215900;
	rsigma_dot(  7, 18,  1,  1 ) = -0.215900;
	rsigma_dot(  8,  1,  1,  1 ) = -0.169300;
	rsigma_dot(  8,  2,  1,  1 ) = -0.169300;
	rsigma_dot(  8,  3,  1,  1 ) = -0.169300;
	rsigma_dot(  8,  4,  1,  1 ) = -0.169300;
	rsigma_dot(  8,  5,  1,  1 ) = -0.169300;
	rsigma_dot(  8,  6,  1,  1 ) = -6.132930;
	rsigma_dot(  8,  7,  1,  1 ) = -8.735140;
	rsigma_dot(  8,  8,  1,  1 ) = -7.789560;
	rsigma_dot(  8,  9,  1,  1 ) = -5.199790;
	rsigma_dot(  8, 10,  1,  1 ) = -4.981530;
	rsigma_dot(  8, 11,  1,  1 ) = -3.882920;
	rsigma_dot(  8, 12,  1,  1 ) = -5.059700;
	rsigma_dot(  8, 13,  1,  1 ) = -7.329420;
	rsigma_dot(  8, 14,  1,  1 ) = -7.217740;
	rsigma_dot(  8, 15,  1,  1 ) = -4.723230;
	rsigma_dot(  8, 16,  1,  1 ) = -0.169300;
	rsigma_dot(  8, 17,  1,  1 ) = -0.169300;
	rsigma_dot(  8, 18,  1,  1 ) = -0.169300;
	rsigma_dot(  9,  1,  1,  1 ) = -0.124900;
	rsigma_dot(  9,  2,  1,  1 ) = -0.124900;
	rsigma_dot(  9,  3,  1,  1 ) = -0.124900;
	rsigma_dot(  9,  4,  1,  1 ) = -1.223500;
	rsigma_dot(  9,  5,  1,  1 ) = -3.069300;
	rsigma_dot(  9,  6,  1,  1 ) = -6.756900;
	rsigma_dot(  9,  7,  1,  1 ) = -8.997660;
	rsigma_dot(  9,  8,  1,  1 ) = -7.292710;
	rsigma_dot(  9,  9,  1,  1 ) = -5.704630;
	rsigma_dot(  9, 10,  1,  1 ) = -4.442390;
	rsigma_dot(  9, 11,  1,  1 ) = -4.202430;
	rsigma_dot(  9, 12,  1,  1 ) = -4.567550;
	rsigma_dot(  9, 13,  1,  1 ) = -7.550850;
	rsigma_dot(  9, 14,  1,  1 ) = -8.291970;
	rsigma_dot(  9, 15,  1,  1 ) = -5.617960;
	rsigma_dot(  9, 16,  1,  1 ) = -1.223500;
	rsigma_dot(  9, 17,  1,  1 ) = -1.223500;
	rsigma_dot(  9, 18,  1,  1 ) = -0.124900;
	rsigma_dot( 10,  1,  1,  1 ) = -0.082300;
	rsigma_dot( 10,  2,  1,  1 ) = -0.082300;
	rsigma_dot( 10,  3,  1,  1 ) = -0.082300;
	rsigma_dot( 10,  4,  1,  1 ) = -1.180900;
	rsigma_dot( 10,  5,  1,  1 ) = -3.932490;
	rsigma_dot( 10,  6,  1,  1 ) = -6.858840;
	rsigma_dot( 10,  7,  1,  1 ) = -8.186740;
	rsigma_dot( 10,  8,  1,  1 ) = -6.228670;
	rsigma_dot( 10,  9,  1,  1 ) = -5.720690;
	rsigma_dot( 10, 10,  1,  1 ) = -4.548250;
	rsigma_dot( 10, 11,  1,  1 ) = -4.451790;
	rsigma_dot( 10, 12,  1,  1 ) = -4.548250;
	rsigma_dot( 10, 13,  1,  1 ) = -7.242410;
	rsigma_dot( 10, 14,  1,  1 ) = -8.619920;
	rsigma_dot( 10, 15,  1,  1 ) = -6.349540;
	rsigma_dot( 10, 16,  1,  1 ) = -3.026800;
	rsigma_dot( 10, 17,  1,  1 ) = -1.180900;
	rsigma_dot( 10, 18,  1,  1 ) = -1.180900;
	rsigma_dot( 11,  1,  1,  1 ) = -0.041500;
	rsigma_dot( 11,  2,  1,  1 ) = -0.041500;
	rsigma_dot( 11,  3,  1,  1 ) = -0.041500;
	rsigma_dot( 11,  4,  1,  1 ) = -1.987400;
	rsigma_dot( 11,  5,  1,  1 ) = -4.275620;
	rsigma_dot( 11,  6,  1,  1 ) = -6.582550;
	rsigma_dot( 11,  7,  1,  1 ) = -6.915710;
	rsigma_dot( 11,  8,  1,  1 ) = -6.166200;
	rsigma_dot( 11,  9,  1,  1 ) = -5.483930;
	rsigma_dot( 11, 10,  1,  1 ) = -4.595390;
	rsigma_dot( 11, 11,  1,  1 ) = -4.616230;
	rsigma_dot( 11, 12,  1,  1 ) = -4.931860;
	rsigma_dot( 11, 13,  1,  1 ) = -6.246070;
	rsigma_dot( 11, 14,  1,  1 ) = -8.296560;
	rsigma_dot( 11, 15,  1,  1 ) = -6.734840;
	rsigma_dot( 11, 16,  1,  1 ) = -3.337400;
	rsigma_dot( 11, 17,  1,  1 ) = -1.140100;
	rsigma_dot( 11, 18,  1,  1 ) = -0.041500;
	rsigma_dot( 12,  1,  1,  1 ) = -0.002300;
	rsigma_dot( 12,  2,  1,  1 ) = -0.002300;
	rsigma_dot( 12,  3,  1,  1 ) = -0.002300;
	rsigma_dot( 12,  4,  1,  1 ) = -2.199500;
	rsigma_dot( 12,  5,  1,  1 ) = -4.729680;
	rsigma_dot( 12,  6,  1,  1 ) = -6.131350;
	rsigma_dot( 12,  7,  1,  1 ) = -5.729140;
	rsigma_dot( 12,  8,  1,  1 ) = -5.604410;
	rsigma_dot( 12,  9,  1,  1 ) = -5.211780;
	rsigma_dot( 12, 10,  1,  1 ) = -4.892640;
	rsigma_dot( 12, 11,  1,  1 ) = -4.992730;
	rsigma_dot( 12, 12,  1,  1 ) = -5.275290;
	rsigma_dot( 12, 13,  1,  1 ) = -5.945090;
	rsigma_dot( 12, 14,  1,  1 ) = -7.716970;
	rsigma_dot( 12, 15,  1,  1 ) = -6.693140;
	rsigma_dot( 12, 16,  1,  1 ) = -3.972590;
	rsigma_dot( 12, 17,  1,  1 ) = -1.948200;
	rsigma_dot( 12, 18,  1,  1 ) = -0.002300;

	// section 22
	rsigma_dot(  1,  1,  2,  2 ) = -0.550000;
	rsigma_dot(  1,  2,  2,  2 ) = -0.550000;
	rsigma_dot(  1,  3,  2,  2 ) = -0.550000;
	rsigma_dot(  1,  4,  2,  2 ) = -0.550000;
	rsigma_dot(  1,  5,  2,  2 ) = -0.550000;
	rsigma_dot(  1,  6,  2,  2 ) = -0.550000;
	rsigma_dot(  1,  7,  2,  2 ) = -0.550000;
	rsigma_dot(  1,  8,  2,  2 ) = -2.159400;
	rsigma_dot(  1,  9,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 10,  2,  2 ) = -2.159400;
	rsigma_dot(  1, 11,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 12,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 13,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 14,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 15,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 16,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 17,  2,  2 ) = -0.550000;
	rsigma_dot(  1, 18,  2,  2 ) = -0.550000;
	rsigma_dot(  2,  1,  2,  2 ) = -0.485500;
	rsigma_dot(  2,  2,  2,  2 ) = -0.485500;
	rsigma_dot(  2,  3,  2,  2 ) = -0.485500;
	rsigma_dot(  2,  4,  2,  2 ) = -0.485500;
	rsigma_dot(  2,  5,  2,  2 ) = -0.485500;
	rsigma_dot(  2,  6,  2,  2 ) = -1.584100;
	rsigma_dot(  2,  7,  2,  2 ) = -3.050400;
	rsigma_dot(  2,  8,  2,  2 ) = -2.094900;
	rsigma_dot(  2,  9,  2,  2 ) = -2.431400;
	rsigma_dot(  2, 10,  2,  2 ) = -1.584100;
	rsigma_dot(  2, 11,  2,  2 ) = -0.485500;
	rsigma_dot(  2, 12,  2,  2 ) = -0.485500;
	rsigma_dot(  2, 13,  2,  2 ) = -0.485500;
	rsigma_dot(  2, 14,  2,  2 ) = -0.485500;
	rsigma_dot(  2, 15,  2,  2 ) = -0.485500;
	rsigma_dot(  2, 16,  2,  2 ) = -0.485500;
	rsigma_dot(  2, 17,  2,  2 ) = -0.485500;
	rsigma_dot(  2, 18,  2,  2 ) = -0.485500;
	rsigma_dot(  3,  1,  2,  2 ) = -0.424800;
	rsigma_dot(  3,  2,  2,  2 ) = -0.424800;
	rsigma_dot(  3,  3,  2,  2 ) = -0.424800;
	rsigma_dot(  3,  4,  2,  2 ) = -0.424800;
	rsigma_dot(  3,  5,  2,  2 ) = -0.424800;
	rsigma_dot(  3,  6,  2,  2 ) = -3.258000;
	rsigma_dot(  3,  7,  2,  2 ) = -4.231500;
	rsigma_dot(  3,  8,  2,  2 ) = -3.980180;
	rsigma_dot(  3,  9,  2,  2 ) = -2.370700;
	rsigma_dot(  3, 10,  2,  2 ) = -1.523400;
	rsigma_dot(  3, 11,  2,  2 ) = -2.822700;
	rsigma_dot(  3, 12,  2,  2 ) = -0.424800;
	rsigma_dot(  3, 13,  2,  2 ) = -0.424800;
	rsigma_dot(  3, 14,  2,  2 ) = -0.424800;
	rsigma_dot(  3, 15,  2,  2 ) = -0.424800;
	rsigma_dot(  3, 16,  2,  2 ) = -0.424800;
	rsigma_dot(  3, 17,  2,  2 ) = -0.424800;
	rsigma_dot(  3, 18,  2,  2 ) = -0.424800;
	rsigma_dot(  4,  1,  2,  2 ) = -0.367700;
	rsigma_dot(  4,  2,  2,  2 ) = -0.367700;
	rsigma_dot(  4,  3,  2,  2 ) = -0.367700;
	rsigma_dot(  4,  4,  2,  2 ) = -0.367700;
	rsigma_dot(  4,  5,  2,  2 ) = -3.312100;
	rsigma_dot(  4,  6,  2,  2 ) = -3.412200;
	rsigma_dot(  4,  7,  2,  2 ) = -5.509340;
	rsigma_dot(  4,  8,  2,  2 ) = -4.128880;
	rsigma_dot(  4,  9,  2,  2 ) = -3.075700;
	rsigma_dot(  4, 10,  2,  2 ) = -2.765600;
	rsigma_dot(  4, 11,  2,  2 ) = -2.765600;
	rsigma_dot(  4, 12,  2,  2 ) = -2.765600;
	rsigma_dot(  4, 13,  2,  2 ) = -0.367700;
	rsigma_dot(  4, 14,  2,  2 ) = -0.367700;
	rsigma_dot(  4, 15,  2,  2 ) = -0.367700;
	rsigma_dot(  4, 16,  2,  2 ) = -0.367700;
	rsigma_dot(  4, 17,  2,  2 ) = -0.367700;
	rsigma_dot(  4, 18,  2,  2 ) = -0.367700;
	rsigma_dot(  5,  1,  2,  2 ) = -0.313600;
	rsigma_dot(  5,  2,  2,  2 ) = -0.313600;
	rsigma_dot(  5,  3,  2,  2 ) = -0.313600;
	rsigma_dot(  5,  4,  2,  2 ) = -1.412200;
	rsigma_dot(  5,  5,  2,  2 ) = -3.021700;
	rsigma_dot(  5,  6,  2,  2 ) = -4.779520;
	rsigma_dot(  5,  7,  2,  2 ) = -6.825360;
	rsigma_dot(  5,  8,  2,  2 ) = -4.657420;
	rsigma_dot(  5,  9,  2,  2 ) = -3.358100;
	rsigma_dot(  5, 10,  2,  2 ) = -3.609400;
	rsigma_dot(  5, 11,  2,  2 ) = -4.245440;
	rsigma_dot(  5, 12,  2,  2 ) = -3.021700;
	rsigma_dot(  5, 13,  2,  2 ) = -1.412200;
	rsigma_dot(  5, 14,  2,  2 ) = -0.313600;
	rsigma_dot(  5, 15,  2,  2 ) = -0.313600;
	rsigma_dot(  5, 16,  2,  2 ) = -0.313600;
	rsigma_dot(  5, 17,  2,  2 ) = -0.313600;
	rsigma_dot(  5, 18,  2,  2 ) = -0.313600;
	rsigma_dot(  6,  1,  2,  2 ) = -0.262300;
	rsigma_dot(  6,  2,  2,  2 ) = -0.262300;
	rsigma_dot(  6,  3,  2,  2 ) = -0.262300;
	rsigma_dot(  6,  4,  2,  2 ) = -1.871800;
	rsigma_dot(  6,  5,  2,  2 ) = -2.208200;
	rsigma_dot(  6,  6,  2,  2 ) = -6.862190;
	rsigma_dot(  6,  7,  2,  2 ) = -7.463490;
	rsigma_dot(  6,  8,  2,  2 ) = -4.837030;
	rsigma_dot(  6,  9,  2,  2 ) = -4.436700;
	rsigma_dot(  6, 10,  2,  2 ) = -4.916280;
	rsigma_dot(  6, 11,  2,  2 ) = -5.687270;
	rsigma_dot(  6, 12,  2,  2 ) = -4.405450;
	rsigma_dot(  6, 13,  2,  2 ) = -2.208200;
	rsigma_dot(  6, 14,  2,  2 ) = -0.262300;
	rsigma_dot(  6, 15,  2,  2 ) = -1.360900;
	rsigma_dot(  6, 16,  2,  2 ) = -0.262300;
	rsigma_dot(  6, 17,  2,  2 ) = -0.262300;
	rsigma_dot(  6, 18,  2,  2 ) = -0.262300;
	rsigma_dot(  7,  1,  2,  2 ) = -0.213500;
	rsigma_dot(  7,  2,  2,  2 ) = -0.213500;
	rsigma_dot(  7,  3,  2,  2 ) = -0.213500;
	rsigma_dot(  7,  4,  2,  2 ) = -0.213500;
	rsigma_dot(  7,  5,  2,  2 ) = -4.256580;
	rsigma_dot(  7,  6,  2,  2 ) = -8.696750;
	rsigma_dot(  7,  7,  2,  2 ) = -7.275720;
	rsigma_dot(  7,  8,  2,  2 ) = -4.746130;
	rsigma_dot(  7,  9,  2,  2 ) = -4.145350;
	rsigma_dot(  7, 10,  2,  2 ) = -4.923060;
	rsigma_dot(  7, 11,  2,  2 ) = -6.725270;
	rsigma_dot(  7, 12,  2,  2 ) = -5.620700;
	rsigma_dot(  7, 13,  2,  2 ) = -2.611400;
	rsigma_dot(  7, 14,  2,  2 ) = -0.213500;
	rsigma_dot(  7, 15,  2,  2 ) = -0.213500;
	rsigma_dot(  7, 16,  2,  2 ) = -0.213500;
	rsigma_dot(  7, 17,  2,  2 ) = -0.213500;
	rsigma_dot(  7, 18,  2,  2 ) = -0.213500;
	rsigma_dot(  8,  1,  2,  2 ) = -0.167000;
	rsigma_dot(  8,  2,  2,  2 ) = -0.167000;
	rsigma_dot(  8,  3,  2,  2 ) = -0.167000;
	rsigma_dot(  8,  4,  2,  2 ) = -1.265600;
	rsigma_dot(  8,  5,  2,  2 ) = -6.383610;
	rsigma_dot(  8,  6,  2,  2 ) = -9.098560;
	rsigma_dot(  8,  7,  2,  2 ) = -6.418910;
	rsigma_dot(  8,  8,  2,  2 ) = -4.561460;
	rsigma_dot(  8,  9,  2,  2 ) = -4.174340;
	rsigma_dot(  8, 10,  2,  2 ) = -4.979190;
	rsigma_dot(  8, 11,  2,  2 ) = -7.462060;
	rsigma_dot(  8, 12,  2,  2 ) = -7.337890;
	rsigma_dot(  8, 13,  2,  2 ) = -3.722400;
	rsigma_dot(  8, 14,  2,  2 ) = -0.167000;
	rsigma_dot(  8, 15,  2,  2 ) = -0.167000;
	rsigma_dot(  8, 16,  2,  2 ) = -0.167000;
	rsigma_dot(  8, 17,  2,  2 ) = -0.167000;
	rsigma_dot(  8, 18,  2,  2 ) = -0.167000;
	rsigma_dot(  9,  1,  2,  2 ) = -0.122600;
	rsigma_dot(  9,  2,  2,  2 ) = -0.122600;
	rsigma_dot(  9,  3,  2,  2 ) = -0.122600;
	rsigma_dot(  9,  4,  2,  2 ) = -1.221200;
	rsigma_dot(  9,  5,  2,  2 ) = -7.827370;
	rsigma_dot(  9,  6,  2,  2 ) = -8.067400;
	rsigma_dot(  9,  7,  2,  2 ) = -5.809530;
	rsigma_dot(  9,  8,  2,  2 ) = -4.966740;
	rsigma_dot(  9,  9,  2,  2 ) = -4.327250;
	rsigma_dot(  9, 10,  2,  2 ) = -4.884730;
	rsigma_dot(  9, 11,  2,  2 ) = -7.072410;
	rsigma_dot(  9, 12,  2,  2 ) = -8.543900;
	rsigma_dot(  9, 13,  2,  2 ) = -5.252450;
	rsigma_dot(  9, 14,  2,  2 ) = -1.732000;
	rsigma_dot(  9, 15,  2,  2 ) = -0.122600;
	rsigma_dot(  9, 16,  2,  2 ) = -0.122600;
	rsigma_dot(  9, 17,  2,  2 ) = -0.122600;
	rsigma_dot(  9, 18,  2,  2 ) = -0.122600;
	rsigma_dot( 10,  1,  2,  2 ) = -0.080000;
	rsigma_dot( 10,  2,  2,  2 ) = -0.080000;
	rsigma_dot( 10,  3,  2,  2 ) = -1.689400;
	rsigma_dot( 10,  4,  2,  2 ) = -2.913200;
	rsigma_dot( 10,  5,  2,  2 ) = -7.579420;
	rsigma_dot( 10,  6,  2,  2 ) = -6.907620;
	rsigma_dot( 10,  7,  2,  2 ) = -5.422330;
	rsigma_dot( 10,  8,  2,  2 ) = -5.161400;
	rsigma_dot( 10,  9,  2,  2 ) = -4.590850;
	rsigma_dot( 10, 10,  2,  2 ) = -4.970340;
	rsigma_dot( 10, 11,  2,  2 ) = -6.331900;
	rsigma_dot( 10, 12,  2,  2 ) = -8.504850;
	rsigma_dot( 10, 13,  2,  2 ) = -6.881280;
	rsigma_dot( 10, 14,  2,  2 ) = -2.644900;
	rsigma_dot( 10, 15,  2,  2 ) = -0.080000;
	rsigma_dot( 10, 16,  2,  2 ) = -1.178600;
	rsigma_dot( 10, 17,  2,  2 ) = -0.080000;
	rsigma_dot( 10, 18,  2,  2 ) = -0.080000;
	rsigma_dot( 11,  1,  2,  2 ) = -0.039200;
	rsigma_dot( 11,  2,  2,  2 ) = -1.137800;
	rsigma_dot( 11,  3,  2,  2 ) = -2.437100;
	rsigma_dot( 11,  4,  2,  2 ) = -4.356660;
	rsigma_dot( 11,  5,  2,  2 ) = -6.159470;
	rsigma_dot( 11,  6,  2,  2 ) = -6.109910;
	rsigma_dot( 11,  7,  2,  2 ) = -5.082600;
	rsigma_dot( 11,  8,  2,  2 ) = -5.203960;
	rsigma_dot( 11,  9,  2,  2 ) = -5.029610;
	rsigma_dot( 11, 10,  2,  2 ) = -5.180840;
	rsigma_dot( 11, 11,  2,  2 ) = -5.633880;
	rsigma_dot( 11, 12,  2,  2 ) = -7.850740;
	rsigma_dot( 11, 13,  2,  2 ) = -8.236990;
	rsigma_dot( 11, 14,  2,  2 ) = -3.258000;
	rsigma_dot( 11, 15,  2,  2 ) = -2.604100;
	rsigma_dot( 11, 16,  2,  2 ) = -1.648600;
	rsigma_dot( 11, 17,  2,  2 ) = -0.039200;
	rsigma_dot( 11, 18,  2,  2 ) = -0.039200;
	rsigma_dot( 12,  1,  2,  2 ) = 0.000000;
	rsigma_dot( 12,  2,  2,  2 ) = -2.197200;
	rsigma_dot( 12,  3,  2,  2 ) = -2.708000;
	rsigma_dot( 12,  4,  2,  2 ) = -5.093700;
	rsigma_dot( 12,  5,  2,  2 ) = -5.802070;
	rsigma_dot( 12,  6,  2,  2 ) = -5.587200;
	rsigma_dot( 12,  7,  2,  2 ) = -5.198450;
	rsigma_dot( 12,  8,  2,  2 ) = -5.459540;
	rsigma_dot( 12,  9,  2,  2 ) = -5.068860;
	rsigma_dot( 12, 10,  2,  2 ) = -5.272950;
	rsigma_dot( 12, 11,  2,  2 ) = -5.370590;
	rsigma_dot( 12, 12,  2,  2 ) = -6.597100;
	rsigma_dot( 12, 13,  2,  2 ) = -7.631870;
	rsigma_dot( 12, 14,  2,  2 ) = -5.043380;
	rsigma_dot( 12, 15,  2,  2 ) = -3.218800;
	rsigma_dot( 12, 16,  2,  2 ) = -2.833200;
	rsigma_dot( 12, 17,  2,  2 ) = -2.197200;
	rsigma_dot( 12, 18,  2,  2 ) = 0.000000;

	// section 21
	rsigma_dot(  1,  1,  2,  1 ) = -0.974100;
	rsigma_dot(  1,  2,  2,  1 ) = -0.974100;
	rsigma_dot(  1,  3,  2,  1 ) = -0.974100;
	rsigma_dot(  1,  4,  2,  1 ) = -0.974100;
	rsigma_dot(  1,  5,  2,  1 ) = -0.974100;
	rsigma_dot(  1,  6,  2,  1 ) = -0.974100;
	rsigma_dot(  1,  7,  2,  1 ) = -0.974100;
	rsigma_dot(  1,  8,  2,  1 ) = -2.920000;
	rsigma_dot(  1,  9,  2,  1 ) = -3.171400;
	rsigma_dot(  1, 10,  2,  1 ) = -0.974100;
	rsigma_dot(  1, 11,  2,  1 ) = -2.583600;
	rsigma_dot(  1, 12,  2,  1 ) = -0.974100;
	rsigma_dot(  1, 13,  2,  1 ) = -0.974100;
	rsigma_dot(  1, 14,  2,  1 ) = -0.974100;
	rsigma_dot(  1, 15,  2,  1 ) = -0.974100;
	rsigma_dot(  1, 16,  2,  1 ) = -0.974100;
	rsigma_dot(  1, 17,  2,  1 ) = -0.974100;
	rsigma_dot(  1, 18,  2,  1 ) = -0.974100;
	rsigma_dot(  2,  1,  2,  1 ) = -0.909600;
	rsigma_dot(  2,  2,  2,  1 ) = -0.909600;
	rsigma_dot(  2,  3,  2,  1 ) = -0.909600;
	rsigma_dot(  2,  4,  2,  1 ) = -0.909600;
	rsigma_dot(  2,  5,  2,  1 ) = -0.909600;
	rsigma_dot(  2,  6,  2,  1 ) = -0.909600;
	rsigma_dot(  2,  7,  2,  1 ) = -0.909600;
	rsigma_dot(  2,  8,  2,  1 ) = -4.343580;
	rsigma_dot(  2,  9,  2,  1 ) = -6.015540;
	rsigma_dot(  2, 10,  2,  1 ) = -4.205430;
	rsigma_dot(  2, 11,  2,  1 ) = -2.008200;
	rsigma_dot(  2, 12,  2,  1 ) = -2.519000;
	rsigma_dot(  2, 13,  2,  1 ) = -0.909600;
	rsigma_dot(  2, 14,  2,  1 ) = -0.909600;
	rsigma_dot(  2, 15,  2,  1 ) = -0.909600;
	rsigma_dot(  2, 16,  2,  1 ) = -0.909600;
	rsigma_dot(  2, 17,  2,  1 ) = -0.909600;
	rsigma_dot(  2, 18,  2,  1 ) = -0.909600;
	rsigma_dot(  3,  1,  2,  1 ) = -0.849000;
	rsigma_dot(  3,  2,  2,  1 ) = -0.849000;
	rsigma_dot(  3,  3,  2,  1 ) = -0.849000;
	rsigma_dot(  3,  4,  2,  1 ) = -0.849000;
	rsigma_dot(  3,  5,  2,  1 ) = -0.849000;
	rsigma_dot(  3,  6,  2,  1 ) = -0.849000;
	rsigma_dot(  3,  7,  2,  1 ) = -3.046200;
	rsigma_dot(  3,  8,  2,  1 ) = -5.990630;
	rsigma_dot(  3,  9,  2,  1 ) = -7.627760;
	rsigma_dot(  3, 10,  2,  1 ) = -6.013760;
	rsigma_dot(  3, 11,  2,  1 ) = -2.794900;
	rsigma_dot(  3, 12,  2,  1 ) = -2.458400;
	rsigma_dot(  3, 13,  2,  1 ) = -0.849000;
	rsigma_dot(  3, 14,  2,  1 ) = -0.849000;
	rsigma_dot(  3, 15,  2,  1 ) = -0.849000;
	rsigma_dot(  3, 16,  2,  1 ) = -0.849000;
	rsigma_dot(  3, 17,  2,  1 ) = -0.849000;
	rsigma_dot(  3, 18,  2,  1 ) = -0.849000;
	rsigma_dot(  4,  1,  2,  1 ) = -0.791800;
	rsigma_dot(  4,  2,  2,  1 ) = -0.791800;
	rsigma_dot(  4,  3,  2,  1 ) = -0.791800;
	rsigma_dot(  4,  4,  2,  1 ) = -0.791800;
	rsigma_dot(  4,  5,  2,  1 ) = -0.791800;
	rsigma_dot(  4,  6,  2,  1 ) = -2.401300;
	rsigma_dot(  4,  7,  2,  1 ) = -3.736250;
	rsigma_dot(  4,  8,  2,  1 ) = -6.512120;
	rsigma_dot(  4,  9,  2,  1 ) = -8.856760;
	rsigma_dot(  4, 10,  2,  1 ) = -7.235940;
	rsigma_dot(  4, 11,  2,  1 ) = -4.010690;
	rsigma_dot(  4, 12,  2,  1 ) = -2.401300;
	rsigma_dot(  4, 13,  2,  1 ) = -1.890400;
	rsigma_dot(  4, 14,  2,  1 ) = -0.791800;
	rsigma_dot(  4, 15,  2,  1 ) = -0.791800;
	rsigma_dot(  4, 16,  2,  1 ) = -0.791800;
	rsigma_dot(  4, 17,  2,  1 ) = -0.791800;
	rsigma_dot(  4, 18,  2,  1 ) = -0.791800;
	rsigma_dot(  5,  1,  2,  1 ) = -0.737700;
	rsigma_dot(  5,  2,  2,  1 ) = -0.737700;
	rsigma_dot(  5,  3,  2,  1 ) = -0.737700;
	rsigma_dot(  5,  4,  2,  1 ) = -0.737700;
	rsigma_dot(  5,  5,  2,  1 ) = -0.737700;
	rsigma_dot(  5,  6,  2,  1 ) = -2.683700;
	rsigma_dot(  5,  7,  2,  1 ) = -4.171730;
	rsigma_dot(  5,  8,  2,  1 ) = -7.436010;
	rsigma_dot(  5,  9,  2,  1 ) = -9.773380;
	rsigma_dot(  5, 10,  2,  1 ) = -8.510500;
	rsigma_dot(  5, 11,  2,  1 ) = -4.669570;
	rsigma_dot(  5, 12,  2,  1 ) = -3.445800;
	rsigma_dot(  5, 13,  2,  1 ) = -1.836400;
	rsigma_dot(  5, 14,  2,  1 ) = -0.737700;
	rsigma_dot(  5, 15,  2,  1 ) = -0.737700;
	rsigma_dot(  5, 16,  2,  1 ) = -0.737700;
	rsigma_dot(  5, 17,  2,  1 ) = -0.737700;
	rsigma_dot(  5, 18,  2,  1 ) = -0.737700;
	rsigma_dot(  6,  1,  2,  1 ) = -0.686500;
	rsigma_dot(  6,  2,  2,  1 ) = -0.686500;
	rsigma_dot(  6,  3,  2,  1 ) = -0.686500;
	rsigma_dot(  6,  4,  2,  1 ) = -0.686500;
	rsigma_dot(  6,  5,  2,  1 ) = -0.686500;
	rsigma_dot(  6,  6,  2,  1 ) = -0.686500;
	rsigma_dot(  6,  7,  2,  1 ) = -4.297370;
	rsigma_dot(  6,  8,  2,  1 ) = -7.670240;
	rsigma_dot(  6,  9,  2,  1 ) = -9.274480;
	rsigma_dot(  6, 10,  2,  1 ) = -8.780530;
	rsigma_dot(  6, 11,  2,  1 ) = -5.395980;
	rsigma_dot(  6, 12,  2,  1 ) = -2.632400;
	rsigma_dot(  6, 13,  2,  1 ) = -3.084300;
	rsigma_dot(  6, 14,  2,  1 ) = -0.686500;
	rsigma_dot(  6, 15,  2,  1 ) = -0.686500;
	rsigma_dot(  6, 16,  2,  1 ) = -0.686500;
	rsigma_dot(  6, 17,  2,  1 ) = -0.686500;
	rsigma_dot(  6, 18,  2,  1 ) = -0.686500;
	rsigma_dot(  7,  1,  2,  1 ) = -0.637700;
	rsigma_dot(  7,  2,  2,  1 ) = -0.637700;
	rsigma_dot(  7,  3,  2,  1 ) = -0.637700;
	rsigma_dot(  7,  4,  2,  1 ) = -0.637700;
	rsigma_dot(  7,  5,  2,  1 ) = -1.736300;
	rsigma_dot(  7,  6,  2,  1 ) = -3.682200;
	rsigma_dot(  7,  7,  2,  1 ) = -4.645000;
	rsigma_dot(  7,  8,  2,  1 ) = -6.967380;
	rsigma_dot(  7,  9,  2,  1 ) = -7.766160;
	rsigma_dot(  7, 10,  2,  1 ) = -7.608390;
	rsigma_dot(  7, 11,  2,  1 ) = -5.416790;
	rsigma_dot(  7, 12,  2,  1 ) = -3.773160;
	rsigma_dot(  7, 13,  2,  1 ) = -3.345700;
	rsigma_dot(  7, 14,  2,  1 ) = -2.834900;
	rsigma_dot(  7, 15,  2,  1 ) = -2.247100;
	rsigma_dot(  7, 16,  2,  1 ) = -0.637700;
	rsigma_dot(  7, 17,  2,  1 ) = -0.637700;
	rsigma_dot(  7, 18,  2,  1 ) = -0.637700;
	rsigma_dot(  8,  1,  2,  1 ) = -0.591100;
	rsigma_dot(  8,  2,  2,  1 ) = -0.591100;
	rsigma_dot(  8,  3,  2,  1 ) = -0.591100;
	rsigma_dot(  8,  4,  2,  1 ) = -0.591100;
	rsigma_dot(  8,  5,  2,  1 ) = -2.788400;
	rsigma_dot(  8,  6,  2,  1 ) = -4.087650;
	rsigma_dot(  8,  7,  2,  1 ) = -4.702020;
	rsigma_dot(  8,  8,  2,  1 ) = -6.084200;
	rsigma_dot(  8,  9,  2,  1 ) = -6.877140;
	rsigma_dot(  8, 10,  2,  1 ) = -6.827510;
	rsigma_dot(  8, 11,  2,  1 ) = -5.980210;
	rsigma_dot(  8, 12,  2,  1 ) = -4.087650;
	rsigma_dot(  8, 13,  2,  1 ) = -3.635700;
	rsigma_dot(  8, 14,  2,  1 ) = -4.202060;
	rsigma_dot(  8, 15,  2,  1 ) = -2.537100;
	rsigma_dot(  8, 16,  2,  1 ) = -0.591100;
	rsigma_dot(  8, 17,  2,  1 ) = -0.591100;
	rsigma_dot(  8, 18,  2,  1 ) = -0.591100;
	rsigma_dot(  9,  1,  2,  1 ) = -0.546700;
	rsigma_dot(  9,  2,  2,  1 ) = -0.546700;
	rsigma_dot(  9,  3,  2,  1 ) = -0.546700;
	rsigma_dot(  9,  4,  2,  1 ) = -0.546700;
	rsigma_dot(  9,  5,  2,  1 ) = -3.254700;
	rsigma_dot(  9,  6,  2,  1 ) = -4.102040;
	rsigma_dot(  9,  7,  2,  1 ) = -4.916140;
	rsigma_dot(  9,  8,  2,  1 ) = -5.788440;
	rsigma_dot(  9,  9,  2,  1 ) = -5.777800;
	rsigma_dot(  9, 10,  2,  1 ) = -6.126420;
	rsigma_dot(  9, 11,  2,  1 ) = -5.997730;
	rsigma_dot(  9, 12,  2,  1 ) = -4.721080;
	rsigma_dot(  9, 13,  2,  1 ) = -4.102040;
	rsigma_dot(  9, 14,  2,  1 ) = -4.157610;
	rsigma_dot(  9, 15,  2,  1 ) = -3.111600;
	rsigma_dot(  9, 16,  2,  1 ) = -1.645300;
	rsigma_dot(  9, 17,  2,  1 ) = -0.546700;
	rsigma_dot(  9, 18,  2,  1 ) = -0.546700;
	rsigma_dot( 10,  1,  2,  1 ) = -0.504100;
	rsigma_dot( 10,  2,  2,  1 ) = -0.504100;
	rsigma_dot( 10,  3,  2,  1 ) = -0.504100;
	rsigma_dot( 10,  4,  2,  1 ) = -0.504100;
	rsigma_dot( 10,  5,  2,  1 ) = -3.548700;
	rsigma_dot( 10,  6,  2,  1 ) = -4.873580;
	rsigma_dot( 10,  7,  2,  1 ) = -5.176960;
	rsigma_dot( 10,  8,  2,  1 ) = -5.756400;
	rsigma_dot( 10,  9,  2,  1 ) = -5.573030;
	rsigma_dot( 10, 10,  2,  1 ) = -5.702630;
	rsigma_dot( 10, 11,  2,  1 ) = -5.668920;
	rsigma_dot( 10, 12,  2,  1 ) = -5.078840;
	rsigma_dot( 10, 13,  2,  1 ) = -4.678520;
	rsigma_dot( 10, 14,  2,  1 ) = -4.310790;
	rsigma_dot( 10, 15,  2,  1 ) = -4.217700;
	rsigma_dot( 10, 16,  2,  1 ) = -2.113600;
	rsigma_dot( 10, 17,  2,  1 ) = -0.504100;
	rsigma_dot( 10, 18,  2,  1 ) = -0.504100;
	rsigma_dot( 11,  1,  2,  1 ) = -0.463300;
	rsigma_dot( 11,  2,  2,  1 ) = -0.463300;
	rsigma_dot( 11,  3,  2,  1 ) = -0.463300;
	rsigma_dot( 11,  4,  2,  1 ) = -2.409200;
	rsigma_dot( 11,  5,  2,  1 ) = -4.313460;
	rsigma_dot( 11,  6,  2,  1 ) = -5.225480;
	rsigma_dot( 11,  7,  2,  1 ) = -5.493750;
	rsigma_dot( 11,  8,  2,  1 ) = -5.353660;
	rsigma_dot( 11,  9,  2,  1 ) = -5.154660;
	rsigma_dot( 11, 10,  2,  1 ) = -5.397780;
	rsigma_dot( 11, 11,  2,  1 ) = -5.726000;
	rsigma_dot( 11, 12,  2,  1 ) = -5.480590;
	rsigma_dot( 11, 13,  2,  1 ) = -4.018660;
	rsigma_dot( 11, 14,  2,  1 ) = -4.313460;
	rsigma_dot( 11, 15,  2,  1 ) = -4.574180;
	rsigma_dot( 11, 16,  2,  1 ) = -2.861200;
	rsigma_dot( 11, 17,  2,  1 ) = -0.463300;
	rsigma_dot( 11, 18,  2,  1 ) = -1.561900;
	rsigma_dot( 12,  1,  2,  1 ) = -0.424100;
	rsigma_dot( 12,  2,  2,  1 ) = -0.424100;
	rsigma_dot( 12,  3,  2,  1 ) = -2.822000;
	rsigma_dot( 12,  4,  2,  1 ) = -3.132100;
	rsigma_dot( 12,  5,  2,  1 ) = -5.115440;
	rsigma_dot( 12,  6,  2,  1 ) = -5.697090;
	rsigma_dot( 12,  7,  2,  1 ) = -5.747100;
	rsigma_dot( 12,  8,  2,  1 ) = -5.386930;
	rsigma_dot( 12,  9,  2,  1 ) = -5.019210;
	rsigma_dot( 12, 10,  2,  1 ) = -5.186260;
	rsigma_dot( 12, 11,  2,  1 ) = -5.775950;
	rsigma_dot( 12, 12,  2,  1 ) = -5.600240;
	rsigma_dot( 12, 13,  2,  1 ) = -4.394380;
	rsigma_dot( 12, 14,  2,  1 ) = -3.257300;
	rsigma_dot( 12, 15,  2,  1 ) = -3.920600;
	rsigma_dot( 12, 16,  2,  1 ) = -3.643000;
	rsigma_dot( 12, 17,  2,  1 ) = -1.522700;
	rsigma_dot( 12, 18,  2,  1 ) = -0.424100;
}

void
SecondaryStructureEval::centroid_pack_correction_initializer(
	FArray1D_float & centroid_pack_correction
)
{
	// These values are obtained by comparing the centroid-centroid distance
	// distribution calculated for real PDB structures (with Rosetta-guessed
	// centroids) and the distribution calculated for a large pool of Rosetta
	// decoys. The PDB shows a distinct peak in the correlation function at
	// 5 angstroms, suggesting "semi-crystalline" packing of sidechains, and
	// almost no observed centroid pairs below 3 angstroms.
	// These features are not apparent in the decoys. Below values are
	// -log ( PDB / decoys) for 0.1 angstrom distance bins.

	centroid_pack_correction( 1) = 1.41640;
	centroid_pack_correction( 2) = 1.41640;
	centroid_pack_correction( 3) = 1.41640;
	centroid_pack_correction( 4) = 1.41640;
	centroid_pack_correction( 5) = 1.41640;
	centroid_pack_correction( 6) = 1.41640;
	centroid_pack_correction( 7) = 1.41640;
	centroid_pack_correction( 8) = 1.41640;
	centroid_pack_correction( 9) = 1.41640;
	centroid_pack_correction(10) = 1.41640;
	centroid_pack_correction(11) = 1.41640;
	centroid_pack_correction(12) = 1.41640;
	centroid_pack_correction(13) = 1.41640;
	centroid_pack_correction(14) = 1.41640;
	centroid_pack_correction(15) = 1.41640;
	centroid_pack_correction(16) = 1.34879;
	centroid_pack_correction(17) = 1.26321;
	centroid_pack_correction(18) = 1.41126;
	centroid_pack_correction(19) = 1.04062;
	centroid_pack_correction(20) = 1.32634;
	centroid_pack_correction(21) = 1.21590;
	centroid_pack_correction(22) = 1.17593;
	centroid_pack_correction(23) = 1.17736;
	centroid_pack_correction(24) = 1.12760;
	centroid_pack_correction(25) = 1.14841;
	centroid_pack_correction(26) = 0.95829;
	centroid_pack_correction(27) = 1.00708;
	centroid_pack_correction(28) = 0.92431;
	centroid_pack_correction(29) = 0.81809;
	centroid_pack_correction(30) = 0.79747;
	centroid_pack_correction(31) = 0.65308;
	centroid_pack_correction(32) = 0.64273;
	centroid_pack_correction(33) = 0.51349;
	centroid_pack_correction(34) = 0.46635;
	centroid_pack_correction(35) = 0.37186;
	centroid_pack_correction(36) = 0.20699;
	centroid_pack_correction(37) = 0.16137;
	centroid_pack_correction(38) = 0.09686;
	centroid_pack_correction(39) = -0.00143;
	centroid_pack_correction(40) = -0.04904;
	centroid_pack_correction(41) = -0.13305;
	centroid_pack_correction(42) = -0.14444;
	centroid_pack_correction(43) = -0.16689;
	centroid_pack_correction(44) = -0.25425;
	centroid_pack_correction(45) = -0.25223;
	centroid_pack_correction(46) = -0.27366;
	centroid_pack_correction(47) = -0.28487;
	centroid_pack_correction(48) = -0.30148;
	centroid_pack_correction(49) = -0.30376;
	centroid_pack_correction(50) = -0.29090;
	centroid_pack_correction(51) = -0.26104;
	centroid_pack_correction(52) = -0.26818;
	centroid_pack_correction(53) = -0.18952;
	centroid_pack_correction(54) = -0.18254;
	centroid_pack_correction(55) = -0.13800;
	centroid_pack_correction(56) = -0.12473;
	centroid_pack_correction(57) = -0.08532;
	centroid_pack_correction(58) = -0.06890;
	centroid_pack_correction(59) = -0.03183;
	centroid_pack_correction(60) = -0.02379;
	centroid_pack_correction(61) = -0.01371;
	centroid_pack_correction(62) = 0.02235;
	centroid_pack_correction(63) = 0.04125;
	centroid_pack_correction(64) = 0.09542;
	centroid_pack_correction(65) = 0.11937;
	centroid_pack_correction(66) = 0.11762;
	centroid_pack_correction(67) = 0.13437;
	centroid_pack_correction(68) = 0.13544;
	centroid_pack_correction(69) = 0.16683;
	centroid_pack_correction(70) = 0.17090;
	centroid_pack_correction(71) = 0.16978;
	centroid_pack_correction(72) = 0.14705;
	centroid_pack_correction(73) = 0.16138;
	centroid_pack_correction(74) = 0.15677;
	centroid_pack_correction(75) = 0.12790;
	centroid_pack_correction(76) = 0.14213;
	centroid_pack_correction(77) = 0.13141;
	centroid_pack_correction(78) = 0.10548;
	centroid_pack_correction(79) = 0.09670;
	centroid_pack_correction(80) = 0.09010;
	centroid_pack_correction(81) = 0.09170;
	centroid_pack_correction(82) = 0.06510;
	centroid_pack_correction(83) = 0.05086;
	centroid_pack_correction(84) = 0.05124;
	centroid_pack_correction(85) = 0.04903;
	centroid_pack_correction(86) = 0.03258;
	centroid_pack_correction(87) = 0.02944;
	centroid_pack_correction(88) = 0.02203;
	centroid_pack_correction(89) = 0.01674;
	centroid_pack_correction(90) = 0.00446;
	centroid_pack_correction(91) = 0.01098;
	centroid_pack_correction(92) = 0.01808;
	centroid_pack_correction(93) = -0.01358;
	centroid_pack_correction(94) = 0.00296;
	centroid_pack_correction(95) = 0.00350;
	centroid_pack_correction(96) = -0.00199;
	centroid_pack_correction(97) = -0.00737;
	centroid_pack_correction(98) = -0.00698;
	centroid_pack_correction(99) = -0.00489;
	centroid_pack_correction(100) = -0.05050;
	centroid_pack_correction(101) = -0.05704;
	centroid_pack_correction(102) = -0.05773;
	centroid_pack_correction(103) = -0.05163;
	centroid_pack_correction(104) = -0.04760;
	centroid_pack_correction(105) = -0.04140;
	centroid_pack_correction(106) = -0.05405;
	centroid_pack_correction(107) = -0.04531;
	centroid_pack_correction(108) = -0.03361;
	centroid_pack_correction(109) = -0.04713;
	centroid_pack_correction(110) = -0.04211;
	centroid_pack_correction(111) = -0.03348;
	centroid_pack_correction(112) = -0.03796;
	centroid_pack_correction(113) = -0.02672;
	centroid_pack_correction(114) = -0.02450;
	centroid_pack_correction(115) = -0.01789;
	centroid_pack_correction(116) = -0.00972;
	centroid_pack_correction(117) = -0.02324;
	centroid_pack_correction(118) = -0.02444;
	centroid_pack_correction(119) = -0.01871;
	centroid_pack_correction(120) = -0.02417;
}

/* seq_struct */
void
SecondaryStructureEval::env_log_initializer_master(
	FArray2D_float & env_log
)
{
	env_log_initializer_section1(env_log);
	env_log_initializer_section2(env_log);
	env_log_initializer_section3(env_log);
}

void
SecondaryStructureEval::env_log_initializer_section1(
	FArray2D_float & env_log
)
{
	// section 1
	env_log( 1, 1) = 0.7895;
	env_log( 1, 2) = 0.7895;
	env_log( 1, 3) = 0.7895;
	env_log( 1, 4) = 0.7895;
	env_log( 1, 5) = 0.7895;
	env_log( 1, 6) = 0.8900;
	env_log( 1, 7) = 0.7714;
	env_log( 1, 8) = 0.5847;
	env_log( 1, 9) = 0.4103;
	env_log( 1,10) = 0.2816;
	env_log( 1,11) = 0.2036;
	env_log( 1,12) = 0.1656;
	env_log( 1,13) = 0.1516;
	env_log( 1,14) = 0.1456;
	env_log( 1,15) = 0.1351;
	env_log( 1,16) = 0.1133;
	env_log( 1,17) = 0.0783;
	env_log( 1,18) = 0.0326;
	env_log( 1,19) = -0.0183;
	env_log( 1,20) = -0.0682;
	env_log( 1,21) = -0.1113;
	env_log( 1,22) = -0.1439;
	env_log( 1,23) = -0.1649;
	env_log( 1,24) = -0.1767;
	env_log( 1,25) = -0.1843;
	env_log( 1,26) = -0.1951;
	env_log( 1,27) = -0.2169;
	env_log( 1,28) = -0.2568;
	env_log( 1,29) = -0.3191;
	env_log( 1,30) = -0.4037;
	env_log( 1,31) = -0.5052;
	env_log( 1,32) = -0.6129;
	env_log( 1,33) = -0.7114;
	env_log( 1,34) = -0.7845;
	env_log( 1,35) = -0.8204;
	env_log( 1,36) = -0.8207;
	env_log( 1,37) = -0.8207;
	env_log( 1,38) = -0.8207;
	env_log( 1,39) = -0.8207;
	env_log( 1,40) = -0.8207;
	env_log( 2, 1) = 1.6892;
	env_log( 2, 2) = 1.6892;
	env_log( 2, 3) = 1.6892;
	env_log( 2, 4) = 1.6892;
	env_log( 2, 5) = 1.6892;
	env_log( 2, 6) = 1.6892;
	env_log( 2, 7) = 1.6892;
	env_log( 2, 8) = 1.6892;
	env_log( 2, 9) = 1.9186;
	env_log( 2,10) = 1.7989;
	env_log( 2,11) = 1.5422;
	env_log( 2,12) = 1.2597;
	env_log( 2,13) = 0.9989;
	env_log( 2,14) = 0.7712;
	env_log( 2,15) = 0.5711;
	env_log( 2,16) = 0.3880;
	env_log( 2,17) = 0.2137;
	env_log( 2,18) = 0.0453;
	env_log( 2,19) = -0.1149;
	env_log( 2,20) = -0.2606;
	env_log( 2,21) = -0.3840;
	env_log( 2,22) = -0.4786;
	env_log( 2,23) = -0.5417;
	env_log( 2,24) = -0.5750;
	env_log( 2,25) = -0.5859;
	env_log( 2,26) = -0.5859;
	env_log( 2,27) = -0.5884;
	env_log( 2,28) = -0.6057;
	env_log( 2,29) = -0.6445;
	env_log( 2,30) = -0.7014;
	env_log( 2,31) = -0.7589;
	env_log( 2,32) = -0.7815;
	env_log( 2,33) = -0.7148;
	env_log( 2,34) = -0.4860;
	env_log( 2,35) = -0.4860;
	env_log( 2,36) = -0.4860;
	env_log( 2,37) = -0.4860;
	env_log( 2,38) = -0.4860;
	env_log( 2,39) = -0.4860;
	env_log( 2,40) = -0.4860;
	env_log( 3, 1) = 0.1714;
	env_log( 3, 2) = 0.1714;
	env_log( 3, 3) = 0.1714;
	env_log( 3, 4) = 0.1714;
	env_log( 3, 5) = -0.3157;
	env_log( 3, 6) = -0.5377;
	env_log( 3, 7) = -0.6085;
	env_log( 3, 8) = -0.6025;
	env_log( 3, 9) = -0.5645;
	env_log( 3,10) = -0.5180;
	env_log( 3,11) = -0.4717;
	env_log( 3,12) = -0.4256;
	env_log( 3,13) = -0.3752;
	env_log( 3,14) = -0.3146;
	env_log( 3,15) = -0.2394;
	env_log( 3,16) = -0.1471;
	env_log( 3,17) = -0.0387;
	env_log( 3,18) = 0.0824;
	env_log( 3,19) = 0.2102;
	env_log( 3,20) = 0.3375;
	env_log( 3,21) = 0.4573;
	env_log( 3,22) = 0.5634;
	env_log( 3,23) = 0.6519;
	env_log( 3,24) = 0.7215;
	env_log( 3,25) = 0.7735;
	env_log( 3,26) = 0.8112;
	env_log( 3,27) = 0.8374;
	env_log( 3,28) = 0.8513;
	env_log( 3,29) = 0.8436;
	env_log( 3,30) = 0.7893;
	env_log( 3,31) = 0.6388;
	env_log( 3,32) = 0.6388;
	env_log( 3,33) = 0.6388;
	env_log( 3,34) = 0.6388;
	env_log( 3,35) = 0.6388;
	env_log( 3,36) = 0.6388;
	env_log( 3,37) = 0.6388;
	env_log( 3,38) = 0.6388;
	env_log( 3,39) = 0.6388;
	env_log( 3,40) = 0.6388;
	env_log( 4, 1) = -0.8010;
	env_log( 4, 2) = -0.8010;
	env_log( 4, 3) = -0.8010;
	env_log( 4, 4) = -0.9240;
	env_log( 4, 5) = -1.0035;
	env_log( 4, 6) = -1.0243;
	env_log( 4, 7) = -0.9910;
	env_log( 4, 8) = -0.9173;
	env_log( 4, 9) = -0.8190;
	env_log( 4,10) = -0.7091;
	env_log( 4,11) = -0.5962;
	env_log( 4,12) = -0.4840;
	env_log( 4,13) = -0.3719;
	env_log( 4,14) = -0.2568;
	env_log( 4,15) = -0.1345;
	env_log( 4,16) = -0.0014;
	env_log( 4,17) = 0.1443;
	env_log( 4,18) = 0.3012;
	env_log( 4,19) = 0.4654;
	env_log( 4,20) = 0.6298;
	env_log( 4,21) = 0.7858;
	env_log( 4,22) = 0.9244;
	env_log( 4,23) = 1.0379;
	env_log( 4,24) = 1.1215;
	env_log( 4,25) = 1.1750;
	env_log( 4,26) = 1.2030;
	env_log( 4,27) = 1.2147;
	env_log( 4,28) = 1.2215;
	env_log( 4,29) = 1.2323;
	env_log( 4,30) = 1.2457;
	env_log( 4,31) = 1.2384;
	env_log( 4,32) = 1.1492;
	env_log( 4,33) = 1.1492;
	env_log( 4,34) = 1.1492;
	env_log( 4,35) = 1.1492;
	env_log( 4,36) = 1.1492;
	env_log( 4,37) = 1.1492;
	env_log( 4,38) = 1.1492;
	env_log( 4,39) = 1.1492;
	env_log( 4,40) = 1.1492;
	env_log( 5, 1) = 0.6512;
	env_log( 5, 2) = 0.6512;
	env_log( 5, 3) = 0.6512;
	env_log( 5, 4) = 0.6512;
	env_log( 5, 5) = 0.6512;
	env_log( 5, 6) = 0.8320;
	env_log( 5, 7) = 1.0349;
	env_log( 5, 8) = 1.1879;
	env_log( 5, 9) = 1.2595;
	env_log( 5,10) = 1.2446;
	env_log( 5,11) = 1.1541;
	env_log( 5,12) = 1.0067;
	env_log( 5,13) = 0.8236;
	env_log( 5,14) = 0.6244;
	env_log( 5,15) = 0.4255;
	env_log( 5,16) = 0.2385;
	env_log( 5,17) = 0.0707;
	env_log( 5,18) = -0.0747;
	env_log( 5,19) = -0.1972;
	env_log( 5,20) = -0.2984;
	env_log( 5,21) = -0.3806;
	env_log( 5,22) = -0.4457;
	env_log( 5,23) = -0.4947;
	env_log( 5,24) = -0.5273;
	env_log( 5,25) = -0.5415;
	env_log( 5,26) = -0.5344;
	env_log( 5,27) = -0.5025;
	env_log( 5,28) = -0.4428;
	env_log( 5,29) = -0.3540;
	env_log( 5,30) = -0.2378;
	env_log( 5,31) = -0.0997;
	env_log( 5,32) = 0.0497;
	env_log( 5,33) = 0.1948;
	env_log( 5,34) = 0.3148;
	env_log( 5,35) = 0.3859;
	env_log( 5,36) = 0.3859;
	env_log( 5,37) = 0.3859;
	env_log( 5,38) = 0.3859;
	env_log( 5,39) = 0.3859;
	env_log( 5,40) = 0.3859;
	env_log( 6, 1) = 0.9040;
	env_log( 6, 2) = 0.9040;
	env_log( 6, 3) = 0.9040;
	env_log( 6, 4) = 0.9040;
	env_log( 6, 5) = 0.9040;
	env_log( 6, 6) = 0.9040;
	env_log( 6, 7) = 0.6035;
	env_log( 6, 8) = 0.3487;
	env_log( 6, 9) = 0.1586;
	env_log( 6,10) = 0.0319;
	env_log( 6,11) = -0.0425;
	env_log( 6,12) = -0.0789;
	env_log( 6,13) = -0.0906;
	env_log( 6,14) = -0.0880;
	env_log( 6,15) = -0.0774;
	env_log( 6,16) = -0.0620;
	env_log( 6,17) = -0.0424;
	env_log( 6,18) = -0.0182;
	env_log( 6,19) = 0.0111;
	env_log( 6,20) = 0.0447;
	env_log( 6,21) = 0.0799;
	env_log( 6,22) = 0.1120;
	env_log( 6,23) = 0.1344;
	env_log( 6,24) = 0.1397;
	env_log( 6,25) = 0.1204;
	env_log( 6,26) = 0.0708;
	env_log( 6,27) = -0.0117;
	env_log( 6,28) = -0.1255;
	env_log( 6,29) = -0.2636;
	env_log( 6,30) = -0.4140;
	env_log( 6,31) = -0.5612;
	env_log( 6,32) = -0.6898;
	env_log( 6,33) = -0.7895;
	env_log( 6,34) = -0.8650;
	env_log( 6,35) = -0.9476;
	env_log( 6,36) = -1.1130;
	env_log( 6,37) = -1.5031;
	env_log( 6,38) = -1.5031;
	env_log( 6,39) = -1.5031;
	env_log( 6,40) = -1.5031;
	env_log( 7, 1) = -0.1070;
	env_log( 7, 2) = -0.1070;
	env_log( 7, 3) = -0.1070;
	env_log( 7, 4) = -0.1070;
	env_log( 7, 5) = -0.1070;
	env_log( 7, 6) = 0.1507;
	env_log( 7, 7) = 0.2179;
	env_log( 7, 8) = 0.2017;
	env_log( 7, 9) = 0.1575;
	env_log( 7,10) = 0.1095;
	env_log( 7,11) = 0.0644;
	env_log( 7,12) = 0.0213;
	env_log( 7,13) = -0.0222;
	env_log( 7,14) = -0.0671;
	env_log( 7,15) = -0.1116;
	env_log( 7,16) = -0.1512;
	env_log( 7,17) = -0.1802;
	env_log( 7,18) = -0.1931;
	env_log( 7,19) = -0.1857;
	env_log( 7,20) = -0.1567;
	env_log( 7,21) = -0.1079;
	env_log( 7,22) = -0.0447;
	env_log( 7,23) = 0.0251;
	env_log( 7,24) = 0.0922;
	env_log( 7,25) = 0.1476;
	env_log( 7,26) = 0.1841;
	env_log( 7,27) = 0.1988;
	env_log( 7,28) = 0.1933;
	env_log( 7,29) = 0.1741;
	env_log( 7,30) = 0.1512;
	env_log( 7,31) = 0.1332;
	env_log( 7,32) = 0.1205;
	env_log( 7,33) = 0.0938;
	env_log( 7,34) = 0.0938;
	env_log( 7,35) = 0.0938;
	env_log( 7,36) = 0.0938;
	env_log( 7,37) = 0.0938;
	env_log( 7,38) = 0.0938;
	env_log( 7,39) = 0.0938;
	env_log( 7,40) = 0.0938;
	env_log( 8, 1) = 0.9650;
	env_log( 8, 2) = 0.9650;
	env_log( 8, 3) = 0.9650;
	env_log( 8, 4) = 0.9650;
	env_log( 8, 5) = 0.9650;
	env_log( 8, 6) = 1.3157;
	env_log( 8, 7) = 1.4056;
	env_log( 8, 8) = 1.3556;
	env_log( 8, 9) = 1.2414;
	env_log( 8,10) = 1.1061;
	env_log( 8,11) = 0.9704;
	env_log( 8,12) = 0.8414;
	env_log( 8,13) = 0.7182;
	env_log( 8,14) = 0.5966;
	env_log( 8,15) = 0.4723;
	env_log( 8,16) = 0.3425;
	env_log( 8,17) = 0.2071;
	env_log( 8,18) = 0.0685;
	env_log( 8,19) = -0.0688;
	env_log( 8,20) = -0.1987;
	env_log( 8,21) = -0.3152;
	env_log( 8,22) = -0.4129;
	env_log( 8,23) = -0.4882;
	env_log( 8,24) = -0.5400;
	env_log( 8,25) = -0.5698;
	env_log( 8,26) = -0.5818;
	env_log( 8,27) = -0.5816;
	env_log( 8,28) = -0.5746;
	env_log( 8,29) = -0.5638;
	env_log( 8,30) = -0.5462;
	env_log( 8,31) = -0.5084;
	env_log( 8,32) = -0.4211;
	env_log( 8,33) = -0.2330;
	env_log( 8,34) = 0.1376;
	env_log( 8,35) = 0.1376;
	env_log( 8,36) = 0.1376;
	env_log( 8,37) = 0.1376;
	env_log( 8,38) = 0.1376;
	env_log( 8,39) = 0.1376;
	env_log( 8,40) = 0.1376;
	env_log( 9, 1) = -0.9806;
	env_log( 9, 2) = -0.9806;
	env_log( 9, 3) = -0.9806;
	env_log( 9, 4) = -1.0962;
	env_log( 9, 5) = -1.0972;
	env_log( 9, 6) = -1.0394;
	env_log( 9, 7) = -0.9555;
	env_log( 9, 8) = -0.8633;
	env_log( 9, 9) = -0.7713;
	env_log( 9,10) = -0.6824;
	env_log( 9,11) = -0.5962;
	env_log( 9,12) = -0.5105;
	env_log( 9,13) = -0.4221;
	env_log( 9,14) = -0.3269;
	env_log( 9,15) = -0.2207;
	env_log( 9,16) = -0.0988;
	env_log( 9,17) = 0.0429;
	env_log( 9,18) = 0.2077;
	env_log( 9,19) = 0.3969;
	env_log( 9,20) = 0.6095;
	env_log( 9,21) = 0.8406;
	env_log( 9,22) = 1.0813;
	env_log( 9,23) = 1.3185;
	env_log( 9,24) = 1.5354;
	env_log( 9,25) = 1.7132;
	env_log( 9,26) = 1.8352;
	env_log( 9,27) = 1.8916;
	env_log( 9,28) = 1.8883;
	env_log( 9,29) = 1.8582;
	env_log( 9,30) = 1.8765;
	env_log( 9,31) = 2.0815;
	env_log( 9,32) = 2.0815;
	env_log( 9,33) = 2.0815;
	env_log( 9,34) = 2.0815;
	env_log( 9,35) = 2.0815;
	env_log( 9,36) = 2.0815;
	env_log( 9,37) = 2.0815;
	env_log( 9,38) = 2.0815;
	env_log( 9,39) = 2.0815;
	env_log( 9,40) = 2.0815;
	env_log(10, 1) = 0.4926;
	env_log(10, 2) = 0.4926;
	env_log(10, 3) = 0.4926;
	env_log(10, 4) = 0.4926;
	env_log(10, 5) = 0.6852;
	env_log(10, 6) = 0.8767;
	env_log(10, 7) = 1.0278;
	env_log(10, 8) = 1.1198;
	env_log(10, 9) = 1.1472;
	env_log(10,10) = 1.1140;
	env_log(10,11) = 1.0291;
	env_log(10,12) = 0.9045;
	env_log(10,13) = 0.7523;
	env_log(10,14) = 0.5844;
	env_log(10,15) = 0.4110;
	env_log(10,16) = 0.2408;
	env_log(10,17) = 0.0805;
	env_log(10,18) = -0.0652;
	env_log(10,19) = -0.1930;
	env_log(10,20) = -0.3008;
	env_log(10,21) = -0.3878;
	env_log(10,22) = -0.4538;
	env_log(10,23) = -0.4992;
	env_log(10,24) = -0.5245;
	env_log(10,25) = -0.5308;
	env_log(10,26) = -0.5186;
	env_log(10,27) = -0.4886;
	env_log(10,28) = -0.4407;
	env_log(10,29) = -0.3736;
	env_log(10,30) = -0.2841;
	env_log(10,31) = -0.1661;
	env_log(10,32) = -0.0086;
	env_log(10,33) = 0.2061;
	env_log(10,34) = 0.5061;
	env_log(10,35) = 0.5061;
	env_log(10,36) = 0.5061;
	env_log(10,37) = 0.5061;
	env_log(10,38) = 0.5061;
	env_log(10,39) = 0.5061;
	env_log(10,40) = 0.5061;
}

void
SecondaryStructureEval::env_log_initializer_section2(
	FArray2D_float & env_log
)
{
	// section 2
	env_log(11, 1) = -0.4017;
	env_log(11, 2) = -0.4017;
	env_log(11, 3) = -0.4017;
	env_log(11, 4) = -0.4017;
	env_log(11, 5) = -0.1017;
	env_log(11, 6) = 0.2915;
	env_log(11, 7) = 0.6024;
	env_log(11, 8) = 0.7708;
	env_log(11, 9) = 0.8022;
	env_log(11,10) = 0.7338;
	env_log(11,11) = 0.6110;
	env_log(11,12) = 0.4739;
	env_log(11,13) = 0.3502;
	env_log(11,14) = 0.2535;
	env_log(11,15) = 0.1854;
	env_log(11,16) = 0.1384;
	env_log(11,17) = 0.1009;
	env_log(11,18) = 0.0611;
	env_log(11,19) = 0.0105;
	env_log(11,20) = -0.0541;
	env_log(11,21) = -0.1306;
	env_log(11,22) = -0.2119;
	env_log(11,23) = -0.2891;
	env_log(11,24) = -0.3537;
	env_log(11,25) = -0.4013;
	env_log(11,26) = -0.4332;
	env_log(11,27) = -0.4558;
	env_log(11,28) = -0.4767;
	env_log(11,29) = -0.4944;
	env_log(11,30) = -0.4814;
	env_log(11,31) = -0.3563;
	env_log(11,32) = 0.0556;
	env_log(11,33) = 0.0556;
	env_log(11,34) = 0.0556;
	env_log(11,35) = 0.0556;
	env_log(11,36) = 0.0556;
	env_log(11,37) = 0.0556;
	env_log(11,38) = 0.0556;
	env_log(11,39) = 0.0556;
	env_log(11,40) = 0.0556;
	env_log(12, 1) = 0.0696;
	env_log(12, 2) = 0.0696;
	env_log(12, 3) = 0.0696;
	env_log(12, 4) = 0.0696;
	env_log(12, 5) = -0.3629;
	env_log(12, 6) = -0.5013;
	env_log(12, 7) = -0.5045;
	env_log(12, 8) = -0.4590;
	env_log(12, 9) = -0.4058;
	env_log(12,10) = -0.3588;
	env_log(12,11) = -0.3187;
	env_log(12,12) = -0.2806;
	env_log(12,13) = -0.2395;
	env_log(12,14) = -0.1922;
	env_log(12,15) = -0.1377;
	env_log(12,16) = -0.0771;
	env_log(12,17) = -0.0122;
	env_log(12,18) = 0.0552;
	env_log(12,19) = 0.1241;
	env_log(12,20) = 0.1939;
	env_log(12,21) = 0.2647;
	env_log(12,22) = 0.3356;
	env_log(12,23) = 0.4045;
	env_log(12,24) = 0.4665;
	env_log(12,25) = 0.5137;
	env_log(12,26) = 0.5369;
	env_log(12,27) = 0.5279;
	env_log(12,28) = 0.4866;
	env_log(12,29) = 0.4306;
	env_log(12,30) = 0.4110;
	env_log(12,31) = 0.5348;
	env_log(12,32) = 0.5348;
	env_log(12,33) = 0.5348;
	env_log(12,34) = 0.5348;
	env_log(12,35) = 0.5348;
	env_log(12,36) = 0.5348;
	env_log(12,37) = 0.5348;
	env_log(12,38) = 0.5348;
	env_log(12,39) = 0.5348;
	env_log(12,40) = 0.5348;
	env_log(13, 1) = 0.5092;
	env_log(13, 2) = 0.5092;
	env_log(13, 3) = 0.5092;
	env_log(13, 4) = 0.5092;
	env_log(13, 5) = 0.5092;
	env_log(13, 6) = 0.2103;
	env_log(13, 7) = 0.0426;
	env_log(13, 8) = -0.0573;
	env_log(13, 9) = -0.1256;
	env_log(13,10) = -0.1798;
	env_log(13,11) = -0.2259;
	env_log(13,12) = -0.2633;
	env_log(13,13) = -0.2886;
	env_log(13,14) = -0.2978;
	env_log(13,15) = -0.2878;
	env_log(13,16) = -0.2572;
	env_log(13,17) = -0.2068;
	env_log(13,18) = -0.1390;
	env_log(13,19) = -0.0577;
	env_log(13,20) = 0.0322;
	env_log(13,21) = 0.1255;
	env_log(13,22) = 0.2170;
	env_log(13,23) = 0.3017;
	env_log(13,24) = 0.3757;
	env_log(13,25) = 0.4356;
	env_log(13,26) = 0.4794;
	env_log(13,27) = 0.5061;
	env_log(13,28) = 0.5162;
	env_log(13,29) = 0.5121;
	env_log(13,30) = 0.4989;
	env_log(13,31) = 0.4859;
	env_log(13,32) = 0.4882;
	env_log(13,33) = 0.5299;
	env_log(13,34) = 0.6481;
	env_log(13,35) = 0.6481;
	env_log(13,36) = 0.6481;
	env_log(13,37) = 0.6481;
	env_log(13,38) = 0.6481;
	env_log(13,39) = 0.6481;
	env_log(13,40) = 0.6481;
	env_log(14, 1) = -0.6318;
	env_log(14, 2) = -0.6318;
	env_log(14, 3) = -0.6318;
	env_log(14, 4) = -0.6318;
	env_log(14, 5) = -0.5230;
	env_log(14, 6) = -0.5316;
	env_log(14, 7) = -0.5717;
	env_log(14, 8) = -0.5980;
	env_log(14, 9) = -0.5924;
	env_log(14,10) = -0.5537;
	env_log(14,11) = -0.4895;
	env_log(14,12) = -0.4103;
	env_log(14,13) = -0.3260;
	env_log(14,14) = -0.2433;
	env_log(14,15) = -0.1653;
	env_log(14,16) = -0.0910;
	env_log(14,17) = -0.0168;
	env_log(14,18) = 0.0626;
	env_log(14,19) = 0.1519;
	env_log(14,20) = 0.2542;
	env_log(14,21) = 0.3692;
	env_log(14,22) = 0.4923;
	env_log(14,23) = 0.6146;
	env_log(14,24) = 0.7236;
	env_log(14,25) = 0.8048;
	env_log(14,26) = 0.8457;
	env_log(14,27) = 0.8399;
	env_log(14,28) = 0.7942;
	env_log(14,29) = 0.7369;
	env_log(14,30) = 0.7292;
	env_log(14,31) = 0.8771;
	env_log(14,32) = 0.8771;
	env_log(14,33) = 0.8771;
	env_log(14,34) = 0.8771;
	env_log(14,35) = 0.8771;
	env_log(14,36) = 0.8771;
	env_log(14,37) = 0.8771;
	env_log(14,38) = 0.8771;
	env_log(14,39) = 0.8771;
	env_log(14,40) = 0.8771;
	env_log(15, 1) = -0.9497;
	env_log(15, 2) = -0.9497;
	env_log(15, 3) = -0.9497;
	env_log(15, 4) = -0.8585;
	env_log(15, 5) = -0.7471;
	env_log(15, 6) = -0.6405;
	env_log(15, 7) = -0.5506;
	env_log(15, 8) = -0.4807;
	env_log(15, 9) = -0.4286;
	env_log(15,10) = -0.3900;
	env_log(15,11) = -0.3594;
	env_log(15,12) = -0.3315;
	env_log(15,13) = -0.3019;
	env_log(15,14) = -0.2670;
	env_log(15,15) = -0.2243;
	env_log(15,16) = -0.1720;
	env_log(15,17) = -0.1089;
	env_log(15,18) = -0.0338;
	env_log(15,19) = 0.0539;
	env_log(15,20) = 0.1550;
	env_log(15,21) = 0.2699;
	env_log(15,22) = 0.3986;
	env_log(15,23) = 0.5401;
	env_log(15,24) = 0.6920;
	env_log(15,25) = 0.8496;
	env_log(15,26) = 1.0053;
	env_log(15,27) = 1.1480;
	env_log(15,28) = 1.2622;
	env_log(15,29) = 1.3275;
	env_log(15,30) = 1.3185;
	env_log(15,31) = 1.2050;
	env_log(15,32) = 1.2050;
	env_log(15,33) = 1.2050;
	env_log(15,34) = 1.2050;
	env_log(15,35) = 1.2050;
	env_log(15,36) = 1.2050;
	env_log(15,37) = 1.2050;
	env_log(15,38) = 1.2050;
	env_log(15,39) = 1.2050;
	env_log(15,40) = 1.2050;
	env_log(16, 1) = 0.4247;
	env_log(16, 2) = 0.4247;
	env_log(16, 3) = 0.4247;
	env_log(16, 4) = 0.4247;
	env_log(16, 5) = 0.2875;
	env_log(16, 6) = 0.1294;
	env_log(16, 7) = -0.0105;
	env_log(16, 8) = -0.1152;
	env_log(16, 9) = -0.1814;
	env_log(16,10) = -0.2134;
	env_log(16,11) = -0.2187;
	env_log(16,12) = -0.2051;
	env_log(16,13) = -0.1789;
	env_log(16,14) = -0.1451;
	env_log(16,15) = -0.1063;
	env_log(16,16) = -0.0639;
	env_log(16,17) = -0.0183;
	env_log(16,18) = 0.0300;
	env_log(16,19) = 0.0801;
	env_log(16,20) = 0.1304;
	env_log(16,21) = 0.1779;
	env_log(16,22) = 0.2187;
	env_log(16,23) = 0.2484;
	env_log(16,24) = 0.2621;
	env_log(16,25) = 0.2562;
	env_log(16,26) = 0.2283;
	env_log(16,27) = 0.1785;
	env_log(16,28) = 0.1102;
	env_log(16,29) = 0.0298;
	env_log(16,30) = -0.0534;
	env_log(16,31) = -0.1289;
	env_log(16,32) = -0.1869;
	env_log(16,33) = -0.2231;
	env_log(16,34) = -0.2444;
	env_log(16,35) = -0.2771;
	env_log(16,36) = -0.2771;
	env_log(16,37) = -0.2771;
	env_log(16,38) = -0.2771;
	env_log(16,39) = -0.2771;
	env_log(16,40) = -0.2771;
	env_log(17, 1) = 0.4832;
	env_log(17, 2) = 0.4832;
	env_log(17, 3) = 0.4832;
	env_log(17, 4) = 0.4832;
	env_log(17, 5) = 0.4832;
	env_log(17, 6) = 0.4269;
	env_log(17, 7) = 0.3348;
	env_log(17, 8) = 0.2280;
	env_log(17, 9) = 0.1202;
	env_log(17,10) = 0.0207;
	env_log(17,11) = -0.0644;
	env_log(17,12) = -0.1312;
	env_log(17,13) = -0.1773;
	env_log(17,14) = -0.2016;
	env_log(17,15) = -0.2043;
	env_log(17,16) = -0.1867;
	env_log(17,17) = -0.1514;
	env_log(17,18) = -0.1022;
	env_log(17,19) = -0.0439;
	env_log(17,20) = 0.0179;
	env_log(17,21) = 0.0773;
	env_log(17,22) = 0.1285;
	env_log(17,23) = 0.1664;
	env_log(17,24) = 0.1870;
	env_log(17,25) = 0.1884;
	env_log(17,26) = 0.1707;
	env_log(17,27) = 0.1362;
	env_log(17,28) = 0.0894;
	env_log(17,29) = 0.0361;
	env_log(17,30) = -0.0184;
	env_log(17,31) = -0.0714;
	env_log(17,32) = -0.1262;
	env_log(17,33) = -0.1972;
	env_log(17,34) = -0.3161;
	env_log(17,35) = -0.3161;
	env_log(17,36) = -0.3161;
	env_log(17,37) = -0.3161;
	env_log(17,38) = -0.3161;
	env_log(17,39) = -0.3161;
	env_log(17,40) = -0.3161;
	env_log(18, 1) = 0.9433;
	env_log(18, 2) = 0.9433;
	env_log(18, 3) = 0.9433;
	env_log(18, 4) = 0.9433;
	env_log(18, 5) = 0.9433;
	env_log(18, 6) = 1.1665;
	env_log(18, 7) = 1.2476;
	env_log(18, 8) = 1.2303;
	env_log(18, 9) = 1.1490;
	env_log(18,10) = 1.0296;
	env_log(18,11) = 0.8911;
	env_log(18,12) = 0.7460;
	env_log(18,13) = 0.6024;
	env_log(18,14) = 0.4648;
	env_log(18,15) = 0.3351;
	env_log(18,16) = 0.2138;
	env_log(18,17) = 0.1005;
	env_log(18,18) = -0.0052;
	env_log(18,19) = -0.1034;
	env_log(18,20) = -0.1941;
	env_log(18,21) = -0.2763;
	env_log(18,22) = -0.3491;
	env_log(18,23) = -0.4112;
	env_log(18,24) = -0.4616;
	env_log(18,25) = -0.4995;
	env_log(18,26) = -0.5250;
	env_log(18,27) = -0.5391;
	env_log(18,28) = -0.5431;
	env_log(18,29) = -0.5389;
	env_log(18,30) = -0.5275;
	env_log(18,31) = -0.5080;
	env_log(18,32) = -0.4750;
	env_log(18,33) = -0.4160;
	env_log(18,34) = -0.3072;
	env_log(18,35) = -0.3072;
	env_log(18,36) = -0.3072;
	env_log(18,37) = -0.3072;
	env_log(18,38) = -0.3072;
	env_log(18,39) = -0.3072;
	env_log(18,40) = -0.3072;
	env_log(19, 1) = 0.7399;
	env_log(19, 2) = 0.7399;
	env_log(19, 3) = 0.7399;
	env_log(19, 4) = 0.7399;
	env_log(19, 5) = 0.7399;
	env_log(19, 6) = 0.7399;
	env_log(19, 7) = 0.7399;
	env_log(19, 8) = 1.0855;
	env_log(19, 9) = 1.1414;
	env_log(19,10) = 1.0724;
	env_log(19,11) = 0.9576;
	env_log(19,12) = 0.8266;
	env_log(19,13) = 0.6848;
	env_log(19,14) = 0.5295;
	env_log(19,15) = 0.3593;
	env_log(19,16) = 0.1774;
	env_log(19,17) = -0.0075;
	env_log(19,18) = -0.1832;
	env_log(19,19) = -0.3364;
	env_log(19,20) = -0.4560;
	env_log(19,21) = -0.5346;
	env_log(19,22) = -0.5704;
	env_log(19,23) = -0.5663;
	env_log(19,24) = -0.5292;
	env_log(19,25) = -0.4674;
	env_log(19,26) = -0.3879;
	env_log(19,27) = -0.2942;
	env_log(19,28) = -0.1848;
	env_log(19,29) = -0.0547;
	env_log(19,30) = 0.0992;
	env_log(19,31) = 0.2673;
	env_log(19,32) = 0.4088;
	env_log(19,33) = 0.4088;
	env_log(19,34) = 0.4088;
	env_log(19,35) = 0.4088;
	env_log(19,36) = 0.4088;
	env_log(19,37) = 0.4088;
	env_log(19,38) = 0.4088;
	env_log(19,39) = 0.4088;
	env_log(19,40) = 0.4088;
	env_log(20, 1) = 0.7893;
	env_log(20, 2) = 0.7893;
	env_log(20, 3) = 0.7893;
	env_log(20, 4) = 0.7893;
	env_log(20, 5) = 0.7893;
	env_log(20, 6) = 0.7893;
	env_log(20, 7) = 0.7983;
	env_log(20, 8) = 0.8278;
	env_log(20, 9) = 0.8508;
	env_log(20,10) = 0.8467;
	env_log(20,11) = 0.8031;
	env_log(20,12) = 0.7158;
	env_log(20,13) = 0.5881;
	env_log(20,14) = 0.4290;
	env_log(20,15) = 0.2515;
	env_log(20,16) = 0.0703;
	env_log(20,17) = -0.1001;
	env_log(20,18) = -0.2470;
	env_log(20,19) = -0.3609;
	env_log(20,20) = -0.4365;
	env_log(20,21) = -0.4726;
	env_log(20,22) = -0.4718;
	env_log(20,23) = -0.4398;
	env_log(20,24) = -0.3844;
	env_log(20,25) = -0.3132;
	env_log(20,26) = -0.2325;
	env_log(20,27) = -0.1454;
	env_log(20,28) = -0.0506;
	env_log(20,29) = 0.0585;
	env_log(20,30) = 0.1934;
	env_log(20,31) = 0.3689;
	env_log(20,32) = 0.5996;
	env_log(20,33) = 0.5996;
	env_log(20,34) = 0.5996;
	env_log(20,35) = 0.5996;
	env_log(20,36) = 0.5996;
	env_log(20,37) = 0.5996;
	env_log(20,38) = 0.5996;
	env_log(20,39) = 0.5996;
	env_log(20,40) = 0.5996;
}

void
SecondaryStructureEval::env_log_initializer_section3(
	FArray2D_float & env_log
)
{
	// section 3
	env_log(21, 1) = 0.0000;
	env_log(21, 2) = 0.0000;
	env_log(21, 3) = 0.0000;
	env_log(21, 4) = 0.0000;
	env_log(21, 5) = 0.0000;
	env_log(21, 6) = 0.0000;
	env_log(21, 7) = 0.0000;
	env_log(21, 8) = 0.0000;
	env_log(21, 9) = 0.0000;
	env_log(21,10) = 0.0000;
	env_log(21,11) = 0.0000;
	env_log(21,12) = 0.0000;
	env_log(21,13) = 0.0000;
	env_log(21,14) = 0.0000;
	env_log(21,15) = 0.0000;
	env_log(21,16) = 0.0000;
	env_log(21,17) = 0.0000;
	env_log(21,18) = 0.0000;
	env_log(21,19) = 0.0000;
	env_log(21,20) = 0.0000;
	env_log(21,21) = 0.0000;
	env_log(21,22) = 0.0000;
	env_log(21,23) = 0.0000;
	env_log(21,24) = 0.0000;
	env_log(21,25) = 0.0000;
	env_log(21,26) = 0.0000;
	env_log(21,27) = 0.0000;
	env_log(21,28) = 0.0000;
	env_log(21,29) = 0.0000;
	env_log(21,30) = 0.0000;
	env_log(21,31) = 0.0000;
	env_log(21,32) = 0.0000;
	env_log(21,33) = 0.0000;
	env_log(21,34) = 0.0000;
	env_log(21,35) = 0.0000;
	env_log(21,36) = 0.0000;
	env_log(21,37) = 0.0000;
	env_log(21,38) = 0.0000;
	env_log(21,39) = 0.0000;
	env_log(21,40) = 0.0000;
	env_log(22, 1) = 0.0000;
	env_log(22, 2) = 0.0000;
	env_log(22, 3) = 0.0000;
	env_log(22, 4) = 0.0000;
	env_log(22, 5) = 0.0000;
	env_log(22, 6) = 0.0000;
	env_log(22, 7) = 0.0000;
	env_log(22, 8) = 0.0000;
	env_log(22, 9) = 0.0000;
	env_log(22,10) = 0.0000;
	env_log(22,11) = 0.0000;
	env_log(22,12) = 0.0000;
	env_log(22,13) = 0.0000;
	env_log(22,14) = 0.0000;
	env_log(22,15) = 0.0000;
	env_log(22,16) = 0.0000;
	env_log(22,17) = 0.0000;
	env_log(22,18) = 0.0000;
	env_log(22,19) = 0.0000;
	env_log(22,20) = 0.0000;
	env_log(22,21) = 0.0000;
	env_log(22,22) = 0.0000;
	env_log(22,23) = 0.0000;
	env_log(22,24) = 0.0000;
	env_log(22,25) = 0.0000;
	env_log(22,26) = 0.0000;
	env_log(22,27) = 0.0000;
	env_log(22,28) = 0.0000;
	env_log(22,29) = 0.0000;
	env_log(22,30) = 0.0000;
	env_log(22,31) = 0.0000;
	env_log(22,32) = 0.0000;
	env_log(22,33) = 0.0000;
	env_log(22,34) = 0.0000;
	env_log(22,35) = 0.0000;
	env_log(22,36) = 0.0000;
	env_log(22,37) = 0.0000;
	env_log(22,38) = 0.0000;
	env_log(22,39) = 0.0000;
	env_log(22,40) = 0.0000;
	env_log(23, 1) = 0.0000;
	env_log(23, 2) = 0.0000;
	env_log(23, 3) = 0.0000;
	env_log(23, 4) = 0.0000;
	env_log(23, 5) = 0.0000;
	env_log(23, 6) = 0.0000;
	env_log(23, 7) = 0.0000;
	env_log(23, 8) = 0.0000;
	env_log(23, 9) = 0.0000;
	env_log(23,10) = 0.0000;
	env_log(23,11) = 0.0000;
	env_log(23,12) = 0.0000;
	env_log(23,13) = 0.0000;
	env_log(23,14) = 0.0000;
	env_log(23,15) = 0.0000;
	env_log(23,16) = 0.0000;
	env_log(23,17) = 0.0000;
	env_log(23,18) = 0.0000;
	env_log(23,19) = 0.0000;
	env_log(23,20) = 0.0000;
	env_log(23,21) = 0.0000;
	env_log(23,22) = 0.0000;
	env_log(23,23) = 0.0000;
	env_log(23,24) = 0.0000;
	env_log(23,25) = 0.0000;
	env_log(23,26) = 0.0000;
	env_log(23,27) = 0.0000;
	env_log(23,28) = 0.0000;
	env_log(23,29) = 0.0000;
	env_log(23,30) = 0.0000;
	env_log(23,31) = 0.0000;
	env_log(23,32) = 0.0000;
	env_log(23,33) = 0.0000;
	env_log(23,34) = 0.0000;
	env_log(23,35) = 0.0000;
	env_log(23,36) = 0.0000;
	env_log(23,37) = 0.0000;
	env_log(23,38) = 0.0000;
	env_log(23,39) = 0.0000;
	env_log(23,40) = 0.0000;
	env_log(24, 1) = 0.0000;
	env_log(24, 2) = 0.0000;
	env_log(24, 3) = 0.0000;
	env_log(24, 4) = 0.0000;
	env_log(24, 5) = 0.0000;
	env_log(24, 6) = 0.0000;
	env_log(24, 7) = 0.0000;
	env_log(24, 8) = 0.0000;
	env_log(24, 9) = 0.0000;
	env_log(24,10) = 0.0000;
	env_log(24,11) = 0.0000;
	env_log(24,12) = 0.0000;
	env_log(24,13) = 0.0000;
	env_log(24,14) = 0.0000;
	env_log(24,15) = 0.0000;
	env_log(24,16) = 0.0000;
	env_log(24,17) = 0.0000;
	env_log(24,18) = 0.0000;
	env_log(24,19) = 0.0000;
	env_log(24,20) = 0.0000;
	env_log(24,21) = 0.0000;
	env_log(24,22) = 0.0000;
	env_log(24,23) = 0.0000;
	env_log(24,24) = 0.0000;
	env_log(24,25) = 0.0000;
	env_log(24,26) = 0.0000;
	env_log(24,27) = 0.0000;
	env_log(24,28) = 0.0000;
	env_log(24,29) = 0.0000;
	env_log(24,30) = 0.0000;
	env_log(24,31) = 0.0000;
	env_log(24,32) = 0.0000;
	env_log(24,33) = 0.0000;
	env_log(24,34) = 0.0000;
	env_log(24,35) = 0.0000;
	env_log(24,36) = 0.0000;
	env_log(24,37) = 0.0000;
	env_log(24,38) = 0.0000;
	env_log(24,39) = 0.0000;
	env_log(24,40) = 0.0000;
	env_log(25, 1) = 0.0000;
	env_log(25, 2) = 0.0000;
	env_log(25, 3) = 0.0000;
	env_log(25, 4) = 0.0000;
	env_log(25, 5) = 0.0000;
	env_log(25, 6) = 0.0000;
	env_log(25, 7) = 0.0000;
	env_log(25, 8) = 0.0000;
	env_log(25, 9) = 0.0000;
	env_log(25,10) = 0.0000;
	env_log(25,11) = 0.0000;
	env_log(25,12) = 0.0000;
	env_log(25,13) = 0.0000;
	env_log(25,14) = 0.0000;
	env_log(25,15) = 0.0000;
	env_log(25,16) = 0.0000;
	env_log(25,17) = 0.0000;
	env_log(25,18) = 0.0000;
	env_log(25,19) = 0.0000;
	env_log(25,20) = 0.0000;
	env_log(25,21) = 0.0000;
	env_log(25,22) = 0.0000;
	env_log(25,23) = 0.0000;
	env_log(25,24) = 0.0000;
	env_log(25,25) = 0.0000;
	env_log(25,26) = 0.0000;
	env_log(25,27) = 0.0000;
	env_log(25,28) = 0.0000;
	env_log(25,29) = 0.0000;
	env_log(25,30) = 0.0000;
	env_log(25,31) = 0.0000;
	env_log(25,32) = 0.0000;
	env_log(25,33) = 0.0000;
	env_log(25,34) = 0.0000;
	env_log(25,35) = 0.0000;
	env_log(25,36) = 0.0000;
	env_log(25,37) = 0.0000;
	env_log(25,38) = 0.0000;
	env_log(25,39) = 0.0000;
	env_log(25,40) = 0.0000;
	env_log(26, 1) = 0.0000;
	env_log(26, 2) = 0.0000;
	env_log(26, 3) = 0.0000;
	env_log(26, 4) = 0.0000;
	env_log(26, 5) = 0.0000;
	env_log(26, 6) = 0.0000;
	env_log(26, 7) = 0.0000;
	env_log(26, 8) = 0.0000;
	env_log(26, 9) = 0.0000;
	env_log(26,10) = 0.0000;
	env_log(26,11) = 0.0000;
	env_log(26,12) = 0.0000;
	env_log(26,13) = 0.0000;
	env_log(26,14) = 0.0000;
	env_log(26,15) = 0.0000;
	env_log(26,16) = 0.0000;
	env_log(26,17) = 0.0000;
	env_log(26,18) = 0.0000;
	env_log(26,19) = 0.0000;
	env_log(26,20) = 0.0000;
	env_log(26,21) = 0.0000;
	env_log(26,22) = 0.0000;
	env_log(26,23) = 0.0000;
	env_log(26,24) = 0.0000;
	env_log(26,25) = 0.0000;
	env_log(26,26) = 0.0000;
	env_log(26,27) = 0.0000;
	env_log(26,28) = 0.0000;
	env_log(26,29) = 0.0000;
	env_log(26,30) = 0.0000;
	env_log(26,31) = 0.0000;
	env_log(26,32) = 0.0000;
	env_log(26,33) = 0.0000;
	env_log(26,34) = 0.0000;
	env_log(26,35) = 0.0000;
	env_log(26,36) = 0.0000;
	env_log(26,37) = 0.0000;
	env_log(26,38) = 0.0000;
	env_log(26,39) = 0.0000;
	env_log(26,40) = 0.0000;
	env_log(27, 1) = 0.0000;
	env_log(27, 2) = 0.0000;
	env_log(27, 3) = 0.0000;
	env_log(27, 4) = 0.0000;
	env_log(27, 5) = 0.0000;
	env_log(27, 6) = 0.0000;
	env_log(27, 7) = 0.0000;
	env_log(27, 8) = 0.0000;
	env_log(27, 9) = 0.0000;
	env_log(27,10) = 0.0000;
	env_log(27,11) = 0.0000;
	env_log(27,12) = 0.0000;
	env_log(27,13) = 0.0000;
	env_log(27,14) = 0.0000;
	env_log(27,15) = 0.0000;
	env_log(27,16) = 0.0000;
	env_log(27,17) = 0.0000;
	env_log(27,18) = 0.0000;
	env_log(27,19) = 0.0000;
	env_log(27,20) = 0.0000;
	env_log(27,21) = 0.0000;
	env_log(27,22) = 0.0000;
	env_log(27,23) = 0.0000;
	env_log(27,24) = 0.0000;
	env_log(27,25) = 0.0000;
	env_log(27,26) = 0.0000;
	env_log(27,27) = 0.0000;
	env_log(27,28) = 0.0000;
	env_log(27,29) = 0.0000;
	env_log(27,30) = 0.0000;
	env_log(27,31) = 0.0000;
	env_log(27,32) = 0.0000;
	env_log(27,33) = 0.0000;
	env_log(27,34) = 0.0000;
	env_log(27,35) = 0.0000;
	env_log(27,36) = 0.0000;
	env_log(27,37) = 0.0000;
	env_log(27,38) = 0.0000;
	env_log(27,39) = 0.0000;
	env_log(27,40) = 0.0000;
	env_log(28, 1) = 0.0000;
	env_log(28, 2) = 0.0000;
	env_log(28, 3) = 0.0000;
	env_log(28, 4) = 0.0000;
	env_log(28, 5) = 0.0000;
	env_log(28, 6) = 0.0000;
	env_log(28, 7) = 0.0000;
	env_log(28, 8) = 0.0000;
	env_log(28, 9) = 0.0000;
	env_log(28,10) = 0.0000;
	env_log(28,11) = 0.0000;
	env_log(28,12) = 0.0000;
	env_log(28,13) = 0.0000;
	env_log(28,14) = 0.0000;
	env_log(28,15) = 0.0000;
	env_log(28,16) = 0.0000;
	env_log(28,17) = 0.0000;
	env_log(28,18) = 0.0000;
	env_log(28,19) = 0.0000;
	env_log(28,20) = 0.0000;
	env_log(28,21) = 0.0000;
	env_log(28,22) = 0.0000;
	env_log(28,23) = 0.0000;
	env_log(28,24) = 0.0000;
	env_log(28,25) = 0.0000;
	env_log(28,26) = 0.0000;
	env_log(28,27) = 0.0000;
	env_log(28,28) = 0.0000;
	env_log(28,29) = 0.0000;
	env_log(28,30) = 0.0000;
	env_log(28,31) = 0.0000;
	env_log(28,32) = 0.0000;
	env_log(28,33) = 0.0000;
	env_log(28,34) = 0.0000;
	env_log(28,35) = 0.0000;
	env_log(28,36) = 0.0000;
	env_log(28,37) = 0.0000;
	env_log(28,38) = 0.0000;
	env_log(28,39) = 0.0000;
	env_log(28,40) = 0.0000;

	//KMa phospho_ser
	for (int i=1;i<=40;i++) {
		env_log(29, i) = 0.0000;
	}
	for (int i=1;i<=40;i++) {
		env_log(21, i) = env_log(16, i);
	}
}

void
SecondaryStructureEval::pair_log_initializer_master(
	FArray3D_float & pair_log
)
{
	pair_log_initializer_section1a(pair_log);
	pair_log_initializer_section1b(pair_log);
	pair_log_initializer_section2a(pair_log);
	pair_log_initializer_section2b(pair_log);
	pair_log_initializer_section3a(pair_log);
	pair_log_initializer_section3b(pair_log);
	pair_log_initializer_section4a(pair_log);
	pair_log_initializer_section4b(pair_log);
	pair_log_initializer_section5a(pair_log);
	pair_log_initializer_section5b(pair_log);
}

void
SecondaryStructureEval::pair_log_initializer_section1a(
	FArray3D_float & pair_log
)
{
	// section 1a
	pair_log(1, 1, 1) = -0.13571;
	pair_log(1, 1, 2) = 0.23079;
	pair_log(1, 1, 3) = 0.33397;
	pair_log(1, 1, 4) = 0.26471;
	pair_log(1, 1, 5) = 0.08139;
	pair_log(1, 1, 6) = -0.01467;
	pair_log(1, 1, 7) = 0.32413;
	pair_log(1, 1, 8) = -0.22176;
	pair_log(1, 1, 9) = 0.39894;
	pair_log(1, 1,10) = -0.15025;
	pair_log(1, 1,11) = 0.03521;
	pair_log(1, 1,12) = 0.25935;
	pair_log(1, 1,13) = 0.03615;
	pair_log(1, 1,14) = 0.26575;
	pair_log(1, 1,15) = 0.37121;
	pair_log(1, 1,16) = 0.10475;
	pair_log(1, 1,17) = -0.04679;
	pair_log(1, 1,18) = -0.27134;
	pair_log(1, 1,19) = 0.20001;
	pair_log(1, 1,20) = 0.12835;
	pair_log(1, 1,21) = 0.00000;
	pair_log(1, 1,22) = 0.00000;
	pair_log(1, 1,23) = 0.00000;
	pair_log(1, 1,24) = 0.00000;
	pair_log(1, 1,25) = 0.00000;
	pair_log(1, 1,26) = 0.00000;
	pair_log(1, 1,27) = 0.00000;
	pair_log(1, 1,28) = 0.00000;
	pair_log(1, 2, 1) = 0.23079;
	pair_log(1, 2, 2) = -1.79243;
	pair_log(1, 2, 3) = 0.53024;
	pair_log(1, 2, 4) = 0.66360;
	pair_log(1, 2, 5) = 0.09641;
	pair_log(1, 2, 6) = 0.20423;
	pair_log(1, 2, 7) = 0.24383;
	pair_log(1, 2, 8) = 0.12534;
	pair_log(1, 2, 9) = 0.73178;
	pair_log(1, 2,10) = 0.10747;
	pair_log(1, 2,11) = 0.04462;
	pair_log(1, 2,12) = 0.32481;
	pair_log(1, 2,13) = -0.04477;
	pair_log(1, 2,14) = 0.25200;
	pair_log(1, 2,15) = 0.49103;
	pair_log(1, 2,16) = 0.11283;
	pair_log(1, 2,17) = 0.27539;
	pair_log(1, 2,18) = 0.09778;
	pair_log(1, 2,19) = 0.09401;
	pair_log(1, 2,20) = 0.16464;
	pair_log(1, 2,21) = 0.00000;
	pair_log(1, 2,22) = 0.00000;
	pair_log(1, 2,23) = 0.00000;
	pair_log(1, 2,24) = 0.00000;
	pair_log(1, 2,25) = 0.00000;
	pair_log(1, 2,26) = 0.00000;
	pair_log(1, 2,27) = 0.00000;
	pair_log(1, 2,28) = 0.00000;
	pair_log(1, 3, 1) = 0.33397;
	pair_log(1, 3, 2) = 0.53024;
	pair_log(1, 3, 3) = 0.06704;
	pair_log(1, 3, 4) = 0.24572;
	pair_log(1, 3, 5) = 0.53784;
	pair_log(1, 3, 6) = -0.22435;
	pair_log(1, 3, 7) = -0.47402;
	pair_log(1, 3, 8) = 0.44972;
	pair_log(1, 3, 9) = -0.90014;
	pair_log(1, 3,10) = 0.38200;
	pair_log(1, 3,11) = 0.32317;
	pair_log(1, 3,12) = -0.41830;
	pair_log(1, 3,13) = -0.21449;
	pair_log(1, 3,14) = -0.00061;
	pair_log(1, 3,15) = -0.78243;
	pair_log(1, 3,16) = -0.50285;
	pair_log(1, 3,17) = -0.36579;
	pair_log(1, 3,18) = 0.44658;
	pair_log(1, 3,19) = 0.50378;
	pair_log(1, 3,20) = 0.46473;
	pair_log(1, 3,21) = 0.00000;
	pair_log(1, 3,22) = 0.00000;
	pair_log(1, 3,23) = 0.00000;
	pair_log(1, 3,24) = 0.00000;
	pair_log(1, 3,25) = 0.00000;
	pair_log(1, 3,26) = 0.00000;
	pair_log(1, 3,27) = 0.00000;
	pair_log(1, 3,28) = 0.00000;
	pair_log(1, 4, 1) = 0.26471;
	pair_log(1, 4, 2) = 0.66360;
	pair_log(1, 4, 3) = 0.24572;
	pair_log(1, 4, 4) = 0.51101;
	pair_log(1, 4, 5) = 0.25134;
	pair_log(1, 4, 6) = 0.14922;
	pair_log(1, 4, 7) = -0.10674;
	pair_log(1, 4, 8) = 0.21945;
	pair_log(1, 4, 9) = -0.92364;
	pair_log(1, 4,10) = 0.19892;
	pair_log(1, 4,11) = 0.15161;
	pair_log(1, 4,12) = -0.18010;
	pair_log(1, 4,13) = -0.14194;
	pair_log(1, 4,14) = -0.05835;
	pair_log(1, 4,15) = -0.78607;
	pair_log(1, 4,16) = -0.35312;
	pair_log(1, 4,17) = -0.29144;
	pair_log(1, 4,18) = 0.22764;
	pair_log(1, 4,19) = 0.16071;
	pair_log(1, 4,20) = 0.18883;
	pair_log(1, 4,21) = 0.00000;
	pair_log(1, 4,22) = 0.00000;
	pair_log(1, 4,23) = 0.00000;
	pair_log(1, 4,24) = 0.00000;
	pair_log(1, 4,25) = 0.00000;
	pair_log(1, 4,26) = 0.00000;
	pair_log(1, 4,27) = 0.00000;
	pair_log(1, 4,28) = 0.00000;
	pair_log(1, 5, 1) = 0.08139;
	pair_log(1, 5, 2) = 0.09641;
	pair_log(1, 5, 3) = 0.53784;
	pair_log(1, 5, 4) = 0.25134;
	pair_log(1, 5, 5) = -0.24651;
	pair_log(1, 5, 6) = 0.21293;
	pair_log(1, 5, 7) = 0.03923;
	pair_log(1, 5, 8) = -0.19106;
	pair_log(1, 5, 9) = 0.04828;
	pair_log(1, 5,10) = -0.22682;
	pair_log(1, 5,11) = -0.23360;
	pair_log(1, 5,12) = 0.26608;
	pair_log(1, 5,13) = -0.10484;
	pair_log(1, 5,14) = 0.14340;
	pair_log(1, 5,15) = 0.03136;
	pair_log(1, 5,16) = 0.28005;
	pair_log(1, 5,17) = 0.37472;
	pair_log(1, 5,18) = -0.08226;
	pair_log(1, 5,19) = -0.13588;
	pair_log(1, 5,20) = -0.01526;
	pair_log(1, 5,21) = 0.00000;
	pair_log(1, 5,22) = 0.00000;
	pair_log(1, 5,23) = 0.00000;
	pair_log(1, 5,24) = 0.00000;
	pair_log(1, 5,25) = 0.00000;
	pair_log(1, 5,26) = 0.00000;
	pair_log(1, 5,27) = 0.00000;
	pair_log(1, 5,28) = 0.00000;
	pair_log(1, 6, 1) = -0.01467;
	pair_log(1, 6, 2) = 0.20423;
	pair_log(1, 6, 3) = -0.22435;
	pair_log(1, 6, 4) = 0.14922;
	pair_log(1, 6, 5) = 0.21293;
	pair_log(1, 6, 6) = -0.48115;
	pair_log(1, 6, 7) = 0.08603;
	pair_log(1, 6, 8) = 0.36527;
	pair_log(1, 6, 9) = 0.00501;
	pair_log(1, 6,10) = 0.30617;
	pair_log(1, 6,11) = 0.14609;
	pair_log(1, 6,12) = -0.37069;
	pair_log(1, 6,13) = -0.18438;
	pair_log(1, 6,14) = -0.01890;
	pair_log(1, 6,15) = -0.08319;
	pair_log(1, 6,16) = -0.27119;
	pair_log(1, 6,17) = -0.24551;
	pair_log(1, 6,18) = 0.23105;
	pair_log(1, 6,19) = 0.24344;
	pair_log(1, 6,20) = 0.10640;
	pair_log(1, 6,21) = 0.00000;
	pair_log(1, 6,22) = 0.00000;
	pair_log(1, 6,23) = 0.00000;
	pair_log(1, 6,24) = 0.00000;
	pair_log(1, 6,25) = 0.00000;
	pair_log(1, 6,26) = 0.00000;
	pair_log(1, 6,27) = 0.00000;
	pair_log(1, 6,28) = 0.00000;
	pair_log(1, 7, 1) = 0.32413;
	pair_log(1, 7, 2) = 0.24383;
	pair_log(1, 7, 3) = -0.47402;
	pair_log(1, 7, 4) = -0.10674;
	pair_log(1, 7, 5) = 0.03923;
	pair_log(1, 7, 6) = 0.08603;
	pair_log(1, 7, 7) = -0.23317;
	pair_log(1, 7, 8) = 0.10553;
	pair_log(1, 7, 9) = 0.00361;
	pair_log(1, 7,10) = 0.11443;
	pair_log(1, 7,11) = 0.01416;
	pair_log(1, 7,12) = -0.00420;
	pair_log(1, 7,13) = -0.27877;
	pair_log(1, 7,14) = -0.03046;
	pair_log(1, 7,15) = -0.10894;
	pair_log(1, 7,16) = -0.11302;
	pair_log(1, 7,17) = -0.25624;
	pair_log(1, 7,18) = 0.15787;
	pair_log(1, 7,19) = -0.17229;
	pair_log(1, 7,20) = 0.02691;
	pair_log(1, 7,21) = 0.00000;
	pair_log(1, 7,22) = 0.00000;
	pair_log(1, 7,23) = 0.00000;
	pair_log(1, 7,24) = 0.00000;
	pair_log(1, 7,25) = 0.00000;
	pair_log(1, 7,26) = 0.00000;
	pair_log(1, 7,27) = 0.00000;
	pair_log(1, 7,28) = 0.00000;
	pair_log(1, 8, 1) = -0.22176;
	pair_log(1, 8, 2) = 0.12534;
	pair_log(1, 8, 3) = 0.44972;
	pair_log(1, 8, 4) = 0.21945;
	pair_log(1, 8, 5) = -0.19106;
	pair_log(1, 8, 6) = 0.36527;
	pair_log(1, 8, 7) = 0.10553;
	pair_log(1, 8, 8) = -0.31702;
	pair_log(1, 8, 9) = 0.21697;
	pair_log(1, 8,10) = -0.12606;
	pair_log(1, 8,11) = -0.08790;
	pair_log(1, 8,12) = 0.38282;
	pair_log(1, 8,13) = 0.56031;
	pair_log(1, 8,14) = 0.20555;
	pair_log(1, 8,15) = 0.33584;
	pair_log(1, 8,16) = 0.31299;
	pair_log(1, 8,17) = 0.28667;
	pair_log(1, 8,18) = -0.19634;
	pair_log(1, 8,19) = -0.15983;
	pair_log(1, 8,20) = -0.16964;
	pair_log(1, 8,21) = 0.00000;
	pair_log(1, 8,22) = 0.00000;
	pair_log(1, 8,23) = 0.00000;
	pair_log(1, 8,24) = 0.00000;
	pair_log(1, 8,25) = 0.00000;
	pair_log(1, 8,26) = 0.00000;
	pair_log(1, 8,27) = 0.00000;
	pair_log(1, 8,28) = 0.00000;
	pair_log(1, 9, 1) = 0.39894;
	pair_log(1, 9, 2) = 0.73178;
	pair_log(1, 9, 3) = -0.90014;
	pair_log(1, 9, 4) = -0.92364;
	pair_log(1, 9, 5) = 0.04828;
	pair_log(1, 9, 6) = 0.00501;
	pair_log(1, 9, 7) = 0.00361;
	pair_log(1, 9, 8) = 0.21697;
	pair_log(1, 9, 9) = 0.56407;
	pair_log(1, 9,10) = 0.21292;
	pair_log(1, 9,11) = 0.15363;
	pair_log(1, 9,12) = -0.07038;
	pair_log(1, 9,13) = 0.04081;
	pair_log(1, 9,14) = -0.24804;
	pair_log(1, 9,15) = 0.55155;
	pair_log(1, 9,16) = 0.02715;
	pair_log(1, 9,17) = -0.13219;
	pair_log(1, 9,18) = 0.23784;
	pair_log(1, 9,19) = -0.24586;
	pair_log(1, 9,20) = -0.16896;
	pair_log(1, 9,21) = 0.00000;
	pair_log(1, 9,22) = 0.00000;
	pair_log(1, 9,23) = 0.00000;
	pair_log(1, 9,24) = 0.00000;
	pair_log(1, 9,25) = 0.00000;
	pair_log(1, 9,26) = 0.00000;
	pair_log(1, 9,27) = 0.00000;
	pair_log(1, 9,28) = 0.00000;
	pair_log(1,10, 1) = -0.15025;
	pair_log(1,10, 2) = 0.10747;
	pair_log(1,10, 3) = 0.38200;
	pair_log(1,10, 4) = 0.19892;
	pair_log(1,10, 5) = -0.22682;
	pair_log(1,10, 6) = 0.30617;
	pair_log(1,10, 7) = 0.11443;
	pair_log(1,10, 8) = -0.12606;
	pair_log(1,10, 9) = 0.21292;
	pair_log(1,10,10) = -0.19983;
	pair_log(1,10,11) = -0.12860;
	pair_log(1,10,12) = 0.35359;
	pair_log(1,10,13) = 0.35217;
	pair_log(1,10,14) = 0.07523;
	pair_log(1,10,15) = 0.19784;
	pair_log(1,10,16) = 0.27135;
	pair_log(1,10,17) = 0.31011;
	pair_log(1,10,18) = -0.10641;
	pair_log(1,10,19) = -0.16843;
	pair_log(1,10,20) = -0.20609;
	pair_log(1,10,21) = 0.00000;
	pair_log(1,10,22) = 0.00000;
	pair_log(1,10,23) = 0.00000;
	pair_log(1,10,24) = 0.00000;
	pair_log(1,10,25) = 0.00000;
	pair_log(1,10,26) = 0.00000;
	pair_log(1,10,27) = 0.00000;
	pair_log(1,10,28) = 0.00000;
	pair_log(1,11, 1) = 0.03521;
	pair_log(1,11, 2) = 0.04462;
	pair_log(1,11, 3) = 0.32317;
	pair_log(1,11, 4) = 0.15161;
	pair_log(1,11, 5) = -0.23360;
	pair_log(1,11, 6) = 0.14609;
	pair_log(1,11, 7) = 0.01416;
	pair_log(1,11, 8) = -0.08790;
	pair_log(1,11, 9) = 0.15363;
	pair_log(1,11,10) = -0.12860;
	pair_log(1,11,11) = -0.13998;
	pair_log(1,11,12) = 0.02882;
	pair_log(1,11,13) = 0.11287;
	pair_log(1,11,14) = -0.03542;
	pair_log(1,11,15) = 0.12999;
	pair_log(1,11,16) = 0.19696;
	pair_log(1,11,17) = 0.30090;
	pair_log(1,11,18) = 0.00637;
	pair_log(1,11,19) = -0.09998;
	pair_log(1,11,20) = -0.22924;
	pair_log(1,11,21) = 0.00000;
	pair_log(1,11,22) = 0.00000;
	pair_log(1,11,23) = 0.00000;
	pair_log(1,11,24) = 0.00000;
	pair_log(1,11,25) = 0.00000;
	pair_log(1,11,26) = 0.00000;
	pair_log(1,11,27) = 0.00000;
	pair_log(1,11,28) = 0.00000;
	pair_log(1,12, 1) = 0.25935;
	pair_log(1,12, 2) = 0.32481;
	pair_log(1,12, 3) = -0.41830;
	pair_log(1,12, 4) = -0.18010;
	pair_log(1,12, 5) = 0.26608;
	pair_log(1,12, 6) = -0.37069;
	pair_log(1,12, 7) = -0.00420;
	pair_log(1,12, 8) = 0.38282;
	pair_log(1,12, 9) = -0.07038;
	pair_log(1,12,10) = 0.35359;
	pair_log(1,12,11) = 0.02882;
	pair_log(1,12,12) = -0.61455;
	pair_log(1,12,13) = -0.20175;
	pair_log(1,12,14) = -0.26143;
	pair_log(1,12,15) = -0.27050;
	pair_log(1,12,16) = -0.28825;
	pair_log(1,12,17) = -0.28531;
	pair_log(1,12,18) = 0.47451;
	pair_log(1,12,19) = 0.28602;
	pair_log(1,12,20) = 0.12638;
	pair_log(1,12,21) = 0.00000;
	pair_log(1,12,22) = 0.00000;
	pair_log(1,12,23) = 0.00000;
	pair_log(1,12,24) = 0.00000;
	pair_log(1,12,25) = 0.00000;
	pair_log(1,12,26) = 0.00000;
	pair_log(1,12,27) = 0.00000;
	pair_log(1,12,28) = 0.00000;
	pair_log(1,13, 1) = 0.03615;
	pair_log(1,13, 2) = -0.04477;
	pair_log(1,13, 3) = -0.21449;
	pair_log(1,13, 4) = -0.14194;
	pair_log(1,13, 5) = -0.10484;
	pair_log(1,13, 6) = -0.18438;
	pair_log(1,13, 7) = -0.27877;
	pair_log(1,13, 8) = 0.56031;
	pair_log(1,13, 9) = 0.04081;
	pair_log(1,13,10) = 0.35217;
	pair_log(1,13,11) = 0.11287;
	pair_log(1,13,12) = -0.20175;
	pair_log(1,13,13) = -0.04170;
	pair_log(1,13,14) = -0.16569;
	pair_log(1,13,15) = -0.06999;
	pair_log(1,13,16) = -0.10791;
	pair_log(1,13,17) = 0.02844;
	pair_log(1,13,18) = 0.39761;
	pair_log(1,13,19) = -0.55908;
	pair_log(1,13,20) = -0.41613;
	pair_log(1,13,21) = 0.00000;
	pair_log(1,13,22) = 0.00000;
	pair_log(1,13,23) = 0.00000;
	pair_log(1,13,24) = 0.00000;
	pair_log(1,13,25) = 0.00000;
	pair_log(1,13,26) = 0.00000;
	pair_log(1,13,27) = 0.00000;
	pair_log(1,13,28) = 0.00000;
	pair_log(1,14, 1) = 0.26575;
	pair_log(1,14, 2) = 0.25200;
	pair_log(1,14, 3) = -0.00061;
	pair_log(1,14, 4) = -0.05835;
	pair_log(1,14, 5) = 0.14340;
	pair_log(1,14, 6) = -0.01890;
	pair_log(1,14, 7) = -0.03046;
	pair_log(1,14, 8) = 0.20555;
	pair_log(1,14, 9) = -0.24804;
	pair_log(1,14,10) = 0.07523;
	pair_log(1,14,11) = -0.03542;
	pair_log(1,14,12) = -0.26143;
	pair_log(1,14,13) = -0.16569;
	pair_log(1,14,14) = -0.24068;
	pair_log(1,14,15) = -0.25307;
	pair_log(1,14,16) = -0.06140;
	pair_log(1,14,17) = -0.23014;
	pair_log(1,14,18) = 0.08581;
	pair_log(1,14,19) = -0.04570;
	pair_log(1,14,20) = -0.03777;
	pair_log(1,14,21) = 0.00000;
	pair_log(1,14,22) = 0.00000;
	pair_log(1,14,23) = 0.00000;
	pair_log(1,14,24) = 0.00000;
	pair_log(1,14,25) = 0.00000;
	pair_log(1,14,26) = 0.00000;
	pair_log(1,14,27) = 0.00000;
	pair_log(1,14,28) = 0.00000;
	pair_log(1,15, 1) = 0.37121;
	pair_log(1,15, 2) = 0.49103;
	pair_log(1,15, 3) = -0.78243;
	pair_log(1,15, 4) = -0.78607;
	pair_log(1,15, 5) = 0.03136;
	pair_log(1,15, 6) = -0.08319;
	pair_log(1,15, 7) = -0.10894;
	pair_log(1,15, 8) = 0.33584;
	pair_log(1,15, 9) = 0.55155;
	pair_log(1,15,10) = 0.19784;
	pair_log(1,15,11) = 0.12999;
	pair_log(1,15,12) = -0.27050;
	pair_log(1,15,13) = -0.06999;
	pair_log(1,15,14) = -0.25307;
	pair_log(1,15,15) = 0.23245;
	pair_log(1,15,16) = -0.02548;
	pair_log(1,15,17) = -0.05313;
	pair_log(1,15,18) = 0.33279;
	pair_log(1,15,19) = -0.33116;
	pair_log(1,15,20) = -0.14488;
	pair_log(1,15,21) = 0.00000;
	pair_log(1,15,22) = 0.00000;
	pair_log(1,15,23) = 0.00000;
	pair_log(1,15,24) = 0.00000;
	pair_log(1,15,25) = 0.00000;
	pair_log(1,15,26) = 0.00000;
	pair_log(1,15,27) = 0.00000;
	pair_log(1,15,28) = 0.00000;
}

void
SecondaryStructureEval::pair_log_initializer_section1b(
	FArray3D_float & pair_log
)
{
	// section 1b
	pair_log(1,16, 1) = 0.10475;
	pair_log(1,16, 2) = 0.11283;
	pair_log(1,16, 3) = -0.50285;
	pair_log(1,16, 4) = -0.35312;
	pair_log(1,16, 5) = 0.28005;
	pair_log(1,16, 6) = -0.27119;
	pair_log(1,16, 7) = -0.11302;
	pair_log(1,16, 8) = 0.31299;
	pair_log(1,16, 9) = 0.02715;
	pair_log(1,16,10) = 0.27135;
	pair_log(1,16,11) = 0.19696;
	pair_log(1,16,12) = -0.28825;
	pair_log(1,16,13) = -0.10791;
	pair_log(1,16,14) = -0.06140;
	pair_log(1,16,15) = -0.02548;
	pair_log(1,16,16) = -0.20955;
	pair_log(1,16,17) = -0.32381;
	pair_log(1,16,18) = 0.15369;
	pair_log(1,16,19) = 0.36554;
	pair_log(1,16,20) = 0.31614;
	pair_log(1,16,21) = 0.00000;
	pair_log(1,16,22) = 0.00000;
	pair_log(1,16,23) = 0.00000;
	pair_log(1,16,24) = 0.00000;
	pair_log(1,16,25) = 0.00000;
	pair_log(1,16,26) = 0.00000;
	pair_log(1,16,27) = 0.00000;
	pair_log(1,16,28) = 0.00000;
	pair_log(1,17, 1) = -0.04679;
	pair_log(1,17, 2) = 0.27539;
	pair_log(1,17, 3) = -0.36579;
	pair_log(1,17, 4) = -0.29144;
	pair_log(1,17, 5) = 0.37472;
	pair_log(1,17, 6) = -0.24551;
	pair_log(1,17, 7) = -0.25624;
	pair_log(1,17, 8) = 0.28667;
	pair_log(1,17, 9) = -0.13219;
	pair_log(1,17,10) = 0.31011;
	pair_log(1,17,11) = 0.30090;
	pair_log(1,17,12) = -0.28531;
	pair_log(1,17,13) = 0.02844;
	pair_log(1,17,14) = -0.23014;
	pair_log(1,17,15) = -0.05313;
	pair_log(1,17,16) = -0.32381;
	pair_log(1,17,17) = -0.19546;
	pair_log(1,17,18) = 0.14755;
	pair_log(1,17,19) = 0.33614;
	pair_log(1,17,20) = 0.36576;
	pair_log(1,17,21) = 0.00000;
	pair_log(1,17,22) = 0.00000;
	pair_log(1,17,23) = 0.00000;
	pair_log(1,17,24) = 0.00000;
	pair_log(1,17,25) = 0.00000;
	pair_log(1,17,26) = 0.00000;
	pair_log(1,17,27) = 0.00000;
	pair_log(1,17,28) = 0.00000;
	pair_log(1,18, 1) = -0.27134;
	pair_log(1,18, 2) = 0.09778;
	pair_log(1,18, 3) = 0.44658;
	pair_log(1,18, 4) = 0.22764;
	pair_log(1,18, 5) = -0.08226;
	pair_log(1,18, 6) = 0.23105;
	pair_log(1,18, 7) = 0.15787;
	pair_log(1,18, 8) = -0.19634;
	pair_log(1,18, 9) = 0.23784;
	pair_log(1,18,10) = -0.10641;
	pair_log(1,18,11) = 0.00637;
	pair_log(1,18,12) = 0.47451;
	pair_log(1,18,13) = 0.39761;
	pair_log(1,18,14) = 0.08581;
	pair_log(1,18,15) = 0.33279;
	pair_log(1,18,16) = 0.15369;
	pair_log(1,18,17) = 0.14755;
	pair_log(1,18,18) = -0.29733;
	pair_log(1,18,19) = 0.12174;
	pair_log(1,18,20) = 0.02059;
	pair_log(1,18,21) = 0.00000;
	pair_log(1,18,22) = 0.00000;
	pair_log(1,18,23) = 0.00000;
	pair_log(1,18,24) = 0.00000;
	pair_log(1,18,25) = 0.00000;
	pair_log(1,18,26) = 0.00000;
	pair_log(1,18,27) = 0.00000;
	pair_log(1,18,28) = 0.00000;
	pair_log(1,19, 1) = 0.20001;
	pair_log(1,19, 2) = 0.09401;
	pair_log(1,19, 3) = 0.50378;
	pair_log(1,19, 4) = 0.16071;
	pair_log(1,19, 5) = -0.13588;
	pair_log(1,19, 6) = 0.24344;
	pair_log(1,19, 7) = -0.17229;
	pair_log(1,19, 8) = -0.15983;
	pair_log(1,19, 9) = -0.24586;
	pair_log(1,19,10) = -0.16843;
	pair_log(1,19,11) = -0.09998;
	pair_log(1,19,12) = 0.28602;
	pair_log(1,19,13) = -0.55908;
	pair_log(1,19,14) = -0.04570;
	pair_log(1,19,15) = -0.33116;
	pair_log(1,19,16) = 0.36554;
	pair_log(1,19,17) = 0.33614;
	pair_log(1,19,18) = 0.12174;
	pair_log(1,19,19) = 0.05462;
	pair_log(1,19,20) = -0.03280;
	pair_log(1,19,21) = 0.00000;
	pair_log(1,19,22) = 0.00000;
	pair_log(1,19,23) = 0.00000;
	pair_log(1,19,24) = 0.00000;
	pair_log(1,19,25) = 0.00000;
	pair_log(1,19,26) = 0.00000;
	pair_log(1,19,27) = 0.00000;
	pair_log(1,19,28) = 0.00000;
	pair_log(1,20, 1) = 0.12835;
	pair_log(1,20, 2) = 0.16464;
	pair_log(1,20, 3) = 0.46473;
	pair_log(1,20, 4) = 0.18883;
	pair_log(1,20, 5) = -0.01526;
	pair_log(1,20, 6) = 0.10640;
	pair_log(1,20, 7) = 0.02691;
	pair_log(1,20, 8) = -0.16964;
	pair_log(1,20, 9) = -0.16896;
	pair_log(1,20,10) = -0.20609;
	pair_log(1,20,11) = -0.22924;
	pair_log(1,20,12) = 0.12638;
	pair_log(1,20,13) = -0.41613;
	pair_log(1,20,14) = -0.03777;
	pair_log(1,20,15) = -0.14488;
	pair_log(1,20,16) = 0.31614;
	pair_log(1,20,17) = 0.36576;
	pair_log(1,20,18) = 0.02059;
	pair_log(1,20,19) = -0.03280;
	pair_log(1,20,20) = -0.01669;
	pair_log(1,20,21) = 0.00000;
	pair_log(1,20,22) = 0.00000;
	pair_log(1,20,23) = 0.00000;
	pair_log(1,20,24) = 0.00000;
	pair_log(1,20,25) = 0.00000;
	pair_log(1,20,26) = 0.00000;
	pair_log(1,20,27) = 0.00000;
	pair_log(1,20,28) = 0.00000;
	pair_log(1,21, 1) = 0.00000;
	pair_log(1,21, 2) = 0.00000;
	pair_log(1,21, 3) = 0.00000;
	pair_log(1,21, 4) = 0.00000;
	pair_log(1,21, 5) = 0.00000;
	pair_log(1,21, 6) = 0.00000;
	pair_log(1,21, 7) = 0.00000;
	pair_log(1,21, 8) = 0.00000;
	pair_log(1,21, 9) = 0.00000;
	pair_log(1,21,10) = 0.00000;
	pair_log(1,21,11) = 0.00000;
	pair_log(1,21,12) = 0.00000;
	pair_log(1,21,13) = 0.00000;
	pair_log(1,21,14) = 0.00000;
	pair_log(1,21,15) = 0.00000;
	pair_log(1,21,16) = 0.00000;
	pair_log(1,21,17) = 0.00000;
	pair_log(1,21,18) = 0.00000;
	pair_log(1,21,19) = 0.00000;
	pair_log(1,21,20) = 0.00000;
	pair_log(1,21,21) = 0.00000;
	pair_log(1,21,22) = 0.00000;
	pair_log(1,21,23) = 0.00000;
	pair_log(1,21,24) = 0.00000;
	pair_log(1,21,25) = 0.00000;
	pair_log(1,21,26) = 0.00000;
	pair_log(1,21,27) = 0.00000;
	pair_log(1,21,28) = 0.00000;
	pair_log(1,22, 1) = 0.00000;
	pair_log(1,22, 2) = 0.00000;
	pair_log(1,22, 3) = 0.00000;
	pair_log(1,22, 4) = 0.00000;
	pair_log(1,22, 5) = 0.00000;
	pair_log(1,22, 6) = 0.00000;
	pair_log(1,22, 7) = 0.00000;
	pair_log(1,22, 8) = 0.00000;
	pair_log(1,22, 9) = 0.00000;
	pair_log(1,22,10) = 0.00000;
	pair_log(1,22,11) = 0.00000;
	pair_log(1,22,12) = 0.00000;
	pair_log(1,22,13) = 0.00000;
	pair_log(1,22,14) = 0.00000;
	pair_log(1,22,15) = 0.00000;
	pair_log(1,22,16) = 0.00000;
	pair_log(1,22,17) = 0.00000;
	pair_log(1,22,18) = 0.00000;
	pair_log(1,22,19) = 0.00000;
	pair_log(1,22,20) = 0.00000;
	pair_log(1,22,21) = 0.00000;
	pair_log(1,22,22) = 0.00000;
	pair_log(1,22,23) = 0.00000;
	pair_log(1,22,24) = 0.00000;
	pair_log(1,22,25) = 0.00000;
	pair_log(1,22,26) = 0.00000;
	pair_log(1,22,27) = 0.00000;
	pair_log(1,22,28) = 0.00000;
	pair_log(1,23, 1) = 0.00000;
	pair_log(1,23, 2) = 0.00000;
	pair_log(1,23, 3) = 0.00000;
	pair_log(1,23, 4) = 0.00000;
	pair_log(1,23, 5) = 0.00000;
	pair_log(1,23, 6) = 0.00000;
	pair_log(1,23, 7) = 0.00000;
	pair_log(1,23, 8) = 0.00000;
	pair_log(1,23, 9) = 0.00000;
	pair_log(1,23,10) = 0.00000;
	pair_log(1,23,11) = 0.00000;
	pair_log(1,23,12) = 0.00000;
	pair_log(1,23,13) = 0.00000;
	pair_log(1,23,14) = 0.00000;
	pair_log(1,23,15) = 0.00000;
	pair_log(1,23,16) = 0.00000;
	pair_log(1,23,17) = 0.00000;
	pair_log(1,23,18) = 0.00000;
	pair_log(1,23,19) = 0.00000;
	pair_log(1,23,20) = 0.00000;
	pair_log(1,23,21) = 0.00000;
	pair_log(1,23,22) = 0.00000;
	pair_log(1,23,23) = 0.00000;
	pair_log(1,23,24) = 0.00000;
	pair_log(1,23,25) = 0.00000;
	pair_log(1,23,26) = 0.00000;
	pair_log(1,23,27) = 0.00000;
	pair_log(1,23,28) = 0.00000;
	pair_log(1,24, 1) = 0.00000;
	pair_log(1,24, 2) = 0.00000;
	pair_log(1,24, 3) = 0.00000;
	pair_log(1,24, 4) = 0.00000;
	pair_log(1,24, 5) = 0.00000;
	pair_log(1,24, 6) = 0.00000;
	pair_log(1,24, 7) = 0.00000;
	pair_log(1,24, 8) = 0.00000;
	pair_log(1,24, 9) = 0.00000;
	pair_log(1,24,10) = 0.00000;
	pair_log(1,24,11) = 0.00000;
	pair_log(1,24,12) = 0.00000;
	pair_log(1,24,13) = 0.00000;
	pair_log(1,24,14) = 0.00000;
	pair_log(1,24,15) = 0.00000;
	pair_log(1,24,16) = 0.00000;
	pair_log(1,24,17) = 0.00000;
	pair_log(1,24,18) = 0.00000;
	pair_log(1,24,19) = 0.00000;
	pair_log(1,24,20) = 0.00000;
	pair_log(1,24,21) = 0.00000;
	pair_log(1,24,22) = 0.00000;
	pair_log(1,24,23) = 0.00000;
	pair_log(1,24,24) = 0.00000;
	pair_log(1,24,25) = 0.00000;
	pair_log(1,24,26) = 0.00000;
	pair_log(1,24,27) = 0.00000;
	pair_log(1,24,28) = 0.00000;
	pair_log(1,25, 1) = 0.00000;
	pair_log(1,25, 2) = 0.00000;
	pair_log(1,25, 3) = 0.00000;
	pair_log(1,25, 4) = 0.00000;
	pair_log(1,25, 5) = 0.00000;
	pair_log(1,25, 6) = 0.00000;
	pair_log(1,25, 7) = 0.00000;
	pair_log(1,25, 8) = 0.00000;
	pair_log(1,25, 9) = 0.00000;
	pair_log(1,25,10) = 0.00000;
	pair_log(1,25,11) = 0.00000;
	pair_log(1,25,12) = 0.00000;
	pair_log(1,25,13) = 0.00000;
	pair_log(1,25,14) = 0.00000;
	pair_log(1,25,15) = 0.00000;
	pair_log(1,25,16) = 0.00000;
	pair_log(1,25,17) = 0.00000;
	pair_log(1,25,18) = 0.00000;
	pair_log(1,25,19) = 0.00000;
	pair_log(1,25,20) = 0.00000;
	pair_log(1,25,21) = 0.00000;
	pair_log(1,25,22) = 0.00000;
	pair_log(1,25,23) = 0.00000;
	pair_log(1,25,24) = 0.00000;
	pair_log(1,25,25) = 0.00000;
	pair_log(1,25,26) = 0.00000;
	pair_log(1,25,27) = 0.00000;
	pair_log(1,25,28) = 0.00000;
	pair_log(1,26, 1) = 0.00000;
	pair_log(1,26, 2) = 0.00000;
	pair_log(1,26, 3) = 0.00000;
	pair_log(1,26, 4) = 0.00000;
	pair_log(1,26, 5) = 0.00000;
	pair_log(1,26, 6) = 0.00000;
	pair_log(1,26, 7) = 0.00000;
	pair_log(1,26, 8) = 0.00000;
	pair_log(1,26, 9) = 0.00000;
	pair_log(1,26,10) = 0.00000;
	pair_log(1,26,11) = 0.00000;
	pair_log(1,26,12) = 0.00000;
	pair_log(1,26,13) = 0.00000;
	pair_log(1,26,14) = 0.00000;
	pair_log(1,26,15) = 0.00000;
	pair_log(1,26,16) = 0.00000;
	pair_log(1,26,17) = 0.00000;
	pair_log(1,26,18) = 0.00000;
	pair_log(1,26,19) = 0.00000;
	pair_log(1,26,20) = 0.00000;
	pair_log(1,26,21) = 0.00000;
	pair_log(1,26,22) = 0.00000;
	pair_log(1,26,23) = 0.00000;
	pair_log(1,26,24) = 0.00000;
	pair_log(1,26,25) = 0.00000;
	pair_log(1,26,26) = 0.00000;
	pair_log(1,26,27) = 0.00000;
	pair_log(1,26,28) = 0.00000;
	pair_log(1,27, 1) = 0.00000;
	pair_log(1,27, 2) = 0.00000;
	pair_log(1,27, 3) = 0.00000;
	pair_log(1,27, 4) = 0.00000;
	pair_log(1,27, 5) = 0.00000;
	pair_log(1,27, 6) = 0.00000;
	pair_log(1,27, 7) = 0.00000;
	pair_log(1,27, 8) = 0.00000;
	pair_log(1,27, 9) = 0.00000;
	pair_log(1,27,10) = 0.00000;
	pair_log(1,27,11) = 0.00000;
	pair_log(1,27,12) = 0.00000;
	pair_log(1,27,13) = 0.00000;
	pair_log(1,27,14) = 0.00000;
	pair_log(1,27,15) = 0.00000;
	pair_log(1,27,16) = 0.00000;
	pair_log(1,27,17) = 0.00000;
	pair_log(1,27,18) = 0.00000;
	pair_log(1,27,19) = 0.00000;
	pair_log(1,27,20) = 0.00000;
	pair_log(1,27,21) = 0.00000;
	pair_log(1,27,22) = 0.00000;
	pair_log(1,27,23) = 0.00000;
	pair_log(1,27,24) = 0.00000;
	pair_log(1,27,25) = 0.00000;
	pair_log(1,27,26) = 0.00000;
	pair_log(1,27,27) = 0.00000;
	pair_log(1,27,28) = 0.00000;
	pair_log(1,28, 1) = 0.00000;
	pair_log(1,28, 2) = 0.00000;
	pair_log(1,28, 3) = 0.00000;
	pair_log(1,28, 4) = 0.00000;
	pair_log(1,28, 5) = 0.00000;
	pair_log(1,28, 6) = 0.00000;
	pair_log(1,28, 7) = 0.00000;
	pair_log(1,28, 8) = 0.00000;
	pair_log(1,28, 9) = 0.00000;
	pair_log(1,28,10) = 0.00000;
	pair_log(1,28,11) = 0.00000;
	pair_log(1,28,12) = 0.00000;
	pair_log(1,28,13) = 0.00000;
	pair_log(1,28,14) = 0.00000;
	pair_log(1,28,15) = 0.00000;
	pair_log(1,28,16) = 0.00000;
	pair_log(1,28,17) = 0.00000;
	pair_log(1,28,18) = 0.00000;
	pair_log(1,28,19) = 0.00000;
	pair_log(1,28,20) = 0.00000;
	pair_log(1,28,21) = 0.00000;
	pair_log(1,28,22) = 0.00000;
	pair_log(1,28,23) = 0.00000;
	pair_log(1,28,24) = 0.00000;
	pair_log(1,28,25) = 0.00000;
	pair_log(1,28,26) = 0.00000;
	pair_log(1,28,27) = 0.00000;
	pair_log(1,28,28) = 0.00000;
	//KMa phospho_ser
	for (int i=1;i<=29;i++) {
		pair_log(1,29,i) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(1,i,29) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(1,21,i) = pair_log(1,16,i);
	}
	for (int i=1;i<=29;i++) {
		pair_log(1,i,21) = pair_log(1,i,16);
	}
}

void
SecondaryStructureEval::pair_log_initializer_section2a(
	FArray3D_float & pair_log
)
{
	// section 2a
	pair_log(2, 1, 1) = -0.02226;
	pair_log(2, 1, 2) = -0.04411;
	pair_log(2, 1, 3) = 0.13313;
	pair_log(2, 1, 4) = 0.15220;
	pair_log(2, 1, 5) = -0.14905;
	pair_log(2, 1, 6) = -0.05662;
	pair_log(2, 1, 7) = 0.02143;
	pair_log(2, 1, 8) = -0.03193;
	pair_log(2, 1, 9) = 0.14682;
	pair_log(2, 1,10) = 0.01865;
	pair_log(2, 1,11) = -0.08634;
	pair_log(2, 1,12) = 0.02956;
	pair_log(2, 1,13) = 0.08000;
	pair_log(2, 1,14) = 0.08891;
	pair_log(2, 1,15) = 0.09517;
	pair_log(2, 1,16) = 0.00350;
	pair_log(2, 1,17) = 0.00526;
	pair_log(2, 1,18) = -0.02354;
	pair_log(2, 1,19) = -0.11261;
	pair_log(2, 1,20) = -0.05314;
	pair_log(2, 1,21) = 0.00000;
	pair_log(2, 1,22) = 0.00000;
	pair_log(2, 1,23) = 0.00000;
	pair_log(2, 1,24) = 0.00000;
	pair_log(2, 1,25) = 0.00000;
	pair_log(2, 1,26) = 0.00000;
	pair_log(2, 1,27) = 0.00000;
	pair_log(2, 1,28) = 0.00000;
	pair_log(2, 2, 1) = -0.04411;
	pair_log(2, 2, 2) = -0.35421;
	pair_log(2, 2, 3) = 0.16627;
	pair_log(2, 2, 4) = 0.46641;
	pair_log(2, 2, 5) = -0.08872;
	pair_log(2, 2, 6) = -0.15618;
	pair_log(2, 2, 7) = -0.08947;
	pair_log(2, 2, 8) = 0.08276;
	pair_log(2, 2, 9) = 0.30183;
	pair_log(2, 2,10) = 0.03309;
	pair_log(2, 2,11) = -0.03164;
	pair_log(2, 2,12) = 0.05061;
	pair_log(2, 2,13) = -0.09590;
	pair_log(2, 2,14) = 0.06070;
	pair_log(2, 2,15) = 0.21192;
	pair_log(2, 2,16) = -0.06203;
	pair_log(2, 2,17) = -0.02885;
	pair_log(2, 2,18) = -0.01849;
	pair_log(2, 2,19) = -0.17325;
	pair_log(2, 2,20) = 0.00201;
	pair_log(2, 2,21) = 0.00000;
	pair_log(2, 2,22) = 0.00000;
	pair_log(2, 2,23) = 0.00000;
	pair_log(2, 2,24) = 0.00000;
	pair_log(2, 2,25) = 0.00000;
	pair_log(2, 2,26) = 0.00000;
	pair_log(2, 2,27) = 0.00000;
	pair_log(2, 2,28) = 0.00000;
	pair_log(2, 3, 1) = 0.13313;
	pair_log(2, 3, 2) = 0.16627;
	pair_log(2, 3, 3) = -0.21301;
	pair_log(2, 3, 4) = -0.05421;
	pair_log(2, 3, 5) = 0.33380;
	pair_log(2, 3, 6) = -0.13454;
	pair_log(2, 3, 7) = -0.51082;
	pair_log(2, 3, 8) = 0.64069;
	pair_log(2, 3, 9) = -0.83773;
	pair_log(2, 3,10) = 0.67107;
	pair_log(2, 3,11) = 0.34501;
	pair_log(2, 3,12) = -0.36779;
	pair_log(2, 3,13) = -0.10799;
	pair_log(2, 3,14) = -0.22107;
	pair_log(2, 3,15) = -0.75762;
	pair_log(2, 3,16) = -0.17464;
	pair_log(2, 3,17) = -0.04369;
	pair_log(2, 3,18) = 0.49624;
	pair_log(2, 3,19) = -0.00891;
	pair_log(2, 3,20) = -0.12981;
	pair_log(2, 3,21) = 0.00000;
	pair_log(2, 3,22) = 0.00000;
	pair_log(2, 3,23) = 0.00000;
	pair_log(2, 3,24) = 0.00000;
	pair_log(2, 3,25) = 0.00000;
	pair_log(2, 3,26) = 0.00000;
	pair_log(2, 3,27) = 0.00000;
	pair_log(2, 3,28) = 0.00000;
	pair_log(2, 4, 1) = 0.15220;
	pair_log(2, 4, 2) = 0.46641;
	pair_log(2, 4, 3) = -0.05421;
	pair_log(2, 4, 4) = -0.06802;
	pair_log(2, 4, 5) = 0.29542;
	pair_log(2, 4, 6) = -0.01456;
	pair_log(2, 4, 7) = -0.38576;
	pair_log(2, 4, 8) = 0.51696;
	pair_log(2, 4, 9) = -0.84899;
	pair_log(2, 4,10) = 0.47591;
	pair_log(2, 4,11) = 0.21535;
	pair_log(2, 4,12) = -0.29050;
	pair_log(2, 4,13) = -0.08512;
	pair_log(2, 4,14) = -0.22637;
	pair_log(2, 4,15) = -0.73107;
	pair_log(2, 4,16) = -0.21143;
	pair_log(2, 4,17) = -0.12650;
	pair_log(2, 4,18) = 0.41691;
	pair_log(2, 4,19) = -0.09799;
	pair_log(2, 4,20) = -0.03815;
	pair_log(2, 4,21) = 0.00000;
	pair_log(2, 4,22) = 0.00000;
	pair_log(2, 4,23) = 0.00000;
	pair_log(2, 4,24) = 0.00000;
	pair_log(2, 4,25) = 0.00000;
	pair_log(2, 4,26) = 0.00000;
	pair_log(2, 4,27) = 0.00000;
	pair_log(2, 4,28) = 0.00000;
	pair_log(2, 5, 1) = -0.14905;
	pair_log(2, 5, 2) = -0.08872;
	pair_log(2, 5, 3) = 0.33380;
	pair_log(2, 5, 4) = 0.29542;
	pair_log(2, 5, 5) = -0.19925;
	pair_log(2, 5, 6) = 0.05265;
	pair_log(2, 5, 7) = 0.09431;
	pair_log(2, 5, 8) = -0.09249;
	pair_log(2, 5, 9) = 0.31158;
	pair_log(2, 5,10) = -0.12690;
	pair_log(2, 5,11) = -0.11997;
	pair_log(2, 5,12) = 0.27039;
	pair_log(2, 5,13) = 0.16121;
	pair_log(2, 5,14) = 0.11689;
	pair_log(2, 5,15) = 0.27997;
	pair_log(2, 5,16) = 0.09814;
	pair_log(2, 5,17) = 0.14806;
	pair_log(2, 5,18) = -0.12269;
	pair_log(2, 5,19) = -0.05715;
	pair_log(2, 5,20) = -0.03704;
	pair_log(2, 5,21) = 0.00000;
	pair_log(2, 5,22) = 0.00000;
	pair_log(2, 5,23) = 0.00000;
	pair_log(2, 5,24) = 0.00000;
	pair_log(2, 5,25) = 0.00000;
	pair_log(2, 5,26) = 0.00000;
	pair_log(2, 5,27) = 0.00000;
	pair_log(2, 5,28) = 0.00000;
	pair_log(2, 6, 1) = -0.05662;
	pair_log(2, 6, 2) = -0.15618;
	pair_log(2, 6, 3) = -0.13454;
	pair_log(2, 6, 4) = -0.01456;
	pair_log(2, 6, 5) = 0.05265;
	pair_log(2, 6, 6) = -0.14051;
	pair_log(2, 6, 7) = -0.17065;
	pair_log(2, 6, 8) = 0.27635;
	pair_log(2, 6, 9) = -0.13111;
	pair_log(2, 6,10) = 0.33203;
	pair_log(2, 6,11) = 0.15193;
	pair_log(2, 6,12) = -0.20722;
	pair_log(2, 6,13) = -0.05692;
	pair_log(2, 6,14) = -0.16506;
	pair_log(2, 6,15) = -0.11493;
	pair_log(2, 6,16) = -0.20971;
	pair_log(2, 6,17) = -0.08635;
	pair_log(2, 6,18) = 0.14201;
	pair_log(2, 6,19) = -0.09435;
	pair_log(2, 6,20) = -0.00928;
	pair_log(2, 6,21) = 0.00000;
	pair_log(2, 6,22) = 0.00000;
	pair_log(2, 6,23) = 0.00000;
	pair_log(2, 6,24) = 0.00000;
	pair_log(2, 6,25) = 0.00000;
	pair_log(2, 6,26) = 0.00000;
	pair_log(2, 6,27) = 0.00000;
	pair_log(2, 6,28) = 0.00000;
	pair_log(2, 7, 1) = 0.02143;
	pair_log(2, 7, 2) = -0.08947;
	pair_log(2, 7, 3) = -0.51082;
	pair_log(2, 7, 4) = -0.38576;
	pair_log(2, 7, 5) = 0.09431;
	pair_log(2, 7, 6) = -0.17065;
	pair_log(2, 7, 7) = -0.55055;
	pair_log(2, 7, 8) = 0.51369;
	pair_log(2, 7, 9) = 0.15045;
	pair_log(2, 7,10) = 0.31403;
	pair_log(2, 7,11) = 0.04739;
	pair_log(2, 7,12) = -0.22095;
	pair_log(2, 7,13) = -0.05316;
	pair_log(2, 7,14) = -0.13022;
	pair_log(2, 7,15) = 0.02841;
	pair_log(2, 7,16) = -0.19378;
	pair_log(2, 7,17) = -0.08904;
	pair_log(2, 7,18) = 0.34177;
	pair_log(2, 7,19) = -0.02796;
	pair_log(2, 7,20) = 0.00739;
	pair_log(2, 7,21) = 0.00000;
	pair_log(2, 7,22) = 0.00000;
	pair_log(2, 7,23) = 0.00000;
	pair_log(2, 7,24) = 0.00000;
	pair_log(2, 7,25) = 0.00000;
	pair_log(2, 7,26) = 0.00000;
	pair_log(2, 7,27) = 0.00000;
	pair_log(2, 7,28) = 0.00000;
	pair_log(2, 8, 1) = -0.03193;
	pair_log(2, 8, 2) = 0.08276;
	pair_log(2, 8, 3) = 0.64069;
	pair_log(2, 8, 4) = 0.51696;
	pair_log(2, 8, 5) = -0.09249;
	pair_log(2, 8, 6) = 0.27635;
	pair_log(2, 8, 7) = 0.51369;
	pair_log(2, 8, 8) = -0.38531;
	pair_log(2, 8, 9) = 0.37502;
	pair_log(2, 8,10) = -0.35708;
	pair_log(2, 8,11) = -0.15359;
	pair_log(2, 8,12) = 0.49079;
	pair_log(2, 8,13) = 0.19667;
	pair_log(2, 8,14) = 0.31459;
	pair_log(2, 8,15) = 0.34774;
	pair_log(2, 8,16) = 0.35197;
	pair_log(2, 8,17) = 0.11428;
	pair_log(2, 8,18) = -0.27482;
	pair_log(2, 8,19) = 0.13403;
	pair_log(2, 8,20) = 0.08237;
	pair_log(2, 8,21) = 0.00000;
	pair_log(2, 8,22) = 0.00000;
	pair_log(2, 8,23) = 0.00000;
	pair_log(2, 8,24) = 0.00000;
	pair_log(2, 8,25) = 0.00000;
	pair_log(2, 8,26) = 0.00000;
	pair_log(2, 8,27) = 0.00000;
	pair_log(2, 8,28) = 0.00000;
	pair_log(2, 9, 1) = 0.14682;
	pair_log(2, 9, 2) = 0.30183;
	pair_log(2, 9, 3) = -0.83773;
	pair_log(2, 9, 4) = -0.84899;
	pair_log(2, 9, 5) = 0.31158;
	pair_log(2, 9, 6) = -0.13111;
	pair_log(2, 9, 7) = 0.15045;
	pair_log(2, 9, 8) = 0.37502;
	pair_log(2, 9, 9) = 0.06908;
	pair_log(2, 9,10) = 0.42522;
	pair_log(2, 9,11) = 0.32782;
	pair_log(2, 9,12) = -0.28908;
	pair_log(2, 9,13) = -0.01420;
	pair_log(2, 9,14) = -0.21644;
	pair_log(2, 9,15) = 0.05722;
	pair_log(2, 9,16) = -0.13103;
	pair_log(2, 9,17) = -0.12997;
	pair_log(2, 9,18) = 0.33110;
	pair_log(2, 9,19) = 0.14321;
	pair_log(2, 9,20) = -0.03110;
	pair_log(2, 9,21) = 0.00000;
	pair_log(2, 9,22) = 0.00000;
	pair_log(2, 9,23) = 0.00000;
	pair_log(2, 9,24) = 0.00000;
	pair_log(2, 9,25) = 0.00000;
	pair_log(2, 9,26) = 0.00000;
	pair_log(2, 9,27) = 0.00000;
	pair_log(2, 9,28) = 0.00000;
	pair_log(2,10, 1) = 0.01865;
	pair_log(2,10, 2) = 0.03309;
	pair_log(2,10, 3) = 0.67107;
	pair_log(2,10, 4) = 0.47591;
	pair_log(2,10, 5) = -0.12690;
	pair_log(2,10, 6) = 0.33203;
	pair_log(2,10, 7) = 0.31403;
	pair_log(2,10, 8) = -0.35708;
	pair_log(2,10, 9) = 0.42522;
	pair_log(2,10,10) = -0.36593;
	pair_log(2,10,11) = -0.14099;
	pair_log(2,10,12) = 0.49954;
	pair_log(2,10,13) = 0.20002;
	pair_log(2,10,14) = 0.32215;
	pair_log(2,10,15) = 0.31116;
	pair_log(2,10,16) = 0.34207;
	pair_log(2,10,17) = 0.15723;
	pair_log(2,10,18) = -0.27941;
	pair_log(2,10,19) = 0.13555;
	pair_log(2,10,20) = 0.03594;
	pair_log(2,10,21) = 0.00000;
	pair_log(2,10,22) = 0.00000;
	pair_log(2,10,23) = 0.00000;
	pair_log(2,10,24) = 0.00000;
	pair_log(2,10,25) = 0.00000;
	pair_log(2,10,26) = 0.00000;
	pair_log(2,10,27) = 0.00000;
	pair_log(2,10,28) = 0.00000;
	pair_log(2,11, 1) = -0.08634;
	pair_log(2,11, 2) = -0.03164;
	pair_log(2,11, 3) = 0.34501;
	pair_log(2,11, 4) = 0.21535;
	pair_log(2,11, 5) = -0.11997;
	pair_log(2,11, 6) = 0.15193;
	pair_log(2,11, 7) = 0.04739;
	pair_log(2,11, 8) = -0.15359;
	pair_log(2,11, 9) = 0.32782;
	pair_log(2,11,10) = -0.14099;
	pair_log(2,11,11) = -0.16514;
	pair_log(2,11,12) = 0.13447;
	pair_log(2,11,13) = 0.08185;
	pair_log(2,11,14) = 0.13752;
	pair_log(2,11,15) = 0.23747;
	pair_log(2,11,16) = 0.10816;
	pair_log(2,11,17) = 0.10441;
	pair_log(2,11,18) = -0.07385;
	pair_log(2,11,19) = 0.00104;
	pair_log(2,11,20) = -0.01117;
	pair_log(2,11,21) = 0.00000;
	pair_log(2,11,22) = 0.00000;
	pair_log(2,11,23) = 0.00000;
	pair_log(2,11,24) = 0.00000;
	pair_log(2,11,25) = 0.00000;
	pair_log(2,11,26) = 0.00000;
	pair_log(2,11,27) = 0.00000;
	pair_log(2,11,28) = 0.00000;
	pair_log(2,12, 1) = 0.02956;
	pair_log(2,12, 2) = 0.05061;
	pair_log(2,12, 3) = -0.36779;
	pair_log(2,12, 4) = -0.29050;
	pair_log(2,12, 5) = 0.27039;
	pair_log(2,12, 6) = -0.20722;
	pair_log(2,12, 7) = -0.22095;
	pair_log(2,12, 8) = 0.49079;
	pair_log(2,12, 9) = -0.28908;
	pair_log(2,12,10) = 0.49954;
	pair_log(2,12,11) = 0.13447;
	pair_log(2,12,12) = -0.43720;
	pair_log(2,12,13) = -0.22235;
	pair_log(2,12,14) = -0.29003;
	pair_log(2,12,15) = -0.20477;
	pair_log(2,12,16) = -0.23763;
	pair_log(2,12,17) = -0.15047;
	pair_log(2,12,18) = 0.40194;
	pair_log(2,12,19) = 0.03693;
	pair_log(2,12,20) = 0.05800;
	pair_log(2,12,21) = 0.00000;
	pair_log(2,12,22) = 0.00000;
	pair_log(2,12,23) = 0.00000;
	pair_log(2,12,24) = 0.00000;
	pair_log(2,12,25) = 0.00000;
	pair_log(2,12,26) = 0.00000;
	pair_log(2,12,27) = 0.00000;
	pair_log(2,12,28) = 0.00000;
	pair_log(2,13, 1) = 0.08000;
	pair_log(2,13, 2) = -0.09590;
	pair_log(2,13, 3) = -0.10799;
	pair_log(2,13, 4) = -0.08512;
	pair_log(2,13, 5) = 0.16121;
	pair_log(2,13, 6) = -0.05692;
	pair_log(2,13, 7) = -0.05316;
	pair_log(2,13, 8) = 0.19667;
	pair_log(2,13, 9) = -0.01420;
	pair_log(2,13,10) = 0.20002;
	pair_log(2,13,11) = 0.08185;
	pair_log(2,13,12) = -0.22235;
	pair_log(2,13,13) = -0.20538;
	pair_log(2,13,14) = -0.18800;
	pair_log(2,13,15) = -0.11752;
	pair_log(2,13,16) = -0.11199;
	pair_log(2,13,17) = -0.13377;
	pair_log(2,13,18) = 0.15123;
	pair_log(2,13,19) = -0.13591;
	pair_log(2,13,20) = -0.10661;
	pair_log(2,13,21) = 0.00000;
	pair_log(2,13,22) = 0.00000;
	pair_log(2,13,23) = 0.00000;
	pair_log(2,13,24) = 0.00000;
	pair_log(2,13,25) = 0.00000;
	pair_log(2,13,26) = 0.00000;
	pair_log(2,13,27) = 0.00000;
	pair_log(2,13,28) = 0.00000;
	pair_log(2,14, 1) = 0.08891;
	pair_log(2,14, 2) = 0.06070;
	pair_log(2,14, 3) = -0.22107;
	pair_log(2,14, 4) = -0.22637;
	pair_log(2,14, 5) = 0.11689;
	pair_log(2,14, 6) = -0.16506;
	pair_log(2,14, 7) = -0.13022;
	pair_log(2,14, 8) = 0.31459;
	pair_log(2,14, 9) = -0.21644;
	pair_log(2,14,10) = 0.32215;
	pair_log(2,14,11) = 0.13752;
	pair_log(2,14,12) = -0.29003;
	pair_log(2,14,13) = -0.18800;
	pair_log(2,14,14) = -0.25399;
	pair_log(2,14,15) = -0.10846;
	pair_log(2,14,16) = -0.19343;
	pair_log(2,14,17) = -0.16004;
	pair_log(2,14,18) = 0.28120;
	pair_log(2,14,19) = -0.03032;
	pair_log(2,14,20) = -0.02419;
	pair_log(2,14,21) = 0.00000;
	pair_log(2,14,22) = 0.00000;
	pair_log(2,14,23) = 0.00000;
	pair_log(2,14,24) = 0.00000;
	pair_log(2,14,25) = 0.00000;
	pair_log(2,14,26) = 0.00000;
	pair_log(2,14,27) = 0.00000;
	pair_log(2,14,28) = 0.00000;
	pair_log(2,15, 1) = 0.09517;
	pair_log(2,15, 2) = 0.21192;
	pair_log(2,15, 3) = -0.75762;
	pair_log(2,15, 4) = -0.73107;
	pair_log(2,15, 5) = 0.27997;
	pair_log(2,15, 6) = -0.11493;
	pair_log(2,15, 7) = 0.02841;
	pair_log(2,15, 8) = 0.34774;
	pair_log(2,15, 9) = 0.05722;
	pair_log(2,15,10) = 0.31116;
	pair_log(2,15,11) = 0.23747;
	pair_log(2,15,12) = -0.20477;
	pair_log(2,15,13) = -0.11752;
	pair_log(2,15,14) = -0.10846;
	pair_log(2,15,15) = 0.07551;
	pair_log(2,15,16) = -0.18824;
	pair_log(2,15,17) = -0.05800;
	pair_log(2,15,18) = 0.33249;
	pair_log(2,15,19) = 0.04019;
	pair_log(2,15,20) = 0.01551;
	pair_log(2,15,21) = 0.00000;
	pair_log(2,15,22) = 0.00000;
	pair_log(2,15,23) = 0.00000;
	pair_log(2,15,24) = 0.00000;
	pair_log(2,15,25) = 0.00000;
	pair_log(2,15,26) = 0.00000;
	pair_log(2,15,27) = 0.00000;
	pair_log(2,15,28) = 0.00000;
}

void
SecondaryStructureEval::pair_log_initializer_section2b(
	FArray3D_float & pair_log
)
{
	// section 2b
	pair_log(2,16, 1) = 0.00350;
	pair_log(2,16, 2) = -0.06203;
	pair_log(2,16, 3) = -0.17464;
	pair_log(2,16, 4) = -0.21143;
	pair_log(2,16, 5) = 0.09814;
	pair_log(2,16, 6) = -0.20971;
	pair_log(2,16, 7) = -0.19378;
	pair_log(2,16, 8) = 0.35197;
	pair_log(2,16, 9) = -0.13103;
	pair_log(2,16,10) = 0.34207;
	pair_log(2,16,11) = 0.10816;
	pair_log(2,16,12) = -0.23763;
	pair_log(2,16,13) = -0.11199;
	pair_log(2,16,14) = -0.19343;
	pair_log(2,16,15) = -0.18824;
	pair_log(2,16,16) = -0.18928;
	pair_log(2,16,17) = -0.07783;
	pair_log(2,16,18) = 0.25468;
	pair_log(2,16,19) = -0.03151;
	pair_log(2,16,20) = -0.02162;
	pair_log(2,16,21) = 0.00000;
	pair_log(2,16,22) = 0.00000;
	pair_log(2,16,23) = 0.00000;
	pair_log(2,16,24) = 0.00000;
	pair_log(2,16,25) = 0.00000;
	pair_log(2,16,26) = 0.00000;
	pair_log(2,16,27) = 0.00000;
	pair_log(2,16,28) = 0.00000;
	pair_log(2,17, 1) = 0.00526;
	pair_log(2,17, 2) = -0.02885;
	pair_log(2,17, 3) = -0.04369;
	pair_log(2,17, 4) = -0.12650;
	pair_log(2,17, 5) = 0.14806;
	pair_log(2,17, 6) = -0.08635;
	pair_log(2,17, 7) = -0.08904;
	pair_log(2,17, 8) = 0.11428;
	pair_log(2,17, 9) = -0.12997;
	pair_log(2,17,10) = 0.15723;
	pair_log(2,17,11) = 0.10441;
	pair_log(2,17,12) = -0.15047;
	pair_log(2,17,13) = -0.13377;
	pair_log(2,17,14) = -0.16004;
	pair_log(2,17,15) = -0.05800;
	pair_log(2,17,16) = -0.07783;
	pair_log(2,17,17) = -0.08219;
	pair_log(2,17,18) = 0.08308;
	pair_log(2,17,19) = 0.03798;
	pair_log(2,17,20) = 0.06792;
	pair_log(2,17,21) = 0.00000;
	pair_log(2,17,22) = 0.00000;
	pair_log(2,17,23) = 0.00000;
	pair_log(2,17,24) = 0.00000;
	pair_log(2,17,25) = 0.00000;
	pair_log(2,17,26) = 0.00000;
	pair_log(2,17,27) = 0.00000;
	pair_log(2,17,28) = 0.00000;
	pair_log(2,18, 1) = -0.02354;
	pair_log(2,18, 2) = -0.01849;
	pair_log(2,18, 3) = 0.49624;
	pair_log(2,18, 4) = 0.41691;
	pair_log(2,18, 5) = -0.12269;
	pair_log(2,18, 6) = 0.14201;
	pair_log(2,18, 7) = 0.34177;
	pair_log(2,18, 8) = -0.27482;
	pair_log(2,18, 9) = 0.33110;
	pair_log(2,18,10) = -0.27941;
	pair_log(2,18,11) = -0.07385;
	pair_log(2,18,12) = 0.40194;
	pair_log(2,18,13) = 0.15123;
	pair_log(2,18,14) = 0.28120;
	pair_log(2,18,15) = 0.33249;
	pair_log(2,18,16) = 0.25468;
	pair_log(2,18,17) = 0.08308;
	pair_log(2,18,18) = -0.25929;
	pair_log(2,18,19) = 0.07499;
	pair_log(2,18,20) = 0.04100;
	pair_log(2,18,21) = 0.00000;
	pair_log(2,18,22) = 0.00000;
	pair_log(2,18,23) = 0.00000;
	pair_log(2,18,24) = 0.00000;
	pair_log(2,18,25) = 0.00000;
	pair_log(2,18,26) = 0.00000;
	pair_log(2,18,27) = 0.00000;
	pair_log(2,18,28) = 0.00000;
	pair_log(2,19, 1) = -0.11261;
	pair_log(2,19, 2) = -0.17325;
	pair_log(2,19, 3) = -0.00891;
	pair_log(2,19, 4) = -0.09799;
	pair_log(2,19, 5) = -0.05715;
	pair_log(2,19, 6) = -0.09435;
	pair_log(2,19, 7) = -0.02796;
	pair_log(2,19, 8) = 0.13403;
	pair_log(2,19, 9) = 0.14321;
	pair_log(2,19,10) = 0.13555;
	pair_log(2,19,11) = 0.00104;
	pair_log(2,19,12) = 0.03693;
	pair_log(2,19,13) = -0.13591;
	pair_log(2,19,14) = -0.03032;
	pair_log(2,19,15) = 0.04019;
	pair_log(2,19,16) = -0.03151;
	pair_log(2,19,17) = 0.03798;
	pair_log(2,19,18) = 0.07499;
	pair_log(2,19,19) = -0.05100;
	pair_log(2,19,20) = -0.00889;
	pair_log(2,19,21) = 0.00000;
	pair_log(2,19,22) = 0.00000;
	pair_log(2,19,23) = 0.00000;
	pair_log(2,19,24) = 0.00000;
	pair_log(2,19,25) = 0.00000;
	pair_log(2,19,26) = 0.00000;
	pair_log(2,19,27) = 0.00000;
	pair_log(2,19,28) = 0.00000;
	pair_log(2,20, 1) = -0.05314;
	pair_log(2,20, 2) = 0.00201;
	pair_log(2,20, 3) = -0.12981;
	pair_log(2,20, 4) = -0.03815;
	pair_log(2,20, 5) = -0.03704;
	pair_log(2,20, 6) = -0.00928;
	pair_log(2,20, 7) = 0.00739;
	pair_log(2,20, 8) = 0.08237;
	pair_log(2,20, 9) = -0.03110;
	pair_log(2,20,10) = 0.03594;
	pair_log(2,20,11) = -0.01117;
	pair_log(2,20,12) = 0.05800;
	pair_log(2,20,13) = -0.10661;
	pair_log(2,20,14) = -0.02419;
	pair_log(2,20,15) = 0.01551;
	pair_log(2,20,16) = -0.02162;
	pair_log(2,20,17) = 0.06792;
	pair_log(2,20,18) = 0.04100;
	pair_log(2,20,19) = -0.00889;
	pair_log(2,20,20) = 0.00158;
	pair_log(2,20,21) = 0.00000;
	pair_log(2,20,22) = 0.00000;
	pair_log(2,20,23) = 0.00000;
	pair_log(2,20,24) = 0.00000;
	pair_log(2,20,25) = 0.00000;
	pair_log(2,20,26) = 0.00000;
	pair_log(2,20,27) = 0.00000;
	pair_log(2,20,28) = 0.00000;
	pair_log(2,21, 1) = 0.00000;
	pair_log(2,21, 2) = 0.00000;
	pair_log(2,21, 3) = 0.00000;
	pair_log(2,21, 4) = 0.00000;
	pair_log(2,21, 5) = 0.00000;
	pair_log(2,21, 6) = 0.00000;
	pair_log(2,21, 7) = 0.00000;
	pair_log(2,21, 8) = 0.00000;
	pair_log(2,21, 9) = 0.00000;
	pair_log(2,21,10) = 0.00000;
	pair_log(2,21,11) = 0.00000;
	pair_log(2,21,12) = 0.00000;
	pair_log(2,21,13) = 0.00000;
	pair_log(2,21,14) = 0.00000;
	pair_log(2,21,15) = 0.00000;
	pair_log(2,21,16) = 0.00000;
	pair_log(2,21,17) = 0.00000;
	pair_log(2,21,18) = 0.00000;
	pair_log(2,21,19) = 0.00000;
	pair_log(2,21,20) = 0.00000;
	pair_log(2,21,21) = 0.00000;
	pair_log(2,21,22) = 0.00000;
	pair_log(2,21,23) = 0.00000;
	pair_log(2,21,24) = 0.00000;
	pair_log(2,21,25) = 0.00000;
	pair_log(2,21,26) = 0.00000;
	pair_log(2,21,27) = 0.00000;
	pair_log(2,21,28) = 0.00000;
	pair_log(2,22, 1) = 0.00000;
	pair_log(2,22, 2) = 0.00000;
	pair_log(2,22, 3) = 0.00000;
	pair_log(2,22, 4) = 0.00000;
	pair_log(2,22, 5) = 0.00000;
	pair_log(2,22, 6) = 0.00000;
	pair_log(2,22, 7) = 0.00000;
	pair_log(2,22, 8) = 0.00000;
	pair_log(2,22, 9) = 0.00000;
	pair_log(2,22,10) = 0.00000;
	pair_log(2,22,11) = 0.00000;
	pair_log(2,22,12) = 0.00000;
	pair_log(2,22,13) = 0.00000;
	pair_log(2,22,14) = 0.00000;
	pair_log(2,22,15) = 0.00000;
	pair_log(2,22,16) = 0.00000;
	pair_log(2,22,17) = 0.00000;
	pair_log(2,22,18) = 0.00000;
	pair_log(2,22,19) = 0.00000;
	pair_log(2,22,20) = 0.00000;
	pair_log(2,22,21) = 0.00000;
	pair_log(2,22,22) = 0.00000;
	pair_log(2,22,23) = 0.00000;
	pair_log(2,22,24) = 0.00000;
	pair_log(2,22,25) = 0.00000;
	pair_log(2,22,26) = 0.00000;
	pair_log(2,22,27) = 0.00000;
	pair_log(2,22,28) = 0.00000;
	pair_log(2,23, 1) = 0.00000;
	pair_log(2,23, 2) = 0.00000;
	pair_log(2,23, 3) = 0.00000;
	pair_log(2,23, 4) = 0.00000;
	pair_log(2,23, 5) = 0.00000;
	pair_log(2,23, 6) = 0.00000;
	pair_log(2,23, 7) = 0.00000;
	pair_log(2,23, 8) = 0.00000;
	pair_log(2,23, 9) = 0.00000;
	pair_log(2,23,10) = 0.00000;
	pair_log(2,23,11) = 0.00000;
	pair_log(2,23,12) = 0.00000;
	pair_log(2,23,13) = 0.00000;
	pair_log(2,23,14) = 0.00000;
	pair_log(2,23,15) = 0.00000;
	pair_log(2,23,16) = 0.00000;
	pair_log(2,23,17) = 0.00000;
	pair_log(2,23,18) = 0.00000;
	pair_log(2,23,19) = 0.00000;
	pair_log(2,23,20) = 0.00000;
	pair_log(2,23,21) = 0.00000;
	pair_log(2,23,22) = 0.00000;
	pair_log(2,23,23) = 0.00000;
	pair_log(2,23,24) = 0.00000;
	pair_log(2,23,25) = 0.00000;
	pair_log(2,23,26) = 0.00000;
	pair_log(2,23,27) = 0.00000;
	pair_log(2,23,28) = 0.00000;
	pair_log(2,24, 1) = 0.00000;
	pair_log(2,24, 2) = 0.00000;
	pair_log(2,24, 3) = 0.00000;
	pair_log(2,24, 4) = 0.00000;
	pair_log(2,24, 5) = 0.00000;
	pair_log(2,24, 6) = 0.00000;
	pair_log(2,24, 7) = 0.00000;
	pair_log(2,24, 8) = 0.00000;
	pair_log(2,24, 9) = 0.00000;
	pair_log(2,24,10) = 0.00000;
	pair_log(2,24,11) = 0.00000;
	pair_log(2,24,12) = 0.00000;
	pair_log(2,24,13) = 0.00000;
	pair_log(2,24,14) = 0.00000;
	pair_log(2,24,15) = 0.00000;
	pair_log(2,24,16) = 0.00000;
	pair_log(2,24,17) = 0.00000;
	pair_log(2,24,18) = 0.00000;
	pair_log(2,24,19) = 0.00000;
	pair_log(2,24,20) = 0.00000;
	pair_log(2,24,21) = 0.00000;
	pair_log(2,24,22) = 0.00000;
	pair_log(2,24,23) = 0.00000;
	pair_log(2,24,24) = 0.00000;
	pair_log(2,24,25) = 0.00000;
	pair_log(2,24,26) = 0.00000;
	pair_log(2,24,27) = 0.00000;
	pair_log(2,24,28) = 0.00000;
	pair_log(2,25, 1) = 0.00000;
	pair_log(2,25, 2) = 0.00000;
	pair_log(2,25, 3) = 0.00000;
	pair_log(2,25, 4) = 0.00000;
	pair_log(2,25, 5) = 0.00000;
	pair_log(2,25, 6) = 0.00000;
	pair_log(2,25, 7) = 0.00000;
	pair_log(2,25, 8) = 0.00000;
	pair_log(2,25, 9) = 0.00000;
	pair_log(2,25,10) = 0.00000;
	pair_log(2,25,11) = 0.00000;
	pair_log(2,25,12) = 0.00000;
	pair_log(2,25,13) = 0.00000;
	pair_log(2,25,14) = 0.00000;
	pair_log(2,25,15) = 0.00000;
	pair_log(2,25,16) = 0.00000;
	pair_log(2,25,17) = 0.00000;
	pair_log(2,25,18) = 0.00000;
	pair_log(2,25,19) = 0.00000;
	pair_log(2,25,20) = 0.00000;
	pair_log(2,25,21) = 0.00000;
	pair_log(2,25,22) = 0.00000;
	pair_log(2,25,23) = 0.00000;
	pair_log(2,25,24) = 0.00000;
	pair_log(2,25,25) = 0.00000;
	pair_log(2,25,26) = 0.00000;
	pair_log(2,25,27) = 0.00000;
	pair_log(2,25,28) = 0.00000;
	pair_log(2,26, 1) = 0.00000;
	pair_log(2,26, 2) = 0.00000;
	pair_log(2,26, 3) = 0.00000;
	pair_log(2,26, 4) = 0.00000;
	pair_log(2,26, 5) = 0.00000;
	pair_log(2,26, 6) = 0.00000;
	pair_log(2,26, 7) = 0.00000;
	pair_log(2,26, 8) = 0.00000;
	pair_log(2,26, 9) = 0.00000;
	pair_log(2,26,10) = 0.00000;
	pair_log(2,26,11) = 0.00000;
	pair_log(2,26,12) = 0.00000;
	pair_log(2,26,13) = 0.00000;
	pair_log(2,26,14) = 0.00000;
	pair_log(2,26,15) = 0.00000;
	pair_log(2,26,16) = 0.00000;
	pair_log(2,26,17) = 0.00000;
	pair_log(2,26,18) = 0.00000;
	pair_log(2,26,19) = 0.00000;
	pair_log(2,26,20) = 0.00000;
	pair_log(2,26,21) = 0.00000;
	pair_log(2,26,22) = 0.00000;
	pair_log(2,26,23) = 0.00000;
	pair_log(2,26,24) = 0.00000;
	pair_log(2,26,25) = 0.00000;
	pair_log(2,26,26) = 0.00000;
	pair_log(2,26,27) = 0.00000;
	pair_log(2,26,28) = 0.00000;
	pair_log(2,27, 1) = 0.00000;
	pair_log(2,27, 2) = 0.00000;
	pair_log(2,27, 3) = 0.00000;
	pair_log(2,27, 4) = 0.00000;
	pair_log(2,27, 5) = 0.00000;
	pair_log(2,27, 6) = 0.00000;
	pair_log(2,27, 7) = 0.00000;
	pair_log(2,27, 8) = 0.00000;
	pair_log(2,27, 9) = 0.00000;
	pair_log(2,27,10) = 0.00000;
	pair_log(2,27,11) = 0.00000;
	pair_log(2,27,12) = 0.00000;
	pair_log(2,27,13) = 0.00000;
	pair_log(2,27,14) = 0.00000;
	pair_log(2,27,15) = 0.00000;
	pair_log(2,27,16) = 0.00000;
	pair_log(2,27,17) = 0.00000;
	pair_log(2,27,18) = 0.00000;
	pair_log(2,27,19) = 0.00000;
	pair_log(2,27,20) = 0.00000;
	pair_log(2,27,21) = 0.00000;
	pair_log(2,27,22) = 0.00000;
	pair_log(2,27,23) = 0.00000;
	pair_log(2,27,24) = 0.00000;
	pair_log(2,27,25) = 0.00000;
	pair_log(2,27,26) = 0.00000;
	pair_log(2,27,27) = 0.00000;
	pair_log(2,27,28) = 0.00000;
	pair_log(2,28, 1) = 0.00000;
	pair_log(2,28, 2) = 0.00000;
	pair_log(2,28, 3) = 0.00000;
	pair_log(2,28, 4) = 0.00000;
	pair_log(2,28, 5) = 0.00000;
	pair_log(2,28, 6) = 0.00000;
	pair_log(2,28, 7) = 0.00000;
	pair_log(2,28, 8) = 0.00000;
	pair_log(2,28, 9) = 0.00000;
	pair_log(2,28,10) = 0.00000;
	pair_log(2,28,11) = 0.00000;
	pair_log(2,28,12) = 0.00000;
	pair_log(2,28,13) = 0.00000;
	pair_log(2,28,14) = 0.00000;
	pair_log(2,28,15) = 0.00000;
	pair_log(2,28,16) = 0.00000;
	pair_log(2,28,17) = 0.00000;
	pair_log(2,28,18) = 0.00000;
	pair_log(2,28,19) = 0.00000;
	pair_log(2,28,20) = 0.00000;
	pair_log(2,28,21) = 0.00000;
	pair_log(2,28,22) = 0.00000;
	pair_log(2,28,23) = 0.00000;
	pair_log(2,28,24) = 0.00000;
	pair_log(2,28,25) = 0.00000;
	pair_log(2,28,26) = 0.00000;
	pair_log(2,28,27) = 0.00000;
	pair_log(2,28,28) = 0.00000;
	//KMa phospho_ser 29 aa-dna
	for (int i=1;i<=29;i++) {
		pair_log(2,29,i) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(2,i,29) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(2,21,i) = pair_log(2,16,i);
	}
	for (int i=1;i<=29;i++) {
		pair_log(2,i,21) = pair_log(2,i,16);
	}
}

void
SecondaryStructureEval::pair_log_initializer_section3a(
	FArray3D_float & pair_log
)
{
	// section 3a
	pair_log(3, 1, 1) = -0.06711;
	pair_log(3, 1, 2) = 0.04917;
	pair_log(3, 1, 3) = 0.08686;
	pair_log(3, 1, 4) = 0.10110;
	pair_log(3, 1, 5) = 0.02583;
	pair_log(3, 1, 6) = 0.02605;
	pair_log(3, 1, 7) = 0.06456;
	pair_log(3, 1, 8) = -0.10028;
	pair_log(3, 1, 9) = 0.06167;
	pair_log(3, 1,10) = -0.08988;
	pair_log(3, 1,11) = -0.03220;
	pair_log(3, 1,12) = 0.09263;
	pair_log(3, 1,13) = 0.04566;
	pair_log(3, 1,14) = 0.07189;
	pair_log(3, 1,15) = 0.06154;
	pair_log(3, 1,16) = 0.05356;
	pair_log(3, 1,17) = 0.04740;
	pair_log(3, 1,18) = -0.07279;
	pair_log(3, 1,19) = 0.08732;
	pair_log(3, 1,20) = 0.03904;
	pair_log(3, 1,21) = 0.00000;
	pair_log(3, 1,22) = 0.00000;
	pair_log(3, 1,23) = 0.00000;
	pair_log(3, 1,24) = 0.00000;
	pair_log(3, 1,25) = 0.00000;
	pair_log(3, 1,26) = 0.00000;
	pair_log(3, 1,27) = 0.00000;
	pair_log(3, 1,28) = 0.00000;
	pair_log(3, 2, 1) = 0.04917;
	pair_log(3, 2, 2) = -0.17110;
	pair_log(3, 2, 3) = 0.08274;
	pair_log(3, 2, 4) = 0.12837;
	pair_log(3, 2, 5) = -0.05523;
	pair_log(3, 2, 6) = -0.06868;
	pair_log(3, 2, 7) = -0.05122;
	pair_log(3, 2, 8) = 0.00854;
	pair_log(3, 2, 9) = 0.10403;
	pair_log(3, 2,10) = -0.00733;
	pair_log(3, 2,11) = -0.10678;
	pair_log(3, 2,12) = 0.02032;
	pair_log(3, 2,13) = 0.01966;
	pair_log(3, 2,14) = 0.02698;
	pair_log(3, 2,15) = 0.10341;
	pair_log(3, 2,16) = -0.00836;
	pair_log(3, 2,17) = -0.04399;
	pair_log(3, 2,18) = 0.00772;
	pair_log(3, 2,19) = -0.02572;
	pair_log(3, 2,20) = -0.01389;
	pair_log(3, 2,21) = 0.00000;
	pair_log(3, 2,22) = 0.00000;
	pair_log(3, 2,23) = 0.00000;
	pair_log(3, 2,24) = 0.00000;
	pair_log(3, 2,25) = 0.00000;
	pair_log(3, 2,26) = 0.00000;
	pair_log(3, 2,27) = 0.00000;
	pair_log(3, 2,28) = 0.00000;
	pair_log(3, 3, 1) = 0.08686;
	pair_log(3, 3, 2) = 0.08274;
	pair_log(3, 3, 3) = -0.17210;
	pair_log(3, 3, 4) = -0.14442;
	pair_log(3, 3, 5) = 0.14568;
	pair_log(3, 3, 6) = -0.01798;
	pair_log(3, 3, 7) = -0.12234;
	pair_log(3, 3, 8) = 0.17048;
	pair_log(3, 3, 9) = -0.28130;
	pair_log(3, 3,10) = 0.14879;
	pair_log(3, 3,11) = 0.06630;
	pair_log(3, 3,12) = -0.15017;
	pair_log(3, 3,13) = -0.05572;
	pair_log(3, 3,14) = -0.12196;
	pair_log(3, 3,15) = -0.30946;
	pair_log(3, 3,16) = 0.00009;
	pair_log(3, 3,17) = 0.00758;
	pair_log(3, 3,18) = 0.14779;
	pair_log(3, 3,19) = 0.02263;
	pair_log(3, 3,20) = -0.05851;
	pair_log(3, 3,21) = 0.00000;
	pair_log(3, 3,22) = 0.00000;
	pair_log(3, 3,23) = 0.00000;
	pair_log(3, 3,24) = 0.00000;
	pair_log(3, 3,25) = 0.00000;
	pair_log(3, 3,26) = 0.00000;
	pair_log(3, 3,27) = 0.00000;
	pair_log(3, 3,28) = 0.00000;
	pair_log(3, 4, 1) = 0.10110;
	pair_log(3, 4, 2) = 0.12837;
	pair_log(3, 4, 3) = -0.14442;
	pair_log(3, 4, 4) = -0.17005;
	pair_log(3, 4, 5) = 0.11831;
	pair_log(3, 4, 6) = 0.04571;
	pair_log(3, 4, 7) = -0.08473;
	pair_log(3, 4, 8) = 0.14349;
	pair_log(3, 4, 9) = -0.32208;
	pair_log(3, 4,10) = 0.14917;
	pair_log(3, 4,11) = 0.07144;
	pair_log(3, 4,12) = -0.09284;
	pair_log(3, 4,13) = -0.10634;
	pair_log(3, 4,14) = -0.10631;
	pair_log(3, 4,15) = -0.28982;
	pair_log(3, 4,16) = 0.00702;
	pair_log(3, 4,17) = -0.02133;
	pair_log(3, 4,18) = 0.11895;
	pair_log(3, 4,19) = -0.04921;
	pair_log(3, 4,20) = -0.08186;
	pair_log(3, 4,21) = 0.00000;
	pair_log(3, 4,22) = 0.00000;
	pair_log(3, 4,23) = 0.00000;
	pair_log(3, 4,24) = 0.00000;
	pair_log(3, 4,25) = 0.00000;
	pair_log(3, 4,26) = 0.00000;
	pair_log(3, 4,27) = 0.00000;
	pair_log(3, 4,28) = 0.00000;
	pair_log(3, 5, 1) = 0.02583;
	pair_log(3, 5, 2) = -0.05523;
	pair_log(3, 5, 3) = 0.14568;
	pair_log(3, 5, 4) = 0.11831;
	pair_log(3, 5, 5) = -0.10140;
	pair_log(3, 5, 6) = 0.07023;
	pair_log(3, 5, 7) = 0.03701;
	pair_log(3, 5, 8) = -0.08883;
	pair_log(3, 5, 9) = 0.14062;
	pair_log(3, 5,10) = -0.11808;
	pair_log(3, 5,11) = -0.11397;
	pair_log(3, 5,12) = 0.09736;
	pair_log(3, 5,13) = 0.12370;
	pair_log(3, 5,14) = 0.06410;
	pair_log(3, 5,15) = 0.15212;
	pair_log(3, 5,16) = -0.01294;
	pair_log(3, 5,17) = 0.03108;
	pair_log(3, 5,18) = -0.04855;
	pair_log(3, 5,19) = -0.11108;
	pair_log(3, 5,20) = -0.02173;
	pair_log(3, 5,21) = 0.00000;
	pair_log(3, 5,22) = 0.00000;
	pair_log(3, 5,23) = 0.00000;
	pair_log(3, 5,24) = 0.00000;
	pair_log(3, 5,25) = 0.00000;
	pair_log(3, 5,26) = 0.00000;
	pair_log(3, 5,27) = 0.00000;
	pair_log(3, 5,28) = 0.00000;
	pair_log(3, 6, 1) = 0.02605;
	pair_log(3, 6, 2) = -0.06868;
	pair_log(3, 6, 3) = -0.01798;
	pair_log(3, 6, 4) = 0.04571;
	pair_log(3, 6, 5) = 0.07023;
	pair_log(3, 6, 6) = -0.08089;
	pair_log(3, 6, 7) = -0.00788;
	pair_log(3, 6, 8) = 0.02672;
	pair_log(3, 6, 9) = 0.00219;
	pair_log(3, 6,10) = 0.01380;
	pair_log(3, 6,11) = -0.02177;
	pair_log(3, 6,12) = -0.02792;
	pair_log(3, 6,13) = -0.04698;
	pair_log(3, 6,14) = -0.02051;
	pair_log(3, 6,15) = -0.05025;
	pair_log(3, 6,16) = 0.00867;
	pair_log(3, 6,17) = -0.00796;
	pair_log(3, 6,18) = 0.02340;
	pair_log(3, 6,19) = 0.10405;
	pair_log(3, 6,20) = 0.04000;
	pair_log(3, 6,21) = 0.00000;
	pair_log(3, 6,22) = 0.00000;
	pair_log(3, 6,23) = 0.00000;
	pair_log(3, 6,24) = 0.00000;
	pair_log(3, 6,25) = 0.00000;
	pair_log(3, 6,26) = 0.00000;
	pair_log(3, 6,27) = 0.00000;
	pair_log(3, 6,28) = 0.00000;
	pair_log(3, 7, 1) = 0.06456;
	pair_log(3, 7, 2) = -0.05122;
	pair_log(3, 7, 3) = -0.12234;
	pair_log(3, 7, 4) = -0.08473;
	pair_log(3, 7, 5) = 0.03701;
	pair_log(3, 7, 6) = -0.00788;
	pair_log(3, 7, 7) = -0.36006;
	pair_log(3, 7, 8) = 0.16715;
	pair_log(3, 7, 9) = 0.05396;
	pair_log(3, 7,10) = 0.12272;
	pair_log(3, 7,11) = 0.03768;
	pair_log(3, 7,12) = -0.07118;
	pair_log(3, 7,13) = -0.05851;
	pair_log(3, 7,14) = -0.07235;
	pair_log(3, 7,15) = -0.02235;
	pair_log(3, 7,16) = -0.06447;
	pair_log(3, 7,17) = -0.03512;
	pair_log(3, 7,18) = 0.07075;
	pair_log(3, 7,19) = -0.11836;
	pair_log(3, 7,20) = -0.03306;
	pair_log(3, 7,21) = 0.00000;
	pair_log(3, 7,22) = 0.00000;
	pair_log(3, 7,23) = 0.00000;
	pair_log(3, 7,24) = 0.00000;
	pair_log(3, 7,25) = 0.00000;
	pair_log(3, 7,26) = 0.00000;
	pair_log(3, 7,27) = 0.00000;
	pair_log(3, 7,28) = 0.00000;
	pair_log(3, 8, 1) = -0.10028;
	pair_log(3, 8, 2) = 0.00854;
	pair_log(3, 8, 3) = 0.17048;
	pair_log(3, 8, 4) = 0.14349;
	pair_log(3, 8, 5) = -0.08883;
	pair_log(3, 8, 6) = 0.02672;
	pair_log(3, 8, 7) = 0.16715;
	pair_log(3, 8, 8) = -0.12746;
	pair_log(3, 8, 9) = 0.16985;
	pair_log(3, 8,10) = -0.08644;
	pair_log(3, 8,11) = -0.00634;
	pair_log(3, 8,12) = 0.16954;
	pair_log(3, 8,13) = 0.12930;
	pair_log(3, 8,14) = 0.13563;
	pair_log(3, 8,15) = 0.19111;
	pair_log(3, 8,16) = 0.04214;
	pair_log(3, 8,17) = 0.00418;
	pair_log(3, 8,18) = -0.13605;
	pair_log(3, 8,19) = 0.03430;
	pair_log(3, 8,20) = 0.02135;
	pair_log(3, 8,21) = 0.00000;
	pair_log(3, 8,22) = 0.00000;
	pair_log(3, 8,23) = 0.00000;
	pair_log(3, 8,24) = 0.00000;
	pair_log(3, 8,25) = 0.00000;
	pair_log(3, 8,26) = 0.00000;
	pair_log(3, 8,27) = 0.00000;
	pair_log(3, 8,28) = 0.00000;
	pair_log(3, 9, 1) = 0.06167;
	pair_log(3, 9, 2) = 0.10403;
	pair_log(3, 9, 3) = -0.28130;
	pair_log(3, 9, 4) = -0.32208;
	pair_log(3, 9, 5) = 0.14062;
	pair_log(3, 9, 6) = 0.00219;
	pair_log(3, 9, 7) = 0.05396;
	pair_log(3, 9, 8) = 0.16985;
	pair_log(3, 9, 9) = -0.19507;
	pair_log(3, 9,10) = 0.10764;
	pair_log(3, 9,11) = 0.15021;
	pair_log(3, 9,12) = -0.17589;
	pair_log(3, 9,13) = -0.12627;
	pair_log(3, 9,14) = -0.16154;
	pair_log(3, 9,15) = -0.03348;
	pair_log(3, 9,16) = -0.03768;
	pair_log(3, 9,17) = -0.01572;
	pair_log(3, 9,18) = 0.12223;
	pair_log(3, 9,19) = 0.14702;
	pair_log(3, 9,20) = 0.06447;
	pair_log(3, 9,21) = 0.00000;
	pair_log(3, 9,22) = 0.00000;
	pair_log(3, 9,23) = 0.00000;
	pair_log(3, 9,24) = 0.00000;
	pair_log(3, 9,25) = 0.00000;
	pair_log(3, 9,26) = 0.00000;
	pair_log(3, 9,27) = 0.00000;
	pair_log(3, 9,28) = 0.00000;
	pair_log(3,10, 1) = -0.08988;
	pair_log(3,10, 2) = -0.00733;
	pair_log(3,10, 3) = 0.14879;
	pair_log(3,10, 4) = 0.14917;
	pair_log(3,10, 5) = -0.11808;
	pair_log(3,10, 6) = 0.01380;
	pair_log(3,10, 7) = 0.12272;
	pair_log(3,10, 8) = -0.08644;
	pair_log(3,10, 9) = 0.10764;
	pair_log(3,10,10) = -0.06027;
	pair_log(3,10,11) = 0.01406;
	pair_log(3,10,12) = 0.12818;
	pair_log(3,10,13) = 0.07467;
	pair_log(3,10,14) = 0.12921;
	pair_log(3,10,15) = 0.18513;
	pair_log(3,10,16) = 0.01289;
	pair_log(3,10,17) = 0.01068;
	pair_log(3,10,18) = -0.06701;
	pair_log(3,10,19) = -0.06007;
	pair_log(3,10,20) = 0.00677;
	pair_log(3,10,21) = 0.00000;
	pair_log(3,10,22) = 0.00000;
	pair_log(3,10,23) = 0.00000;
	pair_log(3,10,24) = 0.00000;
	pair_log(3,10,25) = 0.00000;
	pair_log(3,10,26) = 0.00000;
	pair_log(3,10,27) = 0.00000;
	pair_log(3,10,28) = 0.00000;
	pair_log(3,11, 1) = -0.03220;
	pair_log(3,11, 2) = -0.10678;
	pair_log(3,11, 3) = 0.06630;
	pair_log(3,11, 4) = 0.07144;
	pair_log(3,11, 5) = -0.11397;
	pair_log(3,11, 6) = -0.02177;
	pair_log(3,11, 7) = 0.03768;
	pair_log(3,11, 8) = -0.00634;
	pair_log(3,11, 9) = 0.15021;
	pair_log(3,11,10) = 0.01406;
	pair_log(3,11,11) = -0.00959;
	pair_log(3,11,12) = 0.11448;
	pair_log(3,11,13) = 0.09356;
	pair_log(3,11,14) = 0.01964;
	pair_log(3,11,15) = 0.07068;
	pair_log(3,11,16) = -0.02411;
	pair_log(3,11,17) = -0.00949;
	pair_log(3,11,18) = -0.01508;
	pair_log(3,11,19) = -0.06644;
	pair_log(3,11,20) = -0.00096;
	pair_log(3,11,21) = 0.00000;
	pair_log(3,11,22) = 0.00000;
	pair_log(3,11,23) = 0.00000;
	pair_log(3,11,24) = 0.00000;
	pair_log(3,11,25) = 0.00000;
	pair_log(3,11,26) = 0.00000;
	pair_log(3,11,27) = 0.00000;
	pair_log(3,11,28) = 0.00000;
	pair_log(3,12, 1) = 0.09263;
	pair_log(3,12, 2) = 0.02032;
	pair_log(3,12, 3) = -0.15017;
	pair_log(3,12, 4) = -0.09284;
	pair_log(3,12, 5) = 0.09736;
	pair_log(3,12, 6) = -0.02792;
	pair_log(3,12, 7) = -0.07118;
	pair_log(3,12, 8) = 0.16954;
	pair_log(3,12, 9) = -0.17589;
	pair_log(3,12,10) = 0.12818;
	pair_log(3,12,11) = 0.11448;
	pair_log(3,12,12) = -0.17967;
	pair_log(3,12,13) = -0.09111;
	pair_log(3,12,14) = -0.11794;
	pair_log(3,12,15) = -0.15773;
	pair_log(3,12,16) = -0.07509;
	pair_log(3,12,17) = -0.02649;
	pair_log(3,12,18) = 0.10802;
	pair_log(3,12,19) = 0.00987;
	pair_log(3,12,20) = -0.01661;
	pair_log(3,12,21) = 0.00000;
	pair_log(3,12,22) = 0.00000;
	pair_log(3,12,23) = 0.00000;
	pair_log(3,12,24) = 0.00000;
	pair_log(3,12,25) = 0.00000;
	pair_log(3,12,26) = 0.00000;
	pair_log(3,12,27) = 0.00000;
	pair_log(3,12,28) = 0.00000;
	pair_log(3,13, 1) = 0.04566;
	pair_log(3,13, 2) = 0.01966;
	pair_log(3,13, 3) = -0.05572;
	pair_log(3,13, 4) = -0.10634;
	pair_log(3,13, 5) = 0.12370;
	pair_log(3,13, 6) = -0.04698;
	pair_log(3,13, 7) = -0.05851;
	pair_log(3,13, 8) = 0.12930;
	pair_log(3,13, 9) = -0.12627;
	pair_log(3,13,10) = 0.07467;
	pair_log(3,13,11) = 0.09356;
	pair_log(3,13,12) = -0.09111;
	pair_log(3,13,13) = -0.13648;
	pair_log(3,13,14) = -0.08858;
	pair_log(3,13,15) = -0.16615;
	pair_log(3,13,16) = -0.02111;
	pair_log(3,13,17) = 0.02446;
	pair_log(3,13,18) = 0.11169;
	pair_log(3,13,19) = 0.01048;
	pair_log(3,13,20) = -0.05590;
	pair_log(3,13,21) = 0.00000;
	pair_log(3,13,22) = 0.00000;
	pair_log(3,13,23) = 0.00000;
	pair_log(3,13,24) = 0.00000;
	pair_log(3,13,25) = 0.00000;
	pair_log(3,13,26) = 0.00000;
	pair_log(3,13,27) = 0.00000;
	pair_log(3,13,28) = 0.00000;
	pair_log(3,14, 1) = 0.07189;
	pair_log(3,14, 2) = 0.02698;
	pair_log(3,14, 3) = -0.12196;
	pair_log(3,14, 4) = -0.10631;
	pair_log(3,14, 5) = 0.06410;
	pair_log(3,14, 6) = -0.02051;
	pair_log(3,14, 7) = -0.07235;
	pair_log(3,14, 8) = 0.13563;
	pair_log(3,14, 9) = -0.16154;
	pair_log(3,14,10) = 0.12921;
	pair_log(3,14,11) = 0.01964;
	pair_log(3,14,12) = -0.11794;
	pair_log(3,14,13) = -0.08858;
	pair_log(3,14,14) = -0.09693;
	pair_log(3,14,15) = -0.13486;
	pair_log(3,14,16) = -0.05424;
	pair_log(3,14,17) = -0.01817;
	pair_log(3,14,18) = 0.11813;
	pair_log(3,14,19) = -0.06867;
	pair_log(3,14,20) = -0.04713;
	pair_log(3,14,21) = 0.00000;
	pair_log(3,14,22) = 0.00000;
	pair_log(3,14,23) = 0.00000;
	pair_log(3,14,24) = 0.00000;
	pair_log(3,14,25) = 0.00000;
	pair_log(3,14,26) = 0.00000;
	pair_log(3,14,27) = 0.00000;
	pair_log(3,14,28) = 0.00000;
	pair_log(3,15, 1) = 0.06154;
	pair_log(3,15, 2) = 0.10341;
	pair_log(3,15, 3) = -0.30946;
	pair_log(3,15, 4) = -0.28982;
	pair_log(3,15, 5) = 0.15212;
	pair_log(3,15, 6) = -0.05025;
	pair_log(3,15, 7) = -0.02235;
	pair_log(3,15, 8) = 0.19111;
	pair_log(3,15, 9) = -0.03348;
	pair_log(3,15,10) = 0.18513;
	pair_log(3,15,11) = 0.07068;
	pair_log(3,15,12) = -0.15773;
	pair_log(3,15,13) = -0.16615;
	pair_log(3,15,14) = -0.13486;
	pair_log(3,15,15) = -0.08474;
	pair_log(3,15,16) = -0.09178;
	pair_log(3,15,17) = -0.04650;
	pair_log(3,15,18) = 0.17288;
	pair_log(3,15,19) = 0.07735;
	pair_log(3,15,20) = 0.02232;
	pair_log(3,15,21) = 0.00000;
	pair_log(3,15,22) = 0.00000;
	pair_log(3,15,23) = 0.00000;
	pair_log(3,15,24) = 0.00000;
	pair_log(3,15,25) = 0.00000;
	pair_log(3,15,26) = 0.00000;
	pair_log(3,15,27) = 0.00000;
	pair_log(3,15,28) = 0.00000;
}

void
SecondaryStructureEval::pair_log_initializer_section3b(
	FArray3D_float & pair_log
)
{
	// section 3b
	pair_log(3,16, 1) = 0.05356;
	pair_log(3,16, 2) = -0.00836;
	pair_log(3,16, 3) = 0.00009;
	pair_log(3,16, 4) = 0.00702;
	pair_log(3,16, 5) = -0.01294;
	pair_log(3,16, 6) = 0.00867;
	pair_log(3,16, 7) = -0.06447;
	pair_log(3,16, 8) = 0.04214;
	pair_log(3,16, 9) = -0.03768;
	pair_log(3,16,10) = 0.01289;
	pair_log(3,16,11) = -0.02411;
	pair_log(3,16,12) = -0.07509;
	pair_log(3,16,13) = -0.02111;
	pair_log(3,16,14) = -0.05424;
	pair_log(3,16,15) = -0.09178;
	pair_log(3,16,16) = 0.01170;
	pair_log(3,16,17) = -0.01390;
	pair_log(3,16,18) = 0.03754;
	pair_log(3,16,19) = 0.01889;
	pair_log(3,16,20) = 0.00139;
	pair_log(3,16,21) = 0.00000;
	pair_log(3,16,22) = 0.00000;
	pair_log(3,16,23) = 0.00000;
	pair_log(3,16,24) = 0.00000;
	pair_log(3,16,25) = 0.00000;
	pair_log(3,16,26) = 0.00000;
	pair_log(3,16,27) = 0.00000;
	pair_log(3,16,28) = 0.00000;
	pair_log(3,17, 1) = 0.04740;
	pair_log(3,17, 2) = -0.04399;
	pair_log(3,17, 3) = 0.00758;
	pair_log(3,17, 4) = -0.02133;
	pair_log(3,17, 5) = 0.03108;
	pair_log(3,17, 6) = -0.00796;
	pair_log(3,17, 7) = -0.03512;
	pair_log(3,17, 8) = 0.00418;
	pair_log(3,17, 9) = -0.01572;
	pair_log(3,17,10) = 0.01068;
	pair_log(3,17,11) = -0.00949;
	pair_log(3,17,12) = -0.02649;
	pair_log(3,17,13) = 0.02446;
	pair_log(3,17,14) = -0.01817;
	pair_log(3,17,15) = -0.04650;
	pair_log(3,17,16) = -0.01390;
	pair_log(3,17,17) = -0.01606;
	pair_log(3,17,18) = 0.01024;
	pair_log(3,17,19) = 0.01650;
	pair_log(3,17,20) = 0.00633;
	pair_log(3,17,21) = 0.00000;
	pair_log(3,17,22) = 0.00000;
	pair_log(3,17,23) = 0.00000;
	pair_log(3,17,24) = 0.00000;
	pair_log(3,17,25) = 0.00000;
	pair_log(3,17,26) = 0.00000;
	pair_log(3,17,27) = 0.00000;
	pair_log(3,17,28) = 0.00000;
	pair_log(3,18, 1) = -0.07279;
	pair_log(3,18, 2) = 0.00772;
	pair_log(3,18, 3) = 0.14779;
	pair_log(3,18, 4) = 0.11895;
	pair_log(3,18, 5) = -0.04855;
	pair_log(3,18, 6) = 0.02340;
	pair_log(3,18, 7) = 0.07075;
	pair_log(3,18, 8) = -0.13605;
	pair_log(3,18, 9) = 0.12223;
	pair_log(3,18,10) = -0.06701;
	pair_log(3,18,11) = -0.01508;
	pair_log(3,18,12) = 0.10802;
	pair_log(3,18,13) = 0.11169;
	pair_log(3,18,14) = 0.11813;
	pair_log(3,18,15) = 0.17288;
	pair_log(3,18,16) = 0.03754;
	pair_log(3,18,17) = 0.01024;
	pair_log(3,18,18) = -0.10756;
	pair_log(3,18,19) = 0.01594;
	pair_log(3,18,20) = 0.03319;
	pair_log(3,18,21) = 0.00000;
	pair_log(3,18,22) = 0.00000;
	pair_log(3,18,23) = 0.00000;
	pair_log(3,18,24) = 0.00000;
	pair_log(3,18,25) = 0.00000;
	pair_log(3,18,26) = 0.00000;
	pair_log(3,18,27) = 0.00000;
	pair_log(3,18,28) = 0.00000;
	pair_log(3,19, 1) = 0.08732;
	pair_log(3,19, 2) = -0.02572;
	pair_log(3,19, 3) = 0.02263;
	pair_log(3,19, 4) = -0.04921;
	pair_log(3,19, 5) = -0.11108;
	pair_log(3,19, 6) = 0.10405;
	pair_log(3,19, 7) = -0.11836;
	pair_log(3,19, 8) = 0.03430;
	pair_log(3,19, 9) = 0.14702;
	pair_log(3,19,10) = -0.06007;
	pair_log(3,19,11) = -0.06644;
	pair_log(3,19,12) = 0.00987;
	pair_log(3,19,13) = 0.01048;
	pair_log(3,19,14) = -0.06867;
	pair_log(3,19,15) = 0.07735;
	pair_log(3,19,16) = 0.01889;
	pair_log(3,19,17) = 0.01650;
	pair_log(3,19,18) = 0.01594;
	pair_log(3,19,19) = -0.07522;
	pair_log(3,19,20) = -0.09071;
	pair_log(3,19,21) = 0.00000;
	pair_log(3,19,22) = 0.00000;
	pair_log(3,19,23) = 0.00000;
	pair_log(3,19,24) = 0.00000;
	pair_log(3,19,25) = 0.00000;
	pair_log(3,19,26) = 0.00000;
	pair_log(3,19,27) = 0.00000;
	pair_log(3,19,28) = 0.00000;
	pair_log(3,20, 1) = 0.03904;
	pair_log(3,20, 2) = -0.01389;
	pair_log(3,20, 3) = -0.05851;
	pair_log(3,20, 4) = -0.08186;
	pair_log(3,20, 5) = -0.02173;
	pair_log(3,20, 6) = 0.04000;
	pair_log(3,20, 7) = -0.03306;
	pair_log(3,20, 8) = 0.02135;
	pair_log(3,20, 9) = 0.06447;
	pair_log(3,20,10) = 0.00677;
	pair_log(3,20,11) = -0.00096;
	pair_log(3,20,12) = -0.01661;
	pair_log(3,20,13) = -0.05590;
	pair_log(3,20,14) = -0.04713;
	pair_log(3,20,15) = 0.02232;
	pair_log(3,20,16) = 0.00139;
	pair_log(3,20,17) = 0.00633;
	pair_log(3,20,18) = 0.03319;
	pair_log(3,20,19) = -0.09071;
	pair_log(3,20,20) = -0.03925;
	pair_log(3,20,21) = 0.00000;
	pair_log(3,20,22) = 0.00000;
	pair_log(3,20,23) = 0.00000;
	pair_log(3,20,24) = 0.00000;
	pair_log(3,20,25) = 0.00000;
	pair_log(3,20,26) = 0.00000;
	pair_log(3,20,27) = 0.00000;
	pair_log(3,20,28) = 0.00000;
	pair_log(3,21, 1) = 0.00000;
	pair_log(3,21, 2) = 0.00000;
	pair_log(3,21, 3) = 0.00000;
	pair_log(3,21, 4) = 0.00000;
	pair_log(3,21, 5) = 0.00000;
	pair_log(3,21, 6) = 0.00000;
	pair_log(3,21, 7) = 0.00000;
	pair_log(3,21, 8) = 0.00000;
	pair_log(3,21, 9) = 0.00000;
	pair_log(3,21,10) = 0.00000;
	pair_log(3,21,11) = 0.00000;
	pair_log(3,21,12) = 0.00000;
	pair_log(3,21,13) = 0.00000;
	pair_log(3,21,14) = 0.00000;
	pair_log(3,21,15) = 0.00000;
	pair_log(3,21,16) = 0.00000;
	pair_log(3,21,17) = 0.00000;
	pair_log(3,21,18) = 0.00000;
	pair_log(3,21,19) = 0.00000;
	pair_log(3,21,20) = 0.00000;
	pair_log(3,21,21) = 0.00000;
	pair_log(3,21,22) = 0.00000;
	pair_log(3,21,23) = 0.00000;
	pair_log(3,21,24) = 0.00000;
	pair_log(3,21,25) = 0.00000;
	pair_log(3,21,26) = 0.00000;
	pair_log(3,21,27) = 0.00000;
	pair_log(3,21,28) = 0.00000;
	pair_log(3,22, 1) = 0.00000;
	pair_log(3,22, 2) = 0.00000;
	pair_log(3,22, 3) = 0.00000;
	pair_log(3,22, 4) = 0.00000;
	pair_log(3,22, 5) = 0.00000;
	pair_log(3,22, 6) = 0.00000;
	pair_log(3,22, 7) = 0.00000;
	pair_log(3,22, 8) = 0.00000;
	pair_log(3,22, 9) = 0.00000;
	pair_log(3,22,10) = 0.00000;
	pair_log(3,22,11) = 0.00000;
	pair_log(3,22,12) = 0.00000;
	pair_log(3,22,13) = 0.00000;
	pair_log(3,22,14) = 0.00000;
	pair_log(3,22,15) = 0.00000;
	pair_log(3,22,16) = 0.00000;
	pair_log(3,22,17) = 0.00000;
	pair_log(3,22,18) = 0.00000;
	pair_log(3,22,19) = 0.00000;
	pair_log(3,22,20) = 0.00000;
	pair_log(3,22,21) = 0.00000;
	pair_log(3,22,22) = 0.00000;
	pair_log(3,22,23) = 0.00000;
	pair_log(3,22,24) = 0.00000;
	pair_log(3,22,25) = 0.00000;
	pair_log(3,22,26) = 0.00000;
	pair_log(3,22,27) = 0.00000;
	pair_log(3,22,28) = 0.00000;
	pair_log(3,23, 1) = 0.00000;
	pair_log(3,23, 2) = 0.00000;
	pair_log(3,23, 3) = 0.00000;
	pair_log(3,23, 4) = 0.00000;
	pair_log(3,23, 5) = 0.00000;
	pair_log(3,23, 6) = 0.00000;
	pair_log(3,23, 7) = 0.00000;
	pair_log(3,23, 8) = 0.00000;
	pair_log(3,23, 9) = 0.00000;
	pair_log(3,23,10) = 0.00000;
	pair_log(3,23,11) = 0.00000;
	pair_log(3,23,12) = 0.00000;
	pair_log(3,23,13) = 0.00000;
	pair_log(3,23,14) = 0.00000;
	pair_log(3,23,15) = 0.00000;
	pair_log(3,23,16) = 0.00000;
	pair_log(3,23,17) = 0.00000;
	pair_log(3,23,18) = 0.00000;
	pair_log(3,23,19) = 0.00000;
	pair_log(3,23,20) = 0.00000;
	pair_log(3,23,21) = 0.00000;
	pair_log(3,23,22) = 0.00000;
	pair_log(3,23,23) = 0.00000;
	pair_log(3,23,24) = 0.00000;
	pair_log(3,23,25) = 0.00000;
	pair_log(3,23,26) = 0.00000;
	pair_log(3,23,27) = 0.00000;
	pair_log(3,23,28) = 0.00000;
	pair_log(3,24, 1) = 0.00000;
	pair_log(3,24, 2) = 0.00000;
	pair_log(3,24, 3) = 0.00000;
	pair_log(3,24, 4) = 0.00000;
	pair_log(3,24, 5) = 0.00000;
	pair_log(3,24, 6) = 0.00000;
	pair_log(3,24, 7) = 0.00000;
	pair_log(3,24, 8) = 0.00000;
	pair_log(3,24, 9) = 0.00000;
	pair_log(3,24,10) = 0.00000;
	pair_log(3,24,11) = 0.00000;
	pair_log(3,24,12) = 0.00000;
	pair_log(3,24,13) = 0.00000;
	pair_log(3,24,14) = 0.00000;
	pair_log(3,24,15) = 0.00000;
	pair_log(3,24,16) = 0.00000;
	pair_log(3,24,17) = 0.00000;
	pair_log(3,24,18) = 0.00000;
	pair_log(3,24,19) = 0.00000;
	pair_log(3,24,20) = 0.00000;
	pair_log(3,24,21) = 0.00000;
	pair_log(3,24,22) = 0.00000;
	pair_log(3,24,23) = 0.00000;
	pair_log(3,24,24) = 0.00000;
	pair_log(3,24,25) = 0.00000;
	pair_log(3,24,26) = 0.00000;
	pair_log(3,24,27) = 0.00000;
	pair_log(3,24,28) = 0.00000;
	pair_log(3,25, 1) = 0.00000;
	pair_log(3,25, 2) = 0.00000;
	pair_log(3,25, 3) = 0.00000;
	pair_log(3,25, 4) = 0.00000;
	pair_log(3,25, 5) = 0.00000;
	pair_log(3,25, 6) = 0.00000;
	pair_log(3,25, 7) = 0.00000;
	pair_log(3,25, 8) = 0.00000;
	pair_log(3,25, 9) = 0.00000;
	pair_log(3,25,10) = 0.00000;
	pair_log(3,25,11) = 0.00000;
	pair_log(3,25,12) = 0.00000;
	pair_log(3,25,13) = 0.00000;
	pair_log(3,25,14) = 0.00000;
	pair_log(3,25,15) = 0.00000;
	pair_log(3,25,16) = 0.00000;
	pair_log(3,25,17) = 0.00000;
	pair_log(3,25,18) = 0.00000;
	pair_log(3,25,19) = 0.00000;
	pair_log(3,25,20) = 0.00000;
	pair_log(3,25,21) = 0.00000;
	pair_log(3,25,22) = 0.00000;
	pair_log(3,25,23) = 0.00000;
	pair_log(3,25,24) = 0.00000;
	pair_log(3,25,25) = 0.00000;
	pair_log(3,25,26) = 0.00000;
	pair_log(3,25,27) = 0.00000;
	pair_log(3,25,28) = 0.00000;
	pair_log(3,26, 1) = 0.00000;
	pair_log(3,26, 2) = 0.00000;
	pair_log(3,26, 3) = 0.00000;
	pair_log(3,26, 4) = 0.00000;
	pair_log(3,26, 5) = 0.00000;
	pair_log(3,26, 6) = 0.00000;
	pair_log(3,26, 7) = 0.00000;
	pair_log(3,26, 8) = 0.00000;
	pair_log(3,26, 9) = 0.00000;
	pair_log(3,26,10) = 0.00000;
	pair_log(3,26,11) = 0.00000;
	pair_log(3,26,12) = 0.00000;
	pair_log(3,26,13) = 0.00000;
	pair_log(3,26,14) = 0.00000;
	pair_log(3,26,15) = 0.00000;
	pair_log(3,26,16) = 0.00000;
	pair_log(3,26,17) = 0.00000;
	pair_log(3,26,18) = 0.00000;
	pair_log(3,26,19) = 0.00000;
	pair_log(3,26,20) = 0.00000;
	pair_log(3,26,21) = 0.00000;
	pair_log(3,26,22) = 0.00000;
	pair_log(3,26,23) = 0.00000;
	pair_log(3,26,24) = 0.00000;
	pair_log(3,26,25) = 0.00000;
	pair_log(3,26,26) = 0.00000;
	pair_log(3,26,27) = 0.00000;
	pair_log(3,26,28) = 0.00000;
	pair_log(3,27, 1) = 0.00000;
	pair_log(3,27, 2) = 0.00000;
	pair_log(3,27, 3) = 0.00000;
	pair_log(3,27, 4) = 0.00000;
	pair_log(3,27, 5) = 0.00000;
	pair_log(3,27, 6) = 0.00000;
	pair_log(3,27, 7) = 0.00000;
	pair_log(3,27, 8) = 0.00000;
	pair_log(3,27, 9) = 0.00000;
	pair_log(3,27,10) = 0.00000;
	pair_log(3,27,11) = 0.00000;
	pair_log(3,27,12) = 0.00000;
	pair_log(3,27,13) = 0.00000;
	pair_log(3,27,14) = 0.00000;
	pair_log(3,27,15) = 0.00000;
	pair_log(3,27,16) = 0.00000;
	pair_log(3,27,17) = 0.00000;
	pair_log(3,27,18) = 0.00000;
	pair_log(3,27,19) = 0.00000;
	pair_log(3,27,20) = 0.00000;
	pair_log(3,27,21) = 0.00000;
	pair_log(3,27,22) = 0.00000;
	pair_log(3,27,23) = 0.00000;
	pair_log(3,27,24) = 0.00000;
	pair_log(3,27,25) = 0.00000;
	pair_log(3,27,26) = 0.00000;
	pair_log(3,27,27) = 0.00000;
	pair_log(3,27,28) = 0.00000;
	pair_log(3,28, 1) = 0.00000;
	pair_log(3,28, 2) = 0.00000;
	pair_log(3,28, 3) = 0.00000;
	pair_log(3,28, 4) = 0.00000;
	pair_log(3,28, 5) = 0.00000;
	pair_log(3,28, 6) = 0.00000;
	pair_log(3,28, 7) = 0.00000;
	pair_log(3,28, 8) = 0.00000;
	pair_log(3,28, 9) = 0.00000;
	pair_log(3,28,10) = 0.00000;
	pair_log(3,28,11) = 0.00000;
	pair_log(3,28,12) = 0.00000;
	pair_log(3,28,13) = 0.00000;
	pair_log(3,28,14) = 0.00000;
	pair_log(3,28,15) = 0.00000;
	pair_log(3,28,16) = 0.00000;
	pair_log(3,28,17) = 0.00000;
	pair_log(3,28,18) = 0.00000;
	pair_log(3,28,19) = 0.00000;
	pair_log(3,28,20) = 0.00000;
	pair_log(3,28,21) = 0.00000;
	pair_log(3,28,22) = 0.00000;
	pair_log(3,28,23) = 0.00000;
	pair_log(3,28,24) = 0.00000;
	pair_log(3,28,25) = 0.00000;
	pair_log(3,28,26) = 0.00000;
	pair_log(3,28,27) = 0.00000;
	pair_log(3,28,28) = 0.00000;
	//KMa phospho_ser
	for (int i=1;i<=29;i++) {
		pair_log(3,29,i) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(3,i,29) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(3,21,i) = pair_log(3,16,i);
	}
	for (int i=1;i<=29;i++) {
		pair_log(3,i,21) = pair_log(3,i,16);
	}
}

void
SecondaryStructureEval::pair_log_initializer_section4a(
	FArray3D_float & pair_log
)
{
	// section 4a
	pair_log(4, 1, 1) = -0.01427;
	pair_log(4, 1, 2) = 0.01382;
	pair_log(4, 1, 3) = 0.03348;
	pair_log(4, 1, 4) = 0.02897;
	pair_log(4, 1, 5) = -0.02317;
	pair_log(4, 1, 6) = 0.03371;
	pair_log(4, 1, 7) = 0.01652;
	pair_log(4, 1, 8) = -0.04417;
	pair_log(4, 1, 9) = 0.02223;
	pair_log(4, 1,10) = -0.02847;
	pair_log(4, 1,11) = 0.00464;
	pair_log(4, 1,12) = 0.03854;
	pair_log(4, 1,13) = 0.02263;
	pair_log(4, 1,14) = 0.02023;
	pair_log(4, 1,15) = 0.00319;
	pair_log(4, 1,16) = 0.00275;
	pair_log(4, 1,17) = 0.00640;
	pair_log(4, 1,18) = -0.02005;
	pair_log(4, 1,19) = 0.01054;
	pair_log(4, 1,20) = 0.01209;
	pair_log(4, 1,21) = 0.00000;
	pair_log(4, 1,22) = 0.00000;
	pair_log(4, 1,23) = 0.00000;
	pair_log(4, 1,24) = 0.00000;
	pair_log(4, 1,25) = 0.00000;
	pair_log(4, 1,26) = 0.00000;
	pair_log(4, 1,27) = 0.00000;
	pair_log(4, 1,28) = 0.00000;
	pair_log(4, 2, 1) = 0.01382;
	pair_log(4, 2, 2) = -0.03752;
	pair_log(4, 2, 3) = 0.05671;
	pair_log(4, 2, 4) = 0.03232;
	pair_log(4, 2, 5) = -0.06254;
	pair_log(4, 2, 6) = -0.02424;
	pair_log(4, 2, 7) = -0.06586;
	pair_log(4, 2, 8) = 0.04323;
	pair_log(4, 2, 9) = 0.08814;
	pair_log(4, 2,10) = 0.00556;
	pair_log(4, 2,11) = 0.00328;
	pair_log(4, 2,12) = 0.00085;
	pair_log(4, 2,13) = -0.02198;
	pair_log(4, 2,14) = 0.02667;
	pair_log(4, 2,15) = -0.02161;
	pair_log(4, 2,16) = -0.01602;
	pair_log(4, 2,17) = -0.01331;
	pair_log(4, 2,18) = -0.01558;
	pair_log(4, 2,19) = -0.09886;
	pair_log(4, 2,20) = 0.03435;
	pair_log(4, 2,21) = 0.00000;
	pair_log(4, 2,22) = 0.00000;
	pair_log(4, 2,23) = 0.00000;
	pair_log(4, 2,24) = 0.00000;
	pair_log(4, 2,25) = 0.00000;
	pair_log(4, 2,26) = 0.00000;
	pair_log(4, 2,27) = 0.00000;
	pair_log(4, 2,28) = 0.00000;
	pair_log(4, 3, 1) = 0.03348;
	pair_log(4, 3, 2) = 0.05671;
	pair_log(4, 3, 3) = -0.01488;
	pair_log(4, 3, 4) = -0.00697;
	pair_log(4, 3, 5) = 0.02231;
	pair_log(4, 3, 6) = -0.02100;
	pair_log(4, 3, 7) = 0.01351;
	pair_log(4, 3, 8) = 0.00868;
	pair_log(4, 3, 9) = -0.05616;
	pair_log(4, 3,10) = 0.00746;
	pair_log(4, 3,11) = 0.02495;
	pair_log(4, 3,12) = -0.01565;
	pair_log(4, 3,13) = -0.08884;
	pair_log(4, 3,14) = -0.00434;
	pair_log(4, 3,15) = -0.04608;
	pair_log(4, 3,16) = 0.01674;
	pair_log(4, 3,17) = -0.01276;
	pair_log(4, 3,18) = 0.03410;
	pair_log(4, 3,19) = -0.03207;
	pair_log(4, 3,20) = -0.00868;
	pair_log(4, 3,21) = 0.00000;
	pair_log(4, 3,22) = 0.00000;
	pair_log(4, 3,23) = 0.00000;
	pair_log(4, 3,24) = 0.00000;
	pair_log(4, 3,25) = 0.00000;
	pair_log(4, 3,26) = 0.00000;
	pair_log(4, 3,27) = 0.00000;
	pair_log(4, 3,28) = 0.00000;
	pair_log(4, 4, 1) = 0.02897;
	pair_log(4, 4, 2) = 0.03232;
	pair_log(4, 4, 3) = -0.00697;
	pair_log(4, 4, 4) = -0.08009;
	pair_log(4, 4, 5) = 0.01009;
	pair_log(4, 4, 6) = 0.03380;
	pair_log(4, 4, 7) = 0.01615;
	pair_log(4, 4, 8) = 0.03018;
	pair_log(4, 4, 9) = -0.09281;
	pair_log(4, 4,10) = -0.02856;
	pair_log(4, 4,11) = 0.00847;
	pair_log(4, 4,12) = -0.01793;
	pair_log(4, 4,13) = -0.05330;
	pair_log(4, 4,14) = -0.02655;
	pair_log(4, 4,15) = -0.05432;
	pair_log(4, 4,16) = 0.04724;
	pair_log(4, 4,17) = -0.00250;
	pair_log(4, 4,18) = 0.04775;
	pair_log(4, 4,19) = -0.01159;
	pair_log(4, 4,20) = 0.00113;
	pair_log(4, 4,21) = 0.00000;
	pair_log(4, 4,22) = 0.00000;
	pair_log(4, 4,23) = 0.00000;
	pair_log(4, 4,24) = 0.00000;
	pair_log(4, 4,25) = 0.00000;
	pair_log(4, 4,26) = 0.00000;
	pair_log(4, 4,27) = 0.00000;
	pair_log(4, 4,28) = 0.00000;
	pair_log(4, 5, 1) = -0.02317;
	pair_log(4, 5, 2) = -0.06254;
	pair_log(4, 5, 3) = 0.02231;
	pair_log(4, 5, 4) = 0.01009;
	pair_log(4, 5, 5) = 0.03221;
	pair_log(4, 5, 6) = 0.00280;
	pair_log(4, 5, 7) = -0.00113;
	pair_log(4, 5, 8) = -0.02499;
	pair_log(4, 5, 9) = 0.01523;
	pair_log(4, 5,10) = -0.00421;
	pair_log(4, 5,11) = 0.00735;
	pair_log(4, 5,12) = 0.00709;
	pair_log(4, 5,13) = 0.05823;
	pair_log(4, 5,14) = 0.01941;
	pair_log(4, 5,15) = 0.00793;
	pair_log(4, 5,16) = -0.00236;
	pair_log(4, 5,17) = 0.01079;
	pair_log(4, 5,18) = -0.00626;
	pair_log(4, 5,19) = -0.02970;
	pair_log(4, 5,20) = 0.00165;
	pair_log(4, 5,21) = 0.00000;
	pair_log(4, 5,22) = 0.00000;
	pair_log(4, 5,23) = 0.00000;
	pair_log(4, 5,24) = 0.00000;
	pair_log(4, 5,25) = 0.00000;
	pair_log(4, 5,26) = 0.00000;
	pair_log(4, 5,27) = 0.00000;
	pair_log(4, 5,28) = 0.00000;
	pair_log(4, 6, 1) = 0.03371;
	pair_log(4, 6, 2) = -0.02424;
	pair_log(4, 6, 3) = -0.02100;
	pair_log(4, 6, 4) = 0.03380;
	pair_log(4, 6, 5) = 0.00280;
	pair_log(4, 6, 6) = -0.02023;
	pair_log(4, 6, 7) = -0.08543;
	pair_log(4, 6, 8) = 0.00072;
	pair_log(4, 6, 9) = 0.06072;
	pair_log(4, 6,10) = 0.03720;
	pair_log(4, 6,11) = 0.00033;
	pair_log(4, 6,12) = -0.00528;
	pair_log(4, 6,13) = -0.01507;
	pair_log(4, 6,14) = -0.02658;
	pair_log(4, 6,15) = 0.00933;
	pair_log(4, 6,16) = -0.02573;
	pair_log(4, 6,17) = -0.01628;
	pair_log(4, 6,18) = 0.00420;
	pair_log(4, 6,19) = -0.06240;
	pair_log(4, 6,20) = -0.01505;
	pair_log(4, 6,21) = 0.00000;
	pair_log(4, 6,22) = 0.00000;
	pair_log(4, 6,23) = 0.00000;
	pair_log(4, 6,24) = 0.00000;
	pair_log(4, 6,25) = 0.00000;
	pair_log(4, 6,26) = 0.00000;
	pair_log(4, 6,27) = 0.00000;
	pair_log(4, 6,28) = 0.00000;
	pair_log(4, 7, 1) = 0.01652;
	pair_log(4, 7, 2) = -0.06586;
	pair_log(4, 7, 3) = 0.01351;
	pair_log(4, 7, 4) = 0.01615;
	pair_log(4, 7, 5) = -0.00113;
	pair_log(4, 7, 6) = -0.08543;
	pair_log(4, 7, 7) = -0.11699;
	pair_log(4, 7, 8) = 0.06119;
	pair_log(4, 7, 9) = 0.07429;
	pair_log(4, 7,10) = -0.00183;
	pair_log(4, 7,11) = 0.03515;
	pair_log(4, 7,12) = -0.02616;
	pair_log(4, 7,13) = 0.02330;
	pair_log(4, 7,14) = -0.00106;
	pair_log(4, 7,15) = 0.02289;
	pair_log(4, 7,16) = 0.00237;
	pair_log(4, 7,17) = 0.01188;
	pair_log(4, 7,18) = 0.01640;
	pair_log(4, 7,19) = -0.05614;
	pair_log(4, 7,20) = -0.01576;
	pair_log(4, 7,21) = 0.00000;
	pair_log(4, 7,22) = 0.00000;
	pair_log(4, 7,23) = 0.00000;
	pair_log(4, 7,24) = 0.00000;
	pair_log(4, 7,25) = 0.00000;
	pair_log(4, 7,26) = 0.00000;
	pair_log(4, 7,27) = 0.00000;
	pair_log(4, 7,28) = 0.00000;
	pair_log(4, 8, 1) = -0.04417;
	pair_log(4, 8, 2) = 0.04323;
	pair_log(4, 8, 3) = 0.00868;
	pair_log(4, 8, 4) = 0.03018;
	pair_log(4, 8, 5) = -0.02499;
	pair_log(4, 8, 6) = 0.00072;
	pair_log(4, 8, 7) = 0.06119;
	pair_log(4, 8, 8) = -0.03438;
	pair_log(4, 8, 9) = 0.00004;
	pair_log(4, 8,10) = -0.01016;
	pair_log(4, 8,11) = -0.04361;
	pair_log(4, 8,12) = 0.02517;
	pair_log(4, 8,13) = 0.07910;
	pair_log(4, 8,14) = 0.04162;
	pair_log(4, 8,15) = 0.06128;
	pair_log(4, 8,16) = 0.00588;
	pair_log(4, 8,17) = -0.02153;
	pair_log(4, 8,18) = -0.04232;
	pair_log(4, 8,19) = 0.09042;
	pair_log(4, 8,20) = 0.07198;
	pair_log(4, 8,21) = 0.00000;
	pair_log(4, 8,22) = 0.00000;
	pair_log(4, 8,23) = 0.00000;
	pair_log(4, 8,24) = 0.00000;
	pair_log(4, 8,25) = 0.00000;
	pair_log(4, 8,26) = 0.00000;
	pair_log(4, 8,27) = 0.00000;
	pair_log(4, 8,28) = 0.00000;
	pair_log(4, 9, 1) = 0.02223;
	pair_log(4, 9, 2) = 0.08814;
	pair_log(4, 9, 3) = -0.05616;
	pair_log(4, 9, 4) = -0.09281;
	pair_log(4, 9, 5) = 0.01523;
	pair_log(4, 9, 6) = 0.06072;
	pair_log(4, 9, 7) = 0.07429;
	pair_log(4, 9, 8) = 0.00004;
	pair_log(4, 9, 9) = -0.12343;
	pair_log(4, 9,10) = -0.01861;
	pair_log(4, 9,11) = 0.03596;
	pair_log(4, 9,12) = -0.04689;
	pair_log(4, 9,13) = -0.06779;
	pair_log(4, 9,14) = 0.01306;
	pair_log(4, 9,15) = -0.03842;
	pair_log(4, 9,16) = 0.05413;
	pair_log(4, 9,17) = 0.02293;
	pair_log(4, 9,18) = 0.01947;
	pair_log(4, 9,19) = 0.03346;
	pair_log(4, 9,20) = 0.00194;
	pair_log(4, 9,21) = 0.00000;
	pair_log(4, 9,22) = 0.00000;
	pair_log(4, 9,23) = 0.00000;
	pair_log(4, 9,24) = 0.00000;
	pair_log(4, 9,25) = 0.00000;
	pair_log(4, 9,26) = 0.00000;
	pair_log(4, 9,27) = 0.00000;
	pair_log(4, 9,28) = 0.00000;
	pair_log(4,10, 1) = -0.02847;
	pair_log(4,10, 2) = 0.00556;
	pair_log(4,10, 3) = 0.00746;
	pair_log(4,10, 4) = -0.02856;
	pair_log(4,10, 5) = -0.00421;
	pair_log(4,10, 6) = 0.03720;
	pair_log(4,10, 7) = -0.00183;
	pair_log(4,10, 8) = -0.01016;
	pair_log(4,10, 9) = -0.01861;
	pair_log(4,10,10) = -0.01409;
	pair_log(4,10,11) = 0.01546;
	pair_log(4,10,12) = 0.02276;
	pair_log(4,10,13) = 0.04662;
	pair_log(4,10,14) = -0.00516;
	pair_log(4,10,15) = 0.01866;
	pair_log(4,10,16) = -0.00136;
	pair_log(4,10,17) = 0.00533;
	pair_log(4,10,18) = -0.00721;
	pair_log(4,10,19) = 0.05414;
	pair_log(4,10,20) = 0.03288;
	pair_log(4,10,21) = 0.00000;
	pair_log(4,10,22) = 0.00000;
	pair_log(4,10,23) = 0.00000;
	pair_log(4,10,24) = 0.00000;
	pair_log(4,10,25) = 0.00000;
	pair_log(4,10,26) = 0.00000;
	pair_log(4,10,27) = 0.00000;
	pair_log(4,10,28) = 0.00000;
	pair_log(4,11, 1) = 0.00464;
	pair_log(4,11, 2) = 0.00328;
	pair_log(4,11, 3) = 0.02495;
	pair_log(4,11, 4) = 0.00847;
	pair_log(4,11, 5) = 0.00735;
	pair_log(4,11, 6) = 0.00033;
	pair_log(4,11, 7) = 0.03515;
	pair_log(4,11, 8) = -0.04361;
	pair_log(4,11, 9) = 0.03596;
	pair_log(4,11,10) = 0.01546;
	pair_log(4,11,11) = -0.08561;
	pair_log(4,11,12) = -0.00145;
	pair_log(4,11,13) = 0.02119;
	pair_log(4,11,14) = 0.01284;
	pair_log(4,11,15) = 0.04384;
	pair_log(4,11,16) = -0.02691;
	pair_log(4,11,17) = 0.00568;
	pair_log(4,11,18) = 0.03434;
	pair_log(4,11,19) = -0.01871;
	pair_log(4,11,20) = -0.01283;
	pair_log(4,11,21) = 0.00000;
	pair_log(4,11,22) = 0.00000;
	pair_log(4,11,23) = 0.00000;
	pair_log(4,11,24) = 0.00000;
	pair_log(4,11,25) = 0.00000;
	pair_log(4,11,26) = 0.00000;
	pair_log(4,11,27) = 0.00000;
	pair_log(4,11,28) = 0.00000;
	pair_log(4,12, 1) = 0.03854;
	pair_log(4,12, 2) = 0.00085;
	pair_log(4,12, 3) = -0.01565;
	pair_log(4,12, 4) = -0.01793;
	pair_log(4,12, 5) = 0.00709;
	pair_log(4,12, 6) = -0.00528;
	pair_log(4,12, 7) = -0.02616;
	pair_log(4,12, 8) = 0.02517;
	pair_log(4,12, 9) = -0.04689;
	pair_log(4,12,10) = 0.02276;
	pair_log(4,12,11) = -0.00145;
	pair_log(4,12,12) = -0.03011;
	pair_log(4,12,13) = -0.07512;
	pair_log(4,12,14) = -0.02505;
	pair_log(4,12,15) = -0.05011;
	pair_log(4,12,16) = -0.00988;
	pair_log(4,12,17) = 0.00571;
	pair_log(4,12,18) = 0.04989;
	pair_log(4,12,19) = -0.01820;
	pair_log(4,12,20) = -0.02262;
	pair_log(4,12,21) = 0.00000;
	pair_log(4,12,22) = 0.00000;
	pair_log(4,12,23) = 0.00000;
	pair_log(4,12,24) = 0.00000;
	pair_log(4,12,25) = 0.00000;
	pair_log(4,12,26) = 0.00000;
	pair_log(4,12,27) = 0.00000;
	pair_log(4,12,28) = 0.00000;
	pair_log(4,13, 1) = 0.02263;
	pair_log(4,13, 2) = -0.02198;
	pair_log(4,13, 3) = -0.08884;
	pair_log(4,13, 4) = -0.05330;
	pair_log(4,13, 5) = 0.05823;
	pair_log(4,13, 6) = -0.01507;
	pair_log(4,13, 7) = 0.02330;
	pair_log(4,13, 8) = 0.07910;
	pair_log(4,13, 9) = -0.06779;
	pair_log(4,13,10) = 0.04662;
	pair_log(4,13,11) = 0.02119;
	pair_log(4,13,12) = -0.07512;
	pair_log(4,13,13) = -0.08318;
	pair_log(4,13,14) = -0.06324;
	pair_log(4,13,15) = -0.01342;
	pair_log(4,13,16) = 0.00469;
	pair_log(4,13,17) = 0.01434;
	pair_log(4,13,18) = 0.06160;
	pair_log(4,13,19) = -0.01974;
	pair_log(4,13,20) = -0.04774;
	pair_log(4,13,21) = 0.00000;
	pair_log(4,13,22) = 0.00000;
	pair_log(4,13,23) = 0.00000;
	pair_log(4,13,24) = 0.00000;
	pair_log(4,13,25) = 0.00000;
	pair_log(4,13,26) = 0.00000;
	pair_log(4,13,27) = 0.00000;
	pair_log(4,13,28) = 0.00000;
	pair_log(4,14, 1) = 0.02023;
	pair_log(4,14, 2) = 0.02667;
	pair_log(4,14, 3) = -0.00434;
	pair_log(4,14, 4) = -0.02655;
	pair_log(4,14, 5) = 0.01941;
	pair_log(4,14, 6) = -0.02658;
	pair_log(4,14, 7) = -0.00106;
	pair_log(4,14, 8) = 0.04162;
	pair_log(4,14, 9) = 0.01306;
	pair_log(4,14,10) = -0.00516;
	pair_log(4,14,11) = 0.01284;
	pair_log(4,14,12) = -0.02505;
	pair_log(4,14,13) = -0.06324;
	pair_log(4,14,14) = -0.06601;
	pair_log(4,14,15) = -0.02214;
	pair_log(4,14,16) = 0.02246;
	pair_log(4,14,17) = 0.01648;
	pair_log(4,14,18) = 0.01771;
	pair_log(4,14,19) = 0.00147;
	pair_log(4,14,20) = -0.03331;
	pair_log(4,14,21) = 0.00000;
	pair_log(4,14,22) = 0.00000;
	pair_log(4,14,23) = 0.00000;
	pair_log(4,14,24) = 0.00000;
	pair_log(4,14,25) = 0.00000;
	pair_log(4,14,26) = 0.00000;
	pair_log(4,14,27) = 0.00000;
	pair_log(4,14,28) = 0.00000;
	pair_log(4,15, 1) = 0.00319;
	pair_log(4,15, 2) = -0.02161;
	pair_log(4,15, 3) = -0.04608;
	pair_log(4,15, 4) = -0.05432;
	pair_log(4,15, 5) = 0.00793;
	pair_log(4,15, 6) = 0.00933;
	pair_log(4,15, 7) = 0.02289;
	pair_log(4,15, 8) = 0.06128;
	pair_log(4,15, 9) = -0.03842;
	pair_log(4,15,10) = 0.01866;
	pair_log(4,15,11) = 0.04384;
	pair_log(4,15,12) = -0.05011;
	pair_log(4,15,13) = -0.01342;
	pair_log(4,15,14) = -0.02214;
	pair_log(4,15,15) = -0.07401;
	pair_log(4,15,16) = -0.00075;
	pair_log(4,15,17) = 0.03515;
	pair_log(4,15,18) = 0.03622;
	pair_log(4,15,19) = -0.03369;
	pair_log(4,15,20) = -0.04663;
	pair_log(4,15,21) = 0.00000;
	pair_log(4,15,22) = 0.00000;
	pair_log(4,15,23) = 0.00000;
	pair_log(4,15,24) = 0.00000;
	pair_log(4,15,25) = 0.00000;
	pair_log(4,15,26) = 0.00000;
	pair_log(4,15,27) = 0.00000;
	pair_log(4,15,28) = 0.00000;
}

void
SecondaryStructureEval::pair_log_initializer_section4b(
	FArray3D_float & pair_log
)
{
	// section 4b
	pair_log(4,16, 1) = 0.00275;
	pair_log(4,16, 2) = -0.01602;
	pair_log(4,16, 3) = 0.01674;
	pair_log(4,16, 4) = 0.04724;
	pair_log(4,16, 5) = -0.00236;
	pair_log(4,16, 6) = -0.02573;
	pair_log(4,16, 7) = 0.00237;
	pair_log(4,16, 8) = 0.00588;
	pair_log(4,16, 9) = 0.05413;
	pair_log(4,16,10) = -0.00136;
	pair_log(4,16,11) = -0.02691;
	pair_log(4,16,12) = -0.00988;
	pair_log(4,16,13) = 0.00469;
	pair_log(4,16,14) = 0.02246;
	pair_log(4,16,15) = -0.00075;
	pair_log(4,16,16) = -0.01442;
	pair_log(4,16,17) = 0.00670;
	pair_log(4,16,18) = -0.00012;
	pair_log(4,16,19) = -0.03103;
	pair_log(4,16,20) = -0.03088;
	pair_log(4,16,21) = 0.00000;
	pair_log(4,16,22) = 0.00000;
	pair_log(4,16,23) = 0.00000;
	pair_log(4,16,24) = 0.00000;
	pair_log(4,16,25) = 0.00000;
	pair_log(4,16,26) = 0.00000;
	pair_log(4,16,27) = 0.00000;
	pair_log(4,16,28) = 0.00000;
	pair_log(4,17, 1) = 0.00640;
	pair_log(4,17, 2) = -0.01331;
	pair_log(4,17, 3) = -0.01276;
	pair_log(4,17, 4) = -0.00250;
	pair_log(4,17, 5) = 0.01079;
	pair_log(4,17, 6) = -0.01628;
	pair_log(4,17, 7) = 0.01188;
	pair_log(4,17, 8) = -0.02153;
	pair_log(4,17, 9) = 0.02293;
	pair_log(4,17,10) = 0.00533;
	pair_log(4,17,11) = 0.00568;
	pair_log(4,17,12) = 0.00571;
	pair_log(4,17,13) = 0.01434;
	pair_log(4,17,14) = 0.01648;
	pair_log(4,17,15) = 0.03515;
	pair_log(4,17,16) = 0.00670;
	pair_log(4,17,17) = -0.00561;
	pair_log(4,17,18) = -0.01448;
	pair_log(4,17,19) = 0.08390;
	pair_log(4,17,20) = -0.02709;
	pair_log(4,17,21) = 0.00000;
	pair_log(4,17,22) = 0.00000;
	pair_log(4,17,23) = 0.00000;
	pair_log(4,17,24) = 0.00000;
	pair_log(4,17,25) = 0.00000;
	pair_log(4,17,26) = 0.00000;
	pair_log(4,17,27) = 0.00000;
	pair_log(4,17,28) = 0.00000;
	pair_log(4,18, 1) = -0.02005;
	pair_log(4,18, 2) = -0.01558;
	pair_log(4,18, 3) = 0.03410;
	pair_log(4,18, 4) = 0.04775;
	pair_log(4,18, 5) = -0.00626;
	pair_log(4,18, 6) = 0.00420;
	pair_log(4,18, 7) = 0.01640;
	pair_log(4,18, 8) = -0.04232;
	pair_log(4,18, 9) = 0.01947;
	pair_log(4,18,10) = -0.00721;
	pair_log(4,18,11) = 0.03434;
	pair_log(4,18,12) = 0.04989;
	pair_log(4,18,13) = 0.06160;
	pair_log(4,18,14) = 0.01771;
	pair_log(4,18,15) = 0.03622;
	pair_log(4,18,16) = -0.00012;
	pair_log(4,18,17) = -0.01448;
	pair_log(4,18,18) = -0.06684;
	pair_log(4,18,19) = 0.03785;
	pair_log(4,18,20) = 0.03352;
	pair_log(4,18,21) = 0.00000;
	pair_log(4,18,22) = 0.00000;
	pair_log(4,18,23) = 0.00000;
	pair_log(4,18,24) = 0.00000;
	pair_log(4,18,25) = 0.00000;
	pair_log(4,18,26) = 0.00000;
	pair_log(4,18,27) = 0.00000;
	pair_log(4,18,28) = 0.00000;
	pair_log(4,19, 1) = 0.01054;
	pair_log(4,19, 2) = -0.09886;
	pair_log(4,19, 3) = -0.03207;
	pair_log(4,19, 4) = -0.01159;
	pair_log(4,19, 5) = -0.02970;
	pair_log(4,19, 6) = -0.06240;
	pair_log(4,19, 7) = -0.05614;
	pair_log(4,19, 8) = 0.09042;
	pair_log(4,19, 9) = 0.03346;
	pair_log(4,19,10) = 0.05414;
	pair_log(4,19,11) = -0.01871;
	pair_log(4,19,12) = -0.01820;
	pair_log(4,19,13) = -0.01974;
	pair_log(4,19,14) = 0.00147;
	pair_log(4,19,15) = -0.03369;
	pair_log(4,19,16) = -0.03103;
	pair_log(4,19,17) = 0.08390;
	pair_log(4,19,18) = 0.03785;
	pair_log(4,19,19) = -0.11149;
	pair_log(4,19,20) = -0.01196;
	pair_log(4,19,21) = 0.00000;
	pair_log(4,19,22) = 0.00000;
	pair_log(4,19,23) = 0.00000;
	pair_log(4,19,24) = 0.00000;
	pair_log(4,19,25) = 0.00000;
	pair_log(4,19,26) = 0.00000;
	pair_log(4,19,27) = 0.00000;
	pair_log(4,19,28) = 0.00000;
	pair_log(4,20, 1) = 0.01209;
	pair_log(4,20, 2) = 0.03435;
	pair_log(4,20, 3) = -0.00868;
	pair_log(4,20, 4) = 0.00113;
	pair_log(4,20, 5) = 0.00165;
	pair_log(4,20, 6) = -0.01505;
	pair_log(4,20, 7) = -0.01576;
	pair_log(4,20, 8) = 0.07198;
	pair_log(4,20, 9) = 0.00194;
	pair_log(4,20,10) = 0.03288;
	pair_log(4,20,11) = -0.01283;
	pair_log(4,20,12) = -0.02262;
	pair_log(4,20,13) = -0.04774;
	pair_log(4,20,14) = -0.03331;
	pair_log(4,20,15) = -0.04663;
	pair_log(4,20,16) = -0.03088;
	pair_log(4,20,17) = -0.02709;
	pair_log(4,20,18) = 0.03352;
	pair_log(4,20,19) = -0.01196;
	pair_log(4,20,20) = -0.05834;
	pair_log(4,20,21) = 0.00000;
	pair_log(4,20,22) = 0.00000;
	pair_log(4,20,23) = 0.00000;
	pair_log(4,20,24) = 0.00000;
	pair_log(4,20,25) = 0.00000;
	pair_log(4,20,26) = 0.00000;
	pair_log(4,20,27) = 0.00000;
	pair_log(4,20,28) = 0.00000;
	pair_log(4,21, 1) = 0.00000;
	pair_log(4,21, 2) = 0.00000;
	pair_log(4,21, 3) = 0.00000;
	pair_log(4,21, 4) = 0.00000;
	pair_log(4,21, 5) = 0.00000;
	pair_log(4,21, 6) = 0.00000;
	pair_log(4,21, 7) = 0.00000;
	pair_log(4,21, 8) = 0.00000;
	pair_log(4,21, 9) = 0.00000;
	pair_log(4,21,10) = 0.00000;
	pair_log(4,21,11) = 0.00000;
	pair_log(4,21,12) = 0.00000;
	pair_log(4,21,13) = 0.00000;
	pair_log(4,21,14) = 0.00000;
	pair_log(4,21,15) = 0.00000;
	pair_log(4,21,16) = 0.00000;
	pair_log(4,21,17) = 0.00000;
	pair_log(4,21,18) = 0.00000;
	pair_log(4,21,19) = 0.00000;
	pair_log(4,21,20) = 0.00000;
	pair_log(4,21,21) = 0.00000;
	pair_log(4,21,22) = 0.00000;
	pair_log(4,21,23) = 0.00000;
	pair_log(4,21,24) = 0.00000;
	pair_log(4,21,25) = 0.00000;
	pair_log(4,21,26) = 0.00000;
	pair_log(4,21,27) = 0.00000;
	pair_log(4,21,28) = 0.00000;
	pair_log(4,22, 1) = 0.00000;
	pair_log(4,22, 2) = 0.00000;
	pair_log(4,22, 3) = 0.00000;
	pair_log(4,22, 4) = 0.00000;
	pair_log(4,22, 5) = 0.00000;
	pair_log(4,22, 6) = 0.00000;
	pair_log(4,22, 7) = 0.00000;
	pair_log(4,22, 8) = 0.00000;
	pair_log(4,22, 9) = 0.00000;
	pair_log(4,22,10) = 0.00000;
	pair_log(4,22,11) = 0.00000;
	pair_log(4,22,12) = 0.00000;
	pair_log(4,22,13) = 0.00000;
	pair_log(4,22,14) = 0.00000;
	pair_log(4,22,15) = 0.00000;
	pair_log(4,22,16) = 0.00000;
	pair_log(4,22,17) = 0.00000;
	pair_log(4,22,18) = 0.00000;
	pair_log(4,22,19) = 0.00000;
	pair_log(4,22,20) = 0.00000;
	pair_log(4,22,21) = 0.00000;
	pair_log(4,22,22) = 0.00000;
	pair_log(4,22,23) = 0.00000;
	pair_log(4,22,24) = 0.00000;
	pair_log(4,22,25) = 0.00000;
	pair_log(4,22,26) = 0.00000;
	pair_log(4,22,27) = 0.00000;
	pair_log(4,22,28) = 0.00000;
	pair_log(4,23, 1) = 0.00000;
	pair_log(4,23, 2) = 0.00000;
	pair_log(4,23, 3) = 0.00000;
	pair_log(4,23, 4) = 0.00000;
	pair_log(4,23, 5) = 0.00000;
	pair_log(4,23, 6) = 0.00000;
	pair_log(4,23, 7) = 0.00000;
	pair_log(4,23, 8) = 0.00000;
	pair_log(4,23, 9) = 0.00000;
	pair_log(4,23,10) = 0.00000;
	pair_log(4,23,11) = 0.00000;
	pair_log(4,23,12) = 0.00000;
	pair_log(4,23,13) = 0.00000;
	pair_log(4,23,14) = 0.00000;
	pair_log(4,23,15) = 0.00000;
	pair_log(4,23,16) = 0.00000;
	pair_log(4,23,17) = 0.00000;
	pair_log(4,23,18) = 0.00000;
	pair_log(4,23,19) = 0.00000;
	pair_log(4,23,20) = 0.00000;
	pair_log(4,23,21) = 0.00000;
	pair_log(4,23,22) = 0.00000;
	pair_log(4,23,23) = 0.00000;
	pair_log(4,23,24) = 0.00000;
	pair_log(4,23,25) = 0.00000;
	pair_log(4,23,26) = 0.00000;
	pair_log(4,23,27) = 0.00000;
	pair_log(4,23,28) = 0.00000;
	pair_log(4,24, 1) = 0.00000;
	pair_log(4,24, 2) = 0.00000;
	pair_log(4,24, 3) = 0.00000;
	pair_log(4,24, 4) = 0.00000;
	pair_log(4,24, 5) = 0.00000;
	pair_log(4,24, 6) = 0.00000;
	pair_log(4,24, 7) = 0.00000;
	pair_log(4,24, 8) = 0.00000;
	pair_log(4,24, 9) = 0.00000;
	pair_log(4,24,10) = 0.00000;
	pair_log(4,24,11) = 0.00000;
	pair_log(4,24,12) = 0.00000;
	pair_log(4,24,13) = 0.00000;
	pair_log(4,24,14) = 0.00000;
	pair_log(4,24,15) = 0.00000;
	pair_log(4,24,16) = 0.00000;
	pair_log(4,24,17) = 0.00000;
	pair_log(4,24,18) = 0.00000;
	pair_log(4,24,19) = 0.00000;
	pair_log(4,24,20) = 0.00000;
	pair_log(4,24,21) = 0.00000;
	pair_log(4,24,22) = 0.00000;
	pair_log(4,24,23) = 0.00000;
	pair_log(4,24,24) = 0.00000;
	pair_log(4,24,25) = 0.00000;
	pair_log(4,24,26) = 0.00000;
	pair_log(4,24,27) = 0.00000;
	pair_log(4,24,28) = 0.00000;
	pair_log(4,25, 1) = 0.00000;
	pair_log(4,25, 2) = 0.00000;
	pair_log(4,25, 3) = 0.00000;
	pair_log(4,25, 4) = 0.00000;
	pair_log(4,25, 5) = 0.00000;
	pair_log(4,25, 6) = 0.00000;
	pair_log(4,25, 7) = 0.00000;
	pair_log(4,25, 8) = 0.00000;
	pair_log(4,25, 9) = 0.00000;
	pair_log(4,25,10) = 0.00000;
	pair_log(4,25,11) = 0.00000;
	pair_log(4,25,12) = 0.00000;
	pair_log(4,25,13) = 0.00000;
	pair_log(4,25,14) = 0.00000;
	pair_log(4,25,15) = 0.00000;
	pair_log(4,25,16) = 0.00000;
	pair_log(4,25,17) = 0.00000;
	pair_log(4,25,18) = 0.00000;
	pair_log(4,25,19) = 0.00000;
	pair_log(4,25,20) = 0.00000;
	pair_log(4,25,21) = 0.00000;
	pair_log(4,25,22) = 0.00000;
	pair_log(4,25,23) = 0.00000;
	pair_log(4,25,24) = 0.00000;
	pair_log(4,25,25) = 0.00000;
	pair_log(4,25,26) = 0.00000;
	pair_log(4,25,27) = 0.00000;
	pair_log(4,25,28) = 0.00000;
	pair_log(4,26, 1) = 0.00000;
	pair_log(4,26, 2) = 0.00000;
	pair_log(4,26, 3) = 0.00000;
	pair_log(4,26, 4) = 0.00000;
	pair_log(4,26, 5) = 0.00000;
	pair_log(4,26, 6) = 0.00000;
	pair_log(4,26, 7) = 0.00000;
	pair_log(4,26, 8) = 0.00000;
	pair_log(4,26, 9) = 0.00000;
	pair_log(4,26,10) = 0.00000;
	pair_log(4,26,11) = 0.00000;
	pair_log(4,26,12) = 0.00000;
	pair_log(4,26,13) = 0.00000;
	pair_log(4,26,14) = 0.00000;
	pair_log(4,26,15) = 0.00000;
	pair_log(4,26,16) = 0.00000;
	pair_log(4,26,17) = 0.00000;
	pair_log(4,26,18) = 0.00000;
	pair_log(4,26,19) = 0.00000;
	pair_log(4,26,20) = 0.00000;
	pair_log(4,26,21) = 0.00000;
	pair_log(4,26,22) = 0.00000;
	pair_log(4,26,23) = 0.00000;
	pair_log(4,26,24) = 0.00000;
	pair_log(4,26,25) = 0.00000;
	pair_log(4,26,26) = 0.00000;
	pair_log(4,26,27) = 0.00000;
	pair_log(4,26,28) = 0.00000;
	pair_log(4,27, 1) = 0.00000;
	pair_log(4,27, 2) = 0.00000;
	pair_log(4,27, 3) = 0.00000;
	pair_log(4,27, 4) = 0.00000;
	pair_log(4,27, 5) = 0.00000;
	pair_log(4,27, 6) = 0.00000;
	pair_log(4,27, 7) = 0.00000;
	pair_log(4,27, 8) = 0.00000;
	pair_log(4,27, 9) = 0.00000;
	pair_log(4,27,10) = 0.00000;
	pair_log(4,27,11) = 0.00000;
	pair_log(4,27,12) = 0.00000;
	pair_log(4,27,13) = 0.00000;
	pair_log(4,27,14) = 0.00000;
	pair_log(4,27,15) = 0.00000;
	pair_log(4,27,16) = 0.00000;
	pair_log(4,27,17) = 0.00000;
	pair_log(4,27,18) = 0.00000;
	pair_log(4,27,19) = 0.00000;
	pair_log(4,27,20) = 0.00000;
	pair_log(4,27,21) = 0.00000;
	pair_log(4,27,22) = 0.00000;
	pair_log(4,27,23) = 0.00000;
	pair_log(4,27,24) = 0.00000;
	pair_log(4,27,25) = 0.00000;
	pair_log(4,27,26) = 0.00000;
	pair_log(4,27,27) = 0.00000;
	pair_log(4,27,28) = 0.00000;
	pair_log(4,28, 1) = 0.00000;
	pair_log(4,28, 2) = 0.00000;
	pair_log(4,28, 3) = 0.00000;
	pair_log(4,28, 4) = 0.00000;
	pair_log(4,28, 5) = 0.00000;
	pair_log(4,28, 6) = 0.00000;
	pair_log(4,28, 7) = 0.00000;
	pair_log(4,28, 8) = 0.00000;
	pair_log(4,28, 9) = 0.00000;
	pair_log(4,28,10) = 0.00000;
	pair_log(4,28,11) = 0.00000;
	pair_log(4,28,12) = 0.00000;
	pair_log(4,28,13) = 0.00000;
	pair_log(4,28,14) = 0.00000;
	pair_log(4,28,15) = 0.00000;
	pair_log(4,28,16) = 0.00000;
	pair_log(4,28,17) = 0.00000;
	pair_log(4,28,18) = 0.00000;
	pair_log(4,28,19) = 0.00000;
	pair_log(4,28,20) = 0.00000;
	pair_log(4,28,21) = 0.00000;
	pair_log(4,28,22) = 0.00000;
	pair_log(4,28,23) = 0.00000;
	pair_log(4,28,24) = 0.00000;
	pair_log(4,28,25) = 0.00000;
	pair_log(4,28,26) = 0.00000;
	pair_log(4,28,27) = 0.00000;
	pair_log(4,28,28) = 0.00000;
	//KMa phospho_ser
	for (int i=1;i<=29;i++) {
		pair_log(4,29,i) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(4,i,29) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(4,21,i) = pair_log(4,16,i);
	}
	for (int i=1;i<=29;i++) {
		pair_log(4,i,21) = pair_log(4,i,16);
	}
}

void
SecondaryStructureEval::pair_log_initializer_section5a(
	FArray3D_float & pair_log
)
{
	// section 5a
	pair_log(5, 1, 1) = 0.00347;
	pair_log(5, 1, 2) = -0.00180;
	pair_log(5, 1, 3) = -0.00414;
	pair_log(5, 1, 4) = -0.00406;
	pair_log(5, 1, 5) = 0.00212;
	pair_log(5, 1, 6) = -0.00106;
	pair_log(5, 1, 7) = -0.00321;
	pair_log(5, 1, 8) = 0.00746;
	pair_log(5, 1, 9) = -0.00368;
	pair_log(5, 1,10) = 0.00482;
	pair_log(5, 1,11) = 0.00177;
	pair_log(5, 1,12) = -0.00391;
	pair_log(5, 1,13) = -0.00258;
	pair_log(5, 1,14) = -0.00343;
	pair_log(5, 1,15) = -0.00335;
	pair_log(5, 1,16) = -0.00167;
	pair_log(5, 1,17) = -0.00116;
	pair_log(5, 1,18) = 0.00590;
	pair_log(5, 1,19) = -0.00193;
	pair_log(5, 1,20) = -0.00110;
	pair_log(5, 1,21) = 0.00000;
	pair_log(5, 1,22) = 0.00000;
	pair_log(5, 1,23) = 0.00000;
	pair_log(5, 1,24) = 0.00000;
	pair_log(5, 1,25) = 0.00000;
	pair_log(5, 1,26) = 0.00000;
	pair_log(5, 1,27) = 0.00000;
	pair_log(5, 1,28) = 0.00000;
	pair_log(5, 2, 1) = -0.00180;
	pair_log(5, 2, 2) = 0.10838;
	pair_log(5, 2, 3) = -0.01489;
	pair_log(5, 2, 4) = -0.01832;
	pair_log(5, 2, 5) = 0.01487;
	pair_log(5, 2, 6) = 0.00362;
	pair_log(5, 2, 7) = 0.00424;
	pair_log(5, 2, 8) = 0.00508;
	pair_log(5, 2, 9) = -0.01873;
	pair_log(5, 2,10) = 0.00648;
	pair_log(5, 2,11) = 0.00917;
	pair_log(5, 2,12) = -0.00797;
	pair_log(5, 2,13) = -0.00302;
	pair_log(5, 2,14) = -0.00991;
	pair_log(5, 2,15) = -0.01195;
	pair_log(5, 2,16) = -0.00104;
	pair_log(5, 2,17) = 0.00005;
	pair_log(5, 2,18) = 0.00933;
	pair_log(5, 2,19) = 0.01294;
	pair_log(5, 2,20) = 0.00260;
	pair_log(5, 2,21) = 0.00000;
	pair_log(5, 2,22) = 0.00000;
	pair_log(5, 2,23) = 0.00000;
	pair_log(5, 2,24) = 0.00000;
	pair_log(5, 2,25) = 0.00000;
	pair_log(5, 2,26) = 0.00000;
	pair_log(5, 2,27) = 0.00000;
	pair_log(5, 2,28) = 0.00000;
	pair_log(5, 3, 1) = -0.00414;
	pair_log(5, 3, 2) = -0.01489;
	pair_log(5, 3, 3) = 0.00786;
	pair_log(5, 3, 4) = 0.00727;
	pair_log(5, 3, 5) = -0.01138;
	pair_log(5, 3, 6) = 0.00242;
	pair_log(5, 3, 7) = 0.00685;
	pair_log(5, 3, 8) = -0.01451;
	pair_log(5, 3, 9) = 0.01640;
	pair_log(5, 3,10) = -0.01230;
	pair_log(5, 3,11) = -0.00738;
	pair_log(5, 3,12) = 0.00825;
	pair_log(5, 3,13) = 0.00591;
	pair_log(5, 3,14) = 0.00644;
	pair_log(5, 3,15) = 0.01600;
	pair_log(5, 3,16) = 0.00305;
	pair_log(5, 3,17) = 0.00170;
	pair_log(5, 3,18) = -0.01302;
	pair_log(5, 3,19) = -0.00341;
	pair_log(5, 3,20) = -0.00156;
	pair_log(5, 3,21) = 0.00000;
	pair_log(5, 3,22) = 0.00000;
	pair_log(5, 3,23) = 0.00000;
	pair_log(5, 3,24) = 0.00000;
	pair_log(5, 3,25) = 0.00000;
	pair_log(5, 3,26) = 0.00000;
	pair_log(5, 3,27) = 0.00000;
	pair_log(5, 3,28) = 0.00000;
	pair_log(5, 4, 1) = -0.00406;
	pair_log(5, 4, 2) = -0.01832;
	pair_log(5, 4, 3) = 0.00727;
	pair_log(5, 4, 4) = 0.00947;
	pair_log(5, 4, 5) = -0.01030;
	pair_log(5, 4, 6) = -0.00092;
	pair_log(5, 4, 7) = 0.00352;
	pair_log(5, 4, 8) = -0.01407;
	pair_log(5, 4, 9) = 0.01791;
	pair_log(5, 4,10) = -0.01041;
	pair_log(5, 4,11) = -0.00624;
	pair_log(5, 4,12) = 0.00681;
	pair_log(5, 4,13) = 0.00588;
	pair_log(5, 4,14) = 0.00728;
	pair_log(5, 4,15) = 0.01522;
	pair_log(5, 4,16) = 0.00227;
	pair_log(5, 4,17) = 0.00215;
	pair_log(5, 4,18) = -0.01250;
	pair_log(5, 4,19) = -0.00199;
	pair_log(5, 4,20) = -0.00221;
	pair_log(5, 4,21) = 0.00000;
	pair_log(5, 4,22) = 0.00000;
	pair_log(5, 4,23) = 0.00000;
	pair_log(5, 4,24) = 0.00000;
	pair_log(5, 4,25) = 0.00000;
	pair_log(5, 4,26) = 0.00000;
	pair_log(5, 4,27) = 0.00000;
	pair_log(5, 4,28) = 0.00000;
	pair_log(5, 5, 1) = 0.00212;
	pair_log(5, 5, 2) = 0.01487;
	pair_log(5, 5, 3) = -0.01138;
	pair_log(5, 5, 4) = -0.01030;
	pair_log(5, 5, 5) = 0.01272;
	pair_log(5, 5, 6) = -0.00378;
	pair_log(5, 5, 7) = -0.00222;
	pair_log(5, 5, 8) = 0.01376;
	pair_log(5, 5, 9) = -0.01045;
	pair_log(5, 5,10) = 0.01304;
	pair_log(5, 5,11) = 0.00980;
	pair_log(5, 5,12) = -0.00842;
	pair_log(5, 5,13) = -0.00807;
	pair_log(5, 5,14) = -0.00704;
	pair_log(5, 5,15) = -0.00902;
	pair_log(5, 5,16) = -0.00283;
	pair_log(5, 5,17) = -0.00460;
	pair_log(5, 5,18) = 0.01012;
	pair_log(5, 5,19) = 0.00939;
	pair_log(5, 5,20) = 0.00403;
	pair_log(5, 5,21) = 0.00000;
	pair_log(5, 5,22) = 0.00000;
	pair_log(5, 5,23) = 0.00000;
	pair_log(5, 5,24) = 0.00000;
	pair_log(5, 5,25) = 0.00000;
	pair_log(5, 5,26) = 0.00000;
	pair_log(5, 5,27) = 0.00000;
	pair_log(5, 5,28) = 0.00000;
	pair_log(5, 6, 1) = -0.00106;
	pair_log(5, 6, 2) = 0.00362;
	pair_log(5, 6, 3) = 0.00242;
	pair_log(5, 6, 4) = -0.00092;
	pair_log(5, 6, 5) = -0.00378;
	pair_log(5, 6, 6) = 0.00526;
	pair_log(5, 6, 7) = 0.00394;
	pair_log(5, 6, 8) = -0.00567;
	pair_log(5, 6, 9) = 0.00007;
	pair_log(5, 6,10) = -0.00659;
	pair_log(5, 6,11) = -0.00184;
	pair_log(5, 6,12) = 0.00340;
	pair_log(5, 6,13) = 0.00235;
	pair_log(5, 6,14) = 0.00242;
	pair_log(5, 6,15) = 0.00194;
	pair_log(5, 6,16) = 0.00315;
	pair_log(5, 6,17) = 0.00215;
	pair_log(5, 6,18) = -0.00366;
	pair_log(5, 6,19) = -0.00064;
	pair_log(5, 6,20) = -0.00113;
	pair_log(5, 6,21) = 0.00000;
	pair_log(5, 6,22) = 0.00000;
	pair_log(5, 6,23) = 0.00000;
	pair_log(5, 6,24) = 0.00000;
	pair_log(5, 6,25) = 0.00000;
	pair_log(5, 6,26) = 0.00000;
	pair_log(5, 6,27) = 0.00000;
	pair_log(5, 6,28) = 0.00000;
	pair_log(5, 7, 1) = -0.00321;
	pair_log(5, 7, 2) = 0.00424;
	pair_log(5, 7, 3) = 0.00685;
	pair_log(5, 7, 4) = 0.00352;
	pair_log(5, 7, 5) = -0.00222;
	pair_log(5, 7, 6) = 0.00394;
	pair_log(5, 7, 7) = 0.02024;
	pair_log(5, 7, 8) = -0.01233;
	pair_log(5, 7, 9) = -0.00289;
	pair_log(5, 7,10) = -0.00681;
	pair_log(5, 7,11) = -0.00257;
	pair_log(5, 7,12) = 0.00362;
	pair_log(5, 7,13) = 0.00167;
	pair_log(5, 7,14) = 0.00218;
	pair_log(5, 7,15) = -0.00007;
	pair_log(5, 7,16) = 0.00329;
	pair_log(5, 7,17) = 0.00212;
	pair_log(5, 7,18) = -0.00668;
	pair_log(5, 7,19) = 0.00617;
	pair_log(5, 7,20) = 0.00113;
	pair_log(5, 7,21) = 0.00000;
	pair_log(5, 7,22) = 0.00000;
	pair_log(5, 7,23) = 0.00000;
	pair_log(5, 7,24) = 0.00000;
	pair_log(5, 7,25) = 0.00000;
	pair_log(5, 7,26) = 0.00000;
	pair_log(5, 7,27) = 0.00000;
	pair_log(5, 7,28) = 0.00000;
	pair_log(5, 8, 1) = 0.00746;
	pair_log(5, 8, 2) = 0.00508;
	pair_log(5, 8, 3) = -0.01451;
	pair_log(5, 8, 4) = -0.01407;
	pair_log(5, 8, 5) = 0.01376;
	pair_log(5, 8, 6) = -0.00567;
	pair_log(5, 8, 7) = -0.01233;
	pair_log(5, 8, 8) = 0.02696;
	pair_log(5, 8, 9) = -0.01311;
	pair_log(5, 8,10) = 0.01896;
	pair_log(5, 8,11) = 0.00919;
	pair_log(5, 8,12) = -0.01338;
	pair_log(5, 8,13) = -0.01190;
	pair_log(5, 8,14) = -0.01197;
	pair_log(5, 8,15) = -0.01401;
	pair_log(5, 8,16) = -0.00742;
	pair_log(5, 8,17) = -0.00232;
	pair_log(5, 8,18) = 0.02164;
	pair_log(5, 8,19) = -0.00233;
	pair_log(5, 8,20) = -0.00035;
	pair_log(5, 8,21) = 0.00000;
	pair_log(5, 8,22) = 0.00000;
	pair_log(5, 8,23) = 0.00000;
	pair_log(5, 8,24) = 0.00000;
	pair_log(5, 8,25) = 0.00000;
	pair_log(5, 8,26) = 0.00000;
	pair_log(5, 8,27) = 0.00000;
	pair_log(5, 8,28) = 0.00000;
	pair_log(5, 9, 1) = -0.00368;
	pair_log(5, 9, 2) = -0.01873;
	pair_log(5, 9, 3) = 0.01640;
	pair_log(5, 9, 4) = 0.01791;
	pair_log(5, 9, 5) = -0.01045;
	pair_log(5, 9, 6) = 0.00007;
	pair_log(5, 9, 7) = -0.00289;
	pair_log(5, 9, 8) = -0.01311;
	pair_log(5, 9, 9) = 0.00994;
	pair_log(5, 9,10) = -0.01001;
	pair_log(5, 9,11) = -0.00853;
	pair_log(5, 9,12) = 0.00823;
	pair_log(5, 9,13) = 0.00579;
	pair_log(5, 9,14) = 0.00765;
	pair_log(5, 9,15) = 0.00416;
	pair_log(5, 9,16) = 0.00151;
	pair_log(5, 9,17) = 0.00127;
	pair_log(5, 9,18) = -0.01161;
	pair_log(5, 9,19) = -0.00623;
	pair_log(5, 9,20) = -0.00355;
	pair_log(5, 9,21) = 0.00000;
	pair_log(5, 9,22) = 0.00000;
	pair_log(5, 9,23) = 0.00000;
	pair_log(5, 9,24) = 0.00000;
	pair_log(5, 9,25) = 0.00000;
	pair_log(5, 9,26) = 0.00000;
	pair_log(5, 9,27) = 0.00000;
	pair_log(5, 9,28) = 0.00000;
	pair_log(5,10, 1) = 0.00482;
	pair_log(5,10, 2) = 0.00648;
	pair_log(5,10, 3) = -0.01230;
	pair_log(5,10, 4) = -0.01041;
	pair_log(5,10, 5) = 0.01304;
	pair_log(5,10, 6) = -0.00659;
	pair_log(5,10, 7) = -0.00681;
	pair_log(5,10, 8) = 0.01896;
	pair_log(5,10, 9) = -0.01001;
	pair_log(5,10,10) = 0.01700;
	pair_log(5,10,11) = 0.00508;
	pair_log(5,10,12) = -0.01113;
	pair_log(5,10,13) = -0.00832;
	pair_log(5,10,14) = -0.00870;
	pair_log(5,10,15) = -0.01049;
	pair_log(5,10,16) = -0.00567;
	pair_log(5,10,17) = -0.00380;
	pair_log(5,10,18) = 0.01432;
	pair_log(5,10,19) = 0.00152;
	pair_log(5,10,20) = 0.00204;
	pair_log(5,10,21) = 0.00000;
	pair_log(5,10,22) = 0.00000;
	pair_log(5,10,23) = 0.00000;
	pair_log(5,10,24) = 0.00000;
	pair_log(5,10,25) = 0.00000;
	pair_log(5,10,26) = 0.00000;
	pair_log(5,10,27) = 0.00000;
	pair_log(5,10,28) = 0.00000;
	pair_log(5,11, 1) = 0.00177;
	pair_log(5,11, 2) = 0.00917;
	pair_log(5,11, 3) = -0.00738;
	pair_log(5,11, 4) = -0.00624;
	pair_log(5,11, 5) = 0.00980;
	pair_log(5,11, 6) = -0.00184;
	pair_log(5,11, 7) = -0.00257;
	pair_log(5,11, 8) = 0.00919;
	pair_log(5,11, 9) = -0.00853;
	pair_log(5,11,10) = 0.00508;
	pair_log(5,11,11) = 0.00882;
	pair_log(5,11,12) = -0.00508;
	pair_log(5,11,13) = -0.00497;
	pair_log(5,11,14) = -0.00385;
	pair_log(5,11,15) = -0.00669;
	pair_log(5,11,16) = -0.00105;
	pair_log(5,11,17) = -0.00236;
	pair_log(5,11,18) = 0.00330;
	pair_log(5,11,19) = 0.00484;
	pair_log(5,11,20) = 0.00387;
	pair_log(5,11,21) = 0.00000;
	pair_log(5,11,22) = 0.00000;
	pair_log(5,11,23) = 0.00000;
	pair_log(5,11,24) = 0.00000;
	pair_log(5,11,25) = 0.00000;
	pair_log(5,11,26) = 0.00000;
	pair_log(5,11,27) = 0.00000;
	pair_log(5,11,28) = 0.00000;
	pair_log(5,12, 1) = -0.00391;
	pair_log(5,12, 2) = -0.00797;
	pair_log(5,12, 3) = 0.00825;
	pair_log(5,12, 4) = 0.00681;
	pair_log(5,12, 5) = -0.00842;
	pair_log(5,12, 6) = 0.00340;
	pair_log(5,12, 7) = 0.00362;
	pair_log(5,12, 8) = -0.01338;
	pair_log(5,12, 9) = 0.00823;
	pair_log(5,12,10) = -0.01113;
	pair_log(5,12,11) = -0.00508;
	pair_log(5,12,12) = 0.01002;
	pair_log(5,12,13) = 0.00660;
	pair_log(5,12,14) = 0.00689;
	pair_log(5,12,15) = 0.00739;
	pair_log(5,12,16) = 0.00464;
	pair_log(5,12,17) = 0.00245;
	pair_log(5,12,18) = -0.01170;
	pair_log(5,12,19) = -0.00263;
	pair_log(5,12,20) = -0.00175;
	pair_log(5,12,21) = 0.00000;
	pair_log(5,12,22) = 0.00000;
	pair_log(5,12,23) = 0.00000;
	pair_log(5,12,24) = 0.00000;
	pair_log(5,12,25) = 0.00000;
	pair_log(5,12,26) = 0.00000;
	pair_log(5,12,27) = 0.00000;
	pair_log(5,12,28) = 0.00000;
	pair_log(5,13, 1) = -0.00258;
	pair_log(5,13, 2) = -0.00302;
	pair_log(5,13, 3) = 0.00591;
	pair_log(5,13, 4) = 0.00588;
	pair_log(5,13, 5) = -0.00807;
	pair_log(5,13, 6) = 0.00235;
	pair_log(5,13, 7) = 0.00167;
	pair_log(5,13, 8) = -0.01190;
	pair_log(5,13, 9) = 0.00579;
	pair_log(5,13,10) = -0.00832;
	pair_log(5,13,11) = -0.00497;
	pair_log(5,13,12) = 0.00660;
	pair_log(5,13,13) = 0.00713;
	pair_log(5,13,14) = 0.00606;
	pair_log(5,13,15) = 0.00549;
	pair_log(5,13,16) = 0.00189;
	pair_log(5,13,17) = 0.00050;
	pair_log(5,13,18) = -0.00972;
	pair_log(5,13,19) = 0.00255;
	pair_log(5,13,20) = 0.00376;
	pair_log(5,13,21) = 0.00000;
	pair_log(5,13,22) = 0.00000;
	pair_log(5,13,23) = 0.00000;
	pair_log(5,13,24) = 0.00000;
	pair_log(5,13,25) = 0.00000;
	pair_log(5,13,26) = 0.00000;
	pair_log(5,13,27) = 0.00000;
	pair_log(5,13,28) = 0.00000;
	pair_log(5,14, 1) = -0.00343;
	pair_log(5,14, 2) = -0.00991;
	pair_log(5,14, 3) = 0.00644;
	pair_log(5,14, 4) = 0.00728;
	pair_log(5,14, 5) = -0.00704;
	pair_log(5,14, 6) = 0.00242;
	pair_log(5,14, 7) = 0.00218;
	pair_log(5,14, 8) = -0.01197;
	pair_log(5,14, 9) = 0.00765;
	pair_log(5,14,10) = -0.00870;
	pair_log(5,14,11) = -0.00385;
	pair_log(5,14,12) = 0.00689;
	pair_log(5,14,13) = 0.00606;
	pair_log(5,14,14) = 0.00745;
	pair_log(5,14,15) = 0.00609;
	pair_log(5,14,16) = 0.00264;
	pair_log(5,14,17) = 0.00192;
	pair_log(5,14,18) = -0.00952;
	pair_log(5,14,19) = -0.00050;
	pair_log(5,14,20) = -0.00011;
	pair_log(5,14,21) = 0.00000;
	pair_log(5,14,22) = 0.00000;
	pair_log(5,14,23) = 0.00000;
	pair_log(5,14,24) = 0.00000;
	pair_log(5,14,25) = 0.00000;
	pair_log(5,14,26) = 0.00000;
	pair_log(5,14,27) = 0.00000;
	pair_log(5,14,28) = 0.00000;
	pair_log(5,15, 1) = -0.00335;
	pair_log(5,15, 2) = -0.01195;
	pair_log(5,15, 3) = 0.01600;
	pair_log(5,15, 4) = 0.01522;
	pair_log(5,15, 5) = -0.00902;
	pair_log(5,15, 6) = 0.00194;
	pair_log(5,15, 7) = -0.00007;
	pair_log(5,15, 8) = -0.01401;
	pair_log(5,15, 9) = 0.00416;
	pair_log(5,15,10) = -0.01049;
	pair_log(5,15,11) = -0.00669;
	pair_log(5,15,12) = 0.00739;
	pair_log(5,15,13) = 0.00549;
	pair_log(5,15,14) = 0.00609;
	pair_log(5,15,15) = 0.00407;
	pair_log(5,15,16) = 0.00364;
	pair_log(5,15,17) = 0.00080;
	pair_log(5,15,18) = -0.01208;
	pair_log(5,15,19) = -0.00135;
	pair_log(5,15,20) = -0.00062;
	pair_log(5,15,21) = 0.00000;
	pair_log(5,15,22) = 0.00000;
	pair_log(5,15,23) = 0.00000;
	pair_log(5,15,24) = 0.00000;
	pair_log(5,15,25) = 0.00000;
	pair_log(5,15,26) = 0.00000;
	pair_log(5,15,27) = 0.00000;
	pair_log(5,15,28) = 0.00000;
}

void
SecondaryStructureEval::pair_log_initializer_section5b(
	FArray3D_float & pair_log
)
{
	// section 5b
	pair_log(5,16, 1) = -0.00167;
	pair_log(5,16, 2) = -0.00104;
	pair_log(5,16, 3) = 0.00305;
	pair_log(5,16, 4) = 0.00227;
	pair_log(5,16, 5) = -0.00283;
	pair_log(5,16, 6) = 0.00315;
	pair_log(5,16, 7) = 0.00329;
	pair_log(5,16, 8) = -0.00742;
	pair_log(5,16, 9) = 0.00151;
	pair_log(5,16,10) = -0.00567;
	pair_log(5,16,11) = -0.00105;
	pair_log(5,16,12) = 0.00464;
	pair_log(5,16,13) = 0.00189;
	pair_log(5,16,14) = 0.00264;
	pair_log(5,16,15) = 0.00364;
	pair_log(5,16,16) = 0.00249;
	pair_log(5,16,17) = 0.00188;
	pair_log(5,16,18) = -0.00522;
	pair_log(5,16,19) = -0.00112;
	pair_log(5,16,20) = -0.00071;
	pair_log(5,16,21) = 0.00000;
	pair_log(5,16,22) = 0.00000;
	pair_log(5,16,23) = 0.00000;
	pair_log(5,16,24) = 0.00000;
	pair_log(5,16,25) = 0.00000;
	pair_log(5,16,26) = 0.00000;
	pair_log(5,16,27) = 0.00000;
	pair_log(5,16,28) = 0.00000;
	pair_log(5,17, 1) = -0.00116;
	pair_log(5,17, 2) = 0.00005;
	pair_log(5,17, 3) = 0.00170;
	pair_log(5,17, 4) = 0.00215;
	pair_log(5,17, 5) = -0.00460;
	pair_log(5,17, 6) = 0.00215;
	pair_log(5,17, 7) = 0.00212;
	pair_log(5,17, 8) = -0.00232;
	pair_log(5,17, 9) = 0.00127;
	pair_log(5,17,10) = -0.00380;
	pair_log(5,17,11) = -0.00236;
	pair_log(5,17,12) = 0.00245;
	pair_log(5,17,13) = 0.00050;
	pair_log(5,17,14) = 0.00192;
	pair_log(5,17,15) = 0.00080;
	pair_log(5,17,16) = 0.00188;
	pair_log(5,17,17) = 0.00185;
	pair_log(5,17,18) = -0.00176;
	pair_log(5,17,19) = -0.00473;
	pair_log(5,17,20) = -0.00155;
	pair_log(5,17,21) = 0.00000;
	pair_log(5,17,22) = 0.00000;
	pair_log(5,17,23) = 0.00000;
	pair_log(5,17,24) = 0.00000;
	pair_log(5,17,25) = 0.00000;
	pair_log(5,17,26) = 0.00000;
	pair_log(5,17,27) = 0.00000;
	pair_log(5,17,28) = 0.00000;
	pair_log(5,18, 1) = 0.00590;
	pair_log(5,18, 2) = 0.00933;
	pair_log(5,18, 3) = -0.01302;
	pair_log(5,18, 4) = -0.01250;
	pair_log(5,18, 5) = 0.01012;
	pair_log(5,18, 6) = -0.00366;
	pair_log(5,18, 7) = -0.00668;
	pair_log(5,18, 8) = 0.02164;
	pair_log(5,18, 9) = -0.01161;
	pair_log(5,18,10) = 0.01432;
	pair_log(5,18,11) = 0.00330;
	pair_log(5,18,12) = -0.01170;
	pair_log(5,18,13) = -0.00972;
	pair_log(5,18,14) = -0.00952;
	pair_log(5,18,15) = -0.01208;
	pair_log(5,18,16) = -0.00522;
	pair_log(5,18,17) = -0.00176;
	pair_log(5,18,18) = 0.02080;
	pair_log(5,18,19) = -0.00082;
	pair_log(5,18,20) = -0.00012;
	pair_log(5,18,21) = 0.00000;
	pair_log(5,18,22) = 0.00000;
	pair_log(5,18,23) = 0.00000;
	pair_log(5,18,24) = 0.00000;
	pair_log(5,18,25) = 0.00000;
	pair_log(5,18,26) = 0.00000;
	pair_log(5,18,27) = 0.00000;
	pair_log(5,18,28) = 0.00000;
	pair_log(5,19, 1) = -0.00193;
	pair_log(5,19, 2) = 0.01294;
	pair_log(5,19, 3) = -0.00341;
	pair_log(5,19, 4) = -0.00199;
	pair_log(5,19, 5) = 0.00939;
	pair_log(5,19, 6) = -0.00064;
	pair_log(5,19, 7) = 0.00617;
	pair_log(5,19, 8) = -0.00233;
	pair_log(5,19, 9) = -0.00623;
	pair_log(5,19,10) = 0.00152;
	pair_log(5,19,11) = 0.00484;
	pair_log(5,19,12) = -0.00263;
	pair_log(5,19,13) = 0.00255;
	pair_log(5,19,14) = -0.00050;
	pair_log(5,19,15) = -0.00135;
	pair_log(5,19,16) = -0.00112;
	pair_log(5,19,17) = -0.00473;
	pair_log(5,19,18) = -0.00082;
	pair_log(5,19,19) = 0.00844;
	pair_log(5,19,20) = 0.00508;
	pair_log(5,19,21) = 0.00000;
	pair_log(5,19,22) = 0.00000;
	pair_log(5,19,23) = 0.00000;
	pair_log(5,19,24) = 0.00000;
	pair_log(5,19,25) = 0.00000;
	pair_log(5,19,26) = 0.00000;
	pair_log(5,19,27) = 0.00000;
	pair_log(5,19,28) = 0.00000;
	pair_log(5,20, 1) = -0.00110;
	pair_log(5,20, 2) = 0.00260;
	pair_log(5,20, 3) = -0.00156;
	pair_log(5,20, 4) = -0.00221;
	pair_log(5,20, 5) = 0.00403;
	pair_log(5,20, 6) = -0.00113;
	pair_log(5,20, 7) = 0.00113;
	pair_log(5,20, 8) = -0.00035;
	pair_log(5,20, 9) = -0.00355;
	pair_log(5,20,10) = 0.00204;
	pair_log(5,20,11) = 0.00387;
	pair_log(5,20,12) = -0.00175;
	pair_log(5,20,13) = 0.00376;
	pair_log(5,20,14) = -0.00011;
	pair_log(5,20,15) = -0.00062;
	pair_log(5,20,16) = -0.00071;
	pair_log(5,20,17) = -0.00155;
	pair_log(5,20,18) = -0.00012;
	pair_log(5,20,19) = 0.00508;
	pair_log(5,20,20) = 0.00498;
	pair_log(5,20,21) = 0.00000;
	pair_log(5,20,22) = 0.00000;
	pair_log(5,20,23) = 0.00000;
	pair_log(5,20,24) = 0.00000;
	pair_log(5,20,25) = 0.00000;
	pair_log(5,20,26) = 0.00000;
	pair_log(5,20,27) = 0.00000;
	pair_log(5,20,28) = 0.00000;
	pair_log(5,21, 1) = 0.00000;
	pair_log(5,21, 2) = 0.00000;
	pair_log(5,21, 3) = 0.00000;
	pair_log(5,21, 4) = 0.00000;
	pair_log(5,21, 5) = 0.00000;
	pair_log(5,21, 6) = 0.00000;
	pair_log(5,21, 7) = 0.00000;
	pair_log(5,21, 8) = 0.00000;
	pair_log(5,21, 9) = 0.00000;
	pair_log(5,21,10) = 0.00000;
	pair_log(5,21,11) = 0.00000;
	pair_log(5,21,12) = 0.00000;
	pair_log(5,21,13) = 0.00000;
	pair_log(5,21,14) = 0.00000;
	pair_log(5,21,15) = 0.00000;
	pair_log(5,21,16) = 0.00000;
	pair_log(5,21,17) = 0.00000;
	pair_log(5,21,18) = 0.00000;
	pair_log(5,21,19) = 0.00000;
	pair_log(5,21,20) = 0.00000;
	pair_log(5,21,21) = 0.00000;
	pair_log(5,21,22) = 0.00000;
	pair_log(5,21,23) = 0.00000;
	pair_log(5,21,24) = 0.00000;
	pair_log(5,21,25) = 0.00000;
	pair_log(5,21,26) = 0.00000;
	pair_log(5,21,27) = 0.00000;
	pair_log(5,21,28) = 0.00000;
	pair_log(5,22, 1) = 0.00000;
	pair_log(5,22, 2) = 0.00000;
	pair_log(5,22, 3) = 0.00000;
	pair_log(5,22, 4) = 0.00000;
	pair_log(5,22, 5) = 0.00000;
	pair_log(5,22, 6) = 0.00000;
	pair_log(5,22, 7) = 0.00000;
	pair_log(5,22, 8) = 0.00000;
	pair_log(5,22, 9) = 0.00000;
	pair_log(5,22,10) = 0.00000;
	pair_log(5,22,11) = 0.00000;
	pair_log(5,22,12) = 0.00000;
	pair_log(5,22,13) = 0.00000;
	pair_log(5,22,14) = 0.00000;
	pair_log(5,22,15) = 0.00000;
	pair_log(5,22,16) = 0.00000;
	pair_log(5,22,17) = 0.00000;
	pair_log(5,22,18) = 0.00000;
	pair_log(5,22,19) = 0.00000;
	pair_log(5,22,20) = 0.00000;
	pair_log(5,22,21) = 0.00000;
	pair_log(5,22,22) = 0.00000;
	pair_log(5,22,23) = 0.00000;
	pair_log(5,22,24) = 0.00000;
	pair_log(5,22,25) = 0.00000;
	pair_log(5,22,26) = 0.00000;
	pair_log(5,22,27) = 0.00000;
	pair_log(5,22,28) = 0.00000;
	pair_log(5,23, 1) = 0.00000;
	pair_log(5,23, 2) = 0.00000;
	pair_log(5,23, 3) = 0.00000;
	pair_log(5,23, 4) = 0.00000;
	pair_log(5,23, 5) = 0.00000;
	pair_log(5,23, 6) = 0.00000;
	pair_log(5,23, 7) = 0.00000;
	pair_log(5,23, 8) = 0.00000;
	pair_log(5,23, 9) = 0.00000;
	pair_log(5,23,10) = 0.00000;
	pair_log(5,23,11) = 0.00000;
	pair_log(5,23,12) = 0.00000;
	pair_log(5,23,13) = 0.00000;
	pair_log(5,23,14) = 0.00000;
	pair_log(5,23,15) = 0.00000;
	pair_log(5,23,16) = 0.00000;
	pair_log(5,23,17) = 0.00000;
	pair_log(5,23,18) = 0.00000;
	pair_log(5,23,19) = 0.00000;
	pair_log(5,23,20) = 0.00000;
	pair_log(5,23,21) = 0.00000;
	pair_log(5,23,22) = 0.00000;
	pair_log(5,23,23) = 0.00000;
	pair_log(5,23,24) = 0.00000;
	pair_log(5,23,25) = 0.00000;
	pair_log(5,23,26) = 0.00000;
	pair_log(5,23,27) = 0.00000;
	pair_log(5,23,28) = 0.00000;
	pair_log(5,24, 1) = 0.00000;
	pair_log(5,24, 2) = 0.00000;
	pair_log(5,24, 3) = 0.00000;
	pair_log(5,24, 4) = 0.00000;
	pair_log(5,24, 5) = 0.00000;
	pair_log(5,24, 6) = 0.00000;
	pair_log(5,24, 7) = 0.00000;
	pair_log(5,24, 8) = 0.00000;
	pair_log(5,24, 9) = 0.00000;
	pair_log(5,24,10) = 0.00000;
	pair_log(5,24,11) = 0.00000;
	pair_log(5,24,12) = 0.00000;
	pair_log(5,24,13) = 0.00000;
	pair_log(5,24,14) = 0.00000;
	pair_log(5,24,15) = 0.00000;
	pair_log(5,24,16) = 0.00000;
	pair_log(5,24,17) = 0.00000;
	pair_log(5,24,18) = 0.00000;
	pair_log(5,24,19) = 0.00000;
	pair_log(5,24,20) = 0.00000;
	pair_log(5,24,21) = 0.00000;
	pair_log(5,24,22) = 0.00000;
	pair_log(5,24,23) = 0.00000;
	pair_log(5,24,24) = 0.00000;
	pair_log(5,24,25) = 0.00000;
	pair_log(5,24,26) = 0.00000;
	pair_log(5,24,27) = 0.00000;
	pair_log(5,24,28) = 0.00000;
	pair_log(5,25, 1) = 0.00000;
	pair_log(5,25, 2) = 0.00000;
	pair_log(5,25, 3) = 0.00000;
	pair_log(5,25, 4) = 0.00000;
	pair_log(5,25, 5) = 0.00000;
	pair_log(5,25, 6) = 0.00000;
	pair_log(5,25, 7) = 0.00000;
	pair_log(5,25, 8) = 0.00000;
	pair_log(5,25, 9) = 0.00000;
	pair_log(5,25,10) = 0.00000;
	pair_log(5,25,11) = 0.00000;
	pair_log(5,25,12) = 0.00000;
	pair_log(5,25,13) = 0.00000;
	pair_log(5,25,14) = 0.00000;
	pair_log(5,25,15) = 0.00000;
	pair_log(5,25,16) = 0.00000;
	pair_log(5,25,17) = 0.00000;
	pair_log(5,25,18) = 0.00000;
	pair_log(5,25,19) = 0.00000;
	pair_log(5,25,20) = 0.00000;
	pair_log(5,25,21) = 0.00000;
	pair_log(5,25,22) = 0.00000;
	pair_log(5,25,23) = 0.00000;
	pair_log(5,25,24) = 0.00000;
	pair_log(5,25,25) = 0.00000;
	pair_log(5,25,26) = 0.00000;
	pair_log(5,25,27) = 0.00000;
	pair_log(5,25,28) = 0.00000;
	pair_log(5,26, 1) = 0.00000;
	pair_log(5,26, 2) = 0.00000;
	pair_log(5,26, 3) = 0.00000;
	pair_log(5,26, 4) = 0.00000;
	pair_log(5,26, 5) = 0.00000;
	pair_log(5,26, 6) = 0.00000;
	pair_log(5,26, 7) = 0.00000;
	pair_log(5,26, 8) = 0.00000;
	pair_log(5,26, 9) = 0.00000;
	pair_log(5,26,10) = 0.00000;
	pair_log(5,26,11) = 0.00000;
	pair_log(5,26,12) = 0.00000;
	pair_log(5,26,13) = 0.00000;
	pair_log(5,26,14) = 0.00000;
	pair_log(5,26,15) = 0.00000;
	pair_log(5,26,16) = 0.00000;
	pair_log(5,26,17) = 0.00000;
	pair_log(5,26,18) = 0.00000;
	pair_log(5,26,19) = 0.00000;
	pair_log(5,26,20) = 0.00000;
	pair_log(5,26,21) = 0.00000;
	pair_log(5,26,22) = 0.00000;
	pair_log(5,26,23) = 0.00000;
	pair_log(5,26,24) = 0.00000;
	pair_log(5,26,25) = 0.00000;
	pair_log(5,26,26) = 0.00000;
	pair_log(5,26,27) = 0.00000;
	pair_log(5,26,28) = 0.00000;
	pair_log(5,27, 1) = 0.00000;
	pair_log(5,27, 2) = 0.00000;
	pair_log(5,27, 3) = 0.00000;
	pair_log(5,27, 4) = 0.00000;
	pair_log(5,27, 5) = 0.00000;
	pair_log(5,27, 6) = 0.00000;
	pair_log(5,27, 7) = 0.00000;
	pair_log(5,27, 8) = 0.00000;
	pair_log(5,27, 9) = 0.00000;
	pair_log(5,27,10) = 0.00000;
	pair_log(5,27,11) = 0.00000;
	pair_log(5,27,12) = 0.00000;
	pair_log(5,27,13) = 0.00000;
	pair_log(5,27,14) = 0.00000;
	pair_log(5,27,15) = 0.00000;
	pair_log(5,27,16) = 0.00000;
	pair_log(5,27,17) = 0.00000;
	pair_log(5,27,18) = 0.00000;
	pair_log(5,27,19) = 0.00000;
	pair_log(5,27,20) = 0.00000;
	pair_log(5,27,21) = 0.00000;
	pair_log(5,27,22) = 0.00000;
	pair_log(5,27,23) = 0.00000;
	pair_log(5,27,24) = 0.00000;
	pair_log(5,27,25) = 0.00000;
	pair_log(5,27,26) = 0.00000;
	pair_log(5,27,27) = 0.00000;
	pair_log(5,27,28) = 0.00000;
	pair_log(5,28, 1) = 0.00000;
	pair_log(5,28, 2) = 0.00000;
	pair_log(5,28, 3) = 0.00000;
	pair_log(5,28, 4) = 0.00000;
	pair_log(5,28, 5) = 0.00000;
	pair_log(5,28, 6) = 0.00000;
	pair_log(5,28, 7) = 0.00000;
	pair_log(5,28, 8) = 0.00000;
	pair_log(5,28, 9) = 0.00000;
	pair_log(5,28,10) = 0.00000;
	pair_log(5,28,11) = 0.00000;
	pair_log(5,28,12) = 0.00000;
	pair_log(5,28,13) = 0.00000;
	pair_log(5,28,14) = 0.00000;
	pair_log(5,28,15) = 0.00000;
	pair_log(5,28,16) = 0.00000;
	pair_log(5,28,17) = 0.00000;
	pair_log(5,28,18) = 0.00000;
	pair_log(5,28,19) = 0.00000;
	pair_log(5,28,20) = 0.00000;
	pair_log(5,28,21) = 0.00000;
	pair_log(5,28,22) = 0.00000;
	pair_log(5,28,23) = 0.00000;
	pair_log(5,28,24) = 0.00000;
	pair_log(5,28,25) = 0.00000;
	pair_log(5,28,26) = 0.00000;
	pair_log(5,28,27) = 0.00000;
	pair_log(5,28,28) = 0.00000;
	//KMa phospho_ser
	for (int i=1;i<=29;i++) {
		pair_log(5,29,i) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(5,i,29) = 0.00000;
	}
	for (int i=1;i<=29;i++) {
		pair_log(5,21,i) = pair_log(5,16,i);
	}
	for (int i=1;i<=29;i++) {
		pair_log(5,i,21) = pair_log(5,i,16);
	}
}

//////////////////////
/// Private: Setup ///
//////////////////////
bool SecondaryStructureEval::cache_initialized = false; // static
FArray1D_int SecondaryStructureEval::cached_iptsn = FArray1D_int( 36 ); // static
FArray4D_float SecondaryStructureEval::cached_pts = FArray4D_float( 2, 3, 36, 36 ); // static
FArray1D_float SecondaryStructureEval::cached_ds = FArray1D_float( 6 ); // static

// struct_catalog
FArray1D_int const SecondaryStructureEval::idsn = FArray1D_int( 6, idsn_initializer );
FArray1D_int const SecondaryStructureEval::ids = FArray1D_int( 6, ids_initializer );
FArray2D_float const SecondaryStructureEval::ssdist = FArray2D_float( 4, 2, ssdist_initializer );
FArray1D_float const SecondaryStructureEval::cbeta_den6 = FArray1D_float( 45, cbeta_den6_initializer );
FArray1D_float const SecondaryStructureEval::cbeta_den12 = FArray1D_float( 45, cbeta_den12_initializer );
FArray1D_float const SecondaryStructureEval::hs_dp = FArray1D_float( 10, hs_dp_initializer ); // not in use?
FArray4D_float const SecondaryStructureEval::rsigma_dot = FArray4D_float( 12, 18, 2, 2, rsigma_dot_initializer ); // lookup for new rsigma score
FArray1D_float const SecondaryStructureEval::centroid_pack_correction = FArray1D_float( 120, centroid_pack_correction_initializer );

// seq_struct
FArray2D_float const SecondaryStructureEval::env_log = FArray2D_float( 29, 40, env_log_initializer_master );
FArray3D_float const SecondaryStructureEval::pair_log = FArray3D_float( 5, 29, 29, pair_log_initializer_master );

void
SecondaryStructureEval::initialize()
{
	// struct_catalog
	if (!cache_initialized) {
		load_and_cache_phi_theta_bins_from_file(); // failsafe: this is actually done in job_distributor.cc to save time
	}
	iptsn = FArray1D_int( cached_iptsn );
	pts = FArray4D_float( cached_pts );
	ds = FArray1D_float( cached_ds );

	// ss weights
	set_default_parallel_wt(1.0);
	set_default_antiparallel_wt(1.0);
	setup_SS_default_weight(); // set this LAST, after default parallel/antiparallel weights are set

	// rsd weights
	setup_rsd_default_weight();
	set_rsd_wt_helix(1.0);
	set_rsd_wt_loop(1.0);
	set_rsd_wt_strand(1.0);

	// additional
	set_cb_weight(1.0);

	set_new_centroid_packing(false);
	set_centroid_packing_weight(2.0);

	set_localstrandpair_penalty(0.0);

	set_seq_sep_scale(20.0);

	set_stretch_strand_dist_cutoff(false);
	set_max_strand_dist_cutoff(12.0);
	set_strand_dist_cutoff(6.5);

	set_handedness_score_flag(false);

	set_ignore_sspair_barcode_in_score(false);

	set_separate_centroidpackscore(false);

	// HomologStats
	homolog_stats = NULL; // no object by default
}

////////////////////////////////////
/// Private: Inner Class Methods ///
////////////////////////////////////
/* Strands */
SecondaryStructureEval::Strands::Strands() {
	using namespace param;

	SS_resnum = FArray1D_int( MAX_RES() ); // residue number of strand dimer i
	SS_strand = FArray1D_int( MAX_RES() ); // strand number containing SS_dimer i
	SS_strand_end = FArray2D_int( 2, MAX_RES() ); // residue number of first non-E res
	dimer_neighbor = FArray2D_int( 2, MAX_RES() );

	ss_orient = FArray2D_int( MAX_RES(), MAX_RES() ); // does dimer pair i,j interact? 1=yes
}
SecondaryStructureEval::Strands::~Strands() {}

/* Helices */
SecondaryStructureEval::Helices::Helices() {
	using namespace param;

	HH_resnum = FArray1D_int( MAX_RES() );
	HH_helix_end = FArray2D_int( 2, MAX_RES() );
}
SecondaryStructureEval::Helices::~Helices() {}

/* HomologStats */
FArray1D_float const HomologStats::pdb_cen10_prob = FArray1D_float( 30, homolog_env_initializer ); // static
FArray2D_float const HomologStats::pdb_pair_prob = FArray2D_float( 200, 5, homolog_pair_initializer ); // static

HomologStats::HomologStats() {}
HomologStats::~HomologStats() {}

HomologStats::HomologStats(SecondaryStructureEval * sse) {
	set_parent_sse_object(sse);
	initialize();
}

void
HomologStats::initialize()
{
	int const & total_residue = parent_sse_object_->get_total_residue(); // yab: misc removal

	homolog_cen10_log.dimension( total_residue, 30, 0.0 );
	homolog_cen10_log = 0.0;

	homolog_pair_log.dimension( 5, total_residue, total_residue, 0.0 );
	homolog_pair_log  = 0.0;
}

void
HomologStats::read_homolog_env_stats_file(
	std::string homolog_env_filename
)
{
	if ( homolog_env_filename != "-" ) {
		std::cout << "Reading homolog_env_stats file: " << homolog_env_filename << std::endl;
	}

	utility::io::izstream envout( homolog_env_filename.c_str() );

	if ( !envout ) {
		std::cout << "Open failed for file: " << homolog_env_filename << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	std::string line;
	int res_num, bin;
	float env;

	getline( envout, line );

  	while( getline( envout, line) ) {
		std::istringstream line_str( line );
		line_str >> res_num >> bin >> env;
		homolog_cen10_log( res_num, bin ) = env;
	}

	std::cout << "end of reading " << std::endl;
	envout.close();
	envout.clear();
}

void
HomologStats::read_homolog_pair_stats_file(
	std::string homolog_pair_filename
)
{

	if ( homolog_pair_filename != "-" )
		std::cout << "Reading homolog_pair_stats file: " <<  homolog_pair_filename << std::endl;

	utility::io::izstream pairout( homolog_pair_filename.c_str() );

	if ( !pairout ) {
		std::cout << "Open failed for file: " << pairout.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int res_num1, res_num2, icon;
	float pair;
	std::string line;

	getline( pairout, line );
 	while( getline( pairout, line) ) {
		std::istringstream line_str( line );
		line_str >> icon >> res_num1 >> res_num2 >> pair;
		homolog_pair_log( icon, res_num1, res_num2 ) = pair;
	}

	pairout.close();
}

void
HomologStats::set_parent_sse_object(
	SecondaryStructureEval * sse
)
{
	parent_sse_object_ = sse;
}

SecondaryStructureEval *
HomologStats::get_parent_sse_object()
{
	return parent_sse_object_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_homolog_env
///
/// @brief
/// THIS FUNCTION EVALUATES THE ENERGY OF A STRUCTURE BASED ON
///      THE TERTIARY PROBABILITY FILES
/// env_score
///
/// @detailed
///
/// @param  env_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
HomologStats::evaluate_homolog_env(
	float & env_score
)
{
	using namespace cenlist_ns;
	using namespace param;
	using namespace param_aa;

	FArray1D_int const & res = parent_sse_object_->get_res(); // yab: misc removal
	int const & total_residue = parent_sse_object_->get_total_residue(); // yab: misc removal

	int res_id;

	if ( pose_flag() ) pose_update_cendist( score_get_current_pose() );

	compute_CB_environment(); // function is hiding in vdw_compute.cc
	env_score = 0.0;

	for ( int i = 1; i <= total_residue; ++i ) {

//cmd  Several checks added to only consider amino acids, not DNA/RNA
		res_id = res(i);
		if ( is_protein(res_id) ) {

			if ( fcen10(i) >= 31.0f ) fcen10(i) = 30.9999f;

			env_score += homolog_cen10_log( i, static_cast< int >(fcen10(i)) ); // homolog env score

		} // amino acid check

	}

	env_score *= 2.019f;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_homolog_pair
///
/// @brief
/// THIS FUNCTION EVALUATES THE ENERGY OF A STRUCTURE BASED ON
///      THE TERTIARY PROBABILITY FILES
/// pair_score
///
/// @detailed
///
/// @param  pair_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///NOTE:  ASSUMES THE CENDIST ARRAY IS CURRENT!!!!
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
HomologStats::evaluate_homolog_pair(
	float & pair_score
)
{
	using namespace cenlist_ns;
	using namespace param;
	using namespace param_aa;

	FArray1D_int const & res = parent_sse_object_->get_res(); // yab: misc removal
	int const & total_residue = parent_sse_object_->get_total_residue(); // yab: misc removal

	int res_id,icon;
	float interp2;
	float k;

	if ( pose_flag() ) pose_update_cendist( score_get_current_pose() );

	compute_CB_environment(); // function is hiding in vdw_compute.cc

	// make working copy of rsd_wt
  	FArray1D_float rsd_wt(parent_sse_object_->get_rsd_wt()); // rsd_wt is 1 unless
	                                     // command line flag envpair_wt is set
//     PAIR SCORE

	static double const cen_dist5_pad_plus  = cen_dist5_pad  + 25.0;
	static double const cen_dist7_pad_plus  = cen_dist7_pad  + 56.25;
	static double const cen_dist10_pad_plus = cen_dist10_pad + 100.0;

	static double const cen_dist5_pad_minus  = cen_dist5_pad  - 25.0;
	static double const cen_dist7_pad_minus  = cen_dist7_pad  - 56.25;
	static double const cen_dist10_pad_minus = cen_dist10_pad - 100.0;
	static double const cen_dist12_pad_minus = cen_dist12_pad - 144.0;

	static double const cen_dist5_pad_hinv  = 0.5 / cen_dist5_pad;
	static double const cen_dist7_pad_hinv  = 0.5 / cen_dist7_pad;
	static double const cen_dist10_pad_hinv = 0.5 / cen_dist10_pad;
	static double const cen_dist12_pad_hinv = 0.5 / cen_dist12_pad;

	bool const disulf_flag = disulfides::BOUNDARY::get_disulf_flag();
	pair_score = 0.0;

	for ( int i = 1; i <= total_residue; ++i ) {
		res_id = res(i);
		if ( is_protein(res_id) || is_nonnatural(res_id) ) {
			for ( int kk = 1, lcl = cen_list.index(kk,i), kke = cen12up(i);
			 kk <= kke; ++kk, ++lcl ) {
				int j = cen_list[ lcl ]; // cen_list(kk,i)
				int const res_j = res(j);

//CAR  no pair score if a disulfide
				if ( disulf_flag && res_id == 2 && res_j == 2 &&
				 disulfides::BOUNDARY::cys_pair_in_disulf(i,j) ) goto L200;

				if ( j - i > 8 ) {
					k = cendist(i,j);

//$$$  we now try to find which bin the pair distance lies in
//$$$  I note this could in principle be calculated and updatded
//$$$  just like cen_dist is if there is a need for speed.
//$$$  this function interpolates between bins.
//$$$  An important(!) requirement on pair_log is that the
//$$$  value should approach zero as the radius increases.
//$$$  this fact permits us not to have to compute and score pairs are larger
//$$$  than cen_dist > cutoff.

					icon = 5;
					if ( k > cen_dist10_pad_plus ) {
						icon = 4;
						interp2 = ( k + cen_dist12_pad_minus ) * cen_dist12_pad_hinv;
					} else {
						if ( k > cen_dist7_pad_plus ) {
							icon = 3;
							interp2 = ( k + cen_dist10_pad_minus ) * cen_dist10_pad_hinv;
						} else {
							if ( k > cen_dist5_pad_plus ) {
								icon = 2;
								interp2 = ( k + cen_dist7_pad_minus ) * cen_dist7_pad_hinv;
							} else {
								icon = 1;
								interp2 = ( k + cen_dist5_pad_minus ) * cen_dist5_pad_hinv;
							}
						}
					}
					if ( interp2 < 0.0 ) interp2 = 0.0;

// note in theory this will never happen but in practice round off
// error can cause problem
					if ( interp2 > 1.0 ) interp2 = 1.0;

// handle last bin specially since icon+1 would be past array end
					if ( icon != 5 ) {
						int const lpl = homolog_pair_log.index(icon,i,j);
						pair_score += ( ( 1.0f - interp2 ) * homolog_pair_log[ lpl ] +
						 interp2 * homolog_pair_log[ lpl+1 ] ) * rsd_wt(i);
					} else {
						pair_score += ( 1.0f - interp2 ) * homolog_pair_log(icon,i,j) * rsd_wt(i);
					}

				}
L200:; // escape for cys-cys disulfide paires
			}
		} // escape for non-AA's
	}

	pair_score *= 2.019f;
}

void
HomologStats::homolog_env_initializer(
	FArray1D_float & pdb_cen10_prob
)
{
	pdb_cen10_prob( 1 ) = 0.00193;
	pdb_cen10_prob( 2 ) = 0.00063;
	pdb_cen10_prob( 3 ) = 0.00111;
	pdb_cen10_prob( 4 ) = 0.00332;
	pdb_cen10_prob( 5 ) = 0.00761;
	pdb_cen10_prob( 6 ) = 0.01389;
	pdb_cen10_prob( 7 ) = 0.02164;
	pdb_cen10_prob( 8 ) = 0.02965;
	pdb_cen10_prob( 9 ) = 0.03703;
	pdb_cen10_prob( 10 ) = 0.04274;
	pdb_cen10_prob( 11 ) = 0.04689;
	pdb_cen10_prob( 12 ) = 0.04951;
	pdb_cen10_prob( 13 ) = 0.05006;
	pdb_cen10_prob( 14 ) = 0.05007;
	pdb_cen10_prob( 15 ) = 0.04986;
	pdb_cen10_prob( 16 ) = 0.04862;
	pdb_cen10_prob( 17 ) = 0.04810;
	pdb_cen10_prob( 18 ) = 0.04712;
	pdb_cen10_prob( 19 ) = 0.04623;
	pdb_cen10_prob( 20 ) = 0.04651;
	pdb_cen10_prob( 21 ) = 0.04648;
	pdb_cen10_prob( 22 ) = 0.04624;
	pdb_cen10_prob( 23 ) = 0.04578;
	pdb_cen10_prob( 24 ) = 0.04459;
	pdb_cen10_prob( 25 ) = 0.04146;
	pdb_cen10_prob( 26 ) = 0.03674;
	pdb_cen10_prob( 27 ) = 0.02984;
	pdb_cen10_prob( 28 ) = 0.02282;
	pdb_cen10_prob( 29 ) = 0.01963;
	pdb_cen10_prob( 30 ) = 0.02591;
}

void
HomologStats::homolog_pair_initializer(
	FArray2D_float & pdb_pair_prob
)
{
	pdb_pair_prob( 9, 1 ) = 0.062;
	pdb_pair_prob( 9, 2 ) = 0.142;
	pdb_pair_prob( 9, 3 ) = 0.325;
	pdb_pair_prob( 9, 4 ) = 0.470;
	pdb_pair_prob( 9, 5 ) = 0.000;
	pdb_pair_prob( 10, 1 ) = 0.068;
	pdb_pair_prob( 10, 2 ) = 0.149;
	pdb_pair_prob( 10, 3 ) = 0.346;
	pdb_pair_prob( 10, 4 ) = 0.437;
	pdb_pair_prob( 10, 5 ) = 0.000;
	pdb_pair_prob( 11, 1 ) = 0.067;
	pdb_pair_prob( 11, 2 ) = 0.172;
	pdb_pair_prob( 11, 3 ) = 0.335;
	pdb_pair_prob( 11, 4 ) = 0.426;
	pdb_pair_prob( 11, 5 ) = 0.000;
	pdb_pair_prob( 12, 1 ) = 0.065;
	pdb_pair_prob( 12, 2 ) = 0.164;
	pdb_pair_prob( 12, 3 ) = 0.331;
	pdb_pair_prob( 12, 4 ) = 0.440;
	pdb_pair_prob( 12, 5 ) = 0.000;
	pdb_pair_prob( 13, 1 ) = 0.063;
	pdb_pair_prob( 13, 2 ) = 0.161;
	pdb_pair_prob( 13, 3 ) = 0.330;
	pdb_pair_prob( 13, 4 ) = 0.446;
	pdb_pair_prob( 13, 5 ) = 0.000;
	pdb_pair_prob( 14, 1 ) = 0.063;
	pdb_pair_prob( 14, 2 ) = 0.161;
	pdb_pair_prob( 14, 3 ) = 0.342;
	pdb_pair_prob( 14, 4 ) = 0.435;
	pdb_pair_prob( 14, 5 ) = 0.000;
	pdb_pair_prob( 15, 1 ) = 0.060;
	pdb_pair_prob( 15, 2 ) = 0.159;
	pdb_pair_prob( 15, 3 ) = 0.337;
	pdb_pair_prob( 15, 4 ) = 0.444;
	pdb_pair_prob( 15, 5 ) = 0.000;
	pdb_pair_prob( 16, 1 ) = 0.066;
	pdb_pair_prob( 16, 2 ) = 0.160;
	pdb_pair_prob( 16, 3 ) = 0.329;
	pdb_pair_prob( 16, 4 ) = 0.444;
	pdb_pair_prob( 16, 5 ) = 0.000;
	pdb_pair_prob( 17, 1 ) = 0.064;
	pdb_pair_prob( 17, 2 ) = 0.162;
	pdb_pair_prob( 17, 3 ) = 0.337;
	pdb_pair_prob( 17, 4 ) = 0.437;
	pdb_pair_prob( 17, 5 ) = 0.000;
	pdb_pair_prob( 18, 1 ) = 0.064;
	pdb_pair_prob( 18, 2 ) = 0.154;
	pdb_pair_prob( 18, 3 ) = 0.338;
	pdb_pair_prob( 18, 4 ) = 0.444;
	pdb_pair_prob( 18, 5 ) = 0.000;
	pdb_pair_prob( 19, 1 ) = 0.059;
	pdb_pair_prob( 19, 2 ) = 0.156;
	pdb_pair_prob( 19, 3 ) = 0.340;
	pdb_pair_prob( 19, 4 ) = 0.445;
	pdb_pair_prob( 19, 5 ) = 0.000;
	pdb_pair_prob( 20, 1 ) = 0.056;
	pdb_pair_prob( 20, 2 ) = 0.156;
	pdb_pair_prob( 20, 3 ) = 0.344;
	pdb_pair_prob( 20, 4 ) = 0.444;
	pdb_pair_prob( 20, 5 ) = 0.000;
	pdb_pair_prob( 21, 1 ) = 0.061;
	pdb_pair_prob( 21, 2 ) = 0.158;
	pdb_pair_prob( 21, 3 ) = 0.340;
	pdb_pair_prob( 21, 4 ) = 0.440;
	pdb_pair_prob( 21, 5 ) = 0.000;
	pdb_pair_prob( 22, 1 ) = 0.062;
	pdb_pair_prob( 22, 2 ) = 0.155;
	pdb_pair_prob( 22, 3 ) = 0.345;
	pdb_pair_prob( 22, 4 ) = 0.438;
	pdb_pair_prob( 22, 5 ) = 0.000;
	pdb_pair_prob( 23, 1 ) = 0.062;
	pdb_pair_prob( 23, 2 ) = 0.160;
	pdb_pair_prob( 23, 3 ) = 0.342;
	pdb_pair_prob( 23, 4 ) = 0.437;
	pdb_pair_prob( 23, 5 ) = 0.000;
	pdb_pair_prob( 24, 1 ) = 0.061;
	pdb_pair_prob( 24, 2 ) = 0.158;
	pdb_pair_prob( 24, 3 ) = 0.337;
	pdb_pair_prob( 24, 4 ) = 0.444;
	pdb_pair_prob( 24, 5 ) = 0.000;
	pdb_pair_prob( 25, 1 ) = 0.059;
	pdb_pair_prob( 25, 2 ) = 0.160;
	pdb_pair_prob( 25, 3 ) = 0.346;
	pdb_pair_prob( 25, 4 ) = 0.436;
	pdb_pair_prob( 25, 5 ) = 0.000;
	pdb_pair_prob( 26, 1 ) = 0.062;
	pdb_pair_prob( 26, 2 ) = 0.162;
	pdb_pair_prob( 26, 3 ) = 0.344;
	pdb_pair_prob( 26, 4 ) = 0.433;
	pdb_pair_prob( 26, 5 ) = 0.000;
	pdb_pair_prob( 27, 1 ) = 0.062;
	pdb_pair_prob( 27, 2 ) = 0.160;
	pdb_pair_prob( 27, 3 ) = 0.340;
	pdb_pair_prob( 27, 4 ) = 0.438;
	pdb_pair_prob( 27, 5 ) = 0.000;
	pdb_pair_prob( 28, 1 ) = 0.061;
	pdb_pair_prob( 28, 2 ) = 0.159;
	pdb_pair_prob( 28, 3 ) = 0.341;
	pdb_pair_prob( 28, 4 ) = 0.439;
	pdb_pair_prob( 28, 5 ) = 0.000;
	pdb_pair_prob( 29, 1 ) = 0.061;
	pdb_pair_prob( 29, 2 ) = 0.160;
	pdb_pair_prob( 29, 3 ) = 0.337;
	pdb_pair_prob( 29, 4 ) = 0.442;
	pdb_pair_prob( 29, 5 ) = 0.000;
	pdb_pair_prob( 30, 1 ) = 0.061;
	pdb_pair_prob( 30, 2 ) = 0.154;
	pdb_pair_prob( 30, 3 ) = 0.336;
	pdb_pair_prob( 30, 4 ) = 0.449;
	pdb_pair_prob( 30, 5 ) = 0.000;
	pdb_pair_prob( 31, 1 ) = 0.060;
	pdb_pair_prob( 31, 2 ) = 0.152;
	pdb_pair_prob( 31, 3 ) = 0.336;
	pdb_pair_prob( 31, 4 ) = 0.451;
	pdb_pair_prob( 31, 5 ) = 0.000;
	pdb_pair_prob( 32, 1 ) = 0.057;
	pdb_pair_prob( 32, 2 ) = 0.152;
	pdb_pair_prob( 32, 3 ) = 0.339;
	pdb_pair_prob( 32, 4 ) = 0.452;
	pdb_pair_prob( 32, 5 ) = 0.000;
	pdb_pair_prob( 33, 1 ) = 0.058;
	pdb_pair_prob( 33, 2 ) = 0.151;
	pdb_pair_prob( 33, 3 ) = 0.336;
	pdb_pair_prob( 33, 4 ) = 0.455;
	pdb_pair_prob( 33, 5 ) = 0.000;
	pdb_pair_prob( 34, 1 ) = 0.058;
	pdb_pair_prob( 34, 2 ) = 0.150;
	pdb_pair_prob( 34, 3 ) = 0.340;
	pdb_pair_prob( 34, 4 ) = 0.452;
	pdb_pair_prob( 34, 5 ) = 0.000;
	pdb_pair_prob( 35, 1 ) = 0.058;
	pdb_pair_prob( 35, 2 ) = 0.150;
	pdb_pair_prob( 35, 3 ) = 0.333;
	pdb_pair_prob( 35, 4 ) = 0.459;
	pdb_pair_prob( 35, 5 ) = 0.000;
	pdb_pair_prob( 36, 1 ) = 0.056;
	pdb_pair_prob( 36, 2 ) = 0.147;
	pdb_pair_prob( 36, 3 ) = 0.341;
	pdb_pair_prob( 36, 4 ) = 0.457;
	pdb_pair_prob( 36, 5 ) = 0.000;
	pdb_pair_prob( 37, 1 ) = 0.057;
	pdb_pair_prob( 37, 2 ) = 0.151;
	pdb_pair_prob( 37, 3 ) = 0.337;
	pdb_pair_prob( 37, 4 ) = 0.455;
	pdb_pair_prob( 37, 5 ) = 0.000;
	pdb_pair_prob( 38, 1 ) = 0.059;
	pdb_pair_prob( 38, 2 ) = 0.151;
	pdb_pair_prob( 38, 3 ) = 0.336;
	pdb_pair_prob( 38, 4 ) = 0.454;
	pdb_pair_prob( 38, 5 ) = 0.000;
	pdb_pair_prob( 39, 1 ) = 0.057;
	pdb_pair_prob( 39, 2 ) = 0.149;
	pdb_pair_prob( 39, 3 ) = 0.340;
	pdb_pair_prob( 39, 4 ) = 0.454;
	pdb_pair_prob( 39, 5 ) = 0.000;
	pdb_pair_prob( 40, 1 ) = 0.057;
	pdb_pair_prob( 40, 2 ) = 0.147;
	pdb_pair_prob( 40, 3 ) = 0.334;
	pdb_pair_prob( 40, 4 ) = 0.462;
	pdb_pair_prob( 40, 5 ) = 0.000;
	pdb_pair_prob( 41, 1 ) = 0.056;
	pdb_pair_prob( 41, 2 ) = 0.146;
	pdb_pair_prob( 41, 3 ) = 0.333;
	pdb_pair_prob( 41, 4 ) = 0.466;
	pdb_pair_prob( 41, 5 ) = 0.000;
	pdb_pair_prob( 42, 1 ) = 0.053;
	pdb_pair_prob( 42, 2 ) = 0.144;
	pdb_pair_prob( 42, 3 ) = 0.341;
	pdb_pair_prob( 42, 4 ) = 0.462;
	pdb_pair_prob( 42, 5 ) = 0.000;
	pdb_pair_prob( 43, 1 ) = 0.057;
	pdb_pair_prob( 43, 2 ) = 0.149;
	pdb_pair_prob( 43, 3 ) = 0.332;
	pdb_pair_prob( 43, 4 ) = 0.462;
	pdb_pair_prob( 43, 5 ) = 0.000;
	pdb_pair_prob( 44, 1 ) = 0.054;
	pdb_pair_prob( 44, 2 ) = 0.144;
	pdb_pair_prob( 44, 3 ) = 0.336;
	pdb_pair_prob( 44, 4 ) = 0.466;
	pdb_pair_prob( 44, 5 ) = 0.000;
	pdb_pair_prob( 45, 1 ) = 0.054;
	pdb_pair_prob( 45, 2 ) = 0.146;
	pdb_pair_prob( 45, 3 ) = 0.327;
	pdb_pair_prob( 45, 4 ) = 0.473;
	pdb_pair_prob( 45, 5 ) = 0.000;
	pdb_pair_prob( 46, 1 ) = 0.053;
	pdb_pair_prob( 46, 2 ) = 0.141;
	pdb_pair_prob( 46, 3 ) = 0.336;
	pdb_pair_prob( 46, 4 ) = 0.470;
	pdb_pair_prob( 46, 5 ) = 0.000;
	pdb_pair_prob( 47, 1 ) = 0.052;
	pdb_pair_prob( 47, 2 ) = 0.141;
	pdb_pair_prob( 47, 3 ) = 0.334;
	pdb_pair_prob( 47, 4 ) = 0.473;
	pdb_pair_prob( 47, 5 ) = 0.000;
	pdb_pair_prob( 48, 1 ) = 0.054;
	pdb_pair_prob( 48, 2 ) = 0.142;
	pdb_pair_prob( 48, 3 ) = 0.326;
	pdb_pair_prob( 48, 4 ) = 0.478;
	pdb_pair_prob( 48, 5 ) = 0.000;
	pdb_pair_prob( 49, 1 ) = 0.053;
	pdb_pair_prob( 49, 2 ) = 0.144;
	pdb_pair_prob( 49, 3 ) = 0.331;
	pdb_pair_prob( 49, 4 ) = 0.472;
	pdb_pair_prob( 49, 5 ) = 0.000;
	pdb_pair_prob( 50, 1 ) = 0.053;
	pdb_pair_prob( 50, 2 ) = 0.140;
	pdb_pair_prob( 50, 3 ) = 0.335;
	pdb_pair_prob( 50, 4 ) = 0.472;
	pdb_pair_prob( 50, 5 ) = 0.000;
	pdb_pair_prob( 51, 1 ) = 0.057;
	pdb_pair_prob( 51, 2 ) = 0.140;
	pdb_pair_prob( 51, 3 ) = 0.335;
	pdb_pair_prob( 51, 4 ) = 0.468;
	pdb_pair_prob( 51, 5 ) = 0.000;
	pdb_pair_prob( 52, 1 ) = 0.053;
	pdb_pair_prob( 52, 2 ) = 0.142;
	pdb_pair_prob( 52, 3 ) = 0.334;
	pdb_pair_prob( 52, 4 ) = 0.471;
	pdb_pair_prob( 52, 5 ) = 0.000;
	pdb_pair_prob( 53, 1 ) = 0.052;
	pdb_pair_prob( 53, 2 ) = 0.142;
	pdb_pair_prob( 53, 3 ) = 0.328;
	pdb_pair_prob( 53, 4 ) = 0.478;
	pdb_pair_prob( 53, 5 ) = 0.000;
	pdb_pair_prob( 54, 1 ) = 0.053;
	pdb_pair_prob( 54, 2 ) = 0.138;
	pdb_pair_prob( 54, 3 ) = 0.338;
	pdb_pair_prob( 54, 4 ) = 0.472;
	pdb_pair_prob( 54, 5 ) = 0.000;
	pdb_pair_prob( 55, 1 ) = 0.054;
	pdb_pair_prob( 55, 2 ) = 0.137;
	pdb_pair_prob( 55, 3 ) = 0.333;
	pdb_pair_prob( 55, 4 ) = 0.476;
	pdb_pair_prob( 55, 5 ) = 0.000;
	pdb_pair_prob( 56, 1 ) = 0.052;
	pdb_pair_prob( 56, 2 ) = 0.142;
	pdb_pair_prob( 56, 3 ) = 0.326;
	pdb_pair_prob( 56, 4 ) = 0.479;
	pdb_pair_prob( 56, 5 ) = 0.000;
	pdb_pair_prob( 57, 1 ) = 0.049;
	pdb_pair_prob( 57, 2 ) = 0.140;
	pdb_pair_prob( 57, 3 ) = 0.334;
	pdb_pair_prob( 57, 4 ) = 0.478;
	pdb_pair_prob( 57, 5 ) = 0.000;
	pdb_pair_prob( 58, 1 ) = 0.053;
	pdb_pair_prob( 58, 2 ) = 0.146;
	pdb_pair_prob( 58, 3 ) = 0.334;
	pdb_pair_prob( 58, 4 ) = 0.468;
	pdb_pair_prob( 58, 5 ) = 0.000;
	pdb_pair_prob( 59, 1 ) = 0.052;
	pdb_pair_prob( 59, 2 ) = 0.138;
	pdb_pair_prob( 59, 3 ) = 0.337;
	pdb_pair_prob( 59, 4 ) = 0.474;
	pdb_pair_prob( 59, 5 ) = 0.000;
	pdb_pair_prob( 60, 1 ) = 0.054;
	pdb_pair_prob( 60, 2 ) = 0.143;
	pdb_pair_prob( 60, 3 ) = 0.331;
	pdb_pair_prob( 60, 4 ) = 0.472;
	pdb_pair_prob( 60, 5 ) = 0.000;
	pdb_pair_prob( 61, 1 ) = 0.055;
	pdb_pair_prob( 61, 2 ) = 0.139;
	pdb_pair_prob( 61, 3 ) = 0.320;
	pdb_pair_prob( 61, 4 ) = 0.485;
	pdb_pair_prob( 61, 5 ) = 0.000;
	pdb_pair_prob( 62, 1 ) = 0.051;
	pdb_pair_prob( 62, 2 ) = 0.140;
	pdb_pair_prob( 62, 3 ) = 0.341;
	pdb_pair_prob( 62, 4 ) = 0.467;
	pdb_pair_prob( 62, 5 ) = 0.000;
	pdb_pair_prob( 63, 1 ) = 0.052;
	pdb_pair_prob( 63, 2 ) = 0.135;
	pdb_pair_prob( 63, 3 ) = 0.334;
	pdb_pair_prob( 63, 4 ) = 0.479;
	pdb_pair_prob( 63, 5 ) = 0.000;
	pdb_pair_prob( 64, 1 ) = 0.052;
	pdb_pair_prob( 64, 2 ) = 0.135;
	pdb_pair_prob( 64, 3 ) = 0.330;
	pdb_pair_prob( 64, 4 ) = 0.483;
	pdb_pair_prob( 64, 5 ) = 0.000;
	pdb_pair_prob( 65, 1 ) = 0.053;
	pdb_pair_prob( 65, 2 ) = 0.144;
	pdb_pair_prob( 65, 3 ) = 0.331;
	pdb_pair_prob( 65, 4 ) = 0.472;
	pdb_pair_prob( 65, 5 ) = 0.000;
	pdb_pair_prob( 66, 1 ) = 0.052;
	pdb_pair_prob( 66, 2 ) = 0.137;
	pdb_pair_prob( 66, 3 ) = 0.344;
	pdb_pair_prob( 66, 4 ) = 0.468;
	pdb_pair_prob( 66, 5 ) = 0.000;
	pdb_pair_prob( 67, 1 ) = 0.054;
	pdb_pair_prob( 67, 2 ) = 0.136;
	pdb_pair_prob( 67, 3 ) = 0.332;
	pdb_pair_prob( 67, 4 ) = 0.479;
	pdb_pair_prob( 67, 5 ) = 0.000;
	pdb_pair_prob( 68, 1 ) = 0.051;
	pdb_pair_prob( 68, 2 ) = 0.138;
	pdb_pair_prob( 68, 3 ) = 0.336;
	pdb_pair_prob( 68, 4 ) = 0.475;
	pdb_pair_prob( 68, 5 ) = 0.000;
	pdb_pair_prob( 69, 1 ) = 0.052;
	pdb_pair_prob( 69, 2 ) = 0.140;
	pdb_pair_prob( 69, 3 ) = 0.326;
	pdb_pair_prob( 69, 4 ) = 0.482;
	pdb_pair_prob( 69, 5 ) = 0.000;
	pdb_pair_prob( 70, 1 ) = 0.054;
	pdb_pair_prob( 70, 2 ) = 0.142;
	pdb_pair_prob( 70, 3 ) = 0.338;
	pdb_pair_prob( 70, 4 ) = 0.466;
	pdb_pair_prob( 70, 5 ) = 0.000;
	pdb_pair_prob( 71, 1 ) = 0.054;
	pdb_pair_prob( 71, 2 ) = 0.139;
	pdb_pair_prob( 71, 3 ) = 0.330;
	pdb_pair_prob( 71, 4 ) = 0.477;
	pdb_pair_prob( 71, 5 ) = 0.000;
	pdb_pair_prob( 72, 1 ) = 0.048;
	pdb_pair_prob( 72, 2 ) = 0.132;
	pdb_pair_prob( 72, 3 ) = 0.345;
	pdb_pair_prob( 72, 4 ) = 0.476;
	pdb_pair_prob( 72, 5 ) = 0.000;
	pdb_pair_prob( 73, 1 ) = 0.050;
	pdb_pair_prob( 73, 2 ) = 0.136;
	pdb_pair_prob( 73, 3 ) = 0.330;
	pdb_pair_prob( 73, 4 ) = 0.483;
	pdb_pair_prob( 73, 5 ) = 0.000;
	pdb_pair_prob( 74, 1 ) = 0.052;
	pdb_pair_prob( 74, 2 ) = 0.145;
	pdb_pair_prob( 74, 3 ) = 0.318;
	pdb_pair_prob( 74, 4 ) = 0.486;
	pdb_pair_prob( 74, 5 ) = 0.000;
	pdb_pair_prob( 75, 1 ) = 0.050;
	pdb_pair_prob( 75, 2 ) = 0.136;
	pdb_pair_prob( 75, 3 ) = 0.332;
	pdb_pair_prob( 75, 4 ) = 0.482;
	pdb_pair_prob( 75, 5 ) = 0.000;
	pdb_pair_prob( 76, 1 ) = 0.049;
	pdb_pair_prob( 76, 2 ) = 0.136;
	pdb_pair_prob( 76, 3 ) = 0.332;
	pdb_pair_prob( 76, 4 ) = 0.483;
	pdb_pair_prob( 76, 5 ) = 0.000;
	pdb_pair_prob( 77, 1 ) = 0.049;
	pdb_pair_prob( 77, 2 ) = 0.136;
	pdb_pair_prob( 77, 3 ) = 0.329;
	pdb_pair_prob( 77, 4 ) = 0.486;
	pdb_pair_prob( 77, 5 ) = 0.000;
	pdb_pair_prob( 78, 1 ) = 0.050;
	pdb_pair_prob( 78, 2 ) = 0.139;
	pdb_pair_prob( 78, 3 ) = 0.327;
	pdb_pair_prob( 78, 4 ) = 0.484;
	pdb_pair_prob( 78, 5 ) = 0.000;
	pdb_pair_prob( 79, 1 ) = 0.052;
	pdb_pair_prob( 79, 2 ) = 0.138;
	pdb_pair_prob( 79, 3 ) = 0.326;
	pdb_pair_prob( 79, 4 ) = 0.483;
	pdb_pair_prob( 79, 5 ) = 0.000;
	pdb_pair_prob( 80, 1 ) = 0.048;
	pdb_pair_prob( 80, 2 ) = 0.131;
	pdb_pair_prob( 80, 3 ) = 0.332;
	pdb_pair_prob( 80, 4 ) = 0.489;
	pdb_pair_prob( 80, 5 ) = 0.000;
	pdb_pair_prob( 81, 1 ) = 0.051;
	pdb_pair_prob( 81, 2 ) = 0.142;
	pdb_pair_prob( 81, 3 ) = 0.332;
	pdb_pair_prob( 81, 4 ) = 0.475;
	pdb_pair_prob( 81, 5 ) = 0.000;
	pdb_pair_prob( 82, 1 ) = 0.053;
	pdb_pair_prob( 82, 2 ) = 0.139;
	pdb_pair_prob( 82, 3 ) = 0.324;
	pdb_pair_prob( 82, 4 ) = 0.485;
	pdb_pair_prob( 82, 5 ) = 0.000;
	pdb_pair_prob( 83, 1 ) = 0.052;
	pdb_pair_prob( 83, 2 ) = 0.141;
	pdb_pair_prob( 83, 3 ) = 0.329;
	pdb_pair_prob( 83, 4 ) = 0.478;
	pdb_pair_prob( 83, 5 ) = 0.000;
	pdb_pair_prob( 84, 1 ) = 0.052;
	pdb_pair_prob( 84, 2 ) = 0.134;
	pdb_pair_prob( 84, 3 ) = 0.328;
	pdb_pair_prob( 84, 4 ) = 0.486;
	pdb_pair_prob( 84, 5 ) = 0.000;
	pdb_pair_prob( 85, 1 ) = 0.048;
	pdb_pair_prob( 85, 2 ) = 0.137;
	pdb_pair_prob( 85, 3 ) = 0.317;
	pdb_pair_prob( 85, 4 ) = 0.499;
	pdb_pair_prob( 85, 5 ) = 0.000;
	pdb_pair_prob( 86, 1 ) = 0.051;
	pdb_pair_prob( 86, 2 ) = 0.137;
	pdb_pair_prob( 86, 3 ) = 0.340;
	pdb_pair_prob( 86, 4 ) = 0.473;
	pdb_pair_prob( 86, 5 ) = 0.000;
	pdb_pair_prob( 87, 1 ) = 0.055;
	pdb_pair_prob( 87, 2 ) = 0.142;
	pdb_pair_prob( 87, 3 ) = 0.323;
	pdb_pair_prob( 87, 4 ) = 0.481;
	pdb_pair_prob( 87, 5 ) = 0.000;
	pdb_pair_prob( 88, 1 ) = 0.048;
	pdb_pair_prob( 88, 2 ) = 0.147;
	pdb_pair_prob( 88, 3 ) = 0.331;
	pdb_pair_prob( 88, 4 ) = 0.475;
	pdb_pair_prob( 88, 5 ) = 0.000;
	pdb_pair_prob( 89, 1 ) = 0.053;
	pdb_pair_prob( 89, 2 ) = 0.137;
	pdb_pair_prob( 89, 3 ) = 0.333;
	pdb_pair_prob( 89, 4 ) = 0.477;
	pdb_pair_prob( 89, 5 ) = 0.000;
	pdb_pair_prob( 90, 1 ) = 0.052;
	pdb_pair_prob( 90, 2 ) = 0.135;
	pdb_pair_prob( 90, 3 ) = 0.337;
	pdb_pair_prob( 90, 4 ) = 0.475;
	pdb_pair_prob( 90, 5 ) = 0.000;
	pdb_pair_prob( 91, 1 ) = 0.052;
	pdb_pair_prob( 91, 2 ) = 0.140;
	pdb_pair_prob( 91, 3 ) = 0.336;
	pdb_pair_prob( 91, 4 ) = 0.473;
	pdb_pair_prob( 91, 5 ) = 0.000;
	pdb_pair_prob( 92, 1 ) = 0.050;
	pdb_pair_prob( 92, 2 ) = 0.135;
	pdb_pair_prob( 92, 3 ) = 0.338;
	pdb_pair_prob( 92, 4 ) = 0.477;
	pdb_pair_prob( 92, 5 ) = 0.000;
	pdb_pair_prob( 93, 1 ) = 0.050;
	pdb_pair_prob( 93, 2 ) = 0.145;
	pdb_pair_prob( 93, 3 ) = 0.332;
	pdb_pair_prob( 93, 4 ) = 0.472;
	pdb_pair_prob( 93, 5 ) = 0.000;
	pdb_pair_prob( 94, 1 ) = 0.049;
	pdb_pair_prob( 94, 2 ) = 0.135;
	pdb_pair_prob( 94, 3 ) = 0.331;
	pdb_pair_prob( 94, 4 ) = 0.484;
	pdb_pair_prob( 94, 5 ) = 0.000;
	pdb_pair_prob( 95, 1 ) = 0.054;
	pdb_pair_prob( 95, 2 ) = 0.135;
	pdb_pair_prob( 95, 3 ) = 0.325;
	pdb_pair_prob( 95, 4 ) = 0.487;
	pdb_pair_prob( 95, 5 ) = 0.000;
	pdb_pair_prob( 96, 1 ) = 0.049;
	pdb_pair_prob( 96, 2 ) = 0.126;
	pdb_pair_prob( 96, 3 ) = 0.347;
	pdb_pair_prob( 96, 4 ) = 0.478;
	pdb_pair_prob( 96, 5 ) = 0.000;
	pdb_pair_prob( 97, 1 ) = 0.050;
	pdb_pair_prob( 97, 2 ) = 0.141;
	pdb_pair_prob( 97, 3 ) = 0.324;
	pdb_pair_prob( 97, 4 ) = 0.484;
	pdb_pair_prob( 97, 5 ) = 0.000;
	pdb_pair_prob( 98, 1 ) = 0.044;
	pdb_pair_prob( 98, 2 ) = 0.135;
	pdb_pair_prob( 98, 3 ) = 0.333;
	pdb_pair_prob( 98, 4 ) = 0.488;
	pdb_pair_prob( 98, 5 ) = 0.000;
	pdb_pair_prob( 99, 1 ) = 0.055;
	pdb_pair_prob( 99, 2 ) = 0.141;
	pdb_pair_prob( 99, 3 ) = 0.327;
	pdb_pair_prob( 99, 4 ) = 0.477;
	pdb_pair_prob( 99, 5 ) = 0.000;
	pdb_pair_prob( 100, 1 ) = 0.050;
	pdb_pair_prob( 100, 2 ) = 0.138;
	pdb_pair_prob( 100, 3 ) = 0.332;
	pdb_pair_prob( 100, 4 ) = 0.479;
	pdb_pair_prob( 100, 5 ) = 0.000;
	pdb_pair_prob( 101, 1 ) = 0.054;
	pdb_pair_prob( 101, 2 ) = 0.133;
	pdb_pair_prob( 101, 3 ) = 0.328;
	pdb_pair_prob( 101, 4 ) = 0.485;
	pdb_pair_prob( 101, 5 ) = 0.000;
	pdb_pair_prob( 102, 1 ) = 0.047;
	pdb_pair_prob( 102, 2 ) = 0.147;
	pdb_pair_prob( 102, 3 ) = 0.323;
	pdb_pair_prob( 102, 4 ) = 0.483;
	pdb_pair_prob( 102, 5 ) = 0.000;
	pdb_pair_prob( 103, 1 ) = 0.051;
	pdb_pair_prob( 103, 2 ) = 0.136;
	pdb_pair_prob( 103, 3 ) = 0.326;
	pdb_pair_prob( 103, 4 ) = 0.487;
	pdb_pair_prob( 103, 5 ) = 0.000;
	pdb_pair_prob( 104, 1 ) = 0.046;
	pdb_pair_prob( 104, 2 ) = 0.133;
	pdb_pair_prob( 104, 3 ) = 0.329;
	pdb_pair_prob( 104, 4 ) = 0.491;
	pdb_pair_prob( 104, 5 ) = 0.000;
	pdb_pair_prob( 105, 1 ) = 0.046;
	pdb_pair_prob( 105, 2 ) = 0.141;
	pdb_pair_prob( 105, 3 ) = 0.330;
	pdb_pair_prob( 105, 4 ) = 0.483;
	pdb_pair_prob( 105, 5 ) = 0.000;
	pdb_pair_prob( 106, 1 ) = 0.049;
	pdb_pair_prob( 106, 2 ) = 0.134;
	pdb_pair_prob( 106, 3 ) = 0.324;
	pdb_pair_prob( 106, 4 ) = 0.493;
	pdb_pair_prob( 106, 5 ) = 0.000;
	pdb_pair_prob( 107, 1 ) = 0.050;
	pdb_pair_prob( 107, 2 ) = 0.135;
	pdb_pair_prob( 107, 3 ) = 0.323;
	pdb_pair_prob( 107, 4 ) = 0.492;
	pdb_pair_prob( 107, 5 ) = 0.000;
	pdb_pair_prob( 108, 1 ) = 0.047;
	pdb_pair_prob( 108, 2 ) = 0.126;
	pdb_pair_prob( 108, 3 ) = 0.332;
	pdb_pair_prob( 108, 4 ) = 0.495;
	pdb_pair_prob( 108, 5 ) = 0.000;
	pdb_pair_prob( 109, 1 ) = 0.051;
	pdb_pair_prob( 109, 2 ) = 0.138;
	pdb_pair_prob( 109, 3 ) = 0.327;
	pdb_pair_prob( 109, 4 ) = 0.484;
	pdb_pair_prob( 109, 5 ) = 0.000;
	pdb_pair_prob( 110, 1 ) = 0.052;
	pdb_pair_prob( 110, 2 ) = 0.138;
	pdb_pair_prob( 110, 3 ) = 0.339;
	pdb_pair_prob( 110, 4 ) = 0.470;
	pdb_pair_prob( 110, 5 ) = 0.000;
	pdb_pair_prob( 111, 1 ) = 0.048;
	pdb_pair_prob( 111, 2 ) = 0.139;
	pdb_pair_prob( 111, 3 ) = 0.330;
	pdb_pair_prob( 111, 4 ) = 0.483;
	pdb_pair_prob( 111, 5 ) = 0.000;
	pdb_pair_prob( 112, 1 ) = 0.052;
	pdb_pair_prob( 112, 2 ) = 0.130;
	pdb_pair_prob( 112, 3 ) = 0.324;
	pdb_pair_prob( 112, 4 ) = 0.494;
	pdb_pair_prob( 112, 5 ) = 0.000;
	pdb_pair_prob( 113, 1 ) = 0.054;
	pdb_pair_prob( 113, 2 ) = 0.139;
	pdb_pair_prob( 113, 3 ) = 0.325;
	pdb_pair_prob( 113, 4 ) = 0.481;
	pdb_pair_prob( 113, 5 ) = 0.000;
	pdb_pair_prob( 114, 1 ) = 0.057;
	pdb_pair_prob( 114, 2 ) = 0.130;
	pdb_pair_prob( 114, 3 ) = 0.330;
	pdb_pair_prob( 114, 4 ) = 0.483;
	pdb_pair_prob( 114, 5 ) = 0.000;
	pdb_pair_prob( 115, 1 ) = 0.054;
	pdb_pair_prob( 115, 2 ) = 0.135;
	pdb_pair_prob( 115, 3 ) = 0.336;
	pdb_pair_prob( 115, 4 ) = 0.475;
	pdb_pair_prob( 115, 5 ) = 0.000;
	pdb_pair_prob( 116, 1 ) = 0.049;
	pdb_pair_prob( 116, 2 ) = 0.134;
	pdb_pair_prob( 116, 3 ) = 0.327;
	pdb_pair_prob( 116, 4 ) = 0.490;
	pdb_pair_prob( 116, 5 ) = 0.000;
	pdb_pair_prob( 117, 1 ) = 0.051;
	pdb_pair_prob( 117, 2 ) = 0.146;
	pdb_pair_prob( 117, 3 ) = 0.332;
	pdb_pair_prob( 117, 4 ) = 0.471;
	pdb_pair_prob( 117, 5 ) = 0.000;
	pdb_pair_prob( 118, 1 ) = 0.047;
	pdb_pair_prob( 118, 2 ) = 0.130;
	pdb_pair_prob( 118, 3 ) = 0.337;
	pdb_pair_prob( 118, 4 ) = 0.486;
	pdb_pair_prob( 118, 5 ) = 0.000;
	pdb_pair_prob( 119, 1 ) = 0.048;
	pdb_pair_prob( 119, 2 ) = 0.138;
	pdb_pair_prob( 119, 3 ) = 0.337;
	pdb_pair_prob( 119, 4 ) = 0.476;
	pdb_pair_prob( 119, 5 ) = 0.000;
	pdb_pair_prob( 120, 1 ) = 0.052;
	pdb_pair_prob( 120, 2 ) = 0.130;
	pdb_pair_prob( 120, 3 ) = 0.332;
	pdb_pair_prob( 120, 4 ) = 0.486;
	pdb_pair_prob( 120, 5 ) = 0.000;
	pdb_pair_prob( 121, 1 ) = 0.049;
	pdb_pair_prob( 121, 2 ) = 0.135;
	pdb_pair_prob( 121, 3 ) = 0.342;
	pdb_pair_prob( 121, 4 ) = 0.474;
	pdb_pair_prob( 121, 5 ) = 0.000;
	pdb_pair_prob( 122, 1 ) = 0.058;
	pdb_pair_prob( 122, 2 ) = 0.128;
	pdb_pair_prob( 122, 3 ) = 0.317;
	pdb_pair_prob( 122, 4 ) = 0.497;
	pdb_pair_prob( 122, 5 ) = 0.000;
	pdb_pair_prob( 123, 1 ) = 0.048;
	pdb_pair_prob( 123, 2 ) = 0.132;
	pdb_pair_prob( 123, 3 ) = 0.340;
	pdb_pair_prob( 123, 4 ) = 0.479;
	pdb_pair_prob( 123, 5 ) = 0.000;
	pdb_pair_prob( 124, 1 ) = 0.051;
	pdb_pair_prob( 124, 2 ) = 0.134;
	pdb_pair_prob( 124, 3 ) = 0.323;
	pdb_pair_prob( 124, 4 ) = 0.492;
	pdb_pair_prob( 124, 5 ) = 0.000;
	pdb_pair_prob( 125, 1 ) = 0.047;
	pdb_pair_prob( 125, 2 ) = 0.132;
	pdb_pair_prob( 125, 3 ) = 0.330;
	pdb_pair_prob( 125, 4 ) = 0.491;
	pdb_pair_prob( 125, 5 ) = 0.000;
	pdb_pair_prob( 126, 1 ) = 0.047;
	pdb_pair_prob( 126, 2 ) = 0.140;
	pdb_pair_prob( 126, 3 ) = 0.325;
	pdb_pair_prob( 126, 4 ) = 0.487;
	pdb_pair_prob( 126, 5 ) = 0.000;
	pdb_pair_prob( 127, 1 ) = 0.046;
	pdb_pair_prob( 127, 2 ) = 0.138;
	pdb_pair_prob( 127, 3 ) = 0.328;
	pdb_pair_prob( 127, 4 ) = 0.487;
	pdb_pair_prob( 127, 5 ) = 0.000;
	pdb_pair_prob( 128, 1 ) = 0.056;
	pdb_pair_prob( 128, 2 ) = 0.134;
	pdb_pair_prob( 128, 3 ) = 0.316;
	pdb_pair_prob( 128, 4 ) = 0.494;
	pdb_pair_prob( 128, 5 ) = 0.000;
	pdb_pair_prob( 129, 1 ) = 0.054;
	pdb_pair_prob( 129, 2 ) = 0.138;
	pdb_pair_prob( 129, 3 ) = 0.336;
	pdb_pair_prob( 129, 4 ) = 0.472;
	pdb_pair_prob( 129, 5 ) = 0.000;
	pdb_pair_prob( 130, 1 ) = 0.055;
	pdb_pair_prob( 130, 2 ) = 0.132;
	pdb_pair_prob( 130, 3 ) = 0.327;
	pdb_pair_prob( 130, 4 ) = 0.486;
	pdb_pair_prob( 130, 5 ) = 0.000;
	pdb_pair_prob( 131, 1 ) = 0.050;
	pdb_pair_prob( 131, 2 ) = 0.133;
	pdb_pair_prob( 131, 3 ) = 0.342;
	pdb_pair_prob( 131, 4 ) = 0.474;
	pdb_pair_prob( 131, 5 ) = 0.000;
	pdb_pair_prob( 132, 1 ) = 0.053;
	pdb_pair_prob( 132, 2 ) = 0.138;
	pdb_pair_prob( 132, 3 ) = 0.323;
	pdb_pair_prob( 132, 4 ) = 0.487;
	pdb_pair_prob( 132, 5 ) = 0.000;
	pdb_pair_prob( 133, 1 ) = 0.047;
	pdb_pair_prob( 133, 2 ) = 0.144;
	pdb_pair_prob( 133, 3 ) = 0.333;
	pdb_pair_prob( 133, 4 ) = 0.476;
	pdb_pair_prob( 133, 5 ) = 0.000;
	pdb_pair_prob( 134, 1 ) = 0.054;
	pdb_pair_prob( 134, 2 ) = 0.140;
	pdb_pair_prob( 134, 3 ) = 0.321;
	pdb_pair_prob( 134, 4 ) = 0.485;
	pdb_pair_prob( 134, 5 ) = 0.000;
	pdb_pair_prob( 135, 1 ) = 0.054;
	pdb_pair_prob( 135, 2 ) = 0.132;
	pdb_pair_prob( 135, 3 ) = 0.331;
	pdb_pair_prob( 135, 4 ) = 0.483;
	pdb_pair_prob( 135, 5 ) = 0.000;
	pdb_pair_prob( 136, 1 ) = 0.058;
	pdb_pair_prob( 136, 2 ) = 0.131;
	pdb_pair_prob( 136, 3 ) = 0.326;
	pdb_pair_prob( 136, 4 ) = 0.485;
	pdb_pair_prob( 136, 5 ) = 0.000;
	pdb_pair_prob( 137, 1 ) = 0.053;
	pdb_pair_prob( 137, 2 ) = 0.135;
	pdb_pair_prob( 137, 3 ) = 0.346;
	pdb_pair_prob( 137, 4 ) = 0.466;
	pdb_pair_prob( 137, 5 ) = 0.000;
	pdb_pair_prob( 138, 1 ) = 0.047;
	pdb_pair_prob( 138, 2 ) = 0.125;
	pdb_pair_prob( 138, 3 ) = 0.338;
	pdb_pair_prob( 138, 4 ) = 0.490;
	pdb_pair_prob( 138, 5 ) = 0.000;
	pdb_pair_prob( 139, 1 ) = 0.051;
	pdb_pair_prob( 139, 2 ) = 0.140;
	pdb_pair_prob( 139, 3 ) = 0.326;
	pdb_pair_prob( 139, 4 ) = 0.483;
	pdb_pair_prob( 139, 5 ) = 0.000;
	pdb_pair_prob( 140, 1 ) = 0.049;
	pdb_pair_prob( 140, 2 ) = 0.145;
	pdb_pair_prob( 140, 3 ) = 0.321;
	pdb_pair_prob( 140, 4 ) = 0.486;
	pdb_pair_prob( 140, 5 ) = 0.000;
	pdb_pair_prob( 141, 1 ) = 0.053;
	pdb_pair_prob( 141, 2 ) = 0.129;
	pdb_pair_prob( 141, 3 ) = 0.355;
	pdb_pair_prob( 141, 4 ) = 0.464;
	pdb_pair_prob( 141, 5 ) = 0.000;
	pdb_pair_prob( 142, 1 ) = 0.051;
	pdb_pair_prob( 142, 2 ) = 0.136;
	pdb_pair_prob( 142, 3 ) = 0.330;
	pdb_pair_prob( 142, 4 ) = 0.484;
	pdb_pair_prob( 142, 5 ) = 0.000;
	pdb_pair_prob( 143, 1 ) = 0.047;
	pdb_pair_prob( 143, 2 ) = 0.143;
	pdb_pair_prob( 143, 3 ) = 0.342;
	pdb_pair_prob( 143, 4 ) = 0.469;
	pdb_pair_prob( 143, 5 ) = 0.000;
	pdb_pair_prob( 144, 1 ) = 0.051;
	pdb_pair_prob( 144, 2 ) = 0.141;
	pdb_pair_prob( 144, 3 ) = 0.329;
	pdb_pair_prob( 144, 4 ) = 0.479;
	pdb_pair_prob( 144, 5 ) = 0.000;
	pdb_pair_prob( 145, 1 ) = 0.049;
	pdb_pair_prob( 145, 2 ) = 0.140;
	pdb_pair_prob( 145, 3 ) = 0.342;
	pdb_pair_prob( 145, 4 ) = 0.468;
	pdb_pair_prob( 145, 5 ) = 0.000;
	pdb_pair_prob( 146, 1 ) = 0.054;
	pdb_pair_prob( 146, 2 ) = 0.142;
	pdb_pair_prob( 146, 3 ) = 0.321;
	pdb_pair_prob( 146, 4 ) = 0.484;
	pdb_pair_prob( 146, 5 ) = 0.000;
	pdb_pair_prob( 147, 1 ) = 0.051;
	pdb_pair_prob( 147, 2 ) = 0.133;
	pdb_pair_prob( 147, 3 ) = 0.331;
	pdb_pair_prob( 147, 4 ) = 0.486;
	pdb_pair_prob( 147, 5 ) = 0.000;
	pdb_pair_prob( 148, 1 ) = 0.049;
	pdb_pair_prob( 148, 2 ) = 0.133;
	pdb_pair_prob( 148, 3 ) = 0.323;
	pdb_pair_prob( 148, 4 ) = 0.495;
	pdb_pair_prob( 148, 5 ) = 0.000;
	pdb_pair_prob( 149, 1 ) = 0.054;
	pdb_pair_prob( 149, 2 ) = 0.136;
	pdb_pair_prob( 149, 3 ) = 0.332;
	pdb_pair_prob( 149, 4 ) = 0.478;
	pdb_pair_prob( 149, 5 ) = 0.000;
	pdb_pair_prob( 150, 1 ) = 0.052;
	pdb_pair_prob( 150, 2 ) = 0.130;
	pdb_pair_prob( 150, 3 ) = 0.325;
	pdb_pair_prob( 150, 4 ) = 0.493;
	pdb_pair_prob( 150, 5 ) = 0.000;
	pdb_pair_prob( 151, 1 ) = 0.050;
	pdb_pair_prob( 151, 2 ) = 0.148;
	pdb_pair_prob( 151, 3 ) = 0.340;
	pdb_pair_prob( 151, 4 ) = 0.462;
	pdb_pair_prob( 151, 5 ) = 0.000;
	pdb_pair_prob( 152, 1 ) = 0.051;
	pdb_pair_prob( 152, 2 ) = 0.139;
	pdb_pair_prob( 152, 3 ) = 0.326;
	pdb_pair_prob( 152, 4 ) = 0.484;
	pdb_pair_prob( 152, 5 ) = 0.000;
	pdb_pair_prob( 153, 1 ) = 0.049;
	pdb_pair_prob( 153, 2 ) = 0.118;
	pdb_pair_prob( 153, 3 ) = 0.334;
	pdb_pair_prob( 153, 4 ) = 0.499;
	pdb_pair_prob( 153, 5 ) = 0.000;
	pdb_pair_prob( 154, 1 ) = 0.048;
	pdb_pair_prob( 154, 2 ) = 0.132;
	pdb_pair_prob( 154, 3 ) = 0.316;
	pdb_pair_prob( 154, 4 ) = 0.504;
	pdb_pair_prob( 154, 5 ) = 0.000;
	pdb_pair_prob( 155, 1 ) = 0.046;
	pdb_pair_prob( 155, 2 ) = 0.129;
	pdb_pair_prob( 155, 3 ) = 0.341;
	pdb_pair_prob( 155, 4 ) = 0.484;
	pdb_pair_prob( 155, 5 ) = 0.000;
	pdb_pair_prob( 156, 1 ) = 0.051;
	pdb_pair_prob( 156, 2 ) = 0.138;
	pdb_pair_prob( 156, 3 ) = 0.318;
	pdb_pair_prob( 156, 4 ) = 0.494;
	pdb_pair_prob( 156, 5 ) = 0.000;
	pdb_pair_prob( 157, 1 ) = 0.054;
	pdb_pair_prob( 157, 2 ) = 0.145;
	pdb_pair_prob( 157, 3 ) = 0.330;
	pdb_pair_prob( 157, 4 ) = 0.470;
	pdb_pair_prob( 157, 5 ) = 0.000;
	pdb_pair_prob( 158, 1 ) = 0.046;
	pdb_pair_prob( 158, 2 ) = 0.141;
	pdb_pair_prob( 158, 3 ) = 0.338;
	pdb_pair_prob( 158, 4 ) = 0.475;
	pdb_pair_prob( 158, 5 ) = 0.000;
	pdb_pair_prob( 159, 1 ) = 0.041;
	pdb_pair_prob( 159, 2 ) = 0.140;
	pdb_pair_prob( 159, 3 ) = 0.324;
	pdb_pair_prob( 159, 4 ) = 0.494;
	pdb_pair_prob( 159, 5 ) = 0.000;
	pdb_pair_prob( 160, 1 ) = 0.051;
	pdb_pair_prob( 160, 2 ) = 0.135;
	pdb_pair_prob( 160, 3 ) = 0.332;
	pdb_pair_prob( 160, 4 ) = 0.483;
	pdb_pair_prob( 160, 5 ) = 0.000;
	pdb_pair_prob( 161, 1 ) = 0.049;
	pdb_pair_prob( 161, 2 ) = 0.126;
	pdb_pair_prob( 161, 3 ) = 0.341;
	pdb_pair_prob( 161, 4 ) = 0.484;
	pdb_pair_prob( 161, 5 ) = 0.000;
	pdb_pair_prob( 162, 1 ) = 0.052;
	pdb_pair_prob( 162, 2 ) = 0.132;
	pdb_pair_prob( 162, 3 ) = 0.320;
	pdb_pair_prob( 162, 4 ) = 0.497;
	pdb_pair_prob( 162, 5 ) = 0.000;
	pdb_pair_prob( 163, 1 ) = 0.047;
	pdb_pair_prob( 163, 2 ) = 0.131;
	pdb_pair_prob( 163, 3 ) = 0.338;
	pdb_pair_prob( 163, 4 ) = 0.485;
	pdb_pair_prob( 163, 5 ) = 0.000;
	pdb_pair_prob( 164, 1 ) = 0.050;
	pdb_pair_prob( 164, 2 ) = 0.135;
	pdb_pair_prob( 164, 3 ) = 0.332;
	pdb_pair_prob( 164, 4 ) = 0.484;
	pdb_pair_prob( 164, 5 ) = 0.000;
	pdb_pair_prob( 165, 1 ) = 0.058;
	pdb_pair_prob( 165, 2 ) = 0.146;
	pdb_pair_prob( 165, 3 ) = 0.327;
	pdb_pair_prob( 165, 4 ) = 0.470;
	pdb_pair_prob( 165, 5 ) = 0.000;
	pdb_pair_prob( 166, 1 ) = 0.051;
	pdb_pair_prob( 166, 2 ) = 0.140;
	pdb_pair_prob( 166, 3 ) = 0.324;
	pdb_pair_prob( 166, 4 ) = 0.486;
	pdb_pair_prob( 166, 5 ) = 0.000;
	pdb_pair_prob( 167, 1 ) = 0.047;
	pdb_pair_prob( 167, 2 ) = 0.139;
	pdb_pair_prob( 167, 3 ) = 0.345;
	pdb_pair_prob( 167, 4 ) = 0.469;
	pdb_pair_prob( 167, 5 ) = 0.000;
	pdb_pair_prob( 168, 1 ) = 0.044;
	pdb_pair_prob( 168, 2 ) = 0.147;
	pdb_pair_prob( 168, 3 ) = 0.333;
	pdb_pair_prob( 168, 4 ) = 0.477;
	pdb_pair_prob( 168, 5 ) = 0.000;
	pdb_pair_prob( 169, 1 ) = 0.050;
	pdb_pair_prob( 169, 2 ) = 0.149;
	pdb_pair_prob( 169, 3 ) = 0.331;
	pdb_pair_prob( 169, 4 ) = 0.470;
	pdb_pair_prob( 169, 5 ) = 0.000;
	pdb_pair_prob( 170, 1 ) = 0.051;
	pdb_pair_prob( 170, 2 ) = 0.139;
	pdb_pair_prob( 170, 3 ) = 0.331;
	pdb_pair_prob( 170, 4 ) = 0.479;
	pdb_pair_prob( 170, 5 ) = 0.000;
	pdb_pair_prob( 171, 1 ) = 0.049;
	pdb_pair_prob( 171, 2 ) = 0.142;
	pdb_pair_prob( 171, 3 ) = 0.328;
	pdb_pair_prob( 171, 4 ) = 0.482;
	pdb_pair_prob( 171, 5 ) = 0.000;
	pdb_pair_prob( 172, 1 ) = 0.045;
	pdb_pair_prob( 172, 2 ) = 0.142;
	pdb_pair_prob( 172, 3 ) = 0.339;
	pdb_pair_prob( 172, 4 ) = 0.474;
	pdb_pair_prob( 172, 5 ) = 0.000;
	pdb_pair_prob( 173, 1 ) = 0.053;
	pdb_pair_prob( 173, 2 ) = 0.142;
	pdb_pair_prob( 173, 3 ) = 0.322;
	pdb_pair_prob( 173, 4 ) = 0.484;
	pdb_pair_prob( 173, 5 ) = 0.000;
	pdb_pair_prob( 174, 1 ) = 0.054;
	pdb_pair_prob( 174, 2 ) = 0.131;
	pdb_pair_prob( 174, 3 ) = 0.356;
	pdb_pair_prob( 174, 4 ) = 0.460;
	pdb_pair_prob( 174, 5 ) = 0.000;
	pdb_pair_prob( 175, 1 ) = 0.048;
	pdb_pair_prob( 175, 2 ) = 0.134;
	pdb_pair_prob( 175, 3 ) = 0.312;
	pdb_pair_prob( 175, 4 ) = 0.507;
	pdb_pair_prob( 175, 5 ) = 0.000;
	pdb_pair_prob( 176, 1 ) = 0.051;
	pdb_pair_prob( 176, 2 ) = 0.130;
	pdb_pair_prob( 176, 3 ) = 0.320;
	pdb_pair_prob( 176, 4 ) = 0.499;
	pdb_pair_prob( 176, 5 ) = 0.000;
	pdb_pair_prob( 177, 1 ) = 0.054;
	pdb_pair_prob( 177, 2 ) = 0.139;
	pdb_pair_prob( 177, 3 ) = 0.327;
	pdb_pair_prob( 177, 4 ) = 0.480;
	pdb_pair_prob( 177, 5 ) = 0.000;
	pdb_pair_prob( 178, 1 ) = 0.037;
	pdb_pair_prob( 178, 2 ) = 0.151;
	pdb_pair_prob( 178, 3 ) = 0.314;
	pdb_pair_prob( 178, 4 ) = 0.498;
	pdb_pair_prob( 178, 5 ) = 0.000;
	pdb_pair_prob( 179, 1 ) = 0.048;
	pdb_pair_prob( 179, 2 ) = 0.137;
	pdb_pair_prob( 179, 3 ) = 0.329;
	pdb_pair_prob( 179, 4 ) = 0.486;
	pdb_pair_prob( 179, 5 ) = 0.000;
	pdb_pair_prob( 180, 1 ) = 0.048;
	pdb_pair_prob( 180, 2 ) = 0.128;
	pdb_pair_prob( 180, 3 ) = 0.345;
	pdb_pair_prob( 180, 4 ) = 0.479;
	pdb_pair_prob( 180, 5 ) = 0.000;
	pdb_pair_prob( 181, 1 ) = 0.043;
	pdb_pair_prob( 181, 2 ) = 0.153;
	pdb_pair_prob( 181, 3 ) = 0.321;
	pdb_pair_prob( 181, 4 ) = 0.484;
	pdb_pair_prob( 181, 5 ) = 0.000;
	pdb_pair_prob( 182, 1 ) = 0.051;
	pdb_pair_prob( 182, 2 ) = 0.138;
	pdb_pair_prob( 182, 3 ) = 0.331;
	pdb_pair_prob( 182, 4 ) = 0.479;
	pdb_pair_prob( 182, 5 ) = 0.000;
	pdb_pair_prob( 183, 1 ) = 0.040;
	pdb_pair_prob( 183, 2 ) = 0.148;
	pdb_pair_prob( 183, 3 ) = 0.320;
	pdb_pair_prob( 183, 4 ) = 0.492;
	pdb_pair_prob( 183, 5 ) = 0.000;
	pdb_pair_prob( 184, 1 ) = 0.054;
	pdb_pair_prob( 184, 2 ) = 0.133;
	pdb_pair_prob( 184, 3 ) = 0.305;
	pdb_pair_prob( 184, 4 ) = 0.508;
	pdb_pair_prob( 184, 5 ) = 0.000;
	pdb_pair_prob( 185, 1 ) = 0.042;
	pdb_pair_prob( 185, 2 ) = 0.147;
	pdb_pair_prob( 185, 3 ) = 0.358;
	pdb_pair_prob( 185, 4 ) = 0.453;
	pdb_pair_prob( 185, 5 ) = 0.000;
	pdb_pair_prob( 186, 1 ) = 0.054;
	pdb_pair_prob( 186, 2 ) = 0.138;
	pdb_pair_prob( 186, 3 ) = 0.324;
	pdb_pair_prob( 186, 4 ) = 0.484;
	pdb_pair_prob( 186, 5 ) = 0.000;
	pdb_pair_prob( 187, 1 ) = 0.050;
	pdb_pair_prob( 187, 2 ) = 0.135;
	pdb_pair_prob( 187, 3 ) = 0.312;
	pdb_pair_prob( 187, 4 ) = 0.503;
	pdb_pair_prob( 187, 5 ) = 0.000;
	pdb_pair_prob( 188, 1 ) = 0.054;
	pdb_pair_prob( 188, 2 ) = 0.141;
	pdb_pair_prob( 188, 3 ) = 0.334;
	pdb_pair_prob( 188, 4 ) = 0.471;
	pdb_pair_prob( 188, 5 ) = 0.000;
	pdb_pair_prob( 189, 1 ) = 0.044;
	pdb_pair_prob( 189, 2 ) = 0.135;
	pdb_pair_prob( 189, 3 ) = 0.341;
	pdb_pair_prob( 189, 4 ) = 0.480;
	pdb_pair_prob( 189, 5 ) = 0.000;
	pdb_pair_prob( 190, 1 ) = 0.049;
	pdb_pair_prob( 190, 2 ) = 0.129;
	pdb_pair_prob( 190, 3 ) = 0.328;
	pdb_pair_prob( 190, 4 ) = 0.494;
	pdb_pair_prob( 190, 5 ) = 0.000;
	pdb_pair_prob( 191, 1 ) = 0.045;
	pdb_pair_prob( 191, 2 ) = 0.126;
	pdb_pair_prob( 191, 3 ) = 0.330;
	pdb_pair_prob( 191, 4 ) = 0.499;
	pdb_pair_prob( 191, 5 ) = 0.000;
	pdb_pair_prob( 192, 1 ) = 0.048;
	pdb_pair_prob( 192, 2 ) = 0.135;
	pdb_pair_prob( 192, 3 ) = 0.338;
	pdb_pair_prob( 192, 4 ) = 0.479;
	pdb_pair_prob( 192, 5 ) = 0.000;
	pdb_pair_prob( 193, 1 ) = 0.048;
	pdb_pair_prob( 193, 2 ) = 0.133;
	pdb_pair_prob( 193, 3 ) = 0.339;
	pdb_pair_prob( 193, 4 ) = 0.479;
	pdb_pair_prob( 193, 5 ) = 0.000;
	pdb_pair_prob( 194, 1 ) = 0.047;
	pdb_pair_prob( 194, 2 ) = 0.133;
	pdb_pair_prob( 194, 3 ) = 0.320;
	pdb_pair_prob( 194, 4 ) = 0.500;
	pdb_pair_prob( 194, 5 ) = 0.000;
	pdb_pair_prob( 195, 1 ) = 0.043;
	pdb_pair_prob( 195, 2 ) = 0.128;
	pdb_pair_prob( 195, 3 ) = 0.336;
	pdb_pair_prob( 195, 4 ) = 0.493;
	pdb_pair_prob( 195, 5 ) = 0.000;
	pdb_pair_prob( 196, 1 ) = 0.046;
	pdb_pair_prob( 196, 2 ) = 0.144;
	pdb_pair_prob( 196, 3 ) = 0.303;
	pdb_pair_prob( 196, 4 ) = 0.507;
	pdb_pair_prob( 196, 5 ) = 0.000;
	pdb_pair_prob( 197, 1 ) = 0.044;
	pdb_pair_prob( 197, 2 ) = 0.132;
	pdb_pair_prob( 197, 3 ) = 0.330;
	pdb_pair_prob( 197, 4 ) = 0.494;
	pdb_pair_prob( 197, 5 ) = 0.000;
	pdb_pair_prob( 198, 1 ) = 0.049;
	pdb_pair_prob( 198, 2 ) = 0.140;
	pdb_pair_prob( 198, 3 ) = 0.333;
	pdb_pair_prob( 198, 4 ) = 0.478;
	pdb_pair_prob( 198, 5 ) = 0.000;
	pdb_pair_prob( 199, 1 ) = 0.041;
	pdb_pair_prob( 199, 2 ) = 0.142;
	pdb_pair_prob( 199, 3 ) = 0.323;
	pdb_pair_prob( 199, 4 ) = 0.494;
	pdb_pair_prob( 199, 5 ) = 0.000;
	pdb_pair_prob( 200, 1 ) = 0.050;
	pdb_pair_prob( 200, 2 ) = 0.131;
	pdb_pair_prob( 200, 3 ) = 0.338;
	pdb_pair_prob( 200, 4 ) = 0.481;
	pdb_pair_prob( 200, 5 ) = 0.000;
}

}
