// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision:
//  $Date:
//  $Author:

//=============================================================================
//
//  add -surface mode to model proteins with surfaces eg mica, hydroxyapatite
//
//		Kosta Makrodimitris,Ph.D. (KMa) 2006-02
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
// Surface  inherit properties
// and functions from Ligand-Dock modes (translation, rotation, monte-carlo,
// minimize etc)
// but will have additional geometric features because of their symmetry.

// Surfaces have differences comparing to a ligand or a protein
// and for this reason
// they should be a different objects in Rosetta and need different
// functions and treatment. Surfaces have a defined symmetry
//(CUBIC, ORTHOGONAL, HEXAGONAL etc) and space group.
// In the most cases their dimensions are defined through the
// concept of a unit cell // in X,Y,Z directions.
// Usually the protein docks in the XY plane and the Z dimension is
// as thin as possible (Rosetta cutoff is 0.6-0.8nm).
// The protein contact surface that
// we model is around 100nm2 and so an adequate surface area
// should be around 1000nm2 (more atoms comparing to a ligand).
// The number of atoms for a surface (2000-5000)
// is larger than a ligand (100-200).

// cases:
// -Hydroxyapatite-statherin
// -MICA-Top7
// -QUARTZ-Lysozyme
// -GOLD, SILVER,-peptides
// -C4, C18 (chromatography) - peptides
// -Self assembled peptides


// Rosetta Headers
#include "surface.h"
#include "param.h"
#include "misc.h"//KMa surface

// ObjexxFCL Headers
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.io.hh>

//C++ standard template headers
#include <vector>
#include <cmath>
#include <list>


//==========================================================================
//
//     Surface
//
//	kosta makrodimitris 2006-02
//
//==========================================================================


// Default Constructor for Surface object
Surface::Surface()
{}


// Destructor
Surface::~Surface()
{}
////////////////////////////////////////////////////////////////////////////



// ===========================================================================
// function to calculate a parametric equation of a line (pointXYZ and vector)
// ===========================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin Surface::parametric_line
///
/// @brief
/// function to calculate a parametric equation of a line (pointXYZ and vector)
///
/// @detailed
///
/// @param
///		in standard XYZpoint, vector and parameter t
///		out PARAMETRIC line
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kosta Makrodimitris
/// @last_modified	2006-02
/////////////////////////////////////////////////////////////////////////////

numeric::xyzVector<double>
Surface::parametric_line ( numeric::xyzVector<double> point, double t,
						   numeric::xyzVector<double> parallel_vector )
{

	numeric::xyzVector<double> line;
	//??????????????????????????????????

	line = point + t * parallel_vector;

	return line;
}
//////////////////////////////////////////////////////////////////////////////



// ===========================================================================
//  function to calculate the normal for a plane baseD on 3 points
//
//	        A...B
//			.
//			.
//			.
//			C
//
// ==========================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin Surface::normalto3points
///
/// @brief
///  function to calculate the normal for a plane baseD on 3 points
///
/// @detailed
///
/// @param
///		in 3 xyz points
///		out normal vector
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kosta Makrodimitris
/// @last_modified	2006-02
//////////////////////////////////////////////////////////////////////////////

numeric::xyzVector<double>
Surface::normalto3points ( numeric::xyzVector<double> Apoint,
						   numeric::xyzVector<double> Bpoint,
						   numeric::xyzVector<double> Cpoint )
{
	numeric::xyzVector<double> AB,AC,normal;
 	//??????????????????????????????????????

		AB = Bpoint - Apoint;

		AC = Cpoint - Apoint;

		normal = cross_product(AB,AC);

	return normal;
}
//////////////////////////////////////////////////////////////////////////////



//////////////////////////////////////////////////////////////////////////////



// ===========================================================================
//  function to calculate the point from 2 intersected lines
//
//	        A...B
//			.
//			.
//			.
//			C
//
// ===========================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin Surface::point2lines2
///
/// @brief
///  function to calculate the point from 2 intersected lines
///
/// @detailed
///
/// @param
///		in 2 xyz points and 2 vectors
///		out intersection xyz point
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kosta Makrodimitris
/// @last_modified	2006-02
////////////////////////////////////////////////////////////////////////////

numeric::xyzVector<double>
Surface::point2lines2 ( numeric::xyzVector<double> point1,
						numeric::xyzVector<double> point2,
						numeric::xyzVector<double> Bvector,
						numeric::xyzVector<double> Cvector)
{

	double DD,t ;
	numeric::xyzVector<double> crossBC, CB, CB_C, intersection, inv_crossBC ;
 	//???????????????????????????????????????????????????????????????????????

	  CB = point2 - point1 ;

      crossBC = cross_product ( Bvector, Cvector );

	  CB_C = cross_product ( Cvector, crossBC );

	  DD = inner_product ( crossBC,crossBC);

	  t = inner_product( CB, CB_C )/ DD;

	  intersection = point1 + t * Bvector;

		   return intersection;
}
//////////////////////////////////////////////////////////////////////////////



// ==========================================================================
//  function to calculate the point from 2 intersected lines
//
//	        A...B
//			.
//			.
//			.
//			C
//
// ==========================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin Surface::point2lines3
///
/// @brief
///  function to calculate the point from 2 intersected lines
///
/// @detailed
///
/// @param
///		in 2 xyz points and 2 vectors
///		out intersection xyz point
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kosta Makrodimitris
/// @last_modified	2006-02
//////////////////////////////////////////////////////////////////////////////


numeric::xyzVector<double>
Surface::point2lines3 ( numeric::xyzVector<double> point1,
						numeric::xyzVector<double> point2,
						numeric::xyzVector<double> Bvector,
						numeric::xyzVector<double> Cvector)
{

double t,DD,w ;
numeric::xyzVector<double> crossBC,cross1C,CB,CB_C,intersection,inv_crossBC;
 	//??????????????????????????????????????

	  CB = point2 - point1 ;

      cross1C = cross_product ( CB, Cvector );

      crossBC = cross_product ( Bvector, Cvector );

	   w = inner_product ( cross1C, crossBC);

	  DD = inner_product ( crossBC, crossBC);

		t = w / DD;

	  intersection = point1 + t * Bvector;

		   return intersection;
}
//////////////////////////////////////////////////////////////////////////////



// ============================================================================
//  function to calculate the plane from 3 points XYZ
//
//	        A...B
//			.
//			.
//			.
//			C
//
// ============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin Surface::plane3points
///
/// @brief
///  function to calculate the plane ax+by+cz+d
///
/// @detailed
///
/// @param
///		in 3 xyz points
///		out 4 NUMBERS THAT DEFINE THE PLANE
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kosta Makrodimitris
/// @last_modified	2006-02
///////////////////////////////////////////////////////////////////////////////

 std::vector<double>
Surface::plane3points ( numeric::xyzVector<double> Apoint,
						numeric::xyzVector<double> Bpoint,
						numeric::xyzVector<double> Cpoint )
 {

	numeric::xyzVector<double> normalto3 ;
 	//??????????????????????????????????????

    normalto3 = normalto3points(  Apoint,  Bpoint, Cpoint );

	plane123.push_back(normalto3 [0]);

	plane123.push_back(normalto3 [1]);

	plane123.push_back(normalto3 [2]);

	plane123.push_back
		(- normalto3 [0] * Apoint[0]  -
		normalto3 [1] * Apoint[1]  - normalto3 [2] * Apoint[2] );

		   return plane123;
}
///////////////////////////////////////////////////////////////////////////////





// ============================================================================
//function to calculate the point in a plane hit from a 3d point
//
//	        A...B
//			.
//			.
//			.
//			C
//
// ============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin Surface::point_intersect_plane
///
/// @brief
/// function to calculate the point in a plane hit from
/// a point out of plane normal
///
/// @detailed
///
/// @param
///		in  xyz point, plane abcd, normal
///		out point intersection
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kosta Makrodimitris
/// @last_modified	2006-02
////////////////////////////////////////////////////////////////////////////////

numeric::xyzVector<double>
Surface::point_intersect_plane ( std::vector<double>  plane_abcd,
								 numeric::xyzVector<double> point_outofplane,
								 numeric::xyzVector<double> normal_plane )
{
	double t;
	numeric::xyzVector<double> point_intersection;
 	//??????????????????????????????????????

	t=
		-(  plane_abcd [0] * point_outofplane [0] +
			plane_abcd [1] * point_outofplane [1] +
			plane_abcd [2] * point_outofplane [2] + plane_abcd [3] )
		/
		 (  plane_abcd [0] * plane_abcd [0] + plane_abcd [1] * plane_abcd [1]
		  + plane_abcd [2] * plane_abcd [2] );

	point_intersection = point_outofplane - t * normal_plane;

		   return point_intersection;
}
////////////////////////////////////////////////////////////////////////////////

