// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "taboo_search.h"
#include "after_opts.h"
#include "files_paths.h"
#include "force_barcode.h"
#include "misc.h"
#include "monte_carlo.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>
#include <utility/io/orstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>


namespace taboo_map_ns {

	//-------------------------------
	// taboo search params
	//-------------------------------

	bool taboo_mode_flag = false;

	bool taboo_flag = false;

	// size of the phipsi angle bin width
	int BB_BIN_WIDTH;

	int TENURE;

	// taboo_map stores all the history of accepted moves
	cTabooMap taboo_map;

	// active_taboo_map stores the currently active taboos
	cTabooMap active_taboo_map;

	cTabooMap imported_taboo_map;

	cTabooMap trace_taboo_map;

	cTabooMap tried_taboo_map;

	cTrace aspiration_sites;

	cTrace search_trace;

	int taboo_mode = 1; // mode for scoring taboo_bonus

	std::string taboo_list_file;

	bool increment_taboo_flag = false;

	bool taboo_diagnosis = false;

	bool taboo_penalize_trace = false;

	bool taboo_penalize_tried = false;

	std::string nat_code;

	std::string taboo_header;

	bool taboo_continuous = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTaboo::cTaboo
///
/// @brief: constructor for class cTaboo
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
cTaboo::cTaboo( int s, int t, int i )
{
		score = s;
		tenure = t;
		id = i;
}

cTaboo::cTaboo()
{
		score = 0;
		tenure = 100000;
		id = 0;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin cTaboo::~cTaboo
///
/// @brief: destructor for class cTaboo
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
cTaboo::~cTaboo()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTabooMap::cTabooMap
///
/// @brief: constructor for class cTabooMap
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
cTabooMap::cTabooMap()
{
	taboo.clear();
	aspiration = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTabooMap::clear
///
/// @brief: clear a cTabooMap
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/8/04
/////////////////////////////////////////////////////////////////////////////////
void
cTabooMap::clear()
{
	taboo.clear();
	aspiration = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTabooMap::~cTabooMap
///
/// @brief: destructor for class cTabooMap
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
cTabooMap::~cTabooMap()
{
	taboo.clear();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_current_barcode
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
std::string
get_current_barcode()
{
	using namespace taboo_map_ns;
	using namespace misc;

	std::string current_barcode;

	FArray1D_float phi_new;
	FArray1D_float psi_new;
	phi_new.dimension( total_residue );
	psi_new.dimension( total_residue );

	for ( int i = 1; i <= total_residue; ++i ) {
		phi_new(i) = set_phipsi_range( phi(i) );
		psi_new(i) = set_phipsi_range( psi(i) );
	}

	if ( taboo_mode == 1 ) {
		current_barcode = barcode_vector( phi_new, psi_new, 5, total_residue-4 );

	} else if ( taboo_mode == 2 ) {
		current_barcode =
			map_to_string(
				get_closest_barcode( phi_new, psi_new )
				);
	} else if ( taboo_mode == 3 ) {
		current_barcode = feature_barcode_vector( phi_new, psi_new );
	}

	return current_barcode;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_best_barcode
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
std::string
get_best_barcode() {

	using namespace taboo_map_ns;
	using namespace misc;

	std::string best_barcode;

	FArray1D_float phi_new;
	FArray1D_float psi_new;
	phi_new.dimension( total_residue );
	psi_new.dimension( total_residue );

	for ( int i = 1; i <= total_residue; ++i ) {
		phi_new(i) = set_phipsi_range( best_phi(i) );
		psi_new(i) = set_phipsi_range( best_psi(i) );
	}


	if ( taboo_mode == 1 ) {
		best_barcode = barcode_vector( phi_new, psi_new, 5, total_residue-4 );

	} else if ( taboo_mode == 2 ) {
		best_barcode =
			map_to_string(
				get_closest_barcode( phi_new, psi_new )
				);
	} else if ( taboo_mode == 3 ) {
		best_barcode = feature_barcode_vector( phi_new, psi_new );
	}

	return best_barcode;

}

///////////////////////////////////////////////////////////////////////////////_
/// @begin taboo_new_move
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
void
taboo_new_move()
{
	using namespace taboo_map_ns;

	if ( !get_taboo_exist() ) return;

	if ( taboo_penalize_tried ) {
		std::string barcode = get_current_barcode();
		active_taboo_map.update( barcode );
		tried_taboo_map.update( barcode );
	}
	active_taboo_map.reduce_tenure();
	active_taboo_map.remove_expired();
//		std::cout << "taboo_exist in monte carlo! "
//				<< active_taboo_map.taboo.size() << " "
//				<< scores::taboo_score << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin taboo_accept
///
/// @brief: things to do after monte_carlo accept
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
void taboo_accept()
{
	using namespace taboo_map_ns;
	using namespace misc;

	using namespace mc_global_track::mc_score; // yab: misc removal

	if ( !get_taboo_exist() ) return;

	if ( taboo_penalize_trace ) {
		std::string barcode = get_current_barcode();
		active_taboo_map.update( barcode );
		active_taboo_map.remove_expired();
		trace_taboo_map.update( barcode );
		if ( taboo_diagnosis ) search_trace.update( phi, psi, score );
//	} else if ( taboo_penalize_tried ) {
//		if ( taboo_diagnosis ) search_trace.update( phi, psi, score );
	} else if ( taboo_diagnosis ) {
//		taboo_map.update( barcode );
		search_trace.update( phi, psi, score );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin write_taboo_file
///
/// @brief: write out taboo
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/15/04
/////////////////////////////////////////////////////////////////////////////////
void
write_taboo_file( utility::io::orstream & taboo_out_stream, std::string const & tag )
{
	using namespace files_paths;
	using namespace taboo_map_ns;

	std::string current_barcode = get_current_barcode();
	taboo_out_stream
	 << SS( tag )
	 << SS( 1 )
	 << SS( 99999 )
	 << SS( current_barcode )
	 << std::endl;

	if ( taboo_penalize_trace ) {
		// add all the barcodes seen into active_taboo_map
		std::map< std::string, cTaboo >::iterator pos;
		for ( pos = trace_taboo_map.taboo.begin(); pos != trace_taboo_map.taboo.end(); ++pos ) {
			taboo_out_stream
			 << SS( tag + "_t" )
			 << SS( pos->second.score )
			 << SS( 99999 )
			 << SS( pos->first )
			 << std::endl;
		}
	}

	if ( taboo_penalize_tried ) {
		// add all the barcodes seen into active_taboo_map
		std::map< std::string, cTaboo >::iterator pos;
		for ( pos = tried_taboo_map.taboo.begin(); pos != tried_taboo_map.taboo.end(); ++pos ) {
			taboo_out_stream
			 << SS( tag + "_t" )
			 << SS( pos->second.score )
			 << SS( 99999 )
			 << SS( pos->first )
			 << std::endl;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin taboo_bonus
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
float
taboo_bonus()
{
	using namespace taboo_map_ns;
	using namespace misc;

	float bonus = 0.0;

	std::string current_barcode;
	std::string best_barcode;

	current_barcode = get_current_barcode();
	if( best_phi.size() != 0 && best_psi.size() != 0 )
		best_barcode = get_best_barcode();
	else return bonus;

	int current_taboo_score = active_taboo_map.score( current_barcode );
	int best_taboo_score = active_taboo_map.score( best_barcode );
	bonus = ( current_taboo_score - best_taboo_score );

	if ( taboo_mode == 1 ) {
		float influence = taboo_influence();
		bonus *= influence;
	}

	//aspiration should have different effect
	if ( active_taboo_map.aspiration ) bonus = -bonus;

	return bonus;

}

float
taboo_b()
{
	using namespace taboo_map_ns;

	float bonus;

	std::string barcode = get_current_barcode();
	int current_taboo_score =
			active_taboo_map.score( barcode );
	bonus = current_taboo_score;

	return bonus;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin taboo_influence
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
float
taboo_influence()
{
	using namespace taboo_map_ns;
	using namespace misc;

	float influence = 0;
	std::string buf;

	std::string current_barcode = get_current_barcode();
	std::string best_barcode = get_best_barcode();

	std::stringstream current(current_barcode);
	std::stringstream best(best_barcode);

	std::vector< std::string > current_vector;
	std::vector< std::string > best_vector;

	while ( current >> buf )
		current_vector.push_back( buf );

	while ( best >> buf )
		best_vector.push_back( buf );

	int vec_length = current_vector.size();
	for ( int i = 0; i < vec_length; ++i ) {
		if ( current_vector.at( i ) != best_vector.at( i ) )
			++influence;
	}

//	influence /= 2.0; // one pair of phipsi only count once
	//std::cout << influence << std::endl;
	return influence;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_taboo
///
/// @brief: initiate parameters for taboo search
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
void
initialize_taboo()
{
	using namespace taboo_map_ns;

	if ( !get_taboo_mode_exist() ) return;

	intafteroption( "taboo_tenure",99999,TENURE );
	intafteroption( "bb_bin_width",30,BB_BIN_WIDTH );
	stringafteroption( "taboo_file", "", taboo_list_file );
	intafteroption( "taboo_mode", 1 ,taboo_mode );
	if ( truefalseoption ("increment_taboo") ) increment_taboo_flag = true;
	if ( truefalseoption ("taboo_diagnosis") ) taboo_diagnosis = true;
	if ( truefalseoption ("taboo_penalize_trace") ) taboo_penalize_trace = true;
	if ( truefalseoption ("taboo_penalize_tried") ) taboo_penalize_tried = true;
	set_taboo_active( true );

	stringafteroption( "nat_code", "", nat_code );

	stringafteroption( "taboo_header", files_paths::code, taboo_header );

	if ( truefalseoption ("taboo_continuous") ) taboo_continuous = true;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_taboo_start
///
/// @brief: initialize parameters for a new start structure
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/14/04
/////////////////////////////////////////////////////////////////////////////////
void
initialize_taboo_start()
{
	using namespace taboo_map_ns;
	using namespace files_paths;

	if ( !get_taboo_mode_exist() ) return;

	set_taboo_active( true );

	active_taboo_map.clear();
	aspiration_sites.clear();
	imported_taboo_map.clear();
	if ( taboo_list_file_exist() ) {
		std::string taboo_list_file = get_taboo_list_file_name();
		imported_taboo_map.update_from_file( taboo_list_file );
		active_taboo_map = imported_taboo_map;
	}
	trace_taboo_map.clear();
	tried_taboo_map.clear();

	std::string fullname = pdb_out_path + taboo_map_ns::taboo_header + protein_name + ".taboo";
	utility::io::ozstream taboo_out_stream( fullname, std::ios_base::in|std::ios_base::out );
	if ( !taboo_out_stream ) {
		taboo_out_stream.open( fullname );
		if ( !taboo_out_stream ) {
			std::cout << "Open failed for file: " << taboo_out_stream.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		taboo_out_stream << "TABOO LIST FILE: " << '\n';

	}
	taboo_out_stream.close();
	taboo_out_stream.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_taboo_decoy
///
/// @brief: initiate taboo parameters for new output
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
void
initialize_taboo_decoy()
{
	using namespace taboo_map_ns;
	using namespace files_paths;

	if ( !get_taboo_mode_exist() ) return;

	set_taboo_active( true );

	taboo_map.taboo.clear();
	search_trace.clear();
	trace_taboo_map.clear();
	tried_taboo_map.clear();

	if ( get_accum_taboo_flag() ) {//also add in new taboo
// starting over with the original imported taboo map
		active_taboo_map = imported_taboo_map;
		std::cout << "Augmenting taboo list..." << std::endl;
		std::string taboo_list_file = pdb_out_path + taboo_header + protein_name + ".taboo";
		active_taboo_map.update_from_file( taboo_list_file );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin conclude_taboo_decoy
///
/// @brief: things to do after output a decoy
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
void
conclude_taboo_decoy()
{
	using namespace taboo_map_ns;
	using namespace files_paths;

	if ( !get_taboo_mode_exist() ) return;

	if ( get_accum_taboo_flag() ) {//also add in new taboo
// starting over with the original imported taboo map
		active_taboo_map = imported_taboo_map;
		std::cout << "augmenting taboo list..." << std::endl;
		std::string taboo_list_file = pdb_out_path + taboo_header + protein_name + ".taboo";
		active_taboo_map.update_from_file( taboo_list_file );
		//std::string current_barcode = get_current_barcode();
		//active_taboo_map.update( current_barcode );
	}

	if ( taboo_diagnosis ) {
		std::cout << "Taboo history: " << std::endl;
		taboo_map.dump();
		std::cout << "Search Trace: " << std::endl;
		search_trace.dump();
		std::cout << "Active_Taboo_Map: " << std::endl;
		active_taboo_map.dump();
		taboo_map.clear();
		search_trace.clear();
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_accum_taboo_flag()
///
/// @brief: get the flag of wether or not you want to accumualte taboo list
/// during nstruct runs
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/13/04
/////////////////////////////////////////////////////////////////////////////////
bool
get_accum_taboo_flag()
{
	return ( taboo_map_ns::increment_taboo_flag );
}



////////////////////////////////////////////////////////////////////////////////
/// @begin set_taboo_flag
///
/// @brief: set_taboo_flag
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: bqian
///
/// @last_modified: 11/9/04
/////////////////////////////////////////////////////////////////////////////////

void
set_taboo_flag( bool yesno )
{
		taboo_map_ns::taboo_mode_flag = yesno;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_taboo_active
///
/// @brief: turn on - off taboo
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: bqian
///
/// @last_modified: 2/4/05
/////////////////////////////////////////////////////////////////////////////////

void
set_taboo_active( bool yesno )
{
	if ( ! get_taboo_mode_exist() ) return;

	taboo_map_ns::taboo_flag = yesno;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_taboo_mode
///
/// @brief: get_taboo_mode
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/9/04
/////////////////////////////////////////////////////////////////////////////////

int
get_taboo_mode()
{
		return taboo_map_ns::taboo_mode;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin reset_active_taboo_map
///
/// @brief: rest TENURE with a new value, clear active_taboo_map
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/7/04
/////////////////////////////////////////////////////////////////////////////////
void
reset_active_taboo_map()
{
	taboo_map_ns::active_taboo_map.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTabooMap::update_from_file
///
/// @brief: import from a file, and update the TabooMap
///
/// @detailed: taboo_list_file must have the following format:
///            tag       score  tenure barcode
///            string    100    9999   1 0 2 0 2
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/15/04
/////////////////////////////////////////////////////////////////////////////////
void
cTabooMap::update_from_file( std::string const & taboo_list_file )
{
	using namespace std;

	utility::io::izstream infile;
	infile.open( taboo_list_file );

	if ( !infile ) {
		cout << "Could not open taboo_list_file: "
				<< taboo_list_file
				<< endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	string line;
//	std::getline( infile(), line ); //skip the first line

	cout << "[ update_from_file ] Reading in taboo_list_file " << endl << line << endl;

	while ( std::getline( infile(), line ) ) {
		float thisScore;
		string info = "";

		cout << line << endl;
		stringstream line_stream( line );
		line_stream >> info >> thisScore; // the first number is a score;
		if ( line_stream.fail() &&
								(info != "" && info != "TABOO" && info != "score") ) {
			cout << "WARNING: Wrong format in taboo file: " << taboo_list_file
					<< endl << "Line: " << line << endl
					<< "Each line must have the following format: "
					<< "tag score tenure barcode!!" << endl;
//			std::exit( 1 );
		}

		if ( info == "threshhold" || info == "THRESHHOLD" ) {
			barcode_dist_thresh = thisScore;
		} else if ( info == "aspiration" || info == "ASPIRATION" ) {
			if ( thisScore > 0 ) aspiration = true;
			else aspiration = false;
		} else if ( info == "" || info == "TABOO" || info == "score" ) {
			// header lines, nothing needs to be done
		} else { // the barcode lines
			string thisBarcode;
			float thisTenure;
			line_stream >> thisTenure;
			int int_tenure = int( thisTenure );
			if ( line_stream.fail() ) {
				cout << "WARNING: Wrong format in taboo file: " << taboo_list_file
					<< endl << "Line: " << line << endl
					<< "Each line must have the following format: "
					<< "tag score tenure barcode!!" << endl;
//				std::exit( 1 );
			}
			bool noCodeFound = false;
			while ( !line_stream.fail() ) {
				string thisBin;
				line_stream >> thisBin;
				if ( !line_stream.fail() ) {
					thisBarcode += thisBin + ' ';
					// in taboo_mode 2, -1 means no code found
					if ( thisBin == "-1" && taboo_map_ns::taboo_mode == 2 ) noCodeFound = true;
				}
			}
			int int_score = int( thisScore );
			//cout << "this Score " << tenure << endl;
			if ( !noCodeFound ) update( int_score, int_tenure, thisBarcode );
		//debug
		// cout << "thisscore " << thisScore << " thisBarcode "
		//		<< thisBarcode << endl;
		}
	}

	infile.close();
	infile.clear();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_taboo_map
///
/// @brief: rest TENURE with a new value, clear active_taboo_map
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/8/04
/////////////////////////////////////////////////////////////////////////////////
void
cTabooMap::retrieve_from_trace( int thistenure, const cTrace & thistrace )
{
	using namespace taboo_map_ns;
	using namespace misc;

	TENURE = thistenure;
	int steps = thistrace.phi_trace.size();
	for ( int i=0; i< steps; ++i ) {
		std::string thisBarcode = barcode_vector (thistrace.phi_trace.at(i),
																thistrace.psi_trace.at(i), 5, total_residue - 4);
		update ( thisBarcode );
	}

	aspiration = false;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_active_aspiration_map
///
/// @brief: rest TENURE with a new value, clear active_taboo_map
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/8/04
/////////////////////////////////////////////////////////////////////////////////
void
set_active_aspiration_map( int thistenure, int thisscore, int bin_size )
{
	using namespace taboo_map_ns;

	TENURE = thistenure;
	BB_BIN_WIDTH = bin_size;

	active_taboo_map.clear();
	active_taboo_map.retrieve_from_trace ( thistenure, aspiration_sites );
	active_taboo_map.aspiration = true;

	std::map< std::string, cTaboo >::iterator pos;
	for ( pos = active_taboo_map.taboo.begin();
	 pos != active_taboo_map.taboo.end(); ++pos ) {
		active_taboo_map.taboo[ pos->first ].score = thisscore;
		active_taboo_map.taboo[ pos->first ].tenure = thistenure;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_taboo_exist
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
bool
get_taboo_exist()
{
	using namespace taboo_map_ns;

	return taboo_flag;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_taboo_mode_exist
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: bqian
///
/// @last_modified: 2/4/05
/////////////////////////////////////////////////////////////////////////////////
bool
get_taboo_mode_exist()
{
	using namespace taboo_map_ns;

	return taboo_mode_flag;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin taboo_list_file_exist
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/9/04
/////////////////////////////////////////////////////////////////////////////////
bool
taboo_list_file_exist()
{
	using namespace taboo_map_ns;

	return ( taboo_list_file != "" );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_taboo_list_file_name
///
/// @brief:
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/9/04
/////////////////////////////////////////////////////////////////////////////////
std::string
get_taboo_list_file_name()
{
	using namespace taboo_map_ns;

	return taboo_list_file;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin barcode_vector
///
/// @brief: barcode of the current phipsi array
///
/// @detailed: information from start to end of phipsi arrays are put into bins
///            to generate the barcode
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
std::string
barcode_vector(
	FArray1D_float const & phi,
	FArray1D_float const & psi,
	int start,
	int end
)
{
	using std::ostringstream;

	ostringstream barcode;

	for ( int i = start; i < end; ++i ) {
		barcode << phipsi_bin( phi[i] ) << " "
						<< phipsi_bin( psi[i] ) << " ";
	}

	return barcode.str();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin feature_barcode_vector
///
/// @brief: barcode of feature_residues of the current phipsi array
///
/// @detailed: information from start to end of phipsi arrays are put into bins
///            to generate the barcode
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 2/14/05
/////////////////////////////////////////////////////////////////////////////////
std::string
feature_barcode_vector(
	FArray1D_float const & phi,
	FArray1D_float const & psi
)
{
	using std::ostringstream;
	using std::sort;
	using std::vector;

	ostringstream barcode;
	// get all the feature residues
	int const n_residues = 99999;
	vector< int > feature_res = get_feature_residues( n_residues );
	sort( feature_res.begin(), feature_res.end() );

	for ( int i = 0; i < int( feature_res.size() ); ++i ) {
		barcode << phipsi_bin( phi[ feature_res.at(i) ] ) << " "
						<< phipsi_bin( psi[ feature_res.at(i) ] ) << " ";
	}

	return barcode.str();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin phipsi_bin
///
/// @brief: locate the bin a given phipsi array lies in
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
int
phipsi_bin( float angle )
{
	using namespace taboo_map_ns; // defines BB_BIN_WIDTH

	return int( floor( angle / BB_BIN_WIDTH ) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTabooMap::update
///
/// @brief
///
/// @detailed: with a new phipsi array, update the taboo map
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
void
cTabooMap::update( std::string const & barcode )
{
	if ( !check_barcode_validity( barcode ) ) return;

	if (taboo.find( barcode ) != taboo.end()) {
		++taboo[ barcode ].score;
	} else {
		cTaboo thisTaboo( 1, taboo_map_ns::TENURE, taboo.size()+1 );
		taboo[ barcode ] = thisTaboo;
	}

}

void
cTabooMap::update( int score, int tenure, std::string const & barcode )
{
	if ( !check_barcode_validity( barcode ) ) return;

	if (taboo.find( barcode ) != taboo.end()) {
		taboo[ barcode ].score +=score;
		taboo[ barcode ].tenure = tenure;
	} else {
		cTaboo thisTaboo( score, tenure, taboo.size()+1 );
		taboo[ barcode ] = thisTaboo;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin check_barcode_validity
///
/// @brief
/// check to see if the barcode is a valid barcode
///
/// @detailed: for taboo_mode 2, -1 is a space holder for unknown code
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 11/15/04
/////////////////////////////////////////////////////////////////////////////////

bool
check_barcode_validity( std::string const & barcode )
{
	bool valid = true;
	std::string buf;

	std::stringstream code_stream( barcode );

	while ( code_stream >> buf ) {
		if ( taboo_map_ns::taboo_mode == 2 && buf == "-1" )
			valid = false;
	}

	return valid;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTabooMap::get_id
///
/// @brief
///
/// @detailed: with a new phipsi array, update the taboo map
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/28/04
/////////////////////////////////////////////////////////////////////////////////
int
cTabooMap::get_id( std::string const & taboo_name )
{
	if (taboo.find( taboo_name ) != taboo.end())
		return taboo[ taboo_name ].id;
	else {
		return -1;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin return taboo_score
///
/// @brief
///
/// @detailed: Given current phipsi, read taboo count from active_taboo_map
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Bin Qian
///
/// @last_modified: 10/26/04
/////////////////////////////////////////////////////////////////////////////////
int
cTabooMap::score( std::string const & taboo_name )
{
	if ( taboo.find( taboo_name )  != taboo.end() ) {
		return taboo[ taboo_name ].score;
	} else {
		return 0;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin return void
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 10/26/04
/////////////////////////////////////////////////////////////////////////////////
void
cTabooMap::reduce_tenure()
{
	std::map< std::string, cTaboo >::iterator pos;
	for ( pos = taboo.begin(); pos != taboo.end(); ++pos ) {
		--taboo[ pos->first ].tenure;
	}

	remove_expired();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_tenure
///
/// @brief set tenure for the current taboo member
///
/// @detailed assign a time stamp for the current taboo member.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 10/26/04
/////////////////////////////////////////////////////////////////////////////////
void
cTaboo::set_tenure( int t = 10000 )
{
	using namespace misc;

	// will add secstruct dependent tenure here.
	tenure = t;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin expire_active_taboo
///
/// @brief expire active_taboo_map based on the tenure status
///
/// @detailed check taboo_tenure for each active_taboo_map members. If tenure
///           expired (tenure <=0), erase the member from active_taboo_map.
///
/// @global_read active_taboo_map, taboo_tenure
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 10/26/04
/////////////////////////////////////////////////////////////////////////////////
void
cTabooMap::remove_expired()
{
	std::map< std::string, cTaboo >::iterator pos;
	for (pos = taboo.begin(); pos != taboo.end(); ++pos) {
		if ( pos->second.tenure  <= 0 ) taboo.erase( pos );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin dump_taboo_map
///
/// @brief dump taboo map
///
/// @detailed: print out the information in taboo_map
///
/// @global_read: taboo_map
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 10/26/04
/////////////////////////////////////////////////////////////////////////////////
void
cTabooMap::dump()
{
	std::map< std::string, cTaboo >::iterator pos;
	for (pos = taboo.begin(); pos != taboo.end(); ++pos) {
			std::cout << "CODE "
					<< SS( pos->second.score )
					<< SS( pos->second.tenure )
					<< SS( pos->first )
					<< std::endl;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTrace::cTrace
///
/// @brief cTrace constructor
///
/// @detailed:
///
/// @global_read: taboo_map
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 11/8/04
/////////////////////////////////////////////////////////////////////////////////
cTrace::cTrace()
{
	phi_trace.clear();
	psi_trace.clear();
	score_trace.clear();
}

// destructor
cTrace::~cTrace()
{
	phi_trace.clear();
	psi_trace.clear();
	score_trace.clear();
}
////////////////////////////////////////////////////////////////////////////////
/// @begin cTrace::dump
///
/// @brief dump trace
///
/// @detailed: print out the information in taboo_map
///
/// @global_read: taboo_map
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 11/8/04
/////////////////////////////////////////////////////////////////////////////////
void
cTrace::dump()
{
	using namespace misc;

	int steps = phi_trace.size();
	for ( int i = 0; i < steps; ++i ) {
		std::cout << "trace " << i+1
				<< SS(score_trace.at(i) );
		for ( int j = 1; j <= total_residue; ++j ) {
				std::cout << SS( phi_trace.at(i)(j) )
						<< SS( psi_trace.at(i)(j) );
		}
		std::cout << std::endl;
	}

}
////////////////////////////////////////////////////////////////////////////////
/// @begin cTrace::update
///
/// @brief update trace
///
/// @detailed:
///
/// @global_read: taboo_map
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 11/8/04
/////////////////////////////////////////////////////////////////////////////////
void
cTrace::update(
	FArray1D_float const & thisphi,
	FArray1D_float const & thispsi,
	float thisscore
)
{
	phi_trace.push_back( thisphi );
	psi_trace.push_back( thispsi );
	score_trace.push_back( thisscore );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cTrace::clear
///
/// @brief clear trace
///
/// @detailed: clear the trace data
///
/// @global_read: taboo_map
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 11/7/04
/////////////////////////////////////////////////////////////////////////////////
void
cTrace::clear()
{
	phi_trace.clear();
	psi_trace.clear();
	score_trace.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_barcode
///
/// @brief: called from make_pdb.
///
/// @detailed: output the barcode for the current vector
///
/// @global_read:
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 11/9/04
/////////////////////////////////////////////////////////////////////////////////
void
output_decoy_barcode( std::ostream & out )
{
	if ( !get_taboo_mode_exist() ) return;
	std::string barcode = get_current_barcode();

	out << '\n' << "decoy barcode:" << '\n' << barcode << '\n' << '\n';
}
////////////////////////////////////////////////////////////////////////////////
/// @begin map_to_string
///
/// @brief given a map of int or float, convert to string
///
/// @detailed:
///
/// @global_read:
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 11/9/04
/////////////////////////////////////////////////////////////////////////////////
std::string
map_to_string( std::map< std::string, int > const & intmap )
{
	std::string out;

	std::map< std::string, int >::const_iterator pos;
	for ( pos = intmap.begin(); pos != intmap.end(); ++pos ) {
		std::stringstream bin;
		bin << pos->second;
		out += bin.str() + " ";
	}

	return out;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin in_Range
///
/// @brief
///
/// @detailed:
///
/// @global_read:
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Bin Qian
///
/// @last_modified 11/9/04
/////////////////////////////////////////////////////////////////////////////////
float
set_phipsi_range ( float const & phi )
{
	// constrain to (-180,180]
	float new_phi = phi;

	while ( new_phi   >   180.0 ) new_phi -= 360.0;

	while ( new_phi   <= -180.0 ) new_phi += 360.0;

	return new_phi;

}
