// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   numeric/numeric.functions.test.hh
/// @brief  test suite for numeric::numeric_functions
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)


// Test headers
#include <numeric/numeric.functions.test.hh>

// Package headers
#include <numeric/numeric.functions.hh>


namespace test {
namespace numeric {


// --------------- Fixtures --------------- //

// --- define a test fixture (some initial state that several tests share)
TEST_FIXTURE_BEGIN( fixture_numeric_functions )

// Shared initialization goes here.  (invoked when TEST_FIXTURE_USE is called)
TEST_FIXTURE_SETUP_BEGIN( fixture_numeric_functions )
{
	delta_percent = 0.0001;
}
TEST_FIXTURE_SETUP_END

// Shared finalization goes here.  (invoked when a test case that uses the fixture ends)
TEST_FIXTURE_TEARDOWN_BEGIN( fixture_numeric_functions )
{
}
TEST_FIXTURE_TEARDOWN_END

// Shared data elements go here.  (They will be declared public.)

double delta_percent; // percentage difference for floating-point comparisons in TEST_CHECK_CLOSE

TEST_FIXTURE_END  // fixture_numeric_functions


// --------------- Test Suites --------------- //

TEST_SUITE_BEGIN(Tests_numeric_functions)
	TEST_SUITE_USES_CASE( test_numeric_functions_nearest );
	TEST_SUITE_USES_CASE( test_numeric_functions_nint );
	TEST_SUITE_USES_CASE( test_numeric_functions_mod );
	TEST_SUITE_USES_CASE( test_numeric_functions_remainder );
	TEST_SUITE_USES_CASE( test_numeric_functions_gcd );
TEST_SUITE_END


// --------------- Test Cases --------------- //

/// @brief nearest tests
TEST_CASE_BEGIN(test_numeric_functions_nearest)
{
	using ::numeric::nearest;
	TEST_FIXTURE_USE( fixture_numeric_functions, fix );

	TEST_CHECK_EQUAL( nearest< int >( 53.3 ), 53 );
	TEST_CHECK_EQUAL( nearest< int >( 53.6 ), 54 );
	TEST_CHECK_EQUAL( nearest< long >( 53.3 ), 53l );
	TEST_CHECK_EQUAL( nearest< long >( 53.6 ), 54l );
	TEST_CHECK_CLOSE( nearest< float >( 53.3 ), 53.3f, fix.delta_percent );
	TEST_CHECK_CLOSE( nearest< float >( 53.6 ), 53.6f, fix.delta_percent );
}
TEST_CASE_END

/// @brief nint tests
TEST_CASE_BEGIN(test_numeric_functions_nint)
{ 
	using ::numeric::nint;
	TEST_CHECK_EQUAL( nint( 53.3 ), 53 );
	TEST_CHECK_EQUAL( nint( 53.6 ), 54 );
}
TEST_CASE_END

/// @brief mod tests
TEST_CASE_BEGIN(test_numeric_functions_mod)
{
	using ::numeric::mod;
	TEST_FIXTURE_USE( fixture_numeric_functions, fix );
	TEST_CHECK_EQUAL( mod( 7, 2 ), 1 );
	TEST_CHECK_EQUAL( mod( 29, 6 ), 5 );
	TEST_CHECK_CLOSE( mod( 29.0, 6.0 ), 5.0, fix.delta_percent );
	TEST_CHECK_CLOSE( mod( 29.0, 6.5 ), 3.0, fix.delta_percent );
}
TEST_CASE_END

/// @brief remainder tests
TEST_CASE_BEGIN(test_numeric_functions_remainder)
{
	TEST_FIXTURE_USE( fixture_numeric_functions, fix );
	// On GCC and other compilers providing C99 math functions these will call math.h versions
	// (Why check these?  Just want to check our own versions.  Remove for now.  kph, 2006-05-24)
//	TEST_CHECK_EQUAL( remainder( 7, 2 ), -1 );
//	TEST_CHECK_EQUAL( remainder( 29, 6 ), -1 );
//	TEST_CHECK_EQUAL( remainder( 27, 6 ), 3 ); // Special case: n - (x/y) == .5
//	TEST_CHECK_EQUAL( remainder( 33, 6 ), -3 ); // Special case: n - (x/y) == -.5
//	TEST_CHECK_CLOSE( remainder( 29.0, 6.0 ), -1.0, fix.delta_percent );
//	TEST_CHECK_CLOSE( remainder( 29.0, 6.5 ), 3.0, fix.delta_percent );
//	TEST_CHECK_CLOSE( remainder( 27.0, 6.0 ), 3.0, fix.delta_percent ); // Special case: n - (x/y) == .5
//	TEST_CHECK_CLOSE( remainder( 33.0, 6.0 ), -3.0, fix.delta_percent ); // Special case: n - (x/y) == -.5

	// Make sure we call the numeric_functions version
	TEST_CHECK_EQUAL( ::numeric::remainder( 7, 2 ), -1 );
	TEST_CHECK_EQUAL( ::numeric::remainder( 29, 6 ), -1 );
	TEST_CHECK_EQUAL( ::numeric::remainder( 27, 6 ), 3 ); // Special case: n - (x/y) == .5
	TEST_CHECK_EQUAL( ::numeric::remainder( 33, 6 ), -3 ); // Special case: n - (x/y) == -.5
	TEST_CHECK_CLOSE( ::numeric::remainder( 29.0, 6.0 ), -1.0, fix.delta_percent );
	TEST_CHECK_CLOSE( ::numeric::remainder( 29.0, 6.5 ), 3.0, fix.delta_percent );
	TEST_CHECK_CLOSE( ::numeric::remainder( 27.0, 6.0 ), 3.0, fix.delta_percent ); // Special case: n - (x/y) == .5
	TEST_CHECK_CLOSE( ::numeric::remainder( 33.0, 6.0 ), -3.0, fix.delta_percent ); // Special case: n - (x/y) == -.5
}
TEST_CASE_END

/// @brief gcd tests
TEST_CASE_BEGIN(test_numeric_functions_gcd)
{
	using ::numeric::gcd;
	TEST_FIXTURE_USE( fixture_numeric_functions, fix );
	TEST_CHECK_EQUAL( gcd( 7, 2 ), 1 );
	TEST_CHECK_EQUAL( gcd( 28, 6 ), 2 );
	TEST_CHECK_EQUAL( gcd( 16, 12 ), 4 );
	TEST_CHECK_CLOSE( gcd( 7.0, 2.0 ), 1.0, fix.delta_percent );
	TEST_CHECK_CLOSE( gcd( 28.0, 6.0 ), 2.0, fix.delta_percent );
	TEST_CHECK_CLOSE( gcd( 16.0, 12.0 ), 4.0, fix.delta_percent );
}
TEST_CASE_END


} // namespace numeric
} // namespace test
