// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   numeric/xyzVector.test.hh
/// @brief  test suite for numeric::xyzVector
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)


// Test headers
#include <numeric/xyzVector.test.hh>

// Package Headers
#include <numeric/xyzVector.hh>
#include <numeric/xyzVector.io.hh>


namespace test {
namespace numeric {


// --------------- Fixtures --------------- //

// --- define a test fixture (some initial state that several tests share)
TEST_FIXTURE_BEGIN( fixture_xyzVector )

// Shared initialization goes here.  (invoked when TEST_FIXTURE_USE is called)
TEST_FIXTURE_SETUP_BEGIN( fixture_xyzVector )
{
	delta_percent = 0.0001;
}
TEST_FIXTURE_SETUP_END

// Shared finalization goes here.  (invoked when a test case that uses the fixture ends)
TEST_FIXTURE_TEARDOWN_BEGIN( fixture_xyzVector )
{
}
TEST_FIXTURE_TEARDOWN_END

// Shared data elements go here.  (They will be declared public.)

double delta_percent; // percentage difference for floating-point comparisons in TEST_CHECK_CLOSE

TEST_FIXTURE_END  // fixture_xyzVector


// --------------- Test Suites --------------- //

TEST_SUITE_BEGIN( Tests_xyzVector )
	TEST_SUITE_USES_CASE( test_xyzVector_ConstructFromSingleValue );
	TEST_SUITE_USES_CASE( test_xyzVector_ConstructFrom1DArrayPointer );
	TEST_SUITE_USES_CASE( test_xyzVector_ConstructFrom2DArrayPointer );
	TEST_SUITE_USES_CASE( test_xyzVector_Length );
	TEST_SUITE_USES_CASE( test_xyzVector_Accessors );
	TEST_SUITE_USES_CASE( test_xyzVector_Relations );
	TEST_SUITE_USES_CASE( test_xyzVector_BinaryOperations );
TEST_SUITE_END


// --------------- Test Cases --------------- //

/// @brief test construction from 1D array
TEST_CASE_BEGIN(test_xyzVector_ConstructFromSingleValue)
{
	using ::numeric::xyzVector_float;

	xyzVector_float v( 15.0 );
	TEST_CHECK_EQUAL( v.x(), 15.0f );
	TEST_CHECK_EQUAL( v.y(), 15.0f );
	TEST_CHECK_EQUAL( v.z(), 15.0f );
}
TEST_CASE_END


/// @brief test construction from 1D array
TEST_CASE_BEGIN(test_xyzVector_ConstructFrom1DArrayPointer)
{
	using ::numeric::xyzVector_float;
	TEST_FIXTURE_USE( fixture_xyzVector, fix );

	// set up a 1D array
	float v[ 10 ];
	for ( int i = 0; i < 10; ++i ) {
		v[ i ] = i;
	}

	// initialize a vector from one spot
	xyzVector_float u( &v[ 1 ] );
	TEST_CHECK_EQUAL( u.x(), 1.0 );
	TEST_CHECK_EQUAL( u.y(), 2.0 );
	TEST_CHECK_EQUAL( u.z(), 3.0 );

	// initialize a vector from another spot
	xyzVector_float w( &v[ 4 ] );
	TEST_CHECK_EQUAL( w.x(), 4.0 );
	TEST_CHECK_EQUAL( w.y(), 5.0 );
	TEST_CHECK_EQUAL( w.z(), 6.0 );

	// Tests if constructor catches double to float conversion.
	// . caught at compilation time:
	// . no matching function for call to ::numeric::xyzVector<float>::xyzVector(double*)
	//FArray1D_double a( 3, 0.0 );
	//xyzVector_float b( &a( 1 ) );
}
TEST_CASE_END


/// @brief test construction from 2D array
TEST_CASE_BEGIN(test_xyzVector_ConstructFrom2DArrayPointer)
{
	using ::numeric::xyzVector_float;
	// build a simple 2D array where the row values are sequential:
	//  0  1  2  3
	//  4  5  6  7
	//  8  9 10 11
	// 12 13 14 15
	float a [ 4 ] [ 4 ];
	for ( int i = 0; i < 4; ++i ) {
		for ( int j = 0; j < 4; ++j) {
			a[i][j] = 4*i + j;
		}
	}

	// initialize a vector from one spot
	xyzVector_float u( &a[0][0] );
	TEST_CHECK_EQUAL( u.x(), 0.0 );
	TEST_CHECK_EQUAL( u.y(), 1.0 );
	TEST_CHECK_EQUAL( u.z(), 2.0 );
	
	// initialize vector from another spot (across rows)
	xyzVector_float v( &a[2][3] );
	TEST_CHECK_EQUAL( v.x(), 11.0 );
	TEST_CHECK_EQUAL( v.y(), 12.0 );
	TEST_CHECK_EQUAL( v.z(), 13.0 );
}
TEST_CASE_END


/// @brief Length tests
TEST_CASE_BEGIN(test_xyzVector_Length)
{
	using ::numeric::xyzVector_float;
	TEST_FIXTURE_USE( fixture_xyzVector, fix );
	xyzVector_float v( 2.0, 1.0, -1.0 );

	// normalized vector should have length 1
	v.normalize();
	TEST_CHECK_CLOSE( v.length(), 1.0f, fix.delta_percent );
	// Note: First two parameters must be of same type for call to template function.

	// zeroed vector should have length 0
	v.zero();
	TEST_CHECK_CLOSE( v.length(), 0.0f, fix.delta_percent );
}
TEST_CASE_END


/// @brief Accessors tests
TEST_CASE_BEGIN(test_xyzVector_Accessors)
{
	using ::numeric::xyzVector_float;
	float x = 1.0;
	float y = 2.0;
	float z = 3.0;

	xyzVector_float v( x, y, z );

	// test access to x/y/z components
	TEST_CHECK_EQUAL( v.x(), x );
	TEST_CHECK_EQUAL( v.y(), y );
	TEST_CHECK_EQUAL( v.z(), z );
}
TEST_CASE_END


/// @brief Relations tests
TEST_CASE_BEGIN(test_xyzVector_Relations)
{
	using ::numeric::xyzVector_float;
	xyzVector_float v( 1.0, 2.0, 3.0 );
	xyzVector_float w( 1.0, 2.0, 3.0 );

	TEST_CHECK_EQUAL( v, w );
	
	// Reduce v and test inequality
	v -= 0.5;
	TEST_CHECK( v != w );
	TEST_CHECK( ! ( v == w ) );
	TEST_CHECK( v < w );
	TEST_CHECK( v <= w );

	// Increase v and test inequality
	v += 1.0;
	TEST_CHECK( v != w );
	TEST_CHECK( ! ( v == w ) );
	TEST_CHECK( v > w );
	TEST_CHECK( v >= w );

	// Test partial ordering: Set v.x to 0 but leave v.y > w.y and v.z > w.z so v and w are not orderable
	v.x( 0.0 );
	TEST_CHECK( v != w );
	TEST_CHECK( ! ( v == w ) );
	TEST_CHECK( ! ( v < w ) );
	TEST_CHECK( ! ( v <= w ) );
	TEST_CHECK( ! ( v > w ) );
	TEST_CHECK( ! ( v >= w ) );

	// Test length relations
	TEST_CHECK( ! equal_length( v, w ) );
	TEST_CHECK( ! v.equal_length(w) );
	TEST_CHECK( not_equal_length( v, w ) );
	TEST_CHECK( v.not_equal_length(w) );
	TEST_CHECK( v.longer(w) );
	TEST_CHECK( ! v.shorter(w) );
	TEST_CHECK( v.longer_or_equal(w) );
	TEST_CHECK( ! v.shorter_or_equal(w) );
}
TEST_CASE_END


/// @brief binary operator tests
TEST_CASE_BEGIN(test_xyzVector_BinaryOperations)
{
	using ::numeric::xyzVector_float;
	TEST_FIXTURE_USE( fixture_xyzVector, fix );

	xyzVector_float v( 1.0, 2.0, 3.0 );
	xyzVector_float w( 1.0, 2.0, 3.0 );

	// Check dot product of two equal vectors
	float x = dot( v, w );
	TEST_CHECK_CLOSE( x, v.length_squared(), fix.delta_percent ); // v == w here

	// Tweak the vectors and compute cross product
	v += 1.0; w -= 1.0;
	xyzVector_float t = cross( v, w );
	TEST_CHECK_CLOSE( dot( t, v ), 0.0f, fix.delta_percent ); // t and v are orthogonal
	TEST_CHECK_CLOSE( dot( t, w ), 0.0f, fix.delta_percent ); // t and w are orthogonal

	// Check midpoint (should match original vector)
	xyzVector_float original( 1.0, 2.0, 3.0 );
	xyzVector_float mid;
	mid = midpoint ( v, w );
	TEST_CHECK_CLOSE( original.x(), mid.x(), fix.delta_percent );
	TEST_CHECK_CLOSE( original.y(), mid.y(), fix.delta_percent );
	TEST_CHECK_CLOSE( original.z(), mid.z(), fix.delta_percent );
}
TEST_CASE_END


} // namespace numeric
} // namespace test
