// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/BitSet.test.cc
/// @brief  BitSet.test: test suite for utility::BitSet
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


// Package headers
#include <utility/BitSet.hh>

// Test headers
#include <utility/BitSet.test.hh>


namespace test {
namespace utility {


enum Bit { zero, one, two, three, four, nine=9, ten };

inline
::utility::BitSet< Bit >
operator |( Bit const & a, Bit const & b )
{
	return ::utility::BitSet< Bit >( a, b );
}


namespace b {

class Bit
{
public:

	inline
	explicit
	Bit( int ii ) :
		i( ii )
	{}

	friend
	inline
	bool
	operator <( Bit const & a, Bit const & b )
	{
		return ( a.i < b.i );
	}

	friend
	inline
	::utility::BitSet< Bit >
	operator |( Bit const & a, Bit const & b )
	{
		return ::utility::BitSet< Bit >( a, b );
	}

	int i;
};

} // namespace b


// Test suite
TEST_SUITE_BEGIN(BitSet_Tests)
	TEST_SUITE_USES_CASE(test_BitSet_constructor)
	TEST_SUITE_USES_CASE(test_BitSet_assignment)
	TEST_SUITE_USES_CASE(test_BitSet_method)
TEST_SUITE_END


// Test cases


/// @brief Constructor Test
TEST_CASE_BEGIN(test_BitSet_constructor)
{

	{ // enum Bit: Copy construction
		::utility::BitSet< Bit > s( one | nine |= ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( s.size() == 3 );
		TEST_CHECK( ! s.empty() );
	}

	{ // class Bit Bit: Copy construction
		b::Bit one(1), two(2), nine(9), ten(10);
		::utility::BitSet< b::Bit > s( one | nine |= ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
	}

	{ // enum Bit: Multiple bit construction
		::utility::BitSet< Bit > s( one, nine, ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
	}

}
TEST_CASE_END


/// @brief Assignment Test
TEST_CASE_BEGIN(test_BitSet_assignment)
{

	{ // |= BitSet
		::utility::BitSet< Bit > s( one | nine |= ten );
		::utility::BitSet< Bit > const t( two );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( t[ two ] );
		s |= t;
		TEST_CHECK( s[ two ] );
	}

	{ // += BitSet
		::utility::BitSet< Bit > s( one | nine |= ten );
		::utility::BitSet< Bit > const t( two );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( t[ two ] );
		s += t;
		TEST_CHECK( s[ two ] );
	}

	{ // -= BitSet
		::utility::BitSet< Bit > s( one | nine |= ten );
		::utility::BitSet< Bit > const t( nine );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		TEST_CHECK( t[ nine ] );
		s -= t;
		TEST_CHECK( ! s[ nine ] );
	}

	{ // |=, -=, += Bit
		::utility::BitSet< Bit > s( one | nine |= ten );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( s[ ten ] );
		s |= two;
		TEST_CHECK( s[ two ] );
		s -= nine;
		TEST_CHECK( ! s[ nine ] );
		s += nine;
		TEST_CHECK( s[ nine ] );
	}
}
TEST_CASE_END


/// @brief Method Test
TEST_CASE_BEGIN(test_BitSet_method)
{

	{ // BitSet | BitSet
		::utility::BitSet< Bit > const s( one | nine );
		::utility::BitSet< Bit > const t( two | ten );
		::utility::BitSet< Bit > const u( s | t );
		TEST_CHECK( u[ one ] );
		TEST_CHECK( u[ two ] );
		TEST_CHECK( u[ nine ] );
		TEST_CHECK( u[ ten ] );
		TEST_CHECK( ! u[ three ] );
	}

	{ // BitSet + BitSet
		::utility::BitSet< Bit > const s( one | nine );
		::utility::BitSet< Bit > const t( two | ten );
		::utility::BitSet< Bit > const u( s + t );
		TEST_CHECK( ! s[ ten ] );
		TEST_CHECK( u[ one ] );
		TEST_CHECK( u[ two ] );
		TEST_CHECK( u[ nine ] );
		TEST_CHECK( u[ ten ] );
		TEST_CHECK( ! u[ three ] );
	}

	{ // BitSet - BitSet
		::utility::BitSet< Bit > const s( one | nine );
		::utility::BitSet< Bit > const t( two | nine );
		::utility::BitSet< Bit > const u( s - t );
		TEST_CHECK( u[ one ] );
		TEST_CHECK( ! u[ two ] );
		TEST_CHECK( ! u[ nine ] );
		TEST_CHECK( ! u[ ten ] );
	}

	{ // Swap
		::utility::BitSet< Bit > s( one | nine );
		::utility::BitSet< Bit > t( two | nine );
		TEST_CHECK( s != t );
		TEST_CHECK( s[ one ] );
		TEST_CHECK( ! s[ two ] );
		TEST_CHECK( s[ nine ] );
		TEST_CHECK( t[ two ] );
		TEST_CHECK( ! t[ three ] );
		TEST_CHECK( t[ nine ] );
		swap( s, t );
		TEST_CHECK( t[ one ] );
		TEST_CHECK( ! t[ two ] );
		TEST_CHECK( t[ nine ] );
		TEST_CHECK( s[ two ] );
		TEST_CHECK( ! s[ three ] );
		TEST_CHECK( s[ nine ] );
	}

	{ // BitSet ==, != BitSet
		::utility::BitSet< Bit > s( one | nine |= ten );
		::utility::BitSet< Bit > t;
		TEST_CHECK( t.empty() );
		t = s;
		TEST_CHECK( s == t );
		t |= two;
		TEST_CHECK( s != t );
	}

}
TEST_CASE_END


} // namespace utility
} // namespace test
