// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   utility/pointer/owning_ptr.hh
/// @brief  Unit tests for utility::pointer::owning_ptr
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)


// Headers
#include <iostream> // Put this first to allow debug print-outs in project headers
#include <utility/pointer/owning_ptr.test.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/ReferenceCount.hh>
#include <set>


namespace test {
namespace utility {
namespace pointer {


// --- set up some helper classes/functions for these tests
// --- (these go in a special namespace to be used from this file only!)

namespace owning_ptr_test {


class Atom_ :
	public ::utility::pointer::ReferenceCount
{
public:

	Atom_( double const weight, double const charge ) :
		weight_( weight ),
		charge_( charge )
	{
		++nAtoms;
	}

	virtual
	~Atom_()
	{
		--nAtoms;
	}

	double weight_;
	double charge_;
	static int nAtoms; // For tracking destructions
}; // class Atom_

int Atom_::nAtoms = 0;

typedef  ::utility::pointer::owning_ptr< Atom_ >  AtomP_;
typedef  std::set< AtomP_ >  Atoms_;


class Molecule_ :
	public ::utility::pointer::ReferenceCount
{
public:

//	typedef  ::utility::pointer::owning_ptr< Atom_ >  AtomP_;
//	typedef  std::set< AtomP_ >  Atoms_;

	inline
	Molecule_()
	{}

	virtual
	~Molecule_()
	{}

	inline
	void
	add( Atom_ * atom_p )
	{
		atoms_.insert( atom_p );
	}

	inline
	void
	add( AtomP_ const & atom_p )
	{
		atoms_.insert( atom_p );
	}

	inline
	std::set< AtomP_ >::size_type
	n_atom() const
	{
		return atoms_.size();
	}

	Atoms_ atoms_;

}; // class Molecule_


} // namespace owning_ptr_test


// The tests live in the utility::pointer namespace, but
// grant them access to our local namespace for classes/functions.

using namespace owning_ptr_test;


// --- set up the test suite
TEST_SUITE_BEGIN(Owning_Ptr_Tests)
	TEST_SUITE_USES_CASE(test_owning_ptr_explicit)
	TEST_SUITE_USES_CASE(test_owning_ptr_containers)
TEST_SUITE_END

// --- set up the individual test cases

/// @brief Explicit owning pointer
TEST_CASE_BEGIN(test_owning_ptr_explicit)
{
	Atom_ * Cp = new Atom_( 12.0, 0.0 ); // Raw heap pointer
	TEST_CHECK_EQUAL( Atom_::nAtoms, 1 );
	TEST_CHECK( Cp->ref_count() == 0 );
	{
		AtomP_ C( Cp ); // First owning pointer has sole ownership
		TEST_CHECK( Cp->ref_count() == 1 );
		{
			AtomP_ C2( Cp ); // Second owning pointer shares ownership
			TEST_CHECK( Cp->ref_count() == 2 );
		}
		TEST_CHECK( Cp->ref_count() == 1 );
	}
	// Atom should have been destructed
	TEST_CHECK( Atom_::nAtoms == 0 );
}
TEST_CASE_END

/// @brief Containers
TEST_CASE_BEGIN(test_owning_ptr_containers)
{
	Atom_ * Cp = new Atom_( 13.0, 0.0 ); // Raw heap pointer
	Atom_ * Hp = new Atom_( 1.0, 0.0 ); // Raw heap pointer
	TEST_CHECK( Atom_::nAtoms == 2 );
	TEST_CHECK( Cp->ref_count() == 0 );
	TEST_CHECK( Hp->ref_count() == 0 );
	{
		Molecule_ m1; m1.add( Cp ); m1.add( Hp );
		TEST_CHECK( m1.n_atom() == 2 );
		TEST_CHECK( Cp->ref_count() == 1 );
		TEST_CHECK( Hp->ref_count() == 1 );
		{
			Molecule_ m2; m2.add( Cp ); m2.add( Hp );
			TEST_CHECK( m2.n_atom() == 2 );
			TEST_CHECK( Cp->ref_count() == 2 );
			TEST_CHECK( Hp->ref_count() == 2 );
			m2.add( Cp ); // Repeat: Won't be added again
			TEST_CHECK( m2.n_atom() == 2 );
			TEST_CHECK( Cp->ref_count() == 2 );
		}
		TEST_CHECK( Cp->ref_count() == 1 );
		TEST_CHECK( Hp->ref_count() == 1 );
	}
	// Atoms should have been destructed
	TEST_CHECK( Atom_::nAtoms == 0 );
}
TEST_CASE_END


} // namespace pointer
} // namespace utility
} // namespace test
