// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "tether.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "cenlist.h"
#include "fullatom.h"
#include "minimize.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "param_torsion.h"
#include "pdb.h"
#include "runlevel.h"
#include "score_ns.h"
#include "tether_ns.h"
#include "util_basic.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>

//Utility Headers
#include <utility/basic_sys_util.hh>

/// @brief Allocate space for the #dis2_tether array
/// @see   #tether::tether_data::dis2_tether.
static void init_dis2_tether_();

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//------------------------------------------------------------------------------
//  TETHER INTERFACE
//------------------------------------------------------------------------------
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

////////////////////////////////////////////////////////////////////////////////
/// @begin tether_reset
///
/// @brief
/// set tethering energy to ground state
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
tether_reset()
{
	using namespace scorefxns;
	using namespace tether;

	tether_window_size = -1;

	tether_weight = 0.0; // also in score_reset_weights
	phipsi_weight = 0.0;
	omega_weight = 0.0;

	tether_init_residue_weights();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin tether_set_window_size
///
/// @brief
///
/// @detailed
///
/// @param  window - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
tether_set_window_size( int window )
{
	using namespace tether;

	tether_window_size = window;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin tether_set_res_weight
///
/// @brief
///
/// @detailed
///
/// @param  residue - [in/out]? -
/// @param  weight - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
tether_set_res_weight(
	int residue,
	float weight
)
{
	using namespace tether;

	tether_res_weight(residue) = weight;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin tether_set_angle_res_weight
///
/// @brief
///
/// @detailed
///
/// @param  residue - [in/out]? -
/// @param  weight - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
tether_set_angle_res_weight(
	int residue,
	float weight
)
{
	using namespace tether;

	tether_angle_res_weight(residue) = weight;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin tether_init_residue_weights
///
/// @brief
///
/// @detailed
///These weights determine the contribution of each residue to tether,
///phipsi, and omega scores (scores designed to anchor structure to
///the starting coordinates and anges)
///This function sets all weights to 1.0; individual weights can be
///subsequently altered with tether_set_residue_weight
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
tether_init_residue_weights()
{
	using namespace param;
	using namespace tether;

	tether_res_weight = 1.0;
	tether_angle_res_weight = 1.0;
}

/// @begin init_dis2_tether_
///
/// @brief
///
/// Redimension the dimensions of #dis2_tether appropriately.
///
/// @detailed
///
/// #dis2_tether is only used if #tether_setup is called, a point at which
/// the dimensions of it can be dynamically determined.  This function
/// This function allocates the array with an appropriate size only
/// if idealization (which uses tethers) is actually set up.
///
/// @global_read
///
/// #param::MAX_RES
/// #tether::max_tether_atoms
/// #tether::tether_data::dis2_tether
///
/// @global_write
///
/// @remarks
///
/// - This is a separate function to allow for the possibility that
///   #dis2_tether might need to be resized somewhere other than
///   #tether_setup().
/// - The function guards against being called more than once.
///
/// @references
///
/// @authors ion
///
/// @last_modified 5/2/06

void
init_dis2_tether_()
{
	using param::MAX_RES;
	using tether::max_tether_atoms;
	using tether::tether_data::dis2_tether;

	// Prevent this function from being called more than once.
	static bool called = false;

	if ( !called )
	{
		dis2_tether.redimension( MAX_RES(), max_tether_atoms,
														 MAX_RES(), max_tether_atoms );
		called = true;
	}
}

//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
//------------------------------------------------------------------------------
//  END TETHER INTERFACE
//------------------------------------------------------------------------------
//cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc


////////////////////////////////////////////////////////////////////////////////
/// @begin tether_setup
///
/// @brief
///car calculate all pairwise distances squared between all atoms
///car return distance matrix
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
tether_setup()
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace tether;

	using param::MAX_RES;

//car local
	int i_maxatm,j_maxatm;

	init_dis2_tether_();

	minimize_update_fullcoord();
	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = i; j <= total_residue; ++j ) {
			if ( get_fullatom_flag() ) {
				i_maxatm = std::min(natoms(res(i),res_variant(i)),max_tether_atoms);
				j_maxatm = std::min(natoms(res(j),res_variant(j)),max_tether_atoms);
				cutoff_atm = 2; // set global in minimize.h
			} else {
				i_maxatm = max_tether_atoms;
				j_maxatm = max_tether_atoms;
				cutoff_atm = 6; // set global in minimize.h
			}
			for ( int ii = 1; ii <= i_maxatm; ++ii ) {
				int jjstart = 1;
				if ( i == j ) jjstart = ii;
				for ( int jj = jjstart; jj <= j_maxatm; ++jj ) {
					dis2_tether(i,ii,j,jj) = vec_dist2(full_coord(1,ii,i),
					full_coord(1,jj,j));
					dis2_tether(j,jj,i,ii) = dis2_tether(i,ii,j,jj);
				}
			}
		}                  // next j
	}                     // next i

	for ( int i = 1; i <= total_residue; ++i ) {
		omega_tether(i) = omega(i); // save for omega_score
		phi_tether(i) = phi(i);
		psi_tether(i) = psi(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_tether_dme
///
/// @brief
///car calculate score reflecting how far bb and centroid coordinates
///   are from their starting points
///
/// @detailed
///
/// @param  tether_dme - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_tether_dme( float & tether_dme )
{
	using namespace aaproperties_pack;
	using namespace cenlist_ns;
	using namespace misc;
	using namespace param_aa;
	using namespace tether;

//car local
	int i_restype, j_restype; // residue types of i and j
	int i_maxatm, j_maxatm;
	int jjstart;
	float dist_dif, temp_weight;
	bool fullatom;
	int const fa_atm_cb = { 5 };

//  check that the tetherer is in the proper state for this function
	if ( tether_window_size < 0 ) {
		std::cout << "calculate_tether_dme called with window_size < 0 " << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	minimize_update_fullcoord();
	fullatom = get_fullatom_flag();
	tether_norm = 0.0; // all atoms with centroids/ca <12A
	tether_dme = 0.0;
	for ( int i = 1; i <= total_residue; ++i ) {
		int jend = std::min( i + tether_window_size, total_residue );
		i_restype = res(i);
		for ( int j = i; j <= jend; ++j ) {
			j_restype = res(j);
			if ( dis2_tether(i,cutoff_atm,j,cutoff_atm) > cen_dist_cutoff2 ) goto L100;
			if ( fullatom ) {
				i_maxatm = std::min(natoms(i_restype,res_variant(i)),max_tether_atoms);
				j_maxatm = std::min(natoms(j_restype,res_variant(j)),max_tether_atoms);
			} else {
				i_maxatm = max_tether_atoms;
				j_maxatm = max_tether_atoms;
			}
			for ( int ii = 1; ii <= i_maxatm; ++ii ) {
				if ( !fullatom && i_restype == aa_gly && ii == fa_atm_cb ) goto L50;
				jjstart = 1;
				if ( i == j ) jjstart = ii+1;
				for ( int jj = jjstart; jj <= j_maxatm; ++jj ) {
					if ( !fullatom && j_restype == aa_gly && jj == fa_atm_cb ) goto L20;
					temp_weight = tether_res_weight(i)*tether_res_weight(j);
					tether_norm += temp_weight;
// this is the only line different from tether_score
					dist_dif = std::sqrt( dis2_tether(i,ii,j,jj) ) -
					 std::sqrt( vec_dist2(full_coord(1,ii,i),full_coord(1,jj,j)) );
					tether_dme += temp_weight * ( dist_dif * dist_dif );
L20:;
				}            // jj
L50:;
			}               // ii
L100:;
		}                  // j
	}                     // i
	if ( tether_norm != 0.0 ) tether_dme = std::sqrt(tether_dme/tether_norm);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_tether_score
///
/// @brief
///car calculate score reflecting how far coordinates are from their
///car starting points
///
/// @detailed
///
/// @param  tether_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_tether_score( float & tether_score )
{
	using namespace aaproperties_pack;
	using namespace cenlist_ns;
	using namespace misc;
	using namespace param_aa;
	using namespace scorefxns;
	using namespace tether;

//car local
	int i_maxatm,j_maxatm;
	int i_restype, j_restype; // residue types of i and j
	int jjstart;
	float dist_dif,temp_weight;
	int const fa_atm_cb = { 5 };

// check that the tetherer is in the proper state for this function
	if ( tether_window_size < 0 ) {
		std::cout << "calculate_tether_score called with window_size < 0" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	tether_norm = 0.0;
	tether_score = 0.0;
	if ( tether_weight == 0.0 ) return;
	bool const fullatom = get_fullatom_flag();
	for ( int i = 1; i <= total_residue; ++i ) {
		int jend = std::min( i + tether_window_size, total_residue );
		i_restype = res(i);
		for ( int j = i; j <= jend; ++j ) {
			j_restype = res(j);
			if ( dis2_tether(i,cutoff_atm,j,cutoff_atm) > cen_dist_cutoff2 ) goto L100;
			if ( fullatom ) {
				i_maxatm = std::min(natoms(i_restype,res_variant(i)),max_tether_atoms);
				j_maxatm = std::min(natoms(j_restype,res_variant(j)),max_tether_atoms);
			} else {
				i_maxatm = max_tether_atoms;
				j_maxatm = max_tether_atoms;
			}
			for ( int ii = 1; ii <= i_maxatm; ++ii ) {
				if ( !fullatom && i_restype == aa_gly && ii == fa_atm_cb ) goto L50;
				jjstart = 1;
				if ( i == j ) jjstart = ii+1;
				for ( int jj = jjstart; jj <= j_maxatm; ++jj ) {
					if ( !fullatom && j_restype == aa_gly && jj == fa_atm_cb ) goto L20;
					temp_weight = tether_res_weight(i)*tether_res_weight(j);
					tether_norm += temp_weight;
// this is the only line different from tether_dme
					dist_dif = dis2_tether(i,ii,j,jj) -
					 vec_dist2(full_coord(1,ii,i),full_coord(1,jj,j));
					tether_score += temp_weight * ( dist_dif * dist_dif );
L20:;
				}            // jj
L50:;
			}               // ii
L100:;
		}                  // j
	}                     // i
	if ( tether_norm != 0.0 ) tether_score /= tether_norm;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_phipsi_torsion_tether_score
///
/// @brief
///car calculate score reflecting how far angles are from their
///car starting points
///ctsa ....and derivatives for each torsion angle
///
/// @detailed
///
/// @param  phipsi_score - [in/out]? -
/// @param  torsion_tether_score_deriv - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_phipsi_torsion_tether_score(
	float & phipsi_score,
	FArray2Da_float torsion_tether_score_deriv
)
{
	using namespace misc;
	using namespace param;
	using namespace param_torsion;
	using namespace tether;

	torsion_tether_score_deriv.dimension( total_bb_torsion, total_residue );

	phipsi_score = 0.0;
	float phipsi_norm = 0.0;

	for ( int i = 1; i <= total_residue; ++i ) {
		float const weight = tether_angle_res_weight(i);

		if ( i != 1 ) { //// phi
			float const angle = subtract_degree_angles(phi(i),phi_tether(i));
			phipsi_norm += weight;
			phipsi_score += weight * ( angle * angle );
			torsion_tether_score_deriv(phi_torsion,i) = 2*weight*angle;
		} else {
			torsion_tether_score_deriv(phi_torsion,i) = 0.;
		}

		if ( i != total_residue ) { //// psi,omega
			float angle = subtract_degree_angles(psi(i),psi_tether(i));
			phipsi_norm += weight;
			phipsi_score += weight * ( angle * angle );
			torsion_tether_score_deriv(psi_torsion,i) = 2*weight*angle;
		} else {
			torsion_tether_score_deriv(psi_torsion,i) = 0.;
		}

	}

//car note angle_norm isnt quite the correct normalization
//car because of end effects, but close enough

	if ( phipsi_norm != 0.0 ) {
		phipsi_score /= phipsi_norm;
		for ( int i = 1; i <= total_residue; ++i ) {
			torsion_tether_score_deriv(phi_torsion,i) /= phipsi_norm;
			torsion_tether_score_deriv(psi_torsion,i) /= phipsi_norm;
		}
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_omega_torsion_tether_score
///
/// @brief
///car calculate score reflecting how far angles are from their
///car starting points
///ctsa ....and derivatives for each torsion angle
///
/// @detailed
///
/// @param  phipsi_score - [in/out]? -
/// @param  omega_score - [in/out]? -
/// @param  torsion_tether_score_deriv - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_omega_torsion_tether_score(
	float & omega_score,
	FArray2Da_float torsion_tether_score_deriv
)
{
	using namespace misc;
	using namespace param;
	using namespace param_torsion;
	using namespace tether;

	torsion_tether_score_deriv.dimension( total_bb_torsion, total_residue );


	omega_score = 0.0;
	float omega_norm = 0.0;

	for ( int i = 1; i <= total_residue; ++i ) {
		float const weight = tether_angle_res_weight(i);

		if ( i != total_residue ) { //// omega
			float angle = subtract_degree_angles(omega(i),omega_tether(i));
			omega_norm += weight;
			omega_score += weight * ( angle * angle );
			torsion_tether_score_deriv(omega_torsion,i) = 2*weight*angle;
		} else {
			torsion_tether_score_deriv(omega_torsion,i) = 0.;
		}

	}

//car note angle_norm isnt quite the correct normalization
//car because of end effects, but close enough

	if ( omega_norm != 0.0 ) {
		omega_score /= omega_norm;
		for ( int i = 1; i <= total_residue; ++i ) {
			torsion_tether_score_deriv(omega_torsion,i) /= omega_norm;
		}
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_angle_rms
///
/// @brief
///car calculate score reflecting how far angles are from their
///car starting points
///
/// @detailed
///
/// @param  angle_rms - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calculate_angle_rms( float & angle_rms )
{
	using namespace misc;
	using namespace runlevel_ns;
	using namespace tether;

	angle_rms = 0.0;
	float angle_norm = 0.0;
	for ( int i = 1; i <= total_residue; ++i ) {
		float const weight = tether_angle_res_weight(i);
		float const angle_delta_phi = angle_delta(phi_tether(i),phi(i));
		float const angle_delta_psi = angle_delta(psi_tether(i),psi(i));
		float const angle_delta_omega = angle_delta(omega_tether(i),omega(i));
		if ( i != 1 ) {
			angle_norm += weight;
			angle_rms += weight * ( angle_delta_phi * angle_delta_phi );
		}
		if ( i != total_residue ) {
			angle_norm += 2.0 * weight;
			angle_rms += weight * ( ( angle_delta_psi * angle_delta_psi ) +
			 ( angle_delta_omega * angle_delta_omega ) );
		}

		if ( runlevel > standard ) {
			if ( angle_delta_phi > 3.0 ) std::cout <<
			 "phi" << SS( i ) << SS( phi(i) ) << SS( phi_tether(i) ) <<
			 SS( angle_delta_phi ) << std::endl;
			if ( angle_delta_psi > 3.0 ) std::cout <<
			 "psi" << SS( i ) << SS( psi(i) ) << SS( psi_tether(i) ) <<
			 SS( angle_delta_psi ) << std::endl;
			if ( angle_delta_omega > 3.0 ) std::cout <<
			 "omega" << SS( i ) << SS( omega(i) ) << SS( omega_tether(i) ) <<
			 SS( angle_delta_omega ) << std::endl;
		}

	}
	if ( angle_norm != 0.0 ) angle_rms = std::sqrt(angle_rms/angle_norm);
}

//  move me to util.angle.cc (or similar)....
////////////////////////////////////////////////////////////////////////////////
/// @begin angle_delta
///
/// @brief
///car returns the difference between angles a and b in degrees
///car a and b are assumed to be in the range from -180.0 to 180.0
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
angle_delta(
	float a,
	float b
)
{
	return std::min(std::abs(a-b),std::abs(std::abs(a-b)-360.0f));
}

namespace omega_wt_ns{
float omega_wt_default = 0.5;
float omega_wt = 0.0;
}

float
get_omega_weight()
{
	using namespace omega_wt_ns;
	static bool init = {false};

	if (!init) {
		realafteroption("omega_weight",omega_wt_default,omega_wt);
		init = true;
	}

	return(omega_wt);
}

void
set_omega_weight_default(float omega_wt)
{
	using namespace omega_wt_ns;
	omega_wt_default=omega_wt;
}
