// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "util_basic.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.Project.hh>
#include <ObjexxFCL/Fmath.hh>
//#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/constants.hh>

// C++ Headers
#include <cmath>


//
//     util_basic.cc - general utility functions that don't fit into any
//     other util*.cc files
//



////////////////////////////////////////////////////////////////////////////////
/// @begin calc_quadratic
///
/// @brief calculates quadratic polynomial solutions
///
/// @detailed
///
///     solves for solutions of x in the polynomial: a*x^2+b*x+c=0
///
/// @param[in]   a - in - x^2 term
/// @param[in]   b - in - x term
/// @param[in]   c - in - constant term
/// @param[out]   n1 - out - one solution
/// @param[out]   n2 - out - another solution
///
/// @remarks courtesy of Jerry Tsai
///
/// @references
///
/// @authors ctsa 8=19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
calc_quadratic(
	float a,
	float b,
	float c,
	float & n1,
	float & n2
)
{
//cj

	float bsq = b*b;
	float ac4 = 4*a*c;
	float st = std::sqrt( bsq - ac4 );

//cj    std::cout << F( 8, 3, a ) << ' ' << F( 8, 3, b ) << ' ' << F( 8, 3, c ) << std::endl;
//cj    std::cout << F( 8, 3, bsq ) << ' ' << F( 8, 3, ac4 ) << ' ' << F( 8, 3, st ) << std::endl;

	n1 = ((-b)+st)/(2*a);
	n2 = ((-b)-st)/(2*a);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin subtract_degree_angles
///
/// @brief subtract angles in degrees, restricting the range of the result
///
/// @detailed
///
///     given angles a and b in degrees, get a-b restricted to
///     [-180.,180.), assuming that a-b=a-b+n*360, n=any integer
///
/// @param[in]   a - in - angle in degrees
/// @param[in]   b - in - angle in degrees
///
/// @return  angle a-b in degrees, restricted to the specified range
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
subtract_degree_angles(
	float a,
	float b
)
{
	return periodic_range( a - b, 360.0f );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin subtract_radian_angles
///
/// @brief subtract angles in radians, restricting the range of the result
///
/// @detailed
///
///     given angles a and b in degrees, get a-b restricted to
///     [-pi,pi), assuming that a-b=a-b+n*2*pi, n=any integer
///
/// @param[in]   a - in - angle in radians
/// @param[in]   b - in - angle in radians
///
/// @return  angle a-b in degrees, restricted to the specified range
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
subtract_radian_angles(
	float a,
	float b
)
{
	using namespace numeric::constants::f;
	return periodic_range( a - b, pi_2 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin periodic_range
///
/// @brief  a is restricted to [-x/2,x/2), assuming that a=a+n*x, n=any integer
///
/// @detailed
///
/// @param[in]   a - in - input value with periodicity x
/// @param[in]   x - in - periodicity of a
///
/// @return  a restricted to [-x/2,x/2)
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
periodic_range(
	float a,
	float x
)
{
	float const halfx = 0.5f * x;
	return ( ( a >= halfx || a < -halfx ) ? mod( mod( a, x ) + ( x + halfx ), x ) - halfx : a );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsigned_periodic_range
///
/// @brief a is restricted to [0.,x), assuming that a=a+n*x,, n=any integer
///
/// @detailed
///
/// @param[in]   a - in - input value with periodicity x
/// @param[in]   x - in - periodicity of a
///
/// @return  a restricted to [0.,x)
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
unsigned_periodic_range(
	float a,
	float x
)
{
	return ( ( a >= x || a < 0.0 ) ? mod( mod( a, x ) + x, x ) : a );
}
