// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7630 $
//  $Date: 2006-03-10 09:37:52 -0800 (Fri, 10 Mar 2006) $
//  $Author: stuartm $

#ifndef INCLUDED_util_interpolate
#define INCLUDED_util_interpolate


// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/Fmath.hh>


// util_interpolate Function Declarations


void
interpolate_get_angle_bins(
	float const x,
	float const binrange,
	int & xbin,
	int & xbin_next,
	float & xd
);


void
interpolate_bilinear(
	int const xbin,
	int const xbin_next,
	float const xd,
	int const ybin,
	int const ybin_next,
	float const yd,
	FArray2Da_float xy_func,
	int const xbin_count,
	int const ybin_count,
	float const binrange,
	bool const angles,
	float & val,
	float & dval_dx,
	float & dval_dy
);


void
interpolate_bilinear_by_value(
	float const x0y0,
	float const x1y0,
	float const x0y1,
	float const x1y1,
	float const xd,
	float const yd,
	float const binrange,
	bool const angles,
	float & val,
	float & dval_dx,
	float & dval_dy
);


void
interpolate_2d_func_of_angles(
	float const x,
	float const y,
	FArray2Da_float xy_func,
	float & val,
	float & dval_dx,
	float & dval_dy
);


////////////////////////////////////////////////////////////////////////////////
/// @begin interpolate_get_angle_bins
///
/// @brief get bin information for a periodic value w/ periodic bins
///
/// @detailed
///
///     for 'x', an angle in degrees, and angular bins of width
///     'binrange' aligned to start bin 1 at a value of 0. degrees, find
///     the two bins, whose average bin values 'x' falls between, and
///     report the error between the lower average bin value and the real
///     value of 'x'
///
/// @param[in]   x - in - angle in degrees
/// @param[in]   binrange - in - degrees per bin
/// @param[out]   xbin - out - anglular bin whose average value is the lower
///                      of the two average bin values bracketing 'x'
/// @param[out]   xbin_next - out - anglular bin whose average value is the lower
///                      of the two average bin values bracketing 'x'
/// @param[out]   xd - out - error between the average value of bin 'xbin' and 'x'
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
interpolate_get_angle_bins(
	float const x,
	float const binrange,
	int & xbin,
	int & xbin_next,
	float & xd
)
{
//------------------------------------------------------------------------------
// ctsa - get nbins for binrange
//
	int const nbins = static_cast< int >( 360.0 / binrange );

// ctsa -  convert angle to float(anglebin) and
//         mod value to range: [1.,float(nbins)+1.)
//
//   note that:  float(anglebin) = 1. + (angle-.5*binrange)/binrange
//     ...this solves for the lowest of the two bracketing bin averages,
//       rather  than the nearest bin average, and is thus only
//       appropriate for interpolation
//

	float const xbin_real = 1.0 + mod(
	 mod( (x-(0.5*binrange))/binrange, static_cast< double >(nbins) ) + nbins,
	 static_cast< double >(nbins) );

// ctsa -  convert float bin values to array lookup bin values
//
	xbin = static_cast< int >( xbin_real );

// ctsa -  get next lookup bin and convert to range: (1,nbins)
//
	xbin_next = 1 + mod( xbin, nbins );

// ctsa -  get error
//
	xd = xbin_real - xbin;


	//// ctsa -- 5-2003
	////   ...another float resolution bug of type:
	////
	//// static_cast< int >(float(intval)-epsilon) == intval, epsilon << 1
	////
	//// this causes xbin to sporatically resolve to bins+1, a
	//// very bad thing; check below should fix...
	////
	if ( xbin == nbins + 1 ) {
		xbin = 1;
		xd = 0.0;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin interpolate_bilinear
///
/// @brief get bilinear interpolate of a 2d periodic function
///
/// @detailed
///
///     Value and derivatives of bilinear interpolation on a 2d binned
///     periodic function, represented as a 2d array.  (see Num. Recipes
///     v2, sec 3.6, "Interpolation in two or more dimensions")
///
///     Note that it is automatically assumed that the arguments are
///     periodic, that is f(a,b),a,b=periodic value; the *value* of the
///     function can be treated as periodic ( fixed to a periodicity of
///     360.) as well when the input parameter angle is set to true
///
///     The derivatives are of the bilinear interpolation only. This is
///     not the same value as a numerical derivative of function
///     values. The derivatives here are discontinuous every time you
///     cross over a bin boundary in either dimension
///
/// @param[in]   xbin - in - low bin in 1st dim
/// @param[in]   xbin_next - in - high bin in 1st dim
/// @param[in]   xd - in - error between low and real val in 1st dim
/// @param[in]   ybin - in - " 2nd dim "
/// @param[in]   ybin_next - in - " 2nd dim "
/// @param[in]   yd - in - " 2nd dim "
/// @param[in]   xy_func - in - 2d-array specifying the 2d binned function values,
///                        assumed to be periodic in each dimension
/// @param[in]   xbin_count - in - number of bins in the 1st dim of the periodic
///                        function
/// @param[in]   ybin_count - in - " 2nd dim "
/// @param[in]   binrange - in - range of bin in angles ( for both dimensions )
/// @param[in]   angles - in - if true, treat the the *values* of xy_func as
///                       as having a periodicity of 360. ( note that
///                       the bin ranges are already assumed periodic )
/// @param[out]   val - out - bilinear interpolated value of xy_func
/// @param[out]   dval_dx - out - derivative of interpolation w.r.t 1st dim
/// @param[out]   dval_dy - out - " 2nd dim "
///
/// @remarks
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
interpolate_bilinear(
	int const xbin,
	int const xbin_next,
	float const xd,
	int const ybin,
	int const ybin_next,
	float const yd,
	FArray2Da_float xy_func,
	int const xbin_count,
	int const ybin_count,
	float const binrange,
	bool const angles,
	float & val,
	float & dval_dx,
	float & dval_dy
)
{
	xy_func.dimension( xbin_count, ybin_count );

//------------------------------------------------------------------------------
	float const x0y0 = xy_func(xbin,ybin);
	float const x1y0 = xy_func(xbin_next,ybin);
	float const x0y1 = xy_func(xbin,ybin_next);
	float const x1y1 = xy_func(xbin_next,ybin_next);

	interpolate_bilinear_by_value(x0y0,x1y0,x0y1,x1y1,xd,yd,binrange,angles,val,
	 dval_dx,dval_dy);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin interpolate_2d_func_of_angles
///
/// @brief get bilinear interpolate of a 2d function with degree angle arguments
///
/// @detailed
///
///     Value and derivatives of bilinear interpolation on a 2d function
///     with degree angle arguments represented by a 2d array with binned
///     degree angle indices.  (see Num. Recipes v2, sec 3.6,
///     "Interpolation in two or more dimensions")
///
///     The derivatives are of the bilinear interpolation only. This is
///     not the same value as a numerical derivative of function
///     values. The derivatives here are discontinuous every time you
///     cross over a bin boundary in either dimension
///
/// @param[in]   x - in - function argument value in 1st dim to find interpolate for
/// @param[in]   y - in - " 2nd dim "
/// @param[in]   xy_func - in - 2d array with binned angle value indicies;
///               represents known values of 2d function being interpolated.
/// @param[out]   val - out - bilinear interpolated value
/// @param[out]   dval_dx - out - derivative of interpolated value w.r.t. 1st dim angle
/// @param[out]   dval_dy - out - " 2nd dim "
///
/// @remarks
///
///     !!! assumes 10 degree bins in both dimensions of xy_func
///
/// @references
///
/// @authors ctsa 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
interpolate_2d_func_of_angles(
	float const x,
	float const y,
	FArray2Da_float xy_func,
	float & val,
	float & dval_dx,
	float & dval_dy
)
{
	int const nbins = { 36 };
	xy_func.dimension( nbins, nbins );

//ctsa fixed parameters
	float const binrange = { 10.0f };

//ctsa local
	int xbin,ybin,xbin_next,ybin_next;
	float xd,yd;

//------------------------------------------------------------------------------
	interpolate_get_angle_bins(x,binrange,xbin,xbin_next,xd);
	interpolate_get_angle_bins(y,binrange,ybin,ybin_next,yd);

	bool const treat_as_angles = false;

	interpolate_bilinear(xbin,xbin_next,xd,ybin,ybin_next,yd,xy_func,nbins,nbins,
	 binrange,treat_as_angles,val,dval_dx,dval_dy);
}


#endif
