// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "util_vector.h"
#include "angles.h"
#include "pack_geom_inline.h"

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>


//Objexx: Linear, zero-based indexing used for speed (dimension calls not needed)


////////////////////////////////////////////////////////////////////////////////
/// @begin cross
///
/// @brief
/// a X b = c, length of c  = d
/// cross product A X B
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
/// @param  c - [in/out]? -
/// @param  d - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
cross(
	FArray1Da_double a,
	FArray1Da_double b,
	FArray1Da_double c,
	double & d
)
{
	c[ 0 ] = a[ 1 ] * b[ 2 ] - a[ 2 ] * b[ 1 ];
	c[ 1 ] = a[ 2 ] * b[ 0 ] - a[ 0 ] * b[ 2 ];
	c[ 2 ] = a[ 0 ] * b[ 1 ] - a[ 1 ] * b[ 0 ];
	d = std::sqrt( ( c[ 0 ] * c[ 0 ] ) + ( c[ 1 ] * c[ 1 ] ) + ( c[ 2 ] * c[ 2 ] ) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin align
///
/// @brief
/// rotate around vector a1->a2 to put a3 in the plane defined by a1,a2,b1
/// return the matrix and vector for the operation.
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  a3 - [in/out]? -
/// @param  b1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
align(
	FArray1Da_double a1,
	FArray1Da_double a2,
	FArray1Da_double a3,
	FArray1Da_double b1,
	FArray2Da_double mat,
	FArray1Da_double vec
)
{
	a1.dimension( 3 );
	a2.dimension( 3 );
	a3.dimension( 3 );
	b1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

	FArray1D_double a( 3 );
	FArray1D_double b( 3 );
	FArray1D_double c( 3 );
	FArray1D_double ab( 3 );
	FArray1D_double ac( 3 );
	FArray1D_double aba( 3 );
	double dab,dac,daba,ab_ac,aba_ac,chi;

//* get difference vectors
	for ( int i = 1; i <= 3; ++i ) {
		a(i) = a1(i) - a2(i);
		b(i) = a3(i) - a2(i);
		c(i) = b1(i) - a2(i);
	}
	cross(b,a,ab,dab);
	cross(c,a,ac,dac);
	cross(ab,a,aba,daba);
//* get vector lengths
//* get ab.dot.ac
	ab_ac = ( ab(1) * ac(1) ) + ( ab(2) * ac(2) ) + ( ab(3) * ac(3) );
//*     aba.dot.ac
	aba_ac = ( aba(1) * ac(1) ) + ( aba(2) * ac(2) ) + ( aba(3) * ac(3) );
//* cosine ab_ac
	if ( dab == 0.0 ) {
		ab_ac = 0.0;
	} else {
		ab_ac /= dab;
	}
//* cosine aba_ac = sine ab_ac
	if ( daba == 0.0 ) {
		aba_ac = 0.0;
	} else {
		aba_ac /= daba;
	}
//* get chi
	if ( aba_ac != 0.0 ) {
		chi = std::atan2( aba_ac, ab_ac );
	} else {
		chi = 0.0;
	}
//* get matrix for chi rotation
	getrot( a1, a2, chi, mat, vec );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin lineup
///
/// @brief
/// find the matrix (mat) and vector (vec) that line up a1->a2 with b1->b2,
/// with a1 superimposed on b1.
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  b1 - [in/out]? -
/// @param  b2 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
lineup(
	FArray1Da_double a1,
	FArray1Da_double a2,
	FArray1Da_double b1,
	FArray1Da_double b2,
	FArray2Da_double mat,
	FArray1Da_double vec
)
{
	using numeric::sin_cos_range;

	a1.dimension( 3 );
	a2.dimension( 3 );
	b1.dimension( 3 );
	b2.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

	FArray1D_double a( 3 );
	FArray1D_double b( 3 );
	FArray1D_double c( 3 );

//*
	double chi = 0.0;
	double phi = 0.0;
	double psi = 0.0;
//* get difference vectors
	for ( int i = 1; i <= 3; ++i ) {
		a(i) = a2(i) - a1(i);
		b(i) = b2(i) - b1(i);
	}
//* get vector lengths
//* get a.dot.b
	double da = ( a(1) * a(1) ) + ( a(2) * a(2) ) + ( a(3) * a(3) );
	double db = ( b(1) * b(1) ) + ( b(2) * b(2) ) + ( b(3) * b(3) );
	double dab = ( a(1) * b(1) ) + ( a(2) * b(2) ) + ( a(3) * b(3) );
	da = std::sqrt(da);
	db = std::sqrt(db);
//* get chi
//cb The following is to protect against underflows on the Mac:
	if ( std::abs(dab-(da*db)) < 0.000001 ) {
		chi = 0.0;
	} else {
		chi = std::acos( sin_cos_range(dab/(da*db)) );
	}
	if ( chi >= 1.0E-4 ) {
//* cross product A X B = C
		cross(a,b,c,dab);
//* get phi, psi
		psi = std::acos( sin_cos_range(c(3)/dab) );
		if ( c(1) != 0.0 || c(2) != 0.0) phi = std::atan2( -c(1), c(2) );
	}
//* get matrix
	getmat(phi,psi,chi,mat);
//* get vector, = -Mat*a1 + b1
	rotate(mat,a1,c);
	for ( int i = 1; i <= 3; ++i ) {
		vec(i) = -c(i) + b1(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin vector_midpoint
///
/// @brief
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
/// @param  c - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vector_midpoint(
	FArray1Da_float a,
	FArray1Da_float b,
	FArray1Da_float c
)
{
//     jjg 3/29/01
	c[ 0 ] = ( a[ 0 ] + b[ 0 ] ) * 0.5f;
	c[ 1 ] = ( a[ 1 ] + b[ 1 ] ) * 0.5f;
	c[ 2 ] = ( a[ 2 ] + b[ 2 ] ) * 0.5f;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin vector_copy
///
/// @brief
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vector_copy(
	FArray1Da_float a,
	FArray1Da_float b
)
{
//     jjg 3/29/01
	b[ 0 ] = a[ 0 ];
	b[ 1 ] = a[ 1 ];
	b[ 2 ] = a[ 2 ];
}


////////////////////////////////////////////////////////////////////////////////
/// @begin vector_project_normal
///
/// @brief
/// (project b normal to a)
///
/// @detailed
///
/// @param  a - [in/out]? -
/// @param  b - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vector_project_normal(
	FArray1Da_float a,
	FArray1Da_float b
)
{
	a.dimension( 3 );
	b.dimension( 3 );

//     jjg 3/29/01                               // (project b normal to a)

	float ab = dotprod(a,b);
	float aa = dotprod(a,a);
	float c = ab / aa;

	b(1) -= a(1) * c;
	b(2) -= a(2) * c;
	b(3) -= a(3) * c;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin vector_write
///
/// @brief
///
/// @detailed
///
/// @param  unit - [in/out]? -
/// @param  a - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vector_write(
	std::ostream & unit,
	FArray1Da_float a
)
{
//     jjg 3/30/01
	unit << '(' <<
	 F( 10, 3, a[ 0 ] ) << ',' <<
	 F( 10, 3, a[ 1 ] ) << ',' <<
	 F( 10, 3, a[ 2 ] ) << ')';
}

////////////////////////////////////////////////////////////////////////////////
/// @begin vector_write_line
///
/// @brief
///
/// @detailed
///
/// @param  unit - [in/out]? -
/// @param  a - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vector_write_line(
	std::ostream & unit,
	FArray1Da_float a
)
{
//     jjg 3/30/01
	unit << '(' <<
	 F( 10, 3, a[ 0 ] ) << ',' <<
	 F( 10, 3, a[ 1 ] ) << ',' <<
	 F( 10, 3, a[ 2 ] ) << ')' << '\n';
}

////////////////////////////////////////////////////////////////////////////////
/// @begin vector_name_write_line
///
/// @brief
///
/// @detailed
///
/// @param  unit - [in/out]? -
/// @param  a - [in/out]? -
/// @param  name - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vector_name_write_line(
	std::ostream & unit,
	FArray1Da_float a,
	std::string const & name
)
{
//     jjg 4/30/01
	unit << A( 10, name ) << ": (" <<
	 F( 10, 3, a[ 0 ] ) << ',' <<
	 F( 10, 3, a[ 1 ] ) << ',' <<
	 F( 10, 3, a[ 2 ] ) << ')' << '\n';
}

void
Dvector_name_write_line(
	std::ostream & unit,
	FArray1Da_double a,
	std::string const & name
)
{
//     jjg 4/30/01
	unit << A( 10, name ) << ": (" <<
	 F( 10, 3, a[ 0 ] ) << ',' <<
	 F( 10, 3, a[ 1 ] ) << ',' <<
	 F( 10, 3, a[ 2 ] ) << ')' << '\n';
}

////////////////////////////////////////////////////////////////////////////////
/// @begin matrix_write
///
/// @brief
///
/// @detailed
///
/// @param  unit - [in/out]? -
/// @param  M - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
matrix_write(
	std::ostream & unit,
	FArray2Da_float M
)
{
	M.dimension( 3, 3 );

//     jjg 4/30/01
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			unit << F( 5, 0, M(i,j) );
		}
		unit << '\n';
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin matrix_name_write
///
/// @brief
///
/// @detailed
///
/// @param  unit - [in/out]? -
/// @param  M - [in/out]? -
/// @param  name - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
matrix_name_write(
	std::ostream & unit,
	FArray2Da_float M,
	std::string const & name
)
{
	M.dimension( 3, 3 );

//     jjg 4/30/01
	unit << A( 70, name ) << ':' << '\n';
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			unit << F( 10, 3, M(i,j) );
		}
		unit << '\n';
	}
}

void
Dmatrix_name_write(
	std::ostream & unit,
	FArray2Da_double M,
	std::string const & name
)
{
	M.dimension( 3, 3 );

//     jjg 4/30/01
	unit << A( 70, name ) << ':' << '\n';
	for ( int i = 1; i <= 3; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			unit << F( 10, 3, M(i,j) );
		}
		unit << '\n';
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin vec_angle
///
/// @brief
///
/// @detailed
///
/// @param[in]   a - in -
/// @param[in]   b - in -
/// @param[in]   c - in -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
vec_angle(
	FArray1Da_float a,
	FArray1Da_float b,
	FArray1Da_float c
)
{
	using numeric::conversions::degrees;
	using numeric::in_sin_cos_range;
	using numeric::sin_cos_range;

	a.dimension( 3 );
	b.dimension( 3 );
	c.dimension( 3 );

	static FArray1D_float const zero( 3, 0.0 );
	static FArray1D_float ba( 3 );
	static FArray1D_float bc( 3 );

	subvec(a,b,ba);
	subvec(c,b,bc);

	float const cosine = dotprod(ba,bc) / ( vec_dist(ba,zero) * vec_dist(bc,zero) );

	if ( !in_sin_cos_range(cosine) ) {
		//debug
		//std::cout << "dotprod = " << dotprod(ba,bc) << '\n';
		//std::cout << "|ba| = " << vec_dist(ba,zero) << '\n';
		//std::cout << "|bc| = " << vec_dist(bc,zero) << '\n';
		std::cout << "ERROR: function vec_angle in util_vector.cc" << '\n';
		std::cout << "cos() value: " << cosine << " is not within [-1,1], something is wrong" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return degrees( std::acos( sin_cos_range( cosine ) ) );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rotate_in_plane
///
/// @brief rotate the vector in the defined plane
///
/// @detailed  rotate the vetor a1->a2 in the plane of b1,b2, and b3
///       return the position of a2
///
/// @param[in]   a1,a2    - in - the vetor a1->a2
/// @param[out]   a2       - out- return the position of a2
/// @param[in]   b1,b2,b3 - in - define the plane of b1,b2 and b3
/// @param[in]   rot_ang  - in - the rotation angle
///
/// @global_read none
///
/// @global_write none
///
/// @remarks  a2 is changed
///
/// @references
///
/// @authors   Lin Jiang 09/19/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rotate_in_plane(
								FArray1Da_float a1,
								FArray1Da_float a2,
								FArray1Da_float b1,
								FArray1Da_float b2,
								FArray1Da_float b3,
								float & rot_ang // rotate angle
								)
{
	using numeric::conversions::radians;

	a1.dimension( 3 );
	a2.dimension( 3 );
	b1.dimension( 3 );
	b2.dimension( 3 );
	b3.dimension( 3 );

	//lin  rotate the vetor a1->a2 in the plane of b1,b2, and b3
	//lin  return the position of a2


	FArray1D_float b12( 3 );
	FArray1D_float b13( 3 );
	FArray1D_float norm1( 3 );
	FArray1D_float vec( 3 );
	FArray2D_float mat( 3, 3 );
	float dis1,dis2;

	bool const safe_check = { false };

	if ( safe_check ) distance_bk(a1(1),a2(1),dis1);
	for ( int I = 1; I <= 3; ++I ) {
		b12(I) = b2(I)-b1(I);
		b13(I) = b3(I)-b1(I);
	}
	//lin generate the normal line of the plane
	float const rot_ang1 = radians( rot_ang );
	cross_bk(b13,b12,norm1);
	norm1 += a1;
	//lin generate rotation vector and matrix
	getrot_bk(a1, norm1, rot_ang1, mat, vec);
	//lin  move atom
	move_bk(a2(1),mat,vec);
	if ( safe_check ) {
		distance_bk(a1(1),a2(1),dis2);
		if ( dis1-dis2 > 0.001 ) {
			std::cout << "rotation angle error!!!dis1 is" << SS( dis1 ) <<
			" dis2 is" << SS( dis2 ) << ",.....exiting" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

}
