// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13616 $
//  $Date: 2007-03-17 23:39:36 -0700 (Sat, 17 Mar 2007) $
//  $Author: stuartm $


// Rosetta Headers
#include "vdw.h"
#include "after_opts.h"
#include "cenlist.h"
#include "count_pair_position.h"
#include "current_pose.h"
#include "ligand_ns.h"
#include "maps.h"
#include "maps_ns.h"
#include "minimize.h"
#include "misc.h"
#include "param.h"
#include "pose_vdw.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "structure.h"

// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpressions.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2Dp.hh>
#include <ObjexxFCL/FArray3Dp.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <iostream>

//car  10-18-2004
//car tried grouping this data into various types of structures
//car and STL containers, but all trials resulted in very large
//car slowdowns to the ab initio benchmark
namespace vdw_bumps {

	int n_bump = { 0 };
	int n_best_bump = { 0 };

	inline
	void
	bumps_initializer( FArray1D_float & )
	{ // Reset counters to zero when arrays reallocated
		n_bump = 0;
		n_best_bump = 0;
	}

	Dimension const MAXBUMPS( param::MAX_RES() * param::MAX_RES() );

	FArray1D_int best_bump_res1( MAXBUMPS );
	FArray1D_int best_bump_res2( MAXBUMPS );
	FArray1D_float best_bump( MAXBUMPS );
	FArray1D_int bump_res1( MAXBUMPS );
	FArray1D_int bump_res2( MAXBUMPS );
	FArray1D_float bump( MAXBUMPS, bumps_initializer );
}


//car   10-18-2004
//car  tried replacing hard-coded copies of this function in copy_best_bumps
//car  and vdw_compute with this inlined function but observed a small but
//car  significant slowdown for the abinitio benchmark
inline
void
add_bump(
	int const res1,
	int const res2,
	float const vdw
)
{
	using namespace vdw_bumps;
	++n_bump;
	if ( n_bump > MAXBUMPS ) {
		std::cout << "Increase MAXBUMPS in vdw.cc" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	bump_res1(n_bump) = res1;
	bump_res2(n_bump) = res2;
	bump(n_bump) = vdw;
}


float
copy_best_bumps(
	int const first,
	int const last,
	FArray2DB_bool const & pair_moved
)
{
	using namespace vdw_bumps;

	float sum = 0.0;
	int res1, res2;
	float vdw;

	for ( int i = 1; i <= n_best_bump; ++i ) {
		res1 = best_bump_res1(i);
		res2 = best_bump_res2(i);
		if ( ( res1 > last  && res2 > last ) || ( res1 < first && res2 < first )
		 || ! pair_moved(res1,res2) ) {
			vdw = best_bump(i);
			sum += vdw;

			//car add_bump
			++n_bump;
			if ( n_bump > MAXBUMPS ) {
				std::cout << "Increase MAXBUMPS in vdw.cc" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			bump_res1(n_bump) = res1;
			bump_res2(n_bump) = res2;
			bump(n_bump) = vdw;
		}
	}

	return sum;
}



//
// public functions
//
///////////////////////////////////////////////////////////////////////////////
/// @begin vdw_compute
///
/// @brief
///
/// @detailed
///
/// @param  vdw_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vdw_compute( float & vdw_score )
{
	using namespace cenlist_ns;
	using namespace ligand;
	using namespace misc;
	using namespace param;
	using namespace runlevel_ns;
	using namespace vdw;
	using namespace vdw_bumps;
	using numeric::xyzVector_float;

	if ( score_check_current_pose() ) {
		pose_vdw_compute( vdw_score );
		return;
	}

	int first = 1;
	int last = total_residue;
	int const cendist_size1 = cendist.size1();
	int const Eposition_size12 = Eposition.size1() * Eposition.size2();

	int type0; // centroid i
	int type1; // atom j
	int type2; // atom i
	int type3; // centroid j
	float cendist_ij, dist2, alpha_alpha;
	float atomvdw; // lower distance bound, temp

//   compute the distance between all atom pairs, and score for bumps
//
//  1) if the centroids >  12 A apart, no atoms to residues can bump
//  2) if CA(i) CA(j) distance > 7A, no backbone atoms of pair bump
//  3) dont check any pairs of atoms who's distance is unchanged.
//    maintain internal arrays to determine what parts of molecule are unchanged
//------------------------------------------------------------------------------
//

// what range of protein changed?
	FArray2DB_bool const & pair_moved( retrieve_pair_moved(true) );
	maps_get_pair_moved_region( first, last );

// compute i-centroid to j-centroid distance.  store it for use elsewhere
	update_cendist(total_residue,centroid);

	float vdw_score_local = 0.0;
	n_bump = 0;

	for ( int i = 1, lci = 0; i <= last; ++i, lci += 3 ) { // residue 1
		type0 = atom_type_cen(i);
		xyzVector_float const cen_i( &centroid[ lci ] ); // centroid(1-3,i)

		int const res_i = res(i);
		int const lEt_1i = Eatom_type.index(1,i);
		int const lEp_11i = Eposition.index(1,1,i);

		xyzVector_float const Epos_2i( &Eposition(1,2,i) ); // Eposition(1-3,2,i)

		for ( int j = std::max( i, first ), lcj = centroid.index(1,j),
		 lcd = cendist.index(i,j), lEp_12j = Eposition.index(1,2,j);
		 j <= total_residue;
		 ++j, lcj += 3, lcd += cendist_size1, lEp_12j += Eposition_size12 ) { // residue 2
			float vdw_score_j = 0.0;
			if ( pair_moved[ lcd ] ) { //pair_moved(i,j)
				cendist_ij = cendist[ lcd ]; // cendist(i,j); // look up prev calc dist
//--------
// as long as were at it, we now do a clash check
//  first do a coarse grain reality check on centroid seperations
//  and if it passes carefully check atom-atom distances
//--------

				if ( cendist_ij <= cen_dist_cutoff2 ) { // reality check
	  //std::cout << "vdw" << SS( i ) << SS( j ) << SS( cendist_ij ) << std::endl;

					type3 = atom_type_cen(j);
					int const res_j = res(j);

//     centroid-centroid clash check
					if ( count_pair_position( i, type0, res_i, j, type3, res_j ) ) {
						atomvdw = atom_vdw(type0,type3);
						dist2 = atomvdw - cendist_ij;
						if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / atomvdw; // clash score
					}

					xyzVector_float const cen_j( &centroid[ lcj ] ); // centroid(1-3,j)

// As a prefilter we first compute the calpha-calpha distance.
// . If this is greater than 49 (7 squared) none of the backbone/cbeta can clash and we skip the distance backbone checks.
// . Additionally if this is greater than 60 then we can skip the centriod backbone checks.
//
//car these cutoffs defined in param.h
					alpha_alpha = distance_squared( Epos_2i, // Eposition(1-3,2,i)
					 xyzVector_float( &Eposition[ lEp_12j ] ) ); // Eposition(1-3,2,j)

// minimum  centroid-pentamer radius
					int const lEt_1j = Eatom_type.index(1,j);
					int const lEp_11j = Eposition.index(1,1,j);

//---------
//  check i-centroid to j-atom  clash
//---------
					for ( int jatom = 1, lEtj = lEt_1j, lEpj = lEp_11j; jatom <= MAX_POS; ++jatom, ++lEtj, lEpj += 3 ) {
						type1 = Eatom_type[ lEtj ]; // Eatom_type(jatom,j)
						if ( count_pair_position( i, type0, res_i, j, type1, res_j ) ) {
							atomvdw = atom_vdw(type0,type1);
							dist2 = atomvdw - distance_squared( cen_i, // centroid(1-3,i)
							 xyzVector_float( &Eposition[ lEpj ] ) ); // Eposition(1-3,jatom,j)

							if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / atomvdw; // clash score
						}
					} // jatom

//---------
//  check j-centroid to i-atom  clash
//---------
					for ( int iatom = 1, lEti = lEt_1i, lEpi = lEp_11i; iatom <= MAX_POS; ++iatom, ++lEti, lEpi += 3 ) {
						type2 = Eatom_type[ lEti ]; // Eatom_type(iatom,i) // vdw calc
						if ( count_pair_position( j, type3, res_j, i, type2, res_i ) ) {
							atomvdw = atom_vdw(type3,type2);
							dist2 = atomvdw - distance_squared( cen_j, // centroid(1-3,j)
							 xyzVector_float( &Eposition[ lEpi ] ) ); // Eposition(1-3,iatom,i)

							if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / atomvdw; // clash score
						}
					} // iatom

//------
//   check helix backbone to backbone non-helix nitrogen/oxygen clashes
//------
					if (get_ss_vdw()) {
						char const ssi = secstruct(i);
						char const ssj = secstruct(j);
						ss_vdw( Eposition, MAX_POS, i, ssi, j, ssj, lEt_1j, lEp_11j, lEt_1i, lEp_11i, vdw_score_j );
					}

//------
//   check i-pentamer to j-pentamer
//------
					if ( alpha_alpha < ca_dist_cutoff2 ) { // clash radius CA-CA

						for ( int jatom = 1, lEtj = lEt_1j, lEpj = lEp_11j;
						 jatom <= MAX_POS; ++jatom, ++lEtj, lEpj += 3 ) {
							type1 = Eatom_type[ lEtj ]; // Eatom_type(jatom,j)

							if ( type1 != 0 ) { // count_pair_position would catch but leave so we can avoid loop
								xyzVector_float const Epos_jatom( &Eposition[ lEpj ] ); // Eposition(1-3,jatom,j)

								for ( int iatom = 1, lEti = lEt_1i, lEpi = lEp_11i;
								 iatom <= MAX_POS; ++iatom, ++lEti, lEpi += 3 ) {
									// all backbone atoms in residue i
									type2 = Eatom_type[ lEti ]; // Eatom_type(iatom,i)
									if ( count_pair_position( i, type2, res_i, j, type1, res_j ) ) {
										atomvdw = atom_vdw(type1,type2);
										dist2 = atomvdw - distance_squared( Epos_jatom, // Eposition(1-3,jatom,j)
										 xyzVector_float( &Eposition[ lEpi ] ) ); // Eposition(1-3,iatom,i)

										if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / atomvdw; // clash score
									} // end count_pair check
								} // end iatom
							} // end type1 glycine check
						} // end jatom
					} // end alpha-alpha < ca_dist_cutoff2 check

				} // end cen_dist_cutoff2
			}

			if ( vdw_score_j > 0.0 ) {
				vdw_score_local += vdw_score_j;

				//car add_bump
				++n_bump;
				if ( n_bump > MAXBUMPS ) {
					std::cout << "Increase MAXBUMPS in vdw.cc" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
				bump_res1(n_bump) = i;
				bump_res2(n_bump) = j;
				bump(n_bump) = vdw_score_j;
			}
		} // end j
	} // end i=1,end_res

	vdw_score_local += copy_best_bumps( first, last, pair_moved );

	vdw_hetero( Eposition, centroid, atom_type_cen, total_residue, ligand::ligand_one->atom_vector,
		ligand::ligand_one->atom_vector.size(), atom_vdw, vdw_score_local );

	vdw_score = vdw_score_local * 0.8f;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin ss_vdw
///
/// @brief  Increases the backbone atom vdw scores for helix to non-helix nitrogen/oxygen
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors rvernon 11/01/06
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
ss_vdw( FArray3DB_float const & Eposition,
				int const & MAX_POS,
				int const & i,
				char const & ssi,
				int const & j,
				char const & ssj,
				int const & lEt_1j,
				int const & lEp_11j,
				int const & lEt_1i,
				int const & lEp_11i,
				float & vdw_score_j)
{
//------
//   check i non-helical Oxygen and Nitrogen for clash with j helical backbone
//------

	using numeric::xyzVector_float;

	if ( std::abs(i - j) > get_ss_vdw_neighborspacer() ) { // Only test Ox/Ni to helix backbone clashes at least 2 residues from a helix termini

		if ((ssi != 'H') && (ssj == 'H')) { // Check for j = helix, i = strand or loop
			for ( int jatom = 1, lEtj = lEt_1j, lEpj = lEp_11j; jatom <= MAX_POS; ++jatom, ++lEtj, lEpj += 3 ) {
				//type1 = Eatom_type(jatom,j); // Eatom_type(jatom,j) // vdw calc

				//if ((type1 != 3)) {
				if (ssi == 'E') {
					//vdw = 3.5 A if sheet Oxygen to helix backbone
					float dist2 = 12.25 - distance_squared( xyzVector_float( &Eposition[ lEpj ] ), // Eposition(1-3,jatom,j)
																						xyzVector_float( &Eposition(1,5,i) )); // Eposition(1-3,OXYGEN,i)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 12.25; // clash score

					//vdw = 4 A if sheet Nitrogen to helix backbone
					dist2 = 16 - distance_squared( xyzVector_float( &Eposition[ lEpj ] ), // Eposition(1-3,jatom,j)
																				 xyzVector_float( &Eposition(1,1,i) )); // Eposition(1-3,NITROGEN,i)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 16; // clash score

				} else {

					//vdw = 3.2 A if loop Oxygen to helix backbone
					float dist2 = 10.24 - distance_squared( xyzVector_float( &Eposition[ lEpj ] ), // Eposition(1-3,jatom,j)
																						xyzVector_float( &Eposition(1,5,i) )); // Eposition(1-3,OXYGEN,i)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 10.24; // clash score

									//vdw = 3 A if loop Nitrogen to helix backbone
					dist2 = 9 - distance_squared( xyzVector_float( &Eposition[ lEpj ] ), // Eposition(1-3,jatom,j)
																				xyzVector_float( &Eposition(1,1,i) )); // Eposition(1-3,NITROGEN,i)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 9; // clash score

				}
			}
//------
//   check j non-helical Oxygen and Nitrogen for clash with i helical backbone
//------
		} else if ((ssj != 'H') && (ssi == 'H')) { // Do Reverse: Check for i = helix, j = strand or loop
			//int 1Epj_o = Eposition.index(1,5,j);
			//xyzVector_float 1Eposi_o = Eposition[1Epj_o];
			//int 1Epj_n = Eposition.index(1,2,j);
			//xyzVector_float 1Eposi_n = Eposition[1Epj_n];

			for ( int iatom = 1, lEti = lEt_1i, lEpi = lEp_11i; iatom <= MAX_POS; ++iatom, ++lEti, lEpi += 3 ) {
				//type1 = Eatom_type(iatom,i); // Eatom_type(iatom,i) // vdw calc


				if (ssj == 'E') {
					//vdw = 3.5 A if sheet Oxygen to helix backbone
					float dist2 = 12.25 - distance_squared( xyzVector_float( &Eposition[ lEpi ] ), // Eposition(1-3,iatom,i)
																							xyzVector_float( &Eposition(1,5,j) )); // Eposition(1-3,OXYGEN,j)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 12.25; // clash score
					//vdw = 4 A if sheet Nitrogen to helix backbone
					dist2 = 16 - distance_squared( xyzVector_float( &Eposition[ lEpi ] ), // Eposition(1-3,iatom,i)
																					xyzVector_float( &Eposition(1,1,j) )); // Eposition(1-3,NITROGEN,j)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 16; // clash score

				} else {
					//vdw = 3.2 A if loop Oxygen to helix backbone
					float dist2 = 10.24 - distance_squared( xyzVector_float( &Eposition[ lEpi ] ), // Eposition(1-3,iatom,i)
																							xyzVector_float( &Eposition(1,5,j) )); // Eposition(1-3,OXYGEN,j)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 10.24; // clash score
					//vdw = 3 A if loop Nitrogen to helix backbone
					dist2 = 9 - distance_squared( xyzVector_float( &Eposition[ lEpi ] ), // Eposition(1-3,iatom,i)
																					xyzVector_float( &Eposition(1,1,j) )); // Eposition(1-3,NITROGEN,j)

					if ( dist2 > 0.0 ) vdw_score_j += ( dist2 * dist2 ) / 9; // clash score

				}
			}
		}
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin accept_best_vdw_bumps
///
/// @brief  copies current bump list to best bump_list
///
/// @detailed
///
/// @global_read
///  current_bumps in vdw_bumps namespace
///
/// @global_write
///  best_bumps in vdw_bumps namespace
///
/// @remarks
///
/// @references
///
/// @authors car 9/25/04
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
accept_best_vdw_bumps()
{
	using namespace vdw_bumps;

	for ( int i = 1; i <= n_bump; ++i ) {
		best_bump_res1(i) = bump_res1(i);
		best_bump_res2(i) = bump_res2(i);
		best_bump(i) = bump(i);
	}
	n_best_bump = n_bump;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin vdw_hetero
///
/// @brief
///
/// @detailed
///
/// @param  Eposition - [in/out]? - positon array for residues
/// @param  centroid - [in/out]? -
/// @param  atom_type_cen - [in/out]? - residue type array
/// @param  total_residue - [in/out]? - logical size of Eposition
/// @param  hetero_atom_cent - [in/out]? - hetero atom centroid types
/// @param  hetero_atom_coord - [in/out]? - hetero atom coordinates
/// @param  atom_vdw - [in/out]? -
/// @param  vdw_score - [in/out]? - adds the hetero score to the input score
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
vdw_hetero(
	FArray3DB_float const & Eposition, // positon array for residues
	FArray2DB_float const & centroid,
	FArray1DB_short const & atom_type_cen, // residue type array
	int total_residue, // logical size of Eposition
	std::vector< Atom * > const & ligand, // ligand information
//	FArray1DB_int const & hetero_atom_cent, // hetero atom centroid types
//	FArray2DB_float const & hetero_atom_coord, // hetero atom coordinates
	int hetero_atom_count, // logical size of Het_position
	FArray2DB_float const & atom_vdw,
	float & vdw_score // adds the hetero score to the input score
)
{
	using namespace param;
	using namespace cenlist_ns;
	using numeric::xyzVector_double;

	if ( hetero_atom_count == 0 ) return;

//***    parameters
	int const NITRO = { 1 };
	int const CALPHA = { 2 };
	int const CBETA = { 3 };
	int const CCARB = { 4 };
	int const OXYGEN = { 5 };
	int const GLYCINE = { 0 };

//****   local
	int type_het, type_res; // atom type, residue type
	float dist2;

// retrieve i-centroid to j-centroid positions

	for ( int i = 1; i <= total_residue; ++i ) {

		xyzVector_double const cen_i( &centroid( 1, i ) ); // centroid(1-3,i)
		type_res = atom_type_cen(i);

		xyzVector_double const Eposition_CALPHA_i( &Eposition(1,CALPHA,i) );
		xyzVector_double const Eposition_NITRO_i( &Eposition(1,NITRO,i) );
		xyzVector_double const Eposition_CCARB_i( &Eposition(1,CCARB,i) );
		xyzVector_double const Eposition_OXYGEN_i( &Eposition(1,OXYGEN,i) );
		xyzVector_double const Eposition_CBETA_i( &Eposition(1,CBETA,i) );

		for ( int j = 0; j < hetero_atom_count; ++j ) {
			if ( ligand[j]->get_ros_atom_cent() != -1 ) { // exclude hydrogens here
				type_het = ligand[j]->get_ros_atom_cent(); // fourth column is atom type
				xyzVector_double const hetero_atom_coord_j=ligand[j]->get_coordinates(); // hetero_atom_coord(1-3,j)

				if ( type_res != GLYCINE ) {
					dist2 = atom_vdw(type_het,type_res) - distance_squared( cen_i, hetero_atom_coord_j );
					if ( dist2 > 0.0 ) vdw_score += dist2;
				}

				dist2 = distance_squared( hetero_atom_coord_j, Eposition_CALPHA_i );

				if ( dist2 <= ca_dist_cutoff2 ) {

					dist2 = atom_vdw(type_het,CALPHA) - dist2;
					if ( dist2 > 0.0 ) vdw_score += dist2;

					dist2 = atom_vdw(type_het,NITRO) - distance_squared( hetero_atom_coord_j, Eposition_NITRO_i );
					if ( dist2 > 0.0 ) vdw_score += dist2;

					dist2 = atom_vdw(type_het,CCARB) - distance_squared( hetero_atom_coord_j, Eposition_CCARB_i );
					if ( dist2 > 0.0 ) vdw_score += dist2;

					dist2 = atom_vdw(type_het,OXYGEN) - distance_squared( hetero_atom_coord_j, Eposition_OXYGEN_i );
					if ( dist2 > 0.0 ) vdw_score += dist2;

					if ( type_res != GLYCINE ) { // cbeta is a special case
						dist2 = atom_vdw(type_het,CBETA) - distance_squared( hetero_atom_coord_j, Eposition_CBETA_i );
						if ( dist2 > 0.0 ) vdw_score += dist2;
					}
				}
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin update_cendist
///
/// @brief
///
/// @detailed
/// uses pair_moved to do fast updates
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
update_cendist(
    int total_residue,
    FArray2D_float centroid
)
{
	using namespace cenlist_ns;
 	using namespace param;
	using numeric::xyzVector_double;

	int const cendist_size1 = cendist.size1();

	if ( score_check_current_pose() ) {
		pose_update_cendist( score_get_current_pose() );
		return;
	}

	FArray2DB_bool const & pair_moved( retrieve_pair_moved( true ) );
	//Objexx:SGM pair_moved from Pose has dimensions of total_residue not MAX_RES()
	// so don't use the lij linear indexing from cendist on pair_moved unless these
	// are guaranteed to be the same (and add an assert test for it)

	for ( int i = 1, lci = centroid.index(1,i); i <= total_residue; ++i, lci += 3 ) {
		xyzVector_double const cen_i( &centroid[ lci ] ); // centroid(1-3,i)
		if ( cen_i.is_zero() ) {
			for ( int j = i; j <= total_residue; ++j ) {
				cendist(i,j) = cendist(j,i) = 999.0;
			}
		} else {
			cendist(i,i) = 0.0;
			for ( int j = i+1, lji = cendist.index(j,i), lij = cendist.index(i,j), lcj = centroid.index(1,j);
			 j <= total_residue; ++j, ++lji, lij += cendist_size1, lcj += 3 ) {
				if ( ! pair_moved(i,j) ) {
					cendist[ lij ] = best_cendist[ lij ]; // cendist(i,j) = best_cendist(i,j)
					cendist[ lji ] = best_cendist[ lji ]; // cendist(j,i) = best_cendist(j,i)
				} else {
					cendist[ lij ] = cendist[ lji ] = distance_squared(
					 cen_i, // centroid(1-3,i)
					 xyzVector_double( &centroid[ lcj ] ) ); // centroid(1-3,j)
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin accept_best_cendist
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
accept_best_cendist()
{
	using namespace cenlist_ns;
	using namespace misc;

	for ( int j = 1; j <= total_residue; ++j ) {
		for ( int i = 1; i <= total_residue; ++i ) {
			best_cendist(i,j) = cendist(i,j);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin compute_CB_environment
///
/// @brief
///
/// @detailed
///car needs cendist list updated!!
///car done by calculate_cendist. vdw_compute calls calculate cen_dist,
///car so if vdw has been called, calc_cendist need not be called first.
///jg cen6(i) is the number of centoids within 6 Angstroms of residue i
///jg cen10(i) is the number of centoids within 10 Angstroms of residue i
///jg cen12(i)+cen6(i) is the number of centoids within 12 Angstroms of residue i
///jg cenlist(*,i) is the list of residues within 12 Angstroms of residue i
///jg cen12up(i) is the number of residues in cenlist(*,i)
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
compute_CB_environment()
{

	using namespace cenlist_ns;
	using namespace misc;
	using namespace runlevel_ns;

	static double const cen_dist6_pad_plus = cen_dist6_pad + 36.0;
	static double const cen_dist10_pad_plus = cen_dist10_pad + 100.0;
	static double const cen_dist12_pad_plus = cen_dist12_pad + 144.0;

	static double const cen_dist6_pad_hinv = -0.5 / cen_dist6_pad;
	static double const cen_dist10_pad_hinv = -0.5 / cen_dist10_pad;
	static double const cen_dist12_pad_hinv = -0.5 / cen_dist12_pad;

	static double const cen_dist_cutoff_12_pad = cen_dist_cutoff2 + cen_dist12_pad;

	float tempdist, interp;

// initialize values
	for ( int i = 1; i <= total_residue; ++i ) {
		cen6(i) = 1; // 1 because I match myself
		fcen6(i) = 1.0;
		cen12(i) = 0; // 0 because I'm closer than 6 angstroms to myself
		fcen12(i) = 0.0;
		cen10(i) = 1; // 1 because I match myself
		fcen10(i) = 1.0;
		cen12up(i) = 0;
	}

// loop over all centroid pairs
	int const cendist_size1 = cendist.size1();
	for ( int i = 1; i < total_residue; ++i ) { // outer loop over residues
		//// Use local (i) values for efficiency
		int cen6_i = cen6(i);
		float fcen6_i = fcen6(i);
		int cen12_i = cen12(i);
		float fcen12_i = fcen12(i);
		int cen10_i = cen10(i);
		float fcen10_i = fcen10(i);
		int cen12up_i = cen12up(i);
		for ( int j = i+1, lcd = cendist.index(i,j); j <= total_residue; ++j, lcd += cendist_size1 ) {

//--------
//     recall  i-centroid to j-centroid distance. (from vdw_compute)
//--------
			tempdist = cendist[ lcd ]; // cendist(i,j)

//  compute arrays needed for C-beta  energy function
//  first do a coarse grain reality check on centroid separations
			if ( tempdist <= cen_dist_cutoff_12_pad ) {

				++cen12up_i;

				if ( cen12up_i <= MAX_CENLIST ) cen_list(cen12up_i,i) = j;

				if ( tempdist <= 100.0 ) {
					if ( tempdist <= 36.0 ) {
						++cen6_i;
						++cen6(j);
					} else {
						++cen12_i;
						++cen12(j);
					}
					++cen10(j);
					++cen10_i;
				} else {
					++cen12_i;
					++cen12(j);
				}


//cems--------------------------------------------------------------------------
//     interpolation notes --Historically we have broken the
//     centroid density statistics into three bins: i) pairs
//     less than 6 angstroms ii) pairs less than 10 angstroms ems
//     iii) and pairs between 6 and 12 angstroms the resulting
//     abruptness in the scoring functions due to the arbitrary radius
//     cutoffs has caused some problems during gradient minimization.
//     therefore this was replaced with an interpolated binning
//     schema as follows: When a pairwise distance lies within "+/-
//     dr" of the bin boundary (6,10,12) then partial credit is given
//     to the enclosing bins.  So for example, if fgap=0.5 angstroms, and
//     a pair radius were 6.4 angstroms, then a fractional count is
//     given to BOTH the "less-than-6" bin AND to the
//     "between-6-and-10" bin.  The sum of these fractions always adds to
//     one.  So that we dont have to re-do the statistics we
//     currently use we want to keep "fgap" small.  ideally fgap
//     should be large compared to the search algorithm step size, and
//     larger than the expected roundoff error in any refold
//     operation, and otherwise as small as possible.  Also we want
//     to cleverly choose the interpolation function so that the average
//     number of counts getting into the bins is the same as under
//     the old schema.  As long as dr is small then we can use either
//     r+/-fgap or alternatively r^2+/-fgap^2 and this will be
//     approximately satsified.  since the squared from is easier to work
//     we will use this.  in the code below the frag^2 term is called
//     a _pad, and we allow for different pad_sizes on the three radii.
//cems--------------------------------------------------------------------------

				if ( tempdist <= cen_dist10_pad_plus ) {

					interp = std::min( ( tempdist - cen_dist10_pad_plus ) * cen_dist10_pad_hinv, 1.0 );

					if ( runlevel >= gush && interp < 0.0 ) {
						std::cout << "interp roundoff 1" << SS( interp ) <<
						 SS( tempdist ) << SS( cen_dist10_pad ) << SS( cen10_i ) <<
						 SS( fcen10_i ) << std::endl;
					}

					fcen10_i += interp;
					fcen10(j) += interp;

				}

				if ( tempdist <= cen_dist6_pad_plus ) { // its sort of a "6"

					interp = std::min( ( tempdist - cen_dist6_pad_plus ) * cen_dist6_pad_hinv, 1.0 );

					if ( runlevel >= gush && interp < 0.0 ) {
						std::cout << "interp roundoff 2 " << SS( interp ) << std::endl;
					}

					fcen6_i += interp;
					fcen6(j) += interp;
					fcen12_i += 1.0f - interp;
					fcen12(j) += 1.0f - interp;

				} else {    // then its sort of a "12" but definitely not a "6"

					interp = std::min( ( tempdist - cen_dist12_pad_plus ) * cen_dist12_pad_hinv, 1.0 );

					if ( runlevel >= gush && interp < 0.0 ) {
						std::cout << "interp roundoff 3 " << SS( interp ) << std::endl;
					}

					fcen12_i += interp;
					fcen12(j) += interp;
				}
			}

		}                  // res j
		if ( cen12up_i > MAX_CENLIST ) {
			if ( runlevel > standard ) {
				std::cout << "WARNING:  residue " << i <<
				 " has more than MAX_CENLIST close neighbors" << std::endl;
				std::cout << "centroid coords: ";
				for ( int j = 1; j <= 3; ++j ) {
					std::cout << SS( centroid(j,i) );
				} std::cout << std::endl;
			}
			cen12up_i = MAX_CENLIST;
		}
		cen6(i) = cen6_i;
		fcen6(i) = fcen6_i;
		cen12(i) = cen12_i;
		fcen12(i) = fcen12_i;
		cen10(i) = cen10_i;
		fcen10(i) = fcen10_i;
		cen12up(i) = cen12up_i;
	}                     // res i
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_atom_type
///
/// @brief
/// ATOM TYPE SETUP FOR VDW SCORE
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_atom_type()
{

	using namespace misc;
	using namespace param;

	for ( int i = 1, e = total_residue * MAX_POS; i <= e; i += MAX_POS ) {
		atom_type(i) = 1;
		atom_type(i+1) = 2;
		atom_type(i+2) = 3;
		atom_type(i+3) = 4;
		atom_type(i+4) = 5;
	}
	for ( int i = 1; i <= total_residue; ++i ) {
		if ( residue1(i) == 'G' ) atom_type((i-1)*MAX_POS+3) = 0;
		atom_type_cen(i) = res(i) + MAX_POS;
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin vdw_compute_ij
///
/// @brief
///     this function is for debugging
///     it assumes that the cendist has already been updated
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  j - [in/out]? -
/// @param  vdw_score - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
vdw_compute_ij(
	int const i,
	int const j,
	float & vdw_score
)
{
	using namespace cenlist_ns;
	using namespace misc;
	using namespace param;
	using namespace vdw;
	using numeric::xyzVector_float;


	vdw_score = 0.0; // accumulate each j-pass individually

//--------
// as long as were at it, we now do a clash check
//  first do a coarse grain reality check on centroid seperations
//  and if it passes carefully check atom-atom distances
//--------

	float const cendist_ij = cendist(i,j); // look up prev calc dist
	if ( cendist_ij <= cen_dist_cutoff2 ) { // reality check

		int const type0 = atom_type_cen(i);
		xyzVector_float const cen_i( &centroid(1,i) ); // centroid(1-3,i)

		int const type3 = atom_type_cen(j);
		xyzVector_float const cen_j( &centroid(1,j) ); // centroid(1-3,j)

		// centroid-centroid clash check
		if ( count_pair_position(i,type0,res(i),j,type3,res(j)) ) {
			float const dist2 = atom_vdw(type0,type3) - cendist_ij;

			if ( dist2 > 0.0 ) vdw_score += ( dist2 * dist2 ) / atom_vdw(type0,type3); // clash score
		}

// As a prefilter we first compute the calpha-calpha distance.
// . If this is greater than 49 (7 squared) none of the backbone/cbeta can clash and we skip the distance backbone checks.
// . Additionally if this is greater than 60 then we can skip the centriod backbone checks.
//car these cutoffs defined in param.h

		float const alpha_alpha = // plus one means calpha
		 distance_squared( xyzVector_float( &Eposition(1,2,i) ), // Eposition(1-3,2,i)
		 xyzVector_float( &Eposition(1,2,j) ) ); // Eposition(1-3,2,j)

// minimum  centroid-pentamer radius

//---------
//  check i-centroid to j-atom  clash
//---------
		for ( int jatom = 1; jatom <= MAX_POS; ++jatom ) {
			int const type1 = Eatom_type(jatom,j);
			if ( count_pair_position(i,type0,res(i),j,type1,res(j))) {
				float const dist2 = atom_vdw(type0,type1) -
				 distance_squared( cen_i, // centroid(1-3,i)
				 xyzVector_float( &Eposition(1,jatom,j) ) ); // Eposition(1-3,jatom,j)

				if ( dist2 > 0.0 ) vdw_score += ( dist2 * dist2 ) / atom_vdw(type0,type1);
			}
		} // jatom

//---------
//  check j-centroid to i-atom  clash
//---------
		for ( int iatom = 1; iatom <= MAX_POS; ++iatom ) {
			int const type2 = Eatom_type(iatom,i); // vdw calc
			if ( count_pair_position(j,type3,res(j),i,type2,res(i)) ) {
				float const dist2 = atom_vdw(type3,type2) -
				 distance_squared( cen_j, // centroid(1-3,j)
				 xyzVector_float( &Eposition(1,iatom,i) ) ); // Eposition(1-3,iatom,i)

				if ( dist2 > 0.0 ) vdw_score += ( dist2 * dist2 ) / atom_vdw(type3,type2);
			}
		} // iatom

//------
//   check i-pentamer to j-pentamer
//------
		if ( alpha_alpha < ca_dist_cutoff2 ) { // clash radius CA-CA

			for ( int jatom = 1; jatom <= MAX_POS; ++jatom ) {
				int const type1 = Eatom_type(jatom,j);

				if ( type1 != 0 ) { // count_pair_position would catch but leave so we can avoid loop

					xyzVector_float const Epos_jatom( &Eposition(1,jatom,j) ); // Eposition(1-3,jatom,j)
					for ( int iatom = 1; iatom <= MAX_POS; ++iatom ) { // all backbone atoms in residue i
						int const type2 = Eatom_type(iatom,i);
						if ( count_pair_position( i, type2, res(i), j, type1, res(j) ) ) {
							float const dist2 = atom_vdw(type1,type2) -
							 distance_squared( Epos_jatom, // Eposition(1-3,jatom,j)
							 xyzVector_float( &Eposition(1,iatom,i) ) ); // Eposition(1-3,iatom,i)

							if ( dist2 > 0.0 ) vdw_score += ( dist2 * dist2 ) / atom_vdw(type1,type2);
						} // end count_pair check
					} // end iatom
				} // end type1 glycine check

			} // end jatom

		} // end alpha-alpha <49 check

	} // end cen_dist_cutoff2

//	vdw_score *= 0.8 //!!!!!!!!!!!!!!! NOT MULTIPLYING
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_default_atomvdw
///
/// @brief changes the default selection of the atom_vdw radii set
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/14/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_default_atomvdw( std::string const & setting )
{
	vdw::atomvdw_set = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_atomvdw
///
/// @brief initializes atom_vdw radii
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/14/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_atomvdw()
{
	using namespace vdw;
	static bool init = { false };

	if ( init ) return;
	stringafteroption( "atom_vdw_set", atomvdw_set, atomvdw_set );
	select_atomvdw( atomvdw_set );
	init = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin select_atomvdw
///
/// @brief
///
/// @detailed
///
/// @param  which - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
select_atomvdw( std::string const & which )
{

	int key;
	static int atomvdw_table = { 0 }; // none in arrays

	if ( which == "highres" ) {
		key = vdw::constants::highres;
	} else if ( which == "hybrid" ) {
		key = vdw::constants::hybrid;
	} else {
		if ( which != "default" ) {
			std::cout << "WARNING:: undefined setting for atomvdw set: " << which
			 << std::endl;
			std::cout << "          Using default setting" << std::endl;
		}
		key = vdw::constants::default_type;
	}

	if ( key == atomvdw_table ) return; // already in arrays
	atomvdw_table = key;
//KMa phospho_ser 29 dna+aa +5
	for ( int i = 1; i <= 34; ++i ) {
		for ( int j = i; j <= 34; ++j ) {
			float tmp = vdw::constants::atom_vdw_tables(i,j,key);
			if (tmp !=  vdw::constants::atom_vdw_tables(j,i,key)) {
				std::cout << "WARNING:: atom_vdw_table is not symmetric!! "
				 << SS(key) << SS(i) << SS(j) << SS(tmp) <<  SS(vdw::constants::atom_vdw_tables(j,i,key))
				 << std::endl;
			}
			vdw::atom_vdw(i,j) = tmp;
			vdw::atom_vdw(j,i) = tmp;
		}
	}
}


bool get_ss_vdw()
{
	static bool init = {false};
	static bool ss_vdw;

	if (!init) {
		ss_vdw = truefalseoption("ss_vdw");
		init = true;
	}

	return ss_vdw;
}

int get_ss_vdw_neighborspacer()
{
	static bool init = {false};
	static int neighbor_spacer;

	if (!init) {
		intafteroption("ss_vdw", 4, neighbor_spacer);
		init = true;
	}

	return neighbor_spacer;
}

namespace vdw {

	std::string atomvdw_set = "default";
	FArray2D_float atom_vdw = FArray2D_float( 34, 34 );

namespace constants {

	int const default_type = 1;
	int const highres = 2;
	int const hybrid = 3;

	FArray3D_float const atom_vdw_tables( 34, 34, 3, initializers::atom_vdw_tables_initializer ); // how close can atoms come before they bump?

}

namespace initializers {

void
atom_vdw_tables_initializer(
	FArray3D_float & atom_vdw_tables
)
{
	using namespace vdw::constants;
	// section 1a
	//------------------------------------------------------------------
	//km 4/09/02 - TABLE 1
	// Default array for vdw_compute- distances are taken as 25th closest
	// approach of atoms in a set of known strctures- Simons et. al. 1999
	// Proteins.
	atom_vdw_tables( 1, 1, default_type) = 5.518;
	atom_vdw_tables( 1, 2, default_type) = 5.803;
	atom_vdw_tables( 1, 3, default_type) = 9.175;
	atom_vdw_tables( 1, 4, default_type) = 7.508;
	//atom_vdw_tables( 1, 4, default_type) = 10.923;
	//atom_vdw_tables( 4, 1, default_type) = 7.508;
	atom_vdw_tables( 1, 5, default_type) = 4.584;
	atom_vdw_tables( 1, 6, default_type) = 6.641;
	atom_vdw_tables( 1, 7, default_type) = 7.458;
	atom_vdw_tables( 1, 8, default_type) = 6.693;
	atom_vdw_tables( 1, 9, default_type) = 5.641;
	atom_vdw_tables( 1,10, default_type) = 5.373;
	atom_vdw_tables( 1,11, default_type) = 5.031;
	atom_vdw_tables( 1,12, default_type) = 6.467;
	atom_vdw_tables( 1,13, default_type) = 9.499;
	atom_vdw_tables( 1,14, default_type) = 4.991;
	atom_vdw_tables( 1,15, default_type) = 7.913;
	atom_vdw_tables( 1,16, default_type) = 5.988;
	atom_vdw_tables( 1,17, default_type) = 6.849;
	atom_vdw_tables( 1,18, default_type) = 9.272;
	atom_vdw_tables( 1,19, default_type) = 5.774;
	atom_vdw_tables( 1,20, default_type) = 3.968;
	atom_vdw_tables( 1,21, default_type) = 7.829;
	atom_vdw_tables( 1,22, default_type) = 7.563;
	atom_vdw_tables( 1,23, default_type) = 9.321;
	atom_vdw_tables( 1,24, default_type) = 4.618;
	atom_vdw_tables( 1,25, default_type) = 3.721;
	atom_vdw_tables( 1,26, default_type) = 0.000;
	atom_vdw_tables( 1,27, default_type) = 0.000;
	atom_vdw_tables( 1,28, default_type) = 0.000;
	atom_vdw_tables( 1,29, default_type) = 0.000;
	atom_vdw_tables( 1,30, default_type) = 0.000;
	atom_vdw_tables( 1,31, default_type) = 0.000;
	atom_vdw_tables( 1,32, default_type) = 0.000;
	atom_vdw_tables( 1,33, default_type) = 0.000;
	atom_vdw_tables( 2, 1, default_type) = 5.803;
	atom_vdw_tables( 2, 2, default_type) = 7.670; // for cis-peptide
	atom_vdw_tables( 2, 3, default_type) = 9.797;
	atom_vdw_tables( 2, 4, default_type) = 5.871;
	atom_vdw_tables( 2, 5, default_type) = 6.744;
	atom_vdw_tables( 2, 6, default_type) = 10.017;
	atom_vdw_tables( 2, 7, default_type) = 6.975;
	atom_vdw_tables( 2, 8, default_type) = 7.236;
	atom_vdw_tables( 2, 9, default_type) = 5.452;
	atom_vdw_tables( 2,10, default_type) = 5.607;
	atom_vdw_tables( 2,11, default_type) = 7.676;
	atom_vdw_tables( 2,12, default_type) = 5.910;
	atom_vdw_tables( 2,13, default_type) = 9.260;
	atom_vdw_tables( 2,14, default_type) = 4.244;
	atom_vdw_tables( 2,15, default_type) = 6.396;
	atom_vdw_tables( 2,16, default_type) = 4.757;
	atom_vdw_tables( 2,17, default_type) = 7.333;
	atom_vdw_tables( 2,18, default_type) = 8.649;
	atom_vdw_tables( 2,19, default_type) = 5.968;
	atom_vdw_tables( 2,20, default_type) = 3.126;
	atom_vdw_tables( 2,21, default_type) = 8.352;
	atom_vdw_tables( 2,22, default_type) = 8.352;
	atom_vdw_tables( 2,23, default_type) = 10.465;
	atom_vdw_tables( 2,24, default_type) = 4.588;
	atom_vdw_tables( 2,25, default_type) = 3.557;
	atom_vdw_tables( 2,26, default_type) = 0.000;
	atom_vdw_tables( 2,27, default_type) = 0.000;
	atom_vdw_tables( 2,28, default_type) = 0.000;
	atom_vdw_tables( 2,29, default_type) = 0.000;
	atom_vdw_tables( 2,30, default_type) = 0.000;
	atom_vdw_tables( 2,31, default_type) = 0.000;
	atom_vdw_tables( 2,32, default_type) = 0.000;
	atom_vdw_tables( 2,33, default_type) = 0.000;
	atom_vdw_tables( 3, 1, default_type) = 9.175;
	atom_vdw_tables( 3, 2, default_type) = 9.797;
	atom_vdw_tables( 3, 3, default_type) = 9.653;
	atom_vdw_tables( 3, 4, default_type) = 9.778;
	atom_vdw_tables( 3, 5, default_type) = 6.589;
	atom_vdw_tables( 3, 6, default_type) = 8.462;
	atom_vdw_tables( 3, 7, default_type) = 6.335;
	atom_vdw_tables( 3, 8, default_type) = 6.776;
	atom_vdw_tables( 3, 9, default_type) = 4.533;
	atom_vdw_tables( 3,10, default_type) = 4.244;
	atom_vdw_tables( 3,11, default_type) = 8.797;
	atom_vdw_tables( 3,12, default_type) = 5.712;
	atom_vdw_tables( 3,13, default_type) = 8.744;
	atom_vdw_tables( 3,14, default_type) = 3.640;
	atom_vdw_tables( 3,15, default_type) = 5.499;
	atom_vdw_tables( 3,16, default_type) = 4.713;
	atom_vdw_tables( 3,17, default_type) = 6.646;
	atom_vdw_tables( 3,18, default_type) = 8.791;
	atom_vdw_tables( 3,19, default_type) = 4.580;
	atom_vdw_tables( 3,20, default_type) = 1.943;
	atom_vdw_tables( 3,21, default_type) = 7.596;
	atom_vdw_tables( 3,22, default_type) = 8.964;
	atom_vdw_tables( 3,23, default_type) = 8.928;
	atom_vdw_tables( 3,24, default_type) = 4.182;
	atom_vdw_tables( 3,25, default_type) = 3.877;
	atom_vdw_tables( 3,26, default_type) = 0.000;
	atom_vdw_tables( 3,27, default_type) = 0.000;
	atom_vdw_tables( 3,28, default_type) = 0.000;
	atom_vdw_tables( 3,29, default_type) = 0.000;
	atom_vdw_tables( 3,30, default_type) = 0.000;
	atom_vdw_tables( 3,31, default_type) = 0.000;
	atom_vdw_tables( 3,32, default_type) = 0.000;
	atom_vdw_tables( 3,33, default_type) = 0.000;
	atom_vdw_tables( 4, 1, default_type) = 7.508;
	atom_vdw_tables( 4, 2, default_type) = 5.871;
	atom_vdw_tables( 4, 3, default_type) = 9.778;
	atom_vdw_tables( 4, 4, default_type) = 6.880;
	atom_vdw_tables( 4, 5, default_type) = 6.600;
	atom_vdw_tables( 4, 6, default_type) = 7.723;
	atom_vdw_tables( 4, 7, default_type) = 7.007;
	atom_vdw_tables( 4, 8, default_type) = 8.364;
	atom_vdw_tables( 4, 9, default_type) = 6.250;
	atom_vdw_tables( 4,10, default_type) = 5.308;
	atom_vdw_tables( 4,11, default_type) = 4.862;
	atom_vdw_tables( 4,12, default_type) = 5.546;
	atom_vdw_tables( 4,13, default_type) = 9.370;
	atom_vdw_tables( 4,14, default_type) = 4.310;
	atom_vdw_tables( 4,15, default_type) = 7.645;
	atom_vdw_tables( 4,16, default_type) = 5.842;
	atom_vdw_tables( 4,17, default_type) = 6.136;
	atom_vdw_tables( 4,18, default_type) = 4.016;
	atom_vdw_tables( 4,19, default_type) = 5.636;
	atom_vdw_tables( 4,20, default_type) = 3.201;
	atom_vdw_tables( 4,21, default_type) = 7.885;
	atom_vdw_tables( 4,22, default_type) = 9.456;
	atom_vdw_tables( 4,23, default_type) = 9.413;
	atom_vdw_tables( 4,24, default_type) = 4.452;
	atom_vdw_tables( 4,25, default_type) = 4.439;
	atom_vdw_tables( 4,26, default_type) = 0.000;
	atom_vdw_tables( 4,27, default_type) = 0.000;
	atom_vdw_tables( 4,28, default_type) = 0.000;
	atom_vdw_tables( 4,29, default_type) = 0.000;
	atom_vdw_tables( 4,30, default_type) = 0.000;
	atom_vdw_tables( 4,31, default_type) = 0.000;
	atom_vdw_tables( 4,32, default_type) = 0.000;
	atom_vdw_tables( 4,33, default_type) = 0.000;
	atom_vdw_tables( 5, 1, default_type) = 4.584;
	atom_vdw_tables( 5, 2, default_type) = 6.744;
	atom_vdw_tables( 5, 3, default_type) = 6.589;
	atom_vdw_tables( 5, 4, default_type) = 6.600;
	atom_vdw_tables( 5, 5, default_type) = 4.889;
	atom_vdw_tables( 5, 6, default_type) = 5.765;
	atom_vdw_tables( 5, 7, default_type) = 4.818;
	atom_vdw_tables( 5, 8, default_type) = 4.946;
	atom_vdw_tables( 5, 9, default_type) = 3.254;
	atom_vdw_tables( 5,10, default_type) = 2.986;
	atom_vdw_tables( 5,11, default_type) = 5.221;
	atom_vdw_tables( 5,12, default_type) = 3.349;
	atom_vdw_tables( 5,13, default_type) = 7.023;
	atom_vdw_tables( 5,14, default_type) = 2.647;
	atom_vdw_tables( 5,15, default_type) = 4.020;
	atom_vdw_tables( 5,16, default_type) = 3.542;
	atom_vdw_tables( 5,17, default_type) = 2.500;
	atom_vdw_tables( 5,18, default_type) = 4.752;
	atom_vdw_tables( 5,19, default_type) = 3.565;
	atom_vdw_tables( 5,20, default_type) = 1.814;
	atom_vdw_tables( 5,21, default_type) = 4.991;
	atom_vdw_tables( 5,22, default_type) = 5.612;
	atom_vdw_tables( 5,23, default_type) = 5.939;
	atom_vdw_tables( 5,24, default_type) = 3.262;
	atom_vdw_tables( 5,25, default_type) = 2.789;
	atom_vdw_tables( 5,26, default_type) = 0.000;
	atom_vdw_tables( 5,27, default_type) = 0.000;
	atom_vdw_tables( 5,28, default_type) = 0.000;
	atom_vdw_tables( 5,29, default_type) = 0.000;
	atom_vdw_tables( 5,30, default_type) = 0.000;
	atom_vdw_tables( 5,31, default_type) = 0.000;
	atom_vdw_tables( 5,32, default_type) = 0.000;
	atom_vdw_tables( 5,33, default_type) = 0.000;
	atom_vdw_tables( 6, 1, default_type) = 6.641;
	atom_vdw_tables( 6, 2, default_type) = 10.017;
	atom_vdw_tables( 6, 3, default_type) = 8.462;
	atom_vdw_tables( 6, 4, default_type) = 7.723;
	atom_vdw_tables( 6, 5, default_type) = 5.765;
	atom_vdw_tables( 6, 6, default_type) = 10.478;
	atom_vdw_tables( 6, 7, default_type) = 8.708;
	atom_vdw_tables( 6, 8, default_type) = 7.896;
	atom_vdw_tables( 6, 9, default_type) = 5.645;
	atom_vdw_tables( 6,10, default_type) = 5.546;
	atom_vdw_tables( 6,11, default_type) = 10.811;
	atom_vdw_tables( 6,12, default_type) = 6.513;
	atom_vdw_tables( 6,13, default_type) = 9.734;
	atom_vdw_tables( 6,14, default_type) = 5.664;
	atom_vdw_tables( 6,15, default_type) = 7.350;
	atom_vdw_tables( 6,16, default_type) = 6.012;
	atom_vdw_tables( 6,17, default_type) = 8.289;
	atom_vdw_tables( 6,18, default_type) = 10.272;
	atom_vdw_tables( 6,19, default_type) = 6.315;
	atom_vdw_tables( 6,20, default_type) = 5.443;
	atom_vdw_tables( 6,21, default_type) = 8.940;
	atom_vdw_tables( 6,22, default_type) = 9.530;
	atom_vdw_tables( 6,23, default_type) = 9.357;
	atom_vdw_tables( 6,24, default_type) = 5.295;
	atom_vdw_tables( 6,25, default_type) = 5.272;
	atom_vdw_tables( 6,26, default_type) = 0.000;
	atom_vdw_tables( 6,27, default_type) = 0.000;
	atom_vdw_tables( 6,28, default_type) = 0.000;
	atom_vdw_tables( 6,29, default_type) = 0.000;
	atom_vdw_tables( 6,30, default_type) = 0.000;
	atom_vdw_tables( 6,31, default_type) = 0.000;
	atom_vdw_tables( 6,32, default_type) = 0.000;
	atom_vdw_tables( 6,33, default_type) = 0.000;
	atom_vdw_tables( 7, 1, default_type) = 7.458;
	atom_vdw_tables( 7, 2, default_type) = 6.975;
	atom_vdw_tables( 7, 3, default_type) = 6.335;
	atom_vdw_tables( 7, 4, default_type) = 7.007;
	atom_vdw_tables( 7, 5, default_type) = 4.818;
	atom_vdw_tables( 7, 6, default_type) = 8.708;
	atom_vdw_tables( 7, 7, default_type) = 3.791;
	atom_vdw_tables( 7, 8, default_type) = 7.678;
	atom_vdw_tables( 7, 9, default_type) = 8.225;
	atom_vdw_tables( 7,10, default_type) = 5.518;
	atom_vdw_tables( 7,11, default_type) = 9.419;
	atom_vdw_tables( 7,12, default_type) = 8.970;
	atom_vdw_tables( 7,13, default_type) = 10.368;
	atom_vdw_tables( 7,14, default_type) = 9.406;
	atom_vdw_tables( 7,15, default_type) = 7.339;
	atom_vdw_tables( 7,16, default_type) = 8.105;
	atom_vdw_tables( 7,17, default_type) = 9.272;
	atom_vdw_tables( 7,18, default_type) = 11.710;
	atom_vdw_tables( 7,19, default_type) = 9.321;
	atom_vdw_tables( 7,20, default_type) = 7.497;
	atom_vdw_tables( 7,21, default_type) = 8.809;
	atom_vdw_tables( 7,22, default_type) = 8.744;
	atom_vdw_tables( 7,23, default_type) = 10.259;
	atom_vdw_tables( 7,24, default_type) = 6.482;
	atom_vdw_tables( 7,25, default_type) = 4.516;
	atom_vdw_tables( 7,26, default_type) = 0.000;
	atom_vdw_tables( 7,27, default_type) = 0.000;
	atom_vdw_tables( 7,28, default_type) = 0.000;
	atom_vdw_tables( 7,29, default_type) = 0.000;
	atom_vdw_tables( 7,30, default_type) = 0.000;
	atom_vdw_tables( 7,31, default_type) = 0.000;
	atom_vdw_tables( 7,32, default_type) = 0.000;
	atom_vdw_tables( 7,33, default_type) = 0.000;
	atom_vdw_tables( 8, 1, default_type) = 6.693;
	atom_vdw_tables( 8, 2, default_type) = 7.236;
	atom_vdw_tables( 8, 3, default_type) = 6.776;
	atom_vdw_tables( 8, 4, default_type) = 8.364;
	atom_vdw_tables( 8, 5, default_type) = 4.946;
	atom_vdw_tables( 8, 6, default_type) = 7.896;
	atom_vdw_tables( 8, 7, default_type) = 7.678;
	atom_vdw_tables( 8, 8, default_type) = 8.791;
	atom_vdw_tables( 8, 9, default_type) = 6.275;
	atom_vdw_tables( 8,10, default_type) = 4.554;
	atom_vdw_tables( 8,11, default_type) = 8.509;
	atom_vdw_tables( 8,12, default_type) = 6.022;
	atom_vdw_tables( 8,13, default_type) = 7.684;
	atom_vdw_tables( 8,14, default_type) = 5.130;
	atom_vdw_tables( 8,15, default_type) = 6.155;
	atom_vdw_tables( 8,16, default_type) = 6.290;
	atom_vdw_tables( 8,17, default_type) = 6.854;
	atom_vdw_tables( 8,18, default_type) = 9.511;
	atom_vdw_tables( 8,19, default_type) = 5.185;
	atom_vdw_tables( 8,20, default_type) = 3.806;
	atom_vdw_tables( 8,21, default_type) = 6.559;
	atom_vdw_tables( 8,22, default_type) = 7.166;
	atom_vdw_tables( 8,23, default_type) = 9.205;
	atom_vdw_tables( 8,24, default_type) = 5.240;
	atom_vdw_tables( 8,25, default_type) = 5.669;
	atom_vdw_tables( 8,26, default_type) = 0.000;
	atom_vdw_tables( 8,27, default_type) = 0.000;
	atom_vdw_tables( 8,28, default_type) = 0.000;
	atom_vdw_tables( 8,29, default_type) = 0.000;
	atom_vdw_tables( 8,30, default_type) = 0.000;
	atom_vdw_tables( 8,31, default_type) = 0.000;
	atom_vdw_tables( 8,32, default_type) = 0.000;
	atom_vdw_tables( 8,33, default_type) = 0.000;
	atom_vdw_tables( 9, 1, default_type) = 5.641;
	atom_vdw_tables( 9, 2, default_type) = 5.452;
	atom_vdw_tables( 9, 3, default_type) = 4.533;
	atom_vdw_tables( 9, 4, default_type) = 6.250;
	atom_vdw_tables( 9, 5, default_type) = 3.254;
	atom_vdw_tables( 9, 6, default_type) = 5.645;
	atom_vdw_tables( 9, 7, default_type) = 8.225;
	atom_vdw_tables( 9, 8, default_type) = 6.275;
	atom_vdw_tables( 9, 9, default_type) = 5.698;
	atom_vdw_tables( 9,10, default_type) = 3.497;
	atom_vdw_tables( 9,11, default_type) = 5.818;
	atom_vdw_tables( 9,12, default_type) = 2.907;
	atom_vdw_tables( 9,13, default_type) = 7.437;
	atom_vdw_tables( 9,14, default_type) = 3.404;
	atom_vdw_tables( 9,15, default_type) = 5.929;
	atom_vdw_tables( 9,16, default_type) = 7.458;
	atom_vdw_tables( 9,17, default_type) = 5.939;
	atom_vdw_tables( 9,18, default_type) = 7.590;
	atom_vdw_tables( 9,19, default_type) = 6.682;
	atom_vdw_tables( 9,20, default_type) = 2.641;
	atom_vdw_tables( 9,21, default_type) = 4.713;
	atom_vdw_tables( 9,22, default_type) = 7.204;
	atom_vdw_tables( 9,23, default_type) = 5.626;
	atom_vdw_tables( 9,24, default_type) = 6.265;
	atom_vdw_tables( 9,25, default_type) = 3.419;
	atom_vdw_tables( 9,26, default_type) = 0.000;
	atom_vdw_tables( 9,27, default_type) = 0.000;
	atom_vdw_tables( 9,28, default_type) = 0.000;
	atom_vdw_tables( 9,29, default_type) = 0.000;
	atom_vdw_tables( 9,30, default_type) = 0.000;
	atom_vdw_tables( 9,31, default_type) = 0.000;
	atom_vdw_tables( 9,32, default_type) = 0.000;
	atom_vdw_tables( 9,33, default_type) = 0.000;
	atom_vdw_tables(10, 1, default_type) = 5.373;
	atom_vdw_tables(10, 2, default_type) = 5.607;
	atom_vdw_tables(10, 3, default_type) = 4.244;
	atom_vdw_tables(10, 4, default_type) = 5.308;
	atom_vdw_tables(10, 5, default_type) = 2.986;
	atom_vdw_tables(10, 6, default_type) = 5.546;
	atom_vdw_tables(10, 7, default_type) = 5.518;
	atom_vdw_tables(10, 8, default_type) = 4.554;
	atom_vdw_tables(10, 9, default_type) = 3.497;
	atom_vdw_tables(10,10, default_type) = 3.309;
	atom_vdw_tables(10,11, default_type) = 5.988;
	atom_vdw_tables(10,12, default_type) = 4.260;
	atom_vdw_tables(10,13, default_type) = 5.522;
	atom_vdw_tables(10,14, default_type) = 3.764;
	atom_vdw_tables(10,15, default_type) = 2.631;
	atom_vdw_tables(10,16, default_type) = 4.339;
	atom_vdw_tables(10,17, default_type) = 5.281;
	atom_vdw_tables(10,18, default_type) = 4.928;
	atom_vdw_tables(10,19, default_type) = 2.722;
	atom_vdw_tables(10,20, default_type) = 3.197;
	atom_vdw_tables(10,21, default_type) = 5.240;
	atom_vdw_tables(10,22, default_type) = 6.651;
	atom_vdw_tables(10,23, default_type) = 4.661;
	atom_vdw_tables(10,24, default_type) = 4.235;
	atom_vdw_tables(10,25, default_type) = 2.076;
	atom_vdw_tables(10,26, default_type) = 0.000;
	atom_vdw_tables(10,27, default_type) = 0.000;
	atom_vdw_tables(10,28, default_type) = 0.000;
	atom_vdw_tables(10,29, default_type) = 0.000;
	atom_vdw_tables(10,30, default_type) = 0.000;
	atom_vdw_tables(10,31, default_type) = 0.000;
	atom_vdw_tables(10,32, default_type) = 0.000;
	atom_vdw_tables(10,33, default_type) = 0.000;
	atom_vdw_tables(11, 1, default_type) = 5.031;
	atom_vdw_tables(11, 2, default_type) = 7.676;
	atom_vdw_tables(11, 3, default_type) = 8.797;
	atom_vdw_tables(11, 4, default_type) = 4.862;
	atom_vdw_tables(11, 5, default_type) = 5.221;
	atom_vdw_tables(11, 6, default_type) = 10.811;
	atom_vdw_tables(11, 7, default_type) = 9.419;
	atom_vdw_tables(11, 8, default_type) = 8.509;
	atom_vdw_tables(11, 9, default_type) = 5.818;
	atom_vdw_tables(11,10, default_type) = 5.988;
	atom_vdw_tables(11,11, default_type) = 10.936;
	atom_vdw_tables(11,12, default_type) = 7.740;
	atom_vdw_tables(11,13, default_type) = 9.791;
	atom_vdw_tables(11,14, default_type) = 5.013;
	atom_vdw_tables(11,15, default_type) = 7.981;
	atom_vdw_tables(11,16, default_type) = 8.134;
	atom_vdw_tables(11,17, default_type) = 8.934;
	atom_vdw_tables(11,18, default_type) = 11.089;
	atom_vdw_tables(11,19, default_type) = 7.268;
	atom_vdw_tables(11,20, default_type) = 4.364;
	atom_vdw_tables(11,21, default_type) = 9.066;
	atom_vdw_tables(11,22, default_type) = 9.272;
	atom_vdw_tables(11,23, default_type) = 10.969;
	atom_vdw_tables(11,24, default_type) = 5.532;
	atom_vdw_tables(11,25, default_type) = 5.636;
	atom_vdw_tables(11,26, default_type) = 0.000;
	atom_vdw_tables(11,27, default_type) = 0.000;
	atom_vdw_tables(11,28, default_type) = 0.000;
	atom_vdw_tables(11,29, default_type) = 0.000;
	atom_vdw_tables(11,30, default_type) = 0.000;
	atom_vdw_tables(11,31, default_type) = 0.000;
	atom_vdw_tables(11,32, default_type) = 0.000;
	atom_vdw_tables(11,33, default_type) = 0.000;

	// section 1b
	atom_vdw_tables(12, 1, default_type) = 6.467;
	atom_vdw_tables(12, 2, default_type) = 5.910;
	atom_vdw_tables(12, 3, default_type) = 5.712;
	atom_vdw_tables(12, 4, default_type) = 5.546;
	atom_vdw_tables(12, 5, default_type) = 3.349;
	atom_vdw_tables(12, 6, default_type) = 6.513;
	atom_vdw_tables(12, 7, default_type) = 8.970;
	atom_vdw_tables(12, 8, default_type) = 6.022;
	atom_vdw_tables(12, 9, default_type) = 2.907;
	atom_vdw_tables(12,10, default_type) = 4.260;
	atom_vdw_tables(12,11, default_type) = 7.740;
	atom_vdw_tables(12,12, default_type) = 7.241;
	atom_vdw_tables(12,13, default_type) = 8.105;
	atom_vdw_tables(12,14, default_type) = 4.924;
	atom_vdw_tables(12,15, default_type) = 6.140;
	atom_vdw_tables(12,16, default_type) = 6.350;
	atom_vdw_tables(12,17, default_type) = 5.359;
	atom_vdw_tables(12,18, default_type) = 7.290;
	atom_vdw_tables(12,19, default_type) = 5.958;
	atom_vdw_tables(12,20, default_type) = 3.190;
	atom_vdw_tables(12,21, default_type) = 6.205;
	atom_vdw_tables(12,22, default_type) = 8.020;
	atom_vdw_tables(12,23, default_type) = 6.391;
	atom_vdw_tables(12,24, default_type) = 5.139;
	atom_vdw_tables(12,25, default_type) = 4.406;
	atom_vdw_tables(12,26, default_type) = 0.000;
	atom_vdw_tables(12,27, default_type) = 0.000;
	atom_vdw_tables(12,28, default_type) = 0.000;
	atom_vdw_tables(12,29, default_type) = 0.000;
	atom_vdw_tables(12,30, default_type) = 0.000;
	atom_vdw_tables(12,31, default_type) = 0.000;
	atom_vdw_tables(12,32, default_type) = 0.000;
	atom_vdw_tables(12,33, default_type) = 0.000;
	atom_vdw_tables(13, 1, default_type) = 9.499;
	atom_vdw_tables(13, 2, default_type) = 9.260;
	atom_vdw_tables(13, 3, default_type) = 8.744;
	atom_vdw_tables(13, 4, default_type) = 9.370;
	atom_vdw_tables(13, 5, default_type) = 7.023;
	atom_vdw_tables(13, 6, default_type) = 9.734;
	atom_vdw_tables(13, 7, default_type) = 10.368;
	atom_vdw_tables(13, 8, default_type) = 7.684;
	atom_vdw_tables(13, 9, default_type) = 7.437;
	atom_vdw_tables(13,10, default_type) = 5.522;
	atom_vdw_tables(13,11, default_type) = 9.791;
	atom_vdw_tables(13,12, default_type) = 8.105;
	atom_vdw_tables(13,13, default_type) = 11.129;
	atom_vdw_tables(13,14, default_type) = 6.140;
	atom_vdw_tables(13,15, default_type) = 7.447;
	atom_vdw_tables(13,16, default_type) = 7.458;
	atom_vdw_tables(13,17, default_type) = 8.260;
	atom_vdw_tables(13,18, default_type) = 11.710;
	atom_vdw_tables(13,19, default_type) = 7.585;
	atom_vdw_tables(13,20, default_type) = 5.688;
	atom_vdw_tables(13,21, default_type) = 9.986;
	atom_vdw_tables(13,22, default_type) = 9.828;
	atom_vdw_tables(13,23, default_type) = 9.598;
	atom_vdw_tables(13,24, default_type) = 6.523;
	atom_vdw_tables(13,25, default_type) = 5.076;
	atom_vdw_tables(13,26, default_type) = 0.000;
	atom_vdw_tables(13,27, default_type) = 0.000;
	atom_vdw_tables(13,28, default_type) = 0.000;
	atom_vdw_tables(13,29, default_type) = 0.000;
	atom_vdw_tables(13,30, default_type) = 0.000;
	atom_vdw_tables(13,31, default_type) = 0.000;
	atom_vdw_tables(13,32, default_type) = 0.000;
	atom_vdw_tables(13,33, default_type) = 0.000;
	atom_vdw_tables(14, 1, default_type) = 4.991;
	atom_vdw_tables(14, 2, default_type) = 4.244;
	atom_vdw_tables(14, 3, default_type) = 3.640;
	atom_vdw_tables(14, 4, default_type) = 4.310;
	atom_vdw_tables(14, 5, default_type) = 2.647;
	atom_vdw_tables(14, 6, default_type) = 5.664;
	atom_vdw_tables(14, 7, default_type) = 9.406;
	atom_vdw_tables(14, 8, default_type) = 5.130;
	atom_vdw_tables(14, 9, default_type) = 3.404;
	atom_vdw_tables(14,10, default_type) = 3.764;
	atom_vdw_tables(14,11, default_type) = 5.013;
	atom_vdw_tables(14,12, default_type) = 4.924;
	atom_vdw_tables(14,13, default_type) = 6.140;
	atom_vdw_tables(14,14, default_type) = 6.411;
	atom_vdw_tables(14,15, default_type) = 4.866;
	atom_vdw_tables(14,16, default_type) = 4.889;
	atom_vdw_tables(14,17, default_type) = 4.627;
	atom_vdw_tables(14,18, default_type) = 6.734;
	atom_vdw_tables(14,19, default_type) = 4.884;
	atom_vdw_tables(14,20, default_type) = 6.933;
	atom_vdw_tables(14,21, default_type) = 5.664;
	atom_vdw_tables(14,22, default_type) = 5.929;
	atom_vdw_tables(14,23, default_type) = 7.023;
	atom_vdw_tables(14,24, default_type) = 3.045;
	atom_vdw_tables(14,25, default_type) = 2.503;
	atom_vdw_tables(14,26, default_type) = 0.000;
	atom_vdw_tables(14,27, default_type) = 0.000;
	atom_vdw_tables(14,28, default_type) = 0.000;
	atom_vdw_tables(14,29, default_type) = 0.000;
	atom_vdw_tables(14,30, default_type) = 0.000;
	atom_vdw_tables(14,31, default_type) = 0.000;
	atom_vdw_tables(14,32, default_type) = 0.000;
	atom_vdw_tables(14,33, default_type) = 0.000;
	atom_vdw_tables(15, 1, default_type) = 7.913;
	atom_vdw_tables(15, 2, default_type) = 6.396;
	atom_vdw_tables(15, 3, default_type) = 5.499;
	atom_vdw_tables(15, 4, default_type) = 7.645;
	atom_vdw_tables(15, 5, default_type) = 4.020;
	atom_vdw_tables(15, 6, default_type) = 7.350;
	atom_vdw_tables(15, 7, default_type) = 7.339;
	atom_vdw_tables(15, 8, default_type) = 6.155;
	atom_vdw_tables(15, 9, default_type) = 5.929;
	atom_vdw_tables(15,10, default_type) = 2.631;
	atom_vdw_tables(15,11, default_type) = 7.981;
	atom_vdw_tables(15,12, default_type) = 6.140;
	atom_vdw_tables(15,13, default_type) = 7.447;
	atom_vdw_tables(15,14, default_type) = 4.866;
	atom_vdw_tables(15,15, default_type) = 6.870;
	atom_vdw_tables(15,16, default_type) = 4.601;
	atom_vdw_tables(15,17, default_type) = 7.044;
	atom_vdw_tables(15,18, default_type) = 8.644;
	atom_vdw_tables(15,19, default_type) = 5.789;
	atom_vdw_tables(15,20, default_type) = 4.356;
	atom_vdw_tables(15,21, default_type) = 6.610;
	atom_vdw_tables(15,22, default_type) = 6.896;
	atom_vdw_tables(15,23, default_type) = 8.009;
	atom_vdw_tables(15,24, default_type) = 3.378;
	atom_vdw_tables(15,25, default_type) = 2.696;
	atom_vdw_tables(15,26, default_type) = 0.000;
	atom_vdw_tables(15,27, default_type) = 0.000;
	atom_vdw_tables(15,28, default_type) = 0.000;
	atom_vdw_tables(15,29, default_type) = 0.000;
	atom_vdw_tables(15,30, default_type) = 0.000;
	atom_vdw_tables(15,31, default_type) = 0.000;
	atom_vdw_tables(15,32, default_type) = 0.000;
	atom_vdw_tables(15,33, default_type) = 0.000;
	atom_vdw_tables(16, 1, default_type) = 5.988;
	atom_vdw_tables(16, 2, default_type) = 4.757;
	atom_vdw_tables(16, 3, default_type) = 4.713;
	atom_vdw_tables(16, 4, default_type) = 5.842;
	atom_vdw_tables(16, 5, default_type) = 3.542;
	atom_vdw_tables(16, 6, default_type) = 6.012;
	atom_vdw_tables(16, 7, default_type) = 8.105;
	atom_vdw_tables(16, 8, default_type) = 6.290;
	atom_vdw_tables(16, 9, default_type) = 7.458;
	atom_vdw_tables(16,10, default_type) = 4.339;
	atom_vdw_tables(16,11, default_type) = 8.134;
	atom_vdw_tables(16,12, default_type) = 6.350;
	atom_vdw_tables(16,13, default_type) = 7.458;
	atom_vdw_tables(16,14, default_type) = 4.889;
	atom_vdw_tables(16,15, default_type) = 4.601;
	atom_vdw_tables(16,16, default_type) = 7.475;
	atom_vdw_tables(16,17, default_type) = 5.660;
	atom_vdw_tables(16,18, default_type) = 9.809;
	atom_vdw_tables(16,19, default_type) = 8.260;
	atom_vdw_tables(16,20, default_type) = 3.629;
	atom_vdw_tables(16,21, default_type) = 7.773;
	atom_vdw_tables(16,22, default_type) = 8.532;
	atom_vdw_tables(16,23, default_type) = 6.096;
	atom_vdw_tables(16,24, default_type) = 5.434;
	atom_vdw_tables(16,25, default_type) = 3.378;
	atom_vdw_tables(16,26, default_type) = 0.000;
	atom_vdw_tables(16,27, default_type) = 0.000;
	atom_vdw_tables(16,28, default_type) = 0.000;
	atom_vdw_tables(16,29, default_type) = 0.000;
	atom_vdw_tables(16,30, default_type) = 0.000;
	atom_vdw_tables(16,31, default_type) = 0.000;
	atom_vdw_tables(16,32, default_type) = 0.000;
	atom_vdw_tables(16,33, default_type) = 0.000;
	atom_vdw_tables(17, 1, default_type) = 6.849;
	atom_vdw_tables(17, 2, default_type) = 7.333;
	atom_vdw_tables(17, 3, default_type) = 6.646;
	atom_vdw_tables(17, 4, default_type) = 6.136;
	atom_vdw_tables(17, 5, default_type) = 2.500;
	atom_vdw_tables(17, 6, default_type) = 8.289;
	atom_vdw_tables(17, 7, default_type) = 9.272;
	atom_vdw_tables(17, 8, default_type) = 6.854;
	atom_vdw_tables(17, 9, default_type) = 5.939;
	atom_vdw_tables(17,10, default_type) = 5.281;
	atom_vdw_tables(17,11, default_type) = 8.934;
	atom_vdw_tables(17,12, default_type) = 5.359;
	atom_vdw_tables(17,13, default_type) = 8.260;
	atom_vdw_tables(17,14, default_type) = 4.627;
	atom_vdw_tables(17,15, default_type) = 7.044;
	atom_vdw_tables(17,16, default_type) = 5.660;
	atom_vdw_tables(17,17, default_type) = 7.464;
	atom_vdw_tables(17,18, default_type) = 10.983;
	atom_vdw_tables(17,19, default_type) = 7.065;
	atom_vdw_tables(17,20, default_type) = 5.331;
	atom_vdw_tables(17,21, default_type) = 7.118;
	atom_vdw_tables(17,22, default_type) = 7.728;
	atom_vdw_tables(17,23, default_type) = 8.294;
	atom_vdw_tables(17,24, default_type) = 5.736;
	atom_vdw_tables(17,25, default_type) = 4.326;
	atom_vdw_tables(17,26, default_type) = 0.000;
	atom_vdw_tables(17,27, default_type) = 0.000;
	atom_vdw_tables(17,28, default_type) = 0.000;
	atom_vdw_tables(17,29, default_type) = 0.000;
	atom_vdw_tables(17,30, default_type) = 0.000;
	atom_vdw_tables(17,31, default_type) = 0.000;
	atom_vdw_tables(17,32, default_type) = 0.000;
	atom_vdw_tables(17,33, default_type) = 0.000;
	atom_vdw_tables(18, 1, default_type) = 9.272;
	atom_vdw_tables(18, 2, default_type) = 8.649;
	atom_vdw_tables(18, 3, default_type) = 8.791;
	atom_vdw_tables(18, 4, default_type) = 4.016;
	atom_vdw_tables(18, 5, default_type) = 4.752;
	atom_vdw_tables(18, 6, default_type) = 10.272;
	atom_vdw_tables(18, 7, default_type) = 11.710;
	atom_vdw_tables(18, 8, default_type) = 9.511;
	atom_vdw_tables(18, 9, default_type) = 7.590;
	atom_vdw_tables(18,10, default_type) = 4.928;
	atom_vdw_tables(18,11, default_type) = 11.089;
	atom_vdw_tables(18,12, default_type) = 7.290;
	atom_vdw_tables(18,13, default_type) = 11.710;
	atom_vdw_tables(18,14, default_type) = 6.734;
	atom_vdw_tables(18,15, default_type) = 8.644;
	atom_vdw_tables(18,16, default_type) = 9.809;
	atom_vdw_tables(18,17, default_type) = 10.983;
	atom_vdw_tables(18,18, default_type) = 13.440;
	atom_vdw_tables(18,19, default_type) = 7.382;
	atom_vdw_tables(18,20, default_type) = 7.038;
	atom_vdw_tables(18,21, default_type) = 9.382;
	atom_vdw_tables(18,22, default_type) = 11.296;
	atom_vdw_tables(18,23, default_type) = 12.510;
	atom_vdw_tables(18,24, default_type) = 6.345;
	atom_vdw_tables(18,25, default_type) = 4.431;
	atom_vdw_tables(18,26, default_type) = 0.000;
	atom_vdw_tables(18,27, default_type) = 0.000;
	atom_vdw_tables(18,28, default_type) = 0.000;
	atom_vdw_tables(18,29, default_type) = 0.000;
	atom_vdw_tables(18,30, default_type) = 0.000;
	atom_vdw_tables(18,31, default_type) = 0.000;
	atom_vdw_tables(18,32, default_type) = 0.000;
	atom_vdw_tables(18,33, default_type) = 0.000;
	atom_vdw_tables(19, 1, default_type) = 5.774;
	atom_vdw_tables(19, 2, default_type) = 5.968;
	atom_vdw_tables(19, 3, default_type) = 4.580;
	atom_vdw_tables(19, 4, default_type) = 5.636;
	atom_vdw_tables(19, 5, default_type) = 3.565;
	atom_vdw_tables(19, 6, default_type) = 6.315;
	atom_vdw_tables(19, 7, default_type) = 9.321;
	atom_vdw_tables(19, 8, default_type) = 5.185;
	atom_vdw_tables(19, 9, default_type) = 6.682;
	atom_vdw_tables(19,10, default_type) = 2.722;
	atom_vdw_tables(19,11, default_type) = 7.268;
	atom_vdw_tables(19,12, default_type) = 5.958;
	atom_vdw_tables(19,13, default_type) = 7.585;
	atom_vdw_tables(19,14, default_type) = 4.884;
	atom_vdw_tables(19,15, default_type) = 5.789;
	atom_vdw_tables(19,16, default_type) = 8.260;
	atom_vdw_tables(19,17, default_type) = 7.065;
	atom_vdw_tables(19,18, default_type) = 7.382;
	atom_vdw_tables(19,19, default_type) = 5.570;
	atom_vdw_tables(19,20, default_type) = 3.976;
	atom_vdw_tables(19,21, default_type) = 5.731;
	atom_vdw_tables(19,22, default_type) = 5.603;
	atom_vdw_tables(19,23, default_type) = 7.530;
	atom_vdw_tables(19,24, default_type) = 4.722;
	atom_vdw_tables(19,25, default_type) = 3.000;
	atom_vdw_tables(19,26, default_type) = 0.000;
	atom_vdw_tables(19,27, default_type) = 0.000;
	atom_vdw_tables(19,28, default_type) = 0.000;
	atom_vdw_tables(19,29, default_type) = 0.000;
	atom_vdw_tables(19,30, default_type) = 0.000;
	atom_vdw_tables(19,31, default_type) = 0.000;
	atom_vdw_tables(19,32, default_type) = 0.000;
	atom_vdw_tables(19,33, default_type) = 0.000;
	atom_vdw_tables(20, 1, default_type) = 3.968;
	atom_vdw_tables(20, 2, default_type) = 3.126;
	atom_vdw_tables(20, 3, default_type) = 1.943;
	atom_vdw_tables(20, 4, default_type) = 3.201;
	atom_vdw_tables(20, 5, default_type) = 1.814;
	atom_vdw_tables(20, 6, default_type) = 5.443;
	atom_vdw_tables(20, 7, default_type) = 7.497;
	atom_vdw_tables(20, 8, default_type) = 3.806;
	atom_vdw_tables(20, 9, default_type) = 2.641;
	atom_vdw_tables(20,10, default_type) = 3.197;
	atom_vdw_tables(20,11, default_type) = 4.364;
	atom_vdw_tables(20,12, default_type) = 3.190;
	atom_vdw_tables(20,13, default_type) = 5.688;
	atom_vdw_tables(20,14, default_type) = 6.933;
	atom_vdw_tables(20,15, default_type) = 4.356;
	atom_vdw_tables(20,16, default_type) = 3.629;
	atom_vdw_tables(20,17, default_type) = 5.331;
	atom_vdw_tables(20,18, default_type) = 7.038;
	atom_vdw_tables(20,19, default_type) = 3.976;
	atom_vdw_tables(20,20, default_type) = 4.068;
	atom_vdw_tables(20,21, default_type) = 4.439;
	atom_vdw_tables(20,22, default_type) = 6.260;
	atom_vdw_tables(20,23, default_type) = 5.244;
	atom_vdw_tables(20,24, default_type) = 2.802;
	atom_vdw_tables(20,25, default_type) = 2.742;
	atom_vdw_tables(20,26, default_type) = 0.000;
	atom_vdw_tables(20,27, default_type) = 0.000;
	atom_vdw_tables(20,28, default_type) = 0.000;
	atom_vdw_tables(20,29, default_type) = 0.000;
	atom_vdw_tables(20,30, default_type) = 0.000;
	atom_vdw_tables(20,31, default_type) = 0.000;
	atom_vdw_tables(20,32, default_type) = 0.000;
	atom_vdw_tables(20,33, default_type) = 0.000;
	atom_vdw_tables(21, 1, default_type) = 7.829;
	atom_vdw_tables(21, 2, default_type) = 8.352;
	atom_vdw_tables(21, 3, default_type) = 7.596;
	atom_vdw_tables(21, 4, default_type) = 7.885;
	atom_vdw_tables(21, 5, default_type) = 4.991;
	atom_vdw_tables(21, 6, default_type) = 8.940;
	atom_vdw_tables(21, 7, default_type) = 8.809;
	atom_vdw_tables(21, 8, default_type) = 6.559;
	atom_vdw_tables(21, 9, default_type) = 4.713;
	atom_vdw_tables(21,10, default_type) = 5.240;
	atom_vdw_tables(21,11, default_type) = 9.066;
	atom_vdw_tables(21,12, default_type) = 6.205;
	atom_vdw_tables(21,13, default_type) = 9.986;
	atom_vdw_tables(21,14, default_type) = 5.664;
	atom_vdw_tables(21,15, default_type) = 6.610;
	atom_vdw_tables(21,16, default_type) = 7.773;
	atom_vdw_tables(21,17, default_type) = 7.118;
	atom_vdw_tables(21,18, default_type) = 9.382;
	atom_vdw_tables(21,19, default_type) = 5.731;
	atom_vdw_tables(21,20, default_type) = 4.439;
	atom_vdw_tables(21,21, default_type) = 7.497;
	atom_vdw_tables(21,22, default_type) = 9.090;
	atom_vdw_tables(21,23, default_type) = 9.042;
	atom_vdw_tables(21,24, default_type) = 5.895;
	atom_vdw_tables(21,25, default_type) = 5.022;
	atom_vdw_tables(21,26, default_type) = 0.000;
	atom_vdw_tables(21,27, default_type) = 0.000;
	atom_vdw_tables(21,28, default_type) = 0.000;
	atom_vdw_tables(21,29, default_type) = 0.000;
	atom_vdw_tables(21,30, default_type) = 0.000;
	atom_vdw_tables(21,31, default_type) = 0.000;
	atom_vdw_tables(21,32, default_type) = 0.000;
	atom_vdw_tables(21,33, default_type) = 0.000;
	atom_vdw_tables(22, 1, default_type) = 7.563;
	atom_vdw_tables(22, 2, default_type) = 8.352;
	atom_vdw_tables(22, 3, default_type) = 8.964;
	atom_vdw_tables(22, 4, default_type) = 9.456;
	atom_vdw_tables(22, 5, default_type) = 5.612;
	atom_vdw_tables(22, 6, default_type) = 9.530;
	atom_vdw_tables(22, 7, default_type) = 8.744;
	atom_vdw_tables(22, 8, default_type) = 7.166;
	atom_vdw_tables(22, 9, default_type) = 7.204;
	atom_vdw_tables(22,10, default_type) = 6.651;
	atom_vdw_tables(22,11, default_type) = 9.272;
	atom_vdw_tables(22,12, default_type) = 8.020;
	atom_vdw_tables(22,13, default_type) = 9.828;
	atom_vdw_tables(22,14, default_type) = 5.929;
	atom_vdw_tables(22,15, default_type) = 6.896;
	atom_vdw_tables(22,16, default_type) = 8.532;
	atom_vdw_tables(22,17, default_type) = 7.728;
	atom_vdw_tables(22,18, default_type) = 11.296;
	atom_vdw_tables(22,19, default_type) = 5.603;
	atom_vdw_tables(22,20, default_type) = 6.260;
	atom_vdw_tables(22,21, default_type) = 9.090;
	atom_vdw_tables(22,22, default_type) = 9.828;
	atom_vdw_tables(22,23, default_type) = 11.169;
	atom_vdw_tables(22,24, default_type) = 7.964;
	atom_vdw_tables(22,25, default_type) = 6.160;
	atom_vdw_tables(22,26, default_type) = 0.000;
	atom_vdw_tables(22,27, default_type) = 0.000;
	atom_vdw_tables(22,28, default_type) = 0.000;
	atom_vdw_tables(22,29, default_type) = 0.000;
	atom_vdw_tables(22,30, default_type) = 0.000;
	atom_vdw_tables(22,31, default_type) = 0.000;
	atom_vdw_tables(22,32, default_type) = 0.000;
	atom_vdw_tables(22,33, default_type) = 0.000;

	// section 1c
	atom_vdw_tables(23, 1, default_type) = 9.321;
	atom_vdw_tables(23, 2, default_type) = 10.465;
	atom_vdw_tables(23, 3, default_type) = 8.928;
	atom_vdw_tables(23, 4, default_type) = 9.413;
	atom_vdw_tables(23, 5, default_type) = 5.939;
	atom_vdw_tables(23, 6, default_type) = 9.357;
	atom_vdw_tables(23, 7, default_type) = 10.259;
	atom_vdw_tables(23, 8, default_type) = 9.205;
	atom_vdw_tables(23, 9, default_type) = 5.626;
	atom_vdw_tables(23,10, default_type) = 4.661;
	atom_vdw_tables(23,11, default_type) = 10.969;
	atom_vdw_tables(23,12, default_type) = 6.391;
	atom_vdw_tables(23,13, default_type) = 9.598;
	atom_vdw_tables(23,14, default_type) = 7.023;
	atom_vdw_tables(23,15, default_type) = 8.009;
	atom_vdw_tables(23,16, default_type) = 6.096;
	atom_vdw_tables(23,17, default_type) = 8.294;
	atom_vdw_tables(23,18, default_type) = 12.510;
	atom_vdw_tables(23,19, default_type) = 7.530;
	atom_vdw_tables(23,20, default_type) = 5.244;
	atom_vdw_tables(23,21, default_type) = 9.042;
	atom_vdw_tables(23,22, default_type) = 11.169;
	atom_vdw_tables(23,23, default_type) = 11.451;
	atom_vdw_tables(23,24, default_type) = 5.717;
	atom_vdw_tables(23,25, default_type) = 5.944;
	atom_vdw_tables(23,26, default_type) = 0.000;
	atom_vdw_tables(23,27, default_type) = 0.000;
	atom_vdw_tables(23,28, default_type) = 0.000;
	atom_vdw_tables(23,29, default_type) = 0.000;
	atom_vdw_tables(23,30, default_type) = 0.000;
	atom_vdw_tables(23,31, default_type) = 0.000;
	atom_vdw_tables(23,32, default_type) = 0.000;
	atom_vdw_tables(23,33, default_type) = 0.000;
	atom_vdw_tables(24, 1, default_type) = 4.618;
	atom_vdw_tables(24, 2, default_type) = 4.588;
	atom_vdw_tables(24, 3, default_type) = 4.182;
	atom_vdw_tables(24, 4, default_type) = 4.452;
	atom_vdw_tables(24, 5, default_type) = 3.262;
	atom_vdw_tables(24, 6, default_type) = 5.295;
	atom_vdw_tables(24, 7, default_type) = 6.482;
	atom_vdw_tables(24, 8, default_type) = 5.240;
	atom_vdw_tables(24, 9, default_type) = 6.265;
	atom_vdw_tables(24,10, default_type) = 4.235;
	atom_vdw_tables(24,11, default_type) = 5.532;
	atom_vdw_tables(24,12, default_type) = 5.139;
	atom_vdw_tables(24,13, default_type) = 6.523;
	atom_vdw_tables(24,14, default_type) = 3.045;
	atom_vdw_tables(24,15, default_type) = 3.378;
	atom_vdw_tables(24,16, default_type) = 5.434;
	atom_vdw_tables(24,17, default_type) = 5.736;
	atom_vdw_tables(24,18, default_type) = 6.345;
	atom_vdw_tables(24,19, default_type) = 4.722;
	atom_vdw_tables(24,20, default_type) = 2.802;
	atom_vdw_tables(24,21, default_type) = 5.895;
	atom_vdw_tables(24,22, default_type) = 7.964;
	atom_vdw_tables(24,23, default_type) = 5.717;
	atom_vdw_tables(24,24, default_type) = 7.480;
	atom_vdw_tables(24,25, default_type) = 2.776;
	atom_vdw_tables(24,26, default_type) = 0.000;
	atom_vdw_tables(24,27, default_type) = 0.000;
	atom_vdw_tables(24,28, default_type) = 0.000;
	atom_vdw_tables(24,29, default_type) = 0.000;
	atom_vdw_tables(24,30, default_type) = 0.000;
	atom_vdw_tables(24,31, default_type) = 0.000;
	atom_vdw_tables(24,32, default_type) = 0.000;
	atom_vdw_tables(24,33, default_type) = 0.000;
	atom_vdw_tables(25, 1, default_type) = 3.721;
	atom_vdw_tables(25, 2, default_type) = 3.557;
	atom_vdw_tables(25, 3, default_type) = 3.877;
	atom_vdw_tables(25, 4, default_type) = 4.439;
	atom_vdw_tables(25, 5, default_type) = 2.789;
	atom_vdw_tables(25, 6, default_type) = 5.272;
	atom_vdw_tables(25, 7, default_type) = 4.516;
	atom_vdw_tables(25, 8, default_type) = 5.669;
	atom_vdw_tables(25, 9, default_type) = 3.419;
	atom_vdw_tables(25,10, default_type) = 2.076;
	atom_vdw_tables(25,11, default_type) = 5.636;
	atom_vdw_tables(25,12, default_type) = 4.406;
	atom_vdw_tables(25,13, default_type) = 5.076;
	atom_vdw_tables(25,14, default_type) = 2.503;
	atom_vdw_tables(25,15, default_type) = 2.696;
	atom_vdw_tables(25,16, default_type) = 3.378;
	atom_vdw_tables(25,17, default_type) = 4.326;
	atom_vdw_tables(25,18, default_type) = 4.431;
	atom_vdw_tables(25,19, default_type) = 3.000;
	atom_vdw_tables(25,20, default_type) = 2.742;
	atom_vdw_tables(25,21, default_type) = 5.022;
	atom_vdw_tables(25,22, default_type) = 6.160;
	atom_vdw_tables(25,23, default_type) = 5.944;
	atom_vdw_tables(25,24, default_type) = 2.776;
	atom_vdw_tables(25,25, default_type) = 2.722;
	atom_vdw_tables(25,26, default_type) = 0.000;
	atom_vdw_tables(25,27, default_type) = 0.000;
	atom_vdw_tables(25,28, default_type) = 0.000;
	atom_vdw_tables(25,29, default_type) = 0.000;
	atom_vdw_tables(25,30, default_type) = 0.000;
	atom_vdw_tables(25,31, default_type) = 0.000;
	atom_vdw_tables(25,32, default_type) = 0.000;
	atom_vdw_tables(25,33, default_type) = 0.000;
	atom_vdw_tables(26, 1, default_type) = 0.000;
	atom_vdw_tables(26, 2, default_type) = 0.000;
	atom_vdw_tables(26, 3, default_type) = 0.000;
	atom_vdw_tables(26, 4, default_type) = 0.000;
	atom_vdw_tables(26, 5, default_type) = 0.000;
	atom_vdw_tables(26, 6, default_type) = 0.000;
	atom_vdw_tables(26, 7, default_type) = 0.000;
	atom_vdw_tables(26, 8, default_type) = 0.000;
	atom_vdw_tables(26, 9, default_type) = 0.000;
	atom_vdw_tables(26,10, default_type) = 0.000;
	atom_vdw_tables(26,11, default_type) = 0.000;
	atom_vdw_tables(26,12, default_type) = 0.000;
	atom_vdw_tables(26,13, default_type) = 0.000;
	atom_vdw_tables(26,14, default_type) = 0.000;
	atom_vdw_tables(26,15, default_type) = 0.000;
	atom_vdw_tables(26,16, default_type) = 0.000;
	atom_vdw_tables(26,17, default_type) = 0.000;
	atom_vdw_tables(26,18, default_type) = 0.000;
	atom_vdw_tables(26,19, default_type) = 0.000;
	atom_vdw_tables(26,20, default_type) = 0.000;
	atom_vdw_tables(26,21, default_type) = 0.000;
	atom_vdw_tables(26,22, default_type) = 0.000;
	atom_vdw_tables(26,23, default_type) = 0.000;
	atom_vdw_tables(26,24, default_type) = 0.000;
	atom_vdw_tables(26,25, default_type) = 0.000;
	atom_vdw_tables(26,26, default_type) = 0.000;
	atom_vdw_tables(26,27, default_type) = 0.000;
	atom_vdw_tables(26,28, default_type) = 0.000;
	atom_vdw_tables(26,29, default_type) = 0.000;
	atom_vdw_tables(26,30, default_type) = 0.000;
	atom_vdw_tables(26,31, default_type) = 0.000;
	atom_vdw_tables(26,32, default_type) = 0.000;
	atom_vdw_tables(26,33, default_type) = 0.000;
	atom_vdw_tables(27, 1, default_type) = 0.000;
	atom_vdw_tables(27, 2, default_type) = 0.000;
	atom_vdw_tables(27, 3, default_type) = 0.000;
	atom_vdw_tables(27, 4, default_type) = 0.000;
	atom_vdw_tables(27, 5, default_type) = 0.000;
	atom_vdw_tables(27, 6, default_type) = 0.000;
	atom_vdw_tables(27, 7, default_type) = 0.000;
	atom_vdw_tables(27, 8, default_type) = 0.000;
	atom_vdw_tables(27, 9, default_type) = 0.000;
	atom_vdw_tables(27,10, default_type) = 0.000;
	atom_vdw_tables(27,11, default_type) = 0.000;
	atom_vdw_tables(27,12, default_type) = 0.000;
	atom_vdw_tables(27,13, default_type) = 0.000;
	atom_vdw_tables(27,14, default_type) = 0.000;
	atom_vdw_tables(27,15, default_type) = 0.000;
	atom_vdw_tables(27,16, default_type) = 0.000;
	atom_vdw_tables(27,17, default_type) = 0.000;
	atom_vdw_tables(27,18, default_type) = 0.000;
	atom_vdw_tables(27,19, default_type) = 0.000;
	atom_vdw_tables(27,20, default_type) = 0.000;
	atom_vdw_tables(27,21, default_type) = 0.000;
	atom_vdw_tables(27,22, default_type) = 0.000;
	atom_vdw_tables(27,23, default_type) = 0.000;
	atom_vdw_tables(27,24, default_type) = 0.000;
	atom_vdw_tables(27,25, default_type) = 0.000;
	atom_vdw_tables(27,26, default_type) = 0.000;
	atom_vdw_tables(27,27, default_type) = 0.000;
	atom_vdw_tables(27,28, default_type) = 0.000;
	atom_vdw_tables(27,29, default_type) = 0.000;
	atom_vdw_tables(27,30, default_type) = 0.000;
	atom_vdw_tables(27,31, default_type) = 0.000;
	atom_vdw_tables(27,32, default_type) = 0.000;
	atom_vdw_tables(27,33, default_type) = 0.000;
	atom_vdw_tables(28, 1, default_type) = 0.000;
	atom_vdw_tables(28, 2, default_type) = 0.000;
	atom_vdw_tables(28, 3, default_type) = 0.000;
	atom_vdw_tables(28, 4, default_type) = 0.000;
	atom_vdw_tables(28, 5, default_type) = 0.000;
	atom_vdw_tables(28, 6, default_type) = 0.000;
	atom_vdw_tables(28, 7, default_type) = 0.000;
	atom_vdw_tables(28, 8, default_type) = 0.000;
	atom_vdw_tables(28, 9, default_type) = 0.000;
	atom_vdw_tables(28,10, default_type) = 0.000;
	atom_vdw_tables(28,11, default_type) = 0.000;
	atom_vdw_tables(28,12, default_type) = 0.000;
	atom_vdw_tables(28,13, default_type) = 0.000;
	atom_vdw_tables(28,14, default_type) = 0.000;
	atom_vdw_tables(28,15, default_type) = 0.000;
	atom_vdw_tables(28,16, default_type) = 0.000;
	atom_vdw_tables(28,17, default_type) = 0.000;
	atom_vdw_tables(28,18, default_type) = 0.000;
	atom_vdw_tables(28,19, default_type) = 0.000;
	atom_vdw_tables(28,20, default_type) = 0.000;
	atom_vdw_tables(28,21, default_type) = 0.000;
	atom_vdw_tables(28,22, default_type) = 0.000;
	atom_vdw_tables(28,23, default_type) = 0.000;
	atom_vdw_tables(28,24, default_type) = 0.000;
	atom_vdw_tables(28,25, default_type) = 0.000;
	atom_vdw_tables(28,26, default_type) = 0.000;
	atom_vdw_tables(28,27, default_type) = 0.000;
	atom_vdw_tables(28,28, default_type) = 0.000;
	atom_vdw_tables(28,29, default_type) = 0.000;
	atom_vdw_tables(28,30, default_type) = 0.000;
	atom_vdw_tables(28,31, default_type) = 0.000;
	atom_vdw_tables(28,32, default_type) = 0.000;
	atom_vdw_tables(28,33, default_type) = 0.000;
	atom_vdw_tables(29, 1, default_type) = 0.000;
	atom_vdw_tables(29, 2, default_type) = 0.000;
	atom_vdw_tables(29, 3, default_type) = 0.000;
	atom_vdw_tables(29, 4, default_type) = 0.000;
	atom_vdw_tables(29, 5, default_type) = 0.000;
	atom_vdw_tables(29, 6, default_type) = 0.000;
	atom_vdw_tables(29, 7, default_type) = 0.000;
	atom_vdw_tables(29, 8, default_type) = 0.000;
	atom_vdw_tables(29, 9, default_type) = 0.000;
	atom_vdw_tables(29,10, default_type) = 0.000;
	atom_vdw_tables(29,11, default_type) = 0.000;
	atom_vdw_tables(29,12, default_type) = 0.000;
	atom_vdw_tables(29,13, default_type) = 0.000;
	atom_vdw_tables(29,14, default_type) = 0.000;
	atom_vdw_tables(29,15, default_type) = 0.000;
	atom_vdw_tables(29,16, default_type) = 0.000;
	atom_vdw_tables(29,17, default_type) = 0.000;
	atom_vdw_tables(29,18, default_type) = 0.000;
	atom_vdw_tables(29,19, default_type) = 0.000;
	atom_vdw_tables(29,20, default_type) = 0.000;
	atom_vdw_tables(29,21, default_type) = 0.000;
	atom_vdw_tables(29,22, default_type) = 0.000;
	atom_vdw_tables(29,23, default_type) = 0.000;
	atom_vdw_tables(29,24, default_type) = 0.000;
	atom_vdw_tables(29,25, default_type) = 0.000;
	atom_vdw_tables(29,26, default_type) = 0.000;
	atom_vdw_tables(29,27, default_type) = 0.000;
	atom_vdw_tables(29,28, default_type) = 0.000;
	atom_vdw_tables(29,29, default_type) = 0.000;
	atom_vdw_tables(29,30, default_type) = 0.000;
	atom_vdw_tables(29,31, default_type) = 0.000;
	atom_vdw_tables(29,32, default_type) = 0.000;
	atom_vdw_tables(29,33, default_type) = 0.000;
	atom_vdw_tables(30, 1, default_type) = 0.000;
	atom_vdw_tables(30, 2, default_type) = 0.000;
	atom_vdw_tables(30, 3, default_type) = 0.000;
	atom_vdw_tables(30, 4, default_type) = 0.000;
	atom_vdw_tables(30, 5, default_type) = 0.000;
	atom_vdw_tables(30, 6, default_type) = 0.000;
	atom_vdw_tables(30, 7, default_type) = 0.000;
	atom_vdw_tables(30, 8, default_type) = 0.000;
	atom_vdw_tables(30, 9, default_type) = 0.000;
	atom_vdw_tables(30,10, default_type) = 0.000;
	atom_vdw_tables(30,11, default_type) = 0.000;
	atom_vdw_tables(30,12, default_type) = 0.000;
	atom_vdw_tables(30,13, default_type) = 0.000;
	atom_vdw_tables(30,14, default_type) = 0.000;
	atom_vdw_tables(30,15, default_type) = 0.000;
	atom_vdw_tables(30,16, default_type) = 0.000;
	atom_vdw_tables(30,17, default_type) = 0.000;
	atom_vdw_tables(30,18, default_type) = 0.000;
	atom_vdw_tables(30,19, default_type) = 0.000;
	atom_vdw_tables(30,20, default_type) = 0.000;
	atom_vdw_tables(30,21, default_type) = 0.000;
	atom_vdw_tables(30,22, default_type) = 0.000;
	atom_vdw_tables(30,23, default_type) = 0.000;
	atom_vdw_tables(30,24, default_type) = 0.000;
	atom_vdw_tables(30,25, default_type) = 0.000;
	atom_vdw_tables(30,26, default_type) = 0.000;
	atom_vdw_tables(30,27, default_type) = 0.000;
	atom_vdw_tables(30,28, default_type) = 0.000;
	atom_vdw_tables(30,29, default_type) = 0.000;
	atom_vdw_tables(30,30, default_type) = 0.000;
	atom_vdw_tables(30,31, default_type) = 0.000;
	atom_vdw_tables(30,32, default_type) = 0.000;
	atom_vdw_tables(30,33, default_type) = 0.000;
	atom_vdw_tables(31, 1, default_type) = 0.000;
	atom_vdw_tables(31, 2, default_type) = 0.000;
	atom_vdw_tables(31, 3, default_type) = 0.000;
	atom_vdw_tables(31, 4, default_type) = 0.000;
	atom_vdw_tables(31, 5, default_type) = 0.000;
	atom_vdw_tables(31, 6, default_type) = 0.000;
	atom_vdw_tables(31, 7, default_type) = 0.000;
	atom_vdw_tables(31, 8, default_type) = 0.000;
	atom_vdw_tables(31, 9, default_type) = 0.000;
	atom_vdw_tables(31,10, default_type) = 0.000;
	atom_vdw_tables(31,11, default_type) = 0.000;
	atom_vdw_tables(31,12, default_type) = 0.000;
	atom_vdw_tables(31,13, default_type) = 0.000;
	atom_vdw_tables(31,14, default_type) = 0.000;
	atom_vdw_tables(31,15, default_type) = 0.000;
	atom_vdw_tables(31,16, default_type) = 0.000;
	atom_vdw_tables(31,17, default_type) = 0.000;
	atom_vdw_tables(31,18, default_type) = 0.000;
	atom_vdw_tables(31,19, default_type) = 0.000;
	atom_vdw_tables(31,20, default_type) = 0.000;
	atom_vdw_tables(31,21, default_type) = 0.000;
	atom_vdw_tables(31,22, default_type) = 0.000;
	atom_vdw_tables(31,23, default_type) = 0.000;
	atom_vdw_tables(31,24, default_type) = 0.000;
	atom_vdw_tables(31,25, default_type) = 0.000;
	atom_vdw_tables(31,26, default_type) = 0.000;
	atom_vdw_tables(31,27, default_type) = 0.000;
	atom_vdw_tables(31,28, default_type) = 0.000;
	atom_vdw_tables(31,29, default_type) = 0.000;
	atom_vdw_tables(31,30, default_type) = 0.000;
	atom_vdw_tables(31,31, default_type) = 0.000;
	atom_vdw_tables(31,32, default_type) = 0.000;
	atom_vdw_tables(31,33, default_type) = 0.000;
	atom_vdw_tables(32, 1, default_type) = 0.000;
	atom_vdw_tables(32, 2, default_type) = 0.000;
	atom_vdw_tables(32, 3, default_type) = 0.000;
	atom_vdw_tables(32, 4, default_type) = 0.000;
	atom_vdw_tables(32, 5, default_type) = 0.000;
	atom_vdw_tables(32, 6, default_type) = 0.000;
	atom_vdw_tables(32, 7, default_type) = 0.000;
	atom_vdw_tables(32, 8, default_type) = 0.000;
	atom_vdw_tables(32, 9, default_type) = 0.000;
	atom_vdw_tables(32,10, default_type) = 0.000;
	atom_vdw_tables(32,11, default_type) = 0.000;
	atom_vdw_tables(32,12, default_type) = 0.000;
	atom_vdw_tables(32,13, default_type) = 0.000;
	atom_vdw_tables(32,14, default_type) = 0.000;
	atom_vdw_tables(32,15, default_type) = 0.000;
	atom_vdw_tables(32,16, default_type) = 0.000;
	atom_vdw_tables(32,17, default_type) = 0.000;
	atom_vdw_tables(32,18, default_type) = 0.000;
	atom_vdw_tables(32,19, default_type) = 0.000;
	atom_vdw_tables(32,20, default_type) = 0.000;
	atom_vdw_tables(32,21, default_type) = 0.000;
	atom_vdw_tables(32,22, default_type) = 0.000;
	atom_vdw_tables(32,23, default_type) = 0.000;
	atom_vdw_tables(32,24, default_type) = 0.000;
	atom_vdw_tables(32,25, default_type) = 0.000;
	atom_vdw_tables(32,26, default_type) = 0.000;
	atom_vdw_tables(32,27, default_type) = 0.000;
	atom_vdw_tables(32,28, default_type) = 0.000;
	atom_vdw_tables(32,29, default_type) = 0.000;
	atom_vdw_tables(32,30, default_type) = 0.000;
	atom_vdw_tables(32,31, default_type) = 0.000;
	atom_vdw_tables(32,32, default_type) = 0.000;
	atom_vdw_tables(32,33, default_type) = 0.000;
	atom_vdw_tables(33, 1, default_type) = 0.000;
	atom_vdw_tables(33, 2, default_type) = 0.000;
	atom_vdw_tables(33, 3, default_type) = 0.000;
	atom_vdw_tables(33, 4, default_type) = 0.000;
	atom_vdw_tables(33, 5, default_type) = 0.000;
	atom_vdw_tables(33, 6, default_type) = 0.000;
	atom_vdw_tables(33, 7, default_type) = 0.000;
	atom_vdw_tables(33, 8, default_type) = 0.000;
	atom_vdw_tables(33, 9, default_type) = 0.000;
	atom_vdw_tables(33,10, default_type) = 0.000;
	atom_vdw_tables(33,11, default_type) = 0.000;
	atom_vdw_tables(33,12, default_type) = 0.000;
	atom_vdw_tables(33,13, default_type) = 0.000;
	atom_vdw_tables(33,14, default_type) = 0.000;
	atom_vdw_tables(33,15, default_type) = 0.000;
	atom_vdw_tables(33,16, default_type) = 0.000;
	atom_vdw_tables(33,17, default_type) = 0.000;
	atom_vdw_tables(33,18, default_type) = 0.000;
	atom_vdw_tables(33,19, default_type) = 0.000;
	atom_vdw_tables(33,20, default_type) = 0.000;
	atom_vdw_tables(33,21, default_type) = 0.000;
	atom_vdw_tables(33,22, default_type) = 0.000;
	atom_vdw_tables(33,23, default_type) = 0.000;
	atom_vdw_tables(33,24, default_type) = 0.000;
	atom_vdw_tables(33,25, default_type) = 0.000;
	atom_vdw_tables(33,26, default_type) = 0.000;
	atom_vdw_tables(33,27, default_type) = 0.000;
	atom_vdw_tables(33,28, default_type) = 0.000;
	atom_vdw_tables(33,29, default_type) = 0.000;
	atom_vdw_tables(33,30, default_type) = 0.000;
	atom_vdw_tables(33,31, default_type) = 0.000;
	atom_vdw_tables(33,32, default_type) = 0.000;
	atom_vdw_tables(33,33, default_type) = 0.000;
	//KMa phospho_ser
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( i, 34, default_type) = 0.000;
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( 34, i, default_type) = 0.000;
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( i, 26, default_type) = atom_vdw_tables( i, 21, default_type);
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( 26, i, default_type) = atom_vdw_tables( 21, i, default_type);
	}

	// section 2a
	//------------------------------------------------------------------------------
	//km 4/09/02 - TABLE 2
	// Array 2 used in vdw_compute - distances are takes as 5th closest
	// approach in set of 190 crystal structures with 1.3A resolution or
	// better.  Sequences share no greater than 40% sequence identity.
	//------------------------------------------------------------------------------
	atom_vdw_tables( 1, 1, highres) = 12.417;
	atom_vdw_tables( 1, 2, highres) = 11.976;
	atom_vdw_tables( 1, 3, highres) = 11.859;
	atom_vdw_tables( 1, 4, highres) = 8.464;
	atom_vdw_tables( 1, 5, highres) = 6.853;
	atom_vdw_tables( 1, 6, highres) = 7.378;
	atom_vdw_tables( 1, 7, highres) = 10.357;
	atom_vdw_tables( 1, 8, highres) = 8.915;
	atom_vdw_tables( 1, 9, highres) = 9.509;
	atom_vdw_tables( 1,10, highres) = 8.545;
	atom_vdw_tables( 1,11, highres) = 5.548;
	atom_vdw_tables( 1,12, highres) = 10.252;
	atom_vdw_tables( 1,13, highres) = 14.246;
	atom_vdw_tables( 1,14, highres) = 10.573;
	atom_vdw_tables( 1,15, highres) = 11.918;
	atom_vdw_tables( 1,16, highres) = 11.389;
	atom_vdw_tables( 1,17, highres) = 10.807;
	atom_vdw_tables( 1,18, highres) = 12.000;
	atom_vdw_tables( 1,19, highres) = 8.787;
	atom_vdw_tables( 1,20, highres) = 8.078;
	atom_vdw_tables( 1,21, highres) = 9.808;
	atom_vdw_tables( 1,22, highres) = 10.986;
	atom_vdw_tables( 1,23, highres) = 11.163;
	atom_vdw_tables( 1,24, highres) = 7.904;
	atom_vdw_tables( 1,25, highres) = 8.805;
	atom_vdw_tables( 1,26, highres) = 0.000;
	atom_vdw_tables( 1,27, highres) = 0.000;
	atom_vdw_tables( 1,28, highres) = 0.000;
	atom_vdw_tables( 1,29, highres) = 0.000;
	atom_vdw_tables( 1,30, highres) = 0.000;
	atom_vdw_tables( 1,31, highres) = 0.000;
	atom_vdw_tables( 1,32, highres) = 0.000;
	atom_vdw_tables( 1,33, highres) = 0.000;
	atom_vdw_tables( 2, 1, highres) = 11.976;
	atom_vdw_tables( 2, 2, highres) = 13.744;
	atom_vdw_tables( 2, 3, highres) = 12.249;
	atom_vdw_tables( 2, 4, highres) = 10.464; // 12.450 3/23/02
	atom_vdw_tables( 2, 5, highres) = 8.564;
	atom_vdw_tables( 2, 6, highres) = 12.853;
	atom_vdw_tables( 2, 7, highres) = 10.804;
	atom_vdw_tables( 2, 8, highres) = 10.084;
	atom_vdw_tables( 2, 9, highres) = 9.786;
	atom_vdw_tables( 2,10, highres) = 8.093;
	atom_vdw_tables( 2,11, highres) = 13.031;
	atom_vdw_tables( 2,12, highres) = 9.033;
	atom_vdw_tables( 2,13, highres) = 14.682;
	atom_vdw_tables( 2,14, highres) = 9.530;
	atom_vdw_tables( 2,15, highres) = 11.590;
	atom_vdw_tables( 2,16, highres) = 11.705;
	atom_vdw_tables( 2,17, highres) = 11.056;
	atom_vdw_tables( 2,18, highres) = 12.090;
	atom_vdw_tables( 2,19, highres) = 11.200;
	atom_vdw_tables( 2,20, highres) = 7.092;
	atom_vdw_tables( 2,21, highres) = 11.788;
	atom_vdw_tables( 2,22, highres) = 12.498;
	atom_vdw_tables( 2,23, highres) = 12.329;
	atom_vdw_tables( 2,24, highres) = 8.221;
	atom_vdw_tables( 2,25, highres) = 6.992;
	atom_vdw_tables( 2,26, highres) = 0.000;
	atom_vdw_tables( 2,27, highres) = 0.000;
	atom_vdw_tables( 2,28, highres) = 0.000;
	atom_vdw_tables( 2,29, highres) = 0.000;
	atom_vdw_tables( 2,30, highres) = 0.000;
	atom_vdw_tables( 2,31, highres) = 0.000;
	atom_vdw_tables( 2,32, highres) = 0.000;
	atom_vdw_tables( 2,33, highres) = 0.000;
	atom_vdw_tables( 3, 1, highres) = 11.859;
	atom_vdw_tables( 3, 2, highres) = 12.249;
	atom_vdw_tables( 3, 3, highres) = 12.024;
	atom_vdw_tables( 3, 4, highres) = 11.663;
	atom_vdw_tables( 3, 5, highres) = 8.739; // 8.815 3/23/02
	atom_vdw_tables( 3, 6, highres) = 12.191;
	atom_vdw_tables( 3, 7, highres) = 8.580;
	atom_vdw_tables( 3, 8, highres) = 10.533;
	atom_vdw_tables( 3, 9, highres) = 8.919;
	atom_vdw_tables( 3,10, highres) = 6.867;
	atom_vdw_tables( 3,11, highres) = 12.538;
	atom_vdw_tables( 3,12, highres) = 7.630;
	atom_vdw_tables( 3,13, highres) = 11.959;
	atom_vdw_tables( 3,14, highres) = 8.014;
	atom_vdw_tables( 3,15, highres) = 9.848;
	atom_vdw_tables( 3,16, highres) = 8.512;
	atom_vdw_tables( 3,17, highres) = 10.582;
	atom_vdw_tables( 3,18, highres) = 12.525;
	atom_vdw_tables( 3,19, highres) = 8.142;
	atom_vdw_tables( 3,20, highres) = 5.563;
	atom_vdw_tables( 3,21, highres) = 10.335;
	atom_vdw_tables( 3,22, highres) = 10.859;
	atom_vdw_tables( 3,23, highres) = 12.036;
	atom_vdw_tables( 3,24, highres) = 6.592;
	atom_vdw_tables( 3,25, highres) = 5.676;
	atom_vdw_tables( 3,26, highres) = 0.000;
	atom_vdw_tables( 3,27, highres) = 0.000;
	atom_vdw_tables( 3,28, highres) = 0.000;
	atom_vdw_tables( 3,29, highres) = 0.000;
	atom_vdw_tables( 3,30, highres) = 0.000;
	atom_vdw_tables( 3,31, highres) = 0.000;
	atom_vdw_tables( 3,32, highres) = 0.000;
	atom_vdw_tables( 3,33, highres) = 0.000;
	atom_vdw_tables( 4, 1, highres) = 8.464;
	atom_vdw_tables( 4, 2, highres) = 12.450; // 12.450 3/23/02
	atom_vdw_tables( 4, 3, highres) = 11.663;
	atom_vdw_tables( 4, 4, highres) = 13.674;
	atom_vdw_tables( 4, 5, highres) = 8.511; // 9.044 3/23/02
	atom_vdw_tables( 4, 6, highres) = 8.989;
	atom_vdw_tables( 4, 7, highres) = 11.088;
	atom_vdw_tables( 4, 8, highres) = 10.310;
	atom_vdw_tables( 4, 9, highres) = 8.330;
	atom_vdw_tables( 4,10, highres) = 8.443;
	atom_vdw_tables( 4,11, highres) = 5.414;
	atom_vdw_tables( 4,12, highres) = 8.958;
	atom_vdw_tables( 4,13, highres) = 13.473;
	atom_vdw_tables( 4,14, highres) = 8.614;
	atom_vdw_tables( 4,15, highres) = 12.148;
	atom_vdw_tables( 4,16, highres) = 10.470;
	atom_vdw_tables( 4,17, highres) = 10.065;
	atom_vdw_tables( 4,18, highres) = 8.137;
	atom_vdw_tables( 4,19, highres) = 9.172;
	atom_vdw_tables( 4,20, highres) = 6.041;
	atom_vdw_tables( 4,21, highres) = 10.081;
	atom_vdw_tables( 4,22, highres) = 11.383;
	atom_vdw_tables( 4,23, highres) = 11.936;
	atom_vdw_tables( 4,24, highres) = 8.508;
	atom_vdw_tables( 4,25, highres) = 8.177;
	atom_vdw_tables( 4,26, highres) = 0.000;
	atom_vdw_tables( 4,27, highres) = 0.000;
	atom_vdw_tables( 4,28, highres) = 0.000;
	atom_vdw_tables( 4,29, highres) = 0.000;
	atom_vdw_tables( 4,30, highres) = 0.000;
	atom_vdw_tables( 4,31, highres) = 0.000;
	atom_vdw_tables( 4,32, highres) = 0.000;
	atom_vdw_tables( 4,33, highres) = 0.000;
	atom_vdw_tables( 5, 1, highres) = 6.853;
	atom_vdw_tables( 5, 2, highres) = 8.564;
	atom_vdw_tables( 5, 3, highres) = 8.739; // 8.815 3/23/02
	atom_vdw_tables( 5, 4, highres) = 8.511; // 9.044 3/23/02
	atom_vdw_tables( 5, 5, highres) = 8.830;
	atom_vdw_tables( 5, 6, highres) = 7.860;
	atom_vdw_tables( 5, 7, highres) = 9.868;
	atom_vdw_tables( 5, 8, highres) = 7.226;
	atom_vdw_tables( 5, 9, highres) = 6.689;
	atom_vdw_tables( 5,10, highres) = 6.501;
	atom_vdw_tables( 5,11, highres) = 6.688;
	atom_vdw_tables( 5,12, highres) = 6.699;
	atom_vdw_tables( 5,13, highres) = 8.810;
	atom_vdw_tables( 5,14, highres) = 5.189;
	atom_vdw_tables( 5,15, highres) = 9.107;
	atom_vdw_tables( 5,16, highres) = 7.205;
	atom_vdw_tables( 5,17, highres) = 6.461;
	atom_vdw_tables( 5,18, highres) = 9.357;
	atom_vdw_tables( 5,19, highres) = 4.932;
	atom_vdw_tables( 5,20, highres) = 2.724;
	atom_vdw_tables( 5,21, highres) = 7.369;
	atom_vdw_tables( 5,22, highres) = 8.144;
	atom_vdw_tables( 5,23, highres) = 9.408;
	atom_vdw_tables( 5,24, highres) = 6.558;
	atom_vdw_tables( 5,25, highres) = 5.682;
	atom_vdw_tables( 5,26, highres) = 0.000;
	atom_vdw_tables( 5,27, highres) = 0.000;
	atom_vdw_tables( 5,28, highres) = 0.000;
	atom_vdw_tables( 5,29, highres) = 0.000;
	atom_vdw_tables( 5,30, highres) = 0.000;
	atom_vdw_tables( 5,31, highres) = 0.000;
	atom_vdw_tables( 5,32, highres) = 0.000;
	atom_vdw_tables( 5,33, highres) = 0.000;
	atom_vdw_tables( 6, 1, highres) = 7.378;
	atom_vdw_tables( 6, 2, highres) = 12.853;
	atom_vdw_tables( 6, 3, highres) = 12.191;
	atom_vdw_tables( 6, 4, highres) = 8.989;
	atom_vdw_tables( 6, 5, highres) = 7.860;
	atom_vdw_tables( 6, 6, highres) = 13.101;
	atom_vdw_tables( 6, 7, highres) = 15.814;
	atom_vdw_tables( 6, 8, highres) = 14.016;
	atom_vdw_tables( 6, 9, highres) = 11.585;
	atom_vdw_tables( 6,10, highres) = 7.182;
	atom_vdw_tables( 6,11, highres) = 13.453;
	atom_vdw_tables( 6,12, highres) = 12.724;
	atom_vdw_tables( 6,13, highres) = 14.252;
	atom_vdw_tables( 6,14, highres) = 14.845;
	atom_vdw_tables( 6,15, highres) = 11.513;
	atom_vdw_tables( 6,16, highres) = 16.654;
	atom_vdw_tables( 6,17, highres) = 13.566;
	atom_vdw_tables( 6,18, highres) = 14.798;
	atom_vdw_tables( 6,19, highres) = 15.631;
	atom_vdw_tables( 6,20, highres) = 9.847;
	atom_vdw_tables( 6,21, highres) = 13.792;
	atom_vdw_tables( 6,22, highres) = 12.931;
	atom_vdw_tables( 6,23, highres) = 13.611;
	atom_vdw_tables( 6,24, highres) = 13.943;
	atom_vdw_tables( 6,25, highres) = 6.519;
	atom_vdw_tables( 6,26, highres) = 0.000;
	atom_vdw_tables( 6,27, highres) = 0.000;
	atom_vdw_tables( 6,28, highres) = 0.000;
	atom_vdw_tables( 6,29, highres) = 0.000;
	atom_vdw_tables( 6,30, highres) = 0.000;
	atom_vdw_tables( 6,31, highres) = 0.000;
	atom_vdw_tables( 6,32, highres) = 0.000;
	atom_vdw_tables( 6,33, highres) = 0.000;
	atom_vdw_tables( 7, 1, highres) = 10.357;
	atom_vdw_tables( 7, 2, highres) = 10.804;
	atom_vdw_tables( 7, 3, highres) = 8.580;
	atom_vdw_tables( 7, 4, highres) = 11.088;
	atom_vdw_tables( 7, 5, highres) = 9.868;
	atom_vdw_tables( 7, 6, highres) = 15.814;
	atom_vdw_tables( 7, 7, highres) = 5.636;
	atom_vdw_tables( 7, 8, highres) = 27.842;
	atom_vdw_tables( 7, 9, highres) = 25.389;
	atom_vdw_tables( 7,10, highres) = 13.485;
	atom_vdw_tables( 7,11, highres) = 18.391;
	atom_vdw_tables( 7,12, highres) = 24.499;
	atom_vdw_tables( 7,13, highres) = 18.772;
	atom_vdw_tables( 7,14, highres) = 30.995;
	atom_vdw_tables( 7,15, highres) = 15.421;
	atom_vdw_tables( 7,16, highres) = 19.697;
	atom_vdw_tables( 7,17, highres) = 16.402;
	atom_vdw_tables( 7,18, highres) = 22.328;
	atom_vdw_tables( 7,19, highres) = 23.147;
	atom_vdw_tables( 7,20, highres) = 21.921;
	atom_vdw_tables( 7,21, highres) = 17.038;
	atom_vdw_tables( 7,22, highres) = 17.673;
	atom_vdw_tables( 7,23, highres) = 14.368;
	atom_vdw_tables( 7,24, highres) = 31.079;
	atom_vdw_tables( 7,25, highres) = 21.162;
	atom_vdw_tables( 7,26, highres) = 0.000;
	atom_vdw_tables( 7,27, highres) = 0.000;
	atom_vdw_tables( 7,28, highres) = 0.000;
	atom_vdw_tables( 7,29, highres) = 0.000;
	atom_vdw_tables( 7,30, highres) = 0.000;
	atom_vdw_tables( 7,31, highres) = 0.000;
	atom_vdw_tables( 7,32, highres) = 0.000;
	atom_vdw_tables( 7,33, highres) = 0.000;
	atom_vdw_tables( 8, 1, highres) = 8.915;
	atom_vdw_tables( 8, 2, highres) = 10.084;
	atom_vdw_tables( 8, 3, highres) = 10.533;
	atom_vdw_tables( 8, 4, highres) = 10.310;
	atom_vdw_tables( 8, 5, highres) = 7.226;
	atom_vdw_tables( 8, 6, highres) = 14.016;
	atom_vdw_tables( 8, 7, highres) = 27.842;
	atom_vdw_tables( 8, 8, highres) = 20.717;
	atom_vdw_tables( 8, 9, highres) = 16.994;
	atom_vdw_tables( 8,10, highres) = 14.973;
	atom_vdw_tables( 8,11, highres) = 11.989;
	atom_vdw_tables( 8,12, highres) = 16.496;
	atom_vdw_tables( 8,13, highres) = 18.184;
	atom_vdw_tables( 8,14, highres) = 12.355;
	atom_vdw_tables( 8,15, highres) = 12.956;
	atom_vdw_tables( 8,16, highres) = 18.549;
	atom_vdw_tables( 8,17, highres) = 13.096;
	atom_vdw_tables( 8,18, highres) = 16.023;
	atom_vdw_tables( 8,19, highres) = 20.899;
	atom_vdw_tables( 8,20, highres) = 10.842;
	atom_vdw_tables( 8,21, highres) = 11.938;
	atom_vdw_tables( 8,22, highres) = 10.505;
	atom_vdw_tables( 8,23, highres) = 15.052;
	atom_vdw_tables( 8,24, highres) = 18.853;
	atom_vdw_tables( 8,25, highres) = 16.286;
	atom_vdw_tables( 8,26, highres) = 0.000;
	atom_vdw_tables( 8,27, highres) = 0.000;
	atom_vdw_tables( 8,28, highres) = 0.000;
	atom_vdw_tables( 8,29, highres) = 0.000;
	atom_vdw_tables( 8,30, highres) = 0.000;
	atom_vdw_tables( 8,31, highres) = 0.000;
	atom_vdw_tables( 8,32, highres) = 0.000;
	atom_vdw_tables( 8,33, highres) = 0.000;
	atom_vdw_tables( 9, 1, highres) = 9.509;
	atom_vdw_tables( 9, 2, highres) = 9.786;
	atom_vdw_tables( 9, 3, highres) = 8.919;
	atom_vdw_tables( 9, 4, highres) = 8.330;
	atom_vdw_tables( 9, 5, highres) = 6.689;
	atom_vdw_tables( 9, 6, highres) = 11.585;
	atom_vdw_tables( 9, 7, highres) = 25.389;
	atom_vdw_tables( 9, 8, highres) = 16.994;
	atom_vdw_tables( 9, 9, highres) = 29.097;
	atom_vdw_tables( 9,10, highres) = 11.877;
	atom_vdw_tables( 9,11, highres) = 16.832;
	atom_vdw_tables( 9,12, highres) = 20.591;
	atom_vdw_tables( 9,13, highres) = 16.409;
	atom_vdw_tables( 9,14, highres) = 8.798;
	atom_vdw_tables( 9,15, highres) = 13.833;
	atom_vdw_tables( 9,16, highres) = 12.168;
	atom_vdw_tables( 9,17, highres) = 17.815;
	atom_vdw_tables( 9,18, highres) = 16.878;
	atom_vdw_tables( 9,19, highres) = 19.535;
	atom_vdw_tables( 9,20, highres) = 9.903;
	atom_vdw_tables( 9,21, highres) = 16.777;
	atom_vdw_tables( 9,22, highres) = 15.167;
	atom_vdw_tables( 9,23, highres) = 16.534;
	atom_vdw_tables( 9,24, highres) = 31.291;
	atom_vdw_tables( 9,25, highres) = 9.029;
	atom_vdw_tables( 9,26, highres) = 0.000;
	atom_vdw_tables( 9,27, highres) = 0.000;
	atom_vdw_tables( 9,28, highres) = 0.000;
	atom_vdw_tables( 9,29, highres) = 0.000;
	atom_vdw_tables( 9,30, highres) = 0.000;
	atom_vdw_tables( 9,31, highres) = 0.000;
	atom_vdw_tables( 9,32, highres) = 0.000;
	atom_vdw_tables( 9,33, highres) = 0.000;
	atom_vdw_tables(10, 1, highres) = 8.545;
	atom_vdw_tables(10, 2, highres) = 8.093;
	atom_vdw_tables(10, 3, highres) = 6.867;
	atom_vdw_tables(10, 4, highres) = 8.443;
	atom_vdw_tables(10, 5, highres) = 6.501;
	atom_vdw_tables(10, 6, highres) = 7.182;
	atom_vdw_tables(10, 7, highres) = 13.485;
	atom_vdw_tables(10, 8, highres) = 14.973;
	atom_vdw_tables(10, 9, highres) = 11.877;
	atom_vdw_tables(10,10, highres) = 12.555;
	atom_vdw_tables(10,11, highres) = 13.766;
	atom_vdw_tables(10,12, highres) = 10.285;
	atom_vdw_tables(10,13, highres) = 12.750;
	atom_vdw_tables(10,14, highres) = 14.897;
	atom_vdw_tables(10,15, highres) = 7.324;
	atom_vdw_tables(10,16, highres) = 10.708;
	atom_vdw_tables(10,17, highres) = 13.825;
	atom_vdw_tables(10,18, highres) = 12.599;
	atom_vdw_tables(10,19, highres) = 11.476;
	atom_vdw_tables(10,20, highres) = 8.701;
	atom_vdw_tables(10,21, highres) = 15.513;
	atom_vdw_tables(10,22, highres) = 11.885;
	atom_vdw_tables(10,23, highres) = 9.845;
	atom_vdw_tables(10,24, highres) = 11.888;
	atom_vdw_tables(10,25, highres) = 12.593;
	atom_vdw_tables(10,26, highres) = 0.000;
	atom_vdw_tables(10,27, highres) = 0.000;
	atom_vdw_tables(10,28, highres) = 0.000;
	atom_vdw_tables(10,29, highres) = 0.000;
	atom_vdw_tables(10,30, highres) = 0.000;
	atom_vdw_tables(10,31, highres) = 0.000;
	atom_vdw_tables(10,32, highres) = 0.000;
	atom_vdw_tables(10,33, highres) = 0.000;
	atom_vdw_tables(11, 1, highres) = 5.548;
	atom_vdw_tables(11, 2, highres) = 13.031;
	atom_vdw_tables(11, 3, highres) = 12.538;
	atom_vdw_tables(11, 4, highres) = 5.414;
	atom_vdw_tables(11, 5, highres) = 6.688;
	atom_vdw_tables(11, 6, highres) = 13.453;
	atom_vdw_tables(11, 7, highres) = 18.391;
	atom_vdw_tables(11, 8, highres) = 11.989;
	atom_vdw_tables(11, 9, highres) = 16.832;
	atom_vdw_tables(11,10, highres) = 13.766;
	atom_vdw_tables(11,11, highres) = 14.049;
	atom_vdw_tables(11,12, highres) = 11.632;
	atom_vdw_tables(11,13, highres) = 16.820;
	atom_vdw_tables(11,14, highres) = 14.845;
	atom_vdw_tables(11,15, highres) = 13.372;
	atom_vdw_tables(11,16, highres) = 17.733;
	atom_vdw_tables(11,17, highres) = 11.395;
	atom_vdw_tables(11,18, highres) = 12.403;
	atom_vdw_tables(11,19, highres) = 16.403;
	atom_vdw_tables(11,20, highres) = 13.185;
	atom_vdw_tables(11,21, highres) = 12.895;
	atom_vdw_tables(11,22, highres) = 13.388;
	atom_vdw_tables(11,23, highres) = 15.317;
	atom_vdw_tables(11,24, highres) = 21.201;
	atom_vdw_tables(11,25, highres) = 12.878;
	atom_vdw_tables(11,26, highres) = 0.000;
	atom_vdw_tables(11,27, highres) = 0.000;
	atom_vdw_tables(11,28, highres) = 0.000;
	atom_vdw_tables(11,29, highres) = 0.000;
	atom_vdw_tables(11,30, highres) = 0.000;
	atom_vdw_tables(11,31, highres) = 0.000;
	atom_vdw_tables(11,32, highres) = 0.000;
	atom_vdw_tables(11,33, highres) = 0.000;

	// section 2b
	atom_vdw_tables(12, 1, highres) = 10.252;
	atom_vdw_tables(12, 2, highres) = 9.033;
	atom_vdw_tables(12, 3, highres) = 7.630;
	atom_vdw_tables(12, 4, highres) = 8.958;
	atom_vdw_tables(12, 5, highres) = 6.699;
	atom_vdw_tables(12, 6, highres) = 12.724;
	atom_vdw_tables(12, 7, highres) = 24.499;
	atom_vdw_tables(12, 8, highres) = 16.496;
	atom_vdw_tables(12, 9, highres) = 20.591;
	atom_vdw_tables(12,10, highres) = 10.285;
	atom_vdw_tables(12,11, highres) = 11.632;
	atom_vdw_tables(12,12, highres) = 16.412;
	atom_vdw_tables(12,13, highres) = 16.486;
	atom_vdw_tables(12,14, highres) = 21.969;
	atom_vdw_tables(12,15, highres) = 12.214;
	atom_vdw_tables(12,16, highres) = 18.066;
	atom_vdw_tables(12,17, highres) = 15.185;
	atom_vdw_tables(12,18, highres) = 14.138;
	atom_vdw_tables(12,19, highres) = 19.308;
	atom_vdw_tables(12,20, highres) = 13.018;
	atom_vdw_tables(12,21, highres) = 12.508;
	atom_vdw_tables(12,22, highres) = 10.313;
	atom_vdw_tables(12,23, highres) = 13.107;
	atom_vdw_tables(12,24, highres) = 15.974;
	atom_vdw_tables(12,25, highres) = 13.927;
	atom_vdw_tables(12,26, highres) = 0.000;
	atom_vdw_tables(12,27, highres) = 0.000;
	atom_vdw_tables(12,28, highres) = 0.000;
	atom_vdw_tables(12,29, highres) = 0.000;
	atom_vdw_tables(12,30, highres) = 0.000;
	atom_vdw_tables(12,31, highres) = 0.000;
	atom_vdw_tables(12,32, highres) = 0.000;
	atom_vdw_tables(12,33, highres) = 0.000;
	atom_vdw_tables(13, 1, highres) = 14.246;
	atom_vdw_tables(13, 2, highres) = 14.682;
	atom_vdw_tables(13, 3, highres) = 11.959;
	atom_vdw_tables(13, 4, highres) = 13.473;
	atom_vdw_tables(13, 5, highres) = 8.810;
	atom_vdw_tables(13, 6, highres) = 14.252;
	atom_vdw_tables(13, 7, highres) = 18.772;
	atom_vdw_tables(13, 8, highres) = 18.184;
	atom_vdw_tables(13, 9, highres) = 16.409;
	atom_vdw_tables(13,10, highres) = 12.750;
	atom_vdw_tables(13,11, highres) = 16.820;
	atom_vdw_tables(13,12, highres) = 16.486;
	atom_vdw_tables(13,13, highres) = 17.938;
	atom_vdw_tables(13,14, highres) = 15.035;
	atom_vdw_tables(13,15, highres) = 13.297;
	atom_vdw_tables(13,16, highres) = 15.209;
	atom_vdw_tables(13,17, highres) = 13.785;
	atom_vdw_tables(13,18, highres) = 19.701;
	atom_vdw_tables(13,19, highres) = 19.713;
	atom_vdw_tables(13,20, highres) = 14.167;
	atom_vdw_tables(13,21, highres) = 15.026;
	atom_vdw_tables(13,22, highres) = 18.226;
	atom_vdw_tables(13,23, highres) = 13.612;
	atom_vdw_tables(13,24, highres) = 12.191;
	atom_vdw_tables(13,25, highres) = 10.501;
	atom_vdw_tables(13,26, highres) = 0.000;
	atom_vdw_tables(13,27, highres) = 0.000;
	atom_vdw_tables(13,28, highres) = 0.000;
	atom_vdw_tables(13,29, highres) = 0.000;
	atom_vdw_tables(13,30, highres) = 0.000;
	atom_vdw_tables(13,31, highres) = 0.000;
	atom_vdw_tables(13,32, highres) = 0.000;
	atom_vdw_tables(13,33, highres) = 0.000;
	atom_vdw_tables(14, 1, highres) = 10.573;
	atom_vdw_tables(14, 2, highres) = 9.530;
	atom_vdw_tables(14, 3, highres) = 8.014;
	atom_vdw_tables(14, 4, highres) = 8.614;
	atom_vdw_tables(14, 5, highres) = 5.189;
	atom_vdw_tables(14, 6, highres) = 14.845;
	atom_vdw_tables(14, 7, highres) = 30.995;
	atom_vdw_tables(14, 8, highres) = 12.355;
	atom_vdw_tables(14, 9, highres) = 8.798;
	atom_vdw_tables(14,10, highres) = 14.897;
	atom_vdw_tables(14,11, highres) = 14.845;
	atom_vdw_tables(14,12, highres) = 21.969;
	atom_vdw_tables(14,13, highres) = 15.035;
	atom_vdw_tables(14,14, highres) = 17.571;
	atom_vdw_tables(14,15, highres) = 12.518;
	atom_vdw_tables(14,16, highres) = 22.326;
	atom_vdw_tables(14,17, highres) = 15.828;
	atom_vdw_tables(14,18, highres) = 20.675;
	atom_vdw_tables(14,19, highres) = 14.543;
	atom_vdw_tables(14,20, highres) = 28.984;
	atom_vdw_tables(14,21, highres) = 21.558;
	atom_vdw_tables(14,22, highres) = 15.011;
	atom_vdw_tables(14,23, highres) = 13.729;
	atom_vdw_tables(14,24, highres) = 8.752;
	atom_vdw_tables(14,25, highres) = 12.693;
	atom_vdw_tables(14,26, highres) = 0.000;
	atom_vdw_tables(14,27, highres) = 0.000;
	atom_vdw_tables(14,28, highres) = 0.000;
	atom_vdw_tables(14,29, highres) = 0.000;
	atom_vdw_tables(14,30, highres) = 0.000;
	atom_vdw_tables(14,31, highres) = 0.000;
	atom_vdw_tables(14,32, highres) = 0.000;
	atom_vdw_tables(14,33, highres) = 0.000;
	atom_vdw_tables(15, 1, highres) = 11.918;
	atom_vdw_tables(15, 2, highres) = 11.590;
	atom_vdw_tables(15, 3, highres) = 9.848;
	atom_vdw_tables(15, 4, highres) = 12.148;
	atom_vdw_tables(15, 5, highres) = 9.107;
	atom_vdw_tables(15, 6, highres) = 11.513;
	atom_vdw_tables(15, 7, highres) = 15.421;
	atom_vdw_tables(15, 8, highres) = 12.956;
	atom_vdw_tables(15, 9, highres) = 13.833;
	atom_vdw_tables(15,10, highres) = 7.324;
	atom_vdw_tables(15,11, highres) = 13.372;
	atom_vdw_tables(15,12, highres) = 12.214;
	atom_vdw_tables(15,13, highres) = 13.297;
	atom_vdw_tables(15,14, highres) = 12.518;
	atom_vdw_tables(15,15, highres) = 12.130;
	atom_vdw_tables(15,16, highres) = 14.782;
	atom_vdw_tables(15,17, highres) = 14.596;
	atom_vdw_tables(15,18, highres) = 18.342;
	atom_vdw_tables(15,19, highres) = 12.636;
	atom_vdw_tables(15,20, highres) = 8.859;
	atom_vdw_tables(15,21, highres) = 11.876;
	atom_vdw_tables(15,22, highres) = 11.691;
	atom_vdw_tables(15,23, highres) = 11.642;
	atom_vdw_tables(15,24, highres) = 10.834;
	atom_vdw_tables(15,25, highres) = 9.255;
	atom_vdw_tables(15,26, highres) = 0.000;
	atom_vdw_tables(15,27, highres) = 0.000;
	atom_vdw_tables(15,28, highres) = 0.000;
	atom_vdw_tables(15,29, highres) = 0.000;
	atom_vdw_tables(15,30, highres) = 0.000;
	atom_vdw_tables(15,31, highres) = 0.000;
	atom_vdw_tables(15,32, highres) = 0.000;
	atom_vdw_tables(15,33, highres) = 0.000;
	atom_vdw_tables(16, 1, highres) = 11.389;
	atom_vdw_tables(16, 2, highres) = 11.705;
	atom_vdw_tables(16, 3, highres) = 8.512;
	atom_vdw_tables(16, 4, highres) = 10.470;
	atom_vdw_tables(16, 5, highres) = 7.205;
	atom_vdw_tables(16, 6, highres) = 16.654;
	atom_vdw_tables(16, 7, highres) = 19.697;
	atom_vdw_tables(16, 8, highres) = 18.549;
	atom_vdw_tables(16, 9, highres) = 12.168;
	atom_vdw_tables(16,10, highres) = 10.708;
	atom_vdw_tables(16,11, highres) = 17.733;
	atom_vdw_tables(16,12, highres) = 18.066;
	atom_vdw_tables(16,13, highres) = 15.209;
	atom_vdw_tables(16,14, highres) = 22.326;
	atom_vdw_tables(16,15, highres) = 14.782;
	atom_vdw_tables(16,16, highres) = 18.418;
	atom_vdw_tables(16,17, highres) = 22.348;
	atom_vdw_tables(16,18, highres) = 19.218;
	atom_vdw_tables(16,19, highres) = 18.893;
	atom_vdw_tables(16,20, highres) = 24.865;
	atom_vdw_tables(16,21, highres) = 18.080;
	atom_vdw_tables(16,22, highres) = 14.189;
	atom_vdw_tables(16,23, highres) = 13.589;
	atom_vdw_tables(16,24, highres) = 20.580;
	atom_vdw_tables(16,25, highres) = 14.707;
	atom_vdw_tables(16,26, highres) = 0.000;
	atom_vdw_tables(16,27, highres) = 0.000;
	atom_vdw_tables(16,28, highres) = 0.000;
	atom_vdw_tables(16,29, highres) = 0.000;
	atom_vdw_tables(16,30, highres) = 0.000;
	atom_vdw_tables(16,31, highres) = 0.000;
	atom_vdw_tables(16,32, highres) = 0.000;
	atom_vdw_tables(16,33, highres) = 0.000;
	atom_vdw_tables(17, 1, highres) = 10.807;
	atom_vdw_tables(17, 2, highres) = 11.056;
	atom_vdw_tables(17, 3, highres) = 10.582;
	atom_vdw_tables(17, 4, highres) = 10.065;
	atom_vdw_tables(17, 5, highres) = 6.461;
	atom_vdw_tables(17, 6, highres) = 13.566;
	atom_vdw_tables(17, 7, highres) = 16.402;
	atom_vdw_tables(17, 8, highres) = 13.096;
	atom_vdw_tables(17, 9, highres) = 17.815;
	atom_vdw_tables(17,10, highres) = 13.825;
	atom_vdw_tables(17,11, highres) = 11.395;
	atom_vdw_tables(17,12, highres) = 15.185;
	atom_vdw_tables(17,13, highres) = 13.785;
	atom_vdw_tables(17,14, highres) = 15.828;
	atom_vdw_tables(17,15, highres) = 14.596;
	atom_vdw_tables(17,16, highres) = 22.348;
	atom_vdw_tables(17,17, highres) = 19.102;
	atom_vdw_tables(17,18, highres) = 17.165;
	atom_vdw_tables(17,19, highres) = 15.301;
	atom_vdw_tables(17,20, highres) = 13.389;
	atom_vdw_tables(17,21, highres) = 12.651;
	atom_vdw_tables(17,22, highres) = 15.775;
	atom_vdw_tables(17,23, highres) = 14.585;
	atom_vdw_tables(17,24, highres) = 19.169;
	atom_vdw_tables(17,25, highres) = 7.611;
	atom_vdw_tables(17,26, highres) = 0.000;
	atom_vdw_tables(17,27, highres) = 0.000;
	atom_vdw_tables(17,28, highres) = 0.000;
	atom_vdw_tables(17,29, highres) = 0.000;
	atom_vdw_tables(17,30, highres) = 0.000;
	atom_vdw_tables(17,31, highres) = 0.000;
	atom_vdw_tables(17,32, highres) = 0.000;
	atom_vdw_tables(17,33, highres) = 0.000;
	atom_vdw_tables(18, 1, highres) = 12.000;
	atom_vdw_tables(18, 2, highres) = 12.090;
	atom_vdw_tables(18, 3, highres) = 12.525;
	atom_vdw_tables(18, 4, highres) = 8.137;
	atom_vdw_tables(18, 5, highres) = 9.357;
	atom_vdw_tables(18, 6, highres) = 14.798;
	atom_vdw_tables(18, 7, highres) = 22.328;
	atom_vdw_tables(18, 8, highres) = 16.023;
	atom_vdw_tables(18, 9, highres) = 16.878;
	atom_vdw_tables(18,10, highres) = 12.599;
	atom_vdw_tables(18,11, highres) = 12.403;
	atom_vdw_tables(18,12, highres) = 14.138;
	atom_vdw_tables(18,13, highres) = 19.701;
	atom_vdw_tables(18,14, highres) = 20.675;
	atom_vdw_tables(18,15, highres) = 18.342;
	atom_vdw_tables(18,16, highres) = 19.218;
	atom_vdw_tables(18,17, highres) = 17.165;
	atom_vdw_tables(18,18, highres) = 24.158;
	atom_vdw_tables(18,19, highres) = 17.030;
	atom_vdw_tables(18,20, highres) = 19.398;
	atom_vdw_tables(18,21, highres) = 11.809;
	atom_vdw_tables(18,22, highres) = 16.068;
	atom_vdw_tables(18,23, highres) = 17.670;
	atom_vdw_tables(18,24, highres) = 10.106;
	atom_vdw_tables(18,25, highres) = 10.689;
	atom_vdw_tables(18,26, highres) = 0.000;
	atom_vdw_tables(18,27, highres) = 0.000;
	atom_vdw_tables(18,28, highres) = 0.000;
	atom_vdw_tables(18,29, highres) = 0.000;
	atom_vdw_tables(18,30, highres) = 0.000;
	atom_vdw_tables(18,31, highres) = 0.000;
	atom_vdw_tables(18,32, highres) = 0.000;
	atom_vdw_tables(18,33, highres) = 0.000;
	atom_vdw_tables(19, 1, highres) = 8.787;
	atom_vdw_tables(19, 2, highres) = 11.200;
	atom_vdw_tables(19, 3, highres) = 8.142;
	atom_vdw_tables(19, 4, highres) = 9.172;
	atom_vdw_tables(19, 5, highres) = 4.932;
	atom_vdw_tables(19, 6, highres) = 15.631;
	atom_vdw_tables(19, 7, highres) = 23.147;
	atom_vdw_tables(19, 8, highres) = 20.899;
	atom_vdw_tables(19, 9, highres) = 19.535;
	atom_vdw_tables(19,10, highres) = 11.476;
	atom_vdw_tables(19,11, highres) = 16.403;
	atom_vdw_tables(19,12, highres) = 19.308;
	atom_vdw_tables(19,13, highres) = 19.713;
	atom_vdw_tables(19,14, highres) = 14.543;
	atom_vdw_tables(19,15, highres) = 12.636;
	atom_vdw_tables(19,16, highres) = 18.893;
	atom_vdw_tables(19,17, highres) = 15.301;
	atom_vdw_tables(19,18, highres) = 17.030;
	atom_vdw_tables(19,19, highres) = 29.367;
	atom_vdw_tables(19,20, highres) = 20.131;
	atom_vdw_tables(19,21, highres) = 17.077;
	atom_vdw_tables(19,22, highres) = 15.515;
	atom_vdw_tables(19,23, highres) = 12.847;
	atom_vdw_tables(19,24, highres) = 18.930;
	atom_vdw_tables(19,25, highres) = 9.606;
	atom_vdw_tables(19,26, highres) = 0.000;
	atom_vdw_tables(19,27, highres) = 0.000;
	atom_vdw_tables(19,28, highres) = 0.000;
	atom_vdw_tables(19,29, highres) = 0.000;
	atom_vdw_tables(19,30, highres) = 0.000;
	atom_vdw_tables(19,31, highres) = 0.000;
	atom_vdw_tables(19,32, highres) = 0.000;
	atom_vdw_tables(19,33, highres) = 0.000;
	atom_vdw_tables(20, 1, highres) = 8.078;
	atom_vdw_tables(20, 2, highres) = 7.092;
	atom_vdw_tables(20, 3, highres) = 5.563;
	atom_vdw_tables(20, 4, highres) = 6.041;
	atom_vdw_tables(20, 5, highres) = 2.724;
	atom_vdw_tables(20, 6, highres) = 9.847;
	atom_vdw_tables(20, 7, highres) = 21.921;
	atom_vdw_tables(20, 8, highres) = 10.842;
	atom_vdw_tables(20, 9, highres) = 9.903;
	atom_vdw_tables(20,10, highres) = 8.701;
	atom_vdw_tables(20,11, highres) = 13.185;
	atom_vdw_tables(20,12, highres) = 13.018;
	atom_vdw_tables(20,13, highres) = 14.167;
	atom_vdw_tables(20,14, highres) = 28.984;
	atom_vdw_tables(20,15, highres) = 8.859;
	atom_vdw_tables(20,16, highres) = 24.865;
	atom_vdw_tables(20,17, highres) = 13.389;
	atom_vdw_tables(20,18, highres) = 19.398;
	atom_vdw_tables(20,19, highres) = 20.131;
	atom_vdw_tables(20,20, highres) = 23.565;
	atom_vdw_tables(20,21, highres) = 14.734;
	atom_vdw_tables(20,22, highres) = 15.227;
	atom_vdw_tables(20,23, highres) = 11.563;
	atom_vdw_tables(20,24, highres) = 16.984;
	atom_vdw_tables(20,25, highres) = 6.742;
	atom_vdw_tables(20,26, highres) = 0.000;
	atom_vdw_tables(20,27, highres) = 0.000;
	atom_vdw_tables(20,28, highres) = 0.000;
	atom_vdw_tables(20,29, highres) = 0.000;
	atom_vdw_tables(20,30, highres) = 0.000;
	atom_vdw_tables(20,31, highres) = 0.000;
	atom_vdw_tables(20,32, highres) = 0.000;
	atom_vdw_tables(20,33, highres) = 0.000;
	atom_vdw_tables(21, 1, highres) = 9.808;
	atom_vdw_tables(21, 2, highres) = 11.788;
	atom_vdw_tables(21, 3, highres) = 10.335;
	atom_vdw_tables(21, 4, highres) = 10.081;
	atom_vdw_tables(21, 5, highres) = 7.369;
	atom_vdw_tables(21, 6, highres) = 13.792;
	atom_vdw_tables(21, 7, highres) = 17.038;
	atom_vdw_tables(21, 8, highres) = 11.938;
	atom_vdw_tables(21, 9, highres) = 16.777;
	atom_vdw_tables(21,10, highres) = 15.513;
	atom_vdw_tables(21,11, highres) = 12.895;
	atom_vdw_tables(21,12, highres) = 12.508;
	atom_vdw_tables(21,13, highres) = 15.026;
	atom_vdw_tables(21,14, highres) = 21.558;
	atom_vdw_tables(21,15, highres) = 11.876;
	atom_vdw_tables(21,16, highres) = 18.080;
	atom_vdw_tables(21,17, highres) = 12.651;
	atom_vdw_tables(21,18, highres) = 11.809;
	atom_vdw_tables(21,19, highres) = 17.077;
	atom_vdw_tables(21,20, highres) = 14.734;
	atom_vdw_tables(21,21, highres) = 14.778;
	atom_vdw_tables(21,22, highres) = 13.555;
	atom_vdw_tables(21,23, highres) = 14.309;
	atom_vdw_tables(21,24, highres) = 16.396;
	atom_vdw_tables(21,25, highres) = 11.899;
	atom_vdw_tables(21,26, highres) = 0.000;
	atom_vdw_tables(21,27, highres) = 0.000;
	atom_vdw_tables(21,28, highres) = 0.000;
	atom_vdw_tables(21,29, highres) = 0.000;
	atom_vdw_tables(21,30, highres) = 0.000;
	atom_vdw_tables(21,31, highres) = 0.000;
	atom_vdw_tables(21,32, highres) = 0.000;
	atom_vdw_tables(21,33, highres) = 0.000;
	atom_vdw_tables(22, 1, highres) = 10.986;
	atom_vdw_tables(22, 2, highres) = 12.498;
	atom_vdw_tables(22, 3, highres) = 10.859;
	atom_vdw_tables(22, 4, highres) = 11.383;
	atom_vdw_tables(22, 5, highres) = 8.144;
	atom_vdw_tables(22, 6, highres) = 12.931;
	atom_vdw_tables(22, 7, highres) = 17.673;
	atom_vdw_tables(22, 8, highres) = 10.505;
	atom_vdw_tables(22, 9, highres) = 15.167;
	atom_vdw_tables(22,10, highres) = 11.885;
	atom_vdw_tables(22,11, highres) = 13.388;
	atom_vdw_tables(22,12, highres) = 10.313;
	atom_vdw_tables(22,13, highres) = 18.226;
	atom_vdw_tables(22,14, highres) = 15.011;
	atom_vdw_tables(22,15, highres) = 11.691;
	atom_vdw_tables(22,16, highres) = 14.189;
	atom_vdw_tables(22,17, highres) = 15.775;
	atom_vdw_tables(22,18, highres) = 16.068;
	atom_vdw_tables(22,19, highres) = 15.515;
	atom_vdw_tables(22,20, highres) = 15.227;
	atom_vdw_tables(22,21, highres) = 13.555;
	atom_vdw_tables(22,22, highres) = 17.234;
	atom_vdw_tables(22,23, highres) = 13.633;
	atom_vdw_tables(22,24, highres) = 22.182;
	atom_vdw_tables(22,25, highres) = 18.054;
	atom_vdw_tables(22,26, highres) = 0.000;
	atom_vdw_tables(22,27, highres) = 0.000;
	atom_vdw_tables(22,28, highres) = 0.000;
	atom_vdw_tables(22,29, highres) = 0.000;
	atom_vdw_tables(22,30, highres) = 0.000;
	atom_vdw_tables(22,31, highres) = 0.000;
	atom_vdw_tables(22,32, highres) = 0.000;
	atom_vdw_tables(22,33, highres) = 0.000;

	// section 2c
	atom_vdw_tables(23, 1, highres) = 11.163;
	atom_vdw_tables(23, 2, highres) = 12.329;
	atom_vdw_tables(23, 3, highres) = 12.036;
	atom_vdw_tables(23, 4, highres) = 11.936;
	atom_vdw_tables(23, 5, highres) = 9.408;
	atom_vdw_tables(23, 6, highres) = 13.611;
	atom_vdw_tables(23, 7, highres) = 14.368;
	atom_vdw_tables(23, 8, highres) = 15.052;
	atom_vdw_tables(23, 9, highres) = 16.534;
	atom_vdw_tables(23,10, highres) = 9.845;
	atom_vdw_tables(23,11, highres) = 15.317;
	atom_vdw_tables(23,12, highres) = 13.107;
	atom_vdw_tables(23,13, highres) = 13.612;
	atom_vdw_tables(23,14, highres) = 13.729;
	atom_vdw_tables(23,15, highres) = 11.642;
	atom_vdw_tables(23,16, highres) = 13.589;
	atom_vdw_tables(23,17, highres) = 14.585;
	atom_vdw_tables(23,18, highres) = 17.670;
	atom_vdw_tables(23,19, highres) = 12.847;
	atom_vdw_tables(23,20, highres) = 11.563;
	atom_vdw_tables(23,21, highres) = 14.309;
	atom_vdw_tables(23,22, highres) = 13.633;
	atom_vdw_tables(23,23, highres) = 16.439;
	atom_vdw_tables(23,24, highres) = 13.377;
	atom_vdw_tables(23,25, highres) = 9.423;
	atom_vdw_tables(23,26, highres) = 0.000;
	atom_vdw_tables(23,27, highres) = 0.000;
	atom_vdw_tables(23,28, highres) = 0.000;
	atom_vdw_tables(23,29, highres) = 0.000;
	atom_vdw_tables(23,30, highres) = 0.000;
	atom_vdw_tables(23,31, highres) = 0.000;
	atom_vdw_tables(23,32, highres) = 0.000;
	atom_vdw_tables(23,33, highres) = 0.000;
	atom_vdw_tables(24, 1, highres) = 7.904;
	atom_vdw_tables(24, 2, highres) = 8.221;
	atom_vdw_tables(24, 3, highres) = 6.592;
	atom_vdw_tables(24, 4, highres) = 8.508;
	atom_vdw_tables(24, 5, highres) = 6.558;
	atom_vdw_tables(24, 6, highres) = 13.943;
	atom_vdw_tables(24, 7, highres) = 31.079;
	atom_vdw_tables(24, 8, highres) = 18.853;
	atom_vdw_tables(24, 9, highres) = 31.291;
	atom_vdw_tables(24,10, highres) = 11.888;
	atom_vdw_tables(24,11, highres) = 21.201;
	atom_vdw_tables(24,12, highres) = 15.974;
	atom_vdw_tables(24,13, highres) = 12.191;
	atom_vdw_tables(24,14, highres) = 8.752;
	atom_vdw_tables(24,15, highres) = 10.834;
	atom_vdw_tables(24,16, highres) = 20.580;
	atom_vdw_tables(24,17, highres) = 19.169;
	atom_vdw_tables(24,18, highres) = 10.106;
	atom_vdw_tables(24,19, highres) = 18.930;
	atom_vdw_tables(24,20, highres) = 16.984;
	atom_vdw_tables(24,21, highres) = 16.396;
	atom_vdw_tables(24,22, highres) = 22.182;
	atom_vdw_tables(24,23, highres) = 13.377;
	atom_vdw_tables(24,24, highres) = 44.161;
	atom_vdw_tables(24,25, highres) = 12.616;
	atom_vdw_tables(24,26, highres) = 0.000;
	atom_vdw_tables(24,27, highres) = 0.000;
	atom_vdw_tables(24,28, highres) = 0.000;
	atom_vdw_tables(24,29, highres) = 0.000;
	atom_vdw_tables(24,30, highres) = 0.000;
	atom_vdw_tables(24,31, highres) = 0.000;
	atom_vdw_tables(24,32, highres) = 0.000;
	atom_vdw_tables(24,33, highres) = 0.000;
	atom_vdw_tables(25, 1, highres) = 8.805;
	atom_vdw_tables(25, 2, highres) = 6.992;
	atom_vdw_tables(25, 3, highres) = 5.676;
	atom_vdw_tables(25, 4, highres) = 8.177;
	atom_vdw_tables(25, 5, highres) = 5.682;
	atom_vdw_tables(25, 6, highres) = 6.519;
	atom_vdw_tables(25, 7, highres) = 21.162;
	atom_vdw_tables(25, 8, highres) = 16.286;
	atom_vdw_tables(25, 9, highres) = 9.029;
	atom_vdw_tables(25,10, highres) = 12.593;
	atom_vdw_tables(25,11, highres) = 12.878;
	atom_vdw_tables(25,12, highres) = 13.927;
	atom_vdw_tables(25,13, highres) = 10.501;
	atom_vdw_tables(25,14, highres) = 12.693;
	atom_vdw_tables(25,15, highres) = 9.255;
	atom_vdw_tables(25,16, highres) = 14.707;
	atom_vdw_tables(25,17, highres) = 7.611;
	atom_vdw_tables(25,18, highres) = 10.689;
	atom_vdw_tables(25,19, highres) = 9.606;
	atom_vdw_tables(25,20, highres) = 6.742;
	atom_vdw_tables(25,21, highres) = 11.899;
	atom_vdw_tables(25,22, highres) = 18.054;
	atom_vdw_tables(25,23, highres) = 9.423;
	atom_vdw_tables(25,24, highres) = 12.616;
	atom_vdw_tables(25,25, highres) = 13.136;
	atom_vdw_tables(25,26, highres) = 0.000;
	atom_vdw_tables(25,27, highres) = 0.000;
	atom_vdw_tables(25,28, highres) = 0.000;
	atom_vdw_tables(25,29, highres) = 0.000;
	atom_vdw_tables(25,30, highres) = 0.000;
	atom_vdw_tables(25,31, highres) = 0.000;
	atom_vdw_tables(25,32, highres) = 0.000;
	atom_vdw_tables(25,33, highres) = 0.000;
	atom_vdw_tables(26, 1, highres) = 0.000;
	atom_vdw_tables(26, 2, highres) = 0.000;
	atom_vdw_tables(26, 3, highres) = 0.000;
	atom_vdw_tables(26, 4, highres) = 0.000;
	atom_vdw_tables(26, 5, highres) = 0.000;
	atom_vdw_tables(26, 6, highres) = 0.000;
	atom_vdw_tables(26, 7, highres) = 0.000;
	atom_vdw_tables(26, 8, highres) = 0.000;
	atom_vdw_tables(26, 9, highres) = 0.000;
	atom_vdw_tables(26,10, highres) = 0.000;
	atom_vdw_tables(26,11, highres) = 0.000;
	atom_vdw_tables(26,12, highres) = 0.000;
	atom_vdw_tables(26,13, highres) = 0.000;
	atom_vdw_tables(26,14, highres) = 0.000;
	atom_vdw_tables(26,15, highres) = 0.000;
	atom_vdw_tables(26,16, highres) = 0.000;
	atom_vdw_tables(26,17, highres) = 0.000;
	atom_vdw_tables(26,18, highres) = 0.000;
	atom_vdw_tables(26,19, highres) = 0.000;
	atom_vdw_tables(26,20, highres) = 0.000;
	atom_vdw_tables(26,21, highres) = 0.000;
	atom_vdw_tables(26,22, highres) = 0.000;
	atom_vdw_tables(26,23, highres) = 0.000;
	atom_vdw_tables(26,24, highres) = 0.000;
	atom_vdw_tables(26,25, highres) = 0.000;
	atom_vdw_tables(26,26, highres) = 0.000;
	atom_vdw_tables(26,27, highres) = 0.000;
	atom_vdw_tables(26,28, highres) = 0.000;
	atom_vdw_tables(26,29, highres) = 0.000;
	atom_vdw_tables(26,30, highres) = 0.000;
	atom_vdw_tables(26,31, highres) = 0.000;
	atom_vdw_tables(26,32, highres) = 0.000;
	atom_vdw_tables(26,33, highres) = 0.000;
	atom_vdw_tables(27, 1, highres) = 0.000;
	atom_vdw_tables(27, 2, highres) = 0.000;
	atom_vdw_tables(27, 3, highres) = 0.000;
	atom_vdw_tables(27, 4, highres) = 0.000;
	atom_vdw_tables(27, 5, highres) = 0.000;
	atom_vdw_tables(27, 6, highres) = 0.000;
	atom_vdw_tables(27, 7, highres) = 0.000;
	atom_vdw_tables(27, 8, highres) = 0.000;
	atom_vdw_tables(27, 9, highres) = 0.000;
	atom_vdw_tables(27,10, highres) = 0.000;
	atom_vdw_tables(27,11, highres) = 0.000;
	atom_vdw_tables(27,12, highres) = 0.000;
	atom_vdw_tables(27,13, highres) = 0.000;
	atom_vdw_tables(27,14, highres) = 0.000;
	atom_vdw_tables(27,15, highres) = 0.000;
	atom_vdw_tables(27,16, highres) = 0.000;
	atom_vdw_tables(27,17, highres) = 0.000;
	atom_vdw_tables(27,18, highres) = 0.000;
	atom_vdw_tables(27,19, highres) = 0.000;
	atom_vdw_tables(27,20, highres) = 0.000;
	atom_vdw_tables(27,21, highres) = 0.000;
	atom_vdw_tables(27,22, highres) = 0.000;
	atom_vdw_tables(27,23, highres) = 0.000;
	atom_vdw_tables(27,24, highres) = 0.000;
	atom_vdw_tables(27,25, highres) = 0.000;
	atom_vdw_tables(27,26, highres) = 0.000;
	atom_vdw_tables(27,27, highres) = 0.000;
	atom_vdw_tables(27,28, highres) = 0.000;
	atom_vdw_tables(27,29, highres) = 0.000;
	atom_vdw_tables(27,30, highres) = 0.000;
	atom_vdw_tables(27,31, highres) = 0.000;
	atom_vdw_tables(27,32, highres) = 0.000;
	atom_vdw_tables(27,33, highres) = 0.000;
	atom_vdw_tables(28, 1, highres) = 0.000;
	atom_vdw_tables(28, 2, highres) = 0.000;
	atom_vdw_tables(28, 3, highres) = 0.000;
	atom_vdw_tables(28, 4, highres) = 0.000;
	atom_vdw_tables(28, 5, highres) = 0.000;
	atom_vdw_tables(28, 6, highres) = 0.000;
	atom_vdw_tables(28, 7, highres) = 0.000;
	atom_vdw_tables(28, 8, highres) = 0.000;
	atom_vdw_tables(28, 9, highres) = 0.000;
	atom_vdw_tables(28,10, highres) = 0.000;
	atom_vdw_tables(28,11, highres) = 0.000;
	atom_vdw_tables(28,12, highres) = 0.000;
	atom_vdw_tables(28,13, highres) = 0.000;
	atom_vdw_tables(28,14, highres) = 0.000;
	atom_vdw_tables(28,15, highres) = 0.000;
	atom_vdw_tables(28,16, highres) = 0.000;
	atom_vdw_tables(28,17, highres) = 0.000;
	atom_vdw_tables(28,18, highres) = 0.000;
	atom_vdw_tables(28,19, highres) = 0.000;
	atom_vdw_tables(28,20, highres) = 0.000;
	atom_vdw_tables(28,21, highres) = 0.000;
	atom_vdw_tables(28,22, highres) = 0.000;
	atom_vdw_tables(28,23, highres) = 0.000;
	atom_vdw_tables(28,24, highres) = 0.000;
	atom_vdw_tables(28,25, highres) = 0.000;
	atom_vdw_tables(28,26, highres) = 0.000;
	atom_vdw_tables(28,27, highres) = 0.000;
	atom_vdw_tables(28,28, highres) = 0.000;
	atom_vdw_tables(28,29, highres) = 0.000;
	atom_vdw_tables(28,30, highres) = 0.000;
	atom_vdw_tables(28,31, highres) = 0.000;
	atom_vdw_tables(28,32, highres) = 0.000;
	atom_vdw_tables(28,33, highres) = 0.000;
	atom_vdw_tables(29, 1, highres) = 0.000;
	atom_vdw_tables(29, 2, highres) = 0.000;
	atom_vdw_tables(29, 3, highres) = 0.000;
	atom_vdw_tables(29, 4, highres) = 0.000;
	atom_vdw_tables(29, 5, highres) = 0.000;
	atom_vdw_tables(29, 6, highres) = 0.000;
	atom_vdw_tables(29, 7, highres) = 0.000;
	atom_vdw_tables(29, 8, highres) = 0.000;
	atom_vdw_tables(29, 9, highres) = 0.000;
	atom_vdw_tables(29,10, highres) = 0.000;
	atom_vdw_tables(29,11, highres) = 0.000;
	atom_vdw_tables(29,12, highres) = 0.000;
	atom_vdw_tables(29,13, highres) = 0.000;
	atom_vdw_tables(29,14, highres) = 0.000;
	atom_vdw_tables(29,15, highres) = 0.000;
	atom_vdw_tables(29,16, highres) = 0.000;
	atom_vdw_tables(29,17, highres) = 0.000;
	atom_vdw_tables(29,18, highres) = 0.000;
	atom_vdw_tables(29,19, highres) = 0.000;
	atom_vdw_tables(29,20, highres) = 0.000;
	atom_vdw_tables(29,21, highres) = 0.000;
	atom_vdw_tables(29,22, highres) = 0.000;
	atom_vdw_tables(29,23, highres) = 0.000;
	atom_vdw_tables(29,24, highres) = 0.000;
	atom_vdw_tables(29,25, highres) = 0.000;
	atom_vdw_tables(29,26, highres) = 0.000;
	atom_vdw_tables(29,27, highres) = 0.000;
	atom_vdw_tables(29,28, highres) = 0.000;
	atom_vdw_tables(29,29, highres) = 0.000;
	atom_vdw_tables(29,30, highres) = 0.000;
	atom_vdw_tables(29,31, highres) = 0.000;
	atom_vdw_tables(29,32, highres) = 0.000;
	atom_vdw_tables(29,33, highres) = 0.000;
	atom_vdw_tables(30, 1, highres) = 0.000;
	atom_vdw_tables(30, 2, highres) = 0.000;
	atom_vdw_tables(30, 3, highres) = 0.000;
	atom_vdw_tables(30, 4, highres) = 0.000;
	atom_vdw_tables(30, 5, highres) = 0.000;
	atom_vdw_tables(30, 6, highres) = 0.000;
	atom_vdw_tables(30, 7, highres) = 0.000;
	atom_vdw_tables(30, 8, highres) = 0.000;
	atom_vdw_tables(30, 9, highres) = 0.000;
	atom_vdw_tables(30,10, highres) = 0.000;
	atom_vdw_tables(30,11, highres) = 0.000;
	atom_vdw_tables(30,12, highres) = 0.000;
	atom_vdw_tables(30,13, highres) = 0.000;
	atom_vdw_tables(30,14, highres) = 0.000;
	atom_vdw_tables(30,15, highres) = 0.000;
	atom_vdw_tables(30,16, highres) = 0.000;
	atom_vdw_tables(30,17, highres) = 0.000;
	atom_vdw_tables(30,18, highres) = 0.000;
	atom_vdw_tables(30,19, highres) = 0.000;
	atom_vdw_tables(30,20, highres) = 0.000;
	atom_vdw_tables(30,21, highres) = 0.000;
	atom_vdw_tables(30,22, highres) = 0.000;
	atom_vdw_tables(30,23, highres) = 0.000;
	atom_vdw_tables(30,24, highres) = 0.000;
	atom_vdw_tables(30,25, highres) = 0.000;
	atom_vdw_tables(30,26, highres) = 0.000;
	atom_vdw_tables(30,27, highres) = 0.000;
	atom_vdw_tables(30,28, highres) = 0.000;
	atom_vdw_tables(30,29, highres) = 0.000;
	atom_vdw_tables(30,30, highres) = 0.000;
	atom_vdw_tables(30,31, highres) = 0.000;
	atom_vdw_tables(30,32, highres) = 0.000;
	atom_vdw_tables(30,33, highres) = 0.000;
	atom_vdw_tables(31, 1, highres) = 0.000;
	atom_vdw_tables(31, 2, highres) = 0.000;
	atom_vdw_tables(31, 3, highres) = 0.000;
	atom_vdw_tables(31, 4, highres) = 0.000;
	atom_vdw_tables(31, 5, highres) = 0.000;
	atom_vdw_tables(31, 6, highres) = 0.000;
	atom_vdw_tables(31, 7, highres) = 0.000;
	atom_vdw_tables(31, 8, highres) = 0.000;
	atom_vdw_tables(31, 9, highres) = 0.000;
	atom_vdw_tables(31,10, highres) = 0.000;
	atom_vdw_tables(31,11, highres) = 0.000;
	atom_vdw_tables(31,12, highres) = 0.000;
	atom_vdw_tables(31,13, highres) = 0.000;
	atom_vdw_tables(31,14, highres) = 0.000;
	atom_vdw_tables(31,15, highres) = 0.000;
	atom_vdw_tables(31,16, highres) = 0.000;
	atom_vdw_tables(31,17, highres) = 0.000;
	atom_vdw_tables(31,18, highres) = 0.000;
	atom_vdw_tables(31,19, highres) = 0.000;
	atom_vdw_tables(31,20, highres) = 0.000;
	atom_vdw_tables(31,21, highres) = 0.000;
	atom_vdw_tables(31,22, highres) = 0.000;
	atom_vdw_tables(31,23, highres) = 0.000;
	atom_vdw_tables(31,24, highres) = 0.000;
	atom_vdw_tables(31,25, highres) = 0.000;
	atom_vdw_tables(31,26, highres) = 0.000;
	atom_vdw_tables(31,27, highres) = 0.000;
	atom_vdw_tables(31,28, highres) = 0.000;
	atom_vdw_tables(31,29, highres) = 0.000;
	atom_vdw_tables(31,30, highres) = 0.000;
	atom_vdw_tables(31,31, highres) = 0.000;
	atom_vdw_tables(31,32, highres) = 0.000;
	atom_vdw_tables(31,33, highres) = 0.000;
	atom_vdw_tables(32, 1, highres) = 0.000;
	atom_vdw_tables(32, 2, highres) = 0.000;
	atom_vdw_tables(32, 3, highres) = 0.000;
	atom_vdw_tables(32, 4, highres) = 0.000;
	atom_vdw_tables(32, 5, highres) = 0.000;
	atom_vdw_tables(32, 6, highres) = 0.000;
	atom_vdw_tables(32, 7, highres) = 0.000;
	atom_vdw_tables(32, 8, highres) = 0.000;
	atom_vdw_tables(32, 9, highres) = 0.000;
	atom_vdw_tables(32,10, highres) = 0.000;
	atom_vdw_tables(32,11, highres) = 0.000;
	atom_vdw_tables(32,12, highres) = 0.000;
	atom_vdw_tables(32,13, highres) = 0.000;
	atom_vdw_tables(32,14, highres) = 0.000;
	atom_vdw_tables(32,15, highres) = 0.000;
	atom_vdw_tables(32,16, highres) = 0.000;
	atom_vdw_tables(32,17, highres) = 0.000;
	atom_vdw_tables(32,18, highres) = 0.000;
	atom_vdw_tables(32,19, highres) = 0.000;
	atom_vdw_tables(32,20, highres) = 0.000;
	atom_vdw_tables(32,21, highres) = 0.000;
	atom_vdw_tables(32,22, highres) = 0.000;
	atom_vdw_tables(32,23, highres) = 0.000;
	atom_vdw_tables(32,24, highres) = 0.000;
	atom_vdw_tables(32,25, highres) = 0.000;
	atom_vdw_tables(32,26, highres) = 0.000;
	atom_vdw_tables(32,27, highres) = 0.000;
	atom_vdw_tables(32,28, highres) = 0.000;
	atom_vdw_tables(32,29, highres) = 0.000;
	atom_vdw_tables(32,30, highres) = 0.000;
	atom_vdw_tables(32,31, highres) = 0.000;
	atom_vdw_tables(32,32, highres) = 0.000;
	atom_vdw_tables(32,33, highres) = 0.000;
	atom_vdw_tables(33, 1, highres) = 0.000;
	atom_vdw_tables(33, 2, highres) = 0.000;
	atom_vdw_tables(33, 3, highres) = 0.000;
	atom_vdw_tables(33, 4, highres) = 0.000;
	atom_vdw_tables(33, 5, highres) = 0.000;
	atom_vdw_tables(33, 6, highres) = 0.000;
	atom_vdw_tables(33, 7, highres) = 0.000;
	atom_vdw_tables(33, 8, highres) = 0.000;
	atom_vdw_tables(33, 9, highres) = 0.000;
	atom_vdw_tables(33,10, highres) = 0.000;
	atom_vdw_tables(33,11, highres) = 0.000;
	atom_vdw_tables(33,12, highres) = 0.000;
	atom_vdw_tables(33,13, highres) = 0.000;
	atom_vdw_tables(33,14, highres) = 0.000;
	atom_vdw_tables(33,15, highres) = 0.000;
	atom_vdw_tables(33,16, highres) = 0.000;
	atom_vdw_tables(33,17, highres) = 0.000;
	atom_vdw_tables(33,18, highres) = 0.000;
	atom_vdw_tables(33,19, highres) = 0.000;
	atom_vdw_tables(33,20, highres) = 0.000;
	atom_vdw_tables(33,21, highres) = 0.000;
	atom_vdw_tables(33,22, highres) = 0.000;
	atom_vdw_tables(33,23, highres) = 0.000;
	atom_vdw_tables(33,24, highres) = 0.000;
	atom_vdw_tables(33,25, highres) = 0.000;
	atom_vdw_tables(33,26, highres) = 0.000;
	atom_vdw_tables(33,27, highres) = 0.000;
	atom_vdw_tables(33,28, highres) = 0.000;
	atom_vdw_tables(33,29, highres) = 0.000;
	atom_vdw_tables(33,30, highres) = 0.000;
	atom_vdw_tables(33,31, highres) = 0.000;
	atom_vdw_tables(33,32, highres) = 0.000;
	atom_vdw_tables(33,33, highres) = 0.000;
	//KMa phospho_ser
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( i, 34, highres) = 0.000;
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( 34, i, highres) = 0.000;
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( i, 26, highres) = atom_vdw_tables( i, 21, highres);
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( 26, i, highres) = atom_vdw_tables( 21, i, highres);
	}

	// section 3a
	  //--------------------------------------------------
	//     5/16/02
	//     atom-atom distances from the highres xray structures, identical
	//          to highres radii
	//     atom-centroid and centroid-centroid distances from 2A or
	//          better xray structures (without counting i,i+1 contacts)
	//          to increase counts
	//--------------------------------------------------
	atom_vdw_tables( 1, 1, hybrid) = 12.417;
	atom_vdw_tables( 1, 2, hybrid) = 11.976;
	atom_vdw_tables( 1, 3, hybrid) = 11.859;
	atom_vdw_tables( 1, 4, hybrid) = 8.464;
	atom_vdw_tables( 1, 5, hybrid) = 6.853;
	atom_vdw_tables( 1, 6, hybrid) = 7.258;
	atom_vdw_tables( 1, 7, hybrid) = 9.025;
	atom_vdw_tables( 1, 8, hybrid) = 7.751;
	atom_vdw_tables( 1, 9, hybrid) = 7.023;
	atom_vdw_tables( 1,10, hybrid) = 6.661;
	atom_vdw_tables( 1,11, hybrid) = 5.304;
	atom_vdw_tables( 1,12, hybrid) = 7.558;
	atom_vdw_tables( 1,13, hybrid) = 11.877;
	atom_vdw_tables( 1,14, hybrid) = 6.455;
	atom_vdw_tables( 1,15, hybrid) = 9.204;
	atom_vdw_tables( 1,16, hybrid) = 8.112;
	atom_vdw_tables( 1,17, hybrid) = 8.499;
	atom_vdw_tables( 1,18, hybrid) = 10.968;
	atom_vdw_tables( 1,19, hybrid) = 7.721;
	atom_vdw_tables( 1,20, hybrid) = 5.740;
	atom_vdw_tables( 1,21, hybrid) = 8.468;
	atom_vdw_tables( 1,22, hybrid) = 10.170;
	atom_vdw_tables( 1,23, hybrid) = 10.647;
	atom_vdw_tables( 1,24, hybrid) = 6.279;
	atom_vdw_tables( 1,25, hybrid) = 5.542;
	atom_vdw_tables( 1,26, hybrid) = 0.000;
	atom_vdw_tables( 1,27, hybrid) = 0.000;
	atom_vdw_tables( 1,28, hybrid) = 0.000;
	atom_vdw_tables( 1,29, hybrid) = 0.000;
	atom_vdw_tables( 1,30, hybrid) = 0.000;
	atom_vdw_tables( 1,31, hybrid) = 0.000;
	atom_vdw_tables( 1,32, hybrid) = 0.000;
	atom_vdw_tables( 1,33, hybrid) = 0.000;
	atom_vdw_tables( 2, 1, hybrid) = 11.976;
	atom_vdw_tables( 2, 2, hybrid) = 13.744;
	atom_vdw_tables( 2, 3, hybrid) = 12.249;
	atom_vdw_tables( 2, 4, hybrid) = 10.464;
	atom_vdw_tables( 2, 5, hybrid) = 8.564;
	atom_vdw_tables( 2, 6, hybrid) = 11.382;
	atom_vdw_tables( 2, 7, hybrid) = 8.363;
	atom_vdw_tables( 2, 8, hybrid) = 8.524;
	atom_vdw_tables( 2, 9, hybrid) = 6.918;
	atom_vdw_tables( 2,10, hybrid) = 6.481;
	atom_vdw_tables( 2,11, hybrid) = 8.252;
	atom_vdw_tables( 2,12, hybrid) = 7.013;
	atom_vdw_tables( 2,13, hybrid) = 11.793;
	atom_vdw_tables( 2,14, hybrid) = 5.852;
	atom_vdw_tables( 2,15, hybrid) = 8.778;
	atom_vdw_tables( 2,16, hybrid) = 8.098;
	atom_vdw_tables( 2,17, hybrid) = 8.390;
	atom_vdw_tables( 2,18, hybrid) = 10.723;
	atom_vdw_tables( 2,19, hybrid) = 7.518;
	atom_vdw_tables( 2,20, hybrid) = 4.681;
	atom_vdw_tables( 2,21, hybrid) = 9.730;
	atom_vdw_tables( 2,22, hybrid) = 11.122;
	atom_vdw_tables( 2,23, hybrid) = 11.436;
	atom_vdw_tables( 2,24, hybrid) = 5.832;
	atom_vdw_tables( 2,25, hybrid) = 5.597;
	atom_vdw_tables( 2,26, hybrid) = 0.000;
	atom_vdw_tables( 2,27, hybrid) = 0.000;
	atom_vdw_tables( 2,28, hybrid) = 0.000;
	atom_vdw_tables( 2,29, hybrid) = 0.000;
	atom_vdw_tables( 2,30, hybrid) = 0.000;
	atom_vdw_tables( 2,31, hybrid) = 0.000;
	atom_vdw_tables( 2,32, hybrid) = 0.000;
	atom_vdw_tables( 2,33, hybrid) = 0.000;
	atom_vdw_tables( 3, 1, hybrid) = 11.859;
	atom_vdw_tables( 3, 2, hybrid) = 12.249;
	atom_vdw_tables( 3, 3, hybrid) = 12.024;
	atom_vdw_tables( 3, 4, hybrid) = 11.663;
	atom_vdw_tables( 3, 5, hybrid) = 8.739;
	atom_vdw_tables( 3, 6, hybrid) = 11.151;
	atom_vdw_tables( 3, 7, hybrid) = 7.422;
	atom_vdw_tables( 3, 8, hybrid) = 7.991;
	atom_vdw_tables( 3, 9, hybrid) = 5.855;
	atom_vdw_tables( 3,10, hybrid) = 5.451;
	atom_vdw_tables( 3,11, hybrid) = 11.245;
	atom_vdw_tables( 3,12, hybrid) = 6.440;
	atom_vdw_tables( 3,13, hybrid) = 9.628;
	atom_vdw_tables( 3,14, hybrid) = 5.018;
	atom_vdw_tables( 3,15, hybrid) = 7.305;
	atom_vdw_tables( 3,16, hybrid) = 6.392;
	atom_vdw_tables( 3,17, hybrid) = 7.954;
	atom_vdw_tables( 3,18, hybrid) = 10.892;
	atom_vdw_tables( 3,19, hybrid) = 6.281;
	atom_vdw_tables( 3,20, hybrid) = 3.945;
	atom_vdw_tables( 3,21, hybrid) = 8.752;
	atom_vdw_tables( 3,22, hybrid) = 10.176;
	atom_vdw_tables( 3,23, hybrid) = 10.028;
	atom_vdw_tables( 3,24, hybrid) = 5.128;
	atom_vdw_tables( 3,25, hybrid) = 4.649;
	atom_vdw_tables( 3,26, hybrid) = 0.000;
	atom_vdw_tables( 3,27, hybrid) = 0.000;
	atom_vdw_tables( 3,28, hybrid) = 0.000;
	atom_vdw_tables( 3,29, hybrid) = 0.000;
	atom_vdw_tables( 3,30, hybrid) = 0.000;
	atom_vdw_tables( 3,31, hybrid) = 0.000;
	atom_vdw_tables( 3,32, hybrid) = 0.000;
	atom_vdw_tables( 3,33, hybrid) = 0.000;
	atom_vdw_tables( 4, 1, hybrid) = 8.464;
	atom_vdw_tables( 4, 2, hybrid) = 10.464;
	atom_vdw_tables( 4, 3, hybrid) = 11.663;
	atom_vdw_tables( 4, 4, hybrid) = 13.674;
	atom_vdw_tables( 4, 5, hybrid) = 8.511;
	atom_vdw_tables( 4, 6, hybrid) = 8.394;
	atom_vdw_tables( 4, 7, hybrid) = 9.022;
	atom_vdw_tables( 4, 8, hybrid) = 8.959;
	atom_vdw_tables( 4, 9, hybrid) = 7.018;
	atom_vdw_tables( 4,10, hybrid) = 6.567;
	atom_vdw_tables( 4,11, hybrid) = 5.242;
	atom_vdw_tables( 4,12, hybrid) = 7.316;
	atom_vdw_tables( 4,13, hybrid) = 11.428;
	atom_vdw_tables( 4,14, hybrid) = 5.917;
	atom_vdw_tables( 4,15, hybrid) = 8.660;
	atom_vdw_tables( 4,16, hybrid) = 7.820;
	atom_vdw_tables( 4,17, hybrid) = 7.939;
	atom_vdw_tables( 4,18, hybrid) = 7.398;
	atom_vdw_tables( 4,19, hybrid) = 7.718;
	atom_vdw_tables( 4,20, hybrid) = 4.692;
	atom_vdw_tables( 4,21, hybrid) = 9.001;
	atom_vdw_tables( 4,22, hybrid) = 10.543;
	atom_vdw_tables( 4,23, hybrid) = 10.964;
	atom_vdw_tables( 4,24, hybrid) = 6.134;
	atom_vdw_tables( 4,25, hybrid) = 5.639;
	atom_vdw_tables( 4,26, hybrid) = 0.000;
	atom_vdw_tables( 4,27, hybrid) = 0.000;
	atom_vdw_tables( 4,28, hybrid) = 0.000;
	atom_vdw_tables( 4,29, hybrid) = 0.000;
	atom_vdw_tables( 4,30, hybrid) = 0.000;
	atom_vdw_tables( 4,31, hybrid) = 0.000;
	atom_vdw_tables( 4,32, hybrid) = 0.000;
	atom_vdw_tables( 4,33, hybrid) = 0.000;
	atom_vdw_tables( 5, 1, hybrid) = 6.853;
	atom_vdw_tables( 5, 2, hybrid) = 8.564;
	atom_vdw_tables( 5, 3, hybrid) = 8.739;
	atom_vdw_tables( 5, 4, hybrid) = 8.511;
	atom_vdw_tables( 5, 5, hybrid) = 9.100; // 8.83 10/22/03
	atom_vdw_tables( 5, 6, hybrid) = 7.332;
	atom_vdw_tables( 5, 7, hybrid) = 6.560;
	atom_vdw_tables( 5, 8, hybrid) = 5.961;
	atom_vdw_tables( 5, 9, hybrid) = 4.542;
	atom_vdw_tables( 5,10, hybrid) = 4.485;
	atom_vdw_tables( 5,11, hybrid) = 6.181;
	atom_vdw_tables( 5,12, hybrid) = 5.022;
	atom_vdw_tables( 5,13, hybrid) = 8.160;
	atom_vdw_tables( 5,14, hybrid) = 3.264;
	atom_vdw_tables( 5,15, hybrid) = 5.767;
	atom_vdw_tables( 5,16, hybrid) = 4.724;
	atom_vdw_tables( 5,17, hybrid) = 5.142;
	atom_vdw_tables( 5,18, hybrid) = 7.413;
	atom_vdw_tables( 5,19, hybrid) = 4.986;
	atom_vdw_tables( 5,20, hybrid) = 2.106;
	atom_vdw_tables( 5,21, hybrid) = 6.309;
	atom_vdw_tables( 5,22, hybrid) = 7.349;
	atom_vdw_tables( 5,23, hybrid) = 8.198;
	atom_vdw_tables( 5,24, hybrid) = 4.267;
	atom_vdw_tables( 5,25, hybrid) = 3.485;
	atom_vdw_tables( 5,26, hybrid) = 0.000;
	atom_vdw_tables( 5,27, hybrid) = 0.000;
	atom_vdw_tables( 5,28, hybrid) = 0.000;
	atom_vdw_tables( 5,29, hybrid) = 0.000;
	atom_vdw_tables( 5,30, hybrid) = 0.000;
	atom_vdw_tables( 5,31, hybrid) = 0.000;
	atom_vdw_tables( 5,32, hybrid) = 0.000;
	atom_vdw_tables( 5,33, hybrid) = 0.000;
	atom_vdw_tables( 6, 1, hybrid) = 7.258;
	atom_vdw_tables( 6, 2, hybrid) = 11.382;
	atom_vdw_tables( 6, 3, hybrid) = 11.151;
	atom_vdw_tables( 6, 4, hybrid) = 8.394;
	atom_vdw_tables( 6, 5, hybrid) = 7.332;
	atom_vdw_tables( 6, 6, hybrid) = 12.353;
	atom_vdw_tables( 6, 7, hybrid) = 10.279;
	atom_vdw_tables( 6, 8, hybrid) = 9.361;
	atom_vdw_tables( 6, 9, hybrid) = 7.346;
	atom_vdw_tables( 6,10, hybrid) = 6.619;
	atom_vdw_tables( 6,11, hybrid) = 11.934;
	atom_vdw_tables( 6,12, hybrid) = 8.216;
	atom_vdw_tables( 6,13, hybrid) = 11.061;
	atom_vdw_tables( 6,14, hybrid) = 7.559;
	atom_vdw_tables( 6,15, hybrid) = 8.332;
	atom_vdw_tables( 6,16, hybrid) = 7.952;
	atom_vdw_tables( 6,17, hybrid) = 10.006;
	atom_vdw_tables( 6,18, hybrid) = 12.854;
	atom_vdw_tables( 6,19, hybrid) = 8.597;
	atom_vdw_tables( 6,20, hybrid) = 6.035;
	atom_vdw_tables( 6,21, hybrid) = 10.044;
	atom_vdw_tables( 6,22, hybrid) = 11.051;
	atom_vdw_tables( 6,23, hybrid) = 11.027;
	atom_vdw_tables( 6,24, hybrid) = 6.942;
	atom_vdw_tables( 6,25, hybrid) = 6.143;
	atom_vdw_tables( 6,26, hybrid) = 0.000;
	atom_vdw_tables( 6,27, hybrid) = 0.000;
	atom_vdw_tables( 6,28, hybrid) = 0.000;
	atom_vdw_tables( 6,29, hybrid) = 0.000;
	atom_vdw_tables( 6,30, hybrid) = 0.000;
	atom_vdw_tables( 6,31, hybrid) = 0.000;
	atom_vdw_tables( 6,32, hybrid) = 0.000;
	atom_vdw_tables( 6,33, hybrid) = 0.000;
	atom_vdw_tables( 7, 1, hybrid) = 9.025;
	atom_vdw_tables( 7, 2, hybrid) = 8.363;
	atom_vdw_tables( 7, 3, hybrid) = 7.422;
	atom_vdw_tables( 7, 4, hybrid) = 9.022;
	atom_vdw_tables( 7, 5, hybrid) = 6.560;
	atom_vdw_tables( 7, 6, hybrid) = 10.279;
	atom_vdw_tables( 7, 7, hybrid) = 4.310;
	atom_vdw_tables( 7, 8, hybrid) = 11.907;
	atom_vdw_tables( 7, 9, hybrid) = 10.622;
	atom_vdw_tables( 7,10, hybrid) = 7.294;
	atom_vdw_tables( 7,11, hybrid) = 10.536;
	atom_vdw_tables( 7,12, hybrid) = 13.235;
	atom_vdw_tables( 7,13, hybrid) = 11.812;
	atom_vdw_tables( 7,14, hybrid) = 13.499;
	atom_vdw_tables( 7,15, hybrid) = 9.427;
	atom_vdw_tables( 7,16, hybrid) = 10.310;
	atom_vdw_tables( 7,17, hybrid) = 11.879;
	atom_vdw_tables( 7,18, hybrid) = 13.546;
	atom_vdw_tables( 7,19, hybrid) = 12.350;
	atom_vdw_tables( 7,20, hybrid) = 9.945;
	atom_vdw_tables( 7,21, hybrid) = 11.385;
	atom_vdw_tables( 7,22, hybrid) = 12.386;
	atom_vdw_tables( 7,23, hybrid) = 11.659;
	atom_vdw_tables( 7,24, hybrid) = 11.396;
	atom_vdw_tables( 7,25, hybrid) = 8.389;
	atom_vdw_tables( 7,26, hybrid) = 0.000;
	atom_vdw_tables( 7,27, hybrid) = 0.000;
	atom_vdw_tables( 7,28, hybrid) = 0.000;
	atom_vdw_tables( 7,29, hybrid) = 0.000;
	atom_vdw_tables( 7,30, hybrid) = 0.000;
	atom_vdw_tables( 7,31, hybrid) = 0.000;
	atom_vdw_tables( 7,32, hybrid) = 0.000;
	atom_vdw_tables( 7,33, hybrid) = 0.000;
	atom_vdw_tables( 8, 1, hybrid) = 7.751;
	atom_vdw_tables( 8, 2, hybrid) = 8.524;
	atom_vdw_tables( 8, 3, hybrid) = 7.991;
	atom_vdw_tables( 8, 4, hybrid) = 8.959;
	atom_vdw_tables( 8, 5, hybrid) = 5.961;
	atom_vdw_tables( 8, 6, hybrid) = 9.361;
	atom_vdw_tables( 8, 7, hybrid) = 11.907;
	atom_vdw_tables( 8, 8, hybrid) = 11.843;
	atom_vdw_tables( 8, 9, hybrid) = 9.671;
	atom_vdw_tables( 8,10, hybrid) = 6.965;
	atom_vdw_tables( 8,11, hybrid) = 9.447;
	atom_vdw_tables( 8,12, hybrid) = 8.645;
	atom_vdw_tables( 8,13, hybrid) = 10.323;
	atom_vdw_tables( 8,14, hybrid) = 6.005;
	atom_vdw_tables( 8,15, hybrid) = 8.173;
	atom_vdw_tables( 8,16, hybrid) = 8.385;
	atom_vdw_tables( 8,17, hybrid) = 8.726;
	atom_vdw_tables( 8,18, hybrid) = 10.644;
	atom_vdw_tables( 8,19, hybrid) = 9.143;
	atom_vdw_tables( 8,20, hybrid) = 6.293;
	atom_vdw_tables( 8,21, hybrid) = 8.244;
	atom_vdw_tables( 8,22, hybrid) = 9.086;
	atom_vdw_tables( 8,23, hybrid) = 10.664;
	atom_vdw_tables( 8,24, hybrid) = 10.085;
	atom_vdw_tables( 8,25, hybrid) = 6.947;
	atom_vdw_tables( 8,26, hybrid) = 0.000;
	atom_vdw_tables( 8,27, hybrid) = 0.000;
	atom_vdw_tables( 8,28, hybrid) = 0.000;
	atom_vdw_tables( 8,29, hybrid) = 0.000;
	atom_vdw_tables( 8,30, hybrid) = 0.000;
	atom_vdw_tables( 8,31, hybrid) = 0.000;
	atom_vdw_tables( 8,32, hybrid) = 0.000;
	atom_vdw_tables( 8,33, hybrid) = 0.000;
	atom_vdw_tables( 9, 1, hybrid) = 7.023;
	atom_vdw_tables( 9, 2, hybrid) = 6.918;
	atom_vdw_tables( 9, 3, hybrid) = 5.855;
	atom_vdw_tables( 9, 4, hybrid) = 7.018;
	atom_vdw_tables( 9, 5, hybrid) = 4.542;
	atom_vdw_tables( 9, 6, hybrid) = 7.346;
	atom_vdw_tables( 9, 7, hybrid) = 10.622;
	atom_vdw_tables( 9, 8, hybrid) = 9.671;
	atom_vdw_tables( 9, 9, hybrid) = 11.338;
	atom_vdw_tables( 9,10, hybrid) = 5.551;
	atom_vdw_tables( 9,11, hybrid) = 9.012;
	atom_vdw_tables( 9,12, hybrid) = 6.903;
	atom_vdw_tables( 9,13, hybrid) = 8.790;
	atom_vdw_tables( 9,14, hybrid) = 5.582;
	atom_vdw_tables( 9,15, hybrid) = 8.255;
	atom_vdw_tables( 9,16, hybrid) = 8.770;
	atom_vdw_tables( 9,17, hybrid) = 8.165;
	atom_vdw_tables( 9,18, hybrid) = 8.952;
	atom_vdw_tables( 9,19, hybrid) = 7.976;
	atom_vdw_tables( 9,20, hybrid) = 5.228;
	atom_vdw_tables( 9,21, hybrid) = 7.283;
	atom_vdw_tables( 9,22, hybrid) = 8.589;
	atom_vdw_tables( 9,23, hybrid) = 8.943;
	atom_vdw_tables( 9,24, hybrid) = 8.278;
	atom_vdw_tables( 9,25, hybrid) = 5.941;
	atom_vdw_tables( 9,26, hybrid) = 0.000;
	atom_vdw_tables( 9,27, hybrid) = 0.000;
	atom_vdw_tables( 9,28, hybrid) = 0.000;
	atom_vdw_tables( 9,29, hybrid) = 0.000;
	atom_vdw_tables( 9,30, hybrid) = 0.000;
	atom_vdw_tables( 9,31, hybrid) = 0.000;
	atom_vdw_tables( 9,32, hybrid) = 0.000;
	atom_vdw_tables( 9,33, hybrid) = 0.000;
	atom_vdw_tables(10, 1, hybrid) = 6.661;
	atom_vdw_tables(10, 2, hybrid) = 6.481;
	atom_vdw_tables(10, 3, hybrid) = 5.451;
	atom_vdw_tables(10, 4, hybrid) = 6.567;
	atom_vdw_tables(10, 5, hybrid) = 4.485;
	atom_vdw_tables(10, 6, hybrid) = 6.619;
	atom_vdw_tables(10, 7, hybrid) = 7.294;
	atom_vdw_tables(10, 8, hybrid) = 6.965;
	atom_vdw_tables(10, 9, hybrid) = 5.551;
	atom_vdw_tables(10,10, hybrid) = 4.640;
	atom_vdw_tables(10,11, hybrid) = 7.662;
	atom_vdw_tables(10,12, hybrid) = 5.658;
	atom_vdw_tables(10,13, hybrid) = 6.930;
	atom_vdw_tables(10,14, hybrid) = 5.874;
	atom_vdw_tables(10,15, hybrid) = 4.309;
	atom_vdw_tables(10,16, hybrid) = 5.564;
	atom_vdw_tables(10,17, hybrid) = 7.149;
	atom_vdw_tables(10,18, hybrid) = 7.807;
	atom_vdw_tables(10,19, hybrid) = 5.676;
	atom_vdw_tables(10,20, hybrid) = 4.448;
	atom_vdw_tables(10,21, hybrid) = 7.120;
	atom_vdw_tables(10,22, hybrid) = 8.059;
	atom_vdw_tables(10,23, hybrid) = 7.058;
	atom_vdw_tables(10,24, hybrid) = 4.767;
	atom_vdw_tables(10,25, hybrid) = 3.435;
	atom_vdw_tables(10,26, hybrid) = 0.000;
	atom_vdw_tables(10,27, hybrid) = 0.000;
	atom_vdw_tables(10,28, hybrid) = 0.000;
	atom_vdw_tables(10,29, hybrid) = 0.000;
	atom_vdw_tables(10,30, hybrid) = 0.000;
	atom_vdw_tables(10,31, hybrid) = 0.000;
	atom_vdw_tables(10,32, hybrid) = 0.000;
	atom_vdw_tables(10,33, hybrid) = 0.000;
	atom_vdw_tables(11, 1, hybrid) = 5.304;
	atom_vdw_tables(11, 2, hybrid) = 8.252;
	atom_vdw_tables(11, 3, hybrid) = 11.245;
	atom_vdw_tables(11, 4, hybrid) = 5.242;
	atom_vdw_tables(11, 5, hybrid) = 6.181;
	atom_vdw_tables(11, 6, hybrid) = 11.934;
	atom_vdw_tables(11, 7, hybrid) = 10.536;
	atom_vdw_tables(11, 8, hybrid) = 9.447;
	atom_vdw_tables(11, 9, hybrid) = 9.012;
	atom_vdw_tables(11,10, hybrid) = 7.662;
	atom_vdw_tables(11,11, hybrid) = 12.897;
	atom_vdw_tables(11,12, hybrid) = 8.735;
	atom_vdw_tables(11,13, hybrid) = 12.712;
	atom_vdw_tables(11,14, hybrid) = 8.280;
	atom_vdw_tables(11,15, hybrid) = 10.293;
	atom_vdw_tables(11,16, hybrid) = 9.830;
	atom_vdw_tables(11,17, hybrid) = 9.745;
	atom_vdw_tables(11,18, hybrid) = 12.661;
	atom_vdw_tables(11,19, hybrid) = 9.648;
	atom_vdw_tables(11,20, hybrid) = 7.146;
	atom_vdw_tables(11,21, hybrid) = 10.432;
	atom_vdw_tables(11,22, hybrid) = 11.548;
	atom_vdw_tables(11,23, hybrid) = 12.516;
	atom_vdw_tables(11,24, hybrid) = 8.347;
	atom_vdw_tables(11,25, hybrid) = 6.967;
	atom_vdw_tables(11,26, hybrid) = 0.000;
	atom_vdw_tables(11,27, hybrid) = 0.000;
	atom_vdw_tables(11,28, hybrid) = 0.000;
	atom_vdw_tables(11,29, hybrid) = 0.000;
	atom_vdw_tables(11,30, hybrid) = 0.000;
	atom_vdw_tables(11,31, hybrid) = 0.000;
	atom_vdw_tables(11,32, hybrid) = 0.000;
	atom_vdw_tables(11,33, hybrid) = 0.000;

	// section 3b
	atom_vdw_tables(12, 1, hybrid) = 7.558;
	atom_vdw_tables(12, 2, hybrid) = 7.013;
	atom_vdw_tables(12, 3, hybrid) = 6.440;
	atom_vdw_tables(12, 4, hybrid) = 7.316;
	atom_vdw_tables(12, 5, hybrid) = 5.022;
	atom_vdw_tables(12, 6, hybrid) = 8.216;
	atom_vdw_tables(12, 7, hybrid) = 13.235;
	atom_vdw_tables(12, 8, hybrid) = 8.645;
	atom_vdw_tables(12, 9, hybrid) = 6.903;
	atom_vdw_tables(12,10, hybrid) = 5.658;
	atom_vdw_tables(12,11, hybrid) = 8.735;
	atom_vdw_tables(12,12, hybrid) = 9.466;
	atom_vdw_tables(12,13, hybrid) = 9.737;
	atom_vdw_tables(12,14, hybrid) = 9.387;
	atom_vdw_tables(12,15, hybrid) = 7.817;
	atom_vdw_tables(12,16, hybrid) = 9.914;
	atom_vdw_tables(12,17, hybrid) = 9.032;
	atom_vdw_tables(12,18, hybrid) = 9.500;
	atom_vdw_tables(12,19, hybrid) = 9.323;
	atom_vdw_tables(12,20, hybrid) = 6.330;
	atom_vdw_tables(12,21, hybrid) = 8.787;
	atom_vdw_tables(12,22, hybrid) = 9.324;
	atom_vdw_tables(12,23, hybrid) = 8.597;
	atom_vdw_tables(12,24, hybrid) = 7.220;
	atom_vdw_tables(12,25, hybrid) = 5.627;
	atom_vdw_tables(12,26, hybrid) = 0.000;
	atom_vdw_tables(12,27, hybrid) = 0.000;
	atom_vdw_tables(12,28, hybrid) = 0.000;
	atom_vdw_tables(12,29, hybrid) = 0.000;
	atom_vdw_tables(12,30, hybrid) = 0.000;
	atom_vdw_tables(12,31, hybrid) = 0.000;
	atom_vdw_tables(12,32, hybrid) = 0.000;
	atom_vdw_tables(12,33, hybrid) = 0.000;
	atom_vdw_tables(13, 1, hybrid) = 11.877;
	atom_vdw_tables(13, 2, hybrid) = 11.793;
	atom_vdw_tables(13, 3, hybrid) = 9.628;
	atom_vdw_tables(13, 4, hybrid) = 11.428;
	atom_vdw_tables(13, 5, hybrid) = 8.160;
	atom_vdw_tables(13, 6, hybrid) = 11.061;
	atom_vdw_tables(13, 7, hybrid) = 11.812;
	atom_vdw_tables(13, 8, hybrid) = 10.323;
	atom_vdw_tables(13, 9, hybrid) = 8.790;
	atom_vdw_tables(13,10, hybrid) = 6.930;
	atom_vdw_tables(13,11, hybrid) = 12.712;
	atom_vdw_tables(13,12, hybrid) = 9.737;
	atom_vdw_tables(13,13, hybrid) = 13.459;
	atom_vdw_tables(13,14, hybrid) = 9.063;
	atom_vdw_tables(13,15, hybrid) = 9.668;
	atom_vdw_tables(13,16, hybrid) = 10.217;
	atom_vdw_tables(13,17, hybrid) = 11.623;
	atom_vdw_tables(13,18, hybrid) = 13.740;
	atom_vdw_tables(13,19, hybrid) = 10.077;
	atom_vdw_tables(13,20, hybrid) = 8.090;
	atom_vdw_tables(13,21, hybrid) = 11.311;
	atom_vdw_tables(13,22, hybrid) = 12.891;
	atom_vdw_tables(13,23, hybrid) = 12.102;
	atom_vdw_tables(13,24, hybrid) = 8.123;
	atom_vdw_tables(13,25, hybrid) = 6.972;
	atom_vdw_tables(13,26, hybrid) = 0.000;
	atom_vdw_tables(13,27, hybrid) = 0.000;
	atom_vdw_tables(13,28, hybrid) = 0.000;
	atom_vdw_tables(13,29, hybrid) = 0.000;
	atom_vdw_tables(13,30, hybrid) = 0.000;
	atom_vdw_tables(13,31, hybrid) = 0.000;
	atom_vdw_tables(13,32, hybrid) = 0.000;
	atom_vdw_tables(13,33, hybrid) = 0.000;
	atom_vdw_tables(14, 1, hybrid) = 6.455;
	atom_vdw_tables(14, 2, hybrid) = 5.852;
	atom_vdw_tables(14, 3, hybrid) = 5.018;
	atom_vdw_tables(14, 4, hybrid) = 5.917;
	atom_vdw_tables(14, 5, hybrid) = 3.264;
	atom_vdw_tables(14, 6, hybrid) = 7.559;
	atom_vdw_tables(14, 7, hybrid) = 13.499;
	atom_vdw_tables(14, 8, hybrid) = 6.005;
	atom_vdw_tables(14, 9, hybrid) = 5.582;
	atom_vdw_tables(14,10, hybrid) = 5.874;
	atom_vdw_tables(14,11, hybrid) = 8.280;
	atom_vdw_tables(14,12, hybrid) = 9.387;
	atom_vdw_tables(14,13, hybrid) = 9.063;
	atom_vdw_tables(14,14, hybrid) = 11.732;
	atom_vdw_tables(14,15, hybrid) = 7.382;
	atom_vdw_tables(14,16, hybrid) = 9.005;
	atom_vdw_tables(14,17, hybrid) = 7.947;
	atom_vdw_tables(14,18, hybrid) = 10.446;
	atom_vdw_tables(14,19, hybrid) = 8.393;
	atom_vdw_tables(14,20, hybrid) = 9.902;
	atom_vdw_tables(14,21, hybrid) = 7.184;
	atom_vdw_tables(14,22, hybrid) = 9.175;
	atom_vdw_tables(14,23, hybrid) = 8.827;
	atom_vdw_tables(14,24, hybrid) = 5.166;
	atom_vdw_tables(14,25, hybrid) = 4.582;
	atom_vdw_tables(14,26, hybrid) = 0.000;
	atom_vdw_tables(14,27, hybrid) = 0.000;
	atom_vdw_tables(14,28, hybrid) = 0.000;
	atom_vdw_tables(14,29, hybrid) = 0.000;
	atom_vdw_tables(14,30, hybrid) = 0.000;
	atom_vdw_tables(14,31, hybrid) = 0.000;
	atom_vdw_tables(14,32, hybrid) = 0.000;
	atom_vdw_tables(14,33, hybrid) = 0.000;
	atom_vdw_tables(15, 1, hybrid) = 9.204;
	atom_vdw_tables(15, 2, hybrid) = 8.778;
	atom_vdw_tables(15, 3, hybrid) = 7.305;
	atom_vdw_tables(15, 4, hybrid) = 8.660;
	atom_vdw_tables(15, 5, hybrid) = 5.767;
	atom_vdw_tables(15, 6, hybrid) = 8.332;
	atom_vdw_tables(15, 7, hybrid) = 9.427;
	atom_vdw_tables(15, 8, hybrid) = 8.173;
	atom_vdw_tables(15, 9, hybrid) = 8.255;
	atom_vdw_tables(15,10, hybrid) = 4.309;
	atom_vdw_tables(15,11, hybrid) = 10.293;
	atom_vdw_tables(15,12, hybrid) = 7.817;
	atom_vdw_tables(15,13, hybrid) = 9.668;
	atom_vdw_tables(15,14, hybrid) = 7.382;
	atom_vdw_tables(15,15, hybrid) = 8.877;
	atom_vdw_tables(15,16, hybrid) = 8.090;
	atom_vdw_tables(15,17, hybrid) = 9.423;
	atom_vdw_tables(15,18, hybrid) = 12.013;
	atom_vdw_tables(15,19, hybrid) = 8.791;
	atom_vdw_tables(15,20, hybrid) = 5.894;
	atom_vdw_tables(15,21, hybrid) = 9.052;
	atom_vdw_tables(15,22, hybrid) = 10.386;
	atom_vdw_tables(15,23, hybrid) = 9.920;
	atom_vdw_tables(15,24, hybrid) = 5.109;
	atom_vdw_tables(15,25, hybrid) = 4.932;
	atom_vdw_tables(15,26, hybrid) = 0.000;
	atom_vdw_tables(15,27, hybrid) = 0.000;
	atom_vdw_tables(15,28, hybrid) = 0.000;
	atom_vdw_tables(15,29, hybrid) = 0.000;
	atom_vdw_tables(15,30, hybrid) = 0.000;
	atom_vdw_tables(15,31, hybrid) = 0.000;
	atom_vdw_tables(15,32, hybrid) = 0.000;
	atom_vdw_tables(15,33, hybrid) = 0.000;
	atom_vdw_tables(16, 1, hybrid) = 8.112;
	atom_vdw_tables(16, 2, hybrid) = 8.098;
	atom_vdw_tables(16, 3, hybrid) = 6.392;
	atom_vdw_tables(16, 4, hybrid) = 7.820;
	atom_vdw_tables(16, 5, hybrid) = 4.724;
	atom_vdw_tables(16, 6, hybrid) = 7.952;
	atom_vdw_tables(16, 7, hybrid) = 10.310;
	atom_vdw_tables(16, 8, hybrid) = 8.385;
	atom_vdw_tables(16, 9, hybrid) = 8.770;
	atom_vdw_tables(16,10, hybrid) = 5.564;
	atom_vdw_tables(16,11, hybrid) = 9.830;
	atom_vdw_tables(16,12, hybrid) = 9.914;
	atom_vdw_tables(16,13, hybrid) = 10.217;
	atom_vdw_tables(16,14, hybrid) = 9.005;
	atom_vdw_tables(16,15, hybrid) = 8.090;
	atom_vdw_tables(16,16, hybrid) = 11.357;
	atom_vdw_tables(16,17, hybrid) = 9.369;
	atom_vdw_tables(16,18, hybrid) = 11.869;
	atom_vdw_tables(16,19, hybrid) = 9.779;
	atom_vdw_tables(16,20, hybrid) = 9.236;
	atom_vdw_tables(16,21, hybrid) = 10.315;
	atom_vdw_tables(16,22, hybrid) = 11.435;
	atom_vdw_tables(16,23, hybrid) = 9.808;
	atom_vdw_tables(16,24, hybrid) = 8.501;
	atom_vdw_tables(16,25, hybrid) = 5.737;
	atom_vdw_tables(16,26, hybrid) = 0.000;
	atom_vdw_tables(16,27, hybrid) = 0.000;
	atom_vdw_tables(16,28, hybrid) = 0.000;
	atom_vdw_tables(16,29, hybrid) = 0.000;
	atom_vdw_tables(16,30, hybrid) = 0.000;
	atom_vdw_tables(16,31, hybrid) = 0.000;
	atom_vdw_tables(16,32, hybrid) = 0.000;
	atom_vdw_tables(16,33, hybrid) = 0.000;
	atom_vdw_tables(17, 1, hybrid) = 8.499;
	atom_vdw_tables(17, 2, hybrid) = 8.390;
	atom_vdw_tables(17, 3, hybrid) = 7.954;
	atom_vdw_tables(17, 4, hybrid) = 7.939;
	atom_vdw_tables(17, 5, hybrid) = 5.142;
	atom_vdw_tables(17, 6, hybrid) = 10.006;
	atom_vdw_tables(17, 7, hybrid) = 11.879;
	atom_vdw_tables(17, 8, hybrid) = 8.726;
	atom_vdw_tables(17, 9, hybrid) = 8.165;
	atom_vdw_tables(17,10, hybrid) = 7.149;
	atom_vdw_tables(17,11, hybrid) = 9.745;
	atom_vdw_tables(17,12, hybrid) = 9.032;
	atom_vdw_tables(17,13, hybrid) = 11.623;
	atom_vdw_tables(17,14, hybrid) = 7.947;
	atom_vdw_tables(17,15, hybrid) = 9.423;
	atom_vdw_tables(17,16, hybrid) = 9.369;
	atom_vdw_tables(17,17, hybrid) = 11.248;
	atom_vdw_tables(17,18, hybrid) = 12.755;
	atom_vdw_tables(17,19, hybrid) = 8.943;
	atom_vdw_tables(17,20, hybrid) = 8.182;
	atom_vdw_tables(17,21, hybrid) = 8.967;
	atom_vdw_tables(17,22, hybrid) = 10.608;
	atom_vdw_tables(17,23, hybrid) = 11.073;
	atom_vdw_tables(17,24, hybrid) = 8.804;
	atom_vdw_tables(17,25, hybrid) = 6.082;
	atom_vdw_tables(17,26, hybrid) = 0.000;
	atom_vdw_tables(17,27, hybrid) = 0.000;
	atom_vdw_tables(17,28, hybrid) = 0.000;
	atom_vdw_tables(17,29, hybrid) = 0.000;
	atom_vdw_tables(17,30, hybrid) = 0.000;
	atom_vdw_tables(17,31, hybrid) = 0.000;
	atom_vdw_tables(17,32, hybrid) = 0.000;
	atom_vdw_tables(17,33, hybrid) = 0.000;
	atom_vdw_tables(18, 1, hybrid) = 10.968;
	atom_vdw_tables(18, 2, hybrid) = 10.723;
	atom_vdw_tables(18, 3, hybrid) = 10.892;
	atom_vdw_tables(18, 4, hybrid) = 7.398;
	atom_vdw_tables(18, 5, hybrid) = 7.413;
	atom_vdw_tables(18, 6, hybrid) = 12.854;
	atom_vdw_tables(18, 7, hybrid) = 13.546;
	atom_vdw_tables(18, 8, hybrid) = 10.644;
	atom_vdw_tables(18, 9, hybrid) = 8.952;
	atom_vdw_tables(18,10, hybrid) = 7.807;
	atom_vdw_tables(18,11, hybrid) = 12.661;
	atom_vdw_tables(18,12, hybrid) = 9.500;
	atom_vdw_tables(18,13, hybrid) = 13.740;
	atom_vdw_tables(18,14, hybrid) = 10.446;
	atom_vdw_tables(18,15, hybrid) = 12.013;
	atom_vdw_tables(18,16, hybrid) = 11.869;
	atom_vdw_tables(18,17, hybrid) = 12.755;
	atom_vdw_tables(18,18, hybrid) = 17.165;
	atom_vdw_tables(18,19, hybrid) = 11.088;
	atom_vdw_tables(18,20, hybrid) = 8.417;
	atom_vdw_tables(18,21, hybrid) = 11.151;
	atom_vdw_tables(18,22, hybrid) = 13.372;
	atom_vdw_tables(18,23, hybrid) = 13.686;
	atom_vdw_tables(18,24, hybrid) = 7.538;
	atom_vdw_tables(18,25, hybrid) = 6.476;
	atom_vdw_tables(18,26, hybrid) = 0.000;
	atom_vdw_tables(18,27, hybrid) = 0.000;
	atom_vdw_tables(18,28, hybrid) = 0.000;
	atom_vdw_tables(18,29, hybrid) = 0.000;
	atom_vdw_tables(18,30, hybrid) = 0.000;
	atom_vdw_tables(18,31, hybrid) = 0.000;
	atom_vdw_tables(18,32, hybrid) = 0.000;
	atom_vdw_tables(18,33, hybrid) = 0.000;
	atom_vdw_tables(19, 1, hybrid) = 7.721;
	atom_vdw_tables(19, 2, hybrid) = 7.518;
	atom_vdw_tables(19, 3, hybrid) = 6.281;
	atom_vdw_tables(19, 4, hybrid) = 7.718;
	atom_vdw_tables(19, 5, hybrid) = 4.986;
	atom_vdw_tables(19, 6, hybrid) = 8.597;
	atom_vdw_tables(19, 7, hybrid) = 12.350;
	atom_vdw_tables(19, 8, hybrid) = 9.143;
	atom_vdw_tables(19, 9, hybrid) = 7.976;
	atom_vdw_tables(19,10, hybrid) = 5.676;
	atom_vdw_tables(19,11, hybrid) = 9.648;
	atom_vdw_tables(19,12, hybrid) = 9.323;
	atom_vdw_tables(19,13, hybrid) = 10.077;
	atom_vdw_tables(19,14, hybrid) = 8.393;
	atom_vdw_tables(19,15, hybrid) = 8.791;
	atom_vdw_tables(19,16, hybrid) = 9.779;
	atom_vdw_tables(19,17, hybrid) = 8.943;
	atom_vdw_tables(19,18, hybrid) = 11.088;
	atom_vdw_tables(19,19, hybrid) = 13.223;
	atom_vdw_tables(19,20, hybrid) = 8.232;
	atom_vdw_tables(19,21, hybrid) = 8.484;
	atom_vdw_tables(19,22, hybrid) = 9.723;
	atom_vdw_tables(19,23, hybrid) = 9.750;
	atom_vdw_tables(19,24, hybrid) = 7.328;
	atom_vdw_tables(19,25, hybrid) = 5.593;
	atom_vdw_tables(19,26, hybrid) = 0.000;
	atom_vdw_tables(19,27, hybrid) = 0.000;
	atom_vdw_tables(19,28, hybrid) = 0.000;
	atom_vdw_tables(19,29, hybrid) = 0.000;
	atom_vdw_tables(19,30, hybrid) = 0.000;
	atom_vdw_tables(19,31, hybrid) = 0.000;
	atom_vdw_tables(19,32, hybrid) = 0.000;
	atom_vdw_tables(19,33, hybrid) = 0.000;
	atom_vdw_tables(20, 1, hybrid) = 5.740;
	atom_vdw_tables(20, 2, hybrid) = 4.681;
	atom_vdw_tables(20, 3, hybrid) = 3.945;
	atom_vdw_tables(20, 4, hybrid) = 4.692;
	atom_vdw_tables(20, 5, hybrid) = 2.106;
	atom_vdw_tables(20, 6, hybrid) = 6.035;
	atom_vdw_tables(20, 7, hybrid) = 9.945;
	atom_vdw_tables(20, 8, hybrid) = 6.293;
	atom_vdw_tables(20, 9, hybrid) = 5.228;
	atom_vdw_tables(20,10, hybrid) = 4.448;
	atom_vdw_tables(20,11, hybrid) = 7.146;
	atom_vdw_tables(20,12, hybrid) = 6.330;
	atom_vdw_tables(20,13, hybrid) = 8.090;
	atom_vdw_tables(20,14, hybrid) = 9.902;
	atom_vdw_tables(20,15, hybrid) = 5.894;
	atom_vdw_tables(20,16, hybrid) = 9.236;
	atom_vdw_tables(20,17, hybrid) = 8.182;
	atom_vdw_tables(20,18, hybrid) = 8.417;
	atom_vdw_tables(20,19, hybrid) = 8.232;
	atom_vdw_tables(20,20, hybrid) = 10.191;
	atom_vdw_tables(20,21, hybrid) = 7.026;
	atom_vdw_tables(20,22, hybrid) = 8.112;
	atom_vdw_tables(20,23, hybrid) = 8.305;
	atom_vdw_tables(20,24, hybrid) = 5.549;
	atom_vdw_tables(20,25, hybrid) = 4.721;
	atom_vdw_tables(20,26, hybrid) = 0.000;
	atom_vdw_tables(20,27, hybrid) = 0.000;
	atom_vdw_tables(20,28, hybrid) = 0.000;
	atom_vdw_tables(20,29, hybrid) = 0.000;
	atom_vdw_tables(20,30, hybrid) = 0.000;
	atom_vdw_tables(20,31, hybrid) = 0.000;
	atom_vdw_tables(20,32, hybrid) = 0.000;
	atom_vdw_tables(20,33, hybrid) = 0.000;
	atom_vdw_tables(21, 1, hybrid) = 8.468;
	atom_vdw_tables(21, 2, hybrid) = 9.730;
	atom_vdw_tables(21, 3, hybrid) = 8.752;
	atom_vdw_tables(21, 4, hybrid) = 9.001;
	atom_vdw_tables(21, 5, hybrid) = 6.309;
	atom_vdw_tables(21, 6, hybrid) = 10.044;
	atom_vdw_tables(21, 7, hybrid) = 11.385;
	atom_vdw_tables(21, 8, hybrid) = 8.244;
	atom_vdw_tables(21, 9, hybrid) = 7.283;
	atom_vdw_tables(21,10, hybrid) = 7.120;
	atom_vdw_tables(21,11, hybrid) = 10.432;
	atom_vdw_tables(21,12, hybrid) = 8.787;
	atom_vdw_tables(21,13, hybrid) = 11.311;
	atom_vdw_tables(21,14, hybrid) = 7.184;
	atom_vdw_tables(21,15, hybrid) = 9.052;
	atom_vdw_tables(21,16, hybrid) = 10.315;
	atom_vdw_tables(21,17, hybrid) = 8.967;
	atom_vdw_tables(21,18, hybrid) = 11.151;
	atom_vdw_tables(21,19, hybrid) = 8.484;
	atom_vdw_tables(21,20, hybrid) = 7.026;
	atom_vdw_tables(21,21, hybrid) = 9.588;
	atom_vdw_tables(21,22, hybrid) = 10.451;
	atom_vdw_tables(21,23, hybrid) = 10.456;
	atom_vdw_tables(21,24, hybrid) = 7.804;
	atom_vdw_tables(21,25, hybrid) = 6.074;
	atom_vdw_tables(21,26, hybrid) = 0.000;
	atom_vdw_tables(21,27, hybrid) = 0.000;
	atom_vdw_tables(21,28, hybrid) = 0.000;
	atom_vdw_tables(21,29, hybrid) = 0.000;
	atom_vdw_tables(21,30, hybrid) = 0.000;
	atom_vdw_tables(21,31, hybrid) = 0.000;
	atom_vdw_tables(21,32, hybrid) = 0.000;
	atom_vdw_tables(21,33, hybrid) = 0.000;
	atom_vdw_tables(22, 1, hybrid) = 10.170;
	atom_vdw_tables(22, 2, hybrid) = 11.122;
	atom_vdw_tables(22, 3, hybrid) = 10.176;
	atom_vdw_tables(22, 4, hybrid) = 10.543;
	atom_vdw_tables(22, 5, hybrid) = 7.349;
	atom_vdw_tables(22, 6, hybrid) = 11.051;
	atom_vdw_tables(22, 7, hybrid) = 12.386;
	atom_vdw_tables(22, 8, hybrid) = 9.086;
	atom_vdw_tables(22, 9, hybrid) = 8.589;
	atom_vdw_tables(22,10, hybrid) = 8.059;
	atom_vdw_tables(22,11, hybrid) = 11.548;
	atom_vdw_tables(22,12, hybrid) = 9.324;
	atom_vdw_tables(22,13, hybrid) = 12.891;
	atom_vdw_tables(22,14, hybrid) = 9.175;
	atom_vdw_tables(22,15, hybrid) = 10.386;
	atom_vdw_tables(22,16, hybrid) = 11.435;
	atom_vdw_tables(22,17, hybrid) = 10.608;
	atom_vdw_tables(22,18, hybrid) = 13.372;
	atom_vdw_tables(22,19, hybrid) = 9.723;
	atom_vdw_tables(22,20, hybrid) = 8.112;
	atom_vdw_tables(22,21, hybrid) = 10.451;
	atom_vdw_tables(22,22, hybrid) = 12.897;
	atom_vdw_tables(22,23, hybrid) = 12.538;
	atom_vdw_tables(22,24, hybrid) = 10.763;
	atom_vdw_tables(22,25, hybrid) = 8.655;
	atom_vdw_tables(22,26, hybrid) = 0.000;
	atom_vdw_tables(22,27, hybrid) = 0.000;
	atom_vdw_tables(22,28, hybrid) = 0.000;
	atom_vdw_tables(22,29, hybrid) = 0.000;
	atom_vdw_tables(22,30, hybrid) = 0.000;
	atom_vdw_tables(22,31, hybrid) = 0.000;
	atom_vdw_tables(22,32, hybrid) = 0.000;
	atom_vdw_tables(22,33, hybrid) = 0.000;

	// section 3c
	atom_vdw_tables(23, 1, hybrid) = 10.647;
	atom_vdw_tables(23, 2, hybrid) = 11.436;
	atom_vdw_tables(23, 3, hybrid) = 10.028;
	atom_vdw_tables(23, 4, hybrid) = 10.964;
	atom_vdw_tables(23, 5, hybrid) = 8.198;
	atom_vdw_tables(23, 6, hybrid) = 11.027;
	atom_vdw_tables(23, 7, hybrid) = 11.659;
	atom_vdw_tables(23, 8, hybrid) = 10.664;
	atom_vdw_tables(23, 9, hybrid) = 8.943;
	atom_vdw_tables(23,10, hybrid) = 7.058;
	atom_vdw_tables(23,11, hybrid) = 12.516;
	atom_vdw_tables(23,12, hybrid) = 8.597;
	atom_vdw_tables(23,13, hybrid) = 12.102;
	atom_vdw_tables(23,14, hybrid) = 8.827;
	atom_vdw_tables(23,15, hybrid) = 9.920;
	atom_vdw_tables(23,16, hybrid) = 9.808;
	atom_vdw_tables(23,17, hybrid) = 11.073;
	atom_vdw_tables(23,18, hybrid) = 13.686;
	atom_vdw_tables(23,19, hybrid) = 9.750;
	atom_vdw_tables(23,20, hybrid) = 8.305;
	atom_vdw_tables(23,21, hybrid) = 10.456;
	atom_vdw_tables(23,22, hybrid) = 12.538;
	atom_vdw_tables(23,23, hybrid) = 12.277;
	atom_vdw_tables(23,24, hybrid) = 7.948;
	atom_vdw_tables(23,25, hybrid) = 6.552;
	atom_vdw_tables(23,26, hybrid) = 0.000;
	atom_vdw_tables(23,27, hybrid) = 0.000;
	atom_vdw_tables(23,28, hybrid) = 0.000;
	atom_vdw_tables(23,29, hybrid) = 0.000;
	atom_vdw_tables(23,30, hybrid) = 0.000;
	atom_vdw_tables(23,31, hybrid) = 0.000;
	atom_vdw_tables(23,32, hybrid) = 0.000;
	atom_vdw_tables(23,33, hybrid) = 0.000;
	atom_vdw_tables(24, 1, hybrid) = 6.279;
	atom_vdw_tables(24, 2, hybrid) = 5.832;
	atom_vdw_tables(24, 3, hybrid) = 5.128;
	atom_vdw_tables(24, 4, hybrid) = 6.134;
	atom_vdw_tables(24, 5, hybrid) = 4.267;
	atom_vdw_tables(24, 6, hybrid) = 6.942;
	atom_vdw_tables(24, 7, hybrid) = 11.396;
	atom_vdw_tables(24, 8, hybrid) = 10.085;
	atom_vdw_tables(24, 9, hybrid) = 8.278;
	atom_vdw_tables(24,10, hybrid) = 4.767;
	atom_vdw_tables(24,11, hybrid) = 8.347;
	atom_vdw_tables(24,12, hybrid) = 7.220;
	atom_vdw_tables(24,13, hybrid) = 8.123;
	atom_vdw_tables(24,14, hybrid) = 5.166;
	atom_vdw_tables(24,15, hybrid) = 5.109;
	atom_vdw_tables(24,16, hybrid) = 8.501;
	atom_vdw_tables(24,17, hybrid) = 8.804;
	atom_vdw_tables(24,18, hybrid) = 7.538;
	atom_vdw_tables(24,19, hybrid) = 7.328;
	atom_vdw_tables(24,20, hybrid) = 5.549;
	atom_vdw_tables(24,21, hybrid) = 7.804;
	atom_vdw_tables(24,22, hybrid) = 10.763;
	atom_vdw_tables(24,23, hybrid) = 7.948;
	atom_vdw_tables(24,24, hybrid) = 11.560;
	atom_vdw_tables(24,25, hybrid) = 6.281;
	atom_vdw_tables(24,26, hybrid) = 0.000;
	atom_vdw_tables(24,27, hybrid) = 0.000;
	atom_vdw_tables(24,28, hybrid) = 0.000;
	atom_vdw_tables(24,29, hybrid) = 0.000;
	atom_vdw_tables(24,30, hybrid) = 0.000;
	atom_vdw_tables(24,31, hybrid) = 0.000;
	atom_vdw_tables(24,32, hybrid) = 0.000;
	atom_vdw_tables(24,33, hybrid) = 0.000;
	atom_vdw_tables(25, 1, hybrid) = 5.542;
	atom_vdw_tables(25, 2, hybrid) = 5.597;
	atom_vdw_tables(25, 3, hybrid) = 4.649;
	atom_vdw_tables(25, 4, hybrid) = 5.639;
	atom_vdw_tables(25, 5, hybrid) = 3.485;
	atom_vdw_tables(25, 6, hybrid) = 6.143;
	atom_vdw_tables(25, 7, hybrid) = 8.389;
	atom_vdw_tables(25, 8, hybrid) = 6.947;
	atom_vdw_tables(25, 9, hybrid) = 5.941;
	atom_vdw_tables(25,10, hybrid) = 3.435;
	atom_vdw_tables(25,11, hybrid) = 6.967;
	atom_vdw_tables(25,12, hybrid) = 5.627;
	atom_vdw_tables(25,13, hybrid) = 6.972;
	atom_vdw_tables(25,14, hybrid) = 4.582;
	atom_vdw_tables(25,15, hybrid) = 4.932;
	atom_vdw_tables(25,16, hybrid) = 5.737;
	atom_vdw_tables(25,17, hybrid) = 6.082;
	atom_vdw_tables(25,18, hybrid) = 6.476;
	atom_vdw_tables(25,19, hybrid) = 5.593;
	atom_vdw_tables(25,20, hybrid) = 4.721;
	atom_vdw_tables(25,21, hybrid) = 6.074;
	atom_vdw_tables(25,22, hybrid) = 8.655;
	atom_vdw_tables(25,23, hybrid) = 6.552;
	atom_vdw_tables(25,24, hybrid) = 6.281;
	atom_vdw_tables(25,25, hybrid) = 4.317;
	atom_vdw_tables(25,26, hybrid) = 0.000;
	atom_vdw_tables(25,27, hybrid) = 0.000;
	atom_vdw_tables(25,28, hybrid) = 0.000;
	atom_vdw_tables(25,29, hybrid) = 0.000;
	atom_vdw_tables(25,30, hybrid) = 0.000;
	atom_vdw_tables(25,31, hybrid) = 0.000;
	atom_vdw_tables(25,32, hybrid) = 0.000;
	atom_vdw_tables(25,33, hybrid) = 0.000;
	atom_vdw_tables(26, 1, hybrid) = 0.000;
	atom_vdw_tables(26, 2, hybrid) = 0.000;
	atom_vdw_tables(26, 3, hybrid) = 0.000;
	atom_vdw_tables(26, 4, hybrid) = 0.000;
	atom_vdw_tables(26, 5, hybrid) = 0.000;
	atom_vdw_tables(26, 6, hybrid) = 0.000;
	atom_vdw_tables(26, 7, hybrid) = 0.000;
	atom_vdw_tables(26, 8, hybrid) = 0.000;
	atom_vdw_tables(26, 9, hybrid) = 0.000;
	atom_vdw_tables(26,10, hybrid) = 0.000;
	atom_vdw_tables(26,11, hybrid) = 0.000;
	atom_vdw_tables(26,12, hybrid) = 0.000;
	atom_vdw_tables(26,13, hybrid) = 0.000;
	atom_vdw_tables(26,14, hybrid) = 0.000;
	atom_vdw_tables(26,15, hybrid) = 0.000;
	atom_vdw_tables(26,16, hybrid) = 0.000;
	atom_vdw_tables(26,17, hybrid) = 0.000;
	atom_vdw_tables(26,18, hybrid) = 0.000;
	atom_vdw_tables(26,19, hybrid) = 0.000;
	atom_vdw_tables(26,20, hybrid) = 0.000;
	atom_vdw_tables(26,21, hybrid) = 0.000;
	atom_vdw_tables(26,22, hybrid) = 0.000;
	atom_vdw_tables(26,23, hybrid) = 0.000;
	atom_vdw_tables(26,24, hybrid) = 0.000;
	atom_vdw_tables(26,25, hybrid) = 0.000;
	atom_vdw_tables(26,26, hybrid) = 0.000;
	atom_vdw_tables(26,27, hybrid) = 0.000;
	atom_vdw_tables(26,28, hybrid) = 0.000;
	atom_vdw_tables(26,29, hybrid) = 0.000;
	atom_vdw_tables(26,30, hybrid) = 0.000;
	atom_vdw_tables(26,31, hybrid) = 0.000;
	atom_vdw_tables(26,32, hybrid) = 0.000;
	atom_vdw_tables(26,33, hybrid) = 0.000;
	atom_vdw_tables(27, 1, hybrid) = 0.000;
	atom_vdw_tables(27, 2, hybrid) = 0.000;
	atom_vdw_tables(27, 3, hybrid) = 0.000;
	atom_vdw_tables(27, 4, hybrid) = 0.000;
	atom_vdw_tables(27, 5, hybrid) = 0.000;
	atom_vdw_tables(27, 6, hybrid) = 0.000;
	atom_vdw_tables(27, 7, hybrid) = 0.000;
	atom_vdw_tables(27, 8, hybrid) = 0.000;
	atom_vdw_tables(27, 9, hybrid) = 0.000;
	atom_vdw_tables(27,10, hybrid) = 0.000;
	atom_vdw_tables(27,11, hybrid) = 0.000;
	atom_vdw_tables(27,12, hybrid) = 0.000;
	atom_vdw_tables(27,13, hybrid) = 0.000;
	atom_vdw_tables(27,14, hybrid) = 0.000;
	atom_vdw_tables(27,15, hybrid) = 0.000;
	atom_vdw_tables(27,16, hybrid) = 0.000;
	atom_vdw_tables(27,17, hybrid) = 0.000;
	atom_vdw_tables(27,18, hybrid) = 0.000;
	atom_vdw_tables(27,19, hybrid) = 0.000;
	atom_vdw_tables(27,20, hybrid) = 0.000;
	atom_vdw_tables(27,21, hybrid) = 0.000;
	atom_vdw_tables(27,22, hybrid) = 0.000;
	atom_vdw_tables(27,23, hybrid) = 0.000;
	atom_vdw_tables(27,24, hybrid) = 0.000;
	atom_vdw_tables(27,25, hybrid) = 0.000;
	atom_vdw_tables(27,26, hybrid) = 0.000;
	atom_vdw_tables(27,27, hybrid) = 0.000;
	atom_vdw_tables(27,28, hybrid) = 0.000;
	atom_vdw_tables(27,29, hybrid) = 0.000;
	atom_vdw_tables(27,30, hybrid) = 0.000;
	atom_vdw_tables(27,31, hybrid) = 0.000;
	atom_vdw_tables(27,32, hybrid) = 0.000;
	atom_vdw_tables(27,33, hybrid) = 0.000;
	atom_vdw_tables(28, 1, hybrid) = 0.000;
	atom_vdw_tables(28, 2, hybrid) = 0.000;
	atom_vdw_tables(28, 3, hybrid) = 0.000;
	atom_vdw_tables(28, 4, hybrid) = 0.000;
	atom_vdw_tables(28, 5, hybrid) = 0.000;
	atom_vdw_tables(28, 6, hybrid) = 0.000;
	atom_vdw_tables(28, 7, hybrid) = 0.000;
	atom_vdw_tables(28, 8, hybrid) = 0.000;
	atom_vdw_tables(28, 9, hybrid) = 0.000;
	atom_vdw_tables(28,10, hybrid) = 0.000;
	atom_vdw_tables(28,11, hybrid) = 0.000;
	atom_vdw_tables(28,12, hybrid) = 0.000;
	atom_vdw_tables(28,13, hybrid) = 0.000;
	atom_vdw_tables(28,14, hybrid) = 0.000;
	atom_vdw_tables(28,15, hybrid) = 0.000;
	atom_vdw_tables(28,16, hybrid) = 0.000;
	atom_vdw_tables(28,17, hybrid) = 0.000;
	atom_vdw_tables(28,18, hybrid) = 0.000;
	atom_vdw_tables(28,19, hybrid) = 0.000;
	atom_vdw_tables(28,20, hybrid) = 0.000;
	atom_vdw_tables(28,21, hybrid) = 0.000;
	atom_vdw_tables(28,22, hybrid) = 0.000;
	atom_vdw_tables(28,23, hybrid) = 0.000;
	atom_vdw_tables(28,24, hybrid) = 0.000;
	atom_vdw_tables(28,25, hybrid) = 0.000;
	atom_vdw_tables(28,26, hybrid) = 0.000;
	atom_vdw_tables(28,27, hybrid) = 0.000;
	atom_vdw_tables(28,28, hybrid) = 0.000;
	atom_vdw_tables(28,29, hybrid) = 0.000;
	atom_vdw_tables(28,30, hybrid) = 0.000;
	atom_vdw_tables(28,31, hybrid) = 0.000;
	atom_vdw_tables(28,32, hybrid) = 0.000;
	atom_vdw_tables(28,33, hybrid) = 0.000;
	atom_vdw_tables(29, 1, hybrid) = 0.000;
	atom_vdw_tables(29, 2, hybrid) = 0.000;
	atom_vdw_tables(29, 3, hybrid) = 0.000;
	atom_vdw_tables(29, 4, hybrid) = 0.000;
	atom_vdw_tables(29, 5, hybrid) = 0.000;
	atom_vdw_tables(29, 6, hybrid) = 0.000;
	atom_vdw_tables(29, 7, hybrid) = 0.000;
	atom_vdw_tables(29, 8, hybrid) = 0.000;
	atom_vdw_tables(29, 9, hybrid) = 0.000;
	atom_vdw_tables(29,10, hybrid) = 0.000;
	atom_vdw_tables(29,11, hybrid) = 0.000;
	atom_vdw_tables(29,12, hybrid) = 0.000;
	atom_vdw_tables(29,13, hybrid) = 0.000;
	atom_vdw_tables(29,14, hybrid) = 0.000;
	atom_vdw_tables(29,15, hybrid) = 0.000;
	atom_vdw_tables(29,16, hybrid) = 0.000;
	atom_vdw_tables(29,17, hybrid) = 0.000;
	atom_vdw_tables(29,18, hybrid) = 0.000;
	atom_vdw_tables(29,19, hybrid) = 0.000;
	atom_vdw_tables(29,20, hybrid) = 0.000;
	atom_vdw_tables(29,21, hybrid) = 0.000;
	atom_vdw_tables(29,22, hybrid) = 0.000;
	atom_vdw_tables(29,23, hybrid) = 0.000;
	atom_vdw_tables(29,24, hybrid) = 0.000;
	atom_vdw_tables(29,25, hybrid) = 0.000;
	atom_vdw_tables(29,26, hybrid) = 0.000;
	atom_vdw_tables(29,27, hybrid) = 0.000;
	atom_vdw_tables(29,28, hybrid) = 0.000;
	atom_vdw_tables(29,29, hybrid) = 0.000;
	atom_vdw_tables(29,30, hybrid) = 0.000;
	atom_vdw_tables(29,31, hybrid) = 0.000;
	atom_vdw_tables(29,32, hybrid) = 0.000;
	atom_vdw_tables(29,33, hybrid) = 0.000;
	atom_vdw_tables(30, 1, hybrid) = 0.000;
	atom_vdw_tables(30, 2, hybrid) = 0.000;
	atom_vdw_tables(30, 3, hybrid) = 0.000;
	atom_vdw_tables(30, 4, hybrid) = 0.000;
	atom_vdw_tables(30, 5, hybrid) = 0.000;
	atom_vdw_tables(30, 6, hybrid) = 0.000;
	atom_vdw_tables(30, 7, hybrid) = 0.000;
	atom_vdw_tables(30, 8, hybrid) = 0.000;
	atom_vdw_tables(30, 9, hybrid) = 0.000;
	atom_vdw_tables(30,10, hybrid) = 0.000;
	atom_vdw_tables(30,11, hybrid) = 0.000;
	atom_vdw_tables(30,12, hybrid) = 0.000;
	atom_vdw_tables(30,13, hybrid) = 0.000;
	atom_vdw_tables(30,14, hybrid) = 0.000;
	atom_vdw_tables(30,15, hybrid) = 0.000;
	atom_vdw_tables(30,16, hybrid) = 0.000;
	atom_vdw_tables(30,17, hybrid) = 0.000;
	atom_vdw_tables(30,18, hybrid) = 0.000;
	atom_vdw_tables(30,19, hybrid) = 0.000;
	atom_vdw_tables(30,20, hybrid) = 0.000;
	atom_vdw_tables(30,21, hybrid) = 0.000;
	atom_vdw_tables(30,22, hybrid) = 0.000;
	atom_vdw_tables(30,23, hybrid) = 0.000;
	atom_vdw_tables(30,24, hybrid) = 0.000;
	atom_vdw_tables(30,25, hybrid) = 0.000;
	atom_vdw_tables(30,26, hybrid) = 0.000;
	atom_vdw_tables(30,27, hybrid) = 0.000;
	atom_vdw_tables(30,28, hybrid) = 0.000;
	atom_vdw_tables(30,29, hybrid) = 0.000;
	atom_vdw_tables(30,30, hybrid) = 0.000;
	atom_vdw_tables(30,31, hybrid) = 0.000;
	atom_vdw_tables(30,32, hybrid) = 0.000;
	atom_vdw_tables(30,33, hybrid) = 0.000;
	atom_vdw_tables(31, 1, hybrid) = 0.000;
	atom_vdw_tables(31, 2, hybrid) = 0.000;
	atom_vdw_tables(31, 3, hybrid) = 0.000;
	atom_vdw_tables(31, 4, hybrid) = 0.000;
	atom_vdw_tables(31, 5, hybrid) = 0.000;
	atom_vdw_tables(31, 6, hybrid) = 0.000;
	atom_vdw_tables(31, 7, hybrid) = 0.000;
	atom_vdw_tables(31, 8, hybrid) = 0.000;
	atom_vdw_tables(31, 9, hybrid) = 0.000;
	atom_vdw_tables(31,10, hybrid) = 0.000;
	atom_vdw_tables(31,11, hybrid) = 0.000;
	atom_vdw_tables(31,12, hybrid) = 0.000;
	atom_vdw_tables(31,13, hybrid) = 0.000;
	atom_vdw_tables(31,14, hybrid) = 0.000;
	atom_vdw_tables(31,15, hybrid) = 0.000;
	atom_vdw_tables(31,16, hybrid) = 0.000;
	atom_vdw_tables(31,17, hybrid) = 0.000;
	atom_vdw_tables(31,18, hybrid) = 0.000;
	atom_vdw_tables(31,19, hybrid) = 0.000;
	atom_vdw_tables(31,20, hybrid) = 0.000;
	atom_vdw_tables(31,21, hybrid) = 0.000;
	atom_vdw_tables(31,22, hybrid) = 0.000;
	atom_vdw_tables(31,23, hybrid) = 0.000;
	atom_vdw_tables(31,24, hybrid) = 0.000;
	atom_vdw_tables(31,25, hybrid) = 0.000;
	atom_vdw_tables(31,26, hybrid) = 0.000;
	atom_vdw_tables(31,27, hybrid) = 0.000;
	atom_vdw_tables(31,28, hybrid) = 0.000;
	atom_vdw_tables(31,29, hybrid) = 0.000;
	atom_vdw_tables(31,30, hybrid) = 0.000;
	atom_vdw_tables(31,31, hybrid) = 0.000;
	atom_vdw_tables(31,32, hybrid) = 0.000;
	atom_vdw_tables(31,33, hybrid) = 0.000;
	atom_vdw_tables(32, 1, hybrid) = 0.000;
	atom_vdw_tables(32, 2, hybrid) = 0.000;
	atom_vdw_tables(32, 3, hybrid) = 0.000;
	atom_vdw_tables(32, 4, hybrid) = 0.000;
	atom_vdw_tables(32, 5, hybrid) = 0.000;
	atom_vdw_tables(32, 6, hybrid) = 0.000;
	atom_vdw_tables(32, 7, hybrid) = 0.000;
	atom_vdw_tables(32, 8, hybrid) = 0.000;
	atom_vdw_tables(32, 9, hybrid) = 0.000;
	atom_vdw_tables(32,10, hybrid) = 0.000;
	atom_vdw_tables(32,11, hybrid) = 0.000;
	atom_vdw_tables(32,12, hybrid) = 0.000;
	atom_vdw_tables(32,13, hybrid) = 0.000;
	atom_vdw_tables(32,14, hybrid) = 0.000;
	atom_vdw_tables(32,15, hybrid) = 0.000;
	atom_vdw_tables(32,16, hybrid) = 0.000;
	atom_vdw_tables(32,17, hybrid) = 0.000;
	atom_vdw_tables(32,18, hybrid) = 0.000;
	atom_vdw_tables(32,19, hybrid) = 0.000;
	atom_vdw_tables(32,20, hybrid) = 0.000;
	atom_vdw_tables(32,21, hybrid) = 0.000;
	atom_vdw_tables(32,22, hybrid) = 0.000;
	atom_vdw_tables(32,23, hybrid) = 0.000;
	atom_vdw_tables(32,24, hybrid) = 0.000;
	atom_vdw_tables(32,25, hybrid) = 0.000;
	atom_vdw_tables(32,26, hybrid) = 0.000;
	atom_vdw_tables(32,27, hybrid) = 0.000;
	atom_vdw_tables(32,28, hybrid) = 0.000;
	atom_vdw_tables(32,29, hybrid) = 0.000;
	atom_vdw_tables(32,30, hybrid) = 0.000;
	atom_vdw_tables(32,31, hybrid) = 0.000;
	atom_vdw_tables(32,32, hybrid) = 0.000;
	atom_vdw_tables(32,33, hybrid) = 0.000;
	atom_vdw_tables(33, 1, hybrid) = 0.000;
	atom_vdw_tables(33, 2, hybrid) = 0.000;
	atom_vdw_tables(33, 3, hybrid) = 0.000;
	atom_vdw_tables(33, 4, hybrid) = 0.000;
	atom_vdw_tables(33, 5, hybrid) = 0.000;
	atom_vdw_tables(33, 6, hybrid) = 0.000;
	atom_vdw_tables(33, 7, hybrid) = 0.000;
	atom_vdw_tables(33, 8, hybrid) = 0.000;
	atom_vdw_tables(33, 9, hybrid) = 0.000;
	atom_vdw_tables(33,10, hybrid) = 0.000;
	atom_vdw_tables(33,11, hybrid) = 0.000;
	atom_vdw_tables(33,12, hybrid) = 0.000;
	atom_vdw_tables(33,13, hybrid) = 0.000;
	atom_vdw_tables(33,14, hybrid) = 0.000;
	atom_vdw_tables(33,15, hybrid) = 0.000;
	atom_vdw_tables(33,16, hybrid) = 0.000;
	atom_vdw_tables(33,17, hybrid) = 0.000;
	atom_vdw_tables(33,18, hybrid) = 0.000;
	atom_vdw_tables(33,19, hybrid) = 0.000;
	atom_vdw_tables(33,20, hybrid) = 0.000;
	atom_vdw_tables(33,21, hybrid) = 0.000;
	atom_vdw_tables(33,22, hybrid) = 0.000;
	atom_vdw_tables(33,23, hybrid) = 0.000;
	atom_vdw_tables(33,24, hybrid) = 0.000;
	atom_vdw_tables(33,25, hybrid) = 0.000;
	atom_vdw_tables(33,26, hybrid) = 0.000;
	atom_vdw_tables(33,27, hybrid) = 0.000;
	atom_vdw_tables(33,28, hybrid) = 0.000;
	atom_vdw_tables(33,29, hybrid) = 0.000;
	atom_vdw_tables(33,30, hybrid) = 0.000;
	atom_vdw_tables(33,31, hybrid) = 0.000;
	atom_vdw_tables(33,32, hybrid) = 0.000;
	atom_vdw_tables(33,33, hybrid) = 0.000;
	//KMa phospho_ser
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( i, 34, hybrid) = 0.000;
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( 34, i, hybrid) = 0.000;
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( i, 26, hybrid) = atom_vdw_tables( i, 21, hybrid);
	}
	for (int i=1;i<=34;i++) {
		atom_vdw_tables( 26, i, hybrid) = atom_vdw_tables( 21, i, hybrid);
	}
}

} // initializers

} // vdw

