// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   AtomPoint.hh
/// @brief  Subclasses numeric:xyzTriple< float > to track atom position (x, y, z) and identity (atom number, residue number).
/// @brief  Primarily for use with rootstock::Octree.
/// @note   This was originally meant to be an abstract lightweight class (concrete implementations would include
/// @note   one where atom coordinates are copied, like now, and a second where atom coordinates are handled either
/// @note   by pointer or by indexing into external storage to save memory).  xyzTriple provides a number of
/// @note   useful already-implemented functions such as lexicographic ordering and various operators, so it's
/// @note   used for now until the need for a lighter weight class (primarily the memory saving implementation
/// @note   becomes necessary.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)


#ifndef INCLUDED_epigraft_AtomPoint_HH_
#define INCLUDED_epigraft_AtomPoint_HH_

// rootstock headers
#include <rootstock_types.hh>

// numeric headers
#include <numeric/xyzTriple.hh>


namespace epigraft {


/// @brief  Subclasses numeric:xyzTriple< float > to track atom position (x, y, z) and identity (atom number, residue number).
/// @brief  Primarily for use with rootstock::Octree.
class AtomPoint : public numeric::xyzTriple< Real > {

	private: // typedefs

		typedef numeric::xyzTriple< Real > Super;

	public: // construct/destruct

		/// @brief default constructor
		inline
		AtomPoint() : Super()
		{}

		/// @brief constructor
		inline
		AtomPoint(
			Integer const & residue_id,
			Integer const & atom_id,
			Real const & x,
			Real const & y,
			Real const & z
		) : Super( x, y, z ),
		    residue_id_( residue_id ),
		    atom_id_( atom_id )
		{}

		/// @brief copy constructor
		inline
		AtomPoint(
			AtomPoint const & ap
		) : Super( ap ),
		    residue_id_( ap.residue_id_ ),
		    atom_id_( ap.atom_id_ )
		{}

		/// @brief default destructor
		inline
		virtual
		~AtomPoint() {}


	public: // assignment

		/// @brief copy assignment
		inline
		AtomPoint &
		operator =( AtomPoint const & ap )
		{
			if ( this != &ap ) {
				Super::operator=( ap );
				residue_id_ = ap.residue_id_;
				atom_id_ = ap.atom_id_;
			}

			return *this;
		}


	public: // accessors

		/// @brief get residue id (residue number)
		inline
		Integer const &
		residue_id() const
		{
			return residue_id_;
		}

		/// @brief get atom id (in Rosetta, specific to residue type )
		inline
		Integer const &
		atom_id() const
		{
			return atom_id_;
		}


	private: // data

		// atom info
		Integer residue_id_;
		Integer atom_id_; // the atom index within the residue so we can access properties in Rosetta properly

};


} // namespace epigraft

#endif /*INCLUDED_epigraft_AtomPoint_HH_*/
