// /////////////////////////////////////////////////////////////////////////////////////
// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
// @DomainInsertionMode.cc
//  domins - functions called in this mode that are specific to the mode
//  start with domins
//
// @brief
// Routine to insert one protein domain into another at a specific insertion site
//
// @detailed:
// To use this mode, enter -domain_insertion -insertion_file {name of
//   insertion file} -l
// The two proteins must face each other with the insertion sites near each other.
// The input file for this mode requires the name of the protein to insert
// (insertp) into the host protein (hostp), the insertion
// sites on each protein (hostp_begin, hostp_end, insertp_begin, insertp_end)
//
// @authors: Monica Berrondo
//
// @last_modified: April 13 2006
// /////////////////////////////////////////////////////////////////////////////////////


// Rosetta Headers
#include "DomainInsertionMode.h"
#include "after_opts.h"
#include "current_pose.h"
#include "docking.h"
#include "docking_score.h"
#include "dock_structure.h"
#include "docking_constraints.h" // for the docking_constraints
#include "domins_fullatom.h"
#include "domins_ns.h"
#include "domins_score.h"
#include "files_paths.h"
#include "fragments.h"
#include "fragments_ns.h"
#include "fullatom.h"
#include "initialize.h" // in order to get the start pose
#include "interface.h" // for the docking constraints
#include "jumping_loops.h"
#include "jumping_pairings.h" // get_loop_fraction
#include "jumping_refold.h"
#include "jumping_util.h"
#include "loop_class.h"
#include "loop_relax.h"
#include "minimize_ns.h"
#include "monte_carlo.h"
#include "misc.h" // damn
#include "orient_rms.h"
#include "param_torsion.h"
#include "pdb.h"
#include "pose.h"
#include "pose_docking.h"
#include "pose_io.h"
#include "pose_looping.h"
#include "pose_loops.h"
#include "pose_rms.h"
#include "prof.h"
#include "random_numbers.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"
#include "structure.h" // for setting per residue weighting
#include "SplicedProtein.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include "utility/basic_sys_util.hh"
#include "utility/io/izstream.hh"

// Numeric Headers
#include "numeric/xyzVector.hh"

// C++ Headers
#include <list> // list class definition
#include <map> // map class definition
#include <cstdlib> // (for use of has_suffix)
#include <string>


///////////////////////////////////////////////////////////
//
// @insertion_main
//
// @brief
// main routine for setting up the domain insertion mode.
// makes calls to other functions depending on inputs and options
//
// @detailed
// Creates pose, copies coordinates in from misc
// Modes and flags:
// 		chain_break_tol - default is 0.2, otherwise must be set
// 		loop_only - takes the starting structure and only rebuilds the loop
// 		repack and repack_only - sets relax_mode to repack only and doesn't do refinement
// 		fast_relax and fast_relax_only - runs full fast_relax protocol
// 		no_loop - only rebuilds the loop once (quick and dirty)
// 		small_preturbations - small perturbations of 0.5 0.7 instead of 3 8
// 		fa_output - fullatom mode, will run fast_relax protocol
// 			must be specified for use with fast_relax or repack
// 		fa_input - sets fa_output to true, uses the sidechains from the input pdb
// 			instead of from the separate domains.
// 			To be used only if doing a fast_relax_only or repack_only
//
// @param
//
// @global_read
//     namespaces misc and domins_ns
// @global_write
//     structure and scores in usual global arrays
// @remarks
//
// @authors: Monica Berrondo
//
// @last_modified: March 21 2006
////////////////////////////////////////////////////////////////

void
domain_insertion_main()
{
	using namespace pose_ns;
	using namespace files_paths;
	using namespace misc;
	using namespace domins_ns;

	// reads in the values for loop_begin and loop_end from the command line
	bool fullatom ( false );
	//bool ideal_pos ( false );
	std::string filename (start_path + protein_name + ".pdb" );
	int iter(0), cycles(0);
	Pose starting_pose;
	Pose &combined_pose (initialize_get_decoy_pose());

	domins_get_mode( fullatom, iter, cycles );

	// Generate a pose
	score_reset_current_pose();
	starting_pose = initialize_get_decoy_pose();

	// copies the pose to a pose that will be worked with
	combined_pose.set_native_pose(initialize_get_native_pose());

	// all the centroid mode stuff
	combined_pose.set_fullatom_flag( false );

	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score8di );
	if (!no_loop)
		weight_map.set_weight( CHAINBREAK, 1.0 );
	combined_pose.score( weight_map );
	float const start_score ( combined_pose.get_0D_score( SCORE ) );
	std::cout << "starting score.......: " << start_score << std::endl;
	initial_vdw = combined_pose.get_0D_score (VDW );
	std::cout << "initial_vdw.......: " << initial_vdw << std::endl;

	// centroid main protocol
	if ( do_main_protocol ) {
		domins_centroid_main_protocol(combined_pose, weight_map, iter, cycles);
	}
	Monte_carlo mc (combined_pose, weight_map, 2);
	if ( do_insertion_flop ) {
		domins_insertion_flop(combined_pose, cycles, weight_map, mc);
	}

	// gather and store centroid information
	std::cout << "calculating rmsd of the centroid....." << std::endl;
	centroid_rmsd = domins_calc_ins_rms(combined_pose);
	combined_pose.score( weight_map );
	std::cout << "centroid_rmsd.......: " << centroid_rmsd << std::endl;
	centroid_score = combined_pose.get_0D_score( SCORE );
	std::cout << "centroid score.......: " << centroid_score << std::endl;
	//output_all_scores();

	// calls repack and adds sidechains back
	// fullatom main protocol
	if ( fullatom ) {
		// get the rms of the centroid
		combined_pose.copy_to_misc();
		combined_pose.set_fullatom_flag( fullatom, false /*don't do a full repack*/ );
		domins_fullatom_protocol(combined_pose, starting_pose, fa_input_flag);
		std::cout << "domain_scorefxn: " << scorefxns::domain_scorefxn << std::endl;
		float fa_score = combined_pose.get_0D_score( SCORE );
		std::cout << "full atom score.......: " << fa_score << std::endl;
		// for degugging, uncomment line below
		//output_all_scores();
	}

	// get the final rms to native
	std::cout << "calculating global rmsd to native....." << std::endl;
	rmsd_to_native = domins_calc_rms(combined_pose);
	std::cout << "rmsd_to_native.......: " << rmsd_to_native << std::endl;
	std::cout << "calculating insert rmsd to native....." << std::endl;
	insert_rmsd = domins_calc_ins_rms(combined_pose);
	std::cout << "insert_rmsd.......: " << insert_rmsd << std::endl;
	std::cout << "calculating rigid body rmsd to native....." << std::endl;
	rb_rmsd = domins_calc_rb_rms(combined_pose);
	std::cout << "rigid body rmsd.......: " << rb_rmsd << std::endl;
	std::cout << "calculating loop rmsd....." << std::endl;
	loop1_rmsd = domins_calc_loop_rms(combined_pose, 1);
	loop2_rmsd = domins_calc_loop_rms(combined_pose, 2);
	loop_rmsd = loop1_rmsd + loop2_rmsd;
	std::cout << "loop_rmsd.......: " << loop_rmsd << std::endl;

	//copy the coordinates back to misc for output
	combined_pose.copy_to_misc();

	// monte_carlo_reset() needs to be called to update the best and low arrays in misc.
	// For this to work, pose flag must be set to false.
	// This is only necessary in centroid mode because the pose flag must be set to true
	// for score8di, however, for fullatom mode, score12 is used so pose flag must be false
	score_set_current_pose( combined_pose );
	set_pose_flag( false );
	monte_carlo_reset();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_centroid_main_protocol()
///
/// @brief
///   this is the centroid-mode protocol to optimize (1) rigid body positon between
///   host and insert domains and (2) loop conformations for the
///   inter-domain connecting segments.
///		This protocol runs while the chain break is less than a certain
///		chain break tolerance (set by -chain_break_tol)
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, April 25 2006
///
/// @last_modified: April 25 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_centroid_main_protocol(
		pose_ns::Pose & combined_pose,
		pose_ns::Score_weight_map & weight_map,
		int const iter,
		int const cycles
		)
{
	pose_ns::Pose recover_pose;
	float const chain_break_tol ( get_chain_break_tolerance() );
	float chain_break;
	int rejected (0);

	recover_pose = combined_pose;

	do {
		combined_pose = recover_pose;
		combined_pose.score( weight_map );
		if (!domins_ns::no_loop && !runlevel_ns::benchmark) {
			chain_break = combined_pose.get_0D_score( pose_ns::CHAINBREAK );
			std::cout << "chain break score before.......: " << chain_break << std::endl;
		}
		rejected++;
		// dock proteins together
		domins_starting_position_rigid_body_move(combined_pose, iter);

		// alternate between closing loop and rigid body moves
		domins_rigid_body_and_loop_move_protocol(combined_pose, cycles);

		// don't calculate the chain break score and exit out if not creating loops
		if (!domins_ns::no_loop && !runlevel_ns::benchmark) {
			chain_break = combined_pose.get_0D_score( pose_ns::CHAINBREAK );
			std::cout << "chain break score after.......: " << chain_break << std::endl;
		} else {
			return;
		}
	} while ( chain_break > chain_break_tol);

	rejected = rejected -1;

	std::cout << "CHAIN_BREAK_TOL of " << chain_break_tol << std::endl;
	std::cout << "Rejected " << rejected << " centroid decoys to create this decoy." << std::endl;
	domins_ns::centroid_rejected = rejected;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_insertion_flop()
///
/// @brief
/// 	closes the two linker loops and does rigid-body domain perturbations
///   as many times as determined by
///		'cycles'
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, April 25 2006
///
/// @last_modified: April 25 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_insertion_flop(
		pose_ns::Pose & pose,
		int const cycles,
		pose_ns::Score_weight_map & weight_map,
		pose_ns::Monte_carlo & mc
		)
{
	using namespace pose_ns;
	using namespace domins_ns;

	int loop_size, loop_begin, loop_end, cutpoint, accepted(0), ntrial(0);
	int const nmoves ( 10 );
	int const ccd_times ( 10 );

	// allow the backbone to move for the loops
	// allow repacking of the loops
	pose_loops_set_allow_bb_move( pose, loops );

	score_set_try_rotamers( true );
	pose.score( weight_map );
	score_set_try_rotamers( false );

	show_decoy_status(pose, "insertion_flop");

	// alternate between small moves and ccd loop closures between
	// the two loops
	for (int i=1; i<=cycles; ++i) {
		ntrial++;
		// set up new fold tree
		if (mod(i,2)==0) {
			domins_create_fold_tree(pose, false, 1);
			loop_begin = loops.at(1).start();
			loop_end = loops.at(1).stop();
		} else {
			domins_create_fold_tree(pose, false, 2);
			loop_begin = loops.at(0).start();
			loop_end = loops.at(0).stop();
		}
		loop_size = loop_end - loop_begin + 1;
		cutpoint = loop_begin + loop_size / 2 -1;
		pose_small_moves(pose, nmoves);
		std::cout << "loop: " << loop_begin << ' ' << loop_end << ' ' << cutpoint << std::endl;
		ccd_moves(ccd_times, pose, loop_begin, loop_end, cutpoint);
		pose.score( weight_map );
		show_decoy_status(pose, "insertion_flop");
		if ( mc.boltzmann( pose, "insertion_flop" ) ) { accepted++; }
	}

	mc.show_scores();

	std::cout << "insertion_flop --- accepted/trial/ratio:\n"
						<< accepted << "/" << ntrial << "/"
						<< float(accepted)/float(ntrial) << "for insertion flop moves"
						<< std::endl;
	// go back to the old fold tree
	domins_create_fold_tree(pose, true, 0);

	pose = mc.low_pose();

}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_starting_posistion_rigid_body_move()
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, April 25 2006
///
/// @last_modified: April 25 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_starting_position_rigid_body_move(
		pose_ns::Pose & pose,
		int iter
		)
{
	using namespace pose_ns;
	using namespace domins_ns;

	if ( loop_only || no_perturb ) return;

	// monica fix - these commented out pieces will be needed is using per residue weighting
	int const dock_jump(1), loop_jump(2);//, nres( pose.total_residue() );
	float current_score, full_vdw, vdw_diff;//, loop_weight ( 0.2 );
	Jump perturbed_jump( pose.get_jump( dock_jump ) );
	Pose starting_pose;

	starting_pose = pose;

	// damp the repulsive
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score8di );
	pose.score( weight_map );

	do {
		pose = starting_pose;
		// sets the first jump to be flexible and the second one to be fixed
		// first jump is the one between the two proteins, second jump is the one
		// between the two parts of the first protein
		pose.set_allow_jump_move( dock_jump, true );
		pose.set_allow_jump_move( loop_jump, false );

		if (!small_perturbation) {
			randomize_insert_domain(pose);
		} else {
			perturb_insert_domain(pose);
		}

		pose.score( weight_map );
		full_vdw = pose.get_0D_score( VDW );
		std::cout << "full_vdw: " << full_vdw << std::endl;
		vdw_diff = full_vdw - initial_vdw;
		std::cout << "vdw_diff: " << vdw_diff << std::endl;
	} while ( vdw_diff > 100.0 );

	pose.score( weight_map );
	current_score = pose.get_0D_score( SCORE );
	std::cout << "current score........: " << current_score << std::endl;

	std::cout << "calculating starting rmsd....." << std::endl;
	start_rmsd = domins_calc_ins_rms(pose);
	std::cout << "start_rmsd.......: " << start_rmsd << std::endl;

	if (!no_loop) {
		domins_build_loops(pose, iter);
	}

	std::cout << "done!" << std::endl;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_rigid_body_and_loop_move_protocol()
///
/// @brief
/// 	closes the two linker loops and does rigid-body domain perturbations
///   as many times as determined by
///		'cycles'
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, April 25 2006
///
/// @last_modified: April 25 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_rigid_body_and_loop_move_protocol(
		pose_ns::Pose & pose,
		int const cycles
		)
{
	using namespace pose_ns;
	using namespace domins_ns;

	if ( no_loop ) return;

	int overlap(0), iter(0);
	float const final_ch_weight(5.0), start_rep_weight ( 0.02 ), start_rsd_rep_weight ( 0.5 ), end_rep_weight ( 1.0 );
	float const delta_ch_weight(final_ch_weight/cycles);
  float normal_perturbation ( 0.7 ), rotational_perturbation ( 5.0 ), current_score;
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score8di );
	pose.score( weight_map );
	numeric::xyzVector <double> rot_center, pos;
	std::string move;

	// setup geometric ramping of the repulsive term
	float const rep_delta
		( std::exp( std::log( end_rep_weight / start_rep_weight ) /
								(cycles-1) ) );
	float rep_weight = start_rep_weight / rep_delta;
	float const rep_rsd_delta
		( std::exp( std::log( end_rep_weight / start_rsd_rep_weight ) /
								(cycles-1) ) );
	float rep_rsd_weight = start_rsd_rep_weight / rep_rsd_delta;

	// over cycles, alternate rigid-body moves and loop moves
	for (int i=1; i<=cycles; ++i) {

		// set the chain break weight map
		weight_map.set_weight( CHAINBREAK, i * delta_ch_weight );
		if (i > cycles/2) overlap = 1;
		weight_map.set_weight( CHAINBREAK_OVERLAP, overlap );
		rep_rsd_weight *= rep_rsd_delta;
		rep_weight *= rep_delta;

		// rigid body moves
		if ( !loop_only ) {
			pose_docking_centroid_rigid_body_adaptive( pose, 5/*outer_cycle*/, 50/*inner_cycle*/, weight_map, normal_perturbation/*trans_mag*/, rotational_perturbation/*rot_mag*/);
			move = "rigid_body";
			show_decoy_status(pose, move);
		}

		current_score = pose.get_0D_score( SCORE );
		std::cout << "current score........: " << current_score << std::endl;

		// number of iterations for the insert_loop protocol
		iter = (i<=3) ? 5 : (i<=6) ? 10 : 20;

		domins_build_loops(pose, iter);
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin randomize_insert_domain()
///
/// @brief randomizes the location of the insert domain with respect to the parent domain
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 07 2006
////////////////////////////////////////////////////////////////////////////////
void
randomize_insert_domain(
		pose_ns::Pose & pose
		)
{
	using namespace domins_ns;
	using namespace pose_ns;

	if(truefalseoption("check_random_distribution")) check_random_distribution(pose);

	int const dock_jump(1);
	Jump perturbed_jump( pose.get_jump( dock_jump ) );
	int const pos1( pose.fold_tree().get_jump_point()(1, dock_jump) );
	int const pos2( pose.fold_tree().get_jump_point()(2, dock_jump) );
	float rotation_angle;
	numeric::xyzVector <double> spin_axis, rot_center, host_linker_com, insert_linker_com;
	std::string move;

	// create spin_axis between the center of mass of the beginning of the linkers of the two domains
	rot_center = get_loop_com(pose);
	std::cout << "rot_center for spin: " << rot_center << std::endl;
	host_linker_com = domins_calc_midpoint(pose, hostp_begin-linker_size, hostp_end+linker_size);
	insert_linker_com = domins_calc_midpoint(pose, insertp_begin+linker_size, insertp_end-linker_size);
	spin_axis = host_linker_com - insert_linker_com;

	rotation_angle = 360.0 * ran3();
	perturbed_jump.rotation_by_axis( pose.Eposition()(1,1,hostp_com)/*center*/, spin_axis, rot_center, rotation_angle);
	pose.set_jump( dock_jump, perturbed_jump );


	std::cout << "finished spin..moving on......" << std::endl;
	move = "spin";
	show_decoy_status(pose, move);

	// randomize starting location of the insertion protein by rotating it about the midpoint of the
	// host protein's loop ends
	std::cout << "randomizing starting position......" << std::endl;

	rot_center = get_loop_com(pose);
	std::cout << "rot_center for random: " << rot_center << std::endl;
	perturbed_jump.set_rb_center( 1, pose.Eposition()(1,1,pos2)/*downstream_jump*/, rot_center );
	perturbed_jump.rotation_by_matrix( pose.Eposition()(1,1,pos1), rot_center, random_reorientation_matrix() );
	pose.set_jump ( dock_jump, perturbed_jump );

	std::cout << "finished random..moving on......" << std::endl;
	move = "random";
	show_decoy_status(pose, move);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin perturb_insert_domain()
///
/// @brief does a dock 3 8 8 type perturbation of the insert domain with repect
///		to the parent domain
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 07 2006
////////////////////////////////////////////////////////////////////////////////
void
perturb_insert_domain(
		pose_ns::Pose & pose
		)
{
	using namespace domins_ns;
	using namespace pose_ns;

	int const dock_jump(1);
	Jump perturbed_jump( pose.get_jump( dock_jump ) );
	float normal_perturbation, rotational_perturbation;
	numeric::xyzVector <double> spin_axis, rot_center, dist, pos;
	std::string move;

	// small perturbations to make new starting position
	std::cout << "starting perturbation......" << std::endl;
	normal_perturbation = 3;
	rotational_perturbation = 8;

	rot_center = get_loop_com(pose);
	perturbed_jump.set_rb_center( 1, pose.Eposition()(1,1,insertp_com), rot_center );
	perturbed_jump.gaussian_move( 1, normal_perturbation, rotational_perturbation );
	pose.set_jump( dock_jump, perturbed_jump );
	std::cout << "finished perturbation!!" << std::endl;
	move = "perturbation";
	show_decoy_status(pose, move);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_close_gap()
///
/// @brief moves the two domains relative to each other along the axis going through
///		the centers of the two loops until the domains are within a certain distance
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 07 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_close_gap(
		pose_ns::Pose & pose
		)
{
	using namespace pose_ns;
	using namespace domins_ns;

	int const dock_jump(1);
	int count(0);
	Jump perturbed_jump( pose.get_jump( dock_jump ) );

	float const step_size( 1.0 );
	float const desired_dist( (loops.at(0).stop()-loops.at(0).start())*2.0);
	numeric::xyzVector<double> trans_axis, host_linker_com, insert_linker_com, dist, start_dist;
	std::string move;

	// move the proteins towards each other
	host_linker_com = domins_calc_midpoint(pose, hostp_begin-linker_size, hostp_end+linker_size);
	insert_linker_com = domins_calc_midpoint(pose, insertp_begin+linker_size, insertp_end-linker_size);
	trans_axis = host_linker_com - insert_linker_com;
	trans_axis = pose.Eposition()(1,1,hostp_com) - pose.Eposition()(1,1,insertp_com);
	dist = insert_linker_com - host_linker_com;
	start_dist = dist;
	std::cout << "starting distance: " << dist.length() << std::endl;

	// check axis direction
	////perturbed_jump.translation_along_axis( pose.Eposition()(1,1,hostp_com), trans_axis, step_size );
	////pose.set_jump( dock_jump, perturbed_jump );
	////host_linker_com = domins_calc_midpoint(pose, hostp_begin-5, hostp_end+5);
	////insert_linker_com = domins_calc_midpoint(pose, insertp_begin+5, insertp_end-5);
	////dist = insert_linker_com - host_linker_com;
	////if ( dist.length() > start_dist.length() ) trans_axis.negate();

	while ( dist.length() > desired_dist ) {
		count++;
		//perturbed_jump.translation_along_axis( pose.Eposition()(1,1,hostp_com), trans_axis, step_size );
		perturbed_jump.translation_along_axis( pose.Eposition()(1,1,insertp_com), trans_axis, step_size );
		pose.set_jump( dock_jump, perturbed_jump );
		host_linker_com = domins_calc_midpoint(pose, hostp_begin-linker_size, hostp_end+linker_size);
		insert_linker_com = domins_calc_midpoint(pose, insertp_begin+linker_size, insertp_end-linker_size);
		std::cout << "insert_com: " << insertp_com << std::endl;
		std::cout << "host_com: " << hostp_com << std::endl;
		dist = insert_linker_com - host_linker_com;
		std::cout << "distance (decreasing): " << dist.length() << std::endl;
		if ( count == 5 ) exit(0);
	}

	// make it so that the proteins move away from each other
	trans_axis.negate();
	while ( dist.length() < desired_dist ) {
		perturbed_jump.translation_along_axis( pose.Eposition()(1,1,hostp_com), trans_axis, step_size );
		pose.set_jump( dock_jump, perturbed_jump );
		host_linker_com = domins_calc_midpoint(pose, hostp_begin-linker_size, hostp_end+linker_size);
		insert_linker_com = domins_calc_midpoint(pose, insertp_begin+linker_size, insertp_end-linker_size);
		dist = insert_linker_com - host_linker_com;
		std::cout << "distance (increasing): " << dist.length() << std::endl;
	}

	// correct for last movement
	trans_axis.negate();
	perturbed_jump.translation_along_axis( pose.Eposition()(1,1,hostp_com), trans_axis, step_size );
	pose.set_jump( dock_jump, perturbed_jump );

	// debugging information
	std::cout << "finished translation..moving on....." << std::endl;
	move = "translation";
	show_decoy_status(pose, move);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_build_loops()
///
/// @brief sets up the loops and calls the functions to build each one
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 07 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_build_loops(
		pose_ns::Pose & pose,
		int iter
		)
{
	using namespace domins_ns;

	// build the loops between the two domains
	// allow the backbone to move for the loops
	// allow repacking of the loops
	pose_loops_set_allow_bb_move( pose, loops );

	// couts for debugging purposes
	int max_loop_size (12);
	int min_good_loops (3);
	std::cout << "fold_tree: " << pose.fold_tree() << std::endl;
	std::cout << "iter: " << iter << std::endl;
	std::cout << "loop1: " << loops.at(0).start() << ' ' << loops.at(0).stop() << std::endl;
	build_insertion_loop(pose, max_loop_size, min_good_loops, loops.at(0).start(), loops.at(0).stop(), iter);
	std::cout << "fold_tree: " << pose.fold_tree() << std::endl;
	std::cout << "fold_tree: " << pose.fold_tree() << std::endl;
	std::cout << "loop2: " << loops.at(1).start() << ' ' << loops.at(1).stop() << std::endl;
	build_insertion_loop(pose, max_loop_size, min_good_loops, loops.at(1).start(), loops.at(1).stop(), iter);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_generate_start
///
/// @brief
///		generates the combined pose and pdb for an insertion protein
///
/// @detailed
///		In the case that this is called from initialize, it actually outputs the
///		pdb of the combined protein from the pose
///		Otherwise, it just creates the map and pose for the combined protein from
///		the separate pieces.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 03 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_generate_start(
	//	pose_ns::Pose & pose
		std::vector <int> & sizes,
		std::vector <int> & new_start,
		std::vector <int> & start,
		std::map <std::string, int> & ip
		)
{
	using namespace domins_ns;
	using namespace files_paths;

	std::string filename;
	bool circular (true);
	//std::map<std::string,int> ip;

	domins_fill_sizes(circular, ip);

	//std::vector < int> sizes, new_start, start;
	sizes.push_back(ip["hostp_begin_insertion"]-ip["hostp_start_protein"]+1);
	sizes.push_back(ip["insertp_end_protein"]-ip["insertp_begin_insertion"]+1);
	if (!circular) {
		sizes.push_back(0);
	} else {
		sizes.push_back(ip["insertp_end_insertion"]-ip["insertp_start_protein"]+1);
	}
	sizes.push_back(ip["hostp_end_protein"]-ip["hostp_end_insertion"]+1);
	new_start.push_back(ip["hostp_start_protein"]);
	new_start.push_back(new_start.at(0) + sizes.at(0));
	new_start.push_back(new_start.at(1) + sizes.at(1));
	new_start.push_back(new_start.at(2) + sizes.at(2));
	start.push_back(ip["hostp_start_protein"]);
	start.push_back(ip["insertp_begin_insertion"]);
	start.push_back(ip["insertp_start_protein"]);
	start.push_back(ip["hostp_end_insertion"]);

	// for debugging purposes prints the entire contents of the protein map
	if(truefalseoption("printmap")) {
		spliced_protein_map.print_map();
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_fill_sizes
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified February 22 2007
////////////////////////////////////////////////////////////////////////////////
void
domins_fill_sizes(
		bool & circular,
		std::map<std::string,int> & ip
		)
{
	using namespace domins_ns;

	int value;
	// idealize the bonds to avoid problems when copying over to the combined pose
	host_protein_pose.set_allow_bb_move( false );
	insert_protein_pose.set_allow_bb_move( false );

	// values that are read in from the insertion parameter file
	// start and end points of the host and insertion protein
	ip["hostp_start_protein_init"] = atoi(params["hostp_start_res"].c_str());
	ip["hostp_end_protein_init"] = atoi(params["hostp_end_res"].c_str());
	res_num_from_pdb_res_num_chain(value, host_protein_pose.pdb_info(), ip["hostp_start_protein_init"], 'A', hostp_nres);
	ip["hostp_start_protein"] = value;
	res_num_from_pdb_res_num_chain(value, host_protein_pose.pdb_info(), ip["hostp_end_protein_init"], 'A', hostp_nres);
	ip["hostp_end_protein"] = value;

	ip["insertp_start_protein_init"] = atoi(params["insertp_start_res"].c_str());
	ip["insertp_end_protein_init"] = atoi(params["insertp_end_res"].c_str());
	res_num_from_pdb_res_num_chain(value, insert_protein_pose.pdb_info(), ip["insertp_start_protein_init"], 'A', insertp_nres);
	ip["insertp_start_protein"] = value;
	res_num_from_pdb_res_num_chain(value, insert_protein_pose.pdb_info(), ip["insertp_end_protein_init"], 'A', insertp_nres);
	ip["insertp_end_protein"] = value;

	// insertion point in the host protein
	ip["hostp_begin_insertion_init"] = atoi(params["hostp_begin"].c_str());
	ip["hostp_end_insertion_init"] = atoi(params["hostp_end"].c_str());
	res_num_from_pdb_res_num_chain(ip["hostp_begin_insertion"], host_protein_pose.pdb_info(), ip["hostp_begin_insertion_init"], 'A', hostp_nres);
	res_num_from_pdb_res_num_chain(ip["hostp_end_insertion"], host_protein_pose.pdb_info(), ip["hostp_end_insertion_init"], 'A', hostp_nres);

	// circular permutation point in the insertion protein
	ip["insertp_begin_insertion_init"] = atoi(params["insertp_begin"].c_str());
	ip["insertp_end_insertion_init"] = atoi(params["insertp_end"].c_str());
	res_num_from_pdb_res_num_chain(ip["insertp_begin_insertion"], insert_protein_pose.pdb_info(), ip["insertp_begin_insertion_init"], 'A', insertp_nres);
	res_num_from_pdb_res_num_chain(ip["insertp_end_insertion"], insert_protein_pose.pdb_info(), ip["insertp_end_insertion_init"], 'A', insertp_nres);

	// check to see if insertion protein must be circularly permuted
	if ((ip["insertp_begin_insertion_init"] ==  ip["insertp_start_protein_init"]) && (ip["insertp_end_insertion_init"] == ip["insertp_end_protein_init"])) {
		circular = false;
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_input_map
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified February 22 2007
////////////////////////////////////////////////////////////////////////////////
void
domins_input_map(
		std::vector <int> sizes,
		std::vector <int> new_start,
		std::vector <int> start,
		std::map<std::string,int> & ip
		)
{
	using namespace domins_ns;

	// sets up the map of information about the residues and where they came from
	// spliced_protein_res_info contains information of the type:
	//   original protein name, type of the original protein (host or insertion), residue number in
	//   the fusion protein, sequential rosetta residue number in the original protein, residue number in the original
	//   protein from the pdb
	// determine if any of the residues are repeated and add them to an array
	std::vector <int> repeated_residues;
	if (ip["insertp_begin_insertion_init"] < ip["insertp_end_insertion_init"]) {
		for (int res=ip["insertp_begin_insertion_init"]; res<=ip["insertp_end_insertion_init"]; ++res) {
			repeated_residues.push_back(res);
		}
	}

	spliced_protein_map.set_repetition(new_start.at(0), start.at(0), sizes.at(0), "host", repeated_residues, 0);
	spliced_protein_map.set_repetition(new_start.at(1), start.at(1), sizes.at(1), "insert", repeated_residues, 1);
	spliced_protein_map.set_repetition(new_start.at(2), start.at(2), sizes.at(2), "insert", repeated_residues, 2);
	spliced_protein_map.set_repetition(new_start.at(3), start.at(3), sizes.at(3), "host", repeated_residues, 0);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_initial_setup()
///
/// @brief
///		calls the functions for the initial setup
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, April 25 2006
///
/// @last_modified: April 25 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_initial_setup(
		pose_ns::Pose & pose
		)
{
	// set up the map with information about the proteins
	domins_setup_map();

	// create a 2 jump fold tree
	domins_create_fold_tree(pose, true/**/, 0/*fold_tree type*/);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_setup_map()
///
/// @brief
///		get information out of the spliced protein map and gather all the
///		information necessary to run the rest of the protocol
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, April 25 2006
///
/// @last_modified: April 25 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_setup_map(
		)
{
	using namespace domins_ns;

	// initialize values
	int last_repeated ( 1 ), loop_begin, loop_end;
	bool circular (true);
	linker_size = atoi(params["linker_size"].c_str());

	// information for host protein
	std::string hostp = params["hostp"].c_str();
	if ( has_suffix( hostp, ".pdb" ) ) hostp.erase( hostp.length() - 4 );

	// information for insertion protein
	std::string insertp = params["insertp"].c_str();
	if ( has_suffix( insertp, ".pdb" ) ) insertp.erase( insertp.length() - 4 );


	if (linker_size == 0)	linker_size = 5;

	// check to see if the input parameters indicate that the insertion protein
	// is circularly permuted
	// If it is, then check to see if any of the residues are repeated and set them to
	// 1 or 2 depending on whether or not the residue has been seen yet.
	if ((insertp_begin ==  insertp_start_res) && (insertp_end == insertp_end_res)) {
		circular = false;
	}
	if (circular && (insertp_end >= insertp_begin)) {
		last_repeated = 2;
	} else {
		last_repeated = 1;
	}

	// residue numbers in the combined protein
	// test get_combined_residue routine
	spliced_protein_map.get_combined_residue_from_pdb(hostp_combined_begin, hostp+".pdb", hostp_begin, 1, nres, 0);
	spliced_protein_map.get_combined_residue_from_pdb(hostp_combined_end, hostp+".pdb", hostp_end, 1, nres, 0);
	spliced_protein_map.get_combined_residue_from_pdb(insertp_combined_begin, insertp+".pdb", insertp_begin, 1, nres, 1);
	spliced_protein_map.get_combined_residue_from_pdb(insertp_combined_end, insertp+".pdb", insertp_end, 1, nres, last_repeated);

	// clear the loops in case there is anything there (for example when start_pose is called after
	// native_pose has been set)
	loops.clear();

	// define the first loop
	loop_begin = hostp_combined_begin-linker_size;
	loop_end = insertp_combined_begin+linker_size;
	loops.add_loop( loop_begin, loop_end, loop_begin + linker_size, 0/*offset*/, false/*is extended?*/ );

	// define the second loop
	loop_begin = insertp_combined_end-linker_size;
	loop_end = hostp_combined_end+linker_size;
	loops.add_loop( loop_begin, loop_end, loop_begin + linker_size, 0, false );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin create_insertion_map
///
/// @brief
///		function to copy pieces of poses into a combined pose
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified July 03 2006
////////////////////////////////////////////////////////////////////////////////
void create_insertion_map(
		std::vector <int> sizes,
		std::vector <int> new_start,
		std::vector <int> start,
		pose_ns::Pose const & host_protein,
		pose_ns::Pose const & insert_protein
		)
{
	using namespace domins_ns;

	spliced_protein_map.reset_spliced_protein_map();

	// protein information
	std::string host_name (params["hostp"].c_str());
	std::string insert_name (params["insertp"].c_str());

	spliced_protein_res_info.push_back(SplicedProtein(0, "", "", 0, 0, 0));
	spliced_protein_map.pushback_segment(host_name, "host_protein", new_start.at(0), start.at(0), sizes.at(0), host_protein);
	spliced_protein_map.pushback_segment(insert_name, "insert_protein", new_start.at(1), start.at(1), sizes.at(1), insert_protein);
	spliced_protein_map.pushback_segment(insert_name, "insert_protein", new_start.at(2), start.at(2), sizes.at(2), insert_protein);
	spliced_protein_map.pushback_segment(host_name, "host_protein", new_start.at(3), start.at(3), sizes.at(3), host_protein);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin initialize_struct()
///
/// @brief
///		set up all the information in the loop information
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, February 07 2007
///
/// @last_modified: February 07 2007
////////////////////////////////////////////////////////////////////////////////
void
initialize_struct(
		)
{
	using namespace domins_ns;

	// an instance of it is created in the domins_ns namespace that can be accessed
	// throughout all the functions in this file.
	hostp_nres = host_protein_pose.total_residue();
	insertp_nres = insert_protein_pose.total_residue();
	hostp_begin = atoi(params["hostp_begin"].c_str());
	hostp_end = atoi(params["hostp_end"].c_str());
	hostp_start_res = atoi(params["hostp_start_res"].c_str());
	hostp_end_res = atoi(params["hostp_end_res"].c_str());
	insertp_begin = atoi(params["insertp_begin"].c_str());
	insertp_end = atoi(params["insertp_end"].c_str());
	insertp_start_res = atoi(params["insertp_start_res"].c_str());
	insertp_end_res = atoi(params["insertp_end_res"].c_str());
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_create_fold_tree()
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, April 25 2006
///
/// @last_modified: April 25 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_create_fold_tree(
		pose_ns::Pose & pose,
		bool const centroid, // true creates the normal tree, false creates insertion_flop tree
		int const type
		)
{
	using namespace domins_ns;

	// initialize values
	int const num_jump(2), dock_jump(1), loop_jump(2);
	FArray1D_int cuts( num_jump );
	FArray2D_int jump_point( 2, num_jump );

	// information for host protein
	std::string hostp = params["hostp"].c_str();

	// information for insertion protein
	std::string insertp = params["insertp"].c_str();

	// number of jumps, jump number for the dock and loop jumps
	// dock jump with be flexible while the loop jump will remain fixed
	cuts(dock_jump) = hostp_combined_begin;
	cuts(loop_jump) = hostp_combined_end-1;

	int last_repeated ( 1 ), overlap (0), temp_com (0);//, start(0), stop(0);
	int recover_hostp_end = hostp_end;
	int recover_insertp_start_res = insertp_start_res;

	domins_set_info(last_repeated, overlap);

	jump_point(1,loop_jump) = hostp_combined_begin-linker_size-1;
	jump_point(2,loop_jump) = hostp_combined_end+linker_size+1;

	// if centroid is true, create the normal domain insertion tree
	if ( centroid ) {
		temp_com = host_protein_pose.residue_center_of_mass(hostp_start_res,hostp_end_res);
		if ( temp_com > hostp_begin ) {
			temp_com = temp_com+overlap;
		}
		spliced_protein_map.get_combined_residue(hostp_com,hostp,temp_com,1,nres,0);

		temp_com = insert_protein_pose.residue_center_of_mass(insertp_start_res,insertp_end_res);
		if ( temp_com > insertp_nres ) {
			temp_com = temp_com-hostp_begin;
		}
		spliced_protein_map.get_combined_residue(insertp_com,insertp,temp_com,1,nres,last_repeated);

		if ( hostp_part1_com == hostp_com )
			hostp_com += 1;

		if ( hostp_part2_com == hostp_com )
			hostp_com -= 1;

		jump_point(1,dock_jump) = hostp_com;
		jump_point(2,dock_jump) = insertp_com;

	// if centroid is false, create the insertion flop tree
	} else {
		if (type == 1 ) {
			jump_point(1,dock_jump) = hostp_combined_begin-linker_size;
			jump_point(2,dock_jump) = hostp_combined_begin+linker_size;
			std::cout << "type1 tree: " << jump_point(1,dock_jump) << ' ' << jump_point(2,dock_jump) << std::endl;
		} else if (type == 2 ) {
			jump_point(1,dock_jump) = hostp_combined_end-linker_size;
			jump_point(2,dock_jump) = hostp_combined_end+linker_size;
			std::cout << "type2 tree: " << jump_point(1,dock_jump) << ' ' << jump_point(2,dock_jump) << std::endl;
		}
	}
	pose_ns::Fold_tree f;
	f.tree_from_jumps_and_cuts( nres, num_jump, jump_point, cuts );
	pose.set_fold_tree(f);
	std::cout << "fold_tree: " << pose.fold_tree() << std::endl;

	hostp_end = recover_hostp_end;
	insertp_start_res = recover_insertp_start_res;

	// allow the backbone to move for the loops
	// allow repacking of the loops
	pose_loops_set_allow_bb_move( pose, loops);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_set_info
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified February 22 2007
////////////////////////////////////////////////////////////////////////////////
void
domins_set_info(
		int & last_repeated,
		int & overlap
		)
{
	using namespace domins_ns;

	bool circular (true);

	if ((insertp_begin ==  insertp_start_res) && (insertp_end == insertp_end_res)) {
		circular = false;
	}

	// determine whether any of the residues are repeated due to circular permutation
	if (circular && (insertp_end >= insertp_begin)) {
		last_repeated = 0;
	} else {
		last_repeated = 1;
	}

	// determine if the host domain is sequential
	// or whether the insertion domain is sequential with the host domain
	if (hostp_begin+1 == insertp_start_res) {
		// this is the case where the the insertion protein is sequential in pdb numbering following the
		// first part of the host domain
		insertp_start_res = 1;
		hostp_end = hostp_end - insertp_nres;
//		overlap = insertp_nres-1;
		overlap = 0;
	} else if (hostp_begin+1 != hostp_end) {
		// this is the case where the host protein is not sequential in pdb numbering
		insertp_start_res = 1;
		overlap = hostp_end - hostp_begin-1;
		std::cout << "overlap: " << overlap << std::endl;
		std::cout << "insertp_nres: " << insertp_nres << std::endl;
		if (overlap == insertp_nres) {
			hostp_end = hostp_end - insertp_nres;
		}
//		overlap = insertp_nres-2;
	} else {
		overlap = 0;
	}

	hostp_end_res = hostp_nres;
	insertp_end_res = insertp_nres-1;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin domins_initialize_pose
///
/// @brief read in two seperate pdbs and combine into one pose, then spit
///		out the pdb
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified August 16 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_initialize_pose(
		)
{
	using namespace domins_ns; // for domain_insertion
	using namespace files_paths;

	std::string insertion_file (""), filename("");
	bool ideal_pos (false);
	bool fullatom (false);
	set_pose_flag (true);
	// params and poses are kept in DomainInsertionMode.cc in a namespace
	// params is a map
	stringafteroption("insertion_file", "none", insertion_file);

	std::string loop_file( start_path + insertion_file);
	domins_get_parameters(params, loop_file);

	// information for host protein
	std::string hostp = params["hostp"].c_str();
	if ( has_suffix( hostp, ".pdb" ) ) hostp.erase( hostp.length() - 4 );
	filename = start_path + hostp + ".pdb";
	pose_from_pdb(host_protein_pose, filename, fullatom, ideal_pos);
	if (runlevel_ns::runlevel >= runlevel_ns::chat)
		host_protein_pose.pdb_info().print_pdb_information(host_protein_pose.total_residue());

	// information for insertion protein
	std::string insertp = params["insertp"].c_str();
	if ( has_suffix( insertp, ".pdb" ) ) insertp.erase( insertp.length() - 4 );
	filename = start_path + insertp + ".pdb";
	pose_from_pdb(insert_protein_pose, filename, fullatom, ideal_pos);
	if (runlevel_ns::runlevel >= runlevel_ns::chat)
		insert_protein_pose.pdb_info().print_pdb_information(insert_protein_pose.total_residue());

	// sets the size of the poses and all the information for the separate proteins
	initialize_struct();
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_get_parameters
///
/// @brief read insertion parameters from the text file
///
/// @detailed
///		file format:
///		-definition		XX
///
///		eg.
///		hostp 				1OMP.pdb
///		insertp				1N3W.pdb
///		hostp_start_res		1
///		hostp_end_res			632
///		hostp_begin				316
///		hostp_end					318
///		insertp_start_res	26
///		insertp_end_res		270
///		insertp_begin			226
///		insertp_end				227
///		linker_size					5
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified March 28 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_get_parameters(
				std::map<std::string, std::string> & params,
				std::string filename
				)
{
		using namespace files_paths;

		//open the file for reading
		utility::io::izstream param_file;
		std::string key, value;

		param_file.open(filename);

		if (!param_file) {
				std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
				std::cout << "ERROR: NO LOOP FILE DEFINED                     " << std::endl;
				std::cout << "Must define insertion file for -domain_insertion mode" << std::endl;
				std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// splits the line into words
		// puts the first word in the map
		// puts the second word in the value for the map
		while(!param_file.eof()) {
				param_file >> key >> value;
				params[key] = value;
		}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin copy_combined_pose
///
/// @brief
///		function to copy pieces of poses into a combined pose
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 03 2006
////////////////////////////////////////////////////////////////////////////////
void copy_combined_pose(
		std::vector <int> sizes,
		std::vector <int> new_start,
		std::vector <int> start,
		pose_ns::Pose const & host_protein,
		pose_ns::Pose const & insert_protein,
		pose_ns::Pose & return_protein
		)
{
		using namespace domins_ns;

		// size of the final pdb/pose is the sum of all the pieces
		int const nres ( sizes.at(0)+sizes.at(1)+sizes.at(2)+sizes.at(3) );

		// combine all the little pieces into one pose/pdb
		return_protein.simple_fold_tree(nres);
		return_protein.set_allow_bb_move( true );

		// create the combined pose (only when create_domins_fasta is true)
		return_protein.copy_segment(sizes.at(0), host_protein, new_start.at(0), start.at(0));
		return_protein.copy_segment(sizes.at(1), insert_protein, new_start.at(1), start.at(1));
		return_protein.copy_segment(sizes.at(2), insert_protein, new_start.at(2), start.at(2));
		return_protein.copy_segment(sizes.at(3), host_protein, new_start.at(3), start.at(3));
		create_insertion_map(sizes, new_start, start, host_protein, insert_protein);
		return_protein.set_allow_bb_move( false );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_native
///
/// @brief
///		reads in the coordinates of the native pose and stores them
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 03 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_native(
		pose_ns::Pose & native_pose
		)
{
	using namespace pose_ns;
	using namespace files_paths;
	using namespace domins_ns;

	if (create_domins_fasta) return;

	std::string filename("");
	std::vector < int> sizes, new_start, start;
	std::map<std::string,int> ip;

	// read in native
	// if file does not exist, use the starting pose as the fake_native
	utility::io::izstream param_file;

	filename = start_path + protein_name + ".native.pdb";
	param_file.open(filename);
	if (!param_file) {
		std::cout << "Could not find " << filename << ", using starting structure instead." << std::endl;
		filename = start_path + protein_name + ".pdb";
		param_file.open(filename);
		if (!param_file) {
			std::cout << "Could not find " << filename << " exiting...." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
	std::cout << "native_pose: " << filename << std::endl;
	pose_from_pdb(native_pose, filename, false, false);
	std::cout << "native_pose length: " << native_pose.total_residue() << std::endl;
	nres = native_pose.total_residue();
	domins_initialize_pose(); // reads in the insertion file
	pose_to_misc(native_pose);// forces the misc arrays to be the right size
	domins_generate_start(sizes, new_start, start, ip); // creates the insertion map which is needed for creating the fold tree
	create_insertion_map(sizes, new_start, start, host_protein_pose, insert_protein_pose);
	domins_input_map(sizes, new_start, start, ip);
	domins_initial_setup(native_pose); // sets the sizes and other initial steps
	score_set_current_pose( native_pose );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_start
///
/// @brief
///		reads in the coordinates of the native pose and stores them
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 03 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_start(
		pose_ns::Pose & start_pose
		)
{
	using namespace pose_ns;
	using namespace domins_ns;
	using namespace files_paths;

	if (create_domins_fasta) {
		domins_fasta_start(start_pose);
		return;
	}

	std::string filename("");
	std::vector < int> sizes, new_start, start;
	std::map<std::string,int> ip;

	// resets the current_score_pose that was set when the native pose was initialized
	// so that next time a score is called it doesn't think anything has been set
	score_reset_current_pose();

	// read in native
	// if file does not exist, use the starting pose as the fake_native
	utility::io::izstream param_file;

	filename = start_path + protein_name + ".pdb";
	param_file.open(filename);
	if (!param_file) {
		std::cout << "Could not find " << filename << " exiting...." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// now do all the initialization
	domins_initialize_pose(); // reads in the insertion file
	std::cout << "start_pose: " << filename << std::endl;
	pose_from_pdb(start_pose, filename, false, false);
	std::cout << "start_pose length: " << start_pose.total_residue() << std::endl;
	nres = start_pose.total_residue();
	pose_to_misc(start_pose);// forces the misc arrays to be the right size
	domins_generate_start(sizes, new_start, start, ip); // creates the insertion map which is needed for creating the fold tree
	create_insertion_map(sizes, new_start, start, host_protein_pose, insert_protein_pose);
	domins_input_map(sizes, new_start, start, ip);
	domins_initial_setup(start_pose); // sets the sizes and other initial steps
	score_set_current_pose( start_pose );
}

void
domins_fasta_start(
		pose_ns::Pose & start_pose
		)
{
	using namespace pose_ns;
	using namespace domins_ns;
	using namespace files_paths;

	std::vector < int> sizes, new_start, start;
	std::map<std::string,int> ip;
	std::string filename("");

	domins_initialize_pose(); // reads in the insertion file
//	domins_set_start_position(); // orients the two poses before going on to output the combined pose
	domins_generate_start(sizes, new_start, start, ip); // creates the insertion map which is needed for creating the fold tree
	copy_combined_pose(sizes, new_start, start, host_protein_pose, insert_protein_pose, start_pose);
	domins_input_map(sizes, new_start, start, ip);
	// copy the combined protein into the misc and then output a pdb
	pose_to_misc( start_pose );
	filename = start_path + protein_name + ".pdb";
	make_named_pdb(filename, false);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin find_min_loop_length()
///
/// @brief finds the conformation with the minimum length between the two loops
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified April 07 2006
////////////////////////////////////////////////////////////////////////////////
int
find_min_loop_length(
		std::vector <double> average_dist
		)
{
	// set a large distance to compare to
	double temp (1000.0);
	int x(0);

	for (int i=0; i < (int)average_dist.size(); i++) {
		if (average_dist.at(i) < temp ) {
			x=i;
			temp = average_dist.at(i);
		}
	}
	return x+1;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_calc_ins_rms
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified February 22 2007
////////////////////////////////////////////////////////////////////////////////
float
domins_calc_ins_rms(
		pose_ns::Pose const & pose
		)
{
	using namespace domins_ns;
	using namespace pose_ns;

	float rms (0.0);
	int count(0);

	for (int i=loops.at(0).start(); i<loops.at(1).stop(); ++i) {
		++count;
		rms += distance_squared(numeric::xyzVector_float(&pose.Eposition()(1,2,i)), numeric::xyzVector_float(&pose.native_pose().Eposition()(1,2,i)));
	}

	rms = std::sqrt(rms/count);
	return rms;
}

float
domins_calc_rb_rms(
		pose_ns::Pose const & pose
		)
{
	using namespace domins_ns;
	using namespace pose_ns;

	float rms(0.0);
	int count(0);

	for (int i=loops.at(0).stop()+1; i<loops.at(1).start()-1; ++i) {
		++count;
		rms += distance_squared(numeric::xyzVector_float(&pose.Eposition()(1,2,i)), numeric::xyzVector_float(&pose.native_pose().Eposition()(1,2,i)));
	}

	rms = std::sqrt(rms/count);
	return rms;
}

float
domins_calc_loop_rms(
		pose_ns::Pose const & pose,
		int const & loop
		)
{
	using namespace pose_ns;
	using namespace domins_ns;

	FArray1D_bool subset (pose.total_residue());
	subset = false;
	float rms(0.00);

	int loop_begin, loop_end;
	loop_begin = loops.at(loop-1).start();
	loop_end = loops.at(loop-1).stop();
	std::cout << "loops: " << loop_begin << ' ' << loop_end << std::endl;

	// set the loop to true so that the rms is calculated over this region
	for ( int i=loop_begin; i<=loop_end; ++i ) {
		subset(i) = true;
	}

	rms = CA_rmsd_by_subset(pose, pose.native_pose(), subset);
	return rms;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_calc_rms
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified February 22 2007
////////////////////////////////////////////////////////////////////////////////
float
domins_calc_rms(
		pose_ns::Pose const & pose
		)
{
	using namespace pose_ns;

	float rms;
	rms = CA_rmsd(pose, pose.native_pose());
	return rms;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_calc_midpoint
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified February 22 2007
////////////////////////////////////////////////////////////////////////////////
numeric::xyzVector <double>
domins_calc_midpoint(
		pose_ns::Pose const & pose,
		int const start,
		int const stop
		)
{
	using numeric::xyzVector_float;

	assert( start < stop );

	xyzVector_float center(0.0);
	xyzVector_float start_pos( &pose.Eposition()(1,2,start) ), stop_pos( &pose.Eposition()(1,2,stop) );
	center = midpoint(start_pos, stop_pos);
	return center;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin check_random_distribution
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Monica Berrondo
///
/// @last_modified February 22 2007
////////////////////////////////////////////////////////////////////////////////
void
check_random_distribution(
		pose_ns::Pose & pose
		)
{
	using namespace domins_ns;
	using namespace pose_ns;

	int const dock_jump(1);
	if ( ! pose.get_allow_jump_move( dock_jump ) ) return;
	int count(0);
	Jump perturbed_jump( pose.get_jump( dock_jump ) );
	int const pos1( pose.fold_tree().get_jump_point()(1, dock_jump) );
	int const pos2( pose.fold_tree().get_jump_point()(2, dock_jump) );
	pose_ns::Pose start_pose, out_pose;
	double rotation, latitude, longitude;
	float rotation_angle, current_rmsd(9999.0), best_rmsd(9999.0), good_rmsd(5.0);
	numeric::xyzVector <double> spin_axis, rot_center, host_linker_com, insert_linker_com, dist;
	numeric::xyzMatrix_double rotation_matrix;
	std::string move;

	out_pose = pose;
	start_pose = pose;

	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4 );
	float start_rmsd = domins_calc_rms(pose);
	pose.score( weight_map );
	float start_score = pose.get_0D_score( SCORE );
	std::cout << "start_rmsd: " <<  start_rmsd << std::endl;
	std::cout << "start_score: " <<  start_score << std::endl;

	for ( int i=1; i<=6; ++i ) {
		rotation_angle = 20.0*i;
		host_linker_com = domins_calc_midpoint(pose, hostp_begin-linker_size, hostp_end+linker_size);
		spin_axis = pos2 - pos1;
		numeric::xyzVector <double> perpendicular( 1, -(spin_axis.y()/spin_axis.x()), 0 );
		perpendicular.normalize();

		rot_center = get_loop_com(pose);
		perturbed_jump.set_rb_center( 1, pose.Eposition()(1,1,insertp_com), rot_center );
		perturbed_jump.rotation_by_axis( pose.Eposition()(1,1,hostp_com)/*center*/, spin_axis, rot_center, rotation_angle);
		pose.set_jump( dock_jump, perturbed_jump );
		pose.dump_pdb("cover_"+string_of(i)+"_pose.pdb");
		float current_rmsd = domins_calc_rms(pose);
		pose.score( weight_map );
		float current_score = pose.get_0D_score( SCORE );
		std::cout << "rmsd: " <<  current_rmsd << std::endl;
		std::cout << "score: " <<  current_score << std::endl;
	}
	exit(0);
	for ( int i=0; i<=10; ++i) {
				pose = start_pose;

				longitude = 0.0;
				latitude = 0.0;
				rotation = 36.0;
				rotation_matrix = Z_rot(  longitude   ) *
				Y_rot(  latitude ) *
				Z_rot(  rotation   );
				rot_center = get_loop_com(pose);
				std::cout << "rotation center: " << rot_center << std::endl;
				perturbed_jump.set_rb_center( 1, pose.Eposition()(1,1,pos2)/*downstream_jump*/, rot_center );
				perturbed_jump.rotation_by_matrix( pose.Eposition()(1,1,pos1), rot_center, rotation_matrix );
				pose.set_jump ( dock_jump, perturbed_jump );
				pose.dump_pdb("pose_distribution"+string_of(i)+".pdb");
	}
				exit(0);

	// create spin_axis between the center of mass of the beginning of the linkers of the two domains
	for ( int i=0; i<=10; ++i) {
		count = 0;
		for ( int k=0; k<=36/*long*/; ++k ) {
			for ( int n=0; n<=18/*lat*/; ++n ) {
				pose = start_pose;
				rotation_angle = i*36.0;
				longitude = k;
				latitude = n;
				rotation = rotation_angle;
				std::cout << "longitude: " << longitude << " latitude: " << latitude << " rotation: " << rotation << std::endl;
				rot_center = get_loop_com(pose);
				host_linker_com = domins_calc_midpoint(pose, hostp_begin-linker_size, hostp_end+linker_size);
				insert_linker_com = domins_calc_midpoint(pose, insertp_begin+linker_size, insertp_end-linker_size);
				spin_axis = host_linker_com - insert_linker_com;

				// spin about an axis
				perturbed_jump.set_rb_center( 1, pose.Eposition()(1,1,insertp_com), rot_center );
				perturbed_jump.rotation_by_axis( pose.Eposition()(1,1,hostp_com)/*center*/, spin_axis, rot_center, rotation_angle);
				pose.set_jump ( dock_jump, perturbed_jump );
//				pose.dump_pdb("spin_"+string_of(i)+"pose.pdb");

				// do a rotation by matrix
				// change this so that the rotation varies with degrees of latitude and longitude
				rotation_matrix = Z_rot(  longitude   ) *
				Y_rot(  latitude ) *
				Z_rot(  rotation   );
				rot_center = get_loop_com(pose);
				perturbed_jump.set_rb_center( 1, pose.Eposition()(1,1,insertp_com), rot_center );
				perturbed_jump.rotation_by_matrix( pose.Eposition()(1,1,insertp_com), rot_center, rotation_matrix );
				pose.set_jump ( dock_jump, perturbed_jump );

				// calculate the rms to the native for this move
				current_rmsd = domins_calc_rms(pose);
				std::cout << "current_rmsd: " << current_rmsd << " best_rmsd: " << best_rmsd << std::endl;
				// if the rms is the lowest seen, store the coordinates to a pose that will be dumped
				if (current_rmsd < best_rmsd) {
					best_rmsd = current_rmsd;
					out_pose = pose;
				}
				if (current_rmsd < good_rmsd) {
					pose.dump_pdb("spin"+string_of(i)+"_good"+string_of(count)+".pdb");
					count++;
				}
			}
		}
		// dump the coordinates for the best pose seen for this degree of spin
		out_pose.dump_pdb("best_pose_at_spin"+string_of(i)+".pdb");
	}
	pose = out_pose;
	pose.dump_pdb("best_pose.pdb");

	std::cout << "finished random..moving on......" << std::endl;
	// exit
	exit(0);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_loop_com()
///
/// @brief
///		calculates the center of mass of the linker regions
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, August 08 2006
///
/// @last_modified: August 08 2006
////////////////////////////////////////////////////////////////////////////////
numeric::xyzVector <double>
get_loop_com(
		pose_ns::Pose & pose
		)
{
	using namespace domins_ns;

	std::vector < numeric::xyzVector<double> > loops_1, loops_2;
	std::vector < numeric::xyzVector<double> > loops_positions;

	numeric::xyzVector <double> pos, center, cen1, cen2;

	for (int j=loops.at(0).start(); j<=loops.at(0).stop(); ++j) {
		pos = &pose.Eposition()(1,2,j);
		loops_positions.push_back(pos);
		loops_1.push_back(pos);
	}
	cen1 = vector_centroid(loops_1);

	for (int j=loops.at(1).start(); j<=loops.at(1).stop(); ++j) {
		pos = &pose.Eposition()(1,2,j);
		loops_positions.push_back(pos);
		loops_2.push_back(pos);
	}
	cen2 = vector_centroid(loops_2);

	center = midpoint( cen1, cen2 );
	return center;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin domins_get_mode()
///
/// @brief
///		gets the truefalseoption flags and decides which mode to run domain_insertion in
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: Monica Berrondo, August 08 2006
///
/// @last_modified: August 08 2006
////////////////////////////////////////////////////////////////////////////////
void
domins_get_mode(
		bool & fullatom,
		int & iter,
		int & cycles
		)
{
	using namespace runlevel_ns;
	using namespace domins_ns;

	do_main_protocol = true;
	do_insertion_flop = true;

	if ( benchmark ) {
		iter = 2;
		cycles = 1;
	} else {
		iter = 10;
		cycles = 5;
	}
	fullatom = false;
	fa_input_flag = false;
	if (truefalseoption("fa_output")) {
		fullatom = true;
		fa_set = true;
	}
	if (truefalseoption("fa_input")) {
		fullatom = true;
		fa_set = true;
		fa_input_flag = true;
	}
	if (truefalseoption("rb_relax")) rb_relax = true;
	if (truefalseoption("fast_relax_only")) {
		if( !fa_set ) {
			std::cout << "ERROR -- fast_relax requires fa_output or fa_input\n"
				        << "exit......." << std::endl;
			assert( false );
			utility::exit ( EXIT_FAILURE, __FILE__, __LINE__ );
		}
		do_main_protocol = false;
		do_insertion_flop = false;
	}
	if (truefalseoption("repack")) {
		if( !fa_set ) {
			std::cout << "ERROR -- repack requires fa_output or fa_input\n"
				        << "exit......." << std::endl;
			assert( false );
			utility::exit ( EXIT_FAILURE, __FILE__, __LINE__ );
		}
		repack_relax = true;
	}
	if (truefalseoption("repack_only")) {
		if( !fa_set ) {
			std::cout << "ERROR -- repack requires fa_output or fa_input\n"
				        << "exit......." << std::endl;
			assert( false );
			utility::exit ( EXIT_FAILURE, __FILE__, __LINE__ );
		}
		repack_relax = true;
		do_main_protocol = false;
		do_insertion_flop = false;
	}
	// if this flag is set to true no rigid body perturbations are used, initial or otherwise
	if (truefalseoption("loop_only")) {
		loop_only = true;
		do_insertion_flop = false;
	}
	// if this flag is set to true the initial perturbation is a dock_pert 3 8 8
	// instead of a randomly distributed perturbation
	if (truefalseoption("small_perturbation")) small_perturbation = true;
	// if this flag is set to true only rebuilds the loop once
	if (truefalseoption("no_loop")) {
		no_loop = true;
		do_insertion_flop = false;
	}
	// if this flag is set to true loop modeling runs without fragment insertions
	if (truefalseoption("no_frags")) no_frags = true;
	// if this flag is set to true no initial perturbation takes place (small pert or random)
	if (truefalseoption("no_perturb")) no_perturb = true;
}
