// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   Octree.cc
/// @brief  Modified "octree" class based on Stuart G. Mentzer's original implementation for neighbor list update/scoring.
/// @note   This was an experiment in template classes; currently only usable template types are those that provide an interface .x() .y() .z() to obtain coordinates.
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Will Sheffler (willsheffler@gmail.com)

#ifndef OCTREE_HH_
#define OCTREE_HH_


// package headers
#include <rootstock_types.hh>

// Numeric headers
#include <numeric/xyzTriple.hh>
#include <numeric/numeric.functions.hh>

// ObjexxFCL headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray3D.hh>

// C++ headers
#include <cmath>
#include <vector>
#include <limits>

namespace rootstock {


// typedefs
typedef numeric::xyzTriple< Size > CubeDim; // cube dimensions
typedef numeric::xyzTriple< Size > CubeIndex; // cube index


// global parameters
Length const OCTREE_EPSILON = 0.1; // expansion of original bounding box parameters (avoid floating point error)
Real const OCTREE_INFINITY = std::numeric_limits<Real>::infinity();


/// @brief Modified "octree" class based on Stuart G. Mentzer's original implementation for neighbor list update/scoring.
/// @note  Current implementation uses FArray3D< std::vector< T > > to store cube "tree" a la
///        Will Sheffler's discovery that arrays can be up to 100x - 300x faster than using std::map
///        for larger systems.  With arrays there is a memory hit due to unused cubes (which could
///        otherwise inactivated using a map), but for now this should be minor.  For unknown,
///        potentially non-compact, large point distributions consider using a map to save memory.
/// @note  This is not really "oct" nor "tree".  ;)  See libRosetta residue_neighbors.cc for more info.

template< typename T >
class Octree
{


private: // types


	typedef FArray3D< std::vector< T > > CubeTree; // cube "tree"


public: // construct/destruct


	/// @brief constructor
	/// @note  Bounding box is expanded by distance cutoff so to properly resolve
	/// @note  near neighbor queries for points outside the box.  This expansion
	/// @note  may potentially cause a memory usage issue.  If this happens, change
	/// @note  to using a data structure such as a map.
	inline
	Octree(
		Distance const distance_cutoff,
		PointPosition const & lower_corner,
		PointPosition const & upper_corner
	) :
		distance_cutoff_( distance_cutoff ),
		lower_corner_( lower_corner - distance_cutoff ),
		upper_corner_( upper_corner + distance_cutoff )
	{
		initialize();
	}


	/// @brief default constructor
	inline
	Octree() {}


	/// @brief default destructor
	inline
	~Octree() {}


public: // point manipulation


	/// @brief add point to tree
	inline
	void
	add(
		T const & p
	)
	{
		// check and make sure point is within expanded bounding box
		// due to FArray implementation cube index runs from 1 to cube_dim_
		assert( within_bounds( p.x(), p.y(), p.z() ) );

		CubeIndex const ci = cube_index( p.x(), p.y(), p.z() );

		// add point to tree
		cube_tree_( ci.x(), ci.y(), ci.z() ).push_back( p );
	}
//
//
//	void
//	remove();


public: // query


	/// @brief find nearest neighbor wrt point x,y,z and distance cutoff that Octree was constructed with
	inline
	T const *
	nearest_neighbor(
		Real const & x,
		Real const & y,
		Real const & z
	) const
	{
		using numeric::min;
		using numeric::max;
		using numeric::square;

		// initialize closest
		T const * closest_point = NULL;
		DistanceSquared closest_d_sq( OCTREE_INFINITY );

		// Query point lies outside of bounding box, return no neighbors.
		// This is proper behavior due to the expansion of the bounding box by
		// the distance cutoff.
		if ( !within_bounds( x, y, z ) ) {
			return closest_point;
		}

		// constants
		Distance const D_ZERO( 0.0 );
		DistanceSquared const distance_cutoff_sq( distance_cutoff_ * distance_cutoff_ );

		// cube index for point
		CubeIndex ci = cube_index( x, y, z );

		// cube-relative position (for fast cube exclusion tests)
		PointPosition cp = cube_relative_position( ci, x, y, z );

		// check its cube and adjacent cubes (<= all 27 of them with side_factor == 1 ; see cube_size() )
		for ( Size ix = max( ci.x() - 1, Size( 1 ) ), ixe = min( ci.x() + 1, cube_dim_.x()  ); ix <= ixe; ++ix) {
			for ( Size iy = max( ci.y() - 1, Size( 1 ) ), iye = min( ci.y() + 1, cube_dim_.y() ); iy <= iye; ++iy ) {
				for ( Size iz = max( ci.z() - 1, Size( 1 ) ), ize = min( ci.z() + 1, cube_dim_.z() ); iz <= ize; ++iz ) {

					// grab cube
					std::vector< T > const & ic = cube_tree_(ix, iy, iz);

					if ( // Test gives a 10% speed up in trials.
					 ( ix != ci.x() ? square( cp.x() - ( ix > ci.x() ? side_ : D_ZERO ) ) : D_ZERO ) +
					 ( iy != ci.y() ? square( cp.y() - ( iy > ci.y() ? side_ : D_ZERO ) ) : D_ZERO ) +
					 ( iz != ci.z() ? square( cp.z() - ( iz > ci.z() ? side_ : D_ZERO ) ) : D_ZERO )
					 <= distance_cutoff_sq
					)
					{ // max cutoff sphere intersects this cube so check each point in it
						// loop through points
						for (typename std::vector< T >::const_iterator ia = ic.begin(), iae = ic.end(); ia != iae; ++ia ) {
							T const & point( *ia );
							DistanceSquared const d_sq( distance_squared( x, y, z, point.x(), point.y(), point.z() ) );
							if ( d_sq < closest_d_sq ) {
								closest_point = &point;
								closest_d_sq = d_sq;
							}
						}
					}

				}
			}
		}

		return closest_point;
	}


	/// @brief find near neighbors wrt point x,y,z and a given query distance
	inline
	std::vector< T >
	near_neighbors(
		Distance const & qd,
		Real const & x,
		Real const & y,
		Real const & z
	) const
	{
		using numeric::min;
		using numeric::max;
		using numeric::square;

		assert( qd <= distance_cutoff_ );
		Real const qd_sq = qd * qd;

		Distance const D_ZERO( 0.0 );
		DistanceSquared const distance_cutoff_sq( distance_cutoff_ * distance_cutoff_ );

		// std::vector to return
		std::vector< T > nn;

		// Query point lies outside of bounding box, return no neighbors.
		// This is proper behavior due to the expansion of the bounding box by
		// the distance cutoff.
		if ( !within_bounds( x, y, z ) ) {
			return nn; // empty vector
		}

		// cube index for point
		CubeIndex ci = cube_index( x, y, z );

		// cube-relative position (for fast cube exclusion tests)
		PointPosition cp = cube_relative_position( ci, x, y, z );

		// check its cube and adjacent cubes (<= all 27 of them with side_factor == 1 ; see cube_size() )
		for ( Size ix = max( ci.x() - 1, Size( 1 ) ), ixe = min( ci.x() + 1, cube_dim_.x()  ); ix <= ixe; ++ix) {
			for ( Size iy = max( ci.y() - 1, Size( 1 ) ), iye = min( ci.y() + 1, cube_dim_.y() ); iy <= iye; ++iy ) {
				for ( Size iz = max( ci.z() - 1, Size( 1 ) ), ize = min( ci.z() + 1, cube_dim_.z() ); iz <= ize; ++iz ) {

					// grab cube
					std::vector< T > const & ic = cube_tree_(ix, iy, iz);

					if ( // Test gives a 10% speed up in trials.
					 ( ix != ci.x() ? square( cp.x() - ( ix > ci.x() ? side_ : D_ZERO ) ) : D_ZERO ) +
					 ( iy != ci.y() ? square( cp.y() - ( iy > ci.y() ? side_ : D_ZERO ) ) : D_ZERO ) +
					 ( iz != ci.z() ? square( cp.z() - ( iz > ci.z() ? side_ : D_ZERO ) ) : D_ZERO )
					 <= distance_cutoff_sq
					)
					{ // max cutoff sphere intersects this cube so check each point in it
						// loop through points
						for (typename std::vector< T >::const_iterator ia = ic.begin(), iae = ic.end(); ia != iae; ++ia ) {
							T const & point( *ia );
							DistanceSquared const d_sq( distance_squared( x, y, z, point.x(), point.y(), point.z() ) );
							if ( d_sq <= qd_sq ) {
								nn.push_back(point);
							}
						}
					}

				}
			}
		}

		return nn;
	}


//	void
//	upper_neighbors();


public: // cube methods


	/// @brief finds cube index for a point, due to FArray implementation, index runs from 1 to cube_dim_
	/// @note CubeIndex is typically xyzTriple< size_t >, so within computation of the index only the integral value is kept
	inline
	CubeIndex
	cube_index(
		Real const & x,
		Real const & y,
		Real const & z
	) const
	{
		CubeIndex ci(
		 (Size)( ( x - lower_corner_.x() ) * side_inv_ ) + 1,
		 (Size)( ( y - lower_corner_.y() ) * side_inv_ ) + 1,
		 (Size)( ( z - lower_corner_.z() ) * side_inv_ ) + 1
		);

		return ci;
	}


	/// @brief returns the number of points in a cube
	inline
	Size
	cube_point_count(
		CubeIndex const & ci
	)
	{
		return cube_tree_(ci.x(), ci.y(), ci.z()).size();
	}


	/// @brief returns list of neighboring cubes
	inline
	std::vector< CubeIndex >
	neighboring_cubes(
		CubeIndex const & ci
	)
	{
		using numeric::min;
		using numeric::max;

		std::vector< CubeIndex > v;

		for ( Size ix = max( ci.x() - 1, Size( 1 ) ), ixe = min( ci.x() + 1, cube_dim_.x()  ); ix <= ixe; ++ix) {
			for ( Size iy = max( ci.y() - 1, Size( 1 ) ), iye = min( ci.y() + 1, cube_dim_.y() ); iy <= iye; ++iy ) {
				for ( Size iz = max( ci.z() - 1, Size( 1 ) ), ize = min( ci.z() + 1, cube_dim_.z() ); iz <= ize; ++iz ) {
					// query cube
					if ( cube_tree_(ix, iy, iz).size() > 0 ) {
						v.push_back( CubeIndex( ix, iy, iz ) );
					}
				}
			}
		}

		return v;
	}


private: // methods


	/// @brief setup bounding box, this routine is currently pseudocode and does nothing!
//	void
//	bounding_box()
//	{
		// init upper and lower corners
		// for each Point:
		//    lower_corner_ <- min(Point, lower_corner_) // when using libR triple: lower_corner_.min(Point)
		//    upper_corner_ <- max(Point, upper_corner_) // when using libR triple: upper_corner_.max(Point)
		// expand corners:
		//    epsilon <- std::numeric_limits< Real >::epsilon(); // fine grained control
		//    epsilon <- 0.1; // better to catch most everything, should be greater than fp-error of queries
		//    lower_corner_ -= epsilon
		//    upper_corner_ +- epsilon
//	}


	/// @brief sets cube size and dimensions with respect to bounding box
	inline
	void
	cube_size()
	{
		using numeric::max;

		Size const side_factor( 1 ); // 1 factor => Check <= 27 adjacent cubes // 2 factor => Check <= 8 adjacent cubes
		side_ = side_factor * distance_cutoff_;
		side_inv_ = Distance( 1 ) / side_;

		cube_dim_ = CubeDim( // Cube dimensions
			Size( std::ceil( ( upper_corner_.x() - lower_corner_.x() ) * side_inv_ ) ),
			Size( std::ceil( ( upper_corner_.y() - lower_corner_.y() ) * side_inv_ ) ),
			Size( std::ceil( ( upper_corner_.z() - lower_corner_.z() ) * side_inv_ ) )
		);
		// We rounded up the number of cubes in each dimension
		// We use cubes of exactly side x side x side dimensions
		// We treat the (1,1,1) cube as touching lower_corner_ at its low corner
		// The "highest" cube generally extends beyond upper_corner_
		// We call this the expanded bounding box

		// Number of potential cubes in expanded bounding box.
		n_cube_ = cube_dim_.x() * cube_dim_.y() * cube_dim_.z();
	}


	/// @brief finds cube-relative position (for fast cube exclusion tests)
	inline
	PointPosition
	cube_relative_position(
		CubeIndex const & ci,
		Real const & x,
		Real const & y,
		Real const & z
	) const
	{
		PointPosition cp(
		 x - ( lower_corner_.x() + ( ( ci.x() - 1 ) * side_ ) ),
		 y - ( lower_corner_.y() + ( ( ci.y() - 1 ) * side_ ) ),
		 z - ( lower_corner_.z() + ( ( ci.z() - 1 ) * side_ ) )
		);

		return cp;
	}


private: // math -- remove if libR available? (i.e. use xyzTriple/xyzVector in combination with libR objects where possible)


	inline
	DistanceSquared
	distance_squared( Real const a, Real const b, Real const c, Real const x, Real const y, Real const z ) const
	{
		return numeric::square( a - x ) + numeric::square( b - y ) + numeric::square( c - z );
	}


private: // handlers for points outside of bounding box


	/// @brief is point within the bounding box of this octree?
	inline
	bool
	within_bounds(
		Real const & x,
		Real const & y,
		Real const & z
	) const
	{
		return lower_corner_.x() <= x && lower_corner_.y() <= y && lower_corner_.z() <= z &&
		       x <= upper_corner_.x() && y <= upper_corner_.y() && z <= upper_corner_.z();
	}


	/// @brief the closest cube to this point (which may be outside the bounding box)
	/// @note  this routine works properly because the bounding box/cubes are axis aligned.
	inline
	CubeIndex
	closest_cube(
		Real const & x,
		Real const & y,
		Real const & z
	) const
	{
		if ( within_bounds ) {
			return cube_index( x, y, z );
		}

		Real const x_prime = x < lower_corner_.x() ? lower_corner_.x() : ( upper_corner_.x() < x ? upper_corner_.x() : x );
		Real const y_prime = y < lower_corner_.y() ? lower_corner_.y() : ( upper_corner_.y() < y ? upper_corner_.y() : y );
		Real const z_prime = z < lower_corner_.z() ? lower_corner_.z() : ( upper_corner_.z() < z ? upper_corner_.z() : z );

		return cube_index( x_prime, y_prime, z_prime );
	}


private: // initialization


	/// @brief initialization
	inline
	void
	initialize()
	{
		// setup cube size and dimensions wrt bounding box
		cube_size();

		// setup cube tree
		cube_tree_ = CubeTree( cube_dim_.x(), cube_dim_.y(), cube_dim_.z() );
	}


private: // data

	Distance distance_cutoff_; // maximum possible distance for which neighbors can be found

	PointPosition lower_corner_; // lower corner of bounding box
	PointPosition upper_corner_; // upper corner of bounding box

	CubeDim cube_dim_; // cube dimensions
	Size n_cube_; // number of cubes

	Distance side_; // cube size
	Distance side_inv_; // invert cube size

	CubeTree cube_tree_; // cube "tree", FArray3D< std::vector< T > > ; each (cube-indexed) entry stores std::vector< T >

};

} // namespace rootstock

#endif /*OCTREE_HH_*/
