// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

//xa   Implementation for new simulated annealing routine that will optimize
//xa   a single sequence for multiple structures (Ambroggio & Kuhlman 2006)

// Rosetta Headers
#include "design.h"              // needed for conv_limit_mod
#include "files_paths.h"         // needed for reading in equiv_resfile
#include "EquivResfileReader.h"
#include "FixbbSimAnnealer.h"
#include "InteractionGraphBase.h" // needed for calling ig functions
#include "misc.h"                // needed for res_num_from_pdb_res_num_chain
#include "pack.h"                // needed for rotindex
#include "OneGeneTwoProteins.h"
#include "param.h"               // needed for MAX_RES
#include "pdb.h"                 // needed for res_num_from_pdb_res_num_chain
#include "random_numbers.h"      // needed for random_range
#include "ResidueData.h"
#include "RotamerSet.h"
#include "XUtilities.cc"

// C++ Headers
#include <fstream>
#include <iostream>
#include <set>
#include <iterator>

//Utility Headers
#include <utility/basic_sys_util.hh>

namespace pack{


  ////////////////////////////////////////////////////////////////////////////////

  OneGeneTwoProteins::OneGeneTwoProteins(
				       const RotamerSet * p_rotamer_set,
				       FArray2DB_bool & neighborlist,
				       FArray1D_int & bestrotamer_at_seqpos,
				       float & bestenergy,
				       bool start_with_current, // start simulation with current rotamers
				       pack::InteractionGraphBase * ig,
				       FArray1DB_int & current_rot_index,
				       bool calc_rot_freq,
				       FArray1D_float & rot_freq
				       ):
    ParallelSimAnnealBase(
			  p_rotamer_set,
			  neighborlist,
			  bestrotamer_at_seqpos,
			  bestenergy,
			  start_with_current,
			  ig,
			  current_rot_index,
			  calc_rot_freq,
			  rot_freq
			  )

  {
    enforce_complementary_amino_acids_for_equivalent_sequence_positions();
  }

  ////////////////////////////////////////////////////////////////////////////////

  OneGeneTwoProteins::~OneGeneTwoProteins()
  {}

  void OneGeneTwoProteins::check_that_only_pairs_of_equivalent_residues()
  {
    for( unsigned int i = 0; i < equivalent_residues.size(); i++)
      if( equivalent_residues.at(i).size() != 2 ){
	cout << "More than 2 sequence positions set to be complementary, exiting\n";
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
      }
  }

  void OneGeneTwoProteins::enforce_complementary_amino_acids_for_equivalent_sequence_positions()
  {
    check_that_only_pairs_of_equivalent_residues();

    list<string> sense_codons, antisense_codons, complementary_codons;
    map<int, ResidueData>::iterator ResDataMap_iter;

    for( ResDataMap_iter = master_sequence_map.begin(); ResDataMap_iter != master_sequence_map.end(); ResDataMap_iter++)
      for( unsigned int i = 0; i < equivalent_residues.size(); i++)
	if( ResDataMap_iter->first == equivalent_residues.at(i).front() ){
	  ResDataMap_iter->second.return_possible_codons( sense_codons );
	  sense_codons.sort();

	  master_sequence_map.find( equivalent_residues.at(i).back() )->second.return_possible_anticodons( antisense_codons );
	  antisense_codons.sort();
	  set_intersection( sense_codons.begin(), sense_codons.end(), antisense_codons.begin(), antisense_codons.end(), back_inserter( complementary_codons ) );

	  if( complementary_codons.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__); // complementary sequence impossible for input

	  ResDataMap_iter->second.delete_possible_amino_acids_not_in_codon_list( complementary_codons );
	  ResDataMap_iter->second.delete_possible_codons_not_in_codon_list( complementary_codons );
	  master_sequence_map.find( equivalent_residues.at(i).back() )->second.delete_possible_amino_acids_not_in_anticodon_list( complementary_codons );
	  master_sequence_map.find( equivalent_residues.at(i).back() )->second.delete_possible_codons_not_in_anticodon_list( complementary_codons );

	  sense_codons.clear();
	  antisense_codons.clear();
	  complementary_codons.clear();
	}
  }

  void OneGeneTwoProteins::create_a_random_sequence()
  {
    set<int> mutated_residues;
    string mutant_codon;
    list<int>::iterator list_oI_iter;
    std::map<int, ResidueData>::iterator intResDataMap_iter;

    for(unsigned int i = 0; i < equivalent_residues.size(); i++){
      master_sequence_map.find( equivalent_residues.at( i ).front() )->second.mutate_codon();
      master_sequence_map.find( equivalent_residues.at( i ).front() )->second.get_curr_codon( mutant_codon );
      mutated_residues.insert( equivalent_residues.at( i ).front() );

      master_sequence_map.find( equivalent_residues.at( i ).back() )->second.set_curr_codon( ResidueData::return_reverse_complement( mutant_codon ) );
      mutated_residues.insert( equivalent_residues.at( i ).back() );
    }

    for( intResDataMap_iter = master_sequence_map.begin(); intResDataMap_iter != master_sequence_map.end(); intResDataMap_iter++)
      if( mutated_residues.find( intResDataMap_iter->first ) == mutated_residues.end() ) intResDataMap_iter->second.mutate_codon();
  }

void OneGeneTwoProteins::propogate_mutations_to_complementary_residues()
{
  if( mutant_sequence_positions.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

  string mutant_codon;
  list<int>::iterator mutant_seq_iter;
  list<int> additional_mutant_sequence_positions;

  for( mutant_seq_iter = mutant_sequence_positions.begin(); mutant_seq_iter != mutant_sequence_positions.end(); mutant_seq_iter++)
    for(unsigned int i = 0; i < equivalent_residues.size(); i++)
      if( * mutant_seq_iter == equivalent_residues.at(i).front() ){
	master_sequence_map.find( * mutant_seq_iter )->second.get_curr_codon( mutant_codon );
	master_sequence_map.find( equivalent_residues.at( i ).back() )->second.set_curr_codon( ResidueData::return_reverse_complement( mutant_codon ) );
	additional_mutant_sequence_positions.push_back( equivalent_residues.at( i ).back() );
      }

  mutant_sequence_positions.sort();
  additional_mutant_sequence_positions.sort();
  mutant_sequence_positions.merge( additional_mutant_sequence_positions );
  mutant_sequence_positions.unique();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OneGeneTwoProteins::run
///
/// @brief Find sequences for two proteins that can be encoded by one gene
///
/// @refrences
///
/// @authors Xavier Ambroggio
////////////////////////////////////////////////////////////////////////////////

  void OneGeneTwoProteins::run()
  {
    using namespace design;
    std::cout << "Starting Reverse Complementary parallel simulated annealing\n";

    create_a_random_sequence();

    fill_rot_to_pack_for_current_sequence();

    start_with_current_ = false;

    if( rot_to_pack.empty() ) return;

    pack_set_of_rotamers();

    previous_energy = energy_of_best_run = bestenergy_;

    accept_new_sequence_from_packing_run();

    start_with_current_ = true;

    setup_iterations( number_of_possible_mutations() );

    for( int i = 1; i <= get_outeriterations(); i++ ){ // annealing outer loop
      setup_temperature( i );

      for( int j = 1; j <= get_inneriterations() + ( i * 200 * conv_limit_mod ); j++ ){ // inner loop for mutations

	make_a_sequential_mutation();

	propogate_mutations_to_complementary_residues();

	add_rotamers_to_pack_for_residue_list( mutant_sequence_positions );

	add_rotamers_to_pack_for_neighbors_of_mutated_residues();

	add_current_rotamers_and_start_with_them();

	if( rot_to_pack.empty() ) continue;

	pack_set_of_rotamers();

	delta_energy = bestenergy_ - previous_energy;

	if( pass_metropolis( delta_energy ) ){
	  accept_new_sequence_from_packing_run();
	}else{
	  reject_mutations();
	}
      }
      revert_to_best_sequence();
      print_current_sequences();
    }
    output_best_rotamers();
  }


} //end of namespace
