// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.4 $
//  $Date: 2006/02/16 20:23:54 $
//  $Author: plato $

// Rosetta Headers
#include "PrecomputedInteractionGraph.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>

// STL Headers
#include <iostream>

namespace pack{

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::set_aatypes_for_node
///
/// @brief
/// sets the amino acid types for each state of a particular node
///
/// @detailed
///
/// @param
/// node_ind - [in] - the index of the node in question
/// aatype - [in] - the vector of amino acid types, one entry for each state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::set_aatypes_for_node
(
	int node_ind,
	std::vector< int > const & aatypes
)
{
	((PrecomputedPairEnergiesNode*) get_node(node_ind ))->
		set_amino_acid_types( aatypes );
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::add_to_nodes_one_body_energy
///
/// @brief
/// interface to PrecomputedPairEnergiesNode::add_to_nodes_one_body_energy
///
/// @detailed
///
/// @param
/// node_ind - [in] - the node in question
/// one_body_energies - [in] - the energies to be added to the one-body energies
/// 	on that node.  One entry per state.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::add_to_nodes_one_body_energy
(
	int node_ind,
	std::vector< float > const & one_body_energies
)
{
	int num_states_for_node = get_node( node_ind )->get_num_states();
	for (int ii = 1; ii <= num_states_for_node; ++ii )
	{
		((PrecomputedPairEnergiesNode*) get_node( node_ind ))->
			add_to_one_body_energy( ii, one_body_energies[ ii ] );
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::add_to_nodes_one_body_energy
///
/// @brief
/// interface to PrecomputedPairEnergiesNode::add_to_nodes_one_body_energy
///
/// @detailed
///
/// @param
/// node_ind - [in] - the node in question
/// one_body_energies - [in] - the energies to be added to the one-body energies
/// 	on that node.  One entry per state.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::add_to_nodes_one_body_energy
(
	int node_ind,
	FArray1DB_float const & one_body_energies
)
{
	int num_states_for_node = get_node( node_ind )->get_num_states();
	for (int ii = 1; ii <= num_states_for_node; ++ii )
	{
		((PrecomputedPairEnergiesNode*) get_node( node_ind ))->
			add_to_one_body_energy( ii, one_body_energies( ii ) );
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::add_to_nodes_one_body_energy
///
/// @brief
/// interface to PDNode::add_to_nodes_one_body_energy
///
/// @detailed
///
/// @param
/// node_ind - [in] - the index of the node in question
/// state_id - [in] - the state in question
/// one_body_energy - [in] - the energy to be added
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::add_to_nodes_one_body_energy
(
	int node_ind,
	int state_id,
	float const one_body_energy
)
{
	((PrecomputedPairEnergiesNode*) get_node( node_ind ))->
		add_to_one_body_energy( state_id, one_body_energy );
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::set_sparse_aa_info_for_edge
///
/// @brief
/// interface to PDEdge::set_sparse_aa_info
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// sparse_conn_info - [in] - the boolean 2-D array of amino-acid neighbor info
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::set_sparse_aa_info_for_edge(
	int node1,
	int node2,
	FArray2DB_bool const & sparse_conn_info)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{	std::cerr <<
			"WARNING:: you've input sparse aa info for an edge that does not exist"
			<< std::endl;
		return;
	}
	edge->set_sparse_aa_info( sparse_conn_info );
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::get_sparse_aa_info_for_edge
///
/// @brief
/// returns true if node1aa and node2aa are amino acid neighbors
///
/// @detailed
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// aa_node1 - [in] - the amino acid type for the node with the smaller index
/// aa_node2 - [in] - the amino acid type for the node with the larger index
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool PrecomputedPairEnergiesInteractionGraph::get_sparse_aa_info_for_edge(
	int node1,
	int node2,
	int node1aa,
	int node2aa
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{
	std::cerr <<
	"WARNING:: you've requested sparse aa info for an edge that does not exist"
	<< std::endl;
		return false;
	}

	return edge->get_sparse_aa_info( node1aa, node2aa );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::force_aa_neighbors_for_edge
///
/// @brief
/// interface to PrecomputedPairEnergiesEdge::force_aa_neighbors
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// aa_node1 - [in] - the amino acid type for the node with the smaller index
/// aa_node2 - [in] - the amino acid type for the node with the larger index
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::force_aa_neighbors_for_edge
(
	int node1,
	int node2,
	int aa_node1,
	int aa_node2
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{
		return;
	}
	edge->force_aa_neighbors(aa_node1, aa_node2);
	return;
}




////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::
/// force_all_aa_neighbors_for_edge
///
/// @brief
/// interface to PDEdge::force_aa_neighbors
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// aa_node1 - [in] - the amino acid type for the node with the smaller index
/// aa_node2 - [in] - the amino acid type for the node with the larger index
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::force_all_aa_neighbors_for_edge
(
	int node1,
	int node2
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge(node1, node2 );
	if (edge == NULL)
	{
		add_edge( node1, node2 );
		edge = (PrecomputedPairEnergiesEdge*) find_edge(node1, node2);

		FArray2D_bool all_aa_neighbors( get_num_aatypes() , get_num_aatypes(), true);
		edge->set_sparse_aa_info( all_aa_neighbors );
	}
	else
	{
		edge->force_all_aa_neighbors();
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::
///  add_to_two_body_energies_for_edge
///
/// @brief
/// interface for PrecomputedPairEnergiesEdge::add_to_two_body_energies
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// oversized_res_res_energy_array - [in] - the large table
/// 	of rotamer pair energies
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::add_to_two_body_energies_for_edge
(
	int node1,
	int node2,
	FArray2DB_float const & oversized_res_res_energy_array
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{	std::cerr <<
			"WARNING:: you've input edge energies for an edge that does not exist"
			<< std::endl;
		return;
	}
	edge->add_to_two_body_energies( oversized_res_res_energy_array );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::add_to_two_body_energies_for_edge
///
/// @brief
/// interface to PrecomputedPairEnergiesEdge::add_to_two_body_energies
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// state_node1 - [in] - state on smaller-indexed node
/// state_node2 - [in] - state on larger-indexed node
/// two_body_energy  - [in] - the energy for this state pair
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
PrecomputedPairEnergiesInteractionGraph::add_to_two_body_energies_for_edge
(
	int node1,
	int node2,
	int state_node1,
	int state_node2,
	float const two_body_energy
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{	std::cerr <<
			"WARNING:: you've input edge energies for an edge that does not exist"
			<< std::endl;
		return;
	}
	edge->add_to_two_body_energy( state_node1, state_node2, two_body_energy );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::set_two_body_energy_for_edge
///
/// @brief
/// interface to PDEdge::set_two_body_energy
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// state_node1 - [in] - state on smaller-indexed node
/// state_node2 - [in] - state on larger-indexed node
/// two_body_energy  - [in] - the energy for this state pair
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::set_two_body_energy_for_edge(
	int node1,
	int node2,
	int state_node1,
	int state_node2,
	float const two_body_energy
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{	std::cerr <<
			"WARNING:: you've input edge energies for an edge that does not exist"
			<< std::endl;
		return;
	}
	edge->set_two_body_energy( state_node1, state_node2, two_body_energy );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::clear_two_body_energy_for_edge
///
/// @brief
/// interface to PDEdge::clear_two_body_energy
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// state_node1 - [in] - state on smaller-indexed node
/// state_node2 - [in] - state on larger-indexed node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::clear_two_body_energy_for_edge(
	int node1,
	int node2,
	int state_node1,
	int state_node2
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL) return;
	edge->set_two_body_energy( state_node1, state_node2, 0 );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::get_two_body_energy_for_edge
///
/// @brief
/// interface to PrecomputedPairEnergiesEdge::get_two_body_energy
///  - returns the state pair energy
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
/// state_node1 - [in] - state on smaller-indexed node
/// state_node2 - [in] - state on larger-indexed node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PrecomputedPairEnergiesInteractionGraph::get_two_body_energy_for_edge(
	int node1,
	int node2,
	int state_node1,
	int state_node2
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{
		return 0;
	}
	return edge->get_two_body_energy( state_node1, state_node2 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PrecomputedPairEnergiesInteractionGraph::declare_edge_energies_final
///
/// @brief
/// call this if you're done storing energies in an edge - it will reduce
/// the memory usage for that edge if possible
///
/// @detailed
///
/// @param
/// node1 - [in] - the index of the smaller-indexed node
/// node2 - [in] - the index of the larger-indexed node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PrecomputedPairEnergiesInteractionGraph::declare_edge_energies_final
(
	int node1,
	int node2
)
{
	PrecomputedPairEnergiesEdge* edge =
		(PrecomputedPairEnergiesEdge*) find_edge( node1, node2 );
	if (edge == NULL)
	{
		return;
	}
	edge->declare_energies_final();
}

} //end namespace
