// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

//C++ Headers
#include <algorithm>
#include <iostream>
#include <map>
#include <set>
#include <list>
#include <vector>
#include <string>
#include <functional>

//Rosetta Headers
#include "random_numbers.h"
#include "ResidueData.h"
#include "XUtilities.cc"

//Utility Headers
#include <utility/basic_sys_util.hh>

using namespace std;

  //////////////////////////////////////////////////////////////////////////////////////
  // ResidueData: Data structure to store and manipulate residue information
  //////////////////////////////////////////////////////////////////////////////////////

ResidueData::ResidueData( const map<int, list<int> > & possible_amino_acids_with_rotamers_input )
{
  if( possible_amino_acids_with_rotamers_input.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

  only_one_possible_amino_acid = false;
  no_rotamers = false;

  possible_amino_acids_with_rotamers = possible_amino_acids_with_rotamers_input;

  if( possible_amino_acids_with_rotamers.size() == 1 ){
    only_one_possible_amino_acid = true;
    curr_aa = prev_aa = possible_amino_acids_with_rotamers.begin()->first;
    return_codon_for_amino_acid( curr_aa, curr_codon );
    prev_codon = curr_codon = best_codon;
    if( possible_amino_acids_with_rotamers.begin()->second.empty() )
      no_rotamers = true;
  }else{
    curr_aa = prev_aa = 0;
    curr_codon.clear();
    prev_codon.clear();
  }

  curr_rot = best_rot = prev_rot = 0;

  map<int, list<int> >::iterator map_i_loi_iter;

  for( map_i_loi_iter = possible_amino_acids_with_rotamers.begin(); map_i_loi_iter != possible_amino_acids_with_rotamers.end(); map_i_loi_iter++)
    if( ! no_rotamers ){
      map_i_loi_iter->second.sort();
      map_i_loi_iter->second.unique();
    }

  list<int> possible_amino_acids;
  XUtilities::return_all_keys( possible_amino_acids_with_rotamers, possible_amino_acids );
  XUtilities::return_keys_given_values( CodonTable, possible_amino_acids, possible_codons );
  possible_codons.sort();
}

void ResidueData::fill_neighbor_list( const list<int> & neighbors_input )
{
  neighbors = neighbors_input;
  neighbors.sort();
  neighbors.unique();
}

void ResidueData::mutate_residue()
{
  if( only_one_possible_amino_acid ) return;

  accept_aa();
  curr_aa = XUtilities::random_element_from_container( possible_amino_acids_with_rotamers );
  assign_rand_rot();
}

void ResidueData::mutate_codon()
{
  string trial_codon;

  if( curr_codon.empty() ){
    curr_codon = XUtilities::random_element_from_container( possible_codons );
    curr_aa = CodonTable.find( curr_codon )->second;
  }else if( possible_codons.size() > 1){
    accept_codon();
    if( ! only_one_possible_amino_acid ) accept_aa();
    do{
      trial_codon = curr_codon;
      trial_codon.at( static_cast<int>( trial_codon.size() * ran3() ) ) = DNA_bases.at( static_cast<int>( DNA_bases.size() * ran3() ) );
      curr_aa = CodonTable.find( trial_codon )->second;
    }while( possible_amino_acids_with_rotamers.find( curr_aa ) == possible_amino_acids_with_rotamers.end()
	    || find( possible_codons.begin(), possible_codons.end(), trial_codon ) == possible_codons.end() );
    curr_codon = trial_codon;
  }else return;

  if( ! no_rotamers ) assign_rand_rot();
  return;
}

void ResidueData::assign_rand_rot()
{
  if( no_rotamers || possible_amino_acids_with_rotamers.find( curr_aa ) == possible_amino_acids_with_rotamers.end() ) return;

  list<int> possible_rotamers_for_current_amino_acid;

  if( curr_rot != 0 ) accept_rot();
  possible_rotamers_for_current_amino_acid = possible_amino_acids_with_rotamers.find( curr_aa )->second;

  curr_rot = XUtilities::random_element_from_container( possible_rotamers_for_current_amino_acid );
}

void ResidueData::accept_aa()
{
  if( only_one_possible_amino_acid || curr_aa == 0 ) return;
  prev_aa = curr_aa;
}

void ResidueData::accept_codon()
{
  if( only_one_possible_amino_acid || curr_codon.empty() ) return;
  prev_codon = curr_codon;
}

void ResidueData::accept_rot()
{
  if( curr_rot == 0 ) return;
  prev_rot = curr_rot;
}

void ResidueData::accept_residue()
{
  accept_aa();
  accept_codon();
  accept_rot();
}


void ResidueData::revert_aa()
{
  if( only_one_possible_amino_acid ) return;
  curr_aa = prev_aa;
}

void ResidueData::revert_codon()
{
  if( only_one_possible_amino_acid ) return;
  curr_codon = prev_codon;
}


void ResidueData::revert_rot()
{
  if( no_rotamers ) return;
  curr_rot = prev_rot;
}

void ResidueData::revert_residue()
{
  revert_aa();
  revert_codon();
  revert_rot();
}

void ResidueData::print_state()
{
  if( no_rotamers ) cout << "only amino acid " << curr_aa << ", no rotamers\n";
  else if( only_one_possible_amino_acid ) cout << "only amino acid " << curr_aa  << ", current codon " << curr_codon
					       << ", current rotamer " << curr_rot << ", best rotamer " << best_rot << endl;
  else cout << "current amino acid " << curr_aa << ", current codon " << curr_codon
	    << ", current rotamer " << curr_rot  << ", best rotamer " << best_rot << endl;
}

void ResidueData::print_amino_acid_rotamer_maps()
{

  map<int, list<int> >::iterator map_i_loi_iter;
  list<int>::iterator intList_iter;

  for( map_i_loi_iter = possible_amino_acids_with_rotamers.begin(); map_i_loi_iter != possible_amino_acids_with_rotamers.end(); map_i_loi_iter++){
    if( no_rotamers ) cout << "Amino acid " << map_i_loi_iter->first << ", has no rotamers." << endl;
    else{
      cout << "Amino acid " << map_i_loi_iter->first << ", has rotamers:" << endl;
      for(  intList_iter = map_i_loi_iter->second.begin(); intList_iter != map_i_loi_iter->second.end(); intList_iter++ )
	cout << * intList_iter << " ";
    }
    cout << "\n";
  }
}

void ResidueData::print_possible_amino_acids()
{
  map<int, list<int> >::iterator map_i_loi_iter;
  list<int>::iterator intList_iter;

  for( map_i_loi_iter = possible_amino_acids_with_rotamers.begin(); map_i_loi_iter != possible_amino_acids_with_rotamers.end(); map_i_loi_iter++)
    cout << map_i_loi_iter->first << " ";
  cout << "\n";
}

void ResidueData::print_neighbors()
{
  list<int>::iterator intList_iter;

  for(  intList_iter = neighbors.begin(); intList_iter != neighbors.end(); intList_iter++ )
    cout << * intList_iter << " ";
  cout << "\n";
}

void ResidueData::return_possible_amino_acids( list<int> & possible_amino_acids_output )
{
  XUtilities::return_all_keys( possible_amino_acids_with_rotamers, possible_amino_acids_output );
}

void ResidueData::return_possible_codons( list<string> & possible_codons_output )
{
  possible_codons_output = possible_codons;
}

void ResidueData::return_possible_anticodons( list<string> & possible_anticodons_output )
{
  return_reverse_complement_of_strings( possible_codons, possible_anticodons_output );
}

template< template< class, class> class Container, class AllocT >
bool ResidueData::return_reverse_complement_of_strings( const Container<string, AllocT> & sense_strings_input, Container<string, AllocT> & antisense_strings_output )
{
  if( sense_strings_input.empty() ) return false;

  typename Container<string, AllocT>::const_iterator strings_inp_iter;

  for( strings_inp_iter = sense_strings_input.begin(); strings_inp_iter != sense_strings_input.end(); strings_inp_iter++ )
    antisense_strings_output.push_back( return_reverse_complement( * strings_inp_iter ) );

  if( antisense_strings_output.empty() ) return false;
  else return true;
}

bool ResidueData::return_possible_rotamers_for_amino_acid( const int & amino_acid_input, list<int> & possible_rotamers_output )
{
  if( no_rotamers || possible_amino_acids_with_rotamers.find( amino_acid_input ) ==  possible_amino_acids_with_rotamers.end() ) return false;
  possible_rotamers_output = possible_amino_acids_with_rotamers.find( amino_acid_input )->second;
  return true;
}

bool ResidueData::return_possible_rotamers_for_current_amino_acid( list<int> & possible_rotamers_output )
{
  return return_possible_rotamers_for_amino_acid( curr_aa, possible_rotamers_output );
}

void ResidueData::delete_possible_amino_acids_not_in_codon_list( const list<string> & codons_input )
{
  if( codons_input.empty() ) return;
  list<int> amino_acids_from_codons;
  XUtilities::return_values_given_keys( CodonTable, codons_input, amino_acids_from_codons );
  delete_possible_amino_acids_not_in_list( amino_acids_from_codons );
}

void ResidueData::delete_possible_amino_acids_not_in_anticodon_list( const list<string> & anticodons_input )
{
  if( anticodons_input.empty() ) return;
  list<string> codons_output;
  return_reverse_complement_of_strings( anticodons_input, codons_output );
  delete_possible_amino_acids_not_in_codon_list( codons_output );
}

void ResidueData::delete_possible_codons_not_in_codon_list( const list<string> & codons_input )
{
  if( codons_input.empty() ) return;
  list<string> codons( codons_input );
  list<string> common_codons;
  codons.sort();
  set_intersection( possible_codons.begin(), possible_codons.end(), codons.begin(), codons.end(), back_inserter( common_codons ) );
  possible_codons = common_codons;
}

void ResidueData::delete_possible_codons_not_in_anticodon_list( const list<string> & anticodons_input )
{
  if( anticodons_input.empty() ) return;
  list<string> codons_output;
  return_reverse_complement_of_strings( anticodons_input, codons_output );
  delete_possible_codons_not_in_codon_list( codons_output );
}

void ResidueData::delete_possible_amino_acids_not_in_list( const list<int> & amino_acids_input )
{
  if( amino_acids_input.empty() ) return;

  map<int, list<int> >::iterator map_i_loi_iter;

  for( map_i_loi_iter = possible_amino_acids_with_rotamers.begin(); map_i_loi_iter != possible_amino_acids_with_rotamers.end(); map_i_loi_iter++)
    if( find( amino_acids_input.begin(), amino_acids_input.end(), map_i_loi_iter->first ) == amino_acids_input.end() )
      possible_amino_acids_with_rotamers.erase( map_i_loi_iter->first );
}

void ResidueData::revert_to_best_state()
{
  if( no_rotamers ) return;
  curr_rot = best_rot;
  curr_codon = best_codon;
  curr_aa = CodonTable.find( best_codon )->second;
}

void ResidueData::set_best_state()
{
  if( no_rotamers ) return;
  best_rot = curr_rot;
  best_codon = curr_codon;
}

bool ResidueData::set_curr_aa( const int & amino_acid_input )
{
  if( possible_amino_acids_with_rotamers.find( amino_acid_input ) ==  possible_amino_acids_with_rotamers.end() ) return false;
  accept_aa();
  accept_codon();
  curr_aa = amino_acid_input;
  return_codon_for_amino_acid( curr_aa, curr_codon );
  assign_rand_rot();
  return true;
}

bool ResidueData::set_curr_codon( const string & curr_codon_input )
{
  int amino_acid_from_codon_input = CodonTable.find( curr_codon_input )->second;
  if( find( possible_codons.begin(), possible_codons.end(), curr_codon_input ) == possible_codons.end()
      || possible_amino_acids_with_rotamers.find( amino_acid_from_codon_input ) ==  possible_amino_acids_with_rotamers.end() ) return false;
  accept_aa();
  accept_codon();
  curr_codon = curr_codon_input;
  curr_aa = amino_acid_from_codon_input;
  assign_rand_rot();
  return true;
}

bool ResidueData::accept_new_state( const int & new_rotamer_input )
{
  map<int, list<int> >::iterator map_i_loi_iter;

  for( map_i_loi_iter = possible_amino_acids_with_rotamers.begin(); map_i_loi_iter != possible_amino_acids_with_rotamers.end(); map_i_loi_iter++)
    if( find( map_i_loi_iter->second.begin(), map_i_loi_iter->second.end(), new_rotamer_input ) != map_i_loi_iter->second.end() ){
      accept_residue();
      curr_rot = new_rotamer_input;
      return true;
    }
  return false;
}

bool ResidueData::not_a_possible_amino_acid( const int & amino_acid_input )
{
  if( possible_amino_acids_with_rotamers.find( amino_acid_input ) == possible_amino_acids_with_rotamers.end() ) return true;
  else return false;
}

bool ResidueData::return_codon_for_amino_acid( const int & amino_acid_input, string & codon_output )
{
  if( amino_acid_input <= 0 || amino_acid_input > 20 ) return false;

  map<string, int>::const_iterator CodonTable_iter;

  for( CodonTable_iter = CodonTable.begin(); CodonTable_iter != CodonTable.end(); CodonTable_iter++ )
    if( CodonTable_iter->second == amino_acid_input ){
      codon_output = CodonTable_iter->first;
      return true;
    }
  return false;
}

const map<string,int> ResidueData::initialize_CodonTable()
{
  map<string, int> CT;
  CT["TTT"] =  5;
  CT["TTC"] =  5;
  CT["TTA"] = 10;
  CT["TTG"] = 10;
  CT["TCT"] = 16;
  CT["TCC"] = 16;
  CT["TCA"] = 16;
  CT["TCG"] = 16;

  CT["TAT"] = 20;
  CT["TAC"] = 20;
  CT["TAA"] =  0;
  CT["TAG"] =  0;
  CT["TGT"] =  2;
  CT["TGC"] =  2;
  CT["TGA"] =  0;
  CT["TGG"] = 19;

  CT["CTT"] = 10;
  CT["CTC"] = 10;
  CT["CTA"] = 10;
  CT["CTG"] = 10;
  CT["CCT"] = 13;
  CT["CCC"] = 13;
  CT["CCA"] = 13;
  CT["CCG"] = 13;

  CT["CAT"] =  7;
  CT["CAC"] =  7;
  CT["CAA"] = 14;
  CT["CAG"] = 14;
  CT["CGT"] = 15;
  CT["CGC"] = 15;
  CT["CGA"] = 15;
  CT["CGG"] = 15;

  CT["ATT"] =  8;
  CT["ATC"] =  8;
  CT["ATA"] =  8;
  CT["ATG"] = 11;
  CT["ACT"] = 17;
  CT["ACC"] = 17;
  CT["ACA"] = 17;
  CT["ACG"] = 17;

  CT["AAT"] = 12;
  CT["AAC"] = 12;
  CT["AAA"] =  9;
  CT["AAG"] =  9;
  CT["AGT"] = 16;
  CT["AGC"] = 16;
  CT["AGA"] = 15;
  CT["AGG"] = 15;

  CT["GTT"] = 18;
  CT["GTC"] = 18;
  CT["GTA"] = 18;
  CT["GTG"] = 18;
  CT["GCT"] =  1;
  CT["GCC"] =  1;
  CT["GCA"] =  1;
  CT["GCG"] =  1;

  CT["GAT"] =  3;
  CT["GAC"] =  3;
  CT["GAA"] =  4;
  CT["GAG"] =  4;
  CT["GGT"] =  6;
  CT["GGC"] =  6;
  CT["GGA"] =  6;
  CT["GGG"] =  6;

  const map<string, int> const_CT( CT );
  return const_CT;
}

const map<string, int> ResidueData::CodonTable = initialize_CodonTable();

const string ResidueData::initialize_DNA_bases()
{
  string DB = "ACGT";
  const string const_DB( DB );
  return const_DB;
}

const string ResidueData::DNA_bases = initialize_DNA_bases();

const map<int, char> ResidueData::initialize_AminoAcidLookup()
{
  map<int, char> AAL;
  AAL[1] = 'A';
  AAL[2] = 'C';
  AAL[3] = 'D';
  AAL[4] = 'E';
  AAL[5] = 'F';
  AAL[6] = 'G';
  AAL[7] = 'H';
  AAL[8] = 'I';
  AAL[9] = 'K';
  AAL[10] = 'L';
  AAL[11] = 'M';
  AAL[12] = 'N';
  AAL[13] = 'P';
  AAL[14] = 'Q';
  AAL[15] = 'R';
  AAL[16] = 'S';
  AAL[17] = 'T';
  AAL[18] = 'V';
  AAL[19] = 'W';
  AAL[20] = 'Y';

  const map<int, char> const_AAL( AAL );
  return const_AAL;
}

const map<int, char> ResidueData::AminoAcidLookup = initialize_AminoAcidLookup();

string ResidueData::return_reverse_complement( const string & DNA_sequence_input )
{
  if( DNA_sequence_input.empty() ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

  string reverse_complement_output;

  for( unsigned int i = 0; i < DNA_sequence_input.size(); i++ ){
    if( DNA_sequence_input.at( i ) == 'A' ) reverse_complement_output.insert( reverse_complement_output.begin(), 'T' );
    else if( DNA_sequence_input.at( i ) == 'C' ) reverse_complement_output.insert( reverse_complement_output.begin(), 'G' );
    else if( DNA_sequence_input.at( i ) == 'G' ) reverse_complement_output.insert( reverse_complement_output.begin(), 'C' );
    else if( DNA_sequence_input.at( i ) == 'T' ) reverse_complement_output.insert( reverse_complement_output.begin(), 'A' );
    else utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }
  return reverse_complement_output;
}
