// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "Rotamer.h"
#include "aaproperties_pack.h"
#include "fullatom_energy.h"
#include "input_pdb.h"
#include "misc.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "rotamer_functions.h"
#include "template_pack.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>

// Numeric Headers
#include <numeric/conversions.hh>


////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer::initialize
///
/// @brief
/// initialize a Rotamer object
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void Rotamer::initialize( int const seqpos, int const aa, int const aav, int const rotnum ) {

	using namespace param;

  seqpos_ = seqpos;
  aa_ = aa;
  aav_ = aav;
  rotnum_ = rotnum;

  rotactcoord_.dimension( 3 );
  rotcoord_.dimension( 3, MAX_ATOM() );
  rchi_.dimension( MAX_CHI );
  rrot_.dimension( MAX_CHI );
  rot_born_radius_.dimension( MAX_ATOM(), 0. );

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer::change_chi_angle_update_coors
///
/// @brief
/// change a chi angle, move the coordinates as required
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void Rotamer::change_chi_angle_update_coors( int const ichi, float const chi ) {

	using namespace aaproperties_pack;
	using namespace param;
	using numeric::conversions::radians;

	FArray2D_float mat( 3, 3 );
	FArray1D_float vec( 3 );

	if ( ichi > nchi(aa_,aav_) || ichi == 0 ) return;

	rchi_(ichi) = chi;

	//bk retrieve atoms that determine chi angle
	int const c1 = chi_atoms(1,ichi,aa_,aav_);
	int const c2 = chi_atoms(2,ichi,aa_,aav_);
	int const c3 = chi_atoms(3,ichi,aa_,aav_);
	int const c4 = chi_atoms(4,ichi,aa_,aav_);

	//bk calculate initial chi angle
	float schi;
	dihedral_bk( rotcoord_(1,c1), rotcoord_(1,c2), rotcoord_(1,c3), rotcoord_(1,c4), schi );

	//bk rotate angle by new-initial degrees
	float const rot = radians( chi - schi );

	//bk generate rotation vector and matrix
	getrot_bk( rotcoord_(1,c2), rotcoord_(1,c3), rot, mat, vec );

	//bk move atoms
	for ( int i = 1, ie = natoms( aa_, aav_ ); i <= ie; ++i ) {
		if ( chi_required( ichi, i, aa_, aav_ ) ) {
			move_bk( rotcoord_(1,i), mat, vec );
		}
	}

	fill_rotactcoord();

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin	fill_chi_from_coors
///
/// @brief  member function to get chi angles from coordinates
///
/// @detailed  member function to get chi angles from coordinates
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors psh
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Rotamer::fill_chi_from_coors(){

  using namespace aaproperties_pack;

  for (int i = 1; i<= nchi(aa_,aav_); i++){
		rchi_(i) = dihedral( rotcoord_(1,chi_atoms(1,i,aa_,aav_)),
												 rotcoord_(1,chi_atoms(2,i,aa_,aav_)),
												 rotcoord_(1,chi_atoms(3,i,aa_,aav_)),
												 rotcoord_(1,chi_atoms(4,i,aa_,aav_)));
  }
  set_all_chi_to_periodic_range(rchi_, aa_);
  rotamer_from_chi(rchi_, aa_, rrot_);

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin	fill_coors_from_chi
///
/// @brief  member function to get coors from chi angles
///
/// @detailed  member function to get coors from chi angles
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors psh
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Rotamer::fill_coors_from_chi(){

	// Rebuild the coors of this rotamer from its chi angles
	using namespace misc;  // full_coord is here
	get_rot_coord(full_coord,aa_,aav_,rchi_,rotcoord_,seqpos_);
	fill_rotactcoord();

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin	compute_rperc
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Rotamer::compute_rperc(){

	get_rotamer_probability(aa_,aav_,template_pack::phi(seqpos_),template_pack::psi(seqpos_),
													seqpos_,misc::total_residue,rchi_,rrot_,rperc_);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin	fill_rotactcoord
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void Rotamer::fill_rotactcoord(){

	put_wcentroid(rotcoord_,rotactcoord_,aa_);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Rotamer::operator <
///
/// @brief: "less-than" operator for class Rotamer (for sorting)
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 12/08/06
/////////////////////////////////////////////////////////////////////////////////
bool operator < ( Rotamer const & lhs, Rotamer const & rhs ) {

	// jk sort first on seqpos (lowest to highest)
	if ( lhs.seqpos_ < rhs.seqpos_ ) return true;
	if ( rhs.seqpos_ < lhs.seqpos_ ) return false;

	// jk sort next on aa (lowest to highest)
	if ( lhs.aa_ < rhs.aa_ ) return true;
	if ( rhs.aa_ < lhs.aa_ ) return false;

	// jk break all remaining ties by sorting on aav
	if ( lhs.aav_ < rhs.aav_ ) return true;
	return false;

}

