// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "RotamerSet.h"
#include "aaproperties_pack.h"
#include "design.h"
#include "dna.h"
#include "docking_ns.h"
#include "dunbrack_pack.h"
#include "Dunbrack4D.h"
#include "Dunbrack5D.h"
#include "enzyme.h"
#include "files_paths.h"
#include "force_barcode.h"
#include "fullatom_energy.h"
#include "dna_ns.h"
#include "ligand_ns.h"
#include "ligand.h"
#include "misc.h"
#include "pack.h"
#include "pack_bump_selector.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "PackerTask.h"
#include "pdb.h"
#include "PDInteractionGraph.h"
#include "pH_main.h"
#include "pH_ns.h"
#include "pose.h"
#include "pose_dna.h"
#include "pose_ligand.h"
#include "read_aaproperties.h"
#include "refold.h"
#include "Rotamer.h"
#include "RotamerOptions.h"
#include "rotamer_trials.h"
#include "rotamer_explosion.h"
#include "rotamer_functions.h"
#include "symmetric_design.h"
#include "template_pack.h"
#include "termini.h"
#include "util_interpolate.h"
#include "water.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/byte.hh>
#include <ObjexxFCL/ChunkVector.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/FArray.io.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>

// Numeric Headers
#include <numeric/constants.hh>
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/vector1.hh>
#include <utility/io/izstream.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
//#include <ctime>
#include <fstream>
#include <iostream>
#include <list>


typedef struct {
	int aa;
	int aav;
	float chi[4];
} rotamer_geometry;




////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::reset
///
/// @brief
/// reset a RotamerSetBase object (clear stored rotamers)
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::reset() {

  rotamers_.clear();
  base_rotamer_.clear();
  nrotamers_=0;

	reset_indexing_arrays();

  return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::reset_indexing_arrays
///
/// @brief
/// reset the indexing arrays
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::reset_indexing_arrays() {

	using namespace param;

	nmoltenres_ = 0;
	effective_numres_ = misc::total_residue;
	if( get_ligand_flexible_flag() && !get_enable_ligaa_flag()) effective_numres_ += int(ligand::ligand_ptr_vector.size());
	resid_2_moltenres_.assign( effective_numres_, 0 );
	rotindex_offsets_.assign( effective_numres_, 0 );
	moltenres_2_resid_.clear();
	num_states_for_moltenres_.clear();
	nrotoffset_.clear();
  first_rot_for_aa_.clear();

	// flexible ligands
	for (int ii = misc::total_residue + 1; ii <= effective_numres_; ++ii) {
		++nmoltenres_;
		resid_2_moltenres_[ii] = nmoltenres_;
		//kwk this part of the function store a negative int
		//if you take -1*moltenres_2_resid(x)-1
		// you will get the index of the appropriate ligand pointer in
		// the ligand_ptr_vector located in ligand_ns.h
		moltenres_2_resid_.push_back( misc::total_residue - ii );
		int const num_states = ligand::ligand_ptr_vector[nmoltenres_-1]->ligand_conformations.size();
		num_states_for_moltenres_.push_back(num_states);
		nrotoffset_.push_back(0);
		utility::vector1<int> fr(int(MAX_AA()),0);
		first_rot_for_aa_.push_back(fr);
	}

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::push_rotamer
///
/// @brief
/// add a new rotamer to the end of this RotamerSetBase
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::push_rotamer( int const seqpos, int const aa, int const aav, int const rotnum ) {

  Rotamer r( seqpos, aa, aav, rotnum );
	rotamers_.push_back(r);
	++nrotamers_;
	update_indexing_arrays(seqpos, aa, nrotamers_);

	// jk Start by setting the base rotamer to "self"
	base_rotamer_.push_back(nrotamers_);

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::push_rotamer
///
/// @brief
/// add a new rotamer to the end of this RotamerSetBase
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::push_rotamer( Rotamer const & r ) {

	rotamers_.push_back(r);
	++nrotamers_;
	update_indexing_arrays(r.seqpos(), r.aa(), nrotamers_);

	// jk Start by setting the base rotamer to "self"
	base_rotamer_.push_back(nrotamers_);

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::pop_rotamer
///
/// @brief
/// remove the last rotamer
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::pop_rotamer() {

	int const seqpos = report_seqpos(nrotamers_);
	int const aa = report_aa(nrotamers_);
  int const moltenres = resid_2_moltenres_[seqpos];

	if ( first_rot_for_aa_[moltenres][aa] == nrotamers_ ) {
		// jk This was the first rotamer for this aa at this seqpos
		first_rot_for_aa_[moltenres][aa] = 0;
	}

	--(num_states_for_moltenres_[moltenres]);

	if ( num_states_for_moltenres_[moltenres] == 0 ) {
		// jk This was the only rotamer at this seqpos
		resid_2_moltenres_[seqpos] = 0;
		rotindex_offsets_[seqpos] = 0;
		moltenres_2_resid_.pop_back();
		num_states_for_moltenres_.pop_back();
		nrotoffset_.pop_back();
		first_rot_for_aa_.pop_back();
		--nmoltenres_;
	}

	rotamers_.pop_back();
	base_rotamer_.pop_back();
	--nrotamers_;

  return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::find_suitable_base_rotamer
///
/// @brief
/// find a base rotamer (ie. same chi angles, but variant #1)
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::find_suitable_base_rotamer( int const rotinx ) {

	using namespace aaproperties_pack;
	using namespace param_aa;

	if ( nrotamers() == 0 ) return;

	int const seqpos = report_seqpos(rotinx);
	int const aa = report_aa(rotinx);
	int const aav = report_aav(rotinx);

	// jk Start by assuming "self" is the base rotamer
	set_base_rotamer(rotinx) = rotinx;

	// jk Conditions immediately confirm that "self" is the base rotamer
	if ( is_DNA(aa) || ( aav == 1 ) ||
			 variant_type( aav_Nterm, aa, aav ) || variant_type( aav_Cterm, aa, aav ) ||
			 ( aa == aa_his && variant_type( aav_his_taut, aa, aav ) ) ) {
		return;
	}

	// jk pH mode - conditions for identifying base rotamer not clear,
	// so make everything a base rotamer until this is clarified
	if ( get_pH_packing_flag() ) {
		return;
	}

	// jk Look for a rotamer with matching chi angles
	// (starting with the first rotamer for this seqpos+aa)
	int const base_aav(1);
	int const base_nchi = nchi(aa,base_aav);
	int const start_rotinx = first_rot_for_aa(aa,resid_2_moltenres(seqpos));

	for ( int ii = start_rotinx; ii <= rotinx; ++ii ) {
		assert( seqpos == report_seqpos(ii) );
		assert( aa == report_aa(ii) );
		if ( report_aav(ii) != 1 ) continue;

		bool found_match(true);
		for ( int chiinx = 1; chiinx <= base_nchi; ++chiinx ) {
			if ( std::abs( get_rchi(chiinx,rotinx) - get_rchi(chiinx,ii) ) > 0.1 ) {
				found_match = false;
				break;
			}
		}

		if ( found_match ) {
			set_base_rotamer(rotinx) = ii;
			break;
		}

	}

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::update_indexing_arrays
///
/// @brief
/// update the resid_2_moltenres and related arrays after adding a rotamer
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::update_indexing_arrays( int const seqpos, int const aa, int const rotinx ) {

	using namespace param;

	if ( resid_2_moltenres_[seqpos] == 0 ) {
		// jk This is the first rotamer for this seqpos
		++nmoltenres_;
		resid_2_moltenres_[seqpos] = nmoltenres_;
		moltenres_2_resid_.push_back(seqpos);
		num_states_for_moltenres_.push_back(0);
		// jk Note: offset arrays expect rotamers indexed from zero
		rotindex_offsets_[seqpos] = rotinx-1;
		nrotoffset_.push_back(rotinx-1);
		utility::vector1<int> fr(int(MAX_AA()),0);
		first_rot_for_aa_.push_back(fr);
	}

  int const moltenres = resid_2_moltenres_[seqpos];
	++(num_states_for_moltenres_[moltenres]);

	if ( first_rot_for_aa_[moltenres][aa] == 0 ) {
		// jk Note: first_rot_for_aa_ expects the actual rotamer index
		// (unlike the offset arrays)
		first_rot_for_aa_[moltenres][aa] = rotinx;
	}

  return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::setup_indexing_arrays
///
/// @brief
/// fill the resid_2_moltenres and related arrays (from scratch)
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::setup_indexing_arrays() {

	reset_indexing_arrays();
	for ( int ii = 1; ii <= nrotamers(); ++ii ) {
		update_indexing_arrays(report_seqpos(ii), report_aa(ii), ii);
	}

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::sort
///
/// @brief
/// ensure that the RotamerSet is sorted (important for usage)
///
///  Note: when rotamers are added through member functions,
///  they are typically added in the correct sorted order,
///  so a call to this function is not needed.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::sort() {

	std::sort( rotamers_.begin(), rotamers_.end() );
	for ( int ii = 1; ii <= nrotamers(); ++ii ) {
		find_suitable_base_rotamer(ii);
	}
	setup_indexing_arrays();

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::remove( std::vector<int> const & exclude )
{

	int num_bumped = exclude.size();

	std::cout << num_bumped <<
	 " rotamers will be removed from this rotamer set." << std::endl;

	std::list<int> rots_to_copy;
	int i = 0;
	for ( int rot = 1; rot <= nrotamers(); ++rot ) {

	// found a rotamer to skip, go to next
		if ( rot == exclude[i] ) {
			++i;
			assert( i <= num_bumped );
			continue;
			// note: will need to make sure that a 'one and only' rotamer is not
			// skipped!! If design_map.repack_residue(pos) == true later but there
			// are no rotamers at pos, you'll end up in trouble
		}

		rots_to_copy.push_back( rot );
	}

	i = 0;
	for ( std::list<int>::iterator it = rots_to_copy.begin();
	      it != rots_to_copy.end(); ++it ) {
		++i;

		copy_rotamer( *this, (*it), i, true, false );
	}

	nrotamers_ = rots_to_copy.size();
	while ( int(rotamers_.size()) > nrotamers_ ) {
		rotamers_.pop_back();
	}

	setup_indexing_arrays();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::copy_rotamer
///
/// @brief
///
/// @authors
/// ashworth
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerSetBase::copy_rotamer(
	RotamerSetBase & source,
	int const old_index,
	int const new_index,
	bool const same,
	bool const update_index_arrays
)
{

	using namespace param;
	using namespace aaproperties_pack;

	assert( new_index <= nrotamers_ && new_index > 0 );
	assert( old_index <= nrotamers_ && old_index > 0 );

	if ( same && new_index == old_index ) return; // nothing to be done

  rotamers_[new_index] = source.rotamers_[old_index];

	find_suitable_base_rotamer(new_index);

	if ( update_index_arrays ) setup_indexing_arrays();

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::dump_member_data
///
/// @brief
/// dump all the member data for debugging, etc.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::dump_member_data() const {

	using namespace aaproperties_pack;
	using namespace param;

	std::cout << "nrotamers is: " << nrotamers() << std::endl;

	for (int i = 1; i<= nrotamers_; i++){

		std::cout << "rot_seqpos(" << i << ") = " << report_seqpos(i) << std::endl;
		std::cout << "rot_aa(" << i << ") = " << report_aa(i) << std::endl;
		std::cout << "rot_aav(" << i << ") = " << report_aav(i) << std::endl;
		std::cout << "rot_num(" << i << ") = " << report_rotnum(i) << std::endl;

		int const aa = report_aa(i);
		int const aav = report_aav(i);

		for ( int j = 1, je = natoms( aa, aav ); j <= je; ++j ) {
			for (int k = 1; k<= 3; k++){
				std::cout << "rotcoord(" << k << ',' << j << ',' << i << ") = " << get_rotcoord(k,j,i) << std::endl;
			}
		}

		for (int j = 1; j<= 3; j++){
			std::cout << "rotactcoord(" << j << ',' << i << ") = " << get_rotactcoord(j,i) << std::endl;
		}

		for ( int j = 1, je = natoms( aa, aav ); j <= je; ++j ) {
			std::cout << "rot_born_radius(" << j << ',' << i << ") = " << get_rot_born_radius(j,i) << std::endl;
		}

		for ( int j = 1, nchi_aa = nchi(aa,aav); j <= nchi_aa; ++j ) {
			std::cout << "rchi(" << j << ',' << i << ") = " << get_rchi(j,i) << std::endl;
			std::cout << "rrot(" << j << ',' << i << ") = " << get_rrot(j,i) << std::endl;
		}

		std::cout << "rperc(" << i << ") = " << get_rperc(i) << std::endl;
		std::cout << "base_rotamer(" << i << ") = " << get_base_rotamer(i) << std::endl;

	}

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::dump_rchi
///
/// @brief
/// dump contents of the rchi array for debugging, etc.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::dump_rchi() const {

	using namespace aaproperties_pack;
	using namespace param;

	for (int i = 1; i<= nrotamers(); i++){
		for ( int j = 1, nchi_aa = nchi(report_aa(i),report_aav(i)); j <= nchi_aa; ++j ) {
			std::cout << "rchi(" << j << ',' << i << ") = " << get_rchi(j,i) << std::endl;
		}
	}

  return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSetBase::dump_rotcoord
///
/// @brief
/// dump contents of the rotcoord array for debugging, etc.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSetBase::dump_rotcoord() const {

	using namespace aaproperties_pack;
	using namespace param;

	for (int i = 1; i<= nrotamers(); i++){
		for ( int j = 1, je = natoms( report_aa(i), report_aav(i) ); j <= je; ++j ) {
			for (int k = 1; k<= 3; k++){
				std::cout << "rotcoord(" << k << ',' << j << ',' << i << ") = " << get_rotcoord(k,j,i) << std::endl;
			}
		}
	}

  return;
}







////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSet::write_rotamer_set_to_file
///
/// @brief
/// writes information describing the residues that the interaction graph file
/// will represent, and the rotamers that each residue represents.
///
/// @detailed
/// The output file will contains the number of amino acid types at its creation,
/// then the number of nodes in the interaction graph, then the
/// residue id for each node, and the number of states for each node.  Finally
/// the rotamer geometry for each state is listed.  The states are ordered first
/// by the index of the node that they correspond to, then listed in whatever order
/// they appeared at the time the graph was created.
///
/// @param
/// outfile - [in/out] - the file to write to
/// ig - [in] - the interaction graph
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerSet::write_rotamer_set_to_file
(
	std::ofstream & outfile,
	pack::PDInteractionGraph & ig
)
{

	//apl order of data in output file:
	//apl number of aa types during file's creation
	//apl number of nodes
	//apl the residue these nodes represent
	//apl the number of states for each node
	//apl the aa aav rchi1 rchi2 rchi3 and rchi4 data for each state
	int num_nodes = ig.get_num_nodes();
	int num_aa_types = ig.get_num_aatypes();

	outfile.write((char*) &num_aa_types, sizeof( int ) );
	outfile.write((char*) &num_nodes, sizeof( int ) );
	int * node_info_buffer = new int[ num_nodes ];
	for (int ii = 1; ii <= num_nodes; ++ii)
	{
		node_info_buffer[ ii - 1] = moltenres_2_resid( ii );
	}
	outfile.write( (char*) node_info_buffer, sizeof( int ) * num_nodes );

	int total_rots = 0;
	for (int ii = 1; ii <= num_nodes; ++ii)
	{
		node_info_buffer[ ii - 1 ] = ig.get_num_states_for_node(ii);
		total_rots += node_info_buffer[ ii - 1];
	}
	outfile.write( (char*) node_info_buffer, sizeof( int ) * num_nodes );
	delete [] node_info_buffer;

	rotamer_geometry* rot_geom = new rotamer_geometry[ total_rots ];

	for (int ii = 1; ii <= num_nodes; ++ii)
	{
		int ii_num_states = ig.get_num_states_for_node( ii );
		int ii_offset = rotindex_offsets( moltenres_2_resid( ii ) );
		for (int jj = 1; jj <= ii_num_states; ++jj)
		{
			int jjrotindex = ii_offset + jj;
			rot_geom[ jjrotindex - 1 ].aa = report_aa(jjrotindex );
			rot_geom[ jjrotindex - 1 ].aav = report_aav( jjrotindex );
			rot_geom[ jjrotindex - 1 ].chi[0] = get_rchi( 1, jjrotindex );
			rot_geom[ jjrotindex - 1 ].chi[1] = get_rchi( 2, jjrotindex );
			rot_geom[ jjrotindex - 1 ].chi[2] = get_rchi( 3, jjrotindex );
			rot_geom[ jjrotindex - 1 ].chi[3] = get_rchi( 4, jjrotindex );
		}
	}
	outfile.write( (char* ) rot_geom, sizeof( rotamer_geometry ) * total_rots );
	delete [] rot_geom;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSet::read_rotamer_set_from_file
///
/// @brief
/// reads the information at the begining of an interaction graph file
/// that describes which residue each node corresponds to and which
/// rotamer each state corresponds to.  Rotamers are described by their amino
/// acid types and their internal geoemetry and not by their coordinates.
/// This allows an interaction graph file to accomodate rigid body translations
/// and rotations of the protein.  This method stores the correspondence
/// information in the interaction graph.
///
/// @detailed
/// The file contains the number of amino acid types at its creation, then
/// contains the number of nodes in the interaction graph, then the
/// residue id for each node, and the number of states for each node.  Finally
/// the rotamer geometry for each state is listed.  The states are ordered first
/// by the index of the node that they correspond to, then listed in whatever order
/// they appeared in at the time the graph was created.
///
/// @param
/// infile - [in/out] - the binary input file to read from, advanced at the end
///   of this method to point at the first piece of data that the interaction
///   graph will read inside read_edge_energies_from_file()
/// ig - [in/out] - the interaction graph in which node and statecorrespondence
///    information is stored.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerSet::read_rotamer_set_from_file
(
	std::ifstream & infile,
	pack::PDInteractionGraph & ig
)
{
	int num_aatypes_in_file;
	int num_nodes_in_file;
	int num_instance_nodes = ig.get_num_nodes();

	infile.read( (char*) &num_aatypes_in_file, sizeof( int ) );
	ig.set_num_file_aatypes( num_aatypes_in_file );

	//apl find out how many nodes there are in the input file
	infile.read( (char*) &num_nodes_in_file, sizeof( int ) );

	FArray1D_int nodes_in_file_2_resid( num_nodes_in_file);
	FArray1D_int nodes_in_file_2_instance_nodes( num_nodes_in_file, -1 );
	FArray1D_int instance_nodes_2_nodes_in_file( num_instance_nodes, -1 );

	//apl read in the residue that each node represents
	int * node_info_buffer = new int[ num_nodes_in_file ];
	infile.read( (char*) node_info_buffer, sizeof(int) * num_nodes_in_file );
	for (int ii = 1; ii <= num_nodes_in_file; ++ii)
	{
		nodes_in_file_2_resid( ii ) = node_info_buffer[ ii - 1 ];
	}
	ig.set_num_nodes_in_file( num_nodes_in_file );

	//apl determine correspondence between nodes in the instance graph
	//apl and the nodes represented in the file
	for (int ii = 1; ii <= num_nodes_in_file; ++ii)
	{
		int iiresid = nodes_in_file_2_resid(ii);
		if ( resid_2_moltenres( iiresid ) != 0 )
		{
			int ii_instance_node = resid_2_moltenres( iiresid );
			instance_nodes_2_nodes_in_file( ii_instance_node) = ii;
			nodes_in_file_2_instance_nodes( ii ) = ii_instance_node;
			ig.set_node_correspondence( ii_instance_node, ii );
		}
	}

	//apl read in the number of states for each node in the file
	FArray1D_int num_states_per_node_in_file( num_nodes_in_file );
	infile.read( (char*) node_info_buffer, sizeof(int) * num_nodes_in_file );
	int total_rotamers = 0;
	for (int ii = 1; ii <= num_nodes_in_file; ++ii )
	{
		num_states_per_node_in_file( ii ) = node_info_buffer[ ii - 1 ];
		total_rotamers += node_info_buffer[ ii - 1 ];
	}

	delete [] node_info_buffer;

	//apl allocate space for and read in the
	//apl aa aav rchi1 rchi2 rchi3 and rchi4 data
	FArray1D< FArray1D_int > file_states_aa( num_nodes_in_file );
	FArray1D< FArray1D_int > file_states_aav( num_nodes_in_file );
	FArray1D< FArray2D_float > file_states_rchi( num_nodes_in_file );
	for (int ii = 1; ii <= num_nodes_in_file; ++ii )
	{
		int ii_num_states = num_states_per_node_in_file( ii );
		file_states_aa(ii).dimension( ii_num_states );
		file_states_aav(ii).dimension( ii_num_states);
		file_states_rchi(ii).dimension( 4, ii_num_states );
		ig.set_num_states_for_file_node( ii, ii_num_states );
	}

	//read in contents of file
	rotamer_geometry * rot_geom = new rotamer_geometry[ total_rotamers ];
	infile.read( (char*) rot_geom, sizeof( rotamer_geometry) * total_rotamers );
	int rotamer_index = 0;
	for (int ii = 1; ii <= num_nodes_in_file; ++ii )
	{
		int ii_num_states = num_states_per_node_in_file( ii );
		for( int jj = 1; jj <= ii_num_states; ++jj )
		{
			file_states_aa(ii)(jj) = rot_geom[ rotamer_index ].aa ;
			file_states_aav(ii)(jj) = rot_geom[ rotamer_index ].aav;
			for (int kk = 1; kk <= 4; ++kk )
			{
				file_states_rchi(ii)(kk, jj) = rot_geom[ rotamer_index].chi[ kk - 1];
			}
			++rotamer_index;
			ig.set_aa_for_file_node_state( ii, jj, file_states_aa(ii)(jj) );
		}
	}

	delete [] rot_geom;

	//apl find the correspondence between rotamers for nodes in the file
	//apl and nodes in this graph instance
	for (int ii = 1; ii <= num_instance_nodes; ++ii)
	{
		int iiresid = moltenres_2_resid( ii );
		int ii_file_node = instance_nodes_2_nodes_in_file( ii );

		if ( ii_file_node == -1 ) continue;

		find_correspondence_between_file_and_instance( ii, ig,
			file_states_aa(ii_file_node), file_states_aav(ii_file_node),
			file_states_rchi(ii_file_node), rotindex_offsets( iiresid ) );
	}

	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSet::find_correspondence_between_file_and_instance
///
/// @brief
/// finds the correspondence between the rotamers created for the current
/// invocation of pack_rotamers and those described in the interaction graph
/// file.
///
/// @detailed
///
/// @param
/// node_index - [in] - the index of the instance node
/// ig - [in/out] - the interaction graph to be updated with correspondence
///   information.
/// file_states_aa - [in] - an array describing the aa of each file state
///   for the file node that corresponds to this instance node
/// file_states_aav - [in] - like file_states_aa, except for variants
/// file_states_rchi - [in] - array of the four chi angles for each file rotamer
/// rotamer_offset_for_node - [in] - the index one before the starting position
///   for the rotamers on the node within those rotamers
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
RotamerSet::find_correspondence_between_file_and_instance
(
	int node_index,
	pack::PDInteractionGraph & ig,
	FArray1DB_int & file_states_aa,
	FArray1DB_int & file_states_aav,
	FArray2DB_float & file_states_rchi,
	int rotamer_offset_for_node
)
{

	int num_states_this_node = ig.get_num_states_for_node( node_index );
	int num_states_in_file = file_states_aa.size();

	float const close_enough = 0.001;

	int last_correspondence = 0;
	for (int ii = 1; ii <= num_states_this_node; ++ii)
	{
		int iirotindex = rotamer_offset_for_node + ii;
		bool found_aa = false;
		for (int jj = last_correspondence + 1; jj <= num_states_in_file; ++jj )
		{
			if ( file_states_aa( jj ) == report_aa( iirotindex ) )
			{
				found_aa = true;
				if( (file_states_aav( jj ) == report_aav( iirotindex ) )
						&& ( std::abs( file_states_rchi( 1, jj ) - get_rchi( 1, iirotindex )) < close_enough)
						&& ( std::abs( file_states_rchi( 2, jj ) - get_rchi( 2, iirotindex )) < close_enough)
						&& ( std::abs( file_states_rchi( 3, jj ) - get_rchi( 3, iirotindex )) < close_enough)
						&& ( std::abs( file_states_rchi( 4, jj ) - get_rchi( 4, iirotindex )) < close_enough)
					)
				{
					ig.set_correspondence_for_state( node_index, ii, jj );
					last_correspondence = jj;
					break;
				}
			}
			else
			{
				//apl ASSUMPTION: rotamers of the same amino acid are contiguous in rotindex
				//apl once we're looking at a rotamer for another amino acid, we can be sure
				//apl no correspondence exists, and we may bail out.
				if (found_aa)
				{
					//std::cerr << "Not found: " << node_index << " ";
					//std::cerr <<  report_aa(iirotindex) << " ";
					//std::cerr <<  report_aav( iirotindex ) << " ";
					//std::cerr <<  get_rchi( 1, iirotindex ) << " ";
					//std::cerr <<  get_rchi( 2, iirotindex ) << " ";
					//std::cerr <<  get_rchi( 3, iirotindex ) << " ";
					//std::cerr <<  get_rchi( 4, iirotindex ) << std::endl;

					break;
				}

			}
		}
	}

}



////////////////////////////////////////////////////////////////////////////////
/// @begin get_rotamers
///
/// @brief
///bk get rotamers and rotamer coordinates uses namespace template_pack as input
///bk output variables are in namespace rotamers
///
/// @detailed
///
/// @param  design_map
/// @param  include_current - [in/out]? -
/// @param  current_rot_index - [in/out]? - rotno's of current/native rot at each
///       res pos
/// @param  include_extra - [in/out]? -
/// @param  extra_rot - [in/out]? -
/// @param  extra_chi - [in/out]? -
/// @param  mode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::get_rotamers(
	int total_residue,
	const FArray1D_int & res,
	const FArray1D_int & res_variant,
	const FArray3D_float & full_coord,
	FArray1D_int & current_rot_index,
  PackerTask const & Task
)
{

	using namespace aaproperties_pack;
	using namespace design;
	using namespace design_sym;
	using namespace param;
	using namespace template_pack;
	using namespace water;
	using namespace termini_ns;
	using namespace files_paths; // for use_conformer switch

	reset();

	// jk Get chi's from current coors if we'll need them
	FArray2D_float chi_curr( MAX_CHI, total_residue );
	FArray2D_int rot_curr( MAX_CHI, total_residue );

	if ( Task.get_include_current() && active_rotamer_options.rot_pert_input ) {
		get_chi_and_rot_from_coords( total_residue, res, res_variant, full_coord,
		 chi_curr, rot_curr );
	}

	//ctsa initialize current_rot_index with a senseless number for safety
	//apl block assignment
	current_rot_index = -1;

	//flo need to know residue pos of ligand for exploded rotamer screening
	if ( (get_enable_ligaa_flag() ) ){
		ligand::ligand_residue_numbers.clear();
		rotamer_explosion::rotamers_to_minimize.clear();

		for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
			if( param_aa::is_ligand( res(seqpos) ) ) {
				ligand::ligand_residue_numbers.push_back(seqpos);
			}
		}
	}

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( clone_follows(seqpos) != 0 ) continue;

		for ( int aa = 1, aae = MAX_AA(); aa <= aae; ++aa ) {
			int nrotaa = 0;

//psh conformers also use default rotamer builder so this if..else structure is essential
			if (use_conformer &&
					(Task.get_mode() == "design" || Task.get_mode() == "packrot" ) &&
					Task.get_designmap().get(seqpos,aa) &&
					confLibHandler.isConformerKeyValid(confLibHandler.conformerKeyFromCoord(seqpos,aa))) {
				buildBDAconformers( total_residue, res, res_variant, full_coord,Task.get_designmap(), seqpos, aa );
				//confLibHandler.printChi(12, -80, -10);
			} else { // default

//ctsa add_native_rot moved to allow native and dunbrack rotamers to be loaded
//ctsa in global residue and aa order

//bk  add rotamers with current coordinates
				if ( Task.get_include_current() && res(seqpos) == aa &&
					   Task.get_designmap().get(seqpos,aa) ) {

					add_extra_rot_from_coord( seqpos, res, res_variant, nrotaa,
					 Task.get_designmap(), template_pack::phi, template_pack::psi,
					 rotarray, chiarray, total_residue, actcoord, full_coord,
					 current_rot_index );

					// jk add perturbed rotamers around current coordinates
					if ( active_rotamer_options.rot_pert_input ) {
						add_rot_pert_input( seqpos, res, res_variant, nrotaa,
						 Task.get_designmap(), template_pack::phi, template_pack::psi,
						 total_residue, full_coord, chi_curr, rot_curr );
					}
				}
//chu include extra chi angles
				if ( Task.get_include_extra() && res(seqpos) == aa ) {
					add_extra_rot_from_chi( seqpos, res, res_variant, nrotaa,
					 Task.get_designmap(), template_pack::phi, template_pack::psi,
					 Task.get_extra_rot(), Task.get_extra_chi(), total_residue,
					 full_coord, current_rot_index );
				}
				if ( !Task.get_designmap().repack_residue(seqpos) ){continue;}
				int const aave = nvar(aa);
				for ( int aav = 1; aav <= aave; ++aav ) {

					// jk jumpout if this aav corresponds to a variant we're not using
					bool jumpout( true );
					for ( int vtype = 1; vtype <= number_aav_type; ++vtype ) {
						if ( variant_type( vtype, aa, aav ) &&
							Task.get_designmap().variant_type_is_allowed(vtype,seqpos) ) {
//						std::cout << "Jump type is " << vtype << std::endl;
							jumpout = false;
							break;
						}
					}
					//if ( jumpout ) std::cout << " Jumping out pos aa aav type " <<
					//seqpos << " " << aa << " " << aav << " " << std::endl;
					if ( jumpout ) continue;
					if ( Task.get_designmap().get(seqpos,aa) ) {
						if ( param_aa::is_ligand(res(seqpos))
							&& res(seqpos) == aa && enable_ligaa_ns::nligandrotamers(aa,1)!=0 ) {

							//get ligand rotamers for this residue.
							get_rotamers_for_ligand_aa(seqpos, total_residue, res, res_variant, aa, 1, nrotaa, full_coord, Task);
						}else{
							get_rotamers_seqpos_aa_aav( total_residue,res,res_variant,full_coord, aa, aav, seqpos, nrotaa,
					 			Task.get_include_current(), Task.get_mode(), Task );
						}
					}

//jk Additional call with optimizeH if use_input_sc, to get "similar" rotamers
//jk This also builds rotamer variants around NATRO rotamers, where desired

					if ( res(seqpos) == aa && Task.get_mode() != "optimizeH" &&
							 ( Task.get_designmap().num_allowed_aa(seqpos) == 0 ||
								 Task.get_designmap().get(seqpos,aa) ) ) {

						// If you don't pass false, it won't build current var
						bool include_current( Task.get_designmap().get(seqpos,aa) ?
																	Task.get_include_current() : false );

						if ( ( !active_rotamer_options.use_input_sc && hydrate_dna && param_aa::is_DNA(aa)) ||
								 active_rotamer_options.use_input_sc ) {
							get_rotamers_seqpos_aa_aav( total_residue, res, res_variant, full_coord, aa, aav, seqpos, nrotaa,
								include_current, "optimizeH", Task );
						}
					}
				}
			}
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_rotamers_seqpos_aa_aav
///
/// @brief
///
/// @detailed
///
/// @param  aa - [in/out]? - rotamers for this amino acid
/// @param  aav - [in/out]? - rotamers for this amino acid variant
/// @param  seqpos - [in/out]? - sequence position
/// @param  nrotaa - [in/out]? -
/// @param  current_chi_already_included - [in/out]? -
/// @param  mode - [in/out]? -
/// @param  Task - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::get_rotamers_seqpos_aa_aav(
 	int total_residue,
	const FArray1D_int & res,
	const FArray1D_int & res_variant,
  const FArray3D_float & full_coord,
	int const aa,
	int const aav,
	int const seqpos,
	int & nrotaa,
	bool const current_chi_already_included,
	std::string const & mode,
	const PackerTask & Task
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace dunbrack_pack;
	using namespace exchi_flags;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace template_pack;
	using namespace water;

	FArray1D_int base_rotno( MAX_CHI );

	int rot_limit, linenum, iphi, ipsi, start_nrotamers, rotate_proton = 0,
	 rotnum;
	FArray1D_float base_chi( MAX_CHI );
	FArray1D_float base_sd( MAX_CHI );
	FArray1D_float tmp3( MAX_CHI );
	float perc_limit, perc, perc_total, bumpenergy, tmp1, tmp2;

	int packed_rotno;

	int phibin,phibin_next,psibin,psibin_next;
	float phi_err,psi_err;

// fixed parameters
	float const binrange = { 10.0 };
	int const nbins = { 36 };

	std::list< FArray1D_float > total_chi_set;
	std::list< FArray1D_int > total_rot_set;

	int is_buried;
	bool is_extra;

	FArray1D_float tmp_hchi( MAXANGLES_PER_CHI );

	bool treat_as_angles;

	FArray1D_int h2o_nb( MAX_H2O_PER_AA );
	bool is_exposed_water;
	float h2o_intra_repE,h2o_intra_hbE;

	bool delete_current_rot;
	bool delete_previous_rot;

	start_nrotamers = nrotamers();

	bool force_rotamer( barcode_get_rot_cst_exists( seqpos ) );

	// HACK: mechanism for looping back here if we dont find any rotamers
	bool try_again( true );
	while ( try_again ) {
	try_again = false;


	// bk set limits which will determine how many rotamers to include
	if ( neighbors(seqpos) < active_rotamer_options.base_rotamer_buried_cutoff ) {
		perc_limit = perc_limit_surface;
		rot_limit = rot_limit_surface;
	} else {
		perc_limit = perc_limit_buried;
		rot_limit = rot_limit_buried;
	}

	//bk read in rotamers
	perc = 0.0;
	perc_total = 0.0;
	linenum = 0;
	rotnum = nrotaa;

	iphi = 0;
	ipsi = 0;

	get_phibin( template_pack::phi(seqpos), template_pack::psi(seqpos), iphi,
	 ipsi, seqpos, total_residue );

	get_phi_interpolate_bin( template_pack::phi(seqpos),
	 template_pack::psi(seqpos), phibin, psibin, phibin_next, psibin_next,
	 phi_err, psi_err, seqpos, total_residue );

	// ctsa - bump selector ensures that at least one rotamer passes the
	// bump filter for each aa-pos; problem initially found by jim h.
	reset_bump_selector();

	while ( ( force_rotamer && perc_total < 0.99999 ) ||
	        ( perc_total < 0.999 && ( perc_total < perc_limit ||
					  linenum < rot_min_count ) && linenum < rot_limit ) ) {

		total_chi_set.clear();
		total_rot_set.clear();

		//bk optimizing hydrogen (and waters if present)
		if ( mode == "optimizeH" ) {
			perc = 1.0; // dummy value to keep from looping through

			//bk get chi angles for hydrogens, this only pertains to some amino acids
			//cy add RNA ribose 2'-OH
			if ( aa == aa_ser || aa == aa_thr || aa == aa_tyr || aa == aa_asn ||
			     aa == aa_gln || aa == aa_his ||
			     aa == na_rgu || aa == na_rad || aa == na_rcy || aa == na_ura ||
			     ( get_pH_packing_flag() &&
			       ( ( aa == aa_asp && variant_type( aav_pH, aa, aav ) ) ||
			         ( aa == aa_glu &&  variant_type( aav_pH, aa, aav ) ) ) ) ) {

				if ( aa == aa_ser || aa == aa_thr ) {

					rotate_proton = 2;
					is_buried = 1;
					if ( neighbors(seqpos) >= active_rotamer_options.extrachi_cutoff ) is_buried = 2;
					is_extra = active_rotamer_options.extrachi_flag(aa,2,is_buried) != NO_EXTRA_CHI_SAMPLES;
					int count;
					get_ser_thr_hchi( is_extra, count, tmp_hchi );

					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					new_chi_set(1) = chiarray(1,seqpos);
					for ( int i = 1; i <= count; ++i ) {
						new_chi_set(2) = tmp_hchi(i);
						rotamer_from_chi( new_chi_set, aa, new_rot_set );
						total_chi_set.push_back(new_chi_set);
						total_rot_set.push_back(new_rot_set);
					}

				} else if ( aa == na_rgu || aa == na_rad || aa == na_rcy ||
				 aa == na_ura ) { // RNA

					rotate_proton = 1;

					is_buried = 1;
					if ( neighbors(seqpos) >= active_rotamer_options.extrachi_cutoff ) is_buried = 2;
					is_extra = active_rotamer_options.extrachi_flag(aa,1,is_buried) != NO_EXTRA_CHI_SAMPLES ;
					int count;
					get_rna_hchi( is_extra, count, tmp_hchi );

					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					for ( int i = 1; i <= count; ++i ) {
						new_chi_set(1) = tmp_hchi(i);
						// jk Note: rotamer_from_chi not called if RNA
						//						rotamer_from_chi( new_chi_set, aa, new_rot_set );
						total_chi_set.push_back(new_chi_set);
						total_rot_set.push_back(new_rot_set);
					}

				} else if ( aa == aa_tyr ) {

					rotate_proton = 3;

					is_buried = 1;
					if ( neighbors(seqpos) >= active_rotamer_options.extrachi_cutoff ) is_buried = 2;
					is_extra = active_rotamer_options.extrachi_flag(aa,3,is_buried) != NO_EXTRA_CHI_SAMPLES;
					int count;
					get_tyr_hchi( is_extra, count, tmp_hchi );

					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					new_chi_set(1) = chiarray(1,seqpos);
					new_chi_set(2) = chiarray(2,seqpos);
					for ( int i = 1; i <= count; ++i ) {
						new_chi_set(3) = tmp_hchi(i);
						rotamer_from_chi( new_chi_set, aa, new_rot_set );
						total_chi_set.push_back(new_chi_set);
						total_rot_set.push_back(new_rot_set);
					}

				} else if ( aa == aa_asn ) { // amide plane flips

					rotate_proton = 2;

					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					new_chi_set(1) = chiarray(1,seqpos);
					new_chi_set(2) = chiarray(2,seqpos);
					rotamer_from_chi( new_chi_set, aa, new_rot_set );
					total_chi_set.push_back(new_chi_set);
					total_rot_set.push_back(new_rot_set);

					new_chi_set(2) = set_chi_to_periodic_range( (new_chi_set(2)-180.), aa, 2 );
					rotamer_from_chi( new_chi_set, aa, new_rot_set );
					total_chi_set.push_back(new_chi_set);
					total_rot_set.push_back(new_rot_set);

				} else if ( aa == aa_gln ) { // amide plane flips

					rotate_proton = 3;

					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					new_chi_set(1) = chiarray(1,seqpos);
					new_chi_set(2) = chiarray(2,seqpos);
					new_chi_set(3) = chiarray(3,seqpos);
					rotamer_from_chi( new_chi_set, aa, new_rot_set );
					total_chi_set.push_back(new_chi_set);
					total_rot_set.push_back(new_rot_set);

					new_chi_set(3) = set_chi_to_periodic_range( (new_chi_set(3)-180.), aa, 3 );
					rotamer_from_chi( new_chi_set, aa, new_rot_set );
					total_chi_set.push_back(new_chi_set);
					total_rot_set.push_back(new_rot_set);

				} else if ( aa == aa_his ) { //jk ring plane flip

					rotate_proton = 2;

					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					new_chi_set(1) = chiarray(1,seqpos);
					new_chi_set(2) = chiarray(2,seqpos);
					rotamer_from_chi( new_chi_set, aa, new_rot_set );
					total_chi_set.push_back(new_chi_set);
					total_rot_set.push_back(new_rot_set);

					new_chi_set(2) = set_chi_to_periodic_range( (new_chi_set(2)-180.), aa, 2 );
					rotamer_from_chi( new_chi_set, aa, new_rot_set );
					total_chi_set.push_back(new_chi_set);
					total_rot_set.push_back(new_rot_set);

					//rh Extra pH rotamers - Protonated ASP
					//rh Represents Hydrogen degrees of freedom for optimized Hbond angles
				} else if ( get_pH_packing_flag() &&
				            aa == aa_asp && variant_type( aav_pH, aa, aav )  ) {
					rotate_proton = 3;

					is_buried = 1;
					if ( neighbors(seqpos) >= active_rotamer_options.extrachi_cutoff ) is_buried = 2;
					is_extra = active_rotamer_options.extrachi_flag(aa,3,is_buried) != NO_EXTRA_CHI_SAMPLES;
					int count;
					get_ser_thr_hchi( is_extra, count, tmp_hchi );

					// Hydrogen Rotamers for OD1 or OD2
					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					new_chi_set(1) = chiarray(1,seqpos);
					new_chi_set(2) = chiarray(2,seqpos);
					for ( int i = 1; i <= count; ++i ) {
						new_chi_set(3) = tmp_hchi(i);
						rotamer_from_chi( new_chi_set, aa, new_rot_set );
						total_chi_set.push_back(new_chi_set);
						total_rot_set.push_back(new_rot_set);
						}

					//rh Extra pH rotamers - Protonated GLU
					//rh Represents Hydrogen degrees of freedom for optimized Hbond angles
				} else if ( get_pH_packing_flag() && aa == aa_glu &&
				            variant_type( aav_pH, aa, aav ) ) {
					rotate_proton = 4;

					is_buried = 1;
					if ( neighbors(seqpos) >= active_rotamer_options.extrachi_cutoff ) is_buried = 2;
					is_extra = active_rotamer_options.extrachi_flag(aa,4,is_buried) != NO_EXTRA_CHI_SAMPLES;
					int count;
					get_ser_thr_hchi( is_extra, count, tmp_hchi );

					// Hydrogen Rotamers for OE1 or OE2
					FArray1D_float new_chi_set( MAX_CHI, 0. );
					FArray1D_int new_rot_set( MAX_CHI, 0 );
					new_chi_set(1) = chiarray(1,seqpos);
					new_chi_set(2) = chiarray(2,seqpos);
					new_chi_set(3) = chiarray(3,seqpos);
					for ( int i = 1; i <= count; ++i ) {
						new_chi_set(4) = tmp_hchi(i);
						rotamer_from_chi( new_chi_set, aa, new_rot_set );
						total_chi_set.push_back(new_chi_set);
						total_rot_set.push_back(new_rot_set);
					}

				}

				//bk no rotatable hydrogens, set chi angles and rotamer definitions
				//bk to be the same as the input rotamer
			} else {                // no rotatable hydrogens
				rotate_proton = 0;

				FArray1D_float new_chi_set( MAX_CHI, 0. );
				FArray1D_int new_rot_set( MAX_CHI, 0 );
				for ( int i = 1; i <= MAX_CHI; ++i ) {
					new_chi_set(i) = chiarray(i,seqpos);
					new_rot_set(i) = rotarray(i,seqpos);
				}
				total_chi_set.push_back(new_chi_set);
				total_rot_set.push_back(new_rot_set);

			}              // rotatable hydrogens ?
			//finished optimize hydrogens - end optimizeH

		} else if ( aa == aa_ala || aa == aa_gly ) { // no rot for gly and ala

			FArray1D_float new_chi_set( MAX_CHI, 0. );
			FArray1D_int new_rot_set( MAX_CHI, 0 );
			total_chi_set.push_back(new_chi_set);
			total_rot_set.push_back(new_rot_set);
			perc = 1.0;

		// use rotamer from the Dunbrack Library
		} else if ( !is_DNA(aa) ) {
			++linenum;
			base_rotno = 0;
			for ( int i = 1, nchi_aa = nchi(aa,1); i <= nchi_aa; ++i ) {
				base_rotno(i) = dun_group_rotno_to_rotno(iphi,ipsi,aa,linenum,i);
			}

			packed_rotno = get_aa_rotno_to_packedrotno(aa,aav,base_rotno);

			// ctsa -change to interpolated starting positions, so that
			//   interpolated scoring always starts out with a canonical
			//   rotamer (canoncal interpolate, that is)
			base_chi = 0.0;
			base_sd = 0.0;
			for ( int i = 1, nchi_aa = nchi(aa,aav); i <= nchi_aa; ++i ) {
				if ( is_chi_proton_rotamer(aa,aav,i) ) goto L123;

				treat_as_angles = true;

				//Objexx:SGM Replaced call below since dun_chi is no longer sliceable
				interpolate_bilinear_by_value(
					dun_chi( phibin, psibin, packed_rotno, i ),
					dun_chi( phibin_next, psibin, packed_rotno, i ),
					dun_chi( phibin, psibin_next, packed_rotno, i ),
					dun_chi( phibin_next, psibin_next, packed_rotno, i ),
					phi_err, psi_err, binrange, treat_as_angles, base_chi(i),
					tmp1, tmp2
				);
//				interpolate_bilinear( phibin, phibin_next, phi_err, psibin, psibin_next,
//				 psi_err, dun_chi(1,1,packed_rotno,i), nbins, nbins, binrange,
//				 treat_as_angles, base_chi(i), tmp1, tmp2 );

				treat_as_angles = false;

				//Objexx:SGM Replaced call below since dun_sd is no longer sliceable
				interpolate_bilinear_by_value(
					dun_sd( phibin, psibin, packed_rotno, i ),
					dun_sd( phibin_next, psibin, packed_rotno, i ),
					dun_sd( phibin, psibin_next, packed_rotno, i ),
					dun_sd( phibin_next, psibin_next, packed_rotno, i ),
					phi_err, psi_err, binrange, treat_as_angles, base_sd(i),
					tmp1, tmp2
				);
//				interpolate_bilinear( phibin, phibin_next, phi_err, psibin, psibin_next,
//				 psi_err, dun_sd(1,1,packed_rotno,i), nbins, nbins, binrange,
//				 treat_as_angles, base_sd(i), tmp1, tmp2 );

				// ctsa - check validity of result
				if ( base_sd(i) < 0.0 ) {
				 //mj from chu: there are 0.0 prob rotamers in library
					std::cout << "ABORT:: invalid dunbrack rotamer sd:" << std::endl;
					std::cout << "  possible invalid aa-rotamer combination" << std::endl;
					std::cout << "aa,rot_id: " << SS( aa ) << SS( base_rotno ) << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}

L123:;
			}

			treat_as_angles = false;

			interpolate_bilinear(phibin,phibin_next,phi_err,psibin,psibin_next,
			 psi_err,dun_prob(1,1,packed_rotno),nbins,nbins,binrange,treat_as_angles,
			 perc,tmp1,tmp2);

			// ctsa - check validity of result
			if ( perc < 0.0 ) { //mj from chu: there are 0.0 prob rotamers in library
				std::cout << "ABORT:: invalid dunbrack rotamer prob:" << std::endl;
				std::cout << "  possible invalid aa-rotamer combination" << std::endl;
				std::cout << "aa,rot_id: " << SS( aa ) << SS( base_rotno ) << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			get_extrachi(aa,aav,base_chi,base_sd,neighbors(seqpos),
				total_chi_set,total_rot_set,seqpos,Task.get_designmap());
		}
		else { // dna rotamer: must make dummy containers for the next loop
			FArray1D_float new_chi_set( MAX_CHI, 0. ); // dummy
			total_chi_set.push_back(new_chi_set);
			FArray1D_int new_rot_set( MAX_CHI, 0 ); // dummy
			total_rot_set.push_back(new_rot_set);
		}

		// ctsa - go through the list of chisets for this base rotamer and
		// decide which ones to keep; enter those into the corresponding
		// rotamer arrays

		unsigned int chiset_id = 0;
		//ja iterating total_rot_set in this fashion is risky!
		std::list<FArray1D_int>::iterator rotset = total_rot_set.begin();
		for( std::list<FArray1D_float>::iterator chiset = total_chi_set.begin();
				 chiset != total_chi_set.end(); ++chiset, ++rotset ) {

			++chiset_id;

			// bk if current_chi_already_included then make sure that rotamer being added
			// bk doesn't have the same chi angles as the current
			if ( current_chi_already_included ) {
				if ( !is_DNA(aa) && aa == res(seqpos) && aav == res_variant(seqpos) &&
				 coord_match_aa(aa,aav,full_coord(1,1,seqpos)) ) {
					for ( int i = 1, ie = nchi(aa,aav); i <= ie; ++i ) {
						if ( std::abs(subtract_chi_angles( (*chiset)(i),
						  chiarray(i,seqpos),aa,i)) > 0.1 ) goto L150;
					}
					goto L200; // this rotamer is already included
L150:;
				}
			}

			// check for barcode constraint
			if ( force_rotamer &&
			 !barcode_allow_rotamer( seqpos, aa, aav, *chiset, *rotset ) ) {
				goto L200;
			}

			// increment counters
			++rotnum;

			// jk Note: push_rotamer updates nrotamers_
			push_rotamer( seqpos, aa, aav, rotnum );

			for ( int i = 1; i <= MAX_CHI; ++i ) {
				set_rchi(i,nrotamers()) = (*chiset)(i);
				set_rrot(i,nrotamers()) = (*rotset)(i);
			}

			//bk calculate the probability of the sidechain conformation
			compute_rperc(nrotamers());

			// ctsa - filter rotamers again at this stage to get rid of
			//   highly unlikely extra chi rots ( check against base rot
			//   by preventing elim of chiset_id == 1 )

			if ( ( get_rperc(nrotamers()) < min_extrachi_rot_prob ) &&
					 ( mode == "design" ) && ( chiset_id != 1 ) ) {

				// jk Note: pop_rotamer updates nrotamers_
				pop_rotamer();

				--rotnum;
				goto L200;
			}

			if ( mode == "optimizeH" ) {
				for ( int j = 1, je = natoms( aa, aav ); j <= je; ++j ) {
					if ( atom_type_char( j, aa, aav ) == "HOH " ) {
						place_atom( aa, aav, j, get_rotcoord(nrotamers()) );
					} else {
						for ( int i = 1; i <= 3; ++i ) {
							set_rotcoord( i, j, nrotamers() ) = full_coord( i, j, seqpos );
						}
					}
				}

//				//jjh For histidine, the proton placement isn't a rotation
//				if ( aa == aa_his && aav != res_variant(seqpos) ) {
//					get_coords_by_name_and_placement( aa,
//								res_variant(seqpos), full_coord(1,1,seqpos),
//								aav, get_rotcoord( nrotamers() ) );
//				}

//				//jjh For carboxylic acids, assign by name, then place rotatable
//				//jjh to prepare for rotation below
//				if ( (aa == aa_asp || aa == aa_glu) && aav != res_variant(seqpos) ) {
//					get_coords_by_name_and_placement( aa,
//								res_variant(seqpos), full_coord(1,1,seqpos),
//								aav, get_rotcoord( nrotamers() ) );
//				}

				//jjh Protonation changes involve different covalent structure
				//jjh and must be handled specially
				if( ( aav != res_variant( seqpos ) ) &&
						( variant_type( aav_pH, aa, aav ) ||
							variant_type( aav_pH, aa, res_variant( seqpos ) ) ||
							variant_type( aav_his_taut, aa, aav ) ||
							variant_type( aav_his_taut, aa, res_variant( seqpos ) ) ) ) {
					get_coords_by_name_and_placement( aa,
								res_variant(seqpos), full_coord(1,1,seqpos),
								aav, get_rotcoord( nrotamers() ) );
				}



				//bk set the hydrogen torsion angle
				if ( rotate_proton > 0 && rotate_proton <= nchi(aa,aav) ) {
					change_chi_angle_update_coors(nrotamers(),rotate_proton,get_rchi(rotate_proton,nrotamers()));
				}

			} else if ( is_DNA(aa) ) {
			//ja add DNA base(root) 'rotamer'

//				std::cout << "Adding DNA rotamer for type " << aa_name3(aa) <<
//				 " at " << seqpos << " (" << pdb::pdb_res_num(seqpos) << ") " <<
//				 std::endl;

				int const nataa( res(seqpos) ), nataav( res_variant(seqpos) );
				align_dna_base( aa, aav, get_rotcoord( nrotamers() ),
				                nataa, nataav, full_coord(1,1,seqpos) );
				// escape the for loop (just making one 'rotamer' per DNA type)
				perc_total = 1.0;

			} else {
				get_rot_coord( full_coord, aa, aav, get_rchi(nrotamers()),
					get_rotcoord(nrotamers()), seqpos );
			}

			fill_rotactcoord(nrotamers());

			//bk throw out rotamers that bump the backbone or are very improbable
			//bk for an environment
			//ctsa ...but retain at least the single best rotamer for each aa-pos,
			//ctsa regardless of bump score

			//jjh skip this for the rotamerize option, since you want the closest,
			//jjh regardless of whether it bumps
			if ( ( aav != 1 || mode == "design" ) && ( !rotamerize ) ) {
				bump_check( aa, aav, seqpos, get_rotcoord(nrotamers()), full_coord,
				 res, res_variant, Task.get_designmap(), total_residue, bumpenergy );

				//lin check if attached water are bured
				water_rotamer_check( aa, aav, get_rotcoord(nrotamers()), full_coord, res,
				 res_variant, total_residue, h2o_nb, is_exposed_water, h2o_intra_repE,
				 h2o_intra_hbE );

				// ctsa - decide whether to keep the current rotamer
				// based on the bump energy and the best rot found
				// for this aa-pos
				if ( ( is_exposed_water && !hydrate_dna ) ||
					     h2o_intra_repE > h2oE_clash_cut) {
					delete_current_rot = true;
				} else {
					iterate_bump_selector( bumpenergy, delete_current_rot,
					                       delete_previous_rot );
				}

				if ( delete_current_rot ) {
					// jk Note: pop_rotamer updates nrotamers_
					pop_rotamer();
					--rotnum;
				} else if ( delete_previous_rot ) {
					// delete previous rotamer and rerun current rotamer
					// jk Note: pop_rotamer updates nrotamers_
					pop_rotamer();
					pop_rotamer();
					rotnum -= 2;
					--chiset_id;
					--chiset;
					--rotset;
				}

			}     // end check bump

			// jk Look for a suitable base rotamer
			// (same chi angles but a different amino acid variant, for speeding up energy calculations)
			find_suitable_base_rotamer(nrotamers());

L200:;
		} // end chi set while loop

		perc_total += perc;

	} // end perc while loop
	nrotaa = rotnum;

	if ( force_rotamer && nrotamers() == start_nrotamers ) {
		std::cout << "No rotamers found -- turning off force_rotamer!" <<
			std::endl;
		force_rotamer = false;
		try_again = true;
	}
	} // end 'while'

  if (  (Task.get_mode() != "optimizeH") && ( active_rotamer_options.do_rotamer_bias || active_rotamer_options.minimize_best_rotamers) ) {
		if ( active_rotamer_options.minimize_best_rotamers || ( ! active_rotamer_options.bias_rotamers_by_replacement ) ) {
			build_biased_rotamers_by_appending( total_residue, res, res_variant, full_coord, seqpos, aa, Task );
		} else {
			build_biased_rotamers_by_replacement( total_residue, res, res_variant, full_coord,seqpos, aa, Task );
		}
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin add_extra_rot_from_coord
///
/// @brief
///bk add native rotamers and coordinates to rotamer list
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  res - [in/out]? - amino acid at each residue position
/// @param  res_variant - [in/out]? - amino acid variant at each sequence position
/// @param  nrotaa - [in/out]? -
/// @param  design_map
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  pdbrot - [in/out]? - pdb rotamer
/// @param  pdbchi - [in/out]? - pdb chi angles
/// @param  total_residue - [in/out]? - number of residues
/// @param  actcoord - [in/out]? -
/// @param  full_coord - [in/out]? - template coordinates
/// @param  current_rot_index - [in/out]? - rotno for current/native rot at
///       each res pos
/// @param  rot_aa_variant - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::add_extra_rot_from_coord(
	int const i,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	int & nrotaa,
	const DesignMap & design_map,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray2Da_int pdbrot,
	FArray2Da_float pdbchi,
	int const total_residue,
	FArray2D_float const & actcoord,
	FArray3D_float const & full_coord,
	FArray1D_int & current_rot_index
)
{

	using namespace aaproperties_pack;
	using namespace param;

	phi.dimension( total_residue );
	psi.dimension( total_residue );
	pdbrot.dimension( 4, total_residue );
	pdbchi.dimension( 4, total_residue );

	if ( !design_map.repack_residue(i) ) return;

	if ( ! coord_match_aa(res(i),res_variant(i),full_coord(1,1,i)) ) return;

	++nrotaa;
	// jk Note: push_rotamer updates nrotamers_
	push_rotamer( i, res(i), res_variant(i), nrotaa );
	current_rot_index(i) = nrotamers();
	for ( int j = 1; j <= MAX_CHI; ++j ) {
		set_rchi(j,nrotamers()) = pdbchi(j,i);
		set_rrot(j,nrotamers()) = pdbrot(j,i);
	}

	compute_rperc(nrotamers());

	for ( int j = 1, je = natoms(res(i),res_variant(i)); j <= je; ++j ) {
		for ( int k = 1; k <= 3; ++k ) {
			set_rotcoord(k,j,nrotamers()) = full_coord(k,j,i);
		}
	}
	for ( int k = 1; k <= 3; ++k ) {
		set_rotactcoord(k,nrotamers()) = actcoord(k,i);
	}

}


/////////////////////////////////////////////////////////////////////////
/// @begin add_rot_pert_input
///
/// @brief
///bk add new rotamers to rotamer list around the native rotamers
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  res - [in/out]? - amino acid at each residue position
/// @param  res_variant - [in/out]? - amino acid variant at each sequence position
/// @param  nrotaa - [in/out]? -
/// @param  design_map
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  pdbrot - [in/out]? - pdb rotamer
/// @param  pdbchi - [in/out]? - pdb chi angles
/// @param  total_residue - [in/out]? - number of residues
/// @param  actcoord - [in/out]? -
/// @param  full_coord - [in/out]? - template coordinates
/// @param  current_rot_index - [in/out]? - rotno for current/native rot at
///       each res pos
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::add_rot_pert_input(
	int & i,
	FArray1Da_int res,
	FArray1Da_int res_variant,
	int & nrotaa,
	const DesignMap & design_map,
	FArray1Da_float phi,
	FArray1Da_float psi,
	int & total_residue,
	FArray3Da_float full_coord,
	FArray2Da_float chi_curr,
	FArray2Da_int rot_curr
)
{

	using namespace aaproperties_pack;
	using namespace design;
	using namespace param;

	res.dimension( total_residue );
	res_variant.dimension( total_residue );
	phi.dimension( total_residue );
	psi.dimension( total_residue );
	full_coord.dimension( 3, MAX_ATOM(), total_residue );
	chi_curr.dimension( MAX_CHI, total_residue );
	rot_curr.dimension( MAX_CHI, total_residue );

	if ( !design_map.repack_residue(i) ) return;

	int const aa=res(i);
	int const aav=res_variant(i);

	if ( ! coord_match_aa(aa,aav,full_coord(1,1,i)) ) return;

	for ( int j = 1; j <= MAX_CHI; ++j ) {
		if ( nchi(aa,aav) >= j ) {
			for ( int add = 0; add <= 1; ++add ) {

				++nrotaa;
				// jk Note: push_rotamer updates nrotamers_
				push_rotamer( i, aa, aav, nrotaa );

				FArray1D_float new_chi( MAX_CHI );
				for ( int k = 1; k <= MAX_CHI; ++k ) {
					if ( k == j ) {
						if ( add == 0 ) {
							new_chi(k)=chi_curr(k,i)+active_rotamer_options.pert_size;
						} else {
							new_chi(k)=chi_curr(k,i)-active_rotamer_options.pert_size;
						}
					} else {
						new_chi(k)=chi_curr(k,i);
					}
				}

				set_all_chi_to_periodic_range(new_chi,aa);
				for ( int k = 1; k <= MAX_CHI; ++k ) {
					set_rchi(k,nrotamers()) = new_chi(k);
					set_rrot(k,nrotamers()) = rot_curr(k,i);
				}

				compute_rperc(nrotamers());

				get_rot_coord(full_coord,res(i),aav,get_rchi(nrotamers()),get_rotcoord(nrotamers()),i);

				fill_rotactcoord(nrotamers());

			}
		}
	}
	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin add_extra_rot_from_chi
///
/// @brief
///bk add native rotamers and coordinates to rotamer list
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  res - [in/out]? - amino acid at each residue position
/// @param  aav - [in/out]? - amino acid variant at each position
/// @param  nrotaa - [in/out]? - number of rotamers
/// @param  design_map
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  pdbrot - [in/out]? - pdb rotamer
/// @param  pdbchi - [in/out]? - pdb chi angles
/// @param  total_residue - [in/out]? - number of residues
/// @param  actcoord - [in/out]? -
/// @param  full_coord - [in/out]? - template coordinates
/// @param  rotindex - [in/out]? - identifies rotamars with amino acid and position
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::add_extra_rot_from_chi(
	int const i,
	FArray1D_int const & res,
	FArray1D_int const & aav,
	int & nrotaa,
	const DesignMap & design_map,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray2Da_int pdbrot,
	FArray2Da_float pdbchi,
	int const total_residue,
	FArray3D_float const & full_coord,
	FArray1D_int & current_rot_index
)
{
	using namespace param;

	phi.dimension( total_residue );
	psi.dimension( total_residue );
	pdbrot.dimension( 4, total_residue );
	pdbchi.dimension( 4, total_residue );

	if ( !design_map.repack_residue(i) ) goto L52;
	for ( int j = 1; j <= 4; ++j ) {
		if ( pdbrot(j,i) != 0 ) goto L40;
	}
	return; // invalid rotamer
L40:
	++nrotaa;

	// jk Note: push_rotamer updates nrotamers_
	push_rotamer( i, res(i), aav(i), nrotaa );
	current_rot_index(i) = nrotamers();
	for ( int j = 1; j <= MAX_CHI; ++j ) {
		set_rchi(j,nrotamers()) = pdbchi(j,i);
		set_rrot(j,nrotamers()) = pdbrot(j,i);
	}

	compute_rperc(nrotamers());

	get_rot_coord(full_coord,res(i),aav(i),get_rchi(nrotamers()),get_rotcoord(nrotamers()),i);

	fill_rotactcoord(nrotamers());

L52:;

}


//////////////////////////////////////////////////////////////////////////////
/// @begin add_extra_rot_to_seqpos
///
/// @brief
///
/// @detailed
///
/// @param  aa - [in/out]? -
/// @param  aav - [in/out]? -
/// @param  seqpos - [in/out]? -
/// @param  total_residue - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  full_coord - [in/out]? -
/// @param  extra_chi - [in/out]? -
/// @param  extra_rot - [in/out]? -
/// @param  rotchi - [in/out]? -
/// @param  rrot - [in/out]? -
/// @param  rperc - [in/out]? -
/// @param  rotindex - [in/out]? -
/// @param  & nrot - [in/out]? -
/// @param  & nrotaa - [in/out]? -
/// @param  rot_res_variant - [in/out]? -
/// @param  baserotamer - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSet::add_extra_rot_to_seqpos(
	int & aa,
	int & aav,
	int & seqpos,
	FArray1Da_float phi,
	FArray1Da_float psi,
	FArray3Da_float full_coord,
	FArray2Da_float extra_chi,
	FArray2Da_int extra_rot,
	int & nrotaa
)
{
	using namespace param;

	phi.dimension( MAX_RES() );
	psi.dimension( MAX_RES() );
	full_coord.dimension( 3, MAX_ATOM(), MAX_RES() );
	extra_chi.dimension( MAX_CHI, MAX_RES() );
	extra_rot.dimension( MAX_CHI, MAX_RES() );

//chu increase the counter
	for ( int j = 1; j <= MAX_CHI; ++j ) {
		if ( extra_rot(j,seqpos) != 0 ) goto L10;
	}
	return;
L10:

	++nrotaa;
	// jk Note: push_rotamer updates nrotamers_
	push_rotamer( seqpos, aa, aav, nrotaa );

//chu   add in angles and rotno
	for ( int j = 1; j <= MAX_CHI; ++j ) {
		set_rchi(j,nrotamers()) = extra_chi(j,seqpos);
		set_rrot(j,nrotamers()) = extra_rot(j,seqpos);
	}

	//chu   calculate the probability
	compute_rperc(nrotamers());

//chu   determine the coordinate and action coordinate
	get_rot_coord(full_coord,aa,aav,get_rchi(nrotamers()),get_rotcoord(nrotamers()),seqpos);

	fill_rotactcoord(nrotamers());

}

////////////////////////////////////////////////////////////////////////////////
///@begin get_rotamers_for_ligand_aa
///
///@brief loads rotamers from the enable_ligaa_ns namespace for ligand rotamers
///
///@param[in] int const seqpos the sequence position in pose of the ligand
///@param[in] FArray1Da_int res aa type at each sequence position
///@param[in] FArray1Da_int res_variant aa variant type at each sequence position
///@param[in] int const aa the ligand amino acid number
///@param[in] int const aav amino acid variant
///@param[in,out] int & nrotaa rotamer number
///@param[in] FArray_3D_float full_coord coordinates of current protein
///@param[in] PackerTask const & Task
///
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
void
RotamerSet::get_rotamers_for_ligand_aa(
	int const seqpos,
	int const total_residue,
	FArray1Da_int res,
	FArray1Da_int res_variant,
	int const aa,
	int const aav,
	int & nrotaa,
	FArray3D_float full_coord,
	PackerTask const & Task
){
	//must add a rotamer for each non bumping ligand rotamer
	//assert that we are dealing with a ligand position
	assert( res(seqpos) == aa && res_variant(seqpos) == aav &&  param_aa::is_ligand(aa) );
	std::cout << "getting ligand rotamers" << std::endl;
	//The idea is to iterate through all the ligand rotamers and cache the bumpenergies
	std::multimap< float , FArray2D_float > bump_energy_rot_map;
	FArray2D_float input_coord(3,aaproperties_pack::natoms(aa,aav));
	for( int atom =1 ; atom <=aaproperties_pack::natoms(aa,aav); atom++){
		input_coord(1,atom)=full_coord(1,atom,seqpos);
		input_coord(2,atom)=full_coord(2,atom,seqpos);
		input_coord(3,atom)=full_coord(3,atom,seqpos);
	}
	std::cout << "before small_dock" << std::endl;
	FArray3D_float rotamer_library(4,aaproperties_pack::natoms(aa,aav),1000,0.0);
	int nconformations=0;
	float trans, rot;
	//load a rotamer library enable_ligaa_ns
	Task.get_perturb_ligand_rotamers( trans, rot);
	rotamer_library=small_dock_perturbation_pose_ligand(aa,aav,trans, rot, input_coord, nconformations);

	//evaluate the bump energy of all the rotamers in the rotamer_library
	int total_residue2=total_residue;
	for( int conf=1;conf<=nconformations;conf++){
		float bumpenergy=0.0;
		FArray2D_float conformation(4, param::MAX_ATOM());
		conformation=load_conformation_from_FArray3D_float(rotamer_library, conf, aa, aav);
		bump_check( aa, aav, seqpos, conformation, full_coord, res, res_variant, Task.get_designmap(), total_residue2, bumpenergy);
		std::pair< float, FArray2D_float > p(bumpenergy, conformation);
		bump_energy_rot_map.insert(p);
	}

	//added suitable rotamers up to 25 to the RotamerSet
	std::multimap< float , FArray2D_float >::iterator cconf=bump_energy_rot_map.begin();
	int added_rotamers=0;
	//utility::io::ozstream iunit( "coordinates2.txt");
	//for( int conf=1; conf<=nconformations; conf++){
		//for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
			//iunit << F(10,3, rotamer_library(1,atom,conf)) << " ";
			//iunit << F(10,3, rotamer_library(2,atom,conf)) << " ";
			//iunit << F(10,3, rotamer_library(3,atom,conf)) << std::endl;
		//}
		//iunit << "TER" << std::endl;
	//}
  //iunit.close();
	do{
		added_rotamers++;
		push_rotamer( seqpos, aa, aav, ++nrotaa );
		std::cout << "Added ligand rotamer " << added_rotamers << " nrotamers " << nrotamers() << " energy " << ((*cconf).first) << std::endl;
//		get_rotcoord( nrotaa )=((*cconf).second  );
		for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
				set_rotcoord(1,atom,nrotamers())=((*cconf).second)(1,atom);
				set_rotcoord(2,atom,nrotamers())=((*cconf).second)(2,atom);
				set_rotcoord(3,atom,nrotamers())=((*cconf).second)(3,atom);
     }
//		FArray2D_float rcoord=get_rotcoord(nrotamers());
//		for( int atom=1; atom<=aaproperties_pack::natoms(aa,aav); atom++){
//      iunit << F(10,3, rcoord(1,atom)) << " ";
//       iunit << F(10,3, rcoord(2,atom)) << " ";
//       iunit << F(10,3, rcoord(3,atom)) << std::endl;
//     }
//     iunit << "TER" << std::endl;

		//There is a general issue with the size of chi arrays.
		//the number of ligand chi angles varies from 0 up to 20
		//param::MAX_CHI is set to 4 and there are areas of the code
		//such as in DNA that depend on MAX_CHI being 4. rrot rchi both
		//are sized based on this parameter. This creates the probability that
		//some information is lost in the rchi array. Thus I zero out this
		//information at this point.
		FArray1D_int rrot( param::MAX_CHI );
//		FArray1D_float rchi( param::MAX_CHI );
		rrot=create_rotid( int(((*cconf).second)(4,1)) );
//		rchi=get_current_chi( ((*cconf).second), aa, aav );
		for ( int i = 1; i <= param::MAX_CHI; ++i ) {
//			set_rchi( i,nrotamers() ) = rchi(i);
			set_rchi( i,nrotamers() ) = 0;
			set_rrot( i,nrotamers() ) = rrot(i);
		}
		set_rperc(nrotamers(), 1.0);
		cconf++;
	}while( (*cconf).first < 5.0 && added_rotamers < 100  && added_rotamers < nconformations );
//	iunit.close();
	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_optimize
///
/// @brief
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSet::rotamer_optimize(
	pose_ns::Pose & pose,
  const DesignMap & design_map
) {

	using namespace param;
	using namespace param_aa;
	using namespace template_pack;

	int rloop;
	int seqpos, raa, raav;
	int site, cloop;
	int rotamers_minimized(0);
	FArray1D_int backup_aa( MAX_RES()() );
	FArray1D_int backup_aav( MAX_RES()() );
	FArray3D_float backup_coords( 3, MAX_ATOM()(), MAX_RES()() );
	FArray1D_float dummy_chi( MAX_CHI );
	FArray2D_float temp_rot( 3, MAX_ATOM()() );

//jjh store the template

	backup_aa = pose.res();
	backup_aav = pose.res_variant();
	backup_coords = pose.full_coord();

//jjh At each position represented by rotamers, overwrite
//jjh the aa to be glycine.

	for ( site = 1; site <= pose.total_residue(); ++site ) {
		if ( design_map.repack_residue(site) ) {
			pose.set_res(site,aa_gly);
			pose.set_res_variant(site,1);
//jjh build glycine
			for ( cloop = 1; cloop <= 4; ++cloop ) {
				dummy_chi(cloop) = 0.0;
			}
			get_rot_coord(pose.full_coord(), pose.res(site), pose.res_variant(site), dummy_chi, temp_rot,
			 site);

			pose.copy_sidechain(site, aa_gly, 1, temp_rot, true);
		}
	}

	pre_design_trials(pose);

//jjh loop over rotamers
#ifndef WIN32
	float minstart = clock();
#endif

	for ( rloop = 1; rloop <= nrotamers(); ++rloop ) {

		if(!design::active_rotamer_options.rot_opt && design::active_rotamer_options.minimize_best_rotamers && (!rotamer_explosion::rotamers_to_minimize[rloop]) ) { continue;}

		seqpos = report_seqpos(rloop);
		raa   = report_aa(rloop);
		raav   = report_aav(rloop);


		get_trial_energies(get_rperc(rloop),get_rotcoord(rloop),
											 get_rot_born_radius(rloop), seqpos,raa,raav, get_rotactcoord(rloop),
											 pose.total_residue(), pose.res(), pose.res_variant(), pose.full_coord());

		minimize_trial_energies(get_rchi(rloop), get_rrot(rloop), seqpos, raa, raav,
														get_rotcoord(rloop), get_rot_born_radius(rloop),
														get_rotactcoord(rloop), 0.005, "dfpmin", pose.total_residue(),
														pose.res(), pose.res_variant(), pose.full_coord());

		rotamers_minimized++;

	}

#ifndef WIN32
	float minend = clock();
	float mintime = (minend - minstart) / CLOCKS_PER_SEC;

	std::cout << "Rotamer one-body energy minimization: " << rotamers_minimized << " rotamers were minimized in " << mintime << " seconds." << std::endl ;
#endif

	//jjh restore the native aa and aav information to the full-coord set

	pose.set_res(backup_aa);
	pose.set_res_variant(backup_aav);
	pose.set_full_coord(backup_coords);

}


////////////////////////////////////////////////////////////////////////////////
/// @begin select_closest_rotamer
///
/// @brief
///
/// Given a number of rotamers at each position, selects the rotamer
/// with the smallest heavy atom rmsd from the native.
///
/// @detailed
///
/// This function is intended as a quick way to 'rotamerize' a structure
/// by selecting the rotamer closest to the native at each site.  In this
/// sense, it serves as a replacement for sim_annealing() as a function
/// for selecting rotamers, here based on rmsd from native rather than
/// based on energy.  Phe, Tyr, Asp, and Glu are flipped to see if a
/// symmetrically related rotamer fits better due to naming convention
/// differences.  The output is sent to standard error like most Rosetta
/// output.
///
/// @param[out]  bestrotamer_at_seqpos - out
///
/// This parameter is an array that returns the best rotamer at each sequence
/// position.
///
/// @global_read
///
/// Lots of rotamer information is accessed, as is information about the
/// template structure. Amino acid information is needed to determine
/// how many heavy atoms there are, and which ones they are.
///
/// @global_write
///
/// No global variables are altered by this function. Certain rotamer
/// atoms are flipped, but should also be flipped back.
///
/// @remarks
///
/// @references
///
/// @authors
///
/// jjh 8-19-03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::select_closest_rotamer(
    pose_ns::Pose & pose,
    FArray1Da_int bestrotamer_at_seqpos
) {

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace template_pack;
	using namespace numeric::constants::f;

	bestrotamer_at_seqpos.dimension( MAX_RES() );

//jjh Picks the library-derived rotamer that is closest to the
//jjh native side chain as judged by coordinate rmsd.

//jjh internal variables
	float test_rmsd, best_rmsd;
	float accum, accum2;
	float dis;
	FArray1D_float natchi( 4 );
	FArray1D_float rotchi( 4 );
	FArray1D_float dchi( 4 );
	int this_pos, this_aa, this_aav;
	FArray1D_int first_at_pos( MAX_RES()() );
	FArray1D_int last_at_pos( MAX_RES()() );
	int c1,c2,c3,c4;
	FArray2D_float mat( 3, 3 );
	FArray1D_float vec( 3 );


//jjh     std::cout << "starting rotamerize" << std::endl;


//jjh initialize variables
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		bestrotamer_at_seqpos(seqpos) = 0;
		first_at_pos(seqpos) = 0;
		last_at_pos(seqpos) = 0;
	}

	if ( nrotamers() == 0 ) return;

//jjh figure out the first and last rotamer at each position
	for ( int i = nrotamers(); i >= 1; --i ) {
		this_pos = report_seqpos(i);
		first_at_pos(this_pos) = i;
	}

	for ( int i = 1; i <= nrotamers(); ++i ) {
		this_pos = report_seqpos(i);
		last_at_pos(this_pos) = i;
	}


	for ( int i = 1; i <= pose.total_residue(); ++i ) {
		if ( first_at_pos(i) > 0 ) {
			this_aav = pose.res_variant(i);
			this_aa = pose.res(i);
			best_rmsd = 9999.9;
			bestrotamer_at_seqpos(i) = -1;
			for ( int j = first_at_pos(i), je = last_at_pos(i); j <= je; ++j ) {
				if ( this_aa == pose.res(i) ) {
//jjh test the rmsd for this side chain

					accum = 0.0;
					for ( int k = 1, ke = nheavyatoms( this_aa, report_aav(j) ); k <= ke; ++k ) {
						distance_bk( get_rotcoord(j)(1,k), pose.full_coord()(1,k,i), dis );
						accum += dis * dis;
					}


//jjh check to see if a re-naming of symmetrically related atoms
//jjh gives an even better fit for certain amino acids
					if ( ( this_aa == aa_phe ) || ( this_aa == aa_tyr ) ||
					 ( this_aa == aa_asp ) ) {

//jjh flip about chi2
						c1 = chi_atoms(1,2,this_aa,pose.res_variant(i));
						c2 = chi_atoms(2,2,this_aa,pose.res_variant(i));
						c3 = chi_atoms(3,2,this_aa,pose.res_variant(i));
						c4 = chi_atoms(4,2,this_aa,pose.res_variant(i));

						dihedral_bk(pose.full_coord()(1,c1,i),pose.full_coord()(1,c2,i),pose.full_coord()(1,c3,i),pose.full_coord()(1,c4,i),
						 rotchi(2));

						dihedral_bk(get_rotcoord(j)(1,c1), get_rotcoord(j)(1,c2), get_rotcoord(j)(1,c3),
						 get_rotcoord(j)(1,c4), rotchi(2));

						change_chi_angle_update_coors(j,2,rotchi(2)-180.);

						accum2 = 0.0;
						for ( int k = 1, ke = nheavyatoms( this_aa, report_aav(j) ); k <= ke; ++k ) {
							distance_bk( get_rotcoord(j)(1,k), pose.full_coord()(1,k,i), dis );
							accum2 += dis * dis;
						}

						if ( accum2 < accum ) {
							accum = accum2;
						} else {
							change_chi_angle_update_coors(j,2,rotchi(2));
						}

					} else if ( (this_aa == aa_glu) ) {


//jjh flip about chi3
						c1 = chi_atoms(1,3,this_aa,pose.res_variant()(i));
						c2 = chi_atoms(2,3,this_aa,pose.res_variant()(i));
						c3 = chi_atoms(3,3,this_aa,pose.res_variant()(i));
						c4 = chi_atoms(4,3,this_aa,pose.res_variant(i));

						dihedral_bk(get_rotcoord(j)(1,c1),get_rotcoord(j)(1,c2),get_rotcoord(j)(1,c3),
						 get_rotcoord(j)(1,c4), rotchi(3));

						change_chi_angle_update_coors(j,3,rotchi(3)-180.);

						accum2 = 0.0;
						for ( int k = 1, ke = nheavyatoms( this_aa, report_aav(j) ); k <= ke; ++k ) {
							distance_bk(get_rotcoord(j)(1,k),pose.full_coord()(1,k,i),dis);
							accum2 += dis * dis;
						}

						if ( accum2 < accum ) {
							accum = accum2;
						} else {
							change_chi_angle_update_coors(j,3,rotchi(3));
						}

					} else if (this_aa == aa_arg) {
//jjh flip about virtual chi 5
						c1 = 7;
						c2 = 8;
						c3 = 9;
						c4 = 10;

						dihedral_bk( get_rotcoord(j)(1,c1), get_rotcoord(j)(1,c2), get_rotcoord(j)(1,c3),
						 get_rotcoord(j)(1,c4), rotchi(4) );

						getrot_bk(get_rotcoord(j)(1,c2), get_rotcoord(j)(1,c3), pi, mat, vec);

						move_bk( get_rotcoord(j)(1,10), mat, vec );
						move_bk( get_rotcoord(j)(1,11), mat, vec );
						move_bk( get_rotcoord(j)(1,13), mat, vec );
						move_bk( get_rotcoord(j)(1,14), mat, vec );
						move_bk( get_rotcoord(j)(1,15), mat, vec );
						move_bk( get_rotcoord(j)(1,16), mat, vec );

						accum2 = 0.0;
						for ( int k = 1, ke = nheavyatoms( this_aa, report_aav(j) ); k <= ke; ++k ) {
							distance_bk( get_rotcoord(j)(1,k), pose.full_coord()(1,k,i), dis );
							accum2 += dis * dis;
						}

						if ( accum2 < accum ) {
							accum = accum2;
						} else {
							move_bk( get_rotcoord(j)(1,10), mat, vec );
							move_bk( get_rotcoord(j)(1,11), mat, vec );
							move_bk( get_rotcoord(j)(1,13), mat, vec );
							move_bk( get_rotcoord(j)(1,14), mat, vec );
							move_bk( get_rotcoord(j)(1,15), mat, vec );
							move_bk( get_rotcoord(j)(1,16), mat, vec );
						}
					}

					test_rmsd = (accum/
												(float(nheavyatoms(this_aa, report_aav(j)))))* 0.5;
					test_rmsd = std::sqrt(test_rmsd);
					if ( test_rmsd < best_rmsd ) {
						best_rmsd = test_rmsd;
						bestrotamer_at_seqpos(i) = j;
					}
				}
			}
			if ( bestrotamer_at_seqpos(i) == -1 ) {
				std::cout << "Fatal error in rotamerize - native a.a. not found!" << std::endl;
			}

// compute some stuff for the 'best' rotamer
// first, absolute error in dihedral values

			dchi(1) = 0.0;
			dchi(2) = 0.0;
			dchi(3) = 0.0;
			dchi(4) = 0.0;
			natchi(1) = 0.0;
			natchi(2) = 0.0;
			natchi(3) = 0.0;
			natchi(4) = 0.0;
			rotchi(1) = 0.0;
			rotchi(2) = 0.0;
			rotchi(3) = 0.0;
			rotchi(4) = 0.0;

			for ( int ichi = 1, ichie = nchi( this_aa, pose.res_variant(i) ); ichi <= ichie; ++ichi ) {
				c1 = chi_atoms( 1, ichi, this_aa, pose.res_variant(i) );
				c2 = chi_atoms( 2, ichi, this_aa, pose.res_variant(i) );
				c3 = chi_atoms( 3, ichi, this_aa, pose.res_variant(i) );
				c4 = chi_atoms( 4, ichi, this_aa, pose.res_variant(i) );

				dihedral_bk( pose.full_coord()(1,c1,i), pose.full_coord()(1,c2,i), pose.full_coord()(1,c3,i), pose.full_coord()(1,c4,i), natchi(ichi) );

				dihedral_bk(get_rotcoord(bestrotamer_at_seqpos(i))(1,c1),
				get_rotcoord(bestrotamer_at_seqpos(i))(1,c2),
				get_rotcoord(bestrotamer_at_seqpos(i))(1,c3),
				get_rotcoord(bestrotamer_at_seqpos(i))(1,c4), rotchi(ichi));
				dchi(ichi) = std::abs(rotchi(ichi)-natchi(ichi));
				if ( dchi(ichi) > 180.0 ) {
					dchi(ichi) = 360.0-dchi(ichi);
				}
			}

			std::cout << " ROTS: " << I( 3, i ) << " AA: " << aa_name3(this_aa)
			  << " rmsd: " << F( 8, 3, best_rmsd ) << std::endl;
			std::cout << " ROTS: " << I( 3, i ) << " AA: " << aa_name3(this_aa)
			  << " native chis: " << F( 8, 3, natchi(1) ) << " dc2: "
			  << F( 8, 3, natchi(2) ) << " dc3: " << F( 8, 3, natchi(3) )
			  << " dc4: " << F( 8, 3, natchi(4) ) << std::endl;
			std::cout << " ROTS: " << I( 3, i ) << " AA: " << aa_name3(this_aa)
			  << " rotamer chis: " << F( 8, 3, rotchi(1) ) << " dc2: "
			  << F( 8, 3, rotchi(2) ) << " dc3: " << F( 8, 3, rotchi(3) )
			  << " dc4: " << F( 8, 3, rotchi(4) ) << std::endl;
			std::cout << " ROTS: " << I( 3, i ) << " AA: " << aa_name3(this_aa)
			  << " delta chis: " << F( 8, 3, dchi(1) ) << " dc2: "
			  << F( 8, 3, dchi(2) ) << " dc3: " << F( 8, 3, dchi(3) )
			  << " dc4: " << F( 8, 3, dchi(4) ) << std::endl;
		}
	}

	std::cout << "ROTAMERIZED" << std::endl;

}



////////////////////////////////////////////////////////////////////////////////
/// @begin update_rotamer_alignment
///
/// @brief
///    Carry out translation / rotation to adjust *this onto the pose backbone
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSet::update_rotamer_alignment(
	pose_ns::Pose & curr_pose
) {

	using namespace aaproperties_pack;
	using namespace param;

	std::cout << "Updating rotamer alignment for new backbone orientation" << std::endl;

	// jk Transform each rotamer onto the new backbone
	//	int prev_seqpos(0);
	FArray2D_float Mgl( 4, 4 );
	FArray2D_float start_rot( 3, MAX_ATOM() );
	FArray1D_float start_actrot( 3 );
	for ( int i = 1; i <= nrotamers(); ++i ) {

		// Store the starting rotamer
		int const seqpos = report_seqpos(i);
		int const aa = report_aa(i);
		int const aav = report_aav(i);
		for ( int j = 1, je = natoms( aa, aav ); j <= je; ++j ) {
			for ( int k = 1; k <= 3; ++k ) {
				start_rot( k, j ) = get_rotcoord( k, j, i );
			}
		}
		for ( int k = 1; k <= 3; ++k ) {
			start_actrot( k ) = get_rotactcoord( k, i );
		}

		// jk Find the required transform
		get_sym_rotamer_transform(start_rot, curr_pose.full_coord()(1,1,seqpos),Mgl);

		// jk Apply the transform
		if ( ! identity_Mgl( Mgl ) ) {
			transfer_sym_rotamer(aa, aav, Mgl, start_rot, start_actrot,
        get_rotcoord(i), get_rotactcoord(i));
		}

	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin update_rotamer_alignment
///
/// @brief
///    Carry out translation / rotation to adjust starting_rotamer_set onto the pose backbone
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSet::update_rotamer_alignment(
  RotamerSetBase & starting_rotamer_set,
	pose_ns::Pose & curr_pose
) {

	using namespace param;

	std::cout << "Updating rotamer alignment while copying rotamers" << std::endl;

	// jk Transform each rotamer onto the new backbone
	//	int prev_seqpos(0);
	FArray2D_float Mgl( 4, 4 );
	for ( int i = 1; i <= starting_rotamer_set.nrotamers(); ++i ) {
		int const seqpos = starting_rotamer_set.report_seqpos(i);

		assert ( starting_rotamer_set.report_seqpos(i) == report_seqpos(i));
		assert ( starting_rotamer_set.report_aa(i) == report_aa(i));

		// jk Find the required transform
		get_sym_rotamer_transform(starting_rotamer_set.get_rotcoord(i),
			curr_pose.full_coord()(1,1,seqpos),Mgl);

		if ( ! identity_Mgl( Mgl ) ) {
			int const aa = starting_rotamer_set.report_aa(i);
			int const aav = starting_rotamer_set.report_aav(i);
			transfer_sym_rotamer(aa, aav, Mgl,
	      starting_rotamer_set.get_rotcoord(i), starting_rotamer_set.get_rotactcoord(i),
        get_rotcoord(i), get_rotactcoord(i));
		}
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin rot_to_pack_single_sequence
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
std::vector<int>
RotamerSet::rot_to_pack_single_sequence( FArray1DB_int const & sequence ) const
{
	std::vector<int> rot_to_pack;
	for ( int rot(1); rot <= nrotamers(); ++rot ) {

//		std::cout << "rotamer " << rot << " aa " <<
//		 param_aa::aa_name3(report_aa(rot)) << " at pos " <<
//		 pdb::pdb_res_num(report_seqpos(rot));

		if ( report_aa(rot) != sequence( report_seqpos(rot) ) ) continue;
		rot_to_pack.push_back( rot );

//		std::cout << " allowed" << std::endl;
	}
	return rot_to_pack;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin makeSerThrTyrOHConformer
///
/// @brief makes new rotamers for ser, thr, tyr
///
/// @detailed
///        this function runs through the built ser/thr/tyr rotamers and
///        generates new rotamer by varying the hydroxyl hydrogen positions
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors psh, jk
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::makeSerThrTyrOHConformer() {

  using namespace param_aa;
  using namespace aaproperties_pack;
  using namespace param; // for MAXANGLES_PER_CHI

  FArray1D_float chi(MAXANGLES_PER_CHI);
  bool is_extra=false;
  int chiCount = 3;
  //  int aav = 1; // there's no variant for these types here... yet
  int const orig_nrotamers = nrotamers();
  for ( int rotinx = 1; rotinx < orig_nrotamers; ++rotinx ) {
    int const aa = report_aa(rotinx);
    if ( ( aa == aa_ser ) || ( aa == aa_thr ) ) {
      get_ser_thr_hchi(is_extra,chiCount,chi(1));
      for (int nchi = 1; nchi <= 3 ; nchi++) {
				// jk make a duplicate copy of this rotamer. Note: we'll need to sort rotamer_set later!
				push_rotamer(get_rotamer(rotinx));
				change_chi_angle_update_coors(nrotamers(),2,chi(nchi));
      }
    } else if ( aa == aa_tyr ) {
      get_tyr_hchi(is_extra, chiCount, chi);
      for (int nchi = 1; nchi <= 2 ; nchi++) {
				// jk make a duplicate copy of this rotamer. Note: we'll need to sort rotamer_set later!
				push_rotamer(get_rotamer(rotinx));
				change_chi_angle_update_coors(nrotamers(),3,chi(nchi));
      }
    }
  }

  sort();
	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin  buildBDAconformers
///
/// @brief  builds conformers from storage
///
/// @detailed
///					use BDA library to build conformers by fetching the key lookup for
///					index ranges before orienting the residues and load the entire block
///					to rotcoord array.  At this stage rotindex is not properly setup
///
/// @param  map - [in] - design map to tell whether this site is used
/// @param  seqpos - [in] - sequence position
/// @param  aa - [in] - aa
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors psh, jk
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::buildBDAconformers(
  int total_residue,
  const FArray1D_int & res,
	const FArray1D_int & res_variant,
	const FArray3D_float & full_coord,
	const DesignMap & design_map,
	int const seqpos,
	int const aa
) {

  using namespace design::conformer;
  using namespace param_aa;
  using namespace param_pack;
  using namespace param;
  using namespace aaproperties_pack;

  std::string key;
  std::map<std::string, FArray1D_int>::iterator iterator;

  FArray2D_float mat(3,3); // used for align and lineup_bk
  FArray1D_float vec(3); //used for align and lineup_bk

  //std::cout << "coming in with: " << nrotamers() << "seqpos" << seqpos << "aa" << aa << std::endl;
  if ( design_map.get(seqpos,aa) ) {
    //std::cout << "seqpos: " << seqpos << std::endl;
    //std::cout << "aa: " << aa << std::endl;
    key = confLibHandler.conformerKeyFromCoord(seqpos, aa);
    iterator = confLibHandler.rotamerSerialRanges.find(key);
    if (iterator != confLibHandler.rotamerSerialRanges.end()){ // validate key
      //std::cout	<< key << std::endl;
      int fromIndex = confLibHandler.rotamerSerialRanges[key](1);
      int toIndex = confLibHandler.rotamerSerialRanges[key](2);
      //std::cout << "key " << key << " with " << toIndex - fromIndex + 1 << " rotamers at " << seqpos << "with nrotamers " << nrotamers() << std::endl;
      int maxatom;
      for (int i=fromIndex, rotIndex3 =1; i <= toIndex; i++, rotIndex3++){

				// jk Note: push_rotamer updates nrotamers_
				int const aav = confLibHandler.conformer_aav(aa,i);
				push_rotamer( seqpos, aa, aav, rotIndex3 );

				FArray2D_float tempArray(3,MAX_ATOM());
				FArray2Dp_float dummy(confLibHandler.getSliceWithSerialNumber(i,aa));
				int & natoms_aa( natoms(aa,aav) );
				for ( int p = 1, q = 0, r = dummy.index(1,p); p <= natoms_aa; ++p ) {
					for ( int s = 1; s <= 3; ++s, ++q, ++r) {
						tempArray[q] = dummy[r]; // tempArray(i,j) = dummy(i,j);
						//copy from reference to real FArray for transformation
					}
				}
				lineup_bk(tempArray(1,2),tempArray(1,1),full_coord(1,2,seqpos),full_coord(1,1,seqpos),mat,vec);
				maxatom = natoms(aa,aav);
				for (int j=1; j <= maxatom ; j++){
					move_bk(tempArray(1,j),mat,vec);
				}

				if (res(seqpos) == aa_gly) { //if the template at seqpos is gly without cb
					if (aa == aa_gly) { //if the residue to design is also gly
						// align with hydrogen?
						align_bk(tempArray(1,1),tempArray(1,2),tempArray(1,6),full_coord(1,6,seqpos),mat(1,1),vec(1)); // 2HA to 2HA
					}
					else { // other amino acids
						align_bk(tempArray(1,1),tempArray(1,2),tempArray(1,5),full_coord(1,6,seqpos),mat(1,1),vec(1)); // CB to 2HA
					}
				}
				else { // template has cb
					if (aa == aa_gly) { // if the residue to design is gly
						align_bk(tempArray(1,1),tempArray(1,2),tempArray(1,6),full_coord(1,5,seqpos),mat(1,1),vec(1)); // 2HA to CB
					}
					else { // other amino acids
						align_bk(tempArray(1,1),tempArray(1,2),tempArray(1,5),full_coord(1,5,seqpos),mat(1,1),vec(1)); // CB to CB
					}
				}
				for (int j=3; j <= maxatom; j++){
					move_bk(tempArray(1,j),mat,vec);
				}

				// set probability and bump check
				if (aa == aa_gly){ // if designing gly set prob to 1
					set_rperc(nrotamers(),1.0);

				} else { // other amino acids set prob according to bilinear interpolation

					// fill chi angles then compute probability
					fill_chi_from_coors(nrotamers());
					compute_rperc(nrotamers());

					float bumpenergy = -1000;
					//bump check
					bump_check(aa,aav,seqpos,tempArray,full_coord,res,res_variant,
										 design_map,total_residue,bumpenergy);
					//std::cout << "bump:" << bumpenergy <<
					//" itr: " << itr << std::endl;
					if (bumpenergy > max_rot_bumpenergy){
						pop_rotamer();
						continue;
					}
				}

				//finally assign the coord to rotcoord
				// for BDA, use two parts..  p = 5 starts copying from the sidechain
				// but somehow the atom order is mesed up, so start from 1 and
				// overwrite the backbone with full_coord coordinates to fill missing atoms

				int r=0;
				for (int p = 1;p<=natoms_aa;p++){
					for (int s=1;s<=3;s++){
						set_rotcoord(s,p,nrotamers()) = tempArray[r];
						++r;
					}
				} // rotcoord assignment
				// copy the backbone from fullcoord
				r=0;
				for (int p = 1;p<=4;p++){
					for (int s=1;s<=3;s++){
						set_rotcoord(s,p,nrotamers()) = full_coord(s,p,seqpos);
						++r;
					}
				} // rotcoord assignment
				fill_rotactcoord(nrotamers());
      } // from ... to rotamer range
    }
    else { // should be using bbind conformers? or dunbrack?
      std::cout << "key " << key << " does not exist" << std::endl;
    }
  }

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin  buildConformers
///
/// @brief
///
/// @detailed
/// overload buildRotcoord to handle two types of tasks
/// the difference is basically single function call versus being called inside another loop
/// for the one-pass version (to be called inside of a loop), it takes two extra arguments, seqpos and nrotamers
/// nrotamers is returned to the main function for the next iteration
///
/// @param  see buildBDAbuild
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors  psh
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::buildConformers(
	const DesignMap & design_map,
	int const seqpos,
	int const aa
) {

  using namespace design::conformer;
  using namespace misc;
  using namespace aaproperties_pack;
  using namespace param;
  std::string key;
  std::map<std::string, FArray1D_int>::iterator iterator;

  FArray2D_float mat(3,3); // used for align and lineup_bk
  FArray1D_float vec(3); //used for align and lineup_bk

  //std::cout << "coming in with: " << nrotamers() << "seqpos" << seqpos << "aa" << aa << std::endl;
  //for (int aa = 1,lastAA = MAX_AA(); aa <= lastAA ; aa++) {
  if( design_map.get(seqpos,aa) ){
    //std::cout << "map.get?" << std::endl;
    key = confLibHandler.conformerKeyFromCoord(seqpos, aa);
    iterator = confLibHandler.rotamerSerialRanges.find(key);
    if (iterator != confLibHandler.rotamerSerialRanges.end()){ // validate key
      //	std::cout	<< key << std::endl;
      int fromIndex = confLibHandler.rotamerSerialRanges[key](1);
      int toIndex = confLibHandler.rotamerSerialRanges[key](2);
      int maxatom;
      int const aav=1; // shouldn't have to worry about variants
      for (int i=fromIndex, rotIndex3 = 1; i <= toIndex; i++, rotIndex3++){

				// jk Note: push_rotamer updates nrotamers_
				push_rotamer( seqpos, aa, aav, rotIndex3 );
				// could have problems later, but force to use no variant for now

				FArray2D_float tempArray(3,MAX_ATOM());
				FArray2Dp_float dummy(confLibHandler.getSliceWithSerialNumber(i,aa));
				int & natoms_aa( natoms(aa,aav) );
				for ( int p = 1, q = 0, r = dummy.index(1,p); p <= natoms_aa; ++p ) {
					for ( int s = 1; s <= 3; ++s, ++q, ++r) {
						// tempArray(i,j) = dummy(i,j); copy from reference to real FArray to do transformation
						tempArray[q] = dummy[r];
					}
				}

				lineup_bk(tempArray(1,2),tempArray(1,1),full_coord(1,2,seqpos),full_coord(1,1,seqpos),mat,vec);
				maxatom = natoms(aa,aav);
				for (int j=1; j <= maxatom ; j++){
					move_bk(tempArray(1,j),mat,vec);
				}
				align_bk(tempArray(1,1),tempArray(1,2),tempArray(1,3),full_coord(1,3,seqpos),mat(1,1),vec(1));
				for (int j=3; j <= maxatom; j++){
					move_bk(tempArray(1,j),mat,vec);
				}
				//finally assign the coord to rotcoord
				int r(0);
				for (int p = 1;p<=natoms_aa;p++){
					for (int s=1;s<=3;s++){
						set_rotcoord(s,p,nrotamers()) = tempArray[r];
						++r;
					}
				} // rotcoord assignment
				fill_rotactcoord(nrotamers());
      } // from ... to rotamer range

    } else { // should be using bbind conformers? or dunbrack?
      std::cout << "key " << key << " does not exist" << std::endl;
    }

  }

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin bumpCheck
///
/// @brief bump Check rotamers
///
/// @detailed bump check rotamers against backbone
///
/// @global_read  full_coord, total_residue, res_variant
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors psh
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void RotamerSet::conformerBumpFilter( const DesignMap & design_map ){

  using namespace misc;
  using namespace param_pack;

	float bumpenergy;
  std::vector <int> removalList;

  for ( int rotinx = 1; rotinx <= nrotamers(); ++rotinx ){
    bump_check(report_aa(rotinx),report_aav(rotinx),report_seqpos(rotinx),
							 get_rotcoord(rotinx),full_coord,res,res_variant,
							 design_map,total_residue,bumpenergy);
    //std::cout << "bump:" << bumpenergy <<  " rotinx: " << rotinx << std::endl;
    if (bumpenergy > max_rot_bumpenergy){
      removalList.push_back(rotinx);
    }
  }
  remove( removalList );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSet::build_biased_rotamers_by_replacement
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSet::build_biased_rotamers_by_replacement(
		int total_residue,
	  const FArray1D_int & res,
	  const FArray1D_int & res_variant,
    const FArray3D_float & full_coord,
		int const rot_seqpos,
		int const rot_aa,
		const PackerTask & Task
) {

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;

	if ( ( rot_aa == aa_gly ) ||  ( rot_aa == aa_ala ) ) return;
	if ( ! Task.get_designmap().get(rot_seqpos,rot_aa) ) return;

	// First check that there's at least one currently allowed rotamer,
	// collect a list of energies (1-body + interaction with bias residues) for the original rotamers
	// note: (sorted) multimap has key=energy, data=rotindex
	std::multimap<float, int> orig_enemap;
	float worst_orig_rot_ene(-9999.);
	bool found_allowed_rot(false);
	for ( int rotnum = 1; rotnum <= nrotamers(); ++rotnum ) {
		if ( ( rot_seqpos == report_seqpos(rotnum) ) &&
				 ( rot_aa == report_aa(rotnum) ) ) {
			found_allowed_rot=true;
			float const curr_ene = get_rotamer_biasing_ene( rot_seqpos, rot_aa, report_aav(rotnum),
        get_rotcoord(rotnum), get_rotactcoord(rotnum), rotnum, Task );
			orig_enemap.insert(std::pair<float, int>(curr_ene, rotnum));
			if ( curr_ene > worst_orig_rot_ene ) worst_orig_rot_ene = curr_ene;
		}
	}
	//	std::cout << "Found " << orig_enemap.size() << " rotamers eligible for replacement at seqpos " <<
	//		rot_seqpos << " for aa " << rot_aa << std::endl;
	if ( ! found_allowed_rot ) return;

	// Next, build an exploded RotamerSet for the requested seqpos + aa
	bool const current_chi_already_included(false);
	int const rot_aav(1);
	int nrotaa(0);

	bool const do_explode(true);

	RotamerOptions saved_rotamer_options = design::active_rotamer_options;
	if ( do_explode ) {
		design::active_rotamer_options.extrachi_flag=exchi_flags::EX_TWO_HALF_STEP_STDDEVS;
		design::active_rotamer_options.rot_explode=true;
		design::active_rotamer_options.rot_explode_accept_all=true;
	}

	RotamerSet explosion_set;
	explosion_set.get_rotamers_seqpos_aa_aav(total_residue, res, res_variant, full_coord, rot_aa, rot_aav, rot_seqpos,
		nrotaa, current_chi_already_included, "design", Task);

	if ( do_explode ) {
		design::active_rotamer_options = saved_rotamer_options;
	}

	if ( explosion_set.nrotamers() == 0 ) return;

	// Next, determine the 1-body + interaction with bias residues for each exploded rotamer
	// note: (sorted) multimap has key=energy, data=rotindex
	std::multimap<float, int> explosion_enemap;
	float best_new_rot_ene(9999.);
	for ( int rotnum = 1; rotnum <= explosion_set.nrotamers(); ++rotnum ) {
		float const curr_ene = explosion_set.get_rotamer_biasing_ene(rot_seqpos, rot_aa, rot_aav,
			explosion_set.get_rotcoord(rotnum), explosion_set.get_rotactcoord(rotnum),
			rotnum, Task );
		// save only exploded rotamers which are better than the worst original rotamer
		if ( curr_ene < worst_orig_rot_ene ) {
			explosion_enemap.insert(std::pair<float, int>(curr_ene, rotnum));
		}
		if ( curr_ene < best_new_rot_ene ) best_new_rot_ene = curr_ene;
	}
	if ( best_new_rot_ene >= worst_orig_rot_ene ) {
		return;
	}

	// to carry out swap, exploded rotamer must be better than the original rotamer by at least improvement_thres
	//	float const improvement_thres = 1.;
	float const improvement_thres = 0.01;
	int num_replacements(0);
	float const best_orig_ene = (*(orig_enemap.begin())).first;
	while ( ( ! orig_enemap.empty() ) && ( ! explosion_enemap.empty() ) ) {
		// try replacing the worst original rotamer with the best exploded rotamer
		// note: multimap sorting puts the lowest key at the front of the list
		float const explosion_ene = (*(explosion_enemap.begin())).first;
		// get the last element by using a reverse iterator (rbegin)
		float const orig_ene = (*(orig_enemap.rbegin())).first;

		if ( explosion_ene < ( orig_ene - improvement_thres ) ) {

			int const explosion_rot_inx = (*(explosion_enemap.begin())).second;
			int const orig_rot_inx = (*(orig_enemap.rbegin())).second;

			for (int j = 1; j<= MAX_ATOM(); j++){
				for (int k = 1; k<= 3; k++){
					set_rotcoord(k,j,orig_rot_inx) = explosion_set.get_rotcoord(k,j,explosion_rot_inx);
				}
			}

			fill_rotactcoord(orig_rot_inx);
			fill_chi_from_coors(orig_rot_inx);
			compute_rperc(orig_rot_inx);
			find_suitable_base_rotamer(orig_rot_inx);

			// remove these entries from the enemaps
			explosion_enemap.erase(explosion_enemap.begin());
			orig_enemap.erase(--(orig_enemap.end()));

			// increment the counter
			++num_replacements;

		} else {
			// we're done making replacements
			orig_enemap.clear();
			explosion_enemap.clear();
		}

	}

	// If the explosion_enemap is not empty, this means that we didn't jump out because we wish to keep the
	// remaining original rotamers: rather, we still have more exploded rotamers better than the best original
	// rotamer. Append at most "max_appended_rotamers" of these to the RotamerSet, then resort
	int const max_appended_rotamers = 20;
	int num_rotamers_appended(0);
	while ( ! explosion_enemap.empty() ) {
		float const explosion_ene = (*(explosion_enemap.begin())).first;
		if ( ( num_rotamers_appended < max_appended_rotamers ) &&
				 ( explosion_ene < ( best_orig_ene - improvement_thres ) ) ) {
			int const explosion_rot_inx = (*(explosion_enemap.begin())).second;
			append_rotamer( explosion_set,explosion_rot_inx);
			explosion_enemap.erase(explosion_enemap.begin());
			++num_rotamers_appended;
		} else {
			explosion_enemap.clear();
		}
	}

	std::cout << "Made " << num_replacements << " rotamer replacements and appended " <<
		num_rotamers_appended << " rotamers at seqpos " << rot_seqpos << " for aa " << rot_aa << std::endl;

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSet::build_biased_rotamers_by_appending
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void RotamerSet::build_biased_rotamers_by_appending(
    int total_residue,
	  const FArray1D_int & res,
	  const FArray1D_int & res_variant,
    const FArray3D_float & full_coord,
		int const rot_seqpos,
		int const rot_aa,
		const PackerTask & Task
) {

	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;

	if ( ( rot_aa == aa_gly ) ||  ( rot_aa == aa_ala ) ) return;
	if ( ! Task.get_designmap().get(rot_seqpos,rot_aa) ) return;

	// First check that there's at least one currently allowed rotamer
	int orig_allowed_rot(0);
	int num_rotamers_before_this_res(0);
	for ( int rotnum = 1; rotnum <= nrotamers(); ++rotnum ) {
		if ( ( rot_seqpos == report_seqpos(rotnum) ) && ( rot_aa == report_aa(rotnum) ) ) {
			if(orig_allowed_rot == 0) { num_rotamers_before_this_res = rotnum;}
			++orig_allowed_rot;
		}
	}
	if ( orig_allowed_rot == 0 ) return;

	// Next, build an exploded RotamerSet for the requested seqpos + aa
	bool const current_chi_already_included(false);
	int const rot_aav(1);
	int nrotaa(0);
	RotamerSet explosion_set;
	std::multimap<float, int> explosion_enemap;

	if(design::active_rotamer_options.do_rotamer_bias){
		bool const do_explode(true);
		RotamerOptions saved_rotamer_options = design::active_rotamer_options;
		if ( do_explode ) {
			design::active_rotamer_options.extrachi_flag=exchi_flags::EX_THREE_THIRD_STEP_STDDEVS;
			design::active_rotamer_options.rot_explode=true;
			design::active_rotamer_options.rot_explode_accept_all=true;
			design::active_rotamer_options.do_rotamer_bias=false;
		}

		explosion_set.get_rotamers_seqpos_aa_aav(total_residue, res, res_variant, full_coord, rot_aa, rot_aav, rot_seqpos,
		nrotaa, current_chi_already_included, "design", Task );

		if ( do_explode ) {
			design::active_rotamer_options = saved_rotamer_options;
		}

		if ( explosion_set.nrotamers() == 0 ) return;

		// Next, determine the 1-body + interaction with bias residues for each exploded rotamer
		// note: (sorted) multimap has key=energy, data=rotindex
		for ( int rotnum = 1; rotnum <= explosion_set.nrotamers(); ++rotnum ) {
			float const curr_ene = explosion_set.get_rotamer_biasing_ene(rot_seqpos, rot_aa, rot_aav,
																																	 explosion_set.get_rotcoord(rotnum), explosion_set.get_rotactcoord(rotnum),
																																	 rotnum, Task);
			explosion_enemap.insert(std::pair<float, int>(curr_ene, rotnum));
		}
	}

	if (design::active_rotamer_options.minimize_best_rotamers) {

		screen_rotamer_set_for_ligand_contacts(full_coord,*this,rotamer_explosion::rotamers_to_minimize,num_rotamers_before_this_res);

		if(!design::active_rotamer_options.rot_opt) {
			for(std::map<int,bool>::iterator CurRot ( rotamer_explosion::rotamers_to_minimize.begin() );
					CurRot != rotamer_explosion::rotamers_to_minimize.end(); CurRot++) {
					if( (*CurRot).first > num_rotamers_before_this_res ) {   //energetically good rotamers are duplicated, and will later be minimized
						append_rotamer(*this,(*CurRot).first);
					}
				}
		}
	}

		// Append at most "max_appended_rotamers" of these to the RotamerSet, then resort
	int const max_appended_rotamers = orig_allowed_rot;
	int num_rotamers_appended(0);
	while ( ! explosion_enemap.empty() ) {
		if ( num_rotamers_appended < max_appended_rotamers ) {
			int const explosion_rot_inx = (*(explosion_enemap.begin())).second;
			append_rotamer( explosion_set,explosion_rot_inx);
			explosion_enemap.erase(explosion_enemap.begin());
			++num_rotamers_appended;
		} else {
			explosion_enemap.clear();
		}
	}

	if( !design::active_rotamer_options.minimize_best_rotamers ) {
		std::cout << "Appended " << num_rotamers_appended << " rotamers at seqpos " << rot_seqpos << " for aa " << rot_aa << std::endl;
	}
	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerSet::get_rotamer_biasing_ene
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float RotamerSet::get_rotamer_biasing_ene(
	int const rot_seqpos,
	int const rot_aa,
	int const rot_aav,
	FArray2Da_float rotcoord,
	FArray1Da_float rotactcoord,
	int const rotinx,
	const PackerTask & Task
) {

	using namespace design;
	using namespace misc;
	using namespace template_pack;

	if ( param_pack::gen_born ) {
		std::cerr << "Gen Born not yet supported with rotamer_biasing..." << std::endl;
		std::cerr << "  Aborting..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	FArray1D_float energy1b_base( nrotamers(), 0. );
	float ene = get_1body_energy( *this, rotinx, rot_seqpos,
		rot_aa, rot_aav, rotcoord, energy1b_base, Task );

  for (std::map<int, float>::iterator it = active_rotamer_options.extra_bias_weights.begin();
			 it != active_rotamer_options.extra_bias_weights.end(); ++it) {
		// JK NOTE: GEN_BORN IS NOT INCLUDED HERE
		float const reweighting = it->second - 1.;
		if ( std::abs(reweighting) > 0.0001 ) {
			int const seqpos = it->first;
			int const aa = res(seqpos);
			put_wcentroid(full_coord(1,1,seqpos),actcoord,aa);
			ene += reweighting * get_base_2body_energy( seqpos, aa, res_variant(seqpos),
				full_coord(1,1,seqpos), actcoord(1,seqpos), rot_seqpos, rot_aa, rot_aav,
				rotcoord, rotactcoord, Task );
		}
	}

	return ene;

}
