// -*- Mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.13 $
//  $Date: 2007/03/26 13:02:54 $
//  $Author: plato $

#ifndef SOLVENT_ACCESSIBLE_SURFACE_AREA_INTERACTION_GRAPH_H
#define SOLVENT_ACCESSIBLE_SURFACE_AREA_INTERACTION_GRAPH_H

//Rosetta Headers
#include "AdditionalBackgroundNodesInteractionGraph.h"
#include "PrecomputedInteractionGraph.h"
#include "SparseMatrixIndex.h"
#include "AminoAcidNeighborSparseMatrix.h"
#include "RotamerDots.h"
#include "SASAGraph.h"

//Utility Headers
#include <utility/vector1.hh>

//ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/ubyte.hh>

//STL Headers
#include <ctime>
#include <vector>
#include <iosfwd>

namespace pack
{

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G > class SASANode;
template < typename V, typename E, typename G > class SASABackgroundResidueNode;
template < typename V, typename E, typename G > class SASAEdge;
template < typename V, typename E, typename G > class SASABackgroundEdge;
template < typename V, typename E, typename G > class SASAInteractionGraph;

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////

template < typename V, typename E, typename G >
class SASANode : public FirstClassNode< V, E, G >
{
public:
	typedef  FirstClassNode< V, E, G >  parent;
public:
	SASANode( G * owner, int node_index, int num_states );
	virtual ~SASANode();

	virtual void assign_zero_state();
	virtual bool state_unassigned() const { return parent::get_current_state() == 0;}
	virtual float get_totalE() const { return parent::get_totalE();}

	void assign_state_SASA( int state );

	float
	get_curr_state_SASA_energy() const;

	float
	project_deltaE_for_substitution_SASA(
		int alternate_state,
		float & prev_energy_for_node,
		float deltaE_thresh_for_avoiding_SASA_calcs
	);

	float
	get_SASA_deltaE_for_neighbors_state_substitution
	(
		RotamerDots & neighbors_alternate_state,
		RotamerDotsCache const & neighbors_curr_state_overlap_with_this,
		RotamerDotsCache & this_overlap_with_neighbors_alternate,
		RotamerDotsCache & neighbors_alternate_overlap_with_this,
		FArray3DB< short > & alt_atom_overlap_mask_inds,
		FArray2DB< ubyte > & mask_this_covered_by_other_small,
		FArray2DB< ubyte > & mask_other_covered_by_this_small,
		FArray2DB< ubyte > & mask_this_covered_by_other_large,
		FArray2DB< ubyte > & mask_other_covered_by_this_large,
		int num_atoms_same
	);

	float
	commit_considered_substitution_SASA();

	void
	acknowledge_neighbors_substitution_SASA();

	//SASA Related Methods
	virtual void setRotamerCoordsForState
	(
		int state,
		RotamerCoords const & coords
	);

	bool
	detect_any_overlap_with_rotamer
	(
		RotamerDots const & rotamer
	) const;

	void initialize_overlap_with_background
	(
		RotamerDots const & bg_rotamer,
		std::vector< RotamerDotsCache * > &  overlap_with_bg_cache
	);

	virtual
	void initialize_overlap_with_ligand( Ligand & ligand);

	//Virtual methods from NodeBase class
	virtual void print() const;
	virtual void prepare_for_simulated_annealing();
	virtual unsigned int getMemoryUsageInBytes() const;
	//end NodeBase methods

	bool overlaps( SASANode< V, E, G > * neighbor );

	static void print_SASA_avoidance_stats();
	static void reset_SASA_avoidance_stats();

	void write_dot_kinemage( std::ofstream & output_kin );

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	inline
	SASAEdge< V, E, G > * get_incident_sasa_edge( int index ) const
	{	return (SASAEdge< V, E, G > *) parent::get_incident_edge( index ); }

	inline
	SASABackgroundEdge< V, E, G > * get_edge_to_sasa_bg_node( int index ) const
	{	return (SASABackgroundEdge< V, E, G > *) parent::get_edge_to_bg_node( index ); }

	inline
	SASAInteractionGraph< V, E, G > * get_sasa_owner() const
	{	return (SASAInteractionGraph< V, E, G > *) parent::get_owner();}

	float get_SASA_score_difference() const;

private:

	void initialize_self_overlap();
	float project_SASA_deltaE();

	bool decide_procrastinate_SASA_computations(
		float const pd_deltaE,
		float const threshold
	) const;

	void track_SASA_E_min();

	bool calculated_SASA_deltaE_;
	float deltaE_for_substitution_;
	//float SASA_deltaE_for_sub_;
	//mutable float predicted_max_mag_SASA_deltaE_for_sub_;
	//mutable bool suggest_procrastinating_SASA_computations_;

	bool have_prepared_for_simA_;
	float SASA_score_min_last_100_;
	float SASA_score_min_recent_;
	int num_substitutions_since_SASA_min_update_;
	bool observed_sufficient_SASA_E_to_predict_min_;

	utility::vector1< RotamerDots > self_and_bg_dots_for_states_;

	RotamerDots curr_state_dots_;
	RotamerDots alt_state_dots_;

	static int num_state_substitutions_considered_;
	static int num_SASA_comps_procrastinated_;
	static int num_SASA_comps_later_made_;

	//no default constructor, uncopyable
	SASANode();
	SASANode( SASANode< V, E, G > const & );
	SASANode< V, E, G > & operator = ( SASANode< V, E, G > const & );
};

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////

template < typename V, typename E, typename G >
class SASABackgroundResidueNode : public BackgroundNode< V, E, G >
{
public:
	typedef BackgroundNode< V, E, G > parent;

public:
	SASABackgroundResidueNode(
		AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner,
		int node_index
	);

	virtual ~SASABackgroundResidueNode();

	void
	set_rotamer_coords( RotamerCoords const & input_coords );

	bool
	detect_overlap( SASANode< V, E, G >* node ) const;

	void initialize_bg_bg_overlap(
		SASABackgroundResidueNode< V, E, G > & other
	);
	void initialize_overlap_with_ligand( Ligand & ligand);

	float
	project_SASA_deltaE_for_substitution(
		RotamerDotsCache const & nodes_curr_overlap_with_bg_res,
		RotamerDotsCache const & nodes_alt_overlap_with_bg_res
	);

	void
	acknowledge_substitution_SASA();

	float get_SASA_score() const;

	virtual void prepare_for_simulated_annealing();
	virtual void print() const;
	void write_dot_kinemage( std::ofstream & output_kin );

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	inline
	SASABackgroundEdge< V, E, G > * get_sasa_bg_edge( int index )
	{	return (SASABackgroundEdge< V, E, G > *) parent::get_incident_edge( index );}

private:
	void initialize_self_overlap();

	bool have_preped_for_simA_;

	RotamerDots current_state_;
	RotamerDots alternate_state_;

	//no default constructor, uncopyable
	SASABackgroundResidueNode();
	SASABackgroundResidueNode( SASABackgroundResidueNode< V, E, G > const & );
	SASABackgroundResidueNode< V, E, G > & operator =
		( SASABackgroundResidueNode< V, E, G > const & );

};

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////

template < typename V, typename E, typename G >
class  SASAEdge : public FirstClassEdge< V, E, G >
{
public:
	typedef  FirstClassEdge< V, E, G >  parent;

public:
	SASAEdge( G * owner, int node1, int node2 );
	virtual ~SASAEdge();

	void
	acknowledge_state_zeroed_SASA( int node_index );

	float
	get_SASA_deltaE_for_neighbor
	(
		int node_considering_substitution,
		int alt_state,
		RotamerDots & alt_state_dots,
		int num_atoms_same_as_current
	);

	float get_currernt_two_body_energy() const;

	void acknowledge_substitution_SASA();

	//Virtual methods from EdgeBase
	virtual void declare_energies_final();
	virtual void prepare_for_simulated_annealing();
	virtual unsigned int getMemoryUsageInBytes() const;
	//end methods from EdgeBase

	float get_max_sasa_deltaE_guess( int node_changing ) const;

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	inline
	SASANode< V, E, G > * get_sasa_node( int index )
	{	return (SASANode< V, E, G > *) E::get_node( index ); }

private:
	inline
	void inform_non_changing_node_of_neighbors_change();

	void prepare_masks(
		FArray2DB< ubyte > & mask_unchanging_covered_by_changing_small,
		FArray2DB< ubyte > & mask_unchanging_covered_by_changing_large,
		FArray2DB< ubyte > & mask_changing_covered_by_unchanging_small,
		FArray2DB< ubyte > & mask_changing_covered_by_unchanging_large
	);

	void
	copy_dot_masks_indices_from_alt_to_curr();

	static
	inline
	void
	or_dot_overlap_masks(
		FArray1DB< ubyte > & collected_masks,
		int mask_index
	);

	int node_changing_;
	int node_not_changing_;
	int nodes_curr_states_[2];
	int nodes_alt_states_[2];

	RotamerDotsCache nodes_curr_pair_dot_counts_[2];
	RotamerDotsCache nodes_alt_pair_dot_counts_[2];

	int num_atoms_curr_[2];
	int num_atoms_same_as_current_;
	int changing_num_atoms_;
	FArray3D< short > curr_atom_pair_overlap_mask_indices_;
	FArray3D< short > alt_atom_pair_overlap_mask_indices_;

	float max_sasa_deltaE_last_50_commits_[2];
	float max_sasa_deltaE_recent_50_commits_[2];
	float magnitude_last_sasa_deltaE_[2];
	int num_sasa_deltaE_observations_since_update_[2];

	void track_max_magnitude_SASA_deltaE();

	//no default constructor, uncopyable
	SASAEdge();
	SASAEdge( SASAEdge< V, E, G > const & );
	SASAEdge< V, E, G > & operator = ( SASAEdge< V, E, G > const & );
};

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////

template < typename V, typename E, typename G >
class SASABackgroundEdge : public BackgroundToFirstClassEdge< V, E, G >
{
public:
	typedef  BackgroundToFirstClassEdge< V, E, G >  parent;

public:
	SASABackgroundEdge(
		AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner,
		int first_class_node_index,
		int background_node_index
	);
	virtual ~SASABackgroundEdge();

	float
	get_SASA_deltaE_for_substitution( int alt_state );

	void
	acknowledge_substitution_SASA();

	void
	acknowledge_state_change( int new_state );

	void
	initialize_overlap_cache( RotamerDots const & bg_residue );

	void
	prepare_for_simulated_annealing();

	float
	get_max_sasa_deltaE_guess() const;

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	inline
	SASANode< V, E, G > * get_sasa_node() const
	{	return (SASANode< V, E, G > *) parent::get_first_class_node();}

	inline
	SASABackgroundResidueNode< V, E, G > * get_sasa_bg_node() const
	{	return (SASABackgroundResidueNode< V, E, G > *) parent::get_background_node();}

private:
	int bg_res_num_atoms_;

	int nodes_curr_state_;
	int nodes_alt_state_;

	RotamerDotsCache curr_dots_;
	RotamerDotsCache alt_dots_;

	//uses index 0; do not change to utility::vector1
	//std::vector< RotamerDotsCache* > node_states_coverage_of_bg_res_;
	std::vector< FArray3D_ubyte > node_states_coverage_of_bg_res_;
	std::vector< FArray1D_bool > node_states_overlap_bg_atoms_;

	float max_sasa_deltaE_last_50_commits_;
	float max_sasa_deltaE_recent_50_commits_;
	float magnitude_last_sasa_deltaE_;
	int num_sasa_deltaE_observations_since_update_;

	void track_max_magnitude_SASA_deltaE();

	//no default constructor, uncopyable
	SASABackgroundEdge();
	SASABackgroundEdge( SASABackgroundEdge< V, E, G > const & );
	SASABackgroundEdge< V, E, G > & operator = ( SASABackgroundEdge< V, E, G > const & );
};

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
class SASAInteractionGraph : public AdditionalBackgroundNodesInteractionGraph< V, E, G >,
	public SASAGraph
{
public:
	typedef  AdditionalBackgroundNodesInteractionGraph< V, E, G >  parent;

public:
	SASAInteractionGraph( int num_nodes );
	virtual ~SASAInteractionGraph();

	//Virtual public methods from InteractionGraphBase
	//virtual float get_one_body_energy_for_node_state( int node, int state);
	virtual void prepare_for_simulated_annealing();
   virtual void  blanket_assign_state_0();
   virtual float set_state_for_node(int node_ind, int new_state);
	virtual float set_network_state( FArray1DB_int & node_states);
   virtual void consider_substitution
	(
   	int node_ind,
   	int new_state,
		float & delta_energy,
		float & prev_energy_for_node
	);
   virtual float commit_considered_substitution();
	virtual float get_energy_current_state_assignment();
	virtual int get_edge_memory_usage() const;

	virtual void print_current_state_assignment() const;
	virtual void set_errorfull_deltaE_threshold( float deltaE );
	virtual float get_energy_sum_for_vertex_group( int group_id );
	//End public methods from InteractionGraphBase

	//Other Methods
	void setRotamerCoordsForNodeState
	(
		int node_index,
		int state,
		RotamerCoords const & coords
	);

	void setNumResiduesInProtein( int num_res );
	void setNumBackgroundResidues( int num_background_residues );
	void setResidueAsBackgroundResidue( int residue );
	void setBackgroundResidueRotamerCoords( int residue, RotamerCoords const & coords);
	void initialize_overlap_with_ligand( Ligand & ligand);

	void write_dot_kinemage( std::ofstream & output_kin );
	void print_internal_energies_for_current_state_assignment();

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	//virtual unsigned int getMemoryUsageInBytes() const;

	//Factory Methods:
	//From InteractionGraphBase
	virtual NodeBase*
	create_new_node( int node_index, int num_states);

	virtual EdgeBase*
	create_new_edge( int index1, int index2);
	//End InteractionGraphBase factory methods

	//From AdditionalBackgroundNodesInteractionGraph
	virtual BackgroundNode< V, E, G > *
	create_background_node( int node_index );

	virtual BackgroundToFirstClassEdge< V, E, G > *
	create_background_edge( int fc_node_index, int bg_node_index);
	//End AdditionalBackgroundNodesInteractionGraph factory methods

	inline
	SASANode< V, E, G > * get_sasa_node( int index ) const
	{	return (SASANode< V, E, G > *) G::get_node( index );}

	inline
	SASABackgroundResidueNode< V, E, G > * get_sasa_bg_node( int index ) const
	{	return (SASABackgroundResidueNode< V, E, G > *) parent::get_background_node( index );}

	void
	update_internal_energy_totals_SASA();

private:
	void detect_background_residue_and_first_class_residue_overlap();
	void initialize_bg_bg_overlaps();

	int num_total_residues_;
	int num_residues_assigned_as_background_;
	utility::vector1< int > resid_2_bgenumeration_;
	utility::vector1< int > bgenumeration_2_resid_;

	int num_commits_since_last_update_;
	float total_energy_current_state_assignment_;
	float total_energy_alternate_state_assignment_;
	int node_considering_alt_state_;
	float deltaE_thresh_for_avoiding_SASA_calcs_;
	bool prepared_for_simulated_annealing_;

	static const int COMMIT_LIMIT_BETWEEN_UPDATES = 1024; // 2^10

	//no default constructor, uncopyable
	SASAInteractionGraph();
	SASAInteractionGraph( SASAInteractionGraph< V, E, G > const & );
	SASAInteractionGraph< V, E, G > & operator = ( SASAInteractionGraph< V, E, G > const & );

};

// Begin implementation: previously in SASAInteractionGraph.cc


//----------------------------------------------------------------------------//
//--------------------------- SASA-Pack Node Class ---------------------------//
//----------------------------------------------------------------------------//

template < typename V, typename E, typename G >
int SASANode< V, E, G >::num_state_substitutions_considered_( 0 );

template < typename V, typename E, typename G >
int SASANode< V, E, G >::num_SASA_comps_procrastinated_( 0 );

template < typename V, typename E, typename G >
int SASANode< V, E, G >::num_SASA_comps_later_made_( 0 );

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode< V, E, G >::SASANode
///
/// @brief
/// SASANode constructor
///
/// @detailed
///
/// @param
/// owner - [in] - the owning interaction graph
/// node_id - [in] - the index for this node amongst its owners set
/// num_states - [in] - the number of states for this node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASANode< V, E, G >::SASANode(
	G * owner,
	int node_index,
	int num_states
) :
	FirstClassNode< V, E, G > ( owner, node_index, num_states ),
	calculated_SASA_deltaE_( false ),
	deltaE_for_substitution_( 0.0f ),
	have_prepared_for_simA_( false ),
	SASA_score_min_last_100_( 0 ),
	SASA_score_min_recent_( 0 ),
	num_substitutions_since_SASA_min_update_( 0 ),
	observed_sufficient_SASA_E_to_predict_min_( false ),
	self_and_bg_dots_for_states_( parent::get_num_states() ),
	curr_state_dots_(),
	alt_state_dots_()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode< V, E, G >::~SASANode
///
/// @brief
/// destructor -- no dynamically allocated data, does nothing
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASANode< V, E, G >::~SASANode() {}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::assign_zero_state
///
/// @brief
/// Assign the node to state 0 -- the "unassigned" state.
///
/// @detailed
/// A node in state 0 produces no SASApack score.  Its neighbors have to adjust
/// their scores appropriately.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::assign_zero_state()
{
	parent::assign_zero_state();
	for ( int ii = 1; ii <= parent::get_num_incident_edges(); ++ii )
	{
		get_incident_sasa_edge( ii )->
			acknowledge_state_zeroed_SASA( parent::get_node_index() );
	}

	for (int ii = 1; ii <= parent::get_num_edges_to_background_nodes(); ++ii )
	{
		get_edge_to_sasa_bg_node( ii )->acknowledge_state_change( 0 );
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::assign_state
///
/// @brief
/// assigns the node to one of the states in its state space or to the
/// unassigned state.
///
/// @detailed
/// piggy backs on the standard project_deltaE/commit_considered_substitution
/// protocol.  Slightly inefficient; however, this code is executed once per
/// simulated annealing run.  Not worth optimizing.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::assign_state_SASA( int state )
{
	if ( state == 0 )
	{
		assign_zero_state();
		return;
	}

	float temp(0.0f);
	project_deltaE_for_substitution_SASA( state, temp, -1.0f );
	commit_considered_substitution_SASA();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::get_curr_state_SASA_energy()
///
/// @brief
/// returns the SASA pack energy for the node in its current state assignment
///
/// @detailed
/// state 0 is handled distinctly; there's no good way to make a RotamerDots
/// object think of itself as "unassigned", so the SASANode carries this check
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASANode< V, E, G >::get_curr_state_SASA_energy() const
{
	if ( parent::get_current_state() == 0)
	{
		return 0;
	}

	return curr_state_dots_.get_score();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::project_deltaE_for_substitution_SASA
///
/// @brief
/// returns the (possibly approximate) change in energy induced by changing
/// a node from its current state into some alternate state.
///
/// @detailed
/// Iterates across the edges first to determine the sum of the pairwise
/// decomposable (PD) energy.  If the PD difference implies a collision, then
/// the SASANode pretends as if the state substitution causes the best
/// improvement possible in SASApack score for it and its neighbors, returns
/// the PD difference + pretend SASApack difference.  It will procrastinate
/// computing the actual SASApack score difference until the guiding SimAnnealer
/// decides to commit the substiution.  If the SimAnnealer rejects the
/// substitution, then the work to compute the SASApack score is never done.
/// If it is unclear that the SimAnnealer will reject the substitution
/// based on the PD difference, then the vertex computes the change in
/// SASApack score that it induces on its neighbors.
///
/// @param
/// alternate_state - [in] - the alternate state that the node must consider
/// prev_PDenergies_for_node - [out] - the sum of the PD energies (one-
///   and two-body) for the node in its current state
/// deltaE_thresh_for_avoiding_SASA_calcs - [in] - the minimum deltaE
///   for which the SimAnnealer will accept slopyness in the deltaE projections
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASANode< V, E, G >::project_deltaE_for_substitution_SASA
(
	int alternate_state,
	float & prev_PDenergies_for_node,
	float deltaE_thresh_for_avoiding_SASA_calcs
)
{
	assert( alternate_state > 0  && alternate_state <= parent::get_num_states() );

	++num_state_substitutions_considered_;
	calculated_SASA_deltaE_ = false;

	prev_PDenergies_for_node = parent::get_curr_pd_energy_total();

	// have base class perform deltaE computations for PD portion of energy function
	//std::cout << "calling parent::calc_deltaEpd" << std::endl;
	parent::calc_deltaEpd( alternate_state );

	deltaE_for_substitution_ = parent::get_alt_pd_energy_total() - parent::get_curr_pd_energy_total();

	if ( decide_procrastinate_SASA_computations(
		deltaE_for_substitution_,
		deltaE_thresh_for_avoiding_SASA_calcs) )
	{
		//std::cerr << "procrastinated" << std::endl;
		++num_SASA_comps_procrastinated_;
	}
	else
	{
		//std::cout << "( pd, " << deltaE_for_substitution_ << ") ";

		deltaE_for_substitution_ += project_SASA_deltaE();

		//std::cout << std::endl;
	}

	return deltaE_for_substitution_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode< V, E, G >::project_SASA_deltaE()
///
/// @brief
/// returns the change in SASApack score for this node and all of its neighbors
/// produced by switching from its current state to an alternate state.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float SASANode< V, E, G >::project_SASA_deltaE()
{
	calculated_SASA_deltaE_ = true;
	alt_state_dots_ = self_and_bg_dots_for_states_[ parent::get_alternate_state() ];
	int num_atoms_same = alt_state_dots_.get_num_shared_atoms( curr_state_dots_ );
	//int num_atoms_same = 0;
	float SASA_deltaE = 0;
	for ( int ii = 1; ii <= parent::get_num_incident_edges(); ++ii )
	{
		SASA_deltaE += get_incident_sasa_edge(ii)->
			get_SASA_deltaE_for_neighbor( parent::get_node_index(),
				parent::get_alternate_state(),
				alt_state_dots_,
				num_atoms_same );
	}

	for ( int ii = 1; ii <= parent::get_num_edges_to_background_nodes(); ++ii )
	{
		SASA_deltaE += get_edge_to_sasa_bg_node( ii )->
			get_SASA_deltaE_for_substitution( parent::get_alternate_state() );
	}

	//std::cout << "( self, " << get_SASA_score_difference() << ", "<< alt_state_dots_.get_score() << " ) ";

	SASA_deltaE += get_SASA_score_difference();
	return SASA_deltaE;
}

template < typename V, typename E, typename G >
bool
SASANode< V, E, G >::decide_procrastinate_SASA_computations(
	float const pd_deltaE,
	float const threshold
) const
{

	float sasa_deltaE_max = 0;
	if ( ! observed_sufficient_SASA_E_to_predict_min_ ) return false;

	//std::cerr << "(t: " << threshold << ",d: " << pd_deltaE << ")";
	if (threshold < 0 || pd_deltaE < 0 ) return false;

	//std::cout << "FC Edges: ";
	for( int ii = 1; ii <= parent::get_num_incident_edges(); ++ii)
	{
		float mag_deltaE = get_incident_sasa_edge(ii)->
			get_max_sasa_deltaE_guess( parent::get_node_index() );
		//std::cout << "(ii: " << ii << ", " << mag_deltaE << ")";
		if ( mag_deltaE < 0.0f ) {/*std::cout << std::endl;*/ return false;}
		sasa_deltaE_max += mag_deltaE;
	}
	//std::cout << "; BG Edges: ";
	for ( int ii = 1; ii <= parent::get_num_edges_to_background_nodes(); ++ii )
	{
		float mag_deltaE = get_edge_to_sasa_bg_node( ii )->
			get_max_sasa_deltaE_guess();
		//std::cout << "(ii: " << ii << ", " << mag_deltaE << ")";
		if ( mag_deltaE < 0.0f ) {/*std::cout << std::endl;*/ return false;}
		sasa_deltaE_max += mag_deltaE;
	}
	//std::cout << std::endl << "total: " << sasa_deltaE_max << std::endl;

	sasa_deltaE_max += curr_state_dots_.get_score() - SASA_score_min_last_100_;

	//predicted_max_mag_SASA_deltaE_for_sub_ = sasa_deltaE_max;

	//std::cerr << "considering procrastination: " << pd_deltaE << " " << sasa_deltaE_max << " " << pd_deltaE - sasa_deltaE_max << " with threshold: " << threshold << std::endl;
	//suggest_procrastinating_SASA_computations_ = pd_deltaE - sasa_deltaE_max > threshold;
	//return suggest_procrastinating_SASA_computations_;

	return pd_deltaE - sasa_deltaE_max > threshold;
}

template < typename V, typename E, typename G >
void
SASANode< V, E, G >::track_SASA_E_min()
{
	++num_substitutions_since_SASA_min_update_;
	float alt_sasaE = alt_state_dots_.get_score();
	if ( SASA_score_min_recent_ > alt_sasaE ) SASA_score_min_recent_ = alt_sasaE;
	if ( num_substitutions_since_SASA_min_update_ == 100 )
	{
		SASA_score_min_last_100_ = SASA_score_min_recent_;
		float curr_sasaE = curr_state_dots_.get_score();
		if (curr_sasaE < SASA_score_min_last_100_ ) {SASA_score_min_last_100_ = curr_sasaE;}
		observed_sufficient_SASA_E_to_predict_min_ = true;
		num_substitutions_since_SASA_min_update_ = 0;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode< V, E, G >::get_SASA_deltaE_for_neighbors_state_substitution
///
/// @brief
/// returns the change in SASApack score for this node induced by a state
/// substitution at a neighboring node.  The node increments the dot covereage
/// count for the RotamerDots object representing the alternate state
/// dots for that neighbor.
///
/// @detailed
/// The procedure is simple at heart -- the caching makes it complex.
///
/// alt_state_dots_ = curr_state_dots_; //copy dot coverage counts
/// alt_state_dots_.decrement( neighbors_curr_state_overlap_with_this );
/// alt_state_dots_.increment_both( neighbors_alternate_state );
/// return ( alt_state_dots_.get_score() - curr_state_dots_.get_score() );
///
/// Extensive caching techniques save time.  Each SASAEdge stores the dot
/// coverage for each pair of SASANodes in their current state.  These are
/// stored in the RotamerDotsCache objects.
/// Let's name the vertice: this vertex is vertex B.  Vertex B is projecting
/// its SASApack deltaE while vertex A is considering a substitution from
/// one state to another.  Vertex A has determined that the first
/// num_atoms_same atoms are shared between its current state and the
/// alternate state it's considering.  All of the atoms are ordered according
/// to the definitions in param_rotamer_trie, which is why it's the first
/// num_atoms_same that are shared and not some difficult-to-determine subset
/// of atoms. In addition to the RotamerDotsCache objects, the edge connecting
/// A and B provides node B with the set of starting masks for 1) the atoms on
/// A's alternate state that are shared with its current state in their overlap
/// with all atoms of vertex B and 2) all of the atoms on B in their overlap
/// with the atoms on vertex A that are shared.  These starting bit arrays
/// are the four FArray2D< ubyte > arrays in the parameter list.  As vertex B
/// computes its overlap with the atoms of A that differ, it stores the
/// mask index for each atom pair for each of the two sphere sizes.  It writes
/// these indices into the 3D array alt_atom_overlap_mask_inds.  Actually,
/// vertex B's alt_state_dots object does the writing.  This alt_state_dots
/// object does not think about whether it should be writing mask indices in
/// row or column order into this table; it writes it into a canonical form
/// and the SASAEdge will later transpose the table if necessary.
///
/// @param
/// neighbors_alternate_state - [in/out] - the RotamerDots object representing
///   the alternate state vertex A (the neighbor) is considering.  This object
///   is incremented during
/// neighbors_curr_state_overlap_with_this - [in] - the RotamerDotsCache object
///   held by the SASAEdge connecting A and B that represents the coverage of A's
///   current state of B's atoms.
/// this_overlap_with_neighbors_alternate - [out] - the dot cache to store the
///   results of computing the overlap of B's current state on A's alt state
/// neighbors_alternate_overlap_with_this - [out] - the dot cache to store the
///   results of computing the overlap of A's alt state on B's current state
/// alt_atom_overlap_mask_inds - [out] - the table of overlap-mask indices
///   to write the results of computing each atom pair overlap
/// mask_this_covered_by_other_small - [in/out] - starting set of overlaps
/// mask_other_covered_by_this_small - [in/out] - starting set of overlaps
/// mask_this_covered_by_other_large - [in/out] - starting set of overlaps
/// mask_other_covered_by_this_large - [in/out] - starting set of overlaps
/// num_atoms_same - [in] - the number of atoms shared between A's current
///   and alternate states.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASANode< V, E, G >::get_SASA_deltaE_for_neighbors_state_substitution
(
	RotamerDots & neighbors_alternate_state,
	RotamerDotsCache const & neighbors_curr_state_overlap_with_this,
	RotamerDotsCache & this_overlap_with_neighbors_alternate,
	RotamerDotsCache & neighbors_alternate_overlap_with_this,
	FArray3DB< short > & alt_atom_overlap_mask_inds,
	FArray2DB< ubyte > & mask_this_covered_by_other_small,
	FArray2DB< ubyte > & mask_other_covered_by_this_small,
	FArray2DB< ubyte > & mask_this_covered_by_other_large,
	FArray2DB< ubyte > & mask_other_covered_by_this_large,
	int num_atoms_same
)
{
	parent::set_alternate_state( parent::get_current_state() );

	//std::cout << "get_SASA_deltaE_for_neighbors_state_substitution node: " << parent::get_node_index() << std::endl;
	//std::cout << "curr_state_dots_" << std::endl;
	//curr_state_dots_.print();
	//std::cout << "alt_state_dots_" << std::endl;
	//alt_state_dots_.print();
	alt_state_dots_.copy_spheres_not_doubly_covered( &curr_state_dots_ );
	//std::cout << "alt_state_dots_.copy_spheres_not_doubly_covered( &curr_state_dots_ );" << std::endl;
	//std::cout << "alt_state_dots_" << std::endl;
	//alt_state_dots_.print();

	alt_state_dots_.decrement_from_cached( neighbors_curr_state_overlap_with_this );
	//std::cout << "	alt_state_dots_.decrement_from_cached( neighbors_curr_state_overlap_with_this );" << std::endl;
	//std::cout << "alt_state_dots_" << std::endl;
	//alt_state_dots_.print();

	alt_state_dots_.increment_both_and_cache(
		neighbors_alternate_state,
		this_overlap_with_neighbors_alternate,
		neighbors_alternate_overlap_with_this,
		alt_atom_overlap_mask_inds,
		mask_this_covered_by_other_small,
		mask_other_covered_by_this_small,
		mask_this_covered_by_other_large,
		mask_other_covered_by_this_large,
		num_atoms_same
	);

	//std::cout << "( " << parent::get_node_index() << ", " << get_SASA_score_difference() << ", "<< alt_state_dots_.get_score() <<  ") ";

	return get_SASA_score_difference();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::commit_considered_substitution_SASA
///
/// @brief
/// Sets the current state to the alternate state this node was asked to
/// consider.  Copies appropriate score information.  Notifies all of its
/// neighbors that it is going through with the state substitution it had been
/// considering.
///
/// @detailed
/// If the node had procrastinated the SASApack deltaE computation because it
/// thought the sim-annealer unlikely to accept the substitition, then the
/// node must now do the work it procrastinated.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASANode< V, E, G >::commit_considered_substitution_SASA()
{
	assert( parent::considering_alternate_state() );
	if ( ! calculated_SASA_deltaE_ )
	{
		++num_SASA_comps_later_made_;
		float temp = project_SASA_deltaE();
		deltaE_for_substitution_ = parent::get_alt_pd_energy_total()
			- parent::get_curr_pd_energy_total() + temp;
		//if ( deltaE_for_substitution_ < 10.0f ) std::cerr << "sasa_deltaE : " << temp << std::endl;
	}

	//call base class method
	parent::commit_considered_substitution();

	curr_state_dots_ = alt_state_dots_;
	//std::cout << "Committed substitution node " << parent::get_node_index() << std::endl;
	//curr_state_dots_.print();


	for ( int ii = 1; ii <= parent::get_num_incident_edges(); ++ii )
	{
		get_incident_sasa_edge(ii)->acknowledge_substitution_SASA();
	}

	for (int ii = 1; ii <= parent::get_num_edges_to_background_nodes(); ++ii)
	{
		get_edge_to_sasa_bg_node( ii )->acknowledge_substitution_SASA();
	}

	//if ( suggest_procrastinating_SASA_computations_ )
	//{
	//	suggest_procrastinating_SASA_computations_ = false;
	//	if ( predicted_max_mag_SASA_deltaE_for_sub_ < std::abs( SASA_deltaE_for_sub_ ) )
	//	{
	//		std::cerr << "sasaDeltaE mis-prediction: " << deltaE_for_substitution_ << " pred SASA deltaE: ";
	//		std::cerr << predicted_max_mag_SASA_deltaE_for_sub_ << " actual: ";
	//		std::cerr << SASA_deltaE_for_sub_ << std::endl;
	//	}
	//}


	track_SASA_E_min();
	//considering_alternate_state_ = false;
	return deltaE_for_substitution_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::acknowledge_neighbors_substitution_SASA
///
/// @brief
/// bookkeeping to follow a neighbors state substitution.
///
/// @detailed
/// note use of RotamerDots assignment operator (operator = ).
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::acknowledge_neighbors_substitution_SASA()
{
	//std::cout << "acknowledge_neighbors_substitution_SASA FC Node: " << parent::get_node_index() << std::endl;
	curr_state_dots_ = alt_state_dots_;
	//std::cout << "FC Node: " << parent::get_node_index() << std::endl;
	//curr_state_dots_.print();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::setRotamerCoordsForState
///
/// @brief
/// stores the coordinates for a state.
///
/// @detailed
/// RotamerCoords class stores its atoms in the order defined by
/// param_rotamer_trie.  The SASANode knows this.
///
/// @param
/// state - [in] - the index of the state in question
/// coords - [in] - the RotamerCoords object constructed for the state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASANode< V, E, G >::setRotamerCoordsForState
(
	int state,
	RotamerCoords const & coords
)
{
	assert( state > 0 && state <= parent::get_num_states() );
	self_and_bg_dots_for_states_[ state ].set_coords( coords );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::detect_any_overlap_with_rotamer
///
/// @brief
/// determines if any atom for any rotamer of this vertex overlaps with any
/// atom from some background residue.
///
/// @detailed
///
/// @param
/// bg_dots - [in] - the RotamerDots object for a background residue
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
bool
SASANode< V, E, G >::detect_any_overlap_with_rotamer(
	RotamerDots const & bg_dots
) const
{
	for (int ii = 1; ii <= parent::get_num_states(); ++ii)
	{
		if ( self_and_bg_dots_for_states_[ ii ].overlaps( bg_dots ) )
		{
			return true;
		}
	}
	return false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode< V, E, G >::initialize_overlap_with_background
///
/// @brief
/// This method computes and caches each set of overlaps this node's states
/// have with a background residue.
///
/// @detailed
/// The node stores these overlaps in two places: 1) in the input vector from
/// the SASABackgroundEdge object, and 2) in its own array of dot coverage
/// counts for the self-and-background sphere overlaps.
///
/// @param
/// bg_rotamer - [in] - the RotamerDots object for the background residue
/// overlap_with_bg_cache -[out] - the array of RotamerDotsCache objects
///   that the SASABackgroundEdge stores for the overlap of each state on this
///   residue on the background residue.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASANode< V, E, G >::initialize_overlap_with_background
(
	RotamerDots const & bg_rotamer,
	std::vector< RotamerDotsCache * > &  overlap_with_bg_cache
)
{
	for (int ii = 1; ii <= parent::get_num_states(); ++ii)
	{
		if ( self_and_bg_dots_for_states_[ ii ].overlaps( bg_rotamer ) )
		{
			overlap_with_bg_cache[ ii ] = new RotamerDotsCache();

			self_and_bg_dots_for_states_[ ii ].increment_this_and_cache(
				bg_rotamer,
				*overlap_with_bg_cache[ ii ]);
		}
		else
		{
			overlap_with_bg_cache[ ii ] = overlap_with_bg_cache[ 0 ];
		}
	}
}

template < typename V, typename E, typename G >
void
SASANode< V, E, G >::initialize_overlap_with_ligand( Ligand & ligand)
{
	for (int ii = 1; ii <= parent::get_num_states(); ++ii)
	{
		self_and_bg_dots_for_states_[ ii ].increment( ligand );
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::print
///
/// @brief
/// useful for debugging - should write information about a node to standard
/// error, but currently unimplemented.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::print() const
{
	std::cerr << "Node " << parent::get_node_index() << std::endl;
	std::cerr << "current_state: " << parent::get_current_state();
	std::cerr << " one body energy: " << parent::get_one_body_energy_current_state();
	std::cerr << std::endl;
	std::cerr << "Dots: " << std::endl;
	curr_state_dots_.force_rescore();
	curr_state_dots_.print();
}


template < typename V, typename E, typename G >
unsigned int
SASANode< V, E, G >::getMemoryUsageInBytes() const
{
	return 0; //unimplemented -- but should be!
}

template < typename V, typename E, typename G >
unsigned int
SASANode< V, E, G >::count_static_memory() const
{
	//std::cout << "calling SASANode< V, E, G >::count_static_memory()" << std::endl;
	return sizeof ( SASANode< V, E, G > );
}

template < typename V, typename E, typename G >
unsigned int
SASANode< V, E, G >::count_dynamic_memory() const
{
	//std::cout << "calling SASANode< V, E, G >::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = parent::count_dynamic_memory();
	total_memory += self_and_bg_dots_for_states_.size() * sizeof ( RotamerDots );
	return total_memory;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::prepare_for_simulated_annealing
///
/// @brief
/// invokes V's prep_for_simA method and computes the
/// "self-overlap" that each of its rotamers has.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::prepare_for_simulated_annealing()
{
	V::prepare_for_simulated_annealing(); // I want V's prep_for_simA not parent's

	if ( ! parent::get_bg_edge_vector_up_to_date_() ) { parent::update_bg_edge_vector(); }

	if ( ! have_prepared_for_simA_ )
	{
		initialize_self_overlap();
		for (int ii = 1; ii <= parent::get_num_states(); ++ii)
		{
			self_and_bg_dots_for_states_[ ii ].double_counts(); //HA - I just discovered a bug.
		}
		have_prepared_for_simA_ = true;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::overlaps
///
/// @brief
/// returns true if any sphere for any atom of any state on this vertex overlaps
/// with any atom on any sphere on a neighboring vertex.
///
/// @detailed
/// neighbor - [in] - the vertex that neighbors this vertex
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// This method could be faster if I built a rotamer trie for each vertex, so
/// I've got code in here to measure just how much time I spend on this task.
/// In a complete redesign of UBQ, I spend 8 to 10 seconds in this method.
/// In comparison, I spend 15 minutes in simulated annealing.
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
bool
SASANode< V, E, G >::overlaps( SASANode< V, E, G > * neighbor )
{
	static double time_spent_here( 0.0 );
	bool found_overlap = false;

	std::clock_t starttime = clock();
	for ( int ii = 1; ii <= parent::get_num_states(); ++ii )
	{
		for (int jj = 1; jj <= neighbor->parent::get_num_states(); ++jj )
		{
			if (self_and_bg_dots_for_states_[ ii ].overlaps(
				neighbor->self_and_bg_dots_for_states_[jj] ) )
			{
				found_overlap = true;
				break;
			}
		}
		if (found_overlap) break;
	}

	std::clock_t stoptime = clock();
	time_spent_here += ((double) (stoptime - starttime)) / CLOCKS_PER_SEC;
	std::cerr << "time spent computing overlaps: " << time_spent_here << " found overlap? " << found_overlap;
	return found_overlap;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::print_SASA_avoidance_stats
///
/// @brief
/// reports on the level of success for SASApack calculation procrastination
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::print_SASA_avoidance_stats()
{
	if (num_state_substitutions_considered_ == 0) return;

	std::cerr << "SASA Avoidance Statistics:" << std::endl;
	std::cerr << "num state substitutions considered: " << num_state_substitutions_considered_ << std::endl;
	std::cerr << "num SASA calcs procrastinated: " << num_SASA_comps_procrastinated_ << std::endl;
	std::cerr << "num SASA calcs later computed: " << num_SASA_comps_later_made_ << std::endl;
	std::cerr << "Percent Avoided: " << (double) (num_SASA_comps_procrastinated_ - num_SASA_comps_later_made_) / num_state_substitutions_considered_ << std::endl;
	if (num_SASA_comps_procrastinated_ != 0 )
	{
		std::cerr << "Worthwhile Procrastination: " << (double) (num_SASA_comps_procrastinated_ - num_SASA_comps_later_made_) / num_SASA_comps_procrastinated_ << std::endl;
	}
	else
	{
		std::cerr << "Worthwhile Procrastination: " << "N/A" << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::reset_SASA_avoidance_stats
///
/// @brief
/// resets static member variables of SASANode that measure how worthwhile
/// SASApack calculation procrastination is.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::reset_SASA_avoidance_stats()
{
	num_state_substitutions_considered_ = 0;
	num_SASA_comps_procrastinated_ = 0;
	num_SASA_comps_later_made_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::get_SASA_score_difference
///
/// @brief
/// Returns alt_state_dots_.get_score() - curr_state_dots.get_score() except
/// when either the current state or the alternate state is the unassigned
/// state; state 0.
///
/// @detailed
/// This method requires that the variables current_state_ and
/// alternate_state_  correspond to the rotamers held in curr_state_dots_
/// and alt_state_dots_.  Usually, alternate_state_ holds meaningful data
/// only if a vertex is considering a state substitution.  This method will
/// be invoked by a vertex as it considers a state substitition; it will
/// also be invoked by each of its neighbors.  That means the statement
/// alternate_state_ = current_state_;
/// must be present in
/// SASANode::get_SASA_deltaE_for_neighbors_state_substitution().
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASANode< V, E, G >::get_SASA_score_difference() const
{
	if ( parent::get_alternate_state() != 0 && parent::get_current_state() != 0)
	{
		return alt_state_dots_.get_score() - curr_state_dots_.get_score();
	}
	else
	{
		if ( parent::get_alternate_state() == 0 )
		{
			if ( parent::get_current_state() == 0 )
			{
				return 0;
			}
			else { return -1 * curr_state_dots_.get_score();}
		}
		else
		{
			return alt_state_dots_.get_score();
		}
	}

	return 1234; //makes the compiler happy; I guarantee this never to executes
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASANode::initialize_self_overlap
///
/// @brief
/// Initializes rotamer self overlap; called by
/// SASAInteractionGraph;:prepare_for_sim_annealing right before sim annealing
/// begins.  All rotamer coordinates must be loaded into the graph before
/// simulated annealing begins.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASANode< V, E, G >::initialize_self_overlap()
{
	for (int ii = 1; ii <= parent::get_num_states(); ++ii)
	{
		self_and_bg_dots_for_states_[ii].increment_self_overlap();
	}
}


template < typename V, typename E, typename G >
void
SASANode< V, E, G >::write_dot_kinemage( std::ofstream & output_kin )
{
	curr_state_dots_.write_dot_kinemage( output_kin );
}

//----------------------------------------------------------------------------//
//------------------ SASA-Pack Background Residue Node Class -----------------//
//----------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::SASABackgroundResidueNode
///
/// @brief
/// main constructor.  No default constructor, copy constructor or
/// assignment operator
///
/// @detailed
///
/// @param
/// owner - [in] - the owning interaction graph
/// node_id - [in] - the index for this node amongst its owners set
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASABackgroundResidueNode< V, E, G >::SASABackgroundResidueNode
(
	AdditionalBackgroundNodesInteractionGraph< V, E, G > * owner,
	int node_index
) :
	BackgroundNode< V, E, G > ( owner, node_index ),
	have_preped_for_simA_( false ),
	current_state_(),
	alternate_state_()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::~SASABackgroundResidueNode
///
/// @brief
/// destructor - no dynamicly allocated memory to manage -> empty
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASABackgroundResidueNode< V, E, G >::~SASABackgroundResidueNode() {}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::set_rotamer_coords
///
/// @brief
/// sets the coordinates for a background rotamer
///
/// @detailed
///
/// @param
/// input_coords - [in] - the RotamerCoords object to copy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundResidueNode< V, E, G >::set_rotamer_coords(
	RotamerCoords const & input_coords
)
{
	current_state_.set_coords( input_coords );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::detect_overlap
///
/// @brief
/// returns true if this background residue overlaps with any atom on any
/// rotamer of a SASANode.
///
/// @detailed
/// Hands the SASANode the RotamerDots object its coordinates in
/// its RotamerDots object, current_state_
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
bool
SASABackgroundResidueNode< V, E, G >::detect_overlap( SASANode< V, E, G > * node ) const
{
	return node->detect_any_overlap_with_rotamer( current_state_ );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::initialize_bg_bg_overlap
///
/// @brief
/// stores the sphere overlap for a pair of background nodes
///
/// @detailed
/// background nodes may overlap with each other, of course, but since they
/// do not move, their overlap will not change during the course of the
/// optimization.  Here we precompute that mutual overlap.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundResidueNode< V, E, G >::initialize_bg_bg_overlap
(
	SASABackgroundResidueNode< V, E, G > & other
)
{
	current_state_.increment_both( other.current_state_ );
}

template < typename V, typename E, typename G >
void
SASABackgroundResidueNode< V, E, G >::
initialize_overlap_with_ligand( Ligand & ligand)
{
	current_state_.increment( ligand );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::project_SASA_deltaE_for_substitution
///
/// @brief
/// returns the change SASApack induced by a SASANode undergoing a state
/// substitution.  The overlap between the SASANode and this
/// SASABackgroundResidueNode has been precomputed and stored in the
/// SASABackgroundEdge that connects them.  There is little work to do
/// in this subroutine.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASABackgroundResidueNode< V, E, G >::project_SASA_deltaE_for_substitution
(
	RotamerDotsCache const & nodes_curr_overlap_with_bg_res,
	RotamerDotsCache const & nodes_alt_overlap_with_bg_res
)
{
	alternate_state_.copy_spheres_not_doubly_covered( &current_state_ );
	alternate_state_.decrement_from_cached( nodes_curr_overlap_with_bg_res );
	alternate_state_.increment_from_cached( nodes_alt_overlap_with_bg_res );

	//std::cout << "( bg " << parent::get_node_index() << ", " << alternate_state_.get_score() - current_state_.get_score() <<  ", "<< alternate_state_.get_score() << ") " << std::endl;
	return alternate_state_.get_score() - current_state_.get_score();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::acknowledge_substitution
///
/// @brief
/// bookkeeping to reflect a SASANode's state substitution.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundResidueNode< V, E, G >::acknowledge_substitution_SASA()
{
	//std::cout << "BG Node: " << parent::get_node_index() << std::endl;
	current_state_ = alternate_state_;
	//std::cout << "BG Node: " << parent::get_node_index() << std::endl;
	//current_state_.print();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode< V, E, G >::get_SASA_score
///
/// @brief
/// returns the SASApack score under the current state assignment
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASABackgroundResidueNode< V, E, G >::get_SASA_score() const
{
	return current_state_.get_score();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::prepare_for_simulated_annealing
///
/// @brief
/// detects self overlap and asks its incident SASABackgroundEdges to
/// compute and cache the overlaps of this residue with all states on
/// the neighboring SASANode.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundResidueNode< V, E, G >::prepare_for_simulated_annealing()
{
	if ( ! parent::get_edge_vector_up_to_date() ) parent::update_edge_vector();

	if ( ! have_preped_for_simA_ )
	{
		have_preped_for_simA_ = true;
		initialize_self_overlap();
		for (int ii = 1; ii <= parent::get_num_incident_edges(); ++ii)
		{
			get_sasa_bg_edge( ii )->initialize_overlap_cache( current_state_ );
		}

		current_state_.double_counts();
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::print
///
/// @brief
/// useful for debugging and yet unimplemented
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASABackgroundResidueNode< V, E, G >::print() const {}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundResidueNode::initialize_self_overlap
///
/// @brief
/// initializes the self overlap for its RotamerDots object
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundResidueNode< V, E, G >::initialize_self_overlap()
{
	current_state_.increment_self_overlap();
}

template < typename V, typename E, typename G >
void
SASABackgroundResidueNode< V, E, G >::write_dot_kinemage(
	std::ofstream & output_kin
)
{
	current_state_.write_dot_kinemage( output_kin );
}

template < typename V, typename E, typename G >
unsigned int
SASABackgroundResidueNode< V, E, G >::count_static_memory() const
{
	//std::cout << "calling SASABackgroundResidueNode< V, E, G >::count_static_memory()" << std::endl;
	return sizeof ( SASABackgroundResidueNode< V, E, G > );
}

template < typename V, typename E, typename G >
unsigned int
SASABackgroundResidueNode< V, E, G >::count_dynamic_memory() const
{
	//std::cout << "calling SASABackgroundResidueNode< V, E, G >::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = parent::count_dynamic_memory();
	return total_memory;
}

//----------------------------------------------------------------------------//
//--------------------------- SASA-Pack Edge Class ---------------------------//
//----------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::SASAEdge
///
/// @brief
/// main constructor.  No default, or copy constructors, no assignment operator
///
/// @detailed
///
/// @param
/// owner - [in] - the owning interaction graph object
/// node1 - [in] - the index of the lower-indexed SASANode
/// node2 - [in] - the index of the higher-indexed SASANode
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASAEdge< V, E, G >::SASAEdge
(
	G * owner,
	int node1,
	int node2
) :
	FirstClassEdge< V, E, G > ( owner, node1, node2 ),
	node_changing_( -1 ),
	node_not_changing_( -1 ),
	num_atoms_same_as_current_( 0 ),
	changing_num_atoms_( 0 ),
	curr_atom_pair_overlap_mask_indices_(
		4, param::MAX_ATOM(), param::MAX_ATOM(), static_cast< short > ( 0 ) ),
	alt_atom_pair_overlap_mask_indices_(
		4, param::MAX_ATOM(), param::MAX_ATOM(), static_cast< short > ( 0 ) )
{
	nodes_curr_states_[ 0 ] = nodes_curr_states_[ 1 ] = 0;
	nodes_alt_states_[ 0 ] = nodes_alt_states_[ 1 ] = 0;
	num_atoms_curr_[ 0 ] = num_atoms_curr_[ 1 ] = 0;

	//variables for tracking the magnitude of the sasa_deltaE's
	max_sasa_deltaE_last_50_commits_[0] =
		max_sasa_deltaE_last_50_commits_[1] = -1.0f;
	max_sasa_deltaE_recent_50_commits_[0] =
		max_sasa_deltaE_recent_50_commits_[1] = -1.0f;
	magnitude_last_sasa_deltaE_[0] =
		magnitude_last_sasa_deltaE_[1] = -1.0f;
	num_sasa_deltaE_observations_since_update_[0] =
		num_sasa_deltaE_observations_since_update_[1] = 0;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::~SASAEdge
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASAEdge< V, E, G >::~SASAEdge() {}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::acknowledge_state_zeroed
///
/// @brief
/// respond to when one of its vertices enters the "unassigned" state
///
/// @detailed
///
/// @param
/// node_that_changed - [in] - the index of the node that changed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAEdge< V, E, G >::acknowledge_state_zeroed_SASA( int node_that_changed )
{
	node_changing_ = ( node_that_changed == parent::get_node_index(0) ? 0 : 1 );
	node_not_changing_ = ! node_changing_;
	nodes_curr_states_[ node_changing_ ] = 0;
	num_atoms_curr_[ node_changing_ ] = 0;
	inform_non_changing_node_of_neighbors_change();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::get_SASA_deltaE_for_neighbor
///
/// @brief
/// returns the change in SASApack score for the neighbor of a node that is
/// produced by the state substition it is considering.
///
/// @detailed
/// Very complicated.  See SASANode::project_deltaE_for_neighbors_state_sub
/// This edge collects cached sphere overlaps and hands this cached data to
/// the node that is not considering the state substitution.  That node
/// computes more sphere overlaps and returns a deltaE; this method then
/// passes that deltaE along.
///
/// @param
/// changing_node_alt_state_dots - [in] - the RotamerDots object for
///   the alternate state
/// num_atoms_same_as_current - [in] - the number of atoms shared
///   between the changing node's current state, and its
///   alternate state.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASAEdge< V, E, G >::get_SASA_deltaE_for_neighbor
(
	int node_considering_substitution,
	int alt_state,
	RotamerDots & changing_node_alt_state_dots,
	int num_atoms_same_as_current
)
{
	using namespace fullatom_sasa;
	using namespace param;

	static FArray2D< ubyte > mask_unchanging_covered_by_changing_small( nbytes, MAX_ATOM() );
	static FArray2D< ubyte > mask_unchanging_covered_by_changing_large( nbytes, MAX_ATOM() );
	static FArray2D< ubyte > mask_changing_covered_by_unchanging_small( nbytes, MAX_ATOM() );
	static FArray2D< ubyte > mask_changing_covered_by_unchanging_large( nbytes, MAX_ATOM() );

	node_changing_ = ( node_considering_substitution == parent::get_node_index(0) ? 0 : 1 );
	node_not_changing_ = ! node_changing_;

	nodes_alt_states_[ node_changing_ ] = alt_state;
	nodes_alt_states_[ node_not_changing_ ] = nodes_curr_states_[ node_not_changing_ ];


	num_atoms_same_as_current_ = num_atoms_same_as_current;
	changing_num_atoms_ = changing_node_alt_state_dots.get_num_atoms();

	prepare_masks(
		mask_unchanging_covered_by_changing_small,
		mask_unchanging_covered_by_changing_large,
		mask_changing_covered_by_unchanging_small,
		mask_changing_covered_by_unchanging_large);

	 float const sasa_deltaE = get_sasa_node( node_not_changing_ )->
		get_SASA_deltaE_for_neighbors_state_substitution
		(
		changing_node_alt_state_dots,
		nodes_curr_pair_dot_counts_[ node_not_changing_ ],
		nodes_alt_pair_dot_counts_[ node_changing_ ],
		nodes_alt_pair_dot_counts_[ node_not_changing_ ],
		alt_atom_pair_overlap_mask_indices_,
		mask_unchanging_covered_by_changing_small,
		mask_changing_covered_by_unchanging_small,
		mask_unchanging_covered_by_changing_large,
		mask_changing_covered_by_unchanging_large,
		num_atoms_same_as_current_
		);

	magnitude_last_sasa_deltaE_[ node_changing_ ] = std::abs( sasa_deltaE );
	return sasa_deltaE;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::acknowledge_substitution_SASA
///
/// @brief
/// bookkeeping following the decision to substitute a nodes current state
/// with the alternate it was asked to consider.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAEdge< V, E, G >::acknowledge_substitution_SASA()
{
	inform_non_changing_node_of_neighbors_change();

	nodes_curr_states_[ node_changing_ ] = nodes_alt_states_[ node_changing_ ];

	nodes_curr_pair_dot_counts_[0] = nodes_alt_pair_dot_counts_[0];
	nodes_curr_pair_dot_counts_[1] = nodes_alt_pair_dot_counts_[1];

	num_atoms_curr_[ node_changing_ ] =  changing_num_atoms_;
	copy_dot_masks_indices_from_alt_to_curr();

	track_max_magnitude_SASA_deltaE();

	return;
}

template < typename V, typename E, typename G >
void
SASAEdge< V, E, G >::track_max_magnitude_SASA_deltaE()
{
	++num_sasa_deltaE_observations_since_update_[ node_changing_ ];

	//std::cerr << "(e: " << get_node_index( 0 ) << " " << get_node_index( 1 );
	//std::cerr << "-- "<< get_node_index( node_changing_ ) << ", " <<  num_sasa_deltaE_observations_since_update_[ node_changing_ ] << ") ";

	if ( magnitude_last_sasa_deltaE_[ node_changing_ ] >
		max_sasa_deltaE_recent_50_commits_[ node_changing_ ] )
	{
		max_sasa_deltaE_recent_50_commits_[ node_changing_ ] =
			magnitude_last_sasa_deltaE_[ node_changing_ ];
	}

	if ( num_sasa_deltaE_observations_since_update_[ node_changing_] == 50 )
	{
		//std::cerr << "Edge: " << get_node_index( 0 ) << " " << get_node_index( 1 );
		//std::cerr << " max_sasa_delatE for node "<< get_node_index( node_changing_ ) << ": " << max_sasa_deltaE_recent_50_commits_[ node_changing_ ] << std::endl;

		max_sasa_deltaE_last_50_commits_[ node_changing_ ] = max_sasa_deltaE_recent_50_commits_[ node_changing_ ];
		num_sasa_deltaE_observations_since_update_[ node_changing_] = 0;
		max_sasa_deltaE_recent_50_commits_[ node_changing_ ] = -1.0f;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::declare_energies_final
///
/// @brief
/// Reduces memory usage in the two body energy table after the energy
/// calculating function declares that the energies will not change thereafter
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// In the PDEdge's version of this method, after invoking
/// 	two_body_energies_.drop_zero_submatrices_where_possible();
/// the PDEdge checks if the two body energy table it now holds is empty.  If
/// the table is empty, the edge deletes itself.  A SASAEdge should not delete
/// itself if the pair energies are all zero since the Minkowski sum of a water
/// and a van der Waal's sphere extends further out from an atoms center
/// than its (lj_atr, lj_rep, lksolv) interaction sphere.
/// However, if a SASAEdge holds no pair energies, it's
/// a very good candidate for removal  -- it just first needs to check
/// that no (vdw + 1.4 A) spheres overlap between any pair of rotamers on
/// the edges it connects.
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAEdge< V, E, G >::declare_energies_final()
{
	parent::declare_energies_final_no_deletion();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::prepare_for_simulated_annealing
///
/// @brief
/// drops zero submatrices of the AminoAcidNeighborSparseMatrix
/// and if the two_body_energies_ member then holds nothing, it checks
/// whether or not its incident nodes have any sphere overlaps.  If they don't
/// then the edge deletes itself.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAEdge< V, E, G >::prepare_for_simulated_annealing()
{
	parent::prepare_for_simulated_annealing_no_deletion();
	if ( parent::pd_edge_table_all_zeros() )
	{
		if ( ! (get_sasa_node(0)->overlaps( get_sasa_node(1)) ) )
		{
			std::cerr << " - dropped" << std::endl;
			delete this;
		}
		else
		{
			std::cerr << "- kept" << std::endl;
		}
	}
}

template < typename V, typename E, typename G >
float
SASAEdge< V, E, G >::get_max_sasa_deltaE_guess( int /*node_index*/ ) const
{
	//return max_sasa_deltaE_last_50_commits_[ ( node_index == parent::get_node_index(0) ? 0 : 1 ) ];
	return max_sasa_deltaE_last_50_commits_[ node_changing_ ];
}

template < typename V, typename E, typename G >
unsigned int
SASAEdge< V, E, G >::getMemoryUsageInBytes() const
{
	return 0; //unimplemented but should be
}

template < typename V, typename E, typename G >
unsigned int
SASAEdge< V, E, G >::count_static_memory() const
{
	//std::cout << "calling SASAEdge< V, E, G >::count_static_memory()" << std::endl;
	return sizeof ( SASAEdge< V, E, G > );
}

template < typename V, typename E, typename G >
unsigned int
SASAEdge< V, E, G >::count_dynamic_memory() const
{
	///std::cout << "calling SASAEdge< V, E, G >::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = parent::count_dynamic_memory();
	total_memory += curr_atom_pair_overlap_mask_indices_.size() * sizeof ( short );
	total_memory += alt_atom_pair_overlap_mask_indices_.size() * sizeof ( short );
	
	return total_memory;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::inform_non_changing_node_of_neighbors_change
///
/// @brief
/// tells the node that isn't considering a substitution or changing state
/// that its neighbor who is has changed.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
inline
void
SASAEdge< V, E, G >::inform_non_changing_node_of_neighbors_change()
{
	get_sasa_node( node_not_changing_ )->acknowledge_neighbors_substitution_SASA();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::prepare_masks
///
/// @brief
/// loads the ubyte arrays with cached atom pair overlap data in anticipation
/// of a call to SASANode::project_deltaE_for_neighbors_state_substitution()
///
/// @detailed
/// The first num_atoms_same_as_current_ atoms are shared between the changing
/// node's current stste and its alternate state.  This method retrieves
/// cached atom-atom-overlap-mask indices for those atoms that don't change.
/// This method looks up the overlap masks using the cached indices and preps
/// the input/output ubyte arrays.  In particular it preps the masks for all
/// atoms on the unchanging node in their overlap with the atoms shared
/// by the current and alternate state.  The overlaps for the shared atoms
/// are the same for the alternate and the current, so they are simply copied.
///
/// @param
/// mask_unchanging_covered_by_changing_small - [out]
/// mask_unchanging_covered_by_changing_large - [out]
/// mask_changing_covered_by_unchanging_small - [out]
/// mask_changing_covered_by_unchanging_large - [out]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAEdge< V, E, G >::prepare_masks(
	FArray2DB< ubyte > & mask_unchanging_covered_by_changing_small,
	FArray2DB< ubyte > & mask_unchanging_covered_by_changing_large,
	FArray2DB< ubyte > & mask_changing_covered_by_unchanging_small,
	FArray2DB< ubyte > & mask_changing_covered_by_unchanging_large
)
{
	using namespace fullatom_sasa;

	mask_unchanging_covered_by_changing_small = static_cast< ubyte > ( 0 );
	mask_unchanging_covered_by_changing_large = static_cast< ubyte > ( 0 );
	mask_changing_covered_by_unchanging_small = static_cast< ubyte > ( 0 );
	mask_changing_covered_by_unchanging_large = static_cast< ubyte > ( 0 );

	nodes_alt_pair_dot_counts_[ node_not_changing_ ].zero();
	nodes_alt_pair_dot_counts_[ node_changing_ ].copy_some_and_zero(
		nodes_curr_pair_dot_counts_[ node_changing_ ],
		num_atoms_same_as_current_,
		changing_num_atoms_);

	if (num_atoms_same_as_current_ == 0 ) return;

	int offset_for_unchanging_node;
	int offset_for_changing_node;
	int index_multiplier_for_unchanging;
	int index_multiplier_for_changing;

	if (node_changing_ == 0 )
	{
		offset_for_changing_node = 0;
		offset_for_unchanging_node = 1;
		index_multiplier_for_unchanging = 1;
		index_multiplier_for_changing = alt_atom_pair_overlap_mask_indices_.size2();
	}
	else
	{
		offset_for_changing_node = 1;
		offset_for_unchanging_node = 0;
		index_multiplier_for_unchanging = alt_atom_pair_overlap_mask_indices_.size2();
		index_multiplier_for_changing = 1;
	}

	const int unchanging_num_atoms = num_atoms_curr_[ node_not_changing_ ];

	for (int ii = 1; ii <= unchanging_num_atoms; ++ii)
	{
		FArray1Da< ubyte > ii_small_mask( mask_unchanging_covered_by_changing_small( 1, ii ), nbytes);
		FArray1Da< ubyte > ii_large_mask( mask_unchanging_covered_by_changing_large( 1, ii ), nbytes);

		for (int jj = 1; jj <= num_atoms_same_as_current_; ++jj)
		{

			int start_index_for_pair =
				4 * (
				index_multiplier_for_unchanging * ii +
				index_multiplier_for_changing * jj -
				alt_atom_pair_overlap_mask_indices_.size2() -
				1
				);

			int ii_masknum_small = curr_atom_pair_overlap_mask_indices_[
				start_index_for_pair + offset_for_unchanging_node + 2 ];
			or_dot_overlap_masks(ii_small_mask, ii_masknum_small );

			int ii_masknum_large = curr_atom_pair_overlap_mask_indices_[
				start_index_for_pair + offset_for_unchanging_node ];
			or_dot_overlap_masks(ii_large_mask, ii_masknum_large );

			if (ii_masknum_large != 0 | ii_masknum_small != 0 )
				nodes_alt_pair_dot_counts_[ node_not_changing_ ].set_atom_has_overlap( ii );

		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::copy_dot_masks_indices_from_alt_to_curr
///
/// @brief
/// after committing a considered substitution, the SASAEdge must copy the
/// cached atom-pair-overlap-mask indices from the table that the
/// unchanging node's RotamerDots object wrote them to,
/// (alt_atom_pair_overlap_mask_indices_) into the "current" table.
///
/// @detailed
/// At this point, the SASAEdge switches from the row-major or column-major
/// way that the RotamerDots wrote the indices into the alt_..._indices_ array.
/// In the new enumeration, the atoms from the higher-indexed node run across
/// the second dimension, the atoms from the smaller-indexed node run across
/// the third dimension.  The first dimension holds the set of four masks for
/// each atom pair;
/// 1 large sphere overlap on the lower-indexed node by the higher-indexed node
/// 2 large sphere overlap on the higher-indexed node by the lower-indexed node
/// 3 small sphere overlap on the lower-indexed node by the higher-indexed node
/// 4 small sphere overlap on the higher-indexed node by the lower-indexed node
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAEdge< V, E, G >::copy_dot_masks_indices_from_alt_to_curr()
{
	int offset_for_unchanging_node;
	int offset_for_changing_node;
	int index_multiplier_for_unchanging;
	int index_multiplier_for_changing;

	if (node_changing_ == 0 )
	{
		offset_for_changing_node = 0;
		offset_for_unchanging_node = 1;
		index_multiplier_for_unchanging = 1;
		index_multiplier_for_changing = alt_atom_pair_overlap_mask_indices_.size2();
	}
	else
	{
		offset_for_changing_node = 1;
		offset_for_unchanging_node = 0;
		index_multiplier_for_unchanging = alt_atom_pair_overlap_mask_indices_.size2();
		index_multiplier_for_changing = 1;
	}

	const int unchanging_num_atoms = num_atoms_curr_[ node_not_changing_ ];

	for (int ii = 1; ii <= unchanging_num_atoms; ++ii)
	{
		for (int jj = num_atoms_same_as_current_ + 1; jj <= changing_num_atoms_; ++jj)
		{
			int start_index_for_pair =
				4 * (
				index_multiplier_for_unchanging * ii +
				index_multiplier_for_changing * jj -
				alt_atom_pair_overlap_mask_indices_.size2() -
				1
				);

			int alt_linear_index = alt_atom_pair_overlap_mask_indices_.index(1, jj, ii);

			curr_atom_pair_overlap_mask_indices_[
				start_index_for_pair + offset_for_unchanging_node]
				=
				alt_atom_pair_overlap_mask_indices_[ alt_linear_index ];

			curr_atom_pair_overlap_mask_indices_[
				start_index_for_pair + offset_for_changing_node]
				=
				alt_atom_pair_overlap_mask_indices_[ alt_linear_index + 1];

			curr_atom_pair_overlap_mask_indices_[
				start_index_for_pair + offset_for_unchanging_node + 2]
				=
				alt_atom_pair_overlap_mask_indices_[ alt_linear_index + 2 ];

			curr_atom_pair_overlap_mask_indices_[
				start_index_for_pair + offset_for_changing_node + 2]
				=
				alt_atom_pair_overlap_mask_indices_[ alt_linear_index + 3];
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAEdge::or_dot_overlap_masks
///
/// @brief
/// Takes an index into the fullatom_sasa::masks array and performs a logical OR
/// to the elements of the ubyte array this is handed.
///
/// @detailed
///
/// @param
/// collected_masks - [out] - the dot coverage ubyte-array to OR the overlap mask
///   with.  I need to go back and change the naming conventions; too many
///   variables (this one included) are called "masks" when they are not
/// mask_index - [in] - the mask index
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// This method does not belong in this class
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
inline
void
SASAEdge< V, E, G >::or_dot_overlap_masks(
	FArray1DB<ubyte > & collected_masks,
	int mask_index
)
{
	if (mask_index == 0) return;
	for (int ii = 1,
		mask_li = fullatom_sasa::masks.index(1, mask_index);
		ii <= fullatom_sasa::nbytes;
		++ii,
		++mask_li)
	{
		collected_masks( ii ) |= fullatom_sasa::masks[ mask_li ];
	}
}

//----------------------------------------------------------------------------//
//---- SASA-Pack Edge Between SASANode and SASABackgroundResidueNode Class ---//
//----------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundEdge< V, E, G >::SASABackgroundEdge
///
/// @brief
/// main constructor
///
/// @detailed
///
/// @param
/// owner - [in] - the owning graph
/// first_class_node_index - [in] - the index of the first class node upon which
///   this new edge is incident
/// second_class_node_index - [in] - the index of the second class node upon
///   which this new edge is incident
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASABackgroundEdge< V, E, G >::SASABackgroundEdge
(
	AdditionalBackgroundNodesInteractionGraph < V, E, G > * owner,
	int first_class_node_index,
	int background_node_index
)
:
	BackgroundToFirstClassEdge< V, E, G >( owner, first_class_node_index, background_node_index ),
	nodes_curr_state_( 0 ),
	nodes_alt_state_( 0 ),
	node_states_coverage_of_bg_res_(
		parent::get_first_class_node()->get_num_states() + 1),
	node_states_overlap_bg_atoms_(
		parent::get_first_class_node()->get_num_states() + 1)

{
	max_sasa_deltaE_last_50_commits_ = -1.0f;
	max_sasa_deltaE_recent_50_commits_ = -1.0f;
	magnitude_last_sasa_deltaE_ = -1.0f;
	num_sasa_deltaE_observations_since_update_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundEdge::~SASABackgroundEdge
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASABackgroundEdge< V, E, G >::~SASABackgroundEdge()
{
	/*
	for (int ii = 1; ii <= get_sasa_node()->get_num_states(); ++ii)
	{
		if ( node_states_coverage_of_bg_res_[ ii ] != 0 )
		{
			if (node_states_coverage_of_bg_res_[ ii ] != node_states_coverage_of_bg_res_[ 0 ] )
			{
				delete node_states_coverage_of_bg_res_[ ii ];
			}
			node_states_coverage_of_bg_res_[ ii ] = 0;
		}
	}
	if ( node_states_coverage_of_bg_res_[ 0 ] != 0 )
	{
		delete node_states_coverage_of_bg_res_[ 0 ];
	}
	*/
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundEdge::get_SASA_deltaE_for_substitution
///
/// @brief
/// returns the change in sasa-pack energy of produced by a background
/// node in response to a considered state substitution of the first class node
///
/// @detailed
///
/// @param
/// alt_state - [in] - the alternate state that the first class node is
///   considering
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASABackgroundEdge< V, E, G >::get_SASA_deltaE_for_substitution( int alt_state )
{
	if ( node_states_coverage_of_bg_res_[ alt_state ].size() != 0 )
	{
		alt_dots_.zero();
		alt_dots_.set_non_zero_overlap( node_states_overlap_bg_atoms_[ alt_state ] );
		FArray2Da_ubyte small_dots( node_states_coverage_of_bg_res_[ alt_state ]( 1, 1, 1 ),
			fullatom_sasa::nbytes, bg_res_num_atoms_ );
		FArray2Da_ubyte large_dots( node_states_coverage_of_bg_res_[ alt_state ]( 1, 1, 2 ),
			fullatom_sasa::nbytes, bg_res_num_atoms_ );

		alt_dots_.increment_count_for_some( small_dots, large_dots, bg_res_num_atoms_ );
	}
	else
	{
		alt_dots_.zero();
	}

	nodes_alt_state_ = alt_state;
	float const sasa_deltaE = get_sasa_bg_node()->project_SASA_deltaE_for_substitution(
		curr_dots_,
		alt_dots_ );

	magnitude_last_sasa_deltaE_ = std::abs( sasa_deltaE );
	return sasa_deltaE;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundEdge::acknowledge_substitution_SASA
///
/// @brief
/// bookkeeping in response to the SASANode committing the considered
/// substitution
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundEdge< V, E, G >::acknowledge_substitution_SASA()
{
	get_sasa_bg_node()->acknowledge_substitution_SASA();

	nodes_curr_state_ = nodes_alt_state_;
	curr_dots_ = alt_dots_;
	track_max_magnitude_SASA_deltaE();
}

template < typename V, typename E, typename G >
void
SASABackgroundEdge< V, E, G >::track_max_magnitude_SASA_deltaE()
{
	++num_sasa_deltaE_observations_since_update_;
	if ( magnitude_last_sasa_deltaE_ > max_sasa_deltaE_recent_50_commits_)
	{
		max_sasa_deltaE_recent_50_commits_ =
			magnitude_last_sasa_deltaE_;
	}
	if ( num_sasa_deltaE_observations_since_update_ == 1000 )
	{
		max_sasa_deltaE_last_50_commits_ =
			max_sasa_deltaE_recent_50_commits_;
		num_sasa_deltaE_observations_since_update_= 0;
		max_sasa_deltaE_recent_50_commits_ = -1.0f;
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundEdge::acknowledge_state_change
///
/// @brief
/// bookkeeping in response to a SASANode switching states (without having
/// gone through the usual consider-substitution/commit-substitution pattern).
///
/// @detailed
/// State "0" is handled by the SASABackgroundEdge just like any other state.
/// The dot overlap cache is simply empty: the unassigned state induces no
/// overlap on the background node.  The SASABackgroundEdge keeps a stl vector
/// of RotamerDotCaches.  Position 0 in that vector holds a Cache object
/// of nothing but 0's.
///
/// @param
/// new_state - [in] - the state the SASANode switched to
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundEdge< V, E, G >::acknowledge_state_change( int new_state )
{
	if (new_state == nodes_curr_state_ ) return;

	get_SASA_deltaE_for_substitution( new_state );
	acknowledge_substitution_SASA();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundEdge::initialize_overlap_cache
///
/// @brief
/// compute the sphere overlaps of the background node with every state on
/// the first class node.  The SASABackgroundEdge hands its stl vector
/// of RotamerDotsCache objects (node_states_coverage_of_bg_res) to
/// the SASANode
///
/// @detailed
///
/// @param
/// bg_residue - [in] - the RotamerDots object representing the background
///   residue
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundEdge< V, E, G >::
initialize_overlap_cache( RotamerDots const & bg_residue )
{
	std::vector< RotamerDotsCache* > dots_for_bg_res(
		get_sasa_node()->get_num_states() + 1,
		static_cast< RotamerDotsCache* > (0) );

	bg_res_num_atoms_ = bg_residue.get_num_atoms();

	dots_for_bg_res[ 0 ] = new RotamerDotsCache();

	get_sasa_node()->initialize_overlap_with_background(
		bg_residue,
		dots_for_bg_res);

	for (int ii = 1; ii <= get_sasa_node()->get_num_states(); ++ii)
	{
		if ( dots_for_bg_res[ ii ] != dots_for_bg_res[ 0 ] )
		{

			node_states_coverage_of_bg_res_[ ii ].dimension( fullatom_sasa::nbytes,
				bg_res_num_atoms_, 2);
			node_states_coverage_of_bg_res_[ ii ] = static_cast< ubyte > (0);
			dots_for_bg_res[ ii ]->write_to_compact_array(
				node_states_coverage_of_bg_res_[ ii ] );

			node_states_overlap_bg_atoms_[ ii ].dimension( bg_res_num_atoms_ );
			node_states_overlap_bg_atoms_[ ii ] = dots_for_bg_res[ ii ]->get_non_zero_overlap();
		}
	}


	//delete rotamerDotsCache objects
	for (int ii = 1; ii <= get_sasa_node()->get_num_states(); ++ii)
	{
		if ( dots_for_bg_res[ ii ] != 0 )
		{
			if (dots_for_bg_res[ ii ] != dots_for_bg_res[ 0 ] )
			{
				delete dots_for_bg_res[ ii ];
			}
			dots_for_bg_res[ ii ] = 0;
		}
	}
	if ( dots_for_bg_res[ 0 ] != 0 )
	{
		delete dots_for_bg_res[ 0 ];
	}


}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASABackgroundEdge::prepare_for_simulated_annealing
///
/// @brief
/// Invoked by
/// AdditionalBackgroundNodesInteractionGraph::prepare_for_simulated_annealing.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// The SASABackgroundEdge has no responsibilities in this function.  However,
/// when the AdditionalBackgroundNodesInteractionGraph invokes
/// prepare_for_simulated_annealing on the SASABackgroundNode that this edge
/// is incident upon, that node will invoke initialize_overlap_cache on
/// this edge
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASABackgroundEdge< V, E, G >::prepare_for_simulated_annealing() {}

template < typename V, typename E, typename G >
float
SASABackgroundEdge< V, E, G >::get_max_sasa_deltaE_guess() const
{
	return max_sasa_deltaE_last_50_commits_;
}

template < typename V, typename E, typename G >
unsigned int
SASABackgroundEdge< V, E, G >::count_static_memory() const
{
	//std::cout << "calling SASABackgroundEdge< V, E, G >::count_static_memory()" << std::endl;
	return sizeof ( SASABackgroundEdge< V, E, G > );
}

template < typename V, typename E, typename G >
unsigned int
SASABackgroundEdge< V, E, G >::count_dynamic_memory() const
{
	//std::cout << "calling SASABackgroundEdge< V, E, G >::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = parent::count_dynamic_memory();

	for ( unsigned int ii = 0; ii < node_states_coverage_of_bg_res_.size(); ++ii )
	{
		total_memory += node_states_coverage_of_bg_res_[ ii ].size() * sizeof ( ubyte );
	}
	total_memory += node_states_coverage_of_bg_res_.size() * sizeof ( FArray3D_ubyte );
	total_memory += node_states_overlap_bg_atoms_.size() * sizeof ( bool );

	return total_memory;
}


//----------------------------------------------------------------------------//
//--------------------------- SASA-Pack Edge Class ---------------------------//
//----------------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::SASAInteractionGraph
///
/// @brief
/// Main constructor
///
/// @detailed
///
/// @param
/// num_nodes - [in] - the number of first class nodes in the graph.  The number
///   of second class nodes is reported to the graph much later.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASAInteractionGraph< V, E, G >::
SASAInteractionGraph( int num_nodes ) :
	AdditionalBackgroundNodesInteractionGraph< V, E, G > ( num_nodes ),
	num_total_residues_( 0 ),
	num_residues_assigned_as_background_( 0 ),
	num_commits_since_last_update_( 0 ),
	total_energy_current_state_assignment_( 0 ),
	total_energy_alternate_state_assignment_( 0 ),
	node_considering_alt_state_( 0 ),
	deltaE_thresh_for_avoiding_SASA_calcs_( -1.0f ),
	prepared_for_simulated_annealing_( false )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::~SASAInteractionGraph
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
SASAInteractionGraph< V, E, G >::~SASAInteractionGraph()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::prepare_for_simulated_annealing
///
/// @brief
/// Prepares the graph to begin simulated annealing.
///
/// @detailed
/// Invokes both base-class prepare_for_simulated_annealing subroutines:
/// InteractionGraphBase first, to prepare the SASANodes and SASAEdges.
/// Then the AdditionalBackgroundNodesInteractionGraph, to prepare the
/// SASABackgroundNodes, the SASABackgroundEdges, and to do a little more
/// preparing of the SASANodes.
/// Also computes background/background overlap
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// As it is written, it should only be executed once; that will change, however
/// if we make SASAInteractionGraph work with ligands (ligands that stay fixed
/// during any single sim annealing process, but that move between anealings.
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::prepare_for_simulated_annealing()
{
	if ( prepared_for_simulated_annealing_ ) return;
	prepared_for_simulated_annealing_ = true;

	detect_background_residue_and_first_class_residue_overlap();
	//InteractionGraphBase::prepare_for_simulated_annealing();
	G::prepare_for_simulated_annealing();
	parent::prepare_for_simulated_annealing();
	initialize_bg_bg_overlaps();

	std::cerr << "Num Edges In Graph: " << parent::get_num_edges() << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::blanket_assign_state_0
///
/// @brief
/// assigns state 0 -- the unassigned state -- to all (first class) vertices
/// in the graph
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::blanket_assign_state_0()
{
	for (int ii = 1; ii <= parent::get_num_nodes(); ++ii)
	{
		get_sasa_node( ii )->assign_zero_state();
	}
	update_internal_energy_totals_SASA();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::set_state_for_node
///
/// @brief
/// Assign new_state to the first class node with index node_ind
///
/// @detailed
///
/// @param
/// node_ind - [in] - the index of the (first class) node
/// new_state - [in] - the new state assigned to the node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASAInteractionGraph< V, E, G >::set_state_for_node(int node_ind, int new_state)
{
	get_sasa_node( node_ind )->assign_state_SASA( new_state );
	update_internal_energy_totals_SASA();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::set_network_state
///
/// @brief
/// Switch the state assignment of every first class node in the graph.
/// Useful, for instance, if you want to switch to the best network state
/// that you've found so far.
///
/// @detailed
///
/// @param
/// node_states - [in] - the array of states, one for each vertex in the graph
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASAInteractionGraph< V, E, G >::set_network_state( FArray1DB_int & node_states)
{
	for (int ii = 1; ii <= parent::get_num_nodes(); ++ii)
	{
		get_sasa_node( ii )->assign_state_SASA( node_states( ii ) );
	}
	update_internal_energy_totals_SASA();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::consider_substitution
///
/// @brief
/// returns the change in energy induced by switching a particular node from its
/// currently assigned state to some alternate state.
///
/// @detailed
/// Also returns the sum of the two body energies for the node in its current
/// state; the sim-annealer accepts state substitutions at higher chance if the
/// original state was also at a poor energy.
///
/// @param
/// node_ind - [in] - the index of the (first class) node
/// new_state - [in] - the alternate state that the node should consider
/// delta_energy - [out] - the change in energy induced on the entire graph
///   by substituting a node's current state with the alternate.  This energy
///   may be inaccurate if it exceeds a threshold set by the sim-annealer.
/// prev_energy_for_node - [out] - the sum of the pair-wise decomposable portion
///   of the energy function for the node's currently assigned state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::consider_substitution
(
	int node_ind,
	int new_state,
	float & delta_energy,
	float & prev_energy_for_node
)
{
	node_considering_alt_state_ = node_ind;
	delta_energy = get_sasa_node( node_ind)->
		project_deltaE_for_substitution_SASA(
		new_state,
		prev_energy_for_node,
		deltaE_thresh_for_avoiding_SASA_calcs_ );

	total_energy_alternate_state_assignment_ =
		delta_energy + total_energy_current_state_assignment_;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::commit_considered_substitution
///
/// @brief
/// Commits the substitution that the sim annealer had previously asked the
/// graph to consider.  Returns the accurate total energy for the graph.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASAInteractionGraph< V, E, G >::commit_considered_substitution()
{
	float deltaE = get_sasa_node( node_considering_alt_state_ )->
		commit_considered_substitution_SASA();

	total_energy_current_state_assignment_ =
		total_energy_current_state_assignment_ + deltaE;

	//if ( std::abs( total_energy_current_state_assignment_ - total_energy_alternate_state_assignment_) > 3.0f )
	//{
	//      std::cerr << "Tot curr: " << total_energy_current_state_assignment_;
	//      std::cerr << " Tot alt: " << total_energy_alternate_state_assignment_;
	//      std::cerr << " diff: " << std::abs( total_energy_current_state_assignment_ - total_energy_alternate_state_assignment_);
	//      std::cerr << std::endl;
	//}

	node_considering_alt_state_ = -1;
	++num_commits_since_last_update_;
	if ( num_commits_since_last_update_ == COMMIT_LIMIT_BETWEEN_UPDATES )
	{
		update_internal_energy_totals_SASA();
	}
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::get_energy_current_state_assignment
///
/// @brief
/// returns the energy of the entire graph under the current network state
/// assignment.  Also sends a bunch of information to standard error.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float SASAInteractionGraph< V, E, G >::get_energy_current_state_assignment()
{
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::get_edge_memory_usage
///
/// @brief
/// Should return a measurement of the memory used by the interaction graph
/// to store the rotamer pair energies.  Unimplemented.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
int SASAInteractionGraph< V, E, G >::get_edge_memory_usage() const
{
	//unimplemented
	return 0;
}

template < typename V, typename E, typename G >
unsigned int
SASAInteractionGraph< V, E, G >::count_static_memory() const
{
	//std::cout << "calling SASAInteractionGraph< V, E, G >::count_static_memory()" << std::endl;
	return sizeof ( SASAInteractionGraph< V, E, G > );
}

template < typename V, typename E, typename G >
unsigned int
SASAInteractionGraph< V, E, G >::count_dynamic_memory() const
{
	//std::cout << "calling SASAInteractionGraph< V, E, G >::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = parent::count_dynamic_memory();
	
	total_memory += resid_2_bgenumeration_.size() * sizeof ( int );
	total_memory += bgenumeration_2_resid_.size() * sizeof ( int );
	
	return total_memory;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::print_current_state_assignment
///
/// @brief
/// Should write the state assigned to each first class vertex to the screen.
/// Unimplemented.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::print_current_state_assignment() const
{
	//unimplemented
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::set_errorfull_deltaE_threshold
///
/// @brief
/// Allows the sim-annealer to specify a deltaE threshold above which, it is
/// no longer necessary to be very accurate.
///
/// @detailed
/// When the annealer asks the graph to consider a state substitution that
/// produces a large collision, the graph may approximate the SASApack deltaE
/// instead of performing expensive sphere overlap computations.  The deltaE
/// returned by consider_substitution() will be inaccurate, but if the annealer
/// is unlikely to accept the substitution, then time can be saved.
/// The graph guarantees that if the annealer does commit that substitution
/// that it will go back and perform the SASApack computations and return
/// an accurate total energy for the graph.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::set_errorfull_deltaE_threshold
(
	float deltaE
)
{
	std::cerr << "SASAInteractionGraph: setting errorfull deltaE threshold to " << deltaE << std::endl;
	SASANode< V, E, G >::print_SASA_avoidance_stats();
	SASANode< V, E, G >::reset_SASA_avoidance_stats();
	deltaE_thresh_for_avoiding_SASA_calcs_ = deltaE;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::get_energy_sum_for_vertex_group
///
/// @brief
/// returns the sum of the PD energy and the SASApack energy for all members
/// first class members of a user-defined vertex subset.  Unimplemented.
///
/// @detailed
///
/// @param
/// integer input - [in] - the group identifier.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
float
SASAInteractionGraph< V, E, G >::get_energy_sum_for_vertex_group( int )
{
	//apl functionality stubbed out for now
	return 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::setRotamerCoordsForNodeState
///
/// @brief
/// store the coordinates for a particular rotamer
///
/// @detailed
///
/// @param
/// node_index - [in] - the index of the (first class) node
/// state - [in] - the state identifier for the rotamer
/// coords - [in] - the (trie-ordered) RotamerCoords object for the rotamer
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::setRotamerCoordsForNodeState
(
	int node_index,
	int state,
	RotamerCoords const & coords
)
{
	get_sasa_node( node_index )->
		setRotamerCoordsForState(state, coords);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::setNumResiduesInProtein
///
/// @brief
/// tells the graph how many residues there are total in the protein
///
/// @detailed
/// The graph maintains its own enumeration for the background residues;
/// but asks that anyone wanting to refer to them use their original resid.
/// The graph has to switch back and forth between enumeration schemes and
/// must know how many residues there are total to do that efficiently.
///
/// @param
/// num_res - [in] - the total number of residues
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::setNumResiduesInProtein( int num_res )
{
	num_total_residues_ = num_res;
	resid_2_bgenumeration_.resize( num_total_residues_ );
	for (int ii = 1; ii <= num_total_residues_; ++ii)
	{
		resid_2_bgenumeration_[ii] = 0;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::setNumBackgroundResidues
///
/// @brief
/// tells the graph how many residues there are as part of the protein that are
/// not part of the combinatorial optimization process -- they are part of the
/// background
///
/// @detailed
/// The other half of switching between enumeration schemes for the background
/// residues is knowing how many background residues there are.
///
/// @param
/// num_background_residues - [in] - the number of background residues in the
///   protein.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::setNumBackgroundResidues
(
	int num_background_residues
)
{
	parent::set_num_background_nodes( num_background_residues );
	if ( parent::get_num_background_nodes() == 0) return;

	bgenumeration_2_resid_.resize( parent::get_num_background_nodes() );
	for (int ii = 1; ii <= parent::get_num_background_nodes(); ++ii)
	{
		bgenumeration_2_resid_[ii] = 0;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::setResidueAsBackgroundResidue
///
/// @brief
/// informs the graph that a particular residue is part of the background
///
/// @detailed
///
/// @param
/// residue - [in] - the residue identifier
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::setResidueAsBackgroundResidue( int residue )
{
	assert( resid_2_bgenumeration_[ residue ] == 0 );
	++num_residues_assigned_as_background_;
	resid_2_bgenumeration_[ residue ] = num_residues_assigned_as_background_;
	bgenumeration_2_resid_[ num_residues_assigned_as_background_ ] = residue;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::setBackgroundResidueRotamerCoords
///
/// @brief
/// stores the coordinates for a background residue.  The residue must first
/// have been declared to be a background residue.
///
/// @detailed
///
/// @param
/// residue - [in] - the identifier for the background residue
/// coords - [in] - the (trie-ordered) coordinates for that residue
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void SASAInteractionGraph< V, E, G >::setBackgroundResidueRotamerCoords
(
	int residue,
	RotamerCoords const & coords
)
{
	int bgid = resid_2_bgenumeration_[ residue ];
	get_sasa_bg_node( bgid )->set_rotamer_coords( coords );
}

template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::initialize_overlap_with_ligand( Ligand & ligand )
{
	std::cerr << "initializing ligand overlap" << std::endl;
	for (int ii = 1; ii <= parent::get_num_nodes(); ++ii)
	{
		get_sasa_node( ii )->initialize_overlap_with_ligand( ligand );
	}
	for (int ii = 1; ii <= parent::get_num_background_nodes(); ++ii)
	{
		get_sasa_bg_node( ii )->initialize_overlap_with_ligand( ligand );
	}
	std::cerr << "done initializing ligand overlap" << std::endl;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::create_new_node
///
/// @brief
/// factory method pattern for instantiation of SASANode objects, used by
/// InteractionGraphBase class.
///
/// @detailed
///
/// @param
/// node_index - [in] - the index of the (first class) node
/// num_states - [in] - the number of states for that node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
NodeBase*
SASAInteractionGraph< V, E, G >::create_new_node( int node_index, int num_states)
{
	return new SASANode< V, E, G >( this, node_index, num_states );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::create_new_edge
///
/// @brief
/// factory method pattern for instantiation of SASAEdge objects, used by
/// InteractionGraphBase class.
///
/// @detailed
///
/// @param
/// index1 - [in] - the index of the lower-indexed (first class) node
/// index2 - [in] - the index of the higher-indexed (first class) node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
EdgeBase*
SASAInteractionGraph< V, E, G >::create_new_edge( int index1, int index2)
{
	return new SASAEdge< V, E, G >( this, index1, index2 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::create_background_node
///
/// @brief
/// factory method pattern for instantiation of SASABackgroundResidueNode
/// objects, used by AdditionalBackgroundNodesInteractionGraph class.
///
/// @detailed
///
/// @param
/// node_index - [in] - the index of the (second class) node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundNode< V, E, G > *
SASAInteractionGraph< V, E, G >::create_background_node( int node_index )
{
	return new SASABackgroundResidueNode< V, E, G >( this, node_index );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::create_background_edge
///
/// @brief
/// factory method pattern for instantiation of SASABackgroundEdge
/// objects, used by AdditionalBackgroundNodesInteractionGraph class.
///
/// @detailed
///
/// @param
/// fc_node_index - [in] - the index of the first class node
/// sc_node_index - [in] - the index of the second class node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
BackgroundToFirstClassEdge< V, E, G > *
SASAInteractionGraph< V, E, G >::create_background_edge(
	int fc_node_index,
	int bg_node_index
)
{
	return new SASABackgroundEdge< V, E, G >( this, fc_node_index, bg_node_index );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::update_internal_energy_totals_SASA
///
/// @brief
/// After every 2^10 commits, the graph traverses its nodes and edges and
/// re-tallies the total energy of the current state assignment.  This update
/// prevents the accumulation of numerical drift, increasing accuracy.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::update_internal_energy_totals_SASA()
{
	//std::cerr << "updating internal energy sums: " << std::endl;
	//SASANode::print_SASA_avoidance_stats();

	parent::update_internal_energy_totals();

	total_energy_current_state_assignment_ = parent::get_energy_PD_current_state_assignment();
	for (int ii = 1; ii <= parent::get_num_nodes(); ++ii)
	{

		float sasa =
			get_sasa_node( ii )->get_curr_state_SASA_energy();
		//std::cerr << ", sasa = " << sasa;
		//if ( ii % 3 == 0) std::cerr << std::endl;

		total_energy_current_state_assignment_ += sasa;

	}
	//std::cerr << std::endl;
	for (int ii = 1; ii <= parent::get_num_background_nodes(); ++ii)
	{
		float bg_sasa =
			get_sasa_bg_node( ii )->get_SASA_score();
		//std::cerr << "bg res: " << bgenumeration_2_resid_[ ii ] << " sasa: " << bg_sasa << std::endl;

		total_energy_current_state_assignment_ += bg_sasa;
	}
	num_commits_since_last_update_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::
///   detect_background_residue_and_first_class_residue_overlap
///
/// @brief
/// iterates across all pairs of first- and second-class nodes to determine
/// which share sphere overlaps.  Adds a SASABackgroundEdge between any pair
/// that do.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::
detect_background_residue_and_first_class_residue_overlap()
{
	for (int ii = 1; ii <= parent::get_num_background_nodes(); ++ii)
	{
		for (int jj = 1; jj <= parent::get_num_nodes(); ++jj)
		{
			if ( get_sasa_bg_node(ii)->detect_overlap( get_sasa_node( jj ) ) )
			{
				parent::add_background_edge( jj, ii );

				//std::cerr << "Adding fc/sc edge: residue" << bgenumeration_2_resid_[ ii ] << " node: " << jj << std::endl;
			}
		}
	}
	//std::cerr << "done" << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin SASAInteractionGraph::initialize_bg_bg_overlaps
///
/// @brief
/// computes the background overlaps for all background node pairs
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::initialize_bg_bg_overlaps()
{
	for (int ii = 1; ii <= parent::get_num_background_nodes(); ++ii)
	{
		for (int jj = ii + 1; jj <= parent::get_num_background_nodes(); ++jj)
		{
			get_sasa_bg_node( ii )->initialize_bg_bg_overlap( * get_sasa_bg_node(jj) );
		}
	}
}

template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::write_dot_kinemage( std::ofstream & output_kin )
{
	output_kin << "@group {dots} off" << std::endl;
	output_kin << "@subgroup {molten_residues} dominant" << std::endl;

	for (int ii = 1; ii <= parent::get_num_nodes(); ++ii)
	{
		get_sasa_node( ii )->write_dot_kinemage( output_kin );
	}

	output_kin << "@subgroup {background_residues} dominant" << std::endl;
	for (int ii = 1; ii <= parent::get_num_background_nodes(); ++ii)
	{
		get_sasa_bg_node( ii )->write_dot_kinemage( output_kin );
	}
}


template < typename V, typename E, typename G >
void
SASAInteractionGraph< V, E, G >::print_internal_energies_for_current_state_assignment()
{
	std::cerr << "internal energiess: " << std::endl;
	for (int ii = 1; ii <= parent::get_num_nodes(); ++ii)
	{
		float one_body =
			get_sasa_node( ii )->get_curr_state_one_body_energy();
		std::cerr << "node " << ii << " 1b: " << one_body;
		float sasa =
		get_sasa_node( ii )->get_curr_state_SASA_energy();
		std::cerr << ", sasa = " << sasa;
		if ( ii % 3 == 0) std::cerr << std::endl;
	}
	std::cerr << std::endl;
	for (int ii = 1; ii <= parent::get_num_background_nodes(); ++ii)
	{
		float bg_sasa =
		get_sasa_bg_node( ii )->get_SASA_score();
		std::cerr << "bg res: " << bgenumeration_2_resid_[ ii ] << " sasa: " << bg_sasa << std::endl;
	}
	int count_edges = 0;
	for (std::list< EdgeBase*>::const_iterator iter = parent::get_edge_list_begin();
		iter != parent::get_edge_list_end();
		++iter)
	{
		float edge_energy =
			((SASAEdge< V, E, G > *) (*iter))->get_currernt_two_body_energy();
		std::cerr << "edge: " << edge_energy << " ";
		if ( count_edges % 5 == 0) std::cerr << std::endl;
		++count_edges;
	}
}

} //end namespace


#endif
